# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN

|hland_96| closely emulates the "land components" of HBV96
:cite:p:`ref-Lindstrom1997HBV96` while providing additional functionalities (for
example, the land-use type |SEALED|).  We implemented it on behalf of the `German
Federal Institute of Hydrology (BfG)`_ to model large river basins in central Europe.

The following list summarises the main components of |hland_96|:

 * Apply different correction factors to the liquid and the frozen amount of
   precipitation.
 * Use a submodel like |evap_aet_hbv96| for calculating actual evapotranspiration
   (which, for best compatibility with :cite:t:`ref-Lindstrom1997HBV96`, could use
   |evap_pet_hbv96| for estimating potential evapotranspiration).
 * Simulate interception via simple "bucket" storages.
 * Reflect local spatial heterogeneity of snow by defining snow classes.
 * Redistribute snow from higher to lower zones to prevent "snow towers" in high
   mountain areas.
 * Use a degree-day factor reflecting the seasonal variability of radiation for
   calculating snowmelt.
 * Consider both the melting of ice and the (re)freezing of water within the snow
   layer.
 * Apply a saturation excess mechanism for the generation of direct runoff.
 * Calculate a (nearly) complete and immediate runoff response for sealed areas.
 * Provide an optional "response area" option, which modifies the usual direct runoff
   and percolation calculation.
 * Distinguish between an upper zone layer related to direct runoff and a lower zone
   layer related to base flow.
 * Pass percolation from the upper to the lower zone layer and capillary rise from the
   upper zone layer to the soil layer.
 * Consider water areas as "groundwater lakes" that are part of the lower zone layer.
 * In contrast to the original HBV96 implementation, handle both the upper and the
   lower zone layers as nonlinear storages.
 * Conceptualise the melting of glacial ice with an additional application of the
   degree-day method.
 * Optionally, use a submodel like |rconc_uh| to calculate runoff concentration.


The following figure shows the general structure of |hland_96|.  Note that zones of
type |FIELD| and |FOREST| are based on the same set of process equations. In this
scheme, the submodel |rconc_uh| is included (which is optional):

.. image:: HydPy-H-HBV96.png

Integration tests
=================

.. how_to_understand_integration_tests::

The following integration tests rely on the meteorological input data used for testing
the application model |lland_dd|. The values of the input sequences |P| (precipitation)
and |T| (temperature) are copy-pasted.  The |evap_inputs.NormalEvapotranspiration|
values are the |evap_fluxes.ReferenceEvapotranspiration| values calcuted by
|evap_ret_tw2002| but divided by 0.4 to account for the selected value of the
evaporation adjustment factor |evap_control.EvapotranspirationFactor|.  Hopefully, this
will make it easier to draw comparisons between both models.

We perform all integration tests over five days with a simulation step of one hour:

>>> from hydpy import pub
>>> pub.timegrids = "01.01.2000", "05.01.2000", "1h"

First, we prepare the model instance and build the connections to element `land` and
node `outlet`:

>>> from hydpy.models.hland_96 import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> land = Element("land", outlets=outlet)
>>> land.model = model

We perform most tests using a single zone of 1 km³ at an altitude of 100 m, consisting
of a single snow class:

>>> nmbzones(1)
>>> sclass(1)
>>> area(1.0)
>>> zonearea(1.0)
>>> zonez(1.0)

We assume a runoff coefficient of one for zones defined as |SEALED|:

>>> psi(1.0)

For a start, we select |FIELD| as the only zone type (note that the results for the
land-use type |FOREST| would be the same):

>>> zonetype(FIELD)

The following set of control parameter values tries to configure application model
|hland_96| in a manner that allows retracing the influence of the different implemented
methods on the shown results:

>>> pcorr(1.1)
>>> pcalt(0.1)
>>> rfcf(1.1)
>>> sfcf(1.3)
>>> tcorr(0.6)
>>> tcalt(0.6)
>>> icmax(2.0)
>>> sfdist(1.0)
>>> smax(inf)
>>> sred(0.0)
>>> tt(0.0)
>>> ttint(2.0)
>>> dttm(1.0)
>>> cfmax(0.5)
>>> cfvar(0.1)
>>> gmelt(1.0)
>>> gvar(0.2)
>>> cfr(0.1)
>>> whc(0.4)
>>> fc(200.0)
>>> beta(2.0)
>>> percmax(0.5)
>>> cflux(0.1)
>>> alpha(2.0)
>>> k(0.001)
>>> k4(0.005)
>>> gamma(0.0)

|hland_96| requires a submodel for calculating actual evapotranspiration.  Therefore,
we select |evap_aet_hbv96|, which also closely follows equations used by HBV96
:cite:p:`ref-Lindstrom1997HBV96`.  The same holds for |evap_pet_hbv96|, which will
provide |evap_aet_hbv96| with HBV96-like potential evapotranspiration estimates:

>>> with model.add_aetmodel_v1("evap_aet_hbv96"):
...     temperaturethresholdice(0.0)
...     soilmoisturelimit(0.8)
...     excessreduction(0.5)
...     with model.add_petmodel_v1("evap_pet_hbv96"):
...         evapotranspirationfactor(0.7)
...         airtemperaturefactor(0.1)
...         altitudefactor(-0.1)
...         precipitationfactor(0.1)

|hland_96| can use a submodel that follows the |RConcModel_V1| to consider additional
runoff concentration processes. A possible choice is |rconc_uh|, which implements the
unit hydrograph method and provides a convenient way to set the required ordinates in
the triangle shape typical for HBV96:

>>> with model.add_rconcmodel_v1("rconc_uh"):
...     uh("triangle", tb=3.0)

We initialise a test function object, which prepares and runs the tests and prints
their results for the given sequences:

>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = inputs.p, fluxes.rt, fluxes.qt
>>> IntegrationTest.plotting_options.axis2 = inputs.t
>>> test = IntegrationTest(land)
>>> test.dateformat = "%d/%m %H:00"

Initially, relative soil moisture is 50 %, the lower zone layer contains only 10 mm,
and all other storages are empty:

>>> test.inits = ((states.ic, 0.0),
...               (states.sp, 0.0),
...               (states.wc, 0.0),
...               (states.sm, 100.0),
...               (states.uz, 0.0),
...               (states.lz, 10.0),
...               (model.rconcmodel.sequences.logs.quh, 0.05))

As mentioned above, the values of the input sequences |P| and |hland_inputs.T| of
|hland_96| stem from :ref:`here <lland_dd_acker_summer>`.  For educational purposes, we
again use unrealistically high values of |evap_inputs.NormalEvapotranspiration|.  For
the sake of simplicity, we define |evap_inputs.NormalAirTemperature| to be constantly
1 °C below |hland_inputs.T|:

>>> inputs.p.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0, 2.1, 10.4, 3.5, 3.4,
...     1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1, 20.7, 37.9, 8.2, 3.6, 7.5, 18.5,
...     15.4, 6.3, 1.9, 4.9, 2.7, 0.5, 0.2, 0.5, 2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6,
...     0.7, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     1.3, 0.0, 0.0, 0.0, 0.7, 0.4, 0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
>>> inputs.t.series = (
...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2, 35.2, 37.1,
...     31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8, 19.6, 19.2, 19.2, 19.2,
...     18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8, 19.0, 19.2, 19.3, 19.0, 18.8, 18.7,
...     17.8, 17.4, 17.3, 16.8, 16.5, 16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3,
...     14.9, 15.7, 16.0, 16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3,
...     18.1, 16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1, 11.9,
...     12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8, 22.2, 20.1, 17.8,
...     15.2, 14.5, 12.4, 11.7, 11.9)
>>> petinputs = model.aetmodel.petmodel.sequences.inputs
>>> petinputs.normalairtemperature.series = inputs.t.series - 1.0
>>> petinputs.normalevapotranspiration.series = (
...     0.100707, 0.097801, 0.096981, 0.09599, 0.096981, 0.102761, 0.291908, 1.932875,
...     4.369536, 7.317556, 8.264362, 9.369867, 5.126178, 6.62503, 7.397619, 2.39151,
...     1.829834, 1.136569, 0.750986, 0.223895, 0.099425, 0.098454, 0.098128, 0.097474,
...     0.097474, 0.097474, 0.096981, 0.096652, 0.096321, 0.09599, 0.187298, 1.264612,
...     3.045538, 1.930758, 2.461001, 6.215945, 3.374783, 8.821555, 4.046025, 2.110757,
...     2.239257, 2.877848, 1.591452, 0.291604, 0.092622, 0.092451, 0.091248, 0.089683,
...     0.089858, 0.089683, 0.088805, 0.089157, 0.090207, 0.091593, 0.154861, 0.470369,
...     1.173726, 4.202296, 4.359715, 5.305753, 5.376027, 4.658915, 7.789594, 4.851567,
...     5.30692, 3.286036, 1.506216, 0.274762, 0.087565, 0.085771, 0.084317, 0.083215,
...     0.082289, 0.0845, 0.084864, 0.083584, 0.0834, 0.084864, 0.310229, 1.391958,
...     3.195876, 5.191651, 7.155036, 8.391432, 8.391286, 10.715238, 9.383394, 7.861915,
...     6.298329, 2.948416, 1.309232, 0.32955, 0.089508, 0.085771, 0.0845, 0.084864)

We memorise the initial conditions to check later if |hland_96| holds the water balance:

>>> test.reset_inits()
>>> conditions = model.conditions

.. _hland_96_field:

field
_____

In the first example, we disable the |RespArea| option and set a relatively large value
for the accuracy-related parameter |RecStep|:

>>> resparea(False)
>>> recstep(100)

The following results show the response of application model |hland_96| to the given
extreme precipitation event.  The intense evaporation weakens the response markedly.
One striking difference to other models like |lland_dd| is the block-like appearance of
percolation (|Perc|), which is one reason for the unusual transitions between event
periods (consisting of both runoff components |Q0| and |Q1|) and the subsequent pure
base flow periods (consisting of |Q1| only):

.. integration-test::

    >>> test("hland_96_field")
    |        date |    p |    t |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inrc |     outrc |        rt |       qt |       ic |  sp |  wc |         sm |        uz |        lz |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.048465 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.061111 |  0.061111 | 0.016975 |      0.0 | 0.0 | 0.0 |  99.951535 |       0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.047044 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.088833 |  0.088833 | 0.024676 |      0.0 | 0.0 | 0.0 |  99.904491 |       0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046628 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |   0.04975 |   0.04975 |  0.01382 |      0.0 | 0.0 | 0.0 |  99.857863 |       0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04613 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049502 |  0.049502 |  0.01375 |      0.0 | 0.0 | 0.0 |  99.811734 |       0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046584 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049254 |  0.049254 | 0.013682 |      0.0 | 0.0 | 0.0 |   99.76515 |       0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.049338 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049008 |  0.049008 | 0.013613 |      0.0 | 0.0 | 0.0 |  99.715812 |       0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.140081 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048763 |  0.048763 | 0.013545 |      0.0 | 0.0 | 0.0 |   99.57573 |       0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.92625 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048519 |  0.048519 | 0.013477 |      0.0 | 0.0 | 0.0 |  98.649481 |       0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  2.07444 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048276 |  0.048276 |  0.01341 |      0.0 | 0.0 | 0.0 |  96.575041 |       0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.400961 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048035 |  0.048035 | 0.013343 |      0.0 | 0.0 | 0.0 |   93.17408 |       0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.705742 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047795 |  0.047795 | 0.013276 |      0.0 | 0.0 | 0.0 |  89.468338 |       0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.03435 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047556 |  0.047556 |  0.01321 |      0.0 | 0.0 | 0.0 |  85.433988 |       0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.107634 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047318 |  0.047318 | 0.013144 |      0.0 | 0.0 | 0.0 |  83.326354 |       0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.656691 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047081 |  0.047081 | 0.013078 |      0.0 | 0.0 | 0.0 |  80.669664 |       0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.871924 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046846 |  0.046846 | 0.013013 |      0.0 | 0.0 | 0.0 |   77.79774 |       0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.242 |    0.242 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.873977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046612 |  0.046612 | 0.012948 |      0.0 | 0.0 | 0.0 |  76.923763 |       0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.677397 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046379 |  0.046379 | 0.012883 |      0.0 | 0.0 | 0.0 |  76.246366 |       0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.417048 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046147 |  0.046147 | 0.012819 |      0.0 | 0.0 | 0.0 |  75.829319 |       0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  1.573 | 0.494092 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.117083 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045916 |  0.045916 | 0.012754 | 1.078908 | 0.0 | 0.0 |  75.712235 |       0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  6.776 |  0.08755 |  5.854908 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.854908 |  0.839058 | 0.0 | 0.022103 | 0.059636 |  0.779422 |      0.5 |  0.000006 | 0.0 | 0.047958 |  0.047963 |  0.046243 |  0.046243 | 0.012845 |  1.91245 | 0.0 | 0.0 |  80.765617 |  0.279417 |  9.543605 | 0.012845 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  3.509 |   0.0539 |   3.42145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.42145 |   0.55796 | 0.0 | 0.014096 | 0.058185 |  0.499775 |      0.5 |  0.000022 | 0.0 | 0.050218 |   0.05024 |  0.047963 |  0.047963 | 0.013323 |   1.9461 | 0.0 | 0.0 |  83.673196 |   0.27917 |  9.993387 | 0.013323 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  5.929 | 0.041902 |    5.8751 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    5.8751 |  1.028319 | 0.0 | 0.011598 |  0.05574 |  0.972579 |      0.5 |  0.000168 | 0.0 | 0.052467 |  0.052635 |  0.050266 |  0.050266 | 0.013963 | 1.958098 | 0.0 | 0.0 |  88.564118 |  0.751581 |  10.44092 | 0.013963 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 | 12.826 | 0.020954 | 12.784098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.784098 |  2.506835 | 0.0 | 0.006475 | 0.050579 |  2.456256 |      0.5 |  0.006898 | 0.0 | 0.054705 |  0.061602 |  0.054095 |  0.054095 | 0.015026 | 1.979046 | 0.0 | 0.0 |  98.885486 |  2.700939 | 10.886215 | 0.015026 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.121 | 0.074152 |  0.100046 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.100046 |  0.024457 | 0.0 | 0.022944 | 0.050519 | -0.026062 |      0.5 |  0.014479 | 0.0 | 0.056931 |   0.07141 |  0.061789 |  0.061789 | 0.017164 | 1.925848 | 0.0 | 0.0 |  98.988651 |  2.160398 | 11.329284 | 0.017164 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.847 |  0.06896 |  0.772848 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.772848 |  0.189324 | 0.0 | 0.021469 | 0.050214 |   0.13911 |      0.5 |  0.007758 | 0.0 | 0.059146 |  0.066905 |  0.068229 |  0.068229 | 0.018953 |  1.93104 | 0.0 | 0.0 |   99.60092 |  1.791749 | 11.770138 | 0.018953 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   3.63 | 0.052207 |   3.56104 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.56104 |  0.883169 | 0.0 | 0.016695 | 0.048861 |  0.834308 |      0.5 |  0.007549 | 0.0 | 0.061351 |    0.0689 |  0.068349 |  0.068349 | 0.018986 | 1.947793 | 0.0 | 0.0 | 102.310958 |  2.118508 | 12.208787 | 0.018986 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.541 | 0.057919 |  2.488793 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.488793 |  0.651288 | 0.0 | 0.018859 | 0.047926 |  0.603362 |      0.5 |  0.010164 | 0.0 | 0.063544 |  0.073708 |  0.069525 |  0.069525 | 0.019312 | 1.942081 | 0.0 | 0.0 |  104.17753 |  2.211707 | 12.645243 | 0.019312 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 12.584 | 0.021144 | 12.526081 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.526081 |  3.398626 | 0.0 | 0.007489 | 0.043348 |  3.355278 |      0.5 |  0.055198 | 0.0 | 0.065726 |  0.120925 |  0.083132 |  0.083132 | 0.023092 | 1.978856 | 0.0 | 0.0 | 113.340843 |  5.011787 | 13.079517 | 0.023092 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.235 | 0.048561 |  4.213856 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.213856 |  1.353295 | 0.0 |  0.01764 | 0.041899 |  1.311396 |      0.5 |  0.153723 | 0.0 | 0.067898 |  0.221621 |  0.132809 |  0.132809 | 0.036891 | 1.951439 | 0.0 | 0.0 | 116.225662 |   5.66946 | 13.511619 | 0.036891 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.114 | 0.048983 |  4.065439 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.065439 |   1.37294 | 0.0 | 0.018209 | 0.040541 |  1.332399 |      0.5 |  0.215358 | 0.0 | 0.070058 |  0.285416 |  0.213421 |  0.213421 | 0.059283 | 1.951017 | 0.0 | 0.0 | 118.940493 |    6.2865 | 13.941561 | 0.059283 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.452 | 0.124728 |  1.403017 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.403017 |  0.496206 | 0.0 | 0.046729 | 0.040076 |   0.45613 |      0.5 |  0.232354 | 0.0 | 0.072208 |  0.304562 |  0.275494 |  0.275494 | 0.076526 | 1.875272 | 0.0 | 0.0 | 119.840651 |  6.010276 | 14.369353 | 0.076526 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.121 |  0.96204 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.360406 |  0.04008 |  -0.04008 |      0.5 |  0.180378 | 0.0 | 0.074347 |  0.254725 |  0.289232 |  0.289232 | 0.080342 | 1.034232 | 0.0 | 0.0 | 119.520324 |  5.289819 | 14.795007 | 0.080342 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    0.0 | 1.034232 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.531595 |  0.04024 |  -0.04024 |      0.5 |  0.122017 | 0.0 | 0.076475 |  0.198492 |  0.253303 |  0.253303 | 0.070362 |      0.0 | 0.0 | 0.0 | 118.028969 |  4.627562 | 15.218532 | 0.070362 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.097079 | 0.040986 | -0.040986 |      0.5 |  0.080568 | 0.0 | 0.078593 |  0.159161 |  0.202248 |  0.202248 |  0.05618 |      0.0 | 0.0 | 0.0 | 116.972875 |  4.006009 | 15.639939 |  0.05618 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.320389 | 0.041514 | -0.041514 |      0.5 |  0.051191 | 0.0 |   0.0807 |  0.131891 |  0.161841 |  0.161841 | 0.044956 |      0.0 | 0.0 | 0.0 |    115.694 |  3.413304 | 16.059239 | 0.044956 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.420519 | 0.042153 | -0.042153 |      0.5 |  0.030713 | 0.0 | 0.082796 |  0.113509 |  0.133866 |  0.133866 | 0.037185 |      0.0 | 0.0 | 0.0 | 112.315634 |  2.840438 | 16.476443 | 0.037185 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.356 | 1.680962 |     2.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     2.356 |  0.743012 | 0.0 | 0.598694 | 0.043036 |  0.699976 |      0.5 |  0.025159 | 0.0 | 0.084882 |  0.110042 |  0.116823 |  0.116823 | 0.032451 | 0.319038 | 0.0 | 0.0 | 113.372964 |  3.015255 | 16.891561 | 0.032451 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.139 |      2.0 |  5.458038 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.458038 |  1.753863 | 0.0 | 1.880771 | 0.041461 |  1.712401 |      0.5 |  0.048171 | 0.0 | 0.086958 |  0.135128 |  0.116387 |  0.116387 |  0.03233 |      0.0 | 0.0 | 0.0 |  115.23783 |  4.179486 | 17.304603 |  0.03233 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.331 |    1.331 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.680563 | 0.042381 | -0.042381 |      0.5 |  0.058507 | 0.0 | 0.089023 |   0.14753 |  0.132309 |  0.132309 | 0.036753 |      0.0 | 0.0 | 0.0 | 113.599648 |  3.578598 |  17.71558 | 0.036753 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 25.047 | 0.132786 |    23.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    23.047 |  7.435469 | 0.0 | 0.053632 | 0.035394 |  7.400075 |      0.5 |   0.40815 | 0.0 | 0.091078 |  0.499228 |  0.222929 |  0.222929 | 0.061925 | 1.867214 | 0.0 | 0.0 | 129.192941 | 10.070522 | 18.124502 | 0.061925 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 45.859 | 0.017578 | 45.726214 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.726214 | 19.080196 | 0.0 | 0.008562 | 0.022081 | 19.058115 |      0.5 |  5.990579 | 0.0 | 0.093123 |  6.083701 |  1.662067 |  1.662067 | 0.461685 | 1.982422 | 0.0 | 0.0 | 155.852479 | 22.638059 |  18.53138 | 0.461685 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.922 | 0.821583 |  9.904422 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.904422 |  6.014459 | 0.0 | 0.410182 | 0.020129 |   5.99433 |      0.5 |   8.87815 | 0.0 | 0.095157 |  8.973307 |  5.484842 |  5.484842 | 1.523567 | 1.178417 | 0.0 | 0.0 | 159.352388 | 19.254239 | 18.936223 | 1.523567 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.356 | 0.792694 |  3.534417 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.534417 |  2.243752 | 0.0 | 0.396347 | 0.019678 |  2.224074 |      0.5 |  5.214741 | 0.0 | 0.097181 |  5.311922 |  7.517531 |  7.517531 | 2.088203 | 1.207306 | 0.0 | 0.0 | 160.266384 | 15.763572 | 19.339042 | 2.088203 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.075 | 0.090607 |  8.282306 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.282306 |  5.318341 | 0.0 | 0.045304 | 0.018385 |  5.299956 |      0.5 |  4.208086 | 0.0 | 0.099195 |  4.307281 |   5.90231 |   5.90231 |  1.63953 | 1.909393 | 0.0 | 0.0 | 163.203431 | 16.355442 | 19.739846 |  1.63953 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 22.385 | 0.007604 | 22.294393 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.294393 | 14.845479 | 0.0 | 0.003802 | 0.014674 | 14.830806 |      0.5 |  8.285963 | 0.0 | 0.101199 |  8.387162 |  5.437175 |  5.437175 | 1.510326 | 1.992396 | 0.0 | 0.0 | 170.663216 | 22.400285 | 20.138647 | 1.510326 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 18.634 | 0.011044 | 18.626396 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.626396 | 13.562779 | 0.0 | 0.005522 | 0.012137 | 13.550643 |      0.5 | 12.233902 | 0.0 | 0.103193 | 12.337096 |  8.358285 |  8.358285 | 2.321746 | 1.988956 | 0.0 | 0.0 | 175.733447 | 23.217025 | 20.535454 | 2.321746 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.623 | 0.032783 |  7.611956 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.611956 |  5.876857 | 0.0 | 0.016392 | 0.011266 |  5.865591 |      0.5 |  9.245942 | 0.0 | 0.105177 |   9.35112 | 10.795782 | 10.795782 | 2.998828 | 1.967217 | 0.0 | 0.0 |  177.46342 | 19.336674 | 20.930277 | 2.998828 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.299 | 0.054873 |  2.266217 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.266217 |  1.784264 | 0.0 | 0.027436 | 0.011027 |  1.773237 |      0.5 |   5.11831 | 0.0 | 0.107151 |  5.225461 |  9.097857 |  9.097857 | 2.527182 | 1.945127 | 0.0 | 0.0 | 177.928964 | 15.491601 | 21.323125 | 2.527182 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  5.929 | 0.038243 |  5.874127 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.874127 |  4.649183 | 0.0 | 0.019122 | 0.010423 |   4.63876 |      0.5 |  3.860955 | 0.0 | 0.109116 |   3.97007 |  5.863298 |  5.863298 | 1.628694 | 1.961757 | 0.0 | 0.0 | 179.145209 | 15.769407 |  21.71401 | 1.628694 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.267 |  0.04981 |  3.228757 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.228757 |  2.590513 | 0.0 | 0.024905 | 0.010108 |  2.580405 |      0.5 |   3.41746 | 0.0 |  0.11107 |   3.52853 |  4.150926 |  4.150926 | 1.153035 |  1.95019 | 0.0 | 0.0 | 179.768657 | 14.432352 |  22.10294 | 1.153035 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.605 | 0.064366 |   0.55519 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.55519 |  0.448549 | 0.0 | 0.032183 | 0.010062 |  0.438486 |      0.5 |  2.298065 | 0.0 | 0.113015 |   2.41108 |  3.378328 |  3.378328 | 0.938424 | 1.935634 | 0.0 | 0.0 | 179.853178 | 12.072772 | 22.489925 | 0.938424 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.242 | 0.067009 |  0.177634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.177634 |  0.143649 | 0.0 | 0.033505 | 0.010056 |  0.133593 |      0.5 |  1.392466 | 0.0 |  0.11495 |  1.507416 |  2.458588 |  2.458588 | 0.682941 | 1.932991 | 0.0 | 0.0 | 179.863714 | 10.313899 | 22.874975 | 0.682941 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.605 | 0.065382 |  0.537991 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.537991 |  0.435113 | 0.0 | 0.032691 | 0.010017 |  0.425096 |      0.5 |  0.939834 | 0.0 | 0.116875 |  1.056709 |  1.608073 |  1.608073 | 0.446687 | 1.934618 | 0.0 | 0.0 | 179.943918 |  9.299161 |   23.2581 | 0.446687 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  2.904 | 0.052751 |  2.838618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.838618 |  2.297848 | 0.0 | 0.026376 | 0.009758 |  2.288091 |      0.5 |  0.931867 | 0.0 | 0.118791 |  1.050657 |  1.155521 |  1.155521 | 0.320978 | 1.947249 | 0.0 | 0.0 |  180.46807 | 10.155384 |  23.63931 | 0.320978 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.484 | 0.113609 |  0.431249 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.431249 |   0.35113 | 0.0 | 0.056805 | 0.009726 |  0.341405 |      0.5 |  0.890249 | 0.0 | 0.120697 |  1.010945 |  1.043177 |  1.043177 | 0.289771 | 1.886391 | 0.0 | 0.0 |  180.50111 |   9.10654 | 24.018613 | 0.289771 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.242 | 0.353524 |  0.128391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.128391 |  0.104577 | 0.0 | 0.176762 | 0.009738 |  0.094839 |      0.5 |  0.631758 | 0.0 | 0.122593 |  0.754351 |  0.962749 |  0.962749 |  0.26743 | 1.646476 | 0.0 | 0.0 | 180.357899 |  8.069621 |  24.39602 |  0.26743 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 | 0.903769 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.451885 | 0.009821 | -0.009821 |      0.5 |  0.437555 | 0.0 |  0.12448 |  0.562035 |  0.768635 |  0.768635 |  0.21351 | 0.742707 | 0.0 | 0.0 | 179.915836 |  7.122246 |  24.77154 |  0.21351 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 | 0.742707 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.864415 | 0.010042 | -0.010042 |      0.5 |  0.302641 | 0.0 | 0.126358 |  0.428999 |  0.575208 |  0.575208 |  0.15978 |      0.0 | 0.0 | 0.0 | 177.061463 |  6.309562 | 25.145182 |  0.15978 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.055807 | 0.011469 | -0.011469 |      0.5 |  0.210436 | 0.0 | 0.128226 |  0.338662 |  0.438487 |  0.438487 | 0.121802 |      0.0 | 0.0 | 0.0 | 174.017125 |  5.587657 | 25.516957 | 0.121802 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     1.146 |  0.867578 | 0.0 | 1.982694 | 0.012852 |  0.854726 |      0.5 |  0.183057 | 0.0 | 0.130085 |  0.313141 |  0.353065 |  0.353065 | 0.098074 |      0.0 | 0.0 | 0.0 | 172.325705 |  5.759327 | 25.886872 | 0.098074 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.37986 | 0.013837 | -0.013837 |      0.5 |  0.159478 | 0.0 | 0.131934 |  0.291412 |  0.313984 |  0.313984 | 0.087218 |      0.0 | 0.0 | 0.0 | 168.959683 |  5.086012 | 26.254937 | 0.087218 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.277978 |  0.01552 |  -0.01552 |      0.5 |  0.108875 | 0.0 | 0.133775 |   0.24265 |  0.285405 |  0.285405 | 0.079279 |      0.0 | 0.0 | 0.0 | 165.697224 |  4.461616 | 26.621163 | 0.079279 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.602665 | 0.017151 | -0.017151 |      0.5 |  0.072459 | 0.0 | 0.135606 |  0.208064 |    0.2458 |    0.2458 | 0.068278 |      0.0 | 0.0 | 0.0 | 160.111711 |  3.872006 | 26.985557 | 0.068278 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.735707 | 0.019944 | -0.019944 |      0.5 |   0.04635 | 0.0 | 0.137428 |  0.183778 |  0.210353 |  0.210353 | 0.058431 |      0.0 | 0.0 | 0.0 | 156.395948 |  3.305712 | 27.348129 | 0.058431 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.994839 | 0.021802 | -0.021802 |      0.5 |  0.027967 | 0.0 | 0.139241 |  0.167208 |  0.185492 |  0.185492 | 0.051526 |      0.0 | 0.0 | 0.0 | 152.422911 |  2.755943 | 27.708889 | 0.051526 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.410799 | 0.023789 | -0.023789 |      0.5 |  0.015488 | 0.0 | 0.141044 |  0.156532 |  0.168518 |  0.168518 |  0.04681 |      0.0 | 0.0 | 0.0 |   150.0359 |  2.216667 | 28.067844 |  0.04681 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.087741 | 0.024982 | -0.024982 |      0.5 |  0.007522 | 0.0 | 0.142839 |  0.150362 |  0.157533 |  0.157533 | 0.043759 |      0.0 | 0.0 | 0.0 | 148.973141 |  1.684163 | 28.425005 | 0.043759 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.19702 | 0.025513 | -0.025513 |      0.5 |  0.002944 | 0.0 | 0.144625 |  0.147569 |  0.151112 |  0.151112 | 0.041976 |      0.0 | 0.0 | 0.0 | 148.801635 |  1.155705 |  28.78038 | 0.041976 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062717 | 0.025599 | -0.025599 |      0.5 |  0.000766 | 0.0 | 0.146402 |  0.147168 |  0.148101 |  0.148101 | 0.041139 |      0.0 | 0.0 | 0.0 | 148.764517 |   0.62934 | 29.133978 | 0.041139 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061417 | 0.025618 | -0.025618 |      0.5 |  0.000073 | 0.0 |  0.14817 |  0.148243 |  0.147496 |  0.147496 | 0.040971 |      0.0 | 0.0 | 0.0 | 148.728719 |  0.103649 | 29.485808 | 0.040971 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060361 | 0.025636 | -0.025636 | 0.078013 |       0.0 | 0.0 | 0.147819 |  0.147819 |   0.14791 |   0.14791 | 0.041086 |      0.0 | 0.0 | 0.0 | 148.693993 |       0.0 | 29.416002 | 0.041086 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.059548 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.14708 |   0.14708 |  0.147749 |  0.147749 | 0.041041 |      0.0 | 0.0 | 0.0 | 148.634445 |       0.0 | 29.268922 | 0.041041 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.058862 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.146345 |  0.146345 |  0.147081 |  0.147081 | 0.040856 |      0.0 | 0.0 | 0.0 | 148.575584 |       0.0 | 29.122577 | 0.040856 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060419 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.145613 |  0.145613 |  0.146345 |  0.146345 | 0.040652 |      0.0 | 0.0 | 0.0 | 148.515165 |       0.0 | 28.976964 | 0.040652 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060655 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.144885 |  0.144885 |  0.145614 |  0.145614 | 0.040448 |      0.0 | 0.0 | 0.0 |  148.45451 |       0.0 | 28.832079 | 0.040448 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  1.573 | 0.054992 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.025512 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.14416 |   0.14416 |  0.144886 |  0.144886 | 0.040246 | 1.518008 | 0.0 | 0.0 | 148.428998 |       0.0 | 28.687919 | 0.040246 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 | 0.064218 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.029787 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.14344 |   0.14344 |  0.144161 |  0.144161 | 0.040045 |  1.45379 | 0.0 | 0.0 | 148.399211 |       0.0 | 28.544479 | 0.040045 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 | 0.065345 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.030304 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.142722 |  0.142722 |   0.14344 |   0.14344 | 0.039845 | 1.388445 | 0.0 | 0.0 | 148.368907 |       0.0 | 28.401757 | 0.039845 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 | 0.238876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.110756 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.142009 |  0.142009 |  0.142723 |  0.142723 | 0.039645 | 1.149568 | 0.0 | 0.0 | 148.258152 |       0.0 | 28.259748 | 0.039645 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.847 | 0.984764 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.456248 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.141299 |  0.141299 |   0.14201 |   0.14201 | 0.039447 | 1.011805 | 0.0 | 0.0 | 147.801904 |       0.0 |  28.11845 | 0.039447 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.484 | 1.495805 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.507282 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.140592 |  0.140592 |    0.1413 |    0.1413 |  0.03925 |      0.0 | 0.0 | 0.0 | 146.294622 |       0.0 | 27.977857 |  0.03925 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.611184 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.139889 |  0.139889 |  0.140593 |  0.140593 | 0.039054 |      0.0 | 0.0 | 0.0 | 142.683437 |       0.0 | 27.837968 | 0.039054 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.680975 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.13919 |   0.13919 |   0.13989 |   0.13989 | 0.038858 |      0.0 | 0.0 | 0.0 | 138.002463 |       0.0 | 27.698778 | 0.038858 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.573059 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.138494 |  0.138494 |  0.139191 |  0.139191 | 0.038664 |      0.0 | 0.0 | 0.0 | 132.429404 |       0.0 | 27.560284 | 0.038664 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.347905 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.137801 |  0.137801 |  0.138495 |  0.138495 | 0.038471 |      0.0 | 0.0 | 0.0 | 127.081499 |       0.0 | 27.422483 | 0.038471 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.553222 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.137112 |  0.137112 |  0.137802 |  0.137802 | 0.038278 |      0.0 | 0.0 | 0.0 | 120.528276 |       0.0 |  27.28537 | 0.038278 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.442766 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.136427 |  0.136427 |  0.137113 |  0.137113 | 0.038087 |      0.0 | 0.0 | 0.0 | 115.085511 |       0.0 | 27.148944 | 0.038087 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.354314 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.135745 |  0.135745 |  0.136428 |  0.136428 | 0.037897 |      0.0 | 0.0 | 0.0 | 110.731197 |       0.0 | 27.013199 | 0.037897 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.356341 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.135066 |  0.135066 |  0.135745 |  0.135745 | 0.037707 |      0.0 | 0.0 | 0.0 | 107.374856 |       0.0 | 26.878133 | 0.037707 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.523569 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.134391 |  0.134391 |  0.135067 |  0.135067 | 0.037519 |      0.0 | 0.0 | 0.0 | 105.851287 |       0.0 | 26.743742 | 0.037519 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.666935 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.133719 |  0.133719 |  0.134391 |  0.134391 | 0.037331 |      0.0 | 0.0 | 0.0 | 105.184352 |       0.0 | 26.610023 | 0.037331 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.166818 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.13305 |   0.13305 |  0.133719 |  0.133719 | 0.037144 |      0.0 | 0.0 | 0.0 | 105.017534 |       0.0 | 26.476973 | 0.037144 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.045237 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.132385 |  0.132385 |  0.133051 |  0.133051 | 0.036959 |      0.0 | 0.0 | 0.0 | 104.972297 |       0.0 | 26.344588 | 0.036959 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04333 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.131723 |  0.131723 |  0.132386 |  0.132386 | 0.036774 |      0.0 | 0.0 | 0.0 | 104.928967 |       0.0 | 26.212866 | 0.036774 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04267 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.131064 |  0.131064 |  0.131724 |  0.131724 |  0.03659 |      0.0 | 0.0 | 0.0 | 104.886297 |       0.0 | 26.081801 |  0.03659 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042836 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.130409 |  0.130409 |  0.131065 |  0.131065 | 0.036407 |      0.0 | 0.0 | 0.0 | 104.843461 |       0.0 | 25.951392 | 0.036407 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96_no_rconc_submodel:

no rconc submodel
_________________

Using a submodel like |rconc_uh| as an additional means for calculating runoff
concentration is optional. We repeat the last integration test without one, so there is
less delay in the catchment's simulated outflow.

>>> model.rconcmodel = None

.. integration-test::

    >>> test("hland_96_no_rconc_submodel")
    |        date |    p |    t |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inrc |     outrc |        rt |       qt |       ic |  sp |  wc |         sm |        uz |        lz |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.048465 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 |      0.05 |      0.05 | 0.013889 |      0.0 | 0.0 | 0.0 |  99.951535 |       0.0 |      9.95 | 0.013889 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.047044 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |   0.04975 |   0.04975 | 0.013819 |      0.0 | 0.0 | 0.0 |  99.904491 |       0.0 |   9.90025 | 0.013819 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046628 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |  0.049501 |  0.049501 |  0.01375 |      0.0 | 0.0 | 0.0 |  99.857863 |       0.0 |  9.850749 |  0.01375 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04613 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049254 |  0.049254 | 0.013682 |      0.0 | 0.0 | 0.0 |  99.811734 |       0.0 |  9.801495 | 0.013682 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046584 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049007 |  0.049007 | 0.013613 |      0.0 | 0.0 | 0.0 |   99.76515 |       0.0 |  9.752488 | 0.013613 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.049338 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.048762 |  0.048762 | 0.013545 |      0.0 | 0.0 | 0.0 |  99.715812 |       0.0 |  9.703725 | 0.013545 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.140081 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048519 |  0.048519 | 0.013477 |      0.0 | 0.0 | 0.0 |   99.57573 |       0.0 |  9.655206 | 0.013477 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.92625 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048276 |  0.048276 |  0.01341 |      0.0 | 0.0 | 0.0 |  98.649481 |       0.0 |   9.60693 |  0.01341 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  2.07444 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048035 |  0.048035 | 0.013343 |      0.0 | 0.0 | 0.0 |  96.575041 |       0.0 |  9.558896 | 0.013343 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.400961 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.047794 |  0.047794 | 0.013276 |      0.0 | 0.0 | 0.0 |   93.17408 |       0.0 |  9.511101 | 0.013276 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.705742 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047556 |  0.047556 |  0.01321 |      0.0 | 0.0 | 0.0 |  89.468338 |       0.0 |  9.463546 |  0.01321 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.03435 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047318 |  0.047318 | 0.013144 |      0.0 | 0.0 | 0.0 |  85.433988 |       0.0 |  9.416228 | 0.013144 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.107634 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047081 |  0.047081 | 0.013078 |      0.0 | 0.0 | 0.0 |  83.326354 |       0.0 |  9.369147 | 0.013078 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.656691 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.046846 |  0.046846 | 0.013013 |      0.0 | 0.0 | 0.0 |  80.669664 |       0.0 |  9.322301 | 0.013013 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.871924 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046612 |  0.046612 | 0.012948 |      0.0 | 0.0 | 0.0 |   77.79774 |       0.0 |   9.27569 | 0.012948 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.242 |    0.242 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.873977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046378 |  0.046378 | 0.012883 |      0.0 | 0.0 | 0.0 |  76.923763 |       0.0 |  9.229311 | 0.012883 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.677397 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046147 |  0.046147 | 0.012818 |      0.0 | 0.0 | 0.0 |  76.246366 |       0.0 |  9.183165 | 0.012818 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.417048 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.045916 |  0.045916 | 0.012754 |      0.0 | 0.0 | 0.0 |  75.829319 |       0.0 |  9.137249 | 0.012754 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  1.573 | 0.494092 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.117083 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045686 |  0.045686 | 0.012691 | 1.078908 | 0.0 | 0.0 |  75.712235 |       0.0 |  9.091563 | 0.012691 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  6.776 |  0.08755 |  5.854908 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.854908 |  0.839058 | 0.0 | 0.022103 | 0.059636 |  0.779422 |      0.5 |  0.000006 | 0.0 | 0.047958 |  0.047963 |  0.047963 |  0.047963 | 0.013323 |  1.91245 | 0.0 | 0.0 |  80.765617 |  0.279417 |  9.543605 | 0.013323 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  3.509 |   0.0539 |   3.42145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.42145 |   0.55796 | 0.0 | 0.014096 | 0.058185 |  0.499775 |      0.5 |  0.000022 | 0.0 | 0.050218 |   0.05024 |   0.05024 |   0.05024 | 0.013956 |   1.9461 | 0.0 | 0.0 |  83.673196 |   0.27917 |  9.993387 | 0.013956 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  5.929 | 0.041902 |    5.8751 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    5.8751 |  1.028319 | 0.0 | 0.011598 |  0.05574 |  0.972579 |      0.5 |  0.000168 | 0.0 | 0.052467 |  0.052635 |  0.052635 |  0.052635 | 0.014621 | 1.958098 | 0.0 | 0.0 |  88.564118 |  0.751581 |  10.44092 | 0.014621 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 | 12.826 | 0.020954 | 12.784098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.784098 |  2.506835 | 0.0 | 0.006475 | 0.050579 |  2.456256 |      0.5 |  0.006898 | 0.0 | 0.054705 |  0.061602 |  0.061602 |  0.061602 | 0.017112 | 1.979046 | 0.0 | 0.0 |  98.885486 |  2.700939 | 10.886215 | 0.017112 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.121 | 0.074152 |  0.100046 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.100046 |  0.024457 | 0.0 | 0.022944 | 0.050519 | -0.026062 |      0.5 |  0.014479 | 0.0 | 0.056931 |   0.07141 |   0.07141 |   0.07141 | 0.019836 | 1.925848 | 0.0 | 0.0 |  98.988651 |  2.160398 | 11.329284 | 0.019836 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.847 |  0.06896 |  0.772848 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.772848 |  0.189324 | 0.0 | 0.021469 | 0.050214 |   0.13911 |      0.5 |  0.007758 | 0.0 | 0.059146 |  0.066905 |  0.066905 |  0.066905 | 0.018585 |  1.93104 | 0.0 | 0.0 |   99.60092 |  1.791749 | 11.770138 | 0.018585 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   3.63 | 0.052207 |   3.56104 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.56104 |  0.883169 | 0.0 | 0.016695 | 0.048861 |  0.834308 |      0.5 |  0.007549 | 0.0 | 0.061351 |    0.0689 |    0.0689 |    0.0689 | 0.019139 | 1.947793 | 0.0 | 0.0 | 102.310958 |  2.118508 | 12.208787 | 0.019139 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.541 | 0.057919 |  2.488793 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.488793 |  0.651288 | 0.0 | 0.018859 | 0.047926 |  0.603362 |      0.5 |  0.010164 | 0.0 | 0.063544 |  0.073708 |  0.073708 |  0.073708 | 0.020474 | 1.942081 | 0.0 | 0.0 |  104.17753 |  2.211707 | 12.645243 | 0.020474 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 12.584 | 0.021144 | 12.526081 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.526081 |  3.398626 | 0.0 | 0.007489 | 0.043348 |  3.355278 |      0.5 |  0.055198 | 0.0 | 0.065726 |  0.120925 |  0.120925 |  0.120925 |  0.03359 | 1.978856 | 0.0 | 0.0 | 113.340843 |  5.011787 | 13.079517 |  0.03359 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.235 | 0.048561 |  4.213856 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.213856 |  1.353295 | 0.0 |  0.01764 | 0.041899 |  1.311396 |      0.5 |  0.153723 | 0.0 | 0.067898 |  0.221621 |  0.221621 |  0.221621 | 0.061561 | 1.951439 | 0.0 | 0.0 | 116.225662 |   5.66946 | 13.511619 | 0.061561 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.114 | 0.048983 |  4.065439 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.065439 |   1.37294 | 0.0 | 0.018209 | 0.040541 |  1.332399 |      0.5 |  0.215358 | 0.0 | 0.070058 |  0.285416 |  0.285416 |  0.285416 | 0.079282 | 1.951017 | 0.0 | 0.0 | 118.940493 |    6.2865 | 13.941561 | 0.079282 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.452 | 0.124728 |  1.403017 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.403017 |  0.496206 | 0.0 | 0.046729 | 0.040076 |   0.45613 |      0.5 |  0.232354 | 0.0 | 0.072208 |  0.304562 |  0.304562 |  0.304562 |   0.0846 | 1.875272 | 0.0 | 0.0 | 119.840651 |  6.010276 | 14.369353 |   0.0846 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.121 |  0.96204 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.360406 |  0.04008 |  -0.04008 |      0.5 |  0.180378 | 0.0 | 0.074347 |  0.254725 |  0.254725 |  0.254725 | 0.070757 | 1.034232 | 0.0 | 0.0 | 119.520324 |  5.289819 | 14.795007 | 0.070757 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    0.0 | 1.034232 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.531595 |  0.04024 |  -0.04024 |      0.5 |  0.122017 | 0.0 | 0.076475 |  0.198492 |  0.198492 |  0.198492 | 0.055137 |      0.0 | 0.0 | 0.0 | 118.028969 |  4.627562 | 15.218532 | 0.055137 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.097079 | 0.040986 | -0.040986 |      0.5 |  0.080568 | 0.0 | 0.078593 |  0.159161 |  0.159161 |  0.159161 | 0.044211 |      0.0 | 0.0 | 0.0 | 116.972875 |  4.006009 | 15.639939 | 0.044211 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.320389 | 0.041514 | -0.041514 |      0.5 |  0.051191 | 0.0 |   0.0807 |  0.131891 |  0.131891 |  0.131891 | 0.036636 |      0.0 | 0.0 | 0.0 |    115.694 |  3.413304 | 16.059239 | 0.036636 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.420519 | 0.042153 | -0.042153 |      0.5 |  0.030713 | 0.0 | 0.082796 |  0.113509 |  0.113509 |  0.113509 |  0.03153 |      0.0 | 0.0 | 0.0 | 112.315634 |  2.840438 | 16.476443 |  0.03153 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.356 | 1.680962 |     2.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     2.356 |  0.743012 | 0.0 | 0.598694 | 0.043036 |  0.699976 |      0.5 |  0.025159 | 0.0 | 0.084882 |  0.110042 |  0.110042 |  0.110042 | 0.030567 | 0.319038 | 0.0 | 0.0 | 113.372964 |  3.015255 | 16.891561 | 0.030567 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.139 |      2.0 |  5.458038 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.458038 |  1.753863 | 0.0 | 1.880771 | 0.041461 |  1.712401 |      0.5 |  0.048171 | 0.0 | 0.086958 |  0.135128 |  0.135128 |  0.135128 | 0.037536 |      0.0 | 0.0 | 0.0 |  115.23783 |  4.179486 | 17.304603 | 0.037536 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.331 |    1.331 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.680563 | 0.042381 | -0.042381 |      0.5 |  0.058507 | 0.0 | 0.089023 |   0.14753 |   0.14753 |   0.14753 |  0.04098 |      0.0 | 0.0 | 0.0 | 113.599648 |  3.578598 |  17.71558 |  0.04098 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 25.047 | 0.132786 |    23.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    23.047 |  7.435469 | 0.0 | 0.053632 | 0.035394 |  7.400075 |      0.5 |   0.40815 | 0.0 | 0.091078 |  0.499228 |  0.499228 |  0.499228 | 0.138675 | 1.867214 | 0.0 | 0.0 | 129.192941 | 10.070522 | 18.124502 | 0.138675 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 45.859 | 0.017578 | 45.726214 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.726214 | 19.080196 | 0.0 | 0.008562 | 0.022081 | 19.058115 |      0.5 |  5.990579 | 0.0 | 0.093123 |  6.083701 |  6.083701 |  6.083701 | 1.689917 | 1.982422 | 0.0 | 0.0 | 155.852479 | 22.638059 |  18.53138 | 1.689917 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.922 | 0.821583 |  9.904422 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.904422 |  6.014459 | 0.0 | 0.410182 | 0.020129 |   5.99433 |      0.5 |   8.87815 | 0.0 | 0.095157 |  8.973307 |  8.973307 |  8.973307 | 2.492585 | 1.178417 | 0.0 | 0.0 | 159.352388 | 19.254239 | 18.936223 | 2.492585 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.356 | 0.792694 |  3.534417 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.534417 |  2.243752 | 0.0 | 0.396347 | 0.019678 |  2.224074 |      0.5 |  5.214741 | 0.0 | 0.097181 |  5.311922 |  5.311922 |  5.311922 | 1.475534 | 1.207306 | 0.0 | 0.0 | 160.266384 | 15.763572 | 19.339042 | 1.475534 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.075 | 0.090607 |  8.282306 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.282306 |  5.318341 | 0.0 | 0.045304 | 0.018385 |  5.299956 |      0.5 |  4.208086 | 0.0 | 0.099195 |  4.307281 |  4.307281 |  4.307281 | 1.196467 | 1.909393 | 0.0 | 0.0 | 163.203431 | 16.355442 | 19.739846 | 1.196467 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 22.385 | 0.007604 | 22.294393 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.294393 | 14.845479 | 0.0 | 0.003802 | 0.014674 | 14.830806 |      0.5 |  8.285963 | 0.0 | 0.101199 |  8.387162 |  8.387162 |  8.387162 | 2.329767 | 1.992396 | 0.0 | 0.0 | 170.663216 | 22.400285 | 20.138647 | 2.329767 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 18.634 | 0.011044 | 18.626396 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.626396 | 13.562779 | 0.0 | 0.005522 | 0.012137 | 13.550643 |      0.5 | 12.233902 | 0.0 | 0.103193 | 12.337096 | 12.337096 | 12.337096 | 3.426971 | 1.988956 | 0.0 | 0.0 | 175.733447 | 23.217025 | 20.535454 | 3.426971 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.623 | 0.032783 |  7.611956 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.611956 |  5.876857 | 0.0 | 0.016392 | 0.011266 |  5.865591 |      0.5 |  9.245942 | 0.0 | 0.105177 |   9.35112 |   9.35112 |   9.35112 | 2.597533 | 1.967217 | 0.0 | 0.0 |  177.46342 | 19.336674 | 20.930277 | 2.597533 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.299 | 0.054873 |  2.266217 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.266217 |  1.784264 | 0.0 | 0.027436 | 0.011027 |  1.773237 |      0.5 |   5.11831 | 0.0 | 0.107151 |  5.225461 |  5.225461 |  5.225461 | 1.451517 | 1.945127 | 0.0 | 0.0 | 177.928964 | 15.491601 | 21.323125 | 1.451517 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  5.929 | 0.038243 |  5.874127 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.874127 |  4.649183 | 0.0 | 0.019122 | 0.010423 |   4.63876 |      0.5 |  3.860955 | 0.0 | 0.109116 |   3.97007 |   3.97007 |   3.97007 | 1.102797 | 1.961757 | 0.0 | 0.0 | 179.145209 | 15.769407 |  21.71401 | 1.102797 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.267 |  0.04981 |  3.228757 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.228757 |  2.590513 | 0.0 | 0.024905 | 0.010108 |  2.580405 |      0.5 |   3.41746 | 0.0 |  0.11107 |   3.52853 |   3.52853 |   3.52853 | 0.980147 |  1.95019 | 0.0 | 0.0 | 179.768657 | 14.432352 |  22.10294 | 0.980147 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.605 | 0.064366 |   0.55519 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.55519 |  0.448549 | 0.0 | 0.032183 | 0.010062 |  0.438486 |      0.5 |  2.298065 | 0.0 | 0.113015 |   2.41108 |   2.41108 |   2.41108 | 0.669744 | 1.935634 | 0.0 | 0.0 | 179.853178 | 12.072772 | 22.489925 | 0.669744 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.242 | 0.067009 |  0.177634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.177634 |  0.143649 | 0.0 | 0.033505 | 0.010056 |  0.133593 |      0.5 |  1.392466 | 0.0 |  0.11495 |  1.507416 |  1.507416 |  1.507416 | 0.418727 | 1.932991 | 0.0 | 0.0 | 179.863714 | 10.313899 | 22.874975 | 0.418727 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.605 | 0.065382 |  0.537991 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.537991 |  0.435113 | 0.0 | 0.032691 | 0.010017 |  0.425096 |      0.5 |  0.939834 | 0.0 | 0.116875 |  1.056709 |  1.056709 |  1.056709 |  0.29353 | 1.934618 | 0.0 | 0.0 | 179.943918 |  9.299161 |   23.2581 |  0.29353 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  2.904 | 0.052751 |  2.838618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.838618 |  2.297848 | 0.0 | 0.026376 | 0.009758 |  2.288091 |      0.5 |  0.931867 | 0.0 | 0.118791 |  1.050657 |  1.050657 |  1.050657 | 0.291849 | 1.947249 | 0.0 | 0.0 |  180.46807 | 10.155384 |  23.63931 | 0.291849 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.484 | 0.113609 |  0.431249 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.431249 |   0.35113 | 0.0 | 0.056805 | 0.009726 |  0.341405 |      0.5 |  0.890249 | 0.0 | 0.120697 |  1.010945 |  1.010945 |  1.010945 | 0.280818 | 1.886391 | 0.0 | 0.0 |  180.50111 |   9.10654 | 24.018613 | 0.280818 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.242 | 0.353524 |  0.128391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.128391 |  0.104577 | 0.0 | 0.176762 | 0.009738 |  0.094839 |      0.5 |  0.631758 | 0.0 | 0.122593 |  0.754351 |  0.754351 |  0.754351 | 0.209542 | 1.646476 | 0.0 | 0.0 | 180.357899 |  8.069621 |  24.39602 | 0.209542 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 | 0.903769 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.451885 | 0.009821 | -0.009821 |      0.5 |  0.437555 | 0.0 |  0.12448 |  0.562035 |  0.562035 |  0.562035 | 0.156121 | 0.742707 | 0.0 | 0.0 | 179.915836 |  7.122246 |  24.77154 | 0.156121 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 | 0.742707 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.864415 | 0.010042 | -0.010042 |      0.5 |  0.302641 | 0.0 | 0.126358 |  0.428999 |  0.428999 |  0.428999 | 0.119166 |      0.0 | 0.0 | 0.0 | 177.061463 |  6.309562 | 25.145182 | 0.119166 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.055807 | 0.011469 | -0.011469 |      0.5 |  0.210436 | 0.0 | 0.128226 |  0.338662 |  0.338662 |  0.338662 | 0.094073 |      0.0 | 0.0 | 0.0 | 174.017125 |  5.587657 | 25.516957 | 0.094073 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     1.146 |  0.867578 | 0.0 | 1.982694 | 0.012852 |  0.854726 |      0.5 |  0.183057 | 0.0 | 0.130085 |  0.313141 |  0.313141 |  0.313141 | 0.086984 |      0.0 | 0.0 | 0.0 | 172.325705 |  5.759327 | 25.886872 | 0.086984 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.37986 | 0.013837 | -0.013837 |      0.5 |  0.159478 | 0.0 | 0.131934 |  0.291412 |  0.291412 |  0.291412 | 0.080948 |      0.0 | 0.0 | 0.0 | 168.959683 |  5.086012 | 26.254937 | 0.080948 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.277978 |  0.01552 |  -0.01552 |      0.5 |  0.108875 | 0.0 | 0.133775 |   0.24265 |   0.24265 |   0.24265 | 0.067403 |      0.0 | 0.0 | 0.0 | 165.697224 |  4.461616 | 26.621163 | 0.067403 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.602665 | 0.017151 | -0.017151 |      0.5 |  0.072459 | 0.0 | 0.135606 |  0.208064 |  0.208064 |  0.208064 | 0.057796 |      0.0 | 0.0 | 0.0 | 160.111711 |  3.872006 | 26.985557 | 0.057796 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.735707 | 0.019944 | -0.019944 |      0.5 |   0.04635 | 0.0 | 0.137428 |  0.183778 |  0.183778 |  0.183778 | 0.051049 |      0.0 | 0.0 | 0.0 | 156.395948 |  3.305712 | 27.348129 | 0.051049 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.994839 | 0.021802 | -0.021802 |      0.5 |  0.027967 | 0.0 | 0.139241 |  0.167208 |  0.167208 |  0.167208 | 0.046447 |      0.0 | 0.0 | 0.0 | 152.422911 |  2.755943 | 27.708889 | 0.046447 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.410799 | 0.023789 | -0.023789 |      0.5 |  0.015488 | 0.0 | 0.141044 |  0.156532 |  0.156532 |  0.156532 | 0.043481 |      0.0 | 0.0 | 0.0 |   150.0359 |  2.216667 | 28.067844 | 0.043481 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.087741 | 0.024982 | -0.024982 |      0.5 |  0.007522 | 0.0 | 0.142839 |  0.150362 |  0.150362 |  0.150362 | 0.041767 |      0.0 | 0.0 | 0.0 | 148.973141 |  1.684163 | 28.425005 | 0.041767 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.19702 | 0.025513 | -0.025513 |      0.5 |  0.002944 | 0.0 | 0.144625 |  0.147569 |  0.147569 |  0.147569 | 0.040992 |      0.0 | 0.0 | 0.0 | 148.801635 |  1.155705 |  28.78038 | 0.040992 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062717 | 0.025599 | -0.025599 |      0.5 |  0.000766 | 0.0 | 0.146402 |  0.147168 |  0.147168 |  0.147168 |  0.04088 |      0.0 | 0.0 | 0.0 | 148.764517 |   0.62934 | 29.133978 |  0.04088 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061417 | 0.025618 | -0.025618 |      0.5 |  0.000073 | 0.0 |  0.14817 |  0.148243 |  0.148243 |  0.148243 | 0.041179 |      0.0 | 0.0 | 0.0 | 148.728719 |  0.103649 | 29.485808 | 0.041179 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060361 | 0.025636 | -0.025636 | 0.078013 |       0.0 | 0.0 | 0.147819 |  0.147819 |  0.147819 |  0.147819 | 0.041061 |      0.0 | 0.0 | 0.0 | 148.693993 |       0.0 | 29.416002 | 0.041061 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.059548 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.14708 |   0.14708 |   0.14708 |   0.14708 | 0.040856 |      0.0 | 0.0 | 0.0 | 148.634445 |       0.0 | 29.268922 | 0.040856 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.058862 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.146345 |  0.146345 |  0.146345 |  0.146345 | 0.040651 |      0.0 | 0.0 | 0.0 | 148.575584 |       0.0 | 29.122577 | 0.040651 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060419 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.145613 |  0.145613 |  0.145613 |  0.145613 | 0.040448 |      0.0 | 0.0 | 0.0 | 148.515165 |       0.0 | 28.976964 | 0.040448 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060655 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.144885 |  0.144885 |  0.144885 |  0.144885 | 0.040246 |      0.0 | 0.0 | 0.0 |  148.45451 |       0.0 | 28.832079 | 0.040246 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  1.573 | 0.054992 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.025512 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.14416 |   0.14416 |   0.14416 |   0.14416 | 0.040045 | 1.518008 | 0.0 | 0.0 | 148.428998 |       0.0 | 28.687919 | 0.040045 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 | 0.064218 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.029787 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.14344 |   0.14344 |   0.14344 |   0.14344 | 0.039844 |  1.45379 | 0.0 | 0.0 | 148.399211 |       0.0 | 28.544479 | 0.039844 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 | 0.065345 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.030304 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.142722 |  0.142722 |  0.142722 |  0.142722 | 0.039645 | 1.388445 | 0.0 | 0.0 | 148.368907 |       0.0 | 28.401757 | 0.039645 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 | 0.238876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.110756 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.142009 |  0.142009 |  0.142009 |  0.142009 | 0.039447 | 1.149568 | 0.0 | 0.0 | 148.258152 |       0.0 | 28.259748 | 0.039447 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.847 | 0.984764 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.456248 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.141299 |  0.141299 |  0.141299 |  0.141299 |  0.03925 | 1.011805 | 0.0 | 0.0 | 147.801904 |       0.0 |  28.11845 |  0.03925 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.484 | 1.495805 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.507282 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.140592 |  0.140592 |  0.140592 |  0.140592 | 0.039053 |      0.0 | 0.0 | 0.0 | 146.294622 |       0.0 | 27.977857 | 0.039053 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.611184 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.139889 |  0.139889 |  0.139889 |  0.139889 | 0.038858 |      0.0 | 0.0 | 0.0 | 142.683437 |       0.0 | 27.837968 | 0.038858 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.680975 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.13919 |   0.13919 |   0.13919 |   0.13919 | 0.038664 |      0.0 | 0.0 | 0.0 | 138.002463 |       0.0 | 27.698778 | 0.038664 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.573059 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.138494 |  0.138494 |  0.138494 |  0.138494 | 0.038471 |      0.0 | 0.0 | 0.0 | 132.429404 |       0.0 | 27.560284 | 0.038471 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.347905 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.137801 |  0.137801 |  0.137801 |  0.137801 | 0.038278 |      0.0 | 0.0 | 0.0 | 127.081499 |       0.0 | 27.422483 | 0.038278 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.553222 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.137112 |  0.137112 |  0.137112 |  0.137112 | 0.038087 |      0.0 | 0.0 | 0.0 | 120.528276 |       0.0 |  27.28537 | 0.038087 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.442766 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.136427 |  0.136427 |  0.136427 |  0.136427 | 0.037896 |      0.0 | 0.0 | 0.0 | 115.085511 |       0.0 | 27.148944 | 0.037896 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.354314 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.135745 |  0.135745 |  0.135745 |  0.135745 | 0.037707 |      0.0 | 0.0 | 0.0 | 110.731197 |       0.0 | 27.013199 | 0.037707 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.356341 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.135066 |  0.135066 |  0.135066 |  0.135066 | 0.037518 |      0.0 | 0.0 | 0.0 | 107.374856 |       0.0 | 26.878133 | 0.037518 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.523569 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.134391 |  0.134391 |  0.134391 |  0.134391 | 0.037331 |      0.0 | 0.0 | 0.0 | 105.851287 |       0.0 | 26.743742 | 0.037331 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.666935 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.133719 |  0.133719 |  0.133719 |  0.133719 | 0.037144 |      0.0 | 0.0 | 0.0 | 105.184352 |       0.0 | 26.610023 | 0.037144 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.166818 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.13305 |   0.13305 |   0.13305 |   0.13305 | 0.036958 |      0.0 | 0.0 | 0.0 | 105.017534 |       0.0 | 26.476973 | 0.036958 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.045237 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.132385 |  0.132385 |  0.132385 |  0.132385 | 0.036774 |      0.0 | 0.0 | 0.0 | 104.972297 |       0.0 | 26.344588 | 0.036774 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04333 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.131723 |  0.131723 |  0.131723 |  0.131723 |  0.03659 |      0.0 | 0.0 | 0.0 | 104.928967 |       0.0 | 26.212866 |  0.03659 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04267 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.131064 |  0.131064 |  0.131064 |  0.131064 | 0.036407 |      0.0 | 0.0 | 0.0 | 104.886297 |       0.0 | 26.081801 | 0.036407 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042836 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.130409 |  0.130409 |  0.130409 |  0.130409 | 0.036225 |      0.0 | 0.0 | 0.0 | 104.843461 |       0.0 | 25.951392 | 0.036225 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

For the next tests, we use the rconc submodel again:

>>> with model.add_rconcmodel_v1("rconc_uh"):
...     uh("triangle", tb=3.0)

.. _hland_96_resparea:

contributing area
_________________

We can substantially change the functioning of |hland_96| by enabling its |RespArea|
option, which decreases |Perc| but increases |Q0| in dry periods (more concretely: in
periods with dry soils).  Hence the simulated result of |Perc| appears less "block-like"
and reaches its maximum at the same time as the result of |SM| does, whereas |Q0| shows
more pronounced peaks in the initial subperiod when the soil is not saturated yet:

.. integration-test::

    >>> resparea(True)
    >>> test("hland_96_resparea")
    |        date |    p |    t |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inrc |    outrc |       rt |       qt |       ic |  sp |  wc |         sm |        uz |        lz |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249758 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.048465 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 | 0.061111 | 0.061111 | 0.016975 |      0.0 | 0.0 | 0.0 |  99.951535 |       0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249523 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.047044 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 | 0.088833 | 0.088833 | 0.024676 |      0.0 | 0.0 | 0.0 |  99.904491 |       0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24929 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046628 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |  0.04975 |  0.04975 |  0.01382 |      0.0 | 0.0 | 0.0 |  99.857863 |       0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24906 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04613 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 | 0.049502 | 0.049502 |  0.01375 |      0.0 | 0.0 | 0.0 |  99.811734 |       0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248827 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046584 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 | 0.049254 | 0.049254 | 0.013682 |      0.0 | 0.0 | 0.0 |   99.76515 |       0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248581 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.049338 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 | 0.049008 | 0.049008 | 0.013613 |      0.0 | 0.0 | 0.0 |  99.715812 |       0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.247883 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.140081 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 | 0.048763 | 0.048763 | 0.013545 |      0.0 | 0.0 | 0.0 |   99.57573 |       0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.243293 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.92625 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 | 0.048519 | 0.048519 | 0.013477 |      0.0 | 0.0 | 0.0 |  98.649481 |       0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.233168 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  2.07444 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 | 0.048276 | 0.048276 |  0.01341 |      0.0 | 0.0 | 0.0 |  96.575041 |       0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.217035 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.400961 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 | 0.048035 | 0.048035 | 0.013343 |      0.0 | 0.0 | 0.0 |   93.17408 |       0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.200115 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.705742 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 | 0.047795 | 0.047795 | 0.013276 |      0.0 | 0.0 | 0.0 |  89.468338 |       0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.182474 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.03435 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 | 0.047556 | 0.047556 |  0.01321 |      0.0 | 0.0 | 0.0 |  85.433988 |       0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.173582 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.107634 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 | 0.047318 | 0.047318 | 0.013144 |      0.0 | 0.0 | 0.0 |  83.326354 |       0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.16269 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.656691 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 | 0.047081 | 0.047081 | 0.013078 |      0.0 | 0.0 | 0.0 |  80.669664 |       0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.151312 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.871924 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 | 0.046846 | 0.046846 | 0.013013 |      0.0 | 0.0 | 0.0 |   77.79774 |       0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.147932 |  0.242 |    0.242 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.873977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 | 0.046612 | 0.046612 | 0.012948 |      0.0 | 0.0 | 0.0 |  76.923763 |       0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.145338 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.677397 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 | 0.046379 | 0.046379 | 0.012883 |      0.0 | 0.0 | 0.0 |  76.246366 |       0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.143752 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.417048 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 | 0.046147 | 0.046147 | 0.012819 |      0.0 | 0.0 | 0.0 |  75.829319 |       0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.143309 |  1.573 | 0.494092 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.117083 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 | 0.045916 | 0.045916 | 0.012754 | 1.078908 | 0.0 | 0.0 |  75.712235 |       0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.163077 |  6.776 |  0.08755 |  5.854908 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.854908 |  0.839058 | 0.0 | 0.022103 | 0.059636 |  0.779422 | 0.081539 |  0.019081 | 0.0 | 0.045866 |  0.064946 | 0.050017 | 0.050017 | 0.013894 |  1.91245 | 0.0 | 0.0 |  80.765617 |  0.678803 |  9.127236 | 0.013894 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.17503 |  3.509 |   0.0539 |   3.42145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.42145 |   0.55796 | 0.0 | 0.014096 | 0.058185 |  0.499775 | 0.087515 |  0.114159 | 0.0 | 0.046074 |  0.160233 | 0.081841 | 0.081841 | 0.022734 |   1.9461 | 0.0 | 0.0 |  83.673196 |  0.976904 |  9.168677 | 0.022734 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.19609 |  5.929 | 0.041902 |    5.8751 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    5.8751 |  1.028319 | 0.0 | 0.011598 |  0.05574 |  0.972579 | 0.098045 |  0.305008 | 0.0 | 0.046334 |  0.351342 | 0.181527 | 0.181527 | 0.050424 | 1.958098 | 0.0 | 0.0 |  88.564118 |   1.54643 |  9.220388 | 0.050424 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.244458 | 12.826 | 0.020954 | 12.784098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.784098 |  2.506835 | 0.0 | 0.006475 | 0.050579 |  2.456256 | 0.122229 |  0.981496 | 0.0 | 0.046713 |  1.028209 | 0.459288 | 0.459288 |  0.12758 | 1.979046 | 0.0 | 0.0 |  98.885486 |  2.898961 |  9.295905 |  0.12758 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.244969 |  0.121 | 0.074152 |  0.100046 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.100046 |  0.024457 | 0.0 | 0.022944 | 0.050519 | -0.026062 | 0.122484 |  0.868647 | 0.0 | 0.047092 |  0.915739 | 0.852801 | 0.852801 | 0.236889 | 1.925848 | 0.0 | 0.0 |  98.988651 |  1.881767 |  9.371297 | 0.236889 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.248009 |  0.847 |  0.06896 |  0.772848 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.772848 |  0.189324 | 0.0 | 0.021469 | 0.050214 |   0.13911 | 0.124004 |  0.330832 | 0.0 | 0.047477 |  0.378309 | 0.821303 | 0.821303 |  0.22814 |  1.93104 | 0.0 | 0.0 |   99.60092 |   1.56604 |  9.447825 |  0.22814 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.261688 |   3.63 | 0.052207 |   3.56104 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.56104 |  0.883169 | 0.0 | 0.016695 | 0.048861 |  0.834308 | 0.130844 |  0.318669 | 0.0 | 0.047893 |  0.366562 | 0.495127 | 0.495127 | 0.137535 | 1.947793 | 0.0 | 0.0 | 102.310958 |  1.950835 |  9.530776 | 0.137535 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.271324 |  2.541 | 0.057919 |  2.488793 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.488793 |  0.651288 | 0.0 | 0.018859 | 0.047926 |  0.603362 | 0.135662 |  0.397046 | 0.0 | 0.048332 |  0.445378 | 0.386687 | 0.386687 | 0.107413 | 1.942081 | 0.0 | 0.0 |  104.17753 |   2.02149 |  9.618105 | 0.107413 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.321154 | 12.584 | 0.021144 | 12.526081 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.526081 |  3.398626 | 0.0 | 0.007489 | 0.043348 |  3.355278 | 0.160577 |  1.129778 | 0.0 | 0.048893 |  1.178671 | 0.590817 | 0.590817 | 0.164116 | 1.978856 | 0.0 | 0.0 | 113.340843 |  4.086413 |  9.729789 | 0.164116 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.33771 |  4.235 | 0.048561 |  4.213856 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.213856 |  1.353295 | 0.0 |  0.01764 | 0.041899 |  1.311396 | 0.168855 |  1.516311 | 0.0 | 0.049493 |  1.565804 | 1.101747 | 1.101747 | 0.306041 | 1.951439 | 0.0 | 0.0 | 116.225662 |  3.712643 |  9.849151 | 0.306041 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.353671 |  4.114 | 0.048983 |  4.065439 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.065439 |   1.37294 | 0.0 | 0.018209 | 0.040541 |  1.332399 | 0.176836 |  1.163402 | 0.0 |  0.05013 |  1.213532 | 1.401492 | 1.401492 | 0.389303 | 1.951017 | 0.0 | 0.0 | 118.940493 |  3.704804 |  9.975856 | 0.389303 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.359045 |  1.452 | 0.124728 |  1.403017 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.403017 |  0.496206 | 0.0 | 0.046729 | 0.040076 |   0.45613 | 0.179522 |   0.84735 | 0.0 | 0.050777 |  0.898127 | 1.221725 | 1.221725 | 0.339368 | 1.875272 | 0.0 | 0.0 | 119.840651 |  3.134062 | 10.104602 | 0.339368 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.357128 |  0.121 |  0.96204 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.360406 |  0.04008 |  -0.04008 | 0.178564 |  0.469808 | 0.0 | 0.051416 |  0.521224 | 0.884461 | 0.884461 | 0.245684 | 1.034232 | 0.0 | 0.0 | 119.520324 |   2.44561 |  10.23175 | 0.245684 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.348271 |    0.0 | 1.034232 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.531595 |  0.04024 |  -0.04024 | 0.174135 |  0.254248 | 0.0 | 0.052029 |  0.306277 | 0.557214 | 0.557214 | 0.154782 |      0.0 | 0.0 | 0.0 | 118.028969 |  1.976987 | 10.353856 | 0.154782 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.342066 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.097079 | 0.040986 | -0.040986 | 0.171033 |  0.144956 | 0.0 | 0.052624 |   0.19758 | 0.329888 | 0.329888 | 0.091636 |      0.0 | 0.0 | 0.0 | 116.972875 |  1.620012 | 10.472264 | 0.091636 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.334628 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.320389 | 0.041514 | -0.041514 | 0.167314 |  0.085292 | 0.0 | 0.053198 |   0.13849 | 0.208604 | 0.208604 | 0.057946 |      0.0 | 0.0 | 0.0 |    115.694 |  1.325893 |  10.58638 | 0.057946 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.31537 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.420519 | 0.042153 | -0.042153 | 0.157685 |  0.055002 | 0.0 |  0.05372 |  0.108722 | 0.145006 | 0.145006 |  0.04028 |      0.0 | 0.0 | 0.0 | 112.315634 |  1.071053 | 10.690345 |  0.04028 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.321336 |  4.356 | 1.680962 |     2.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     2.356 |  0.743012 | 0.0 | 0.598694 | 0.043036 |  0.699976 | 0.160668 |  0.070647 | 0.0 | 0.054255 |  0.124902 | 0.118933 | 0.118933 | 0.033037 | 0.319038 | 0.0 | 0.0 | 113.372964 |  1.539714 | 10.796758 | 0.033037 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.331994 |  7.139 |      2.0 |  5.458038 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.458038 |  1.753863 | 0.0 | 1.880771 | 0.041461 |  1.712401 | 0.165997 |  0.316119 | 0.0 | 0.054814 |  0.370933 |  0.17598 |  0.17598 | 0.048883 |      0.0 | 0.0 | 0.0 |  115.23783 |  2.769999 | 10.907941 | 0.048883 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.322622 |  1.331 |    1.331 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.680563 | 0.042381 | -0.042381 | 0.161311 |  0.432074 | 0.0 | 0.055346 |   0.48742 | 0.342145 | 0.342145 |  0.09504 |      0.0 | 0.0 | 0.0 | 113.599648 |  2.134233 | 11.013906 |  0.09504 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.41727 | 25.047 | 0.132786 |    23.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    23.047 |  7.435469 | 0.0 | 0.053632 | 0.035394 |  7.400075 | 0.208635 |  2.252485 | 0.0 | 0.056113 |  2.308598 |  0.86624 |  0.86624 | 0.240622 | 1.867214 | 0.0 | 0.0 | 129.192941 |  7.073188 | 11.166428 | 0.240622 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.60725 | 45.859 | 0.017578 | 45.726214 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.726214 | 19.080196 | 0.0 | 0.008562 | 0.022081 | 19.058115 | 0.303625 | 10.375901 | 0.0 |  0.05735 | 10.433252 |  3.70937 |  3.70937 | 1.030381 | 1.982422 | 0.0 | 0.0 | 155.852479 | 15.451777 | 11.412703 | 1.030381 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.63483 |  9.922 | 0.821583 |  9.904422 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.904422 |  6.014459 | 0.0 | 0.410182 | 0.020129 |   5.99433 | 0.317415 |  9.160802 | 0.0 | 0.058651 |  9.219452 |  8.35804 |  8.35804 | 2.321678 | 1.178417 | 0.0 | 0.0 | 159.352388 | 11.967891 | 11.671467 | 2.321678 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.642133 |  4.356 | 0.792694 |  3.534417 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.534417 |  2.243752 | 0.0 | 0.396347 | 0.019678 |  2.224074 | 0.321066 |  4.414385 | 0.0 | 0.059963 |  4.474348 | 8.434718 | 8.434718 | 2.342977 | 1.207306 | 0.0 | 0.0 | 160.266384 |  9.456514 | 11.932571 | 2.342977 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.665884 |  9.075 | 0.090607 |  8.282306 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.282306 |  5.318341 | 0.0 | 0.045304 | 0.018385 |  5.299956 | 0.332942 |  3.672759 | 0.0 | 0.061328 |  3.734087 | 5.364313 | 5.364313 | 1.490087 | 1.909393 | 0.0 | 0.0 | 163.203431 | 10.750768 | 12.204185 | 1.490087 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.728148 | 22.385 | 0.007604 | 22.294393 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.294393 | 14.845479 | 0.0 | 0.003802 | 0.014674 | 14.830806 | 0.364074 |  8.480469 | 0.0 | 0.062841 |   8.54331 | 4.967305 | 4.967305 | 1.379807 | 1.992396 | 0.0 | 0.0 | 170.663216 | 16.737031 | 12.505418 | 1.379807 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.772056 | 18.634 | 0.011044 | 18.626396 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.626396 | 13.562779 | 0.0 | 0.005522 | 0.012137 | 13.550643 | 0.386028 | 11.966569 | 0.0 | 0.064457 | 12.031027 | 8.249642 | 8.249642 | 2.291567 | 1.988956 | 0.0 | 0.0 | 175.733447 | 17.935076 | 12.826989 | 2.291567 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.787332 |  7.623 | 0.032783 |  7.611956 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.611956 |  5.876857 | 0.0 | 0.016392 | 0.011266 |  5.865591 | 0.393666 |  8.563157 | 0.0 | 0.066103 |   8.62926 | 10.50003 | 10.50003 | 2.916675 | 1.967217 | 0.0 | 0.0 |  177.46342 | 14.843845 | 13.154552 | 2.916675 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.791468 |  2.299 | 0.054873 |  2.266217 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.266217 |  1.784264 | 0.0 | 0.027436 | 0.011027 |  1.773237 | 0.395734 |  4.526098 | 0.0 | 0.067751 |   4.59385 |  8.48845 |  8.48845 | 2.357903 | 1.945127 | 0.0 | 0.0 | 177.928964 |  11.69525 | 13.482534 | 2.357903 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.802325 |  5.929 | 0.038243 |  5.874127 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.874127 |  4.649183 | 0.0 | 0.019122 | 0.010423 |   4.63876 | 0.401163 |  3.486825 | 0.0 | 0.069418 |  3.556244 | 5.260028 | 5.260028 | 1.461119 | 1.961757 | 0.0 | 0.0 | 179.145209 | 12.446022 | 13.814278 | 1.461119 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.807919 |  3.267 |  0.04981 |  3.228757 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.228757 |  2.590513 | 0.0 | 0.024905 | 0.010108 |  2.580405 |  0.40396 |  3.196903 | 0.0 | 0.071091 |  3.267995 | 3.722767 | 3.722767 | 1.034102 |  1.95019 | 0.0 | 0.0 | 179.768657 | 11.425564 | 14.147147 | 1.034102 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.808679 |  0.605 | 0.064366 |   0.55519 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.55519 |  0.448549 | 0.0 | 0.032183 | 0.010062 |  0.438486 |  0.40434 |  2.089712 | 0.0 | 0.072757 |  2.162469 | 3.086378 | 3.086378 | 0.857327 | 1.935634 | 0.0 | 0.0 | 179.853178 |  9.369998 | 14.478729 | 0.857327 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.808774 |  0.242 | 0.067009 |  0.177634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.177634 |  0.143649 | 0.0 | 0.033505 | 0.010056 |  0.133593 | 0.404387 |   1.20495 | 0.0 | 0.074416 |  1.279365 | 2.211896 | 2.211896 | 0.614416 | 1.932991 | 0.0 | 0.0 | 179.863714 |  7.894254 |   14.8087 | 0.614416 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.809495 |  0.605 | 0.065382 |  0.537991 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.537991 |  0.435113 | 0.0 | 0.032691 | 0.010017 |  0.425096 | 0.404748 |  0.794496 | 0.0 | 0.076067 |  0.870563 | 1.384766 | 1.384766 | 0.384657 | 1.934618 | 0.0 | 0.0 | 179.943918 |  7.120107 | 15.137381 | 0.384657 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.814218 |  2.904 | 0.052751 |  2.838618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.838618 |  2.297848 | 0.0 | 0.026376 | 0.009758 |  2.288091 | 0.407109 |  0.843324 | 0.0 | 0.077722 |  0.921046 | 0.972627 | 0.972627 | 0.270174 | 1.947249 | 0.0 | 0.0 |  180.46807 |  8.157764 | 15.466767 | 0.270174 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.814516 |  0.484 | 0.113609 |  0.431249 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.431249 |   0.35113 | 0.0 | 0.056805 | 0.009726 |  0.341405 | 0.407258 |  0.842125 | 0.0 |  0.07937 |  0.921495 | 0.909928 | 0.909928 | 0.252758 | 1.886391 | 0.0 | 0.0 |  180.50111 |  7.249786 | 15.794655 | 0.252758 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.813224 |  0.242 | 0.353524 |  0.128391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.128391 |  0.104577 | 0.0 | 0.176762 | 0.009738 |  0.094839 | 0.406612 |  0.583181 | 0.0 | 0.081006 |  0.664188 | 0.864216 | 0.864216 |  0.24006 | 1.646476 | 0.0 | 0.0 | 180.357899 |  6.354831 | 16.120261 |  0.24006 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.809243 |    0.0 | 0.903769 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.451885 | 0.009821 | -0.009821 | 0.404621 |  0.396477 | 0.0 | 0.082624 |  0.479102 | 0.680237 | 0.680237 | 0.188955 | 0.742707 | 0.0 | 0.0 | 179.915836 |  5.543912 | 16.442258 | 0.188955 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.783769 |    0.0 | 0.742707 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.864415 | 0.010042 | -0.010042 | 0.391885 |   0.29117 | 0.0 | 0.084171 |  0.375341 | 0.497174 | 0.497174 | 0.138104 |      0.0 | 0.0 | 0.0 | 177.061463 |  4.850815 | 16.749972 | 0.138104 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.757049 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.055807 | 0.011469 | -0.011469 | 0.378524 |  0.216558 | 0.0 | 0.085642 |  0.302201 | 0.382145 | 0.382145 | 0.106151 |      0.0 | 0.0 | 0.0 | 174.017125 |  4.244263 | 17.042854 | 0.106151 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.742404 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     1.146 |  0.867578 | 0.0 | 1.982694 | 0.012852 |  0.854726 | 0.371202 |   0.20687 | 0.0 |  0.08707 |   0.29394 | 0.316618 | 0.316618 |  0.08795 |      0.0 | 0.0 | 0.0 | 172.325705 |  4.520918 | 17.326985 |  0.08795 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.713684 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.37986 | 0.013837 | -0.013837 | 0.356842 |  0.208231 | 0.0 | 0.088419 |   0.29665 | 0.296378 | 0.296378 | 0.082327 |      0.0 | 0.0 | 0.0 | 168.959683 |  3.942007 | 17.595408 | 0.082327 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.686389 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.277978 |  0.01552 |  -0.01552 | 0.343195 |  0.154721 | 0.0 | 0.089693 |  0.244414 |  0.28444 |  0.28444 | 0.079011 |      0.0 | 0.0 | 0.0 | 165.697224 |  3.428572 |  17.84891 | 0.079011 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.640894 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.602665 | 0.017151 | -0.017151 | 0.320447 |  0.124208 | 0.0 | 0.090847 |  0.215054 | 0.249497 | 0.249497 | 0.069305 |      0.0 | 0.0 | 0.0 | 160.111711 |  2.966766 |  18.07851 | 0.069305 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.611492 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.735707 | 0.019944 | -0.019944 | 0.305746 |  0.091815 | 0.0 | 0.091921 |  0.183736 | 0.214619 | 0.214619 | 0.059616 |      0.0 | 0.0 | 0.0 | 156.395948 |   2.54926 | 18.292335 | 0.059616 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.580819 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.994839 | 0.021802 | -0.021802 | 0.290409 |  0.067147 | 0.0 | 0.092914 |  0.160061 | 0.185435 | 0.185435 |  0.05151 |      0.0 | 0.0 | 0.0 | 152.422911 |  2.169902 | 18.489831 |  0.05151 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.562769 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.410799 | 0.023789 | -0.023789 | 0.281385 |  0.044672 | 0.0 | 0.093856 |  0.138528 | 0.160537 | 0.160537 | 0.044594 |      0.0 | 0.0 | 0.0 |   150.0359 |  1.820057 | 18.677359 | 0.044594 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.554825 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.087741 | 0.024982 | -0.024982 | 0.277412 |  0.026698 | 0.0 | 0.094774 |  0.121472 | 0.139523 | 0.139523 | 0.038756 |      0.0 | 0.0 | 0.0 | 148.973141 |  1.490964 | 18.859998 | 0.038756 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.553548 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.19702 | 0.025513 | -0.025513 | 0.276774 |  0.014083 | 0.0 | 0.095684 |  0.109766 | 0.122661 | 0.122661 | 0.034073 |      0.0 | 0.0 | 0.0 | 148.801635 |  1.174594 | 19.041088 | 0.034073 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.553272 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062717 | 0.025599 | -0.025599 | 0.276636 |  0.006378 | 0.0 | 0.096589 |  0.102967 | 0.110857 | 0.110857 | 0.030794 |      0.0 | 0.0 | 0.0 | 148.764517 |   0.86598 | 19.221135 | 0.030794 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.553006 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061417 | 0.025618 | -0.025618 | 0.276503 |  0.002232 | 0.0 | 0.097488 |  0.099721 | 0.103757 | 0.103757 | 0.028821 |      0.0 | 0.0 | 0.0 | 148.728719 |  0.561627 |  19.40015 | 0.028821 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.552748 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060361 | 0.025636 | -0.025636 | 0.276374 |   0.00046 | 0.0 | 0.098383 |  0.098843 | 0.100247 | 0.100247 | 0.027846 |      0.0 | 0.0 | 0.0 | 148.693993 |  0.259158 | 19.578141 | 0.027846 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.552496 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.059558 | 0.025653 | -0.025653 | 0.233484 |  0.000021 | 0.0 | 0.099058 |  0.099079 |  0.09909 |  0.09909 | 0.027525 |      0.0 | 0.0 | 0.0 | 148.660088 |       0.0 | 19.712567 | 0.027525 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.552058 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.058872 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.098563 |  0.098563 | 0.098912 | 0.098912 | 0.027476 |      0.0 | 0.0 | 0.0 | 148.601216 |       0.0 | 19.614004 | 0.027476 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.551609 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.06043 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.09807 |   0.09807 | 0.098568 | 0.098568 |  0.02738 |      0.0 | 0.0 | 0.0 | 148.540787 |       0.0 | 19.515934 |  0.02738 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.551159 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060665 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.09758 |   0.09758 | 0.098071 | 0.098071 | 0.027242 |      0.0 | 0.0 | 0.0 | 148.480122 |       0.0 | 19.418354 | 0.027242 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.550969 |  1.573 | 0.054992 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.025516 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.097092 |  0.097092 |  0.09758 |  0.09758 | 0.027106 | 1.518008 | 0.0 | 0.0 | 148.454605 |       0.0 | 19.321262 | 0.027106 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.550748 |    0.0 | 0.064218 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.029792 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.096606 |  0.096606 | 0.097092 | 0.097092 |  0.02697 |  1.45379 | 0.0 | 0.0 | 148.424813 |       0.0 | 19.224656 |  0.02697 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.550523 |    0.0 | 0.065345 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.030309 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.096123 |  0.096123 | 0.096607 | 0.096607 | 0.026835 | 1.388445 | 0.0 | 0.0 | 148.394504 |       0.0 | 19.128533 | 0.026835 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.549702 |    0.0 | 0.238876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.110775 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095643 |  0.095643 | 0.096124 | 0.096124 | 0.026701 | 1.149568 | 0.0 | 0.0 |  148.28373 |       0.0 |  19.03289 | 0.026701 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.546324 |  0.847 | 0.984764 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.456326 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095164 |  0.095164 | 0.095643 | 0.095643 | 0.026568 | 1.011805 | 0.0 | 0.0 | 147.827403 |       0.0 | 18.937726 | 0.026568 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.535238 |  0.484 | 1.495805 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.507469 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094689 |  0.094689 | 0.095165 | 0.095165 | 0.026435 |      0.0 | 0.0 | 0.0 | 146.319934 |       0.0 | 18.843037 | 0.026435 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.50914 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.611809 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094215 |  0.094215 | 0.094689 | 0.094689 | 0.026303 |      0.0 | 0.0 | 0.0 | 142.708125 |       0.0 | 18.748822 | 0.026303 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.476282 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.681784 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093744 |  0.093744 | 0.094216 | 0.094216 | 0.026171 |      0.0 | 0.0 | 0.0 |  138.02634 |       0.0 | 18.655078 | 0.026171 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.43859 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.574024 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093275 |  0.093275 | 0.093745 | 0.093745 |  0.02604 |      0.0 | 0.0 | 0.0 | 132.452317 |       0.0 | 18.561802 |  0.02604 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.403882 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  5.34883 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092809 |  0.092809 | 0.093276 | 0.093276 |  0.02591 |      0.0 | 0.0 | 0.0 | 127.103487 |       0.0 | 18.468993 |  0.02591 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.363302 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.554356 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092345 |  0.092345 |  0.09281 |  0.09281 |  0.02578 |      0.0 | 0.0 | 0.0 | 120.549131 |       0.0 | 18.376648 |  0.02578 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.331231 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.443707 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091883 |  0.091883 | 0.092345 | 0.092345 | 0.025652 |      0.0 | 0.0 | 0.0 | 115.105423 |       0.0 | 18.284765 | 0.025652 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.306641 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.355067 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091424 |  0.091424 | 0.091884 | 0.091884 | 0.025523 |      0.0 | 0.0 | 0.0 | 110.750356 |       0.0 | 18.193341 | 0.025523 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.288334 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.356922 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090967 |  0.090967 | 0.091424 | 0.091424 | 0.025396 |      0.0 | 0.0 | 0.0 | 107.393434 |       0.0 | 18.102375 | 0.025396 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.280209 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.523832 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090512 |  0.090512 | 0.090967 | 0.090967 | 0.025269 |      0.0 | 0.0 | 0.0 | 105.869602 |       0.0 | 18.011863 | 0.025269 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.276689 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.66705 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090059 |  0.090059 | 0.090512 | 0.090512 | 0.025142 |      0.0 | 0.0 | 0.0 | 105.202551 |       0.0 | 17.921804 | 0.025142 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.275812 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.166847 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089609 |  0.089609 |  0.09006 |  0.09006 | 0.025017 |      0.0 | 0.0 | 0.0 | 105.035704 |       0.0 | 17.832194 | 0.025017 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.275575 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.045245 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089161 |  0.089161 |  0.08961 |  0.08961 | 0.024892 |      0.0 | 0.0 | 0.0 | 104.990459 |       0.0 | 17.743034 | 0.024892 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.275347 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043337 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088715 |  0.088715 | 0.089161 | 0.089161 | 0.024767 |      0.0 | 0.0 | 0.0 | 104.947122 |       0.0 | 17.654318 | 0.024767 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.275124 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042677 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088272 |  0.088272 | 0.088716 | 0.088716 | 0.024643 |      0.0 | 0.0 | 0.0 | 104.904445 |       0.0 | 17.566047 | 0.024643 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.274899 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042844 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.08783 |   0.08783 | 0.088272 | 0.088272 |  0.02452 |      0.0 | 0.0 | 0.0 | 104.861601 |       0.0 | 17.478217 |  0.02452 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96_low_accuracy:

low accuracy
____________

The following example indicates caution when reducing the value of |RecStep| to save
computation time.  Setting |RecStep| to 1, which is the smallest possible value, results
in low accuracies.  You can see this by comparing the time series of |Q0| calculated in
this and the last example.  The time series of this example is much more wrong, as the
peak of |Q0| is high above the peak of |R|, which is physically impossible:

.. integration-test::

    >>> recstep(1)
    >>> test("hland_96_low_accuracy")
    |        date |    p |    t |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inrc |     outrc |        rt |       qt |       ic |  sp |  wc |         sm |       uz |        lz |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249758 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.048465 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.061111 |  0.061111 | 0.016975 |      0.0 | 0.0 | 0.0 |  99.951535 |      0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249523 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.047044 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.088833 |  0.088833 | 0.024676 |      0.0 | 0.0 | 0.0 |  99.904491 |      0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24929 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046628 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |   0.04975 |   0.04975 |  0.01382 |      0.0 | 0.0 | 0.0 |  99.857863 |      0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24906 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04613 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049502 |  0.049502 |  0.01375 |      0.0 | 0.0 | 0.0 |  99.811734 |      0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248827 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046584 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049254 |  0.049254 | 0.013682 |      0.0 | 0.0 | 0.0 |   99.76515 |      0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248581 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.049338 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049008 |  0.049008 | 0.013613 |      0.0 | 0.0 | 0.0 |  99.715812 |      0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.247883 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.140081 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048763 |  0.048763 | 0.013545 |      0.0 | 0.0 | 0.0 |   99.57573 |      0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.243293 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.92625 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048519 |  0.048519 | 0.013477 |      0.0 | 0.0 | 0.0 |  98.649481 |      0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.233168 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  2.07444 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048276 |  0.048276 |  0.01341 |      0.0 | 0.0 | 0.0 |  96.575041 |      0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.217035 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.400961 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048035 |  0.048035 | 0.013343 |      0.0 | 0.0 | 0.0 |   93.17408 |      0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.200115 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.705742 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047795 |  0.047795 | 0.013276 |      0.0 | 0.0 | 0.0 |  89.468338 |      0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.182474 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.03435 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047556 |  0.047556 |  0.01321 |      0.0 | 0.0 | 0.0 |  85.433988 |      0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.173582 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.107634 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047318 |  0.047318 | 0.013144 |      0.0 | 0.0 | 0.0 |  83.326354 |      0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.16269 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.656691 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047081 |  0.047081 | 0.013078 |      0.0 | 0.0 | 0.0 |  80.669664 |      0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.151312 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.871924 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046846 |  0.046846 | 0.013013 |      0.0 | 0.0 | 0.0 |   77.79774 |      0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.147932 |  0.242 |    0.242 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.873977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046612 |  0.046612 | 0.012948 |      0.0 | 0.0 | 0.0 |  76.923763 |      0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.145338 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.677397 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046379 |  0.046379 | 0.012883 |      0.0 | 0.0 | 0.0 |  76.246366 |      0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.143752 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.417048 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046147 |  0.046147 | 0.012819 |      0.0 | 0.0 | 0.0 |  75.829319 |      0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.143309 |  1.573 | 0.494092 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.117083 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045916 |  0.045916 | 0.012754 | 1.078908 | 0.0 | 0.0 |  75.712235 |      0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.163077 |  6.776 |  0.08755 |  5.854908 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.854908 |  0.839058 | 0.0 | 0.022103 | 0.059636 |  0.779422 | 0.081539 |  0.078374 | 0.0 | 0.045866 |  0.124239 |  0.063193 |  0.063193 | 0.017554 |  1.91245 | 0.0 | 0.0 |  80.765617 |  0.61951 |  9.127236 | 0.017554 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.17503 |  3.509 |   0.0539 |   3.42145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.42145 |   0.55796 | 0.0 | 0.014096 | 0.058185 |  0.499775 | 0.087515 |  0.204838 | 0.0 | 0.046074 |  0.250912 |  0.134932 |  0.134932 | 0.037481 |   1.9461 | 0.0 | 0.0 |  83.673196 | 0.826932 |  9.168677 | 0.037481 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.19609 |  5.929 | 0.041902 |    5.8751 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    5.8751 |  1.028319 | 0.0 | 0.011598 |  0.05574 |  0.972579 | 0.098045 |  0.653286 | 0.0 | 0.046334 |  0.699619 |  0.322475 |  0.322475 | 0.089576 | 1.958098 | 0.0 | 0.0 |  88.564118 | 1.048181 |  9.220388 | 0.089576 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.244458 | 12.826 | 0.020954 | 12.784098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.784098 |  2.506835 | 0.0 | 0.006475 | 0.050579 |  2.456256 | 0.122229 |   2.64841 | 0.0 | 0.046713 |  2.695123 |  1.043352 |  1.043352 |  0.28982 | 1.979046 | 0.0 | 0.0 |  98.885486 | 0.733797 |  9.295905 |  0.28982 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.244969 |  0.121 | 0.074152 |  0.100046 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.100046 |  0.024457 | 0.0 | 0.022944 | 0.050519 | -0.026062 | 0.122484 |  0.013636 | 0.0 | 0.047092 |  0.060728 |  1.666256 |  1.666256 | 0.462849 | 1.925848 | 0.0 | 0.0 |  98.988651 | 0.571614 |  9.371297 | 0.462849 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.248009 |  0.847 |  0.06896 |  0.772848 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.772848 |  0.189324 | 0.0 | 0.021469 | 0.050214 |   0.13911 | 0.124004 |   0.01324 | 0.0 | 0.047477 |  0.060717 |  0.646147 |  0.646147 | 0.179485 |  1.93104 | 0.0 | 0.0 |   99.60092 |  0.57348 |  9.447825 | 0.179485 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.261688 |   3.63 | 0.052207 |   3.56104 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.56104 |  0.883169 | 0.0 | 0.016695 | 0.048861 |  0.834308 | 0.130844 |  0.116188 | 0.0 | 0.047893 |  0.164082 |  0.083689 |  0.083689 | 0.023247 | 1.947793 | 0.0 | 0.0 | 102.310958 | 1.160755 |  9.530776 | 0.023247 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.271324 |  2.541 | 0.057919 |  2.488793 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.488793 |  0.651288 | 0.0 | 0.018859 | 0.047926 |  0.603362 | 0.135662 |  0.216204 | 0.0 | 0.048332 |  0.264536 |  0.163435 |  0.163435 | 0.045399 | 1.942081 | 0.0 | 0.0 |  104.17753 | 1.412252 |  9.618105 | 0.045399 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.321154 | 12.584 | 0.021144 | 12.526081 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.526081 |  3.398626 | 0.0 | 0.007489 | 0.043348 |  3.355278 | 0.160577 |  2.951907 | 0.0 | 0.048893 |    3.0008 |  0.850272 |  0.850272 | 0.236187 | 1.978856 | 0.0 | 0.0 | 113.340843 | 1.655046 |  9.729789 | 0.236187 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.33771 |  4.235 | 0.048561 |  4.213856 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.213856 |  1.353295 | 0.0 |  0.01764 | 0.041899 |  1.311396 | 0.168855 |  0.568484 | 0.0 | 0.049493 |  0.617978 |  1.863225 |  1.863225 | 0.517563 | 1.951439 | 0.0 | 0.0 | 116.225662 | 2.229102 |  9.849151 | 0.517563 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.353671 |  4.114 | 0.048983 |  4.065439 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.065439 |   1.37294 | 0.0 | 0.018209 | 0.040541 |  1.332399 | 0.176836 |  0.876494 | 0.0 |  0.05013 |  0.926624 |  1.216082 |  1.216082 | 0.337801 | 1.951017 | 0.0 | 0.0 | 118.940493 | 2.508172 |  9.975856 | 0.337801 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.359045 |  1.452 | 0.124728 |  1.403017 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.403017 |  0.496206 | 0.0 | 0.046729 | 0.040076 |   0.45613 | 0.179522 |  0.466581 | 0.0 | 0.050777 |  0.517358 |  0.767088 |  0.767088 |  0.21308 | 1.875272 | 0.0 | 0.0 | 119.840651 | 2.318198 | 10.104602 |  0.21308 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.357128 |  0.121 |  0.96204 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.360406 |  0.04008 |  -0.04008 | 0.178564 |  0.203194 | 0.0 | 0.051416 |   0.25461 |  0.549918 |  0.549918 | 0.152755 | 1.034232 | 0.0 | 0.0 | 119.520324 | 1.896361 |  10.23175 | 0.152755 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.348271 |    0.0 | 1.034232 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.531595 |  0.04024 |  -0.04024 | 0.174135 |  0.112646 | 0.0 | 0.052029 |  0.164675 |  0.293013 |  0.293013 | 0.081392 |      0.0 | 0.0 | 0.0 | 118.028969 | 1.569339 | 10.353856 | 0.081392 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.342066 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.097079 | 0.040986 | -0.040986 | 0.171033 |  0.062476 | 0.0 | 0.052624 |  0.115101 |  0.173644 |  0.173644 | 0.048234 |      0.0 | 0.0 | 0.0 | 116.972875 | 1.294844 | 10.472264 | 0.048234 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.334628 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.320389 | 0.041514 | -0.041514 | 0.167314 |  0.034184 | 0.0 | 0.053198 |  0.087382 |  0.119958 |  0.119958 | 0.033322 |      0.0 | 0.0 | 0.0 |    115.694 | 1.051833 |  10.58638 | 0.033322 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.31537 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.420519 | 0.042153 | -0.042153 | 0.157685 |  0.019717 | 0.0 |  0.05372 |  0.073438 |  0.090443 |  0.090443 | 0.025123 |      0.0 | 0.0 | 0.0 | 112.315634 | 0.832277 | 10.690345 | 0.025123 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.321336 |  4.356 | 1.680962 |     2.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     2.356 |  0.743012 | 0.0 | 0.598694 | 0.043036 |  0.699976 | 0.160668 |  0.077766 | 0.0 | 0.054255 |  0.132021 |  0.089555 |  0.089555 | 0.024876 | 0.319038 | 0.0 | 0.0 | 113.372964 | 1.293819 | 10.796758 | 0.024876 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.331994 |  7.139 |      2.0 |  5.458038 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.458038 |  1.753863 | 0.0 | 1.880771 | 0.041461 |  1.712401 | 0.165997 |  0.626134 | 0.0 | 0.054814 |  0.680948 |  0.240986 |  0.240986 | 0.066941 |      0.0 | 0.0 | 0.0 |  115.23783 |  2.21409 | 10.907941 | 0.066941 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.322622 |  1.331 |    1.331 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.680563 | 0.042381 | -0.042381 | 0.161311 |  0.241971 | 0.0 | 0.055346 |  0.297318 |  0.473713 |  0.473713 | 0.131587 |      0.0 | 0.0 | 0.0 | 113.599648 | 1.768426 | 11.013906 | 0.131587 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.41727 | 25.047 | 0.132786 |    23.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    23.047 |  7.435469 | 0.0 | 0.053632 | 0.035394 |  7.400075 | 0.208635 |  8.959866 | 0.0 | 0.056113 |  9.015979 |  2.320049 |  2.320049 | 0.644458 | 1.867214 | 0.0 | 0.0 | 129.192941 |      0.0 | 11.166428 | 0.644458 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.60725 | 45.859 | 0.017578 | 45.726214 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.726214 | 19.080196 | 0.0 | 0.008562 | 0.022081 | 19.058115 | 0.303625 |  18.75449 | 0.0 |  0.05735 | 18.811841 |  9.255357 |  9.255357 | 2.570932 | 1.982422 | 0.0 | 0.0 | 155.852479 |      0.0 | 11.412703 | 2.570932 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.63483 |  9.922 | 0.821583 |  9.904422 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.904422 |  6.014459 | 0.0 | 0.410182 | 0.020129 |   5.99433 | 0.317415 |  0.715098 | 0.0 | 0.058651 |  0.773749 | 12.626518 | 12.626518 | 3.507366 | 1.178417 | 0.0 | 0.0 | 159.352388 | 4.961817 | 11.671467 | 3.507366 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.642133 |  4.356 | 0.792694 |  3.534417 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.534417 |  2.243752 | 0.0 | 0.396347 | 0.019678 |  2.224074 | 0.321066 |  1.221838 | 0.0 | 0.059963 |  1.281801 |  4.895114 |  4.895114 | 1.359754 | 1.207306 | 0.0 | 0.0 | 160.266384 | 5.642986 | 11.932571 | 1.359754 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.665884 |  9.075 | 0.090607 |  8.282306 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.282306 |  5.318341 | 0.0 | 0.045304 | 0.018385 |  5.299956 | 0.332942 |  4.045297 | 0.0 | 0.061328 |  4.106625 |  1.796639 |  1.796639 | 0.499066 | 1.909393 | 0.0 | 0.0 | 163.203431 | 6.564703 | 12.204185 | 0.499066 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.728148 | 22.385 | 0.007604 | 22.294393 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.294393 | 14.845479 | 0.0 | 0.003802 | 0.014674 | 14.830806 | 0.364074 | 21.031434 | 0.0 | 0.062841 | 21.094276 |   7.25392 |   7.25392 | 2.014978 | 1.992396 | 0.0 | 0.0 | 170.663216 |      0.0 | 12.505418 | 2.014978 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.772056 | 18.634 | 0.011044 | 18.626396 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.626396 | 13.562779 | 0.0 | 0.005522 | 0.012137 | 13.550643 | 0.386028 |  4.957673 | 0.0 | 0.064457 |   5.02213 | 13.747654 | 13.747654 | 3.818793 | 1.988956 | 0.0 | 0.0 | 175.733447 | 8.206942 | 12.826989 | 3.818793 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.787332 |  7.623 | 0.032783 |  7.611956 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.611956 |  5.876857 | 0.0 | 0.016392 | 0.011266 |  5.865591 | 0.393666 |  5.244176 | 0.0 | 0.066103 |   5.31028 |  8.657751 |  8.657751 | 2.404931 | 1.967217 | 0.0 | 0.0 |  177.46342 | 8.434691 | 13.154552 | 2.404931 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.791468 |  2.299 | 0.054873 |  2.266217 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.266217 |  1.784264 | 0.0 | 0.027436 | 0.011027 |  1.773237 | 0.395734 |  1.905454 | 0.0 | 0.067751 |  1.973205 |  4.504674 |  4.504674 | 1.251298 | 1.945127 | 0.0 | 0.0 | 177.928964 |  7.90674 | 13.482534 | 1.251298 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.802325 |  5.929 | 0.038243 |  5.874127 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.874127 |  4.649183 | 0.0 | 0.019122 | 0.010423 |   4.63876 | 0.401163 |   3.46793 | 0.0 | 0.069418 |  3.537348 |  3.062365 |  3.062365 | 0.850657 | 1.961757 | 0.0 | 0.0 | 179.145209 | 8.676408 | 13.814278 | 0.850657 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.807919 |  3.267 |  0.04981 |  3.228757 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.228757 |  2.590513 | 0.0 | 0.024905 | 0.010108 |  2.580405 |  0.40396 |  2.423974 | 0.0 | 0.071091 |  2.495065 |  2.958142 |  2.958142 | 0.821706 |  1.95019 | 0.0 | 0.0 | 179.768657 | 8.428879 | 14.147147 | 0.821706 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.808679 |  0.605 | 0.064366 |   0.55519 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.55519 |  0.448549 | 0.0 | 0.032183 | 0.010062 |  0.438486 |  0.40434 |  1.146168 | 0.0 | 0.072757 |  1.218926 |  2.443097 |  2.443097 | 0.678638 | 1.935634 | 0.0 | 0.0 | 179.853178 | 7.316858 | 14.478729 | 0.678638 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.808774 |  0.242 | 0.067009 |  0.177634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.177634 |  0.143649 | 0.0 | 0.033505 | 0.010056 |  0.133593 | 0.404387 |  0.661239 | 0.0 | 0.074416 |  0.735654 |  1.395118 |  1.395118 | 0.387533 | 1.932991 | 0.0 | 0.0 | 179.863714 | 6.384825 |   14.8087 | 0.387533 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.809495 |  0.605 | 0.065382 |  0.537991 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.537991 |  0.435113 | 0.0 | 0.032691 | 0.010017 |  0.425096 | 0.404748 |  0.495393 | 0.0 | 0.076067 |   0.57146 |   0.80656 |   0.80656 | 0.224045 | 1.934618 | 0.0 | 0.0 | 179.943918 | 5.909781 | 15.137381 | 0.224045 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.814218 |  2.904 | 0.052751 |  2.838618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.838618 |  2.297848 | 0.0 | 0.026376 | 0.009758 |  2.288091 | 0.407109 |  0.876027 | 0.0 | 0.077722 |   0.95375 |  0.692901 |  0.692901 | 0.192472 | 1.947249 | 0.0 | 0.0 |  180.46807 | 6.914735 | 15.466767 | 0.192472 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.814516 |  0.484 | 0.113609 |  0.431249 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.431249 |   0.35113 | 0.0 | 0.056805 | 0.009726 |  0.341405 | 0.407258 |  0.594511 | 0.0 |  0.07937 |  0.673881 |  0.806603 |  0.806603 | 0.224056 | 1.886391 | 0.0 | 0.0 |  180.50111 | 6.254371 | 15.794655 | 0.224056 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.813224 |  0.242 | 0.353524 |  0.128391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.128391 |  0.104577 | 0.0 | 0.176762 | 0.009738 |  0.094839 | 0.406612 |   0.39021 | 0.0 | 0.081006 |  0.471216 |  0.691037 |  0.691037 | 0.191955 | 1.646476 | 0.0 | 0.0 | 180.357899 | 5.552388 | 16.120261 | 0.191955 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.809243 |    0.0 | 0.903769 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.451885 | 0.009821 | -0.009821 | 0.404621 |  0.255936 | 0.0 | 0.082624 |  0.338561 |  0.486774 |  0.486774 | 0.135215 | 0.742707 | 0.0 | 0.0 | 179.915836 | 4.882009 | 16.442258 | 0.135215 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.783769 |    0.0 | 0.742707 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.864415 | 0.010042 | -0.010042 | 0.391885 |  0.186764 | 0.0 | 0.084171 |  0.270935 |  0.353012 |  0.353012 | 0.098059 |      0.0 | 0.0 | 0.0 | 177.061463 | 4.293318 | 16.749972 | 0.098059 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.757049 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.055807 | 0.011469 | -0.011469 | 0.378524 |  0.137067 | 0.0 | 0.085642 |  0.222709 |  0.275246 |  0.275246 | 0.076457 |      0.0 | 0.0 | 0.0 | 174.017125 | 3.766258 | 17.042854 | 0.076457 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.742404 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     1.146 |  0.867578 | 0.0 | 1.982694 | 0.012852 |  0.854726 | 0.371202 |  0.187577 | 0.0 |  0.08707 |  0.274647 |  0.244968 |  0.244968 | 0.068047 |      0.0 | 0.0 | 0.0 | 172.325705 | 4.062205 | 17.326985 | 0.068047 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.713684 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.37986 | 0.013837 | -0.013837 | 0.356842 |  0.138388 | 0.0 | 0.088419 |  0.226807 |  0.252474 |  0.252474 | 0.070132 |      0.0 | 0.0 | 0.0 | 168.959683 | 3.553137 | 17.595408 | 0.070132 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.686389 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.277978 |  0.01552 |  -0.01552 | 0.343195 |  0.100801 | 0.0 | 0.089693 |  0.190494 |  0.229369 |  0.229369 | 0.063714 |      0.0 | 0.0 | 0.0 | 165.697224 | 3.093621 |  17.84891 | 0.063714 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.640894 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.602665 | 0.017151 | -0.017151 | 0.320447 |  0.079522 | 0.0 | 0.090847 |  0.170369 |  0.194092 |  0.194092 | 0.053914 |      0.0 | 0.0 | 0.0 | 160.111711 | 2.676501 |  18.07851 | 0.053914 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.611492 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.735707 | 0.019944 | -0.019944 | 0.305746 |  0.056817 | 0.0 | 0.091921 |  0.148738 |  0.170035 |  0.170035 | 0.047232 |      0.0 | 0.0 | 0.0 | 156.395948 | 2.293993 | 18.292335 | 0.047232 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.580819 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.994839 | 0.021802 | -0.021802 | 0.290409 |  0.039723 | 0.0 | 0.092914 |  0.132637 |  0.149967 |  0.149967 | 0.041658 |      0.0 | 0.0 | 0.0 | 152.422911 | 1.942058 | 18.489831 | 0.041658 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.562769 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.410799 | 0.023789 | -0.023789 | 0.281385 |  0.024607 | 0.0 | 0.093856 |  0.118463 |  0.133065 |  0.133065 | 0.036963 |      0.0 | 0.0 | 0.0 |   150.0359 | 1.612278 | 18.677359 | 0.036963 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.554825 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.087741 | 0.024982 | -0.024982 | 0.277412 |  0.013159 | 0.0 | 0.094774 |  0.107933 |  0.119273 |  0.119273 | 0.033131 |      0.0 | 0.0 | 0.0 | 148.973141 | 1.296724 | 18.859998 | 0.033131 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.553548 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.19702 | 0.025513 | -0.025513 | 0.276774 |  0.005798 | 0.0 | 0.095684 |  0.101482 |  0.108839 |  0.108839 | 0.030233 |      0.0 | 0.0 | 0.0 | 148.801635 | 0.988639 | 19.041088 | 0.030233 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.553272 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062717 | 0.025599 | -0.025599 | 0.276636 |   0.00191 | 0.0 | 0.096589 |  0.098498 |  0.102252 |  0.102252 | 0.028403 |      0.0 | 0.0 | 0.0 | 148.764517 | 0.684494 | 19.221135 | 0.028403 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.553006 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061417 | 0.025618 | -0.025618 | 0.276503 |  0.000331 | 0.0 | 0.097488 |  0.097819 |   0.09901 |   0.09901 | 0.027503 |      0.0 | 0.0 | 0.0 | 148.728719 | 0.382043 |  19.40015 | 0.027503 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.552748 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060361 | 0.025636 | -0.025636 | 0.276374 |  0.000003 | 0.0 | 0.098383 |  0.098386 |  0.098096 |  0.098096 | 0.027249 |      0.0 | 0.0 | 0.0 | 148.693993 |  0.08003 | 19.578141 | 0.027249 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.552496 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.059558 | 0.025653 | -0.025653 | 0.054377 |       0.0 | 0.0 | 0.098163 |  0.098163 |   0.09821 |   0.09821 | 0.027281 |      0.0 | 0.0 | 0.0 | 148.660088 |      0.0 | 19.534356 | 0.027281 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.552058 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.058872 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.097672 |  0.097672 |  0.098103 |  0.098103 | 0.027251 |      0.0 | 0.0 | 0.0 | 148.601216 |      0.0 | 19.436684 | 0.027251 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.551609 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.06043 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.097183 |  0.097183 |  0.097672 |  0.097672 | 0.027131 |      0.0 | 0.0 | 0.0 | 148.540787 |      0.0 | 19.339501 | 0.027131 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.551159 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060665 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.096698 |  0.096698 |  0.097184 |  0.097184 | 0.026996 |      0.0 | 0.0 | 0.0 | 148.480122 |      0.0 | 19.242803 | 0.026996 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.550969 |  1.573 | 0.054992 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.025516 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.096214 |  0.096214 |  0.096698 |  0.096698 | 0.026861 | 1.518008 | 0.0 | 0.0 | 148.454605 |      0.0 | 19.146589 | 0.026861 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.550748 |    0.0 | 0.064218 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.029792 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095733 |  0.095733 |  0.096215 |  0.096215 | 0.026726 |  1.45379 | 0.0 | 0.0 | 148.424813 |      0.0 | 19.050856 | 0.026726 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.550523 |    0.0 | 0.065345 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.030309 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095254 |  0.095254 |  0.095733 |  0.095733 | 0.026593 | 1.388445 | 0.0 | 0.0 | 148.394504 |      0.0 | 18.955602 | 0.026593 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.549702 |    0.0 | 0.238876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.110775 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094778 |  0.094778 |  0.095255 |  0.095255 |  0.02646 | 1.149568 | 0.0 | 0.0 |  148.28373 |      0.0 | 18.860824 |  0.02646 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.546324 |  0.847 | 0.984764 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.456326 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094304 |  0.094304 |  0.094779 |  0.094779 | 0.026327 | 1.011805 | 0.0 | 0.0 | 147.827403 |      0.0 |  18.76652 | 0.026327 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.535238 |  0.484 | 1.495805 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.507469 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093833 |  0.093833 |  0.094305 |  0.094305 | 0.026196 |      0.0 | 0.0 | 0.0 | 146.319934 |      0.0 | 18.672687 | 0.026196 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.50914 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.611809 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093363 |  0.093363 |  0.093833 |  0.093833 | 0.026065 |      0.0 | 0.0 | 0.0 | 142.708125 |      0.0 | 18.579324 | 0.026065 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.476282 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.681784 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092897 |  0.092897 |  0.093364 |  0.093364 | 0.025934 |      0.0 | 0.0 | 0.0 |  138.02634 |      0.0 | 18.486427 | 0.025934 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.43859 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.574024 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092432 |  0.092432 |  0.092897 |  0.092897 | 0.025805 |      0.0 | 0.0 | 0.0 | 132.452317 |      0.0 | 18.393995 | 0.025805 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.403882 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  5.34883 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.09197 |   0.09197 |  0.092433 |  0.092433 | 0.025676 |      0.0 | 0.0 | 0.0 | 127.103487 |      0.0 | 18.302025 | 0.025676 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.363302 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.554356 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.09151 |   0.09151 |   0.09197 |   0.09197 | 0.025547 |      0.0 | 0.0 | 0.0 | 120.549131 |      0.0 | 18.210515 | 0.025547 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.331231 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.443707 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091053 |  0.091053 |  0.091511 |  0.091511 |  0.02542 |      0.0 | 0.0 | 0.0 | 115.105423 |      0.0 | 18.119463 |  0.02542 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.306641 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.355067 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090597 |  0.090597 |  0.091053 |  0.091053 | 0.025293 |      0.0 | 0.0 | 0.0 | 110.750356 |      0.0 | 18.028865 | 0.025293 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.288334 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.356922 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090144 |  0.090144 |  0.090598 |  0.090598 | 0.025166 |      0.0 | 0.0 | 0.0 | 107.393434 |      0.0 | 17.938721 | 0.025166 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.280209 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.523832 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089694 |  0.089694 |  0.090145 |  0.090145 |  0.02504 |      0.0 | 0.0 | 0.0 | 105.869602 |      0.0 | 17.849027 |  0.02504 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.276689 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.66705 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089245 |  0.089245 |  0.089694 |  0.089694 | 0.024915 |      0.0 | 0.0 | 0.0 | 105.202551 |      0.0 | 17.759782 | 0.024915 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.275812 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.166847 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088799 |  0.088799 |  0.089246 |  0.089246 |  0.02479 |      0.0 | 0.0 | 0.0 | 105.035704 |      0.0 | 17.670983 |  0.02479 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.275575 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.045245 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088355 |  0.088355 |  0.088799 |  0.088799 | 0.024667 |      0.0 | 0.0 | 0.0 | 104.990459 |      0.0 | 17.582628 | 0.024667 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.275347 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043337 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087913 |  0.087913 |  0.088355 |  0.088355 | 0.024543 |      0.0 | 0.0 | 0.0 | 104.947122 |      0.0 | 17.494715 | 0.024543 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.275124 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042677 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087474 |  0.087474 |  0.087914 |  0.087914 |  0.02442 |      0.0 | 0.0 | 0.0 | 104.904445 |      0.0 | 17.407242 |  0.02442 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.274899 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042844 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087036 |  0.087036 |  0.087474 |  0.087474 | 0.024298 |      0.0 | 0.0 | 0.0 | 104.861601 |      0.0 | 17.320205 | 0.024298 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96_sealed:

sealed area
___________

|hland_96| tends to underestimate the runoff response to small or medium-sized rain
events that fall on dry soils.  One means to improve the situation is to activate the
:ref:`contributing area <hland_96_resparea>` option.  Unfortunately, this option has the
side-effect that decreasing the soil moisture increases the direct discharge released by
the upper zone storage.  Defining sealed areas provides an alternative to the
"contributing area" approach.  The following example shows that a zone classified as
|SEALED| converts all precipitation, which is not intercepted or stored within a snow
cover, to runoff and directly passes it to the runoff concentration module.  Thus, the
definition of sealed surfaces causes higher dynamics in the simulated flood hydrograph
that is most perceptible during dry conditions:

.. integration-test::

    >>> recstep(100)
    >>> zonetype(SEALED)
    >>> test("hland_96_sealed")
    |        date |    p |    t |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |        sr |  ea |  cf | inuz | perc |  q0 |  el |  q1 |      inrc |     outrc |        rt |       qt |       ic |  sp |  wc |  sm |  uz |  lz |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.05 |      0.05 | 0.013889 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.013889 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.05 |      0.05 | 0.013889 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.013889 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.242 |    0.242 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  1.573 | 0.494092 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.078908 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  6.776 |  0.08755 |  5.854908 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.854908 |  5.854908 |  5.854908 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.854908 |  1.301091 |  1.301091 | 0.361414 |  1.91245 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.361414 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  3.509 |   0.0539 |   3.42145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.42145 |   3.42145 |   3.42145 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   3.42145 |  4.013049 |  4.013049 | 1.114736 |   1.9461 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.114736 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  5.929 | 0.041902 |    5.8751 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    5.8751 |    5.8751 |    5.8751 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |    5.8751 |  4.507474 |  4.507474 | 1.252076 | 1.958098 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.252076 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 | 12.826 | 0.020954 | 12.784098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.784098 | 12.784098 | 12.784098 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 12.784098 |  6.865177 |  6.865177 | 1.906994 | 1.979046 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.906994 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.121 | 0.074152 |  0.100046 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.100046 |  0.100046 |  0.100046 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.100046 |  8.430087 |  8.430087 | 2.341691 | 1.925848 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 2.341691 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.847 |  0.06896 |  0.772848 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.772848 |  0.772848 |  0.772848 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.772848 |  3.068236 |  3.068236 | 0.852288 |  1.93104 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.852288 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   3.63 | 0.052207 |   3.56104 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.56104 |   3.56104 |   3.56104 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   3.56104 |  1.242935 |  1.242935 |  0.34526 | 1.947793 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.34526 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.541 | 0.057919 |  2.488793 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.488793 |  2.488793 |  2.488793 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  2.488793 |  2.703165 |  2.703165 | 0.750879 | 1.942081 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.750879 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 12.584 | 0.021144 | 12.526081 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.526081 | 12.526081 | 12.526081 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 12.526081 |  4.957578 |  4.957578 | 1.377105 | 1.978856 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.377105 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.235 | 0.048561 |  4.213856 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.213856 |  4.213856 |  4.213856 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  4.213856 |  8.448411 |  8.448411 | 2.346781 | 1.951439 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 2.346781 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.114 | 0.048983 |  4.065439 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.065439 |  4.065439 |  4.065439 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  4.065439 |  6.028036 |  6.028036 | 1.674454 | 1.951017 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.674454 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.452 | 0.124728 |  1.403017 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.403017 |  1.403017 |  1.403017 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  1.403017 |  3.506771 |  3.506771 | 0.974103 | 1.875272 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.974103 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.121 |  0.96204 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  1.682885 |  1.682885 | 0.467468 | 1.034232 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.467468 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    0.0 | 1.034232 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.311781 |  0.311781 | 0.086606 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.086606 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.356 | 1.680962 |     2.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     2.356 |     2.356 |     2.356 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |     2.356 |  0.523556 |  0.523556 | 0.145432 | 0.319038 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.145432 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.139 |      2.0 |  5.458038 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.458038 |  5.458038 |  5.458038 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.458038 |  2.521786 |  2.521786 | 0.700496 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.700496 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.331 |    1.331 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  3.555799 |  3.555799 | 0.987722 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.987722 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 25.047 | 0.132786 |    23.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    23.047 |    23.047 |    23.047 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |    23.047 |  6.334453 |  6.334453 |  1.75957 | 1.867214 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.75957 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 45.859 | 0.017578 | 45.726214 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.726214 | 45.726214 | 45.726214 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 45.726214 |  22.96527 |  22.96527 | 6.379242 | 1.982422 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 6.379242 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.922 | 0.821583 |  9.904422 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.904422 |  9.904422 |  9.904422 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  9.904422 | 32.725991 | 32.725991 | 9.090553 | 1.178417 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 9.090553 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.356 | 0.792694 |  3.534417 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.534417 |  3.534417 |  3.534417 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.534417 | 16.449264 | 16.449264 |  4.56924 | 1.207306 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  4.56924 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.075 | 0.090607 |  8.282306 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.282306 |  8.282306 |  8.282306 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  8.282306 |   6.00506 |   6.00506 | 1.668072 | 1.909393 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.668072 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 22.385 | 0.007604 | 22.294393 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.294393 | 22.294393 | 22.294393 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 22.294393 | 10.341017 | 10.341017 | 2.872505 | 1.992396 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 2.872505 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 18.634 | 0.011044 | 18.626396 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.626396 | 18.626396 | 18.626396 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 18.626396 | 18.365485 | 18.365485 | 5.101524 | 1.988956 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 5.101524 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.623 | 0.032783 |  7.611956 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.611956 |  7.611956 |  7.611956 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  7.611956 | 16.993853 | 16.993853 | 4.720515 | 1.967217 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 4.720515 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.299 | 0.054873 |  2.266217 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.266217 |  2.266217 |  2.266217 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  2.266217 |  8.871667 |  8.871667 | 2.464352 | 1.945127 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 2.464352 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  5.929 | 0.038243 |  5.874127 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.874127 |  5.874127 |  5.874127 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.874127 |  4.255917 |  4.255917 | 1.182199 | 1.961757 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.182199 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.267 |  0.04981 |  3.228757 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.228757 |  3.228757 |  3.228757 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.228757 |  4.484509 |  4.484509 | 1.245697 |  1.95019 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.245697 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.605 | 0.064366 |   0.55519 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.55519 |   0.55519 |   0.55519 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   0.55519 |  3.222491 |  3.222491 | 0.895136 | 1.935634 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.895136 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.242 | 0.067009 |  0.177634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.177634 |  0.177634 |  0.177634 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.177634 |  1.065415 |  1.065415 | 0.295949 | 1.932991 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.295949 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.605 | 0.065382 |  0.537991 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.537991 |  0.537991 |  0.537991 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.537991 |  0.341615 |  0.341615 | 0.094893 | 1.934618 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.094893 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  2.904 | 0.052751 |  2.838618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.838618 |  2.838618 |  2.838618 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  2.838618 |  0.969162 |  0.969162 | 0.269212 | 1.947249 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.269212 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.484 | 0.113609 |  0.431249 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.431249 |  0.431249 |  0.431249 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.431249 |  1.792397 |  1.792397 | 0.497888 | 1.886391 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.497888 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.242 | 0.353524 |  0.128391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.128391 |  0.128391 |  0.128391 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.128391 |  0.898918 |  0.898918 | 0.249699 | 1.646476 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.249699 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 | 0.903769 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.167161 |  0.167161 | 0.046434 | 0.742707 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.046434 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 | 0.742707 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.028531 |  0.028531 | 0.007925 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.007925 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     1.146 |     1.146 |     1.146 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |     1.146 |  0.254667 |  0.254667 | 0.070741 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.070741 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.636667 |  0.636667 | 0.176852 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.176852 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.254667 |  0.254667 | 0.070741 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.070741 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  1.573 | 0.054992 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.518008 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 | 0.064218 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |  1.45379 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 | 0.065345 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.388445 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 | 0.238876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.149568 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.847 | 0.984764 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.011805 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.484 | 1.495805 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96_ilake:

internal lake
_____________

In the following example, we demonstrate the functionality of zones of type |ILAKE|.
For such "internal lakes", only the lower zone storage (|LZ|) is relevant (all other
storage values are zero).  Precipitation (|PC|) adds directly to, and evaporation
(|EL|) subtracts directly from |LZ|.  The latter occurs even when |LZ| is empty,
possibly resulting in negative storage values in drought periods.  When using
|evap_aet_hbv96| as the actual evapotranspiration submodel, the only condition
preventing lake evaporation (|EL|) is the occurrence of lake ice, which exists when the
actual temperature ("TC") is below the threshold temperature
(|evap_control.TemperatureThresholdIce|).  In our example, we set it to the unrealistic
value of 13°C, resulting in a zero |EL| for the last day of the simulation period:

.. integration-test::

    >>> zonetype(ILAKE)
    >>> model.aetmodel.parameters.control.temperaturethresholdice(13.0)
    >>> test("hland_96_ilake")
    |        date |    p |    t |   tc | fracrain | rfc | sfc | cfact | swe | gact | contriarea |     pc |  ei |     tf | spl | wcl | spg | wcg | glmelt | melt | refr |    in_ |      r |  sr |  ea |  cf | inuz | perc |  q0 |       el |       q1 |     inrc |    outrc |       rt |       qt |  ic |  sp |  wc |  sm |  uz |         lz |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.077544 | 0.049612 | 0.049612 | 0.061025 | 0.061025 | 0.016951 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.872843 | 0.016951 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.075307 | 0.048988 | 0.048988 | 0.088449 | 0.088449 | 0.024569 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.748549 | 0.024569 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.074675 | 0.048369 | 0.048369 | 0.048989 | 0.048989 | 0.013608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.625504 | 0.013608 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.073912 | 0.047758 | 0.047758 | 0.048371 | 0.048371 | 0.013436 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.503834 | 0.013436 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.074675 | 0.047146 | 0.047146 | 0.047758 | 0.047758 | 0.013266 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.382013 | 0.013266 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.079126 | 0.046514 | 0.046514 | 0.047142 | 0.047142 | 0.013095 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.256372 | 0.013095 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.224769 | 0.045158 | 0.045158 | 0.046353 | 0.046353 | 0.012876 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.986445 | 0.012876 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.488314 | 0.037491 | 0.037491 | 0.043756 | 0.043756 | 0.012154 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   7.460641 | 0.012154 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.364543 |  0.02048 |  0.02048 | 0.035414 | 0.035414 | 0.009837 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   4.075618 | 0.009837 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.634518 |      0.0 |      0.0 | 0.019709 | 0.019709 | 0.005475 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -1.558901 | 0.005475 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.363559 |      0.0 |      0.0 | 0.004551 | 0.004551 | 0.001264 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -7.922459 | 0.001264 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.214798 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -15.137257 |      0.0 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.947157 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -19.084414 |      0.0 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.101273 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -24.185687 |      0.0 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.696167 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -29.881854 |      0.0 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.242 |  0.242 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.797434 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -31.437288 |      0.0 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.408972 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -32.84626 |      0.0 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.875158 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -33.721418 |      0.0 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.573 |  1.573 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.494092 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -32.64251 |      0.0 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  6.776 | 0.0 |  6.776 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  6.776 |  6.776 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.08755 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -25.954061 |      0.0 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  3.509 | 0.0 |  3.509 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.509 |  3.509 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |   0.0539 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -22.498961 |      0.0 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  5.929 | 0.0 |  5.929 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.929 |  5.929 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.041902 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -16.611863 |      0.0 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 12.826 | 0.0 | 12.826 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.826 | 12.826 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.020954 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -3.806816 |      0.0 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.074152 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -3.759969 |      0.0 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.847 |  0.847 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.06896 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -2.981928 |      0.0 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |   3.63 | 0.0 |   3.63 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.63 |   3.63 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.052207 | 0.002979 | 0.002979 | 0.000662 | 0.000662 | 0.000184 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.592885 | 0.000184 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  2.541 | 0.0 |  2.541 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.541 |  2.541 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.057919 |  0.01538 |  0.01538 | 0.005073 | 0.005073 | 0.001409 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   3.060586 | 0.001409 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 12.584 | 0.0 | 12.584 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.584 | 12.584 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.021144 | 0.078117 | 0.078117 | 0.026566 | 0.026566 | 0.007379 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  15.545325 | 0.007379 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.235 | 0.0 |  4.235 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.235 |  4.235 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.048561 | 0.098659 | 0.098659 |  0.06874 |  0.06874 | 0.019095 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  19.633105 | 0.019095 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.114 | 0.0 |  4.114 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.114 |  4.114 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.048983 | 0.118491 | 0.118491 | 0.098501 | 0.098501 | 0.027361 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  23.579631 | 0.027361 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.452 | 0.0 |  1.452 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.452 |  1.452 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.124728 | 0.124535 | 0.124535 | 0.115427 | 0.115427 | 0.032063 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  24.782369 | 0.032063 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.96204 | 0.119707 | 0.119707 | 0.122119 | 0.122119 | 0.033922 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  23.821622 | 0.033922 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.345064 | 0.107383 | 0.107383 | 0.118041 | 0.118041 | 0.032789 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  21.369175 | 0.032789 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.486684 | 0.099412 | 0.099412 |  0.10835 |  0.10835 | 0.030097 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  19.783079 | 0.030097 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.805438 | 0.092308 | 0.092308 | 0.099605 | 0.099605 | 0.027668 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.369332 | 0.027668 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 4.728713 | 0.068808 | 0.068808 | 0.088665 | 0.088665 | 0.024629 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.692812 | 0.024629 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.356 | 0.0 |  4.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.356 |  4.356 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.680962 | 0.081839 | 0.081839 | 0.076926 | 0.076926 | 0.021368 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   16.28601 | 0.021368 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  7.139 | 0.0 |  7.139 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.139 |  7.139 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.326542 | 0.100492 | 0.100492 | 0.083089 | 0.083089 |  0.02308 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  19.997976 |  0.02308 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.331 | 0.0 |  1.331 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.331 |  1.331 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.727186 | 0.093009 | 0.093009 | 0.094684 | 0.094684 | 0.026301 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.508781 | 0.026301 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 25.047 | 0.0 | 25.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 25.047 | 25.047 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.132786 | 0.217115 | 0.217115 | 0.122251 | 0.122251 | 0.033959 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  43.205881 | 0.033959 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 45.859 | 0.0 | 45.859 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.859 | 45.859 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.017578 | 0.445237 | 0.445237 |  0.24023 |  0.24023 |  0.06673 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  88.602066 |  0.06673 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  9.922 | 0.0 |  9.922 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.922 |  9.922 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.821583 | 0.488512 | 0.488512 |  0.40416 |  0.40416 | 0.112267 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  97.213971 | 0.112267 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.356 | 0.0 |  4.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.356 |  4.356 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.792694 | 0.503886 | 0.503886 | 0.482312 | 0.482312 | 0.133976 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  100.27339 | 0.133976 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  9.075 | 0.0 |  9.075 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.075 |  9.075 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.090607 | 0.546289 | 0.546289 | 0.509893 | 0.509893 | 0.141637 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 108.711494 | 0.141637 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 22.385 | 0.0 | 22.385 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.385 | 22.385 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.007604 | 0.655444 | 0.655444 | 0.561123 | 0.561123 | 0.155867 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 130.433446 | 0.155867 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 18.634 | 0.0 | 18.634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.634 | 18.634 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.011044 | 0.745282 | 0.745282 | 0.651152 | 0.651152 | 0.180875 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  148.31112 | 0.180875 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  7.623 | 0.0 |  7.623 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.623 |  7.623 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.032783 | 0.779507 | 0.779507 | 0.732924 | 0.732924 |  0.20359 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  155.12183 |  0.20359 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  2.299 | 0.0 |  2.299 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.299 |  2.299 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.054873 |  0.78683 |  0.78683 | 0.773529 | 0.773529 | 0.214869 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 156.579127 | 0.214869 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  5.929 | 0.0 |  5.929 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.929 |  5.929 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.038243 | 0.812349 | 0.812349 | 0.790873 | 0.790873 | 0.219687 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 161.657535 | 0.219687 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  3.267 | 0.0 |  3.267 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.267 |  3.267 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.04981 | 0.824374 | 0.824374 |  0.80935 |  0.80935 |  0.22482 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 164.050351 |  0.22482 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.605 |  0.605 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.064366 | 0.822955 | 0.822955 | 0.821386 | 0.821386 | 0.228163 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  163.76803 | 0.228163 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.242 |  0.242 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.067009 | 0.819715 | 0.819715 |  0.82255 |  0.82255 | 0.228486 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 163.123306 | 0.228486 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.605 |  0.605 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.065382 | 0.818315 | 0.818315 | 0.820124 | 0.820124 | 0.227812 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  162.84461 | 0.227812 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  2.904 | 0.0 |  2.904 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.904 |  2.904 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.052751 | 0.828479 | 0.828479 | 0.820885 | 0.820885 | 0.228024 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 164.867379 | 0.228024 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.113609 | 0.826189 | 0.826189 | 0.825711 | 0.825711 | 0.229364 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 164.411581 | 0.229364 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.242 |  0.242 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.353524 |   0.8215 |   0.8215 | 0.825656 | 0.825656 | 0.229349 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 163.478556 | 0.229349 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.903769 | 0.812874 | 0.812874 | 0.820625 | 0.820625 | 0.227951 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 161.761913 | 0.227951 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.235768 | 0.792631 | 0.792631 | 0.810292 | 0.810292 | 0.225081 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 157.733515 | 0.225081 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.363 |  0.363 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.237307 | 0.774296 | 0.774296 | 0.793055 | 0.793055 | 0.220293 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.084911 | 0.220293 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  3.146 | 0.0 |  3.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.146 |  3.146 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.982694 | 0.771241 | 0.771241 | 0.777692 | 0.777692 | 0.216025 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 153.476976 | 0.216025 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.847 |  0.847 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  3.80336 | 0.752603 | 0.752603 | 0.767778 | 0.767778 | 0.213272 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 149.768013 | 0.213272 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.363 |  0.363 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.459478 | 0.733358 | 0.733358 | 0.752468 | 0.752468 | 0.209019 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 145.938177 | 0.209019 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.363 |  0.363 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.784165 | 0.702585 | 0.702585 | 0.730796 | 0.730796 | 0.202999 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 139.814427 | 0.202999 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.735707 | 0.680394 | 0.680394 | 0.704492 | 0.704492 | 0.195692 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 135.398327 | 0.195692 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 4.086328 |  0.65656 |  0.65656 | 0.680029 | 0.680029 | 0.188897 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 130.655439 | 0.188897 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.530248 | 0.640626 | 0.640626 | 0.658315 | 0.658315 | 0.182865 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 127.484565 | 0.182865 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.159786 | 0.631624 | 0.631624 | 0.642166 | 0.642166 |  0.17838 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 125.693155 |  0.17838 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.211567 | 0.627408 | 0.627408 | 0.632687 | 0.632687 | 0.175747 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  124.85418 | 0.175747 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.067425 | 0.623934 | 0.623934 | 0.627573 | 0.627573 | 0.174326 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 124.162821 | 0.174326 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.620814 | 0.620814 | 0.624013 | 0.624013 | 0.173337 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 123.542007 | 0.173337 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 |  0.61771 |  0.61771 | 0.620818 | 0.620818 | 0.172449 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 122.924297 | 0.172449 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.614621 | 0.614621 | 0.617713 | 0.617713 | 0.171587 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 122.309676 | 0.171587 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.611548 | 0.611548 | 0.614625 | 0.614625 | 0.170729 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 121.698127 | 0.170729 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.608491 | 0.608491 | 0.611552 | 0.611552 | 0.169875 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 121.089636 | 0.169875 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.605448 | 0.605448 | 0.608494 | 0.608494 | 0.169026 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 120.484188 | 0.169026 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.573 |  1.573 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.610286 | 0.610286 | 0.607199 | 0.607199 | 0.168666 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 121.446902 | 0.168666 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.607235 | 0.607235 | 0.608533 | 0.608533 | 0.169037 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 120.839668 | 0.169037 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.604198 | 0.604198 | 0.607238 | 0.607238 | 0.168677 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  120.23547 | 0.168677 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.601177 | 0.601177 | 0.604202 | 0.604202 | 0.167834 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.634292 | 0.167834 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.847 |  0.847 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.602406 | 0.602406 | 0.602122 | 0.602122 | 0.167256 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.878886 | 0.167256 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.601814 | 0.601814 | 0.602002 | 0.602002 | 0.167223 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.761071 | 0.167223 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 |  0.59941 |  0.59941 | 0.601412 | 0.601412 | 0.167059 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.282661 | 0.167059 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.249074 | 0.572588 | 0.572588 | 0.593984 | 0.593984 | 0.164996 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 113.944999 | 0.164996 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.461403 | 0.537418 | 0.537418 | 0.570733 | 0.570733 | 0.158537 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 106.946178 | 0.158537 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  6.46129 | 0.502424 | 0.502424 | 0.537457 | 0.537457 | 0.149294 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  99.982464 | 0.149294 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 8.250733 | 0.458659 | 0.458659 | 0.500475 | 0.500475 | 0.139021 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  91.273072 | 0.139021 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.225213 | 0.420239 | 0.420239 | 0.459847 | 0.459847 | 0.127735 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  83.627619 | 0.127735 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.053675 |  0.38787 |  0.38787 | 0.421584 | 0.421584 | 0.117107 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  77.186075 | 0.117107 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 4.849713 | 0.361682 | 0.361682 | 0.389243 | 0.389243 | 0.108123 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   71.97468 | 0.108123 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  2.27028 | 0.348522 | 0.348522 | 0.364577 | 0.364577 | 0.101271 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  69.355877 | 0.101271 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.008109 | 0.341739 | 0.341739 | 0.349939 | 0.349939 | 0.097205 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   68.00603 | 0.097205 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.253754 | 0.338761 | 0.338761 | 0.342585 | 0.342585 | 0.095162 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  67.413515 | 0.095162 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.068921 | 0.336723 | 0.336723 |  0.33897 |  0.33897 | 0.094158 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  67.007871 | 0.094158 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.335039 | 0.335039 | 0.336802 | 0.336802 | 0.093556 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  66.672832 | 0.093556 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.333364 | 0.333364 | 0.335041 | 0.335041 | 0.093067 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  66.339467 | 0.093067 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.331697 | 0.331697 | 0.333366 | 0.333366 | 0.092602 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   66.00777 | 0.092602 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96_snow_classes:

snow classes
____________

In this example, we divide the single zone into two snow classes.  All snow classes of
a specific zone are generally identical in size and process parameterisation.  The only
difference is that they receive different amounts of (frozen or liquid) precipitation.
The extent of this difference depends on parameter |SFDist|, for which we apply the
standard "linear-mode" value 0.2:

>>> sclass(2)
>>> sfdist(linear=0.2)

We reset the land-use type to |FIELD| and modify the input temperature series to
activate the snow routines.  In the first half of the simulation period, the
temperature is -20°C, and in the second half, it is +20°C:

>>> zonetype(FIELD)
>>> t_series = inputs.t.series.copy()
>>> tn_series = petinputs.normalairtemperature.series.copy()
>>> inputs.t.series[:48] = -20.0
>>> inputs.t.series[48:] = 20.0
>>> petinputs.normalairtemperature.series = inputs.t.series

The second snow class receives more precipitation and thus builds a deeper snow layer.
At the beginning of the warm period, the melting rates of both classes are identical.
Due to the huger ice content, the second class possesses a higher storage capacity and
releases its meltwater later.  Also, it takes longer until it is finally snow-free:

.. integration-test::

    >>> test("hland_96_snow_classes")
    |        date |    p |     t |    tc | fracrain | rfc | sfc |    cfact |                    swe | gact | contriarea |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt |               melt |      refr |       in_ |        r |  sr |       ea |       cf |      inuz |     perc |       q0 |  el |       q1 |     inrc |    outrc |       rt |       qt |       ic |                     sp |                   wc |         sm |        uz |        lz |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.24978 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.044059 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |     0.05 |     0.05 | 0.061111 | 0.061111 | 0.016975 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.955941 |       0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.249566 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.042769 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |  0.04975 |  0.04975 | 0.088833 | 0.088833 | 0.024676 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.913172 |       0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.249354 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.042392 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.049501 | 0.049501 |  0.04975 |  0.04975 |  0.01382 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.870779 |       0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.249145 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.041941 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.049254 | 0.049254 | 0.049502 | 0.049502 |  0.01375 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.828838 |       0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.248934 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.042357 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.049007 | 0.049007 | 0.049254 | 0.049254 | 0.013682 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.786481 |       0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.24871 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.044862 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048762 | 0.048762 | 0.049008 | 0.049008 | 0.013613 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.741619 |       0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.248075 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  0.12738 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048519 | 0.048519 | 0.048763 | 0.048763 | 0.013545 |      0.0 |        0.0         0.0 |       0.0        0.0 |   99.61424 |       0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.243897 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.842371 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048276 | 0.048276 | 0.048519 | 0.048519 | 0.013477 |      0.0 |        0.0         0.0 |       0.0        0.0 |  98.771869 |       0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.234661 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.888194 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048035 | 0.048035 | 0.048276 | 0.048276 |  0.01341 |      0.0 |        0.0         0.0 |       0.0        0.0 |  96.883675 |       0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.219877 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.101664 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047794 | 0.047794 | 0.048035 | 0.048035 | 0.013343 |      0.0 |        0.0         0.0 |       0.0        0.0 |  93.782011 |       0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.204264 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.390837 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047556 | 0.047556 | 0.047795 | 0.047795 | 0.013276 |      0.0 |        0.0         0.0 |       0.0        0.0 |  90.391174 |       0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.18786 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.705421 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047318 | 0.047318 | 0.047556 | 0.047556 |  0.01321 |      0.0 |        0.0         0.0 |       0.0        0.0 |  86.685753 |       0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.179529 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.944104 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047081 | 0.047081 | 0.047318 | 0.047318 | 0.013144 |      0.0 |        0.0         0.0 |       0.0        0.0 |  84.741649 |       0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.169272 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.456195 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046846 | 0.046846 | 0.047081 | 0.047081 | 0.013078 |      0.0 |        0.0         0.0 |       0.0        0.0 |  82.285455 |       0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.158493 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.663134 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046612 | 0.046612 | 0.046846 | 0.046846 | 0.013013 |      0.0 |        0.0         0.0 |       0.0        0.0 |   79.62232 |       0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.155286 |  0.286 |    0.286 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.809588 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046378 | 0.046378 | 0.046612 | 0.046612 | 0.012948 |      0.0 |        0.0         0.0 |       0.0        0.0 |  78.812732 |       0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.15281 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.630937 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046147 | 0.046147 | 0.046379 | 0.046379 | 0.012883 |      0.0 |        0.0         0.0 |       0.0        0.0 |  78.181795 |       0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.151294 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.388758 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045916 | 0.045916 | 0.046147 | 0.046147 | 0.012819 |      0.0 |        0.0         0.0 |       0.0        0.0 |  77.793036 |       0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.150881 |  1.859 |  0.43651 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.106117 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045686 | 0.045686 | 0.045916 | 0.045916 | 0.012754 |  1.42249 |        0.0         0.0 |       0.0        0.0 |  77.686919 |       0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   5.944392    8.916588 |  0.0 |   0.150881 |  8.008 | 0.070365 |   7.43049 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045458 | 0.045458 | 0.045686 | 0.045686 | 0.012691 | 1.929635 |   5.944392    8.916588 |       0.0        0.0 |  77.686919 |       0.0 |  9.046105 | 0.012691 |
    | 01/01 20:00 |  2.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   9.205699   13.808549 |  0.0 |   0.150881 |  4.147 | 0.045972 |  4.076635 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045231 | 0.045231 | 0.045458 | 0.045458 | 0.012627 | 1.954028 |   9.205699   13.808549 |       0.0        0.0 |  77.686919 |       0.0 |  9.000874 | 0.012627 |
    | 01/01 21:00 |  4.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  14.774522   22.161783 |  0.0 |   0.150881 |  7.007 |   0.0342 |  6.961028 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045004 | 0.045004 | 0.045231 | 0.045231 | 0.012564 |   1.9658 |  14.774522   22.161783 |       0.0        0.0 |  77.686919 |       0.0 |   8.95587 | 0.012564 |
    | 01/01 22:00 | 10.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.873562   40.310343 |  0.0 |   0.150881 | 15.158 | 0.015086 |   15.1238 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044779 | 0.044779 | 0.045005 | 0.045005 | 0.012501 | 1.984914 |  26.873562   40.310343 |       0.0        0.0 |  77.686919 |       0.0 |  8.911091 | 0.012501 |
    | 01/01 23:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.975893    40.46384 |  0.0 |   0.150881 |  0.143 | 0.067263 |  0.127914 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044555 | 0.044555 |  0.04478 |  0.04478 | 0.012439 | 1.932737 |  26.975893    40.46384 |       0.0        0.0 |  77.686919 |       0.0 |  8.866535 | 0.012439 |
    | 02/01 00:00 |  0.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  27.722883   41.584324 |  0.0 |   0.150881 |  1.001 | 0.061733 |  0.933737 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044333 | 0.044333 | 0.044556 | 0.044556 | 0.012377 | 1.938267 |  27.722883   41.584324 |       0.0        0.0 |  77.686919 |       0.0 |  8.822202 | 0.012377 |
    | 02/01 01:00 |  3.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  31.105497   46.658245 |  0.0 |   0.150881 |   4.29 |  0.04443 |  4.228267 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044111 | 0.044111 | 0.044333 | 0.044333 | 0.012315 |  1.95557 |  31.105497   46.658245 |       0.0        0.0 |  77.686919 |       0.0 |  8.778091 | 0.012315 |
    | 02/01 02:00 |  2.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  33.472353   50.208529 |  0.0 |   0.150881 |  3.003 | 0.050277 |   2.95857 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |  0.04389 |  0.04389 | 0.044111 | 0.044111 | 0.012253 | 1.949723 |  33.472353   50.208529 |       0.0        0.0 |  77.686919 |       0.0 |  8.734201 | 0.012253 |
    | 02/01 03:00 | 10.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  45.329731   67.994597 |  0.0 |   0.150881 | 14.872 | 0.015291 | 14.821723 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043671 | 0.043671 | 0.043891 | 0.043891 | 0.012192 | 1.984709 |  45.329731   67.994597 |       0.0        0.0 |  77.686919 |       0.0 |   8.69053 | 0.012192 |
    | 02/01 04:00 |  3.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  49.321499   73.982248 |  0.0 |   0.150881 |  5.005 | 0.040875 |  4.989709 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043453 | 0.043453 | 0.043671 | 0.043671 | 0.012131 | 1.959125 |  49.321499   73.982248 |       0.0        0.0 |  77.686919 |       0.0 |  8.647077 | 0.012131 |
    | 02/01 05:00 |  3.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.178399   79.767599 |  0.0 |   0.150881 |  4.862 | 0.041321 |  4.821125 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043235 | 0.043235 | 0.043453 | 0.043453 |  0.01207 | 1.958679 |  53.178399   79.767599 |       0.0        0.0 |  77.686919 |       0.0 |  8.603842 |  0.01207 |
    | 02/01 06:00 |  1.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.518142   81.777214 |  0.0 |   0.150881 |  1.716 | 0.110435 |  1.674679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043019 | 0.043019 | 0.043236 | 0.043236 |  0.01201 | 1.889565 |  54.518142   81.777214 |       0.0        0.0 |  77.686919 |       0.0 |  8.560823 |  0.01201 |
    | 02/01 07:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |   0.150881 |  0.143 |  0.87266 |  0.032565 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.042804 | 0.042804 | 0.043019 | 0.043019 |  0.01195 |  1.12734 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  8.518019 |  0.01195 |
    | 02/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |   0.150881 |    0.0 |  1.12734 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |  0.04259 |  0.04259 | 0.042804 | 0.042804 |  0.01189 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  8.475429 |  0.01189 |
    | 02/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |   0.150881 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.042377 | 0.042377 |  0.04259 |  0.04259 | 0.011831 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  8.433051 | 0.011831 |
    | 02/01 10:00 |  0.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |   0.150881 |  0.572 |    0.572 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.042165 | 0.042165 | 0.042377 | 0.042377 | 0.011771 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  8.390886 | 0.011771 |
    | 02/01 11:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |   0.150881 |  0.143 |    0.143 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041954 | 0.041954 | 0.042165 | 0.042165 | 0.011713 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  8.348932 | 0.011713 |
    | 02/01 12:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  57.062595   85.593892 |  0.0 |   0.150881 |  5.148 | 1.411787 |     3.148 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041745 | 0.041745 | 0.041955 | 0.041955 | 0.011654 | 0.588213 |  57.062595   85.593892 |       0.0        0.0 |  77.686919 |       0.0 |  8.307187 | 0.011654 |
    | 02/01 13:00 |  5.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  62.682765   94.024148 |  0.0 |   0.150881 |  8.437 |      2.0 |  7.025213 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041536 | 0.041536 | 0.041745 | 0.041745 | 0.011596 |      0.0 |  62.682765   94.024148 |       0.0        0.0 |  77.686919 |       0.0 |  8.265651 | 0.011596 |
    | 02/01 14:00 |  1.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  62.682765   94.024148 |  0.0 |   0.150881 |  1.573 |    1.573 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041328 | 0.041328 | 0.041536 | 0.041536 | 0.011538 |      0.0 |  62.682765   94.024148 |       0.0        0.0 |  77.686919 |       0.0 |  8.224323 | 0.011538 |
    | 02/01 15:00 | 20.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  84.763565  127.145348 |  0.0 |   0.150881 | 29.601 | 0.076556 |    27.601 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041122 | 0.041122 | 0.041328 | 0.041328 |  0.01148 | 1.923444 |  84.763565  127.145348 |       0.0        0.0 |  77.686919 |       0.0 |  8.183201 |  0.01148 |
    | 02/01 16:00 | 37.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  128.05992   192.08988 |  0.0 |   0.150881 | 54.197 | 0.006942 | 54.120444 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040916 | 0.040916 | 0.041122 | 0.041122 | 0.011423 | 1.993058 |  128.05992   192.08988 |       0.0        0.0 |  77.686919 |       0.0 |  8.142285 | 0.011423 |
    | 02/01 17:00 |  8.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 137.435167   206.15275 |  0.0 |   0.150881 | 11.726 | 0.623609 | 11.719058 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040711 | 0.040711 | 0.040916 | 0.040916 | 0.011366 | 1.376391 | 137.435167   206.15275 |       0.0        0.0 |  77.686919 |       0.0 |  8.101574 | 0.011366 |
    | 02/01 18:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  141.05468   211.58202 |  0.0 |   0.150881 |  5.148 | 0.665759 |  4.524391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040508 | 0.040508 | 0.040712 | 0.040712 | 0.011309 | 1.334241 |  141.05468   211.58202 |       0.0        0.0 |  77.686919 |       0.0 |  8.061066 | 0.011309 |
    | 02/01 19:00 |  7.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 149.102073  223.653109 |  0.0 |   0.150881 | 10.725 | 0.069841 | 10.059241 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040305 | 0.040305 | 0.040508 | 0.040508 | 0.011252 | 1.930159 | 149.102073  223.653109 |       0.0        0.0 |  77.686919 |       0.0 |  8.020761 | 0.011252 |
    | 02/01 20:00 | 18.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   170.2102    255.3153 |  0.0 |   0.150881 | 26.455 | 0.004601 | 26.385159 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040104 | 0.040104 | 0.040306 | 0.040306 | 0.011196 | 1.995399 |   170.2102    255.3153 |       0.0        0.0 |  77.686919 |       0.0 |  7.980657 | 0.011196 |
    | 02/01 21:00 | 15.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 187.824119  281.736179 |  0.0 |   0.150881 | 22.022 | 0.007155 | 22.017399 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039903 | 0.039903 | 0.040104 | 0.040104 |  0.01114 | 1.992845 | 187.824119  281.736179 |       0.0        0.0 |  77.686919 |       0.0 |  7.940753 |  0.01114 |
    | 02/01 22:00 |  6.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 195.025595  292.538393 |  0.0 |   0.150881 |  9.009 | 0.025946 |  9.001845 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039704 | 0.039704 | 0.039904 | 0.039904 | 0.011084 | 1.974054 | 195.025595  292.538393 |       0.0        0.0 |  77.686919 |       0.0 |   7.90105 | 0.011084 |
    | 02/01 23:00 |  1.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 197.178439  295.767658 |  0.0 |   0.150881 |  2.717 | 0.047842 |  2.691054 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039505 | 0.039505 | 0.039704 | 0.039704 | 0.011029 | 1.952158 | 197.178439  295.767658 |       0.0        0.0 |  77.686919 |       0.0 |  7.861544 | 0.011029 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 201.883365  302.825047 |  0.0 |   0.150881 |  5.929 | 0.034767 |  5.881158 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039308 | 0.039308 | 0.039505 | 0.039505 | 0.010974 | 1.965233 | 188.332116  286.921335 | 13.551249  15.903712 |  77.686919 |       0.0 |  7.822237 | 0.010974 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 204.469152  306.703727 |  0.0 |   0.150881 |  3.267 | 0.045282 |  3.232233 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039111 | 0.039111 | 0.039308 | 0.039308 | 0.010919 | 1.954718 | 179.485793  278.075012 | 24.983359  28.628715 |  77.686919 |       0.0 |  7.783126 | 0.010919 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 204.916926  307.375389 |  0.0 |   0.150881 |  0.605 | 0.058514 |  0.559718 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.038916 | 0.038916 | 0.039111 | 0.039111 | 0.010864 | 1.941486 |  170.63947  269.228689 | 34.277456    38.1467 |  77.686919 |       0.0 |   7.74421 | 0.010864 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 205.063715  307.595572 |  0.0 |   0.150881 |  0.242 | 0.060918 |  0.183486 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.038721 | 0.038721 | 0.038916 | 0.038916 |  0.01081 | 1.939082 | 161.793147  260.382367 | 43.270568  47.213206 |  77.686919 |       0.0 |  7.705489 |  0.01081 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 205.498981  308.248471 |  0.0 |   0.150881 |  0.605 | 0.059438 |  0.544082 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.038527 | 0.038527 | 0.038721 | 0.038721 | 0.010756 | 1.940562 | 152.946824  251.536044 | 52.552156  56.712427 |  77.686919 |       0.0 |  7.666961 | 0.010756 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 201.740702  311.661946 |  0.0 |   0.161237 |  2.904 | 0.047956 |  2.844562 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  3.016964 | 0.455204 | 0.0 |      0.0 | 0.059876 |  0.395328 | 0.080618 | 0.001878 | 0.0 | 0.038738 | 0.040615 | 0.039034 | 0.039034 | 0.010843 | 1.952044 | 144.100502  242.689721 | 57.640201  68.972225 |  80.308555 |  0.312832 |  7.708842 | 0.010843 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  189.35585  312.185199 |  0.0 |   0.183638 |  0.484 | 0.103281 |  0.436044 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.366844 | 1.026568 | 0.0 |      0.0 | 0.057176 |  0.969393 | 0.091819 | 0.082565 | 0.0 | 0.039003 | 0.121568 | 0.058141 | 0.058141 |  0.01615 | 1.896719 | 135.254179  233.843398 | 54.101671  78.341801 |  85.706006 |  1.107841 |  7.761658 |  0.01615 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 176.970998  312.351661 |  0.0 |   0.206394 |  0.242 | 0.321386 |  0.138719 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.247914 | 1.147354 | 0.0 |      0.0 | 0.054597 |  1.092758 | 0.103197 | 0.370932 | 0.0 | 0.039324 | 0.410256 | 0.167732 | 0.167732 | 0.046592 | 1.678614 | 126.407856  224.997075 | 50.563142  87.354586 |  90.861162 |   1.72647 |   7.82553 | 0.046592 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 164.586146  302.611053 |  0.0 |   0.248456 |    0.0 | 0.821608 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  11.06273 | 2.283279 | 0.0 |      0.0 |  0.05018 |  2.233099 | 0.124228 |  0.97133 | 0.0 | 0.039749 | 1.011078 | 0.479619 | 0.479619 | 0.133228 | 0.857006 | 117.561533  216.150752 | 47.024613  86.460301 |  99.690793 |  2.864011 |   7.91001 | 0.133228 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 152.201294  290.226201 |  0.0 |   0.297265 |    0.0 | 0.857006 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.077095 | 0.0 |      0.0 | 0.045501 |  3.031594 | 0.148633 | 1.774856 | 0.0 | 0.040293 | 1.815149 | 1.056245 | 1.056245 | 0.293401 |      0.0 |  108.71521   207.30443 | 43.486084  82.921772 | 109.044051 |  3.972117 |  8.018349 | 0.293401 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 139.816442  277.841349 |  0.0 |   0.346853 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.681585 | 0.0 |      0.0 | 0.041126 |  3.640458 | 0.173426 | 2.433676 | 0.0 | 0.040959 | 2.474635 | 1.783019 | 1.783019 | 0.495283 |      0.0 |  99.868887  198.458107 | 39.947555  79.383243 | 117.788445 |  5.005473 |  8.150816 | 0.495283 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  127.43159  265.456497 |  0.0 |   0.401086 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.530852 | 4.693216 | 0.0 |      0.0 | 0.036687 |  4.656529 | 0.200543 | 3.155689 | 0.0 | 0.041757 | 3.197446 | 2.488707 | 2.488707 | 0.691308 |      0.0 |  91.022565  189.611784 | 36.409026  75.844714 | 126.662768 |  6.305769 |  8.309603 | 0.691308 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 115.046738  253.071645 |  0.0 |   0.449659 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 4.967396 | 0.0 |      0.0 |  0.03296 |  4.934436 | 0.224829 | 3.730766 | 0.0 | 0.042672 | 3.773438 | 3.164819 | 3.164819 | 0.879117 |      0.0 |  82.176242  180.765461 | 32.870497  72.306184 | 134.113184 |   7.28461 |   8.49176 | 0.879117 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 102.661886  240.686793 |  0.0 |   0.496733 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 5.568956 | 0.0 |      0.0 | 0.029535 |   5.53942 | 0.248367 | 4.209921 | 0.0 | 0.043701 | 4.253622 | 3.752147 | 3.752147 | 1.042263 |      0.0 |  73.329919  171.919138 | 29.331968  68.767655 | 140.958615 |  8.365742 |  8.696426 | 1.042263 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  90.277035  228.301941 |  0.0 |   0.541828 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 6.151968 | 0.0 |      0.0 | 0.026404 |  6.125564 | 0.270914 | 4.759843 | 0.0 | 0.044837 |  4.80468 | 4.269372 | 4.269372 | 1.185937 |      0.0 |  64.483596  163.072815 | 25.793438  65.229126 | 147.217903 |  9.460549 |  8.922503 | 1.185937 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  77.892183   215.91709 |  0.0 |   0.584581 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 6.710457 | 0.0 |      0.0 | 0.023554 |  6.686903 | 0.292291 | 5.316337 | 0.0 | 0.046074 | 5.362411 | 4.806163 | 4.806163 | 1.335045 |      0.0 |  55.637273  154.226493 | 22.254909  61.690597 | 152.915852 | 10.538825 |   9.16872 | 1.335045 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  65.507331  203.532238 |  0.0 |   0.624746 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.239955 | 0.0 |      0.0 |  0.02097 |  7.218985 | 0.312373 |   5.8613 | 0.0 | 0.047405 | 5.908705 | 5.359869 | 5.359869 | 1.488853 |      0.0 |   46.79095   145.38017 |  18.71638  58.152068 | 158.081719 | 11.584137 |  9.433687 | 1.488853 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  53.122479  191.147386 |  0.0 |   0.662171 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.737384 | 0.0 |      0.0 | 0.018635 |  7.718748 | 0.331086 | 6.386677 | 0.0 | 0.048824 | 6.435501 | 5.904372 | 5.904372 | 1.640103 |      0.0 |  37.944628  136.533847 | 15.177851  54.613539 | 162.747823 | 12.585123 |  9.715949 | 1.640103 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  40.737627  178.762534 |  0.0 |   0.696793 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.200894 | 0.0 |      0.0 | 0.016534 |   8.18436 | 0.348397 | 6.887468 | 0.0 | 0.050322 |  6.93779 | 6.430055 | 6.430055 | 1.786126 |      0.0 |  29.098305  127.687524 | 11.639322   51.07501 | 166.948315 | 13.533618 | 10.014024 | 1.786126 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  28.352775  166.377682 |  0.0 |   0.728617 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.629684 | 0.0 |      0.0 | 0.014648 |  8.615036 | 0.364309 |  7.36026 | 0.0 | 0.051892 | 7.412152 | 6.931584 | 6.931584 |  1.92544 |      0.0 |  20.251982  118.841201 |  8.100793   47.53648 | 170.718131 | 14.424086 | 10.326441 |  1.92544 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  15.967923   153.99283 |  0.0 |   0.757702 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 9.023814 | 0.0 |      0.0 |  0.01296 |  9.010853 | 0.378851 | 7.802841 | 0.0 | 0.053526 | 7.856368 | 7.405453 | 7.405453 |  2.05707 |      0.0 |  11.405659  109.994878 |  4.562264  43.997951 |  174.09213 | 15.253247 | 10.651765 |  2.05707 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   3.583071  141.607978 |  0.0 |   0.784149 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 9.384024 | 0.0 |      0.0 | 0.011454 |   9.37257 | 0.392075 | 8.214009 | 0.0 | 0.055219 | 8.269228 |   7.8494 |   7.8494 | 2.180389 |      0.0 |   2.559336  101.148556 |  1.023735  40.459422 | 177.104411 | 16.019734 | 10.988621 | 2.180389 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  129.223126 |  0.0 |   0.799315 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 2.559336  8.846323 | 0.0   0.0 |  7.983961 | 6.260618 | 0.0 | 0.029511 | 0.010586 |  6.250032 | 0.399657 | 7.072846 | 0.0 | 0.056941 | 7.129787 | 7.924272 | 7.924272 | 2.201187 |      0.0 |        0.0   92.302233 |       0.0  36.920893 |  178.80883 | 14.797262 | 11.331337 | 2.201187 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  116.838274 |  0.0 |   0.810292 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.846323 | 0.0   0.0 |  6.192426 | 4.949699 | 0.0 | 0.029125 | 0.009974 |  4.939724 | 0.405146 | 5.483211 | 0.0 | 0.058682 | 5.541894 | 7.030131 | 7.030131 | 1.952814 |      0.0 |        0.0    83.45591 |       0.0  33.382364 | 180.032406 |  13.84863 |   11.6778 | 1.952814 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  104.447803 |  0.0 |    0.82073 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.019948 | 0.0 | 0.028801 | 0.009396 |  5.010552 | 0.410365 | 4.757431 | 0.0 | 0.060441 | 4.817872 | 5.733865 | 5.733865 |  1.59274 |      0.0 |        0.0   74.605574 |       0.0  29.842229 | 181.188289 | 13.691385 | 12.027725 |  1.59274 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   92.057332 |  0.0 |   0.830633 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.084615 | 0.0 | 0.029575 | 0.008851 |  5.075764 | 0.415317 | 4.573807 | 0.0 | 0.062215 | 4.636022 | 4.938355 | 4.938355 | 1.371765 |      0.0 |        0.0   65.755237 |       0.0  26.302095 | 182.278185 | 13.778026 | 12.380826 | 1.371765 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   79.666862 |  0.0 |   0.840028 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 |  5.14597 | 0.0 | 0.029702 | 0.008336 |  5.137633 | 0.420014 | 4.551347 | 0.0 | 0.064004 | 4.615351 |  4.67184 |  4.67184 | 1.297733 |      0.0 |        0.0   56.904901 |       0.0   22.76196 | 183.306085 | 13.944299 | 12.736836 | 1.297733 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   67.276391 |  0.0 |   0.849093 |  1.573 | 0.049993 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.204171 | 0.0 | 0.012498 | 0.007851 |   5.19632 | 0.424547 | 4.581095 | 0.0 | 0.065807 | 4.646902 | 4.626956 | 4.626956 | 1.285265 | 1.523007 |        0.0   48.054565 |       0.0  19.221826 | 184.292502 | 14.134977 | 13.095576 | 1.285265 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    54.88592 |  0.0 |   0.857663 |    0.0 |  0.05838 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.260332 | 0.0 | 0.014595 | 0.007386 |  5.252946 | 0.428832 | 4.630187 | 0.0 | 0.067622 | 4.697809 | 4.651203 | 4.651203 | 1.292001 | 1.464627 |        0.0   39.204229 |       0.0  15.681691 | 185.220197 | 14.328905 | 13.456785 | 1.292001 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   42.495449 |  0.0 |   0.865775 |    0.0 | 0.059405 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.313424 | 0.0 | 0.014851 | 0.006949 |  5.306475 | 0.432888 | 4.684616 | 0.0 | 0.069448 | 4.754064 | 4.698997 | 4.698997 | 1.305277 | 1.405222 |        0.0   30.353892 |       0.0  12.141557 | 186.094105 | 14.517877 | 13.820224 | 1.305277 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   30.104979 |  0.0 |   0.873084 |    0.0 |  0.21716 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.363682 | 0.0 |  0.05429 | 0.006537 |  5.357145 | 0.436542 | 4.743095 | 0.0 | 0.071284 | 4.814379 | 4.754966 | 4.754966 | 1.320824 | 1.188062 |        0.0   21.503556 |       0.0   8.601422 | 186.877906 | 14.695385 | 14.185482 | 1.320824 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   17.714508 |  0.0 |   0.878447 |  0.847 |  0.89524 |  0.035062 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.230298 | 5.439572 | 0.0 |  0.22381 | 0.006166 |  5.433406 | 0.439223 | 4.824331 | 0.0 | 0.073124 | 4.897455 | 4.819437 | 4.819437 | 1.338732 |  1.10476 |        0.0    12.65322 |       0.0   5.061288 | 187.450987 | 14.865237 | 14.551582 | 1.338732 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    5.324037 |  0.0 |   0.879295 |  0.484 |  1.58876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.442185 | 0.0 | 0.668518 | 0.005898 |  5.436287 | 0.439647 | 4.922706 | 0.0 | 0.074956 | 4.997662 | 4.901261 | 4.901261 | 1.361462 |      0.0 |        0.0    3.802884 |       0.0   1.521153 | 187.541418 | 14.939171 | 14.916273 | 1.361462 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.84952 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  3.802884 | 0.0   0.0 |  2.662019 | 2.340698 | 0.0 | 3.529947 | 0.006069 |   2.33463 |  0.42476 | 4.185391 | 0.0 | 0.076705 | 4.262097 | 4.811935 | 4.811935 | 1.336648 |      0.0 |        0.0         0.0 |       0.0        0.0 | 184.338859 | 12.663649 | 15.264328 | 1.336648 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.808352 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.529885 | 0.007831 | -0.007831 | 0.404176 | 2.572771 | 0.0 | 0.078343 | 2.651113 | 4.067559 | 4.067559 | 1.129878 |      0.0 |        0.0         0.0 |       0.0        0.0 | 179.816804 |  9.678872 | 15.590162 | 1.129878 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.75649 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 5.874002 | 0.010092 | -0.010092 | 0.378245 | 1.513533 | 0.0 | 0.079842 | 1.593375 | 2.774057 | 2.774057 | 0.770571 |      0.0 |        0.0         0.0 |       0.0        0.0 | 173.952894 |  7.777002 | 15.888565 | 0.770571 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.706373 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |   5.8739 | 0.013024 | -0.013024 | 0.353187 | 1.002454 | 0.0 | 0.081209 | 1.083663 | 1.715159 | 1.715159 | 0.476433 |      0.0 |        0.0         0.0 |       0.0        0.0 | 168.092017 |  6.408338 | 16.160543 | 0.476433 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.644868 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 7.500667 | 0.015954 | -0.015954 | 0.322434 | 0.746585 | 0.0 | 0.082415 |    0.829 |  1.14034 |  1.14034 | 0.316761 |      0.0 |        0.0         0.0 |       0.0        0.0 | 160.607304 |  5.323365 | 16.400562 | 0.316761 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.593351 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.568376 | 0.019696 | -0.019696 | 0.296676 | 0.555276 | 0.0 | 0.083486 | 0.638762 | 0.843317 | 0.843317 | 0.234255 |      0.0 |        0.0         0.0 |       0.0        0.0 | 154.058625 |  4.451718 | 16.613751 | 0.234255 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.553401 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 5.299772 | 0.022971 | -0.022971 |   0.2767 |  0.40454 | 0.0 | 0.084452 | 0.488993 | 0.647755 | 0.647755 | 0.179932 |      0.0 |        0.0         0.0 |       0.0        0.0 | 148.781824 |  3.747506 |    16.806 | 0.179932 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.523503 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.100417 | 0.025609 | -0.025609 | 0.261752 | 0.287715 | 0.0 | 0.085339 | 0.373054 |  0.49651 |  0.49651 |  0.13792 |      0.0 |        0.0         0.0 |       0.0        0.0 | 144.707016 |  3.172431 | 16.982412 |  0.13792 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.510279 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.866979 | 0.027646 | -0.027646 |  0.25514 |  0.19013 | 0.0 | 0.086188 | 0.276318 | 0.377321 | 0.377321 | 0.104811 |      0.0 |        0.0         0.0 |       0.0        0.0 | 142.867684 |  2.699514 | 17.151364 | 0.104811 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.504652 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.818494 | 0.028566 | -0.028566 | 0.252326 | 0.121304 | 0.0 | 0.087018 | 0.208322 | 0.282705 | 0.282705 | 0.078529 |      0.0 |        0.0         0.0 |       0.0        0.0 | 142.077756 |  2.297318 | 17.316672 | 0.078529 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.503403 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.204887 | 0.028961 | -0.028961 | 0.251702 | 0.074735 | 0.0 | 0.087842 | 0.162577 | 0.213267 | 0.213267 | 0.059241 |      0.0 |        0.0         0.0 |       0.0        0.0 |  141.90183 |   1.94192 | 17.480532 | 0.059241 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.503215 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  0.05558 | 0.029049 | -0.029049 | 0.251608 | 0.044346 | 0.0 | 0.088661 | 0.133006 | 0.166171 | 0.166171 | 0.046159 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.875299 |  1.616918 | 17.643478 | 0.046159 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.503043 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.053249 | 0.029062 | -0.029062 | 0.251522 | 0.024799 | 0.0 | 0.089475 | 0.114274 | 0.135415 | 0.135415 | 0.037615 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.851112 |  1.311535 | 17.805525 | 0.037615 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.502878 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.052451 | 0.029074 | -0.029074 | 0.251439 | 0.012562 | 0.0 | 0.090285 | 0.102847 | 0.115897 | 0.115897 | 0.032194 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.827735 |   1.01846 | 17.966679 | 0.032194 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.50271 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.052669 | 0.029086 | -0.029086 | 0.251355 | 0.005391 | 0.0 |  0.09109 | 0.096481 | 0.103972 | 0.103972 | 0.028881 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.804153 |  0.732627 | 18.126944 | 0.028881 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96_glacier:

glacier
_______

This example demonstrates the functionality of zones of type |GLACIER|.  |GLACIER|
zones are similar to zones of type |FIELD| or |FOREST| but possess neither an
interception nor a soil module.  Instead, they pass precipitation and meltwater to the
upper zone storage (|UZ|) directly. The snow routines of |GLACIER|, |FIELD|, and
|FOREST| zones are identical.  Additional glacier melt can only occur if no snow covers
the glacier.  In the next test run, we can distinguish the simulation period into three
subperiods.  On the first two days (-20°C), the snow layer builds up.  On the third day
(+20°C and |SP| > 0), the snow melts and (with some time delay) releases meltwater. On
the fourth day (+20°C and |SP| = 0), an increased amount of water is passed to |UZ| due
to |GMelt| being larger than |CFMax|:

.. integration-test::

    >>> zonetype(GLACIER)
    >>> test("hland_96_glacier")
    |        date |    p |     t |    tc | fracrain | rfc | sfc |    cfact |                    swe |     gact | contriarea |     pc |  ei |     tf | spl | wcl | spg | wcg |    glmelt |               melt |      refr |       in_ |         r |  sr |  ea |  cf |      inuz | perc |        q0 |  el |       q1 |      inrc |     outrc |        rt |       qt |  ic |                     sp |                   wc |  sm |        uz |        lz |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.061111 |  0.061111 | 0.016975 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.088833 |  0.088833 | 0.024676 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |   0.04975 |   0.04975 |  0.01382 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049502 |  0.049502 |  0.01375 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049254 |  0.049254 | 0.013682 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049008 |  0.049008 | 0.013613 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048763 |  0.048763 | 0.013545 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048519 |  0.048519 | 0.013477 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048276 |  0.048276 |  0.01341 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048035 |  0.048035 | 0.013343 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047795 |  0.047795 | 0.013276 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047556 |  0.047556 |  0.01321 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047318 |  0.047318 | 0.013144 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047081 |  0.047081 | 0.013078 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046846 |  0.046846 | 0.013013 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     0.2288      0.3432 | 0.901953 |        1.0 |  0.286 | 0.0 |  0.286 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046612 |  0.046612 | 0.012948 | 0.0 |     0.2288      0.3432 |       0.0        0.0 | 0.0 |       0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     0.2288      0.3432 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046379 |  0.046379 | 0.012883 | 0.0 |     0.2288      0.3432 |       0.0        0.0 | 0.0 |       0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     0.2288      0.3432 | 0.901953 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046147 |  0.046147 | 0.012819 | 0.0 |     0.2288      0.3432 |       0.0        0.0 | 0.0 |       0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |      1.716       2.574 | 0.901953 |        1.0 |  1.859 | 0.0 |  1.859 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045916 |  0.045916 | 0.012754 | 0.0 |      1.716       2.574 |       0.0        0.0 | 0.0 |       0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     8.1224     12.1836 | 0.901953 |        1.0 |  8.008 | 0.0 |  8.008 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045458 |  0.045458 |  0.045686 |  0.045686 | 0.012691 | 0.0 |     8.1224     12.1836 |       0.0        0.0 | 0.0 |       0.0 |  9.046105 | 0.012691 |
    | 01/01 20:00 |  2.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |      11.44       17.16 | 0.901953 |        1.0 |  4.147 | 0.0 |  4.147 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045231 |  0.045231 |  0.045458 |  0.045458 | 0.012627 | 0.0 |      11.44       17.16 |       0.0        0.0 | 0.0 |       0.0 |  9.000874 | 0.012627 |
    | 01/01 21:00 |  4.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    17.0456     25.5684 | 0.901953 |        1.0 |  7.007 | 0.0 |  7.007 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045004 |  0.045004 |  0.045231 |  0.045231 | 0.012564 | 0.0 |    17.0456     25.5684 |       0.0        0.0 | 0.0 |       0.0 |   8.95587 | 0.012564 |
    | 01/01 22:00 | 10.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     29.172      43.758 | 0.901953 |        1.0 | 15.158 | 0.0 | 15.158 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044779 |  0.044779 |  0.045005 |  0.045005 | 0.012501 | 0.0 |     29.172      43.758 |       0.0        0.0 | 0.0 |       0.0 |  8.911091 | 0.012501 |
    | 01/01 23:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    29.2864     43.9296 | 0.901953 |        1.0 |  0.143 | 0.0 |  0.143 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044555 |  0.044555 |   0.04478 |   0.04478 | 0.012439 | 0.0 |    29.2864     43.9296 |       0.0        0.0 | 0.0 |       0.0 |  8.866535 | 0.012439 |
    | 02/01 00:00 |  0.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    30.0872     45.1308 | 0.902305 |        1.0 |  1.001 | 0.0 |  1.001 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044333 |  0.044333 |  0.044556 |  0.044556 | 0.012377 | 0.0 |    30.0872     45.1308 |       0.0        0.0 | 0.0 |       0.0 |  8.822202 | 0.012377 |
    | 02/01 01:00 |  3.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    33.5192     50.2788 | 0.902305 |        1.0 |   4.29 | 0.0 |   4.29 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044111 |  0.044111 |  0.044333 |  0.044333 | 0.012315 | 0.0 |    33.5192     50.2788 |       0.0        0.0 | 0.0 |       0.0 |  8.778091 | 0.012315 |
    | 02/01 02:00 |  2.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    35.9216     53.8824 | 0.902305 |        1.0 |  3.003 | 0.0 |  3.003 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |  0.04389 |   0.04389 |  0.044111 |  0.044111 | 0.012253 | 0.0 |    35.9216     53.8824 |       0.0        0.0 | 0.0 |       0.0 |  8.734201 | 0.012253 |
    | 02/01 03:00 | 10.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    47.8192     71.7288 | 0.902305 |        1.0 | 14.872 | 0.0 | 14.872 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043671 |  0.043671 |  0.043891 |  0.043891 | 0.012192 | 0.0 |    47.8192     71.7288 |       0.0        0.0 | 0.0 |       0.0 |   8.69053 | 0.012192 |
    | 02/01 04:00 |  3.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    51.8232     77.7348 | 0.902305 |        1.0 |  5.005 | 0.0 |  5.005 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043453 |  0.043453 |  0.043671 |  0.043671 | 0.012131 | 0.0 |    51.8232     77.7348 |       0.0        0.0 | 0.0 |       0.0 |  8.647077 | 0.012131 |
    | 02/01 05:00 |  3.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    55.7128     83.5692 | 0.902305 |        1.0 |  4.862 | 0.0 |  4.862 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043235 |  0.043235 |  0.043453 |  0.043453 |  0.01207 | 0.0 |    55.7128     83.5692 |       0.0        0.0 | 0.0 |       0.0 |  8.603842 |  0.01207 |
    | 02/01 06:00 |  1.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    57.0856     85.6284 | 0.902305 |        1.0 |  1.716 | 0.0 |  1.716 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043019 |  0.043019 |  0.043236 |  0.043236 |  0.01201 | 0.0 |    57.0856     85.6284 |       0.0        0.0 | 0.0 |       0.0 |  8.560823 |  0.01201 |
    | 02/01 07:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |       57.2        85.8 | 0.902305 |        1.0 |  0.143 | 0.0 |  0.143 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.042804 |  0.042804 |  0.043019 |  0.043019 |  0.01195 | 0.0 |       57.2        85.8 |       0.0        0.0 | 0.0 |       0.0 |  8.518019 |  0.01195 |
    | 02/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |       57.2        85.8 | 0.902305 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |  0.04259 |   0.04259 |  0.042804 |  0.042804 |  0.01189 | 0.0 |       57.2        85.8 |       0.0        0.0 | 0.0 |       0.0 |  8.475429 |  0.01189 |
    | 02/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |       57.2        85.8 | 0.902305 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.042377 |  0.042377 |   0.04259 |   0.04259 | 0.011831 | 0.0 |       57.2        85.8 |       0.0        0.0 | 0.0 |       0.0 |  8.433051 | 0.011831 |
    | 02/01 10:00 |  0.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    57.6576     86.4864 | 0.902305 |        1.0 |  0.572 | 0.0 |  0.572 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.042165 |  0.042165 |  0.042377 |  0.042377 | 0.011771 | 0.0 |    57.6576     86.4864 |       0.0        0.0 | 0.0 |       0.0 |  8.390886 | 0.011771 |
    | 02/01 11:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |     57.772      86.658 | 0.902305 |        1.0 |  0.143 | 0.0 |  0.143 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041954 |  0.041954 |  0.042165 |  0.042165 | 0.011713 | 0.0 |     57.772      86.658 |       0.0        0.0 | 0.0 |       0.0 |  8.348932 | 0.011713 |
    | 02/01 12:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    61.8904     92.8356 | 0.902305 |        1.0 |  5.148 | 0.0 |  5.148 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041745 |  0.041745 |  0.041955 |  0.041955 | 0.011654 | 0.0 |    61.8904     92.8356 |       0.0        0.0 | 0.0 |       0.0 |  8.307187 | 0.011654 |
    | 02/01 13:00 |  5.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      68.64      102.96 | 0.902305 |        1.0 |  8.437 | 0.0 |  8.437 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041536 |  0.041536 |  0.041745 |  0.041745 | 0.011596 | 0.0 |      68.64      102.96 |       0.0        0.0 | 0.0 |       0.0 |  8.265651 | 0.011596 |
    | 02/01 14:00 |  1.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    69.8984    104.8476 | 0.902305 |        1.0 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041328 |  0.041328 |  0.041536 |  0.041536 | 0.011538 | 0.0 |    69.8984    104.8476 |       0.0        0.0 | 0.0 |       0.0 |  8.224323 | 0.011538 |
    | 02/01 15:00 | 20.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    93.5792    140.3688 | 0.902305 |        1.0 | 29.601 | 0.0 | 29.601 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041122 |  0.041122 |  0.041328 |  0.041328 |  0.01148 | 0.0 |    93.5792    140.3688 |       0.0        0.0 | 0.0 |       0.0 |  8.183201 |  0.01148 |
    | 02/01 16:00 | 37.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   136.9368    205.4052 | 0.902305 |        1.0 | 54.197 | 0.0 | 54.197 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040916 |  0.040916 |  0.041122 |  0.041122 | 0.011423 | 0.0 |   136.9368    205.4052 |       0.0        0.0 | 0.0 |       0.0 |  8.142285 | 0.011423 |
    | 02/01 17:00 |  8.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   146.3176    219.4764 | 0.902305 |        1.0 | 11.726 | 0.0 | 11.726 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040711 |  0.040711 |  0.040916 |  0.040916 | 0.011366 | 0.0 |   146.3176    219.4764 |       0.0        0.0 | 0.0 |       0.0 |  8.101574 | 0.011366 |
    | 02/01 18:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    150.436     225.654 | 0.902305 |        1.0 |  5.148 | 0.0 |  5.148 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040508 |  0.040508 |  0.040712 |  0.040712 | 0.011309 | 0.0 |    150.436     225.654 |       0.0        0.0 | 0.0 |       0.0 |  8.061066 | 0.011309 |
    | 02/01 19:00 |  7.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    159.016     238.524 | 0.902305 |        1.0 | 10.725 | 0.0 | 10.725 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040305 |  0.040305 |  0.040508 |  0.040508 | 0.011252 | 0.0 |    159.016     238.524 |       0.0        0.0 | 0.0 |       0.0 |  8.020761 | 0.011252 |
    | 02/01 20:00 | 18.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |     180.18      270.27 | 0.902305 |        1.0 | 26.455 | 0.0 | 26.455 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040104 |  0.040104 |  0.040306 |  0.040306 | 0.011196 | 0.0 |     180.18      270.27 |       0.0        0.0 | 0.0 |       0.0 |  7.980657 | 0.011196 |
    | 02/01 21:00 | 15.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   197.7976    296.6964 | 0.902305 |        1.0 | 22.022 | 0.0 | 22.022 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039903 |  0.039903 |  0.040104 |  0.040104 |  0.01114 | 0.0 |   197.7976    296.6964 |       0.0        0.0 | 0.0 |       0.0 |  7.940753 |  0.01114 |
    | 02/01 22:00 |  6.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   205.0048    307.5072 | 0.902305 |        1.0 |  9.009 | 0.0 |  9.009 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039704 |  0.039704 |  0.039904 |  0.039904 | 0.011084 | 0.0 |   205.0048    307.5072 |       0.0        0.0 | 0.0 |       0.0 |   7.90105 | 0.011084 |
    | 02/01 23:00 |  1.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   207.1784    310.7676 | 0.902305 |        1.0 |  2.717 | 0.0 |  2.717 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039505 |  0.039505 |  0.039704 |  0.039704 | 0.011029 | 0.0 |   207.1784    310.7676 |       0.0        0.0 | 0.0 |       0.0 |  7.861544 | 0.011029 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   211.9216    317.8824 | 0.902686 |        1.0 |  5.929 | 0.0 |  5.929 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039308 |  0.039308 |  0.039505 |  0.039505 | 0.010974 | 0.0 | 198.332077  301.921277 | 13.589523  15.961123 | 0.0 |       0.0 |  7.822237 | 0.010974 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   214.5352    321.8028 | 0.902686 |        1.0 |  3.267 | 0.0 |  3.267 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039111 |  0.039111 |  0.039308 |  0.039308 | 0.010919 | 0.0 | 189.485754  293.074954 | 25.049446  28.727846 | 0.0 |       0.0 |  7.783126 | 0.010919 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   215.0192    322.5288 | 0.902686 |        1.0 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.038916 |  0.038916 |  0.039111 |  0.039111 | 0.010864 | 0.0 | 180.639432  284.228632 | 34.379768  38.300168 | 0.0 |       0.0 |   7.74421 | 0.010864 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   215.2128    322.8192 | 0.902686 |        1.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.038721 |  0.038721 |  0.038916 |  0.038916 |  0.01081 | 0.0 | 171.793109  275.382309 | 43.419691  47.436891 | 0.0 |       0.0 |  7.705489 |  0.01081 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   215.6968    323.5452 | 0.902686 |        1.0 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.038527 |  0.038527 |  0.038721 |  0.038721 | 0.010756 | 0.0 | 162.946786  266.535986 | 52.750014  57.009214 | 0.0 |       0.0 |  7.666961 | 0.010756 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 215.740648      327.03 | 0.902686 |        1.0 |  2.904 | 0.0 |  2.904 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  1.139676 |  1.139676 | 0.0 | 0.0 | 0.0 |  1.139676 |  0.5 |  0.000067 | 0.0 | 0.040835 |  0.040902 |  0.039098 |  0.039098 | 0.010861 | 0.0 | 154.100463  257.689663 | 61.640185  69.340337 | 0.0 |  0.639609 |  8.126127 | 0.010861 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 203.355796    327.6108 | 0.902686 |        1.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.386026 |  6.386026 | 0.0 | 0.0 | 0.0 |  6.386026 |  0.5 |  0.076868 | 0.0 | 0.043131 |  0.119999 |  0.057951 |  0.057951 | 0.016098 | 0.0 |  145.25414   248.84334 | 58.101656   78.76746 | 0.0 |  6.448767 |  8.582996 | 0.016098 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 190.970944    327.9012 | 0.902686 |        1.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.289226 |  6.289226 | 0.0 | 0.0 | 0.0 |  6.289226 |  0.5 |  0.804771 | 0.0 | 0.045415 |  0.850186 |  0.264685 |  0.264685 | 0.073524 | 0.0 | 136.407817  239.997017 | 54.563127  87.904183 | 0.0 | 11.433222 |  9.037581 | 0.073524 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 178.586092  323.610972 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |   8.33754 |   8.33754 | 0.0 | 0.0 | 0.0 |   8.33754 |  0.5 |  2.938168 | 0.0 | 0.047688 |  2.985856 |  1.162515 |  1.162515 | 0.322921 | 0.0 | 127.561495  231.150695 | 51.024598  92.460278 | 0.0 | 16.332594 |  9.489893 | 0.322921 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  166.20124   311.22612 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 |  7.185763 | 0.0 | 0.049949 |  7.235713 |  3.455675 |  3.455675 |  0.95991 | 0.0 | 118.715172  222.304372 | 47.486069  88.921749 | 0.0 | 21.031683 |  9.939944 |  0.95991 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 153.816388  298.841268 | 0.902686 |        1.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.747852 | 12.747852 | 0.0 | 0.0 | 0.0 | 12.747852 |  0.5 | 10.758217 | 0.0 |   0.0522 | 10.810416 |  7.085679 |  7.085679 | 1.968244 | 0.0 | 109.868849  213.458049 |  43.94754   85.38322 | 0.0 | 22.521318 | 10.387744 | 1.968244 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 141.431536  286.456416 | 0.902686 |        1.0 |  3.146 | 0.0 |  3.146 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 15.530852 | 15.530852 | 0.0 | 0.0 | 0.0 | 15.530852 |  0.5 | 13.364347 | 0.0 | 0.054439 | 13.418786 | 10.595675 | 10.595675 | 2.943243 | 0.0 | 101.022526  204.611726 |  40.40901   81.84469 | 0.0 | 24.187823 | 10.833305 | 2.943243 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 129.046684  274.071564 | 0.902686 |        1.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.231852 | 13.231852 | 0.0 | 0.0 | 0.0 | 13.231852 |  0.5 | 13.517189 | 0.0 | 0.056667 | 13.573855 | 12.873608 | 12.873608 | 3.576002 | 0.0 |  92.176203  195.765403 | 36.870481  78.306161 | 0.0 | 23.402487 | 11.276639 | 3.576002 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 116.661833  261.686713 | 0.902686 |        1.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.747852 | 12.747852 | 0.0 | 0.0 | 0.0 | 12.747852 |  0.5 | 12.628482 | 0.0 | 0.058883 | 12.687365 | 13.342398 | 13.342398 | 3.706222 | 0.0 |   83.32988   186.91908 | 33.331952  74.767632 | 0.0 | 23.021857 | 11.717755 | 3.706222 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 104.276981  249.301861 | 0.902686 |        1.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.747852 | 12.747852 | 0.0 | 0.0 | 0.0 | 12.747852 |  0.5 | 12.322666 | 0.0 | 0.061089 | 12.383755 | 12.816894 | 12.816894 | 3.560248 | 0.0 |  74.483558  178.072758 | 29.793423  71.229103 | 0.0 | 22.947042 | 12.156667 | 3.560248 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  91.892129  236.917009 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 12.079612 | 0.0 | 0.063283 | 12.142895 |   12.3977 |   12.3977 | 3.443805 | 0.0 |  65.637235  169.226435 | 26.254894  67.690574 | 0.0 | 22.752282 | 12.593383 | 3.443805 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  79.507277  224.532157 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.924763 | 0.0 | 0.065467 |  11.99023 | 12.162494 | 12.162494 | 3.378471 | 0.0 |  56.790912  160.380112 | 22.716365  64.152045 | 0.0 | 22.712371 | 13.027916 | 3.378471 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  67.122425  212.147305 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.893099 | 0.0 |  0.06764 | 11.960738 | 12.017602 | 12.017602 | 3.338223 | 0.0 |  47.944589  151.533789 | 19.177836  60.613516 | 0.0 | 22.704125 | 13.460277 | 3.338223 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  54.737573  199.762453 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.886559 | 0.0 | 0.069801 |  11.95636 | 11.966319 | 11.966319 | 3.323978 | 0.0 |  39.098266  142.687466 | 15.639306  57.074986 | 0.0 | 22.702418 | 13.890476 | 3.323978 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  42.352721  187.377601 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.885205 | 0.0 | 0.071952 | 11.957158 |  11.95751 |  11.95751 | 3.321531 | 0.0 |  30.251943  133.841143 | 12.100777  53.536457 | 0.0 | 22.702064 | 14.318523 | 3.321531 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  29.967869  174.992749 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.884925 | 0.0 | 0.074093 | 11.959018 | 11.957394 | 11.957394 | 3.321498 | 0.0 |  21.405621  124.994821 |  8.562248  49.997928 | 0.0 | 22.701991 | 14.744431 | 3.321498 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  17.583017  162.607897 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.884867 | 0.0 | 0.076222 | 11.961089 | 11.959065 | 11.959065 | 3.321962 | 0.0 |  12.559298  116.148498 |  5.023719  46.459399 | 0.0 | 22.701976 | 15.168208 | 3.321962 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   5.198165  150.223045 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.884855 | 0.0 | 0.078341 | 11.963196 | 11.961097 | 11.961097 | 3.322527 | 0.0 |   3.712975  107.302175 |   1.48519   42.92087 | 0.0 | 22.701973 | 15.589867 | 3.322527 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  137.838193 | 0.902686 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.846323 | 3.712975  8.846323 | 0.0   0.0 | 17.637831 | 17.637831 | 0.0 | 0.0 | 0.0 | 17.637831 |  0.5 | 14.659008 | 0.0 | 0.080449 | 14.739458 | 12.579675 | 12.579675 | 3.494354 | 0.0 |        0.0   98.455852 |       0.0  39.382341 | 0.0 | 25.180796 | 16.009418 | 3.494354 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  125.447722 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 15.315208 | 0.0 | 0.082547 | 15.397755 | 14.268799 | 14.268799 | 3.963555 | 0.0 |        0.0   89.605516 |       0.0  35.842206 | 0.0 | 24.411159 | 16.426871 | 3.963555 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  113.057251 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.667735 | 0.0 | 0.084634 | 14.752369 | 15.108048 | 15.108048 |  4.19668 | 0.0 |        0.0    80.75518 |       0.0  32.302072 | 0.0 | 24.288996 | 16.842237 |  4.19668 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  100.666781 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 |  14.56557 | 0.0 | 0.086711 | 14.652281 | 14.873547 | 14.873547 | 4.131541 | 0.0 |        0.0   71.904843 |       0.0  28.761937 | 0.0 | 24.268997 | 17.255525 | 4.131541 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    88.27631 | 0.903096 |        1.0 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 16.618572 | 16.618572 | 0.0 | 0.0 | 0.0 | 16.618572 |  0.5 | 15.411941 | 0.0 | 0.088778 | 15.500719 | 14.863065 | 14.863065 | 4.128629 | 0.0 |        0.0   63.054507 |       0.0  25.221803 | 0.0 | 24.975628 | 17.666748 | 4.128629 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   75.885839 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 15.141972 | 0.0 | 0.090834 | 15.232806 | 15.252641 | 15.252641 | 4.236845 | 0.0 |        0.0   54.204171 |       0.0  21.681668 | 0.0 | 24.379227 | 18.075914 | 4.236845 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   63.495368 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.641014 | 0.0 |  0.09288 | 14.733894 | 15.181473 | 15.181473 | 4.217076 | 0.0 |        0.0   45.353835 |       0.0  18.141534 | 0.0 | 24.283785 | 18.483034 | 4.217076 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   51.104898 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.561216 | 0.0 | 0.094915 | 14.656131 | 14.827482 | 14.827482 | 4.118745 | 0.0 |        0.0   36.503498 |       0.0  14.601399 | 0.0 |  24.26814 | 18.888119 | 4.118745 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   38.714427 | 0.903096 |        1.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.892572 | 15.892572 | 0.0 | 0.0 | 0.0 | 15.892572 |  0.5 | 15.009934 | 0.0 | 0.096941 | 15.106874 | 14.773577 | 14.773577 | 4.103771 | 0.0 |        0.0   27.653162 |       0.0  11.061265 | 0.0 | 24.650778 | 19.291179 | 4.103771 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   26.323956 | 0.903096 |        1.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.529572 | 15.529572 | 0.0 | 0.0 | 0.0 | 15.529572 |  0.5 | 15.132993 | 0.0 | 0.098956 | 15.231948 | 15.034503 | 15.034503 | 4.176251 | 0.0 |        0.0   18.802826 |       0.0    7.52113 | 0.0 | 24.547358 | 19.692223 | 4.176251 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   13.933485 | 0.903096 |        1.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.166572 | 15.166572 | 0.0 | 0.0 | 0.0 | 15.166572 |  0.5 | 14.847629 | 0.0 | 0.100961 |  14.94859 | 15.141186 | 15.141186 | 4.205885 | 0.0 |        0.0     9.95249 |       0.0   3.980996 | 0.0 |   24.3663 | 20.091262 | 4.205885 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    1.543015 | 0.903096 |        1.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.529572 | 15.529572 | 0.0 | 0.0 | 0.0 | 15.529572 |  0.5 | 14.893578 | 0.0 | 0.102956 | 14.996534 | 15.022213 | 15.022213 | 4.172837 | 0.0 |        0.0    1.102153 |       0.0   0.440861 | 0.0 | 24.502294 | 20.488305 | 4.172837 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0  1.102153 | 0.0   0.0 |  18.47218 |  18.47218 | 0.0 | 0.0 | 0.0 |  18.47218 |  0.5 | 16.659345 | 0.0 | 0.104942 | 16.764286 | 15.378714 | 15.378714 | 4.271865 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.815129 | 20.883364 | 4.271865 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.351313 | 0.0 | 0.106917 |  17.45823 | 16.525662 | 16.525662 | 4.590462 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.664489 | 21.276447 | 4.590462 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.220536 | 0.0 | 0.108882 | 17.329418 | 17.275395 | 17.275395 | 4.798721 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.644625 | 21.667565 | 4.798721 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.203307 | 0.0 | 0.110838 | 17.314145 | 17.354649 | 17.354649 | 4.820736 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641991 | 22.056727 | 4.820736 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.201022 | 0.0 | 0.112784 | 17.313806 | 17.317464 | 17.317464 | 4.810407 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641641 | 22.443943 | 4.810407 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200719 | 0.0 |  0.11472 | 17.315439 | 17.314244 | 17.314244 | 4.809512 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641594 | 22.829224 | 4.809512 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200679 | 0.0 | 0.116646 | 17.317325 | 17.315495 | 17.315495 |  4.80986 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641588 | 23.212577 |  4.80986 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.118563 | 17.319236 |  17.31733 |  17.31733 |  4.81037 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 23.594015 |  4.81037 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 |  0.12047 | 17.321143 | 17.319235 | 17.319235 | 4.810899 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 23.973545 | 4.810899 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.122368 |  17.32304 | 17.321141 | 17.321141 | 4.811428 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 24.351177 | 4.811428 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.124256 | 17.324928 | 17.323038 | 17.323038 | 4.811955 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 24.726921 | 4.811955 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.126135 | 17.326807 | 17.324926 | 17.324926 |  4.81248 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 25.100786 |  4.81248 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.128004 | 17.328676 | 17.326805 | 17.326805 | 4.813001 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 25.472782 | 4.813001 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96_multiple_zones:

multiple zones
______________

All of the above examples deal with a subbasin consisting of a single zone.  Hence,
none of them is suitable to check if, for instance, the aggregation of fluxes
addressing different spatial extents does not introduce errors into the water balance.
The following example fills this gap.  We restore the initial
:ref:`field example <hland_96_field>` settings, except defining five zones of different
land-use types and sizes.  The variations between the responses of the land-use
types are as to be expected:

.. integration-test::

    >>> resparea(False)
    >>> name2value = {par.name: par.value for par in control}
    >>> nmbzones(5)
    >>> sclass(1)
    >>> area(15.0)
    >>> zonearea(5.0, 4.0, 3.0, 2.0, 1.0)
    >>> zonetype(FIELD, FOREST, GLACIER, ILAKE, SEALED)
    >>> sfdist(1.0)
    >>> for name, value in name2value.items():
    ...     if name not in ("nmbzones", "sclass", "area", "zonearea", "zonetype", "sfdist"):
    ...         control[name].value = value
    >>> model.add_aetmodel_v1.update(model, model.aetmodel, refresh=True)
    >>> model.aetmodel.add_petmodel_v1.update(model.aetmodel, model.aetmodel.petmodel, refresh=True)
    >>> aetcontrol = model.aetmodel.parameters.control
    >>> aetcontrol.temperaturethresholdice(0.0)
    >>> aetcontrol.soilmoisturelimit(0.8)
    >>> aetcontrol.excessreduction(0.5)
    >>> petcontrol = model.aetmodel.petmodel.parameters.control
    >>> petcontrol.evapotranspirationfactor(0.7)
    >>> petcontrol.airtemperaturefactor(0.1)
    >>> petcontrol.altitudefactor(-0.1)
    >>> petcontrol.precipitationfactor(0.1)
    >>> inputs.t.series = t_series
    >>> petinputs.normalairtemperature.series = tn_series
    >>> test("hland_96_multiple_zones")
    |        date |    p |    t |                           tc |                     fracrain |                     rfc |                     sfc |                                       cfact |                     swe |                          gact | contriarea |                                     pc |                                     ei |                                              tf |                     spl |                     wcl |                     spg |                     wcg |                           glmelt |                     melt |                     refr |                                                in_ |                                                  r |                            sr |                                ea |                                cf |      inuz | perc |        q0 |                           el |       q1 |      inrc |     outrc |        rt |        qt |                                     ic |                      sp |                      wc |                                    sm |        uz |        lz |    outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.760624     0.0        0.0 |       0.0        0.0  18.760624     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.048465  0.048465  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.690156 |  0.5 |  0.018623 | 0.0  0.0  0.0  0.077544  0.0 | 0.052087 |  0.063513 |  0.064114 |  0.064114 |  0.267142 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.951535   99.951535  0.0  0.0  0.0 |  4.171533 | 10.365406 |  0.267142 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0  20.0  20.0  20.0  20.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.137109  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.137109     0.0        0.0 |       0.0        0.0  17.137109     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.047067  0.047067  0.0  0.0  0.0 | 0.050024  0.050024  0.0  0.0  0.0 |  4.246759 |  0.5 |  0.232421 | 0.0  0.0  0.0  0.075307  0.0 | 0.053916 |  0.236258 |  0.137787 |  0.137787 |  0.574112 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.954492   99.954492  0.0  0.0  0.0 |  7.685872 | 10.729303 |  0.574112 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132     0.0        0.0 |       0.0        0.0  16.686132     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.046674  0.046674  0.0  0.0  0.0 | 0.050023  0.050023  0.0  0.0  0.0 |  4.134016 |  0.5 |  0.794235 | 0.0  0.0  0.0  0.074675  0.0 | 0.055736 |  0.687408 |  0.298126 |  0.298126 |  1.242192 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   99.95784    99.95784  0.0  0.0  0.0 | 10.525652 | 11.091471 |  1.242192 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   16.14496  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   16.14496     0.0        0.0 |       0.0        0.0   16.14496     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.046199  0.046199  0.0  0.0  0.0 | 0.050021  0.050021  0.0  0.0  0.0 |  3.998724 |  0.5 |  1.564789 | 0.0  0.0  0.0  0.073912  0.0 | 0.057547 |  1.305542 |  0.724516 |  0.724516 |  3.018816 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.961662   99.961662  0.0  0.0  0.0 | 12.459588 | 11.451936 |  3.018816 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132     0.0        0.0 |       0.0        0.0  16.686132     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.046678  0.046678  0.0  0.0  0.0 | 0.050019  0.050019  0.0  0.0  0.0 |  4.134019 |  0.5 |  2.310321 | 0.0  0.0  0.0  0.074675  0.0 | 0.059349 |  1.903649 |  1.301092 |  1.301092 |  5.421215 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.965004   99.965004  0.0  0.0  0.0 | 13.783286 |  11.81049 |  5.421215 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1  23.1  23.1  23.1  23.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.933163  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.933163     0.0        0.0 |       0.0        0.0  19.933163     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.049461  0.049461  0.0  0.0  0.0 | 0.050017  0.050017  0.0  0.0  0.0 |  4.945778 |  0.5 |  3.097322 | 0.0  0.0  0.0  0.079126  0.0 | 0.061139 |  2.534921 |  1.911019 |  1.911019 |  7.962579 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   99.96556    99.96556  0.0  0.0  0.0 | 15.131742 | 12.166619 |  7.962579 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7  25.7  25.7  25.7  25.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.278241  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.278241     0.0        0.0 |       0.0        0.0  22.278241     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.140503  0.140503  0.0  0.0  0.0 | 0.050017  0.050017  0.0  0.0  0.0 |  5.532047 |  0.5 |  3.941184 | 0.0  0.0  0.0  0.224769  0.0 | 0.062815 |  3.211575 |  2.545006 |  2.545006 | 10.604191 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.875075   99.875075  0.0  0.0  0.0 | 16.222604 | 12.500265 | 10.604191 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9  28.9  28.9  28.9  28.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  25.164491  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  25.164491     0.0        0.0 |       0.0        0.0  25.164491     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |   0.9295    0.9295  0.0  0.0  0.0 | 0.050062  0.050062  0.0  0.0  0.0 |  6.253576 |  0.5 |  4.774218 | 0.0  0.0  0.0  1.488314  0.0 | 0.063581 |  3.878717 |  3.209461 |  3.209461 | 13.372755 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.995638   98.995638  0.0  0.0  0.0 | 17.201962 | 12.652639 | 13.372755 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4  28.4  28.4  28.4  28.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  24.713515  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  24.713515     0.0        0.0 |       0.0        0.0  24.713515     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.082781  2.082781  0.0  0.0  0.0 | 0.050502  0.050502  0.0  0.0  0.0 |  6.140502 |  0.5 |  5.310776 | 0.0  0.0  0.0  3.364543  0.0 | 0.063003 |  4.307423 |  3.825731 |  3.825731 | 15.940547 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  96.963359   96.963359  0.0  0.0  0.0 | 17.531689 | 12.537559 | 15.940547 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0  32.0  32.0  32.0  32.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.960546  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.960546     0.0        0.0 |       0.0        0.0  27.960546     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.416451  3.416451  0.0  0.0  0.0 | 0.051518  0.051518  0.0  0.0  0.0 |  6.951498 |  0.5 |  5.803473 | 0.0  0.0  0.0  5.634518  0.0 | 0.060806 |  4.699531 |   4.29929 |   4.29929 | 17.913708 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  93.598427   93.598427  0.0  0.0  0.0 | 18.179713 | 12.100393 | 17.913708 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8  32.8  32.8  32.8  32.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  28.682108  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  28.682108     0.0        0.0 |       0.0        0.0  28.682108     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.724735  3.724735  0.0  0.0  0.0 | 0.053201  0.053201  0.0  0.0  0.0 |  7.130626 |  0.5 |  6.280002 | 0.0  0.0  0.0  6.363559  0.0 | 0.058099 |  5.078228 |  4.696551 |  4.696551 | 19.568961 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  89.926892   89.926892  0.0  0.0  0.0 | 18.530338 | 11.561786 | 19.568961 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8  35.8  35.8  35.8  35.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  31.387967  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  31.387967     0.0        0.0 |       0.0        0.0  31.387967     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.057509  4.057509  0.0  0.0  0.0 | 0.055037  0.055037  0.0  0.0  0.0 |  7.805714 |  0.5 |  6.769728 | 0.0  0.0  0.0  7.214798  0.0 | 0.054798 |  5.466927 |   5.08045 |   5.08045 | 21.168543 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   85.92442    85.92442  0.0  0.0  0.0 | 19.066325 | 10.904873 | 21.168543 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7  37.7  37.7  37.7  37.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  33.101678  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  33.101678     0.0        0.0 |       0.0        0.0  33.101678     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.121139  2.121139  0.0  0.0  0.0 | 0.057038  0.057038  0.0  0.0  0.0 |  8.232641 |  0.5 |  7.299913 | 0.0  0.0  0.0  3.947157  0.0 | 0.053848 |  5.890189 |  5.474608 |  5.474608 | 22.810865 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  83.860318   83.860318  0.0  0.0  0.0 | 19.499053 | 10.715717 | 22.810865 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8  31.8  31.8  31.8  31.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.780155  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.780155     0.0        0.0 |       0.0        0.0  27.780155     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.675566  2.675566  0.0  0.0  0.0 |  0.05807   0.05807  0.0  0.0  0.0 |  6.901486 |  0.5 |  7.056908 | 0.0  0.0  0.0  5.101273  0.0 | 0.052078 |  5.694133 |  5.752563 |  5.752563 | 23.969011 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  81.242822   81.242822  0.0  0.0  0.0 | 18.843631 | 10.363458 | 23.969011 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9  24.9  24.9  24.9  24.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.556679  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.556679     0.0        0.0 |       0.0        0.0  21.556679     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.894443  2.894443  0.0  0.0  0.0 | 0.059379  0.059379  0.0  0.0  0.0 |  5.344636 |  0.5 |  6.038241 | 0.0  0.0  0.0  5.696167  0.0 | 0.049891 |  4.877158 |  5.556151 |  5.556151 | 23.150628 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  78.407757   78.407757  0.0  0.0  0.0 | 17.650026 |    9.9284 | 23.150628 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0  26.0  26.0  26.0  26.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.242   0.242   0.242   0.242   0.242 |    0.242     0.242  0.0  0.0     0.242 |       0.0        0.0   0.242   0.242        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.548827  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.790827   0.242        0.0 |       0.0        0.0  22.790827   0.242        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.881513  0.881513  0.0  0.0  0.0 | 0.060796  0.060796  0.0  0.0  0.0 |   5.65211 |  0.5 |  5.409751 | 0.0  0.0  0.0  1.797434  0.0 | 0.050674 |  4.375097 |  4.947139 |  4.947139 | 20.613078 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  77.587041   77.587041  0.0  0.0  0.0 | 17.392384 | 10.084092 | 20.613078 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5  26.5  26.5  26.5  26.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.999804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.999804     0.0        0.0 |       0.0        0.0  22.999804     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.683776  0.683776  0.0  0.0  0.0 | 0.061206  0.061206  0.0  0.0  0.0 |  5.704046 |  0.5 |  5.272798 | 0.0  0.0  0.0  1.408972  0.0 | 0.051557 |  4.266358 |  4.462502 |  4.462502 | 18.593758 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.964471   76.964471  0.0  0.0  0.0 | 17.323632 | 10.259825 | 18.593758 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3  24.3  24.3  24.3  24.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.015507  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.015507     0.0        0.0 |       0.0        0.0  21.015507     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.421312  0.421312  0.0  0.0  0.0 | 0.061518  0.061518  0.0  0.0  0.0 |  5.207738 |  0.5 |  5.061946 | 0.0  0.0  0.0  0.875158  0.0 | 0.052817 |  4.098852 |  4.253299 |  4.253299 | 17.722079 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.604676   76.604676  0.0  0.0  0.0 | 16.969424 | 10.510557 | 17.722079 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2  22.2  22.2  22.2  22.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  1.573   1.573   1.573   1.573   1.573 | 0.494092  0.494092  0.0  0.0  0.494092 |       0.0        0.0   1.573   1.573        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.121405  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  20.694405   1.573        0.0 |       0.0        0.0  20.694405   1.573        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.118376  0.118376  0.0  0.0  0.0 | 0.061698  0.061698  0.0  0.0  0.0 |  5.127328 |  0.5 |  4.828549 | 0.0  0.0  0.0  0.494092  0.0 | 0.055466 |  3.914608 |  4.095133 |  4.095133 | 17.063053 | 1.078908  1.078908  0.0  0.0  1.078908 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.547998   76.547998  0.0  0.0  0.0 | 16.768203 | 11.037792 | 17.063053 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  6.776   6.776   6.776   6.776   6.776 |  0.08755   0.08755  0.0  0.0   0.08755 |  5.854908   5.854908   6.776   6.776   5.854908 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.854908   5.854908  25.536624   6.776   5.854908 |  0.857685   0.857685  25.536624   6.776   5.854908 | 0.0  0.0  0.0  0.0   5.854908 | 0.022327  0.022327  0.0  0.0  0.0 | 0.059227  0.059227  0.0  0.0  0.0 |  6.982999 |  0.5 |  5.347468 | 0.0  0.0  0.0   0.08755  0.0 | 0.062109 |   4.72627 |  4.135921 |  4.135921 | 17.233002 |  1.91245   1.91245  0.0  0.0   1.91245 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  81.582122   81.582122  0.0  0.0  0.0 | 17.903734 | 12.359747 | 17.233002 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0  21.0  21.0  21.0  21.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  3.509   3.509   3.509   3.509   3.509 |   0.0539    0.0539  0.0  0.0    0.0539 |   3.42145    3.42145   3.509   3.509    3.42145 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.039062  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   3.42145    3.42145  21.548062   3.509    3.42145 |  0.569299   0.569299  21.548062   3.509    3.42145 | 0.0  0.0  0.0  0.0    3.42145 | 0.014232  0.014232  0.0  0.0  0.0 | 0.057783  0.057783  0.0  0.0  0.0 |  5.770652 |  0.5 |  5.606212 | 0.0  0.0  0.0    0.0539  0.0 |  0.06641 |  4.775049 |  4.556741 |  4.556741 | 18.986419 |   1.9461    1.9461  0.0  0.0    1.9461 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  84.477824   84.477824  0.0  0.0  0.0 | 17.568174 | 13.215494 | 18.986419 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4  20.4  20.4  20.4  20.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.929   5.929   5.929   5.929   5.929 | 0.041902  0.041902  0.0  0.0  0.041902 |    5.8751     5.8751   5.929   5.929     5.8751 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   17.49789  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    5.8751     5.8751   23.42689   5.929     5.8751 |  1.048192   1.048192   23.42689   5.929     5.8751 | 0.0  0.0  0.0  0.0     5.8751 | 0.011701  0.011701  0.0  0.0  0.0 | 0.055348  0.055348  0.0  0.0  0.0 |  6.601355 |  0.5 |  5.698212 | 0.0  0.0  0.0  0.041902  0.0 | 0.072425 |   5.01784 |  4.818163 |  4.818163 | 20.075678 | 1.958098  1.958098  0.0  0.0  1.958098 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  89.348379   89.348379  0.0  0.0  0.0 | 17.971318 | 14.412655 | 20.075678 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2  20.2  20.2  20.2  20.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 | 12.826  12.826  12.826  12.826  12.826 | 0.020954  0.020954  0.0  0.0  0.020954 | 12.784098  12.784098  12.826  12.826  12.784098 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.317499  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.784098  12.784098  30.143499  12.826  12.784098 |  2.551429   2.551429  30.143499  12.826  12.784098 | 0.0  0.0  0.0  0.0  12.784098 | 0.006524  0.006524  0.0  0.0  0.0 | 0.050209  0.050209  0.0  0.0  0.0 |  9.411789 |  0.5 |  7.043385 | 0.0  0.0  0.0  0.020954  0.0 | 0.083353 |  6.564777 |   5.30765 |   5.30765 | 22.115208 | 1.979046  1.979046  0.0  0.0  1.979046 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.624734   99.624734  0.0  0.0  0.0 | 19.839723 | 16.587166 | 22.115208 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.121   0.121   0.121   0.121   0.121 | 0.074152  0.074152  0.0  0.0  0.074152 |  0.100046   0.100046   0.121   0.121   0.100046 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.956718  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.100046   0.100046  17.077718   0.121   0.100046 |  0.024824   0.024824  17.077718   0.121   0.100046 | 0.0  0.0  0.0  0.0   0.100046 | 0.023115  0.023115  0.0  0.0  0.0 |  0.05015   0.05015  0.0  0.0  0.0 |  4.250435 |  0.5 |  6.274176 | 0.0  0.0  0.0  0.074152  0.0 | 0.085112 |  5.105448 |  5.896718 |  5.896718 | 24.569657 | 1.925848  1.925848  0.0  0.0  1.925848 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.726991   99.726991  0.0  0.0  0.0 | 17.315982 | 16.937318 | 24.569657 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.847   0.847   0.847   0.847   0.847 |  0.06896   0.06896  0.0  0.0   0.06896 |  0.772848   0.772848   0.847   0.847   0.772848 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.772848   0.772848  17.810337   0.847   0.772848 |  0.192158   0.192158  17.810337   0.847   0.772848 | 0.0  0.0  0.0  0.0   0.772848 | 0.021627  0.021627  0.0  0.0  0.0 | 0.049846  0.049846  0.0  0.0  0.0 |  4.559318 |  0.5 |  4.841322 | 0.0  0.0  0.0   0.06896  0.0 | 0.087385 |   4.00614 |  5.185453 |  5.185453 | 21.606053 |  1.93104   1.93104  0.0  0.0   1.93104 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   100.3359    100.3359  0.0  0.0  0.0 | 16.533979 | 17.389652 | 21.606053 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.63    3.63    3.63    3.63    3.63 | 0.052207  0.052207  0.0  0.0  0.052207 |   3.56104    3.56104    3.63    3.63    3.56104 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   3.56104    3.56104  20.593337    3.63    3.56104 |  0.896251   0.896251  20.593337    3.63    3.56104 | 0.0  0.0  0.0  0.0    3.56104 | 0.016812  0.016812  0.0  0.0  0.0 |   0.0485    0.0485  0.0  0.0  0.0 |  5.784148 |  0.5 |  4.796036 | 0.0  0.0  0.0  0.052207  0.0 | 0.091647 |  4.159768 |  4.284571 |  4.284571 | 17.852377 | 1.947793  1.947793  0.0  0.0  1.947793 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 103.032377  103.032377  0.0  0.0  0.0 |  17.02209 |  18.23769 | 17.852377 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.541   2.541   2.541   2.541   2.541 | 0.057919  0.057919  0.0  0.0  0.057919 |  2.488793   2.488793   2.541   2.541   2.488793 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.692645  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.488793   2.488793  19.233645   2.541   2.488793 |  0.660505   0.660505  19.233645   2.541   2.488793 | 0.0  0.0  0.0  0.0   2.488793 | 0.018988  0.018988  0.0  0.0  0.0 |  0.04757   0.04757  0.0  0.0  0.0 |  5.268113 |  0.5 |  4.905892 | 0.0  0.0  0.0  0.057919  0.0 | 0.095105 |  4.179398 |  4.129991 |  4.129991 | 17.208295 | 1.942081  1.942081  0.0  0.0  1.942081 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.889246  104.889246  0.0  0.0  0.0 | 16.884311 | 18.925883 | 17.208295 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 12.584  12.584  12.584  12.584  12.584 | 0.021144  0.021144  0.0  0.0  0.021144 | 12.526081  12.526081  12.584  12.584  12.526081 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.526081  12.526081  29.096184  12.584  12.526081 |  3.445221   3.445221  29.096184  12.584  12.526081 | 0.0  0.0  0.0  0.0  12.526081 | 0.007533  0.007533  0.0  0.0  0.0 | 0.043015  0.043015  0.0  0.0  0.0 |  9.825701 |  0.5 |  6.493702 | 0.0  0.0  0.0  0.021144  0.0 | 0.105746 |  6.128729 |  4.608221 |  4.608221 | 19.200919 | 1.978856  1.978856  0.0  0.0  1.978856 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 114.005587  114.005587  0.0  0.0  0.0 |  19.71631 | 21.043402 | 19.200919 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.235   4.235   4.235   4.235   4.235 | 0.048561  0.048561  0.0  0.0  0.048561 |  4.213856   4.213856   4.235   4.235   4.213856 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  4.213856   4.213856  20.566723   4.235   4.213856 |  1.369216   1.369216  20.566723   4.235   4.213856 | 0.0  0.0  0.0  0.0   4.213856 | 0.017739  0.017739  0.0  0.0  0.0 | 0.041575  0.041575  0.0  0.0  0.0 |  6.137412 |  0.5 |  6.902541 | 0.0  0.0  0.0  0.048561  0.0 |  0.11035 |   5.90595 |  5.646038 |  5.646038 | 23.525159 | 1.951439  1.951439  0.0  0.0  1.951439 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 116.874063  116.874063  0.0  0.0  0.0 | 18.451181 | 21.959686 | 23.525159 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.114   4.114   4.114   4.114   4.114 | 0.048983  0.048983  0.0  0.0  0.048983 |  4.065439   4.065439   4.114   4.114   4.065439 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.151262  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  4.065439   4.065439  20.265262   4.114   4.065439 |  1.388301   1.388301  20.265262   4.114   4.065439 | 0.0  0.0  0.0  0.0   4.065439 | 0.018306  0.018306  0.0  0.0  0.0 | 0.040224  0.040224  0.0  0.0  0.0 |  6.077373 |  0.5 |  6.058646 | 0.0  0.0  0.0  0.048983  0.0 | 0.114845 |  5.225135 |  5.804164 |  5.804164 | 24.184018 | 1.951017  1.951017  0.0  0.0  1.951017 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 119.573119  119.573119  0.0  0.0  0.0 | 17.969908 | 22.854129 | 24.184018 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.452   1.452   1.452   1.452   1.452 | 0.124728  0.124728  0.0  0.0  0.124728 |  1.403017   1.403017   1.452   1.452   1.403017 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.403017   1.403017  17.783723   1.452   1.403017 |  0.501499   0.501499  17.783723   1.452   1.403017 | 0.0  0.0  0.0  0.0   1.403017 | 0.046974  0.046974  0.0  0.0  0.0 | 0.039763  0.039763  0.0  0.0  0.0 |  4.792233 |  0.5 |  5.304618 | 0.0  0.0  0.0  0.124728  0.0 | 0.117362 |  4.446766 |  5.203456 |  5.203456 | 21.681068 | 1.875272  1.875272  0.0  0.0  1.875272 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 120.467426  120.467426  0.0  0.0  0.0 | 16.957523 | 23.354949 | 21.681068 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.121   0.121   0.121   0.121   0.121 |  0.96204   0.96204  0.0  0.0   0.96204 |       0.0        0.0   0.121   0.121        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.723415   0.121        0.0 |       0.0        0.0  16.723415   0.121        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.36229   0.36229  0.0  0.0  0.0 | 0.039766  0.039766  0.0  0.0  0.0 |  4.151029 |  0.5 |  4.505756 | 0.0  0.0  0.0   0.96204  0.0 | 0.118317 |  3.715034 |   4.45713 |   4.45713 | 18.571374 | 1.034232  1.034232  0.0  0.0  1.034232 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 120.144903  120.144903  0.0  0.0  0.0 | 16.102797 | 23.545055 | 18.571374 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 | 1.034232  1.034232  0.0  0.0  1.034232 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.602415     0.0        0.0 |       0.0        0.0  16.602415     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  1.53617   1.53617  0.0  0.0  0.0 | 0.039928  0.039928  0.0  0.0  0.0 |  4.120658 |  0.5 |  4.028869 | 0.0  0.0  0.0  2.345064  0.0 | 0.118193 |  3.333408 |  3.792835 |  3.792835 |  15.80348 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  118.64866   118.64866  0.0  0.0  0.0 | 15.694586 | 23.520425 |  15.80348 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.782876     0.0        0.0 |       0.0        0.0  16.782876     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.102834  1.102834  0.0  0.0  0.0 | 0.040676  0.040676  0.0  0.0  0.0 |  4.165212 |  0.5 |  3.826827 | 0.0  0.0  0.0  1.486684  0.0 | 0.118683 |  3.172232 |  3.382397 |  3.382397 | 14.093321 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 117.586502  117.586502  0.0  0.0  0.0 | 15.532971 |  23.61793 | 14.093321 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.484   0.484   0.484   0.484   0.484 |    0.484     0.484  0.0  0.0     0.484 |       0.0        0.0   0.484   0.484        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.447337   0.484        0.0 |       0.0        0.0  17.447337   0.484        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.324374  1.324374  0.0  0.0  0.0 | 0.041207  0.041207  0.0  0.0  0.0 |  4.330929 |  0.5 |  3.791462 | 0.0  0.0  0.0  1.805438  0.0 | 0.119289 |  3.144506 |  3.201888 |  3.201888 | 13.341199 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 116.303334  116.303334  0.0  0.0  0.0 | 15.572439 | 23.738436 | 13.341199 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9  19.9  19.9  19.9  19.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.121   0.121   0.121   0.121   0.121 |    0.121     0.121  0.0  0.0     0.121 |       0.0        0.0   0.121   0.121        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.053567  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.174567   0.121        0.0 |       0.0        0.0  17.174567   0.121        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.438518  3.438518  0.0  0.0  0.0 | 0.041848  0.041848  0.0  0.0  0.0 |  4.262256 |  0.5 |   3.79179 | 0.0  0.0  0.0  4.728713  0.0 | 0.117544 |  3.143139 |  3.150363 |  3.150363 | 13.126514 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 112.906664  112.906664  0.0  0.0  0.0 | 15.542905 | 23.391218 | 13.126514 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.356   4.356   4.356   4.356   4.356 | 1.680962  1.680962  0.0  0.0  1.680962 |     2.356      2.356   4.356   4.356      2.356 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |     2.356      2.356  21.138876   4.356      2.356 |  0.750852   0.750852  21.138876   4.356      2.356 | 0.0  0.0  0.0  0.0      2.356 | 0.601756  0.601756  0.0  0.0  0.0 | 0.042744  0.042744  0.0  0.0  0.0 |    5.8158 |  0.5 |  4.250394 | 0.0  0.0  0.0  1.680962  0.0 |  0.12101 |  3.670324 |  3.260595 |  3.260595 | 13.585812 | 0.319038  0.319038  0.0  0.0  0.319038 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   113.9528    113.9528  0.0  0.0  0.0 | 16.608311 | 24.080928 | 13.585812 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.139   7.139   7.139   7.139   7.139 |      2.0       2.0  0.0  0.0       2.0 |  5.458038   5.458038   7.139   7.139   5.458038 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.458038   5.458038  23.741415   7.139   5.458038 |  1.771848   1.771848  23.741415   7.139   5.458038 | 0.0  0.0  0.0  0.0   5.458038 | 1.886609  1.886609  0.0  0.0  0.0 | 0.041181  0.041181  0.0  0.0  0.0 |  7.233355 |  0.5 |  5.340478 | 0.0  0.0  0.0  3.326542  0.0 | 0.125271 |  4.753171 |  3.793804 |  3.793804 | 15.807518 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.793562  115.793562  0.0  0.0  0.0 | 18.001188 | 24.928866 | 15.807518 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.331   1.331   1.331   1.331   1.331 |    1.331     1.331  0.0  0.0     1.331 |       0.0        0.0   1.331   1.331        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.843184   1.331        0.0 |       0.0        0.0  17.843184   1.331        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.685297  1.685297  0.0  0.0  0.0 | 0.042103  0.042103  0.0  0.0  0.0 |  4.429219 |  0.5 |  5.199841 | 0.0  0.0  0.0  2.727186  0.0 |  0.12579 |  4.277277 |  4.406784 |  4.406784 | 18.361599 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 114.150368  114.150368  0.0  0.0  0.0 | 16.730566 | 25.032192 | 18.361599 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 25.047  25.047  25.047  25.047  25.047 | 0.132786  0.132786  0.0  0.0  0.132786 |    23.047     23.047  25.047  25.047     23.047 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   15.70011  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    23.047     23.047   40.74711  25.047     23.047 |  7.507737   7.507737   40.74711  25.047     23.047 | 0.0  0.0  0.0  0.0     23.047 |  0.05383   0.05383  0.0  0.0  0.0 | 0.035155  0.035155  0.0  0.0  0.0 | 15.791214 |  0.5 |  9.004531 | 0.0  0.0  0.0  0.132786  0.0 |   0.1451 |  8.875518 |  5.404862 |  5.404862 | 22.520259 | 1.867214  1.867214  0.0  0.0  1.867214 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 129.670956  129.670956  0.0  0.0  0.0 | 23.017249 | 28.874838 | 22.520259 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0  18.0  18.0  18.0  18.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 45.859  45.859  45.859  45.859  45.859 | 0.017578  0.017578  0.0  0.0  0.017578 | 45.726214  45.726214  45.859  45.859  45.726214 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.339188  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 45.726214  45.726214  61.198188  45.859  45.726214 | 19.221651  19.221651  61.198188  45.859  45.726214 | 0.0  0.0  0.0  0.0  45.726214 |  0.00858   0.00858  0.0  0.0  0.0 | 0.021912  0.021912  0.0  0.0  0.0 | 29.699351 |  0.5 | 22.253521 | 0.0  0.0  0.0  0.017578  0.0 | 0.179261 | 21.018541 | 10.552136 | 10.552136 | 43.967234 | 1.982422  1.982422  0.0  0.0  1.982422 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 156.188852  156.188852  0.0  0.0  0.0 | 29.963078 | 35.672923 | 43.967234 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9  17.9  17.9  17.9  17.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  9.922   9.922   9.922   9.922   9.922 | 0.821583  0.821583  0.0  0.0  0.821583 |  9.904422   9.904422   9.922   9.922   9.904422 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.248957  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  9.904422   9.904422  25.170957   9.922   9.904422 |  6.040449   6.040449  25.170957   9.922   9.904422 | 0.0  0.0  0.0  0.0   9.904422 | 0.410792  0.410792  0.0  0.0  0.0 | 0.019974  0.019974  0.0  0.0  0.0 | 10.808096 |  0.5 | 17.103238 | 0.0  0.0  0.0  0.821583  0.0 | 0.187008 | 14.517426 | 16.875399 | 16.875399 | 70.314164 | 1.178417  1.178417  0.0  0.0  1.178417 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 159.662007  159.662007  0.0  0.0  0.0 | 23.167935 | 37.214546 | 70.314164 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4  17.4  17.4  17.4  17.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.356   4.356   4.356   4.356   4.356 | 0.792694  0.792694  0.0  0.0  0.792694 |  3.534417   3.534417   4.356   4.356   3.534417 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.797804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.534417   3.534417  19.153804   4.356   3.534417 |   2.25248    2.25248  19.153804   4.356   3.534417 | 0.0  0.0  0.0  0.0   3.534417 | 0.396347  0.396347  0.0  0.0  0.0 | 0.019528  0.019528  0.0  0.0  0.0 |  6.463165 |  0.5 |  9.467853 | 0.0  0.0  0.0  0.792694  0.0 | 0.190761 |  7.987953 | 14.511125 | 14.511125 | 60.463019 | 1.207306  1.207306  0.0  0.0  1.207306 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 160.567125  160.567125  0.0  0.0  0.0 | 19.663248 |   37.9614 | 60.463019 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  9.075   9.075   9.075   9.075   9.075 | 0.090607  0.090607  0.0  0.0  0.090607 |  8.282306   8.282306   9.075   9.075   8.282306 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.527113  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  8.282306   8.282306  23.602113   9.075   8.282306 |  5.338319   5.338319  23.602113   9.075   8.282306 | 0.0  0.0  0.0  0.0   8.282306 | 0.045304  0.045304  0.0  0.0  0.0 | 0.018244  0.018244  0.0  0.0  0.0 |  9.890584 |  0.5 |  8.416903 | 0.0  0.0  0.0  0.090607  0.0 | 0.198367 |  7.470819 |  9.324029 |  9.324029 | 38.850119 | 1.909393  1.909393  0.0  0.0  1.909393 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 163.484053  163.484053  0.0  0.0  0.0 | 20.636929 | 39.475089 | 38.850119 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 22.385  22.385  22.385  22.385  22.385 | 0.007604  0.007604  0.0  0.0  0.007604 | 22.294393  22.294393  22.385  22.385  22.294393 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.346652  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 22.294393  22.294393  36.731652  22.385  22.294393 | 14.896576  14.896576  36.731652  22.385  22.294393 | 0.0  0.0  0.0  0.0  22.294393 | 0.003802  0.003802  0.0  0.0  0.0 | 0.014559  0.014559  0.0  0.0  0.0 | 20.344425 |  0.5 | 14.449904 | 0.0  0.0  0.0  0.007604  0.0 | 0.215502 | 13.247351 |  8.869412 |  8.869412 | 36.955881 | 1.992396  1.992396  0.0  0.0  1.992396 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 170.892627  170.892627  0.0  0.0  0.0 | 26.031451 | 42.884929 | 36.955881 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 18.634  18.634  18.634  18.634  18.634 | 0.011044  0.011044  0.0  0.0  0.011044 | 18.626396  18.626396  18.634  18.634  18.626396 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.256421  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 18.626396  18.626396  32.890421  18.634  18.626396 | 13.599267  13.599267  32.890421  18.634  18.626396 | 0.0  0.0  0.0  0.0  18.626396 | 0.005522  0.005522  0.0  0.0  0.0 |  0.01204   0.01204  0.0  0.0  0.0 | 18.413025 |  0.5 | 17.952775 | 0.0  0.0  0.0  0.011044  0.0 |  0.22987 | 15.818525 | 12.535049 | 12.535049 | 52.229372 | 1.988956  1.988956  0.0  0.0  1.988956 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 175.926274  175.926274  0.0  0.0  0.0 | 25.991701 | 45.744053 | 52.229372 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1  16.1  16.1  16.1  16.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.623   7.623   7.623   7.623   7.623 | 0.032783  0.032783  0.0  0.0  0.032783 |  7.611956   7.611956   7.623   7.623   7.611956 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.624808  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  7.611956   7.611956  21.247808   7.623   7.611956 |  5.889761   5.889761  21.247808   7.623   7.611956 | 0.0  0.0  0.0  0.0   7.611956 | 0.016392  0.016392  0.0  0.0  0.0 | 0.011176  0.011176  0.0  0.0  0.0 |  9.720891 |  0.5 |  13.19695 | 0.0  0.0  0.0  0.032783  0.0 | 0.236285 | 11.285556 | 14.239827 | 14.239827 | 59.332611 | 1.967217  1.967217  0.0  0.0  1.967217 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.643253  177.643253  0.0  0.0  0.0 | 22.015641 | 47.020657 | 59.332611 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.299   2.299   2.299   2.299   2.299 | 0.054873  0.054873  0.0  0.0  0.054873 |  2.266217   2.266217   2.299   2.299   2.266217 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.812733  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.266217   2.266217  15.111733   2.299   2.266217 |  1.787882   1.787882  15.111733   2.299   2.266217 | 0.0  0.0  0.0  0.0   2.266217 | 0.027436  0.027436  0.0  0.0  0.0 | 0.010939  0.010939  0.0  0.0  0.0 |  5.110641 |  0.5 |   8.06796 | 0.0  0.0  0.0  0.054873  0.0 | 0.238849 |  6.828375 | 11.302398 | 11.302398 | 47.093325 | 1.945127  1.945127  0.0  0.0  1.945127 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  178.10509   178.10509  0.0  0.0  0.0 | 18.558321 | 47.530969 | 47.093325 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3  15.3  15.3  15.3  15.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  5.929   5.929   5.929   5.929   5.929 | 0.038243  0.038243  0.0  0.0  0.038243 |  5.874127   5.874127   5.929   5.929   5.874127 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   12.90841  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.874127   5.874127   18.83741   5.929   5.874127 |  4.658392   4.658392   18.83741   5.929   5.874127 | 0.0  0.0  0.0  0.0   5.874127 | 0.019122  0.019122  0.0  0.0  0.0 |  0.01034   0.01034  0.0  0.0  0.0 |  8.195392 |  0.5 |  6.943194 | 0.0  0.0  0.0  0.038243  0.0 | 0.244005 |  6.173902 |  7.673421 |  7.673421 |  31.97259 | 1.961757  1.961757  0.0  0.0  1.961757 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.312044  179.312044  0.0  0.0  0.0 | 19.310519 | 48.557071 |  31.97259 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.267   3.267   3.267   3.267   3.267 |  0.04981   0.04981  0.0  0.0   0.04981 |  3.228757   3.228757   3.267   3.267   3.228757 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.818141  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.228757   3.228757  16.085141   3.267   3.228757 |   2.59534    2.59534  16.085141   3.267   3.228757 | 0.0  0.0  0.0  0.0   3.228757 | 0.024905  0.024905  0.0  0.0  0.0 | 0.010027  0.010027  0.0  0.0  0.0 |   5.96027 |  0.5 |  6.567203 | 0.0  0.0  0.0   0.04981  0.0 | 0.247226 |  5.699757 |  6.213975 |  6.213975 | 25.891562 |  1.95019   1.95019  0.0  0.0   1.95019 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.930583  179.930583  0.0  0.0  0.0 | 18.203586 | 49.198015 | 25.891562 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7  14.7  14.7  14.7  14.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.605   0.605   0.605   0.605   0.605 | 0.064366  0.064366  0.0  0.0  0.064366 |   0.55519    0.55519   0.605   0.605    0.55519 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.366798  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   0.55519    0.55519  12.971798   0.605    0.55519 |  0.449357   0.449357  12.971798   0.605    0.55519 | 0.0  0.0  0.0  0.0    0.55519 | 0.032183  0.032183  0.0  0.0  0.0 | 0.009982  0.009982  0.0  0.0  0.0 |  3.572481 |  0.5 |  5.034251 | 0.0  0.0  0.0  0.064366  0.0 | 0.248519 |  4.296365 |  5.493258 |  5.493258 | 22.888574 | 1.935634  1.935634  0.0  0.0  1.935634 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 180.014215  180.014215  0.0  0.0  0.0 | 16.241816 | 49.455301 | 22.888574 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9  14.9  14.9  14.9  14.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.242   0.242   0.242   0.242   0.242 | 0.067009  0.067009  0.0  0.0  0.067009 |  0.177634   0.177634   0.242   0.242   0.177634 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.547335  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.177634   0.177634  12.789335   0.242   0.177634 |  0.143907   0.143907  12.789335   0.242   0.177634 | 0.0  0.0  0.0  0.0   0.177634 | 0.033505  0.033505  0.0  0.0  0.0 | 0.009976  0.009976  0.0  0.0  0.0 |  3.297782 |  0.5 |  3.859109 | 0.0  0.0  0.0  0.067009  0.0 | 0.249544 |  3.332038 |  4.393935 |  4.393935 | 18.308062 | 1.932991  1.932991  0.0  0.0  1.932991 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 180.024414  180.024414  0.0  0.0  0.0 | 15.180489 | 49.659327 | 18.308062 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5  15.5  15.5  15.5  15.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.605   0.605   0.605   0.605   0.605 | 0.065382  0.065382  0.0  0.0  0.065382 |  0.537991   0.537991   0.605   0.605   0.537991 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.088947  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.537991   0.537991  13.693947   0.605   0.537991 |  0.435891   0.435891  13.693947   0.605   0.537991 | 0.0  0.0  0.0  0.0   0.537991 | 0.032691  0.032691  0.0  0.0  0.0 | 0.009937  0.009937  0.0  0.0  0.0 |  3.742952 |  0.5 |  3.444335 | 0.0  0.0  0.0  0.065382  0.0 | 0.250825 |  3.025437 |  3.478199 |  3.478199 | 14.492497 | 1.934618  1.934618  0.0  0.0  1.934618 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  180.10376   180.10376  0.0  0.0  0.0 | 14.979106 | 49.914162 | 14.492497 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3  16.3  16.3  16.3  16.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.904   2.904   2.904   2.904   2.904 | 0.052751  0.052751  0.0  0.0  0.052751 |  2.838618   2.838618   2.904   2.904   2.838618 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.811096  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.838618   2.838618  16.715096   2.904   2.838618 |  2.301932   2.301932  16.715096   2.904   2.838618 | 0.0  0.0  0.0  0.0   2.838618 | 0.026376  0.026376  0.0  0.0  0.0 |  0.00968   0.00968  0.0  0.0  0.0 |  5.897963 |  0.5 |  3.972727 | 0.0  0.0  0.0  0.052751  0.0 |  0.25375 |  3.604256 |  3.222197 |  3.222197 | 13.425821 | 1.947249  1.947249  0.0  0.0  1.947249 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  180.62375   180.62375  0.0  0.0  0.0 | 16.404343 | 50.496304 | 13.425821 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6  16.6  16.6  16.6  16.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.484   0.484   0.484   0.484   0.484 | 0.113609  0.113609  0.0  0.0  0.113609 |  0.431249   0.431249   0.484   0.484   0.431249 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.081902  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.431249   0.431249  14.565902   0.484   0.431249 |  0.351736   0.351736  14.565902   0.484   0.431249 | 0.0  0.0  0.0  0.0   0.431249 | 0.056805  0.056805  0.0  0.0  0.0 | 0.009648  0.009648  0.0  0.0  0.0 |  3.898042 |  0.5 |  4.123406 | 0.0  0.0  0.0  0.113609  0.0 | 0.254889 |  3.565371 |  3.466988 |  3.466988 | 14.445785 | 1.886391  1.886391  0.0  0.0  1.886391 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 180.656106  180.656106  0.0  0.0  0.0 | 15.678979 | 50.722899 | 14.445785 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.242   0.242   0.242   0.242   0.242 | 0.353524  0.353524  0.0  0.0  0.353524 |  0.128391   0.128391   0.242   0.242   0.128391 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.128391   0.128391  14.955782   0.242   0.128391 |  0.104756   0.104756  14.955782   0.242   0.128391 | 0.0  0.0  0.0  0.0   0.128391 | 0.176762  0.176762  0.0  0.0  0.0 |  0.00966   0.00966  0.0  0.0  0.0 |  3.810268 |  0.5 |  3.715402 | 0.0  0.0  0.0  0.353524  0.0 | 0.255678 |  3.219513 |  3.497155 |  3.497155 | 14.571478 | 1.646476  1.646476  0.0  0.0  1.646476 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 180.512638  180.512638  0.0  0.0  0.0 | 15.273844 | 50.879861 | 14.571478 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 | 0.903769  0.903769  0.0  0.0  0.903769 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.074856  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.074856     0.0        0.0 |       0.0        0.0  15.074856     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.451885  0.451885  0.0  0.0  0.0 | 0.009744  0.009744  0.0  0.0  0.0 |  3.761406 |  0.5 |  3.496013 | 0.0  0.0  0.0  0.903769  0.0 | 0.255897 |  3.035647 |  3.255511 |  3.255511 | 13.564631 | 0.742707  0.742707  0.0  0.0  0.742707 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 180.070497  180.070497  0.0  0.0  0.0 | 15.039238 | 50.923426 | 13.564631 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 | 0.742707  0.742707  0.0  0.0  0.742707 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.262439  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.262439     0.0        0.0 |       0.0        0.0  14.262439     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.864415  2.864415  0.0  0.0  0.0 | 0.009965  0.009965  0.0  0.0  0.0 |  3.558136 |  0.5 |  3.323904 | 0.0  0.0  0.0  3.235768  0.0 | 0.254449 |  2.896608 |  3.045609 |  3.045609 | 12.690036 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.216048  177.216048  0.0  0.0  0.0 |  14.77347 | 50.635296 | 12.690036 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5  16.5  16.5  16.5  16.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.363   0.363   0.363   0.363   0.363 |    0.363     0.363  0.0  0.0     0.363 |       0.0        0.0   0.363   0.363        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.991633  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.354633   0.363        0.0 |       0.0        0.0  14.354633   0.363        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.055807  3.055807  0.0  0.0  0.0 | 0.011392  0.011392  0.0  0.0  0.0 |  3.580114 |  0.5 |  3.201151 | 0.0  0.0  0.0  3.237307  0.0 | 0.253266 |  2.797303 |  2.905438 |  2.905438 | 12.105991 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 174.171632  174.171632  0.0  0.0  0.0 | 14.652433 | 50.399986 | 12.105991 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.146   3.146   3.146   3.146   3.146 |      2.0       2.0  0.0  0.0       2.0 |     1.146      1.146   3.146   3.146      1.146 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |     1.146      1.146  17.498707   3.146      1.146 |  0.869119   0.869119  17.498707   3.146      1.146 | 0.0  0.0  0.0  0.0      1.146 | 1.982694  1.982694  0.0  0.0  0.0 | 0.012776  0.012776  0.0  0.0  0.0 |  5.016935 |  0.5 |  3.541103 | 0.0  0.0  0.0  2.982694  0.0 | 0.254259 |  3.146591 |   2.89699 |   2.89699 | 12.070793 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 172.478594  172.478594  0.0  0.0  0.0 | 15.628265 | 50.597628 | 12.070793 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.847   0.847   0.847   0.847   0.847 |    0.847     0.847  0.0  0.0     0.847 |       0.0        0.0   0.847   0.847        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.199707   0.847        0.0 |       0.0        0.0  15.199707   0.847        0.0 | 0.0  0.0  0.0  0.0        0.0 |  3.37986   3.37986  0.0  0.0  0.0 | 0.013761  0.013761  0.0  0.0  0.0 |  3.789606 |  0.5 |  3.683428 | 0.0  0.0  0.0   3.80336  0.0 | 0.253019 |  3.182894 |  3.077039 |  3.077039 | 12.820995 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 169.112495  169.112495  0.0  0.0  0.0 | 15.234443 | 50.350842 | 12.820995 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0  17.0  17.0  17.0  17.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.363   0.363   0.363   0.363   0.363 |    0.363     0.363  0.0  0.0     0.363 |       0.0        0.0   0.363   0.363        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.442976  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.805976   0.363        0.0 |       0.0        0.0  14.805976   0.363        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.277978  3.277978  0.0  0.0  0.0 | 0.015444  0.015444  0.0  0.0  0.0 |  3.689911 |  0.5 |  3.456409 | 0.0  0.0  0.0  3.459478  0.0 | 0.251685 |  3.000033 |  3.134191 |  3.134191 | 13.059129 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 165.849961  165.849961  0.0  0.0  0.0 | 14.967945 | 50.085375 | 13.059129 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.363   0.363   0.363   0.363   0.363 |    0.363     0.363  0.0  0.0     0.363 |       0.0        0.0   0.363   0.363        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.533245  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.896245   0.363        0.0 |       0.0        0.0  14.896245   0.363        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.602665  5.602665  0.0  0.0  0.0 | 0.017075  0.017075  0.0  0.0  0.0 |  3.711255 |  0.5 |  3.330846 | 0.0  0.0  0.0  5.784165  0.0 | 0.248697 |  2.896794 |  3.017727 |  3.017727 | 12.573863 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 160.264371  160.264371  0.0  0.0  0.0 | 14.848355 | 49.490796 | 12.573863 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0  19.0  19.0  19.0  19.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.248348  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.248348     0.0        0.0 |       0.0        0.0  16.248348     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.735707  3.735707  0.0  0.0  0.0 | 0.019868  0.019868  0.0  0.0  0.0 |  4.047186 |  0.5 |  3.364609 | 0.0  0.0  0.0  3.735707  0.0 | 0.246928 |  2.922154 |  2.925372 |  2.925372 | 12.189049 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 156.548532  156.548532  0.0  0.0  0.0 | 15.030932 | 49.138767 | 12.189049 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.158079  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.158079     0.0        0.0 |       0.0        0.0  16.158079     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.998734  3.998734  0.0  0.0  0.0 | 0.021726  0.021726  0.0  0.0  0.0 |  4.023226 |  0.5 |  3.448163 | 0.0  0.0  0.0  4.086328  0.0 | 0.244918 |   2.98712 |  2.930955 |  2.930955 | 12.212313 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 152.571523  152.571523  0.0  0.0  0.0 | 15.105995 | 48.738659 | 12.212313 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7  18.7  18.7  18.7  18.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.977542  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.977542     0.0        0.0 |       0.0        0.0  15.977542     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.413148  2.413148  0.0  0.0  0.0 | 0.023714  0.023714  0.0  0.0  0.0 |    3.9766 |  0.5 |  3.472475 | 0.0  0.0  0.0  2.530248  0.0 | 0.244029 |   3.00574 |  2.976821 |  2.976821 |  12.40342 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 150.182089  150.182089  0.0  0.0  0.0 |  15.11012 | 48.561738 |  12.40342 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.713782     0.0        0.0 |       0.0        0.0  14.713782     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |   1.0888    1.0888  0.0  0.0  0.0 | 0.024909  0.024909  0.0  0.0  0.0 |  3.659764 |  0.5 |  3.386509 | 0.0  0.0  0.0  1.159786  0.0 | 0.244123 |  2.937056 |  2.986339 |  2.986339 | 12.443079 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 149.118198  149.118198  0.0  0.0  0.0 | 14.883375 | 48.580503 | 12.443079 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.359753  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.359753     0.0        0.0 |       0.0        0.0  13.359753     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.197211  0.197211  0.0  0.0  0.0 | 0.025441  0.025441  0.0  0.0  0.0 |  3.320858 |  0.5 |  3.184702 | 0.0  0.0  0.0  0.211567  0.0 | 0.244894 |   2.77633 |  2.916602 |  2.916602 | 12.152509 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.946427  148.946427  0.0  0.0  0.0 |  14.51953 | 48.733956 | 12.152509 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0  14.0  14.0  14.0  14.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.734918  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.734918     0.0        0.0 |       0.0        0.0  11.734918     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.062778  0.062778  0.0  0.0  0.0 | 0.025527  0.025527  0.0  0.0  0.0 |  2.914584 |  0.5 |  2.909035 | 0.0  0.0  0.0  0.067425  0.0 | 0.245764 |  2.556608 |   2.76322 |   2.76322 | 11.513416 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.909177  148.909177  0.0  0.0  0.0 | 14.025079 | 48.907131 | 11.513416 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.832232  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.832232     0.0        0.0 |       0.0        0.0  10.832232     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.061476  0.061476  0.0  0.0  0.0 | 0.025545  0.025545  0.0  0.0  0.0 |  2.688899 |  0.5 |  2.632382 | 0.0  0.0  0.0  0.066044  0.0 | 0.246631 |  2.336095 |  2.556432 |  2.556432 | 10.651802 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.873246  148.873246  0.0  0.0  0.0 | 13.581596 | 49.079636 | 10.651802 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.110083  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.110083     0.0        0.0 |       0.0        0.0  10.110083     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.060419  0.060419  0.0  0.0  0.0 | 0.025563  0.025563  0.0  0.0  0.0 |  2.508348 |  0.5 |  2.401079 | 0.0  0.0  0.0  0.064924  0.0 | 0.247495 |  2.151858 |  2.344156 |  2.344156 |  9.767318 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  148.83839   148.83839  0.0  0.0  0.0 | 13.188866 | 49.251438 |  9.767318 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6  11.6  11.6  11.6  11.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.568472  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.568472     0.0        0.0 |       0.0        0.0   9.568472     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.059616  0.059616  0.0  0.0  0.0 | 0.025581  0.025581  0.0  0.0  0.0 |  2.372932 |  0.5 |  2.210642 | 0.0  0.0  0.0  0.064076  0.0 | 0.248354 |  2.000311 |  2.159122 |  2.159122 |  8.996343 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.804355  148.804355  0.0  0.0  0.0 | 12.851156 | 49.422501 |  8.996343 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1  11.1  11.1  11.1  11.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.121265  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.121265     0.0        0.0 |       0.0        0.0   9.121265     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.058939  0.058939  0.0  0.0  0.0 | 0.025598  0.025598  0.0  0.0  0.0 |  2.261118 |  0.5 |  2.054361 | 0.0  0.0  0.0  0.063363  0.0 |  0.24921 |  1.876085 |  2.006382 |  2.006382 |  8.359927 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.771013  148.771013  0.0  0.0  0.0 | 12.557912 | 49.592811 |  8.359927 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498     0.0        0.0 |       0.0        0.0   10.20498     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.060509  0.060509  0.0  0.0  0.0 | 0.025614  0.025614  0.0  0.0  0.0 |  2.532034 |  0.5 |  1.998541 | 0.0  0.0  0.0  0.065065  0.0 |  0.25006 |  1.832222 |  1.893944 |  1.893944 |  7.891432 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.736119  148.736119  0.0  0.0  0.0 | 12.591405 | 49.762027 |  7.891432 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599     0.0        0.0 |       0.0        0.0  10.385599     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.060755  0.060755  0.0  0.0  0.0 | 0.025632  0.025632  0.0  0.0  0.0 |  2.577176 |  0.5 |  2.020653 | 0.0  0.0  0.0  0.065345  0.0 | 0.250906 |  1.850702 |  1.846076 |  1.846076 |  7.691984 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.700995  148.700995  0.0  0.0  0.0 | 12.647928 | 49.930357 |  7.691984 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8  11.8  11.8  11.8  11.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  1.573   1.573   1.573   1.573   1.573 | 0.054992  0.054992  0.0  0.0  0.054992 |       0.0        0.0   1.573   1.573        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.753432  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.326432   1.573        0.0 |       0.0        0.0  11.326432   1.573        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.025559  0.025559  0.0  0.0  0.0 |  0.02565   0.02565  0.0  0.0  0.0 |  2.812371 |  0.5 |  2.091975 | 0.0  0.0  0.0  0.054992  0.0 | 0.252879 |    1.9096 |  1.859684 |  1.859684 |  7.748683 | 1.518008  1.518008  0.0  0.0  1.518008 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.701086  148.701086  0.0  0.0  0.0 | 12.868323 | 50.322908 |  7.748683 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7  11.7  11.7  11.7  11.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 | 0.064218  0.064218  0.0  0.0  0.064218 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.663122  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.663122     0.0        0.0 |       0.0        0.0   9.663122     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.029847  0.029847  0.0  0.0  0.0 | 0.025649  0.025649  0.0  0.0  0.0 |  2.396543 |  0.5 |  2.089396 | 0.0  0.0  0.0  0.064218  0.0 | 0.253712 |  1.908314 |  1.896226 |  1.896226 |  7.900942 |  1.45379   1.45379  0.0  0.0   1.45379 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.696889  148.696889  0.0  0.0  0.0 | 12.675471 | 50.488594 |  7.900942 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 | 0.065345  0.065345  0.0  0.0  0.065345 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599     0.0        0.0 |       0.0        0.0  10.385599     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.03037   0.03037  0.0  0.0  0.0 | 0.025652  0.025652  0.0  0.0  0.0 |  2.577161 |  0.5 |  2.052864 | 0.0  0.0  0.0  0.065345  0.0 | 0.254539 |  1.879861 |  1.902277 |  1.902277 |  7.926154 | 1.388445  1.388445  0.0  0.0  1.388445 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.692171  148.692171  0.0  0.0  0.0 | 12.699768 | 50.653291 |  7.926154 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8  12.8  12.8  12.8  12.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 | 0.238876  0.238876  0.0  0.0  0.238876 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.656527  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.656527     0.0        0.0 |       0.0        0.0  10.656527     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.111016  0.111016  0.0  0.0  0.0 | 0.025654  0.025654  0.0  0.0  0.0 |  2.644891 |  0.5 |  2.076514 | 0.0  0.0  0.0  0.238876  0.0 | 0.255239 |  1.899434 |  1.890533 |  1.890533 |  7.877222 | 1.149568  1.149568  0.0  0.0  1.149568 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.606808  148.606808  0.0  0.0  0.0 | 12.768146 | 50.792498 |  7.877222 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4  12.4  12.4  12.4  12.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.847   0.847   0.847   0.847   0.847 | 0.984764  0.984764  0.0  0.0  0.984764 |       0.0        0.0   0.847   0.847        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.295289  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.142289   0.847        0.0 |       0.0        0.0  11.142289   0.847        0.0 | 0.0  0.0  0.0  0.0        0.0 |   0.4574    0.4574  0.0  0.0  0.0 | 0.025697  0.025697  0.0  0.0  0.0 |    2.7663 |  0.5 |  2.129037 | 0.0  0.0  0.0  0.984764  0.0 | 0.256007 |   1.94217 |  1.904581 |  1.904581 |  7.935754 | 1.011805  1.011805  0.0  0.0  1.011805 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.175105  148.175105  0.0  0.0  0.0 | 12.905409 | 50.945382 |  7.935754 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0  12.0  12.0  12.0  12.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.484   0.484   0.484   0.484   0.484 | 1.495805  1.495805  0.0  0.0  1.495805 |       0.0        0.0   0.484   0.484        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.934051  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.418051   0.484        0.0 |       0.0        0.0  10.418051   0.484        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.510207  1.510207  0.0  0.0  0.0 | 0.025912  0.025912  0.0  0.0  0.0 |  2.585078 |  0.5 |  2.144023 | 0.0  0.0  0.0  2.344557  0.0 | 0.255541 |  1.953723 |   1.93524 |   1.93524 |  8.063501 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 146.690811  146.690811  0.0  0.0  0.0 | 12.846464 | 50.852619 |  8.063501 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.121   0.121   0.121   0.121   0.121 |    0.121     0.121  0.0  0.0     0.121 |       0.0        0.0   0.121   0.121        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.11467  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.23567   0.121        0.0 |       0.0        0.0   10.23567   0.121        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.621622  3.621622  0.0  0.0  0.0 | 0.026655  0.026655  0.0  0.0  0.0 |  2.538927 |  0.5 |  2.111054 | 0.0  0.0  0.0  3.949492  0.0 | 0.253671 |  1.925603 |  1.944907 |  1.944907 |  8.103779 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.095844  143.095844  0.0  0.0  0.0 | 12.774336 | 50.480592 |  8.103779 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6  13.6  13.6  13.6  13.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.484   0.484   0.484   0.484   0.484 |    0.484     0.484  0.0  0.0     0.484 |       0.0        0.0   0.484   0.484        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.379004  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.863004   0.484        0.0 |       0.0        0.0  11.863004   0.484        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.695438  4.695438  0.0  0.0  0.0 | 0.028452  0.028452  0.0  0.0  0.0 |  2.944412 |  0.5 |  2.170019 | 0.0  0.0  0.0  5.249074  0.0 | 0.251142 |  1.970414 |   1.94181 |   1.94181 |  8.090875 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 138.428858  138.428858  0.0  0.0  0.0 |  13.04873 | 49.977296 |  8.090875 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.081695  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.081695     0.0        0.0 |       0.0        0.0  15.081695     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.591522  5.591522  0.0  0.0  0.0 | 0.030786  0.030786  0.0  0.0  0.0 |  3.747335 |  0.5 |  2.464721 | 0.0  0.0  0.0  6.461403  0.0 | 0.247414 |  2.202697 |  2.012075 |  2.012075 |  8.383644 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 132.868122  132.868122  0.0  0.0  0.0 | 13.831343 | 49.235396 |  8.383644 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8  18.8  18.8  18.8  18.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.075101  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.075101     0.0        0.0 |       0.0        0.0  16.075101     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.366977  5.366977  0.0  0.0  0.0 | 0.033566  0.033566  0.0  0.0  0.0 |  3.993601 |  0.5 |  2.868865 | 0.0  0.0  0.0   6.46129  0.0 | 0.243705 |  2.522549 |  2.222157 |  2.222157 |  9.258987 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 127.534711  127.534711  0.0  0.0  0.0 | 14.456079 | 48.497221 |  9.258987 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0  23.0  23.0  23.0  23.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.868102  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.868102     0.0        0.0 |       0.0        0.0  19.868102     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 6.578461  6.578461  0.0  0.0  0.0 | 0.036233  0.036233  0.0  0.0  0.0 |  4.939851 |  0.5 |  3.421084 | 0.0  0.0  0.0  8.250733  0.0 | 0.238736 |  2.959687 |  2.548613 |  2.548613 | 10.619219 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 120.992482  120.992482  0.0  0.0  0.0 | 15.474846 | 47.508381 | 10.619219 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0  22.0  22.0  22.0  22.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.965006  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.965006     0.0        0.0 |       0.0        0.0  18.965006     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.465512  5.465512  0.0  0.0  0.0 | 0.039504  0.039504  0.0  0.0  0.0 |  4.711624 |  0.5 |  3.873912 | 0.0  0.0  0.0  7.225213  0.0 | 0.234524 |  3.318019 |  2.942174 |  2.942174 |  12.25906 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.566474  115.566474  0.0  0.0  0.0 | 15.812558 | 46.670255 |  12.25906 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4  22.4  22.4  22.4  22.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.326244  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.326244     0.0        0.0 |       0.0        0.0  19.326244     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.374109  4.374109  0.0  0.0  0.0 | 0.042217  0.042217  0.0  0.0  0.0 |  4.799899 |  0.5 |  4.079714 | 0.0  0.0  0.0  6.053675  0.0 |  0.23117 |   3.47953 |  3.274281 |  3.274281 | 13.642836 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 111.234582  111.234582  0.0  0.0  0.0 | 16.032743 | 46.002846 | 13.642836 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8  22.8  22.8  22.8  22.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.687483  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.687483     0.0        0.0 |       0.0        0.0  19.687483     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.372944  3.372944  0.0  0.0  0.0 | 0.044383  0.044383  0.0  0.0  0.0 |  4.888584 |  0.5 |  4.226156 | 0.0  0.0  0.0  4.849713  0.0 | 0.228693 |  3.594372 |  3.469159 |  3.469159 | 14.454828 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  107.90602   107.90602  0.0  0.0  0.0 | 16.195171 | 45.509908 | 14.454828 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7  20.7  20.7  20.7  20.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.790982  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.790982     0.0        0.0 |       0.0        0.0  17.790982     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.531759  1.531759  0.0  0.0  0.0 | 0.046047  0.046047  0.0  0.0  0.0 |   4.41321 |  0.5 |   4.16741 | 0.0  0.0  0.0   2.27028  0.0 | 0.228071 |  3.546794 |  3.558278 |  3.558278 |  14.82616 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 106.420308  106.420308  0.0  0.0  0.0 | 15.940971 | 45.386083 |  14.82616 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.713862  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.713862     0.0        0.0 |       0.0        0.0  15.713862     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.670815  0.670815  0.0  0.0  0.0 |  0.04679   0.04679  0.0  0.0  0.0 |  3.893373 |  0.5 |  3.875662 | 0.0  0.0  0.0  1.008109  0.0 | 0.228353 |  3.313659 |  3.505559 |  3.505559 | 14.606496 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 105.796283  105.796283  0.0  0.0  0.0 | 15.458682 | 45.442286 | 14.606496 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.365814  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.365814     0.0        0.0 |       0.0        0.0  13.365814     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.167863  0.167863  0.0  0.0  0.0 | 0.047102  0.047102  0.0  0.0  0.0 |  3.306127 |  0.5 |  3.461637 | 0.0  0.0  0.0  0.253754  0.0 | 0.229173 |  2.983204 |  3.292032 |  3.292032 | 13.716802 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 105.675521  105.675521  0.0  0.0  0.0 | 14.803173 | 45.605434 | 13.716802 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1  15.1  15.1  15.1  15.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.733647  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  12.733647     0.0        0.0 |       0.0        0.0  12.733647     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.045541  0.045541  0.0  0.0  0.0 | 0.047162  0.047162  0.0  0.0  0.0 |   3.14804 |  0.5 |  3.101018 | 0.0  0.0  0.0  0.068921  0.0 | 0.230121 |  2.695594 |  2.992725 |  2.992725 | 12.469688 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 105.677143  105.677143  0.0  0.0  0.0 | 14.350194 | 45.794038 | 12.469688 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.837146  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.837146     0.0        0.0 |       0.0        0.0  10.837146     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.04364   0.04364  0.0  0.0  0.0 | 0.047161  0.047161  0.0  0.0  0.0 |  2.673916 |  0.5 |  2.772496 | 0.0  0.0  0.0  0.066044  0.0 | 0.231066 |  2.433658 |  2.701299 |  2.701299 | 11.255415 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 105.680664  105.680664  0.0  0.0  0.0 | 13.751614 | 45.982109 | 11.255415 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498     0.0        0.0 |       0.0        0.0   10.20498     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.042995  0.042995  0.0  0.0  0.0 |  0.04716   0.04716  0.0  0.0  0.0 |  2.515875 |  0.5 |  2.474215 | 0.0  0.0  0.0  0.065065  0.0 | 0.232007 |  2.195912 |  2.439034 |  2.439034 | 10.162641 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 105.684829  105.684829  0.0  0.0  0.0 | 13.293274 | 46.169379 | 10.162641 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599     0.0        0.0 |       0.0        0.0  10.385599     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.043182  0.043182  0.0  0.0  0.0 | 0.047158  0.047158  0.0  0.0  0.0 |  2.561031 |  0.5 |   2.29409 | 0.0  0.0  0.0  0.065345  0.0 | 0.232943 |  2.052686 |  2.216917 |  2.216917 |  9.237152 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 105.688805  105.688805  0.0  0.0  0.0 | 13.060215 | 46.355672 |  9.237152 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96_snow_redistribution_1:

snow redistribution 1
_____________________

Working with five zones allows demonstrating the redistribution of snow due to
gravitational and wind forcing. Therefore, we change the zone elevations so that the
first one (|FIELD|) is on 0 m, the second one (|FOREST|) on 1000 m, the third one on
3000 m (|GLACIER|), the fourth one (|ILAKE|) on 0 m, and the fifth one (|SEALED|) on
2000 m above sea level.  This random order is not ideal for clarity but demonstrates
that unsorted configurations do not harm the results:

>>> zonez(0.0, 10.0, 30.0, 0.0, 20.0)

We define a small snow threshold value of 200 mm to trigger snow redistribution despite
the short simulation period:

>>> smax(200.0)

We prepare the redistribution parameter |SRed| so that the slipped snow of one zone
deposits in the zone(s) directly below:

>>> sred(n_zones=1)

We define a constant air temperature of -0.4 °C for the subbasin.  However, there is a
substantial difference in the individual zone temperatures due to the high elevation
differences:

>>> inputs.t.series = -0.4

We double the basin's average precipitation by two and halve the altitude-related
precipitation adjustment.  However, there is still a marked difference for the
individual zones, which helps to stress the role of redistribution.

>>> pcorr *= 2.0
>>> pcalt(0.05)

In the following simulation results, the highest zone (|GLACIER|) stores the falling
precipitation as frozen snow (|SP|) and releases the amount exceeding 200 mm (|SPL|) to
the second-highest zone (|SEALED|). The |SEALED| zone receives too much snow itself.
Hence, the third-highest zone (|FOREST|) receives the snow released by both higher
zones (|SPG|), but only for a few hours.  Its available capacity also replenishes soon,
and it then starts releasing snow to the |FIELD| zone at sea level (|ILAKE| zones never
play a role in snow redistribution due to their missing snow module).  However, with
all zones except the |ILAKE| zone releasing snow towards the |FIELD| zone, its snow
cover grows rapidly and soon reaches the |SMax| threshold.  Then, the net transport due
to snow redistribution virtually stops, and the snow covers of all zones (again, except
for the |ILAKE| zone) grow with the same speed (see the documentation on method
|Calc_SPG_WCG_SP_WC_V1| for further details).

In the simulation period's second half, the melting of snow (only taking place in the
low elevation zones) outweighs snowfall (which dominates in the high elevation zones).
Still, snow ablation is at even speed across all zones due to the rebalancing action of
snow redistribution above |SRED| (again, please also read the documentation on method
|Calc_SPG_WCG_SP_WC_V1|):

.. integration-test::

    >>> test("hland_96_snow_redistribution_1",
    ...      axis1=(states.sp, states.wc), axis2=(factors.tc, fluxes.pc))
    |        date |    p |    t |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                 swe |                          gact | contriarea |                                        pc |                                ei |                                          tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                          melt |                               refr |                                    in_ |                                    r |                      sr |                      ea |                                cf |      inuz |     perc |       q0 |                      el |       q1 |     inrc |    outrc |       rt |       qt |                                  ic |                                                  sp |                                                  wc |                                    sm |       uz |        lz |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 |     0.05 | 0.046667 |  0.06037 |  0.06037 | 0.251543 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |      9.95 | 0.251543 |
    | 01/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.04975 | 0.046433 | 0.086244 | 0.086244 | 0.359352 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.90025 | 0.359352 |
    | 01/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049501 | 0.046201 | 0.046434 | 0.046434 | 0.193473 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.850749 | 0.193473 |
    | 01/01 03:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049254 |  0.04597 | 0.046201 | 0.046201 | 0.192506 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.801495 | 0.192506 |
    | 01/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049007 |  0.04574 |  0.04597 |  0.04597 | 0.191543 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.752488 | 0.191543 |
    | 01/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048762 | 0.045512 | 0.045741 | 0.045741 | 0.190586 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.703725 | 0.190586 |
    | 01/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048519 | 0.045284 | 0.045512 | 0.045512 | 0.189633 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.655206 | 0.189633 |
    | 01/01 07:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048276 | 0.045058 | 0.045284 | 0.045284 | 0.188685 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.60693 | 0.188685 |
    | 01/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048035 | 0.044832 | 0.045058 | 0.045058 | 0.187741 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.558896 | 0.187741 |
    | 01/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047794 | 0.044608 | 0.044833 | 0.044833 | 0.186802 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.511101 | 0.186802 |
    | 01/01 10:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047556 | 0.044385 | 0.044608 | 0.044608 | 0.185868 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.463546 | 0.185868 |
    | 01/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047318 | 0.044163 | 0.044385 | 0.044385 | 0.184939 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.416228 | 0.184939 |
    | 01/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047081 | 0.043942 | 0.044163 | 0.044163 | 0.184014 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.369147 | 0.184014 |
    | 01/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046846 | 0.043723 | 0.043943 | 0.043943 | 0.183094 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.322301 | 0.183094 |
    | 01/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046612 | 0.043504 | 0.043723 | 0.043723 | 0.182179 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.27569 | 0.182179 |
    | 01/01 15:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0    1.144   0.242       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046551 | 0.043448 |  0.04354 |  0.04354 | 0.181417 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.26371 | 0.181417 |
    | 01/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046319 | 0.043231 | 0.043412 | 0.043412 | 0.180884 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.217391 | 0.180884 |
    | 01/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046087 | 0.043014 | 0.043231 | 0.043231 | 0.180129 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.171304 | 0.180129 |
    | 01/01 18:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    1.224502        8.58  0.0       4.435 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |    0.0     1.894    7.436   1.573     4.435 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0   0.669498  0.0   1.573  0.0 |      0.0  0.167375  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.049749  0.0  0.0  0.0 |  0.039209 | 0.039209 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047148 | 0.044005 | 0.043283 | 0.043283 | 0.180344 | 1.815       2.0  0.0  0.0       2.0 |        0.0    0.874644        8.58  0.0       4.435 |        0.0    0.349858         0.0  0.0         0.0 |      100.0  100.551873  0.0  0.0  0.0 |      0.0 |  9.382478 | 0.180344 |
    | 01/01 19:00 |  5.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   10.249462      40.612  0.0      28.459 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  6.776  14.5376   32.032   6.776   24.024 | 0.0       0.0  0.0  0.0       0.0 |  6.591   14.5376   32.032   6.776    24.024 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     6.591    5.51264  0.0   6.776  0.0 |  1.64775  1.393413  0.0   6.776  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047528  0.047664  0.0  0.0  0.0 |  1.115342 |      0.5 | 0.000059 | 0.0  0.0  0.0  0.0  0.0 | 0.053895 |  0.05035 | 0.045195 | 0.045195 | 0.188312 |   2.0       2.0  0.0  0.0       2.0 |        0.0    7.321044      40.612  0.0      28.459 |        0.0    2.928418         0.0  0.0         0.0 | 104.990778  104.718764  0.0  0.0  0.0 | 0.615283 | 10.725154 | 0.188312 |
    | 01/01 20:00 |  2.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   14.950102        57.2  0.0        40.9 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  3.509   7.5284   16.588   3.509   12.441 | 0.0       0.0  0.0  0.0       0.0 |  3.509    7.5284   16.588   3.509    12.441 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     3.509    2.82776  0.0   3.509  0.0 | 0.966998  0.775232  0.0   3.509  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046234  0.046614  0.0  0.0  0.0 |  0.626524 |      0.5 | 0.000316 | 0.0  0.0  0.0  0.0  0.0 | 0.058275 | 0.054643 | 0.049894 | 0.049894 | 0.207891 |   2.0       2.0  0.0  0.0       2.0 |        0.0   10.678644        57.2  0.0        40.9 |        0.0    4.271458         0.0  0.0         0.0 | 107.579014  106.817906  0.0  0.0  0.0 | 0.741491 | 11.596736 | 0.207891 |
    | 01/01 21:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   22.853942      85.228  0.0      61.921 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     5.929    4.81656  0.0   5.929  0.0 | 1.715444  1.373932  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.044104   0.04487  0.0  0.0  0.0 |  1.139412 |      0.5 | 0.001313 | 0.0  0.0  0.0  0.0  0.0 | 0.064362 | 0.061121 | 0.055128 | 0.055128 | 0.229701 |   2.0       2.0  0.0  0.0       2.0 |        0.0   16.324244      85.228  0.0      61.921 |        0.0    6.529698         0.0  0.0         0.0 | 111.836673  110.305404  0.0  0.0  0.0 | 1.379591 | 12.807946 | 0.229701 |
    | 01/01 22:00 | 10.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   39.886902      145.86  0.0     107.395 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 | 12.826  27.5176   60.632  12.826   45.474 | 0.0       0.0  0.0  0.0       0.0 | 12.826   27.5176   60.632  12.826    45.474 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.826   10.48464  0.0  12.826  0.0 | 4.010511  3.189239  0.0  12.826  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.039674    0.0412  0.0  0.0  0.0 |  2.703862 |      0.5 | 0.018353 | 0.0  0.0  0.0  0.0  0.0 | 0.075344 | 0.085004 | 0.064989 | 0.064989 | 0.270786 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.490644      145.86  0.0     107.395 |        0.0   11.396258         0.0  0.0         0.0 | 120.691836  117.642005  0.0  0.0  0.0 |   3.5651 | 14.993459 | 0.270786 |
    | 01/01 23:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   40.103062     146.432  0.0     107.824 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121    0.04344  0.0   0.121  0.0 | 0.044064   0.01503  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.039616  0.041165  0.0  0.0  0.0 | -0.006858 |      0.5 | 0.035851 | 0.0  0.0  0.0  0.0  0.0 | 0.077197 | 0.100731 | 0.083191 | 0.083191 | 0.346631 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.645044     146.432  0.0     107.824 |        0.0   11.458018         0.0  0.0         0.0 | 120.808388   117.71158  0.0  0.0  0.0 | 3.022391 |  15.36212 | 0.346631 |
    | 02/01 00:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   41.280182     150.436  0.0     110.827 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.847    0.64008  0.0   0.847  0.0 | 0.309042  0.221724  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.039327  0.040935  0.0  0.0  0.0 |  0.172644 |      0.5 | 0.023113 | 0.0  0.0  0.0  0.0  0.0 | 0.079558 | 0.092745 | 0.095461 | 0.095461 | 0.397756 |   2.0       2.0  0.0  0.0       2.0 |        0.0   29.485844     150.436  0.0     110.827 |        0.0   11.794338         0.0  0.0         0.0 | 121.385673  118.170871  0.0  0.0  0.0 | 2.671922 | 15.832133 | 0.397756 |
    | 02/01 01:00 |  3.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   46.140982     167.596  0.0     123.697 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.63    7.788    17.16    3.63    12.87 | 0.0       0.0  0.0  0.0       0.0 |   3.63     7.788    17.16    3.63     12.87 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |      3.63     2.9272  0.0    3.63  0.0 | 1.337154  1.021911  0.0    3.63  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.038161  0.039962  0.0  0.0  0.0 |  0.868564 |      0.5 | 0.023168 | 0.0  0.0  0.0  0.0  0.0 | 0.083896 | 0.096838 | 0.095429 | 0.095429 | 0.397621 |   2.0       2.0  0.0  0.0       2.0 |        0.0   32.957844     167.596  0.0     123.697 |        0.0   13.183138         0.0  0.0         0.0 | 123.716679  120.116121  0.0  0.0  0.0 | 3.017318 | 16.695379 | 0.397621 |
    | 02/01 02:00 |  2.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   49.560342     179.608  0.0     132.706 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.541   5.4516   12.012   2.541    9.009 | 0.0       0.0  0.0  0.0       0.0 |  2.541    5.4516   12.012   2.541     9.009 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     2.541    2.03224  0.0   2.541  0.0 | 0.972302  0.733023  0.0   2.541  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.037357  0.039292  0.0  0.0  0.0 |  0.620804 |      0.5 |  0.02878 | 0.0  0.0  0.0  0.0  0.0 | 0.087435 |  0.10463 |  0.09766 |  0.09766 | 0.406915 |   2.0       2.0  0.0  0.0       2.0 |        0.0   35.400244     179.608  0.0     132.706 |        0.0   14.160098         0.0  0.0         0.0 | 125.322735  121.454631  0.0  0.0  0.0 | 3.109342 | 17.399516 | 0.406915 |
    | 02/01 03:00 | 10.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   99.386482       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 12.584  26.9984   59.488  12.584   44.616 | 0.0       0.0  0.0  0.0       0.0 | 12.584   26.9984   59.488  12.584    44.616 |        0.0         0.0      39.096  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0     23.6525         0.0  0.0      22.678 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.584    0.82476  0.0  12.584  0.0 | 4.941041  0.304156  0.0  12.584  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.033517  0.039012  0.0  0.0  0.0 |  2.133183 |      0.5 | 0.062162 | 0.0  0.0  0.0  0.0  0.0 | 0.098129 | 0.141317 | 0.111051 | 0.111051 | 0.462712 |   2.0       2.0  0.0  0.0       2.0 |        0.0   70.990344       200.0  0.0       200.0 |        0.0   28.396138         0.0  0.0         0.0 | 132.999211  122.014247  0.0  0.0  0.0 | 4.680362 | 19.527672 | 0.462712 |
    | 02/01 04:00 |  3.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  127.241232       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.235    9.086    20.02   4.235   15.015 | 0.0       0.0  0.0  0.0       0.0 |  4.235     9.086    20.02   4.235    15.015 |        0.0         0.0       20.02  0.0      15.015 |       0.0        0.0        0.0  0.0        0.0 |        0.0    18.76875         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.235        0.0  0.0   4.235  0.0 | 1.872801       0.0  0.0   4.235  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032319  0.038993  0.0  0.0  0.0 |   0.75387 |      0.5 | 0.107612 | 0.0  0.0  0.0  0.0  0.0 | 0.102806 | 0.182042 | 0.142214 | 0.142214 |  0.59256 |   2.0       2.0  0.0  0.0       2.0 |        0.0   93.803094       200.0  0.0       200.0 |        0.0   33.438138         0.0  0.0         0.0 |  135.39373   122.05324  0.0  0.0  0.0 | 4.826619 | 20.458438 |  0.59256 |
    | 02/01 05:00 |  3.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  154.300132       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.114   8.8264   19.448   4.114   14.586 | 0.0       0.0  0.0  0.0       0.0 |  4.114    8.8264   19.448   4.114    14.586 |        0.0         0.0      19.448  0.0      14.586 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.2325         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.114        0.0  0.0   4.114  0.0 | 1.885391       0.0  0.0   4.114  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.031189  0.038973  0.0  0.0  0.0 |  0.759593 |      0.5 | 0.117696 | 0.0  0.0  0.0  0.0  0.0 | 0.107374 | 0.194372 | 0.175732 | 0.175732 | 0.732218 |   2.0       2.0  0.0  0.0       2.0 |        0.0  115.965194       200.0  0.0       200.0 |        0.0   38.334938         0.0  0.0         0.0 | 137.653528  122.092214  0.0  0.0  0.0 | 4.968516 |  21.36735 | 0.732218 |
    | 02/01 06:00 |  1.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  163.850332       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.452   3.1152    6.864   1.452    5.148 | 0.0       0.0  0.0  0.0       0.0 |  1.452    3.1152    6.864   1.452     5.148 |        0.0         0.0       6.864  0.0       5.148 |       0.0        0.0        0.0  0.0        0.0 |        0.0       6.435         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     1.452        0.0  0.0   1.452  0.0 |  0.68783       0.0  0.0   1.452  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030791  0.038954  0.0  0.0  0.0 |  0.260782 |      0.5 |  0.11013 | 0.0  0.0  0.0  0.0  0.0 | 0.110017 | 0.190787 | 0.190835 | 0.190835 | 0.795147 |   2.0       2.0  0.0  0.0       2.0 |        0.0  123.812994       200.0  0.0       200.0 |        0.0   40.037338         0.0  0.0         0.0 | 138.448488  122.131167  0.0  0.0  0.0 | 4.619168 | 21.893333 | 0.795147 |
    | 02/01 07:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 | 0.057983       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030744  0.038934  0.0  0.0  0.0 | -0.001629 |      0.5 | 0.081154 | 0.0  0.0  0.0  0.0  0.0 | 0.111696 | 0.169173 |  0.18678 |  0.18678 | 0.778251 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.503644       200.0  0.0       200.0 |        0.0   40.142538         0.0  0.0         0.0 | 138.542249  122.170102  0.0  0.0  0.0 | 4.036385 | 22.227494 | 0.778251 |
    | 02/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030729  0.038915  0.0  0.0  0.0 | -0.025775 |      0.5 | 0.052742 | 0.0  0.0  0.0  0.0  0.0 |  0.11328 | 0.147922 | 0.169253 | 0.169253 | 0.705222 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.543644       200.0  0.0       200.0 |        0.0   40.102538         0.0  0.0         0.0 | 138.572978  122.209017  0.0  0.0  0.0 | 3.457868 | 22.542785 | 0.705222 |
    | 02/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030714  0.038895  0.0  0.0  0.0 | -0.025762 |      0.5 | 0.032253 | 0.0  0.0  0.0  0.0  0.0 | 0.114857 | 0.133002 | 0.149329 | 0.149329 | 0.622202 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.583644       200.0  0.0       200.0 |        0.0   40.062538         0.0  0.0         0.0 | 138.603692  122.247912  0.0  0.0  0.0 | 2.899853 |   22.8565 | 0.622202 |
    | 02/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  167.829582       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 |        0.0         0.0       2.288  0.0       1.716 |       0.0        0.0        0.0  0.0        0.0 |        0.0       2.145         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.484        0.0  0.0   0.484  0.0 | 0.232453       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030572  0.038876  0.0  0.0  0.0 |  0.071158 |      0.5 | 0.019232 | 0.0  0.0  0.0  0.0  0.0 | 0.116771 | 0.124372 |   0.1344 |   0.1344 | 0.559998 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.226244       200.0  0.0       200.0 |        0.0   40.603338         0.0  0.0         0.0 | 138.885811  122.286788  0.0  0.0  0.0 |  2.45178 | 23.237443 | 0.559998 |
    | 02/01 11:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  168.625432       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 |  0.05835       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030526  0.038857  0.0  0.0  0.0 | -0.001359 |      0.5 | 0.010686 | 0.0  0.0  0.0  0.0  0.0 | 0.118417 | 0.119071 | 0.125112 | 0.125112 | 0.521298 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.916894       200.0  0.0       200.0 |        0.0   40.708538         0.0  0.0         0.0 | 138.978987  122.325645  0.0  0.0  0.0 | 1.939735 | 23.564884 | 0.521298 |
    | 02/01 12:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  197.276032       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |        0.0         0.0      20.592  0.0      15.444 |       0.0        0.0        0.0  0.0        0.0 |        0.0      19.305         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.356        0.0  0.0   4.356  0.0 | 2.103421       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.029384  0.038837  0.0  0.0  0.0 |  0.851236 |      0.5 | 0.009493 | 0.0  0.0  0.0  0.0  0.0 | 0.123079 | 0.122468 | 0.121004 | 0.121004 | 0.504182 |   2.0       2.0  0.0  0.0       2.0 |        0.0  151.380294       200.0  0.0       200.0 |        0.0   45.895738         0.0  0.0         0.0 |  141.26095  122.364482  0.0  0.0  0.0 | 2.281478 | 24.492662 | 0.504182 |
    | 02/01 13:00 |  5.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  42.523945       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.139  15.3164   33.748   7.139   25.311 | 0.0       0.0  0.0  0.0       0.0 |  7.139   15.3164   33.748   7.139    25.311 |        0.0    9.366467      33.748  0.0      25.311 |       0.0   3.225965        0.0  0.0        0.0 |  32.804173         0.0         0.0  0.0         0.0 |  2.580772        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0   7.139  0.0 |      0.0       0.0  0.0   7.139  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.02937  0.038818  0.0  0.0  0.0 | -0.025177 |      0.5 | 0.008283 | 0.0  0.0  0.0  0.0  0.0 | 0.129705 | 0.127685 | 0.122872 | 0.122872 | 0.511968 |   2.0       2.0  0.0  0.0       2.0 |   30.45818  148.803428       200.0  0.0       200.0 |  12.065766   51.196572         0.0  0.0         0.0 |  141.29032    122.4033  0.0  0.0  0.0 | 1.748018 | 25.811385 | 0.511968 |
    | 02/01 14:00 |  1.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  48.329573       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.331   2.8556    6.292   1.331    4.719 | 0.0       0.0  0.0  0.0       0.0 |  1.331    2.8556    6.292   1.331     4.719 |        0.0    2.112422       6.292  0.0       4.719 |       0.0   0.743178        0.0  0.0        0.0 |   6.408937         0.0         0.0  0.0         0.0 |  0.594543        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |  2.528852        0.0  0.0   1.331  0.0 | 1.262084       0.0  0.0   1.331  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028721  0.038798  0.0  0.0  0.0 |  0.500968 |      0.5 | 0.005322 | 0.0  0.0  0.0  0.0  0.0 |  0.13215 | 0.127598 | 0.126506 | 0.126506 |  0.52711 |   2.0       2.0  0.0  0.0       2.0 |  34.521124  147.989406       200.0  0.0       200.0 |  13.808449   52.010594         0.0  0.0         0.0 |  142.58581  122.442098  0.0  0.0  0.0 | 1.743665 | 26.297949 |  0.52711 |
    | 02/01 15:00 | 20.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 201.988205  191.984574  201.988205  0.0  201.988205 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 25.047  53.7372  118.404  25.047   88.803 | 0.0       0.0  0.0  0.0       0.0 | 25.047   53.7372  118.404  25.047    88.803 |        0.0   36.356814     118.404  0.0      88.803 |       0.0  17.380386        0.0  0.0        0.0 | 115.042936    1.778447    1.778447  0.0    1.778447 | 13.568696   0.209758   0.209758  0.0   0.209758 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.209758  0.0  0.209758 |       0.0  10.003631  0.0  25.047  0.0 |      0.0  3.749378  0.0  25.047  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028707  0.035652  0.0  0.0  0.0 |  1.225947 |      0.5 | 0.009615 | 0.0  0.0  0.0  0.0  0.0 | 0.151523 | 0.149114 | 0.132398 | 0.132398 |  0.55166 |   2.0       2.0  0.0  0.0       2.0 | 147.218067  137.131839  201.988205  0.0  201.988205 |  54.770138   54.852736         0.0  0.0         0.0 | 142.614517  128.732003  0.0  0.0  0.0 | 2.459997 |  30.15314 |  0.55166 |
    | 02/01 16:00 | 37.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 290.938862  302.004664  309.356596  0.0  309.356596 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 45.859  98.3884  216.788  45.859  162.591 | 0.0       0.0  0.0  0.0       0.0 | 45.859   98.3884  216.788  45.859   162.591 |  28.420627   56.173848  218.776205  0.0  164.579205 | 19.426578  34.199126        0.0  0.0        0.0 |  91.362027   91.362027   91.362027  0.0   91.362027 | 17.994569  17.994569  17.994569  0.0  17.994569 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 | 18.417734   7.351931  0.0  45.859  0.0 | 9.364911  3.045892  0.0  45.859  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024166  0.033481  0.0  0.0  0.0 |  4.896114 |      0.5 | 0.121417 | 0.0  0.0  0.0  0.0  0.0 | 0.185665 | 0.270421 |  0.17129 |  0.17129 | 0.713706 |   2.0       2.0  0.0  0.0       2.0 | 207.813473  215.717617  292.002027  0.0  291.702027 |  83.125389   86.287047   17.354569  0.0   17.654569 | 151.691505  133.071523  0.0  0.0  0.0 | 6.734694 | 36.947332 | 0.713706 |
    | 02/01 17:00 |  8.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 323.906781  323.906781  323.906781  0.0  323.906781 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  9.922  21.2872   46.904   9.922   35.178 | 0.0       0.0  0.0  0.0       0.0 |  9.922   21.2872   46.904   9.922    35.178 |  69.667573   85.844423  148.648653  0.0  137.128384 | 31.193289  37.447441   7.611943  0.0   7.406211 |  98.060761   98.060761   98.060761  0.0   98.060761 | 25.846019  25.846019  25.846019  0.0  25.846019 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.922  0.0 |      0.0       0.0  0.0   9.922  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024154  0.033464  0.0  0.0  0.0 | -0.021219 |      0.5 | 0.255461 | 0.0  0.0  0.0  0.0  0.0 | 0.193967 | 0.385404 | 0.269016 | 0.269016 | 1.120898 |   2.0       2.0  0.0  0.0       2.0 | 233.860668  237.354756  288.958135  0.0  288.152404 |  90.046113   86.552025   34.948645  0.0   35.754377 |  151.71566  133.104988  0.0  0.0  0.0 | 5.958015 | 38.599366 | 1.120898 |
    | 02/01 18:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 334.397735  334.397735  334.397735  0.0  334.397735 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |  91.376852   96.554068  129.839697  0.0  124.668627 | 36.885929  36.698313  14.659084  0.0  14.682154 | 104.406788  104.406788  104.406788  0.0  104.406788 | 29.990946  29.990946  29.990946  0.0  29.990946 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   4.356  0.0 |      0.0       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024142  0.033448  0.0  0.0  0.0 | -0.021208 |      0.5 | 0.176483 | 0.0  0.0  0.0  0.0  0.0 | 0.198251 | 0.326221 |   0.3467 |   0.3467 | 1.444585 |   2.0       2.0  0.0  0.0       2.0 | 244.544611  249.365877  284.757227  0.0  283.674565 |  89.853124   85.031858   49.640507  0.0   50.723169 | 151.739802  133.138435  0.0  0.0  0.0 | 5.260324 | 39.451972 | 1.444585 |
    | 02/01 19:00 |  7.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 356.253888  356.253888  356.253888  0.0  356.253888 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  9.075    19.47     42.9   9.075   32.175 | 0.0       0.0  0.0  0.0       0.0 |  9.075     19.47     42.9   9.075    32.175 | 102.149255  112.159273  153.970933  0.0  143.523838 | 41.323479  41.708462  23.326802  0.0  23.048897 | 120.370769  120.370769  120.370769  0.0  120.370769 | 35.883119  35.883119  35.883119  0.0  35.883119 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.075  0.0 |      0.0       0.0  0.0   9.075  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.02413  0.033431  0.0  0.0  0.0 | -0.021198 |      0.5 | 0.120588 | 0.0  0.0  0.0  0.0  0.0 | 0.205885 |  0.28863 | 0.331019 | 0.331019 | 1.379245 |   2.0       2.0  0.0  0.0       2.0 | 260.420132  266.197374  294.697064  0.0  293.036497 |  95.833757   90.056515   61.556824  0.0   63.217392 | 151.763932  133.171866  0.0  0.0  0.0 | 4.618538 | 40.971087 | 1.379245 |
    | 02/01 20:00 | 18.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 410.165735  410.165735  410.165735  0.0  410.165735 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 22.385   48.026   105.82  22.385   79.365 | 0.0       0.0  0.0  0.0       0.0 | 22.385    48.026   105.82  22.385    79.365 | 122.864197  145.201755  227.160779  0.0   201.42567 | 55.774692  59.078133   34.91311  0.0  34.193218 | 159.848924  159.848924  159.848924  0.0  159.848924 | 50.316811  50.316811  50.316811  0.0  50.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  22.385  0.0 |      0.0       0.0  0.0  22.385  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024118  0.033414  0.0  0.0  0.0 | -0.021187 |      0.5 | 0.080601 | 0.0  0.0  0.0  0.0  0.0 | 0.222988 | 0.272602 | 0.293421 | 0.293421 |  1.22259 |   2.0       2.0  0.0  0.0       2.0 | 295.058865  302.048542  333.845209  0.0   331.16475 | 115.106869  108.117192   76.320526  0.0   79.000985 |  151.78805   133.20528  0.0  0.0  0.0 |  4.01675 | 44.374528 |  1.22259 |
    | 02/01 21:00 | 15.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 455.043704  455.043704  455.043704  0.0  455.043704 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 18.634  39.9784   88.088  18.634   66.066 | 0.0       0.0  0.0  0.0       0.0 | 18.634   39.9784   88.088  18.634    66.066 | 157.438041  177.637793  252.568414  0.0  230.408289 | 71.361694  72.506342  45.685321  0.0  45.823446 | 191.219608  191.219608  191.219608  0.0  191.219608 | 63.824096  63.824096  63.824096  0.0  63.824096 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  18.634  0.0 |      0.0       0.0  0.0  18.634  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024106  0.033397  0.0  0.0  0.0 | -0.021177 |      0.5 | 0.052027 | 0.0  0.0  0.0  0.0  0.0 | 0.237325 | 0.263125 | 0.274058 | 0.274058 | 1.141908 |   2.0       2.0  0.0  0.0       2.0 | 326.494439  333.287958  361.224403  0.0  358.382069 | 128.549264  121.755746     93.8193  0.0   96.661635 | 151.812156  133.238677  0.0  0.0  0.0 | 3.443547 | 47.227774 | 1.141908 |
    | 02/01 22:00 |  6.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 473.402873  473.402873  473.402873  0.0  473.402873 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.623  16.3548   36.036   7.623   27.027 | 0.0       0.0  0.0  0.0       0.0 |  7.623   16.3548   36.036   7.623    27.027 | 185.358526  196.033453  235.469802  0.0  225.511749 | 77.308178  75.365051  55.609902  0.0  56.558955 | 203.295969  203.295969  203.295969  0.0  203.295969 | 70.106904  70.106904  70.106904  0.0  70.106904 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   7.623  0.0 |      0.0       0.0  0.0   7.623  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024094  0.033381  0.0  0.0  0.0 | -0.021166 |      0.5 | 0.031892 | 0.0  0.0  0.0  0.0  0.0 | 0.243727 | 0.252992 | 0.262979 | 0.262979 | 1.095747 |   2.0       2.0  0.0  0.0       2.0 | 342.085889  347.797674   365.72657  0.0  363.533289 | 131.316984  125.605199  107.676303  0.0  109.869584 |  151.83625  133.272058  0.0  0.0  0.0 | 2.890489 | 48.501618 | 1.095747 |
    | 02/01 23:00 |  1.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 478.939765  478.939765  478.939765  0.0  478.939765 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.299   4.9324   10.868   2.299    8.151 | 0.0       0.0  0.0  0.0       0.0 |  2.299    4.9324   10.868   2.299     8.151 | 198.262243  203.642415  221.064023  0.0  217.315563 |  77.43963  74.692858   63.20685  0.0   64.23831 |  206.64527   206.64527   206.64527  0.0   206.64527 | 72.294496  72.294496  72.294496  0.0  72.294496 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.299  0.0 |      0.0       0.0  0.0   2.299  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024082  0.033364  0.0  0.0  0.0 | -0.021155 |      0.5 | 0.018122 | 0.0  0.0  0.0  0.0  0.0 | 0.246293 | 0.244371 | 0.253328 | 0.253328 | 1.055533 |   2.0       2.0  0.0  0.0       2.0 | 348.122922  353.014129  362.815817  0.0  361.353996 | 130.816843  125.925637  116.123948  0.0   117.58577 | 151.860332  133.305422  0.0  0.0  0.0 | 2.351212 | 49.012325 | 1.055533 |
    | 03/01 00:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 493.219119  493.219119  493.219119  0.0  493.219119 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 | 204.528223  212.738588  236.654993  0.0  229.412995 | 80.340542  78.921577  70.312772  0.0  70.547771 | 216.382573  216.382573  216.382573  0.0  216.382573 | 76.836547  76.836547  76.836547  0.0  76.836547 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   5.929  0.0 |      0.0       0.0  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.02407  0.033347  0.0  0.0  0.0 | -0.021145 |      0.5 | 0.009183 | 0.0  0.0  0.0  0.0  0.0 | 0.251439 | 0.242023 | 0.245765 | 0.245765 | 1.024021 |   2.0       2.0  0.0  0.0       2.0 | 357.630288  362.303713  371.211396  0.0  369.684574 | 135.588831  130.915406  122.007723  0.0  123.534546 | 151.884402  133.338769  0.0  0.0  0.0 | 1.820884 | 50.036457 | 1.024021 |
    | 03/01 01:00 |  2.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 501.087335  501.087335  501.087335  0.0  501.087335 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.267   7.0092   15.444   3.267   11.583 | 0.0       0.0  0.0  0.0       0.0 |  3.267    7.0092   15.444   3.267    11.583 | 213.565721  219.302218  234.627313  0.0  230.211324 | 82.920398  80.926101  74.035807  0.0  74.590795 | 221.471596  221.471596  221.471596  0.0  221.471596 | 79.615739  79.615739  79.615739  0.0  79.615739 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   3.267  0.0 |      0.0       0.0  0.0   3.267  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024058  0.033331  0.0  0.0  0.0 | -0.021134 |      0.5 | 0.003871 | 0.0  0.0  0.0  0.0  0.0 | 0.254659 | 0.240778 | 0.242268 | 0.242268 | 1.009451 |   2.0       2.0  0.0  0.0       2.0 | 363.189179   367.60189  374.139679  0.0  372.867845 | 137.898156  133.485444  126.947655  0.0   128.21949 | 151.908459    133.3721  0.0  0.0  0.0 | 1.295879 | 50.677084 | 1.009451 |
    | 03/01 02:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 502.544412  502.544412  502.544412  0.0  502.544412 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 218.403559  221.603573  227.380998  0.0  225.971212 | 83.288776  80.781762  76.566337  0.0  77.261123 | 222.042022  222.042022  222.042022  0.0  222.042022 | 80.502389  80.502389  80.502389  0.0  80.502389 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024046  0.033314  0.0  0.0  0.0 | -0.021124 |      0.5 | 0.001169 | 0.0  0.0  0.0  0.0  0.0 |  0.25596 | 0.239832 | 0.240845 | 0.240845 | 1.003519 |   2.0       2.0  0.0  0.0       2.0 | 364.480659   368.65234  372.300704  0.0  371.423655 | 138.063753  133.892072  130.243708  0.0  131.120756 | 151.932505  133.405414  0.0  0.0  0.0 | 0.773586 | 50.936124 | 1.003519 |
    | 03/01 03:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 503.127242  503.127242  503.127242  0.0  503.127242 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 219.496367  222.227265  225.160687  0.0  224.375469 | 83.290045  80.836347  78.527724  0.0  79.026943 |  222.01911   222.01911   222.01911  0.0   222.01911 | 81.108133  81.108133  81.108133  0.0  81.108133 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024034  0.033297  0.0  0.0  0.0 | -0.021113 |      0.5 | 0.000168 | 0.0  0.0  0.0  0.0  0.0 | 0.256996 | 0.239997 | 0.240079 | 0.240079 | 1.000329 |   2.0       2.0  0.0  0.0       2.0 | 364.656418  368.712985  370.943126  0.0  370.265296 | 138.470825  134.414258  132.184116  0.0  132.861946 | 151.956539  133.438711  0.0  0.0  0.0 | 0.252305 |  51.14227 | 1.000329 |
    | 03/01 04:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 504.584319  504.584319  504.584319  0.0  504.584319 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 219.874572  222.866862   226.05113  0.0  225.000538 |  83.85767   81.55838  79.936113  0.0  80.271705 | 222.614941  222.614941  222.614941  0.0  222.614941 | 81.969378  81.969378  81.969378  0.0  81.969378 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024022  0.033281  0.0  0.0  0.0 | -0.021103 | 0.231201 | 0.000002 | 0.0  0.0  0.0  0.0  0.0 | 0.257134 | 0.239993 |  0.23996 |  0.23996 | 0.999832 |   2.0       2.0  0.0  0.0       2.0 | 365.049803  369.073064  371.006938  0.0    370.3647 | 139.534516  135.511255  133.577381  0.0  134.219619 | 151.980561  133.471992  0.0  0.0  0.0 |      0.0 | 51.169737 | 0.999832 |
    | 03/01 05:00 |  2.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 511.578288  511.578288  511.578288  0.0  511.578288 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.904   6.2304   13.728   2.904   10.296 | 0.0       0.0  0.0  0.0       0.0 |  2.904    6.2304   13.728   2.904    10.296 |   221.1845  226.239983  236.278139  0.0   232.79694 | 86.303819  84.574736   82.03418  0.0  82.083379 | 227.116446  227.116446  227.116446  0.0  227.116446 | 84.461843  84.461843  84.461843  0.0  84.461843 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.904  0.0 |      0.0       0.0  0.0   2.904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.257923 | 0.240728 | 0.240158 | 0.240158 | 1.000657 |   2.0       2.0  0.0  0.0       2.0 | 368.634765  372.735126  376.213244  0.0  375.320206 | 142.943524  138.843162  135.365044  0.0  136.258083 | 151.980561  133.471992  0.0  0.0  0.0 |      0.0 | 51.326671 | 1.000657 |
    | 03/01 06:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  512.74395   512.74395   512.74395  0.0   512.74395 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 224.654326  227.589692  231.186173  0.0  230.127809 | 87.407963  85.026997  82.680115  0.0   83.16648 | 227.485902  227.485902  227.485902  0.0  227.485902 | 85.258048  85.258048  85.258048  0.0  85.258048 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.484  0.0 |      0.0       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.256979 | 0.239847 | 0.240369 | 0.240369 | 1.001538 |   2.0       2.0  0.0  0.0       2.0 | 369.119357  373.128936  375.440973  0.0  374.734299 | 143.624593  139.615014  137.302977  0.0  138.009651 | 151.980561  133.471992  0.0  0.0  0.0 |      0.0 | 51.138834 | 1.001538 |
    | 03/01 07:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 225.209233  227.873792  230.021905  0.0  229.334171 | 87.776717  85.389358  83.866045  0.0  84.267779 | 227.457017  227.457017  227.457017  0.0  227.457017 | 85.869764  85.869764  85.869764  0.0  85.869764 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255867 | 0.238809 | 0.239812 | 0.239812 | 0.999218 |   2.0       2.0  0.0  0.0       2.0 | 369.020157  372.980961  374.660085  0.0  374.055145 | 144.306624   140.34582  138.666696  0.0  139.271636 | 151.980561  133.471992  0.0  0.0  0.0 |      0.0 | 50.917539 | 0.999218 |
    | 03/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.244235  227.661848   228.68676  0.0  228.317514 | 88.082546  85.664932  84.640021  0.0  85.009267 | 227.018951  227.018951  227.018951  0.0  227.018951 |  86.30783   86.30783   86.30783  0.0   86.30783 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254588 | 0.237615 | 0.238775 | 0.238775 | 0.994894 |   2.0       2.0  0.0  0.0       2.0 |  368.44789  372.378063  373.632276  0.0  373.096582 | 144.878891  140.948718  139.694505  0.0  140.230199 | 151.980561  133.471992  0.0  0.0  0.0 |      0.0 | 50.662951 | 0.994894 |
    | 03/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.894931  227.293849  228.059401  0.0  227.732422 | 88.431849  86.032932   85.26738  0.0  85.594359 | 226.581591  226.581591  226.581591  0.0  226.581591 |  86.74519   86.74519   86.74519  0.0   86.74519 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.253315 | 0.236427 | 0.237617 | 0.237617 | 0.990069 |   2.0       2.0  0.0  0.0       2.0 | 367.787565  371.705805  372.794465  0.0  372.285751 | 145.539216  141.620976  140.532315  0.0   141.04103 | 151.980561  133.471992  0.0  0.0  0.0 |      0.0 | 50.409636 | 0.990069 |
    | 03/01 10:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.201027  514.201027  514.201027  0.0  514.201027 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 |  224.59314  227.312585  229.081589  0.0  228.387074 | 89.096641  86.792995  85.961192  0.0  86.226707 | 226.757529  226.757529  226.757529  0.0  226.757529 | 87.443498  87.443498  87.443498  0.0  87.443498 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.363  0.0 |      0.0       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.252307 | 0.235487 | 0.236482 | 0.236482 | 0.985343 |   2.0       2.0  0.0  0.0       2.0 | 367.604971  371.533949  372.826406  0.0  372.283206 | 146.596056  142.667078  141.374621  0.0  141.917821 | 151.980561  133.471992  0.0  0.0  0.0 |      0.0 | 50.209186 | 0.985343 |
    | 03/01 11:00 |  2.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 519.961252  521.777827  521.777827  0.0  521.777827 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.146   6.7496   14.872   3.146   11.154 | 0.0       0.0  0.0  0.0       0.0 |  3.146    6.7496   14.872   3.146    11.154 | 225.493409  230.729522  241.140791  0.0  237.464602 | 91.853618  90.221105  87.932236  0.0  87.890425 | 231.636316  231.636316  231.636316  0.0  231.636316 | 90.141511  90.141511  90.141511  0.0  90.141511 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.816574        0.0  0.0   3.146  0.0 | 1.048985       0.0  0.0   3.146  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023626       0.0  0.0  0.0  0.0 |  0.427233 | 0.427233 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255124 | 0.238116 |  0.23628 |  0.23628 |   0.9845 |   2.0       2.0  0.0  0.0       2.0 | 371.400895  375.455142  378.833931  0.0   377.94892 | 148.560358  146.322685  142.943896  0.0  143.828907 | 152.771776  133.471992  0.0  0.0  0.0 |      0.0 |  50.76969 |   0.9845 |
    | 03/01 12:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.773236  523.119052  523.119052  0.0  523.119052 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 228.776083   232.53573  237.211776  0.0  235.766768 |  92.03217  91.059297  88.570051  0.0  89.014059 | 232.417341  232.417341  232.417341  0.0  232.417341 | 90.701711  90.701711  90.701711  0.0  90.701711 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.345816        0.0  0.0   0.847  0.0 | 0.785257       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023334       0.0  0.0  0.0  0.0 |  0.317468 | 0.317468 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255814 |  0.23876 | 0.237675 | 0.237675 | 0.990311 |   2.0       2.0  0.0  0.0       2.0 | 372.695169  376.177553  378.683497  0.0  377.942492 | 149.078068  146.941499  144.435556  0.0   145.17656 | 153.355668  133.471992  0.0  0.0  0.0 |      0.0 | 50.906992 | 0.990311 |
    | 03/01 13:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.992672  523.475677  523.475677  0.0  523.475677 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 229.937343  232.782522  235.439858  0.0  234.597488 | 92.198893   91.11533  89.395194  0.0  89.808564 | 232.441067  232.441067  232.441067  0.0  232.441067 |  91.03461   91.03461   91.03461  0.0   91.03461 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.483005        0.0  0.0   0.363  0.0 | 0.871931       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023017       0.0  0.0  0.0  0.0 |  0.353714 | 0.353714 |      0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.25631 | 0.239223 |  0.23872 |  0.23872 | 0.994665 |   2.0       2.0  0.0  0.0       2.0 | 372.851909  376.219298  378.040705  0.0  377.413071 | 149.140763  147.256379  145.434972  0.0  146.062606 | 153.989759  133.471992  0.0  0.0  0.0 |      0.0 | 51.005722 | 0.994665 |
    | 03/01 14:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 522.008863  523.779537  523.779537  0.0  523.779537 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 230.094041  232.906119  235.140283  0.0  234.368133 | 92.261631  91.348357  90.051394  0.0  90.394544 |  232.45259   232.45259   232.45259  0.0   232.45259 | 91.326947  91.326947  91.326947  0.0  91.326947 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.770674        0.0  0.0   0.363  0.0 | 1.049693       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022645       0.0  0.0  0.0  0.0 |  0.427937 | 0.427937 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.257122 |  0.23998 | 0.239288 | 0.239288 | 0.997034 |   2.0       2.0  0.0  0.0       2.0 | 372.863473  376.148968  377.709011  0.0  377.124528 | 149.145389  147.630568  146.070525  0.0  146.655009 | 154.733384  133.471992  0.0  0.0  0.0 |      0.0 | 51.167261 | 0.997034 |
    | 03/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.265413  523.098508  523.098508  0.0  523.098508 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 230.006331  232.520231  233.484587  0.0  233.123282 | 92.002532  91.259306   90.29495  0.0  90.656254 | 231.822278  231.822278  231.822278  0.0  231.822278 |  91.27623   91.27623   91.27623  0.0   91.27623 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.833096        0.0  0.0     0.0  0.0 | 1.097219       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022265       0.0  0.0  0.0  0.0 |  0.447897 | 0.447897 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.257756 | 0.240572 | 0.239944 | 0.239944 | 0.999765 |   2.0       2.0  0.0  0.0       2.0 | 372.332438  375.491016  376.686703  0.0  376.163524 | 148.932975  147.607492  146.411805  0.0  146.934984 | 155.491526  133.471992  0.0  0.0  0.0 |      0.0 | 51.293417 | 0.999765 |
    | 03/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 520.375077  522.393471  522.393471  0.0  522.393471 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 229.475295  231.926846  232.665377  0.0  232.342229 | 91.790118  91.171662  90.433131  0.0  90.756279 | 231.186324  231.186324  231.186324  0.0  231.186324 | 91.207147  91.207147  91.207147  0.0  91.207147 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.018394        0.0  0.0     0.0  0.0 | 1.219999       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021855       0.0  0.0  0.0  0.0 |  0.499227 | 0.499227 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.258607 | 0.241366 | 0.240617 | 0.240617 | 1.002571 |   2.0       2.0  0.0  0.0       2.0 | 371.696484  374.790495   375.84765  0.0  375.347619 | 148.678593  147.602977  146.545821  0.0  147.045852 | 156.311777  133.471992  0.0  0.0  0.0 |      0.0 | 51.462719 | 1.002571 |
    | 03/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  519.45772  521.617166  521.617166  0.0  521.617166 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.839341  231.300763  231.953183  0.0  231.644591 | 91.535736  91.092708  90.440288  0.0  90.748881 | 230.531069  230.531069  230.531069  0.0  230.531069 | 91.086097  91.086097  91.086097  0.0  91.086097 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.159446        0.0  0.0     0.0  0.0 | 1.319064       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021424       0.0  0.0  0.0  0.0 |  0.540683 |      0.5 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.259456 | 0.242159 | 0.241366 | 0.241366 | 1.005692 |   2.0       2.0  0.0  0.0       2.0 | 371.041228    374.0608  375.065536  0.0  374.574097 | 148.416491  147.556365   146.55163  0.0  147.043069 | 157.173583  133.471992  0.0  0.0  0.0 | 0.040683 | 51.631834 | 1.005692 |
    | 03/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.519709   520.78661   520.78661  0.0   520.78661 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.184085    230.6373  231.256796  0.0  230.953787 | 91.273634  90.979866   90.36037  0.0  90.663379 | 229.861062  229.861062  229.861062  0.0  229.861062 | 90.925548  90.925548  90.925548  0.0  90.925548 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |    2.2669        0.0  0.0     0.0  0.0 | 1.400011       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.02098  0.033264  0.0  0.0  0.0 |  0.563509 |      0.5 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.260302 | 0.242949 | 0.242159 | 0.242159 | 1.008994 |   2.0       2.0  0.0  0.0       2.0 | 370.371221  373.324563  374.309801  0.0  373.821372 | 148.148488  147.462047  146.476809  0.0  146.965237 | 158.061452  133.505256  0.0  0.0  0.0 | 0.104191 | 51.800103 | 1.008994 |
    | 03/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 517.565962  519.914725  519.914725  0.0  519.914725 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 227.514078  229.955069  230.561942  0.0  230.261087 | 91.005631  90.831541  90.224668  0.0  90.525523 | 229.179814  229.179814  229.179814  0.0  229.179814 | 90.734911  90.734911  90.734911  0.0  90.734911 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.348763        0.0  0.0     0.0  0.0 | 1.467003       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020528  0.033247  0.0  0.0  0.0 |  0.591615 |      0.5 | 0.000004 | 0.0  0.0  0.0  0.0  0.0 | 0.261143 | 0.243737 | 0.242949 | 0.242949 | 1.012286 |   2.0       2.0  0.0  0.0       2.0 | 369.689973  372.589307  373.567673  0.0  373.080099 | 147.875989  147.325418  146.347052  0.0  146.834625 |  158.96374  133.538503  0.0  0.0  0.0 | 0.195803 | 51.967531 | 1.012286 |
    | 03/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 516.600219  519.011355  519.011355  0.0  519.011355 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  226.83283  229.262223  229.864233  0.0  229.564218 | 90.733132  90.652502  90.050492  0.0  90.350507 | 228.489997  228.489997  228.489997  0.0  228.489997 | 90.521358  90.521358  90.521358  0.0  90.521358 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.411136        0.0  0.0     0.0  0.0 | 1.523203       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020074  0.033231  0.0  0.0  0.0 |  0.615227 |      0.5 | 0.000017 | 0.0  0.0  0.0  0.0  0.0 | 0.261981 | 0.244529 | 0.243738 | 0.243738 | 1.015574 |   2.0       2.0  0.0  0.0       2.0 | 369.000156  371.857081  372.833437  0.0  372.345878 | 147.600062  147.154274  146.177917  0.0  146.665476 | 159.871747  133.571734  0.0  0.0  0.0 | 0.311013 | 52.134122 | 1.015574 |
    | 03/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 515.625325  518.083995  518.083995  0.0  518.083995 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 226.143013   228.56269  229.162809  0.0   228.86313 | 90.457205  90.448665  89.848546  0.0  90.148225 | 227.793645  227.793645  227.793645  0.0  227.793645 |  90.29035   90.29035   90.29035  0.0   90.29035 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.458669        0.0  0.0     0.0  0.0 | 1.571027       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01962  0.033214  0.0  0.0  0.0 |  0.635348 |      0.5 | 0.000056 | 0.0  0.0  0.0  0.0  0.0 | 0.262813 | 0.245338 | 0.244533 | 0.244533 | 1.018886 |   2.0       2.0  0.0  0.0       2.0 | 368.303804  371.128036  372.104273  0.0  371.616393 | 147.321522  146.955959  145.979722  0.0  146.467601 |  160.77901  133.604948  0.0  0.0  0.0 | 0.446304 |  52.29988 | 1.018886 |
    | 03/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.643451  517.138353  517.138353  0.0  517.138353 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.446661  227.858589  228.457962  0.0  228.158422 | 90.178664  90.225405  89.626033  0.0  89.925572 | 227.092305  227.092305  227.092305  0.0  227.092305 | 90.046047  90.046047  90.046047  0.0  90.046047 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.494902        0.0  0.0     0.0  0.0 | 1.612324       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019169  0.033198  0.0  0.0  0.0 |  0.652748 |      0.5 | 0.000146 | 0.0  0.0  0.0  0.0  0.0 | 0.263642 | 0.246183 | 0.245346 | 0.245346 | 1.022274 |   2.0       2.0  0.0  0.0       2.0 | 367.602465  370.401752  371.378616  0.0  370.890276 | 147.040986  146.736601  145.759736  0.0  146.248076 | 161.680757  133.638146  0.0  0.0  0.0 | 0.598906 | 52.464809 | 1.022274 |
    | 03/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.656246  516.178775  516.178775  0.0  516.178775 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.745322  227.151208  227.750276  0.0  227.450799 | 89.898129  89.987145  89.388076  0.0  89.687554 | 226.387159  226.387159  226.387159  0.0  226.387159 | 89.791616  89.791616  89.791616  0.0  89.791616 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.522529        0.0  0.0     0.0  0.0 | 1.648515       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018723  0.033181  0.0  0.0  0.0 |   0.66802 |      0.5 | 0.000324 | 0.0  0.0  0.0  0.0  0.0 | 0.264467 | 0.247095 | 0.246198 | 0.246198 | 1.025825 |   2.0       2.0  0.0  0.0       2.0 | 366.897318  369.677703  370.655499  0.0  370.166637 | 146.758927  146.501072  145.523276  0.0  146.012138 | 162.573494  133.671327  0.0  0.0  0.0 | 0.766602 | 52.628914 | 1.025825 |
    | 04/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.663479  515.208571  515.208571  0.0  515.208571 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0  0.000576  0.0  0.0  0.001152 |    0.0       0.0      0.0     0.0       0.0 | 224.040175  226.441398  227.040334  0.0  226.740888 |  89.61607  89.737377   89.13844  0.0  89.437887 | 225.679105  225.679105  225.679105  0.0  225.679105 | 89.529467  89.529467  89.529467  0.0  89.529467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.545093        0.0  0.0     0.0  0.0 | 1.681679       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018282  0.033164  0.0  0.0  0.0 |  0.682027 |      0.5 | 0.000639 | 0.0  0.0  0.0  0.0  0.0 | 0.265287 | 0.248113 | 0.247119 | 0.247119 | 1.029661 |   2.0  1.999424  0.0  0.0  1.998848 | 366.188199   368.95541   369.93427  0.0  369.444854 |  146.47528  146.253161  145.274302  0.0  145.763718 | 163.455189  133.704491  0.0  0.0  0.0 |  0.94799 | 52.792198 | 1.029661 |
    | 04/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.668518   514.22969   514.22969  0.0   514.22969 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.331056  225.729761  226.328635  0.0  226.029206 | 89.332422   89.47881  88.879937  0.0  89.179365 | 224.968418  224.968418  224.968418  0.0  224.968418 | 89.261271  89.261271  89.261271  0.0  89.261271 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.561172        0.0  0.0     0.0  0.0 | 1.710709       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.017847  0.033148  0.0  0.0  0.0 |   0.69431 |      0.5 | 0.001153 | 0.0  0.0  0.0  0.0  0.0 | 0.266104 | 0.249286 | 0.248147 | 0.248147 | 1.033948 |   2.0  1.999424  0.0  0.0  1.998848 | 365.477513  368.234067  369.214053  0.0  368.724065 | 146.191005  145.995622  145.015637  0.0  145.505624 | 164.323499  133.737639  0.0  0.0  0.0 | 1.141147 | 52.954665 | 1.033948 |
    | 04/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.671177  513.244623  513.244623  0.0  513.244623 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  222.62037  225.016328  225.615167  0.0  225.315751 | 89.048148  89.213361  88.614523  0.0  88.913939 | 224.256032  224.256032  224.256032  0.0  224.256032 | 88.988592  88.988592  88.988592  0.0  88.988592 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.573447        0.0  0.0     0.0  0.0 | 1.737219       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01742  0.033131  0.0  0.0  0.0 |  0.705539 |      0.5 | 0.001938 | 0.0  0.0  0.0  0.0  0.0 | 0.266916 | 0.250672 | 0.249333 | 0.249333 | 1.038889 |   2.0  1.999424  0.0  0.0  1.998848 | 364.765126  367.513771  368.494918  0.0  368.004346 | 145.906051  145.730853  144.749706  0.0  145.240277 | 165.177147   133.77077  0.0  0.0  0.0 | 1.344748 |  53.11632 | 1.038889 |
    | 04/01 03:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.766676   516.04297   516.04297  0.0   516.04297 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |  1.573  3.374224    7.436   1.573  5.575848 | 222.346669  226.148508  231.531107  0.0   229.56884 | 89.897507  90.470339  89.149517  0.0  89.251632 | 226.191503  226.191503  226.191503  0.0  226.191503 | 89.851467  89.851467  89.851467  0.0  89.851467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  3.276294        0.0  0.0   1.573  0.0 | 2.234718       0.0  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016891  0.033115  0.0  0.0  0.0 |  0.913057 |      0.5 | 0.003801 | 0.0  0.0  0.0  0.0  0.0 | 0.268848 | 0.253966 | 0.251096 | 0.251096 | 1.046234 |   2.0       2.0  0.0  0.0       2.0 | 366.261912  369.083712  371.231314  0.0  370.542858 | 146.504765  146.959259  144.811656  0.0  145.500113 | 166.235614  133.803884  0.0  0.0  0.0 | 1.754003 | 53.500758 | 1.046234 |
    | 04/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.270864  514.782857  514.782857  0.0  514.782857 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.404769  226.039922  227.355189  0.0  226.933554 | 89.361908  90.003049  88.687781  0.0  89.109416 | 225.398666  225.398666  225.398666  0.0  225.398666 | 89.384192  89.384192  89.384192  0.0  89.384192 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.511993        0.0  0.0     0.0  0.0 | 1.735428       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016494  0.033098  0.0  0.0  0.0 |   0.70519 |      0.5 | 0.006398 | 0.0  0.0  0.0  0.0  0.0 | 0.269647 | 0.256788 | 0.253861 | 0.253861 | 1.057755 |   2.0       2.0  0.0  0.0       2.0 |  365.90776  368.482456   369.91479  0.0  369.347969 | 146.363104  146.300402  144.868067  0.0  145.434888 | 167.028673  133.836983  0.0  0.0  0.0 | 1.952796 | 53.659683 | 1.057755 |
    | 04/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  511.28054  513.816706  513.816706  0.0  513.816706 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.050617  225.322111  226.197965  0.0  225.851361 | 89.220247  89.460746  88.584892  0.0  88.931496 | 224.691291  224.691291  224.691291  0.0  224.691291 | 89.125415  89.125415  89.125415  0.0  89.125415 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536166        0.0  0.0     0.0  0.0 | 1.768886       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016102  0.033082  0.0  0.0  0.0 |  0.719299 |      0.5 | 0.008757 | 0.0  0.0  0.0  0.0  0.0 | 0.270441 | 0.259418 | 0.256745 | 0.256745 | 1.069772 |   2.0       2.0  0.0  0.0       2.0 | 365.200386  367.891636  369.048117  0.0  368.527899 | 146.080154   145.92507  144.768589  0.0  145.288807 | 167.812055  133.870064  0.0  0.0  0.0 | 2.163338 | 53.817813 | 1.069772 |
    | 04/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.286928  512.841258  512.841258  0.0  512.841258 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 222.343243  224.692074  225.398402  0.0  225.080676 | 88.937297  89.124632  88.418304  0.0   88.73603 | 223.981569  223.981569  223.981569  0.0  223.981569 | 88.859689  88.859689  88.859689  0.0  88.859689 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554329        0.0  0.0     0.0  0.0 | 1.798304       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015716  0.033065  0.0  0.0  0.0 |  0.731724 |      0.5 | 0.011799 | 0.0  0.0  0.0  0.0  0.0 | 0.271232 | 0.262589 | 0.259538 | 0.259538 | 1.081409 |   2.0       2.0  0.0  0.0       2.0 | 364.490663   367.22113  368.271283  0.0  367.768792 | 145.796265  145.620127  144.569974  0.0  145.072466 | 168.583796  133.903129  0.0  0.0  0.0 | 2.383262 | 53.975153 | 1.081409 |
    | 04/01 07:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.955818  513.898731  513.898731  0.0  513.898731 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 221.870249  225.005946  228.218517  0.0  227.018407 | 89.263679  89.652512  88.626741  0.0  88.825851 | 224.696076  224.696076  224.696076  0.0  224.696076 | 89.202655  89.202655  89.202655  0.0  89.202655 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.942913        0.0  0.0   0.847  0.0 | 2.090976       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015282  0.033048  0.0  0.0  0.0 |  0.853856 |      0.5 | 0.016737 | 0.0  0.0  0.0  0.0  0.0 | 0.272624 | 0.267838 | 0.263051 | 0.263051 | 1.096045 |   2.0       2.0  0.0  0.0       2.0 | 364.968442   367.75206  369.392842  0.0  368.789461 | 145.987377  146.146671  144.505889  0.0   145.10927 | 169.451015  133.936177  0.0  0.0  0.0 | 2.720382 |   54.2521 | 1.096045 |
    | 04/01 08:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.181898  513.932503  513.932503  0.0  513.932503 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 222.246491  225.198159  227.670615  0.0   226.79139 | 89.193327  89.738972  88.516116  0.0  88.823341 | 224.756025  224.756025  224.756025  0.0  224.756025 | 89.176478  89.176478  89.176478  0.0  89.176478 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.750605        0.0  0.0   0.484  0.0 | 1.974497       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.014886  0.033032  0.0  0.0  0.0 |  0.805494 |      0.5 | 0.023539 | 0.0  0.0  0.0  0.0  0.0 | 0.273749 |  0.27433 | 0.268114 | 0.268114 | 1.117142 |   2.0       2.0  0.0  0.0       2.0 | 365.129927  367.807526  369.406253  0.0  368.810097 | 146.051971  146.124977   144.52625  0.0  145.122406 | 170.242009  133.969209  0.0  0.0  0.0 | 3.002337 | 54.476066 | 1.117142 |
    | 04/01 09:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.581776  513.165993  513.165993  0.0  513.165993 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 | 222.306591  224.815129  226.159005  0.0  225.667272 | 88.996307  89.376974  88.345498  0.0  88.694231 | 224.225981  224.225981  224.225981  0.0  224.225981 | 88.940012  88.940012  88.940012  0.0  88.940012 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.584217        0.0  0.0   0.121  0.0 | 1.872417       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.014523  0.033015  0.0  0.0  0.0 |  0.763117 |      0.5 | 0.030434 | 0.0  0.0  0.0  0.0  0.0 |  0.27461 |  0.28065 | 0.274292 | 0.274292 | 1.142882 |   2.0       2.0  0.0  0.0       2.0 | 364.701269  367.372779  368.685229  0.0  368.137806 | 145.880507  145.793214  144.480764  0.0  145.028188 | 170.968333  134.002225  0.0  0.0  0.0 |  3.23502 | 54.647313 | 1.142882 |
    | 04/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.579336  513.337725  513.337725  0.0  513.337725 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 221.979417  224.762609  227.032845  0.0   226.18834 | 89.086359  89.441784  88.421148  0.0  88.693653 | 224.325723  224.325723  224.325723  0.0  224.325723 | 89.012002  89.012002  89.012002  0.0  89.012002 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.758389        0.0  0.0   0.484  0.0 | 2.015704       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.014144  0.032999  0.0  0.0  0.0 |  0.822984 |      0.5 | 0.038676 | 0.0  0.0  0.0  0.0  0.0 | 0.275725 | 0.288284 | 0.280942 | 0.280942 | 1.170591 |   2.0       2.0  0.0  0.0       2.0 | 364.699526  367.433493  368.906107  0.0  368.331188 |  145.87981  145.904232  144.431618  0.0  145.006537 | 171.725161  134.035224  0.0  0.0  0.0 | 3.519328 | 54.869302 | 1.170591 |
    | 04/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  509.74056  512.276806  512.276806  0.0  512.276806 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.842383  224.278811  225.177685  0.0  224.826758 | 88.736953  89.058914  88.160041  0.0  88.510967 | 223.591306  223.591306  223.591306  0.0  223.591306 | 88.685501  88.685501  88.685501  0.0  88.685501 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536246        0.0  0.0     0.0  0.0 | 1.869818       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.013804  0.032982  0.0  0.0  0.0 |  0.762345 |      0.5 | 0.047828 | 0.0  0.0  0.0  0.0  0.0 | 0.276489 | 0.296319 | 0.288373 | 0.288373 | 1.201555 |   2.0       2.0  0.0  0.0       2.0 |   364.1004  366.785987  367.959728  0.0  367.435736 |  145.64016  145.490819  144.317078  0.0  144.841071 | 172.405394  134.068206  0.0  0.0  0.0 | 3.733845 | 55.021384 | 1.201555 |
    | 04/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 508.746874  511.301327  511.301327  0.0  511.301327 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.243257  223.587629  224.303125  0.0  223.983707 | 88.497303  88.689177  87.973681  0.0  88.293099 |  222.88153   222.88153   222.88153  0.0   222.88153 | 88.419797  88.419797  88.419797  0.0  88.419797 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554453        0.0  0.0     0.0  0.0 |  1.89819       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.013469  0.032966  0.0  0.0  0.0 |  0.774312 |      0.5 | 0.056891 | 0.0  0.0  0.0  0.0  0.0 |  0.27725 | 0.304279 | 0.296303 | 0.296303 | 1.234594 |   2.0       2.0  0.0  0.0       2.0 | 363.390624  366.119888  367.178132  0.0  366.673558 |  145.35625  145.181439  144.123195  0.0  144.627769 | 173.075127  134.101172  0.0  0.0  0.0 | 3.951266 | 55.172706 | 1.234594 |
    | 04/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 507.750567  510.318845  510.318845  0.0  510.318845 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 220.533481  222.908881  223.553184  0.0  223.245979 | 88.213393  88.392446  87.748143  0.0  88.055348 | 222.169882  222.169882  222.169882  0.0  222.169882 | 88.148963  88.148963  88.148963  0.0  88.148963 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.568278        0.0  0.0     0.0  0.0 | 1.923319       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01314  0.032949  0.0  0.0  0.0 |  0.784925 |      0.5 | 0.067112 | 0.0  0.0  0.0  0.0  0.0 | 0.278006 | 0.313162 | 0.304484 | 0.304484 | 1.268684 |   2.0       2.0  0.0  0.0       2.0 | 362.678976  365.420888   366.43483  0.0  365.937461 | 145.071591  144.897957  143.884015  0.0  144.381384 | 173.733225  134.134121  0.0  0.0  0.0 | 4.169079 | 55.323271 | 1.268684 |
    | 04/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 506.752198  509.331046  509.331046  0.0  509.331046 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.821833  222.208114   222.82468  0.0  222.522236 | 87.928733  88.110731  87.494165  0.0  87.796609 | 221.456761  221.456761  221.456761  0.0  221.456761 | 87.874284  87.874284  87.874284  0.0  87.874284 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.578848        0.0  0.0     0.0  0.0 | 1.945949       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.012817  0.032933  0.0  0.0  0.0 |  0.794494 |      0.5 |  0.07845 | 0.0  0.0  0.0  0.0  0.0 | 0.278759 | 0.322935 |  0.31336 |  0.31336 | 1.305666 |   2.0       2.0  0.0  0.0       2.0 | 361.965856  364.709536  365.706911  0.0  365.211987 | 144.786342   144.62151  143.624134  0.0  144.119059 | 174.378941  134.167054  0.0  0.0  0.0 | 4.385123 | 55.473083 | 1.305666 |
    | 04/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 505.752249  508.339181  508.339181  0.0  508.339181 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.108713  221.498342  222.104076  0.0  221.803495 | 87.643485  87.832704   87.22697  0.0  87.527551 | 220.742512  220.742512  220.742512  0.0  220.742512 | 87.596669  87.596669  87.596669  0.0  87.596669 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.586932        0.0  0.0     0.0  0.0 | 1.966587       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   0.0125  0.032916  0.0  0.0  0.0 |  0.803231 |      0.5 | 0.090834 | 0.0  0.0  0.0  0.0  0.0 | 0.279508 | 0.333541 |  0.32312 |  0.32312 | 1.346335 |   2.0       2.0  0.0  0.0       2.0 | 361.251606  363.993706  364.985347  0.0  364.491004 | 144.500643  144.345475  143.353834  0.0  143.848177 | 175.011787  134.199971  0.0  0.0  0.0 | 4.597519 | 55.622146 | 1.346335 |
    | 04/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 504.751087  507.344207  507.344207  0.0  507.344207 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 218.394464  220.784715  221.386207  0.0  221.086357 | 87.357785  87.554466  86.952974  0.0  87.252824 | 220.027397  220.027397  220.027397  0.0  220.027397 | 87.316811  87.316811  87.316811  0.0  87.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.59312        0.0  0.0     0.0  0.0 | 1.985625       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01219    0.0329  0.0  0.0  0.0 |  0.811298 |      0.5 |  0.10416 | 0.0  0.0  0.0  0.0  0.0 | 0.280254 | 0.344898 | 0.333708 | 0.333708 | 1.390451 |   2.0       2.0  0.0  0.0       2.0 | 360.536491  363.276387  364.266537  0.0  363.772044 | 144.214596   144.06782   143.07767  0.0  143.572164 | 175.631472  134.232871  0.0  0.0  0.0 | 4.804658 | 55.770464 | 1.390451 |
    | 04/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 503.748995  506.346853  506.346853  0.0  506.346853 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 217.679348  220.069317   220.66914  0.0  220.369581 | 87.071739   87.27489  86.675067  0.0  86.974626 | 219.311616  219.311616  219.311616  0.0  219.311616 | 87.035237  87.035237  87.035237  0.0  87.035237 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.597858        0.0  0.0     0.0  0.0 | 2.003365       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.011887  0.032884  0.0  0.0  0.0 |  0.818821 |      0.5 | 0.118298 | 0.0  0.0  0.0  0.0  0.0 | 0.280995 | 0.356901 | 0.345041 | 0.345041 | 1.437672 |   2.0       2.0  0.0  0.0       2.0 | 359.820711  362.558686  363.549013  0.0  363.054079 | 143.928284  143.788167  142.797841  0.0  143.292775 | 176.237852  134.265754  0.0  0.0  0.0 | 5.005181 |  55.91804 | 1.437672 |
    | 04/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 502.746187  505.347677  505.347677  0.0  505.347677 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.963568  219.353023  219.952184  0.0  219.652742 | 86.785427  86.993831   86.39467  0.0  86.694112 | 218.595325  218.595325  218.595325  0.0  218.595325 | 86.752352  86.752352  86.752352  0.0  86.752352 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.60149        0.0  0.0     0.0  0.0 | 2.020043       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01159  0.032867  0.0  0.0  0.0 |  0.825899 |      0.5 | 0.133099 | 0.0  0.0  0.0  0.0  0.0 | 0.281733 |  0.36943 | 0.357018 | 0.357018 | 1.487574 |   2.0       2.0  0.0  0.0       2.0 | 359.104419  361.840989  362.832154  0.0  362.336662 | 143.641768  143.506688  142.515523  0.0  143.011015 |  176.83089  134.298621  0.0  0.0  0.0 | 5.197981 | 56.064879 | 1.487574 |
    | 04/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 501.742828  504.347104  504.347104  0.0  504.347104 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.247276  218.636219  219.235114  0.0  218.935721 | 86.498911  86.711458  86.112564  0.0  86.411956 |  217.87864   217.87864   217.87864  0.0   217.87864 | 86.468464  86.468464  86.468464  0.0  86.468464 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.604276        0.0  0.0     0.0  0.0 | 2.035838       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   0.0113  0.032851  0.0  0.0  0.0 |  0.832607 |      0.5 | 0.148399 | 0.0  0.0  0.0  0.0  0.0 | 0.282467 | 0.382355 | 0.369518 | 0.369518 | 1.539659 |   2.0       2.0  0.0  0.0       2.0 | 358.387734   361.12341   362.11568  0.0  361.619581 | 143.355094  143.223694  142.231424  0.0  142.727523 | 177.410628  134.331472  0.0  0.0  0.0 |  5.38219 | 56.210983 | 1.539659 |
    | 04/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 500.739045   503.34546   503.34546  0.0   503.34546 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 215.530592  217.919094  218.517877  0.0  218.218507 | 86.212237   86.42801  85.829227  0.0  86.128597 | 217.161652  217.161652  217.161652  0.0  217.161652 | 86.183808  86.183808  86.183808  0.0  86.183808 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.606415        0.0  0.0     0.0  0.0 | 2.050892       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.011017  0.032834  0.0  0.0  0.0 |  0.839003 |      0.5 | 0.164024 | 0.0  0.0  0.0  0.0  0.0 | 0.283198 | 0.395537 | 0.382412 | 0.382412 | 1.593384 |   2.0       2.0  0.0  0.0       2.0 | 357.670746  360.405968  361.399455  0.0  360.902725 | 143.068299  142.939492  141.946005  0.0  142.442734 | 177.977167  134.364306  0.0  0.0  0.0 | 5.557169 | 56.356357 | 1.593384 |
    | 04/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 499.734933  502.342992  502.342992  0.0  502.342992 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.813603  217.201748  217.800482  0.0  217.501124 | 85.925441  86.143711  85.544978  0.0  85.844336 | 216.444429  216.444429  216.444429  0.0  216.444429 | 85.898563  85.898563  85.898563  0.0  85.898563 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.608059        0.0  0.0     0.0  0.0 | 2.065314       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01074  0.032818  0.0  0.0  0.0 |  0.845133 |      0.5 | 0.179805 | 0.0  0.0  0.0  0.0  0.0 | 0.283925 |  0.40884 | 0.395564 | 0.395564 | 1.648184 |   2.0       2.0  0.0  0.0       2.0 | 356.953524  359.688648  360.683402  0.0  360.186031 | 142.781409  142.654344   141.65959  0.0  142.156961 | 178.530653  134.397124  0.0  0.0  0.0 | 5.722497 | 56.501003 | 1.648184 |
    | 04/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 498.730567  501.339893  501.339893  0.0  501.339893 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.096381  216.484243  217.082951  0.0    216.7836 | 85.638552   85.85875  85.260042  0.0  85.559392 | 215.727025  215.727025  215.727025  0.0  215.727025 | 85.612867  85.612867  85.612867  0.0  85.612867 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.609325        0.0  0.0     0.0  0.0 | 2.079188       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01047  0.032801  0.0  0.0  0.0 |  0.851032 |      0.5 | 0.195578 | 0.0  0.0  0.0  0.0  0.0 | 0.284648 | 0.422134 | 0.408838 | 0.408838 | 1.703493 |   2.0       2.0  0.0  0.0       2.0 |  356.23612  358.971431  359.967476  0.0  359.469456 | 142.494448  142.368462  141.372416  0.0  141.870437 |  179.07126  134.429926  0.0  0.0  0.0 | 5.877951 | 56.644927 | 1.703493 |
    | 04/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 497.726004  500.336306  500.336306  0.0  500.336306 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 213.378977  215.766617  216.365309  0.0  216.065964 | 85.351591  85.573276  84.974584  0.0  85.273929 |  215.00948   215.00948   215.00948  0.0   215.00948 | 85.326826  85.326826  85.326826  0.0  85.326826 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.610302        0.0  0.0     0.0  0.0 | 2.092582       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.010206  0.032785  0.0  0.0  0.0 |  0.856729 |      0.5 | 0.211192 | 0.0  0.0  0.0  0.0  0.0 | 0.285367 | 0.435297 | 0.422105 | 0.422105 |  1.75877 |   2.0       2.0  0.0  0.0       2.0 | 355.518575  358.254294  359.251648  0.0  358.752972 |  142.20743  142.082012  141.084658  0.0  141.583334 | 179.599185  134.462711  0.0  0.0  0.0 | 6.023488 | 56.788131 |  1.75877 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

>>> sclass(2)
>>> sfdist(linear=0.2)

.. _hland_96_snow_redistribution_2:

snow redistribution 2
_____________________

The example :ref:`hland_96_snow_redistribution_1` assumes uniform snow distributions
within all zones.  Here, we introduce some heterogeneity by defining two snow classes
for each zone:

>>> sclass(2)
>>> sfdist(linear=0.2)

Inspecting the snow pack's evolution reveals that we individually apply the threshold
parameter |SMax| for each snow class.  Hence, one snow class can still receive snowfall
or redistributed snow while the second one already routes it to the next lower zone:

.. integration-test::

    >>> test("hland_96_snow_redistribution_2", axis1=(states.sp, states.wc))
    |        date |    p |    t |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                                                                      swe |                          gact | contriarea |                                        pc |                                ei |                                          tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                                                        melt |                                                   refr |                                  in_ |                                    r |                      sr |                      ea |                                cf |      inuz |     perc |       q0 |                      el |       q1 |     inrc |    outrc |       rt |       qt |                                  ic |                                                                                                       sp |                                                                                                       wc |                                    sm |       uz |        lz |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 |     0.05 | 0.046667 |  0.06037 |  0.06037 | 0.251543 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |      9.95 | 0.251543 |
    | 01/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.04975 | 0.046433 | 0.086244 | 0.086244 | 0.359352 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.90025 | 0.359352 |
    | 01/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049501 | 0.046201 | 0.046434 | 0.046434 | 0.193473 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.850749 | 0.193473 |
    | 01/01 03:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049254 |  0.04597 | 0.046201 | 0.046201 | 0.192506 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.801495 | 0.192506 |
    | 01/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049007 |  0.04574 |  0.04597 |  0.04597 | 0.191543 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.752488 | 0.191543 |
    | 01/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048762 | 0.045512 | 0.045741 | 0.045741 | 0.190586 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.703725 | 0.190586 |
    | 01/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048519 | 0.045284 | 0.045512 | 0.045512 | 0.189633 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.655206 | 0.189633 |
    | 01/01 07:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048276 | 0.045058 | 0.045284 | 0.045284 | 0.188685 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.60693 | 0.188685 |
    | 01/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048035 | 0.044832 | 0.045058 | 0.045058 | 0.187741 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.558896 | 0.187741 |
    | 01/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047794 | 0.044608 | 0.044833 | 0.044833 | 0.186802 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.511101 | 0.186802 |
    | 01/01 10:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047556 | 0.044385 | 0.044608 | 0.044608 | 0.185868 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.463546 | 0.185868 |
    | 01/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047318 | 0.044163 | 0.044385 | 0.044385 | 0.184939 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.416228 | 0.184939 |
    | 01/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047081 | 0.043942 | 0.044163 | 0.044163 | 0.184014 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.369147 | 0.184014 |
    | 01/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046846 | 0.043723 | 0.043943 | 0.043943 | 0.183094 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.322301 | 0.183094 |
    | 01/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046612 | 0.043504 | 0.043723 | 0.043723 | 0.182179 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.27569 | 0.182179 |
    | 01/01 15:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      0.9152  0.0         0.0         0.0         0.0      1.3728  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0    1.144   0.242       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046551 | 0.043448 |  0.04354 |  0.04354 | 0.181417 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0      0.9152  0.0         0.0         0.0         0.0      1.3728  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.26371 | 0.181417 |
    | 01/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      0.9152  0.0         0.0         0.0         0.0      1.3728  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046319 | 0.043231 | 0.043412 | 0.043412 | 0.180884 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0      0.9152  0.0         0.0         0.0         0.0      1.3728  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.217391 | 0.180884 |
    | 01/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      0.9152  0.0         0.0         0.0         0.0      1.3728  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0   0.0  0.0   0.0  0.0   0.0   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046087 | 0.043014 | 0.043231 | 0.043231 | 0.180129 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0      0.9152  0.0         0.0         0.0         0.0      1.3728  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.171304 | 0.180129 |
    | 01/01 18:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    0.990801       6.864  0.0       3.548         0.0    1.458202      10.296  0.0       5.322 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |    0.0     1.894    7.436   1.573     4.435 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |      0.0  0.669498  0.0   1.573  0.0 |      0.0  0.167375  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.049749  0.0  0.0  0.0 |  0.039209 | 0.039209 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047148 | 0.044005 | 0.043283 | 0.043283 | 0.180344 | 1.815       2.0  0.0  0.0       2.0 |        0.0    0.707715       6.864  0.0       3.548         0.0    1.041573      10.296  0.0       5.322 |        0.0    0.283086         0.0  0.0         0.0         0.0    0.416629         0.0  0.0         0.0 |      100.0  100.551873  0.0  0.0  0.0 |      0.0 |  9.382478 | 0.180344 |
    | 01/01 19:00 |  5.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    8.221969     32.4896  0.0     22.7672         0.0   12.276954     48.7344  0.0     34.1508 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  6.776  14.5376   32.032   6.776   24.024 | 0.0       0.0  0.0  0.0       0.0 |  6.591   14.5376   32.032   6.776    24.024 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    6.591   5.51264  0.0   6.776  0.0 |  1.64775  1.393413  0.0   6.776  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047528  0.047664  0.0  0.0  0.0 |  1.115342 |      0.5 | 0.000059 | 0.0  0.0  0.0  0.0  0.0 | 0.053895 |  0.05035 | 0.045195 | 0.045195 | 0.188312 |   2.0       2.0  0.0  0.0       2.0 |        0.0    5.872835     32.4896  0.0     22.7672         0.0    8.769253     48.7344  0.0     34.1508 |        0.0    2.349134         0.0  0.0         0.0         0.0    3.507701         0.0  0.0         0.0 | 104.990778  104.718764  0.0  0.0  0.0 | 0.615283 | 10.725154 | 0.188312 |
    | 01/01 20:00 |  2.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   11.993681       45.76  0.0       32.72         0.0   17.906522       68.64  0.0       49.08 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  3.509   7.5284   16.588   3.509   12.441 | 0.0       0.0  0.0  0.0       0.0 |  3.509    7.5284   16.588   3.509    12.441 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    3.509   2.82776  0.0   3.509  0.0 | 0.966998  0.775232  0.0   3.509  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046234  0.046614  0.0  0.0  0.0 |  0.626524 |      0.5 | 0.000316 | 0.0  0.0  0.0  0.0  0.0 | 0.058275 | 0.054643 | 0.049894 | 0.049894 | 0.207891 |   2.0       2.0  0.0  0.0       2.0 |        0.0    8.566915       45.76  0.0       32.72         0.0   12.790373       68.64  0.0       49.08 |        0.0    3.426766         0.0  0.0         0.0         0.0    5.116149         0.0  0.0         0.0 | 107.579014  106.817906  0.0  0.0  0.0 | 0.741491 | 11.596736 | 0.207891 |
    | 01/01 21:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   18.327953     68.1824  0.0     49.5368         0.0    27.37993    102.2736  0.0     74.3052 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    5.929   4.81656  0.0   5.929  0.0 | 1.715444  1.373932  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.044104   0.04487  0.0  0.0  0.0 |  1.139412 |      0.5 | 0.001313 | 0.0  0.0  0.0  0.0  0.0 | 0.064362 | 0.061121 | 0.055128 | 0.055128 | 0.229701 |   2.0       2.0  0.0  0.0       2.0 |        0.0   13.091395     68.1824  0.0     49.5368         0.0   19.557093    102.2736  0.0     74.3052 |        0.0    5.236558         0.0  0.0         0.0         0.0    7.822837         0.0  0.0         0.0 | 111.836673  110.305404  0.0  0.0  0.0 | 1.379591 | 12.807946 | 0.229701 |
    | 01/01 22:00 | 10.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   31.965521     116.688  0.0      85.916         0.0   47.808282     175.032  0.0     128.874 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 | 12.826  27.5176   60.632  12.826   45.474 | 0.0       0.0  0.0  0.0       0.0 | 12.826   27.5176   60.632  12.826    45.474 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |   12.826  10.48464  0.0  12.826  0.0 | 4.010511  3.189239  0.0  12.826  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.039674    0.0412  0.0  0.0  0.0 |  2.703862 |      0.5 | 0.018353 | 0.0  0.0  0.0  0.0  0.0 | 0.075344 | 0.085004 | 0.064989 | 0.064989 | 0.270786 |   2.0       2.0  0.0  0.0       2.0 |        0.0   22.832515     116.688  0.0      85.916         0.0   34.148773     175.032  0.0     128.874 |        0.0    9.133006         0.0  0.0         0.0         0.0   13.659509         0.0  0.0         0.0 | 120.691836  117.642005  0.0  0.0  0.0 |   3.5651 | 14.993459 | 0.270786 |
    | 01/01 23:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   32.149649    117.1456  0.0     86.2592         0.0   48.056474    175.7184  0.0    129.3888 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    0.121   0.04344  0.0   0.121  0.0 | 0.044064   0.01503  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.039616  0.041165  0.0  0.0  0.0 | -0.006858 |      0.5 | 0.035851 | 0.0  0.0  0.0  0.0  0.0 | 0.077197 | 0.100731 | 0.083191 | 0.083191 | 0.346631 |   2.0       2.0  0.0  0.0       2.0 |        0.0   22.964035    117.1456  0.0     86.2592         0.0   34.326053    175.7184  0.0    129.3888 |        0.0    9.185614         0.0  0.0         0.0         0.0   13.730421         0.0  0.0         0.0 | 120.808388   117.71158  0.0  0.0  0.0 | 3.022391 |  15.36212 | 0.346631 |
    | 02/01 00:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   33.102545    120.3488  0.0     88.6616         0.0   49.457818    180.5232  0.0    132.9924 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    0.847   0.64008  0.0   0.847  0.0 | 0.309042  0.221724  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.039327  0.040935  0.0  0.0  0.0 |  0.172644 |      0.5 | 0.023113 | 0.0  0.0  0.0  0.0  0.0 | 0.079558 | 0.092745 | 0.095461 | 0.095461 | 0.397756 |   2.0       2.0  0.0  0.0       2.0 |        0.0   23.644675    120.3488  0.0     88.6616         0.0   35.327013    180.5232  0.0    132.9924 |        0.0     9.45787         0.0  0.0         0.0         0.0   14.130805         0.0  0.0         0.0 | 121.385673  118.170871  0.0  0.0  0.0 | 2.671922 | 15.832133 | 0.397756 |
    | 02/01 01:00 |  3.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   37.002385    134.0768  0.0   100.29584         0.0   55.279578       200.0  0.0   150.44376 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.63    7.788    17.16    3.63    12.87 | 0.0       0.0  0.0  0.0       0.0 |   3.63     7.788    17.16    3.63     12.87 |        0.0         0.0      0.5576  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0      1.6728 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |     3.63    2.9272  0.0    3.63  0.0 | 1.337154  1.021911  0.0    3.63  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.038161  0.039962  0.0  0.0  0.0 |  0.868564 |      0.5 | 0.023168 | 0.0  0.0  0.0  0.0  0.0 | 0.083896 | 0.096838 | 0.095429 | 0.095429 | 0.397621 |   2.0       2.0  0.0  0.0       2.0 |        0.0   26.430275    134.0768  0.0   100.29584         0.0   39.485413       200.0  0.0   150.44376 |        0.0    10.57211         0.0  0.0         0.0         0.0   15.794165         0.0  0.0         0.0 | 123.716679  120.116121  0.0  0.0  0.0 | 3.017318 | 16.695379 | 0.397621 |
    | 02/01 02:00 |  2.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   39.749073    143.6864  0.0   124.80032         0.0    59.37161       200.0  0.0   187.20048 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.541   5.4516   12.012   2.541    9.009 | 0.0       0.0  0.0  0.0       0.0 |  2.541    5.4516   12.012   2.541     9.009 |        0.0         0.0      7.2072  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0     21.6216 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    2.541   2.03224  0.0   2.541  0.0 | 0.972302  0.733023  0.0   2.541  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.037357  0.039292  0.0  0.0  0.0 |  0.620804 |      0.5 |  0.02878 | 0.0  0.0  0.0  0.0  0.0 | 0.087435 |  0.10463 |  0.09766 |  0.09766 | 0.406915 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.392195    143.6864  0.0   124.80032         0.0   42.408293       200.0  0.0   187.20048 |        0.0   11.356878         0.0  0.0         0.0         0.0   16.963317         0.0  0.0         0.0 | 125.322735  121.454631  0.0  0.0  0.0 | 3.109342 | 17.399516 | 0.406915 |
    | 02/01 03:00 | 10.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   82.886753    191.2768  0.0       200.0         0.0   124.07813       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 12.584  26.9984   59.488  12.584   44.616 | 0.0       0.0  0.0  0.0       0.0 | 12.584   26.9984   59.488  12.584    44.616 |        0.0         0.0     35.6928  0.0    20.36984 |       0.0        0.0        0.0  0.0        0.0 |        0.0     26.9237         0.0  0.0    19.75344 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |   12.584       0.0  0.0  12.584  0.0 | 4.941041       0.0  0.0  12.584  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.033517  0.039273  0.0  0.0  0.0 |  2.031711 |      0.5 | 0.059556 | 0.0  0.0  0.0  0.0  0.0 | 0.098129 | 0.139232 | 0.110587 | 0.110587 | 0.460781 |   2.0       2.0  0.0  0.0       2.0 |        0.0   59.489235    191.2768  0.0       200.0         0.0   89.033853       200.0  0.0       200.0 |        0.0   23.397518         0.0  0.0         0.0         0.0   35.044277         0.0  0.0         0.0 | 132.999211  121.493903  0.0  0.0  0.0 | 4.581497 | 19.527672 | 0.460781 |
    | 02/01 04:00 |  3.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  102.553593       200.0  0.0       200.0         0.0   153.57839       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.235    9.086    20.02   4.235   15.015 | 0.0       0.0  0.0  0.0       0.0 |  4.235     9.086    20.02   4.235    15.015 |        0.0         0.0     15.6584  0.0      15.015 |       0.0        0.0        0.0  0.0        0.0 |        0.0    15.49755         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    4.235       0.0  0.0   4.235  0.0 | 1.872801       0.0  0.0   4.235  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032319  0.039253  0.0  0.0  0.0 |  0.753783 |      0.5 | 0.101246 | 0.0  0.0  0.0  0.0  0.0 | 0.102806 | 0.176949 | 0.139924 | 0.139924 | 0.583016 |   2.0       2.0  0.0  0.0       2.0 |        0.0   75.130475       200.0  0.0       200.0         0.0  112.475713       200.0  0.0       200.0 |        0.0   27.423118         0.0  0.0         0.0         0.0   41.102677         0.0  0.0         0.0 |  135.39373  121.533156  0.0  0.0  0.0 | 4.734034 | 20.458438 | 0.583016 |
    | 02/01 05:00 |  3.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  124.200713       200.0  0.0       200.0         0.0   186.04907       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.114   8.8264   19.448   4.114   14.586 | 0.0       0.0  0.0  0.0       0.0 |  4.114    8.8264   19.448   4.114    14.586 |        0.0         0.0      19.448  0.0      14.586 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.2325         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    4.114       0.0  0.0   4.114  0.0 | 1.885391       0.0  0.0   4.114  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.031189  0.039233  0.0  0.0  0.0 |  0.759506 |      0.5 | 0.111368 | 0.0  0.0  0.0  0.0  0.0 | 0.107374 | 0.189309 | 0.171314 | 0.171314 | 0.713809 |   2.0       2.0  0.0  0.0       2.0 |        0.0   92.868155       200.0  0.0       200.0         0.0  139.062233       200.0  0.0       200.0 |        0.0   31.332558         0.0  0.0         0.0         0.0   46.986837         0.0  0.0         0.0 | 137.653528   121.57239  0.0  0.0  0.0 | 4.882173 |  21.36735 | 0.713809 |
    | 02/01 06:00 |  1.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  131.840873       200.0  0.0       200.0         0.0   197.50931       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.452   3.1152    6.864   1.452    5.148 | 0.0       0.0  0.0  0.0       0.0 |  1.452    3.1152    6.864   1.452     5.148 |        0.0         0.0       6.864  0.0       5.148 |       0.0        0.0        0.0  0.0        0.0 |        0.0       6.435         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    1.452       0.0  0.0   1.452  0.0 |  0.68783       0.0  0.0   1.452  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030791  0.039214  0.0  0.0  0.0 |  0.260695 |      0.5 | 0.104473 | 0.0  0.0  0.0  0.0  0.0 | 0.110017 | 0.186261 | 0.185885 | 0.185885 | 0.774522 |   2.0       2.0  0.0  0.0       2.0 |        0.0   99.154395       200.0  0.0       200.0         0.0  148.471593       200.0  0.0       200.0 |        0.0   32.686478         0.0  0.0         0.0         0.0   49.037717         0.0  0.0         0.0 | 138.448488  121.611603  0.0  0.0  0.0 | 4.538394 | 21.893333 | 0.774522 |
    | 02/01 07:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  132.477553       200.0  0.0       200.0         0.0   198.46433       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |    0.121       0.0  0.0   0.121  0.0 | 0.057983       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030744  0.039194  0.0  0.0  0.0 | -0.001715 |      0.5 | 0.076817 | 0.0  0.0  0.0  0.0  0.0 | 0.111696 | 0.165704 |  0.18237 |  0.18237 | 0.759876 |   2.0       2.0  0.0  0.0       2.0 |        0.0   99.714915       200.0  0.0       200.0         0.0  149.292373       200.0  0.0       200.0 |        0.0   32.762638         0.0  0.0         0.0         0.0   49.171957         0.0  0.0         0.0 | 138.542249  121.650798  0.0  0.0  0.0 | 3.959861 | 22.227494 | 0.759876 |
    | 02/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  132.477553       200.0  0.0       200.0         0.0   198.46433       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030729  0.039175  0.0  0.0  0.0 | -0.025862 |      0.5 | 0.049645 | 0.0  0.0  0.0  0.0  0.0 |  0.11328 | 0.145445 |  0.16577 |  0.16577 | 0.690708 |   2.0       2.0  0.0  0.0       2.0 |        0.0   99.754915       200.0  0.0       200.0         0.0  149.332373       200.0  0.0       200.0 |        0.0   32.722638         0.0  0.0         0.0         0.0   49.131957         0.0  0.0         0.0 | 138.572978  121.689972  0.0  0.0  0.0 | 3.384354 | 22.542785 | 0.690708 |
    | 02/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  132.477553       200.0  0.0       200.0         0.0   198.46433       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030714  0.039155  0.0  0.0  0.0 | -0.025849 |      0.5 | 0.030105 | 0.0  0.0  0.0  0.0  0.0 | 0.114857 | 0.131283 |   0.1468 |   0.1468 | 0.611665 |   2.0       2.0  0.0  0.0       2.0 |        0.0   99.794915       200.0  0.0       200.0         0.0  149.372373       200.0  0.0       200.0 |        0.0   32.682638         0.0  0.0         0.0         0.0   49.091957         0.0  0.0         0.0 | 138.603692  121.729127  0.0  0.0  0.0 | 2.828401 |   22.8565 | 0.611665 |
    | 02/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  135.024273       200.0  0.0       200.0         0.0       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 |        0.0         0.0       2.288  0.0       1.716 |       0.0        0.0        0.0  0.0        0.0 |   0.913764    1.002795         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 0.731011  0.0  0.0  0.0  0.0  1.096517  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 | 1.397764       0.0  0.0   0.484  0.0 | 0.671311       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030335  0.039135  0.0  0.0  0.0 |  0.254028 |      0.5 |   0.0196 | 0.0  0.0  0.0  0.0  0.0 | 0.116771 | 0.124666 |  0.13296 |  0.13296 | 0.553999 |   2.0       2.0  0.0  0.0       2.0 |        0.0  101.916995       200.0  0.0       200.0         0.0  150.251083       200.0  0.0       200.0 |        0.0   33.107278         0.0  0.0         0.0         0.0   49.748917         0.0  0.0         0.0 |  139.36048  121.768263  0.0  0.0  0.0 | 2.562829 | 23.237443 | 0.553999 |
    | 02/01 11:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  135.660953       200.0  0.0       200.0         0.0       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0     0.11694       0.572  0.0       0.429 |       0.0    0.03882        0.0  0.0        0.0 |   0.350952      0.2145         0.0  0.0         0.0 |  0.031056        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 0.280762  0.0  0.0  0.0  0.0  0.421143  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 | 0.503008       0.0  0.0   0.121  0.0 | 0.244227       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.03019  0.039116  0.0  0.0  0.0 |  0.076143 |      0.5 | 0.012954 | 0.0  0.0  0.0  0.0  0.0 | 0.118417 | 0.120885 | 0.125296 | 0.125296 | 0.522068 |   2.0       2.0  0.0  0.0       2.0 |        0.0  102.477515       200.0  0.0       200.0         0.0  150.194482       200.0  0.0       200.0 |        0.0   33.183438         0.0  0.0         0.0         0.0   49.805518         0.0  0.0         0.0 | 139.649451  121.807379  0.0  0.0  0.0 | 2.126019 | 23.564884 | 0.522068 |
    | 02/01 12:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  10.784233  158.581433       200.0  0.0       200.0   17.818544       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |        0.0    4.118589      20.592  0.0      15.444 |       0.0   1.488771        0.0  0.0        0.0 |  12.561271       7.722         0.0  0.0         0.0 |  1.191017        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 | 3.806899       0.0  0.0   4.356  0.0 | 1.856051       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   0.0292  0.039096  0.0  0.0  0.0 |  0.748156 |      0.5 | 0.011362 | 0.0  0.0  0.0  0.0  0.0 | 0.123079 | 0.123963 | 0.122409 | 0.122409 | 0.510038 |   2.0       2.0  0.0  0.0       2.0 |   7.703023  121.256235       200.0  0.0       200.0   12.727532  146.939385       200.0  0.0       200.0 |   3.081209   37.325198         0.0  0.0         0.0    5.091013   53.060615         0.0  0.0         0.0 |   141.6295  121.846475  0.0  0.0  0.0 | 2.362813 | 24.492662 | 0.510038 |
    | 02/01 13:00 |  5.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |   30.35464  196.145553       200.0  0.0       200.0   48.816351  198.842322       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.139  15.3164   33.748   7.139   25.311 | 0.0       0.0  0.0  0.0       0.0 |  7.139   15.3164   33.748   7.139    25.311 |        0.0    6.524337      33.748  0.0      25.311 |       0.0   2.665503        0.0  0.0        0.0 |   20.40607     12.6555         0.0  0.0         0.0 |  2.132402        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 | 4.393365  0.578839  0.0   7.139  0.0 | 2.203153  0.214844  0.0   7.139  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.02809  0.038895  0.0  0.0  0.0 |  0.964926 |      0.5 | 0.017497 | 0.0  0.0  0.0  0.0  0.0 | 0.129705 | 0.135056 | 0.125744 | 0.125744 | 0.523934 |   2.0       2.0  0.0  0.0       2.0 |  21.681886  152.006915       200.0  0.0       200.0   34.868822   142.03023       200.0  0.0       200.0 |   8.672754   44.138638         0.0  0.0         0.0   13.947529   56.812092         0.0  0.0         0.0 | 143.847801  122.249364  0.0  0.0  0.0 | 2.810242 | 25.811385 | 0.523934 |
    | 02/01 14:00 |  1.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  32.373566       200.0       200.0  0.0       200.0   53.486935  198.758117       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.331   2.8556    6.292   1.331    4.719 | 0.0       0.0  0.0  0.0       0.0 |  1.331    2.8556    6.292   1.331     4.719 |        0.0    0.805113       6.292  0.0       4.719 |       0.0   0.329408        0.0  0.0        0.0 |   4.735104    0.784983         0.0  0.0         0.0 |  0.263526        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 | 2.984875  0.620942  0.0   1.331  0.0 |  1.54409  0.231998  0.0   1.331  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.027356  0.038681  0.0  0.0  0.0 |  0.696412 |      0.5 | 0.024353 | 0.0  0.0  0.0  0.0  0.0 |  0.13215 | 0.142823 | 0.134317 | 0.134317 | 0.559654 |   2.0       2.0  0.0  0.0       2.0 |  23.123976  154.623602       200.0  0.0       200.0   38.204954  141.970083       200.0  0.0       200.0 |    9.24959   45.376398         0.0  0.0         0.0   15.281981   56.788033         0.0  0.0         0.0 | 145.315942  122.676989  0.0  0.0  0.0 |   2.9823 | 26.297949 | 0.559654 |
    | 02/01 15:00 | 20.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 198.546515       200.0       200.0  0.0       200.0  196.060668  181.289371       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 25.047  53.7372  118.404  25.047   88.803 | 0.0       0.0  0.0  0.0       0.0 | 25.047   53.7372  118.404  25.047    88.803 |        0.0   35.821169     118.404  0.0      88.803 |       0.0   17.29509        0.0  0.0        0.0 | 117.459935         0.0         0.0  0.0         0.0 | 13.836072        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04   0.0  0.0   0.0  0.0  0.04   0.0  0.0   0.0 | 1.969666  9.355315  0.0  25.047  0.0 | 1.039822  3.519854  0.0  25.047  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026877  0.035744  0.0  0.0  0.0 |   1.58343 |      0.5 |  0.04422 | 0.0  0.0  0.0  0.0  0.0 | 0.151523 | 0.176798 | 0.148647 | 0.148647 | 0.619363 |   2.0       2.0  0.0  0.0       2.0 | 151.513478   142.90054       200.0  0.0       200.0  140.043334  129.492408       200.0  0.0       200.0 |  47.033036    57.09946         0.0  0.0         0.0   56.017334   51.796963         0.0  0.0         0.0 | 146.272663  128.548194  0.0  0.0  0.0 | 4.021511 |  30.15314 | 0.619363 |
    | 02/01 16:00 | 37.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 302.171985  303.573153  306.530792  0.0  306.530792  277.991379  294.949914  306.530792  0.0  306.530792 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 45.859  98.3884  216.788  45.859  162.591 | 0.0       0.0  0.0  0.0       0.0 | 45.859   98.3884  216.788  45.859   162.591 |  25.594811   55.199435     216.788  0.0     162.591 |  17.56778   33.83365        0.0  0.0        0.0 |    89.3636     89.3636     89.3636  0.0     89.3636 | 17.167192  17.167192  17.167192  0.0  17.167192 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 16.44911  7.269259  0.0  45.859  0.0 | 8.798502  3.003047  0.0  45.859  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023038  0.033593  0.0  0.0  0.0 |  4.646261 |      0.5 | 0.242474 | 0.0  0.0  0.0  0.0  0.0 | 0.185665 | 0.367266 | 0.211574 | 0.211574 | 0.881559 |   2.0       2.0  0.0  0.0       2.0 | 215.837132  216.837966    290.0036  0.0    289.7036  198.565271   210.67851    290.0036  0.0    289.7036 |  86.334853   86.735186   16.527192  0.0   16.827192   79.426108   84.271404   16.527192  0.0   16.827192 | 153.946309  132.847998  0.0  0.0  0.0 | 7.925298 | 36.947332 | 0.881559 |
    | 02/01 17:00 |  8.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 321.863578  321.863578  321.863578  0.0  321.863578  321.863578  321.863578  321.863578  0.0  321.863578 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  9.922  21.2872   46.904   9.922   35.178 | 0.0       0.0  0.0  0.0       0.0 |  9.922   21.2872   46.904   9.922    35.178 |  69.105998   83.916001  146.266522  0.0  134.734626 | 30.897685  36.632732    7.16827  0.0   6.974166 |  96.517553   96.517553   96.517553  0.0   96.517553 | 25.346025  25.346025  25.346025  0.0  25.346025 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   9.922  0.0 |      0.0       0.0  0.0   9.922  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023027  0.033576  0.0  0.0  0.0 | -0.020787 |      0.5 | 0.414174 | 0.0  0.0  0.0  0.0  0.0 | 0.193967 | 0.512375 | 0.357186 | 0.357186 | 1.488277 |   2.0       2.0  0.0  0.0       2.0 | 233.372114  236.508001  287.550228  0.0  286.801658  231.161412  235.053179  288.047033  0.0  287.207395 |  88.491464   85.355578    34.31335  0.0    35.06192   90.702166   86.810399   33.816545  0.0   34.656183 | 153.969336  132.881574  0.0  0.0  0.0 | 6.990337 | 38.599366 | 1.488277 |
    | 02/01 18:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 332.354532  332.354532  332.354532  0.0  332.354532  332.354532  332.354532  332.354532  0.0  332.354532 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |  89.864904   95.032179  128.289774  0.0  123.120361 | 36.354674  36.176999  14.165805  0.0  14.187217 | 102.880225  102.880225  102.880225  0.0  102.880225 | 29.474307  29.474307  29.474307  0.0  29.474307 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   4.356  0.0 |      0.0       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023015  0.033559  0.0  0.0  0.0 | -0.020776 |      0.5 | 0.285601 | 0.0  0.0  0.0  0.0  0.0 | 0.198251 | 0.413515 |  0.45816 |  0.45816 | 1.908999 |   2.0       2.0  0.0  0.0       2.0 | 243.994061  248.546433  283.236701  0.0  282.238806   241.87812  247.027637  284.005462  0.0  282.857973 |  88.360471   83.808098   49.117831  0.0   50.115726   90.476412   85.326895    48.34907  0.0   49.496559 | 153.992351  132.915134  0.0  0.0  0.0 |  6.18396 | 39.451972 | 1.908999 |
    | 02/01 19:00 |  7.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 354.210686  354.210686  354.210686  0.0  354.210686  354.210686  354.210686  354.210686  0.0  354.210686 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  9.075    19.47     42.9   9.075   32.175 | 0.0       0.0  0.0  0.0       0.0 |  9.075     19.47     42.9   9.075    32.175 | 100.623515  110.619464  152.512898  0.0  142.061166 | 40.806017  41.205068  22.741634  0.0  22.468366 | 118.861176  118.861176  118.861176  0.0  118.861176 |  35.34951   35.34951   35.34951  0.0   35.34951 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   9.075  0.0 |      0.0       0.0  0.0   9.075  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023004  0.033542  0.0  0.0  0.0 | -0.020766 |      0.5 | 0.197591 | 0.0  0.0  0.0  0.0  0.0 | 0.205885 | 0.350232 | 0.421421 | 0.421421 | 1.755922 |   2.0       2.0  0.0  0.0       2.0 | 260.204005  265.718089  292.710207  0.0  291.210955  257.451511  263.579406  294.308513  0.0  292.515844 |  94.006681   88.492596   61.500479  0.0   62.999731   96.759175    90.63128   59.902173  0.0   61.694842 | 154.015355  132.948676  0.0  0.0  0.0 | 5.465602 | 40.971087 | 1.755922 |
    | 02/01 20:00 | 18.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 408.122532  408.122532  408.122532  0.0  408.122532  407.595072  408.122532  408.122532  0.0  408.122532 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 22.385   48.026   105.82  22.385   79.365 | 0.0       0.0  0.0  0.0       0.0 | 22.385    48.026   105.82  22.385    79.365 | 121.343068  143.657522  225.791472  0.0  200.037615 | 55.252618  58.579164  34.239214  0.0   33.53807 | 158.365958  158.365958  158.365958  0.0  158.365958 | 49.756574  49.756574  49.756574  0.0  49.756574 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |  0.26373       0.0  0.0  22.385  0.0 | 0.156397       0.0  0.0  22.385  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022939  0.033526  0.0  0.0  0.0 |  0.044432 |      0.5 | 0.138101 | 0.0  0.0  0.0  0.0  0.0 | 0.222988 | 0.318602 | 0.357266 | 0.357266 |  1.48861 |   2.0       2.0  0.0  0.0       2.0 | 295.869972  302.382842  330.979006  0.0  328.541091  291.139337  298.739526  334.108687  0.0  331.252394 | 112.252559   105.73969   77.143526  0.0   79.581441  116.455735  109.383006   74.013845  0.0   76.870138 | 154.145627  132.982202  0.0  0.0  0.0 | 4.871934 | 44.374528 |  1.48861 |
    | 02/01 21:00 | 15.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 452.899067  452.899067  452.899067  0.0  452.899067  451.910752  452.899067  452.899067  0.0  452.899067 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 18.634  39.9784   88.088  18.634   66.066 | 0.0       0.0  0.0  0.0       0.0 | 18.634   39.9784   88.088  18.634    66.066 | 155.850179  176.135007  251.155062  0.0  228.997864 | 70.642623  71.965925   45.05547  0.0  45.190668 | 189.711844  189.711844  189.711844  0.0  189.711844 | 63.187222  63.187222  63.187222  0.0  63.187222 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 0.494157       0.0  0.0  18.634  0.0 |  0.29354       0.0  0.0  18.634  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022827  0.033509  0.0  0.0  0.0 |  0.101628 |      0.5 | 0.098844 | 0.0  0.0  0.0  0.0  0.0 | 0.237325 | 0.300579 | 0.321626 | 0.321626 | 1.340108 |   2.0       2.0  0.0  0.0       2.0 | 327.247259  333.570588  358.114207  0.0  355.524426  322.793394  330.020655   361.54305  0.0  358.509019 | 125.651808  119.328479   94.784859  0.0   97.374641  129.117358  122.878412   91.356017  0.0   94.390047 | 154.369071  133.015711  0.0  0.0  0.0 | 4.374718 | 47.227774 | 1.340108 |
    | 02/01 22:00 |  6.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 471.068175  471.068175  471.068175  0.0  471.068175  471.068175  471.068175  471.068175  0.0  471.068175 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.623  16.3548   36.036   7.623   27.027 | 0.0       0.0  0.0  0.0       0.0 |  7.623   16.3548   36.036   7.623    27.027 | 183.712837  194.508916  233.953916  0.0  224.012928 | 76.315073  74.744951   54.98115  0.0  55.913138 |  201.72881   201.72881   201.72881  0.0   201.72881 | 69.339366  69.339366  69.339366  0.0  69.339366 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   7.623  0.0 |      0.0       0.0  0.0   7.623  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022815  0.033492  0.0  0.0  0.0 |  -0.02067 |      0.5 | 0.068053 | 0.0  0.0  0.0  0.0  0.0 | 0.243727 | 0.281921 | 0.300438 | 0.300438 | 1.251824 |   2.0       2.0  0.0  0.0       2.0 | 341.975001   347.41207  363.016775  0.0  361.027545  339.405611   345.11336  365.542269  0.0  363.171662 | 129.093174  123.656106  108.051401  0.0   110.04063  131.662565  125.954815  105.525906  0.0  107.896513 | 154.391886  133.049203  0.0  0.0  0.0 | 3.785994 | 48.501618 | 1.251824 |
    | 02/01 23:00 |  1.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 476.605068  476.605068  476.605068  0.0  476.605068  476.605068  476.605068  476.605068  0.0  476.605068 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.299   4.9324   10.868   2.299    8.151 | 0.0       0.0  0.0  0.0       0.0 |  2.299    4.9324   10.868   2.299     8.151 | 196.746284  202.033428    219.4673  0.0  215.730203 | 76.620891  73.967147  62.468876  0.0  63.488973 |  205.07671   205.07671   205.07671  0.0   205.07671 | 71.528357  71.528357  71.528357  0.0  71.528357 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   2.299  0.0 |      0.0       0.0  0.0   2.299  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022804  0.033475  0.0  0.0  0.0 |  -0.02066 |      0.5 | 0.043152 | 0.0  0.0  0.0  0.0  0.0 | 0.246293 | 0.264395 | 0.282173 | 0.282173 | 1.175719 |   2.0       2.0  0.0  0.0       2.0 | 347.357347  352.123274  360.673013  0.0  359.334981   345.99213  350.915921  362.120852  0.0  360.539242 |  129.24772  124.481794  115.932054  0.0  117.270087  130.612937  125.689147  114.484216  0.0  116.065826 |  154.41469  133.082678  0.0  0.0  0.0 | 3.222182 | 49.012325 | 1.175719 |
    | 03/01 00:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 490.884421  490.884421  490.884421  0.0  490.884421  490.884421  490.884421  490.884421  0.0  490.884421 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 | 202.983948  211.156028  235.093916  0.0  227.853312 |  79.55012   78.16944  69.539152  0.0  69.772755 | 214.821454  214.821454  214.821454  0.0  214.821454 | 76.062967  76.062967  76.062967  0.0  76.062967 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   5.929  0.0 |      0.0       0.0  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022793  0.033459  0.0  0.0  0.0 |  -0.02065 |      0.5 |  0.02577 | 0.0  0.0  0.0  0.0  0.0 | 0.251439 | 0.255293 | 0.266267 | 0.266267 | 1.109446 |   2.0       2.0  0.0  0.0       2.0 | 356.801301  361.378048  368.998259  0.0  367.628058  355.529223    360.2832  370.586684  0.0   368.90445 | 134.083121  129.506373  121.886163  0.0  123.256364  135.355198  130.601222  120.297738  0.0  121.979972 | 154.437483  133.116137  0.0  0.0  0.0 | 2.675762 | 50.036457 | 1.109446 |
    | 03/01 01:00 |  2.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 498.752637  498.752637  498.752637  0.0  498.752637  498.752637  498.752637  498.752637  0.0  498.752637 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.267   7.0092   15.444   3.267   11.583 | 0.0       0.0  0.0  0.0       0.0 |  3.267    7.0092   15.444   3.267    11.583 | 212.012393  217.734739  233.071551  0.0  228.657877 | 82.139028  80.158882   73.25687  0.0  73.809544 | 219.913342  219.913342  219.913342  0.0  219.913342 | 78.839294  78.839294  78.839294  0.0  78.839294 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   3.267  0.0 |      0.0       0.0  0.0   3.267  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022781  0.033442  0.0  0.0  0.0 |  -0.02064 |      0.5 | 0.014084 | 0.0  0.0  0.0  0.0  0.0 | 0.254659 | 0.248949 | 0.255906 | 0.255906 | 1.066274 |   2.0       2.0  0.0  0.0       2.0 | 362.167263  366.521363  372.112736  0.0  370.965664  361.271193  365.754692   373.32379  0.0  371.923773 | 136.585374  132.231274  126.639901  0.0  127.786973  137.481444  132.997945  125.428847  0.0  126.828864 | 154.460264  133.149579  0.0  0.0  0.0 | 2.141039 | 50.677084 | 1.066274 |
    | 03/01 02:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 500.209714  500.209714  500.209714  0.0  500.209714  500.209714  500.209714  500.209714  0.0  500.209714 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 216.845362  220.040775  225.830278  0.0   224.42183 | 82.512275  80.009862  75.782359  0.0  76.475807 | 220.484813  220.484813  220.484813  0.0  220.484813 | 79.724901  79.724901  79.724901  0.0  79.724901 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.02277  0.033425  0.0  0.0  0.0 | -0.020629 |      0.5 | 0.006715 | 0.0  0.0  0.0  0.0  0.0 |  0.25596 | 0.244268 | 0.249318 | 0.249318 | 1.038827 |   2.0       2.0  0.0  0.0       2.0 | 363.226245  367.377772  370.574063  0.0  369.757888  362.797147  367.010359  371.171534  0.0  370.227516 | 136.983469  132.831942  129.635651  0.0  130.451826  137.412567  133.199354   129.03818  0.0  129.982198 | 154.483034  133.183004  0.0  0.0  0.0 | 1.613694 | 50.936124 | 1.038827 |
    | 03/01 03:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 500.792544  500.792544  500.792544  0.0  500.792544  500.792544  500.792544  500.792544  0.0  500.792544 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 217.938071  220.667645   223.61194  0.0  222.826554 | 82.513642  80.061268  77.741774  0.0   78.24116 | 220.463332  220.463332  220.463332  0.0  220.463332 | 80.329213  80.329213  80.329213  0.0  80.329213 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022758  0.033408  0.0  0.0  0.0 | -0.020619 |      0.5 | 0.002549 | 0.0  0.0  0.0  0.0  0.0 | 0.256996 | 0.241902 | 0.244783 | 0.244783 | 1.019928 |   2.0       2.0  0.0  0.0       2.0 | 363.289745  367.344084  369.365472  0.0  368.715954    363.0902   367.17302  369.650908  0.0  368.939005 | 137.502799  133.448461  131.427072  0.0  132.076591  137.702345  133.619525  131.141637  0.0   131.85354 | 154.505793  133.216412  0.0  0.0  0.0 | 1.090527 |  51.14227 | 1.019928 |
    | 03/01 04:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 502.249621  502.249621  502.249621  0.0  502.249621  502.249621  502.249621  502.249621  0.0  502.249621 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 218.318893  221.310909  224.501228  0.0   223.45027 | 83.078652  80.779635  79.151316  0.0  79.487274 | 221.060927  221.060927  221.060927  0.0  221.060927 | 81.188694  81.188694  81.188694  0.0  81.188694 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022747  0.033392  0.0  0.0  0.0 | -0.020609 |      0.5 | 0.000622 | 0.0  0.0  0.0  0.0  0.0 | 0.258286 | 0.241565 | 0.242353 | 0.242353 | 1.009805 |   2.0       2.0  0.0  0.0       2.0 | 363.659782  367.684827  369.452009  0.0  368.834024  363.510264  367.556313   369.68377  0.0  369.012249 | 138.589839  134.564794  132.797613  0.0  133.415598  138.739357  134.693309  132.565851  0.0  133.237372 |  154.52854  133.249804  0.0  0.0  0.0 | 0.569296 | 51.398984 | 1.009805 |
    | 03/01 05:00 |  2.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 509.243591  509.243591  509.243591  0.0  509.243591  509.243591  509.243591  509.243591  0.0  509.243591 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.904   6.2304   13.728   2.904   10.296 | 0.0       0.0  0.0  0.0       0.0 |  2.904    6.2304   13.728   2.904    10.296 |  219.63452  224.689962  234.726843  0.0  231.245333 | 85.519101  83.790059  81.250778  0.0  81.300289 | 225.566024  225.566024  225.566024  0.0  225.566024 | 83.677567  83.677567  83.677567  0.0  83.677567 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   2.904  0.0 |      0.0       0.0  0.0   2.904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022736  0.033375  0.0  0.0  0.0 | -0.020598 |      0.5 |  0.00005 | 0.0  0.0  0.0  0.0  0.0 | 0.261212 | 0.243838 | 0.242145 | 0.242145 | 1.008937 |   2.0       2.0  0.0  0.0       2.0 | 367.364649  371.448515  374.456481  0.0  373.636035  366.974438  371.115949   375.09366  0.0  374.123621 | 141.878942  137.795076   134.78711  0.0  135.607555  142.269153  138.127642  134.149931  0.0   135.11997 | 154.551276  133.283179  0.0  0.0  0.0 | 0.048648 | 51.981201 | 1.008937 |
    | 03/01 06:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 510.409252  510.409252  510.409252  0.0  510.409252  510.409252  510.409252  510.409252  0.0  510.409252 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 223.104517   226.04003  229.638088  0.0  228.580052 | 86.623073   84.24196  81.893502  0.0  82.379539 | 225.936694  225.936694  225.936694  0.0  225.936694 | 84.472558  84.472558  84.472558  0.0  84.472558 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   0.484  0.0 |      0.0       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022724  0.033358  0.0  0.0  0.0 | -0.020588 |  0.02806 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.260372 | 0.243014 | 0.243149 | 0.243149 | 1.013123 |   2.0       2.0  0.0  0.0       2.0 | 367.758648  371.765097  373.830081  0.0  373.161456  367.550825  371.587894  374.173271  0.0  373.423484 | 142.650604  138.644155  136.579171  0.0  137.247796  142.858427  138.821358  136.235981  0.0  136.985768 |    154.574  133.316538  0.0  0.0  0.0 |      0.0 | 51.814023 | 1.013123 |
    | 03/01 07:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 510.992083  510.992083  510.992083  0.0  510.992083  510.992083  510.992083  510.992083  0.0  510.992083 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 223.660279    226.3251   228.47646  0.0    227.7885 | 86.990973  84.603352  83.076792  0.0  83.478752 | 225.909206  225.909206  225.909206  0.0  225.909206 | 85.082877  85.082877  85.082877  0.0  85.082877 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.259243 |  0.24196 | 0.242963 | 0.242963 | 1.012345 |   2.0       2.0  0.0  0.0       2.0 | 367.611025  371.575759  373.127481  0.0  372.541922  367.502334  371.483044  373.309363  0.0  372.680429 | 143.381057  139.416324  137.864602  0.0  138.450161  143.489749  139.509039   137.68272  0.0  138.311654 |    154.574  133.316538  0.0  0.0  0.0 |      0.0 | 51.589352 | 1.012345 |
    | 03/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 510.992083  510.992083  510.992083  0.0  510.992083  510.992083  510.992083  510.992083  0.0  510.992083 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.696651  226.114467  227.142412  0.0   226.77284 | 87.295432  84.877616  83.849671  0.0  84.219243 |   225.4724    225.4724    225.4724  0.0    225.4724 | 85.519683  85.519683  85.519683  0.0  85.519683 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.257947 |  0.24075 | 0.241925 | 0.241925 | 1.008023 |   2.0       2.0  0.0  0.0       2.0 | 367.006716  370.945478  372.152816  0.0  371.623631  366.964175   370.90919  372.224003  0.0  371.677842 | 143.985367  140.046604  138.839267  0.0  139.368452  144.027908  140.082893   138.76808  0.0  139.314241 |    154.574  133.316538  0.0  0.0  0.0 |      0.0 | 51.331405 | 1.008023 |
    | 03/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 510.992083  510.992083  510.992083  0.0  510.992083  510.992083  510.992083  510.992083  0.0  510.992083 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.348995  225.748046  226.515542  0.0  226.188312 | 87.643088  85.244037  84.476541  0.0  84.803771 | 225.036315  225.036315  225.036315  0.0  225.036315 | 85.955768  85.955768  85.955768  0.0  85.955768 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.256657 | 0.239547 | 0.240752 | 0.240752 | 1.003132 |   2.0       2.0  0.0  0.0       2.0 | 366.334107  370.262705  371.335251  0.0  370.828131  366.317457  370.248502  371.363114  0.0  370.849348 | 144.657976  140.729378  139.656832  0.0  140.163952  144.674626  140.743581  139.628969  0.0  140.142734 |    154.574  133.316538  0.0  0.0  0.0 |      0.0 | 51.074748 | 1.003132 |
    | 03/01 10:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 511.866329  511.866329  511.866329  0.0  511.866329  511.866329  511.866329  511.866329  0.0  511.866329 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 223.049299  225.768829  227.537884  0.0   226.84333 | 88.305784  86.002054  85.170199  0.0  85.435753 | 225.213753  225.213753  225.213753  0.0  225.213753 | 86.652576  86.652576  86.652576  0.0  86.652576 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |      0.0       0.0  0.0   0.363  0.0 |      0.0       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255633 | 0.238591 | 0.239602 | 0.239602 |  0.99834 |   2.0       2.0  0.0  0.0       2.0 | 366.166851  370.103758  371.339154  0.0  370.804528  366.119655  370.063697  371.422949  0.0  370.867798 | 145.699478  141.762571  140.527175  0.0  141.061801  145.746674  141.802632   140.44338  0.0  140.998531 |    154.574  133.316538  0.0  0.0  0.0 |      0.0 | 50.870972 |  0.99834 |
    | 03/01 11:00 |  2.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.168336  519.443129  519.443129  0.0  519.443129  517.656277  519.443129  519.443129  0.0  519.443129 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.146   6.7496   14.872   3.146   11.154 | 0.0       0.0  0.0  0.0       0.0 |  3.146    6.7496   14.872   3.146    11.154 | 223.954879  229.190908  239.596424  0.0  235.920329 |  91.05745  89.425021  87.141905  0.0       87.1 | 230.095971  230.095971  230.095971  0.0  230.095971 | 89.347158  89.347158  89.347158  0.0  89.347158 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 1.530823       0.0  0.0   3.146  0.0 | 0.914403       0.0  0.0   3.146  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022405       0.0  0.0  0.0  0.0 |  0.371666 | 0.371666 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.258195 | 0.240982 | 0.239335 | 0.239335 | 0.997227 |   2.0       2.0  0.0  0.0       2.0 |  370.12024  374.155981   377.07547  0.0  376.263684  369.754484  373.850401  377.709728  0.0  376.747927 | 148.048096  145.287148  142.367659  0.0  143.179446  147.901793  145.592728  141.733401  0.0  142.695202 | 155.212824  133.316538  0.0  0.0  0.0 |      0.0 | 51.380776 | 0.997227 |
    | 03/01 12:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 519.723691  520.894259  520.894259  0.0  520.894259  519.592963  520.894259  520.894259  0.0  520.894259 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 227.313476  230.997088  235.671936  0.0  234.227477 | 91.445831  90.263241  87.775193  0.0  88.218652 | 230.907617  230.907617  230.907617  0.0  230.907617 | 89.986642  89.986642  89.986642  0.0  89.986642 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 1.235932       0.0  0.0   0.847  0.0 | 0.744372       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022148       0.0  0.0  0.0  0.0 |  0.300927 | 0.300927 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.258799 | 0.241545 | 0.240576 | 0.240576 | 1.002399 |   2.0       2.0  0.0  0.0       2.0 | 371.231208  374.851959  377.068076  0.0  376.373353  371.137831   374.65708  377.476482  0.0  376.684537 | 148.492483    146.0423  143.826183  0.0  144.520906  148.455132  146.237178  143.417777  0.0  144.209722 | 155.726532  133.316538  0.0  0.0  0.0 |      0.0 | 51.500915 | 1.002399 |
    | 03/01 13:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 519.923683  521.293147  521.293147  0.0  521.293147  519.867843  521.293147  521.293147  0.0  521.293147 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 228.427097  231.292033  233.951676  0.0  233.109169 |  91.59423  90.381026  88.658583  0.0   89.07209 | 230.943678  230.943678  230.943678  0.0  230.943678 | 90.349468  90.349468  90.349468  0.0  90.349468 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 1.397383       0.0  0.0   0.363  0.0 |  0.84719       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021862       0.0  0.0  0.0  0.0 |  0.343887 | 0.343887 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.259238 | 0.241955 | 0.241511 | 0.241511 | 1.006297 |   2.0       2.0  0.0  0.0       2.0 | 371.374059  374.843343  376.506039  0.0  375.903606  371.334174  374.735386  376.734523  0.0  376.077303 | 148.549624  146.449804  144.787107  0.0  145.389541  148.533669   146.55776  144.558623  0.0  145.215843 | 156.298587  133.316538  0.0  0.0  0.0 |      0.0 | 51.588295 | 1.006297 |
    | 03/01 14:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 519.953348  521.629938  521.629938  0.0  521.629938  519.897553  521.629938  521.629938  0.0  521.629938 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 228.596648  231.423409  233.659598  0.0   232.88721 | 91.662116  90.648538  89.349549  0.0  89.692937 | 230.964837  230.964837  230.964837  0.0  230.964837 | 90.665101  90.665101  90.665101  0.0  90.665101 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 1.704487       0.0  0.0   0.363  0.0 | 1.040983       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021519       0.0  0.0  0.0  0.0 |  0.424777 | 0.424777 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.260021 | 0.242686 | 0.242027 | 0.242027 | 1.008444 |   2.0       2.0  0.0  0.0       2.0 | 371.395248  374.751251   376.20153  0.0  375.634452  371.355395  374.676734   376.36151  0.0  375.755711 | 148.558099  146.878686  145.428407  0.0  145.995486  148.542158  146.953204  145.268428  0.0  145.874227 |  156.98361  133.316538  0.0  0.0  0.0 |      0.0 | 51.744225 | 1.008444 |
    | 03/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 519.192792  520.974366  520.974366  0.0  520.974366  519.192792  520.974366  520.974366  0.0  520.974366 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.518179   231.04356  232.010077  0.0  231.648487 | 91.407272  90.586377  89.619861  0.0  89.981451 | 230.341835  230.341835  230.341835  0.0  230.341835 | 90.632531  90.632531  90.632531  0.0  90.632531 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 1.781574       0.0  0.0     0.0  0.0 | 1.097621       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021166       0.0  0.0  0.0  0.0 |  0.448523 | 0.448523 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.260643 | 0.243267 | 0.242653 | 0.242653 | 1.011054 |   2.0       2.0  0.0  0.0       2.0 | 370.851994  374.066552  375.222609  0.0  374.705184  370.851994  374.037981  375.283947  0.0  374.751676 | 148.340798  146.907813  145.751757  0.0  146.269182  148.340798  146.936384  145.690418  0.0   146.22269 | 157.688729  133.316538  0.0  0.0  0.0 |      0.0 |  51.86803 | 1.011054 |
    | 03/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  518.31052  520.289145  520.289145  0.0  520.289145   518.31052  520.289145  520.289145  0.0  520.289145 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 227.994851   230.45508  231.195028  0.0  230.871667 | 91.197941  90.519286  89.779338  0.0  90.102699 |  229.71164   229.71164   229.71164  0.0   229.71164 | 90.577504  90.577504  90.577504  0.0  90.577504 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 1.978625       0.0  0.0     0.0  0.0 | 1.229999       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020781       0.0  0.0  0.0  0.0 |  0.503841 |      0.5 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.261483 | 0.244051 | 0.243312 | 0.243312 | 1.013801 |   2.0       2.0  0.0  0.0       2.0 |   370.2218  373.354312  374.398117  0.0   373.89948    370.2218  373.343343  374.421665  0.0  373.917328 |  148.08872  146.934833  145.891028  0.0  146.389665   148.08872  146.945802   145.86748  0.0  146.371817 | 158.458136  133.316538  0.0  0.0  0.0 | 0.003841 | 52.035118 | 1.013801 |
    | 03/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 517.399379  519.528135  519.528135  0.0  519.528135  517.399379  519.528135  519.528135  0.0  519.528135 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 227.364657  229.832926  230.486115  0.0  230.177401 | 90.945863  90.456219   89.80303  0.0  90.111744 | 229.060826  229.060826  229.060826  0.0  229.060826 | 90.467309  90.467309  90.467309  0.0  90.467309 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.128756       0.0  0.0     0.0  0.0 | 1.336272       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020375  0.003841  0.0  0.0  0.0 |   0.54701 |      0.5 |      0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.262318 | 0.244831 |  0.24405 |  0.24405 | 1.016875 |   2.0       2.0  0.0  0.0       2.0 | 369.570985  372.618835  373.620076  0.0  373.128399  369.570985  372.614619  373.629127  0.0  373.135259 | 147.828394    146.9093   145.90806  0.0  146.399737  147.828394  146.913516  145.899008  0.0  146.392876 | 159.270994  133.320379  0.0  0.0  0.0 | 0.050851 | 52.201371 | 1.016875 |
    | 03/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  516.46616  518.709383  518.709383  0.0  518.709383   516.46616  518.709383  518.709383  0.0  518.709383 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 226.713842  229.172435  229.792314  0.0  229.489241 | 90.685537    90.3557  89.735822  0.0  90.038894 | 228.394241  228.394241  228.394241  0.0  228.394241 | 90.315142  90.315142  90.315142  0.0  90.315142 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.243222       0.0  0.0     0.0  0.0 |  1.42261       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019954   0.03334  0.0  0.0  0.0 |  0.573326 |      0.5 | 0.000001 | 0.0  0.0  0.0  0.0  0.0 |  0.26315 | 0.245607 |  0.24483 |  0.24483 | 1.020124 |   2.0       2.0  0.0  0.0       2.0 |   368.9044  371.879344  372.864787  0.0  372.375508    368.9044  371.877721  372.868271  0.0   372.37815 |  147.56176  146.830038  145.844596  0.0  146.333875   147.56176  146.831662  145.841112  0.0  146.331233 | 160.111562  133.353718  0.0  0.0  0.0 | 0.124177 | 52.366793 | 1.020124 |
    | 03/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 515.516107  517.846605  517.846605  0.0  517.846605  515.516107  517.846605  517.846605  0.0  517.846605 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 226.047257   228.49245  229.099502  0.0  228.798617 | 90.418903  90.216933  89.609881  0.0  89.910766 | 227.715632  227.715632  227.715632  0.0  227.715632 | 90.130973  90.130973  90.130973  0.0  90.130973 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.330498       0.0  0.0     0.0  0.0 | 1.493599       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019526  0.033323  0.0  0.0  0.0 |   0.60309 |      0.5 | 0.000006 | 0.0  0.0  0.0  0.0  0.0 | 0.263977 | 0.246383 | 0.245607 | 0.245607 | 1.023362 |   2.0       2.0  0.0  0.0       2.0 | 368.225791  371.142027  372.121987  0.0  371.633334  368.225791  371.141402   372.12333  0.0  371.634353 | 147.290316  146.704578  145.724618  0.0  146.213271  147.290316  146.705203  145.723275  0.0  146.212252 | 160.967986  133.387041  0.0  0.0  0.0 | 0.227261 | 52.531387 | 1.023362 |
    | 03/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.553211   516.95026   516.95026  0.0   516.95026  514.553211   516.95026   516.95026  0.0   516.95026 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.368648  227.801308  228.403397  0.0   228.10337 | 90.147459  90.045297  89.443208  0.0  89.743235 | 227.027849  227.027849  227.027849  0.0  227.027849 | 89.922411  89.922411  89.922411  0.0  89.922411 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.397049       0.0  0.0     0.0  0.0 |  1.55273       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019094  0.033306  0.0  0.0  0.0 |  0.627913 |      0.5 | 0.000026 | 0.0  0.0  0.0  0.0  0.0 |   0.2648 | 0.247167 | 0.246385 | 0.246385 | 1.026604 |   2.0       2.0  0.0  0.0       2.0 | 367.538008  370.408376  371.386851  0.0  370.898126  367.538008  370.408134   371.38737  0.0  370.898519 | 147.015203  146.541884  145.563409  0.0  146.052134  147.015203  146.542125   145.56289  0.0   146.05174 | 161.831399  133.420348  0.0  0.0  0.0 | 0.355147 | 52.695159 | 1.026604 |
    | 03/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.580513  516.028318  516.028318  0.0  516.028318  513.580513  516.028318  516.028318  0.0  516.028318 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.680865  227.103073  227.703224  0.0  227.403541 | 89.872346  89.847187  89.247035  0.0  89.546718 | 226.333064  226.333064  226.333064  0.0  226.333064 | 89.695254  89.695254  89.695254  0.0  89.695254 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.447805       0.0  0.0     0.0  0.0 | 1.602664       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018662   0.03329  0.0  0.0  0.0 |  0.648904 |      0.5 | 0.000082 | 0.0  0.0  0.0  0.0  0.0 | 0.265619 | 0.247976 | 0.247173 | 0.247173 | 1.029887 |   2.0       2.0  0.0  0.0       2.0 | 366.843223  369.678293  370.656849  0.0  370.167769  366.843223  369.678199   370.65705  0.0  370.167922 | 146.737289  146.350025  145.371469  0.0  145.860548  146.737289  146.350119  145.371268  0.0  145.860396 | 162.695202  133.453638  0.0  0.0  0.0 |  0.50397 | 52.858112 | 1.029887 |
    | 03/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 512.600329  515.086854  515.086854  0.0  515.086854  512.600329  515.086854  515.086854  0.0  515.086854 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  223.98608  226.399967  226.999349  0.0   226.69981 | 89.594432  89.628351  89.028969  0.0  89.328508 | 225.632933  225.632933  225.632933  0.0  225.632933 | 89.453921  89.453921  89.453921  0.0  89.453921 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.486525       0.0  0.0     0.0  0.0 | 1.645441       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018232  0.033273  0.0  0.0  0.0 |  0.666913 |      0.5 | 0.000208 | 0.0  0.0  0.0  0.0  0.0 | 0.266433 | 0.248837 | 0.247988 | 0.247988 | 1.033283 |   2.0       2.0  0.0  0.0       2.0 | 366.143092   368.95123  369.930495  0.0  369.440939  366.143092  368.951194  369.930572  0.0  369.440998 | 146.457237  146.135624   145.15636  0.0  145.645915  146.457237  146.135661  145.156282  0.0  145.645856 | 163.554518  133.486911  0.0  0.0  0.0 | 0.670675 |  53.02025 | 1.033283 |
    | 03/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 511.614427  514.130498  514.130498  0.0  514.130498  511.614427  514.130498  514.130498  0.0  514.130498 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.285949  225.693348  226.292415  0.0   225.99294 |  89.31438  89.393506  88.794439  0.0  89.093914 | 224.928717  224.928717  224.928717  0.0  224.928717 | 89.201781  89.201781  89.201781  0.0  89.201781 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.516071       0.0  0.0     0.0  0.0 | 1.682628       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.017806  0.033257  0.0  0.0  0.0 |   0.68259 |      0.5 |  0.00045 | 0.0  0.0  0.0  0.0  0.0 | 0.267244 | 0.249788 | 0.248857 | 0.248857 | 1.036905 |   2.0       2.0  0.0  0.0       2.0 | 365.438876  368.226588  369.206821  0.0  368.716734  365.438876  368.226574  369.206851  0.0  368.716757 | 146.175551  145.903911  144.923678  0.0  145.413764  146.175551  145.903925  144.923647  0.0  145.413741 | 164.405767  133.520168  0.0  0.0  0.0 | 0.852815 | 53.181577 | 1.036905 |
    | 04/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.622663  513.162779  513.162779  0.0  513.162779  510.622663  513.162779  513.162779  0.0  513.162779 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0  0.000576  0.0  0.0  0.001152 |    0.0       0.0      0.0     0.0       0.0 | 222.581734  224.984123  225.583053  0.0  225.283611 | 89.032693  89.146375  88.547446  0.0  88.846888 | 224.221379  224.221379  224.221379  0.0  224.221379 | 88.941399  88.941399  88.941399  0.0  88.941399 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.540116       0.0  0.0     0.0  0.0 | 1.716436       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.017385   0.03324  0.0  0.0  0.0 |  0.696858 |      0.5 | 0.000872 | 0.0  0.0  0.0  0.0  0.0 | 0.268051 | 0.250878 | 0.249819 | 0.249819 | 1.040912 |   2.0  1.999424  0.0  0.0  1.998848 | 364.730474   367.50384  368.485156  0.0   367.99451  364.730474  367.503834  368.485168  0.0  367.994519 | 145.892189  145.658939  144.677622  0.0  145.168269  145.892189  145.658944  144.677611  0.0   145.16826 | 165.246832  133.553407  0.0  0.0  0.0 | 1.048801 | 53.342098 | 1.040912 |
    | 04/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 509.628474  512.185811  512.185811  0.0  512.185811  509.628474  512.185811  512.185811  0.0  512.185811 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.873331  224.272935  224.871799  0.0  224.572376 | 88.749332  88.889843  88.290979  0.0  88.590403 | 223.511244  223.511244  223.511244  0.0  223.511244 | 88.674567  88.674567  88.674567  0.0  88.674567 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.557338       0.0  0.0     0.0  0.0 | 1.745799       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016971  0.033223  0.0  0.0  0.0 |  0.709271 |      0.5 | 0.001548 | 0.0  0.0  0.0  0.0  0.0 | 0.268853 | 0.252169 | 0.250922 | 0.250922 |  1.04551 |   2.0  1.999424  0.0  0.0  1.998848 | 364.020338  366.782146  367.764605  0.0   367.27338  364.020338  366.782144  367.764609  0.0  367.273383 | 145.608135  145.403665  144.421207  0.0  144.912431  145.608135  145.403667  144.421202  0.0  144.912428 | 166.075341  133.586631  0.0  0.0  0.0 | 1.256524 | 53.501816 |  1.04551 |
    | 04/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 508.631725   511.20222   511.20222  0.0   511.20222  508.631725   511.20222   511.20222  0.0   511.20222 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.163195  223.559847  224.158674  0.0  223.859264 | 88.465278  88.625964  88.027137  0.0  88.326547 | 222.799281  222.799281  222.799281  0.0  222.799281 | 88.402939  88.402939  88.402939  0.0  88.402939 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.570494       0.0  0.0     0.0  0.0 | 1.772421       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016563  0.033207  0.0  0.0  0.0 |  0.720539 |      0.5 | 0.002569 | 0.0  0.0  0.0  0.0  0.0 | 0.269652 |  0.25373 | 0.252229 | 0.252229 | 1.050953 |   2.0  1.999424  0.0  0.0  1.998848 | 363.308375  366.061579  367.045213  0.0  366.553398  363.308375  366.061579  367.045215  0.0  366.553399 |  145.32335   145.14064  144.157007  0.0  144.648822   145.32335  145.140641  144.157005  0.0   144.64882 | 166.889977  133.619837  0.0  0.0  0.0 | 1.474493 | 53.660735 | 1.050953 |
    | 04/01 03:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.848672  514.001702  514.001702  0.0  514.001702  510.602786  514.001702  514.001702  0.0  514.001702 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |  1.573  3.374224    7.436   1.573  5.575848 | 220.891618   224.69365  230.072254  0.0  228.110529 | 89.313108  89.882793  88.565965  0.0  88.667539 | 224.735383  224.735383  224.735383  0.0  224.735383 | 89.266319  89.266319  89.266319  0.0  89.266319 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 3.275973       0.0  0.0   1.573  0.0 | 2.281082       0.0  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016058   0.03319  0.0  0.0  0.0 |  0.932697 |      0.5 | 0.004912 | 0.0  0.0  0.0  0.0  0.0 |  0.27157 | 0.257395 | 0.254198 | 0.254198 | 1.059157 |   2.0       2.0  0.0  0.0       2.0 | 364.891909   367.70202  369.624935  0.0  368.973297  364.716276  367.558496  369.943751  0.0  369.214905 | 145.956763  146.299682  144.376768  0.0  145.028405   145.88651  146.443206  144.057951  0.0  144.786798 | 167.900926  133.653028  0.0  0.0  0.0 | 1.902278 | 54.042451 | 1.059157 |
    | 04/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.231778  512.741713  512.741713  0.0  512.741713  510.231778  512.741713  512.741713  0.0  512.741713 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  221.94695  224.583939  225.899861  0.0  225.478195 |  88.77878  89.417763  88.101841  0.0  88.523507 | 223.942175  223.942175  223.942175  0.0  223.942175 | 88.799537  88.799537  88.799537  0.0  88.799537 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.509935       0.0  0.0     0.0  0.0 | 1.768922       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015679  0.033173  0.0  0.0  0.0 |   0.71946 |      0.5 | 0.008134 | 0.0  0.0  0.0  0.0  0.0 | 0.272355 | 0.260706 | 0.257316 | 0.257316 | 1.072151 |   2.0       2.0  0.0  0.0       2.0 |  364.45127  367.056418  368.404631  0.0  367.851076   364.45127  367.000572  368.528684  0.0  367.945087 | 145.780508  145.685295  144.337082  0.0  144.890636  145.780508  145.741141  144.213029  0.0  144.796626 | 168.657618  133.686201  0.0  0.0  0.0 | 2.113604 | 54.198667 | 1.072151 |
    | 04/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 509.241805  511.776353  511.776353  0.0  511.776353  509.241805  511.776353  511.776353  0.0  511.776353 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.594127  223.865383  224.742576  0.0  224.395779 | 88.637651  88.876329  87.999136  0.0  88.345934 | 223.235052  223.235052  223.235052  0.0  223.235052 | 88.541301  88.541301  88.541301  0.0  88.541301 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.534548       0.0  0.0     0.0  0.0 | 1.802405       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015305  0.033157  0.0  0.0  0.0 |  0.733573 |      0.5 | 0.011063 | 0.0  0.0  0.0  0.0  0.0 | 0.273136 | 0.263777 | 0.260653 | 0.260653 | 1.086052 |   2.0       2.0  0.0  0.0       2.0 | 363.744146  366.449055  367.574939  0.0   367.05902  363.744146  366.427272  367.623327  0.0  367.095689 | 145.497659  145.327298  144.201414  0.0  144.717334  145.497659  145.349081  144.153026  0.0  144.680664 | 169.405066  133.719358  0.0  0.0  0.0 | 2.336114 | 54.354103 | 1.086052 |
    | 04/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  508.24847  510.801527  510.801527  0.0  510.801527   508.24847  510.801527  510.801527  0.0  510.801527 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 220.887004  223.235704  223.942971  0.0  223.625101 | 88.354801   88.54065  87.833382  0.0  88.151252 | 222.525527  222.525527  222.525527  0.0  222.525527 |    88.276     88.276     88.276  0.0     88.276 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.553057       0.0  0.0     0.0  0.0 | 1.831696       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.014937   0.03314  0.0  0.0  0.0 |  0.745936 |      0.5 | 0.014799 | 0.0  0.0  0.0  0.0  0.0 | 0.273913 | 0.267492 |  0.26392 |  0.26392 | 1.099668 |   2.0       2.0  0.0  0.0       2.0 | 363.034621  365.772243  366.812234  0.0  366.310615  363.034621   365.76373  366.831143  0.0  366.324945 | 145.213848  145.029284  143.989293  0.0  144.490912  145.213848  145.037797  143.970383  0.0  144.476582 | 170.141364  133.752498  0.0  0.0  0.0 | 2.567251 | 54.508761 | 1.099668 |
    | 04/01 07:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 508.982907  511.859489  511.859489  0.0  511.859489   508.85003  511.859489  511.859489  0.0  511.859489 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 220.415139  223.550656  226.761452  0.0  225.561605 | 88.680331  89.068071  88.044075  0.0  88.242922 | 223.240329  223.240329  223.240329  0.0  223.240329 |  88.61916   88.61916   88.61916  0.0   88.61916 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.943021       0.0  0.0   0.847  0.0 | 2.129871       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.014523  0.033124  0.0  0.0  0.0 |  0.870354 |      0.5 |  0.02076 | 0.0  0.0  0.0  0.0  0.0 | 0.275292 | 0.273547 | 0.268012 | 0.268012 | 1.116716 |   2.0       2.0  0.0  0.0       2.0 | 363.559219  366.339032  367.853882  0.0  367.270974  363.464307  366.257888  368.035249  0.0  367.408034 | 145.423688  145.520458  144.005607  0.0  144.588516  145.385723  145.601601   143.82424  0.0  144.451455 | 170.969038  133.785622  0.0  0.0  0.0 | 2.916845 |  54.78304 | 1.116716 |
    | 04/01 08:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 509.180927   511.89322   511.89322  0.0   511.89322  509.104939   511.89322   511.89322  0.0   511.89322 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 220.790352   223.74276  226.214602  0.0  225.335486 | 88.610116   89.15513  87.932887  0.0  88.240004 | 223.300161  223.300161  223.300161  0.0  223.300161 | 88.593059  88.593059  88.593059  0.0  88.593059 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.750287       0.0  0.0   0.484  0.0 | 2.009801       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.014145  0.033107  0.0  0.0  0.0 |  0.820487 |      0.5 | 0.028861 | 0.0  0.0  0.0  0.0  0.0 | 0.276404 | 0.281066 | 0.273872 | 0.273872 | 1.141134 |   2.0       2.0  0.0  0.0       2.0 | 363.700662  366.391503  367.873024  0.0  367.295882  363.646385   366.31542  368.043224  0.0  367.424476 | 145.480265  145.501717  144.020196  0.0  144.597338  145.458554    145.5778  143.849995  0.0  144.468743 |  171.72367  133.818729  0.0  0.0  0.0 | 3.208471 | 55.004351 | 1.141134 |
    | 04/01 09:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 508.552825  511.126832  511.126832  0.0  511.126832  508.533814  511.126832  511.126832  0.0  511.126832 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 | 220.850323  223.359065   224.70374  0.0  224.211895 | 88.413609  88.793754   87.76148  0.0  88.110325 | 222.770076  222.770076  222.770076  0.0  222.770076 | 88.356756  88.356756  88.356756  0.0  88.356756 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.583513       0.0  0.0   0.121  0.0 | 1.904632       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.013799  0.033091  0.0  0.0  0.0 |  0.776817 |      0.5 | 0.036994 | 0.0  0.0  0.0  0.0  0.0 | 0.277251 | 0.288363 | 0.281016 | 0.281016 | 1.170902 |   2.0       2.0  0.0  0.0       2.0 | 363.252018  365.939301  367.190709  0.0  366.652811  363.238439  365.898444  367.282213  0.0   366.72191 | 145.300807  145.187532  143.936124  0.0  144.474021  145.295375  145.228389   143.84462  0.0  144.404922 |  172.41635   133.85182  0.0  0.0  0.0 | 3.448293 | 55.172957 | 1.170902 |
    | 04/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 508.578531  511.298835  511.298835  0.0  511.298835   508.50253  511.298835  511.298835  0.0  511.298835 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 220.523914  223.306822  225.576806  0.0  224.732337 | 88.503405  88.858411  87.838027  0.0  88.110495 |  222.86997   222.86997   222.86997  0.0   222.86997 | 88.428865  88.428865  88.428865  0.0  88.428865 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.758305       0.0  0.0   0.484  0.0 | 2.049931       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.013438  0.033074  0.0  0.0  0.0 |  0.837514 |      0.5 | 0.046601 | 0.0  0.0  0.0  0.0  0.0 | 0.278353 | 0.297077 | 0.288678 | 0.288678 | 1.202825 |   2.0       2.0  0.0  0.0       2.0 | 363.270379  366.009873  367.389844  0.0  366.829824  363.216093  365.949368  367.525406  0.0  366.932163 | 145.308152  145.288962  143.908991  0.0  144.469011  145.286437  145.349467  143.773429  0.0  144.366672 | 173.138162  133.884894  0.0  0.0  0.0 | 3.739207 | 55.392318 | 1.202825 |
    | 04/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 507.702392  510.237949  510.237949  0.0  510.237949  507.702392  510.237949  510.237949  0.0  510.237949 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 220.386093  222.822783  223.722651  0.0  223.371575 | 88.154437  88.476052  87.576185  0.0  87.927261 | 222.135471  222.135471  222.135471  0.0  222.135471 | 88.102478  88.102478  88.102478  0.0  88.102478 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.535557       0.0  0.0     0.0  0.0 | 1.900198       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.013113  0.033058  0.0  0.0  0.0 |  0.775266 |      0.5 | 0.057145 | 0.0  0.0  0.0  0.0  0.0 | 0.279104 | 0.306214 | 0.297171 | 0.297171 | 1.238212 |   2.0       2.0  0.0  0.0       2.0 | 362.644566  365.344142  366.483932  0.0  365.964874  362.644566  365.320475  366.536959  0.0  366.004906 | 145.057826  144.893807  143.754016  0.0  144.273074  145.057826  144.917474   143.70099  0.0  144.233043 | 173.786633  133.917951  0.0  0.0  0.0 | 3.957328 | 55.541785 | 1.238212 |
    | 04/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 506.708842  509.262734  509.262734  0.0  509.262734  506.708842  509.262734  509.262734  0.0  509.262734 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.787423  222.131549  222.847887  0.0  222.528336 | 87.914969    88.1064  87.390062  0.0  87.709613 | 221.425793  221.425793  221.425793  0.0  221.425793 | 87.836942  87.836942  87.836942  0.0  87.836942 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.553892       0.0  0.0     0.0  0.0 | 1.928303       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.012794  0.033041  0.0  0.0  0.0 |  0.787115 |      0.5 |  0.06746 | 0.0  0.0  0.0  0.0  0.0 | 0.279852 | 0.315163 | 0.306172 | 0.306172 | 1.275717 |   2.0       2.0  0.0  0.0       2.0 | 361.934887  364.671191  365.717959  0.0  365.214502  361.934887  364.661914  365.738744  0.0  365.230193 | 144.773955  144.591544  143.544776  0.0  144.048233  144.773955  144.600821  143.523991  0.0  144.032542 | 174.425016  133.950992  0.0  0.0  0.0 | 4.176983 | 55.690505 | 1.275717 |
    | 04/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 505.712643  508.280468  508.280468  0.0  508.280468  505.712643  508.280468  508.280468  0.0  508.280468 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.077744  221.453029  222.097833  0.0   221.79055 | 87.631098  87.809706  87.164901  0.0  87.472185 | 220.714222  220.714222  220.714222  0.0  220.714222 | 87.566246  87.566246  87.566246  0.0  87.566246 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.567825       0.0  0.0     0.0  0.0 | 1.953093       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01248  0.033025  0.0  0.0  0.0 |  0.797581 |      0.5 |  0.07896 | 0.0  0.0  0.0  0.0  0.0 | 0.280595 | 0.325057 | 0.315373 | 0.315373 | 1.314054 |   2.0       2.0  0.0  0.0       2.0 | 361.223317  363.969567  364.980659  0.0  364.482939  361.223317  363.965924  364.988821  0.0  364.489101 | 144.489327  144.310901   143.29981  0.0   143.79753  144.489327  144.314545  143.291647  0.0  143.791367 | 175.052228  133.984017  0.0  0.0  0.0 | 4.395604 | 55.838481 | 1.314054 |
    | 04/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 504.714358  507.292843  507.292843  0.0  507.292843  504.714358  507.292843  507.292843  0.0  507.292843 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 218.366174  220.752427  221.369251  0.0  221.066769 | 87.346469  87.528041  86.911217  0.0  87.213699 | 220.001161  220.001161  220.001161  0.0  220.001161 | 87.291682  87.291682  87.291682  0.0  87.291682 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.578485       0.0  0.0     0.0  0.0 | 1.975331       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.012172  0.033008  0.0  0.0  0.0 |   0.80698 |      0.5 | 0.091566 | 0.0  0.0  0.0  0.0  0.0 | 0.281335 | 0.335833 | 0.325253 | 0.325253 |  1.35522 |   2.0       2.0  0.0  0.0       2.0 | 360.510256  363.257197  364.255044  0.0    363.7592  360.510256  363.255763  364.258256  0.0  363.761625 | 144.204102  144.035647  143.037799  0.0  143.533643  144.204102   144.03708  143.034587  0.0  143.531219 | 175.667554  134.017025  0.0  0.0  0.0 | 4.611017 | 55.985717 |  1.35522 |
    | 04/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 503.714473  506.301118  506.301118  0.0  506.301118  503.714473  506.301118  506.301118  0.0  506.301118 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 217.653113  220.042758  220.648612  0.0  220.348016 | 87.061245  87.250085  86.644231  0.0  86.944828 | 219.286958  219.286958  219.286958  0.0  219.286958 | 87.014161  87.014161  87.014161  0.0  87.014161 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.586645       0.0  0.0     0.0  0.0 | 1.995538       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.011871  0.032991  0.0  0.0  0.0 |  0.815531 |      0.5 | 0.105171 | 0.0  0.0  0.0  0.0  0.0 | 0.282071 | 0.347404 | 0.336009 | 0.336009 | 1.400039 |   2.0       2.0  0.0  0.0       2.0 | 359.796052  362.540962  363.534363  0.0  363.038876  359.796052  362.540397  363.535629  0.0  363.039832 | 143.918421  143.760156  142.766755  0.0  143.262242  143.918421  143.760722  142.765489  0.0  143.261286 | 176.270532  134.050016  0.0  0.0  0.0 | 4.821377 | 56.132217 | 1.400039 |
    | 04/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 502.713359  505.306255  505.306255  0.0  505.306255  502.713359  505.306255  505.306255  0.0  505.306255 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.938909  219.329193  219.930733  0.0   219.63088 | 86.775564  86.971926  86.370386  0.0  86.670238 | 218.571876  218.571876  218.571876  0.0  218.571876 | 86.734378  86.734378  86.734378  0.0  86.734378 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.592895       0.0  0.0     0.0  0.0 | 2.014116       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.011575  0.032975  0.0  0.0  0.0 |    0.8234 |      0.5 | 0.119635 | 0.0  0.0  0.0  0.0  0.0 | 0.282804 | 0.359658 | 0.347555 | 0.347555 | 1.448148 |   2.0       2.0  0.0  0.0       2.0 | 359.080971  361.823475   362.81589  0.0  362.320162  359.080971  361.823251   362.81639  0.0   362.32054 | 143.632388   143.48278  142.490365  0.0  142.986093  143.632388  143.483003  142.489865  0.0  142.985715 | 176.860887  134.082991  0.0  0.0  0.0 | 5.025143 | 56.277984 | 1.448148 |
    | 04/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 501.711302  504.308987  504.308987  0.0  504.308987  501.711302  504.308987  504.308987  0.0  504.308987 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.223828   218.61383  219.213667  0.0   218.91411 | 86.489531  86.692424  86.092588  0.0  86.392144 | 217.856121  217.856121  217.856121  0.0  217.856121 | 86.452866  86.452866  86.452866  0.0  86.452866 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.597686       0.0  0.0     0.0  0.0 | 2.031375       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.011286  0.032959  0.0  0.0  0.0 |  0.830718 |      0.5 | 0.134796 | 0.0  0.0  0.0  0.0  0.0 | 0.283533 | 0.372467 | 0.359781 | 0.359781 | 1.499089 |   2.0       2.0  0.0  0.0       2.0 | 358.365215  361.105698  362.098495  0.0  361.602286  358.365215  361.105609  362.098693  0.0  361.602436 | 143.346086  143.203289  142.210492  0.0  142.706701  143.346086  143.203378  142.210294  0.0  142.706551 | 177.438484   134.11595  0.0  0.0  0.0 | 5.221064 | 56.423023 | 1.499089 |
    | 04/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 500.708518  503.309877  503.309877  0.0  503.309877  500.708518  503.309877  503.309877  0.0  503.309877 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 215.508073  217.897556  218.496714  0.0  218.197278 | 86.203229  86.411431  85.812273  0.0  86.111709 | 217.139847  217.139847  217.139847  0.0  217.139847 | 86.170031  86.170031  86.170031  0.0  86.170031 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |  2.60136       0.0  0.0     0.0  0.0 | 2.047557       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.011004  0.032942  0.0  0.0  0.0 |  0.837583 |      0.5 | 0.150478 | 0.0  0.0  0.0  0.0  0.0 | 0.284258 |  0.38569 | 0.372559 | 0.372559 |  1.55233 |   2.0       2.0  0.0  0.0       2.0 | 357.648941  360.387962  361.381688  0.0    360.8849  357.648941  360.387927  361.381766  0.0  360.884959 | 143.059576  142.921915   141.92819  0.0  142.424977  143.059576   142.92195  141.928111  0.0  142.424918 |  178.00329  134.148892  0.0  0.0  0.0 |  5.40817 | 56.567336 |  1.55233 |
    | 04/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 499.705174  502.309354  502.309354  0.0  502.309354  499.705174  502.309354  502.309354  0.0  502.309354 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.791798  217.180763  217.779647  0.0  217.480262 | 85.916719  86.129114  85.530231  0.0  85.829616 | 216.423173  216.423173  216.423173  0.0  216.423173 | 85.886182  85.886182  85.886182  0.0  85.886182 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |  2.60418       0.0  0.0     0.0  0.0 | 2.062847       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.010728  0.032926  0.0  0.0  0.0 |  0.844075 |      0.5 | 0.166496 | 0.0  0.0  0.0  0.0  0.0 |  0.28498 | 0.399178 | 0.385749 | 0.385749 | 1.607287 |   2.0       2.0  0.0  0.0       2.0 | 356.932267  359.670361  360.665237  0.0  360.167829  356.932267  359.670347  360.665269  0.0  360.167853 | 142.772907  142.638993  141.644117  0.0  142.141525  142.772907  142.639007  141.644086  0.0  142.141502 | 178.555351  134.181818  0.0  0.0  0.0 | 5.585748 | 56.710928 | 1.607287 |
    | 04/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 498.701399  501.307747  501.307747  0.0  501.307747  498.701399  501.307747  501.307747  0.0  501.307747 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.075124  216.463643  217.062411  0.0  216.763049 |  85.63005  85.845712  85.246943  0.0  85.546305 |  215.70619   215.70619   215.70619  0.0   215.70619 | 85.601556  85.601556  85.601556  0.0  85.601556 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.606348       0.0  0.0     0.0  0.0 |  2.07739       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.010458  0.032909  0.0  0.0  0.0 |  0.850252 |      0.5 | 0.182667 | 0.0  0.0  0.0  0.0  0.0 | 0.285697 | 0.412785 | 0.399204 | 0.399204 | 1.663351 |   2.0       2.0  0.0  0.0       2.0 | 356.215285  358.952905  359.949026  0.0  359.450977  356.215285  358.952899  359.949038  0.0  359.450987 | 142.486114  142.354842  141.358721  0.0  141.856769  142.486114  142.354847  141.358708  0.0   141.85676 | 179.094766  134.214727  0.0  0.0  0.0 | 5.753333 | 56.853802 | 1.663351 |
    | 04/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  497.69729  500.305305  500.305305  0.0  500.305305   497.69729  500.305305  500.305305  0.0  500.305305 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 213.358142  215.746297  216.345014  0.0  216.045665 | 85.343257  85.561449  84.962732  0.0  85.262082 |  214.98897   214.98897   214.98897  0.0   214.98897 | 85.316335  85.316335  85.316335  0.0  85.316335 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.608015       0.0  0.0     0.0  0.0 | 2.091298       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.010194  0.032893  0.0  0.0  0.0 |  0.856162 |      0.5 | 0.198816 | 0.0  0.0  0.0  0.0  0.0 | 0.286412 | 0.426371 |  0.41278 |  0.41278 | 1.719917 |   2.0       2.0  0.0  0.0       2.0 | 355.498064  358.235576  359.232985  0.0  358.734285  355.498064  358.235574   359.23299  0.0  358.734289 | 142.199226  142.069729   141.07232  0.0   141.57102  142.199226  142.069732  141.072315  0.0  141.571016 | 179.621677  134.247619  0.0  0.0  0.0 | 5.910679 | 56.995962 | 1.719917 |
    | 04/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 496.692922  499.302222  499.302222  0.0  499.302222  496.692922  499.302222  499.302222  0.0  499.302222 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 212.640921  215.028788  215.627479  0.0  215.328137 | 85.056369  85.276517  84.677826  0.0  84.977168 | 214.271564  214.271564  214.271564  0.0  214.271564 | 85.030658  85.030658  85.030658  0.0  85.030658 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 |   2.6093       0.0  0.0     0.0  0.0 | 2.104658       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.009937  0.032876  0.0  0.0  0.0 |  0.861842 |      0.5 | 0.214783 | 0.0  0.0  0.0  0.0  0.0 | 0.287123 | 0.439808 | 0.426338 | 0.426338 | 1.776407 |   2.0       2.0  0.0  0.0       2.0 | 354.780659  357.518351  358.517071  0.0  358.017713  354.780659   357.51835  358.517073  0.0  358.017714 | 141.912263  141.783871  140.785151  0.0  141.284509  141.912263  141.783872  140.785149  0.0  141.284508 | 180.136256  134.280495  0.0  0.0  0.0 | 6.057738 | 57.137411 | 1.776407 |
    | 04/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 495.688353  498.298645  498.298645  0.0  498.298645  495.688353  498.298645  498.298645  0.0  498.298645 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 211.923516  214.311157  214.909831  0.0  214.610496 | 84.769406  84.991066  84.392391  0.0  84.691727 | 213.554015  213.554015  213.554015  0.0  213.554015 |  84.74463   84.74463   84.74463  0.0   84.74463 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04  0.64  0.0  0.34  0.0  0.04  0.64  0.0  0.34 | 2.610292       0.0  0.0     0.0  0.0 | 2.117539       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.009685   0.03286  0.0  0.0  0.0 |  0.867319 |      0.5 | 0.230427 | 0.0  0.0  0.0  0.0  0.0 |  0.28783 | 0.452983 |  0.43975 |  0.43975 |  1.83229 |   2.0       2.0  0.0  0.0       2.0 |  354.06311  356.801209  357.801255  0.0  357.301233   354.06311  356.801209  357.801256  0.0  357.301234 | 141.625244  141.497436   140.49739  0.0  140.997412  141.625244  141.497437  140.497389  0.0  140.997412 | 180.638695  134.313355  0.0  0.0  0.0 |  6.19463 | 57.278152 |  1.83229 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.interfaces import aetinterfaces
from hydpy.interfaces import rconcinterfaces

# ...from hland
from hydpy.models.hland import hland_model
from hydpy.models.hland import hland_masks
from hydpy.models.hland.hland_constants import *


class Model(
    hland_model.Main_AETModel_V1,
    hland_model.Main_RConcModel_V1,
    hland_model.Sub_TempModel_V1,
    hland_model.Sub_PrecipModel_V1,
    hland_model.Sub_IntercModel_V1,
    hland_model.Sub_SoilWaterModel_V1,
    hland_model.Sub_SnowCoverModel_V1,
):
    """|hland_96.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="H-HBV96", description="adoption of SMHI-IHMS-HBV96"
    )
    __HYDPY_ROOTMODEL__ = True

    INLET_METHODS = ()
    OBSERVER_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        hland_model.Calc_TC_V1,
        hland_model.Calc_FracRain_V1,
        hland_model.Calc_RFC_SFC_V1,
        hland_model.Calc_PC_V1,
        hland_model.Calc_TF_Ic_V1,
        hland_model.Calc_SP_WC_V1,
        hland_model.Calc_SPL_WCL_SP_WC_V1,
        hland_model.Calc_SPG_WCG_SP_WC_V1,
        hland_model.Calc_CFAct_V1,
        hland_model.Calc_Melt_SP_WC_V1,
        hland_model.Calc_Refr_SP_WC_V1,
        hland_model.Calc_In_WC_V1,
        hland_model.Calc_SWE_V1,
        hland_model.Calc_SR_V1,
        hland_model.Calc_GAct_V1,
        hland_model.Calc_GlMelt_In_V1,
        hland_model.Calc_EI_Ic_V1,
        hland_model.Calc_R_SM_V1,
        hland_model.Calc_CF_SM_V1,
        hland_model.Calc_EA_SM_V1,
        hland_model.Calc_InUZ_V1,
        hland_model.Calc_ContriArea_V1,
        hland_model.Calc_Q0_Perc_UZ_V1,
        hland_model.Calc_LZ_V1,
        hland_model.Calc_EL_LZ_V1,
        hland_model.Calc_Q1_LZ_V1,
        hland_model.Calc_InRC_V1,
        hland_model.Calc_OutRC_V1,
        hland_model.Calc_RT_V1,
        hland_model.Calc_QT_V1,
    )
    INTERFACE_METHODS = (
        hland_model.Get_Temperature_V1,
        hland_model.Get_MeanTemperature_V1,
        hland_model.Get_Precipitation_V1,
        hland_model.Get_InterceptedWater_V1,
        hland_model.Get_SoilWater_V1,
        hland_model.Get_SnowCover_V1,
    )
    ADD_METHODS = (
        hland_model.Calc_EI_Ic_AETModel_V1,
        hland_model.Calc_EA_SM_AETModel_V1,
        hland_model.Calc_EL_LZ_AETModel_V1,
        hland_model.Calc_OutRC_RConcModel_V1,
    )
    OUTLET_METHODS = (hland_model.Pass_Q_V1,)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (aetinterfaces.AETModel_V1, rconcinterfaces.RConcModel_V1)
    SUBMODELS = ()

    aetmodel = modeltools.SubmodelProperty(aetinterfaces.AETModel_V1)
    rconcmodel = modeltools.SubmodelProperty(
        rconcinterfaces.RConcModel_V1, optional=True
    )

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            \Sigma In_{hru} - \Sigma Out_{hru} - \Sigma Out_{basin}
            + \Delta Vol_{hru} + \Delta Vol_{snow} + \Delta Vol_{basin}
            - \Delta Vol_{rconc}
            \\ \\
            \Sigma In_{hru} =
            \sum_{k=1}^{N_{hru}} A_Z^k \cdot \sum_{t=t0}^{t1} PC_t^k + GLMelt_t^k
            \\
            \Sigma Out_{hru} =
            \sum_{k=1}^{N_{hru}} A_Z^k \cdot \sum_{t=t0}^{t1} EI_t^k + EA_t^k + EL_t^k
            \\
            \Sigma Out_{basin} = \sum_{t=t0}^{t1} RT_t
            \\
            \Delta Vol_{snow} = \sum_{k=1}^{N_{hru}} A_Z^k \cdot \frac{1}{N_{snow}}
            \cdot \sum_{c=1}^{N_{snow}} \big(SP_{t0}^{k,s} - SP_{t1}^{k,s}\big) +
            \big(WC_{t0}^{k,s} - WC_{t1}^{k,s}\big)
            \\
            \Delta Vol_{hru} = \sum_{k=1}^{N_{hru}} A_Z^k \cdot \Big(
            \big(IC_{t0}^k - IC_{t1}^k\big) + \big(SM_{t0}^k - SM_{t1}^k\big) \Big)
            \\
            \Delta Vol_{basin} = A_U \cdot
            \big(UZ_{t0} - UZ_{t1}\big) + A_L \cdot \big(LZ_{t0} - LZ_{t1}\big)
            \\
            \Delta Vol_{rconc} = \begin{cases}
            rconcmodel.get\_waterbalance &|\ rconcmodel \\
            0 &|\ \overline{rconcmodel} \end{cases}
            \\ \\
            N_{hru} = NmbZones \\
            N_{snow} = SClass \\
            A_Z = RelZoneAreas \\
            A_U = RelUpperZoneArea \\
            A_L = RelLowerZoneArea

        The returned error should always be in scale with numerical precision so that
        it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run via
        property |Sequences.conditions|.  See the integration tests of the application
        model |hland_96| for some examples.
        """
        derived = self.parameters.derived
        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["model"]["states"]
        areas = derived.relzoneareas.value
        idxs_lake = self.parameters.control.zonetype.values == ILAKE
        idxs_land = ~idxs_lake
        idxs_glac = self.parameters.control.zonetype.values == GLACIER
        idxs_seal = self.parameters.control.zonetype.values == SEALED
        idxs_soil = idxs_land * ~idxs_glac * ~idxs_seal
        return (
            numpy.sum(fluxes.pc.series * areas)
            + numpy.sum((fluxes.glmelt.series * areas)[:, idxs_glac])
            - numpy.sum((fluxes.ei.series * areas)[:, idxs_land])
            - numpy.sum((fluxes.ea.series * areas)[:, idxs_soil])
            - numpy.sum((fluxes.el.series * areas)[:, idxs_lake])
            - numpy.sum(fluxes.rt.series)
            - numpy.sum(((last.ic - first["ic"]) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.wc - first["wc"], axis=0) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.sp - first["sp"], axis=0) * areas)[idxs_land])
            - numpy.sum(((last.sm - first["sm"]) * areas)[idxs_soil])
            - (last.uz - first["uz"]) * derived.relupperzonearea
            - (last.lz - first["lz"]) * derived.rellowerzonearea
            - self._get_rconcmodel_waterbalance(initial_conditions)
        )


class Masks(masktools.Masks):
    """Masks applicable to |hland_96|."""

    CLASSES = hland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
