# pylint: disable=line-too-long, unused-wildcard-import
"""|kinw_impl_euler| is a kinematic wave routing approach that does not suffer from
stability issues due to high Courant numbers.  Its robustness is a result of solving
the underlying (non-linear) ordinary differential equations with the implicit Euler
method (also known as the Backward Euler method).  For the sake of computational
efficiency, we implemented this method without any adaptive error control.  Hence, it
might sometimes be less accurate than routing models as |musk_mct| (with respect to
physical processes) or |kinw_williams| (with respect to the original differential
equations), but it works efficiently when the represented channels are relatively short
and the simulation step is relatively long (where the simulation with |musk_mct|
produces unreliable results and the simulation with |kinw_williams| takes ages).
Hence, |kinw_impl_euler| is best suited for longer (e.g., daily) simulation steps,
where the fine details of routing processes are lost anyway.

Note that, in comparison with |kinw_williams|, |kinw_impl_euler| is more flexible as it
uses submodels for calculating channel geometry properties and discharges (like
|musk_mct| and the members of the more hydrodynamic model family :ref:`HydPy-SW1d` do).

Integration tests
=================

.. how_to_understand_integration_tests::

For good comparability with |musk_mct| and |kinw_williams|, we define the following
test settings quasi-identical to the ones of the :ref:`musk_mct_base_example` on
|musk_mct| and at least very similar to the ones of the
:ref:`kinw_williams_main_channel_flow` on |kinw_williams| (based on an original test
case taken from :cite:t:`ref-Todini2007`):

>>> from hydpy.models.kinw_impl_euler import *
>>> parameterstep()

>>> from hydpy import pub, Nodes, Element
>>> pub.timegrids = "2000-01-01", "2000-01-05", "30m"

>>> nodes = Nodes("input1", "input2", "output")
>>> stream = Element("stream", inlets=["input1", "input2"], outlets="output")
>>> stream.model = model

>>> length(100.0)
>>> nmbsegments(8)
>>> with model.add_wqmodel_v1("wq_trapeze_strickler"):
...     nmbtrapezes(1)
...     bottomlevels(0.0)
...     bottomwidths(15.0)
...     bottomslope(0.00025)
...     sideslopes(5.0)
...     stricklercoefficients(1.0/0.035)

>>> from hydpy.core.testtools import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = (
...     fluxes.inflow, fluxes.internalflow, fluxes.outflow
... )
>>> test = IntegrationTest(stream)

>>> import numpy
>>> q_base = 100.0
>>> q_peak = 900.0
>>> t_peak = 24.0
>>> beta = 16.0
>>> ts = pub.timegrids.init.to_timepoints()
>>> nodes.input1.sequences.sim.series = q_base
>>> nodes.input2.sequences.sim.series = (
...     (q_peak-q_base)*((ts/t_peak)*numpy.exp(1.0-ts/t_peak))**beta)

.. _kinw_impl_euler_base_example:

base example
____________

The simulation results agree well with those of |kinw_williams| and, in particular,
with those of |musk_mct|.  Compared with the latter, the peak flow is underestimated by
less than 3 %:

.. integration-test::

    >>> model.prepare_states(100.0)
    >>> conditions = test("kinw_impl_euler_base_example", get_conditions="2000-01-01")
    |                date |                                                                       waterdepth |     inflow |                                                                         internalflow |    outflow |                                                                       watervolume | input1 |     input2 |     output |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 1.560986  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 1.560986  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 1.560986  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 1.560986  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 1.560986  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 1.560986  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000003 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 1.560986  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000029 | 100.000004  100.000001       100.0       100.0       100.0       100.0         100.0 |      100.0 | 1.560986  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000029 |      100.0 |
    | 2000-01-01 04:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000175 | 100.000027  100.000004  100.000001       100.0       100.0       100.0         100.0 |      100.0 | 1.560986  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000175 |      100.0 |
    | 2000-01-01 04:30:00 | 3.717835  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000826 | 100.000136  100.000022  100.000004  100.000001       100.0       100.0         100.0 |      100.0 | 1.560988  1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000826 |      100.0 |
    | 2000-01-01 05:00:00 | 3.717842  3.717834  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.003194 |  100.00055  100.000094  100.000016  100.000003       100.0       100.0         100.0 |      100.0 | 1.560992  1.560987  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.003194 |      100.0 |
    | 2000-01-01 05:30:00 | 3.717866  3.717839  3.717834  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.010515 |   100.0019  100.000338  100.000059   100.00001  100.000002       100.0         100.0 |      100.0 | 1.561008   1.56099  1.560987  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.010515 |      100.0 |
    | 2000-01-01 06:00:00 | 3.717934  3.717852  3.717836  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.030315 |  100.00575  100.001072  100.000197  100.000036  100.000006  100.000001         100.0 |      100.0 | 1.561052  1.560998  1.560988  1.560987  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.030315 |      100.0 |
    | 2000-01-01 06:30:00 | 3.718107  3.717886  3.717843  3.717835  3.717833  3.717833  3.717833    3.717833 | 100.078179 | 100.015564  100.003035  100.000581   100.00011   100.00002  100.000004    100.000001 |      100.0 | 1.561165  1.561021  1.560993  1.560987  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.078179 |      100.0 |
    | 2000-01-01 07:00:00 | 3.718507   3.71797   3.71786  3.717838  3.717834  3.717833  3.717833    3.717833 | 100.183351 | 100.038299  100.007813  100.001561  100.000306  100.000059  100.000011    100.000002 |      100.0 | 1.561426  1.561076  1.561004   1.56099  1.560987  1.560986  1.560986     1.560986 |  100.0 |   0.183351 |      100.0 |
    | 2000-01-01 07:30:00 | 3.719361  3.718159  3.717901  3.717847  3.717836  3.717833  3.717833    3.717833 | 100.396214 |   100.0868  100.018516  100.003858  100.000787  100.000158  100.000031    100.000006 | 100.000001 | 1.561983  1.561199   1.56103  1.560995  1.560988  1.560987  1.560986     1.560986 |  100.0 |   0.396214 | 100.000001 |
    | 2000-01-01 08:00:00 | 3.721055  3.718551  3.717989  3.717866   3.71784  3.717834  3.717833    3.717833 | 100.797305 | 100.183096  100.040817  100.008866  100.001882  100.000391   100.00008    100.000016 | 100.000003 | 1.563089  1.561455  1.561088  1.561008  1.560991  1.560987  1.560986     1.560986 |  100.0 |   0.797305 | 100.000003 |
    | 2000-01-01 08:30:00 | 3.724211  3.719319  3.718169  3.717907  3.717849  3.717836  3.717833    3.717833 | 101.507032 | 100.362585   100.08442  100.019103  100.004215   100.00091  100.000192     100.00004 | 100.000008 | 1.565149  1.561956  1.561206  1.561035  1.560997  1.560988  1.560987     1.560986 |  100.0 |   1.507032 | 100.000008 |
    | 2000-01-01 09:00:00 | 3.729764  3.720737  3.718517   3.71799  3.717868   3.71784  3.717834    3.717833 | 102.694853 | 100.678946  100.164995  100.038872  100.008911  100.001994  100.000436    100.000094 |  100.00002 | 1.568777  1.562881  1.561433  1.561089  1.561009  1.560991  1.560987     1.560986 |  100.0 |   2.694853 |  100.00002 |
    | 2000-01-01 09:30:00 | 3.739054  3.723226  3.719156  3.718148  3.717906  3.717849  3.717836    3.717834 | 104.586356 | 101.209449  100.306583  100.075149  100.017886  100.004147  100.000939    100.000208 | 100.000045 | 1.574856  1.564506  1.561849  1.561192  1.561034  1.560997  1.560989     1.560987 |  100.0 |   4.586356 | 100.000045 |
    | 2000-01-01 10:00:00 | 3.753893  3.727404  3.720275  3.718436  3.717978  3.717867  3.717841    3.717835 | 107.466641 | 102.060236  100.544408  100.138745  100.034262  100.008227  100.001927    100.000441 | 100.000099 | 1.584587  1.567234  1.562579   1.56138  1.561081  1.561008  1.560991     1.560987 |  100.0 |   7.466641 | 100.000099 |
    | 2000-01-01 10:30:00 | 3.776614  3.734129  3.722157   3.71894  3.718108  3.717899  3.717849    3.717836 | 111.678559 | 103.371034  100.928003  100.245728  100.062917  100.015638  100.003784    100.000894 | 100.000207 | 1.599541  1.571632  1.563807  1.561709  1.561166   1.56103  1.560996     1.560989 |  100.0 |  11.678559 | 100.000207 |
    | 2000-01-01 11:00:00 | 3.810064  3.744558  3.725204   3.71979  3.718336  3.717959  3.717863     3.71784 | 117.614836 | 105.318823    101.5245  100.419116  100.111189  100.028585  100.007145    100.001741 | 100.000415 | 1.621674  1.578462  1.565797  1.562263  1.561314  1.561068  1.561006     1.560991 |  100.0 |  17.614836 | 100.000415 |
    | 2000-01-01 11:30:00 | 3.857536   3.76018  3.729972  3.721172  3.718721  3.718062   3.71789    3.717847 | 125.703736 | 108.119985   102.42193  100.690808  100.189748  100.050423  100.013009    100.003268 | 100.000801 | 1.653324  1.588718  1.568913  1.563165  1.561565  1.561136  1.561024     1.560995 |  100.0 |  25.703736 | 100.000801 |
    | 2000-01-01 12:00:00 |  3.92261  3.782847  3.737205  3.723351  3.719348  3.718236  3.717937    3.717859 | 136.388647 | 112.030547  103.732402  101.103719  100.313651  100.086094  100.022912    100.005929 | 100.001496 | 1.697169  1.603655  1.573645  1.564587  1.561975  1.561249  1.561054     1.561003 |  100.0 |  36.388647 | 100.001496 |
    | 2000-01-01 12:30:00 | 4.008915  3.814777  3.747865  3.726687  3.720344  3.718522  3.718016     3.71788 | 150.102726 |  117.34411  105.594985  101.714099  100.503569  100.142672  100.039139    100.010429 | 100.002706 | 1.756134  1.624803   1.58063  1.566766  1.562624  1.561436  1.561106     1.561017 |  100.0 |  50.102726 | 100.002706 |
    | 2000-01-01 13:00:00 | 4.119816  3.858513  3.763165  3.731662  3.721881  3.718977  3.718147    3.717916 | 167.240349 | 124.386985  108.178138  102.593982  100.787199  100.230036  100.065007    100.017824 | 100.004755 |  1.83327  1.653979  1.590682  1.570018  1.563627  1.561733  1.561191     1.561041 |  100.0 |  67.240349 | 100.004755 |
    | 2000-01-01 13:30:00 | 4.258065  3.916856  3.784593  3.738904  3.724194  3.719685  3.718355    3.717976 | 188.127569 | 133.509905  111.681431  103.833731  101.200867   100.36166  100.105209    100.029665 |  100.00813 | 1.931582   1.69327  1.604808  1.574757  1.565138  1.562194  1.561327      1.56108 |  100.0 |  88.127569 |  100.00813 |
    | 2000-01-01 14:00:00 | 4.425484  3.992724  3.813918  3.749211  3.727599  3.720759  3.718681    3.718071 | 212.993962 | 145.075607  116.336168  105.544616  101.791309  100.555559  100.166246    100.048172 | 100.013555 | 2.053835  1.745001  1.624233  1.581513  1.567362  1.562895  1.561539     1.561142 |  100.0 | 112.993962 | 100.013555 |
    | 2000-01-01 14:30:00 | 4.622708   4.08898  3.853182  3.763576  3.732507  3.722354  3.719179    3.718222 | 241.948195 | 159.441754  122.404309  107.861294  102.617642  100.835375  100.256949    100.076463 | 100.022079 | 2.202347  1.811669   1.65041  1.590952   1.57057  1.563936  1.561864      1.56124 |  100.0 | 141.948195 | 100.022079 |
    | 2000-01-01 15:00:00 | 4.849057  4.208197  3.904666  3.783211  3.739442  3.724677  3.719925    3.718452 |  274.95935 | 176.939223  130.174918  110.943993  103.753491   101.23164  100.389104    100.118833 | 100.035189 | 2.378783  1.895845  1.685026  1.603895   1.57511  1.565453  1.562351      1.56139 |  100.0 |  174.95935 | 100.035189 |
    | 2000-01-01 15:30:00 | 5.102552  4.352412  3.970809  3.809558  3.749069  3.727997   3.72102      3.7188 | 311.845554 | 197.846626  139.957227  114.980056  105.289248  101.783201  100.578191    100.181093 |  100.05496 | 2.583981  2.000045  1.729985  1.621338  1.581421  1.567622  1.563066     1.561617 |  100.0 | 211.845554 |  100.05496 |
    | 2000-01-01 16:00:00 | 5.380047  4.522888  4.054102  3.844295  3.762209  3.732662  3.722601    3.719316 | 352.270855 | 222.362996  152.069479  120.184367  107.334352  102.538833  100.844252    100.270989 | 100.084235 | 2.817815  2.126574  1.787378  1.644468  1.590053  1.570672  1.564097     1.561953 |  100.0 | 252.270855 | 100.084235 |
    | 2000-01-01 16:30:00 | 5.677459  4.719921  4.156922  3.889315  3.779858  3.739115  3.724845    3.720066 | 395.750601 |  250.58136  166.823051  126.797986  110.019455  103.559018  101.212899    100.398703 | 100.126853 |  3.07912  2.277339  1.859423   1.67467  1.601681  1.574895  1.565563     1.562443 |  100.0 | 295.750601 | 100.126853 |
    | 2000-01-01 17:00:00 | 5.990049  4.942732  4.281349  3.946686  3.803208  3.747906  3.727984     3.72114 | 441.664962 | 282.466342  184.501989  135.084236  113.498205  104.917869  101.716478     100.57744 | 100.187925 | 3.365677  2.453675  1.948375  1.713524  1.617126  1.580658  1.567613     1.563144 |  100.0 | 341.664962 | 100.187925 |
    | 2000-01-01 17:30:00 | 6.312704  5.189453  4.428955  4.018581   3.83365  3.759719   3.73231    3.722657 | 489.279607 | 317.838685  205.339001   145.32142  117.948283  106.705125   102.39539    100.824132 | 100.274169 | 3.674271  2.656174  2.056406  1.762796  1.637364  1.588415  1.570441     1.564134 |  100.0 | 389.279607 | 100.274169 |
    | 2000-01-01 18:00:00 | 6.640194  5.457219  4.600608  4.107164   3.87277  3.775379  3.738194    3.724768 |  537.77215 |  356.36888  229.489852  157.791547  123.571194  109.028123   103.29956    101.160252 | 100.394313 | 4.000797  2.884556  2.185463  1.824393  1.663541  1.598727  1.574292     1.565513 |  100.0 |  437.77215 | 100.394313 |
    | 2000-01-01 18:30:00 | 6.967395  5.742332   4.79632  4.214448   3.92233  3.795876  3.746098     3.72767 | 586.262607 | 397.580969  257.008871  172.764784  130.590193  112.013547  104.490054    101.612763 | 100.559574 | 4.340424  3.137586  2.337103  1.900307  1.696979  1.612269  1.579471     1.567408 |  100.0 | 486.262607 | 100.559574 |
    | 2000-01-01 19:00:00 | 7.289459  6.040482  5.015158  4.342128  3.984218  3.822369  3.756588     3.73161 | 633.846018 | 440.866389  287.828627  190.480026  139.245664  115.808691  106.040786    102.215197 | 100.784224 | 4.687787  3.413054   2.51233  1.992529  1.739166  1.629851  1.586357     1.569984 |  100.0 | 533.846018 | 100.784224 |
    | 2000-01-01 19:30:00 | 7.601935  6.346984  5.255239  4.491404  4.060375  3.856194  3.770353      3.7369 | 679.625318 | 485.506666  321.746566  211.122745   149.78733  120.581866  108.040234    103.008868 | 101.086255 | 5.037201  3.707822  2.711453  2.102933  1.791735  1.652426  1.595414     1.573445 |  100.0 | 579.625318 | 101.086255 |
    | 2000-01-01 20:00:00 | 7.900849  6.657014  5.513805  4.662815  4.152688  3.898848  3.788216    3.743923 | 722.742733 | 530.702888  358.420697  234.802068  162.462852  126.521481  110.593051    104.044214 | 101.488148 | 5.382873   4.01793  2.933967  2.233143   1.85643  1.681097  1.607202     1.578046 |  100.0 | 622.742733 | 101.488148 |
    | 2000-01-01 20:30:00 | 8.182756  6.965815  5.787374   4.85612  4.262861  3.951963  3.811146    3.753154 | 762.408181 | 575.609466  397.375306  261.529627  177.502822  133.833248  113.821361    105.382227 | 102.017758 |  5.71911  4.338752  3.178489  2.384391  1.935035  1.717119  1.622392     1.584102 |  100.0 | 662.408181 | 102.017758 |
    | 2000-01-01 21:00:00 | 8.444753  7.268865  6.071925  5.070232  4.392262  4.017258  3.840266    3.765165 | 797.923464 | 619.369498  438.016425  291.202918  195.102704  142.734965  117.865483     107.09593 | 102.709304 | 6.040507  4.665187  3.442753  2.557372  2.029297  1.761884  1.641777     1.591998 |  100.0 | 697.923464 | 102.709304 |
    | 2000-01-01 21:30:00 | 8.684479  7.562009  6.363115  5.303227  4.541775  4.096463  3.876845    3.780642 | 828.701418 | 661.149243  479.655667  323.595579  215.402989  153.448415  122.883724    109.271798 | 103.604481 | 6.342101  4.991876  3.723661  2.752118  2.140815    1.8169  1.666279     1.602199 |  100.0 | 728.701418 | 103.604481 |
    | 2000-01-01 22:00:00 | 8.900096  7.841549  6.656487  5.552421  4.711662  4.191218  3.922288    3.800396 |  854.27953 | 700.169564  521.540204   358.35626  238.469439  166.188167  129.050826    112.010996 | 104.753649 | 6.619499  5.313408  4.017392  2.967915  2.270922  1.883747  1.696951     1.615262 |  100.0 |  754.27953 | 104.753649 |
    | 2000-01-01 22:30:00 | 9.090256  8.104297  6.947658  5.814505  4.901472   4.30295  3.978101    3.825367 | 874.327871 | 735.732704  562.886168    395.0167  264.275715  181.147459  136.554617    115.430236 | 106.217077 | 6.868971  5.624532  4.319557  3.203249  2.420552  1.964014  1.734974     1.631846 |  100.0 | 774.327871 | 106.217077 |
    | 2000-01-01 23:00:00 | 9.254073  8.347601  7.232477  6.085719  5.109996  4.432748  4.045839    3.856631 | 888.651531 | 767.243343  602.912701  433.008481  292.690686  198.481879  145.590428    119.662003 | 108.066178 | 7.087505  5.920327  4.625385  3.455821  2.590128  2.059219  1.781646     1.652718 |  100.0 | 788.651531 | 108.066178 |
    | 2000-01-01 23:30:00 | 9.391073  8.569344   7.50715  6.362042  5.335294  4.581229  4.127034    3.895387 | 897.187983 | 794.223382   640.87413  471.686918  323.472385  218.292101  156.353001    124.853828 | 110.384642 | 7.272842  6.196356  4.929922  3.722607  2.779453  2.170709  1.838344     1.678763 |  100.0 | 797.187983 | 110.384642 |
    | 2000-01-02 00:00:00 | 9.501156   8.76793  7.768319   6.63938  5.574767  4.748432  4.223099    3.942944 |      900.0 | 816.320425  676.088199  510.359824  356.269777  240.607397   169.02583    131.166255 |  113.26932 | 7.423465  6.448774  5.228233  3.999969  2.987645  2.299556  1.906491     1.710977 |  100.0 |      800.0 |  113.26932 |
    | 2000-01-02 00:30:00 | 9.584554  8.942246  8.013116   6.91373  5.825287  4.933746  4.335224    4.000678 | 897.264896 | 833.310294  707.958944  548.318553  390.632579   265.37192   183.76826    138.769147 | 116.830688 | 7.538583  6.674406  5.515586  4.283804  3.213114  2.446443   1.98749     1.750466 |  100.0 | 797.264896 | 116.830688 |
    | 2000-01-02 01:00:00 | 9.641783  9.091629  8.239187  7.181328  6.083357   5.13589  4.464256     4.06999 | 889.260928 | 845.094234  735.993393  584.868728  426.028381  292.435604  200.701116    147.836012 | 121.192662 | 7.618083  6.870788   5.78761  4.569716  3.453581  2.611565  2.082647     1.798425 |  100.0 | 789.260928 | 121.192662 |
    | 2000-01-02 01:30:00 | 9.673609  9.215817  8.444688   7.43875  6.345276  5.352942  4.610597    4.152228 | 876.351729 | 851.691635  759.811889  619.358391  461.865469  321.551171  219.892054    158.536205 | 126.491474 | 7.662471  7.036171  6.040427  4.853203  3.706147  2.794551  2.193088     1.856105 |  100.0 | 776.351729 | 126.491474 |
    | 2000-01-02 02:00:00 | 9.681005  9.314904  8.628267  7.682991  6.607307  5.582419  4.774116    4.248604 | 858.969595 | 853.229216  779.152303  651.201855   497.51922  352.377988  241.342157    171.025081 | 132.873336 | 7.672804   7.16951  6.270737  5.129832  3.967401  2.994416  2.319658     1.924778 |  100.0 | 758.969595 | 132.873336 |
    | 2000-01-02 02:30:00 | 9.665114  9.389291  8.789031  7.911505  6.865823   5.82139  4.954097      4.3601 |   837.5984 |  849.92764  793.868772  679.898132  532.359734  384.492715  264.975412    185.432499 |  140.49061 | 7.650611  7.270416  6.475884  5.395401  4.233562  3.209547  2.462835     2.005674 |  100.0 |   737.5984 |  140.49061 |
    | 2000-01-02 03:00:00 | 9.627223  9.439638  8.926503  8.122227  7.117431  6.066626  5.149232    4.487368 | 812.756821 | 842.086492  803.925843  705.043435  565.778444  417.405858  290.632532    201.850429 | 149.496274 | 7.597818  7.339105  6.653873  5.646078  4.500633  3.437739  2.622643     2.099911 |  100.0 | 712.756821 | 149.496274 |
    | 2000-01-02 03:30:00 | 9.568722  9.466825  9.040581  8.313566  7.359065  6.314741  5.357658    4.630641 | 784.982427 |  830.06847  809.389009    726.3378  597.211847  450.582719  318.070185    220.320759 | 160.036643 | 7.516663  7.376328  6.803365  5.878505  4.764565  3.676261  2.798592     2.208422 |  100.0 | 684.982427 | 160.036643 |
    | 2000-01-02 04:00:00 | 9.491086  9.471906  9.131483  8.484384  7.588049  6.562345   5.57703    4.789668 | 754.817086 | 814.283535  810.412657  743.586298  626.160975  483.466809  346.966056    240.824574 |  172.24249 | 7.409623  7.383296  6.923652  6.089871  5.021414  3.921963  2.989647      2.33187 |  100.0 | 654.817086 |  172.24249 |
    | 2000-01-02 04:30:00 | 9.395845  9.456076  9.199707  8.633964  7.802132  6.806163   5.80463    4.963677 |  722.79402 | 795.173622  807.226387  756.695582  652.205814  515.503738   376.92949    263.274238 | 186.218996 |  7.27934  7.361601  7.014608  6.277952  5.267478  4.171396  3.194226      2.47057 |  100.0 |  622.79402 | 186.218996 |
    | 2000-01-02 05:00:00 | 9.284564  9.420634  9.245976   8.76197  7.999503  7.043148  6.037495    5.151374 | 689.426711 | 773.198412  800.120586  765.666709  675.014409  546.163713  407.516824    287.509394 | 202.035231 | 7.128551  7.313141  7.076625  6.441126  5.499409  4.420961   3.41024     2.624423 |  100.0 | 589.426711 | 202.035231 |
    | 2000-01-02 05:30:00 | 9.158822  9.366954  9.271202  8.868398  8.178784  7.270556  6.272546    5.350984 | 655.199767 |   748.8225  789.431963  770.585198  694.346852  574.961151  438.250021    313.297628 | 219.714112 |  6.96003  7.240044  7.110549  6.578355  5.714304  4.667041  3.635154     2.792873 |  100.0 | 555.199767 | 219.714112 |
    | 2000-01-02 06:00:00 | 9.020197  9.296461  9.276441  8.953538  8.339009  7.486003  6.506711     5.56032 | 620.561767 |   722.5042  775.529663  771.609294  710.054704  601.470368  468.636974    340.339993 | 239.223902 | 6.776534  7.144598  7.117606  6.689154  5.909756  4.906141  3.866089     2.974882 |  100.0 | 520.561767 | 239.223902 |
    | 2000-01-02 06:30:00 |  8.87025  9.210603  9.262863  9.017919  8.479589  7.687493  6.737041    5.776882 | 585.920022 |   694.6861  758.802391  768.957287  722.076599  625.336785  498.191804    368.281069 | 260.472288 | 6.580755  7.029189  7.099327  6.773539  6.083887  5.135002  4.099928     3.168938 |  100.0 | 485.920022 | 260.472288 |
    | 2000-01-02 07:00:00 | 8.710512  9.110837  9.231714  9.062271  8.600279  7.873427  6.960798    5.997965 |  551.63713 | 665.787415  739.646859  762.894609   730.43091  646.283573  526.453656    396.722704 | 283.303899 | 6.375284  6.896242  7.057481  6.831973  6.235352  5.350696  4.333444     3.373092 |  100.0 |  451.63713 | 283.303899 |
    | 2000-01-02 07:30:00 | 8.542474  8.998609  9.184292  9.087484  8.701132  8.042596  7.175521    6.220773 | 518.029154 | 636.198079  718.457744  753.721323  735.206343  664.113976  553.002822    425.240249 |  307.50176 |  6.16258  6.748174  6.994007    6.8653  6.363319  5.550696  4.563416     3.585021 |  100.0 | 418.029154 |  307.50176 |
    | 2000-01-02 08:00:00 | 8.367578  8.875339  9.121923  9.094566  8.782457   8.19416  7.379075    6.442533 | 485.365238 | 606.274491  695.619257  741.760421  736.551295  678.709873  577.473397    453.399908 | 332.792762 | 5.944944  6.587354  6.910952  6.874677  6.467433  5.732922  4.786749     3.802114 |  100.0 | 385.365238 | 332.792762 |
    | 2000-01-02 08:30:00 | 8.187209  8.742412  9.045937  9.084618  8.844778  8.327621  7.569676    6.660582 |  453.86844 | 576.336758  671.498311  727.347263  734.662689  690.027267  599.562103     480.77585 | 358.856799 | 5.724501  6.416063  6.810424  6.861509  6.547777  5.895759  5.000564     4.021569 |  100.0 |  353.86844 | 358.856799 |
    | 2000-01-02 09:00:00 | 8.002687  8.601167  8.957658    9.0588  8.888791  8.442784  7.745894    6.872452 | 423.717566 | 546.667279  646.439229  710.820328  729.774881  698.089449  619.033267     506.96587 |  385.33881 | 5.503191  6.236473  6.694538  6.827391  6.604811   6.03806  5.202285     4.240497 |  100.0 | 323.717566 |  385.33881 |
    | 2000-01-02 09:30:00 | 7.815267  8.452889  8.858384  9.018305  8.915329  8.539721  7.906651    7.075927 | 395.049792 | 517.510479  620.759901  692.513386  722.149097  702.978609  635.720215    531.604696 | 411.862727 | 5.282762  6.050624  6.565382  6.774047  6.639318  6.159125  5.389693     4.456033 |  100.0 | 295.049792 | 411.862727 |
    | 2000-01-02 10:00:00 | 7.626129  8.298802  8.749376  8.964343   8.92533  8.618732  8.051198    7.269081 | 367.963873 | 489.073499  594.749227   672.74908  712.063734  704.826561   649.52359    554.374329 | 438.046196 | 5.064765  5.860408  6.424982   6.70328  6.652345   6.25867  5.560962     4.665423 |  100.0 | 267.963873 | 438.046196 |
    | 2000-01-02 10:30:00 | 7.436382   8.14007  8.631854   8.89812  8.919805  8.680304   8.17909      7.4503 | 342.523746 | 461.527665  568.665701  651.833876  699.805739  703.805186  660.407174    575.011162 | 463.514974 | 4.850558   5.66756   6.27528  6.616931  6.645146  6.336787  5.714675     4.866117 |  100.0 | 242.523746 | 463.514974 |
    | 2000-01-02 11:00:00 | 7.247055  7.977786  8.506984  8.820822   8.89981  8.725081  8.290156     7.61829 | 318.762374 | 435.010552  542.736966  630.054288  685.663168  700.117052  668.391875    593.309885 | 487.916036 | 4.641311  5.473652  6.118108  6.516835  6.619129  6.393892  5.849822     5.055825 |  100.0 | 218.762374 | 487.916036 |
    | 2000-01-02 11:30:00 | 7.059102  7.812976  8.375876  8.733607  8.866432  8.753823  8.384461    7.772068 |  296.68568 |  409.62848  517.160171  607.674246  669.918986  693.986591  673.548477     609.12445 | 510.928664 | 4.438014  5.280095  5.955183  6.404794  6.575807  6.430681  5.965785     5.232578 |  100.0 |  196.68568 | 510.928664 |
    | 2000-01-02 12:00:00 | 6.873396  7.646594   8.23958  8.637595   8.82076  8.767384  8.462275    7.910948 | 276.276442 | 385.459309  492.102978  584.933466  652.846055  685.652054   675.98975    622.366483 | 532.273051 | 4.241485  5.088136  5.788088  6.282552  6.516756  6.448073  6.062307     5.394746 |  100.0 | 176.276442 | 532.273051 |
    | 2000-01-02 12:30:00 | 6.690733  7.479525  8.099082  8.533859   8.76388  8.766677  8.524036    8.034514 | 257.498082 | 362.555382  467.705055  562.046686  634.703256  675.358424  675.862369    633.001671 | 551.716223 | 4.052381  4.898867  5.618273   6.15177  6.443577  6.447166  6.139457      5.54106 |  100.0 | 157.498082 | 551.716223 |
    | 2000-01-02 13:00:00 | 6.511831  7.312584  7.955306  8.423421  8.696853  8.752659  8.570322    8.142597 | 240.298231 | 340.946539   444.07994  539.203609   615.73264  663.351345   673.33904    641.044639 | 569.075288 | 3.871215  4.713227  5.447051  6.014018  6.357863  6.429188  6.197586     5.670605 |  100.0 | 140.298231 | 569.075288 |
    | 2000-01-02 13:30:00 |  6.33733  7.146515  7.809111   8.30725  8.620713  8.726305  8.601815    8.235241 | 224.612057 | 320.643091  421.317147  516.569428  596.157486  649.872086  668.611114    646.552832 | 584.218257 | 3.698359  4.532014  5.275597  5.870759  6.261177  6.395458  6.237291     5.782807 |  100.0 | 124.612057 | 584.218257 |
    | 2000-01-02 14:00:00 | 6.167794  6.981996  7.661292  8.186256  8.536455  8.688599  8.619279    8.312673 | 210.365288 | 301.638694  399.484414  494.285796  576.181153   635.15349  661.881878    649.619873 | 597.062737 | 3.534067  4.355891  5.104954  5.723348  6.155027  6.347347  6.259363      5.87741 |  100.0 | 110.365288 | 597.062737 |
    | 2000-01-02 14:30:00 | 6.003713   6.81964  7.512581  8.061294  8.445031  8.640511  8.623536    8.375276 |  197.47692 | 283.913071  378.630011  472.472127   555.98659  619.416865  653.360651    650.368769 | 607.572947 | 3.378481  4.185401  4.936038  5.573021  6.040852  6.286248  6.264748     5.954442 |  100.0 |   97.47692 | 607.572947 |
    | 2000-01-02 15:00:00 | 5.845502  6.659995  7.363649  7.933158  8.347345  8.582995  8.615442    8.423555 | 185.861607 |  267.43454  358.785035  451.227144  535.736387  602.869689  643.257732    648.945302 | 615.755459 |  3.23165   4.02097  4.769642  5.420905  5.920012  6.213549  6.254511     6.014184 |  100.0 |  85.861607 | 615.755459 |
    | 2000-01-02 15:30:00 | 5.693508  6.503548  7.215109  7.802587  8.244251  8.516975  8.595875    8.458115 | 175.431719 |  252.16232  339.965649   430.63056  515.573262  585.704071  631.780199    645.511812 | 621.654106 | 3.093535  3.862924  4.606446  5.268008  5.793777  6.130612  6.229794     6.057128 |  100.0 |  75.431719 | 621.654106 |
    | 2000-01-02 16:00:00 | 5.548008  6.350728  7.067516  7.670262  8.136553  8.443342  8.565716    8.479633 | 166.099094 | 238.048617  322.175203   410.74485  495.620873  568.095821  619.128532    640.241548 | 625.344425 | 2.964026  3.711496   4.44702  5.115231  5.663322  6.038753   6.19179     6.083943 |  100.0 |  66.099094 | 625.344425 |
    | 2000-01-02 16:30:00 | 5.409212  6.201909  6.921369  7.536808  8.025001  8.362946  8.525839    8.488843 | 157.776495 | 225.040461  305.406232  391.617036  475.984879   550.20406  605.493994    633.313679 | 626.927965 | 2.842951  3.566838  4.291841  4.963369  5.529727  5.939232  6.141715     6.095437 |  100.0 |  57.776495 | 626.927965 |
    | 2000-01-02 17:00:00 | 5.277269   6.05741  6.777117  7.402801  7.910294  8.276595  8.477098    8.486511 | 150.378786 | 213.081316  289.642289  373.280445  456.754163  532.171251    591.0567    624.908999 | 626.526708 | 2.730086  3.429028  4.141292  4.813116  5.393977  5.833238  6.080781     6.092525 |  100.0 |  50.378786 | 626.526708 |
    | 2000-01-02 17:30:00 | 5.152268  5.917501  6.635161  7.268762   7.79308  8.185051  8.420327    8.473422 | 143.823868 | 202.112459  274.859618  355.756408  438.002164  514.123572  575.984287    615.206351 | 624.277811 | 2.625167  3.298083  3.995678  4.665074  5.256958  5.721888  6.010181     6.076196 |  100.0 |  43.823868 | 624.277811 |
    | 2000-01-02 18:00:00 |  5.03424  5.782406  6.495853  7.135166  7.673957  8.089032  8.356327    8.450367 | 138.033385 | 192.074159  261.028652   339.05587  419.788255  496.171548   560.43109    604.379715 | 620.328796 | 2.527893  3.173965  3.855229  4.519756  5.119468  5.606221  5.931074     6.047488 |  100.0 |  38.033385 | 620.328796 |
    | 2000-01-02 18:30:00 | 4.923166  5.652302  6.359502  7.002442  7.553477  7.989209  8.285862    8.418134 | 132.933235 | 182.906638  248.115342  323.180887  402.159132  478.410876  544.537761    592.595934 |  614.83327 | 2.437941  3.056589  3.720111  4.377595  4.982215  5.487193  5.844569     6.007461 |  100.0 |  32.933235 |  614.83327 |
    | 2000-01-02 19:00:00 | 4.818977  5.527327  6.226378  6.870975  7.432143  7.886206  8.209663    8.377493 | 128.453911 | 174.550876   236.08232  308.126016  385.150175  460.923379   528.43123    580.013005 | 607.947227 | 2.354967  2.945833  3.590432  4.238951  4.845823  5.365679  5.751722     5.957179 |  100.0 |  28.453911 | 607.947227 |
    | 2000-01-02 19:30:00 |  4.72156  5.407578  6.096711   6.74111  7.310416  7.780603  8.128416    8.329197 | 124.530701 | 166.949233  224.889906  293.879573   368.78677  443.778041  512.224951    566.778873 | 599.825925 | 2.278613  2.841539  3.466251  4.104118  4.710839  5.242474  5.653525     5.897695 |  100.0 |  24.530701 | 599.825925 |
    | 2000-01-02 20:00:00 | 4.630765  5.293117  5.970696  6.613154  7.188715  7.672936   8.04277    8.273971 | 121.103761 |  160.04595  214.496965  280.424765  353.085548  427.032091  496.019362    553.030651 | 590.621324 | 2.208517  2.743528  3.347581  3.973329  4.577735  5.118297  5.550904     5.830031 |  100.0 |  21.103761 | 590.621324 |
    | 2000-01-02 20:30:00 | 4.546404  5.183974  5.848495  6.487377   7.06742  7.563699  7.953333    8.212509 | 118.118088 | 153.787505  204.861632  267.740705  338.055555  410.732088  479.902504    538.894207 | 580.480054 | 2.144313  2.651594  3.234399  3.846762  4.446917   4.99379  5.444719     5.755177 |  100.0 |  18.118088 | 580.480054 |
    | 2000-01-02 21:00:00 | 4.468261  5.080146   5.73024  6.364019  6.946873  7.453347   7.86067    8.145474 | 115.523427 | 148.122887   195.94191  255.803303  323.699328  394.914992  463.950755    524.484049 | 569.541858 | 2.085633   2.56552  3.126648  3.724549  4.318729  4.869526  5.335759     5.674073 |  100.0 |  15.523427 | 569.541858 |
    | 2000-01-02 21:30:00 | 4.396094  4.981603  5.616034  6.243285  6.827384  7.342293  7.765309    8.073492 | 113.274094 | 143.003758  187.696168   244.58604  310.013883  379.609204  448.229633    509.903442 | 557.938456 |  2.03212  2.485074  3.024246  3.606779  4.193458  4.746009  5.224746      5.58761 |  100.0 |  13.274094 | 557.938456 |
    | 2000-01-02 22:00:00 | 4.329641   4.88829  5.505955  6.125356  6.709227  7.230918  7.667736    7.997156 | 111.328777 | 138.384558   180.08353  234.060649  296.991608  364.835551  432.794644    495.244719 | 545.792781 |  1.98342  2.410015  2.927087  3.493504  4.071338  4.623683  5.112336     5.496623 |  100.0 |  11.328777 | 545.792781 |
    | 2000-01-02 22:30:00 | 4.268625  4.800128  5.400055  6.010384   6.59265  7.119566  7.568403     7.91702 | 109.650285 | 134.222539  173.064184  224.197687  284.621063  350.608212  417.692139    480.589734 | 533.218521 |  1.93919    2.3401  2.835047  3.384741  3.952562  4.502932  4.999121     5.401891 |  100.0 |   9.650285 | 533.218521 |
    | 2000-01-02 23:00:00 | 4.212756  4.717017  5.298365  5.898496   6.47787  7.008549  7.467725    7.833603 | 108.205287 |  130.47776  166.599624  214.967023  272.887695  336.935586  402.960166    466.010411 | 520.319926 | 1.899099  2.275081  2.747986  3.280484  3.837275  4.384088   4.88563     5.304134 |  100.0 |   8.205287 | 520.319926 |
    | 2000-01-02 23:30:00 |  4.16174   4.63884  5.200898    5.7898  6.365079   6.89815  7.366082     7.74739 | 106.964036 | 127.113034  160.652818  206.338249  261.774469   323.82108  388.629304    451.569361 | 507.191817 | 1.862831   2.21471  2.665752  3.180699  3.725591  4.267433  4.772338     5.204014 |  100.0 |   6.964036 | 507.191817 |
    | 2000-01-03 00:00:00 | 4.115275  4.565462  5.107644   5.68438  6.254445  6.788624  7.263824     7.65883 | 105.900084 | 124.093857  155.188334   198.28102  251.262417  311.263838   374.72345    437.320551 | 493.919773 | 1.830082  2.158739  2.588185  3.085333  3.617589  4.153206  4.659663     5.102135 |  100.0 |   5.900084 | 493.919773 |
    | 2000-01-03 00:30:00 | 4.073064  4.496737   5.01858  5.582304  6.146116  6.680199   7.16127    7.568339 |  104.99001 | 121.388302  150.172415  190.765326   241.33112  299.259395   361.26058    423.309979 | 480.580433 | 1.800565  2.106928  2.515118  2.994314  3.513318  4.041603  4.547974     4.999048 |  100.0 |    4.99001 | 480.580433 |
    | 2000-01-03 01:00:00 | 4.034811  4.432503  4.933666  5.483619  6.040216  6.573079  7.058708    7.476303 | 104.213141 | 118.966917  145.573021  183.761718  231.959123  287.800264  348.253448     409.57636 | 467.241905 | 1.774008  2.059037  2.446378  2.907559  3.412804  3.932788  4.437593     4.895251 |  100.0 |   4.213141 | 467.241905 |
    | 2000-01-03 01:30:00 | 4.000227  4.372593  4.852847  5.388361  5.936854  6.467446  6.956404    7.383074 | 103.551298 | 116.802591  141.359842  177.241478  223.124283  276.876466  335.710242    396.151796 | 453.964236 | 1.750156  2.014834  2.381791   2.82497   3.31605  3.826887  4.328798     4.791188 |  100.0 |   3.551298 | 453.964236 |
    | 2000-01-03 02:00:00 | 3.969031  4.316828  4.776057  5.296547   5.83612   6.36346  6.854596     7.28898 | 102.988541 | 114.870436  137.504287  171.176757  214.804072  266.475992  323.635185    383.062424 | 440.799927 | 1.728769  1.974093  2.321181  2.746441  3.223041     3.724  4.221829     4.687261 |  100.0 |   2.988541 | 440.799927 |
    | 2000-01-03 02:30:00 | 3.940952  4.265027  4.703219  5.208183  5.738087  6.261264  6.753498    7.194318 |  102.51094 | 113.147645  133.979456  165.540672  206.975828  256.585216   312.02908    370.329035 | 427.794474 | 1.709623  1.936596  2.264371  2.671857  3.133744  3.624201  4.116889     4.583823 |  100.0 |    2.51094 | 427.794474 |
    | 2000-01-03 03:00:00 | 3.915733  4.217002  4.634246  5.123262  5.642816  6.160982  6.653305    7.099361 | 102.106358 | 111.613371  130.760096  160.307378  199.616965  247.189248  300.889812    357.967661 | 414.986916 |  1.69251  1.902132  2.211186    2.6011  3.048114   3.52754  4.014149     4.481188 |  100.0 |   2.106358 | 414.986916 |
    | 2000-01-03 03:30:00 | 3.893126  4.172564  4.569039  5.041767  5.550354  6.062722  6.554192    7.004358 | 101.764251 | 110.248589  127.822548   155.45211  192.705138  238.272253   290.21279    345.990115 | 402.410381 | 1.677238  1.870499  2.161452  2.534045  2.966093  3.434047   3.91375     4.379632 |  100.0 |   1.764251 | 402.410381 |
    | 2000-01-03 04:00:00 | 3.872901  4.131523  4.507496  4.963669  5.460734  5.966576  6.456312    6.909536 |  101.47548 | 109.035973   125.14469  150.951214  186.218391  229.817714  279.991356    334.404506 | 390.092618 | 1.663629  1.841503  2.115001  2.470564  2.887614  3.343735  3.815806     4.279393 |  100.0 |    1.47548 | 390.092618 |
    | 2000-01-03 04:30:00 |  3.85484  4.093689  4.449505  4.888932  5.373982  5.872625  6.359806    6.815099 | 101.232151 | 107.959775  122.705868  146.782152  180.135256  221.808663  270.217131    323.215696 | 378.056515 |  1.65152   1.81496  2.071663  2.410528  2.812602    3.2566  3.720409      4.18068 |  100.0 |   1.232151 | 378.056515 |
    | 2000-01-03 05:00:00 |  3.83874  4.058876   4.39495  4.817509   5.29011  5.780934  6.264795    6.721235 | 101.027458 | 107.005705   120.48683  142.923502  174.434848  214.227881  260.880341    312.425733 | 366.320584 | 1.640759  1.790694  2.031277  2.353808  2.740974  3.172625  3.627627     4.083669 |  100.0 |   1.027458 | 366.320584 |
    | 2000-01-03 05:30:00 | 3.824411  4.026899   4.34371  4.749348  5.209121   5.69156  6.171388    6.628111 | 100.855551 | 106.160818  118.469656  139.354941  169.096927  207.058058  251.970088    302.034233 | 354.899427 | 1.631209  1.768538  1.993684  2.300272  2.672644  3.091784  3.537512     3.988512 |  100.0 |   0.855551 | 354.899427 |
    | 2000-01-03 06:00:00 | 3.811679  3.997577  4.295659  4.684389  5.131012  5.604547   6.07968    6.535878 | 100.711411 | 105.413411  116.637691  136.057223   164.10194  200.281934    243.4746    292.038735 | 343.804162 | 1.622746  1.748334  1.958729  2.249792   2.60752  3.014037  3.450096     3.895334 |  100.0 |   0.711411 | 343.804162 |
    | 2000-01-03 06:30:00 | 3.800383  3.970735   4.25067  4.622565  5.055771   5.51993  5.989753    6.444673 | 100.590745 | 104.752918  114.975473  133.012155  159.431062  193.882412  235.381435    282.435011 | 333.042829 | 1.615254  1.729934  1.926263  2.202238  2.545508  2.939339    3.3654      3.80424 |  100.0 |   0.590745 | 333.042829 |
    | 2000-01-03 07:00:00 | 3.790376  3.946203  4.208613  4.563805  4.983376  5.437735  5.901679    6.354618 | 100.489885 | 104.169821  113.468665  130.202556  155.066208  187.842655  227.677673     273.21735 | 322.620757 |  1.60863  1.713196  1.896142  2.157483   2.48651  2.867635  3.283429     3.715314 |  100.0 |   0.489885 | 322.620757 |
    | 2000-01-03 07:30:00 | 3.781523  3.923819  4.169359  4.508032  4.913803   5.35798   5.81552    6.265819 | 100.405708 | 103.655553  112.103993  127.612228  150.990049  182.146155  220.350068    264.378808 | 312.540901 |  1.60278  1.697989  1.868227  2.115403  2.430429  2.798868  3.204177     3.628622 |  100.0 |   0.405708 | 312.540901 |
    | 2000-01-03 08:00:00 | 3.773701  3.903426  4.132775  4.455165  4.847019  5.280676  5.731326    6.178375 |  100.33556 |  103.20242  110.869179  125.225916  147.186007  176.776798  213.385185     255.91143 | 302.804153 |  1.59762  1.684188  1.842385  2.075875  2.377166  2.732973   3.12763     3.544215 |  100.0 |    0.33556 | 302.804153 |
    | 2000-01-03 08:30:00 | 3.766798  3.884874  4.098732  4.405121  4.782988  5.205827   5.64914    6.092369 | 100.277186 | 102.803523  109.752881  123.029265  143.638254  171.718909  206.769516    247.806446 | 293.409621 | 1.593072   1.67168  1.818487  2.038779  2.326621  2.669882  3.053763     3.462129 |  100.0 |   0.277186 | 293.409621 |
    | 2000-01-03 09:00:00 | 3.760714  3.868022  4.067099  4.357812  4.721665  5.133429  5.568999    6.007877 | 100.228681 | 102.452684  108.744634  121.008787  140.331699  166.957283  200.489577    240.054442 | 284.354882 | 1.589069  1.660354  1.796412  2.003997  2.278695  2.609524  2.982546     3.382388 |  100.0 |   0.228681 | 284.354882 |
    | 2000-01-03 09:30:00 | 3.755357  3.852736  4.037748  4.313147  4.663005  5.063475  5.490928    5.924963 | 100.188431 |  102.14438  107.834794  119.151814  137.251975  162.477216  194.531986    232.645513 | 275.636209 | 1.585548  1.650111  1.776041  1.971417  2.233289  2.551826  2.913942     3.305005 |  100.0 |   0.188431 | 275.636209 |
    | 2000-01-03 10:00:00 | 3.750646  3.838888  4.010555  4.271034  4.606956  4.995951  5.414949    5.843685 | 100.155078 |  101.87368  107.014483  117.446463  134.385417  158.264516  188.883531     225.56939 | 267.248772 | 1.582455  1.640858  1.757264  1.940927  2.190307  2.496711  2.847907     3.229982 |  100.0 |   0.155078 | 267.248772 |
    | 2000-01-03 10:30:00 | 3.746506   3.82636  3.985395   4.23138  4.553463  4.930838  5.341077    5.764091 | 100.127477 | 101.636189  106.275538  115.881597  131.719044  154.305516  183.531226    218.815558 | 259.186826 | 1.579739  1.632507  1.739973   1.91242  2.149651  2.444105  2.784396     3.157314 |  100.0 |   0.127477 | 259.186826 |
    | 2000-01-03 11:00:00 | 3.742873  3.815041  3.962149  4.194089  4.502469  4.868114   5.26932    5.686222 | 100.104666 | 101.427994  105.610467  114.446786  129.240538  150.587074  178.462355    212.373346 | 251.443864 | 1.577357  1.624979  1.724067  1.885791  2.111227   2.39393  2.723356     3.086987 |  100.0 |   0.104666 | 251.443864 |
    | 2000-01-03 11:30:00 | 3.739687  3.804826  3.940701  4.159067  4.453913   4.80775  5.199683    5.610113 | 100.085838 | 101.245618  105.012395  113.132271  126.938215  147.096577  173.664503    206.232019 | 244.012765 |  1.57527  1.618198  1.709451   1.86094  2.074942  2.346107  2.664734     3.018982 |  100.0 |   0.085838 | 244.012765 |
    | 2000-01-03 12:00:00 | 3.736895  3.795618  3.920936  4.126218   4.40773  4.749718  5.132164    5.535791 | 100.070317 | 101.085973  104.475025   111.92893  124.801008  143.821933  169.125588     200.38084 | 236.885917 | 1.573442  1.612098  1.696034   1.83777  2.040705  2.300561  2.608475     2.953273 |  100.0 |   0.070317 | 236.885917 |
    | 2000-01-03 12:30:00 |  3.73445  3.787328  3.902748  4.095445  4.363856  4.693981  5.066758     5.46328 | 100.057539 | 100.946323  103.992597  110.828243  122.818437  140.751564  164.833875    194.809136 | 230.055329 | 1.571842  1.606615   1.68373  1.816188  2.008425  2.257213  2.554519     2.889829 |  100.0 |   0.057539 | 230.055329 |
    | 2000-01-03 13:00:00 | 3.732312  3.779872  3.886031  4.066655  4.322221  4.640503  5.003456    5.392597 | 100.047032 | 100.824243  103.559847  109.822259   120.98059  137.874398  160.777997    189.506346 | 223.512723 | 1.570443  1.601691  1.672458  1.796103  1.978016  2.215986  2.502808     2.828618 |  100.0 |   0.047032 | 223.512723 |
    | 2000-01-03 13:30:00 | 3.730442  3.773174  3.870685  4.039753  4.282758  4.589243  4.942243    5.323753 | 100.038402 |  100.71759   103.17197  108.903565  119.278092  135.179852  156.946958    184.462059 | 217.249625 |  1.56922  1.597273  1.662141  1.777429  1.949393  2.176805  2.453281       2.7696 |  100.0 |   0.038402 | 217.249625 |
    | 2000-01-03 14:00:00 | 3.728809  3.767163  3.856615  4.014646  4.245395  4.540159  4.883105    5.256758 | 100.031322 | 100.624471  102.824586  108.065256  117.702089  132.657824  153.330142    179.666053 | 211.257429 | 1.568153  1.593313  1.652708  1.760083  1.922473  2.139595  2.405877     2.712736 |  100.0 |   0.031322 | 211.257429 |
    | 2000-01-03 14:30:00 | 3.727383  3.761773   3.84373  3.991243  4.210061  4.493205   4.82602    5.191614 | 100.025521 | 100.543215  102.513707  107.300908   116.24422  130.298675  149.917312    175.108317 | 205.527466 | 1.567221  1.589766  1.644091  1.743985  1.897175  2.104282  2.360533     2.657981 |  100.0 |   0.025521 | 205.527466 |
    | 2000-01-03 15:00:00 | 3.726139  3.756945  3.831944  3.969455  4.176683  4.448333  4.770965    5.128323 | 100.020773 | 100.472351  102.235709  106.604547  114.896596  128.093216  146.698611    170.779077 | 200.051052 | 1.566408  1.586592  1.636227  1.729059  1.873421  2.070792  2.317188     2.605292 |  100.0 |   0.020773 | 200.051052 |
    | 2000-01-03 15:30:00 | 3.725054  3.752624  3.821176  3.949195  4.145189  4.405495  4.717916    5.066881 | 100.016892 | 100.410581  101.987295  105.970626  113.651779  126.032692  143.664557    166.668806 | 194.819538 | 1.565699  1.583754  1.629057  1.715233  1.851135  2.039054   2.27578      2.55462 |  100.0 |   0.016892 | 194.819538 |
    | 2000-01-03 16:00:00 | 3.724108  3.748761  3.811347  3.930377  4.115506  4.364639  4.666845    5.007281 | 100.013722 | 100.356765  101.765478  105.393997  112.502762  124.108767  140.806038    162.768246 | 189.824345 | 1.565082  1.581218  1.622526  1.702437  1.830244  2.008999  2.236248      2.50592 |  100.0 |   0.013722 | 189.824345 |
    | 2000-01-03 16:30:00 | 3.723285  3.745309  3.802387  3.912919  4.087561  4.325713   4.61772    4.949515 | 100.011136 | 100.309901  101.567551   104.86989  111.442945  122.313508  138.114307    159.068406 | 185.056998 | 1.564544  1.578954  1.616581  1.690606  1.810677  1.980558  2.198531      2.45914 |  100.0 |   0.011136 | 185.056998 |
    | 2000-01-03 17:00:00 | 3.722568  3.742228  3.794225  3.896741  4.061281  4.288662  4.570509    4.893569 | 100.009028 | 100.269109  101.391061  104.393888  110.466119  120.639368  135.580971    155.560575 |  180.50915 | 1.564076  1.576935  1.611176  1.679676  1.792366  1.953663  2.162568     2.414233 |  100.0 |   0.009028 |  180.50915 |
    | 2000-01-03 17:30:00 | 3.721944   3.73948    3.7868  3.881765  4.036595  4.253433  4.525179    4.839429 | 100.007313 | 100.233619  101.233793  103.961905  109.566446  119.079174  133.197981    152.236321 | 176.172609 | 1.563668  1.575134  1.606266  1.669587  1.775243  1.928249  2.128299     2.371147 |  100.0 |   0.007313 | 176.172609 |
    | 2000-01-03 18:00:00 | 3.721401  3.737031   3.78005  3.867919  4.013431  4.219968  4.481692    4.787077 | 100.005918 | 100.202752  101.093744  103.570165  108.738444  117.626112   130.95763    149.087493 | 172.039347 | 1.563314  1.573531  1.601808  1.660284  1.759245  1.904252  2.095665     2.329834 |  100.0 |   0.005918 | 172.039347 |
    | 2000-01-03 18:30:00 | 3.720929  3.734849  3.773921  3.855129  3.991718  4.188213  4.440011    4.736496 | 100.004785 | 100.175918  100.969108  103.215184  107.976963  116.273708  128.852533    146.106222 | 168.101522 | 1.563006  1.572103  1.597765  1.651713  1.744311  1.881611  2.064608     2.290242 |  100.0 |   0.004785 | 168.101522 |
    | 2000-01-03 19:00:00 | 3.720519  3.732908   3.76836  3.843329  3.971389   4.15811  4.400097    4.687662 | 100.003865 | 100.152598  100.858259  102.893749  107.277175   115.01582  126.875624    143.284915 | 164.351483 | 1.562738  1.570833  1.594101  1.643823  1.730381  1.860263  2.035071     2.252323 |  100.0 |   0.003865 | 164.351483 |
    | 2000-01-03 19:30:00 | 3.720162  3.731181   3.76332  3.832453  3.952375  4.129601  4.361909    4.640553 | 100.003119 | 100.132339  100.759731  102.602902  106.634554  113.846621  125.020145     140.61625 | 160.781783 | 1.562506  1.569703  1.590784  1.636566  1.717399  1.840151  2.006998     2.216025 |  100.0 |   0.003119 | 160.781783 |
    | 2000-01-03 20:00:00 | 3.719853  3.729646  3.758755  3.822439  3.934609   4.10263  4.325405    4.595144 | 100.002515 | 100.114744  100.672205  102.339919  106.044864  112.760588  123.279631    138.093178 | 157.385179 | 1.562304    1.5687  1.587782  1.629897  1.705311  1.821216  1.980334     2.181299 |  100.0 |   0.002515 | 157.385179 |
    | 2000-01-03 20:30:00 | 3.719584  3.728283  3.754625  3.813227  3.918028  4.077138  4.290543    4.551409 | 100.002026 | 100.099468  100.594497  102.102299  105.504139  111.752486  121.647904    135.708911 | 154.154638 | 1.562128  1.567809  1.585068  1.623774  1.694064  1.803405  1.955024     2.148097 |  100.0 |   0.002026 | 154.154638 |
    | 2000-01-03 21:00:00 |  3.71935  3.727073  3.750891  3.804762  3.902567   4.05307  4.257279    4.509318 | 100.001631 | 100.086209  100.525544  101.887744  105.008676  110.817359  120.119064     133.45692 |  151.08334 | 1.561976  1.567018  1.582616  1.618156  1.683608  1.786661  1.931016     2.116369 |  100.0 |   0.001631 |  151.08334 |
    | 2000-01-03 21:30:00 | 3.719148  3.725999  3.747517   3.79699  3.888167  4.030367  4.225568    4.468844 | 100.001312 | 100.074704  100.464394  101.694147  104.555014  109.950518  118.687473    131.330924 | 148.164677 | 1.561844  1.566317  1.580402  1.613007  1.673897  1.770935  1.908258     2.086068 |  100.0 |   0.001312 | 148.164677 |
    | 2000-01-03 22:00:00 | 3.718972  3.725047  3.744472  3.789862  3.874767  4.008974  4.195365    4.429954 | 100.001054 | 100.064724   100.41019  101.519579  104.139926  109.147527  117.347754    129.324888 | 145.392253 | 1.561729  1.565695  1.578405   1.60829  1.664883  1.756175  1.886699     2.057147 |  100.0 |   0.001054 | 145.392253 |
    | 2000-01-03 22:30:00 |  3.71882  3.724203  3.741725   3.78333  3.862311  3.988834  4.166625    4.392617 | 100.000846 | 100.056068  100.362169  101.362271  103.760401  108.404196  116.094772    127.433014 | 142.759878 |  1.56163  1.565144  1.576605  1.603973  1.656524  1.742332   1.86629     2.029559 |  100.0 |   0.000846 | 142.759878 |
    | 2000-01-03 23:00:00 | 3.718688  3.723456  3.739249  3.777349  3.850744  3.969894  4.139303    4.356801 | 100.000679 | 100.048562  100.319647   101.22061  103.413641  107.716567  114.923635    125.649734 | 140.261574 | 1.561544  1.564656  1.574983  1.600026  1.648779  1.729359  1.846983     2.003257 |  100.0 |   0.000679 | 140.261574 |
    | 2000-01-03 23:30:00 | 3.718573  3.722794   3.73702  3.771878  3.840012  3.952098   4.11335    4.322471 | 100.000544 | 100.042055  100.282011   101.09312  103.097037  107.080904  113.829675    123.969703 | 137.891561 | 1.561469  1.564224  1.573523  1.596419  1.641608  1.717211  1.828731     1.978198 |  100.0 |   0.000544 | 137.891561 |
    | 2000-01-04 00:00:00 | 3.718474  3.722209  3.735013  3.766878  3.830064  3.935394  4.088722    4.289593 | 100.000436 | 100.036416  100.248715  100.978453  102.808169  106.493687  112.808448     122.38779 | 135.644263 | 1.561404  1.563842   1.57221  1.593125  1.634974  1.705844  1.811488     1.954336 |  100.0 |   0.000436 | 135.644263 |
    | 2000-01-04 00:30:00 | 3.718388  3.721691  3.733208  3.762312  3.820852   3.91973  4.065372     4.25813 | 100.000349 | 100.031529  100.219273  100.875384  102.544788  105.951599  111.855722    120.899076 | 133.514296 | 1.561348  1.563504  1.571029   1.59012  1.628842  1.695217   1.79521     1.931629 |  100.0 |   0.000349 | 133.514296 |
    | 2000-01-04 01:00:00 | 3.718313  3.721234  3.731585  3.758145  3.812329  3.905054  4.043254    4.228048 | 100.000279 | 100.027294   100.19325  100.782795  102.304808  105.451515  110.967468    119.498842 | 131.496469 |   1.5613  1.563205  1.569968  1.587381  1.623178  1.685288  1.779854     1.910033 |  100.0 |   0.000279 | 131.496469 |
    | 2000-01-04 01:30:00 | 3.718249  3.720829  3.730128  3.754347  3.804451  3.891318  4.022321    4.199309 | 100.000223 | 100.023626  100.170258  100.699668  102.086298  104.990498  110.139854    118.182565 | 129.585776 | 1.561258  1.562941  1.569015  1.584885   1.61795  1.676019  1.765377     1.889508 |  100.0 |   0.000223 | 129.585776 |
    | 2000-01-04 02:00:00 | 3.718193  3.720472  3.728819  3.750886  3.797175  3.878473  4.002529    4.171876 | 100.000178 | 100.020449  100.149953  100.625079  101.887468  104.565787  109.369238    116.945912 | 127.777393 | 1.561221  1.562708   1.56816  1.582613  1.613129  1.667373  1.751739     1.870011 |  100.0 |   0.000178 | 127.777393 |
    | 2000-01-04 02:30:00 | 3.718144  3.720157  3.727646  3.747735  3.790461  3.866472  3.983831    4.145711 | 100.000142 | 100.017698  100.132028  100.558189  101.706664   104.17479  108.652158    115.784732 |  126.06667 | 1.561189  1.562502  1.567392  1.580545  1.608686  1.659314    1.7389     1.851503 |  100.0 |   0.000142 |  126.06667 |
    | 2000-01-04 03:00:00 | 3.718102  3.719878  3.726593  3.744869  3.784272   3.85527  3.966184    4.120777 | 100.000113 | 100.015315   100.11621  100.498236  101.542358  103.815075  107.985331    114.695051 | 124.449131 | 1.561162  1.562321  1.566705  1.578666  1.604595  1.651808  1.726823     1.833946 |  100.0 |   0.000113 | 124.449131 |
    | 2000-01-04 03:30:00 | 3.718066  3.719633   3.72565  3.742264   3.77857  3.844824  3.949542    4.097036 |  100.00009 | 100.013252  100.102257  100.444529   101.39314  103.484362  107.365638    113.673064 | 122.920464 | 1.561138   1.56216  1.566089  1.576958  1.600831  1.644821   1.71547     1.817301 |  100.0 |    0.00009 | 122.920464 |
    | 2000-01-04 04:00:00 | 3.718035  3.719416  3.724805  3.739898  3.773322   3.83509  3.933863    4.074449 | 100.000072 | 100.011466  100.089953  100.396443  101.257709  103.180516  106.790124    112.715133 |  121.47652 | 1.561118  1.562019  1.565537  1.575408   1.59737  1.638324  1.704805      1.80153 |  100.0 |   0.000072 |  121.47652 |
    | 2000-01-04 04:30:00 | 3.718007  3.719225  3.724049   3.73775  3.768495  3.826028  3.919104    4.052979 | 100.000057 |  100.00992  100.079107  100.353412  101.134868  102.901541  106.255989    111.817779 | 120.113306 |   1.5611  1.561895  1.565043  1.574002   1.59419  1.632286  1.694793     1.786598 |  100.0 |   0.000057 | 120.113306 |
    | 2000-01-04 05:00:00 | 3.717984  3.719057  3.723373  3.735801   3.76406  3.817599  3.905223    4.032586 | 100.000046 | 100.008582   100.06955  100.314923  101.023512  102.645569  105.760582    110.977677 | 118.826981 | 1.561085  1.561785  1.564602  1.572726  1.591271  1.626679  1.685403      1.77247 |  100.0 |   0.000046 | 118.826981 |
    | 2000-01-04 05:30:00 | 3.717963  3.718909  3.722768  3.734035  3.759987  3.809766   3.89218    4.013234 | 100.000036 | 100.007424  100.061131  100.280514   100.92263  102.410857  105.301395    110.191651 |  117.61385 | 1.561071  1.561688  1.564207   1.57157  1.588592  1.621476    1.6766      1.75911 |  100.0 |   0.000036 |  117.61385 |
    | 2000-01-04 06:00:00 | 3.717946  3.718779  3.722228  3.732435  3.756251  3.802492  3.879933    3.994885 | 100.000029 | 100.006422  100.053718  100.249767   100.83129  102.195779  104.876058    109.456668 | 116.470362 |  1.56106  1.561603  1.563854  1.570524  1.586136  1.616651  1.668355     1.746485 |  100.0 |   0.000029 | 116.470362 |
    | 2000-01-04 06:30:00 | 3.717931  3.718664  3.721745  3.730986  3.752824  3.795743  3.868445      3.9775 | 100.000023 | 100.005555  100.047191  100.222306  100.748636  101.998818  104.482331    108.769835 | 115.393105 |  1.56105  1.561528  1.563539  1.569576  1.583885  1.612181  1.660637     1.734563 |  100.0 |   0.000023 | 115.393105 |
    | 2000-01-04 07:00:00 | 3.717917  3.718563  3.721314  3.729676  3.749686  3.789487  3.857677    3.961044 | 100.000018 | 100.004805  100.041448   100.19779  100.673886  101.818562  104.118103    108.128393 | 114.378799 | 1.561041  1.561462  1.563257  1.568719  1.581825  1.608042  1.653419     1.723312 |  100.0 |   0.000018 | 114.378799 |
    | 2000-01-04 07:30:00 | 3.717906  3.718474  3.720929   3.72849  3.746812  3.783691  3.847593    3.945478 | 100.000014 | 100.004156  100.036395  100.175913  100.606321  101.653699   103.78138    107.529712 | 113.424298 | 1.561034  1.561404  1.563006  1.567944   1.57994  1.604212  1.646672     1.712702 |  100.0 |   0.000014 | 113.424298 |
    | 2000-01-04 08:00:00 | 3.717896  3.718395  3.720585  3.727419  3.744183  3.778327  3.838157    3.930768 | 100.000011 | 100.003594   100.03195    100.1564  100.545284  101.503007  103.470286    106.971288 | 112.526577 | 1.561027  1.561353  1.562782  1.567244  1.578216  1.600671   1.64037     1.702703 |  100.0 |   0.000011 | 112.526577 |
    | 2000-01-04 08:30:00 | 3.717888  3.718327  3.720279  3.726452  3.741779  3.773366  3.829335    3.916877 | 100.000009 | 100.003109  100.028042  100.139003  100.490175  101.365351  103.183053    106.450738 | 111.682738 | 1.561022  1.561308  1.562582  1.566612   1.57664  1.597399  1.634488     1.693285 |  100.0 |   0.000009 | 111.682738 |
    | 2000-01-04 09:00:00 |  3.71788  3.718266  3.720007  3.725578  3.739583  3.768781  3.821094    3.903771 | 100.000007 | 100.002688  100.024607  100.123499  100.440443  101.239681   102.91802    105.965796 | 110.889998 | 1.561017  1.561269  1.562404  1.566042  1.575202  1.594378  1.629002     1.684421 |  100.0 |   0.000007 | 110.889998 |
    | 2000-01-04 09:30:00 | 3.717874  3.718213  3.719764   3.72479  3.737578  3.764547  3.813401    3.891415 | 100.000006 | 100.002325  100.021587  100.109687  100.395588  101.125021  102.673626    105.514305 | 110.145694 | 1.561013  1.561234  1.562246  1.565527  1.573889  1.591591  1.623889     1.676085 |  100.0 |   0.000006 | 110.145694 |
    | 2000-01-04 10:00:00 | 3.717868  3.718166  3.719547   3.72408  3.735748  3.760639  3.806225    3.879776 | 100.000004 |  100.00201  100.018935  100.097388  100.355151  101.020468  102.448405    105.094221 | 109.447268 | 1.561009  1.561204  1.562105  1.565063  1.572691   1.58902  1.619127     1.668249 |  100.0 |   0.000004 | 109.447268 |
    | 2000-01-04 10:30:00 | 3.717863  3.718125  3.719354   3.72344   3.73408  3.757036  3.799538    3.868821 | 100.000004 | 100.001739  100.016605   100.08644  100.318716  100.925186  102.240981    104.703602 | 108.792277 | 1.561006  1.561177  1.561979  1.564645    1.5716  1.586652  1.614694      1.66089 |  100.0 |   0.000004 | 108.792277 |
    | 2000-01-04 11:00:00 | 3.717859  3.718089  3.719183  3.722863   3.73256  3.753716   3.79331    3.858517 | 100.000003 | 100.001503  100.014559  100.076699  100.285903  100.838403  102.050066    104.340605 | 108.178377 | 1.561003  1.561153  1.561867  1.564269  1.570605  1.584471  1.610571     1.653982 |  100.0 |   0.000003 | 108.178377 |
    | 2000-01-04 11:30:00 | 3.717856  3.718057  3.719031  3.722344  3.731175  3.750659  3.787515    3.848835 | 100.000002 |   100.0013  100.012762  100.068035  100.256367  100.759406  101.874453    104.003488 |  107.60333 | 1.561001  1.561133  1.561767   1.56393    1.5697  1.582464  1.606739     1.647502 |  100.0 |   0.000002 |  107.60333 |
    | 2000-01-04 12:00:00 | 3.717853   3.71803  3.718895  3.721876  3.729915  3.747846  3.782127    3.839743 | 100.000002 | 100.001124  100.011185  100.060331  100.229791  100.687535  101.713014    103.690597 | 107.064995 | 1.560999  1.561115  1.561679  1.563625  1.568876  1.580618  1.603179     1.641428 |  100.0 |   0.000002 | 107.064995 |
    | 2000-01-04 12:30:00 |  3.71785  3.718005  3.718774  3.721456  3.728769  3.745259   3.77712    3.831211 | 100.000001 | 100.000972  100.009801  100.053484  100.205892  100.622184  101.564693     103.40037 | 106.561327 | 1.560997  1.561099    1.5616   1.56335  1.568126  1.578921  1.599875     1.635738 |  100.0 |   0.000001 | 106.561327 |
    | 2000-01-04 13:00:00 | 3.717848  3.717984  3.718667  3.721078  3.727726  3.742882  3.772471    3.823213 | 100.000001 |  100.00084  100.008587    100.0474  100.184408  100.562792  101.428505     103.13133 | 106.090372 | 1.560996  1.561085  1.561531  1.563104  1.567445  1.577363   1.59681     1.630412 |  100.0 |   0.000001 | 106.090372 |
    | 2000-01-04 13:30:00 | 3.717846  3.717965  3.718572  3.720739   3.72678  3.740699  3.768158    3.815719 | 100.000001 | 100.000727  100.007522  100.041996  100.165105  100.508846  101.303531    102.882082 | 105.650267 | 1.560995  1.561073  1.561468  1.562882  1.566826  1.575933  1.593968     1.625429 |  100.0 |   0.000001 | 105.650267 |
    | 2000-01-04 14:00:00 | 3.717844  3.717949  3.718488  3.720434   3.72592  3.738695   3.76416    3.808704 | 100.000001 | 100.000628  100.006588  100.037198  100.147769  100.459872  101.188915    102.651309 | 105.239237 | 1.560993  1.561062  1.561413  1.562683  1.566265   1.57462  1.591336     1.620771 |  100.0 |   0.000001 | 105.239237 |
    | 2000-01-04 14:30:00 | 3.717842  3.717934  3.718413   3.72016  3.725139  3.736858  3.760455    3.802142 | 100.000001 | 100.000543  100.005769   100.03294  100.132206  100.415434  101.083857    102.437772 | 104.855588 | 1.560992  1.561052  1.561364  1.562504  1.565755  1.573417  1.588899     1.616419 |  100.0 |   0.000001 | 104.855588 |
    | 2000-01-04 15:00:00 | 3.717841  3.717922  3.718346  3.719914  3.724431  3.735173  3.757024    3.796007 |      100.0 |  100.00047  100.005051  100.029161   100.11824  100.375132  100.987615    102.240299 | 104.497709 | 1.560992  1.561044  1.561321  1.562344  1.565292  1.572315  1.586644     1.612356 |  100.0 |        0.0 | 104.497709 |
    | 2000-01-04 15:30:00 |  3.71784  3.717911  3.718287  3.719694  3.723789   3.73363  3.753851    3.790277 |      100.0 | 100.000406  100.004422  100.025809  100.105713  100.338599  100.899499    102.057792 | 104.164068 | 1.560991  1.561037  1.561283    1.5622  1.564873  1.571305  1.584559     1.608564 |  100.0 |        0.0 | 104.164068 |
    | 2000-01-04 16:00:00 | 3.717839  3.717901  3.718235  3.719496  3.723207  3.732217  3.750916    3.784928 |      100.0 | 100.000351   100.00387  100.022836  100.094482    100.3055  100.818868    101.889214 | 103.853206 |  1.56099  1.561031  1.561248  1.562071  1.564493  1.570381  1.582633     1.605029 |  100.0 |        0.0 | 103.853206 |
    | 2000-01-04 16:30:00 | 3.717838  3.717892  3.718188  3.719319   3.72268  3.730925  3.748205    3.779939 |      100.0 | 100.000304  100.003387  100.020201  100.084415  100.275526  100.745125    101.733595 | 103.563739 |  1.56099  1.561025  1.561218  1.561956  1.564149  1.569536  1.580853     1.601735 |  100.0 |        0.0 | 103.563739 |
    | 2000-01-04 17:00:00 | 3.717837  3.717885  3.718147   3.71916  3.722203  3.729743  3.745701    3.775289 |      100.0 | 100.000262  100.002964  100.017865  100.075397  100.248395  100.677718    101.590019 | 103.294354 | 1.560989   1.56102  1.561191  1.561852  1.563838  1.568763  1.579211     1.598667 |  100.0 |        0.0 | 103.294354 |
    | 2000-01-04 17:30:00 | 3.717837  3.717878  3.718111  3.719018  3.721772  3.728663  3.743391    3.770958 |      100.0 | 100.000227  100.002593  100.015796   100.06732  100.223848  100.616136    101.457632 | 103.043804 | 1.560989  1.561016  1.561168  1.561759  1.563556  1.568057  1.577697     1.595812 |  100.0 |        0.0 | 103.043804 |
    | 2000-01-04 18:00:00 | 3.717836  3.717873  3.718079  3.718891  3.721381  3.727676   3.74126    3.766926 |      100.0 | 100.000196  100.002268  100.013963  100.060089   100.20165  100.559905    101.335631 | 102.810907 | 1.560988  1.561012  1.561147  1.561676  1.563302  1.567412    1.5763     1.593157 |  100.0 |        0.0 | 102.810907 |
    | 2000-01-04 18:30:00 | 3.717836  3.717868   3.71805  3.718777  3.721029  3.726775  3.739296    3.763175 |      100.0 |  100.00017  100.001984   100.01234  100.053619  100.181584  100.508586    101.223263 | 102.594547 | 1.560988  1.561009  1.561128  1.561602  1.563071  1.566824  1.575014     1.590688 |  100.0 |        0.0 | 102.594547 |
    | 2000-01-04 19:00:00 | 3.717835  3.717863  3.718025  3.718675   3.72071  3.725953  3.737487    3.759689 |      100.0 | 100.000147  100.001735  100.010903  100.047831  100.163455  100.461775    101.119827 | 102.393663 | 1.560988  1.561006  1.561111  1.561535  1.562863  1.566287  1.573829     1.588395 |  100.0 |        0.0 | 102.393663 |
    | 2000-01-04 19:30:00 | 3.717835  3.717859  3.718002  3.718584  3.720422  3.725203  3.735822     3.75645 |      100.0 | 100.000127  100.001517  100.009631  100.042654  100.147082  100.419096    101.024663 | 102.207258 | 1.560988  1.561004  1.561097  1.561476  1.562675  1.565797  1.572739     1.586267 |  100.0 |        0.0 | 102.207258 |
    | 2000-01-04 20:00:00 | 3.717835  3.717856  3.717983  3.718502  3.720162   3.72452   3.73429    3.753443 |      100.0 |  100.00011  100.001326  100.008506  100.038027  100.132301  100.380204    100.937159 | 102.034384 | 1.560987  1.561001  1.561084  1.561423  1.562505  1.565351  1.571737     1.584292 |  100.0 |        0.0 | 102.034384 |
    | 2000-01-04 20:30:00 | 3.717834  3.717853  3.717965  3.718429  3.719927  3.723898  3.732881    3.750654 |      100.0 | 100.000095  100.001159   100.00751  100.033892  100.118964   100.34478    100.856741 | 101.874152 | 1.560987  1.560999  1.561072  1.561375  1.562352  1.564944  1.570815      1.58246 |  100.0 |        0.0 | 101.874152 |
    | 2000-01-04 21:00:00 | 3.717834  3.717851   3.71795  3.718364  3.719715  3.723331  3.731587    3.748067 |      100.0 | 100.000082  100.001013   100.00663  100.030198  100.106934  100.312531    100.782874 | 101.725719 | 1.560987  1.560998  1.561062  1.561333  1.562214  1.564574  1.569969     1.580763 |  100.0 |        0.0 | 101.725719 |
    | 2000-01-04 21:30:00 | 3.717834  3.717848  3.717936  3.718306  3.719524  3.722815  3.730398    3.745671 |      100.0 | 100.000071  100.000886  100.005852  100.026899  100.096087  100.283185     100.71506 | 101.588294 | 1.560987  1.560996  1.561053  1.561295   1.56209  1.564237  1.569191     1.579191 |  100.0 |        0.0 | 101.588294 |
    | 2000-01-04 22:00:00 | 3.717834  3.717846  3.717924  3.718255  3.719352  3.722346  3.729306    3.743452 |      100.0 | 100.000061  100.000774  100.005164  100.023954  100.086312  100.256495    100.652836 |  101.46113 | 1.560987  1.560995  1.561045  1.561261  1.561977  1.563931  1.568478     1.577737 |  100.0 |        0.0 |  101.46113 |
    | 2000-01-04 22:30:00 | 3.717834  3.717845  3.717913  3.718208  3.719197  3.721919  3.728305    3.741398 |      100.0 | 100.000053  100.000676  100.004556  100.021325  100.077505   100.23223     100.59577 | 101.343526 | 1.560987  1.560994  1.561038  1.561231  1.561876  1.563653  1.567823     1.576391 |  100.0 |        0.0 | 101.343526 |
    | 2000-01-04 23:00:00 | 3.717834  3.717843  3.717904  3.718167  3.719058  3.721531  3.727387    3.739498 |      100.0 | 100.000046  100.000591  100.004018   100.01898  100.069574  100.210181    100.543462 | 101.234822 | 1.560987  1.560993  1.561032  1.561204  1.561785  1.563399  1.567224     1.575146 |  100.0 |        0.0 | 101.234822 |
    | 2000-01-04 23:30:00 | 3.717833  3.717842  3.717895   3.71813  3.718932  3.721179  3.726546    3.737742 |      100.0 |  100.00004  100.000516  100.003544  100.016889  100.062435  100.190154    100.495537 | 101.134398 | 1.560987  1.560992  1.561027   1.56118  1.561703   1.56317  1.566674     1.573996 |  100.0 |        0.0 | 101.134398 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _kinw_impl_euler_overbank_flow:

overbank flow
_____________

Now, we make the channel's profile more complicated by stacking three trapezes on top
of each other, as in the :ref:`kinw_williams_overbank_flow` example on |kinw_williams|:

>>> with model.add_wqmodel_v1("wq_trapeze_strickler"):
...     nmbtrapezes(3)
...     bottomlevels(0.0, 6.0, 8.0)
...     bottomwidths(15.0, 200.0, 0.0)
...     bottomslope(0.00025)
...     sideslopes(5.0, 10.0, 100.0)
...     stricklercoefficients(1.0/0.035, 10.0, 10.0)

The simulation results are similar to those of |kinw_williams| but with more
recognisable inflexion points related activation and deactivation of overbank flow,
which is due to the "limitation" of |kinw_impl_euler| not implementing an adaptive
numerical integration approach:

.. integration-test::

    >>> model.prepare_states(100.0)
    >>> test("kinw_impl_euler_overbank_flow")
    |                date |                                                                       waterdepth |     inflow |                                                                         internalflow |    outflow |                                                                         watervolume | input1 |     input2 |     output |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000003 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000029 | 100.000004  100.000001       100.0       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000029 |      100.0 |
    | 2000-01-01 04:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000175 | 100.000027  100.000004  100.000001       100.0       100.0       100.0         100.0 |      100.0 |  1.560986   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000175 |      100.0 |
    | 2000-01-01 04:30:00 | 3.717835  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000826 | 100.000136  100.000022  100.000004  100.000001       100.0       100.0         100.0 |      100.0 |  1.560988   1.560986  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.000826 |      100.0 |
    | 2000-01-01 05:00:00 | 3.717842  3.717834  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.003194 |  100.00055  100.000094  100.000016  100.000003       100.0       100.0         100.0 |      100.0 |  1.560992   1.560987  1.560986  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.003194 |      100.0 |
    | 2000-01-01 05:30:00 | 3.717866  3.717839  3.717834  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.010515 |   100.0019  100.000338  100.000059   100.00001  100.000002       100.0         100.0 |      100.0 |  1.561008    1.56099  1.560987  1.560986  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.010515 |      100.0 |
    | 2000-01-01 06:00:00 | 3.717934  3.717852  3.717836  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.030315 |  100.00575  100.001072  100.000197  100.000036  100.000006  100.000001         100.0 |      100.0 |  1.561052   1.560998  1.560988  1.560987  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.030315 |      100.0 |
    | 2000-01-01 06:30:00 | 3.718107  3.717886  3.717843  3.717835  3.717833  3.717833  3.717833    3.717833 | 100.078179 | 100.015564  100.003035  100.000581   100.00011   100.00002  100.000004    100.000001 |      100.0 |  1.561165   1.561021  1.560993  1.560987  1.560986  1.560986  1.560986     1.560986 |  100.0 |   0.078179 |      100.0 |
    | 2000-01-01 07:00:00 | 3.718507   3.71797   3.71786  3.717838  3.717834  3.717833  3.717833    3.717833 | 100.183351 | 100.038299  100.007813  100.001561  100.000306  100.000059  100.000011    100.000002 |      100.0 |  1.561426   1.561076  1.561004   1.56099  1.560987  1.560986  1.560986     1.560986 |  100.0 |   0.183351 |      100.0 |
    | 2000-01-01 07:30:00 | 3.719361  3.718159  3.717901  3.717847  3.717836  3.717833  3.717833    3.717833 | 100.396214 |   100.0868  100.018516  100.003858  100.000787  100.000158  100.000031    100.000006 | 100.000001 |  1.561983   1.561199   1.56103  1.560995  1.560988  1.560987  1.560986     1.560986 |  100.0 |   0.396214 | 100.000001 |
    | 2000-01-01 08:00:00 | 3.721055  3.718551  3.717989  3.717866   3.71784  3.717834  3.717833    3.717833 | 100.797305 | 100.183096  100.040817  100.008866  100.001882  100.000391   100.00008    100.000016 | 100.000003 |  1.563089   1.561455  1.561088  1.561008  1.560991  1.560987  1.560986     1.560986 |  100.0 |   0.797305 | 100.000003 |
    | 2000-01-01 08:30:00 | 3.724211  3.719319  3.718169  3.717907  3.717849  3.717836  3.717833    3.717833 | 101.507032 | 100.362585   100.08442  100.019103  100.004215   100.00091  100.000192     100.00004 | 100.000008 |  1.565149   1.561956  1.561206  1.561035  1.560997  1.560988  1.560987     1.560986 |  100.0 |   1.507032 | 100.000008 |
    | 2000-01-01 09:00:00 | 3.729764  3.720737  3.718517   3.71799  3.717868   3.71784  3.717834    3.717833 | 102.694853 | 100.678946  100.164995  100.038872  100.008911  100.001994  100.000436    100.000094 |  100.00002 |  1.568777   1.562881  1.561433  1.561089  1.561009  1.560991  1.560987     1.560986 |  100.0 |   2.694853 |  100.00002 |
    | 2000-01-01 09:30:00 | 3.739054  3.723226  3.719156  3.718148  3.717906  3.717849  3.717836    3.717834 | 104.586356 | 101.209449  100.306583  100.075149  100.017886  100.004147  100.000939    100.000208 | 100.000045 |  1.574856   1.564506  1.561849  1.561192  1.561034  1.560997  1.560989     1.560987 |  100.0 |   4.586356 | 100.000045 |
    | 2000-01-01 10:00:00 | 3.753893  3.727404  3.720275  3.718436  3.717978  3.717867  3.717841    3.717835 | 107.466641 | 102.060236  100.544408  100.138745  100.034262  100.008227  100.001927    100.000441 | 100.000099 |  1.584587   1.567234  1.562579   1.56138  1.561081  1.561008  1.560991     1.560987 |  100.0 |   7.466641 | 100.000099 |
    | 2000-01-01 10:30:00 | 3.776614  3.734129  3.722157   3.71894  3.718108  3.717899  3.717849    3.717836 | 111.678559 | 103.371034  100.928003  100.245728  100.062917  100.015638  100.003784    100.000894 | 100.000207 |  1.599541   1.571632  1.563807  1.561709  1.561166   1.56103  1.560996     1.560989 |  100.0 |  11.678559 | 100.000207 |
    | 2000-01-01 11:00:00 | 3.810064  3.744558  3.725204   3.71979  3.718336  3.717959  3.717863     3.71784 | 117.614836 | 105.318823    101.5245  100.419116  100.111189  100.028585  100.007145    100.001741 | 100.000415 |  1.621674   1.578462  1.565797  1.562263  1.561314  1.561068  1.561006     1.560991 |  100.0 |  17.614836 | 100.000415 |
    | 2000-01-01 11:30:00 | 3.857536   3.76018  3.729972  3.721172  3.718721  3.718062   3.71789    3.717847 | 125.703736 | 108.119985   102.42193  100.690808  100.189748  100.050423  100.013009    100.003268 | 100.000801 |  1.653324   1.588718  1.568913  1.563165  1.561565  1.561136  1.561024     1.560995 |  100.0 |  25.703736 | 100.000801 |
    | 2000-01-01 12:00:00 |  3.92261  3.782847  3.737205  3.723351  3.719348  3.718236  3.717937    3.717859 | 136.388647 | 112.030547  103.732402  101.103719  100.313651  100.086094  100.022912    100.005929 | 100.001496 |  1.697169   1.603655  1.573645  1.564587  1.561975  1.561249  1.561054     1.561003 |  100.0 |  36.388647 | 100.001496 |
    | 2000-01-01 12:30:00 | 4.008915  3.814777  3.747865  3.726687  3.720344  3.718522  3.718016     3.71788 | 150.102726 |  117.34411  105.594985  101.714099  100.503569  100.142672  100.039139    100.010429 | 100.002706 |  1.756134   1.624803   1.58063  1.566766  1.562624  1.561436  1.561106     1.561017 |  100.0 |  50.102726 | 100.002706 |
    | 2000-01-01 13:00:00 | 4.119816  3.858513  3.763165  3.731662  3.721881  3.718977  3.718147    3.717916 | 167.240349 | 124.386985  108.178138  102.593982  100.787199  100.230036  100.065007    100.017824 | 100.004755 |   1.83327   1.653979  1.590682  1.570018  1.563627  1.561733  1.561191     1.561041 |  100.0 |  67.240349 | 100.004755 |
    | 2000-01-01 13:30:00 | 4.258065  3.916856  3.784593  3.738904  3.724194  3.719685  3.718355    3.717976 | 188.127569 | 133.509905  111.681431  103.833731  101.200867   100.36166  100.105209    100.029665 |  100.00813 |  1.931582    1.69327  1.604808  1.574757  1.565138  1.562194  1.561327      1.56108 |  100.0 |  88.127569 |  100.00813 |
    | 2000-01-01 14:00:00 | 4.425484  3.992724  3.813918  3.749211  3.727599  3.720759  3.718681    3.718071 | 212.993962 | 145.075607  116.336168  105.544616  101.791309  100.555559  100.166246    100.048172 | 100.013555 |  2.053835   1.745001  1.624233  1.581513  1.567362  1.562895  1.561539     1.561142 |  100.0 | 112.993962 | 100.013555 |
    | 2000-01-01 14:30:00 | 4.622708   4.08898  3.853182  3.763576  3.732507  3.722354  3.719179    3.718222 | 241.948195 | 159.441754  122.404309  107.861294  102.617642  100.835375  100.256949    100.076463 | 100.022079 |  2.202347   1.811669   1.65041  1.590952   1.57057  1.563936  1.561864      1.56124 |  100.0 | 141.948195 | 100.022079 |
    | 2000-01-01 15:00:00 | 4.849057  4.208197  3.904666  3.783211  3.739442  3.724677  3.719925    3.718452 |  274.95935 | 176.939223  130.174918  110.943993  103.753491   101.23164  100.389104    100.118833 | 100.035189 |  2.378783   1.895845  1.685026  1.603895   1.57511  1.565453  1.562351      1.56139 |  100.0 |  174.95935 | 100.035189 |
    | 2000-01-01 15:30:00 | 5.102552  4.352412  3.970809  3.809558  3.749069  3.727997   3.72102      3.7188 | 311.845554 | 197.846626  139.957227  114.980056  105.289248  101.783201  100.578191    100.181093 |  100.05496 |  2.583981   2.000045  1.729985  1.621338  1.581421  1.567622  1.563066     1.561617 |  100.0 | 211.845554 |  100.05496 |
    | 2000-01-01 16:00:00 | 5.380047  4.522888  4.054102  3.844295  3.762209  3.732662  3.722601    3.719316 | 352.270855 | 222.362996  152.069479  120.184367  107.334352  102.538833  100.844252    100.270989 | 100.084235 |  2.817815   2.126574  1.787378  1.644468  1.590053  1.570672  1.564097     1.561953 |  100.0 | 252.270855 | 100.084235 |
    | 2000-01-01 16:30:00 | 5.677459  4.719921  4.156922  3.889315  3.779858  3.739115  3.724845    3.720066 | 395.750601 |  250.58136  166.823051  126.797986  110.019455  103.559018  101.212899    100.398703 | 100.126853 |   3.07912   2.277339  1.859423   1.67467  1.601681  1.574895  1.565563     1.562443 |  100.0 | 295.750601 | 100.126853 |
    | 2000-01-01 17:00:00 | 5.990049  4.942732  4.281349  3.946686  3.803208  3.747906  3.727984     3.72114 | 441.664962 | 282.466342  184.501989  135.084236  113.498205  104.917869  101.716478     100.57744 | 100.187925 |  3.365677   2.453675  1.948375  1.713524  1.617126  1.580658  1.567613     1.563144 |  100.0 | 341.664962 | 100.187925 |
    | 2000-01-01 17:30:00 |  6.09782  5.150883  4.422017  4.017479  3.833489  3.759696  3.732307    3.722656 | 489.279607 |  296.62657  201.992874  144.830235  117.879318  106.695649  102.394099    100.823956 | 100.274145 |  3.712453   2.624015  2.051268  1.762036  1.637257    1.5884  1.570439     1.564134 |  100.0 | 389.279607 | 100.274145 |
    | 2000-01-01 18:00:00 | 6.213979  5.350808  4.575159   4.10207  3.871876  3.775234  3.738171    3.724765 |  537.77215 | 313.425001  219.698205  155.903976  123.243691  108.974701  103.291154    101.158958 | 100.394117 |  4.116278   2.792723  2.166097  1.820825  1.662941  1.598631  1.574277      1.56551 |  100.0 |  437.77215 | 100.394117 |
    | 2000-01-01 18:30:00 | 6.343729   5.54841  4.738486  4.200418  3.919437  3.795339  3.746004    3.727654 | 586.262607 | 333.452069  238.086986  168.255414  129.659205   111.83798  104.458763     101.60739 | 100.558676 |  4.571337   2.964381  2.291794  1.890298  1.695019  1.611913   1.57941     1.567398 |  100.0 | 486.262607 | 100.558676 |
    | 2000-01-01 19:00:00 | 6.484685  5.747832  4.910749  4.312258  3.977114  3.820882    3.7563    3.731557 | 633.846018 | 356.550837  257.558344  181.898529  137.191096  115.369207  105.953387     102.19865 | 100.781201 |  5.070468   3.142567  2.427982  1.970771  1.734298  1.628862  1.586168     1.569949 |  100.0 | 533.846018 | 100.781201 |
    | 2000-01-01 19:30:00 | 6.634207  5.951663  5.091347  4.437229  4.045711  3.852768  3.769625    3.736754 | 679.625318 | 382.460797  278.425803  196.892729   145.90849  119.653891  107.836699    102.966796 | 101.077927 |  5.605364    3.32983  2.574741  2.062543  1.781557  1.650133  1.594934     1.573349 |  100.0 | 579.625318 | 101.077927 |
    | 2000-01-01 20:00:00 | 6.789505  6.049992  5.260613  4.571457  4.125339  3.891831  3.786596    3.743574 | 722.742733 | 410.812739  290.072989  211.599564  155.630553  124.744112  110.170677    103.950086 | 101.468151 |  6.166838   3.547162  2.715993  2.163287  1.837152  1.676365  1.606131     1.577817 |  100.0 | 622.742733 | 101.468151 |
    | 2000-01-01 20:30:00 | 6.947738  6.123323  5.417322  4.711275  4.215488  3.938754  3.807863    3.752398 | 762.408181 | 441.137766  300.213659   225.78839  166.158435  130.659407  113.013409    105.190044 | 101.974308 |  6.745125   3.800825  2.849959  2.270621   1.90105  1.708128  1.620213     1.583605 |  100.0 | 662.408181 | 101.974308 |
    | 2000-01-01 21:00:00 | 7.106112  6.206417  5.566583  4.854421  4.315312  3.994029  3.834078     3.76365 | 797.923464 | 472.884662  312.296967  239.823058  177.367134   137.40033  116.417209    106.730435 | 102.621909 |  7.330195   4.089883  2.980412  2.383042  1.972991  1.745897   1.63765     1.591001 |  100.0 | 697.923464 | 102.621909 |
    | 2000-01-01 21:30:00 | 7.261961  6.298514  5.712409  4.999692  4.423835  4.057953  3.865872    3.777794 | 828.701418 | 505.442628  326.332488  254.031911  189.193876  144.958861  120.428303    108.616343 | 103.439381 |  7.912061   4.412281  3.110553   2.49975  2.052614  1.790052  1.658911     1.600319 |  100.0 | 728.701418 | 103.439381 |
    | 2000-01-01 22:00:00 | 7.412817  6.398553  5.857846  5.146631  4.540104   4.13064  3.903829    3.795323 |  854.27953 | 538.166911  342.277085  268.698795   201.62599  153.326111  125.087494    110.893476 | 104.457865 |  8.481063   4.764883  3.242994  2.620481  2.139553  1.840882   1.68446     1.611903 |  100.0 |  754.27953 | 104.457865 |
    | 2000-01-01 22:30:00 | 7.556453  6.505211  6.001556   5.29465  4.663167  4.212041  3.948473    3.816754 | 874.327871 | 570.405101  360.024209  283.716661   214.63421   162.48948  130.430235    113.607573 | 105.710973 |  9.028124   5.143568  3.380347   2.74483  2.233414  1.898589  1.714741     1.626117 |  100.0 | 774.327871 | 105.710973 |
    | 2000-01-01 23:00:00 | 7.690925  6.616962  6.048361  5.427198  4.789204  4.301473  4.000173    3.842598 | 888.651531 | 601.522255  379.401511  289.853788  226.701223  172.206077  136.453737    116.799248 | 107.233896 |  9.544957   5.543386  3.541533  2.858504  2.331505  1.962943  1.750119     1.643335 |  100.0 | 788.651531 | 107.233896 |
    | 2000-01-01 23:30:00 | 7.814587   6.73214  6.101984  5.548925  4.915763  4.397906  4.059104    3.873342 | 897.187983 | 630.923383  400.173747  297.208105   238.13603  182.305175  143.131025    120.501315 | 109.062279 | 10.024233   5.958735  3.726871  2.964834  2.432001  2.033456  1.790853     1.663925 |  100.0 | 797.187983 | 109.062279 |
    | 2000-01-02 00:00:00 |   7.9261   6.84901  6.162275  5.663589  5.041439  4.500189  4.125242    3.909417 |      900.0 | 658.072291  422.050007  305.805342  249.219882  192.677656  150.422013    124.737854 | 111.231094 | 10.459703   6.383575  3.936112  3.066688  2.533777  2.109516  1.837084     1.688237 |  100.0 |      800.0 | 111.231094 |
    | 2000-01-02 00:30:00 | 8.024228  6.965862  6.228896  5.774216  5.165602  4.607192  4.198383    3.951185 | 897.264896 | 682.582901  444.701124  315.663093  260.204069  203.265897  158.282128    129.524486 | 113.773692 | 10.846131   6.811762   4.16838  3.166514  2.636266  2.190487  1.888848     1.716589 |  100.0 | 797.264896 | 113.773692 |
    | 2000-01-02 01:00:00 | 8.105723  7.080836  6.301311  5.883193   5.28816  4.717916  4.278182    3.998917 | 889.260928 | 703.635207  467.726364  326.768597  271.306132  214.053577  166.668764     134.86945 | 116.721076 | 11.180257   7.236398  4.422104  3.266346   2.73932   2.27578  1.946087     1.749256 |  100.0 | 789.260928 | 116.721076 |
    | 2000-01-02 01:30:00 | 8.171091  7.192126   6.37879   5.99232  5.409376  4.831537  4.364195    4.052791 | 876.351729 |  720.79486  490.694547  339.071914  282.706509  225.055512  175.545754    140.775151 | 120.101451 | 11.460259   7.650579  4.695025  3.367804  2.843092  2.364897  2.008674     1.786468 |  100.0 | 776.351729 | 120.101451 |
    | 2000-01-02 02:00:00 | 8.221772  7.298271  6.460459  6.031821  5.517865  4.945308  4.455563    4.112833 | 858.969595 | 734.276828  513.211595  352.486654  287.649581  235.185958  184.712633    147.214388 | 123.936388 | 11.684706   8.048496   4.98433  3.484511  2.937526  2.455749  2.076171     1.828369 |  100.0 | 758.969595 | 123.936388 |
    | 2000-01-02 02:30:00 | 8.258805  7.398087   6.54534   6.07045  5.616175  5.057348   4.55121    4.178896 |   837.5984 | 744.229759  534.920037  366.892274  292.846899  244.599676  194.015339    154.140028 | 128.238711 |  11.85277   8.425254   5.28678  3.617793  3.024371  2.546801  2.147946     1.874991 |  100.0 |   737.5984 | 128.238711 |
    | 2000-01-02 03:00:00 | 8.282981   7.49062  6.632391  6.113766  5.707327  5.166621  4.650015    4.250667 | 812.756821 | 750.774816  555.499992  382.137687   298.86229  253.528362  203.354213    161.494991 | 133.011957 | 11.964338   8.776748  5.598832  3.767688  3.105972  2.637115  2.223293     1.926261 |  100.0 | 712.756821 | 133.011957 |
    | 2000-01-02 03:30:00 | 8.294939  7.575117  6.720541  6.161636   5.79379  5.272704  4.750934    4.327699 | 784.982427 | 754.026056  574.670357  398.045957   305.71251  262.177441  212.674539    169.219982 | 138.250928 | 12.020059   9.099589  5.916756  3.933888  3.184336   2.72622  2.301511     1.982005 |  100.0 | 684.982427 | 138.250928 |
    | 2000-01-02 04:00:00 | 8.295215  7.650998  6.808724   6.21382  5.877562  5.375612  4.853069    4.409449 | 754.817086 | 754.101116  592.189602  414.419821  313.401207  270.725597  221.957543    177.259225 |    143.943 | 12.021348   9.391029  6.236742  4.115722  3.261152  2.814002  2.381968     2.041974 |  100.0 | 654.817086 |    143.943 |
    | 2000-01-02 04:30:00 | 8.284275  7.717832  6.895903  6.269978  5.960231  5.475649  4.955706    4.495317 |  722.79402 | 751.126179   607.85592  431.047714  321.915296  279.324567  231.211959    185.564436 | 150.069861 |  11.97035   9.648916  6.554996   4.31216  3.337815  2.900605  2.464133     2.105864 |  100.0 |  622.79402 | 150.069861 |
    | 2000-01-02 05:00:00 | 8.262536   7.77532  6.981094  6.329673  6.013265   5.56839  5.057449    4.584544 | 689.426711 | 745.237277  621.506605   447.70996  331.223207   285.22012   239.99611    194.023812 | 156.598511 | 11.869891   9.871631   6.86783  4.521836  3.420621  2.982008  2.546884      2.17323 |  100.0 | 589.426711 | 156.598511 |
    | 2000-01-02 05:30:00 | 8.230367  7.823274  7.063383  6.392387  6.040675  5.651084  5.156631    4.676182 | 655.199767 |  736.57913  633.016651  464.184831  341.274166   288.82539   247.99634    202.489448 | 163.477272 | 11.723408  10.058044  7.171728  4.743076  3.515028  3.055501  2.628796     2.243452 |  100.0 | 555.199767 | 163.477272 |
    | 2000-01-02 06:00:00 | 8.188101  7.861602  7.141935  6.457533  6.071452   5.72634  5.252319    4.769252 | 620.561767 | 725.302205  642.296593  480.254189  351.998375  292.983915   255.41522    210.863996 |   170.6454 | 11.534875  10.207454  7.463404  4.973936  3.621254  3.123124  2.708988     2.315845 |  100.0 | 520.561767 |   170.6454 |
    | 2000-01-02 06:30:00 | 8.136025  7.890294  7.216005  6.524472  6.105581  5.796304  5.344122    4.862845 | 585.920022 | 711.558343  649.289643  495.708487  363.307981  297.711685  262.431589    219.091529 | 178.040442 | 11.308726  10.319537   7.73985  5.212257  3.739328  3.186628     2.787     2.389737 |  100.0 | 485.920022 | 178.040442 |
    | 2000-01-02 07:00:00 | 8.074379  7.909399  7.284939  6.592531  6.142986  5.862745  5.432035    4.956184 |  551.63713 | 695.494848  653.968124  510.350971  375.098761  303.019751  269.201552    227.149194 |  185.60364 | 11.049782  10.394285  7.998361  5.455711   3.86907  3.247501  2.862695     2.464519 |  100.0 |  451.63713 |  185.60364 |
    | 2000-01-02 07:30:00 | 8.003347  7.919014  7.348176  6.661015  6.183527  5.927106  5.516318    5.048655 | 518.029154 | 677.245911   656.32912  524.000941  387.252398  308.910837   275.85941    235.039674 |  193.28366 | 10.763192  10.431936  8.236552  5.701858  4.010085  3.306994   2.93617      2.53968 |  100.0 | 418.029154 |  193.28366 |
    | 2000-01-02 08:00:00 | 7.924408  7.919561  7.405275  6.729233  6.227001  5.990544  5.597398    5.139816 | 485.365238 | 657.655691  656.463751  536.503057  399.639792  315.377731  282.518596    242.784476 | 201.038858 | 10.453069  10.434081  8.452481  5.948212  4.161756   3.36614  3.007692     2.614822 |  100.0 | 385.365238 | 201.038858 |
    | 2000-01-02 08:30:00 |  7.84003   7.91155  7.455913  6.796509  6.273144  6.016652  5.669739    5.228316 |  453.86844 | 637.064815  654.495862  547.729949  412.124757   322.40234  285.659763    249.823054 | 208.744007 | 10.123316  10.402705   8.64466  6.192302  4.323257  3.432277  3.072198     2.688764 |  100.0 |  353.86844 | 208.744007 |
    | 2000-01-02 09:00:00 | 7.751096  7.895527  7.499878  6.862198  6.321637  6.038311  5.734322    5.313198 | 423.717566 | 615.734918  650.569422  557.582992  424.567859  329.955261  288.510454    256.209912 | 216.298863 |  9.777685  10.340003  8.812035  6.431729  4.493559  3.506877  3.130339     2.760604 |  100.0 | 323.717566 | 216.298863 |
    | 2000-01-02 09:30:00 | 7.658445  7.872076  7.537062  6.925703  6.372117  6.062436  5.793097    5.394015 | 395.049792 | 593.924397  644.844835  565.992241  436.830195  337.995805  291.754796    262.107423 |  223.64292 |   9.41971  10.248346   8.95397  6.664221  4.671461  3.590111  3.183704      2.82984 |  100.0 | 295.049792 |  223.64292 |
    | 2000-01-02 10:00:00 | 7.562869  7.841802  7.567447  6.986481   6.42418  6.089031  5.847679    5.470685 | 367.963873 | 571.869329  637.493703  572.915483  448.776953  346.472467  295.404618    267.657219 | 230.747343 |  9.052681  10.130223  9.070211   6.88767  4.855609  3.682033  3.233649     2.896278 |  100.0 | 267.963873 | 230.747343 |
    | 2000-01-02 10:30:00 | 7.465105  7.805319  7.591099  7.044048  6.477394   6.11806  5.899405    5.543381 | 342.523746 | 549.782011  628.694052  578.336573  460.280604  355.323814  299.468399    272.981819 | 237.607874 |  8.679616   9.988181  9.160854  7.100171  5.044532  3.782573  3.281325     2.959951 |  100.0 | 242.523746 | 237.607874 |
    | 2000-01-02 11:00:00 | 7.365838  7.763245  7.608156  7.097991  6.531312   6.14945  5.949375    5.612447 | 318.762374 | 527.850309  618.626087  582.263213  471.223621  364.479734  303.949713    278.186113 | 244.238672 |  8.303257   9.824784  9.226307  7.300042  5.236671  3.891527  3.327699     3.021057 |  100.0 | 218.762374 | 244.238672 |
    | 2000-01-02 11:30:00 | 7.265692  7.716191  7.618815  7.147961  6.585475  6.183089  5.998482     5.67833 |  296.68568 | 506.237697  607.468531  584.724365   481.50067  373.862987  308.846374    283.358727 | 250.667034 |  7.926064   9.642569  9.267247  7.485845  5.430419  4.008557  3.373577     3.079902 |  100.0 |  196.68568 | 250.667034 |
    | 2000-01-02 12:00:00 | 7.165233  7.664751  7.623325  7.193681   6.63943  6.218823  6.014634    5.736261 | 276.276442 | 485.083853  595.395547  585.767411  491.020227  383.390983  314.149874    285.397587 |  256.40312 |   7.55021   9.444008  9.284577  7.656389  5.624151  4.133191  3.425331     3.132092 |  100.0 | 176.276442 |  256.40312 |
    | 2000-01-02 12:30:00 | 7.064968  7.609504  7.621976  7.234939  6.692735  6.256468  6.031488    5.787868 | 257.498082 | 464.505726  582.574244  585.455231  499.705638  392.977694  319.845067     287.60552 | 261.579447 |  7.177597   9.231485  9.279392  7.810739  5.816261   4.26483  3.483363     3.178939 |  100.0 | 157.498082 | 261.579447 |
    | 2000-01-02 13:00:00 | 6.965344  7.551002  7.615088  7.271588  6.744965  6.295802  6.050184    5.834683 | 240.298231 | 444.598954  569.162733  583.863279  507.495651  402.535616  325.910064    290.098762 | 266.329424 |  6.809855    9.00727  9.252931    7.9482   6.00519  4.402756  3.547823     3.221723 |  100.0 | 140.298231 | 266.329424 |
    | 2000-01-02 13:30:00 | 6.866752  7.489768  7.603009  7.303539  6.795724  6.336578  6.070741    5.877979 | 224.612057 | 425.439541  555.308697  581.076767  514.344463  411.977701  332.316346    292.886621 | 270.768533 |  6.448366   8.773505  9.206548  8.068319   6.18945  4.546146  3.618796     3.261536 |  100.0 | 124.612057 | 270.768533 |
    | 2000-01-02 14:00:00 | 6.769527  7.426299  7.586098  7.330762  6.844647  6.378525  6.093151    5.918815 | 210.365288 | 407.085728   541.14843  577.188018  520.221347  421.219179   339.02909    295.976507 |  274.99618 |  6.094269   8.532192  9.141677  8.170859  6.367654  4.694088  3.696291     3.299301 |  100.0 | 110.365288 |  274.99618 |
    | 2000-01-02 14:30:00 | 6.673951  7.361055  7.564728  7.353272  6.891405  6.421354  6.117385    5.958066 |  197.47692 | 389.579987  526.806278  572.294044  525.109939  430.179215  346.007686    299.373027 | 279.097337 |  5.748484   8.285185  9.059799   8.25579  6.538529  4.845597  3.780233     3.335797 |  100.0 |   97.47692 | 279.097337 |
    | 2000-01-02 15:00:00 | 6.580257  7.294467  7.539272  7.371132  6.935706  6.464766  6.143388    5.996455 | 185.861607 | 372.951102  512.394456  566.494375  529.007239  438.782351  353.206432    303.077452 | 283.144008 |  5.411722   8.034187  8.962419  8.323267  6.700934  4.999634  3.870466     3.371677 |  100.0 |  85.861607 | 283.144008 |
    | 2000-01-02 15:30:00 |  6.48863  7.226933  7.510106  7.384443  6.977302  6.508454  6.171079    6.010656 | 175.431719 | 357.216329  498.013158  559.889165  531.922423  446.959701  360.575371     307.08734 | 284.882829 |   5.08451   7.780753  8.851042  8.373607  6.853867  5.155126  3.966744     3.411645 |  100.0 |  75.431719 | 284.882829 |
    | 2000-01-02 16:00:00 | 6.399208  7.158817  7.477598  7.393337  7.015984  6.552111  6.200356    6.023639 | 166.099094 | 342.383683  483.750953  552.577579  533.875512  454.649886  368.061244    311.396274 | 286.572201 |  4.767198   7.526292  8.727154  8.407271  6.996473  5.310985  4.068741     3.456328 |  100.0 |  66.099094 | 286.572201 |
    | 2000-01-02 16:30:00 | 6.312088  7.090455  7.442108  7.397977  7.051583  6.595436   6.23109    6.038018 | 157.776495 |  328.45449  469.685421  544.656466   534.89597  461.799706  375.608511    315.993707 | 288.471511 |  4.459978   7.272076  8.592206   8.42484  7.128046  5.466129  4.176048     3.505868 |  100.0 |  57.776495 | 288.471511 |
    | 2000-01-02 17:00:00 | 6.227326  7.022154  7.403987  7.398547  7.083972  6.638137  6.263134    6.053819 | 150.378786 | 315.426609  455.884046  536.219304  535.021277  468.364545   383.16041    320.864921 | 290.588361 |  4.162891   7.019253  8.447603  8.426996  7.248028  5.619497   4.28818     3.560366 |  100.0 |  50.378786 | 290.588361 |
    | 2000-01-02 17:30:00 | 6.144934  6.954197  7.363572  7.395248  7.113059  6.679935  6.296322    6.071053 | 143.823868 | 303.299565  442.405489  527.355439  534.295542  474.308554   390.66001    325.991087 | 292.929399 |  3.875835   6.768862  8.294693  8.414504  7.356005  5.770064  4.404584     3.619877 |  100.0 |  43.823868 | 292.929399 |
    | 2000-01-02 18:00:00 | 6.064877  6.886854  7.321186  7.388297  7.138791   6.72057  6.330471    6.089717 | 138.033385 | 292.086645  429.301761  518.149647  532.768169  479.604604  398.051231    331.349436 | 295.499713 |  3.598539   6.521875  8.134767  8.388191  7.451699   5.91686  4.524647     3.684407 |  100.0 |  38.033385 | 295.499713 |
    | 2000-01-02 18:30:00 | 5.957751  6.819148  7.277038   7.37791  7.161143  6.759801  6.365387    6.109789 | 132.933235 | 279.064302  416.385268   508.66015  530.490586  484.233884  405.279774    336.913535 | 298.302466 |  3.335503   6.274697  7.968672  8.348896  7.534961  6.058978  4.647706     3.753907 |  100.0 |  32.933235 | 298.302466 |
    | 2000-01-02 19:00:00 |   5.7114  6.745476  7.230867  7.364263  7.180118   6.79741  6.400865    6.131235 | 128.453911 | 253.931868  402.629886  498.844418   527.50631  488.184541   412.29388     342.65364 | 301.338631 |  3.109643   6.007041  7.795486  8.297304   7.60574  6.195581  4.773058     3.828274 |  100.0 |  28.453911 | 301.338631 |
    | 2000-01-02 19:30:00 | 5.491977  6.667908  7.182624   7.34751  7.195737  6.833202  6.436694    6.154003 | 124.530701 | 232.744148  388.491316  488.707802  523.856082  491.450848  419.044953    348.537128 |  304.60678 |  2.914859   5.726696  7.615096  8.234037   7.66407  6.325911  4.899973     3.907348 |  100.0 |  24.530701 |  304.60678 |
    | 2000-01-02 20:00:00 | 5.296307  6.588022  7.132405  7.327804   7.20804  6.867005  6.472661    6.178023 | 121.103761 |   214.7826  374.308788  478.286988  519.580946  494.032679  425.488075    354.528985 |  308.10293 |  2.746237   5.439549  7.427935  8.159708  7.710057  6.449292  5.027699     3.990915 |  100.0 |  21.103761 |  308.10293 |
    | 2000-01-02 20:30:00 | 5.121673  6.507021    7.0804  7.305302  7.217081  6.898669  6.508554    6.203212 | 118.118088 | 199.480745  360.331669  467.637826  514.723882  495.935148  431.582419    360.592333 | 311.820433 |  2.599784   5.150017  7.234784  8.074953  7.743876  6.565127  5.155481     4.078705 |  100.0 |  18.118088 | 311.820433 |
    | 2000-01-02 21:00:00 | 4.965729  6.425809  7.026858  7.280172   7.22293   6.92807  6.544161    6.229473 | 115.523427 | 186.387781  346.740622  456.826542  509.330436  497.168316  437.291578    366.688993 | 315.749933 |  2.472228   4.861382   7.03663  7.980446  7.765768  6.672905  5.282566     4.170395 |  100.0 |  15.523427 | 315.749933 |
    | 2000-01-02 21:30:00 | 4.826442  6.345064  6.972059   7.25259  7.225672  6.955104  6.579278    6.256692 | 113.274094 | 175.141761  333.664491  445.923544  503.448689  497.746897  442.583813    372.780058 | 319.879368 |  2.360867   4.576041  6.834563  7.876901  7.776031  6.772198  5.408213     4.265616 |  100.0 |  13.274094 | 319.879368 |
    | 2000-01-02 22:00:00 | 4.702031  6.265281  6.916295  7.222743  7.225402  6.979691  6.613709    6.284747 | 111.328777 | 165.449622  321.194041   434.99926  497.128847  497.689948  447.432212    378.826464 | 324.194035 |  2.263449   4.295701  6.629714  7.765068  7.775021  6.862662  5.531703     4.363955 |  100.0 |  11.328777 | 324.194035 |
    | 2000-01-02 22:30:00 |  4.59093   6.18681  6.859864  7.190827   7.22223  7.001771  6.647267    6.313505 | 109.650285 | 157.072176  309.393889  424.121565  490.422617  497.020496  451.814779    384.789557 | 328.676705 |   2.17809   4.021522  6.423204  7.645726  7.763145  6.944033  5.652348     4.464958 |  100.0 |   9.650285 | 328.676705 |
    | 2000-01-02 23:00:00 | 4.491756  6.109879   6.80306  7.157042  7.216274  7.021306  6.679777    6.342822 | 108.205287 |  149.81273   298.31508  413.354671   483.38256  495.765148  455.714439    390.631622 | 333.307803 |  2.103196   3.754218  6.216133  7.519676  7.740857  7.016124  5.769497     4.568141 |  100.0 |   8.205287 | 333.307803 |
    | 2000-01-02 23:30:00 | 4.403276  6.034604  6.746181  7.121596  7.207664  7.038276   6.71108    6.372551 | 106.964036 | 143.508391  288.018059  402.760106  476.061595  493.953668  459.118982     396.31638 | 338.065625 |  2.037416     3.4941  6.009597  7.387733  7.708651  7.078826  5.882542     4.672992 |  100.0 |   6.964036 | 338.065625 |
    | 2000-01-03 00:00:00 | 4.324389   5.87222  6.685695  7.084392   7.19651  7.052677  6.741027    6.402539 | 105.900084 | 138.023356  270.175454  391.701847   468.45008  491.612824  462.020419    401.809402 | 342.926605 |  1.979594   3.256226   5.79085  7.249586  7.666958  7.132093  5.990922     4.778981 |  100.0 |   5.900084 | 342.926605 |
    | 2000-01-03 00:30:00 | 4.254112  5.645633  6.619513  7.045192  7.182894  7.064516  6.769487    6.432631 |  104.99001 | 133.243702  247.464084  379.852937  460.511033  488.764097  464.414175    407.078409 | 347.865611 |  1.928738    3.05063   5.55255  7.104402  7.616102  7.175922  6.094126     4.885564 |  100.0 |    4.99001 | 347.865611 |
    | 2000-01-03 01:00:00 | 4.191558  5.443057  6.549477  7.003944  7.166887   7.07381  6.796342    6.462671 | 104.213141 | 129.073312  228.171788  367.606167  452.247544  485.427808  466.298676    412.093503 | 352.856257 |   1.88399   2.872252  5.301568  6.952047  7.556378  7.210355  6.191696     4.992191 |  100.0 |   4.213141 | 352.856257 |
    | 2000-01-03 01:30:00 |  4.13593  5.261763   6.47701  6.960727  7.148565  7.080584  6.821488    6.492507 | 103.551298 | 125.430656  211.701614  355.259198  443.689606  481.625646  467.675201    416.827364 | 357.871238 |  1.844607   2.716965  5.043164  6.792873  7.488093  7.235466  6.283222     5.098312 |  100.0 |   3.551298 | 357.871238 |
    | 2000-01-03 02:00:00 | 4.086509  5.099387  6.403205  6.915703  7.128013  7.084872  6.844838    6.521985 | 102.988541 | 122.246241  197.576901   343.03533   434.88427  477.382025  468.547847      421.2554 | 362.882664 |  1.809943    2.58137  4.781339  6.627544  7.411597  7.251367  6.368348     5.203383 |  100.0 |   2.988541 | 362.882664 |
    | 2000-01-03 02:30:00 | 4.042647  4.953871  6.328897  6.869094   7.10533  7.086717  6.866315     6.55096 |  102.51094 | 119.460565  185.413991  331.100485  425.888402  472.724635  468.923549    425.355879 | 367.862403 |  1.779434   2.462653  4.519103  6.456926  7.327292  7.258209   6.44677     5.306871 |  100.0 |    2.51094 | 367.862403 |
    | 2000-01-03 03:00:00 | 4.003758  4.823411  6.254713  6.821151   7.08063   7.08617  6.885859    6.579291 | 102.106358 | 117.022481   174.90175  319.577259  416.763521  467.684448  468.812079    429.110035 |  372.78242 |  1.752584   2.358471  4.258687  6.281991  7.235634  7.256179  6.518234     5.408261 |  100.0 |   2.106358 |  372.78242 |
    | 2000-01-03 03:30:00 | 3.969314  4.706421   6.18112  6.772142  7.054039  7.083291  6.903423    6.606844 | 101.764251 |  114.88786  165.786045  308.557166  407.572322  462.295395  468.226016    432.502135 | 377.615099 |  1.728962   2.266854  4.001699  6.103764  7.137133  7.245504  6.582537     5.507058 |  100.0 |   1.764251 | 377.615099 |
    | 2000-01-03 04:00:00 | 3.938837  4.601499  6.108442  6.722347  7.025697  7.078152  6.918971    6.633492 |  101.47548 | 113.018506  157.857897  298.113192    398.3767   456.59389  467.180662    435.519536 | 382.333565 |  1.708185   2.186143   3.74924  5.923289  7.032342  7.226448  6.639527     5.602792 |  100.0 |    1.47548 | 382.333565 |
    | 2000-01-03 04:30:00 |   3.9119  4.507401  6.036885  6.672053  6.995757   7.07083  6.932483    6.659119 | 101.232151 | 111.381268  150.944348  288.320846  389.237789   450.61839  465.693923    438.152691 | 386.911973 |  1.689916    2.11493  3.501962  5.741639  6.921857  7.199312  6.689101     5.695026 |  100.0 |   1.232151 | 386.911973 |
    | 2000-01-03 05:00:00 | 3.888113  4.423022  5.890484   6.61825  6.964118  7.061391  6.943947    6.683617 | 101.027458 | 109.947297  144.901382  272.058953  379.629367  444.357422  463.781529    440.394738 | 391.325752 |   1.67386   2.052012  3.273079  5.548012  6.805346  7.164349  6.731197      5.78335 |  100.0 |   1.027458 | 391.325752 |
    | 2000-01-03 05:30:00 | 3.867131  4.347374  5.672064  6.558441  6.930507  7.049871  6.953358    6.706888 | 100.855551 | 108.691441  139.608387  250.051262  369.156595  437.766948  461.454296    442.240854 | 395.551765 |  1.659756   1.996362  3.074281  5.333622  6.681848  7.121711  6.765781      5.86739 |  100.0 |   0.855551 | 395.551765 |
    | 2000-01-03 06:00:00 | 3.848639  4.279574  5.476062  6.494419  6.894834  7.036301  6.960719    6.728845 | 100.711411 | 107.591725  134.963802  231.250628  358.194715   430.84128  458.722006    443.687994 | 399.568436 |  1.647371   1.947092  2.900965  5.105123  6.551084  7.071526  6.792842     5.946805 |  100.0 |   0.711411 | 399.568436 |
    | 2000-01-03 06:30:00 | 3.832358  4.218828  5.299973  6.427581  6.857141  7.020714  6.966033    6.749407 | 100.590745 | 106.628928  130.881667  215.111129   347.03265  423.601272   455.59586    444.734866 | 403.355853 |  1.636503   1.903437  2.749352  4.867664   6.41326  7.013936  6.812392     6.021288 |  100.0 |   0.590745 | 403.355853 |
    | 2000-01-03 07:00:00 | 3.818036  4.164425  5.141623  6.359011  6.817559  7.003153  6.969315    6.768505 | 100.489885 | 105.786211  127.288872   201.19444  335.891072  416.085315  452.089878    445.382018 | 406.895863 |  1.626969   1.864732  2.616322   4.62521  6.268911  6.949128  6.824466     6.090563 |  100.0 |   0.489885 | 406.895863 |
    | 2000-01-03 07:30:00 | 3.805449  4.115726  4.999115  6.289544  6.776278  6.983678  6.970582     6.78608 | 100.405708 | 105.048813  124.122956  189.145931  324.937476   408.34261  448.221552    445.631972 | 410.172164 |  1.618612   1.830399  2.499281  4.380786  6.118781  6.877346  6.829128      6.15439 |  100.0 |   0.405708 | 410.172164 |
    | 2000-01-03 08:00:00 | 3.794395  4.072154  4.870786  6.219818  6.733525  6.962364  6.969859    6.802079 |  100.33556 | 104.403781  121.330328  178.676597  314.298972   400.42828  444.011977    445.489362 |  413.17039 |  1.611289   1.799931  2.396057  4.136665  5.963749  6.798895  6.826468     6.212565 |  100.0 |    0.33556 |  413.17039 |
    | 2000-01-03 08:30:00 | 3.784697  4.033188  4.755174  6.150312   6.68955  6.939303   6.96718    6.816462 | 100.277186 | 103.839744  118.864828  169.549303   304.07393  392.400031  439.485656    444.961022 | 415.878192 |  1.604876   1.772886  2.304825  3.894521  5.804762  6.714141  6.816613     6.264914 |  100.0 |   0.277186 | 415.878192 |
    | 2000-01-03 09:00:00 | 3.776194  3.998362  4.650985  6.081366  6.644617  6.914601  6.962588    6.829194 | 100.228681 | 103.346714  116.686547    161.5682  294.345259  384.316312  434.670155    444.056042 | 418.285306 |  1.599264   1.748874  2.224038  3.655522  5.642814  6.623504  6.799718     6.311301 |  100.0 |   0.228681 | 418.285306 |
    | 2000-01-03 09:30:00 | 3.768744  3.967253   4.55707  6.013192  6.599015   6.88838  6.956132    6.840254 | 100.188431 | 102.915922   114.76087  154.570544  285.210691  376.237164   429.59584    442.785777 | 420.383607 |  1.594354   1.727553  2.152381   3.42037  5.478966  6.527458  6.775976     6.351625 |  100.0 |   0.188431 | 420.383607 |
    | 2000-01-03 10:00:00 | 3.762223  3.939479  4.472408  5.822322  6.547633  6.860353  6.947835    6.849622 | 100.155078 |  102.53966  113.057677  148.420307  265.070237  367.287762  424.215003    441.156768 | 422.166548 |  1.590062   1.708621  2.088728    3.2104  5.294974  6.424989  6.745481     6.385807 |  100.0 |   0.155078 | 422.166548 |
    | 2000-01-03 10:30:00 | 3.756518  3.914698  4.396085   5.61914  6.490771  6.830302  6.937702    6.857285 | 100.127477 |  102.21116  111.550688  143.003158  244.887043  357.577846  418.495141    439.172513 | 423.628775 |  1.586311    1.69181  2.032114  3.027009  5.092131  6.315338  6.708262     6.413786 |  100.0 |   0.127477 | 423.628775 |
    | 2000-01-03 11:00:00 |  3.75153    3.8926  4.327286  5.436144  6.430018  6.798168  6.925736    6.863234 | 100.104666 | 101.924477  110.216917  138.222496  227.530052  347.434477  412.435977    436.836574 | 424.766044 |  1.583036   1.676883  1.981704  2.866256  4.876303  6.198335  6.664341     6.435513 |  100.0 |   0.104666 | 424.766044 |
    | 2000-01-03 11:30:00 | 3.747173  3.872905  4.265276  5.271129  6.366622  6.764005  6.911946     6.86746 | 100.085838 | 101.674391  109.036209  133.996298  212.534364  337.111863  406.059714    434.154541 | 425.575307 |  1.580176   1.663632  1.936776  2.724887  4.652064  6.074229   6.61377     6.450956 |  100.0 |   0.085838 | 425.575307 |
    | 2000-01-03 12:00:00 | 3.743368  3.855363  4.209397  5.122177  6.301553  6.727947  6.896355    6.869962 | 100.070317 | 101.456318   107.99086  130.254583  199.524008  326.806399  399.403566    431.135139 | 426.054885 |  1.577681    1.65187  1.896701  2.600202  4.422955  5.943554  6.556653       6.4601 |  100.0 |   0.070317 | 426.054885 |
    | 2000-01-03 12:30:00 | 3.740047  3.839748  4.159054   4.98761  6.235567  6.690179  6.879003    6.870743 | 100.057539 | 101.266238  107.065289  126.937385  188.192706  316.669501  392.514144    427.790725 | 426.204665 |  1.575506   1.641432  1.860931  2.489943  4.191697  5.807034  6.493155     6.462955 |  100.0 |   0.057539 | 426.204665 |
    | 2000-01-03 13:00:00 | 3.737151  3.825854  4.113713  4.865953   6.16924  6.650923  6.859947    6.869813 | 100.047032 | 101.100623  106.245764  123.993102  178.289228  306.818996  385.443399     424.13737 | 426.026287 |  1.573609    1.63217  1.828986   2.39221  3.960343   5.66551  6.423506     6.459555 |  100.0 |   0.047032 | 426.026287 |
    | 2000-01-03 13:30:00 | 3.734626    3.8135  4.072889  4.755906  6.103004  6.610421  6.839259    6.867189 | 100.038402 |  100.95638  105.520162  121.377155  169.606152  297.350772  378.245979    420.194681 | 425.523289 |  1.571957   1.623955  1.800443  2.305397  3.730403  5.519899  6.347998     6.449963 |  100.0 |   0.038402 | 425.523289 |
    | 2000-01-03 14:00:00 | 3.732426  3.802522  4.036146  4.656318  6.037159  6.568937  6.817031    6.862895 | 100.031322 |   100.8308  104.877774  119.050889  161.971161  288.357315  370.978392    415.985553 | 424.701214 |  1.570518   1.616671  1.774932  2.228141  3.502908  5.371181  6.266986     6.434275 |  100.0 |   0.031322 | 424.701214 |
    | 2000-01-03 14:30:00 | 3.730511   3.79277  4.003087  4.566167  5.908081  6.523945  6.793149    6.856947 | 100.025521 | 100.721508  104.309124  116.980676  155.240258   273.88115  363.217749    411.495084 | 423.564169 |  1.569265   1.610213  1.752123  2.159274  3.289354  5.210375  6.180086     6.412551 |  100.0 |   0.025521 | 423.564169 |
    | 2000-01-03 15:00:00 | 3.728843  3.784112  3.973356  4.484539  5.703394  6.472706  6.767314    6.849336 | 100.020773 | 100.626426  103.805827  115.137162   149.29244  253.139136  354.536564    406.674392 | 422.112028 |  1.568175    1.60449  1.731726  2.097794   3.10243   5.02786  6.086238     6.384763 |  100.0 |   0.020773 | 422.112028 |
    | 2000-01-03 15:30:00 | 3.727392  3.776431  3.946628  4.410619  5.518643  6.416903  6.739398    6.840044 | 100.016892 | 100.543734  103.360453  113.494653  144.025482  235.259589  345.276607    401.509192 | 420.343799 |  1.567227    1.59942  1.713485  2.042839  2.938209  4.829829   5.98502     6.350861 |  100.0 |   0.016892 | 420.343799 |
    | 2000-01-03 16:00:00 |  3.72613  3.769618  3.922611  4.343676  5.351655   6.35786  6.709401     6.82906 | 100.013722 | 100.471842  102.966418  112.030599  139.352593  219.775096  335.706851    396.010059 | 418.259852 |  1.566402    1.59493  1.697169  1.993659  2.793449  4.621152  5.876474     6.310811 |  100.0 |   0.013722 | 418.259852 |
    | 2000-01-03 16:30:00 | 3.725032   3.76358  3.901039  4.283051  5.200543  6.296622  6.677414    6.816383 | 100.011136 | 100.409361  102.617882  110.725155  135.199729  206.307243  326.037781    390.204834 | 415.863287 |  1.565685   1.590955  1.682576  1.949605  2.665455  4.405637  5.760973     6.264626 |  100.0 |   0.011136 | 415.863287 |
    | 2000-01-03 17:00:00 | 3.724079   3.75823  3.881671  4.228152  5.063658  6.234006   6.64359    6.802028 | 100.009028 | 100.355074  102.309667  109.560824  131.503436  194.547442  316.433758    384.132812 | 413.160647 |  1.565062   1.587436  1.669524  1.910108  2.551976  4.186241  5.639115     6.212376 |  100.0 |   0.009028 | 413.160647 |
    | 2000-01-03 17:30:00 |  3.72325  3.753492  3.864291  4.178446  4.939554  6.170646  6.608122    6.786027 | 100.007313 | 100.307921  102.037178  108.522142  128.209102  184.242315  307.024024    377.840418 | 410.162198 |  1.564521   1.584324  1.657851  1.874672  2.451116  3.965234  5.511645     6.154197 |  100.0 |   0.007313 | 410.162198 |
    | 2000-01-03 18:00:00 |  3.72253  3.749298  3.848701  4.133447  4.826955  6.107021  6.571235     6.76843 | 100.005918 | 100.266977   101.79634  107.595423  125.269535  175.182461   297.91374    371.378256 | 406.881898 |  1.564051   1.581571  1.647413  1.842858  2.361273  3.744318  5.379409      6.09029 |  100.0 |   0.005918 | 406.881898 |
    | 2000-01-03 18:30:00 | 3.721905  3.745588  3.834724   4.09272  4.724735  6.043482  6.533177    6.749306 | 100.004785 | 100.231432  101.583536  106.768529  122.643796  167.193755  289.200153    364.799765 | 403.337244 |  1.563643   1.579137   1.63808  1.814283  2.281083  3.524707   5.24333     6.020923 |  100.0 |   0.004785 | 403.337244 |
    | 2000-01-03 19:00:00 | 3.721363  3.742306  3.822198  4.055868  4.631892  5.935524  6.492242    6.728587 | 100.003865 | 100.200584  101.395555  106.030684  120.296249  160.130554  276.737585    357.826502 | 399.521112 |  1.563289   1.576986  1.629737  1.788605  2.209381  3.314814   5.09737     5.945872 |  100.0 |   0.003865 | 399.521112 |
    | 2000-01-03 19:30:00 | 3.720892  3.739406  3.810977  4.022532  4.547534  5.734566   6.44482    6.705945 | 100.003119 | 100.173818   101.22955  105.372305  118.195765  153.870365  256.234201     349.88341 | 395.379733 |  1.562982   1.575086   1.62228  1.765522  2.145167  3.130559  4.928801     5.863979 |  100.0 |   0.003119 | 395.379733 |
    | 2000-01-03 20:00:00 | 3.720483  3.736842   3.80093  3.992385  4.470865  5.552759  6.392579    6.681225 | 100.002515 |   100.1506  101.082996  104.784865  116.315068  148.309613  238.501738    341.305365 | 390.893276 |  1.562715   1.573407  1.615616  1.744768  2.087577  2.968213  4.743755     5.774721 |  100.0 |   0.002515 | 390.893276 |
    | 2000-01-03 20:30:00 | 3.720129  3.734578  3.791939   3.96513  4.401169  5.388038  6.336843    6.654404 | 100.002026 | 100.130463  100.953651  104.260765  114.630188  143.360283  223.094642     332.35839 | 386.066725 |  1.562484   1.571926  1.609664  1.726103  2.035862  2.824691   4.54708     5.678046 |  100.0 |   0.002026 | 386.066725 |
    | 2000-01-03 21:00:00 | 3.719822  3.732579  3.783895    3.9405  4.337806  5.238604  6.278659    6.625549 | 100.001631 | 100.113003  100.839529   103.79323  113.120007  138.947225  209.651136     323.25273 | 380.922673 |  1.562284   1.570618  1.604347  1.709315  1.989373  2.697424  4.342597      5.57424 |  100.0 |   0.001631 | 380.922673 |
    | 2000-01-03 21:30:00 | 3.719556  3.730815  3.776703  3.918248  4.280195   5.10289  6.218853    6.594794 | 100.001312 | 100.097866  100.738868  103.376217  111.765867  135.005978  197.875369    314.154332 | 375.495742 |   1.56211   1.569464    1.5996  1.694214  1.947541  2.584259  4.133295     5.463825 |  100.0 |   0.001312 | 375.495742 |
    | 2000-01-03 22:00:00 | 3.719325  3.729258  3.770275  3.898154  4.227817  4.979515  6.158069    6.562316 | 100.001054 | 100.084746  100.650107  103.004327  110.551251  131.481015  187.523776    305.195264 | 369.828393 |  1.561959   1.568447  1.595362  1.680629  1.909868  2.483382  3.921486     5.347486 |  100.0 |   0.001054 | 369.828393 |
    | 2000-01-03 22:30:00 | 3.719125  3.727886  3.764531  3.880014  4.180197  4.867267  6.096801    6.528327 | 100.000846 | 100.073377  100.571861  102.672741  109.461506  128.324308  178.394553    296.484487 | 363.968023 |  1.561829   1.567549  1.591581  1.668409  1.875915  2.393256  3.708924     5.226015 |  100.0 |   0.000846 | 363.968023 |
    | 2000-01-03 23:00:00 | 3.718951  3.726675  3.759402  3.863645  4.136908  4.765072  6.035409    6.493065 | 100.000679 | 100.063525  100.502905  102.377151  108.483605  125.494153  170.319483    288.124846 | 357.965712 |  1.561716   1.566758  1.588207  1.657418  1.845296   2.31257  3.496875     5.100302 |  100.0 |   0.000679 | 357.965712 |
    | 2000-01-03 23:30:00 | 3.718801  3.725608  3.754823  3.848879  4.097562  4.671975  5.915462    6.454192 | 100.000544 | 100.054991  100.442154  102.113709  107.605949  122.954208  163.157543    274.647633 | 351.441616 |  1.561618   1.566061  1.585198  1.647532  1.817669  2.240204  3.296193     4.962073 |  100.0 |   0.000544 | 351.441616 |
    | 2000-01-04 00:00:00 | 3.718671  3.724669  3.750738  3.835565  4.061806  4.587125   5.72222    6.409018 | 100.000436 | 100.047598  100.388647  101.878973  106.818195  120.672692  156.789858    255.005678 | 343.984873 |  1.561533   1.565448  1.582516  1.638641  1.792731  2.175193  3.119404      4.80191 |  100.0 |   0.000436 | 343.984873 |
    | 2000-01-04 00:30:00 | 3.718558  3.723841  3.747094  3.823565   4.02932  4.509764  5.546962    6.359117 | 100.000349 | 100.041196  100.341533  101.669865  106.111105  118.621725  151.115701    237.948915 | 335.908015 |  1.561459   1.564907  1.580125  1.630647  1.770212  2.116704  2.963104     4.625584 |  100.0 |   0.000349 | 335.908015 |
    | 2000-01-04 01:00:00 |  3.71846  3.723112  3.743844  3.812754  3.999812   4.43921  5.387776    6.305744 | 100.000279 | 100.035651  100.300059  101.483637  105.476421  116.776772  146.049304    223.070652 | 327.461112 |  1.561396   1.564431  1.577994   1.62346  1.749871  2.064014  2.824466     4.437681 |  100.0 |   0.000279 | 327.461112 |
    | 2000-01-04 01:30:00 | 3.718376   3.72247  3.740949  3.803018  3.973016   4.37485  5.242998    6.249899 | 100.000223 | 100.030851  100.263562  101.317827   104.90675  115.116184  141.517288    210.039238 |  318.84344 |   1.56134   1.564012  1.576096     1.617  1.731494  2.016492  2.701126     4.241833 |  100.0 |   0.000223 |  318.84344 |
    | 2000-01-04 02:00:00 | 3.718303  3.721906  3.738369  3.794252  3.948689  4.316133  5.111172     6.19237 | 100.000178 | 100.026695  100.231451  101.170236  104.395469  113.620807  137.456594    198.582347 | 310.213929 |  1.561293   1.563644  1.574407  1.611194  1.714888  1.973587    2.5911     4.040896 |  100.0 |   0.000178 | 310.213929 |
    | 2000-01-04 02:30:00 | 3.718239  3.721409  3.736071  3.786365  3.926611  4.262558  4.991019    6.133775 | 100.000142 | 100.023097  100.203207  101.038899  103.936639  112.273653  133.812801    188.474904 | 301.700992 |  1.561251   1.563319  1.572902  1.605978  1.699882  1.934817  2.492708     3.837089 |  100.0 |   0.000142 | 301.700992 |
    | 2000-01-04 03:00:00 | 3.718185  3.720972  3.734025   3.77927   3.90658  4.213674   4.88141     6.07459 | 100.000113 | 100.019983  100.178371  100.922057  103.524931  111.059619  130.538753    179.529731 | 293.413712 |  1.561216   1.563034  1.571564  1.601293  1.686319  1.899754  2.404525     3.632098 |  100.0 |   0.000113 | 293.413712 |
    | 2000-01-04 03:30:00 | 3.718137  3.720588  3.732205   3.77289  3.888412  4.169072  4.781344    6.015155 |  100.00009 | 100.017287  100.156538   100.81814   103.15556  109.965255  127.593431    171.590254 | 285.465173 |  1.561185   1.562784  1.570373  1.597086  1.674062  1.868024   2.32533     3.427123 |  100.0 |    0.00009 | 285.465173 |
    | 2000-01-04 04:00:00 | 3.718096   3.72025  3.730585  3.767156   3.87194   4.12838  4.689933    5.854782 | 100.000072 | 100.014955  100.137348  100.725744   102.82423  108.978557  124.941023    164.524764 | 268.384578 |  1.561158   1.562563  1.569314  1.593309  1.662984  1.839291   2.25408     3.240176 |  100.0 |   0.000072 | 268.384578 |
    | 2000-01-04 04:30:00 | 3.718061  3.719954  3.729145  3.762005  3.857011  4.091259  4.606384    5.672624 | 100.000057 | 100.012936  100.120487  100.643615  102.527077  108.088792   122.55016     158.22188 | 250.106292 |  1.561135    1.56237  1.568372  1.589918  1.652973  1.813261   2.18987     3.074784 |  100.0 |   0.000057 | 250.106292 |
    | 2000-01-04 05:00:00 |  3.71803  3.719693  3.727864  3.757378  3.843484    4.0574  4.529987    5.506953 | 100.000046 | 100.011189  100.105674  100.570632  102.260631  107.286352  120.393271    152.586938 | 234.154831 |  1.561115     1.5622  1.567535  1.586876  1.643927  1.789668  2.131922     2.927962 |  100.0 |   0.000046 | 234.154831 |
    | 2000-01-04 05:30:00 | 3.718003  3.719464  3.726726  3.753224  3.831233  4.026523  4.460107    5.356054 | 100.000036 | 100.009678  100.092665  100.505796   102.02177  106.562621  118.446052    147.539089 | 220.174858 |  1.561097    1.56205  1.566792  1.584148  1.635753  1.768278  2.079554     2.797217 |  100.0 |   0.000036 | 220.174858 |
    | 2000-01-04 06:00:00 |  3.71798  3.719263  3.725715  3.749496  3.820142   3.99837  4.396168    5.218433 | 100.000029 | 100.008371  100.081241  100.448213  101.807686  105.909865  116.687017    143.008972 | 207.874895 |  1.561082   1.561919  1.566131  1.581701  1.628369  1.748879  2.032175     2.680459 |  100.0 |   0.000029 | 207.874895 |
    | 2000-01-04 06:30:00 |  3.71796  3.719086  3.724817  3.746152  3.810103  3.972707  4.337655    5.092777 | 100.000023 |  100.00724  100.071213  100.397085  101.615856  105.321132  115.097117    138.936818 | 197.014271 |  1.561069   1.561804  1.565544  1.579507    1.6217  1.731283  1.989263     2.575919 |  100.0 |   0.000023 | 197.014271 |
    | 2000-01-04 07:00:00 | 3.717943  3.718932  3.724019  3.743153  3.801021   3.94932  4.284099    4.977931 | 100.000018 | 100.006261  100.062411  100.351703  101.444008  104.790168  113.659428     135.27091 | 187.393014 |  1.561058   1.561703  1.565024  1.577541  1.615677  1.715318  1.950363     2.482099 |  100.0 |   0.000018 | 187.393014 |
    | 2000-01-04 07:30:00 | 3.717928  3.718796  3.723312  3.740464  3.792808  3.928012  4.235077    4.872873 | 100.000014 | 100.005415  100.054688  100.311431  101.290101   104.31134  112.358871    131.966319 | 178.843991 |  1.561048   1.561614  1.564562  1.575779  1.610238  1.700832  1.915069      2.39772 |  100.0 |   0.000014 | 178.843991 |
    | 2000-01-04 08:00:00 | 3.717915  3.718676  3.722684  3.738055  3.785383  3.908605  4.190203    4.776696 | 100.000011 | 100.004683  100.047912  100.275703  101.152295  103.879574  111.181988    128.983871 |  171.22672 |   1.56104   1.561536  1.564152  1.574201  1.605329  1.687688  1.883026     2.321682 |  100.0 |   0.000011 |  171.22672 |
    | 2000-01-04 08:30:00 | 3.717904  3.718572  3.722126  3.735897  3.778672  3.890934  4.149128    4.688593 | 100.000009 | 100.004049  100.041968  100.244016  101.028941  103.490294  110.116743    126.289284 | 164.422493 |  1.561033   1.561468  1.563788  1.572788  1.600899   1.67576  1.853915     2.253043 |  100.0 |   0.000009 | 164.422493 |
    | 2000-01-04 09:00:00 | 3.717894   3.71848  3.721632  3.733964  3.772611  3.874848  4.111531     4.60784 | 100.000007 | 100.003502  100.036756   100.21592  100.918551  103.139371  109.152347    123.852454 | 158.330486 |  1.561026   1.561408  1.563465  1.571524  1.596901  1.664937  1.827455     2.190982 |  100.0 |   0.000007 | 158.330486 |
    | 2000-01-04 09:30:00 | 3.717886  3.718399  3.721194  3.732234  3.767137  3.860211  4.077122    4.533792 | 100.000006 | 100.003028  100.032186  100.191014   100.81979  102.823078  108.279121    121.646866 |  152.86465 |  1.561021   1.561356  1.563179  1.570392  1.593295  1.655116  1.803393      2.13479 |  100.0 |   0.000006 |  152.86465 |
    | 2000-01-04 10:00:00 | 3.717879  3.718329  3.720806  3.730685  3.762195  3.846895  4.045635    4.465865 | 100.000004 | 100.002618  100.028179  100.168943  100.731458   102.53805  107.488361    119.649092 | 147.951205 |  1.561016    1.56131  1.562926  1.569379  1.590044  1.646206  1.781504     2.083846 |  100.0 |   0.000004 | 147.951205 |
    | 2000-01-04 10:30:00 | 3.717873  3.718267  3.720462    3.7293  3.757736  3.834787  4.016825    4.403535 | 100.000004 | 100.002264  100.024668  100.149388  100.652474  102.281247  106.772232    117.838372 | 143.526612 |  1.561012   1.561269  1.562702  1.568474  1.587112  1.638122  1.761585     2.037608 |  100.0 |   0.000004 | 143.526612 |
    | 2000-01-04 11:00:00 | 3.717867  3.718213  3.720157  3.728061  3.753714  3.823779  3.990471    4.346327 | 100.000003 | 100.001958   100.02159  100.132067  100.581869  102.049922  106.123676    116.196263 | 139.535921 |  1.561009   1.561234  1.562503  1.567664  1.584469  1.630789  1.743454     1.995596 |  100.0 |   0.000003 | 139.535921 |
    | 2000-01-04 11:30:00 | 3.717863  3.718165  3.719888  3.726954  3.750087  3.813776  3.966367    4.293812 | 100.000002 | 100.001693  100.018894  100.116729  100.518771  101.841594   105.53632    114.706337 |  135.93142 |  1.561006   1.561203  1.562327   1.56694  1.582088  1.624139  1.726948     1.957391 |  100.0 |   0.000002 |  135.93142 |
    | 2000-01-04 12:00:00 | 3.717859  3.718124  3.719649  3.725964  3.746817  3.804689  3.944328    4.245599 | 100.000002 | 100.001464  100.016533   100.10315  100.462398   101.65402  105.004412     113.35393 |  132.67152 |  1.561003   1.561176  1.562171  1.566294  1.579943  1.618108  1.711919     1.922619 |  100.0 |   0.000002 |  132.67152 |
    | 2000-01-04 12:30:00 | 3.717855  3.718087  3.719437   3.72508  3.743871  3.796437   3.92418    4.201334 | 100.000001 | 100.001265  100.014464  100.091131  100.412046  101.485173  104.522749     112.12592 | 129.719843 |  1.561001   1.561152  1.562033  1.565716  1.578012   1.61264  1.698233      1.89095 |  100.0 |   0.000001 | 129.719843 |
    | 2000-01-04 13:00:00 | 3.717852  3.718056   3.71925   3.72429  3.741218  3.788946  3.905768    4.160691 | 100.000001 | 100.001094  100.012652  100.080496  100.367086  101.333221  104.086627    111.010544 | 127.044452 |  1.560999   1.561131  1.561911    1.5652  1.576273  1.607684   1.68577     1.862089 |  100.0 |   0.000001 | 127.044452 |
    | 2000-01-04 13:30:00 | 3.717849  3.718028  3.719084  3.723584  3.738828  3.782147  3.888945    4.123377 | 100.000001 | 100.000946  100.011066  100.071087   100.32695  101.196507  103.691787    109.997236 | 124.617222 |  1.560997   1.561113  1.561803   1.56474  1.574707  1.603193   1.67442     1.835773 |  100.0 |   0.000001 | 124.617222 |
    | 2000-01-04 14:00:00 | 3.717847  3.718003  3.718938  3.722955  3.736677   3.77598  3.873579    4.089121 | 100.000001 | 100.000818  100.009678  100.062765  100.291131  101.073535  103.334369    109.076488 | 122.413305 |  1.560996   1.561097  1.561707  1.564329  1.573299  1.599123  1.664085     1.811767 |  100.0 |   0.000001 | 122.413305 |
    | 2000-01-04 14:30:00 | 3.717845  3.717982  3.718808  3.722393  3.734741  3.770388  3.859549    4.057675 | 100.000001 | 100.000707  100.008462  100.055407  100.259173  100.962953  103.010877    108.239733 | 120.410686 |  1.560994   1.561083  1.561622  1.563962  1.572032  1.595437  1.654673     1.789859 |  100.0 |   0.000001 | 120.410686 |
    | 2000-01-04 15:00:00 | 3.717844  3.717963  3.718694  3.721892     3.733  3.765318  3.846741    4.028812 |      100.0 | 100.000611  100.007398  100.048901  100.230668  100.863538  102.718143    107.479241 | 118.589801 |  1.560993   1.561071  1.561548  1.563635  1.570893  1.592099  1.646103      1.76986 |  100.0 |        0.0 | 118.589801 |
    | 2000-01-04 15:30:00 | 3.717842  3.717947  3.718593  3.721445  3.731434  3.760724  3.835054    4.002325 |      100.0 | 100.000529  100.006468  100.043151  100.205249  100.774187   102.45329    106.788023 | 116.933223 |  1.560992    1.56106  1.561482  1.563343  1.569869  1.589076    1.6383     1.751599 |  100.0 |        0.0 | 116.933223 |
    | 2000-01-04 16:00:00 | 3.717841  3.717932  3.718503  3.721046  3.730027  3.756562  3.824393    3.978023 |      100.0 | 100.000457  100.005653   100.03807  100.182588  100.693903  102.213712    106.159759 | 115.425389 |  1.560991   1.561051  1.561423  1.563083  1.568949   1.58634  1.631197     1.734921 |  100.0 |        0.0 | 115.425389 |
    | 2000-01-04 16:30:00 |  3.71784   3.71792  3.718424  3.720691  3.728762  3.752794   3.81467    3.955729 |      100.0 | 100.000395  100.004941  100.033581  100.162391  100.621785  101.997041    105.588725 | 114.052366 |  1.560991   1.561043  1.561372  1.562851  1.568122  1.583865  1.624732     1.719686 |  100.0 |        0.0 | 114.052366 |
    | 2000-01-04 17:00:00 | 3.717839  3.717909  3.718354  3.720374  3.727625  3.749382  3.805806    3.935283 |      100.0 | 100.000342  100.004318  100.029616  100.144395   100.55702  101.801132    105.069729 | 112.801656 |   1.56099   1.561036  1.561326  1.562644  1.567379  1.581626  1.618849     1.705769 |  100.0 |        0.0 | 112.801656 |
    | 2000-01-04 17:30:00 | 3.717838  3.717899  3.718293  3.720092  3.726604  3.746294  3.797729    3.916536 |      100.0 | 100.000295  100.003773  100.026114  100.128364  100.498874  101.624035    104.598064 | 111.662027 |   1.56099    1.56103  1.561286   1.56246  1.566712    1.5796  1.613496     1.693054 |  100.0 |        0.0 | 111.662027 |
    | 2000-01-04 18:00:00 | 3.717837  3.717891  3.718238  3.719841  3.725688  3.743501   3.79037     3.89935 |      100.0 | 100.000255  100.003296  100.023022  100.114088  100.446685  101.463983    104.169453 |  110.62336 |  1.560989   1.561024  1.561251  1.562296  1.566113  1.577769  1.608626     1.681437 |  100.0 |        0.0 |  110.62336 |
    | 2000-01-04 18:30:00 | 3.717837  3.717883   3.71819  3.719617  3.724865  3.740976  3.783668    3.883602 |      100.0 | 100.000221  100.002879  100.020293  100.101376  100.399855  101.319369    103.780013 |  109.67653 |  1.560989   1.561019  1.561219   1.56215  1.565576  1.576114  1.604197     1.670823 |  100.0 |        0.0 |  109.67653 |
    | 2000-01-04 19:00:00 | 3.717836  3.717877  3.718148  3.719418  3.724127  3.738692  3.777566    3.869173 |      100.0 | 100.000191  100.002515  100.017884  100.090061  100.357846  101.188737    103.426211 | 108.813291 |  1.560988   1.561015  1.561192   1.56202  1.565094  1.574618  1.600169     1.661126 |  100.0 |        0.0 | 108.813291 |
    | 2000-01-04 19:30:00 | 3.717836  3.717871   3.71811  3.719241  3.723465  3.736628  3.772013    3.855958 |      100.0 | 100.000165  100.002197  100.015758  100.079992  100.320171  101.070766    103.104836 |  108.02618 |  1.560988   1.561011  1.561167  1.561905  1.564662  1.573267  1.596508     1.652268 |  100.0 |        0.0 |  108.02618 |
    | 2000-01-04 20:00:00 | 3.717835  3.717867  3.718077  3.719083  3.722872  3.734764  3.766961    3.843858 |      100.0 | 100.000143  100.001918  100.013883  100.071033  100.286393  100.964256    102.812965 | 107.308437 |  1.560988   1.561008  1.561146  1.561802  1.564274  1.572047   1.59318     1.644176 |  100.0 |        0.0 | 107.308437 |
    | 2000-01-04 20:30:00 | 3.717835  3.717862  3.718048  3.718943  3.722339   3.73308  3.762367    3.832782 |      100.0 | 100.000123  100.001675  100.012229  100.063065  100.256116  100.868119    102.547938 | 106.653928 |  1.560988   1.561005  1.561127   1.56171  1.563927  1.570946  1.590156     1.636785 |  100.0 |        0.0 | 106.653928 |
    | 2000-01-04 21:00:00 | 3.717835  3.717859  3.718022  3.718818  3.721862   3.73156  3.758189    3.822647 |      100.0 | 100.000107  100.001463   100.01077  100.055978  100.228985  100.781367    102.307333 | 106.057084 |  1.560987   1.561003   1.56111  1.561629  1.563615  1.569951   1.58741     1.630035 |  100.0 |        0.0 | 106.057084 |
    | 2000-01-04 21:30:00 | 3.717834  3.717855     3.718  3.718707  3.721435  3.730188  3.754393    3.813376 |      100.0 | 100.000092  100.001277  100.009484  100.049678  100.204679  100.703105    102.088945 | 105.512841 |  1.560987   1.561001  1.561095  1.561557  1.563336  1.569054  1.584915     1.623872 |  100.0 |        0.0 | 105.512841 |
    | 2000-01-04 22:00:00 | 3.717834  3.717852   3.71798  3.718609  3.721052   3.72895  3.750943    3.804897 |      100.0 |  100.00008  100.001115   100.00835  100.044078  100.182909   100.63252    101.890766 | 105.016591 |  1.560987   1.560999  1.561082  1.561492  1.563086  1.568245   1.58265     1.618246 |  100.0 |        0.0 | 105.016591 |
    | 2000-01-04 22:30:00 | 3.717834   3.71785  3.717962  3.718521  3.720709  3.727833   3.74781    3.797147 |      100.0 | 100.000069  100.000973   100.00735  100.039101  100.163417  100.568878    101.710966 |  104.56414 |  1.560987   1.560997  1.561071  1.561435  1.562863  1.567515  1.580595      1.61311 |  100.0 |        0.0 |  104.56414 |
    | 2000-01-04 23:00:00 | 3.717834  3.717848  3.717947  3.718443  3.720402  3.726826  3.744966    3.790064 |      100.0 |  100.00006  100.000849   100.00647  100.034679  100.145968   100.51151    101.547879 | 104.151664 |  1.560987   1.560996   1.56106  1.561384  1.562662  1.566857  1.578729     1.608423 |  100.0 |        0.0 | 104.151664 |
    | 2000-01-04 23:30:00 | 3.717834  3.717846  3.717933  3.718374  3.720127  3.725919  3.742384    3.783593 |      100.0 | 100.000051  100.000741  100.005693  100.030752  100.130353  100.459812    101.399986 | 103.775679 |  1.560987   1.560995  1.561052  1.561339  1.562483  1.566264  1.577037     1.604147 |  100.0 |        0.0 | 103.775679 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _kinw_impl_euler_short_segments:

short segments
______________

Short channel segments do neither result in unstable results nor increased computation
time:

.. integration-test::

    >>> length(8.0)
    >>> model.prepare_states(100.0)
    >>> conditions = test("kinw_impl_euler_short_segments", get_conditions="2000-01-01")
    |                date |                                                                       waterdepth |     inflow |                                                                         internalflow |    outflow |                                                                       watervolume | input1 |     input2 |     output |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000003 | 100.000002  100.000002  100.000001  100.000001       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000029 |  100.00002  100.000014  100.000009  100.000006  100.000004  100.000003    100.000002 | 100.000001 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000029 | 100.000001 |
    | 2000-01-01 04:00:00 | 3.717835  3.717834  3.717834  3.717834  3.717833  3.717833  3.717833    3.717833 | 100.000175 | 100.000123  100.000086   100.00006  100.000042  100.000029   100.00002    100.000014 |  100.00001 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000175 |  100.00001 |
    | 2000-01-01 04:30:00 | 3.717843   3.71784  3.717838  3.717836  3.717835  3.717835  3.717834    3.717834 | 100.000826 | 100.000588  100.000418  100.000297  100.000211   100.00015  100.000106    100.000075 | 100.000053 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000826 | 100.000053 |
    | 2000-01-01 05:00:00 | 3.717873  3.717862  3.717854  3.717848  3.717844  3.717841  3.717838    3.717837 | 100.003194 | 100.002313  100.001673  100.001208  100.000871  100.000627  100.000451    100.000324 | 100.000232 | 0.124881   0.12488   0.12488   0.12488  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.003194 | 100.000232 |
    | 2000-01-01 05:30:00 | 3.717969  3.717933  3.717906  3.717887  3.717872  3.717862  3.717854    3.717848 | 100.010515 | 100.007743  100.005692  100.004177   100.00306  100.002238  100.001634    100.001191 | 100.000867 | 0.124886  0.124884  0.124883  0.124882  0.124881   0.12488   0.12488      0.12488 |  100.0 |   0.010515 | 100.000867 |
    | 2000-01-01 06:00:00 | 3.718232  3.718131  3.718055  3.717998  3.717956  3.717924    3.7179    3.717883 | 100.030315 | 100.022688  100.016944   100.01263  100.009395  100.006977  100.005171    100.003826 | 100.002826 |   0.1249  0.124894   0.12489  0.124888  0.124885  0.124884  0.124882     0.124881 |  100.0 |   0.030315 | 100.002826 |
    | 2000-01-01 06:30:00 | 3.718879  3.718626  3.718433  3.718286  3.718175   3.71809  3.718026    3.717977 | 100.078179 | 100.059428  100.045072  100.034108  100.025757  100.019411  100.014599    100.010958 |  100.00821 | 0.124933   0.12492   0.12491  0.124903  0.124897  0.124892  0.124889     0.124886 |  100.0 |   0.078179 |  100.00821 |
    | 2000-01-01 07:00:00 | 3.720323   3.71975  3.719305  3.718961  3.718695  3.718491  3.718334    3.718213 | 100.183351 | 100.141479  100.108903  100.083629  100.064073  100.048981  100.037362    100.028439 | 100.021603 | 0.125009  0.124979  0.124956  0.124938  0.124924  0.124913  0.124905     0.124899 |  100.0 |   0.183351 | 100.021603 |
    | 2000-01-01 07:30:00 | 3.723289  3.722094  3.721152  3.720412  3.719832  3.719379  3.719026    3.718751 | 100.396214 | 100.310161  100.242167  100.188603  100.146527  100.113566  100.087817    100.067752 | 100.052157 | 0.125164  0.125101  0.125052  0.125013  0.124983   0.12496  0.124941     0.124927 |  100.0 |   0.396214 | 100.052157 |
    | 2000-01-01 08:00:00 | 3.728955   3.72664  3.724787   3.72331  3.722135  3.721204  3.720468    3.719888 | 100.797305 | 100.632807  100.500875  100.395397  100.311325  100.244514  100.191572    100.149736 | 100.116765 |  0.12546  0.125339  0.125242  0.125165  0.125103  0.125055  0.125016     0.124986 |  100.0 |   0.797305 | 100.116765 |
    | 2000-01-01 08:30:00 |   3.7391    3.7349  3.731489   3.72873  3.726504  3.724715  3.723281    3.722135 | 101.507032 | 101.212046  100.972005  100.777334  100.619972  100.493167  100.391295    100.309693 | 100.244514 | 0.125991  0.125771  0.125592  0.125448  0.125332  0.125238  0.125163     0.125103 |  100.0 |   1.507032 | 100.244514 |
    | 2000-01-01 09:00:00 |  3.75624  3.749061  3.743145  3.738289  3.734316  3.731078  3.728447    3.726317 | 102.694853 | 102.195156  101.782742  101.443596   101.16567  100.938673  100.753867    100.603876 | 100.482506 |  0.12689  0.126513  0.126203  0.125948   0.12574  0.125571  0.125433     0.125322 |  100.0 |   2.694853 | 100.482506 |
    | 2000-01-01 09:30:00 | 3.783706  3.772089  3.762371  3.754276   3.74756   3.74201   3.73744     3.73369 | 104.586356 | 103.782218  103.109221  102.548183  102.082224  101.696611   101.37858    101.117148 | 100.902922 | 0.128338  0.127725  0.127213  0.126787  0.126434  0.126143  0.125904     0.125708 |  100.0 |   4.586356 | 100.902922 |
    | 2000-01-01 10:00:00 | 3.825614  3.807752  3.792577  3.779747  3.768949    3.7599  3.752348    3.746069 | 107.466641 | 106.231601  105.183525  104.297905  103.552585  102.927754  102.405852    101.971454 | 101.611102 | 0.130561  0.129611  0.128807  0.128129  0.127559  0.127083  0.126686     0.126356 |  100.0 |   7.466641 | 101.611102 |
    | 2000-01-01 10:30:00 | 3.886682  3.860521  3.837936  3.818544   3.80198  3.787899  3.775984    3.765943 | 111.678559 | 109.861374  108.297606    106.9582  105.816032  104.846134  104.025791    103.334561 | 102.754219 | 0.133832  0.132426  0.131218  0.130185  0.129305  0.128559   0.12793     0.127401 |  100.0 |  11.678559 | 102.754219 |
    | 2000-01-01 11:00:00 | 3.971869  3.935302  3.903201  3.875191  3.850893   3.82993  3.811935    3.796562 | 117.614836 | 115.045396  112.802834   110.85556  109.172863  107.725442  106.485786    105.428426 | 104.530061 | 0.138457  0.136463  0.134723  0.133213   0.13191  0.130791  0.129833     0.129018 |  100.0 |  17.614836 | 104.530061 |
    | 2000-01-01 11:30:00 | 4.085835  4.036988  3.993353  3.954635  3.920502  3.890591  3.864529    3.841945 | 125.703736 | 122.203151  119.103918  116.375176  113.985265   111.90255  110.096121    108.536337 | 107.195225 | 0.144758  0.142041  0.139635  0.137515  0.135659  0.134042  0.132641     0.131432 |  100.0 |  25.703736 | 107.195225 |
    | 2000-01-01 12:00:00 | 4.232334  4.169874   4.11308  4.061807  4.015837  3.974894  3.938659    3.906781 | 136.388647 | 131.782829  127.646049  123.952305  120.672726  117.776564  115.232122    113.007591 | 111.071741 | 0.153048  0.149487  0.146283  0.143418  0.140872  0.138622  0.136645     0.134916 |  100.0 |  36.388647 | 111.071741 |
    | 2000-01-01 12:30:00 |  4.41363  4.337042  4.266158  4.201035  4.141633   4.08783   4.03943    3.996173 | 150.102726 | 144.237813  138.894529  134.055882  129.700049  125.801255  122.330744    119.257751 | 116.550419 | 0.163605  0.159105  0.154993  0.151259   0.14789  0.144869  0.142176      0.13979 |  100.0 |  50.102726 | 116.550419 |
    | 2000-01-01 13:00:00 | 4.630122  4.539843   4.45485  4.375408  4.301689   4.23377  4.171637    4.115189 | 167.240349 | 159.997651  153.306983  147.163491  141.556198  136.468501  131.878868    127.761705 | 124.088292 | 0.176642  0.171148  0.166051  0.161352  0.157048  0.153131  0.149587     0.146402 |  100.0 |  67.240349 | 124.088292 |
    | 2000-01-01 13:30:00 |  4.88024   4.77763  4.679481  4.586228  4.498238  4.415795  4.339096    4.268242 | 188.127569 | 179.435689   171.29973   163.72821  156.723341  150.280941  144.390622    139.036249 | 134.196626 | 0.192287  0.185793   0.17968  0.173961  0.168644  0.163733  0.159225     0.155113 |  100.0 |  88.127569 | 134.196626 |
    | 2000-01-01 14:00:00 | 5.160643  5.047796  4.938301  4.832695  4.731481  4.635112  4.543977    4.458384 | 212.993962 | 202.836469   193.21149  184.140005  175.637647  167.714059  160.372486    153.609657 | 147.415957 | 0.210571  0.203118  0.196009  0.189265  0.182907  0.176948  0.171398     0.166262 |  100.0 | 112.993962 | 147.415957 |
    | 2000-01-01 14:30:00 | 5.466609  5.346076   5.22766  5.111919  4.999416  4.890702  4.786298    4.686678 | 241.948195 | 230.366301  219.268735  208.686279  198.646141  189.170966   180.27797    171.978275 | 164.276474 | 0.231418  0.223094  0.215057  0.207337  0.199962  0.192955  0.186338     0.180125 |  100.0 | 141.948195 | 164.276474 |
    | 2000-01-01 15:00:00 | 5.792529  5.667027  5.542427  5.419243  5.298021  5.179328  5.063739    4.951821 |  274.95935 | 262.050081  249.556972  237.517046  225.965793   214.93614  204.457401    194.554268 | 185.245927 | 0.254655  0.245581  0.236729   0.22813  0.219815  0.211817  0.204163      0.19688 |  100.0 |  174.95935 | 185.245927 |
    | 2000-01-01 15:30:00 | 6.069357  5.970188  5.851258  5.730217  5.608138  5.485986  5.364647    5.244949 | 311.845554 | 292.697555  280.371352  268.023551  255.801017  243.821738  232.181247    220.957061 | 210.211675 | 0.289121  0.267769  0.258955   0.25013  0.241378   0.23277  0.224367     0.216222 |  100.0 | 211.845554 | 210.211675 |
    | 2000-01-01 16:00:00 | 6.267803  6.127382  6.039297  5.959759  5.861625  5.756205   5.64561    5.531633 | 352.270855 | 321.581059  300.790067  288.641675  279.275084    269.0865  258.396183    247.461836 | 236.490921 | 0.344363  0.305192  0.280822   0.26699  0.259718  0.252013  0.244049     0.235969 |  100.0 | 252.270855 | 236.490921 |
    | 2000-01-01 16:30:00 | 6.500444  6.318761  6.178148  6.080117  6.017555  5.947324  5.862878    5.769754 | 395.750601 | 359.215195   329.50247  308.121305  294.173286  285.777182  277.971294    269.215129 | 259.755495 | 0.410127  0.358675  0.319308  0.292096  0.274831  0.266063   0.25981     0.252997 |  100.0 | 295.750601 | 259.755495 |
    | 2000-01-01 17:00:00 | 6.745197  6.545286  6.367901  6.224542  6.118429  6.048633  6.002537    5.937806 | 441.664962 | 402.578434  366.883004  337.317411   315.00804   299.52052  289.890263     283.84146 | 276.975844 | 0.480482  0.422927  0.372526  0.332253  0.302708  0.283398  0.270698     0.265355 |  100.0 | 341.664962 | 276.975844 |
    | 2000-01-01 17:30:00 | 6.990122  6.788255  6.592421   6.41606  6.269146  6.157224  6.076722     6.02694 | 489.279607 | 449.499272  410.578919   375.07944  345.138284  321.787452  305.072873    293.706338 | 287.005783 | 0.552087  0.492984  0.436425  0.386147   0.34474  0.313484  0.291157     0.277416 |  100.0 | 389.279607 | 287.005783 |
    | 2000-01-01 18:00:00 | 7.228688  7.034252  6.834829  6.640988  6.464084  6.313922  6.194639    6.108181 |  537.77215 | 498.383862  458.310268  419.353533  383.668619  353.092386  328.742257    310.549393 |  298.07655 | 0.622986  0.565116  0.506547   0.45038  0.399777  0.357314  0.323905     0.299867 |  100.0 |  437.77215 |  298.07655 |
    | 2000-01-01 18:30:00 | 7.457007  7.274685  7.081456   6.88376  6.690763  6.512822  6.358411    6.233676 | 586.262607 | 547.973961    508.1571  467.852463  428.705792  392.619975  361.318949    335.795049 | 316.383894 | 0.691906  0.636786  0.579096  0.520844  0.464731  0.413656  0.369848     0.334807 |  100.0 | 486.262607 | 316.383894 |
    | 2000-01-01 19:00:00 | 7.672428  7.504261  7.322738  7.130859  6.934453   6.74182  6.562111    6.404009 | 633.846018 | 597.189089  558.570589  518.485237  477.968332  438.537541  401.955548    369.792925 | 343.166464 | 0.757888    0.7063  0.651249  0.593775  0.535707  0.479504   0.42774     0.382735 |  100.0 | 533.846018 | 343.166464 |
    | 2000-01-01 19:30:00 | 7.872909  7.719645  7.552519  7.372303  7.181882  6.986613  6.793926    6.612492 | 679.625318 | 645.047616  608.284019  569.508266  529.263364  488.552714  448.803217    411.640668 | 378.611636 | 0.820128  0.772474  0.721046  0.666216  0.608986  0.551053  0.494633     0.442187 |  100.0 | 579.625318 | 378.611636 |
    | 2000-01-01 20:00:00 | 8.055704  7.918662  7.766996   7.60147  7.422942  7.234135  7.039913    6.847148 | 722.742733 |  690.67093  656.242828  619.520114  580.722383  540.405267  499.535533    459.448162 | 421.695186 | 0.877857  0.834445  0.787147  0.736051  0.681557  0.624618   0.56679     0.510142 |  100.0 | 622.742733 | 421.695186 |
    | 2000-01-01 20:30:00 | 8.214291  8.095245  7.962322  7.813359  7.650259  7.473978   7.28708    7.094069 | 762.408181 | 732.276721   700.90763  667.019293  630.627696  592.017647  551.766655    510.809707 | 470.422691 | 0.932094  0.890909  0.848146  0.801556  0.751055   0.69707  0.640513     0.582839 |  100.0 | 662.408181 | 470.422691 |
    | 2000-01-01 21:00:00 | 8.351635  8.245049  8.130596  8.002961  7.857853  7.698069  7.524692    7.340166 | 797.923464 | 769.570179  740.522406  710.134854  677.147491  641.385776  603.200513    563.186977 | 522.260643 |  0.98313  0.943195  0.902843  0.860934  0.815426  0.765803  0.712537     0.656506 |  100.0 | 697.923464 | 522.260643 |
    | 2000-01-01 21:30:00 | 8.469715   8.37377  8.271983  8.162241  8.039944  7.900151    7.7444     7.57455 | 828.701418 | 802.649281  775.697705  747.792698  718.456849  686.614638   651.70132    614.144521 | 574.540561 | 1.030023  0.991708  0.953072  0.913738  0.872742  0.828647  0.780139     0.727793 |  100.0 | 728.701418 | 574.540561 |
    | 2000-01-01 22:00:00 | 8.568892  8.483884  8.392975  8.295819  8.190612  8.073773  7.939888    7.788809 |  854.27953 | 831.201037  806.684513  781.041202   754.26573  725.968992  695.338151    661.470579 |  624.73308 | 1.071565  1.035838   0.99923  0.961934  0.923676  0.883783  0.841101     0.793921 |  100.0 |  754.27953 |  624.73308 |
    | 2000-01-01 22:30:00 | 8.649169  8.576052  8.496051  8.409806  8.317029  8.216309  8.104737    7.977063 | 874.327871 | 854.847071  833.290318  810.161171  785.745016  760.056833  732.815991    703.378148 | 670.678491 |  1.10663   1.07464  1.040863  1.005883  0.969915  0.932816  0.894089      0.85278 |  100.0 | 774.327871 | 670.678491 |
    | 2000-01-01 23:00:00 | 8.710573  8.650358  8.581969   8.50654  8.424628  8.336047  8.239678    8.133099 | 888.651531 | 873.267279  855.200984  835.019739   813.16687  789.903613  765.275164    739.078392 | 710.790932 | 1.134322  1.107159  1.077189  1.045218  1.011789  0.977148  0.941243     0.903698 |  100.0 | 788.651531 | 710.790932 |
    | 2000-01-01 23:30:00 | 8.753259  8.706776   8.65085  8.586787  8.515577   8.43774  8.353175    8.260979 | 897.187983 | 886.246447   872.11964  855.347632  836.430047  815.763204  793.595421    769.995401 | 744.816703 | 1.154017  1.132588  1.107379   1.07927  1.048989   1.01705  0.983723     0.949019 |  100.0 | 797.187983 | 744.816703 |
    | 2000-01-02 00:00:00 | 8.777526  8.745344  8.702641  8.650681   8.59062  8.523349  8.449364    8.368635 |      900.0 | 893.689553  883.828741  870.871176   855.29717  837.553035   818.00047    796.878585 | 774.273148 | 1.165375  1.150337  1.130702  1.107303  1.080928  1.052244  1.021743     0.989709 |  100.0 |      800.0 | 774.273148 |
    | 2000-01-02 00:30:00 | 8.783802   8.76622  8.737301  8.698143  8.649883  8.593561  8.529997    8.459671 | 897.264896 | 895.622205  890.215808  881.377416  869.514921  855.059606   838.41571    819.917902 | 799.797396 | 1.168332  1.160068  1.146611  1.128656  1.106948  1.082203  1.055039     1.025926 |  100.0 | 797.264896 | 799.797396 |
    | 2000-01-02 01:00:00 | 8.772627  8.769688  8.754888  8.729081   8.69327  8.648488  8.595689    8.535634 | 889.260928 | 892.182981  891.280188  886.744566  878.877292  868.047023   854.64461    839.040196 | 821.546261 | 1.163073  1.161694  1.154775  1.142817  1.126442  1.106328  1.083127     1.057415 |  100.0 | 789.260928 | 821.546261 |
    | 2000-01-02 01:30:00 |  8.74463  8.756145  8.755569  8.743471  8.720644  8.688012  8.646526    8.597066 | 876.351729 | 883.611019  887.129026  886.952789  883.257463  876.316652  866.465374    854.060686 | 839.444362 | 1.150006  1.155361  1.155093  1.149468  1.138936   1.12406  1.105455     1.083724 |  100.0 | 776.351729 | 839.444362 |
    | 2000-01-02 02:00:00 | 8.700518  8.726084  8.739618  8.741383  8.731916  8.711957  8.682363    8.644017 | 858.969595 | 870.230825  877.966878  882.083134  882.620812  879.738761  873.685983    864.768547 | 853.314854 | 1.129736  1.141436  1.147683    1.1485  1.144124  1.134955  1.121506     1.104341 |  100.0 | 758.969595 | 853.314854 |
    | 2000-01-02 02:30:00 | 8.641058  8.680074  8.707403  8.722983  8.727071  8.720175  8.702991    8.676317 |   837.5984 | 852.435625  864.081584  872.309113  877.025906   878.26651  876.174486    870.977022 | 862.955129 | 1.103029  1.120473  1.132874   1.14001   1.14189  1.138721  1.130862     1.118781 |  100.0 |   737.5984 | 862.955129 |
    | 2000-01-02 03:00:00 | 8.567071  8.618745  8.659359  8.688519  8.706175  8.712576  8.708206    8.693719 | 812.756821 | 830.670314  845.828175   857.88381  866.617677  871.938227  873.873074    872.551724 | 868.182072 | 1.070784  1.093189  1.111174  1.124289  1.132313  1.135238   1.13324     1.126646 |  100.0 | 712.756821 | 868.182072 |
    | 2000-01-02 03:30:00 | 8.479431  8.542774  8.595976  8.638296   8.66936  8.689128  8.697842    8.695963 | 784.982427 | 805.414769  823.612088   839.12448  851.615608  860.871697   866.80094    869.424129 | 868.858026 | 1.034006  1.060434  1.083251  1.101805  1.115653  1.124565  1.128519     1.127665 |  100.0 | 684.982427 | 868.858026 |
    | 2000-01-02 04:00:00 | 8.379064  8.452877  8.517777  8.572656  8.616797  8.649848  8.671775    8.682812 | 754.817086 | 777.168153    797.8726  816.396037  832.298881  845.253064  855.049139    861.594412 | 864.903221 | 0.993774  1.023166  1.049909   1.07318  1.092335  1.106932  1.116737     1.121709 |  100.0 | 654.817086 | 864.903221 |
    | 2000-01-02 04:30:00 | 8.266965  8.349812  8.425306  8.491956  8.548675  8.594768  8.629899    8.654041 |  722.79402 | 746.434652  769.067139  790.094407  808.989893  825.322173   838.76972    849.126303 | 856.298024 | 0.951221  0.982428   1.01206  1.039168  1.062937  1.082727  1.098095       1.1088 |  100.0 |  622.79402 | 856.298024 |
    | 2000-01-02 05:00:00 | 8.144221  8.234385  8.319123  8.396544  8.465162  8.523905  8.572097    8.609418 | 689.426711 | 713.710794  737.656819  760.630336  782.037007   801.35555  818.160635    832.135777 | 843.077138 |  0.90751  0.939325  0.970708  1.000636  1.028163  1.052478   1.07294     1.089106 |  100.0 | 589.426711 | 843.077138 |
    | 2000-01-02 05:30:00 | 8.012072  8.107477  8.199803  8.286743  8.366377  8.437214  8.498194    8.548657 | 655.199767 | 679.473334  704.092409  728.413294  751.796599  773.647432  793.447155    810.774565 | 825.316947 | 0.863817   0.89501   0.92693  0.958546  0.988832  1.016838  1.041751     1.062929 |  100.0 | 555.199767 | 825.316947 |
    | 2000-01-02 06:00:00 | 7.872396  7.971641  8.069163  8.163514  8.252728  8.334745  8.408005     8.47142 | 620.561767 | 644.922706  669.331323  694.145503  718.793005  742.590513  764.917122    785.240656 | 803.134099 | 0.819968  0.851075  0.882265  0.914181  0.945996   0.97665  1.005168     1.030721 |  100.0 | 520.561767 | 803.134099 |
    | 2000-01-02 06:30:00 | 7.729493  7.831454   7.93131  8.030215  8.126066  8.217348  8.301742    8.377524 | 585.920022 | 610.611924  634.991302  659.355129  684.117243  708.948528  733.093713    755.879793 | 776.740108 | 0.775522  0.807192   0.83841  0.869609    0.9013  0.933189  0.964153     0.993172 |  100.0 | 485.920022 | 776.740108 |
    | 2000-01-02 07:00:00 | 7.583593  7.689227  7.791494  7.891625  7.990969  8.087875  8.180875    8.267506 |  551.63713 | 576.612998  601.123927  625.376345  649.615007  674.139591  698.992975    723.385307 | 746.580929 | 0.730566  0.763072  0.794755  0.825979  0.857156  0.888453  0.920247      0.95142 |  100.0 |  451.63713 | 746.580929 |
    | 2000-01-02 07:30:00 | 7.435371  7.544974  7.650386  7.752623  7.852796  7.951737  8.049495    8.143682 | 518.029154 | 543.160046  567.790635  592.047085    616.0978  640.158106  664.398237    689.071208 | 713.569016 |  0.68533  0.718737  0.751094  0.782688  0.813847   0.84482  0.875836     0.907324 |  100.0 | 418.029154 | 713.569016 |
    | 2000-01-02 08:00:00 | 7.285648  7.399183  7.507998  7.612936  7.714907  7.814745  7.913215     8.01115 | 485.365238 | 510.502931  535.161259  559.413365  583.366182   607.16561  630.961473    654.904666 | 679.237761 | 0.640082  0.674352   0.70744  0.739573  0.771008  0.801988  0.832738     0.863525 |  100.0 | 385.365238 | 679.237761 |
    | 2000-01-02 08:30:00 | 7.135237  7.252532  7.364737  7.472593  7.576866  7.678304  7.777629    7.875594 |  453.86844 | 478.870942  503.436343  527.609833  551.456769  575.070837  598.563757     622.05841 |  645.70172 | 0.595078  0.630135  0.663928  0.696649  0.728503  0.759701  0.790448     0.820967 |  100.0 |  353.86844 |  645.70172 |
    | 2000-01-02 09:00:00 |  6.98488  7.105742  7.221192  7.331954   7.43871  7.542125  7.642851    7.741548 | 423.717566 | 448.459689  472.809035  496.801669  520.479583  543.901283  567.142764    590.295012 | 613.467848 | 0.550542  0.586306  0.620741  0.654028  0.686344  0.717866  0.748774     0.779256 |  100.0 | 323.717566 | 613.467848 |
    | 2000-01-02 09:30:00 |  6.83522    6.9595  7.078011  7.191542  7.300762  7.406284  7.508695    7.608578 | 395.049792 | 419.427759  443.448243  467.152047  490.572304  513.747065  536.725649    559.570643 | 582.360423 | 0.506662  0.543069  0.578074  0.611872  0.644629  0.676505  0.707653     0.738234 |  100.0 | 295.049792 | 582.360423 |
    | 2000-01-02 10:00:00 | 6.686777  6.814419  6.935827  7.051947  7.163498  7.271088  7.375266    7.476557 | 367.963873 | 391.897796  415.492815  438.806037  461.873023   484.72314  507.389007    529.911709 |  552.34433 |  0.46358  0.500598   0.53611  0.570351  0.603499  0.635706  0.667112     0.697855 |  100.0 | 267.963873 |  552.34433 |
    | 2000-01-02 10:30:00 | 6.539927  6.671019  6.795227  6.913756  7.027455  7.136976  7.242863    7.345609 | 342.523746 | 365.959972  389.051585  411.884408  434.505973  456.946215  479.229842    501.383934 | 523.442902 | 0.421395  0.459033  0.495011  0.529632  0.563107  0.595595  0.627234     0.658149 |  100.0 | 242.523746 | 523.442902 |
    | 2000-01-02 11:00:00 | 6.394869  6.529722   6.65673  6.777529  6.893183  7.004444  7.111894    7.216021 | 318.762374 | 341.677475  364.207134  386.483524    408.5759  430.522416  452.347032    474.069524 | 495.711914 | 0.380148   0.41848  0.454914  0.489866  0.523603  0.556311  0.588134     0.619193 |  100.0 | 218.762374 | 495.711914 |
    | 2000-01-02 11:30:00 | 6.251554  6.390843  6.520799  6.643789  6.761218  6.874009  6.982824    7.088176 |  296.68568 | 319.095574  341.023483  362.679943  384.168495  405.542647  426.831325    448.052404 | 469.220834 |  0.33981  0.379009  0.415932  0.451187   0.48513  0.517992  0.549936      0.58109 |  100.0 |  196.68568 | 469.220834 |
    | 2000-01-02 12:00:00 | 6.109508  6.254621  6.387875  6.513046  6.632091  6.746193  6.856143    6.962512 | 276.276442 | 298.262928  319.563125  340.542006  361.357024  382.084386  402.762316    423.410582 | 444.040994 | 0.300235  0.340669   0.37817   0.41372   0.44782  0.480771  0.512769     0.543955 |  100.0 | 176.276442 | 444.040994 |
    | 2000-01-02 12:30:00 | 5.937119  6.113256  6.254819  6.384118  6.505501  6.621104   6.73214    6.839381 | 257.498082 | 276.904123  298.790434  319.593438  339.933357  360.073466  380.134618    400.173715 | 420.217867 | 0.265304  0.301274  0.340725  0.377108  0.411568  0.444661  0.476699     0.507875 |  100.0 | 157.498082 | 420.217867 |
    | 2000-01-02 13:00:00 | 5.679326  5.894708  6.098375  6.246124   6.37621  6.496253  6.609715    6.718393 | 240.298231 |  250.76495   272.49561  296.703895  318.269392   338.65552  358.504972    378.121418 | 397.652836 | 0.246464  0.262159   0.29715   0.33829  0.374873  0.408932  0.441389     0.472719 |  100.0 | 140.298231 | 397.652836 |
    | 2000-01-02 13:30:00 | 5.486275  5.607634  5.830051  6.070995  6.229746  6.363803  6.484594    6.597256 | 224.612057 | 232.208393  243.772958  265.857144  292.921473  315.791043  336.659273    356.535617 | 375.928077 |  0.23279  0.241342  0.257398  0.289574  0.333708  0.371369  0.405612     0.437813 |  100.0 | 124.612057 | 375.928077 |
    | 2000-01-02 14:00:00 | 5.318678  5.405206  5.532272  5.749245  6.033839  6.206954  6.347194     6.47036 | 210.365288 | 216.792155  224.671555  236.551565  257.699571  287.916665  312.377014    334.003664 | 354.143131 | 0.221222  0.227159  0.236014  0.251508  0.279317  0.327341  0.366684     0.401561 |  100.0 | 110.365288 | 354.143131 |
    | 2000-01-02 14:30:00 | 5.165155   5.23773  5.326484  5.453656  5.654995  5.977698  6.176026    6.325558 |  197.47692 | 203.227146  209.573987  217.496082  229.157798   248.37856  281.162383    307.810512 | 330.573348 | 0.210871  0.215735  0.221754  0.230517   0.24472   0.26833  0.318717     0.360588 |  100.0 |   97.47692 | 330.573348 |
    | 2000-01-02 15:00:00 | 5.022458  5.088189  5.160651  5.249452  5.372069  5.554867  5.878273    6.130027 | 185.861607 | 191.088913   196.62439  202.837204  210.610077     221.634  238.702939    270.798813 | 301.166497 | 0.201462  0.205771  0.210571  0.216525  0.224877  0.237606  0.260945     0.305927 |  100.0 |  85.861607 | 301.166497 |
    | 2000-01-02 15:30:00 | 4.889667  4.950871  5.015267  5.086867  5.173763  5.289544  5.468232    5.770947 | 175.431719 | 180.194522  185.168124  190.489145  196.512133  203.973852  214.177294    230.517989 | 259.875318 | 0.192889  0.196819  0.200994  0.205684  0.211446   0.21924  0.231531     0.253083 |  100.0 |  75.431719 | 259.875318 |
    | 2000-01-02 16:00:00 | 4.766486  4.823855  4.883157  4.946006  5.015998  5.099759  5.212282     5.38265 | 166.099094 | 170.429663  174.936882  179.670262  184.769783  190.550028  197.608598    207.335086 | 222.601187 | 0.185094  0.188706  0.192473  0.196505  0.201041  0.206534  0.214024     0.225604 |  100.0 |  66.099094 | 222.601187 |
    | 2000-01-02 16:30:00 | 4.652738  4.706439   4.76167  4.819046  4.880101  4.947931  5.029276    5.137711 | 157.776495 | 161.700559  165.787395  170.054555  174.556362  179.424516  184.927321    191.658664 |  200.85773 | 0.178031  0.181349  0.184793  0.188402  0.192278  0.196629  0.201907     0.209046 |  100.0 |  57.776495 |  200.85773 |
    | 2000-01-02 17:00:00 | 4.548224  4.598275  4.649764  4.702898  4.758298  4.817352  4.883097    4.961814 | 150.378786 | 153.920927  157.617946  161.476051  165.516053  169.792158   174.42239    179.665472 | 186.065932 | 0.171655  0.174695  0.177848   0.18113  0.184581  0.188295   0.19247     0.197525 |  100.0 |  50.378786 | 186.065932 |
    | 2000-01-02 17:30:00 | 4.452683  4.499083  4.546929  4.596271  4.647334  4.700711  4.757754    4.821341 | 143.823868 | 147.008796  150.342025  153.826032  157.468893  161.292722  165.348592    169.749837 | 174.737869 | 0.165922  0.168695  0.171577  0.174573  0.177699  0.180994  0.184547     0.188547 |  100.0 |  43.823868 | 174.737869 |
    | 2000-01-02 18:00:00 | 4.365789  4.408559  4.452807  4.498524  4.545775  4.594792  4.646158    4.701174 | 138.033385 | 140.886035  143.880281  147.017633  150.301601  153.741391  157.358978    161.204045 | 165.384022 | 0.160787  0.163305   0.16593  0.168661  0.171507  0.174482  0.177626     0.181023 |  100.0 |  38.033385 | 165.384022 |
    | 2000-01-02 18:30:00 | 4.287155  4.326352  4.367051  4.409228  4.452887  4.498104  4.545115    4.594492 | 132.933235 | 135.478546  138.158245  140.973857   143.92748  147.023381  150.271259    153.693048 |  157.33672 | 0.156206  0.158482  0.160861  0.163345  0.165934  0.168636  0.171467     0.174464 |  100.0 |  32.933235 |  157.33672 |
    | 2000-01-02 19:00:00 | 4.216347  4.252063   4.28929  4.328003  4.368189   4.40986  4.453101    4.498152 | 128.453911 | 130.716536  133.105831  135.623648  138.271863  141.053051  143.971965    147.038639 | 150.274761 | 0.152133  0.154181  0.156329  0.158578  0.160928  0.163382  0.165947     0.168639 |  100.0 |  28.453911 | 150.274761 |
    | 2000-01-02 19:30:00 | 4.152891  4.185253  4.219113  4.254457  4.291268  4.329538  4.369291    4.410618 | 124.530701 | 126.534728  128.657308  130.900685  133.266927  135.758206  138.377457    141.129786 | 144.025414 | 0.148526   0.15036  0.152291  0.154319  0.156444  0.158668  0.160993     0.163427 |  100.0 |  24.530701 | 144.025414 |
    | 2000-01-02 20:00:00 | 4.096288  4.125451  4.156081  4.188173  4.221717  4.256704  4.293135     4.33104 | 121.103761 | 122.872472  124.751366  126.743048  128.849891  131.074145  133.418223    135.885336 | 138.480844 | 0.145342  0.146978  0.148706  0.150527   0.15244  0.154448  0.156552     0.158755 |  100.0 |  21.103761 | 138.480844 |
    | 2000-01-02 20:30:00 | 4.046025  4.072166  4.099726   4.12871  4.159116  4.190942  4.224183    4.258849 | 118.118088 | 119.673761  121.331143  123.093134  124.962404  126.941432  129.032616    131.238561 |  133.56269 | 0.142542  0.143995  0.145535  0.147162  0.148878  0.150684  0.152581     0.154572 |  100.0 |  18.118088 |  133.56269 |
    | 2000-01-02 21:00:00 | 4.001586  4.024899  4.049568  4.075608  4.103025  4.131826  4.162011    4.193582 | 115.523427 | 116.887176  118.344197  119.897608  121.550321  123.305055  125.164369    127.130781 | 129.207046 | 0.140087  0.141373  0.142739  0.144187  0.145719  0.147337  0.149042     0.150834 |  100.0 |  15.523427 | 129.207046 |
    | 2000-01-02 21:30:00 | 3.962458  3.983148  4.005119  4.028395  4.052991  4.078921  4.106193    4.134812 | 113.274094 |  114.46576  115.742422  117.107335  118.563586  120.114093  121.761602    123.508734 | 125.358099 | 0.137942  0.139075  0.140282  0.141566  0.142929  0.144372  0.145897     0.147506 |  100.0 |  13.274094 | 125.358099 |
    | 2000-01-02 22:00:00 | 3.928143   3.94642  3.965895  3.986598  4.008554  4.031782  4.056298    4.082115 | 111.328777 | 112.366848  113.481908  114.677271  115.956129  117.321537  118.776398    120.323476 | 121.965433 | 0.136074  0.137067   0.13813  0.139264  0.140471  0.141753  0.143112      0.14455 |  100.0 |  11.328777 | 121.965433 |
    | 2000-01-02 22:30:00 | 3.898164  3.914237  3.931421  3.949749  3.969254   3.98996  4.011891    4.035064 | 109.650285 | 110.551859  111.522769  112.566321  113.685741  114.884157  116.164579    117.529889 | 118.982851 | 0.134451   0.13532  0.136252  0.137249  0.138314  0.139448  0.140655     0.141935 |  100.0 |   9.650285 | 118.982851 |
    | 2000-01-02 23:00:00 | 3.872066  3.886142  3.901238  3.917392  3.934638   3.95301  3.972534    3.993235 | 108.205287 | 108.986064  109.828942  110.737162  111.713925  112.762369  113.885549    115.086425 | 116.367848 | 0.133045  0.133803  0.134617  0.135491  0.136426  0.137427  0.138493     0.139628 |  100.0 |   8.205287 | 116.367848 |
    | 2000-01-02 23:30:00 | 3.849425  3.861703   3.87491  3.889086  3.904269  3.920495  3.937796    3.956201 | 106.964036 | 107.638331  108.367962  109.156054  110.005729   110.92008  111.902155    112.954935 | 114.081321 | 0.131832  0.132489  0.133198  0.133961  0.134781  0.135659  0.136598     0.137601 |  100.0 |   6.964036 | 114.081321 |
    | 2000-01-03 00:00:00 | 3.829847  3.840516  3.852025  3.864416  3.877726  3.891995  3.907257    3.923546 | 105.900084 | 106.480873  107.110728  107.792622  108.529555  109.324528  110.180531    111.100517 | 112.087391 | 0.130786  0.131356  0.131971  0.132635   0.13335  0.134118  0.134942     0.135824 |  100.0 |   5.900084 | 112.087391 |
    | 2000-01-03 00:30:00 | 3.812968  3.822207    3.8322  3.842988   3.85461  3.867106  3.880512    3.894866 |  104.99001 | 105.488982  106.031259  106.619637  107.256959  107.946098  108.689941    109.491371 | 110.353249 | 0.129888  0.130379  0.130912  0.131488  0.132109  0.132779    0.1335     0.134273 |  100.0 |    4.99001 | 110.353249 |
    | 2000-01-03 01:00:00 | 3.798461  3.806434   3.81508  3.824438  3.834548  3.845448  3.857177    3.869771 | 104.213141 |  104.64077  105.106453  105.612791  106.162446  106.758135  107.402608     108.09864 |  108.84901 | 0.129118  0.129541      0.13  0.130498  0.131037  0.131619  0.132247     0.132922 |  100.0 |   4.213141 |  108.84901 |
    | 2000-01-03 01:30:00 | 3.786025  3.792885  3.800341  3.808431  3.817193  3.826666  3.836887    3.847894 | 103.551298 | 103.916918  104.315843  104.750465  105.223254  105.736746  106.293533    106.896251 | 107.547561 |  0.12846  0.128823  0.129218  0.129647  0.130113  0.130617  0.131162      0.13175 |  100.0 |   3.551298 | 107.547561 |
    | 2000-01-03 02:00:00 | 3.775394  3.781278  3.787688  3.794659  3.802228  3.810431  3.819305    3.828888 | 102.988541 | 103.300433  103.641362  104.013508   104.41914  104.860603  105.340313    105.860742 | 106.424413 | 0.127899  0.128209  0.128548  0.128917  0.129318  0.129753  0.130225     0.130735 |  100.0 |   2.988541 | 106.424413 |
    | 2000-01-03 02:30:00 | 3.766328  3.771361  3.776855  3.782843   3.78936   3.79644  3.804119    3.812432 |  102.51094 | 102.776417  103.067118  103.385015  103.732174  104.110742  104.522949    104.971093 | 105.457533 | 0.127421  0.127686  0.127976  0.128292  0.128637  0.129011  0.129418      0.12986 |  100.0 |    2.51094 | 105.457533 |
    | 2000-01-03 03:00:00 | 3.758615  3.762909  3.767605  3.772734  3.778328   3.78442  3.791042    3.798228 | 102.106358 | 102.331851   102.57918  102.850118  103.146529  103.470366  103.823666    104.208544 | 104.627187 | 0.127015  0.127241  0.127488  0.127759  0.128054  0.128375  0.128726     0.129106 |  100.0 |   2.106358 | 104.627187 |
    | 2000-01-03 03:30:00 | 3.752069  3.755722  3.759726  3.764107  3.768896  3.774121  3.779813    3.786006 | 101.764251 | 101.955391  102.165375  102.395784  102.648289  102.924653   103.22672    103.556423 | 103.915769 | 0.126671  0.126863  0.127074  0.127304  0.127556  0.127832  0.128132     0.128459 |  100.0 |   1.764251 | 103.915769 |
    | 2000-01-03 04:00:00 | 3.746524  3.749625   3.75303  3.756763   3.76085   3.76532  3.770199    3.775519 |  101.47548 | 101.637182  101.815097  102.010625  102.225257   102.46057  102.718225    102.999971 | 103.307636 |  0.12638  0.126543  0.126722  0.126918  0.127133  0.127368  0.127625     0.127905 |  100.0 |    1.47548 | 103.307636 |
    | 2000-01-03 04:30:00 | 3.741837  3.744464  3.747353  3.750525  3.754006  3.757818  3.761989    3.766545 | 101.232151 | 101.368689  101.519134  101.684724  101.866779    102.0667  102.285974    102.526172 | 102.788945 | 0.126134  0.126272  0.126424   0.12659  0.126773  0.126973  0.127193     0.127432 |  100.0 |   1.232151 | 102.788945 |
    | 2000-01-03 05:00:00 | 3.737884  3.740104   3.74255   3.74524  3.748196   3.75144  3.754995    3.758887 | 101.027458 | 101.142535  101.269511  101.409471  101.563578  101.733072  101.919275    102.123589 | 102.347494 | 0.125927  0.126043  0.126172  0.126313  0.126468  0.126638  0.126825     0.127029 |  100.0 |   1.027458 | 102.347494 |
    | 2000-01-03 05:30:00 | 3.734556  3.736428  3.738494   3.74077  3.743275  3.746029  3.749052    3.752367 | 100.855551 | 100.952367  101.059337  101.177409  101.307603  101.451009  101.608796    101.782207 | 101.972564 | 0.125753  0.125851  0.125959  0.126078   0.12621  0.126354  0.126513     0.126687 |  100.0 |   0.855551 | 101.972564 |
    | 2000-01-03 06:00:00 | 3.731759  3.733335  3.735077  3.736998  3.739116  3.741448  3.744013    3.746831 | 100.711411 | 100.792722  100.882678  100.982102  101.091884  101.212981  101.346418    101.493292 | 101.654774 | 0.125607  0.125689   0.12578  0.125881  0.125992  0.126114  0.126248     0.126396 |  100.0 |   0.711411 | 101.654774 |
    | 2000-01-03 06:30:00 | 3.729413  3.730737  3.732202  3.733821  3.735608  3.737579   3.73975    3.742139 | 100.590745 | 100.658919  100.734435  100.818008  100.910409  101.012472  101.125096    101.249243 | 101.385946 | 0.125484  0.125553   0.12563  0.125714  0.125808  0.125911  0.126025      0.12615 |  100.0 |   0.590745 | 101.385946 |
    | 2000-01-03 07:00:00 | 3.727448  3.728559  3.729789   3.73115  3.732655  3.734317  3.736151    3.738171 | 100.489885 | 100.546951  100.610239  100.680366     100.758  100.843865  100.938742    101.043475 | 101.158967 | 0.125381  0.125439  0.125503  0.125575  0.125653   0.12574  0.125836     0.125942 |  100.0 |   0.489885 | 101.158967 |
    | 2000-01-03 07:30:00 | 3.725806  3.726736  3.727767  3.728909  3.730174  3.731573  3.733118    3.734824 | 100.405708 | 100.453401  100.506354  100.565099  100.630213  100.702322  100.782105    100.870293 | 100.967676 | 0.125295  0.125344  0.125398  0.125457  0.125524  0.125597  0.125678     0.125767 |  100.0 |   0.405708 | 100.967676 |
    | 2000-01-03 08:00:00 | 3.724435  3.725212  3.726075  3.727032  3.728093  3.729268  3.730568    3.732005 |  100.33556 | 100.375356  100.419591  100.468722  100.523245  100.583698  100.650668    100.724791 | 100.806751 | 0.125224  0.125264  0.125309  0.125359  0.125415  0.125476  0.125544     0.125619 |  100.0 |    0.33556 | 100.806751 |
    | 2000-01-03 08:30:00 | 3.723293  3.723941  3.724662  3.725462  3.726351  3.727336  3.728428    3.729636 | 100.277186 | 100.310343  100.347239  100.388264  100.433843  100.484441   100.54056    100.602751 | 100.671607 | 0.125164  0.125198  0.125235  0.125277  0.125324  0.125375  0.125432     0.125495 |  100.0 |   0.277186 | 100.671607 |
    | 2000-01-03 09:00:00 | 3.722342  3.722882  3.723483  3.724152  3.724895  3.725719  3.726634    3.727648 | 100.228681 | 100.256265  100.286992  100.321196  100.359239  100.401519  100.448468    100.500559 | 100.558304 | 0.125114  0.125142  0.125174  0.125209  0.125248  0.125291  0.125339     0.125392 |  100.0 |   0.228681 | 100.558304 |
    | 2000-01-03 09:30:00 | 3.721552  3.722001  3.722502  3.723059  3.723679  3.724368  3.725134    3.725983 | 100.188431 | 100.211346  100.236899  100.265372  100.297077  100.332351  100.371565    100.415124 |  100.46347 | 0.125073  0.125096  0.125123  0.125152  0.125184   0.12522   0.12526     0.125304 |  100.0 |   0.188431 |  100.46347 |
    | 2000-01-03 10:00:00 | 3.720897   3.72127  3.721686   3.72215  3.722667  3.723242  3.723881    3.724591 | 100.155078 | 100.174087  100.195306  100.218975  100.245357  100.274741  100.307443     100.34381 |  100.38422 | 0.125039  0.125058   0.12508  0.125104  0.125131  0.125161  0.125195     0.125232 |  100.0 |   0.155078 |  100.38422 |
    | 2000-01-03 10:30:00 | 3.720354  3.720663  3.721009  3.721394  3.721824  3.722303  3.722836    3.723429 | 100.127477 | 100.143224   100.16082  100.180466  100.202387  100.226828  100.254059    100.284374 | 100.318096 |  0.12501  0.125027  0.125045  0.125065  0.125087  0.125112   0.12514     0.125171 |  100.0 |   0.127477 | 100.318096 |
    | 2000-01-03 11:00:00 | 3.719904  3.720161  3.720447  3.720767  3.721124  3.721523  3.721966     3.72246 | 100.104666 | 100.117693  100.132264  100.148549  100.166738  100.187038  100.209678     100.23491 | 100.263008 | 0.124987     0.125  0.125015  0.125032  0.125051  0.125071  0.125095      0.12512 |  100.0 |   0.104666 | 100.263008 |
    | 2000-01-03 11:30:00 | 3.719533  3.719745  3.719983  3.720248  3.720544  3.720875  3.721243    3.721655 | 100.085838 | 100.096601  100.108651  100.122131  100.137202  100.154039  100.172835    100.193806 | 100.217183 | 0.124968  0.124979  0.124991  0.125005   0.12502  0.125038  0.125057     0.125078 |  100.0 |   0.085838 | 100.217183 |
    | 2000-01-03 12:00:00 | 3.719227  3.719402  3.719598  3.719818  3.720063  3.720337  3.720643    3.720985 | 100.070317 | 100.079198   100.08915  100.100294  100.112764  100.126709  100.142293    100.159696 | 100.179117 | 0.124952  0.124961  0.124971  0.124982  0.124995   0.12501  0.125026     0.125043 |  100.0 |   0.070317 | 100.179117 |
    | 2000-01-03 12:30:00 | 3.718975  3.719119  3.719281  3.719462  3.719665  3.719892  3.720146     3.72043 | 100.057539 | 100.064858  100.073066  100.082266  100.092571  100.104105  100.117008     100.13143 | 100.147541 | 0.124938  0.124946  0.124954  0.124964  0.124975  0.124986     0.125     0.125014 |  100.0 |   0.057539 | 100.147541 |
    | 2000-01-03 13:00:00 | 3.718767  3.718886  3.719019  3.719169  3.719337  3.719525  3.719735    3.719969 | 100.047032 | 100.053055  100.059817  100.067402  100.075907  100.085435  100.096103    100.108039 | 100.121385 | 0.124928  0.124934  0.124941  0.124949  0.124957  0.124967  0.124978      0.12499 |  100.0 |   0.047032 | 100.121385 |
    | 2000-01-03 13:30:00 | 3.718596  3.718694  3.718804  3.718927  3.719066  3.719221  3.719394    3.719589 | 100.038402 | 100.043353  100.048917  100.055163  100.062172  100.070033  100.078841    100.088707 | 100.099748 | 0.124919  0.124924   0.12493  0.124936  0.124943  0.124951   0.12496     0.124971 |  100.0 |   0.038402 | 100.099748 |
    | 2000-01-03 14:00:00 | 3.718456  3.718536  3.718627  3.718728  3.718842   3.71897  3.719114    3.719274 | 100.031322 | 100.035388  100.039959  100.045097  100.050867  100.057343  100.064607    100.072751 | 100.081873 | 0.124911  0.124916   0.12492  0.124926  0.124932  0.124938  0.124946     0.124954 |  100.0 |   0.031322 | 100.081873 |
    | 2000-01-03 14:30:00 | 3.718341  3.718407  3.718481  3.718565  3.718659  3.718764  3.718882    3.719014 | 100.025521 | 100.028855  100.032608  100.036828  100.041572  100.046901  100.052884    100.059597 | 100.067124 | 0.124905  0.124909  0.124913  0.124917  0.124922  0.124927  0.124934     0.124941 |  100.0 |   0.025521 | 100.067124 |
    | 2000-01-03 15:00:00 | 3.718247  3.718301  3.718362   3.71843  3.718507  3.718594  3.718691    3.718801 | 100.020773 | 100.023504   100.02658  100.030043  100.033939  100.038319  100.043241    100.048768 | 100.054971 |   0.1249  0.124903  0.124906   0.12491  0.124914  0.124919  0.124924     0.124929 |  100.0 |   0.020773 | 100.054971 |
    | 2000-01-03 15:30:00 |  3.71817  3.718214  3.718264   3.71832  3.718383  3.718455  3.718535    3.718624 | 100.016892 | 100.019126  100.021645  100.024483  100.027679  100.031274  100.035318    100.039863 | 100.044968 | 0.124896  0.124899  0.124901  0.124904  0.124908  0.124911  0.124916      0.12492 |  100.0 |   0.016892 | 100.044968 |
    | 2000-01-03 16:00:00 | 3.718107  3.718143  3.718184   3.71823  3.718282   3.71834  3.718406     3.71848 | 100.013722 | 100.015548  100.017608  100.019932  100.022549  100.025498  100.028816    100.032549 | 100.036746 | 0.124893  0.124895  0.124897    0.1249  0.124902  0.124905  0.124909     0.124913 |  100.0 |   0.013722 | 100.036746 |
    | 2000-01-03 16:30:00 | 3.718055  3.718085  3.718118  3.718156  3.718198  3.718246    3.7183    3.718361 | 100.011136 | 100.012627   100.01431   100.01621  100.018352  100.020767  100.023487    100.026549 | 100.029995 |  0.12489  0.124892  0.124894  0.124896  0.124898    0.1249  0.124903     0.124906 |  100.0 |   0.011136 | 100.029995 |
    | 2000-01-03 17:00:00 | 3.718013  3.718037  3.718065  3.718095   3.71813  3.718169  3.718214    3.718263 | 100.009028 | 100.010244  100.011618   100.01317  100.014921  100.016897  100.019124    100.021633 | 100.024459 | 0.124888   0.12489  0.124891  0.124893  0.124894  0.124896  0.124899     0.124901 |  100.0 |   0.009028 | 100.024459 |
    | 2000-01-03 17:30:00 | 3.717979  3.717999  3.718021  3.718046  3.718075  3.718107  3.718143    3.718184 | 100.007313 | 100.008304  100.009424   100.01069   100.01212  100.013734  100.015555    100.017609 | 100.019924 | 0.124887  0.124888  0.124889   0.12489  0.124892  0.124893  0.124895     0.124897 |  100.0 |   0.007313 | 100.019924 |
    | 2000-01-03 18:00:00 | 3.717951  3.717967  3.717985  3.718006  3.718029  3.718055  3.718085    3.718118 | 100.005918 | 100.006724  100.007636  100.008668  100.009835  100.011152   100.01264    100.014319 | 100.016213 | 0.124885  0.124886  0.124887  0.124888  0.124889  0.124891  0.124892     0.124894 |  100.0 |   0.005918 | 100.016213 |
    | 2000-01-03 18:30:00 | 3.717929  3.717942  3.717956  3.717973  3.717992  3.718013  3.718038    3.718065 | 100.004785 |  100.00544  100.006182  100.007022  100.007973  100.009047  100.010261    100.011633 | 100.013181 | 0.124884  0.124885  0.124885  0.124886  0.124887  0.124888   0.12489     0.124891 |  100.0 |   0.004785 | 100.013181 |
    | 2000-01-03 19:00:00 |  3.71791  3.717921  3.717933  3.717946  3.717962  3.717979  3.717999    3.718021 | 100.003865 | 100.004397     100.005  100.005684  100.006457  100.007332  100.008322    100.009441 | 100.010705 | 0.124883  0.124883  0.124884  0.124885  0.124886  0.124887  0.124888     0.124889 |  100.0 |   0.003865 | 100.010705 |
    | 2000-01-03 19:30:00 | 3.717895  3.717904  3.717914  3.717925  3.717937  3.717952  3.717968    3.717986 | 100.003119 | 100.003551  100.004041  100.004596  100.005225  100.005937  100.006743    100.007655 | 100.008686 | 0.124882  0.124883  0.124883  0.124884  0.124884  0.124885  0.124886     0.124887 |  100.0 |   0.003119 | 100.008686 |
    | 2000-01-03 20:00:00 | 3.717883   3.71789  3.717898  3.717907  3.717917  3.717929  3.717942    3.717957 | 100.002515 | 100.002865  100.003262  100.003713  100.004224  100.004803  100.005459    100.006201 | 100.007041 | 0.124882  0.124882  0.124882  0.124883  0.124883  0.124884  0.124885     0.124885 |  100.0 |   0.002515 | 100.007041 |
    | 2000-01-03 20:30:00 | 3.717873  3.717879  3.717886  3.717893  3.717901  3.717911  3.717921    3.717933 | 100.002026 |  100.00231  100.002632  100.002997  100.003412  100.003882  100.004415    100.005018 | 100.005702 | 0.124881  0.124881  0.124882  0.124882  0.124882  0.124883  0.124884     0.124884 |  100.0 |   0.002026 | 100.005702 |
    | 2000-01-03 21:00:00 | 3.717866   3.71787  3.717875  3.717881  3.717888  3.717896  3.717904    3.717914 | 100.001631 |  100.00186  100.002121  100.002417  100.002753  100.003135  100.003567    100.004058 | 100.004613 | 0.124881  0.124881  0.124881  0.124881  0.124882  0.124882  0.124883     0.124883 |  100.0 |   0.001631 | 100.004613 |
    | 2000-01-03 21:30:00 | 3.717859  3.717863  3.717867  3.717872  3.717877  3.717883   3.71789    3.717898 | 100.001312 | 100.001497  100.001708  100.001948   100.00222  100.002529   100.00288    100.003278 | 100.003729 |  0.12488   0.12488  0.124881  0.124881  0.124881  0.124882  0.124882     0.124882 |  100.0 |   0.001312 | 100.003729 |
    | 2000-01-03 22:00:00 | 3.717854  3.717857   3.71786  3.717864  3.717869  3.717874  3.717879    3.717886 | 100.001054 | 100.001204  100.001374  100.001568  100.001788  100.002039  100.002323    100.002646 | 100.003012 |  0.12488   0.12488   0.12488  0.124881  0.124881  0.124881  0.124881     0.124882 |  100.0 |   0.001054 | 100.003012 |
    | 2000-01-03 22:30:00 |  3.71785  3.717852  3.717855  3.717858  3.717862  3.717866   3.71787    3.717876 | 100.000846 | 100.000967  100.001105  100.001261  100.001439  100.001642  100.001872    100.002133 |  100.00243 |  0.12488   0.12488   0.12488   0.12488   0.12488  0.124881  0.124881     0.124881 |  100.0 |   0.000846 |  100.00243 |
    | 2000-01-03 23:00:00 | 3.717846  3.717848  3.717851  3.717853  3.717856  3.717859  3.717863    3.717867 | 100.000679 | 100.000776  100.000887  100.001014  100.001158  100.001321  100.001507    100.001719 | 100.001959 |  0.12488   0.12488   0.12488   0.12488   0.12488   0.12488   0.12488     0.124881 |  100.0 |   0.000679 | 100.001959 |
    | 2000-01-03 23:30:00 | 3.717844  3.717845  3.717847  3.717849  3.717851  3.717854  3.717857    3.717861 | 100.000544 | 100.000623  100.000712  100.000814   100.00093  100.001062  100.001213    100.001384 | 100.001578 | 0.124879   0.12488   0.12488   0.12488   0.12488   0.12488   0.12488      0.12488 |  100.0 |   0.000544 | 100.001578 |
    | 2000-01-04 00:00:00 | 3.717842  3.717843  3.717844  3.717846  3.717848   3.71785  3.717852    3.717855 | 100.000436 | 100.000499  100.000571  100.000653  100.000747  100.000854  100.000975    100.001113 |  100.00127 | 0.124879  0.124879  0.124879   0.12488   0.12488   0.12488   0.12488      0.12488 |  100.0 |   0.000436 |  100.00127 |
    | 2000-01-04 00:30:00 |  3.71784  3.717841  3.717842  3.717843  3.717845  3.717847  3.717849    3.717851 | 100.000349 |   100.0004  100.000458  100.000524  100.000599  100.000685  100.000783    100.000895 | 100.001022 | 0.124879  0.124879  0.124879  0.124879   0.12488   0.12488   0.12488      0.12488 |  100.0 |   0.000349 | 100.001022 |
    | 2000-01-04 01:00:00 | 3.717838  3.717839   3.71784  3.717841  3.717842  3.717844  3.717845    3.717847 | 100.000279 |  100.00032  100.000367   100.00042   100.00048   100.00055  100.000628    100.000718 | 100.000821 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879   0.12488      0.12488 |  100.0 |   0.000279 | 100.000821 |
    | 2000-01-04 01:30:00 | 3.717837  3.717838  3.717839   3.71784  3.717841  3.717842  3.717843    3.717844 | 100.000223 | 100.000256  100.000293  100.000336  100.000385   100.00044  100.000504    100.000576 | 100.000659 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000223 | 100.000659 |
    | 2000-01-04 02:00:00 | 3.717836  3.717837  3.717837  3.717838  3.717839   3.71784  3.717841    3.717842 | 100.000178 | 100.000204  100.000234  100.000269  100.000308  100.000353  100.000404    100.000462 | 100.000529 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000178 | 100.000529 |
    | 2000-01-04 02:30:00 | 3.717836  3.717836  3.717837  3.717837  3.717838  3.717838  3.717839     3.71784 | 100.000142 | 100.000163  100.000187  100.000215  100.000246  100.000282  100.000323     100.00037 | 100.000424 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000142 | 100.000424 |
    | 2000-01-04 03:00:00 | 3.717835  3.717835  3.717836  3.717836  3.717837  3.717837  3.717838    3.717839 | 100.000113 |  100.00013   100.00015  100.000172  100.000197  100.000226  100.000259    100.000296 | 100.000339 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000113 | 100.000339 |
    | 2000-01-04 03:30:00 | 3.717835  3.717835  3.717835  3.717836  3.717836  3.717836  3.717837    3.717838 |  100.00009 | 100.000104  100.000119  100.000137  100.000157   100.00018  100.000207    100.000237 | 100.000272 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |    0.00009 | 100.000272 |
    | 2000-01-04 04:00:00 | 3.717834  3.717834  3.717835  3.717835  3.717835  3.717836  3.717836    3.717837 | 100.000072 | 100.000083  100.000095  100.000109  100.000125  100.000144  100.000165     100.00019 | 100.000217 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000072 | 100.000217 |
    | 2000-01-04 04:30:00 | 3.717834  3.717834  3.717834  3.717835  3.717835  3.717835  3.717835    3.717836 | 100.000057 | 100.000066  100.000076  100.000087    100.0001  100.000115  100.000132    100.000151 | 100.000174 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000057 | 100.000174 |
    | 2000-01-04 05:00:00 | 3.717834  3.717834  3.717834  3.717834  3.717834  3.717835  3.717835    3.717835 | 100.000046 | 100.000052   100.00006  100.000069   100.00008  100.000092  100.000105    100.000121 | 100.000139 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000046 | 100.000139 |
    | 2000-01-04 05:30:00 | 3.717833  3.717834  3.717834  3.717834  3.717834  3.717834  3.717834    3.717835 | 100.000036 | 100.000042  100.000048  100.000055  100.000063  100.000073  100.000084    100.000096 | 100.000111 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000036 | 100.000111 |
    | 2000-01-04 06:00:00 | 3.717833  3.717833  3.717834  3.717834  3.717834  3.717834  3.717834    3.717834 | 100.000029 | 100.000033  100.000038  100.000044  100.000051  100.000058  100.000067    100.000077 | 100.000088 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000029 | 100.000088 |
    | 2000-01-04 06:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717834  3.717834  3.717834    3.717834 | 100.000023 | 100.000026   100.00003  100.000035   100.00004  100.000046  100.000053    100.000061 |  100.00007 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000023 |  100.00007 |
    | 2000-01-04 07:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717834  3.717834    3.717834 | 100.000018 | 100.000021  100.000024  100.000028  100.000032  100.000037  100.000042    100.000049 | 100.000056 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000018 | 100.000056 |
    | 2000-01-04 07:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717834 | 100.000014 | 100.000017  100.000019  100.000022  100.000025  100.000029  100.000034    100.000039 | 100.000045 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000014 | 100.000045 |
    | 2000-01-04 08:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000011 | 100.000013  100.000015  100.000017   100.00002  100.000023  100.000027    100.000031 | 100.000035 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000011 | 100.000035 |
    | 2000-01-04 08:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000009 |  100.00001  100.000012  100.000014  100.000016  100.000018  100.000021    100.000024 | 100.000028 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000009 | 100.000028 |
    | 2000-01-04 09:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000007 | 100.000008  100.000009  100.000011  100.000013  100.000015  100.000017    100.000019 | 100.000022 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000007 | 100.000022 |
    | 2000-01-04 09:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000006 | 100.000007  100.000008  100.000009   100.00001  100.000012  100.000013    100.000015 | 100.000018 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000006 | 100.000018 |
    | 2000-01-04 10:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000004 | 100.000005  100.000006  100.000007  100.000008  100.000009  100.000011    100.000012 | 100.000014 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000004 | 100.000014 |
    | 2000-01-04 10:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000004 | 100.000004  100.000005  100.000005  100.000006  100.000007  100.000008     100.00001 | 100.000011 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000004 | 100.000011 |
    | 2000-01-04 11:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000003 | 100.000003  100.000004  100.000004  100.000005  100.000006  100.000007    100.000008 | 100.000009 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000003 | 100.000009 |
    | 2000-01-04 11:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000002 | 100.000003  100.000003  100.000003  100.000004  100.000005  100.000005    100.000006 | 100.000007 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000002 | 100.000007 |
    | 2000-01-04 12:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000002 | 100.000002  100.000002  100.000003  100.000003  100.000004  100.000004    100.000005 | 100.000006 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000002 | 100.000006 |
    | 2000-01-04 12:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000002  100.000002  100.000002  100.000002  100.000003  100.000003    100.000004 | 100.000004 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000001 | 100.000004 |
    | 2000-01-04 13:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000001  100.000001  100.000002  100.000002  100.000002  100.000003    100.000003 | 100.000003 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000001 | 100.000003 |
    | 2000-01-04 13:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000001  100.000001  100.000001  100.000002  100.000002  100.000002    100.000002 | 100.000003 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000001 | 100.000003 |
    | 2000-01-04 14:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000001  100.000001  100.000001  100.000001  100.000001  100.000002    100.000002 | 100.000002 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000001 | 100.000002 |
    | 2000-01-04 14:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000001  100.000001  100.000001  100.000001  100.000001  100.000001    100.000001 | 100.000002 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |   0.000001 | 100.000002 |
    | 2000-01-04 15:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0  100.000001  100.000001  100.000001  100.000001  100.000001    100.000001 | 100.000001 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 | 100.000001 |
    | 2000-01-04 15:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0  100.000001  100.000001  100.000001  100.000001    100.000001 | 100.000001 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 | 100.000001 |
    | 2000-01-04 16:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0  100.000001  100.000001    100.000001 | 100.000001 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 | 100.000001 |
    | 2000-01-04 16:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0    100.000001 | 100.000001 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 | 100.000001 |
    | 2000-01-04 17:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 | 100.000001 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 | 100.000001 |
    | 2000-01-04 17:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 18:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 18:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 19:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 19:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 20:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 20:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 21:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 21:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 22:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 22:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 23:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 23:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.124879  0.124879  0.124879  0.124879  0.124879  0.124879  0.124879     0.124879 |  100.0 |        0.0 |      100.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _kinw_impl_euler_tiny_segments:

tiny segments
_____________

There is no limitation on the channel's "shortness"; even a length of zero is okay:

.. integration-test::

    >>> length(0.0)
    >>> model.prepare_states(100.0)
    >>> conditions = test("kinw_impl_euler_tiny_segments", get_conditions="2000-01-01")
    |                date |                                                                       waterdepth |     inflow |                                                                         internalflow |    outflow |                                    watervolume | input1 |     input2 |     output |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000003 | 100.000003  100.000003  100.000003  100.000003  100.000003  100.000003    100.000003 | 100.000003 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000003 | 100.000003 |
    | 2000-01-01 03:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000029 | 100.000029  100.000029  100.000029  100.000029  100.000029  100.000029    100.000029 | 100.000029 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000029 | 100.000029 |
    | 2000-01-01 04:00:00 | 3.717836  3.717836  3.717836  3.717836  3.717836  3.717836  3.717836    3.717836 | 100.000175 | 100.000175  100.000175  100.000175  100.000175  100.000175  100.000175    100.000175 | 100.000175 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000175 | 100.000175 |
    | 2000-01-01 04:30:00 | 3.717847  3.717847  3.717847  3.717847  3.717847  3.717847  3.717847    3.717847 | 100.000826 | 100.000826  100.000826  100.000826  100.000826  100.000826  100.000826    100.000826 | 100.000826 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000826 | 100.000826 |
    | 2000-01-01 05:00:00 | 3.717889  3.717889  3.717889  3.717889  3.717889  3.717889  3.717889    3.717889 | 100.003194 | 100.003194  100.003194  100.003194  100.003194  100.003194  100.003194    100.003194 | 100.003194 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.003194 | 100.003194 |
    | 2000-01-01 05:30:00 | 3.718018  3.718018  3.718018  3.718018  3.718018  3.718018  3.718018    3.718018 | 100.010515 | 100.010515  100.010515  100.010515  100.010515  100.010515  100.010515    100.010515 | 100.010515 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.010515 | 100.010515 |
    | 2000-01-01 06:00:00 | 3.718367  3.718367  3.718367  3.718367  3.718367  3.718367  3.718367    3.718367 | 100.030315 | 100.030315  100.030315  100.030315  100.030315  100.030315  100.030315    100.030315 | 100.030315 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.030315 | 100.030315 |
    | 2000-01-01 06:30:00 | 3.719209  3.719209  3.719209  3.719209  3.719209  3.719209  3.719209    3.719209 | 100.078179 | 100.078179  100.078179  100.078179  100.078179  100.078179  100.078179    100.078179 | 100.078179 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.078179 | 100.078179 |
    | 2000-01-01 07:00:00 |  3.72106   3.72106   3.72106   3.72106   3.72106   3.72106   3.72106     3.72106 | 100.183351 | 100.183351  100.183351  100.183351  100.183351  100.183351  100.183351    100.183351 | 100.183351 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.183351 | 100.183351 |
    | 2000-01-01 07:30:00 | 3.724801  3.724801  3.724801  3.724801  3.724801  3.724801  3.724801    3.724801 | 100.396214 | 100.396214  100.396214  100.396214  100.396214  100.396214  100.396214    100.396214 | 100.396214 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.396214 | 100.396214 |
    | 2000-01-01 08:00:00 | 3.731839  3.731839  3.731839  3.731839  3.731839  3.731839  3.731839    3.731839 | 100.797305 | 100.797305  100.797305  100.797305  100.797305  100.797305  100.797305    100.797305 | 100.797305 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.797305 | 100.797305 |
    | 2000-01-01 08:30:00 | 3.744253  3.744253  3.744253  3.744253  3.744253  3.744253  3.744253    3.744253 | 101.507032 | 101.507032  101.507032  101.507032  101.507032  101.507032  101.507032    101.507032 | 101.507032 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   1.507032 | 101.507032 |
    | 2000-01-01 09:00:00 | 3.764915  3.764915  3.764915  3.764915  3.764915  3.764915  3.764915    3.764915 | 102.694853 | 102.694853  102.694853  102.694853  102.694853  102.694853  102.694853    102.694853 | 102.694853 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   2.694853 | 102.694853 |
    | 2000-01-01 09:30:00 | 3.797528  3.797528  3.797528  3.797528  3.797528  3.797528  3.797528    3.797528 | 104.586356 | 104.586356  104.586356  104.586356  104.586356  104.586356  104.586356    104.586356 | 104.586356 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   4.586356 | 104.586356 |
    | 2000-01-01 10:00:00 | 3.846529  3.846529  3.846529  3.846529  3.846529  3.846529  3.846529    3.846529 | 107.466641 | 107.466641  107.466641  107.466641  107.466641  107.466641  107.466641    107.466641 | 107.466641 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   7.466641 | 107.466641 |
    | 2000-01-01 10:30:00 | 3.916808  3.916808  3.916808  3.916808  3.916808  3.916808  3.916808    3.916808 | 111.678559 | 111.678559  111.678559  111.678559  111.678559  111.678559  111.678559    111.678559 | 111.678559 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  11.678559 | 111.678559 |
    | 2000-01-01 11:00:00 |  4.01325   4.01325   4.01325   4.01325   4.01325   4.01325   4.01325     4.01325 | 117.614836 | 117.614836  117.614836  117.614836  117.614836  117.614836  117.614836    117.614836 | 117.614836 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  17.614836 | 117.614836 |
    | 2000-01-01 11:30:00 | 4.140133  4.140133  4.140133  4.140133  4.140133  4.140133  4.140133    4.140133 | 125.703736 | 125.703736  125.703736  125.703736  125.703736  125.703736  125.703736    125.703736 | 125.703736 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  25.703736 | 125.703736 |
    | 2000-01-01 12:00:00 | 4.300519  4.300519  4.300519  4.300519  4.300519  4.300519  4.300519    4.300519 | 136.388647 | 136.388647  136.388647  136.388647  136.388647  136.388647  136.388647    136.388647 | 136.388647 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  36.388647 | 136.388647 |
    | 2000-01-01 12:30:00 | 4.495772  4.495772  4.495772  4.495772  4.495772  4.495772  4.495772    4.495772 | 150.102726 | 150.102726  150.102726  150.102726  150.102726  150.102726  150.102726    150.102726 | 150.102726 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  50.102726 | 150.102726 |
    | 2000-01-01 13:00:00 |  4.72534   4.72534   4.72534   4.72534   4.72534   4.72534   4.72534     4.72534 | 167.240349 | 167.240349  167.240349  167.240349  167.240349  167.240349  167.240349    167.240349 | 167.240349 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  67.240349 | 167.240349 |
    | 2000-01-01 13:30:00 | 4.986822  4.986822  4.986822  4.986822  4.986822  4.986822  4.986822    4.986822 | 188.127569 | 188.127569  188.127569  188.127569  188.127569  188.127569  188.127569    188.127569 | 188.127569 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  88.127569 | 188.127569 |
    | 2000-01-01 14:00:00 | 5.276289  5.276289  5.276289  5.276289  5.276289  5.276289  5.276289    5.276289 | 212.993962 | 212.993962  212.993962  212.993962  212.993962  212.993962  212.993962    212.993962 | 212.993962 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 112.993962 | 212.993962 |
    | 2000-01-01 14:30:00 | 5.588719  5.588719  5.588719  5.588719  5.588719  5.588719  5.588719    5.588719 | 241.948195 | 241.948195  241.948195  241.948195  241.948195  241.948195  241.948195    241.948195 | 241.948195 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 141.948195 | 241.948195 |
    | 2000-01-01 15:00:00 |  5.91846   5.91846   5.91846   5.91846   5.91846   5.91846   5.91846     5.91846 |  274.95935 |  274.95935   274.95935   274.95935   274.95935   274.95935   274.95935     274.95935 |  274.95935 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  174.95935 |  274.95935 |
    | 2000-01-01 15:30:00 | 6.203381  6.203381  6.203381  6.203381  6.203381  6.203381  6.203381    6.203381 | 311.845554 | 311.845554  311.845554  311.845554  311.845554  311.845554  311.845554    311.845554 | 311.845554 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 211.845554 | 311.845554 |
    | 2000-01-01 16:00:00 | 6.459166  6.459166  6.459166  6.459166  6.459166  6.459166  6.459166    6.459166 | 352.270855 | 352.270855  352.270855  352.270855  352.270855  352.270855  352.270855    352.270855 | 352.270855 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 252.270855 | 352.270855 |
    | 2000-01-01 16:30:00 | 6.707979  6.707979  6.707979  6.707979  6.707979  6.707979  6.707979    6.707979 | 395.750601 | 395.750601  395.750601  395.750601  395.750601  395.750601  395.750601    395.750601 | 395.750601 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 295.750601 | 395.750601 |
    | 2000-01-01 17:00:00 | 6.950425  6.950425  6.950425  6.950425  6.950425  6.950425  6.950425    6.950425 | 441.664962 | 441.664962  441.664962  441.664962  441.664962  441.664962  441.664962    441.664962 | 441.664962 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 341.664962 | 441.664962 |
    | 2000-01-01 17:30:00 | 7.185361  7.185361  7.185361  7.185361  7.185361  7.185361  7.185361    7.185361 | 489.279607 | 489.279607  489.279607  489.279607  489.279607  489.279607  489.279607    489.279607 | 489.279607 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 389.279607 | 489.279607 |
    | 2000-01-01 18:00:00 | 7.411028  7.411028  7.411028  7.411028  7.411028  7.411028  7.411028    7.411028 |  537.77215 |  537.77215   537.77215   537.77215   537.77215   537.77215   537.77215     537.77215 |  537.77215 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  437.77215 |  537.77215 |
    | 2000-01-01 18:30:00 | 7.625465  7.625465  7.625465  7.625465  7.625465  7.625465  7.625465    7.625465 | 586.262607 | 586.262607  586.262607  586.262607  586.262607  586.262607  586.262607    586.262607 | 586.262607 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 486.262607 | 586.262607 |
    | 2000-01-01 19:00:00 | 7.826711  7.826711  7.826711  7.826711  7.826711  7.826711  7.826711    7.826711 | 633.846018 | 633.846018  633.846018  633.846018  633.846018  633.846018  633.846018    633.846018 | 633.846018 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 533.846018 | 633.846018 |
    | 2000-01-01 19:30:00 | 8.012667  8.012667  8.012667  8.012667  8.012667  8.012667  8.012667    8.012667 | 679.625318 | 679.625318  679.625318  679.625318  679.625318  679.625318  679.625318    679.625318 | 679.625318 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 579.625318 | 679.625318 |
    | 2000-01-01 20:00:00 |  8.17845   8.17845   8.17845   8.17845   8.17845   8.17845   8.17845     8.17845 | 722.742733 | 722.742733  722.742733  722.742733  722.742733  722.742733  722.742733    722.742733 | 722.742733 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 622.742733 | 722.742733 |
    | 2000-01-01 20:30:00 | 8.325609  8.325609  8.325609  8.325609  8.325609  8.325609  8.325609    8.325609 | 762.408181 | 762.408181  762.408181  762.408181  762.408181  762.408181  762.408181    762.408181 | 762.408181 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 662.408181 | 762.408181 |
    | 2000-01-01 21:00:00 | 8.453057  8.453057  8.453057  8.453057  8.453057  8.453057  8.453057    8.453057 | 797.923464 | 797.923464  797.923464  797.923464  797.923464  797.923464  797.923464    797.923464 | 797.923464 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 697.923464 | 797.923464 |
    | 2000-01-01 21:30:00 | 8.560309  8.560309  8.560309  8.560309  8.560309  8.560309  8.560309    8.560309 | 828.701418 | 828.701418  828.701418  828.701418  828.701418  828.701418  828.701418    828.701418 | 828.701418 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 728.701418 | 828.701418 |
    | 2000-01-01 22:00:00 | 8.647261  8.647261  8.647261  8.647261  8.647261  8.647261  8.647261    8.647261 |  854.27953 |  854.27953   854.27953   854.27953   854.27953   854.27953   854.27953     854.27953 |  854.27953 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  754.27953 |  854.27953 |
    | 2000-01-01 22:30:00 | 8.714079  8.714079  8.714079  8.714079  8.714079  8.714079  8.714079    8.714079 | 874.327871 | 874.327871  874.327871  874.327871  874.327871  874.327871  874.327871    874.327871 | 874.327871 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 774.327871 | 874.327871 |
    | 2000-01-01 23:00:00 | 8.761117  8.761117  8.761117  8.761117  8.761117  8.761117  8.761117    8.761117 | 888.651531 | 888.651531  888.651531  888.651531  888.651531  888.651531  888.651531    888.651531 | 888.651531 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 788.651531 | 888.651531 |
    | 2000-01-01 23:30:00 | 8.788879  8.788879  8.788879  8.788879  8.788879  8.788879  8.788879    8.788879 | 897.187983 | 897.187983  897.187983  897.187983  897.187983  897.187983  897.187983    897.187983 | 897.187983 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 797.187983 | 897.187983 |
    | 2000-01-02 00:00:00 | 8.797981  8.797981  8.797981  8.797981  8.797981  8.797981  8.797981    8.797981 |      900.0 |      900.0       900.0       900.0       900.0       900.0       900.0         900.0 |      900.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |      800.0 |      900.0 |
    | 2000-01-02 00:30:00 | 8.789129  8.789129  8.789129  8.789129  8.789129  8.789129  8.789129    8.789129 | 897.264896 | 897.264896  897.264896  897.264896  897.264896  897.264896  897.264896    897.264896 | 897.264896 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 797.264896 | 897.264896 |
    | 2000-01-02 01:00:00 | 8.763106  8.763106  8.763106  8.763106  8.763106  8.763106  8.763106    8.763106 | 889.260928 | 889.260928  889.260928  889.260928  889.260928  889.260928  889.260928    889.260928 | 889.260928 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 789.260928 | 889.260928 |
    | 2000-01-02 01:30:00 |  8.72076   8.72076   8.72076   8.72076   8.72076   8.72076   8.72076     8.72076 | 876.351729 | 876.351729  876.351729  876.351729  876.351729  876.351729  876.351729    876.351729 | 876.351729 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 776.351729 | 876.351729 |
    | 2000-01-02 02:00:00 | 8.662996  8.662996  8.662996  8.662996  8.662996  8.662996  8.662996    8.662996 | 858.969595 | 858.969595  858.969595  858.969595  858.969595  858.969595  858.969595    858.969595 | 858.969595 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 758.969595 | 858.969595 |
    | 2000-01-02 02:30:00 | 8.590774  8.590774  8.590774  8.590774  8.590774  8.590774  8.590774    8.590774 |   837.5984 |   837.5984    837.5984    837.5984    837.5984    837.5984    837.5984      837.5984 |   837.5984 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   737.5984 |   837.5984 |
    | 2000-01-02 03:00:00 |  8.50511   8.50511   8.50511   8.50511   8.50511   8.50511   8.50511     8.50511 | 812.756821 | 812.756821  812.756821  812.756821  812.756821  812.756821  812.756821    812.756821 | 812.756821 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 712.756821 | 812.756821 |
    | 2000-01-02 03:30:00 | 8.407082  8.407082  8.407082  8.407082  8.407082  8.407082  8.407082    8.407082 | 784.982427 | 784.982427  784.982427  784.982427  784.982427  784.982427  784.982427    784.982427 | 784.982427 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 684.982427 | 784.982427 |
    | 2000-01-02 04:00:00 | 8.297843  8.297843  8.297843  8.297843  8.297843  8.297843  8.297843    8.297843 | 754.817086 | 754.817086  754.817086  754.817086  754.817086  754.817086  754.817086    754.817086 | 754.817086 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 654.817086 | 754.817086 |
    | 2000-01-02 04:30:00 | 8.178644  8.178644  8.178644  8.178644  8.178644  8.178644  8.178644    8.178644 |  722.79402 |  722.79402   722.79402   722.79402   722.79402   722.79402   722.79402     722.79402 |  722.79402 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  622.79402 |  722.79402 |
    | 2000-01-02 05:00:00 | 8.050875  8.050875  8.050875  8.050875  8.050875  8.050875  8.050875    8.050875 | 689.426711 | 689.426711  689.426711  689.426711  689.426711  689.426711  689.426711    689.426711 | 689.426711 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 589.426711 | 689.426711 |
    | 2000-01-02 05:30:00 | 7.914417  7.914417  7.914417  7.914417  7.914417  7.914417  7.914417    7.914417 | 655.199767 | 655.199767  655.199767  655.199767  655.199767  655.199767  655.199767    655.199767 | 655.199767 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 555.199767 | 655.199767 |
    | 2000-01-02 06:00:00 | 7.771362  7.771362  7.771362  7.771362  7.771362  7.771362  7.771362    7.771362 | 620.561767 | 620.561767  620.561767  620.561767  620.561767  620.561767  620.561767    620.561767 | 620.561767 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 520.561767 | 620.561767 |
    | 2000-01-02 06:30:00 | 7.623984  7.623984  7.623984  7.623984  7.623984  7.623984  7.623984    7.623984 | 585.920022 | 585.920022  585.920022  585.920022  585.920022  585.920022  585.920022    585.920022 | 585.920022 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 485.920022 | 585.920022 |
    | 2000-01-02 07:00:00 | 7.473399  7.473399  7.473399  7.473399  7.473399  7.473399  7.473399    7.473399 |  551.63713 |  551.63713   551.63713   551.63713   551.63713   551.63713   551.63713     551.63713 |  551.63713 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  451.63713 |  551.63713 |
    | 2000-01-02 07:30:00 | 7.320628  7.320628  7.320628  7.320628  7.320628  7.320628  7.320628    7.320628 | 518.029154 | 518.029154  518.029154  518.029154  518.029154  518.029154  518.029154    518.029154 | 518.029154 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 418.029154 | 518.029154 |
    | 2000-01-02 08:00:00 | 7.166586  7.166586  7.166586  7.166586  7.166586  7.166586  7.166586    7.166586 | 485.365238 | 485.365238  485.365238  485.365238  485.365238  485.365238  485.365238    485.365238 | 485.365238 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 385.365238 | 485.365238 |
    | 2000-01-02 08:30:00 | 7.012072  7.012072  7.012072  7.012072  7.012072  7.012072  7.012072    7.012072 |  453.86844 |  453.86844   453.86844   453.86844   453.86844   453.86844   453.86844     453.86844 |  453.86844 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  353.86844 |  453.86844 |
    | 2000-01-02 09:00:00 |  6.85775   6.85775   6.85775   6.85775   6.85775   6.85775   6.85775     6.85775 | 423.717566 | 423.717566  423.717566  423.717566  423.717566  423.717566  423.717566    423.717566 | 423.717566 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 323.717566 | 423.717566 |
    | 2000-01-02 09:30:00 | 6.704133  6.704133  6.704133  6.704133  6.704133  6.704133  6.704133    6.704133 | 395.049792 | 395.049792  395.049792  395.049792  395.049792  395.049792  395.049792    395.049792 | 395.049792 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 295.049792 | 395.049792 |
    | 2000-01-02 10:00:00 | 6.551548  6.551548  6.551548  6.551548  6.551548  6.551548  6.551548    6.551548 | 367.963873 | 367.963873  367.963873  367.963873  367.963873  367.963873  367.963873    367.963873 | 367.963873 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 267.963873 | 367.963873 |
    | 2000-01-02 10:30:00 | 6.400068  6.400068  6.400068  6.400068  6.400068  6.400068  6.400068    6.400068 | 342.523746 | 342.523746  342.523746  342.523746  342.523746  342.523746  342.523746    342.523746 | 342.523746 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 242.523746 | 342.523746 |
    | 2000-01-02 11:00:00 | 6.249366  6.249366  6.249366  6.249366  6.249366  6.249366  6.249366    6.249366 | 318.762374 | 318.762374  318.762374  318.762374  318.762374  318.762374  318.762374    318.762374 | 318.762374 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 218.762374 | 318.762374 |
    | 2000-01-02 11:30:00 | 6.098244  6.098244  6.098244  6.098244  6.098244  6.098244  6.098244    6.098244 |  296.68568 |  296.68568   296.68568   296.68568   296.68568   296.68568   296.68568     296.68568 |  296.68568 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  196.68568 |  296.68568 |
    | 2000-01-02 12:00:00 | 5.931106  5.931106  5.931106  5.931106  5.931106  5.931106  5.931106    5.931106 | 276.276442 | 276.276442  276.276442  276.276442  276.276442  276.276442  276.276442    276.276442 | 276.276442 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 176.276442 | 276.276442 |
    | 2000-01-02 12:30:00 | 5.747229  5.747229  5.747229  5.747229  5.747229  5.747229  5.747229    5.747229 | 257.498082 | 257.498082  257.498082  257.498082  257.498082  257.498082  257.498082    257.498082 | 257.498082 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 157.498082 | 257.498082 |
    | 2000-01-02 13:00:00 | 5.571543  5.571543  5.571543  5.571543  5.571543  5.571543  5.571543    5.571543 | 240.298231 | 240.298231  240.298231  240.298231  240.298231  240.298231  240.298231    240.298231 | 240.298231 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 140.298231 | 240.298231 |
    | 2000-01-02 13:30:00 |  5.40456   5.40456   5.40456   5.40456   5.40456   5.40456   5.40456     5.40456 | 224.612057 | 224.612057  224.612057  224.612057  224.612057  224.612057  224.612057    224.612057 | 224.612057 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 124.612057 | 224.612057 |
    | 2000-01-02 14:00:00 | 5.246686  5.246686  5.246686  5.246686  5.246686  5.246686  5.246686    5.246686 | 210.365288 | 210.365288  210.365288  210.365288  210.365288  210.365288  210.365288    210.365288 | 210.365288 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 | 110.365288 | 210.365288 |
    | 2000-01-02 14:30:00 | 5.098213  5.098213  5.098213  5.098213  5.098213  5.098213  5.098213    5.098213 |  197.47692 |  197.47692   197.47692   197.47692   197.47692   197.47692   197.47692     197.47692 |  197.47692 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   97.47692 |  197.47692 |
    | 2000-01-02 15:00:00 | 4.959327  4.959327  4.959327  4.959327  4.959327  4.959327  4.959327    4.959327 | 185.861607 | 185.861607  185.861607  185.861607  185.861607  185.861607  185.861607    185.861607 | 185.861607 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  85.861607 | 185.861607 |
    | 2000-01-02 15:30:00 | 4.830099  4.830099  4.830099  4.830099  4.830099  4.830099  4.830099    4.830099 | 175.431719 | 175.431719  175.431719  175.431719  175.431719  175.431719  175.431719    175.431719 | 175.431719 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  75.431719 | 175.431719 |
    | 2000-01-02 16:00:00 | 4.710502  4.710502  4.710502  4.710502  4.710502  4.710502  4.710502    4.710502 | 166.099094 | 166.099094  166.099094  166.099094  166.099094  166.099094  166.099094    166.099094 | 166.099094 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  66.099094 | 166.099094 |
    | 2000-01-02 16:30:00 | 4.600405  4.600405  4.600405  4.600405  4.600405  4.600405  4.600405    4.600405 | 157.776495 | 157.776495  157.776495  157.776495  157.776495  157.776495  157.776495    157.776495 | 157.776495 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  57.776495 | 157.776495 |
    | 2000-01-02 17:00:00 | 4.499591  4.499591  4.499591  4.499591  4.499591  4.499591  4.499591    4.499591 | 150.378786 | 150.378786  150.378786  150.378786  150.378786  150.378786  150.378786    150.378786 | 150.378786 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  50.378786 | 150.378786 |
    | 2000-01-02 17:30:00 | 4.407758  4.407758  4.407758  4.407758  4.407758  4.407758  4.407758    4.407758 | 143.823868 | 143.823868  143.823868  143.823868  143.823868  143.823868  143.823868    143.823868 | 143.823868 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  43.823868 | 143.823868 |
    | 2000-01-02 18:00:00 | 4.324535  4.324535  4.324535  4.324535  4.324535  4.324535  4.324535    4.324535 | 138.033385 | 138.033385  138.033385  138.033385  138.033385  138.033385  138.033385    138.033385 | 138.033385 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  38.033385 | 138.033385 |
    | 2000-01-02 18:30:00 | 4.249496  4.249496  4.249496  4.249496  4.249496  4.249496  4.249496    4.249496 | 132.933235 | 132.933235  132.933235  132.933235  132.933235  132.933235  132.933235    132.933235 | 132.933235 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  32.933235 | 132.933235 |
    | 2000-01-02 19:00:00 | 4.182165  4.182165  4.182165  4.182165  4.182165  4.182165  4.182165    4.182165 | 128.453911 | 128.453911  128.453911  128.453911  128.453911  128.453911  128.453911    128.453911 | 128.453911 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  28.453911 | 128.453911 |
    | 2000-01-02 19:30:00 | 4.122039  4.122039  4.122039  4.122039  4.122039  4.122039  4.122039    4.122039 | 124.530701 | 124.530701  124.530701  124.530701  124.530701  124.530701  124.530701    124.530701 | 124.530701 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  24.530701 | 124.530701 |
    | 2000-01-02 20:00:00 | 4.068593  4.068593  4.068593  4.068593  4.068593  4.068593  4.068593    4.068593 | 121.103761 | 121.103761  121.103761  121.103761  121.103761  121.103761  121.103761    121.103761 | 121.103761 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  21.103761 | 121.103761 |
    | 2000-01-02 20:30:00 | 4.021292  4.021292  4.021292  4.021292  4.021292  4.021292  4.021292    4.021292 | 118.118088 | 118.118088  118.118088  118.118088  118.118088  118.118088  118.118088    118.118088 | 118.118088 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  18.118088 | 118.118088 |
    | 2000-01-02 21:00:00 | 3.979608  3.979608  3.979608  3.979608  3.979608  3.979608  3.979608    3.979608 | 115.523427 | 115.523427  115.523427  115.523427  115.523427  115.523427  115.523427    115.523427 | 115.523427 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  15.523427 | 115.523427 |
    | 2000-01-02 21:30:00 | 3.943022  3.943022  3.943022  3.943022  3.943022  3.943022  3.943022    3.943022 | 113.274094 | 113.274094  113.274094  113.274094  113.274094  113.274094  113.274094    113.274094 | 113.274094 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  13.274094 | 113.274094 |
    | 2000-01-02 22:00:00 | 3.911032  3.911032  3.911032  3.911032  3.911032  3.911032  3.911032    3.911032 | 111.328777 | 111.328777  111.328777  111.328777  111.328777  111.328777  111.328777    111.328777 | 111.328777 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |  11.328777 | 111.328777 |
    | 2000-01-02 22:30:00 | 3.883164  3.883164  3.883164  3.883164  3.883164  3.883164  3.883164    3.883164 | 109.650285 | 109.650285  109.650285  109.650285  109.650285  109.650285  109.650285    109.650285 | 109.650285 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   9.650285 | 109.650285 |
    | 2000-01-02 23:00:00 |  3.85897   3.85897   3.85897   3.85897   3.85897   3.85897   3.85897     3.85897 | 108.205287 | 108.205287  108.205287  108.205287  108.205287  108.205287  108.205287    108.205287 | 108.205287 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   8.205287 | 108.205287 |
    | 2000-01-02 23:30:00 | 3.838034  3.838034  3.838034  3.838034  3.838034  3.838034  3.838034    3.838034 | 106.964036 | 106.964036  106.964036  106.964036  106.964036  106.964036  106.964036    106.964036 | 106.964036 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   6.964036 | 106.964036 |
    | 2000-01-03 00:00:00 | 3.819975  3.819975  3.819975  3.819975  3.819975  3.819975  3.819975    3.819975 | 105.900084 | 105.900084  105.900084  105.900084  105.900084  105.900084  105.900084    105.900084 | 105.900084 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   5.900084 | 105.900084 |
    | 2000-01-03 00:30:00 | 3.804443  3.804443  3.804443  3.804443  3.804443  3.804443  3.804443    3.804443 |  104.99001 |  104.99001   104.99001   104.99001   104.99001   104.99001   104.99001     104.99001 |  104.99001 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |    4.99001 |  104.99001 |
    | 2000-01-03 01:00:00 | 3.791121  3.791121  3.791121  3.791121  3.791121  3.791121  3.791121    3.791121 | 104.213141 | 104.213141  104.213141  104.213141  104.213141  104.213141  104.213141    104.213141 | 104.213141 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   4.213141 | 104.213141 |
    | 2000-01-03 01:30:00 | 3.779725  3.779725  3.779725  3.779725  3.779725  3.779725  3.779725    3.779725 | 103.551298 | 103.551298  103.551298  103.551298  103.551298  103.551298  103.551298    103.551298 | 103.551298 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   3.551298 | 103.551298 |
    | 2000-01-03 02:00:00 | 3.770001  3.770001  3.770001  3.770001  3.770001  3.770001  3.770001    3.770001 | 102.988541 | 102.988541  102.988541  102.988541  102.988541  102.988541  102.988541    102.988541 | 102.988541 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   2.988541 | 102.988541 |
    | 2000-01-03 02:30:00 | 3.761725  3.761725  3.761725  3.761725  3.761725  3.761725  3.761725    3.761725 |  102.51094 |  102.51094   102.51094   102.51094   102.51094   102.51094   102.51094     102.51094 |  102.51094 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |    2.51094 |  102.51094 |
    | 2000-01-03 03:00:00 | 3.754696  3.754696  3.754696  3.754696  3.754696  3.754696  3.754696    3.754696 | 102.106358 | 102.106358  102.106358  102.106358  102.106358  102.106358  102.106358    102.106358 | 102.106358 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   2.106358 | 102.106358 |
    | 2000-01-03 03:30:00 | 3.748739  3.748739  3.748739  3.748739  3.748739  3.748739  3.748739    3.748739 | 101.764251 | 101.764251  101.764251  101.764251  101.764251  101.764251  101.764251    101.764251 | 101.764251 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   1.764251 | 101.764251 |
    | 2000-01-03 04:00:00 | 3.743702  3.743702  3.743702  3.743702  3.743702  3.743702  3.743702    3.743702 |  101.47548 |  101.47548   101.47548   101.47548   101.47548   101.47548   101.47548     101.47548 |  101.47548 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |    1.47548 |  101.47548 |
    | 2000-01-03 04:30:00 | 3.739451  3.739451  3.739451  3.739451  3.739451  3.739451  3.739451    3.739451 | 101.232151 | 101.232151  101.232151  101.232151  101.232151  101.232151  101.232151    101.232151 | 101.232151 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   1.232151 | 101.232151 |
    | 2000-01-03 05:00:00 | 3.735871  3.735871  3.735871  3.735871  3.735871  3.735871  3.735871    3.735871 | 101.027458 | 101.027458  101.027458  101.027458  101.027458  101.027458  101.027458    101.027458 | 101.027458 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   1.027458 | 101.027458 |
    | 2000-01-03 05:30:00 |  3.73286   3.73286   3.73286   3.73286   3.73286   3.73286   3.73286     3.73286 | 100.855551 | 100.855551  100.855551  100.855551  100.855551  100.855551  100.855551    100.855551 | 100.855551 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.855551 | 100.855551 |
    | 2000-01-03 06:00:00 | 3.730334  3.730334  3.730334  3.730334  3.730334  3.730334  3.730334    3.730334 | 100.711411 | 100.711411  100.711411  100.711411  100.711411  100.711411  100.711411    100.711411 | 100.711411 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.711411 | 100.711411 |
    | 2000-01-03 06:30:00 | 3.728217  3.728217  3.728217  3.728217  3.728217  3.728217  3.728217    3.728217 | 100.590745 | 100.590745  100.590745  100.590745  100.590745  100.590745  100.590745    100.590745 | 100.590745 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.590745 | 100.590745 |
    | 2000-01-03 07:00:00 | 3.726447  3.726447  3.726447  3.726447  3.726447  3.726447  3.726447    3.726447 | 100.489885 | 100.489885  100.489885  100.489885  100.489885  100.489885  100.489885    100.489885 | 100.489885 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.489885 | 100.489885 |
    | 2000-01-03 07:30:00 | 3.724968  3.724968  3.724968  3.724968  3.724968  3.724968  3.724968    3.724968 | 100.405708 | 100.405708  100.405708  100.405708  100.405708  100.405708  100.405708    100.405708 | 100.405708 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.405708 | 100.405708 |
    | 2000-01-03 08:00:00 | 3.723736  3.723736  3.723736  3.723736  3.723736  3.723736  3.723736    3.723736 |  100.33556 |  100.33556   100.33556   100.33556   100.33556   100.33556   100.33556     100.33556 |  100.33556 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |    0.33556 |  100.33556 |
    | 2000-01-03 08:30:00 |  3.72271   3.72271   3.72271   3.72271   3.72271   3.72271   3.72271     3.72271 | 100.277186 | 100.277186  100.277186  100.277186  100.277186  100.277186  100.277186    100.277186 | 100.277186 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.277186 | 100.277186 |
    | 2000-01-03 09:00:00 | 3.721857  3.721857  3.721857  3.721857  3.721857  3.721857  3.721857    3.721857 | 100.228681 | 100.228681  100.228681  100.228681  100.228681  100.228681  100.228681    100.228681 | 100.228681 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.228681 | 100.228681 |
    | 2000-01-03 09:30:00 | 3.721149  3.721149  3.721149  3.721149  3.721149  3.721149  3.721149    3.721149 | 100.188431 | 100.188431  100.188431  100.188431  100.188431  100.188431  100.188431    100.188431 | 100.188431 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.188431 | 100.188431 |
    | 2000-01-03 10:00:00 | 3.720562  3.720562  3.720562  3.720562  3.720562  3.720562  3.720562    3.720562 | 100.155078 | 100.155078  100.155078  100.155078  100.155078  100.155078  100.155078    100.155078 | 100.155078 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.155078 | 100.155078 |
    | 2000-01-03 10:30:00 | 3.720077  3.720077  3.720077  3.720077  3.720077  3.720077  3.720077    3.720077 | 100.127477 | 100.127477  100.127477  100.127477  100.127477  100.127477  100.127477    100.127477 | 100.127477 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.127477 | 100.127477 |
    | 2000-01-03 11:00:00 | 3.719675  3.719675  3.719675  3.719675  3.719675  3.719675  3.719675    3.719675 | 100.104666 | 100.104666  100.104666  100.104666  100.104666  100.104666  100.104666    100.104666 | 100.104666 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.104666 | 100.104666 |
    | 2000-01-03 11:30:00 | 3.719344  3.719344  3.719344  3.719344  3.719344  3.719344  3.719344    3.719344 | 100.085838 | 100.085838  100.085838  100.085838  100.085838  100.085838  100.085838    100.085838 | 100.085838 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.085838 | 100.085838 |
    | 2000-01-03 12:00:00 | 3.719071  3.719071  3.719071  3.719071  3.719071  3.719071  3.719071    3.719071 | 100.070317 | 100.070317  100.070317  100.070317  100.070317  100.070317  100.070317    100.070317 | 100.070317 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.070317 | 100.070317 |
    | 2000-01-03 12:30:00 | 3.718846  3.718846  3.718846  3.718846  3.718846  3.718846  3.718846    3.718846 | 100.057539 | 100.057539  100.057539  100.057539  100.057539  100.057539  100.057539    100.057539 | 100.057539 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.057539 | 100.057539 |
    | 2000-01-03 13:00:00 | 3.718661  3.718661  3.718661  3.718661  3.718661  3.718661  3.718661    3.718661 | 100.047032 | 100.047032  100.047032  100.047032  100.047032  100.047032  100.047032    100.047032 | 100.047032 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.047032 | 100.047032 |
    | 2000-01-03 13:30:00 | 3.718509  3.718509  3.718509  3.718509  3.718509  3.718509  3.718509    3.718509 | 100.038402 | 100.038402  100.038402  100.038402  100.038402  100.038402  100.038402    100.038402 | 100.038402 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.038402 | 100.038402 |
    | 2000-01-03 14:00:00 | 3.718384  3.718384  3.718384  3.718384  3.718384  3.718384  3.718384    3.718384 | 100.031322 | 100.031322  100.031322  100.031322  100.031322  100.031322  100.031322    100.031322 | 100.031322 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.031322 | 100.031322 |
    | 2000-01-03 14:30:00 | 3.718282  3.718282  3.718282  3.718282  3.718282  3.718282  3.718282    3.718282 | 100.025521 | 100.025521  100.025521  100.025521  100.025521  100.025521  100.025521    100.025521 | 100.025521 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.025521 | 100.025521 |
    | 2000-01-03 15:00:00 | 3.718199  3.718199  3.718199  3.718199  3.718199  3.718199  3.718199    3.718199 | 100.020773 | 100.020773  100.020773  100.020773  100.020773  100.020773  100.020773    100.020773 | 100.020773 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.020773 | 100.020773 |
    | 2000-01-03 15:30:00 |  3.71813   3.71813   3.71813   3.71813   3.71813   3.71813   3.71813     3.71813 | 100.016892 | 100.016892  100.016892  100.016892  100.016892  100.016892  100.016892    100.016892 | 100.016892 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.016892 | 100.016892 |
    | 2000-01-03 16:00:00 | 3.718074  3.718074  3.718074  3.718074  3.718074  3.718074  3.718074    3.718074 | 100.013722 | 100.013722  100.013722  100.013722  100.013722  100.013722  100.013722    100.013722 | 100.013722 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.013722 | 100.013722 |
    | 2000-01-03 16:30:00 | 3.718029  3.718029  3.718029  3.718029  3.718029  3.718029  3.718029    3.718029 | 100.011136 | 100.011136  100.011136  100.011136  100.011136  100.011136  100.011136    100.011136 | 100.011136 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.011136 | 100.011136 |
    | 2000-01-03 17:00:00 | 3.717992  3.717992  3.717992  3.717992  3.717992  3.717992  3.717992    3.717992 | 100.009028 | 100.009028  100.009028  100.009028  100.009028  100.009028  100.009028    100.009028 | 100.009028 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.009028 | 100.009028 |
    | 2000-01-03 17:30:00 | 3.717962  3.717962  3.717962  3.717962  3.717962  3.717962  3.717962    3.717962 | 100.007313 | 100.007313  100.007313  100.007313  100.007313  100.007313  100.007313    100.007313 | 100.007313 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.007313 | 100.007313 |
    | 2000-01-03 18:00:00 | 3.717937  3.717937  3.717937  3.717937  3.717937  3.717937  3.717937    3.717937 | 100.005918 | 100.005918  100.005918  100.005918  100.005918  100.005918  100.005918    100.005918 | 100.005918 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.005918 | 100.005918 |
    | 2000-01-03 18:30:00 | 3.717917  3.717917  3.717917  3.717917  3.717917  3.717917  3.717917    3.717917 | 100.004785 | 100.004785  100.004785  100.004785  100.004785  100.004785  100.004785    100.004785 | 100.004785 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.004785 | 100.004785 |
    | 2000-01-03 19:00:00 | 3.717901  3.717901  3.717901  3.717901  3.717901  3.717901  3.717901    3.717901 | 100.003865 | 100.003865  100.003865  100.003865  100.003865  100.003865  100.003865    100.003865 | 100.003865 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.003865 | 100.003865 |
    | 2000-01-03 19:30:00 | 3.717888  3.717888  3.717888  3.717888  3.717888  3.717888  3.717888    3.717888 | 100.003119 | 100.003119  100.003119  100.003119  100.003119  100.003119  100.003119    100.003119 | 100.003119 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.003119 | 100.003119 |
    | 2000-01-03 20:00:00 | 3.717877  3.717877  3.717877  3.717877  3.717877  3.717877  3.717877    3.717877 | 100.002515 | 100.002515  100.002515  100.002515  100.002515  100.002515  100.002515    100.002515 | 100.002515 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.002515 | 100.002515 |
    | 2000-01-03 20:30:00 | 3.717868  3.717868  3.717868  3.717868  3.717868  3.717868  3.717868    3.717868 | 100.002026 | 100.002026  100.002026  100.002026  100.002026  100.002026  100.002026    100.002026 | 100.002026 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.002026 | 100.002026 |
    | 2000-01-03 21:00:00 | 3.717861  3.717861  3.717861  3.717861  3.717861  3.717861  3.717861    3.717861 | 100.001631 | 100.001631  100.001631  100.001631  100.001631  100.001631  100.001631    100.001631 | 100.001631 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.001631 | 100.001631 |
    | 2000-01-03 21:30:00 | 3.717856  3.717856  3.717856  3.717856  3.717856  3.717856  3.717856    3.717856 | 100.001312 | 100.001312  100.001312  100.001312  100.001312  100.001312  100.001312    100.001312 | 100.001312 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.001312 | 100.001312 |
    | 2000-01-03 22:00:00 | 3.717851  3.717851  3.717851  3.717851  3.717851  3.717851  3.717851    3.717851 | 100.001054 | 100.001054  100.001054  100.001054  100.001054  100.001054  100.001054    100.001054 | 100.001054 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.001054 | 100.001054 |
    | 2000-01-03 22:30:00 | 3.717848  3.717848  3.717848  3.717848  3.717848  3.717848  3.717848    3.717848 | 100.000846 | 100.000846  100.000846  100.000846  100.000846  100.000846  100.000846    100.000846 | 100.000846 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000846 | 100.000846 |
    | 2000-01-03 23:00:00 | 3.717845  3.717845  3.717845  3.717845  3.717845  3.717845  3.717845    3.717845 | 100.000679 | 100.000679  100.000679  100.000679  100.000679  100.000679  100.000679    100.000679 | 100.000679 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000679 | 100.000679 |
    | 2000-01-03 23:30:00 | 3.717842  3.717842  3.717842  3.717842  3.717842  3.717842  3.717842    3.717842 | 100.000544 | 100.000544  100.000544  100.000544  100.000544  100.000544  100.000544    100.000544 | 100.000544 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000544 | 100.000544 |
    | 2000-01-04 00:00:00 |  3.71784   3.71784   3.71784   3.71784   3.71784   3.71784   3.71784     3.71784 | 100.000436 | 100.000436  100.000436  100.000436  100.000436  100.000436  100.000436    100.000436 | 100.000436 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000436 | 100.000436 |
    | 2000-01-04 00:30:00 | 3.717839  3.717839  3.717839  3.717839  3.717839  3.717839  3.717839    3.717839 | 100.000349 | 100.000349  100.000349  100.000349  100.000349  100.000349  100.000349    100.000349 | 100.000349 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000349 | 100.000349 |
    | 2000-01-04 01:00:00 | 3.717838  3.717838  3.717838  3.717838  3.717838  3.717838  3.717838    3.717838 | 100.000279 | 100.000279  100.000279  100.000279  100.000279  100.000279  100.000279    100.000279 | 100.000279 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000279 | 100.000279 |
    | 2000-01-04 01:30:00 | 3.717837  3.717837  3.717837  3.717837  3.717837  3.717837  3.717837    3.717837 | 100.000223 | 100.000223  100.000223  100.000223  100.000223  100.000223  100.000223    100.000223 | 100.000223 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000223 | 100.000223 |
    | 2000-01-04 02:00:00 | 3.717836  3.717836  3.717836  3.717836  3.717836  3.717836  3.717836    3.717836 | 100.000178 | 100.000178  100.000178  100.000178  100.000178  100.000178  100.000178    100.000178 | 100.000178 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000178 | 100.000178 |
    | 2000-01-04 02:30:00 | 3.717835  3.717835  3.717835  3.717835  3.717835  3.717835  3.717835    3.717835 | 100.000142 | 100.000142  100.000142  100.000142  100.000142  100.000142  100.000142    100.000142 | 100.000142 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000142 | 100.000142 |
    | 2000-01-04 03:00:00 | 3.717835  3.717835  3.717835  3.717835  3.717835  3.717835  3.717835    3.717835 | 100.000113 | 100.000113  100.000113  100.000113  100.000113  100.000113  100.000113    100.000113 | 100.000113 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000113 | 100.000113 |
    | 2000-01-04 03:30:00 | 3.717834  3.717834  3.717834  3.717834  3.717834  3.717834  3.717834    3.717834 |  100.00009 |  100.00009   100.00009   100.00009   100.00009   100.00009   100.00009     100.00009 |  100.00009 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |    0.00009 |  100.00009 |
    | 2000-01-04 04:00:00 | 3.717834  3.717834  3.717834  3.717834  3.717834  3.717834  3.717834    3.717834 | 100.000072 | 100.000072  100.000072  100.000072  100.000072  100.000072  100.000072    100.000072 | 100.000072 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000072 | 100.000072 |
    | 2000-01-04 04:30:00 | 3.717834  3.717834  3.717834  3.717834  3.717834  3.717834  3.717834    3.717834 | 100.000057 | 100.000057  100.000057  100.000057  100.000057  100.000057  100.000057    100.000057 | 100.000057 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000057 | 100.000057 |
    | 2000-01-04 05:00:00 | 3.717834  3.717834  3.717834  3.717834  3.717834  3.717834  3.717834    3.717834 | 100.000046 | 100.000046  100.000046  100.000046  100.000046  100.000046  100.000046    100.000046 | 100.000046 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000046 | 100.000046 |
    | 2000-01-04 05:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000036 | 100.000036  100.000036  100.000036  100.000036  100.000036  100.000036    100.000036 | 100.000036 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000036 | 100.000036 |
    | 2000-01-04 06:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000029 | 100.000029  100.000029  100.000029  100.000029  100.000029  100.000029    100.000029 | 100.000029 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000029 | 100.000029 |
    | 2000-01-04 06:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000023 | 100.000023  100.000023  100.000023  100.000023  100.000023  100.000023    100.000023 | 100.000023 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000023 | 100.000023 |
    | 2000-01-04 07:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000018 | 100.000018  100.000018  100.000018  100.000018  100.000018  100.000018    100.000018 | 100.000018 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000018 | 100.000018 |
    | 2000-01-04 07:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000014 | 100.000014  100.000014  100.000014  100.000014  100.000014  100.000014    100.000014 | 100.000014 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000014 | 100.000014 |
    | 2000-01-04 08:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000011 | 100.000011  100.000011  100.000011  100.000011  100.000011  100.000011    100.000011 | 100.000011 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000011 | 100.000011 |
    | 2000-01-04 08:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000009 | 100.000009  100.000009  100.000009  100.000009  100.000009  100.000009    100.000009 | 100.000009 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000009 | 100.000009 |
    | 2000-01-04 09:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000007 | 100.000007  100.000007  100.000007  100.000007  100.000007  100.000007    100.000007 | 100.000007 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000007 | 100.000007 |
    | 2000-01-04 09:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000006 | 100.000006  100.000006  100.000006  100.000006  100.000006  100.000006    100.000006 | 100.000006 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000006 | 100.000006 |
    | 2000-01-04 10:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000004 | 100.000004  100.000004  100.000004  100.000004  100.000004  100.000004    100.000004 | 100.000004 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000004 | 100.000004 |
    | 2000-01-04 10:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000004 | 100.000004  100.000004  100.000004  100.000004  100.000004  100.000004    100.000004 | 100.000004 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000004 | 100.000004 |
    | 2000-01-04 11:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000003 | 100.000003  100.000003  100.000003  100.000003  100.000003  100.000003    100.000003 | 100.000003 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000003 | 100.000003 |
    | 2000-01-04 11:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000002 | 100.000002  100.000002  100.000002  100.000002  100.000002  100.000002    100.000002 | 100.000002 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000002 | 100.000002 |
    | 2000-01-04 12:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000002 | 100.000002  100.000002  100.000002  100.000002  100.000002  100.000002    100.000002 | 100.000002 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000002 | 100.000002 |
    | 2000-01-04 12:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000001  100.000001  100.000001  100.000001  100.000001  100.000001    100.000001 | 100.000001 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 13:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000001  100.000001  100.000001  100.000001  100.000001  100.000001    100.000001 | 100.000001 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 13:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000001  100.000001  100.000001  100.000001  100.000001  100.000001    100.000001 | 100.000001 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 14:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000001  100.000001  100.000001  100.000001  100.000001  100.000001    100.000001 | 100.000001 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 14:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 | 100.000001 | 100.000001  100.000001  100.000001  100.000001  100.000001  100.000001    100.000001 | 100.000001 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 15:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 15:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 16:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 16:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 17:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 17:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 18:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 18:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 19:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 19:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 20:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 20:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 21:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 21:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 22:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 22:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 23:00:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 23:30:00 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833    3.717833 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0         100.0 |      100.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0          0.0 |  100.0 |        0.0 |      100.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _kinw_impl_euler_no_segments:

no segments
___________

However, if one wishes to forward any inflow directly as outflow (for testing purposes
or so), setting the number of segments to zero is preferable, as it sidesteps
unnecessary calculations:

.. integration-test::

    >>> nmbsegments(0)
    >>> model.prepare_states(100.0)
    >>> conditions = test("kinw_impl_euler_no_segment", get_conditions="2000-01-01")
    |                date | waterdepth |     inflow | internalflow |    outflow | watervolume | input1 |     input2 |     output |
    ------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 |          - | 100.000003 |            - | 100.000003 |           - |  100.0 |   0.000003 | 100.000003 |
    | 2000-01-01 03:30:00 |          - | 100.000029 |            - | 100.000029 |           - |  100.0 |   0.000029 | 100.000029 |
    | 2000-01-01 04:00:00 |          - | 100.000175 |            - | 100.000175 |           - |  100.0 |   0.000175 | 100.000175 |
    | 2000-01-01 04:30:00 |          - | 100.000826 |            - | 100.000826 |           - |  100.0 |   0.000826 | 100.000826 |
    | 2000-01-01 05:00:00 |          - | 100.003194 |            - | 100.003194 |           - |  100.0 |   0.003194 | 100.003194 |
    | 2000-01-01 05:30:00 |          - | 100.010515 |            - | 100.010515 |           - |  100.0 |   0.010515 | 100.010515 |
    | 2000-01-01 06:00:00 |          - | 100.030315 |            - | 100.030315 |           - |  100.0 |   0.030315 | 100.030315 |
    | 2000-01-01 06:30:00 |          - | 100.078179 |            - | 100.078179 |           - |  100.0 |   0.078179 | 100.078179 |
    | 2000-01-01 07:00:00 |          - | 100.183351 |            - | 100.183351 |           - |  100.0 |   0.183351 | 100.183351 |
    | 2000-01-01 07:30:00 |          - | 100.396214 |            - | 100.396214 |           - |  100.0 |   0.396214 | 100.396214 |
    | 2000-01-01 08:00:00 |          - | 100.797305 |            - | 100.797305 |           - |  100.0 |   0.797305 | 100.797305 |
    | 2000-01-01 08:30:00 |          - | 101.507032 |            - | 101.507032 |           - |  100.0 |   1.507032 | 101.507032 |
    | 2000-01-01 09:00:00 |          - | 102.694853 |            - | 102.694853 |           - |  100.0 |   2.694853 | 102.694853 |
    | 2000-01-01 09:30:00 |          - | 104.586356 |            - | 104.586356 |           - |  100.0 |   4.586356 | 104.586356 |
    | 2000-01-01 10:00:00 |          - | 107.466641 |            - | 107.466641 |           - |  100.0 |   7.466641 | 107.466641 |
    | 2000-01-01 10:30:00 |          - | 111.678559 |            - | 111.678559 |           - |  100.0 |  11.678559 | 111.678559 |
    | 2000-01-01 11:00:00 |          - | 117.614836 |            - | 117.614836 |           - |  100.0 |  17.614836 | 117.614836 |
    | 2000-01-01 11:30:00 |          - | 125.703736 |            - | 125.703736 |           - |  100.0 |  25.703736 | 125.703736 |
    | 2000-01-01 12:00:00 |          - | 136.388647 |            - | 136.388647 |           - |  100.0 |  36.388647 | 136.388647 |
    | 2000-01-01 12:30:00 |          - | 150.102726 |            - | 150.102726 |           - |  100.0 |  50.102726 | 150.102726 |
    | 2000-01-01 13:00:00 |          - | 167.240349 |            - | 167.240349 |           - |  100.0 |  67.240349 | 167.240349 |
    | 2000-01-01 13:30:00 |          - | 188.127569 |            - | 188.127569 |           - |  100.0 |  88.127569 | 188.127569 |
    | 2000-01-01 14:00:00 |          - | 212.993962 |            - | 212.993962 |           - |  100.0 | 112.993962 | 212.993962 |
    | 2000-01-01 14:30:00 |          - | 241.948195 |            - | 241.948195 |           - |  100.0 | 141.948195 | 241.948195 |
    | 2000-01-01 15:00:00 |          - |  274.95935 |            - |  274.95935 |           - |  100.0 |  174.95935 |  274.95935 |
    | 2000-01-01 15:30:00 |          - | 311.845554 |            - | 311.845554 |           - |  100.0 | 211.845554 | 311.845554 |
    | 2000-01-01 16:00:00 |          - | 352.270855 |            - | 352.270855 |           - |  100.0 | 252.270855 | 352.270855 |
    | 2000-01-01 16:30:00 |          - | 395.750601 |            - | 395.750601 |           - |  100.0 | 295.750601 | 395.750601 |
    | 2000-01-01 17:00:00 |          - | 441.664962 |            - | 441.664962 |           - |  100.0 | 341.664962 | 441.664962 |
    | 2000-01-01 17:30:00 |          - | 489.279607 |            - | 489.279607 |           - |  100.0 | 389.279607 | 489.279607 |
    | 2000-01-01 18:00:00 |          - |  537.77215 |            - |  537.77215 |           - |  100.0 |  437.77215 |  537.77215 |
    | 2000-01-01 18:30:00 |          - | 586.262607 |            - | 586.262607 |           - |  100.0 | 486.262607 | 586.262607 |
    | 2000-01-01 19:00:00 |          - | 633.846018 |            - | 633.846018 |           - |  100.0 | 533.846018 | 633.846018 |
    | 2000-01-01 19:30:00 |          - | 679.625318 |            - | 679.625318 |           - |  100.0 | 579.625318 | 679.625318 |
    | 2000-01-01 20:00:00 |          - | 722.742733 |            - | 722.742733 |           - |  100.0 | 622.742733 | 722.742733 |
    | 2000-01-01 20:30:00 |          - | 762.408181 |            - | 762.408181 |           - |  100.0 | 662.408181 | 762.408181 |
    | 2000-01-01 21:00:00 |          - | 797.923464 |            - | 797.923464 |           - |  100.0 | 697.923464 | 797.923464 |
    | 2000-01-01 21:30:00 |          - | 828.701418 |            - | 828.701418 |           - |  100.0 | 728.701418 | 828.701418 |
    | 2000-01-01 22:00:00 |          - |  854.27953 |            - |  854.27953 |           - |  100.0 |  754.27953 |  854.27953 |
    | 2000-01-01 22:30:00 |          - | 874.327871 |            - | 874.327871 |           - |  100.0 | 774.327871 | 874.327871 |
    | 2000-01-01 23:00:00 |          - | 888.651531 |            - | 888.651531 |           - |  100.0 | 788.651531 | 888.651531 |
    | 2000-01-01 23:30:00 |          - | 897.187983 |            - | 897.187983 |           - |  100.0 | 797.187983 | 897.187983 |
    | 2000-01-02 00:00:00 |          - |      900.0 |            - |      900.0 |           - |  100.0 |      800.0 |      900.0 |
    | 2000-01-02 00:30:00 |          - | 897.264896 |            - | 897.264896 |           - |  100.0 | 797.264896 | 897.264896 |
    | 2000-01-02 01:00:00 |          - | 889.260928 |            - | 889.260928 |           - |  100.0 | 789.260928 | 889.260928 |
    | 2000-01-02 01:30:00 |          - | 876.351729 |            - | 876.351729 |           - |  100.0 | 776.351729 | 876.351729 |
    | 2000-01-02 02:00:00 |          - | 858.969595 |            - | 858.969595 |           - |  100.0 | 758.969595 | 858.969595 |
    | 2000-01-02 02:30:00 |          - |   837.5984 |            - |   837.5984 |           - |  100.0 |   737.5984 |   837.5984 |
    | 2000-01-02 03:00:00 |          - | 812.756821 |            - | 812.756821 |           - |  100.0 | 712.756821 | 812.756821 |
    | 2000-01-02 03:30:00 |          - | 784.982427 |            - | 784.982427 |           - |  100.0 | 684.982427 | 784.982427 |
    | 2000-01-02 04:00:00 |          - | 754.817086 |            - | 754.817086 |           - |  100.0 | 654.817086 | 754.817086 |
    | 2000-01-02 04:30:00 |          - |  722.79402 |            - |  722.79402 |           - |  100.0 |  622.79402 |  722.79402 |
    | 2000-01-02 05:00:00 |          - | 689.426711 |            - | 689.426711 |           - |  100.0 | 589.426711 | 689.426711 |
    | 2000-01-02 05:30:00 |          - | 655.199767 |            - | 655.199767 |           - |  100.0 | 555.199767 | 655.199767 |
    | 2000-01-02 06:00:00 |          - | 620.561767 |            - | 620.561767 |           - |  100.0 | 520.561767 | 620.561767 |
    | 2000-01-02 06:30:00 |          - | 585.920022 |            - | 585.920022 |           - |  100.0 | 485.920022 | 585.920022 |
    | 2000-01-02 07:00:00 |          - |  551.63713 |            - |  551.63713 |           - |  100.0 |  451.63713 |  551.63713 |
    | 2000-01-02 07:30:00 |          - | 518.029154 |            - | 518.029154 |           - |  100.0 | 418.029154 | 518.029154 |
    | 2000-01-02 08:00:00 |          - | 485.365238 |            - | 485.365238 |           - |  100.0 | 385.365238 | 485.365238 |
    | 2000-01-02 08:30:00 |          - |  453.86844 |            - |  453.86844 |           - |  100.0 |  353.86844 |  453.86844 |
    | 2000-01-02 09:00:00 |          - | 423.717566 |            - | 423.717566 |           - |  100.0 | 323.717566 | 423.717566 |
    | 2000-01-02 09:30:00 |          - | 395.049792 |            - | 395.049792 |           - |  100.0 | 295.049792 | 395.049792 |
    | 2000-01-02 10:00:00 |          - | 367.963873 |            - | 367.963873 |           - |  100.0 | 267.963873 | 367.963873 |
    | 2000-01-02 10:30:00 |          - | 342.523746 |            - | 342.523746 |           - |  100.0 | 242.523746 | 342.523746 |
    | 2000-01-02 11:00:00 |          - | 318.762374 |            - | 318.762374 |           - |  100.0 | 218.762374 | 318.762374 |
    | 2000-01-02 11:30:00 |          - |  296.68568 |            - |  296.68568 |           - |  100.0 |  196.68568 |  296.68568 |
    | 2000-01-02 12:00:00 |          - | 276.276442 |            - | 276.276442 |           - |  100.0 | 176.276442 | 276.276442 |
    | 2000-01-02 12:30:00 |          - | 257.498082 |            - | 257.498082 |           - |  100.0 | 157.498082 | 257.498082 |
    | 2000-01-02 13:00:00 |          - | 240.298231 |            - | 240.298231 |           - |  100.0 | 140.298231 | 240.298231 |
    | 2000-01-02 13:30:00 |          - | 224.612057 |            - | 224.612057 |           - |  100.0 | 124.612057 | 224.612057 |
    | 2000-01-02 14:00:00 |          - | 210.365288 |            - | 210.365288 |           - |  100.0 | 110.365288 | 210.365288 |
    | 2000-01-02 14:30:00 |          - |  197.47692 |            - |  197.47692 |           - |  100.0 |   97.47692 |  197.47692 |
    | 2000-01-02 15:00:00 |          - | 185.861607 |            - | 185.861607 |           - |  100.0 |  85.861607 | 185.861607 |
    | 2000-01-02 15:30:00 |          - | 175.431719 |            - | 175.431719 |           - |  100.0 |  75.431719 | 175.431719 |
    | 2000-01-02 16:00:00 |          - | 166.099094 |            - | 166.099094 |           - |  100.0 |  66.099094 | 166.099094 |
    | 2000-01-02 16:30:00 |          - | 157.776495 |            - | 157.776495 |           - |  100.0 |  57.776495 | 157.776495 |
    | 2000-01-02 17:00:00 |          - | 150.378786 |            - | 150.378786 |           - |  100.0 |  50.378786 | 150.378786 |
    | 2000-01-02 17:30:00 |          - | 143.823868 |            - | 143.823868 |           - |  100.0 |  43.823868 | 143.823868 |
    | 2000-01-02 18:00:00 |          - | 138.033385 |            - | 138.033385 |           - |  100.0 |  38.033385 | 138.033385 |
    | 2000-01-02 18:30:00 |          - | 132.933235 |            - | 132.933235 |           - |  100.0 |  32.933235 | 132.933235 |
    | 2000-01-02 19:00:00 |          - | 128.453911 |            - | 128.453911 |           - |  100.0 |  28.453911 | 128.453911 |
    | 2000-01-02 19:30:00 |          - | 124.530701 |            - | 124.530701 |           - |  100.0 |  24.530701 | 124.530701 |
    | 2000-01-02 20:00:00 |          - | 121.103761 |            - | 121.103761 |           - |  100.0 |  21.103761 | 121.103761 |
    | 2000-01-02 20:30:00 |          - | 118.118088 |            - | 118.118088 |           - |  100.0 |  18.118088 | 118.118088 |
    | 2000-01-02 21:00:00 |          - | 115.523427 |            - | 115.523427 |           - |  100.0 |  15.523427 | 115.523427 |
    | 2000-01-02 21:30:00 |          - | 113.274094 |            - | 113.274094 |           - |  100.0 |  13.274094 | 113.274094 |
    | 2000-01-02 22:00:00 |          - | 111.328777 |            - | 111.328777 |           - |  100.0 |  11.328777 | 111.328777 |
    | 2000-01-02 22:30:00 |          - | 109.650285 |            - | 109.650285 |           - |  100.0 |   9.650285 | 109.650285 |
    | 2000-01-02 23:00:00 |          - | 108.205287 |            - | 108.205287 |           - |  100.0 |   8.205287 | 108.205287 |
    | 2000-01-02 23:30:00 |          - | 106.964036 |            - | 106.964036 |           - |  100.0 |   6.964036 | 106.964036 |
    | 2000-01-03 00:00:00 |          - | 105.900084 |            - | 105.900084 |           - |  100.0 |   5.900084 | 105.900084 |
    | 2000-01-03 00:30:00 |          - |  104.99001 |            - |  104.99001 |           - |  100.0 |    4.99001 |  104.99001 |
    | 2000-01-03 01:00:00 |          - | 104.213141 |            - | 104.213141 |           - |  100.0 |   4.213141 | 104.213141 |
    | 2000-01-03 01:30:00 |          - | 103.551298 |            - | 103.551298 |           - |  100.0 |   3.551298 | 103.551298 |
    | 2000-01-03 02:00:00 |          - | 102.988541 |            - | 102.988541 |           - |  100.0 |   2.988541 | 102.988541 |
    | 2000-01-03 02:30:00 |          - |  102.51094 |            - |  102.51094 |           - |  100.0 |    2.51094 |  102.51094 |
    | 2000-01-03 03:00:00 |          - | 102.106358 |            - | 102.106358 |           - |  100.0 |   2.106358 | 102.106358 |
    | 2000-01-03 03:30:00 |          - | 101.764251 |            - | 101.764251 |           - |  100.0 |   1.764251 | 101.764251 |
    | 2000-01-03 04:00:00 |          - |  101.47548 |            - |  101.47548 |           - |  100.0 |    1.47548 |  101.47548 |
    | 2000-01-03 04:30:00 |          - | 101.232151 |            - | 101.232151 |           - |  100.0 |   1.232151 | 101.232151 |
    | 2000-01-03 05:00:00 |          - | 101.027458 |            - | 101.027458 |           - |  100.0 |   1.027458 | 101.027458 |
    | 2000-01-03 05:30:00 |          - | 100.855551 |            - | 100.855551 |           - |  100.0 |   0.855551 | 100.855551 |
    | 2000-01-03 06:00:00 |          - | 100.711411 |            - | 100.711411 |           - |  100.0 |   0.711411 | 100.711411 |
    | 2000-01-03 06:30:00 |          - | 100.590745 |            - | 100.590745 |           - |  100.0 |   0.590745 | 100.590745 |
    | 2000-01-03 07:00:00 |          - | 100.489885 |            - | 100.489885 |           - |  100.0 |   0.489885 | 100.489885 |
    | 2000-01-03 07:30:00 |          - | 100.405708 |            - | 100.405708 |           - |  100.0 |   0.405708 | 100.405708 |
    | 2000-01-03 08:00:00 |          - |  100.33556 |            - |  100.33556 |           - |  100.0 |    0.33556 |  100.33556 |
    | 2000-01-03 08:30:00 |          - | 100.277186 |            - | 100.277186 |           - |  100.0 |   0.277186 | 100.277186 |
    | 2000-01-03 09:00:00 |          - | 100.228681 |            - | 100.228681 |           - |  100.0 |   0.228681 | 100.228681 |
    | 2000-01-03 09:30:00 |          - | 100.188431 |            - | 100.188431 |           - |  100.0 |   0.188431 | 100.188431 |
    | 2000-01-03 10:00:00 |          - | 100.155078 |            - | 100.155078 |           - |  100.0 |   0.155078 | 100.155078 |
    | 2000-01-03 10:30:00 |          - | 100.127477 |            - | 100.127477 |           - |  100.0 |   0.127477 | 100.127477 |
    | 2000-01-03 11:00:00 |          - | 100.104666 |            - | 100.104666 |           - |  100.0 |   0.104666 | 100.104666 |
    | 2000-01-03 11:30:00 |          - | 100.085838 |            - | 100.085838 |           - |  100.0 |   0.085838 | 100.085838 |
    | 2000-01-03 12:00:00 |          - | 100.070317 |            - | 100.070317 |           - |  100.0 |   0.070317 | 100.070317 |
    | 2000-01-03 12:30:00 |          - | 100.057539 |            - | 100.057539 |           - |  100.0 |   0.057539 | 100.057539 |
    | 2000-01-03 13:00:00 |          - | 100.047032 |            - | 100.047032 |           - |  100.0 |   0.047032 | 100.047032 |
    | 2000-01-03 13:30:00 |          - | 100.038402 |            - | 100.038402 |           - |  100.0 |   0.038402 | 100.038402 |
    | 2000-01-03 14:00:00 |          - | 100.031322 |            - | 100.031322 |           - |  100.0 |   0.031322 | 100.031322 |
    | 2000-01-03 14:30:00 |          - | 100.025521 |            - | 100.025521 |           - |  100.0 |   0.025521 | 100.025521 |
    | 2000-01-03 15:00:00 |          - | 100.020773 |            - | 100.020773 |           - |  100.0 |   0.020773 | 100.020773 |
    | 2000-01-03 15:30:00 |          - | 100.016892 |            - | 100.016892 |           - |  100.0 |   0.016892 | 100.016892 |
    | 2000-01-03 16:00:00 |          - | 100.013722 |            - | 100.013722 |           - |  100.0 |   0.013722 | 100.013722 |
    | 2000-01-03 16:30:00 |          - | 100.011136 |            - | 100.011136 |           - |  100.0 |   0.011136 | 100.011136 |
    | 2000-01-03 17:00:00 |          - | 100.009028 |            - | 100.009028 |           - |  100.0 |   0.009028 | 100.009028 |
    | 2000-01-03 17:30:00 |          - | 100.007313 |            - | 100.007313 |           - |  100.0 |   0.007313 | 100.007313 |
    | 2000-01-03 18:00:00 |          - | 100.005918 |            - | 100.005918 |           - |  100.0 |   0.005918 | 100.005918 |
    | 2000-01-03 18:30:00 |          - | 100.004785 |            - | 100.004785 |           - |  100.0 |   0.004785 | 100.004785 |
    | 2000-01-03 19:00:00 |          - | 100.003865 |            - | 100.003865 |           - |  100.0 |   0.003865 | 100.003865 |
    | 2000-01-03 19:30:00 |          - | 100.003119 |            - | 100.003119 |           - |  100.0 |   0.003119 | 100.003119 |
    | 2000-01-03 20:00:00 |          - | 100.002515 |            - | 100.002515 |           - |  100.0 |   0.002515 | 100.002515 |
    | 2000-01-03 20:30:00 |          - | 100.002026 |            - | 100.002026 |           - |  100.0 |   0.002026 | 100.002026 |
    | 2000-01-03 21:00:00 |          - | 100.001631 |            - | 100.001631 |           - |  100.0 |   0.001631 | 100.001631 |
    | 2000-01-03 21:30:00 |          - | 100.001312 |            - | 100.001312 |           - |  100.0 |   0.001312 | 100.001312 |
    | 2000-01-03 22:00:00 |          - | 100.001054 |            - | 100.001054 |           - |  100.0 |   0.001054 | 100.001054 |
    | 2000-01-03 22:30:00 |          - | 100.000846 |            - | 100.000846 |           - |  100.0 |   0.000846 | 100.000846 |
    | 2000-01-03 23:00:00 |          - | 100.000679 |            - | 100.000679 |           - |  100.0 |   0.000679 | 100.000679 |
    | 2000-01-03 23:30:00 |          - | 100.000544 |            - | 100.000544 |           - |  100.0 |   0.000544 | 100.000544 |
    | 2000-01-04 00:00:00 |          - | 100.000436 |            - | 100.000436 |           - |  100.0 |   0.000436 | 100.000436 |
    | 2000-01-04 00:30:00 |          - | 100.000349 |            - | 100.000349 |           - |  100.0 |   0.000349 | 100.000349 |
    | 2000-01-04 01:00:00 |          - | 100.000279 |            - | 100.000279 |           - |  100.0 |   0.000279 | 100.000279 |
    | 2000-01-04 01:30:00 |          - | 100.000223 |            - | 100.000223 |           - |  100.0 |   0.000223 | 100.000223 |
    | 2000-01-04 02:00:00 |          - | 100.000178 |            - | 100.000178 |           - |  100.0 |   0.000178 | 100.000178 |
    | 2000-01-04 02:30:00 |          - | 100.000142 |            - | 100.000142 |           - |  100.0 |   0.000142 | 100.000142 |
    | 2000-01-04 03:00:00 |          - | 100.000113 |            - | 100.000113 |           - |  100.0 |   0.000113 | 100.000113 |
    | 2000-01-04 03:30:00 |          - |  100.00009 |            - |  100.00009 |           - |  100.0 |    0.00009 |  100.00009 |
    | 2000-01-04 04:00:00 |          - | 100.000072 |            - | 100.000072 |           - |  100.0 |   0.000072 | 100.000072 |
    | 2000-01-04 04:30:00 |          - | 100.000057 |            - | 100.000057 |           - |  100.0 |   0.000057 | 100.000057 |
    | 2000-01-04 05:00:00 |          - | 100.000046 |            - | 100.000046 |           - |  100.0 |   0.000046 | 100.000046 |
    | 2000-01-04 05:30:00 |          - | 100.000036 |            - | 100.000036 |           - |  100.0 |   0.000036 | 100.000036 |
    | 2000-01-04 06:00:00 |          - | 100.000029 |            - | 100.000029 |           - |  100.0 |   0.000029 | 100.000029 |
    | 2000-01-04 06:30:00 |          - | 100.000023 |            - | 100.000023 |           - |  100.0 |   0.000023 | 100.000023 |
    | 2000-01-04 07:00:00 |          - | 100.000018 |            - | 100.000018 |           - |  100.0 |   0.000018 | 100.000018 |
    | 2000-01-04 07:30:00 |          - | 100.000014 |            - | 100.000014 |           - |  100.0 |   0.000014 | 100.000014 |
    | 2000-01-04 08:00:00 |          - | 100.000011 |            - | 100.000011 |           - |  100.0 |   0.000011 | 100.000011 |
    | 2000-01-04 08:30:00 |          - | 100.000009 |            - | 100.000009 |           - |  100.0 |   0.000009 | 100.000009 |
    | 2000-01-04 09:00:00 |          - | 100.000007 |            - | 100.000007 |           - |  100.0 |   0.000007 | 100.000007 |
    | 2000-01-04 09:30:00 |          - | 100.000006 |            - | 100.000006 |           - |  100.0 |   0.000006 | 100.000006 |
    | 2000-01-04 10:00:00 |          - | 100.000004 |            - | 100.000004 |           - |  100.0 |   0.000004 | 100.000004 |
    | 2000-01-04 10:30:00 |          - | 100.000004 |            - | 100.000004 |           - |  100.0 |   0.000004 | 100.000004 |
    | 2000-01-04 11:00:00 |          - | 100.000003 |            - | 100.000003 |           - |  100.0 |   0.000003 | 100.000003 |
    | 2000-01-04 11:30:00 |          - | 100.000002 |            - | 100.000002 |           - |  100.0 |   0.000002 | 100.000002 |
    | 2000-01-04 12:00:00 |          - | 100.000002 |            - | 100.000002 |           - |  100.0 |   0.000002 | 100.000002 |
    | 2000-01-04 12:30:00 |          - | 100.000001 |            - | 100.000001 |           - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 13:00:00 |          - | 100.000001 |            - | 100.000001 |           - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 13:30:00 |          - | 100.000001 |            - | 100.000001 |           - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 14:00:00 |          - | 100.000001 |            - | 100.000001 |           - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 14:30:00 |          - | 100.000001 |            - | 100.000001 |           - |  100.0 |   0.000001 | 100.000001 |
    | 2000-01-04 15:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 15:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 16:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 16:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 17:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 17:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 18:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 18:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 19:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 19:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 20:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 20:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 21:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 21:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 22:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 22:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 23:00:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |
    | 2000-01-04 23:30:00 |          - |      100.0 |            - |      100.0 |           - |  100.0 |        0.0 |      100.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _kinw_impl_euler_numerical_accuracy:

numerical accuracy
__________________

The numerical parameters |WaterVolumeTolerance| and |WaterDepthTolerance| determine the
minimum precision of the iterative search for the "right" water depth ("right" from the
perspective of the implicit Euler method).  This search is the reason for the stability
of |kinw_impl_euler| but causes computational overhead.  Hence, setting the values of
|WaterVolumeTolerance| and |WaterDepthTolerance| is a compromise between numerical
accuracy and computational efficiency.  We hope we found default values sensible for
most applications.  However, you might need to refine them in some cases.  Here, we set
|WaterVolumeTolerance| to an tremendously large value for illustration, which causes
wobbly simulation results but no water balance errors (see method |Calc_WaterDepth_V1|
for more information):

.. integration-test::

    >>> length(100.0)
    >>> nmbsegments(8)
    >>> model.prepare_states(100.0)
    >>> solver.watervolumetolerance(0.01)
    >>> conditions = test("kinw_impl_euler_numerical_accuracy", get_conditions="2000-01-01")
    |                date |                                                                       waterdepth |     inflow |                                                                         internalflow |    outflow |                                                                         watervolume | input1 |     input2 |     output |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 3.711419  3.717025  3.717731   3.71782  3.717831  3.717833  3.717833    3.717833 |      100.0 | 102.322585  102.615209  102.652104  102.656757  102.657343  102.657417    102.657427 | 102.657428 |  1.556806   1.560459   1.56092  1.560978  1.560985  1.560986  1.560986     1.560986 |  100.0 |        0.0 | 102.657428 |
    | 2000-01-01 00:30:00 | 3.705806  3.715614  3.717464  3.717775  3.717824  3.717831  3.717833    3.717833 |      100.0 | 102.030346  102.541485  102.638156  102.654411  102.656974  102.657361    102.657418 | 102.657427 |  1.553151   1.559539  1.560746  1.560949  1.560981  1.560985  1.560986     1.560986 |  100.0 |        0.0 | 102.657427 |
    | 2000-01-01 01:00:00 | 3.700893  3.713765  3.716998  3.717677  3.717806  3.717828  3.717832    3.717833 |      100.0 | 101.775194  102.444929  102.613794   102.64929  102.656005   102.65719     102.65739 | 102.657422 |  1.549956   1.558334  1.560442  1.560885  1.560968  1.560983  1.560986     1.560986 |  100.0 |        0.0 | 102.657422 |
    | 2000-01-01 01:30:00 | 3.696593  3.711609  3.716319  3.717506  3.717768  3.717821  3.717831    3.717832 |      100.0 |  101.55235  102.332513  102.578335  102.640343   102.65403  102.656792    102.657314 | 102.657408 |  1.547161    1.55693  1.559999  1.560773  1.560944  1.560978  1.560985     1.560986 |  100.0 |        0.0 | 102.657408 |
    | 2000-01-01 02:00:00 |  3.69283  3.709254   3.71543  3.717244  3.717702  3.717806  3.717827    3.717832 |      100.0 | 101.357666    102.2098  102.531885  102.626668   102.65058  102.656008    102.657149 | 102.657376 |  1.544718   1.555396   1.55942  1.560602  1.560901  1.560968  1.560983     1.560986 |  100.0 |        0.0 | 102.657376 |
    | 2000-01-01 02:30:00 | 3.689536  3.706784  3.714343  3.716879  3.717598  3.717779  3.717821     3.71783 |      100.0 | 101.187541  102.081193  102.475093  102.607558  102.645155   102.65464    102.656833 | 102.657307 |   1.54258   1.553787  1.558711  1.560364  1.560833  1.560951  1.560979     1.560985 |  100.0 |        0.0 | 102.657307 |
    | 2000-01-01 03:00:00 | 3.686652  3.704263  3.713075    3.7164  3.717447  3.717738  3.717811    3.717828 | 100.000003 | 101.038843  101.950134  102.408961  102.582523  102.637258  102.652448     102.65628 | 102.657178 |   1.54071   1.552147  1.557885  1.560052  1.560735  1.560924  1.560972     1.560983 |  100.0 |   0.000003 | 102.657178 |
    | 2000-01-01 03:30:00 | 3.684128  3.701743  3.711651  3.715802   3.71724  3.717675  3.717794    3.717824 | 100.000029 | 100.908852  101.819288    102.3347  102.551286  102.626418  102.649166    102.655383 | 102.656951 |  1.539074   1.550508  1.556957  1.559662  1.560599  1.560883  1.560961      1.56098 |  100.0 |   0.000029 | 102.656951 |
    | 2000-01-01 04:00:00 | 3.681919  3.699263  3.710096  3.715083  3.716968  3.717586  3.717767    3.717817 | 100.000175 | 100.795209   101.69068  102.253623  102.513772  102.612216  102.644507    102.654012 | 102.656581 |  1.537643   1.548896  1.555944  1.559193  1.560422  1.560825  1.560944     1.560976 |  100.0 |   0.000175 | 102.656581 |
    | 2000-01-01 04:30:00 | 3.679988  3.696853  3.708434  3.714247  3.716625  3.717465  3.717729    3.717806 | 100.000826 |  100.69591  101.565824  102.167068  102.470086  102.594298  102.638176    102.652015 | 102.656005 |  1.536392    1.54733  1.554861  1.558648  1.560199  1.560746  1.560919     1.560968 |  100.0 |   0.000826 | 102.656005 |
    | 2000-01-01 05:00:00 | 3.678302  3.694535   3.70669  3.713296  3.716206  3.717306  3.717676    3.717789 | 100.003194 | 100.609356  101.445843  102.076342  102.420483  102.572388  102.629881    102.649224 |  102.65515 |  1.535301   1.545825  1.553726  1.558029  1.559925  1.560643  1.560884     1.560958 |  100.0 |   0.003194 |  102.65515 |
    | 2000-01-01 05:30:00 | 3.676845  3.692325  3.704889  3.712239  3.715706  3.717104  3.717604    3.717766 | 100.010515 | 100.534561  101.331592  101.982696  102.365345  102.546292  102.619342    102.645456 | 102.653928 |  1.534358    1.54439  1.552554   1.55734  1.559599  1.560511  1.560837     1.560943 |  100.0 |   0.010515 | 102.653928 |
    | 2000-01-01 06:00:00 | 3.675617  3.690239  3.703053  3.711085  3.715124  3.716855  3.717509    3.717733 | 100.030315 |   100.4716  101.223826  101.887316  102.305154  102.515902  102.606301    102.640519 | 102.652237 |  1.533563   1.543036   1.55136  1.556588   1.55922  1.560348  1.560775     1.560921 |  100.0 |   0.030315 | 102.652237 |
    | 2000-01-01 06:30:00 | 3.674659  3.688293  3.701204  3.709843   3.71446  3.716553  3.717389     3.71769 | 100.078179 | 100.422491  101.123459   101.79135  102.240476  102.481196   102.59053    102.634216 | 102.649965 |  1.532943   1.541774  1.550158  1.555779  1.558787  1.560152  1.560697     1.560893 |  100.0 |   0.078179 | 102.649965 |
    | 2000-01-01 07:00:00 | 3.674077  3.686519  3.699366  3.708528  3.713713  3.716195  3.717238    3.717633 | 100.183351 | 100.392645  101.031965  101.695988  102.171957  102.442236  102.571837    102.626351 | 102.646987 |  1.532567   1.540624  1.548963  1.554923    1.5583  1.559918  1.560598     1.560856 |  100.0 |   0.183351 | 102.646987 |
    | 2000-01-01 07:30:00 | 3.674085  3.684967  3.697564  3.707151  3.712888  3.715779  3.717054     3.71756 | 100.396214 |  100.39309  100.952013  101.602601  102.100331  102.399169  102.550073    102.616734 | 102.643172 |  1.532572   1.539617  1.547792  1.554027  1.557762  1.559647  1.560478     1.560808 |  100.0 |   0.396214 | 102.643172 |
    | 2000-01-01 08:00:00 |  3.67507  3.683731  3.695833  3.705731  3.711988  3.715301  3.716833    3.717468 | 100.797305 | 100.443541  100.888401   101.51299  102.026466  102.352231  102.525139    102.605187 | 102.638383 |  1.533209   1.538817  1.546667  1.553102  1.557176  1.559335  1.560334     1.560749 |  100.0 |   0.797305 | 102.638383 |
    | 2000-01-01 08:30:00 | 3.677659  3.682972  3.694224  3.704288   3.71102  3.714762  3.716572    3.717356 | 101.507032 | 100.576333   100.84937  101.429757  101.951449  102.301769  102.496991    102.591547 | 102.632478 |  1.534884   1.538325  1.545623  1.552163  1.556545  1.558984  1.560164     1.560675 |  100.0 |   1.507032 | 102.632478 |
    | 2000-01-01 09:00:00 | 3.682812  3.682952  3.692814  3.702849  3.709993  3.714162  3.716269    3.717219 | 102.694853 | 100.841106  100.848336  101.356861  101.876724  102.248264  102.465652    102.575678 | 102.625316 |  1.538221   1.538312  1.544707  1.551228  1.555877  1.558593  1.559966     1.560586 |  100.0 |   2.694853 | 102.625316 |
    | 2000-01-01 09:30:00 | 3.691907  3.684074  3.691721  3.701454   3.70892  3.713502   3.71592    3.717055 | 104.586356 | 101.309987  100.906079   101.30036  101.804313  102.192393  102.431227     102.55747 | 102.616762 |  1.544119   1.539039  1.543998   1.55032  1.555178  1.558163  1.559739     1.560479 |  100.0 |   4.586356 | 102.616762 |
    | 2000-01-01 10:00:00 | 3.706817  3.686934  3.691121  3.700159   3.70782  3.712787  3.715525    3.716862 | 107.466641 | 102.082959  101.053351  101.269406  101.737132  102.135113  102.393929    102.536858 | 102.606689 |  1.553809   1.540893  1.543609  1.549479  1.554462  1.557697  1.559482     1.560353 |  100.0 |   7.466641 | 102.606689 |
    | 2000-01-01 10:30:00 | 3.729957   3.69237  3.691278  3.699046  3.706718  3.712024  3.715085    3.716638 | 111.678559 | 103.293096  101.333905  101.277488  101.679421  102.077793  102.354116    102.513829 | 102.594982 |  1.568903   1.544419   1.54371  1.548755  1.553745    1.5572  1.559194     1.560207 |  100.0 |  11.678559 | 102.594982 |
    | 2000-01-01 11:00:00 | 3.764271  3.701523  3.692565  3.698234  3.705653  3.711223  3.714598    3.716381 | 117.614836 |  105.11133  101.807892  101.343963  101.637311  102.022401  102.312348     102.48844 | 102.581552 |  1.591409   1.550365  1.544545  1.548227  1.553051  1.556678  1.558877     1.560039 |  100.0 |  17.614836 | 102.581552 |
    | 2000-01-01 11:30:00 | 3.813154  3.715885  3.695503  3.697891  3.704679    3.7104   3.71407     3.71609 | 125.703736 | 107.750435  102.555628  101.495897  101.619562  101.971755  102.269471    102.460851 | 102.566337 |  1.623725   1.559716  1.546453  1.548004  1.552418  1.556142  1.558533      1.55985 |  100.0 |  25.703736 | 102.566337 |
    | 2000-01-01 12:00:00 | 3.880272  3.737336  3.700798  3.698256  3.703872  3.709579  3.713505    3.715764 | 136.388647 | 111.467743  103.681726   101.77025  101.638475  101.929863  102.226726    102.431357 | 102.549324 |  1.668583   1.573731  1.549894  1.548242  1.551893  1.555607  1.558164     1.559637 |  100.0 |  36.388647 | 102.549324 |
    | 2000-01-01 12:30:00 | 3.969296  3.768167  3.709382  3.699656  3.703343  3.708796  3.712912    3.715405 | 150.102726 | 116.565657  105.319537  102.216469  101.711036  101.902362  102.185907     102.40044 |  102.53056 |   1.72895   1.593974  1.555479  1.549151  1.551549  1.555097  1.557778     1.559403 |  100.0 |  50.102726 |  102.53056 |
    | 2000-01-01 13:00:00 | 4.083559  3.811051  3.722458  3.702533  3.703241  3.708098  3.712306    3.715015 | 167.240349 | 123.388375  107.635688  102.899512  101.860311  101.897077  102.149567    102.368844 | 102.510181 |  1.807883   1.622329  1.564004  1.551022  1.551482  1.554642  1.557384     1.559149 |  100.0 |  67.240349 | 102.510181 |
    | 2000-01-01 13:30:00 | 4.225697   3.86897  3.741534  3.707475  3.703774  3.707554  3.711708    3.714598 | 188.127569 | 132.312636  110.834178  103.903359  102.117159  101.924736  102.121283     102.33767 | 102.488443 |   1.90835    1.66099   1.57648  1.554237  1.551829  1.554289  1.556994     1.558877 |  100.0 |  88.127569 | 102.488443 |
    | 2000-01-01 14:00:00 | 4.397321  3.945062  3.768454  3.715247  3.705219  3.707261  3.711149    3.714164 | 212.993962 | 143.730515  115.159002  105.334914  102.522294  101.999851  102.106008    102.308502 | 102.465771 |  2.033024   1.712419  1.594163    1.5593  1.552769  1.554098   1.55663     1.558594 |  100.0 | 112.993962 | 102.465771 |
    | 2000-01-01 14:30:00 | 4.598804  4.042386  3.805403  3.726829  3.707949  3.707347  3.710671    3.713724 | 241.948195 | 158.023943  120.893764  107.328126  103.128751  102.141827  102.110513    102.283576 | 102.442818 |  2.184088   1.779253  1.618581  1.566859  1.554546  1.554154  1.556318     1.558308 |  100.0 | 141.948195 | 102.442818 |
    | 2000-01-01 15:00:00 | 4.829207  4.163634  3.854866  3.743451   3.71245   3.70799  3.710333    3.713297 |  274.95935 | 175.533754  128.356491  110.047866  104.004754  102.376317  102.143951       102.266 | 102.420544 |  2.363054   1.864172  1.651537  1.577736  1.557477  1.554572  1.556098     1.558029 |  100.0 |  174.95935 | 102.420544 |
    | 2000-01-01 15:30:00 | 5.086348  4.310818   3.91954  3.766622  3.719352  3.709423  3.710219     3.71291 | 311.845554 | 196.528949  137.888124  113.692872  105.236955  102.736873  102.218557    102.260028 | 102.400326 |  2.570624   1.969725  1.695088  1.592957  1.561977  1.555505  1.556024     1.557777 |  100.0 | 211.845554 | 102.400326 |
    | 2000-01-01 16:00:00 | 5.345876  4.505635  4.005486  3.798607   3.72952  3.711963  3.710438    3.712599 | 352.270855 | 231.227999   151.29572  118.692648  106.959069  103.270102  102.350932    102.271471 | 102.384097 |  2.788501   2.113604  1.753774  1.614077  1.568617   1.55716  1.556167     1.557574 |  100.0 | 252.270855 | 102.384097 |
    | 2000-01-01 16:30:00 | 5.643503  4.706407  4.111547  3.841391  3.744003  3.716025  3.711142    3.712415 | 395.750601 | 251.180649  166.047505  125.107344  109.301112  104.033951  102.562966    102.308167 | 102.374534 |  3.048727   2.266843  1.827466  1.642529  1.578098  1.559807  1.556625     1.557455 |  100.0 | 295.750601 | 102.374534 |
    | 2000-01-01 17:00:00 | 5.935741   4.95278  4.243376  3.897721  3.764141  3.722153  3.712532     3.71243 | 441.664962 | 293.726964  185.432629  133.456447  112.451898  105.104367  102.883524    102.380616 |   102.3753 |  3.315015   2.461773  1.921023  1.680337  1.591324  1.563805  1.557531     1.557464 |  100.0 | 341.664962 |   102.3753 |
    | 2000-01-01 17:30:00 | 6.076807  5.181268  4.394624  3.969234  3.791406  3.731024   3.71487    3.712737 | 489.279607 | 308.865172  204.664558  143.545717   116.56205  106.569171  103.349224    102.502637 | 102.391337 |  3.639761   2.649334  2.031037  1.728907  1.609311  1.569601  1.559054     1.557665 |  100.0 | 389.279607 | 102.391337 |
    | 2000-01-01 18:00:00 | 6.193364  5.354212  4.575934  4.059189  3.827647  3.743498    3.7185    3.713464 |  537.77215 | 312.994083  231.713813  156.353498  121.906894  108.543955  104.007232    102.692311 | 102.429247 |  4.044362   2.795639  2.166686  1.790911  1.633364  1.577767  1.561421     1.558138 |  100.0 |  437.77215 | 102.429247 |
    | 2000-01-01 18:30:00 | 6.323066  5.541222  4.749637   4.16427  3.874132  3.760529  3.723848    3.714775 | 586.262607 | 333.915923  243.686934  169.346991  128.396553  111.123187  104.911662    102.972384 | 102.497676 |  4.498586   2.958051  2.300498  1.864622  1.664456  1.588948  1.564912     1.558993 |  100.0 | 486.262607 | 102.497676 |
    | 2000-01-01 19:00:00 | 6.463751  5.739984  4.922299  4.282399  3.931742  3.783118  3.731428    3.716881 | 633.846018 | 357.490061  258.929165  182.958128  136.007445  114.391908  106.122035    103.370456 |  102.60768 |  4.996027    3.13546  2.437245  1.949133  1.703364  1.603834  1.569865     1.560365 |  100.0 | 533.846018 |  102.60768 |
    | 2000-01-01 19:30:00 |  6.61299  5.946769  5.099952  4.412662  4.001152  3.812292  3.741834    3.720044 | 679.625318 | 383.461135  278.009785  197.682152  144.785161  118.436256  107.703383    103.919221 | 102.773086 |  5.529122   3.325273  2.581835  2.044348  1.750792  1.623153  1.576676     1.562429 |  100.0 | 579.625318 | 102.773086 |
    | 2000-01-01 20:00:00 | 6.768167  6.044706  5.255571  4.579747  4.086957  3.849668  3.755821    3.724592 | 722.742733 | 411.513288  298.371848  226.206121  156.631181  123.596054  109.759268    104.660942 | 103.011389 |  6.089335   3.528927  2.711733  2.169583  1.810255  1.648059  1.585853     1.565398 |  100.0 | 622.742733 | 103.011389 |
    | 2000-01-01 20:30:00 | 6.926566  6.108356  5.429703  4.738303  4.185866  3.896281  3.774219    3.730927 | 762.408181 | 441.268385  319.038646  236.293708  168.477737   129.76305  112.370425    105.643719 | 103.344088 |  6.667387   3.748941  2.860674  2.291652  1.879942  1.679366  1.597961     1.569537 |  100.0 | 662.408181 | 103.344088 |
    | 2000-01-01 21:00:00 | 7.085456  6.191058  5.567703  4.887486  4.294603  3.952606  3.797856    3.739513 | 797.923464 | 472.288045  312.629639  245.559127  180.158159  136.812694  115.595377    106.918351 |  103.79662 |  7.253531   4.036326  2.981401  2.409373  1.957964  1.717557   1.61358     1.575156 |  100.0 | 697.923464 |  103.79662 |
    | 2000-01-01 21:30:00 | 7.242175  6.282101  5.710394  5.031417  4.410866  4.018713  3.827513    3.750875 | 828.701418 | 504.079131  327.221259  256.479438  191.915665  144.661451  119.477882    108.536559 | 104.398119 |  7.837851    4.35467  3.108737  2.525588  2.043021  1.762887  1.633274     1.582605 |  100.0 | 728.701418 | 104.398119 |
    | 2000-01-01 22:00:00 | 7.394208  6.380607   5.85701  5.173779  4.533231  4.094422  3.863887    3.765578 |  854.27953 |  536.10313  343.450075  269.290028  204.015111  153.267311  124.053337    110.550233 | 105.181157 |  8.410568   4.701446  3.242225  2.643083  2.134367  1.815472   1.65758      1.59227 |  100.0 |  754.27953 | 105.181157 |
    | 2000-01-01 22:30:00 | 7.539251  6.485477       6.0  5.299568  4.684568  4.183286   3.90815      3.7843 | 874.327871 | 567.789709  361.209692  287.445581  228.598511  164.397141  129.599222    113.043615 | 106.185688 |  8.962337    5.07329     3.375  2.749008   2.24993  1.878109   1.68738     1.604614 |  100.0 | 774.327871 | 106.185688 |
    | 2000-01-01 23:00:00 | 7.652173  6.618178  6.044927  5.429912  4.825204  4.281871  3.960873    3.807738 | 888.651531 | 647.974317  384.381784  298.444368  236.306838  175.218981  135.972708    116.075148 | 107.455181 |  9.395556   5.547756  3.529687  2.860855  2.359888  1.948752  1.723196      1.62013 |  100.0 | 788.651531 | 107.455181 |
    | 2000-01-01 23:30:00 | 7.763179  6.743822  6.093298  5.570673   4.95983  4.387815  4.022215    3.836559 | 897.187983 | 658.869342  407.041106  314.200305  245.772045  186.008064  143.079828    119.686478 | 109.034379 |  9.824529   6.001047  3.696801  2.984026  2.467463  2.026023  1.765304     1.639304 |  100.0 | 797.187983 | 109.034379 |
    | 2000-01-02 00:00:00 | 7.870585  6.861962    6.1565  5.678394  5.086465   4.49864  4.091918    3.871343 |      900.0 | 667.781927  428.993846    307.1997  253.903833  196.538826  150.798753    123.899814 | 110.966954 | 10.242522   6.430866   3.91603  3.079959   2.57072  2.108355  1.813722     1.662583 |  100.0 |      800.0 | 110.966954 |
    | 2000-01-02 00:30:00 |      8.0  6.945684  6.222279  5.787131  5.207776  4.612625  4.169449    3.912549 | 897.264896 | 615.332659  444.937658  317.587691   262.97167  206.973661  159.039393    128.723259 | 113.294025 |     10.75   6.737577   4.14526  3.178267  2.671516  2.194637  1.868291     1.690355 |  100.0 | 797.264896 | 113.294025 |
    | 2000-01-02 01:00:00 | 8.058317  7.085633  6.295716  5.897541  5.302394  4.750805   4.25777    3.961041 | 889.260928 | 759.330582  472.323409  329.437574  273.141066  228.755665  169.436792    134.393157 |  116.08491 | 10.983875    7.25419  4.402455  3.279601   2.75141  2.301411  1.931369      1.72331 |  100.0 | 789.260928 |  116.08491 |
    | 2000-01-02 01:30:00 | 8.118144  7.202711  6.374221       6.0  5.415692  4.878466  4.353696    4.017125 | 876.351729 |  738.14951  495.959128  342.384435  289.385097  235.418658  179.437322    140.761948 | 119.383371 | 11.232639   7.690132  4.678889     3.375   2.84855  2.402177  2.000985     1.761792 |  100.0 | 776.351729 | 119.383371 |
    | 2000-01-02 02:00:00 | 8.160218  7.317368  6.457264  6.032455  5.522944   4.99769  4.454702    4.080682 | 858.969595 | 758.798269  519.768789  356.383619  294.329515  242.421446  189.117674    147.703839 | 123.212735 | 11.412947   8.120385  4.972982  3.486697  2.941984  2.498124   2.07553     1.805876 |  100.0 | 758.969595 | 123.212735 |
    | 2000-01-02 02:30:00 | 8.187329  7.426278  6.544003  6.066755  5.634962  5.111608  4.559086    4.151319 |   837.5984 |  771.75056   543.01203  371.333441  305.594686  250.527342  198.673715    155.130745 | 127.582411 | 11.531473   8.532115  5.282004  3.605027  3.041106   2.59146  2.153907     1.855463 |  100.0 |   737.5984 | 127.582411 |
    | 2000-01-02 03:00:00 | 8.201336  7.527111  6.633387  6.103054  5.750181  5.222832  4.665814    4.228497 | 812.756821 | 778.336909  565.093822  387.087273  317.338557  259.788822   208.29232    162.988838 |   132.4933 | 11.593429   8.915952  5.602416  3.730575  3.144695  2.684154  2.235454     1.910355 |  100.0 | 712.756821 |   132.4933 |
    | 2000-01-02 03:30:00 | 8.203599   7.61821  6.724236  6.153258   5.82914  5.303534  4.795133     4.31506 | 784.982427 | 779.394617  585.523129  403.462451  306.693477  266.722201  228.819167    172.866401 | 138.170458 | 11.603487   9.264921  5.930125  3.904759  3.216643  2.752379  2.336169     1.972807 |  100.0 | 684.982427 | 138.170458 |
    | 2000-01-02 04:00:00 | 8.195219  7.676636  6.836388   6.20941  5.909678  5.388823  4.912156    4.407674 | 754.817086 | 775.469413  650.522314  424.187956  315.538097  274.321636  233.772797     182.13941 |  144.44168 | 11.566313   9.489826  6.337527  4.100329  3.290833  2.825367  2.429109     2.040663 |  100.0 | 654.817086 |  144.44168 |
    | 2000-01-02 04:30:00 | 8.177175  7.728488  6.940516  6.270454       6.0  5.471153  5.019179    4.503749 |  722.79402 | 766.931246  655.648266  443.944116  325.332258  278.572797  238.951359    190.897348 | 151.161602 | 11.486866   9.690135  6.718594   4.31383     3.375  2.896686  2.515606     2.112188 |  100.0 |  622.79402 | 151.161602 |
    | 2000-01-02 05:00:00 | 8.150417  7.772317  7.036769  6.335376  6.023351   5.56537  5.120011    4.601665 | 689.426711 | 754.048159   659.68998  462.655753  335.942809  291.311715  245.392342    199.390446 | 158.233782 | 11.370547    9.85998  7.073256  4.541913  3.455336  2.979341  2.598409      2.18627 |  100.0 | 589.426711 | 158.233782 |
    | 2000-01-02 05:30:00 | 8.115921  7.806771  7.125327  6.403211  6.048261  5.660595  5.216654    4.700366 | 655.199767 | 737.039628  662.679768  480.261665  347.236321  299.540861  252.503728    207.750597 | 165.589897 | 11.223235   9.993828  7.401608  4.781359  3.541188  3.064008  2.678965     2.262159 |  100.0 | 555.199767 | 165.589897 |
    | 2000-01-02 06:00:00 | 8.056739  7.848836   7.20703  6.473116  6.074786  5.756286  5.285895    4.821981 | 620.561767 | 757.117114  666.106523   496.84405  359.090765  308.208716  260.307342    227.844209 |  174.96581 | 10.977436  10.157647  7.706281  5.029315  3.632775   3.15023  2.737399      2.35734 |  100.0 | 520.561767 |  174.96581 |
    | 2000-01-02 06:30:00 |      8.0  7.869399  7.281656  6.544265  6.102923  5.851874  5.363075    4.931564 | 585.920022 | 712.273264  667.692293  512.281139   371.37903  317.296733  268.811932    232.234653 | 183.707963 |     10.75  10.237892  7.986021  5.282939  3.730124  3.237503  2.803238     2.444688 |  100.0 | 485.920022 | 183.707963 |
    | 2000-01-02 07:00:00 | 7.944431  7.871755  7.349218  6.615877  6.143795  5.911751  5.441306    5.031728 |  551.63713 | 672.980211  667.870178  526.501883  383.973277  305.219396  274.524583    237.027168 | 191.941599 | 10.531582   10.24709  8.240484   5.53949  3.871881  3.292753  2.870733     2.525842 |  100.0 |  451.63713 | 191.941599 |
    | 2000-01-02 07:30:00 | 7.875412  7.871877  7.410283  6.687271  6.187604       6.0  5.494784    5.121494 | 518.029154 | 668.144837  667.879396  539.559336   396.75535  312.084117  266.391573    240.513741 | 199.517058 | 10.261374  10.247568   8.47146  5.796537  4.024289     3.375  2.917313     2.599636 |  100.0 | 418.029154 | 199.517058 |
    | 2000-01-02 08:00:00 | 7.793997  7.869076  7.465352  6.757859  6.234092  6.016016  5.581387    5.207121 | 485.365238 | 661.590531  667.667801  551.503887  409.616452  319.476298   288.87229    246.544907 | 206.916421 |  9.944169  10.236629  8.680555  6.051935  4.186541  3.430087  2.993502     2.670967 |  100.0 | 385.365238 | 206.916421 |
    | 2000-01-02 08:30:00 | 7.701473  7.862553  7.514859  6.827136  6.282968   6.03316  5.665615    5.263302 |  453.86844 | 653.024195  667.170878  562.381224  422.456644  327.362111  294.562669     252.89625 | 226.619239 |  9.585688  10.211165  8.869176  6.303799  4.357711  3.489126  3.068502     2.718266 |  100.0 |  353.86844 | 226.619239 |
    | 2000-01-02 09:00:00 | 7.625152  7.826887  7.558335  6.894586   6.33391  6.051484  5.747598    5.325814 | 423.717566 | 587.099119  664.349194  572.043936  435.168247  335.701091  300.599486    259.570299 | 230.074506 |  9.291602  10.072115  9.035326  6.550175  4.536752  3.552309  3.142355     2.771358 |  100.0 | 323.717566 | 230.074506 |
    | 2000-01-02 09:30:00 | 7.543592  7.784308  7.596029   6.95976  6.386571  6.071015  5.827454    5.393999 | 395.049792 | 568.755614  660.749162  580.506371  447.650262  344.446834  306.982479    266.568919 | 234.086624 |  8.978931   9.906527  9.179763  6.789316  4.722518  3.619745  3.215099     2.829827 |  100.0 | 295.049792 | 234.086624 |
    | 2000-01-02 10:00:00 | 7.457726  7.734906  7.628125  7.022263  6.440588  6.091756  5.905278    5.467141 | 367.963873 | 549.840099  656.256795  587.774826  459.807714   353.54766  313.703109    273.892132 | 238.689568 |  8.651554   9.714977   9.30303  7.019657  4.913786  3.691465  3.286759     2.893191 |  100.0 | 267.963873 | 238.689568 |
    | 2000-01-02 10:30:00 | 7.368426   7.67893  7.654747  7.081751  6.495585  6.113691       6.0    5.526886 | 342.523746 | 530.588093  650.756963  593.847679   471.55103  362.947221  320.745377    271.722418 | 242.692485 |  8.313038   9.498673  9.405467  7.239791  5.109273  3.767428     3.375     2.945445 |  100.0 | 242.523746 | 242.692485 |
    | 2000-01-02 11:00:00 | 7.276498   7.64171  7.653489  7.135645  6.550967  6.148254  6.009975    5.605443 | 318.762374 | 511.205094  590.868566  593.559821  482.337593   372.54808  305.913411    286.857411 |  248.30863 |  7.966641   9.355278  9.400623  7.439991  5.306894  3.887371  3.409301     3.014833 |  100.0 | 218.762374 |  248.30863 |
    | 2000-01-02 11:30:00 | 7.182682  7.599792  7.648335  7.184049  6.606186  6.184789  6.021014    5.679173 |  296.68568 | 491.867932  581.355483  592.381201   492.14638  382.255702  311.639906    290.535225 | 253.965673 |  7.615313   9.194201  9.380777  7.620414  5.504698  4.014479  3.447289     3.080658 |  100.0 |  196.68568 | 253.965673 |
    | 2000-01-02 12:00:00 | 7.087659  7.553629  7.639296  7.227067   6.66074  6.223105  6.033122    5.748795 | 276.276442 | 472.726642  570.993132  590.318112  500.961434  391.978913  317.719552    294.550017 | 259.671531 |  7.261703   9.017321  9.345992  7.781256  5.700866  4.148146  3.488994     3.143439 |  100.0 | 176.276442 | 259.671531 |
    | 2000-01-02 12:30:00 | 6.992043  7.503675  7.626435  7.264814  6.714172  6.262985  6.046295    5.814953 | 257.498082 | 453.906674   559.91226  587.390505  508.772708  401.630416  324.124234    298.894638 |  265.43961 |  6.908167   8.826511  9.296531  7.922768  5.893722  4.287657  3.534408     3.203658 |  100.0 | 157.498082 |  265.43961 |
    | 2000-01-02 13:00:00 | 6.896391  7.450372  7.609854  7.297412  6.766072  6.304187  6.060516    5.878207 | 240.298231 | 435.511274  548.238661  583.630004  515.576663   411.12735   330.82091    303.557737 | 271.286615 |  6.556784   8.623602  9.232826  8.045264  6.081731  4.432209  3.583481     3.261746 |  100.0 | 140.298231 | 271.286615 |
    | 2000-01-02 13:30:00 | 6.801199  7.394153  7.589691  7.324996  6.816075   6.34645  6.075756    5.939032 | 224.612057 | 417.623927  536.091348  579.077852  521.376596  420.391871  337.772192    308.523864 | 277.230384 |  6.209362    8.41036  9.155451  8.149126  6.263504  4.580924  3.636128     3.318075 |  100.0 | 124.612057 | 277.230384 |
    | 2000-01-02 14:00:00 | 6.706908  7.335433  7.566109  7.347713   6.86386    6.3895  6.091975         6.0 | 210.365288 | 400.310765   523.58131  573.782893  526.182719  429.351708  344.936955    313.773684 | 282.148511 |  5.867461   8.188473  9.065088  8.234806  6.437799  4.732871  3.692222        3.375 |  100.0 | 110.365288 | 282.148511 |
    | 2000-01-02 14:30:00 | 6.613902  7.274606  7.539296  7.365723  6.909151  6.433057  6.109123    6.015931 |  197.47692 | 383.622909  510.810789  567.799656  530.012001  437.940691  352.271002    319.284261 | 288.843797 |  5.532398   7.959535  8.962508  8.302824  6.603528  4.887076  3.751598     3.429793 |  100.0 |   97.47692 | 288.843797 |
    | 2000-01-02 15:00:00 | 6.522514  7.212046  7.509451  7.379197  6.951717  6.476836  6.127137    6.032048 | 185.861607 | 367.598707  497.872957  561.186601  532.887806  446.099199  359.727733    325.029425 | 294.194861 |  5.205271   7.725042  8.848543  8.353762  6.759747  5.042545  3.814055     3.485295 |  100.0 |  85.861607 | 294.194861 |
    | 2000-01-02 15:30:00 | 6.433027  7.148107  7.476787  7.388315  6.991364  6.520553  6.158132    6.037955 | 175.431719 | 352.265901  484.851938  554.004554  534.839376  453.774533  367.258846    307.454643 |  296.14666 |  4.886969   7.486387  8.724069  8.388259  6.905664  5.198273  3.921703     3.505649 |  100.0 |  75.431719 |  296.14666 |
    | 2000-01-02 16:00:00 | 6.345674  7.083119  7.441523  7.393267   7.02794  6.563931  6.190345    6.045246 | 166.099094 | 337.643788  471.823093  546.315365  535.901181  460.921169   374.81503     312.51691 | 298.549627 |  4.578189   7.244864  8.589983  8.407005  7.040628  5.353264   4.03384     3.530791 |  100.0 |  66.099094 | 298.549627 |
    | 2000-01-02 16:30:00 | 6.260641  7.017392  7.403879  7.394251  7.061331    6.6067   6.22364    6.053823 | 157.776495 | 323.745591  458.853547  538.180777    536.1122  467.500909  382.346669    317.804939 | 301.366637 |  4.279445    7.00167  8.447194  8.410728  7.164128  5.506542  4.150015     3.560379 |  100.0 |  57.776495 | 301.366637 |
    | 2000-01-02 17:00:00 |  6.17807  6.951217  7.364078  7.391467  7.091455  6.648604  6.257866    6.063592 | 150.378786 | 310.581573  446.002956  529.661535  535.515157  473.482901   389.80451    323.297802 | 304.562674 |   3.99108   6.757911  8.296608  8.400192  7.275786  5.657163  4.269727     3.594103 |  100.0 |  50.378786 | 304.562674 |
    | 2000-01-02 17:30:00 |  6.09023  6.891641  7.322899  7.385174  7.118269  6.689402  6.292861    6.074464 | 143.823868 | 313.210573  434.608867  520.934234  534.166614  478.844511  397.140403    328.972059 | 308.104277 |  3.686184   6.539394  8.141222  8.376373  7.375366   5.80423   4.39243     3.631665 |  100.0 |  43.823868 | 308.104277 |
    | 2000-01-02 18:00:00 | 6.012202  6.825341  7.279941  7.375567  7.141757  6.728871  6.328452    6.086356 | 138.033385 | 287.600752  422.121175  511.923321  532.112064  483.569865  404.307883    334.801983 | 311.959097 |  3.416962   6.297257  7.979579  8.340034  7.462742  5.946902   4.51754     3.672782 |  100.0 |  38.033385 | 311.959097 |
    | 2000-01-02 18:30:00 | 5.791458  6.753322  7.234939  7.362802  7.161925  6.766807  6.364463    6.099185 | 132.933235 | 263.351384  408.783027  502.584402  529.389637  487.648771  411.262638    340.759844 | 316.095548 |   3.18221    6.03548  7.810736  8.291784  7.537876  6.084397  4.644445     3.717178 |  100.0 |  32.933235 | 316.095548 |
    | 2000-01-02 19:00:00 | 5.557831  6.678505  7.187894  7.347022  7.178795  6.803024  6.400709    6.112869 | 128.453911 |  244.85578  395.173748  492.930663  526.035939  491.076188  417.962925    346.816202 | 320.482549 |  2.972686   5.764908  7.634774  8.232195  7.600803  6.216001  4.772509     3.764578 |  100.0 |  28.453911 | 320.482549 |
    | 2000-01-02 19:30:00 | 5.337672  6.602989  7.138979  7.328372  7.192407  6.837359  6.437007    6.127326 | 124.530701 | 230.753634  381.689895  483.009454   522.08917  493.851981  424.369924    352.940243 |  325.08935 |  2.781485   5.493223  7.452398  8.161851   7.65163  6.341069  4.901083      3.81471 |  100.0 |  24.530701 |  325.08935 |
    | 2000-01-02 20:00:00 | 5.162823    6.5204  7.087857  7.306958  7.202812  6.869668   6.47317    6.154398 | 121.103761 | 203.067345  367.232276  472.766088  517.579492  495.979795  430.447989    359.100111 | 306.871394 |  2.633951   5.197726  7.262438  8.081187   7.69051  6.459026  5.029509     3.908722 |  100.0 |  21.103761 | 306.871394 |
    | 2000-01-02 20:30:00 | 5.006039  6.437787  7.034777  7.282898  7.210068  6.899828  6.509017    6.182337 | 118.118088 |  189.80784  353.072495  462.264075  512.540508  497.467022  436.164866     365.26326 | 311.253438 |  2.504909    4.90385  7.065893   7.99069  7.717642   6.56937  5.157132     4.005939 |  100.0 |  18.118088 | 311.253438 |
    | 2000-01-02 21:00:00 | 4.865366  6.355877  6.980018  7.256322  7.214247  6.927733  6.544367     6.21104 | 115.523427 | 178.393675  339.333901  451.570536  507.009098  498.324732   441.49188    371.396802 | 315.797282 |  2.391743   4.614157  6.863867    7.8909  7.733274  6.671669  5.283303     4.106018 |  100.0 |  15.523427 | 315.797282 |
    | 2000-01-02 21:30:00 | 4.739092   6.27522  6.923871  7.227376  7.215429  6.953299  6.579048    6.240396 | 113.274094 | 168.538186  326.104466  440.752534   501.02493  498.567549  446.404098    377.467866 | 320.486906 |  2.292267   4.330538    6.6575   7.78241  7.737697  6.765563  5.407388     4.208584 |  100.0 |  13.274094 | 320.486906 |
    | 2000-01-02 22:00:00 | 4.625716  6.196227  6.866632  7.196214  7.213705  6.976459  6.612893    6.270284 | 111.328777 | 160.005278  313.447029    429.8748  494.629806   498.21346  450.880434    383.443942 |  325.30475 |  2.204649   4.054343   6.44793  7.665851  7.731246  6.850762  5.528774     4.313235 |  100.0 |  11.328777 |  325.30475 |
    | 2000-01-02 22:30:00 | 4.523908  6.109794  6.816784  7.163672  7.209233   6.99717  6.645746    6.300582 | 109.650285 | 152.599192  319.499034  420.524039  488.003037  497.295776  454.904784    389.293314 | 330.231844 |  2.127341   3.753923  6.266085  7.544389  7.714519  6.927066  5.646875     4.419545 |  100.0 |   9.650285 | 330.231844 |
    | 2000-01-02 23:00:00 | 4.432492  6.032583  6.759554  7.129292  7.202118  7.015406   6.67746     6.33116 | 108.205287 | 146.156694  294.371636  409.928069  481.058709  495.837793  458.464842    394.985376 | 335.247962 |  2.059029   3.487136  6.058084  7.416354  7.687917  6.994338  5.761138     4.527073 |  100.0 |   8.205287 | 335.247962 |
    | 2000-01-02 23:30:00 | 4.350423  5.869095  6.695948  7.092753  7.192429   7.03115    6.7079    6.361888 | 106.964036 | 140.540999   270.42365  398.324179  473.741614  493.856379    461.5513    400.490858 | 340.331799 |   1.99859   3.253347  5.827863  7.280602  7.651711  7.052487  5.871046     4.635359 |  100.0 |   6.964036 | 340.331799 |
    | 2000-01-03 00:00:00 | 4.276766  5.640245  6.628373  7.053968  7.180223  7.044398  6.736942     6.39263 | 105.900084 | 135.636977  250.930812  386.194168  466.045358  491.366886  464.157509    405.782022 | 345.461152 |  1.945064   3.045818  5.584389   7.13687  7.606132  7.101464  5.976122     4.743936 |  100.0 |   5.900084 | 345.461152 |
    | 2000-01-03 00:30:00 | 4.210686  5.425863  6.559141  7.013044  7.165563  7.055153  6.764472    6.423255 |  104.99001 | 131.347339  236.052715  373.976533  458.002987  488.386763  466.279474    410.832836 | 350.613113 |  1.897621   2.857349  5.336126  6.985622  7.551441  7.141257  6.075926     4.852332 |  100.0 |    4.99001 | 350.613113 |
    | 2000-01-03 01:00:00 | 4.151433  5.221517  6.489773    6.9702  7.148529  7.063428  6.790391    6.453629 | 104.213141 | 127.589555  224.422676  361.947647  449.668284  484.937142  467.915972    415.619154 | 355.764275 |  1.855543   2.683049  5.088581  6.827725  7.487957  7.171895  6.170061     4.960071 |  100.0 |   4.213141 | 355.764275 |
    | 2000-01-03 01:30:00 | 4.098332  5.068281  6.411853  6.924996  7.129155  7.069242  6.814609     6.48362 | 103.551298 | 124.293364  195.003965  348.689985  440.967736  481.031105   469.06762    420.118797 |  360.89093 |  1.818208    2.55577  4.811946  6.661625  7.415843  7.193429  6.258168     5.066681 |  100.0 |   3.551298 |  360.89093 |
    | 2000-01-03 02:00:00 | 4.050775  4.930432  6.334287  6.877718    7.1075  7.072618  6.837048    6.513102 | 102.988541 |  121.39874  183.616276  335.763333  431.969701  476.687016  469.737175    424.311653 | 365.969269 |  1.785069   2.443778  4.538081  6.488454  7.335352  7.205939  6.339934     5.171697 |  100.0 |   2.988541 | 365.969269 |
    | 2000-01-03 02:30:00 | 4.008214  4.806327  6.257642  6.828669  7.083646  7.073589  6.857641    6.541949 |  102.51094 | 118.854227  173.739678  323.261646  422.743285  471.927985   469.92976    428.179779 |  370.97559 |  1.755651   2.344985  4.268942  6.309387  7.246819  7.209536  6.415084     5.274665 |  100.0 |    2.51094 |  370.97559 |
    | 2000-01-03 03:00:00 | 3.970151  4.694516   6.18234  6.778156  7.057692  7.072194  6.876332    6.570043 | 102.106358 | 116.615575  165.147568  311.253941  413.355974   466.78117   469.65299    431.707508 | 375.886495 |  1.729535   2.257627   4.00595  6.125603  7.150654  7.204366  6.483386     5.375143 |  100.0 |   2.106358 | 375.886495 |
    | 2000-01-03 03:30:00 | 3.936138  4.593721  6.100069  6.734045  7.030367  7.068536  6.893082     6.59727 | 101.764251 | 114.644609  157.651611  316.379843  405.252568  461.397514  468.927605    434.882442 | 380.679169 |   1.70635   2.180214  3.720239  5.965632  7.049593  7.190812  6.544668     5.472708 |  100.0 |   1.764251 | 380.679169 |
    | 2000-01-03 04:00:00 | 3.905769  4.502805  6.026447  6.682478  7.001207  7.062668   6.90786    6.623524 |  101.47548 | 112.908297  151.094517  292.339286  395.890138  455.691505  467.765471    437.694488 | 385.331557 |  1.685771   2.111479  3.465999  5.779241  6.941951  7.169079  6.598795     5.566962 |  100.0 |    1.47548 | 385.331557 |
    | 2000-01-03 04:30:00 | 3.878674  4.420763  5.857628   6.62442  6.969879  7.054613  6.920644    6.648705 | 101.232151 | 111.377979  145.344369  269.347696  385.490945  449.606132  466.172848    440.135225 | 389.822488 |  1.667508    2.05034  3.242793  5.570183  6.826543  7.139259  6.645663     5.657525 |  100.0 |   1.232151 | 389.822488 |
    | 2000-01-03 05:00:00 | 3.854521  4.346701   5.63813  6.562296  6.936285  7.044385  6.931416     6.67272 | 101.027458 | 110.028721  140.290098  250.769016  374.528791  443.131627  464.155018    442.197713 | 394.131768 |  1.651306   1.995869  3.043931  5.347415  6.703058  7.101417  6.685186     5.744043 |  100.0 |   1.027458 | 394.131768 |
    | 2000-01-03 05:30:00 | 3.833009  4.279821   5.43219  6.498283   6.90051  7.032007  6.940164    6.695484 | 100.855551 |  108.83878  135.837852  236.450218  363.411867  436.294711  461.719558    443.876592 | 398.240282 |  1.636936   1.947271  2.862829  5.118884  6.571869  7.055652  6.717304     5.826189 |  100.0 |   0.855551 | 398.240282 |
    | 2000-01-03 06:00:00 | 3.813863  4.219415    5.2357  6.433838  6.862748  7.017516  6.946882    6.716919 | 100.711411 | 107.789149   131.90805  225.158349  352.403187   429.14213  458.877843    445.168293 | 402.130101 |  1.624196   1.903857  2.694978  4.889844  6.433739  7.002128  6.741981     5.903657 |  100.0 |   0.711411 | 402.130101 |
    | 2000-01-03 06:30:00 | 3.796838  4.164848  5.088174  6.360821  6.822562  7.000914  6.951567    6.736955 | 100.590745 | 106.863186  128.432979  196.683464  340.154585  421.602195  455.634362    446.070306 | 405.784521 |  1.612906   1.865031  2.572127  4.631596  6.287133   6.94087  6.759196     5.976172 |  100.0 |   0.590745 | 405.784521 |
    | 2000-01-03 07:00:00 | 3.781711  4.115551  4.955102  6.287901  6.780221  6.982226   6.95422    6.755528 | 100.489885 | 106.046287  125.354833  185.622014  328.164288  413.737362  451.999009    446.581568 | 409.188131 |  1.602905   1.830276  2.463646   4.37502  6.133102  6.871999  6.768947      6.04348 |  100.0 |   0.489885 | 409.188131 |
    | 2000-01-03 07:30:00 |  3.76828  4.071016  4.834945  6.215631  6.736011  6.961502  6.954848    6.772581 | 100.405708 |  105.32562  122.624092  175.985561  316.527843  405.611814  447.986681     446.70277 | 412.326899 |  1.594049   1.799139  2.367596  4.122043  5.972751  6.795724  6.771258     6.105357 |  100.0 |   0.405708 | 412.326899 |
    | 2000-01-03 08:00:00 | 3.756365  4.030785  4.726347   6.14443  6.690226  6.938811  6.953468    6.788065 |  100.33556 | 104.689895  120.198195  167.563984  305.318149  397.289218  443.616689    446.436588 | 415.188263 |  1.586211   1.771224  2.282337  3.874086  5.807203  6.712335  6.766183     6.161604 |  100.0 |    0.33556 | 415.188263 |
    | 2000-01-03 08:30:00 | 3.745802  3.994446  4.628108  6.068565  6.648483   6.91467  6.950137    6.801942 | 100.277186 | 104.129164  118.040441  160.182253  306.184779  389.782807  438.993649    445.794858 | 417.761826 |  1.579277   1.746183  2.206482  3.611281  5.656726  6.623755   6.75394     6.212063 |  100.0 |   0.277186 | 417.761826 |
    | 2000-01-03 09:00:00 | 3.736444  3.961628  4.539169       6.0  6.600273  6.888703   6.94488    6.814179 | 100.228681 | 103.634656  116.119079  153.693866  284.961295  381.209701  434.051198    444.783013 | 420.038676 |  1.573147   1.723712  2.138847     3.375  5.483479  6.528641  6.734623     6.256603 |  100.0 |   0.228681 | 420.038676 |
    | 2000-01-03 09:30:00 | 3.728159  3.931997  4.458588  5.777518  6.545268  6.860555  6.937689     6.82475 | 100.188431 |  103.19863  114.406554   147.97568  262.133259  371.553811  428.728642    443.401126 | 422.010838 |  1.567728   1.703537  2.078423  3.169516  5.286522  6.425726  6.708213     6.295105 |  100.0 |   0.188431 | 422.010838 |
    | 2000-01-03 10:00:00 |  3.72083  3.905251  4.385531  5.571032  6.486897   6.83018  6.928556    6.833629 | 100.155078 | 102.814248  112.878874  142.923792  245.797798  361.453448  423.025931    441.649604 | 423.671247 |  1.562942   1.685421  2.024342  2.984343  5.078342  6.314895   6.67469     6.327466 |  100.0 |   0.155078 | 423.671247 |
    | 2000-01-03 10:30:00 |  3.71435  3.881115  4.319256  5.375807  6.426968  6.797685  6.917486    6.840795 | 100.127477 | 102.475473  111.515092  138.450209  232.990556  351.240685  416.971479    439.531604 | 425.013924 |  1.558715    1.66915  1.975859  2.814171  4.865492   6.19658  6.634082     6.353598 |  100.0 |   0.127477 | 425.013924 |
    | 2000-01-03 11:00:00 | 3.708624  3.859341  4.259103  5.219981    6.3598  6.762737  6.904456    6.846228 | 100.104666 | 102.176965  110.296866  134.480157  208.042222  339.985049  410.513414    437.045904 | 426.033533 |  1.554985   1.654534  1.932329  2.681759  4.627995  6.069629  6.586323     6.373421 |  100.0 |   0.104666 | 426.033533 |
    | 2000-01-03 11:30:00 | 3.703567  3.839707  4.204481  5.079264  6.291833  6.725533  6.889465    6.849912 | 100.085838 | 101.914005  109.208092  130.949876  195.930064  328.804536  403.698925     434.19578 |  426.72572 |  1.551694   1.641404  1.893194  2.564794  4.388821  5.934819  6.531429     6.386867 |  100.0 |   0.085838 |  426.72572 |
    | 2000-01-03 12:00:00 | 3.699104  3.822008  4.154863  4.952037  6.223753  6.686306  6.872533    6.851836 | 100.070317 |  101.68242  108.234594  127.804817   185.37203  317.822959  396.581064    430.989085 |  427.08743 |  1.548793   1.629611  1.857967  2.461173  4.150409  5.793054  6.469495      6.39389 |  100.0 |   0.070317 |  427.08743 |
    | 2000-01-03 12:30:00 | 3.695167   3.80606  4.109778  4.836873  6.156092  6.645311    6.8537    6.851994 | 100.057539 | 101.478523  107.363862  124.998155  176.137548  307.135967  389.215641    427.438016 | 427.117179 |  1.546235   1.619017  1.826226  2.369123  3.914612  5.645311  6.400694     6.394467 |  100.0 |   0.057539 | 427.117179 |
    | 2000-01-03 13:00:00 | 3.691695  3.791695  4.068802  4.732516  6.082092  6.609153  6.833535    6.850431 | 100.047032 | 101.299051  106.584837  122.489563  168.035024  310.578966  382.781108    423.653651 | 426.823312 |  1.543981   1.609502  1.797597  2.287141  3.658033  5.515347  6.327124     6.388762 |  100.0 |   0.047032 | 426.823312 |
    | 2000-01-03 13:30:00 | 3.688636  3.778763  4.031554  4.637853  6.015423  6.565736  6.811586    6.847151 | 100.038402 | 101.141121   105.88772  120.244195  160.904407  288.674706  375.131325    419.555561 | 426.206911 |  1.541996   1.600959  1.771755  2.213952  3.428046  5.359725   6.24716     6.376789 |  100.0 |   0.038402 | 426.206911 |
    | 2000-01-03 14:00:00 | 3.685941  3.767125  3.997691  4.551901  5.838023  6.515899  6.787519     6.84213 | 100.031322 | 101.002184   105.26381  118.231844  154.611366  267.533632  366.452998    415.087153 | 425.264435 |  1.540249   1.593288  1.748413  2.148469  3.224786   5.18167  6.159619      6.35847 |  100.0 |   0.031322 | 425.264435 |
    | 2000-01-03 14:30:00 | 3.683568  3.756656  3.966905  4.473789  5.635942  6.462001  6.761224    6.835341 | 100.025521 | 100.879988  104.705374   116.42624  149.042693  250.602016  357.191413    410.235089 | 423.991797 |  1.538711   1.586402  1.727315   2.08976  3.041979  4.989809   6.06414     6.333708 |  100.0 |   0.025521 | 423.991797 |
    | 2000-01-03 15:00:00 | 3.681479  3.747243  3.938915  4.402742  5.445359  6.405954  6.732756    6.826764 | 100.020773 | 100.772544  104.205523  114.804466  144.102612  237.285268  347.697455    405.017074 | 422.387053 |  1.537358   1.580223  1.708237  2.037023   2.87425  4.791067  5.960965     6.302442 |  100.0 |   0.020773 | 422.387053 |
    | 2000-01-03 15:30:00 | 3.679641  3.738784  3.913469  4.338074  5.262822  6.349072  6.702267    6.816396 | 100.016892 | 100.678095  103.758116   113.34647  139.709808  226.593478  338.206124    399.469009 |  420.45175 |  1.536167   1.574679  1.690978  1.989569   2.71786  4.590165  5.850692     6.264673 |  100.0 |   0.016892 |  420.45175 |
    | 2000-01-03 16:00:00 | 3.678024  3.731185  3.890339   4.27917  5.125083  6.283727  6.669331    6.804202 | 100.013722 | 100.595089  103.357672  112.034654  135.795003  199.823848  327.485455    393.522112 |   418.1818 |  1.535121   1.569706   1.67536    1.9468  2.602608  4.360374  5.731826     6.220286 |  100.0 |   0.013722 |   418.1818 |
    | 2000-01-03 16:30:00 | 3.676603  3.724362  3.869316  4.225485  4.999991  6.218101  6.634181     6.79017 | 100.011136 | 100.522158  102.999291  110.853528  132.298975  189.307665  316.921409    387.228863 | 415.578201 |  1.534201   1.565247  1.661222  1.908199  2.499992  4.130669  5.605272     6.169257 |  100.0 |   0.011136 | 415.578201 |
    | 2000-01-03 17:00:00 | 3.675354  3.718238  3.850211   4.17653  4.886257  6.152717  6.597075    6.774315 | 100.009028 | 100.458092  102.678593  109.789425  129.170932  180.059843  306.609146    380.644685 | 412.646824 |  1.533393    1.56125  1.648423  1.873312  2.408392   3.90288  5.472008     6.111653 |  100.0 |   0.009028 | 412.646824 |
    | 2000-01-03 17:30:00 | 3.674256  3.712744  3.832854  4.131867  4.782742  6.080912  6.564555     6.75717 | 100.007313 | 100.401827  102.391656  108.830251  126.367165  171.903094  310.196606    374.924382 | 409.489812 |  1.532683   1.557668  1.636833  1.841745  2.326428  3.653952  5.355498     6.049435 |  100.0 |   0.007313 | 409.489812 |
    | 2000-01-03 18:00:00 | 3.673291  3.707817  3.817087  4.091104  4.688432  6.016099  6.524723    6.738276 | 100.005918 | 100.352423  102.134969  107.965281  123.849944  164.688381  288.899802     367.98156 | 406.026039 |  1.532059    1.55446  1.626339  1.813153  2.252918  3.430371  5.213151     5.980955 |  100.0 |   0.005918 | 406.026039 |
    | 2000-01-03 18:30:00 | 3.672444  3.703401  3.802769  4.053889  4.602429   5.84628  6.478395    6.717288 | 100.004785 | 100.309051  101.905386   107.18498  121.586589  158.289859  268.293788    359.994799 | 402.197188 |  1.531511   1.551587  1.616835   1.78723  2.186853  3.232364  5.048089     5.904991 |  100.0 |   0.004785 | 402.197188 |
    | 2000-01-03 19:00:00 | 3.671701  3.699445  3.789771  4.019903  4.523931  5.649505  6.427831    6.694078 | 100.003865 | 100.270984  101.700085  106.480854  119.548707  152.600832  251.642877    351.386617 | 397.985902 |   1.53103   1.549014   1.60823  1.763708  2.127359  3.054089   4.86855     5.821112 |  100.0 |   0.003865 | 397.985902 |
    | 2000-01-03 19:30:00 | 3.671048  3.695901  3.777974   3.98886  4.452223  5.463913  6.374908    6.668681 | 100.003119 | 100.237577   101.51654  105.845322  117.711551  147.530521  238.479615       342.499 | 393.405171 |  1.530608   1.546712  1.600438  1.742349  2.073685   2.89038  4.681316     5.729481 |  100.0 |   0.003119 | 393.405171 |
    | 2000-01-03 20:00:00 | 3.670475  3.692729   3.76727    3.9605  4.386667  5.286175  6.320923    6.641229 | 100.002515 | 100.208266  101.352482  105.271604  116.053485   143.00142  227.859534    333.563819 | 388.486237 |  1.530238   1.544652  1.593384  1.722941  2.025178  2.737636  4.491048     5.630621 |  100.0 |   0.002515 | 388.486237 |
    | 2000-01-03 20:30:00 | 3.669972  3.689891  3.757562   3.93459  4.326695  5.150941  6.258673    6.611312 | 100.002026 | 100.182552  101.205876  104.753626  114.555537  138.947146   202.04266    323.428057 | 383.163625 |  1.529913    1.54281  1.586998  1.705298  1.981273  2.624064  4.272554     5.523097 |  100.0 |   0.002026 | 383.163625 |
    | 2000-01-03 21:00:00 | 3.669531  3.687352   3.74876  3.910916  4.271795  5.027884  6.195942    6.579148 | 100.001631 | 100.159997    101.0749  104.285941   113.20102  135.310665   191.62135    313.401922 | 377.485443 |  1.529628   1.541164  1.581218  1.689251  1.941476  2.522705  4.053349     5.407746 |  100.0 |   0.001631 | 377.485443 |
    | 2000-01-03 21:30:00 | 3.669144  3.685081  3.740782  3.889286  4.221509   4.91577  6.122836    6.554272 | 100.001312 | 100.140216  100.957916  103.863655  111.975215  132.042824  182.430832    323.661618 | 373.125306 |  1.529378   1.539692  1.575987   1.67465  1.905354  2.432006  3.799134     5.318712 |  100.0 |   0.001312 | 373.125306 |
    | 2000-01-03 22:00:00 | 3.668805  3.683052  3.733554  3.869523  4.175427  4.813504  6.056307    6.521403 | 100.001054 | 100.122871  100.853456  103.482367  110.865108  129.101137  174.301181    302.180454 | 367.406108 |  1.529159   1.538377  1.571255  1.661361  1.872529  2.350646  3.568951     5.201306 |  100.0 |   0.001054 | 367.406108 |
    | 2000-01-03 22:30:00 | 3.668507  3.681239  3.727007  3.851468  4.133178   4.72012       6.0    6.478379 | 100.000846 | 100.107662  100.760204  103.138112  109.859154  126.448766  167.089383     274.83994 | 359.992038 |  1.528967   1.537202  1.566975  1.649263  1.842668  2.277493     3.375     5.048032 |  100.0 |   0.000846 | 359.992038 |
    | 2000-01-03 23:00:00 | 3.668246  3.679619   3.72108  3.834975  4.094428  4.634754  5.798903    6.433598 | 100.000679 |  100.09433  100.676979  102.827319  108.947093   124.05368   160.67452    264.008329 | 352.362609 |  1.528798   1.536153  1.563105  1.638248  1.815476  2.211575  3.188999     4.888994 |  100.0 |   0.000679 | 352.362609 |
    | 2000-01-03 23:30:00 | 3.668017  3.678173  3.715715  3.819911  4.058876  4.556642  5.611437    6.385066 | 100.000544 | 100.082642  100.602722  102.546766  108.119782  121.887943  154.953928    248.754229 | 344.195181 |   1.52865   1.535217  1.559605  1.628216  1.790694  2.152057  3.020159       4.7172 |  100.0 |   0.000544 | 344.195181 |
    | 2000-01-04 00:00:00 | 3.667817  3.676882  3.710862  3.806155  4.026249  4.485099  5.433651    6.334424 | 100.000436 | 100.072399  100.536485  102.293545  107.369054   119.92712  149.840122    236.542342 |  335.78585 |  1.528521   1.534382  1.556443   1.61908  1.768089  2.098213  2.864095     4.538562 |  100.0 |   0.000436 |  335.78585 |
    | 2000-01-04 00:30:00 | 3.667641  3.675731  3.706473  3.793595    3.9963  4.419517  5.262643    6.282823 | 100.000349 | 100.063421  100.477417  102.065028  106.687601  118.149775  145.258286    226.583882 | 327.338598 |  1.528407   1.533637  1.553585   1.61076  1.747457  2.049418  2.717709     4.357204 |  100.0 |   0.000349 | 327.338598 |
    | 2000-01-04 01:00:00 | 3.667487  3.674703  3.702505  3.782131  3.968805  4.359352  5.134726    6.222664 | 100.000279 | 100.055553  100.424756  101.858841  106.068869  116.537053  141.144231    200.649448 |   317.6491 |  1.528308   1.532972  1.551004  1.603182  1.728614  2.005125  2.610599     4.146604 |  100.0 |   0.000279 |   317.6491 |
    | 2000-01-04 01:30:00 | 3.667352  3.673787  3.698919  3.771668   3.94356  4.304113  5.017909    6.162098 | 100.000223 | 100.048659  100.377818  101.672838  105.506971  115.072321  137.442712    190.791835 | 308.074946 |   1.52822    1.53238  1.548673   1.59628  1.711397  1.964858  2.514571     3.935495 |  100.0 |   0.000223 | 308.074946 |
    | 2000-01-04 02:00:00 | 3.667234  3.672971   3.69568  3.762122   3.92038  4.253366  4.911101    6.093504 | 100.000178 | 100.042619  100.335994  101.505079  104.996607  113.740867  134.106031     182.05444 | 314.266619 |  1.528144   1.531852  1.546568  1.589995  1.695657  1.928201  2.428264     3.697513 |  100.0 |   0.000178 | 314.266619 |
    | 2000-01-04 02:30:00 |  3.66713  3.672243  3.692755  3.753415  3.899094  4.206715  4.813335    6.031154 | 100.000142 | 100.037327  100.298735  101.353806  104.533002  112.529642  131.092886    174.287927 | 293.898836 |  1.528077   1.531381  1.544669  1.584273  1.681263  1.894787  2.350513     3.482213 |  100.0 |   0.000142 | 293.898836 |
    | 2000-01-04 03:00:00 | 3.667039  3.671594  3.690115  3.745475  3.879548  4.163807  4.723748    5.908643 | 100.000113 | 100.032691  100.265551  101.217431  104.111844  111.427045  128.367403    167.365844 | 274.220493 |  1.528018   1.530962  1.542956  1.579063  1.668096  1.864295  2.280315     3.289875 |  100.0 |   0.000113 | 274.220493 |
    | 2000-01-04 03:30:00 |  3.66696  3.671017  3.687732  3.738237  3.861601  4.124324  4.641573    5.719542 |  100.00009 | 100.028629  100.236004  101.094517  103.729239  110.422738  125.898334    161.180655 | 257.229699 |  1.527967   1.530589   1.54141   1.57432  1.656048  1.836438  2.216807     3.116986 |  100.0 |    0.00009 | 257.229699 |
    | 2000-01-04 04:00:00 |  3.66689  3.670503  3.685583  3.731641  3.845123  4.087975  4.566121    5.541672 | 100.000072 | 100.025072  100.209701  100.983763  103.381661  109.507481  123.658377    155.640556 | 243.718333 |  1.527922   1.530256  1.540017  1.570004  1.645021  1.810967  2.159239     2.958446 |  100.0 |   0.000072 | 243.718333 |
    | 2000-01-04 04:30:00 | 3.666829  3.670045  3.683645  3.725632  3.829996  4.054502  4.496781    5.371766 | 100.000057 | 100.021955  100.186293   100.88399  103.065917  108.673005  121.623612    150.666897 | 232.749594 |  1.527883    1.52996  1.538761  1.566077  1.634928  1.787656  2.106961     2.810698 |  100.0 |   0.000057 | 232.749594 |
    | 2000-01-04 05:00:00 | 3.666776  3.669638  3.681899  3.720159   3.81611  4.023667  4.433002     5.20738 | 100.000046 | 100.019226  100.165465  100.794134  102.779113  107.911887  119.773022    146.192079 | 223.699368 |  1.527848   1.529697   1.53763  1.562504  1.625689  1.766306  2.059407     2.671184 |  100.0 |   0.000046 | 223.699368 |
    | 2000-01-04 05:30:00 | 3.666729  3.669275  3.680325  3.715176  3.803367  3.995254  4.374292    5.089107 | 100.000036 | 100.016835  100.146937   100.71323   102.51862  107.217453  118.088087    142.157819 | 196.762509 |  1.527818   1.529463   1.53661  1.559254  1.617231  1.746739  2.016081     2.572896 |  100.0 |   0.000036 | 196.762509 |
    | 2000-01-04 06:00:00 | 3.666688  3.668953  3.678907  3.710641  3.791674  3.969069  4.320208    4.980686 | 100.000029 | 100.014742  100.130459  100.640406  102.282055  106.583693  116.552444    138.513716 | 187.716899 |  1.527791   1.529255  1.535692  1.556299  1.609488  1.728795  1.976551      2.48433 |  100.0 |   0.000029 | 187.716899 |
    | 2000-01-04 06:30:00 | 3.666652  3.668666   3.67763  3.706515  3.780947  3.944933  4.270352    4.881181 | 100.000023 | 100.012908  100.115808  100.574873  102.067251  106.005181  115.151598    135.216059 | 179.654085 |  1.527768    1.52907  1.534866  1.553613    1.6024  1.712331  1.940435     2.404342 |  100.0 |   0.000023 | 179.654085 |
    | 2000-01-04 07:00:00 |  3.66662  3.668411  3.676481  3.702763  3.771109  3.922684  4.224366    4.789761 | 100.000018 | 100.011302  100.102783  100.515916  101.872237  105.477014  113.872671    132.226831 | 172.448336 |  1.527748   1.528905  1.534122  1.551171  1.595912  1.697219  1.907398     2.331943 |  100.0 |   0.000018 | 172.448336 |
    | 2000-01-04 07:30:00 | 3.666593  3.668185  3.675447  3.699351  3.762087  3.902173  4.181925    4.705681 | 100.000014 | 100.009896  100.091208   100.46289  101.695222  104.994752  112.704197    129.512873 | 165.992456 |   1.52773   1.528758  1.533453  1.548953  1.589972  1.683342  1.877142      2.26628 |  100.0 |   0.000014 | 165.992456 |
    | 2000-01-04 08:00:00 | 3.666569  3.667984  3.674517   3.69625  3.753817  3.883263  4.142738    4.628276 | 100.000011 | 100.008665  100.080921  100.415211  101.534575   104.55437  111.635941    127.045183 | 160.194701 |  1.527715   1.528628  1.532852  1.546938  1.584537  1.670595  1.849405     2.206611 |  100.0 |   0.000011 | 160.194701 |
    | 2000-01-04 08:30:00 | 3.666548  3.667805   3.67368  3.693432  3.746237   3.86583  4.106538     4.55695 | 100.000009 | 100.007587  100.071783  100.372353  101.388813  104.152215   110.65874    124.798326 | 154.976263 |  1.527701   1.528513  1.532311  1.545109  1.579563  1.658884  1.823954      2.15229 |  100.0 |   0.000009 | 154.976263 |
    | 2000-01-04 09:00:00 | 3.666529  3.667646  3.672928  3.690873  3.739292   3.84976  4.073086    4.491168 | 100.000007 | 100.006642  100.063666  100.333837  101.256587  103.784964  109.764381    122.749936 | 150.269232 |  1.527689    1.52841  1.531824  1.543448  1.575012  1.648121   1.80058     2.102756 |  100.0 |   0.000007 | 150.269232 |
    | 2000-01-04 09:30:00 | 3.666513  3.667505  3.672253  3.688549  3.732932  3.834946  4.042162    4.430448 | 100.000006 | 100.005816  100.056458  100.299234  101.136665  103.449598  108.945479    120.880288 | 146.014901 |  1.527679   1.528319  1.531387   1.54194  1.570848  1.638228  1.779098     2.057513 |  100.0 |   0.000006 | 146.014901 |
    | 2000-01-04 10:00:00 | 3.666499  3.667379  3.671645   3.68644  3.727107  3.821292  4.013568    4.374359 | 100.000004 | 100.005092  100.050058  100.268153   101.02793  103.143364  108.195382    119.171945 | 142.162377 |  1.527669   1.528238  1.530995  1.540573  1.567041  1.629134   1.75934     2.016131 |  100.0 |   0.000004 | 142.162377 |
    | 2000-01-04 10:30:00 | 3.666486  3.667268    3.6711  3.684527  3.721775  3.808709  3.987123     4.32251 | 100.000004 | 100.004458  100.044376  100.240245  100.929361  102.863753  107.508089    117.609449 | 138.667412 |  1.527661   1.528166  1.530642  1.539332  1.563559  1.620775  1.741157     1.978226 |  100.0 |   0.000004 | 138.667412 |
    | 2000-01-04 11:00:00 | 3.666476  3.667169   3.67061  3.682791  3.716896  3.797115   3.96266     4.27455 | 100.000003 | 100.003903  100.039334  100.215192  100.840031  102.608477   106.87817    116.179064 | 135.491435 |  1.527654   1.528102  1.530326  1.538208  1.560375  1.613089  1.724416     1.943464 |  100.0 |   0.000003 | 135.491435 |
    | 2000-01-04 11:30:00 | 3.666466  3.667082  3.670171  3.681217  3.712433  3.786433  3.940027     4.23016 | 100.000002 | 100.003417  100.034859  100.192707  100.759094  102.375447   106.30071    114.868552 | 132.600726 |  1.527648   1.528046  1.530041  1.537188  1.557466  1.606024  1.708994     1.911546 |  100.0 |   0.000002 | 132.600726 |
    | 2000-01-04 12:00:00 | 3.666458  3.667004  3.669776   3.67979  3.708351  3.776595  3.919087    4.189054 | 100.000002 | 100.002992  100.030889  100.172533   100.68578  102.162756  105.771249    113.666982 | 129.965733 |  1.527643   1.527996  1.529787  1.536264  1.554807  1.599528  1.694781     1.882208 |  100.0 |   0.000002 | 129.965733 |
    | 2000-01-04 12:30:00 |  3.66645  3.666935  3.669422  3.678498  3.704619  3.767535   3.89971    4.150969 | 100.000001 | 100.002619  100.027367  100.154436   100.61939   101.96866  105.285734    112.564569 | 127.560488 |  1.527638   1.527951  1.529558  1.535427  1.552379  1.593558  1.681679     1.855216 |  100.0 |   0.000001 | 127.560488 |
    | 2000-01-04 13:00:00 | 3.666444  3.666874  3.669105  3.677327  3.701209  3.759193  3.881781    4.115669 | 100.000001 | 100.002293  100.024243  100.138207  100.559285  101.791563  104.840477    111.552535 | 125.362109 |  1.527634   1.527912  1.529353  1.534669  1.550161   1.58807  1.669598     1.830359 |  100.0 |   0.000001 | 125.362109 |
    | 2000-01-04 13:30:00 | 3.666438   3.66682   3.66882  3.676266  3.698093  3.751515   3.86519    4.082937 | 100.000001 | 100.002007  100.021473  100.123657  100.504885  101.630009  104.432119    110.622984 |  123.35039 |   1.52763   1.527877  1.529169  1.533983  1.548135  1.583026  1.658454     1.807449 |  100.0 |   0.000001 |  123.35039 |
    | 2000-01-04 14:00:00 | 3.666434  3.666772  3.668565  3.675306  3.695247   3.74445  3.849839    4.052577 | 100.000001 | 100.001757  100.019017  100.110615  100.455662  101.482663  104.057592      109.7688 | 121.507432 |  1.527627   1.527846  1.529004  1.533362  1.546287  1.578391  1.648174      1.78632 |  100.0 |   0.000001 | 121.507432 |
    | 2000-01-04 14:30:00 | 3.666429  3.666729  3.668336  3.674437  3.692649   3.73795  3.835637     4.02441 | 100.000001 | 100.001539   100.01684  100.098928  100.411134  101.348304  103.714099     108.98356 | 119.817349 |  1.527624   1.527818  1.528856    1.5328    1.5446  1.574132  1.638689     1.766819 |  100.0 |   0.000001 | 119.817349 |
    | 2000-01-04 15:00:00 | 3.666425  3.666691  3.668131  3.673651  3.690277  3.731972  3.822498     3.99827 |      100.0 | 100.001347   100.01491  100.088458  100.370866  101.225816   103.39908     108.26145 | 118.266003 |  1.527622   1.527794  1.528724  1.532292  1.543061  1.570221  1.629937     1.748811 |  100.0 |        0.0 | 118.266003 |
    | 2000-01-04 15:30:00 | 3.666422  3.666658  3.667948  3.672941  3.688113  3.726475  3.810345    3.974007 |      100.0 | 100.001179  100.013199   100.07908  100.334459  101.114175  103.110194      107.5972 |  116.84078 |   1.52762   1.527772  1.528605  1.531832  1.541658  1.566628   1.62186     1.732172 |  100.0 |        0.0 |  116.84078 |
    | 2000-01-04 16:00:00 | 3.666419  3.666628  3.667783  3.672298   3.68614  3.721423  3.799104    3.951484 |      100.0 | 100.001032  100.011684  100.070681  100.301552  101.012446  102.845295    106.986026 | 115.530403 |  1.527618   1.527753  1.528499  1.531417  1.540378  1.563329  1.614407     1.716792 |  100.0 |        0.0 | 115.530403 |
    | 2000-01-04 16:30:00 | 3.666417  3.666602  3.667636  3.671717   3.68434   3.71678   3.78871    3.930573 |      100.0 | 100.000904  100.010341  100.063163  100.271817  100.919771   102.60242    106.423575 | 114.324762 |  1.527616   1.527736  1.528404  1.531041  1.539212    1.5603  1.607529      1.70257 |  100.0 |        0.0 | 114.324762 |
    | 2000-01-04 17:00:00 | 3.666415  3.666578  3.667504  3.671192    3.6827  3.712516  3.779101    3.911158 |      100.0 | 100.000791  100.009151  100.056433  100.244955  100.835366  102.379765    105.905882 | 113.214777 |  1.527615   1.527721  1.528319  1.530702  1.538149   1.55752  1.601182     1.689414 |  100.0 |        0.0 | 113.214777 |
    | 2000-01-04 17:30:00 | 3.666413  3.666558  3.667386  3.670718  3.681206  3.708599  3.770218     3.89313 |      100.0 | 100.000693  100.008097   100.05041  100.220695  100.758513  102.175678     105.42933 | 112.192272 |  1.527614   1.527707  1.528243  1.530395  1.537181  1.554969  1.595325     1.677241 |  100.0 |        0.0 | 112.192272 |
    | 2000-01-04 18:00:00 | 3.666411  3.666539  3.667281   3.67029  3.679844  3.705004  3.762009    3.876392 |      100.0 | 100.000606  100.007164  100.045022   100.19879  100.688553   101.98864    104.990614 | 111.249874 |  1.527613   1.527696  1.528174  1.530118  1.536299  1.552629  1.589921     1.665974 |  100.0 |        0.0 | 111.249874 |
    | 2000-01-04 18:30:00 | 3.666409  3.666523  3.667186  3.669903  3.678605  3.701704  3.754425     3.86085 |      100.0 | 100.000531  100.006338  100.040203  100.179018  100.624887  101.817258    104.586712 | 110.380914 |  1.527612   1.527685  1.528113  1.529868  1.535497  1.550483  1.584936     1.655545 |  100.0 |        0.0 | 110.380914 |
    | 2000-01-04 19:00:00 | 3.666408  3.666509  3.667102  3.669554  3.677476  3.698676  3.747419    3.846421 |      100.0 | 100.000465  100.005606  100.035892  100.161175  100.566962  101.660252    104.214855 | 109.579352 |  1.527611   1.527676  1.528059  1.529643  1.534766  1.548515  1.580338     1.645889 |  100.0 |        0.0 | 109.579352 |
    | 2000-01-04 19:30:00 | 3.666407  3.666496  3.667027  3.669239  3.676449    3.6959   3.74095    3.833026 |      100.0 | 100.000407  100.004959  100.032038  100.145078  100.514276  101.516445    103.872506 | 108.839708 |   1.52761   1.527668   1.52801   1.52944  1.534102  1.546711  1.576097     1.636948 |  100.0 |        0.0 | 108.839708 |
    | 2000-01-04 20:00:00 | 3.666406  3.666485  3.666959  3.668955  3.675515  3.693354  3.734977    3.820591 |      100.0 | 100.000356  100.004385  100.028593  100.130558  100.466368  101.384755    103.557336 | 108.156999 |  1.527609    1.52766  1.527967  1.529256  1.533497  1.545058  1.572187     1.628668 |  100.0 |        0.0 | 108.156999 |
    | 2000-01-04 20:30:00 | 3.666405  3.666475  3.666899  3.668699  3.674665   3.69102  3.729464    3.809051 |      100.0 | 100.000312  100.003878  100.025514  100.117466  100.422816  101.264188    103.267206 | 107.526687 |  1.527609   1.527654  1.527928   1.52909  1.532948  1.543543  1.568581     1.621001 |  100.0 |        0.0 | 107.526687 |
    | 2000-01-04 21:00:00 | 3.666404  3.666466  3.666845  3.668468  3.673893  3.688882  3.724378    3.798341 |      100.0 | 100.000273  100.003429  100.022763  100.105663  100.383235  101.153831    103.000149 | 106.944631 |  1.527608   1.527648  1.527893  1.528941  1.532448  1.542156  1.565258     1.613901 |  100.0 |        0.0 | 106.944631 |
    | 2000-01-04 21:30:00 | 3.666404  3.666458  3.666797   3.66826  3.673191  3.686924  3.719686    3.788404 |      100.0 | 100.000239  100.003032  100.020305  100.095025  100.347272  101.052842    102.754358 |  106.40705 |  1.527608   1.527643  1.527862  1.528807  1.531994  1.540886  1.562195     1.607326 |  100.0 |        0.0 |  106.40705 |
    | 2000-01-04 22:00:00 | 3.666403  3.666452  3.666754  3.668072  3.672553   3.68513  3.715359    3.779186 |      100.0 | 100.000209   100.00268  100.018109   100.08544  100.314606  100.960449    102.528167 | 105.910484 |  1.527608   1.527639  1.527834  1.528686  1.531581  1.539724  1.559373     1.601238 |  100.0 |        0.0 | 105.910484 |
    | 2000-01-04 22:30:00 | 3.666403  3.666445  3.666715  3.667903  3.671973  3.683489   3.71137    3.770637 |      100.0 | 100.000183  100.002369  100.016148  100.076805  100.284942  100.875942    102.320044 | 105.451761 |  1.527607   1.527635  1.527809  1.528576  1.531207   1.53866  1.556774     1.595601 |  100.0 |        0.0 | 105.451761 |
    | 2000-01-04 23:00:00 | 3.666402   3.66644  3.666681  3.667751  3.671447  3.681987  3.707694     3.76271 |      100.0 |  100.00016  100.002094  100.014397  100.069029  100.258012  100.798666    102.128575 | 105.027974 |  1.527607   1.527631  1.527787  1.528478  1.530867  1.537687   1.55438     1.590382 |  100.0 |        0.0 | 105.027974 |
    | 2000-01-04 23:30:00 | 3.666402  3.666435   3.66665  3.667614  3.670969  3.680613  3.704307     3.75536 |      100.0 |  100.00014   100.00185  100.012834  100.062027  100.233571  100.728023    101.952459 | 104.636452 |  1.527607   1.527628  1.527767  1.528389  1.530558  1.536797  1.552176     1.585551 |  100.0 |        0.0 | 104.636452 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _kinw_impl_euler_negative_inflow:

negative inflow
_______________

|kinw_impl_euler| even  sticks to the water balance for negative inflows.  It does so
by successively removing water from the most upstream to the most downstream channel
segment.  As soon as the complete channel becomes dry, it forwards the remaining
deficit as negative outflow:

.. integration-test::

    >>> solver.watervolumetolerance(1e-10)
    >>> model.prepare_states(100.0)
    >>> nodes.input1.sequences.sim.series = -200.0
    >>> conditions = test("kinw_impl_euler_negative_inflow", get_conditions="2000-01-01")
    |                date |                                                                       waterdepth |      inflow |                                                                               internalflow |     outflow |                                                                       watervolume | input1 |     input2 |      output |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 | 2.938452  3.624625  3.705383  3.716149  3.717605  3.717802  3.717829    3.717832 |      -200.0 |   61.316776    94.788843     99.29447    99.904416    99.987049    99.998245     99.999762 |   99.999968 | 1.090616  1.500736  1.552876  1.559888  1.560837  1.560966  1.560983     1.560986 | -200.0 |        0.0 |   99.999968 |
    | 2000-01-01 00:30:00 | 2.109798  3.468149  3.674451  3.710537  3.716648  3.717646  3.717804    3.717828 |      -200.0 |   31.569852     86.40588    97.554181    99.586154    99.932736    99.989369     99.998354 |   99.999749 |  0.67379  1.402032  1.532809  1.556231  1.560214  1.560864  1.560967     1.560983 | -200.0 |        0.0 |   99.999749 |
    | 2000-01-01 01:00:00 | 1.143095  3.271185  3.623462  3.698906  3.714251  3.717186   3.71772    3.717814 |      -200.0 |    9.885214    76.492861    94.724852     98.92855    99.796716    99.963268       99.9936 |   99.998916 | 0.295997  1.282138  1.499992  1.548664  1.558651  1.560564  1.560913     1.560974 | -200.0 |        0.0 |   99.998916 |
    | 2000-01-01 01:30:00 |      0.0  2.965508  3.544295  3.678501  3.709436  3.716137  3.717506    3.717772 |      -200.0 |  -35.557301    62.478813    90.428482    97.780998    99.523804    99.903732     99.981424 |   99.996546 |      0.0  1.105673  1.449682   1.53543  1.555514   1.55988  1.560773     1.560947 | -200.0 |        0.0 |   99.996546 |
    | 2000-01-01 02:00:00 |      0.0  2.139535  3.398406  3.642358   3.70045  3.714017  3.717033    3.717672 |      -200.0 |      -200.0    32.449574     82.81514    95.767632    99.015681    99.783449     99.954602 |   99.990863 |      0.0  0.687264  1.359024  1.512115  1.549667  1.558498  1.560465     1.560881 | -200.0 |        0.0 |   99.990863 |
    | 2000-01-01 02:30:00 |      0.0  1.180344   3.20943  3.588127  3.685525  3.710175  3.716105     3.71746 |      -200.0 |      -200.0    10.485241    73.528838    92.792783    98.175111    99.565686     99.901915 |   99.978812 |      0.0   0.30839  1.245546   1.47744  1.539979  1.555995   1.55986     1.560743 | -200.0 |        0.0 |   99.978812 |
    | 2000-01-01 03:00:00 |      0.0       0.0  2.927275  3.509078  3.662358   3.70375  3.714435     3.71705 | -199.999997 | -199.999997   -28.672128    60.840426     88.55475    96.878696    99.202108     99.807127 |   99.955551 |      0.0       0.0  1.084423  1.427554  1.524996  1.551813  1.558771     1.560476 | -200.0 |   0.000003 |   99.955551 |
    | 2000-01-01 03:30:00 |      0.0       0.0   2.09749  3.363661  3.624012  3.693096  3.711554    3.716306 | -199.999971 | -199.999971  -199.999971    31.209633    81.059367    94.755106    98.601016     99.643806 |   99.913318 |      0.0       0.0  0.668246  1.337825  1.500344  1.544891  1.556893     1.559991 | -200.0 |   0.000029 |   99.913318 |
    | 2000-01-01 04:00:00 |      0.0       0.0    1.1276  3.175592   3.56813   3.67654  3.706839    3.715025 | -199.999825 | -199.999825  -199.999825     9.640875    71.933551    91.709716    97.671104     99.376804 |   99.840644 |      0.0       0.0  0.290892  1.225698  1.464747   1.53416  1.553823     1.559156 | -200.0 |   0.000175 |   99.840644 |
    | 2000-01-01 04:30:00 |      0.0       0.0       0.0  2.867554  3.485157    3.6515  3.699414    3.712916 | -199.999174 | -199.999174  -199.999174   -38.392243    58.331175    87.295056    96.274584      98.95724 |   99.721007 |      0.0       0.0       0.0  1.051596  1.412612  1.517997  1.548994     1.557781 | -200.0 |   0.000826 |   99.721007 |
    | 2000-01-01 05:00:00 |      0.0       0.0       0.0  2.031474  3.337306  3.611325  3.687667     3.70951 | -199.996806 | -199.996806  -199.996806  -199.996806    29.316513    79.742217    94.058559     98.295496 |   99.528019 |      0.0       0.0       0.0  0.638832  1.321845  1.492228  1.541368     1.555562 | -200.0 |   0.003194 |   99.528019 |
    | 2000-01-01 05:30:00 |      0.0       0.0       0.0   1.04372  3.148195  3.553855  3.669975    3.704191 | -199.989485 | -199.989485  -199.989485  -199.989485     8.371519    70.656705     90.94109     97.303833 |   99.227067 |      0.0       0.0       0.0  0.263782  1.209732  1.455716  1.529915       1.5521 | -200.0 |   0.010515 |   99.227067 |
    | 2000-01-01 06:00:00 |      0.0       0.0       0.0       0.0  2.797623  3.465349  3.643282    3.696029 | -199.969685 | -199.969685  -199.969685  -199.969685   -53.424189     55.46978    86.260004     95.818808 |   98.766284 |      0.0       0.0       0.0       0.0  1.013723  1.400293  1.512709     1.546795 | -200.0 |   0.030315 |   98.766284 |
    | 2000-01-01 06:30:00 |      0.0       0.0       0.0       0.0  1.953868  3.313944   3.60132    3.683417 | -199.921821 | -199.921821  -199.921821  -199.921821  -199.921821    27.174105    78.585151     93.511434 |   98.056763 |      0.0       0.0       0.0       0.0   0.60495  1.307753  1.485842     1.538613 | -200.0 |   0.078179 |   98.056763 |
    | 2000-01-01 07:00:00 |      0.0       0.0       0.0       0.0   0.94407  3.123142  3.542217     3.66478 | -199.816649 | -199.816649  -199.816649  -199.816649  -199.816649     6.979395    69.500735     90.317278 |   97.013789 |      0.0       0.0       0.0       0.0  0.232717  1.195215  1.448372      1.52656 | -200.0 |   0.183351 |   97.013789 |
    | 2000-01-01 07:30:00 |      0.0       0.0       0.0       0.0       0.0  2.723251  3.447594    3.636524 | -199.603786 | -199.603786  -199.603786  -199.603786  -199.603786    -70.31637    52.516709     85.338347 |   95.445008 |      0.0       0.0       0.0       0.0       0.0  0.974115  1.389293     1.508368 | -200.0 |   0.396214 |   95.445008 |
    | 2000-01-01 08:00:00 |      0.0       0.0       0.0       0.0       0.0  1.873086  3.292393    3.592824 | -199.202695 | -199.202695  -199.202695  -199.202695  -199.202695  -199.202695    25.038214       77.5265 |   93.048253 |      0.0       0.0       0.0       0.0       0.0  0.570482  1.294814     1.480429 | -200.0 |   0.797305 |   93.048253 |
    | 2000-01-01 08:30:00 |      0.0       0.0       0.0       0.0       0.0  0.844482  3.100037    3.532139 | -198.492968 | -198.492968  -198.492968  -198.492968  -198.492968  -198.492968     5.712287     68.444436 |   89.779086 |      0.0       0.0       0.0       0.0       0.0  0.202912  1.181896     1.442026 | -200.0 |   1.507032 |   89.779086 |
    | 2000-01-01 09:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0  2.657526    3.432243 | -197.305147 | -197.305147  -197.305147  -197.305147  -197.305147  -197.305147   -84.576078     49.983516 |   84.546092 |      0.0       0.0       0.0       0.0       0.0       0.0  0.939689     1.379814 | -200.0 |   2.694853 |   84.546092 |
    | 2000-01-01 09:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0  1.813548    3.274596 | -195.413644 | -195.413644  -195.413644  -195.413644  -195.413644  -195.413644  -195.413644     23.524783 |   76.658555 |      0.0       0.0       0.0       0.0       0.0       0.0    0.5456     1.284173 | -200.0 |   4.586356 |   76.658555 |
    | 2000-01-01 10:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0  0.799043    3.082329 | -192.533359 | -192.533359  -192.533359  -192.533359  -192.533359  -192.533359  -192.533359      5.174978 |   67.641178 |      0.0       0.0       0.0       0.0       0.0       0.0  0.189725     1.171734 | -200.0 |   7.466641 |   67.641178 |
    | 2000-01-01 10:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0    2.645299 | -188.321441 | -188.321441  -188.321441  -188.321441  -188.321441  -188.321441  -188.321441    -82.918715 |   49.520084 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0     0.933344 | -200.0 |  11.678559 |   49.520084 |
    | 2000-01-01 11:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0    1.850576 | -182.385164 | -182.385164  -182.385164  -182.385164  -182.385164  -182.385164  -182.385164   -182.385164 |   24.459996 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0     0.561022 | -200.0 |  17.614836 |   24.459996 |
    | 2000-01-01 11:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0    0.950158 | -174.296264 | -174.296264  -174.296264  -174.296264  -174.296264  -174.296264  -174.296264   -174.296264 |    7.060867 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0      0.23458 | -200.0 |  25.703736 |    7.060867 |
    | 2000-01-01 12:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -163.611353 | -163.611353  -163.611353  -163.611353  -163.611353  -163.611353  -163.611353   -163.611353 |  -33.289327 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |  36.388647 |  -33.289327 |
    | 2000-01-01 12:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -149.897274 | -149.897274  -149.897274  -149.897274  -149.897274  -149.897274  -149.897274   -149.897274 | -149.897274 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |  50.102726 | -149.897274 |
    | 2000-01-01 13:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -132.759651 | -132.759651  -132.759651  -132.759651  -132.759651  -132.759651  -132.759651   -132.759651 | -132.759651 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |  67.240349 | -132.759651 |
    | 2000-01-01 13:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -111.872431 | -111.872431  -111.872431  -111.872431  -111.872431  -111.872431  -111.872431   -111.872431 | -111.872431 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |  88.127569 | -111.872431 |
    | 2000-01-01 14:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |  -87.006038 |  -87.006038   -87.006038   -87.006038   -87.006038   -87.006038   -87.006038    -87.006038 |  -87.006038 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 | 112.993962 |  -87.006038 |
    | 2000-01-01 14:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |  -58.051805 |  -58.051805   -58.051805   -58.051805   -58.051805   -58.051805   -58.051805    -58.051805 |  -58.051805 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 | 141.948195 |  -58.051805 |
    | 2000-01-01 15:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |   -25.04065 |   -25.04065    -25.04065    -25.04065    -25.04065    -25.04065    -25.04065     -25.04065 |   -25.04065 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |  174.95935 |   -25.04065 |
    | 2000-01-01 15:30:00 | 0.108198  0.001615  0.000001       0.0       0.0       0.0       0.0         0.0 |   11.845554 |    0.168443     0.000151          0.0          0.0          0.0          0.0           0.0 |         0.0 | 0.021019  0.000303       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 | 211.845554 |         0.0 |
    | 2000-01-01 16:00:00 | 0.506213  0.023574  0.000128       0.0       0.0       0.0       0.0         0.0 |   52.270855 |    2.319765     0.013165     0.000002          0.0          0.0          0.0           0.0 |         0.0 | 0.110931  0.004455  0.000024       0.0       0.0       0.0       0.0          0.0 | -200.0 | 252.270855 |         0.0 |
    | 2000-01-01 16:30:00 | 1.056557  0.101086  0.001568  0.000001       0.0       0.0       0.0         0.0 |   95.750601 |    8.559971     0.150275     0.000143          0.0          0.0          0.0           0.0 |         0.0 | 0.267874  0.019592  0.000294       0.0       0.0       0.0       0.0          0.0 | -200.0 | 295.750601 |         0.0 |
    | 2000-01-01 17:00:00 | 1.668371  0.265983  0.008898  0.000026       0.0       0.0       0.0         0.0 |  141.664962 |   20.047222     0.768739     0.002591          0.0          0.0          0.0           0.0 |         0.0 | 0.486786  0.054294  0.001673  0.000005       0.0       0.0       0.0          0.0 | -200.0 | 341.664962 |         0.0 |
    | 2000-01-01 17:30:00 | 2.296697  0.530082  0.032483  0.000242       0.0       0.0       0.0         0.0 |  189.279607 |   37.323438     2.513209     0.022485     0.000006          0.0          0.0           0.0 |         0.0 | 0.760307  0.116952  0.006157  0.000045       0.0       0.0       0.0          0.0 | -200.0 | 389.279607 |         0.0 |
    | 2000-01-01 18:00:00 | 2.918314  0.884089  0.088596  0.001397  0.000001       0.0       0.0         0.0 |   237.77215 |   60.460028     6.201451     0.120453     0.000118          0.0          0.0           0.0 |         0.0 | 1.079469  0.214617  0.017102  0.000262       0.0       0.0       0.0          0.0 | -200.0 |  437.77215 |         0.0 |
    | 2000-01-01 18:30:00 | 3.519864  1.308115   0.19578  0.005764  0.000013       0.0       0.0         0.0 |  286.262607 |   89.126141    12.680568     0.457283     0.001256          0.0          0.0           0.0 |         0.0 | 1.434314  0.352219  0.039104  0.001083  0.000002       0.0       0.0          0.0 | -200.0 | 486.262607 |         0.0 |
    | 2000-01-01 19:00:00 | 4.092961  1.780839  0.368485  0.018453  0.000097       0.0       0.0         0.0 |  333.846018 |  122.659194    22.715068     1.341182     0.008749     0.000001          0.0           0.0 |         0.0 | 1.814451  0.532119  0.077577  0.003481  0.000018       0.0       0.0          0.0 | -200.0 | 533.846018 |         0.0 |
    | 2000-01-01 19:30:00 | 4.632069  2.283319  0.612045   0.04838  0.000517       0.0       0.0         0.0 |  379.625318 |  160.143897     36.89373     3.230629     0.043751     0.000023          0.0           0.0 |         0.0 | 2.209517  0.753969  0.138171  0.009218  0.000097       0.0       0.0          0.0 | -200.0 | 579.625318 |         0.0 |
    | 2000-01-01 20:00:00 | 5.133479  2.799737  0.922388  0.107921  0.002123  0.000003       0.0         0.0 |  422.742733 |   200.49384    55.555078     6.693003      0.16772     0.000238          0.0           0.0 |         0.0 | 2.609565  1.014859  0.226123  0.020963  0.000398       0.0       0.0          0.0 | -200.0 | 622.742733 |         0.0 |
    | 2000-01-01 20:30:00 | 5.594755  3.317137  1.289309  0.210562  0.007056  0.000019       0.0         0.0 |  462.408181 |  242.529574    78.742704    12.344014     0.517197      0.00176          0.0           0.0 |         0.0 | 3.005347  1.309675   0.34564  0.042251  0.001326  0.000004       0.0          0.0 | -200.0 | 662.408181 |         0.0 |
    | 2000-01-01 21:00:00 | 6.004438  3.822761  1.699496  0.367021  0.019639  0.000113       0.0         0.0 |  497.923464 |     284.084    106.06381    20.767625     1.332053     0.009707     0.000002           0.0 |         0.0 | 3.390258  1.630112  0.499173  0.077235  0.003706  0.000021       0.0          0.0 | -200.0 | 697.923464 |         0.0 |
    | 2000-01-01 21:30:00 | 6.123514  4.256454  2.126973  0.579656  0.046837   0.00051       0.0         0.0 |  528.701418 |  300.240731   133.401353    32.076311     2.937312     0.041442     0.000022           0.0 |         0.0 | 3.801487  1.930422  0.681558  0.129686  0.008919  0.000096       0.0          0.0 | -200.0 | 728.701418 |         0.0 |
    | 2000-01-01 22:00:00 |  6.24551  4.638014  2.555778  0.843384  0.097731   0.00187  0.000002         0.0 |   554.27953 |  318.175996   160.590705    46.201751        5.699     0.142002     0.000192           0.0 |         0.0 | 4.226473  2.214076  0.887459  0.202591  0.018922  0.000351       0.0          0.0 | -200.0 |  754.27953 |         0.0 |
    | 2000-01-01 22:30:00 | 6.366988  4.979576  2.976306  1.148947  0.181814  0.005723  0.000014         0.0 |  574.327871 |  337.170715   187.528781    62.946081     9.978303     0.403538     0.001241           0.0 |         0.0 | 4.653356  2.483432  1.111707  0.297933  0.036156  0.001075  0.000003          0.0 | -200.0 | 774.327871 |         0.0 |
    | 2000-01-01 23:00:00 | 6.484858  5.288813  3.382775  1.485905  0.306609  0.015001  0.000073         0.0 |  588.651531 |  356.580054   214.111944    82.022546    16.095224     0.979305     0.006192      0.000001 |         0.0 | 5.071085  2.739874  1.349468  0.416602  0.063365  0.002827  0.000014          0.0 | -200.0 | 788.651531 |         0.0 |
    | 2000-01-01 23:30:00 | 6.596409  5.570544  3.771668  1.844339  0.475781  0.034442  0.000311         0.0 |  597.187983 |  375.779414   240.202562   103.084868    24.301074     2.083279     0.024795       0.00001 |         0.0 |  5.46962  2.983912   1.59628  0.558413  0.103357  0.006532  0.000058          0.0 | -200.0 | 797.187983 |         0.0 |
    | 2000-01-02 00:00:00 | 6.699336  5.827695  4.140809   2.21552  0.688666  0.070553    0.0011    0.000001 |       600.0 |  394.176779   265.617131   125.747657    34.758575     3.975552     0.082246      0.000079 |         0.0 | 5.840102   3.21532  1.848045  0.722193  0.158766   0.01354  0.000206          0.0 | -200.0 |      800.0 |         0.0 |
    | 2000-01-02 00:30:00 |  6.79174   6.01912  4.480311  2.590124  0.940769  0.130859  0.003318    0.000006 |  597.264896 |  411.231131   285.981171   148.988166    47.459432     6.935413      0.23186        0.0005 |         0.0 | 6.174963   3.44077  2.094633  0.904945   0.23171  0.025606  0.000623     0.000001 | -200.0 | 797.264896 |         0.0 |
    | 2000-01-02 01:00:00 | 6.872121  6.087589  4.772492  2.956735   1.22447  0.222348  0.008717     0.00003 |  589.260928 |  426.468822   295.204877   170.898237    62.100624    11.219291     0.567162      0.002504 |         0.0 | 6.467989  3.677045  2.318385   1.10078  0.323296   0.04478  0.001639     0.000006 | -200.0 | 789.260928 |         0.0 |
    | 2000-01-02 01:30:00 | 6.939355  6.157347  5.027954  3.308669  1.531165  0.349783  0.020282    0.000128 |  576.351729 |  439.495708   305.090601   191.548092    78.325292    17.032696     1.226632      0.010243 |    0.000002 |  6.71433  3.918974  2.522761  1.304581  0.433623  0.073231  0.003829     0.000024 | -200.0 | 776.351729 |    0.000002 |
    | 2000-01-02 02:00:00 | 6.992667   6.22688  5.254177  3.642272  1.852609  0.514666  0.042403    0.000465 |  558.969595 |  450.004485   315.359576   211.028646    95.762871    24.511922     2.387474      0.035095 |    0.000019 | 6.910467  4.161335  2.710557   1.51206  0.561874  0.113055  0.008063     0.000087 | -200.0 | 758.969595 |    0.000019 |
    | 2000-01-02 02:30:00 | 7.031599  6.294731  5.456503  3.955781  2.181547  0.715277  0.080656    0.001451 |    537.5984 |  457.777432   325.743451   229.423088    114.05553    33.715311     4.251054      0.102912 |    0.000126 | 7.054144  4.398996  2.883933  1.719721  0.706487  0.166091   0.01553     0.000272 | -200.0 |   737.5984 |    0.000126 |
    | 2000-01-02 03:00:00 | 7.055968  6.359539  5.638729  4.248598  2.511905  0.947502  0.141103    0.003967 |  512.756821 |   462.68511   335.975642   246.790949    132.87293    44.622988     7.025265      0.263209 |    0.000674 | 7.144274  4.627073  3.044466  1.924774  0.865337  0.233767  0.027701     0.000745 | -200.0 | 712.756821 |    0.000674 |
    | 2000-01-02 03:30:00 | 7.065837  6.420075  5.803518  4.520813  2.838768  1.205895  0.229205    0.009645 |  484.982427 |  464.681725   345.797435   263.161536   151.918399    57.143318    10.907206      0.597121 |    0.002964 | 7.180815  4.841065  3.193211  2.125011  1.035932  0.316992  0.046259     0.001814 | -200.0 | 684.982427 |    0.002964 |
    | 2000-01-02 04:00:00 | 7.061472  6.475266  5.952692  4.772893  3.158252  1.484542  0.348716    0.021136 |  454.817086 |  463.797952   354.966226   278.533704   170.929541    71.123875    16.067429      1.220231 |    0.010972 | 7.164649  5.036962  3.330789  2.318699  1.215582  0.416093  0.072984     0.003991 | -200.0 | 654.817086 |    0.010972 |
    | 2000-01-02 04:30:00 |  7.04331  6.524211  6.027037  4.994526  3.465067  1.777157  0.500935    0.042224 |   422.79402 |  460.131971   363.263318   287.018494   188.765421    86.245311    22.624897      2.277933 |    0.034848 | 7.097441  5.211325   3.46803  2.495554  1.400118   0.53061  0.109609     0.008028 | -200.0 |  622.79402 |    0.034848 |
    | 2000-01-02 05:00:00 | 7.011927  6.566186  6.067751  5.185657  3.754995  2.077535  0.684625    0.077652 |  389.426711 |  453.839448   370.500321   292.478375   205.008216   102.123551     30.63051      3.934473 |    0.096572 | 6.981498  5.361336  3.608469  2.653001  1.585311  0.659297  0.157662     0.014937 | -200.0 | 589.426711 |    0.096572 |
    | 2000-01-02 05:30:00 | 6.968002  6.600645  6.108541  5.352229  4.025923  2.380084  0.896522    0.132575 |  355.199767 |   445.12307   376.523795   298.127113   219.827271   118.408461    40.064911      6.359028 |    0.236997 | 6.819636  5.484815  3.749583   2.79394  1.767864  0.800316  0.218332     0.025956 | -200.0 | 555.199767 |    0.236997 |
    | 2000-01-02 06:00:00 | 6.912294  6.627214  6.148684  5.498734  4.277112   2.68006  1.132097    0.211712 |  320.561767 |  434.222187   381.218026   303.839303   233.379999   134.796929    50.843969      9.711472 |    0.521987 | 6.615047  5.580222  3.888865  2.920767  1.945314  0.951431  0.292371     0.042497 | -200.0 | 520.561767 |    0.521987 |
    | 2000-01-02 06:30:00 | 6.845613   6.64568  6.187489  5.628575  4.508683   2.97362  1.386261    0.318458 |  285.920022 |  421.403015   384.506024   309.493872   245.802916   151.037274    62.829653     14.129278 |    1.044791 | 6.371178  5.646637  4.023887  3.035411  2.115892  1.110205  0.380032     0.066049 | -200.0 | 485.920022 |    1.044791 |
    | 2000-01-02 07:00:00 | 6.768795  6.655979  6.224312  5.744324  4.721281  3.257783  1.653882    0.454288 |   251.63713 |  406.949771   386.348973   314.973455   257.207752   166.927723    75.843805     19.716513 |    1.923146 | 6.091615  5.683718  4.152363  3.139389  2.278396  1.274156  0.481061     0.098078 | -200.0 |  451.63713 |    1.923146 |
    | 2000-01-02 07:30:00 | 6.682684  6.658185  6.258573  5.847912  4.915839  3.530328  1.930102    0.618646 |  218.029154 |  391.157075   386.744394   320.166777   267.681092   182.311338     89.68261     26.535804 |    3.291985 | 5.779985  5.691661  4.272202  3.233863  2.432062  1.440888  0.594725     0.139916 | -200.0 | 418.029154 |    3.291985 |
    | 2000-01-02 08:00:00 | 6.588109  6.652487  6.289763  5.940776  5.093424  3.789693  2.210519    0.809284 |  185.365238 |  374.323963   385.723369   324.971528   277.286283   197.069287   104.130087     34.603903 |     5.29385 | 5.439859  5.671142  4.381556  3.319697  2.576452  1.608178  0.719872     0.192674 | -200.0 | 385.365238 |     5.29385 |
    | 2000-01-02 08:30:00 | 6.485866  6.639185  6.317454  6.007387  5.252259  4.034283  2.491152    1.022769 |   153.86844 |  356.750005    383.34718   329.297042     284.4619   210.858723   118.933701     43.886993 |    8.068444 | 5.074672  5.623267  4.478846    3.4004  2.708938  1.773643  0.854956     0.257148 | -200.0 |  353.86844 |    8.068444 |
    | 2000-01-02 09:00:00 | 6.376699   6.61867  6.341304   6.03118  5.388323   4.26206  2.768308    1.254977 |  123.717566 |  338.734467    379.70377   333.066536    287.56488   223.120826   133.779201     54.294728 |   11.741613 | 4.687642  5.549522  4.562793  3.482303  2.824937  1.934458  0.998028     0.333743 | -200.0 | 323.717566 |   11.741613 |
    | 2000-01-02 09:30:00 | 6.261281  6.591423  6.361057  6.054922  5.505828  4.472249  3.038805    1.501502 |   95.049792 |  320.581005   374.904573   336.218837   290.737112   234.048693   148.408918     65.690164 |   16.415135 | 4.281686   5.45174  4.632427   3.56417  2.926976   2.08861  1.146922     0.422439 | -200.0 | 295.049792 |   16.415135 |
    | 2000-01-02 10:00:00 | 6.140176  6.558015  6.376546  6.078263  5.608066   4.66491  3.300069    1.757983 |   67.963873 |  302.616622   369.082895   338.709669   293.918121   243.814747   162.621597     77.902601 |   22.158362 | 3.859311  5.332101  4.687099  3.644795  3.017163  2.234757  1.299416     0.522778 | -200.0 | 267.963873 |   22.158362 |
    | 2000-01-02 10:30:00 | 6.013765  6.519157  6.387696  6.100873  5.697621  4.840629  3.550132    2.020302 |   42.523746 |   285.28489   362.399516   340.513048   297.052763   252.568425   176.268026     90.741235 |   29.002584 | 3.422341  5.193294  4.726495  3.723024  3.097234  2.372098  1.453364     0.633908 | -200.0 | 242.523746 |   29.002584 |
    | 2000-01-02 11:00:00 | 5.608145  6.463928    6.3936  6.122373  5.776504  5.000296  3.787592    2.284722 |   18.762374 |  243.822392   353.066268   341.471301   300.078942   260.434278    189.24399    104.007934 |   36.938644 | 3.017233  4.996655  4.747366   3.79753  3.168595  2.500241  1.606789     0.754632 | -200.0 | 218.762374 |   36.938644 |
    | 2000-01-02 11:30:00 | 5.174817  6.394112  6.393639  6.142365  5.846262  5.144962  4.011541    2.547952 |    -3.31432 |  204.065396   341.554461   341.477655   302.930656   267.512251   201.482126    117.508029 |   45.917849 | 2.643949  4.749175  4.747504  3.866914  3.232348  2.619095  1.757943     0.883495 | -200.0 |  196.68568 |   45.917849 |
    | 2000-01-02 12:00:00 | 4.745655  6.312572   6.38755  6.160465  5.908067  5.275727  4.221484    2.807175 |  -23.723558 |  168.810531   328.530406   340.489341    305.54266   273.879667   212.943803    131.058585 |   55.855757 | 2.297388  4.461679  4.725978  3.929818  3.289341  2.728779  1.905336      1.01886 | -200.0 | 176.276442 |   55.855757 |
    | 2000-01-02 12:30:00 | 4.315986  6.221593  6.375347  6.176327  5.962798  5.393676   4.41726    3.060045 |  -42.501918 |  137.446536    314.56522   338.516312   307.854477   279.594241   223.611768    144.494058 |   66.638142 | 1.973481  4.142866  4.682866   3.98501   3.34021  2.829548  2.047748     1.159001 | -200.0 | 157.498082 |   66.638142 |
    | 2000-01-02 13:00:00 | 3.880754  6.122992  6.357268  6.189658   6.00342  5.498556  4.598726    3.304617 |  -59.701769 |  109.505838   300.166679   335.612125     309.8137   283.954004   233.363212    157.651536 |   78.125969 | 1.668907  3.799676  4.619064  4.031447  3.386757  2.920611  2.184029     1.302147 | -200.0 | 140.298231 |   78.125969 |
    | 2000-01-02 13:30:00 | 3.433952  6.018163  6.333764  6.200239  6.016863  5.588101  4.765378    3.539218 |  -75.387943 |   84.634071   285.856413   331.870496   311.378866   285.687178   241.888684    170.343372 |   90.156911 | 1.380867  3.437476  4.536239  4.068332  3.433002  2.999449   2.31281     1.446482 | -200.0 | 124.612057 |   90.156911 |
    | 2000-01-02 14:00:00 | 2.967694  5.695997  6.296113  6.207226   6.02994  5.664976  4.917346    3.762524 |  -89.634712 |   62.573227   252.408009   325.958476   312.417419   287.401136   249.355855    182.433677 |  102.557026 | 1.106893  3.095773  4.403848  4.092706  3.478031   3.06793   2.43327      1.59026 | -200.0 | 110.365288 |  102.557026 |
    | 2000-01-02 14:30:00 | 2.470467  5.336359  6.244915  6.209941  6.042356  5.731279  5.055159    3.973585 |  -102.52308 |   43.160273   218.388423   318.085689   312.822032   289.049621   255.906767    193.830918 |  115.151317 | 0.844663  2.780363  4.224393   4.10218  3.520822  3.127587  2.545007     1.731883 | -200.0 |   97.47692 |  115.151317 |
    | 2000-01-02 15:00:00 | 1.923225   4.98668  6.182694  6.208009  6.053808  5.788657  5.179586    4.171799 | -114.138393 |   26.352646   188.115825   308.788373   312.534053    290.58682   261.659048    204.479847 |  127.772386 | 0.591779  2.489189  4.007182  4.095438  3.560327  3.179657  2.647929     1.869957 | -200.0 |  85.861607 |  127.772386 |
    | 2000-01-02 15:30:00 | 1.288512  4.647139  6.111573  6.201309  6.064011  5.838394  5.291515    4.356877 | -124.568281 |    12.32986   161.277992   298.553343   311.537762   291.968905   266.708155    214.353625 |  140.266841 | 0.345363  2.221082  3.760087  4.072066  3.595551  3.225127  2.742168     2.003313 | -200.0 |  75.431719 |  140.266841 |
    | 2000-01-02 16:00:00 | 0.465136  4.320457  6.033385  6.189951  6.072713  5.881489  5.391877    4.528788 | -133.900906 |    2.003258   137.753239   287.856575   309.856825   293.156492   271.130337    223.446719 |  152.499468 | 0.100735  1.976732  3.489901  4.032466  3.625611  3.264774  2.827998     2.131018 | -200.0 |  66.099094 |  152.499468 |
    | 2000-01-02 16:30:00 |      0.0   3.82333  5.816264  6.168135  6.079275   5.91862  5.481574     4.68771 | -142.223505 |  -86.259607   106.097243   264.454438   306.657951   294.057288   274.975934    231.767264 |  164.355162 |      0.0   1.63049  3.204858  3.956499  3.648293   3.29912  2.905774      2.25236 | -200.0 |  57.776495 |  164.355162 |
    | 2000-01-02 17:00:00 |      0.0  3.184545  5.493219  6.132567  6.082904  5.950169  5.561432    4.833971 | -149.621214 | -149.621214    72.353653   232.860958   301.528625   294.557357   278.269263    239.330234 |   175.73895 |      0.0  1.230935  2.915945  3.832898  3.660841  3.328439  2.975864     2.366824 | -200.0 |  50.378786 |   175.73895 |
    | 2000-01-02 17:30:00 |      0.0  2.519253  5.160134   6.08476  6.083028  5.976326  5.632176    4.968001 | -156.176132 | -156.176132    44.885229   202.792433   294.813611   294.574528   281.017787    246.153021 |  186.574731 |      0.0  0.869025  2.631712  3.667259  3.661271  3.352841  3.038621     2.474065 | -200.0 |  43.823868 |  186.574731 |
    | 2000-01-02 18:00:00 |      0.0  1.790683  4.823133  6.026325  6.079313  5.997179  5.694428    5.090296 | -161.966615 | -161.966615    22.957142   174.879738   286.924867   294.062498   283.220723    252.253113 |  196.803337 |      0.0  0.536162  2.358251  3.465578  3.648423  3.372356  3.094362     2.573875 | -200.0 |  38.033385 |  196.803337 |
    | 2000-01-02 18:30:00 |      0.0  0.914011  4.488618  5.865251  6.067351   6.00365  5.747243    5.201112 | -167.066765 | -167.066765     6.583938   149.586374   269.458911   292.423786   283.983379    257.499451 |  206.356956 |      0.0  0.223591  2.100847  3.249808  3.607087  3.387549  3.142033     2.665931 | -200.0 |  32.933235 |  206.356956 |
    | 2000-01-02 19:00:00 |      0.0       0.0  4.068859  5.619157  6.044192  6.006256  5.791548    5.300854 | -171.546089 | -171.546089   -47.329032   121.120714   244.888777   289.295112   284.316737     261.95094 |  215.190103 |      0.0       0.0  1.797637  3.027025  3.527155   3.39651  3.182292     2.750101 | -200.0 |  28.453911 |  215.190103 |
    | 2000-01-02 19:30:00 |      0.0       0.0  3.360883  5.335591  6.009484  6.006459  5.828312    5.390048 | -175.469299 | -175.469299  -175.469299    80.919949   218.318992   284.731656   284.342834    265.679972 |  223.278951 |      0.0       0.0  1.336136  2.779707  3.407612   3.39721  3.215885     2.826423 | -200.0 |  24.530701 |  223.278951 |
    | 2000-01-02 20:00:00 |      0.0       0.0  2.631016  5.028291  5.881166  5.999866  5.857432    5.469127 | -178.896239 | -178.896239  -178.896239    48.981878    191.57634   271.096992   283.505348    268.656342 |  230.601664 |      0.0       0.0  0.925956  2.523037  3.264475  3.374875  3.242613     2.894921 | -200.0 |  21.103761 |  230.601664 |
    | 2000-01-02 20:30:00 |      0.0       0.0  1.836597  4.707686  5.708014  5.951977  5.873391    5.537492 | -181.881912 | -181.881912  -181.881912    24.104609   165.883063    253.59638     278.4587    270.296014 |  237.047549 |      0.0       0.0   0.55518  2.267836  3.106591  3.330122  3.257306     2.954768 | -200.0 |  18.118088 |  237.047549 |
    | 2000-01-02 21:00:00 |      0.0       0.0  0.875785  4.383667  5.519363  5.882305  5.874884    5.594451 | -184.476573 | -184.476573  -184.476573     6.097283   142.133093   235.327757    271.21447    270.449753 |  242.500284 |      0.0       0.0  0.212147  2.022971  2.938841  3.265526  3.258682     3.005077 | -200.0 |  15.523427 |  242.500284 |
    | 2000-01-02 21:30:00 |      0.0       0.0       0.0  3.922849  5.303897  5.791142  5.860986    5.639315 | -186.725906 | -186.725906  -186.725906   -68.866417   112.045032   215.463125   261.909972     269.02099 |  246.848016 |      0.0       0.0       0.0  1.697331  2.752688  3.181922  3.245883     3.044989 | -200.0 |  13.274094 |  246.848016 |
    | 2000-01-02 22:00:00 |      0.0       0.0       0.0  3.179395  5.043393  5.676408  5.830708    5.671392 | -188.671223 | -188.671223  -188.671223  -188.671223    72.111807   192.841643   250.477994    265.924063 |  249.985232 |      0.0       0.0       0.0  1.227921  2.535375  3.078177   3.21808     3.073679 | -200.0 |  11.328777 |  249.985232 |
    | 2000-01-02 22:30:00 |      0.0       0.0       0.0  2.403585  4.753985  5.539152  5.783535    5.690079 | -190.349715 | -190.349715  -190.349715  -190.349715    40.857282   169.456884   237.205458    261.142485 |  251.823961 |      0.0       0.0       0.0  0.811748  2.303895  2.956229  3.174993     3.090452 | -200.0 |   9.650285 |  251.823961 |
    | 2000-01-02 23:00:00 |      0.0       0.0       0.0  1.536527  4.448228  5.382198  5.719619    5.694966 | -191.794713 | -191.794713  -191.794713  -191.794713    17.145608   146.691104   222.559812    254.747224 |  252.306183 |      0.0       0.0       0.0  0.435656  2.070713  2.819666  3.117056     3.094846 | -200.0 |   8.205287 |  252.306183 |
    | 2000-01-02 23:30:00 |      0.0       0.0       0.0  0.401649  4.139741  5.209661  5.639812    5.685919 | -193.035964 | -193.035964  -193.035964  -193.035964      1.55519   125.678204   207.105319    246.896399 |  251.413936 |      0.0       0.0       0.0  0.085392  1.847292  2.673097  3.045432     3.086714 | -200.0 |   6.964036 |  251.413936 |
    | 2000-01-03 00:00:00 |      0.0       0.0       0.0       0.0  3.504017  4.991072  5.540575    5.662324 | -194.099916 | -194.099916  -194.099916  -194.099916  -146.660021    88.287354   188.479285    237.340789 |   249.09592 |      0.0       0.0       0.0       0.0  1.424387  2.492751  2.957481     3.065555 | -200.0 |   5.900084 |   249.09592 |
    | 2000-01-03 00:30:00 |      0.0       0.0       0.0       0.0  2.731138  4.731064  5.420164    5.623498 |  -195.00999 |  -195.00999   -195.00999   -195.00999   -195.00999    52.825502   167.681887    226.050871 |  245.309847 |      0.0       0.0       0.0       0.0  0.978283   2.28601  2.852417     3.030889 | -200.0 |    4.99001 |  245.309847 |
    | 2000-01-03 01:00:00 |      0.0       0.0       0.0       0.0  1.895043  4.445186  5.279689    5.569168 | -195.786859 | -195.786859  -195.786859  -195.786859  -195.786859    25.609323   146.474437    213.297061 |  240.070693 |      0.0       0.0       0.0       0.0   0.57977  2.068453  2.732136     2.982696 | -200.0 |   4.213141 |  240.070693 |
    | 2000-01-03 01:30:00 |      0.0       0.0       0.0       0.0   0.88528  4.147192   5.12206    5.499663 | -196.448702 | -196.448702  -196.448702  -196.448702  -196.448702     6.216466    126.16312    199.513926 |  233.467483 |      0.0       0.0       0.0       0.0  0.214972  1.852549  2.600105      2.92158 | -200.0 |   3.551298 |  233.467483 |
    | 2000-01-03 02:00:00 |      0.0       0.0       0.0       0.0       0.0  3.680459  4.932781    5.413288 | -197.011459 | -197.011459  -197.011459  -197.011459  -197.011459   -77.582307    97.890756    183.689575 |  225.416175 |      0.0       0.0       0.0       0.0       0.0  1.536697  2.445667     2.846472 | -200.0 |   2.988541 |  225.416175 |
    | 2000-01-03 02:30:00 |      0.0       0.0       0.0       0.0       0.0  2.907034  4.693677    5.306475 |  -197.48906 |  -197.48906   -197.48906   -197.48906   -197.48906   -197.48906    59.983168    164.810669 |  215.694535 |      0.0       0.0       0.0       0.0       0.0  1.073247  2.256977     2.754881 | -200.0 |    2.51094 |  215.694535 |
    | 2000-01-03 03:00:00 |      0.0       0.0       0.0       0.0       0.0  2.082752  4.422503    5.179227 | -197.893642 | -197.893642  -197.893642  -197.893642  -197.893642  -197.893642    30.781337     144.86462 |  204.448632 |      0.0       0.0       0.0       0.0       0.0  0.661632  2.051627      2.64763 | -200.0 |   2.106358 |  204.448632 |
    | 2000-01-03 03:30:00 |      0.0       0.0       0.0       0.0       0.0   1.11794  4.133894    5.033765 | -198.235749 | -198.235749  -198.235749  -198.235749  -198.235749  -198.235749      9.49011    125.298485 |  192.034354 |      0.0       0.0       0.0       0.0       0.0  0.287726  1.843172     2.527505 | -200.0 |   1.764251 |  192.034354 |
    | 2000-01-03 04:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0  3.760805    4.864675 |  -198.52452 |  -198.52452   -198.52452   -198.52452   -198.52452   -198.52452   -38.677011    102.457959 |  178.186934 |      0.0       0.0       0.0       0.0       0.0       0.0  1.589129     2.391193 | -200.0 |    1.47548 |  178.186934 |
    | 2000-01-03 04:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0  2.986233    4.640377 | -198.767849 | -198.767849  -198.767849  -198.767849  -198.767849  -198.767849  -198.767849     63.377417 |  160.768492 |      0.0       0.0       0.0       0.0       0.0       0.0  1.117268     2.215889 | -200.0 |   1.232151 |  160.768492 |
    | 2000-01-03 05:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0  2.165866    4.380078 | -198.972542 | -198.972542  -198.972542  -198.972542  -198.972542  -198.972542  -198.972542     33.239728 |  141.882243 |      0.0       0.0       0.0       0.0       0.0       0.0  0.699286     2.020333 | -200.0 |   1.027458 |  141.882243 |
    | 2000-01-03 05:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0  1.217277    4.099184 | -199.144449 | -199.144449  -199.144449  -199.144449  -199.144449  -199.144449  -199.144449     11.097911 |   123.05831 |      0.0       0.0       0.0       0.0       0.0       0.0   0.32085     1.818804 | -200.0 |   0.855551 |   123.05831 |
    | 2000-01-03 06:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0     3.77049 | -199.288589 | -199.288589  -199.288589  -199.288589  -199.288589  -199.288589  -199.288589    -21.038811 |  103.016759 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0     1.595504 | -200.0 |   0.711411 |  103.016759 |
    | 2000-01-03 06:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0    2.994392 | -199.409255 | -199.409255  -199.409255  -199.409255  -199.409255  -199.409255  -199.409255   -199.409255 |   63.733208 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0     1.121847 | -200.0 |   0.590745 |   63.733208 |
    | 2000-01-03 07:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0    2.172905 | -199.510115 | -199.510115  -199.510115  -199.510115  -199.510115  -199.510115  -199.510115   -199.510115 |   33.452737 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0     0.702514 | -200.0 |   0.489885 |   33.452737 |
    | 2000-01-03 07:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0    1.223806 | -199.594292 | -199.594292  -199.594292  -199.594292  -199.594292  -199.594292  -199.594292   -199.594292 |   11.208068 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0      0.32307 | -200.0 |   0.405708 |   11.208068 |
    | 2000-01-03 08:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |  -199.66444 |  -199.66444   -199.66444   -199.66444   -199.66444   -199.66444   -199.66444    -199.66444 |  -20.181096 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |    0.33556 |  -20.181096 |
    | 2000-01-03 08:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.722814 | -199.722814  -199.722814  -199.722814  -199.722814  -199.722814  -199.722814   -199.722814 | -199.722814 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.277186 | -199.722814 |
    | 2000-01-03 09:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.771319 | -199.771319  -199.771319  -199.771319  -199.771319  -199.771319  -199.771319   -199.771319 | -199.771319 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.228681 | -199.771319 |
    | 2000-01-03 09:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.811569 | -199.811569  -199.811569  -199.811569  -199.811569  -199.811569  -199.811569   -199.811569 | -199.811569 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.188431 | -199.811569 |
    | 2000-01-03 10:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.844922 | -199.844922  -199.844922  -199.844922  -199.844922  -199.844922  -199.844922   -199.844922 | -199.844922 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.155078 | -199.844922 |
    | 2000-01-03 10:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.872523 | -199.872523  -199.872523  -199.872523  -199.872523  -199.872523  -199.872523   -199.872523 | -199.872523 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.127477 | -199.872523 |
    | 2000-01-03 11:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.895334 | -199.895334  -199.895334  -199.895334  -199.895334  -199.895334  -199.895334   -199.895334 | -199.895334 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.104666 | -199.895334 |
    | 2000-01-03 11:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.914162 | -199.914162  -199.914162  -199.914162  -199.914162  -199.914162  -199.914162   -199.914162 | -199.914162 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.085838 | -199.914162 |
    | 2000-01-03 12:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.929683 | -199.929683  -199.929683  -199.929683  -199.929683  -199.929683  -199.929683   -199.929683 | -199.929683 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.070317 | -199.929683 |
    | 2000-01-03 12:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.942461 | -199.942461  -199.942461  -199.942461  -199.942461  -199.942461  -199.942461   -199.942461 | -199.942461 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.057539 | -199.942461 |
    | 2000-01-03 13:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.952968 | -199.952968  -199.952968  -199.952968  -199.952968  -199.952968  -199.952968   -199.952968 | -199.952968 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.047032 | -199.952968 |
    | 2000-01-03 13:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.961598 | -199.961598  -199.961598  -199.961598  -199.961598  -199.961598  -199.961598   -199.961598 | -199.961598 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.038402 | -199.961598 |
    | 2000-01-03 14:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.968678 | -199.968678  -199.968678  -199.968678  -199.968678  -199.968678  -199.968678   -199.968678 | -199.968678 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.031322 | -199.968678 |
    | 2000-01-03 14:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.974479 | -199.974479  -199.974479  -199.974479  -199.974479  -199.974479  -199.974479   -199.974479 | -199.974479 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.025521 | -199.974479 |
    | 2000-01-03 15:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.979227 | -199.979227  -199.979227  -199.979227  -199.979227  -199.979227  -199.979227   -199.979227 | -199.979227 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.020773 | -199.979227 |
    | 2000-01-03 15:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.983108 | -199.983108  -199.983108  -199.983108  -199.983108  -199.983108  -199.983108   -199.983108 | -199.983108 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.016892 | -199.983108 |
    | 2000-01-03 16:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.986278 | -199.986278  -199.986278  -199.986278  -199.986278  -199.986278  -199.986278   -199.986278 | -199.986278 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.013722 | -199.986278 |
    | 2000-01-03 16:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.988864 | -199.988864  -199.988864  -199.988864  -199.988864  -199.988864  -199.988864   -199.988864 | -199.988864 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.011136 | -199.988864 |
    | 2000-01-03 17:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.990972 | -199.990972  -199.990972  -199.990972  -199.990972  -199.990972  -199.990972   -199.990972 | -199.990972 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.009028 | -199.990972 |
    | 2000-01-03 17:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.992687 | -199.992687  -199.992687  -199.992687  -199.992687  -199.992687  -199.992687   -199.992687 | -199.992687 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.007313 | -199.992687 |
    | 2000-01-03 18:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.994082 | -199.994082  -199.994082  -199.994082  -199.994082  -199.994082  -199.994082   -199.994082 | -199.994082 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.005918 | -199.994082 |
    | 2000-01-03 18:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.995215 | -199.995215  -199.995215  -199.995215  -199.995215  -199.995215  -199.995215   -199.995215 | -199.995215 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.004785 | -199.995215 |
    | 2000-01-03 19:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.996135 | -199.996135  -199.996135  -199.996135  -199.996135  -199.996135  -199.996135   -199.996135 | -199.996135 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.003865 | -199.996135 |
    | 2000-01-03 19:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.996881 | -199.996881  -199.996881  -199.996881  -199.996881  -199.996881  -199.996881   -199.996881 | -199.996881 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.003119 | -199.996881 |
    | 2000-01-03 20:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.997485 | -199.997485  -199.997485  -199.997485  -199.997485  -199.997485  -199.997485   -199.997485 | -199.997485 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.002515 | -199.997485 |
    | 2000-01-03 20:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.997974 | -199.997974  -199.997974  -199.997974  -199.997974  -199.997974  -199.997974   -199.997974 | -199.997974 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.002026 | -199.997974 |
    | 2000-01-03 21:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.998369 | -199.998369  -199.998369  -199.998369  -199.998369  -199.998369  -199.998369   -199.998369 | -199.998369 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.001631 | -199.998369 |
    | 2000-01-03 21:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.998688 | -199.998688  -199.998688  -199.998688  -199.998688  -199.998688  -199.998688   -199.998688 | -199.998688 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.001312 | -199.998688 |
    | 2000-01-03 22:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.998946 | -199.998946  -199.998946  -199.998946  -199.998946  -199.998946  -199.998946   -199.998946 | -199.998946 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.001054 | -199.998946 |
    | 2000-01-03 22:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999154 | -199.999154  -199.999154  -199.999154  -199.999154  -199.999154  -199.999154   -199.999154 | -199.999154 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000846 | -199.999154 |
    | 2000-01-03 23:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999321 | -199.999321  -199.999321  -199.999321  -199.999321  -199.999321  -199.999321   -199.999321 | -199.999321 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000679 | -199.999321 |
    | 2000-01-03 23:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999456 | -199.999456  -199.999456  -199.999456  -199.999456  -199.999456  -199.999456   -199.999456 | -199.999456 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000544 | -199.999456 |
    | 2000-01-04 00:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999564 | -199.999564  -199.999564  -199.999564  -199.999564  -199.999564  -199.999564   -199.999564 | -199.999564 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000436 | -199.999564 |
    | 2000-01-04 00:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999651 | -199.999651  -199.999651  -199.999651  -199.999651  -199.999651  -199.999651   -199.999651 | -199.999651 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000349 | -199.999651 |
    | 2000-01-04 01:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999721 | -199.999721  -199.999721  -199.999721  -199.999721  -199.999721  -199.999721   -199.999721 | -199.999721 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000279 | -199.999721 |
    | 2000-01-04 01:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999777 | -199.999777  -199.999777  -199.999777  -199.999777  -199.999777  -199.999777   -199.999777 | -199.999777 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000223 | -199.999777 |
    | 2000-01-04 02:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999822 | -199.999822  -199.999822  -199.999822  -199.999822  -199.999822  -199.999822   -199.999822 | -199.999822 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000178 | -199.999822 |
    | 2000-01-04 02:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999858 | -199.999858  -199.999858  -199.999858  -199.999858  -199.999858  -199.999858   -199.999858 | -199.999858 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000142 | -199.999858 |
    | 2000-01-04 03:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999887 | -199.999887  -199.999887  -199.999887  -199.999887  -199.999887  -199.999887   -199.999887 | -199.999887 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000113 | -199.999887 |
    | 2000-01-04 03:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |  -199.99991 |  -199.99991   -199.99991   -199.99991   -199.99991   -199.99991   -199.99991    -199.99991 |  -199.99991 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |    0.00009 |  -199.99991 |
    | 2000-01-04 04:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999928 | -199.999928  -199.999928  -199.999928  -199.999928  -199.999928  -199.999928   -199.999928 | -199.999928 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000072 | -199.999928 |
    | 2000-01-04 04:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999943 | -199.999943  -199.999943  -199.999943  -199.999943  -199.999943  -199.999943   -199.999943 | -199.999943 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000057 | -199.999943 |
    | 2000-01-04 05:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999954 | -199.999954  -199.999954  -199.999954  -199.999954  -199.999954  -199.999954   -199.999954 | -199.999954 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000046 | -199.999954 |
    | 2000-01-04 05:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999964 | -199.999964  -199.999964  -199.999964  -199.999964  -199.999964  -199.999964   -199.999964 | -199.999964 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000036 | -199.999964 |
    | 2000-01-04 06:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999971 | -199.999971  -199.999971  -199.999971  -199.999971  -199.999971  -199.999971   -199.999971 | -199.999971 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000029 | -199.999971 |
    | 2000-01-04 06:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999977 | -199.999977  -199.999977  -199.999977  -199.999977  -199.999977  -199.999977   -199.999977 | -199.999977 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000023 | -199.999977 |
    | 2000-01-04 07:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999982 | -199.999982  -199.999982  -199.999982  -199.999982  -199.999982  -199.999982   -199.999982 | -199.999982 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000018 | -199.999982 |
    | 2000-01-04 07:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999986 | -199.999986  -199.999986  -199.999986  -199.999986  -199.999986  -199.999986   -199.999986 | -199.999986 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000014 | -199.999986 |
    | 2000-01-04 08:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999989 | -199.999989  -199.999989  -199.999989  -199.999989  -199.999989  -199.999989   -199.999989 | -199.999989 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000011 | -199.999989 |
    | 2000-01-04 08:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999991 | -199.999991  -199.999991  -199.999991  -199.999991  -199.999991  -199.999991   -199.999991 | -199.999991 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000009 | -199.999991 |
    | 2000-01-04 09:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999993 | -199.999993  -199.999993  -199.999993  -199.999993  -199.999993  -199.999993   -199.999993 | -199.999993 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000007 | -199.999993 |
    | 2000-01-04 09:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999994 | -199.999994  -199.999994  -199.999994  -199.999994  -199.999994  -199.999994   -199.999994 | -199.999994 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000006 | -199.999994 |
    | 2000-01-04 10:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999996 | -199.999996  -199.999996  -199.999996  -199.999996  -199.999996  -199.999996   -199.999996 | -199.999996 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000004 | -199.999996 |
    | 2000-01-04 10:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999996 | -199.999996  -199.999996  -199.999996  -199.999996  -199.999996  -199.999996   -199.999996 | -199.999996 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000004 | -199.999996 |
    | 2000-01-04 11:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999997 | -199.999997  -199.999997  -199.999997  -199.999997  -199.999997  -199.999997   -199.999997 | -199.999997 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000003 | -199.999997 |
    | 2000-01-04 11:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999998 | -199.999998  -199.999998  -199.999998  -199.999998  -199.999998  -199.999998   -199.999998 | -199.999998 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000002 | -199.999998 |
    | 2000-01-04 12:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999998 | -199.999998  -199.999998  -199.999998  -199.999998  -199.999998  -199.999998   -199.999998 | -199.999998 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000002 | -199.999998 |
    | 2000-01-04 12:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999999 | -199.999999  -199.999999  -199.999999  -199.999999  -199.999999  -199.999999   -199.999999 | -199.999999 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000001 | -199.999999 |
    | 2000-01-04 13:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999999 | -199.999999  -199.999999  -199.999999  -199.999999  -199.999999  -199.999999   -199.999999 | -199.999999 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000001 | -199.999999 |
    | 2000-01-04 13:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999999 | -199.999999  -199.999999  -199.999999  -199.999999  -199.999999  -199.999999   -199.999999 | -199.999999 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000001 | -199.999999 |
    | 2000-01-04 14:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999999 | -199.999999  -199.999999  -199.999999  -199.999999  -199.999999  -199.999999   -199.999999 | -199.999999 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000001 | -199.999999 |
    | 2000-01-04 14:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 | -199.999999 | -199.999999  -199.999999  -199.999999  -199.999999  -199.999999  -199.999999   -199.999999 | -199.999999 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |   0.000001 | -199.999999 |
    | 2000-01-04 15:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 15:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 16:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 16:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 17:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 17:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 18:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 18:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 19:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 19:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 20:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 20:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 21:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 21:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 22:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 22:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 23:00:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |
    | 2000-01-04 23:30:00 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0         0.0 |      -200.0 |      -200.0       -200.0       -200.0       -200.0       -200.0       -200.0        -200.0 |      -200.0 |      0.0       0.0       0.0       0.0       0.0       0.0       0.0          0.0 | -200.0 |        0.0 |      -200.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from HydPy
from hydpy.core import importtools
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.exe.modelimports import *
from hydpy.interfaces import routinginterfaces

# ...from kinw
from hydpy.models.kinw import kinw_model
from hydpy.models.kinw import kinw_solver


class Model(modeltools.SegmentModel):
    """|kinw_impl_euler.DOCNAME.complete|.

    Please note that the following error report appears correct at first glance but is
    incorrect because the first time when InternalFlow| is required as the flow into a
    channel's second segment, it is already calculated as the flow out of its first
    segment:

    >>> from hydpy import prepare_model
    >>> from hydpy.core.testtools import check_methodorder
    >>> print(check_methodorder(prepare_model("kinw_impl_euler")))
    Method Update_WaterVolume_V1 requires the following sequences, which are not \
among the result sequences of any of its predecessors: InternalFlow
    """

    DOCNAME = modeltools.DocName(
        short="KinW-Implicit-Euler",
        description="A-stable routing based on the implicit Euler method",
    )
    __HYDPY_ROOTMODEL__ = True

    SOLVERPARAMETERS = (
        kinw_solver.NmbRuns,
        kinw_solver.WaterVolumeTolerance,
        kinw_solver.WaterDepthTolerance,
    )
    INLET_METHODS = (kinw_model.Pick_Inflow_V1,)
    OBSERVER_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        kinw_model.Update_WaterVolume_V1,
        kinw_model.Calc_WaterDepth_V1,
        kinw_model.Update_WaterVolume_V2,
        kinw_model.Calc_InternalFlow_Outflow_V1,
    )
    ADD_METHODS = (
        kinw_model.Return_InitialWaterVolume_V1,
        kinw_model.Return_VolumeError_V1,
    )
    OUTLET_METHODS = (kinw_model.Calc_Outflow_V1, kinw_model.Pass_Outflow_V1)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = ()
    SUBMODELS = (kinw_model.PegasusImplicitEuler,)

    wqmodel = modeltools.SubmodelProperty(routinginterfaces.CrossSectionModel_V1)
    wqmodel_is_mainmodel = modeltools.SubmodelIsMainmodelProperty()
    wqmodel_typeid = modeltools.SubmodelTypeIDProperty()

    @importtools.prepare_submodel(
        "wqmodel",
        routinginterfaces.CrossSectionModel_V1,
        routinginterfaces.CrossSectionModel_V1.prepare_bottomslope,
    )
    def add_wqmodel_v1(
        self,
        wqmodel: routinginterfaces.CrossSectionModel_V1,
        /,
        *,
        refresh: bool,  # pylint: disable=unused-argument
    ) -> None:
        """Initialise the given submodel that follows the |CrossSectionModel_V1|
        interface and is responsible for calculating discharge and related properties.

        >>> from hydpy.models.kinw_impl_euler import *
        >>> parameterstep()
        >>> with model.add_wqmodel_v1("wq_trapeze_strickler"):
        ...     nmbtrapezes(2)
        ...     bottomlevels(1.0, 3.0)
        ...     bottomslope(0.01)
        ...     sideslopes(2.0, 4.0)

        >>> model.wqmodel.parameters.control.nmbtrapezes
        nmbtrapezes(2)
        >>> model.wqmodel.parameters.control.bottomslope
        bottomslope(0.01)
        """

    def prepare_states(self, discharge: float) -> None:
        """Determine the initial water volume under the assumption of stationarity.

        Due to the assumptions of stationarity and a single profile geometry for the
        entire channel, the resulting initial volumes are identical for all segments:

        >>> from hydpy import pub, Nodes, Element
        >>> pub.timegrids = "2000-01-01", "2000-01-05", "30m"
        >>> from hydpy.models.kinw_impl_euler import *
        >>> parameterstep()
        >>> length(25.0)
        >>> nmbsegments(2)
        >>> with model.add_wqmodel_v1("wq_trapeze_strickler"):
        ...     nmbtrapezes(1)
        ...     bottomlevels(0.0)
        ...     bottomwidths(15.0)
        ...     bottomslope(0.00025)
        ...     sideslopes(5.0)
        ...     stricklercoefficients(1.0/0.035)
        >>> model.prepare_states(0.0)
        >>> states.watervolume
        watervolume(0.0, 0.0)
        >>> model.prepare_states(100.0)
        >>> states.watervolume
        watervolume(1.560986, 1.560986)
        >>> model.prepare_states(10000.0)
        >>> states.watervolume
        watervolume(48.454829, 48.454829)

        Calling method |Model.prepare_states| can overwrite previously calculated
        values of some sequences but at least resets the channel length (which it sets
        to zero for one calculation step) to its original value:

        >>> length
        length(25.0)
        """

        self.update_parameters()
        if self.parameters.control.nmbsegments > 0:
            self.idx_segment = 0
            length = self.parameters.control.length
            length_orig = length.value
            try:
                length(0.0)
                self.sequences.fluxes.inflow = discharge
                watervolume = self.sequences.states.watervolume
                watervolume(0.0)
                self.update_watervolume_v1()
                self.calc_waterdepth_v1()
                length(length_orig)
                self.update_watervolume_v2()
                watervolume(watervolume.values[0])
                self.new2old()
            finally:
                length(length_orig)

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation run in million
        m³.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            Error = \Sigma InOut - \Delta Vol
            \\ \\
            \Sigma InOut = s \cdot 10^{-6} \cdot \sum_{t=t0}^{t1} I_t - O_t
            \\
            \Delta Vol = \sum_{i=1}^{n} V_{t1}^i - V_{t0}^i
            \\ \\
            s = Seconds \\
            I = Inflow \\
            O = Outflow \\
            n = NmbSegments \\
            V = WaterVolume

        The returned error should always be in scale with numerical precision so that
        it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run via
        property |Sequences.conditions|.  See the integration tests of the application
        model |kinw_impl_euler| for some examples.
        """

        first = initial_conditions["model"]["states"]
        seconds = self.parameters.derived.seconds.value
        fluxes = self.sequences.fluxes
        dv = seconds / 1e6 * numpy.sum(fluxes.inflow.series - fluxes.outflow.series)
        if self.parameters.control.nmbsegments > 0:
            dv -= numpy.sum(
                self.sequences.states.watervolume.values - first["watervolume"]
            )
        return dv


tester = Tester()
cythonizer = Cythonizer()
