# pylint: disable=line-too-long, unused-wildcard-import
"""
|whmod_urban| is an extension of |whmod_rural| that considers water management measures
relevant to urban areas.  Its key additional feature is its ability to collect excess
water from selected sealed (surface runoff) and not sealed areas (percolation) in
cisterns and to use it for irrigation.   We designed |whmod_urban| originally to
develop and analyse Sponge City concepts for Munich, Germany.

Integration tests
=================

The following test settings are identical to those of |whmod_rural|:

.. how_to_understand_integration_tests::

>>> from hydpy import Element, pub, Timegrid
>>> pub.timegrids = "2017-02-10", "2017-04-10", "1d"

>>> element = Element("element")
>>> from hydpy.models.whmod_urban import *
>>> parameterstep("1d")
>>> element.model = model

>>> area(10.0)
>>> nmbzones(1)
>>> zonearea(10.0)
>>> landtype(GRASS)
>>> soiltype(SAND)
>>> interceptioncapacity.grass_feb = 0.4
>>> interceptioncapacity.grass_mar = 0.6
>>> interceptioncapacity.grass_apr = 0.8
>>> degreedayfactor(grass=4.5)
>>> availablefieldcapacity(sand=200.0)
>>> rootingdepth(0.5)
>>> groundwaterdepth(1.0)
>>> withcapillaryrise(True)
>>> capillarythreshold(0.8)
>>> capillarylimit(0.4)
>>> withexternalirrigation(False)
>>> irrigationtrigger(0.6)
>>> irrigationtarget(0.7)
>>> baseflowindex(0.8)
>>> rechargedelay(5.0)

>>> with model.add_aetmodel_v1("evap_aet_minhas") as aetmodel:
...     dissefactor(grass=4.0, deciduous=6.0, corn=3.0, conifer=6.0, springwheat=6.0,
...                 winterwheat=6.0)
...     with model.add_petmodel_v1("evap_pet_mlc") as petmodel:
...         landmonthfactor.grass = 1.0
...         dampingfactor(1.0)
...         with model.add_retmodel_v1("evap_ret_io") as retmodel:
...             evapotranspirationfactor(1.0)


>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = (
...     inputs.precipitation, fluxes.totalevapotranspiration
... )
>>> IntegrationTest.plotting_options.axis2 = (
...     fluxes.cisterninflow,
...     fluxes.cisternoverflow,
...     fluxes.cisterndemand,
...     fluxes.cisternextraction
... )
>>> test = IntegrationTest(element)
>>> test.dateformat = "%Y-%m-%d"

>>> test.inits = (
...     (states.interceptedwater, 0.0),
...     (states.snowpack, 0.0),
...     (states.soilmoisture, 0.0),
...     (states.deepwater, 0.0),
...     (petmodel.sequences.logs.loggedpotentialevapotranspiration, 0.0),
... )

>>> inputs.temperature.series = (
...     -2.8, -1.5, -0.9, -1.6, -1.3, 1.7, 4.4, 4.5, 3.4, 4.8, 6.7, 5.8, 6.5, 5.0, 3.0,
...     3.1, 7.1, 9.4, 4.6, 3.7, 4.7, 5.9, 7.7, 6.3, 3.7, 1.6, 4.0, 5.6, 5.8, 5.7, 4.6,
...     4.2, 7.4, 6.3, 8.7, 6.4, 5.2, 5.1, 8.7, 6.2, 5.9, 5.2, 5.2, 5.9, 6.7, 7.0, 8.3,
...     9.0, 12.4, 15.0, 11.8, 9.4, 8.1, 7.9, 7.5, 7.2, 8.1, 8.6, 10.5)
>>> inputs.precipitation.series = (
...     0.0, 0.4, 0.0, 0.0, 0.0, 0.0, 0.2, 4.5, 0.0, 3.2, 4.6, 2.3, 18.0, 19.2, 0.4,
...     8.3, 5.3, 0.7, 2.7, 1.6, 2.5, 0.6, 0.2, 1.7, 0.3, 0.0, 1.8, 8.9, 0.0, 0.0,
...     0.0, 0.9, 0.1, 0.0, 0.0, 3.9, 8.7, 26.4, 11.5, 0.9, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 1.5, 0.3, 0.2, 4.5, 0.0, 0.0, 0.0, 0.4, 0.0, 0.0, 0.0, 0.0)
>>> retmodel.sequences.inputs.referenceevapotranspiration.series = (
...     0.6, 0.8, 0.7, 0.4, 0.4, 0.4, 0.4, 0.3, 0.3, 0.4, 0.3, 0.6, 0.8, 0.5, 0.8,
...     0.5, 0.4, 1.3, 0.9, 0.7, 0.7, 1.1, 1.0, 0.8, 0.6, 0.7, 0.7, 0.5, 0.8, 1.0,
...     1.2, 0.9, 0.9, 1.2, 1.4, 1.1, 1.1, 0.5, 0.6, 1.5, 2.0, 1.6, 1.6, 1.2, 1.3,
...     1.6, 1.9, 0.8, 1.5, 2.7, 1.5, 1.6, 2.0, 2.1, 1.7, 1.7, 0.8, 1.3, 2.5)


.. _whmod_urban_grassland:

grassland
_________

    We first repeat the :ref:`whmod_rural_grassland` example of |whmod_rural|.

    The cistern has a capacity of 0.1 m³:

    >>> cisterncapacity(0.1)

    We set the boolean |CisternSource| parameter for the single considered zone to
    |False| so that the cistern cannot receive any inflow:

    >>> cisternsource(False)

    We also set the cistern's initial water content to zero:

    >>> test.inits.cisternwater = 0.0

    Due to the effectively "turned off" cistern, the following results are identical
    to the :ref:`whmod_rural_grassland` example of |whmod_rural|:

.. integration-test::

    >>> conditions = test("whmod_urban_grassland", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | cisterninflow | cisternoverflow | cisterndemand | cisternextraction | internalirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | cisternwater | deepwater |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |               0.0375 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |          3.75 |              66.25 |           0.0 |             0.0 |        0.6625 |               0.0 |                0.0 |                0.0 |             -3.75 |      0.0 |          -3.75 |        -0.67976 |              0.0 |      0.0 |         3.75 |          0.0 |  -3.07024 |
    | 2017-02-11 |           0.4 |        -1.5 |             0.070632 |                     0.4 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.030545 |             0.0 |                0.430545 |      3.343748 |          62.936798 |           0.0 |             0.0 |      0.629368 |               0.0 |                0.0 |                0.0 |         -3.343748 |      0.0 |      -3.343748 |       -1.162659 |              0.0 |      0.0 |     7.063202 |          0.0 | -5.251329 |
    | 2017-02-12 |           0.0 |        -0.9 |             0.099731 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.100328 |             0.0 |                0.100328 |      3.010193 |          60.026932 |           0.0 |             0.0 |      0.600269 |               0.0 |                0.0 |                0.0 |         -3.010193 |      0.0 |      -3.010193 |        -1.49756 |              0.0 |      0.0 |     9.973068 |          0.0 | -6.763962 |
    | 2017-02-13 |           0.0 |        -1.6 |             0.126288 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.080508 |             0.0 |                0.080508 |      2.736205 |          57.371235 |           0.0 |             0.0 |      0.573712 |               0.0 |                0.0 |                0.0 |         -2.736205 |      0.0 |      -2.736205 |       -1.722088 |              0.0 |      0.0 |    12.628765 |          0.0 | -7.778079 |
    | 2017-02-14 |           0.0 |        -1.3 |             0.150286 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.101248 |             0.0 |                0.101248 |      2.501132 |          54.971351 |           0.0 |             0.0 |      0.549714 |               0.0 |                0.0 |                0.0 |         -2.501132 |      0.0 |      -2.501132 |       -1.863305 |              0.0 |      0.0 |    15.028649 |          0.0 | -8.415907 |
    | 2017-02-15 |           0.0 |         1.7 |             0.172097 |                     0.0 |         0.0 |              7.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.119564 |             0.0 |                0.119564 |      2.300641 |          52.790274 |           0.0 |             0.0 |      0.527903 |               0.0 |                0.0 |                0.0 |         -2.300641 |      0.0 |      -2.300641 |       -1.942581 |              0.0 |      0.0 |    17.209726 |          0.0 | -8.773967 |
    | 2017-02-16 |           0.2 |         4.4 |             0.192698 |                     0.2 |         0.0 |              19.8 |      0.0 |     0.0 |           0.0 |         0.0 |               0.067899 |             0.0 |                0.267899 |      2.127988 |          50.730184 |           0.0 |             0.0 |      0.507302 |               0.0 |                0.0 |                0.0 |         -2.127988 |      0.0 |      -2.127988 |       -1.976189 |              0.0 |      0.0 |    19.269816 |          0.0 | -8.925766 |
    | 2017-02-17 |           4.5 |         4.5 |             0.252771 |                     0.3 |         4.1 |             20.25 |      0.0 |     4.1 |           0.0 |    0.065722 |                    0.0 |             0.0 |                     0.3 |      1.973055 |          44.722852 |           0.0 |             0.0 |      0.447229 |               0.0 |                0.0 |                0.0 |         -1.907333 |      0.0 |      -1.907333 |       -1.963708 |              0.1 |      0.0 |    25.277148 |          0.0 | -8.869391 |
    | 2017-02-18 |           0.0 |         3.4 |             0.267458 |                     0.1 |         0.0 |              15.3 |      0.0 |     0.0 |           0.0 |         0.0 |               0.095868 |             0.0 |                0.195868 |      1.564558 |          43.254162 |           0.0 |             0.0 |      0.432542 |               0.0 |                0.0 |                0.0 |         -1.564558 |      0.0 |      -1.564558 |       -1.891354 |              0.0 |      0.0 |    26.745838 |          0.0 | -8.542595 |
    | 2017-02-19 |           3.2 |         4.8 |             0.309177 |                     0.4 |         2.8 |              21.6 |      0.0 |     2.8 |           0.0 |    0.102207 |                    0.0 |             0.0 |                     0.4 |      1.474104 |          39.082265 |           0.0 |             0.0 |      0.390823 |               0.0 |                0.0 |                0.0 |         -1.371898 |      0.0 |      -1.371898 |       -1.797193 |              0.0 |      0.0 |    30.917735 |          0.0 |   -8.1173 |
    | 2017-02-20 |           4.6 |         6.7 |             0.361335 |                     0.3 |         4.2 |             30.15 |      0.0 |     4.2 |           0.0 |    0.220591 |                    0.0 |             0.0 |                     0.3 |       1.23632 |          33.866536 |           0.0 |             0.0 |      0.338665 |               0.0 |                0.0 |                0.0 |         -1.015729 |      0.0 |      -1.015729 |       -1.655537 |              0.1 |      0.0 |    36.133464 |          0.0 | -7.477491 |
    | 2017-02-21 |           2.3 |         5.8 |             0.388275 |                     0.4 |         2.0 |              26.1 |      0.0 |     2.0 |           0.0 |     0.15535 |               0.127491 |             0.0 |                0.527491 |      0.976903 |          31.172475 |           0.0 |             0.0 |      0.311725 |               0.0 |                0.0 |                0.0 |         -0.821553 |      0.0 |      -0.821553 |       -1.504361 |              0.0 |      0.0 |    38.827525 |          0.0 | -6.794683 |
    | 2017-02-22 |          18.0 |         6.5 |               0.5538 |                     0.4 |        17.6 |             29.25 |      0.0 |    17.6 |           0.0 |    1.637522 |               0.268404 |             0.0 |                0.668404 |      0.858419 |          14.619981 |           0.0 |             0.0 |        0.1462 |               0.0 |                0.0 |                0.0 |          0.779103 | 0.155821 |       0.623282 |       -1.118685 |              0.0 |      0.0 |    55.380019 |          0.0 | -5.052715 |
    | 2017-02-23 |          19.2 |         5.0 |              0.70165 |                     0.4 |        18.8 |              22.5 |      0.0 |    18.8 |           0.0 |    4.265135 |               0.083065 |             0.0 |                0.483065 |      0.333134 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          3.932001 |   0.7864 |         3.1456 |       -0.345701 |              0.0 |      0.0 |    70.164954 |          0.0 | -1.561413 |
    | 2017-02-24 |           0.4 |         3.0 |             0.698974 |                     0.4 |         0.0 |              13.5 |      0.0 |     0.0 |           0.0 |         0.0 |               0.367107 |             0.0 |                0.767107 |      0.099589 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.099589 |      0.0 |      -0.099589 |       -0.301089 |              0.0 |      0.0 |    69.897436 |          0.0 | -1.359914 |
    | 2017-02-25 |           8.3 |         3.1 |             0.745929 |                     0.4 |         7.9 |             13.95 |      0.0 |     7.9 |           0.0 |    3.215139 |               0.091655 |             0.0 |                0.491655 |      0.102292 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          3.112847 | 0.622569 |       2.490278 |          0.2049 |              0.0 |      0.0 |    74.592934 |          0.0 |  0.925464 |
    | 2017-02-26 |           5.3 |         7.1 |             0.772067 |                     0.4 |         4.9 |             31.95 |      0.0 |     4.9 |           0.0 |    2.347723 |                    0.0 |             0.0 |                     0.4 |      0.061503 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           2.28622 | 0.457244 |       1.828976 |        0.499295 |              0.0 |      0.0 |    77.206714 |          0.0 |  2.255144 |
    | 2017-02-27 |           0.7 |         9.4 |              0.76543 |                     0.4 |         0.3 |              42.3 |      0.0 |     0.3 |           0.0 |    0.156717 |               0.851358 |             0.0 |                1.251358 |      0.044407 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           0.11231 | 0.022462 |       0.089848 |        0.425075 |              0.0 |      0.0 |    76.543046 |          0.0 |  1.919918 |
    | 2017-02-28 |           2.7 |         4.6 |             0.772439 |                     0.4 |         2.3 |              20.7 |      0.0 |     2.3 |           0.0 |    1.175743 |               0.471796 |             0.0 |                0.871796 |        0.0484 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          1.127343 | 0.225469 |       0.901875 |        0.511504 |              0.0 |      0.0 |    77.243907 |          0.0 |  2.310288 |
    | 2017-03-01 |           1.6 |         3.7 |             0.776705 |                     0.6 |         1.0 |             16.65 |      0.0 |     1.0 |           0.0 |    0.523023 |               0.094608 |             0.0 |                0.694608 |       0.04419 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.478833 | 0.095767 |       0.383067 |        0.488222 |              0.0 |      0.0 |    77.670465 |          0.0 |  2.205132 |
    | 2017-03-02 |           2.5 |         4.7 |             0.785099 |                     0.6 |         1.9 |             21.15 |      0.0 |     1.9 |           0.0 |    1.007577 |               0.094757 |             0.0 |                0.694757 |      0.041751 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.965826 | 0.193165 |       0.772661 |        0.539782 |              0.0 |      0.0 |    78.509882 |          0.0 |  2.438011 |
    | 2017-03-03 |           0.6 |         5.9 |             0.780719 |                     0.6 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               0.475214 |             0.0 |                1.075214 |      0.037218 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.037218 |      0.0 |      -0.037218 |         0.43519 |              0.0 |      0.0 |    78.071886 |          0.0 |  1.965603 |
    | 2017-03-04 |           0.2 |         7.7 |             0.773523 |                     0.2 |         0.0 |             34.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.759159 |             0.0 |                0.959159 |       0.03954 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          -0.03954 |      0.0 |       -0.03954 |        0.349136 |              0.0 |      0.0 |    77.352267 |          0.0 |  1.576927 |
    | 2017-03-05 |           1.7 |         6.3 |             0.777292 |                     0.6 |         1.1 |             28.35 |      0.0 |     1.1 |           0.0 |    0.577354 |               0.189293 |             0.0 |                0.789293 |      0.043562 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.533792 | 0.106758 |       0.427034 |        0.363256 |              0.0 |      0.0 |    77.729182 |          0.0 |  1.640704 |
    | 2017-03-06 |           0.3 |         3.7 |             0.774863 |                     0.3 |         0.0 |             16.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.284332 |             0.0 |                0.584332 |      0.041423 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.041423 |      0.0 |      -0.041423 |        0.289901 |              0.0 |      0.0 |    77.486273 |          0.0 |  1.309381 |
    | 2017-03-07 |           0.0 |         1.6 |             0.768662 |                     0.0 |         0.0 |               7.2 |      0.0 |     0.0 |           0.0 |         0.0 |               0.662852 |             0.0 |                0.662852 |      0.042793 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.042793 |      0.0 |      -0.042793 |        0.229593 |              0.0 |      0.0 |    76.866214 |          0.0 |  1.036994 |
    | 2017-03-08 |           1.8 |         4.0 |             0.773982 |                     0.6 |         1.2 |              18.0 |      0.0 |     1.2 |           0.0 |    0.619953 |               0.094475 |             0.0 |                0.694475 |      0.046427 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.573526 | 0.114705 |       0.458821 |        0.271145 |              0.0 |      0.0 |    77.398213 |          0.0 |   1.22467 |
    | 2017-03-09 |           8.9 |         5.6 |             0.813786 |                     0.5 |         8.3 |              25.2 |      0.0 |     8.3 |           0.0 |    4.362895 |                    0.0 |             0.0 |                     0.5 |      0.043297 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          4.319598 |  0.86392 |       3.455679 |        0.848403 |              0.1 |      0.0 |    81.378615 |          0.0 |  3.831945 |
    | 2017-03-10 |           0.0 |         5.8 |             0.807311 |                     0.1 |         0.0 |              26.1 |      0.0 |     0.0 |           0.0 |         0.0 |               0.671692 |             0.0 |                0.771692 |      0.024214 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.024214 |      0.0 |      -0.024214 |        0.690225 |              0.0 |      0.0 |    80.731137 |          0.0 |  3.117507 |
    | 2017-03-11 |           0.0 |         5.7 |             0.798004 |                     0.0 |         0.0 |             25.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.957582 |             0.0 |                0.957582 |      0.026829 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.026829 |      0.0 |      -0.026829 |        0.560245 |              0.0 |      0.0 |    79.800384 |          0.0 |  2.530433 |
    | 2017-03-12 |           0.0 |         4.6 |             0.786857 |                     0.0 |         0.0 |              20.7 |      0.0 |     0.0 |           0.0 |         0.0 |               1.145587 |             0.0 |                1.145587 |      0.030907 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.030907 |      0.0 |      -0.030907 |        0.453087 |              0.0 |      0.0 |    78.685705 |          0.0 |  2.046439 |
    | 2017-03-13 |           0.9 |         4.2 |             0.785723 |                     0.6 |         0.3 |              18.9 |      0.0 |     0.3 |           0.0 |    0.164363 |               0.285305 |             0.0 |                0.885305 |      0.036312 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.128051 |  0.02561 |       0.102441 |        0.389526 |              0.0 |      0.0 |    78.572349 |          0.0 |  1.759354 |
    | 2017-03-14 |           0.1 |         7.4 |             0.778487 |                     0.1 |         0.0 |              33.3 |      0.0 |     0.0 |           0.0 |         0.0 |                0.76051 |             0.0 |                 0.86051 |      0.036894 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.036894 |      0.0 |      -0.036894 |        0.312229 |              0.0 |      0.0 |    77.848732 |          0.0 |  1.410231 |
    | 2017-03-15 |           0.0 |         6.3 |             0.767517 |                     0.0 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |               1.137822 |             0.0 |                1.137822 |      0.040759 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.040759 |      0.0 |      -0.040759 |        0.248243 |              0.0 |      0.0 |     76.75167 |          0.0 |  1.121228 |
    | 2017-03-16 |           0.0 |         8.7 |             0.754767 |                     0.0 |         0.0 |             39.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.322076 |             0.0 |                1.322076 |       0.04712 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          -0.04712 |      0.0 |       -0.04712 |        0.194703 |              0.0 |      0.0 |    75.476714 |          0.0 |  0.879406 |
    | 2017-03-17 |           3.9 |         6.4 |             0.767336 |                     0.6 |         3.3 |              28.8 |      0.0 |     3.3 |           0.0 |    1.628564 |               0.469837 |             0.0 |                1.069837 |      0.055305 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          1.573259 | 0.314652 |       1.258607 |        0.387556 |              0.0 |      0.0 |    76.733618 |          0.0 |  1.750457 |
    | 2017-03-18 |           8.7 |         5.2 |             0.802421 |                     0.6 |         8.1 |              23.4 |      0.0 |     8.1 |           0.0 |    4.166588 |               0.472138 |             0.0 |                1.072138 |       0.04723 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          4.119358 | 0.823872 |       3.295486 |        0.914674 |              0.0 |      0.0 |    80.242122 |          0.0 |  4.131269 |
    | 2017-03-19 |          26.4 |         5.1 |             0.912235 |                     0.5 |        25.8 |             22.95 |      0.0 |    25.8 |           0.0 |   14.847535 |                    0.0 |             0.0 |                     0.5 |      0.028924 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         14.818611 | 2.963722 |      11.854889 |        2.897799 |              0.1 |      0.0 |    91.223511 |          0.0 | 13.088359 |
    | 2017-03-20 |          11.5 |         8.7 |             0.934912 |                     0.6 |        11.0 |             39.15 |      0.0 |    11.0 |           0.0 |    8.734827 |                    0.0 |             0.0 |                     0.6 |      0.002535 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          8.732292 | 1.746458 |       6.985833 |        3.638834 |              0.0 |      0.0 |     93.49122 |          0.0 | 16.435358 |
    | 2017-03-21 |           0.9 |         6.2 |             0.926488 |                     0.6 |         0.3 |              27.9 |      0.0 |     0.3 |           0.0 |    0.253368 |               0.890068 |             0.0 |                1.490068 |      0.001034 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.252334 | 0.050467 |       0.201867 |        3.015817 |              0.0 |      0.0 |    92.648818 |          0.0 | 13.621408 |
    | 2017-03-22 |           0.0 |         5.9 |             0.906757 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                1.97465 |             0.0 |                 1.97465 |       0.00149 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          -0.00149 |      0.0 |       -0.00149 |        2.468872 |              0.0 |      0.0 |    90.675658 |          0.0 | 11.151046 |
    | 2017-03-23 |           0.0 |         5.2 |             0.891055 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.573237 |             0.0 |                1.573237 |       0.00304 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          -0.00304 |      0.0 |       -0.00304 |        2.020791 |              0.0 |      0.0 |    89.105461 |          0.0 |  9.127215 |
    | 2017-03-24 |           0.0 |         5.2 |             0.875426 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.567721 |             0.0 |                1.567721 |      0.004849 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.004849 |      0.0 |      -0.004849 |        1.653604 |              0.0 |      0.0 |     87.54259 |          0.0 |  7.468762 |
    | 2017-03-25 |           0.0 |         5.9 |             0.863784 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.171422 |             0.0 |                1.171422 |       0.00725 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          -0.00725 |      0.0 |       -0.00725 |        1.352543 |              0.0 |      0.0 |    86.378418 |          0.0 |  6.108969 |
    | 2017-03-26 |           0.0 |         6.7 |             0.851226 |                     0.0 |         0.0 |             30.15 |      0.0 |     0.0 |           0.0 |         0.0 |                1.26533 |             0.0 |                 1.26533 |      0.009478 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.009478 |      0.0 |      -0.009478 |         1.10565 |              0.0 |      0.0 |    85.122566 |          0.0 |  4.993841 |
    | 2017-03-27 |           0.0 |         7.0 |             0.835827 |                     0.0 |         0.0 |              31.5 |      0.0 |     0.0 |           0.0 |         0.0 |               1.552174 |             0.0 |                1.552174 |      0.012349 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.012349 |      0.0 |      -0.012349 |        0.902991 |              0.0 |      0.0 |     83.58274 |          0.0 |  4.078501 |
    | 2017-03-28 |           0.0 |         8.3 |              0.81764 |                     0.0 |         0.0 |             37.35 |      0.0 |     0.0 |           0.0 |         0.0 |               1.835299 |             0.0 |                1.835299 |      0.016593 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.016593 |      0.0 |      -0.016593 |        0.736299 |              0.0 |      0.0 |    81.764034 |          0.0 |  3.325609 |
    | 2017-03-29 |           1.5 |         9.0 |             0.819517 |                     0.6 |         0.9 |              40.5 |      0.0 |     0.9 |           0.0 |     0.54295 |               0.192143 |             0.0 |                0.792143 |      0.022741 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.520209 | 0.104042 |       0.416167 |        0.678269 |              0.0 |      0.0 |    81.951683 |          0.0 |  3.063507 |
    | 2017-03-30 |           0.3 |        12.4 |             0.808202 |                     0.3 |         0.0 |              55.8 |      0.0 |     0.0 |           0.0 |         0.0 |               1.153525 |             0.0 |                1.453525 |      0.022047 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.022047 |      0.0 |      -0.022047 |        0.551323 |              0.0 |      0.0 |    80.820205 |          0.0 |  2.490137 |
    | 2017-03-31 |           0.2 |        15.0 |              0.78452 |                     0.2 |         0.0 |              67.5 |      0.0 |     0.0 |           0.0 |         0.0 |               2.394642 |             0.0 |                2.594642 |      0.026458 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.026458 |      0.0 |      -0.026458 |        0.446589 |              0.0 |      0.0 |    78.452021 |          0.0 |   2.01709 |
    | 2017-04-01 |           4.5 |        11.8 |             0.795123 |                     0.8 |         3.7 |              53.1 |      0.0 |     3.7 |           0.0 |    2.012061 |               0.665164 |             0.0 |                1.465164 |      0.037519 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          1.974542 | 0.394908 |       1.579634 |        0.651975 |              0.0 |      0.0 |    79.512315 |          0.0 |  2.944748 |
    | 2017-04-02 |           0.0 |         9.4 |             0.780186 |                     0.0 |         0.0 |              42.3 |      0.0 |     0.0 |           0.0 |         0.0 |               1.525967 |             0.0 |                1.525967 |      0.032249 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.032249 |      0.0 |      -0.032249 |        0.527947 |              0.0 |      0.0 |    78.018597 |          0.0 |  2.384553 |
    | 2017-04-03 |           0.0 |         8.1 |             0.761609 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               1.897533 |             0.0 |                1.897533 |      0.039829 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.039829 |      0.0 |      -0.039829 |        0.425026 |              0.0 |      0.0 |    76.160892 |          0.0 |  1.919698 |
    | 2017-04-04 |           0.0 |         7.9 |             0.742331 |                     0.0 |         0.0 |             35.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.978631 |             0.0 |                1.978631 |      0.050804 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.050804 |      0.0 |      -0.050804 |        0.338773 |              0.0 |      0.0 |    74.233066 |          0.0 |   1.53012 |
    | 2017-04-05 |           0.4 |         7.5 |             0.730818 |                     0.4 |         0.0 |             33.75 |      0.0 |     0.0 |           0.0 |         0.0 |               1.215386 |             0.0 |                1.615386 |      0.064153 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.064153 |      0.0 |      -0.064153 |        0.265735 |              0.0 |      0.0 |    73.081834 |          0.0 |  1.200232 |
    | 2017-04-06 |           0.0 |         7.2 |             0.715734 |                     0.0 |         0.0 |              32.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.581527 |             0.0 |                1.581527 |      0.073142 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.073142 |      0.0 |      -0.073142 |        0.204307 |              0.0 |      0.0 |    71.573449 |          0.0 |  0.922783 |
    | 2017-04-07 |           0.0 |         8.1 |             0.709204 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               0.739189 |             0.0 |                0.739189 |       0.08614 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          -0.08614 |      0.0 |       -0.08614 |        0.151658 |              0.0 |      0.0 |      70.9204 |          0.0 |  0.684986 |
    | 2017-04-08 |           0.0 |         8.6 |             0.698151 |                     0.0 |         0.0 |              38.7 |      0.0 |     0.0 |           0.0 |         0.0 |               1.197483 |             0.0 |                1.197483 |      0.092214 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.092214 |      0.0 |      -0.092214 |        0.107451 |              0.0 |      0.0 |    69.815131 |          0.0 |  0.485321 |
    | 2017-04-09 |           0.0 |        10.5 |             0.676278 |                     0.0 |         0.0 |             47.25 |      0.0 |     0.0 |           0.0 |         0.0 |               2.290433 |             0.0 |                2.290433 |      0.103133 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.103133 |      0.0 |      -0.103133 |        0.069279 |              0.0 |      0.0 |    67.627832 |          0.0 |  0.312908 |

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_urban_external_irrigation:

external irrigation
___________________

External irrigation works as discussed in the :ref:`whmod_rural_irrigation` example of
|whmod_rural|:

.. integration-test::

    >>> withexternalirrigation(True)
    >>> conditions = test("whmod_urban_external_irrigation", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | cisterninflow | cisternoverflow | cisterndemand | cisternextraction | internalirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | cisternwater | deepwater |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |                  0.7 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |          3.75 |              66.25 |           0.0 |             0.0 |        0.6625 |               0.0 |                0.0 |              66.25 |             -3.75 |      0.0 |          -3.75 |        -0.67976 |              0.0 |      0.0 |         70.0 |          0.0 |  -3.07024 |
    | 2017-02-11 |           0.4 |        -1.5 |             0.697344 |                     0.4 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.366807 |             0.0 |                0.766807 |       0.10125 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          -0.10125 |      0.0 |       -0.10125 |       -0.574894 |              0.0 |      0.0 |    69.734443 |          0.0 | -2.596597 |
    | 2017-02-12 |           0.0 |        -0.9 |             0.691973 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.641062 |             0.0 |                0.641062 |      0.103963 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.103963 |      0.0 |      -0.103963 |       -0.489528 |              0.0 |      0.0 |    69.197344 |          0.0 | -2.211031 |
    | 2017-02-13 |           0.0 |        -1.6 |             0.689416 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.365324 |             0.0 |                0.365324 |      0.109596 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.109596 |      0.0 |      -0.109596 |       -0.420658 |              0.0 |      0.0 |    68.941615 |          0.0 | -1.899969 |
    | 2017-02-14 |           0.0 |        -1.3 |             0.686891 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.364843 |             0.0 |                0.364843 |      0.112349 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.112349 |      0.0 |      -0.112349 |       -0.364771 |              0.0 |      0.0 |    68.689121 |          0.0 | -1.647546 |
    | 2017-02-15 |           0.0 |         1.7 |             0.684399 |                     0.0 |         0.0 |              7.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.364364 |             0.0 |                0.364364 |      0.115111 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.115111 |      0.0 |      -0.115111 |       -0.319516 |              0.0 |      0.0 |    68.439868 |          0.0 | -1.443142 |
    | 2017-02-16 |           0.2 |         4.4 |             0.683758 |                     0.2 |         0.0 |              19.8 |      0.0 |     0.0 |           0.0 |         0.0 |               0.181943 |             0.0 |                0.381943 |      0.117882 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.117882 |      0.0 |      -0.117882 |       -0.282966 |              0.0 |      0.0 |    68.375807 |          0.0 | -1.278058 |
    | 2017-02-17 |           4.5 |         4.5 |             0.710155 |                     0.3 |         4.1 |             20.25 |      0.0 |     4.1 |           0.0 |    1.578928 |                    0.0 |             0.0 |                     0.3 |      0.118601 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          1.460327 | 0.292065 |       1.168262 |       -0.019903 |              0.1 |      0.0 |     71.01548 |          0.0 | -0.089894 |
    | 2017-02-18 |           0.0 |         3.4 |             0.709225 |                     0.1 |         0.0 |              15.3 |      0.0 |     0.0 |           0.0 |         0.0 |               0.184312 |             0.0 |                0.284312 |      0.091312 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.091312 |      0.0 |      -0.091312 |       -0.032847 |              0.0 |      0.0 |     70.92248 |          0.0 | -0.148359 |
    | 2017-02-19 |           3.2 |         4.8 |             0.726327 |                     0.4 |         2.8 |              21.6 |      0.0 |     2.8 |           0.0 |    1.181957 |                    0.0 |             0.0 |                     0.4 |      0.092194 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          1.089763 | 0.217953 |        0.87181 |        0.131139 |              0.0 |      0.0 |    72.632718 |          0.0 |  0.592312 |
    | 2017-02-20 |           4.6 |         6.7 |             0.750274 |                     0.3 |         4.2 |             30.15 |      0.0 |     4.2 |           0.0 |    1.882214 |                    0.0 |             0.0 |                     0.3 |      0.076865 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          1.805349 |  0.36107 |       1.444279 |        0.369171 |              0.1 |      0.0 |    75.027368 |          0.0 |   1.66742 |
    | 2017-02-21 |           2.3 |         5.8 |             0.759258 |                     0.4 |         2.0 |              26.1 |      0.0 |     2.0 |           0.0 |    0.972325 |               0.187596 |             0.0 |                0.587596 |      0.058402 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.913923 | 0.182785 |       0.731139 |        0.434785 |              0.0 |      0.0 |    75.925849 |          0.0 |  1.963774 |
    | 2017-02-22 |          18.0 |         6.5 |             0.843856 |                     0.4 |        17.6 |             29.25 |      0.0 |    17.6 |           0.0 |    8.815997 |               0.376537 |             0.0 |                0.776537 |      0.052322 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          8.763675 | 1.752735 |        7.01094 |         1.62684 |              0.0 |      0.0 |    84.385637 |          0.0 |  7.347874 |
    | 2017-02-23 |          19.2 |         5.0 |             0.908264 |                     0.4 |        18.8 |              22.5 |      0.0 |    18.8 |           0.0 |   12.276653 |               0.096815 |             0.0 |                0.496815 |      0.014276 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         12.262377 | 2.452475 |       9.809902 |        3.110177 |              0.0 |      0.0 |    90.826445 |          0.0 | 14.047599 |
    | 2017-02-24 |           0.4 |         3.0 |             0.904359 |                     0.4 |         0.0 |              13.5 |      0.0 |     0.0 |           0.0 |         0.0 |               0.393437 |             0.0 |                0.793437 |      0.002895 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.002895 |      0.0 |      -0.002895 |        2.545873 |              0.0 |      0.0 |    90.435903 |          0.0 | 11.498831 |
    | 2017-02-25 |           8.3 |         3.1 |             0.921028 |                     0.4 |         7.9 |             13.95 |      0.0 |     7.9 |           0.0 |    6.138124 |               0.098276 |             0.0 |                0.498276 |      0.003281 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          6.134843 | 1.226969 |       4.907875 |        2.974031 |              0.0 |      0.0 |    92.102783 |          0.0 | 13.432674 |
    | 2017-02-26 |           5.3 |         7.1 |             0.930188 |                     0.4 |         4.9 |             31.95 |      0.0 |     4.9 |           0.0 |    3.985795 |                    0.0 |             0.0 |                     0.4 |      0.001847 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          3.983948 |  0.79679 |       3.187158 |        3.012665 |              0.0 |      0.0 |    93.018835 |          0.0 | 13.607168 |
    | 2017-02-27 |           0.7 |         9.4 |             0.921807 |                     0.4 |         0.3 |              42.3 |      0.0 |     0.3 |           0.0 |    0.250166 |               0.889246 |             0.0 |                1.289246 |      0.001276 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.248891 | 0.049778 |       0.199112 |        2.502654 |              0.0 |      0.0 |    92.180699 |          0.0 | 11.303627 |
    | 2017-02-28 |           2.7 |         4.6 |             0.921144 |                     0.4 |         2.3 |              20.7 |      0.0 |     2.3 |           0.0 |    1.874859 |               0.493195 |             0.0 |                0.893195 |      0.001793 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          1.873066 | 0.374613 |       1.498453 |        2.320623 |              0.0 |      0.0 |    92.114437 |          0.0 | 10.481456 |
    | 2017-03-01 |           1.6 |         3.7 |              0.92204 |                     0.6 |         1.0 |             16.65 |      0.0 |     1.0 |           0.0 |    0.813686 |               0.098626 |             0.0 |                0.698626 |      0.001839 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.811847 | 0.162369 |       0.649478 |        2.017696 |              0.0 |      0.0 |    92.203964 |          0.0 |  9.113238 |
    | 2017-03-02 |           2.5 |         4.7 |             0.924573 |                     0.6 |         1.9 |             21.15 |      0.0 |     1.9 |           0.0 |    1.549778 |               0.098644 |             0.0 |                0.698644 |      0.001777 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          1.548001 |   0.3096 |       1.238401 |        1.876434 |              0.0 |      0.0 |     92.45732 |          0.0 |  8.475205 |
    | 2017-03-03 |           0.6 |         5.9 |             0.919655 |                     0.6 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               0.493472 |             0.0 |                1.093472 |      0.001609 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.001609 |      0.0 |      -0.001609 |        1.536002 |              0.0 |      0.0 |    91.965457 |          0.0 |  6.937593 |
    | 2017-03-04 |           0.2 |         7.7 |             0.911786 |                     0.2 |         0.0 |             34.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.788765 |             0.0 |                0.988765 |      0.001945 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.001945 |      0.0 |      -0.001945 |         1.25722 |              0.0 |      0.0 |    91.178637 |          0.0 |  5.678428 |
    | 2017-03-05 |           1.7 |         6.3 |             0.912119 |                     0.6 |         1.1 |             28.35 |      0.0 |     1.1 |           0.0 |    0.872404 |               0.196867 |             0.0 |                0.796867 |      0.002574 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           0.86983 | 0.173966 |       0.695864 |        1.155463 |              0.0 |      0.0 |     91.21194 |          0.0 |   5.21883 |
    | 2017-03-06 |           0.3 |         3.7 |             0.909192 |                     0.3 |         0.0 |             16.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.295321 |             0.0 |                0.595321 |      0.002545 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.002545 |      0.0 |      -0.002545 |        0.945552 |              0.0 |      0.0 |    90.919163 |          0.0 |  4.270732 |
    | 2017-03-07 |           0.0 |         1.6 |             0.902333 |                     0.0 |         0.0 |               7.2 |      0.0 |     0.0 |           0.0 |         0.0 |               0.688653 |             0.0 |                0.688653 |      0.002808 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.002808 |      0.0 |      -0.002808 |        0.773643 |              0.0 |      0.0 |    90.233319 |          0.0 |  3.494281 |
    | 2017-03-08 |           1.8 |         4.0 |             0.904114 |                     0.6 |         1.2 |              18.0 |      0.0 |     1.2 |           0.0 |    0.927139 |               0.098232 |             0.0 |                0.698232 |      0.003494 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.923646 | 0.184729 |       0.738917 |        0.767349 |              0.0 |      0.0 |    90.411441 |          0.0 |  3.465849 |
    | 2017-03-09 |           8.9 |         5.6 |             0.922702 |                     0.5 |         8.3 |              25.2 |      0.0 |     8.3 |           0.0 |    6.444537 |                    0.0 |             0.0 |                     0.5 |      0.003306 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          6.441231 | 1.288246 |       5.152985 |         1.56233 |              0.1 |      0.0 |    92.270209 |          0.0 |  7.056505 |
    | 2017-03-10 |           0.0 |         5.8 |             0.915813 |                     0.1 |         0.0 |              26.1 |      0.0 |     0.0 |           0.0 |         0.0 |                 0.6906 |             0.0 |                  0.7906 |      0.001732 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.001732 |      0.0 |      -0.001732 |        1.278813 |              0.0 |      0.0 |    91.581342 |          0.0 |  5.775959 |
    | 2017-03-11 |           0.0 |         5.7 |             0.905984 |                     0.0 |         0.0 |             25.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.985171 |             0.0 |                0.985171 |      0.002237 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.002237 |      0.0 |      -0.002237 |        1.046598 |              0.0 |      0.0 |    90.598408 |          0.0 |  4.727124 |
    | 2017-03-12 |           0.0 |         4.6 |             0.894218 |                     0.0 |         0.0 |              20.7 |      0.0 |     0.0 |           0.0 |         0.0 |                1.17973 |             0.0 |                 1.17973 |      0.003116 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.003116 |      0.0 |      -0.003116 |        0.856317 |              0.0 |      0.0 |    89.421795 |          0.0 |   3.86769 |
    | 2017-03-13 |           0.9 |         4.2 |             0.892055 |                     0.6 |         0.3 |              18.9 |      0.0 |     0.3 |           0.0 |    0.226588 |               0.294161 |             0.0 |                0.894161 |      0.004439 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.222149 |  0.04443 |       0.177719 |        0.733308 |              0.0 |      0.0 |    89.205485 |          0.0 |  3.312101 |
    | 2017-03-14 |           0.1 |         7.4 |             0.884262 |                     0.1 |         0.0 |              33.3 |      0.0 |     0.0 |           0.0 |         0.0 |               0.784041 |             0.0 |                0.884041 |      0.004717 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.004717 |      0.0 |      -0.004717 |        0.599527 |              0.0 |      0.0 |    88.426161 |          0.0 |  2.707857 |
    | 2017-03-15 |           0.0 |         6.3 |             0.872581 |                     0.0 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |               1.173923 |             0.0 |                1.173923 |      0.005814 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.005814 |      0.0 |      -0.005814 |        0.489797 |              0.0 |      0.0 |    87.258052 |          0.0 |  2.212246 |
    | 2017-03-16 |           0.0 |         8.7 |             0.859001 |                     0.0 |         0.0 |             39.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.365698 |             0.0 |                1.365698 |      0.007758 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.007758 |      0.0 |      -0.007758 |        0.399606 |              0.0 |      0.0 |    85.900111 |          0.0 |  1.804882 |
    | 2017-03-17 |           3.9 |         6.4 |             0.864712 |                     0.6 |         3.3 |              28.8 |      0.0 |     3.3 |           0.0 |    2.253344 |               0.486061 |             0.0 |                1.086061 |      0.010512 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          2.242832 | 0.448566 |       1.794266 |        0.652415 |              0.0 |      0.0 |    86.471218 |          0.0 |  2.946733 |
    | 2017-03-18 |           8.7 |         5.2 |               0.8847 |                     0.6 |         8.1 |              23.4 |      0.0 |     8.1 |           0.0 |    5.623705 |               0.486781 |             0.0 |                1.086781 |      0.009286 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          5.614419 | 1.122884 |       4.491535 |        1.348329 |              0.0 |      0.0 |    88.470017 |          0.0 |  6.089939 |
    | 2017-03-19 |          26.4 |         5.1 |             0.953057 |                     0.5 |        25.8 |             22.95 |      0.0 |    25.8 |           0.0 |   18.970089 |                    0.0 |             0.0 |                     0.5 |      0.005748 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         18.964341 | 3.792868 |      15.171473 |         3.85404 |              0.1 |      0.0 |    95.305676 |          0.0 | 17.407372 |
    | 2017-03-20 |          11.5 |         8.7 |             0.965567 |                     0.6 |        11.0 |             39.15 |      0.0 |    11.0 |           0.0 |     9.74939 |                    0.0 |             0.0 |                     0.6 |      0.000388 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          9.749002 |   1.9498 |       7.799201 |        4.569177 |              0.0 |      0.0 |    96.556674 |          0.0 | 20.637397 |
    | 2017-03-21 |           0.9 |         6.2 |              0.95687 |                     0.6 |         0.3 |              27.9 |      0.0 |     0.3 |           0.0 |     0.27474 |               0.895054 |             0.0 |                1.495054 |      0.000153 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.274587 | 0.054917 |        0.21967 |        3.780745 |              0.0 |      0.0 |    95.687033 |          0.0 | 17.076322 |
    | 2017-03-22 |           0.0 |         5.9 |             0.937013 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.985997 |             0.0 |                1.985997 |      0.000301 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.000301 |      0.0 |      -0.000301 |        3.095357 |              0.0 |      0.0 |    93.701337 |          0.0 | 13.980664 |
    | 2017-03-23 |           0.0 |         5.2 |             0.921193 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.582984 |             0.0 |                1.582984 |      0.000937 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.000937 |      0.0 |      -0.000937 |        2.534094 |              0.0 |      0.0 |     92.11929 |          0.0 | 11.445632 |
    | 2017-03-24 |           0.0 |         5.2 |             0.905431 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.578027 |             0.0 |                1.578027 |      0.001835 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.001835 |      0.0 |      -0.001835 |        2.074408 |              0.0 |      0.0 |    90.543098 |          0.0 |  9.369388 |
    | 2017-03-25 |           0.0 |         5.9 |             0.893667 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.179588 |             0.0 |                1.179588 |      0.003172 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.003172 |      0.0 |      -0.003172 |        1.697807 |              0.0 |      0.0 |    89.366681 |          0.0 |   7.66841 |
    | 2017-03-26 |           0.0 |         6.7 |             0.880967 |                     0.0 |         0.0 |             30.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.274537 |             0.0 |                1.274537 |      0.004509 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.004509 |      0.0 |      -0.004509 |         1.38923 |              0.0 |      0.0 |    88.096653 |          0.0 |  6.274671 |
    | 2017-03-27 |           0.0 |         7.0 |              0.86539 |                     0.0 |         0.0 |              31.5 |      0.0 |     0.0 |           0.0 |         0.0 |                  1.564 |             0.0 |                   1.564 |      0.006325 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.006325 |      0.0 |      -0.006325 |        1.136258 |              0.0 |      0.0 |    86.538977 |          0.0 |  5.132088 |
    | 2017-03-28 |           0.0 |         8.3 |              0.84698 |                     0.0 |         0.0 |             37.35 |      0.0 |     0.0 |           0.0 |         0.0 |                1.85009 |             0.0 |                 1.85009 |      0.009147 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.009147 |      0.0 |      -0.009147 |        0.928632 |              0.0 |      0.0 |    84.698034 |          0.0 |   4.19431 |
    | 2017-03-29 |           1.5 |         9.0 |             0.848245 |                     0.6 |         0.9 |              40.5 |      0.0 |     0.9 |           0.0 |    0.593189 |               0.193797 |             0.0 |                0.793797 |      0.013436 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          0.579753 | 0.115951 |       0.463802 |        0.844372 |              0.0 |      0.0 |    84.824484 |          0.0 |   3.81374 |
    | 2017-03-30 |           0.3 |        12.4 |             0.836744 |                     0.3 |         0.0 |              55.8 |      0.0 |     0.0 |           0.0 |         0.0 |               1.163186 |             0.0 |                1.463186 |      0.013106 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.013106 |      0.0 |      -0.013106 |        0.688938 |              0.0 |      0.0 |    83.674404 |          0.0 |  3.111696 |
    | 2017-03-31 |           0.2 |        15.0 |             0.812752 |                     0.2 |         0.0 |              67.5 |      0.0 |     0.0 |           0.0 |         0.0 |               2.415503 |             0.0 |                2.615503 |      0.016317 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.016317 |      0.0 |      -0.016317 |        0.561097 |              0.0 |      0.0 |    81.275217 |          0.0 |  2.534282 |
    | 2017-04-01 |           4.5 |        11.8 |             0.821296 |                     0.8 |         3.7 |              53.1 |      0.0 |     3.7 |           0.0 |    2.198782 |               0.671473 |             0.0 |                1.471473 |       0.02462 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          2.174162 | 0.434832 |        1.73933 |        0.774674 |              0.0 |      0.0 |    82.129582 |          0.0 |  3.498937 |
    | 2017-04-02 |           0.0 |         9.4 |             0.806121 |                     0.0 |         0.0 |              42.3 |      0.0 |     0.0 |           0.0 |         0.0 |               1.538871 |             0.0 |                1.538871 |      0.021401 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.021401 |      0.0 |      -0.021401 |         0.63037 |              0.0 |      0.0 |    80.612112 |          0.0 |  2.847166 |
    | 2017-04-03 |           0.0 |         8.1 |              0.78725 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               1.914427 |             0.0 |                1.914427 |      0.027329 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.027329 |      0.0 |      -0.027329 |         0.51115 |              0.0 |      0.0 |    78.725014 |          0.0 |  2.308687 |
    | 2017-04-04 |           0.0 |         7.9 |             0.767637 |                     0.0 |         0.0 |             35.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.997409 |             0.0 |                1.997409 |      0.036111 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.036111 |      0.0 |      -0.036111 |        0.411948 |              0.0 |      0.0 |    76.763716 |          0.0 |  1.860628 |
    | 2017-04-05 |           0.4 |         7.5 |             0.755831 |                     0.4 |         0.0 |             33.75 |      0.0 |     0.0 |           0.0 |         0.0 |               1.227698 |             0.0 |                1.627698 |      0.047047 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.047047 |      0.0 |      -0.047047 |        0.328746 |              0.0 |      0.0 |    75.583065 |          0.0 |  1.484835 |
    | 2017-04-06 |           0.0 |         7.2 |             0.740395 |                     0.0 |         0.0 |              32.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.598123 |             0.0 |                1.598123 |      0.054589 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.054589 |      0.0 |      -0.054589 |         0.25926 |              0.0 |      0.0 |    74.039531 |          0.0 |  1.170986 |
    | 2017-04-07 |           0.0 |         8.1 |             0.733578 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               0.747322 |             0.0 |                0.747322 |       0.06561 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |          -0.06561 |      0.0 |       -0.06561 |        0.200371 |              0.0 |      0.0 |     73.35782 |          0.0 |  0.905006 |
    | 2017-04-08 |           0.0 |         8.6 |             0.722179 |                     0.0 |         0.0 |              38.7 |      0.0 |     0.0 |           0.0 |         0.0 |                1.21086 |             0.0 |                 1.21086 |      0.070915 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.070915 |      0.0 |      -0.070915 |        0.151195 |              0.0 |      0.0 |    72.217875 |          0.0 |  0.682895 |
    | 2017-04-09 |           0.0 |        10.5 |             0.699815 |                     0.0 |         0.0 |             47.25 |      0.0 |     0.0 |           0.0 |         0.0 |               2.316823 |             0.0 |                2.316823 |      0.080413 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |         -0.080413 |      0.0 |      -0.080413 |        0.109211 |              0.0 |      0.0 |    69.981465 |          0.0 |  0.493271 |

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_urban_internal_irrigation:

internal irrigation
___________________

We turn off external irrigation but activate internal irrigation by setting the boolean
|CisternSource| parameter for the single grassland zone to |True|:

>>> withexternalirrigation(False)
>>> cisternsource(True)

In contrast to external irrigation, internal irrigation is limited to the water
collected in the cistern.  As the cistern is initially empty and can only receive water
from the dry grassland zone, which is also initially dry, it cannot irrigate the same
zone substantially:

.. integration-test::

    >>> conditions = test("whmod_urban_internal_irrigation", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | cisterninflow | cisternoverflow | cisterndemand | cisternextraction | internalirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | cisternwater | deepwater |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |               70.0 |           0.0 |             0.0 |           0.7 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.0 |       0.0 |
    | 2017-02-11 |           0.4 |        -1.5 |                  0.0 |                     0.4 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |               70.0 |           0.0 |             0.0 |           0.7 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.0 |       0.0 |
    | 2017-02-12 |           0.0 |        -0.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |               70.0 |           0.0 |             0.0 |           0.7 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.0 |       0.0 |
    | 2017-02-13 |           0.0 |        -1.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |               70.0 |           0.0 |             0.0 |           0.7 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.0 |       0.0 |
    | 2017-02-14 |           0.0 |        -1.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |               70.0 |           0.0 |             0.0 |           0.7 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.0 |       0.0 |
    | 2017-02-15 |           0.0 |         1.7 |                  0.0 |                     0.0 |         0.0 |              7.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |               70.0 |           0.0 |             0.0 |           0.7 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.0 |       0.0 |
    | 2017-02-16 |           0.2 |         4.4 |                  0.0 |                     0.2 |         0.0 |              19.8 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.2 |           0.0 |               70.0 |           0.0 |             0.0 |           0.7 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.0 |       0.0 |
    | 2017-02-17 |           4.5 |         4.5 |                0.041 |                     0.3 |         4.1 |             20.25 |      0.0 |     4.1 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |               65.9 |           0.0 |             0.0 |         0.659 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.1 |      0.0 |          4.1 |          0.0 |       0.0 |
    | 2017-02-18 |           0.0 |         3.4 |             0.040833 |                     0.1 |         0.0 |              15.3 |      0.0 |     0.0 |           0.0 |         0.0 |               0.016694 |             0.0 |                0.116694 |           0.0 |          65.916694 |           0.0 |             0.0 |      0.659167 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |     4.083306 |          0.0 |       0.0 |
    | 2017-02-19 |           3.2 |         4.8 |             0.068833 |                     0.4 |         2.8 |              21.6 |      0.0 |     2.8 |           0.0 |    0.000913 |                    0.0 |             0.0 |                     0.4 |           0.0 |          63.117607 |      0.000009 |             0.0 |      0.631176 |          0.000009 |           0.000913 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |     6.883306 |          0.0 |       0.0 |
    | 2017-02-20 |           4.6 |         6.7 |             0.110833 |                     0.3 |         4.2 |             30.15 |      0.0 |     4.2 |           0.0 |    0.005081 |                    0.0 |             0.0 |                     0.3 |           0.0 |          58.921775 |      0.000051 |             0.0 |      0.589218 |          0.000051 |           0.005081 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.1 |      0.0 |    11.083306 |          0.0 |       0.0 |
    | 2017-02-21 |           2.3 |         5.8 |             0.130387 |                     0.4 |         2.0 |              26.1 |      0.0 |     2.0 |           0.0 |    0.007998 |               0.044617 |             0.0 |                0.444617 |           0.0 |          56.969309 |       0.00008 |             0.0 |      0.569693 |           0.00008 |           0.007998 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    13.038689 |          0.0 |       0.0 |
    | 2017-02-22 |          18.0 |         6.5 |             0.305343 |                     0.4 |        17.6 |             29.25 |      0.0 |    17.6 |           0.0 |     0.10583 |               0.104408 |             0.0 |                0.504408 |           0.0 |          39.571548 |      0.001058 |             0.0 |      0.395715 |          0.001058 |            0.10583 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    30.534282 |          0.0 |       0.0 |
    | 2017-02-23 |          19.2 |         5.0 |             0.492782 |                     0.4 |        18.8 |              22.5 |      0.0 |    18.8 |           0.0 |    0.956956 |               0.056047 |             0.0 |                0.456047 |           0.0 |          21.678721 |       0.00957 |             0.0 |      0.216787 |           0.00957 |           0.956956 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    49.278235 |          0.0 |       0.0 |
    | 2017-02-24 |           0.4 |         3.0 |              0.48966 |                     0.4 |         0.0 |              13.5 |      0.0 |     0.0 |           0.0 |         0.0 |               0.312225 |             0.0 |                0.712225 |           0.0 |          21.033989 |           0.0 |             0.0 |       0.21034 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    48.966011 |          0.0 |       0.0 |
    | 2017-02-25 |           8.3 |         3.1 |             0.567882 |                     0.4 |         7.9 |             13.95 |      0.0 |     7.9 |           0.0 |     1.31587 |               0.077774 |             0.0 |                0.477774 |           0.0 |          14.527633 |      0.013159 |             0.0 |      0.145276 |          0.013159 |            1.31587 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    56.788237 |          0.0 |       0.0 |
    | 2017-02-26 |           5.3 |         7.1 |             0.605043 |                     0.4 |         4.9 |             31.95 |      0.0 |     4.9 |           0.0 |    1.183982 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |       0.01184 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    60.504255 |      0.01184 |       0.0 |
    | 2017-02-27 |           0.7 |         9.4 |              0.61209 |                     0.4 |         0.3 |              42.3 |      0.0 |     0.3 |           0.0 |     0.08499 |               0.779249 |             0.0 |                1.179249 |           0.0 |          10.059984 |       0.00085 |             0.0 |        0.1006 |           0.01269 |           1.268972 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    61.208988 |          0.0 |       0.0 |
    | 2017-02-28 |           2.7 |         4.6 |             0.624031 |                     0.4 |         2.3 |              20.7 |      0.0 |     2.3 |           0.0 |     0.67081 |               0.435112 |             0.0 |                0.835112 |           0.0 |                0.0 |      0.006708 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    62.403066 |     0.006708 |       0.0 |
    | 2017-03-01 |           1.6 |         3.7 |             0.630092 |                     0.6 |         1.0 |             16.65 |      0.0 |     1.0 |           0.0 |     0.30615 |               0.087743 |             0.0 |                0.687743 |           0.0 |                0.0 |      0.003061 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    63.009173 |      0.00977 |       0.0 |
    | 2017-03-02 |           2.5 |         4.7 |             0.642251 |                     0.6 |         1.9 |             21.15 |      0.0 |     1.9 |           0.0 |     0.59597 |               0.088098 |             0.0 |                0.688098 |           0.0 |                0.0 |       0.00596 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    64.225104 |     0.015729 |       0.0 |
    | 2017-03-03 |           0.6 |         5.9 |             0.637812 |                     0.6 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               0.443945 |             0.0 |                1.043945 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    63.781159 |     0.015729 |       0.0 |
    | 2017-03-04 |           0.2 |         7.7 |             0.630728 |                     0.2 |         0.0 |             34.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.708321 |             0.0 |                0.908321 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    63.072838 |     0.015729 |       0.0 |
    | 2017-03-05 |           1.7 |         6.3 |             0.636507 |                     0.6 |         1.1 |             28.35 |      0.0 |     1.1 |           0.0 |    0.345911 |               0.176271 |             0.0 |                0.776271 |           0.0 |                0.0 |      0.003459 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    63.650656 |     0.019188 |       0.0 |
    | 2017-03-06 |           0.3 |         3.7 |             0.633853 |                     0.3 |         0.0 |             16.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.265399 |             0.0 |                0.565399 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    63.385257 |     0.019188 |       0.0 |
    | 2017-03-07 |           0.0 |         1.6 |             0.627671 |                     0.0 |         0.0 |               7.2 |      0.0 |     0.0 |           0.0 |         0.0 |               0.618205 |             0.0 |                0.618205 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    62.767052 |     0.019188 |       0.0 |
    | 2017-03-08 |           1.8 |         4.0 |             0.635063 |                     0.6 |         1.2 |              18.0 |      0.0 |     1.2 |           0.0 |    0.372782 |               0.087957 |             0.0 |                0.687957 |           0.0 |                0.0 |      0.003728 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    63.506312 |     0.022916 |       0.0 |
    | 2017-03-09 |           8.9 |         5.6 |              0.69151 |                     0.5 |         8.3 |              25.2 |      0.0 |     8.3 |           0.0 |    2.655319 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |      0.026553 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.1 |      0.0 |    69.150993 |     0.049469 |       0.0 |
    | 2017-03-10 |           0.0 |         5.8 |             0.685118 |                     0.1 |         0.0 |              26.1 |      0.0 |     0.0 |           0.0 |         0.0 |               0.639166 |             0.0 |                0.739166 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    68.511827 |     0.049469 |       0.0 |
    | 2017-03-11 |           0.0 |         5.7 |             0.676018 |                     0.0 |         0.0 |             25.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.910062 |             0.0 |                0.910062 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    67.601765 |     0.049469 |       0.0 |
    | 2017-03-12 |           0.0 |         4.6 |              0.66515 |                     0.0 |         0.0 |              20.7 |      0.0 |     0.0 |           0.0 |         0.0 |               1.086751 |             0.0 |                1.086751 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    66.515014 |     0.049469 |       0.0 |
    | 2017-03-13 |           0.9 |         4.2 |             0.664372 |                     0.6 |         0.3 |              18.9 |      0.0 |     0.3 |           0.0 |    0.107801 |               0.270042 |             0.0 |                0.870042 |           0.0 |                0.0 |      0.001078 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    66.437171 |     0.050547 |       0.0 |
    | 2017-03-14 |           0.1 |         7.4 |             0.657174 |                     0.1 |         0.0 |              33.3 |      0.0 |     0.0 |           0.0 |         0.0 |               0.719793 |             0.0 |                0.819793 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    65.717379 |     0.050547 |       0.0 |
    | 2017-03-15 |           0.0 |         6.3 |             0.646422 |                     0.0 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |                1.07518 |             0.0 |                 1.07518 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    64.642199 |     0.050547 |       0.0 |
    | 2017-03-16 |           0.0 |         8.7 |             0.633959 |                     0.0 |         0.0 |             39.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.246271 |             0.0 |                1.246271 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    63.395928 |     0.050547 |       0.0 |
    | 2017-03-17 |           3.9 |         6.4 |             0.652032 |                     0.6 |         3.3 |              28.8 |      0.0 |     3.3 |           0.0 |    1.051129 |               0.441606 |             0.0 |                1.041606 |           0.0 |                0.0 |      0.010511 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    65.203193 |     0.061059 |       0.0 |
    | 2017-03-18 |           8.7 |         5.2 |             0.700879 |                     0.6 |         8.1 |              23.4 |      0.0 |     8.1 |           0.0 |    2.768662 |               0.446621 |             0.0 |                1.046621 |           0.0 |                0.0 |      0.027687 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |     70.08791 |     0.088745 |       0.0 |
    | 2017-03-19 |          26.4 |         5.1 |             0.853159 |                     0.5 |        25.8 |             22.95 |      0.0 |    25.8 |           0.0 |   10.572046 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |       0.10572 |        0.094466 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.1 |      0.0 |    85.315864 |          0.1 |       0.0 |
    | 2017-03-20 |          11.5 |         8.7 |             0.889323 |                     0.6 |        11.0 |             39.15 |      0.0 |    11.0 |           0.0 |    7.383569 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |      0.073836 |        0.073836 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    88.932295 |          0.1 |       0.0 |
    | 2017-03-21 |           0.9 |         6.2 |             0.881273 |                     0.6 |         0.3 |              27.9 |      0.0 |     0.3 |           0.0 |    0.223487 |               0.881489 |             0.0 |                1.481489 |           0.0 |                0.0 |      0.002235 |        0.002235 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    88.127319 |          0.1 |       0.0 |
    | 2017-03-22 |           0.0 |         5.9 |             0.861722 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.955144 |             0.0 |                1.955144 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    86.172175 |          0.1 |       0.0 |
    | 2017-03-23 |           0.0 |         5.2 |             0.846157 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.556499 |             0.0 |                1.556499 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    84.615675 |          0.1 |       0.0 |
    | 2017-03-24 |           0.0 |         5.2 |             0.830657 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.550024 |             0.0 |                1.550024 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    83.065651 |          0.1 |       0.0 |
    | 2017-03-25 |           0.0 |         5.9 |             0.819083 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.157394 |             0.0 |                1.157394 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    81.908257 |          0.1 |       0.0 |
    | 2017-03-26 |           0.0 |         6.7 |             0.806588 |                     0.0 |         0.0 |             30.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.249485 |             0.0 |                1.249485 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    80.658773 |          0.1 |       0.0 |
    | 2017-03-27 |           0.0 |         7.0 |              0.79127 |                     0.0 |         0.0 |              31.5 |      0.0 |     0.0 |           0.0 |         0.0 |               1.531773 |             0.0 |                1.531773 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |       79.127 |          0.1 |       0.0 |
    | 2017-03-28 |           0.0 |         8.3 |             0.773173 |                     0.0 |         0.0 |             37.35 |      0.0 |     0.0 |           0.0 |         0.0 |               1.809702 |             0.0 |                1.809702 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    77.317297 |          0.1 |       0.0 |
    | 2017-03-29 |           1.5 |         9.0 |             0.775562 |                     0.6 |         0.9 |              40.5 |      0.0 |     0.9 |           0.0 |    0.471845 |               0.189268 |             0.0 |                0.789268 |           0.0 |                0.0 |      0.004718 |        0.004718 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    77.556185 |          0.1 |       0.0 |
    | 2017-03-30 |           0.3 |        12.4 |             0.764196 |                     0.3 |         0.0 |              55.8 |      0.0 |     0.0 |           0.0 |         0.0 |               1.136609 |             0.0 |                1.436609 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    76.419575 |          0.1 |       0.0 |
    | 2017-03-31 |           0.2 |        15.0 |             0.740617 |                     0.2 |         0.0 |              67.5 |      0.0 |     0.0 |           0.0 |         0.0 |               2.357864 |             0.0 |                2.557864 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    74.061712 |          0.1 |       0.0 |
    | 2017-04-01 |           4.5 |        11.8 |             0.753665 |                     0.8 |         3.7 |              53.1 |      0.0 |     3.7 |           0.0 |     1.74125 |               0.653968 |             0.0 |                1.453968 |           0.0 |                0.0 |      0.017412 |        0.017412 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    75.366495 |          0.1 |       0.0 |
    | 2017-04-02 |           0.0 |         9.4 |             0.738637 |                     0.0 |         0.0 |              42.3 |      0.0 |     0.0 |           0.0 |         0.0 |               1.502818 |             0.0 |                1.502818 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    73.863677 |          0.1 |       0.0 |
    | 2017-04-03 |           0.0 |         8.1 |             0.719968 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               1.866913 |             0.0 |                1.866913 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    71.996764 |          0.1 |       0.0 |
    | 2017-04-04 |           0.0 |         7.9 |             0.700526 |                     0.0 |         0.0 |             35.55 |      0.0 |     0.0 |           0.0 |         0.0 |                1.94417 |             0.0 |                 1.94417 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    70.052594 |          0.1 |       0.0 |
    | 2017-04-05 |           0.4 |         7.5 |             0.688602 |                     0.4 |         0.0 |             33.75 |      0.0 |     0.0 |           0.0 |         0.0 |               1.192434 |             0.0 |                1.592434 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    68.860161 |          0.1 |       0.0 |
    | 2017-04-06 |           0.0 |         7.2 |             0.673102 |                     0.0 |         0.0 |              32.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.549928 |             0.0 |                1.549928 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    67.310232 |          0.1 |       0.0 |
    | 2017-04-07 |           0.0 |         8.1 |             0.665869 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |                0.72334 |             0.0 |                 0.72334 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    66.586892 |          0.1 |       0.0 |
    | 2017-04-08 |           0.0 |         8.6 |             0.654162 |                     0.0 |         0.0 |              38.7 |      0.0 |     0.0 |           0.0 |         0.0 |               1.170664 |             0.0 |                1.170664 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    65.416228 |          0.1 |       0.0 |
    | 2017-04-09 |           0.0 |        10.5 |             0.631803 |                     0.0 |         0.0 |             47.25 |      0.0 |     0.0 |           0.0 |         0.0 |               2.235957 |             0.0 |                2.235957 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    63.180271 |          0.1 |       0.0 |

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_urban_combined_irrigation:

combined irrigation
___________________

In this example, we turn on external irrigation and fill the cistern to show how both
types of irrigation interact:

>>> withexternalirrigation(True)
>>> test.inits.cisternwater = 0.1

|whmod_urban| first takes as much irrigation water from the cistern as required or
available and then uses the (unlimited) external source to satisfy the remaining
demand:

.. integration-test::

    >>> conditions = test("whmod_urban_combined_irrigation", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | cisterninflow | cisternoverflow | cisterndemand | cisternextraction | internalirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | cisternwater | deepwater |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |                  0.7 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |               70.0 |           0.0 |             0.0 |           0.7 |               0.1 |               10.0 |               60.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |         70.0 |          0.0 |       0.0 |
    | 2017-02-11 |           0.4 |        -1.5 |             0.696332 |                     0.4 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.366807 |             0.0 |                0.766807 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    69.633193 |          0.0 |       0.0 |
    | 2017-02-12 |           0.0 |        -0.9 |             0.689925 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.640736 |             0.0 |                0.640736 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    68.992457 |          0.0 |       0.0 |
    | 2017-02-13 |           0.0 |        -1.6 |             0.686275 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.364939 |             0.0 |                0.364939 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    68.627518 |          0.0 |       0.0 |
    | 2017-02-14 |           0.0 |        -1.3 |             0.682633 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.364246 |             0.0 |                0.364246 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    68.263272 |          0.0 |       0.0 |
    | 2017-02-15 |           0.0 |         1.7 |             0.678997 |                     0.0 |         0.0 |              7.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.363546 |             0.0 |                0.363546 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    67.899726 |          0.0 |       0.0 |
    | 2017-02-16 |           0.2 |         4.4 |             0.677183 |                     0.2 |         0.0 |              19.8 |      0.0 |     0.0 |           0.0 |         0.0 |               0.181419 |             0.0 |                0.381419 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    67.718307 |          0.0 |       0.0 |
    | 2017-02-17 |           4.5 |         4.5 |             0.702772 |                     0.3 |         4.1 |             20.25 |      0.0 |     4.1 |           0.0 |    1.541093 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |      0.015411 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.1 |      0.0 |    70.277214 |     0.015411 |       0.0 |
    | 2017-02-18 |           0.0 |         3.4 |             0.700936 |                     0.1 |         0.0 |              15.3 |      0.0 |     0.0 |           0.0 |         0.0 |               0.183655 |             0.0 |                0.283655 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |     70.09356 |     0.015411 |       0.0 |
    | 2017-02-19 |           3.2 |         4.8 |              0.71746 |                     0.4 |         2.8 |              21.6 |      0.0 |     2.8 |           0.0 |    1.147586 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |      0.011476 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    71.745974 |     0.026887 |       0.0 |
    | 2017-02-20 |           4.6 |         6.7 |             0.741209 |                     0.3 |         4.2 |             30.15 |      0.0 |     4.2 |           0.0 |    1.825063 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |      0.018251 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.1 |      0.0 |    74.120911 |     0.045137 |       0.0 |
    | 2017-02-21 |           2.3 |         5.8 |             0.749909 |                     0.4 |         2.0 |              26.1 |      0.0 |     2.0 |           0.0 |    0.943106 |               0.186894 |             0.0 |                0.586894 |           0.0 |                0.0 |      0.009431 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    74.990911 |     0.054568 |       0.0 |
    | 2017-02-22 |          18.0 |         6.5 |             0.836697 |                     0.4 |        17.6 |             29.25 |      0.0 |    17.6 |           0.0 |    8.546025 |               0.375136 |             0.0 |                0.775136 |           0.0 |                0.0 |       0.08546 |        0.040029 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    83.669749 |          0.1 |       0.0 |
    | 2017-02-23 |          19.2 |         5.0 |             0.903562 |                     0.4 |        18.8 |              22.5 |      0.0 |    18.8 |           0.0 |   12.016894 |               0.096619 |             0.0 |                0.496619 |           0.0 |                0.0 |      0.120169 |        0.120169 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    90.356236 |          0.1 |       0.0 |
    | 2017-02-24 |           0.4 |         3.0 |             0.899632 |                     0.4 |         0.0 |              13.5 |      0.0 |     0.0 |           0.0 |         0.0 |               0.393035 |             0.0 |                0.793035 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    89.963201 |          0.1 |       0.0 |
    | 2017-02-25 |           8.3 |         3.1 |             0.917071 |                     0.4 |         7.9 |             13.95 |      0.0 |     7.9 |           0.0 |    6.057907 |               0.098173 |             0.0 |                0.498173 |           0.0 |                0.0 |      0.060579 |        0.060579 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    91.707121 |          0.1 |       0.0 |
    | 2017-02-26 |           5.3 |         7.1 |             0.926642 |                     0.4 |         4.9 |             31.95 |      0.0 |     4.9 |           0.0 |    3.942954 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |       0.03943 |         0.03943 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    92.664167 |          0.1 |       0.0 |
    | 2017-02-27 |           0.7 |         9.4 |             0.918278 |                     0.4 |         0.3 |              42.3 |      0.0 |     0.3 |           0.0 |    0.247779 |                0.88862 |             0.0 |                 1.28862 |           0.0 |                0.0 |      0.002478 |        0.002478 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    91.827768 |          0.1 |       0.0 |
    | 2017-02-28 |           2.7 |         4.6 |              0.91778 |                     0.4 |         2.3 |              20.7 |      0.0 |     2.3 |           0.0 |    1.856892 |               0.492838 |             0.0 |                0.892838 |           0.0 |                0.0 |      0.018569 |        0.018569 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    91.778038 |          0.1 |       0.0 |
    | 2017-03-01 |           1.6 |         3.7 |             0.918732 |                     0.6 |         1.0 |             16.65 |      0.0 |     1.0 |           0.0 |    0.806247 |               0.098557 |             0.0 |                0.698557 |           0.0 |                0.0 |      0.008062 |        0.008062 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    91.873233 |          0.1 |       0.0 |
    | 2017-03-02 |           2.5 |         4.7 |             0.921388 |                     0.6 |         1.9 |             21.15 |      0.0 |     1.9 |           0.0 |    1.535862 |               0.098577 |             0.0 |                0.698577 |           0.0 |                0.0 |      0.015359 |        0.015359 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    92.138795 |          0.1 |       0.0 |
    | 2017-03-03 |           0.6 |         5.9 |             0.916456 |                     0.6 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               0.493153 |             0.0 |                1.093153 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    91.645642 |          0.1 |       0.0 |
    | 2017-03-04 |           0.2 |         7.7 |             0.908574 |                     0.2 |         0.0 |             34.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.788242 |             0.0 |                0.988242 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |      90.8574 |          0.1 |       0.0 |
    | 2017-03-05 |           1.7 |         6.3 |              0.90896 |                     0.6 |         1.1 |             28.35 |      0.0 |     1.1 |           0.0 |     0.86471 |               0.196732 |             0.0 |                0.796732 |           0.0 |                0.0 |      0.008647 |        0.008647 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    90.895958 |          0.1 |       0.0 |
    | 2017-03-06 |           0.3 |         3.7 |             0.906008 |                     0.3 |         0.0 |             16.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.295122 |             0.0 |                0.595122 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    90.600836 |          0.1 |       0.0 |
    | 2017-03-07 |           0.0 |         1.6 |             0.899127 |                     0.0 |         0.0 |               7.2 |      0.0 |     0.0 |           0.0 |         0.0 |               0.688179 |             0.0 |                0.688179 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    89.912656 |          0.1 |       0.0 |
    | 2017-03-08 |           1.8 |         4.0 |             0.900956 |                     0.6 |         1.2 |              18.0 |      0.0 |     1.2 |           0.0 |    0.918891 |               0.098162 |             0.0 |                0.698162 |           0.0 |                0.0 |      0.009189 |        0.009189 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    90.095603 |          0.1 |       0.0 |
    | 2017-03-09 |           8.9 |         5.6 |             0.920074 |                     0.5 |         8.3 |              25.2 |      0.0 |     8.3 |           0.0 |    6.388175 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |      0.063882 |        0.063882 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.1 |      0.0 |    92.007428 |          0.1 |       0.0 |
    | 2017-03-10 |           0.0 |         5.8 |             0.913172 |                     0.1 |         0.0 |              26.1 |      0.0 |     0.0 |           0.0 |         0.0 |               0.690229 |             0.0 |                0.790229 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    91.317199 |          0.1 |       0.0 |
    | 2017-03-11 |           0.0 |         5.7 |             0.903326 |                     0.0 |         0.0 |             25.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.984624 |             0.0 |                0.984624 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    90.332575 |          0.1 |       0.0 |
    | 2017-03-12 |           0.0 |         4.6 |             0.891535 |                     0.0 |         0.0 |              20.7 |      0.0 |     0.0 |           0.0 |         0.0 |               1.179045 |             0.0 |                1.179045 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    89.153531 |          0.1 |       0.0 |
    | 2017-03-13 |           0.9 |         4.2 |             0.889347 |                     0.6 |         0.3 |              18.9 |      0.0 |     0.3 |           0.0 |    0.224885 |                0.29398 |             0.0 |                 0.89398 |           0.0 |                0.0 |      0.002249 |        0.002249 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    88.934665 |          0.1 |       0.0 |
    | 2017-03-14 |           0.1 |         7.4 |             0.881511 |                     0.1 |         0.0 |              33.3 |      0.0 |     0.0 |           0.0 |         0.0 |                0.78355 |             0.0 |                 0.88355 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    88.151115 |          0.1 |       0.0 |
    | 2017-03-15 |           0.0 |         6.3 |              0.86978 |                     0.0 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |               1.173153 |             0.0 |                1.173153 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    86.977961 |          0.1 |       0.0 |
    | 2017-03-16 |           0.0 |         8.7 |             0.856132 |                     0.0 |         0.0 |             39.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.364743 |             0.0 |                1.364743 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    85.613219 |          0.1 |       0.0 |
    | 2017-03-17 |           3.9 |         6.4 |              0.86193 |                     0.6 |         3.3 |              28.8 |      0.0 |     3.3 |           0.0 |    2.234501 |               0.485693 |             0.0 |                1.085693 |           0.0 |                0.0 |      0.022345 |        0.022345 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    86.193025 |          0.1 |       0.0 |
    | 2017-03-18 |           8.7 |         5.2 |             0.882282 |                     0.6 |         8.1 |              23.4 |      0.0 |     8.1 |           0.0 |      5.5784 |               0.486432 |             0.0 |                1.086432 |           0.0 |                0.0 |      0.055784 |        0.055784 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    88.228193 |          0.1 |       0.0 |
    | 2017-03-19 |          26.4 |         5.1 |              0.95188 |                     0.5 |        25.8 |             22.95 |      0.0 |    25.8 |           0.0 |   18.840199 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |      0.188402 |        0.188402 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.1 |      0.0 |    95.187994 |          0.1 |       0.0 |
    | 2017-03-20 |          11.5 |         8.7 |             0.964688 |                     0.6 |        11.0 |             39.15 |      0.0 |    11.0 |           0.0 |    9.719199 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |      0.097192 |        0.097192 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    96.468795 |          0.1 |       0.0 |
    | 2017-03-21 |           0.9 |         6.2 |             0.955998 |                     0.6 |         0.3 |              27.9 |      0.0 |     0.3 |           0.0 |    0.274113 |               0.894919 |             0.0 |                1.494919 |           0.0 |                0.0 |      0.002741 |        0.002741 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    95.599762 |          0.1 |       0.0 |
    | 2017-03-22 |           0.0 |         5.9 |             0.936141 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.985689 |             0.0 |                1.985689 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    93.614073 |          0.1 |       0.0 |
    | 2017-03-23 |           0.0 |         5.2 |             0.920314 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.582719 |             0.0 |                1.582719 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    92.031355 |          0.1 |       0.0 |
    | 2017-03-24 |           0.0 |         5.2 |             0.904536 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.577743 |             0.0 |                1.577743 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    90.453612 |          0.1 |       0.0 |
    | 2017-03-25 |           0.0 |         5.9 |             0.892743 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.179357 |             0.0 |                1.179357 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    89.274254 |          0.1 |       0.0 |
    | 2017-03-26 |           0.0 |         6.7 |                 0.88 |                     0.0 |         0.0 |             30.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.274268 |             0.0 |                1.274268 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    87.999987 |          0.1 |       0.0 |
    | 2017-03-27 |           0.0 |         7.0 |             0.864364 |                     0.0 |         0.0 |              31.5 |      0.0 |     0.0 |           0.0 |         0.0 |               1.563636 |             0.0 |                1.563636 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |     86.43635 |          0.1 |       0.0 |
    | 2017-03-28 |           0.0 |         8.3 |             0.845867 |                     0.0 |         0.0 |             37.35 |      0.0 |     0.0 |           0.0 |         0.0 |               1.849604 |             0.0 |                1.849604 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    84.586747 |          0.1 |       0.0 |
    | 2017-03-29 |           1.5 |         9.0 |             0.847018 |                     0.6 |         0.9 |              40.5 |      0.0 |     0.9 |           0.0 |    0.591234 |               0.193738 |             0.0 |                0.793738 |           0.0 |                0.0 |      0.005912 |        0.005912 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    84.701775 |          0.1 |       0.0 |
    | 2017-03-30 |           0.3 |        12.4 |              0.83539 |                     0.3 |         0.0 |              55.8 |      0.0 |     0.0 |           0.0 |         0.0 |               1.162794 |             0.0 |                1.462794 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    83.538981 |          0.1 |       0.0 |
    | 2017-03-31 |           0.2 |        15.0 |             0.811244 |                     0.2 |         0.0 |              67.5 |      0.0 |     0.0 |           0.0 |         0.0 |               2.414563 |             0.0 |                2.614563 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    81.124418 |          0.1 |       0.0 |
    | 2017-04-01 |           4.5 |        11.8 |             0.819647 |                     0.8 |         3.7 |              53.1 |      0.0 |     3.7 |           0.0 |    2.188556 |               0.671152 |             0.0 |                1.471152 |           0.0 |                0.0 |      0.021886 |        0.021886 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |     81.96471 |          0.1 |       0.0 |
    | 2017-04-02 |           0.0 |         9.4 |             0.804266 |                     0.0 |         0.0 |              42.3 |      0.0 |     0.0 |           0.0 |         0.0 |               1.538095 |             0.0 |                1.538095 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    80.426616 |          0.1 |       0.0 |
    | 2017-04-03 |           0.0 |         8.1 |             0.785133 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               1.913271 |             0.0 |                1.913271 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    78.513345 |          0.1 |       0.0 |
    | 2017-04-04 |           0.0 |         7.9 |             0.765174 |                     0.0 |         0.0 |             35.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.995925 |             0.0 |                1.995925 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |     76.51742 |          0.1 |       0.0 |
    | 2017-04-05 |           0.4 |         7.5 |             0.752909 |                     0.4 |         0.0 |             33.75 |      0.0 |     0.0 |           0.0 |         0.0 |               1.226549 |             0.0 |                1.626549 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    75.290871 |          0.1 |       0.0 |
    | 2017-04-06 |           0.0 |         7.2 |             0.736946 |                     0.0 |         0.0 |              32.4 |      0.0 |     0.0 |           0.0 |         0.0 |                1.59626 |             0.0 |                 1.59626 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    73.694611 |          0.1 |       0.0 |
    | 2017-04-07 |           0.0 |         8.1 |             0.729484 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               0.746227 |             0.0 |                0.746227 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    72.948385 |          0.1 |       0.0 |
    | 2017-04-08 |           0.0 |         8.6 |             0.717397 |                     0.0 |         0.0 |              38.7 |      0.0 |     0.0 |           0.0 |         0.0 |               1.208693 |             0.0 |                1.208693 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    71.739691 |          0.1 |       0.0 |
    | 2017-04-09 |           0.0 |        10.5 |             0.694279 |                     0.0 |         0.0 |             47.25 |      0.0 |     0.0 |           0.0 |         0.0 |               2.311749 |             0.0 |                2.311749 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |    69.427942 |          0.1 |       0.0 |

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_urban_water:

water
_____

Water areas do neither fill nor empty the cistern.  We show this by setting the
cistern's initial water content to 0.05 m³, half its previously set capacity:

>>> landtype(WATER)
>>> soiltype(NONE)
>>> withexternalirrigation(False)
>>> test.inits.cisternwater = 0.05

The other settings are stem from the :ref:`whmod_rural_water` example of |whmod_rural|.
The cistern's water content remains constant.  Otherwise, the results are identical to
those of the original example:

.. integration-test::

    >>> landmonthfactor = petmodel.parameters.control.landmonthfactor
    >>> landmonthfactor.water_feb = 1.217
    >>> landmonthfactor.water_mar = 1.256
    >>> landmonthfactor.water_apr = 1.283
    >>> conditions = test("whmod_urban_water", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | cisterninflow | cisternoverflow | cisterndemand | cisternextraction | internalirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | cisternwater | deepwater |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.7302 |                  0.7302 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.7302 |      0.0 |        -0.7302 |       -0.132363 |              0.0 |      0.0 |          0.0 |         0.05 | -0.597837 |
    | 2017-02-11 |           0.4 |        -1.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.9736 |                  0.9736 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.5736 |      0.0 |        -0.5736 |       -0.212346 |              0.0 |      0.0 |          0.0 |         0.05 | -0.959092 |
    | 2017-02-12 |           0.0 |        -0.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8519 |                  0.8519 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.8519 |      0.0 |        -0.8519 |       -0.328277 |              0.0 |      0.0 |          0.0 |         0.05 | -1.482715 |
    | 2017-02-13 |           0.0 |        -1.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.4868 |      0.0 |        -0.4868 |       -0.357012 |              0.0 |      0.0 |          0.0 |         0.05 | -1.612502 |
    | 2017-02-14 |           0.0 |        -1.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.4868 |      0.0 |        -0.4868 |       -0.380539 |              0.0 |      0.0 |          0.0 |         0.05 | -1.718763 |
    | 2017-02-15 |           0.0 |         1.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.4868 |      0.0 |        -0.4868 |       -0.399801 |              0.0 |      0.0 |          0.0 |         0.05 | -1.805762 |
    | 2017-02-16 |           0.2 |         4.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.2868 |      0.0 |        -0.2868 |       -0.379317 |              0.0 |      0.0 |          0.0 |         0.05 | -1.713245 |
    | 2017-02-17 |           4.5 |         4.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.3651 |                  0.3651 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            4.1349 |  0.82698 |        3.30792 |        0.289065 |              0.0 |      0.0 |          0.0 |         0.05 |  1.305609 |
    | 2017-02-18 |           0.0 |         3.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.3651 |                  0.3651 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.3651 |      0.0 |        -0.3651 |        0.170485 |              0.0 |      0.0 |          0.0 |         0.05 |  0.770024 |
    | 2017-02-19 |           3.2 |         4.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            2.7132 |  0.54264 |        2.17056 |        0.533037 |              0.0 |      0.0 |          0.0 |         0.05 |  2.407546 |
    | 2017-02-20 |           4.6 |         6.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.3651 |                  0.3651 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            4.2349 |  0.84698 |        3.38792 |         1.05054 |              0.0 |      0.0 |          0.0 |         0.05 |  4.744927 |
    | 2017-02-21 |           2.3 |         5.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.7302 |                  0.7302 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            1.5698 |  0.31396 |        1.25584 |        1.087754 |              0.0 |      0.0 |          0.0 |         0.05 |  4.913012 |
    | 2017-02-22 |          18.0 |         6.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.9736 |                  0.9736 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           17.0264 |  3.40528 |       13.62112 |        3.359668 |              0.0 |      0.0 |          0.0 |         0.05 | 15.174464 |
    | 2017-02-23 |          19.2 |         5.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.6085 |                  0.6085 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           18.5915 |   3.7183 |        14.8732 |        5.446717 |              0.0 |      0.0 |          0.0 |         0.05 | 24.600947 |
    | 2017-02-24 |           0.4 |         3.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.9736 |                  0.9736 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.5736 |      0.0 |        -0.5736 |        4.355419 |              0.0 |      0.0 |          0.0 |         0.05 | 19.671928 |
    | 2017-02-25 |           8.3 |         3.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.6085 |                  0.6085 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            7.6915 |   1.5383 |         6.1532 |        4.681301 |              0.0 |      0.0 |          0.0 |         0.05 | 21.143826 |
    | 2017-02-26 |           5.3 |         7.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            4.8132 |  0.96264 |        3.85056 |        4.530714 |              0.0 |      0.0 |          0.0 |         0.05 | 20.463673 |
    | 2017-02-27 |           0.7 |         9.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.5821 |                  1.5821 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.8821 |      0.0 |        -0.8821 |        3.549537 |              0.0 |      0.0 |          0.0 |         0.05 | 16.032036 |
    | 2017-02-28 |           2.7 |         4.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0953 |                  1.0953 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            1.6047 |  0.32094 |        1.28376 |        3.138821 |              0.0 |      0.0 |          0.0 |         0.05 | 14.176974 |
    | 2017-03-01 |           1.6 |         3.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8792 |                  0.8792 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            0.7208 |  0.14416 |        0.57664 |        2.674377 |              0.0 |      0.0 |          0.0 |         0.05 | 12.079238 |
    | 2017-03-02 |           2.5 |         4.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8792 |                  0.8792 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            1.6208 |  0.32416 |        1.29664 |        2.424635 |              0.0 |      0.0 |          0.0 |         0.05 | 10.951243 |
    | 2017-03-03 |           0.6 |         5.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.3816 |                  1.3816 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.7816 |      0.0 |        -0.7816 |        1.843443 |              0.0 |      0.0 |          0.0 |         0.05 |  8.326199 |
    | 2017-03-04 |           0.2 |         7.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           1.256 |                   1.256 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            -1.056 |      0.0 |         -1.056 |        1.317864 |              0.0 |      0.0 |          0.0 |         0.05 |  5.952336 |
    | 2017-03-05 |           1.7 |         6.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0048 |                  1.0048 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            0.6952 |  0.13904 |        0.55616 |         1.17979 |              0.0 |      0.0 |          0.0 |         0.05 |  5.328705 |
    | 2017-03-06 |           0.3 |         3.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.7536 |                  0.7536 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.4536 |      0.0 |        -0.4536 |        0.883707 |              0.0 |      0.0 |          0.0 |         0.05 |  3.991399 |
    | 2017-03-07 |           0.0 |         1.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8792 |                  0.8792 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.8792 |      0.0 |        -0.8792 |        0.564146 |              0.0 |      0.0 |          0.0 |         0.05 |  2.548053 |
    | 2017-03-08 |           1.8 |         4.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8792 |                  0.8792 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            0.9208 |  0.18416 |        0.73664 |        0.595414 |              0.0 |      0.0 |          0.0 |         0.05 |  2.689279 |
    | 2017-03-09 |           8.9 |         5.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           0.628 |                   0.628 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |             8.272 |   1.6544 |         6.6176 |        1.687051 |              0.0 |      0.0 |          0.0 |         0.05 |  7.619828 |
    | 2017-03-10 |           0.0 |         5.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0048 |                  1.0048 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.0048 |      0.0 |        -1.0048 |        1.199101 |              0.0 |      0.0 |          0.0 |         0.05 |  5.415927 |
    | 2017-03-11 |           0.0 |         5.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           1.256 |                   1.256 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            -1.256 |      0.0 |         -1.256 |        0.754067 |              0.0 |      0.0 |          0.0 |         0.05 |   3.40586 |
    | 2017-03-12 |           0.0 |         4.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.5072 |                  1.5072 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.5072 |      0.0 |        -1.5072 |        0.344169 |              0.0 |      0.0 |          0.0 |         0.05 |  1.554491 |
    | 2017-03-13 |           0.9 |         4.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.1304 |                  1.1304 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -0.2304 |      0.0 |        -0.2304 |        0.240017 |              0.0 |      0.0 |          0.0 |         0.05 |  1.084074 |
    | 2017-03-14 |           0.1 |         7.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.1304 |                  1.1304 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.0304 |      0.0 |        -1.0304 |         0.00973 |              0.0 |      0.0 |          0.0 |         0.05 |  0.043945 |
    | 2017-03-15 |           0.0 |         6.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.5072 |                  1.5072 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.5072 |      0.0 |        -1.5072 |       -0.265243 |              0.0 |      0.0 |          0.0 |         0.05 | -1.198012 |
    | 2017-03-16 |           0.0 |         8.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.7584 |                  1.7584 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.7584 |      0.0 |        -1.7584 |       -0.535907 |              0.0 |      0.0 |          0.0 |         0.05 | -2.420505 |
    | 2017-03-17 |           3.9 |         6.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.3816 |                  1.3816 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            2.5184 |  0.50368 |        2.01472 |       -0.073556 |              0.0 |      0.0 |          0.0 |         0.05 | -0.332229 |
    | 2017-03-18 |           8.7 |         5.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.3816 |                  1.3816 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            7.3184 |  1.46368 |        5.85472 |        1.001058 |              0.0 |      0.0 |          0.0 |         0.05 |  4.521433 |
    | 2017-03-19 |          26.4 |         5.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           0.628 |                   0.628 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            25.772 |   5.1544 |        20.6176 |        4.556934 |              0.0 |      0.0 |          0.0 |         0.05 |   20.5821 |
    | 2017-03-20 |          11.5 |         8.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.7536 |                  0.7536 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           10.7464 |  2.14928 |        8.59712 |        5.289295 |              0.0 |      0.0 |          0.0 |         0.05 | 23.889924 |
    | 2017-03-21 |           0.9 |         6.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           1.884 |                   1.884 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            -0.984 |      0.0 |         -0.984 |         4.15214 |              0.0 |      0.0 |          0.0 |         0.05 | 18.753785 |
    | 2017-03-22 |           0.0 |         5.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           2.512 |                   2.512 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            -2.512 |      0.0 |         -2.512 |        2.944136 |              0.0 |      0.0 |          0.0 |         0.05 | 13.297649 |
    | 2017-03-23 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.0096 |                  2.0096 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -2.0096 |      0.0 |        -2.0096 |        2.046176 |              0.0 |      0.0 |          0.0 |         0.05 |  9.241873 |
    | 2017-03-24 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.0096 |                  2.0096 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -2.0096 |      0.0 |        -2.0096 |        1.310989 |              0.0 |      0.0 |          0.0 |         0.05 |  5.921284 |
    | 2017-03-25 |           0.0 |         5.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.5072 |                  1.5072 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.5072 |      0.0 |        -1.5072 |        0.800138 |              0.0 |      0.0 |          0.0 |         0.05 |  3.613946 |
    | 2017-03-26 |           0.0 |         6.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.6328 |                  1.6328 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.6328 |      0.0 |        -1.6328 |        0.359121 |              0.0 |      0.0 |          0.0 |         0.05 |  1.622025 |
    | 2017-03-27 |           0.0 |         7.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.0096 |                  2.0096 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -2.0096 |      0.0 |        -2.0096 |       -0.070255 |              0.0 |      0.0 |          0.0 |         0.05 | -0.317319 |
    | 2017-03-28 |           0.0 |         8.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.3864 |                  2.3864 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -2.3864 |      0.0 |        -2.3864 |       -0.490101 |              0.0 |      0.0 |          0.0 |         0.05 | -2.213618 |
    | 2017-03-29 |           1.5 |         9.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0048 |                  1.0048 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            0.4952 |  0.09904 |        0.39616 |       -0.329449 |              0.0 |      0.0 |          0.0 |         0.05 | -1.488009 |
    | 2017-03-30 |           0.3 |        12.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           1.884 |                   1.884 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            -1.584 |      0.0 |         -1.584 |       -0.556861 |              0.0 |      0.0 |          0.0 |         0.05 | -2.515148 |
    | 2017-03-31 |           0.2 |        15.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          3.3912 |                  3.3912 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -3.1912 |      0.0 |        -3.1912 |       -1.034385 |              0.0 |      0.0 |          0.0 |         0.05 | -4.671963 |
    | 2017-04-01 |           4.5 |        11.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.9245 |                  1.9245 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            2.5755 |   0.5151 |         2.0604 |       -0.473396 |              0.0 |      0.0 |          0.0 |         0.05 | -2.138167 |
    | 2017-04-02 |           0.0 |         9.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.0528 |                  2.0528 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -2.0528 |      0.0 |        -2.0528 |       -0.759693 |              0.0 |      0.0 |          0.0 |         0.05 | -3.431273 |
    | 2017-04-03 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           2.566 |                   2.566 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |            -2.566 |      0.0 |         -2.566 |       -1.087121 |              0.0 |      0.0 |          0.0 |         0.05 | -4.910152 |
    | 2017-04-04 |           0.0 |         7.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.6943 |                  2.6943 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -2.6943 |      0.0 |        -2.6943 |       -1.378453 |              0.0 |      0.0 |          0.0 |         0.05 | -6.225999 |
    | 2017-04-05 |           0.4 |         7.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.1811 |                  2.1811 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.7811 |      0.0 |        -1.7811 |       -1.451441 |              0.0 |      0.0 |          0.0 |         0.05 | -6.555658 |
    | 2017-04-06 |           0.0 |         7.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.1811 |                  2.1811 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -2.1811 |      0.0 |        -2.1811 |       -1.583706 |              0.0 |      0.0 |          0.0 |         0.05 | -7.153052 |
    | 2017-04-07 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0264 |                  1.0264 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.0264 |      0.0 |        -1.0264 |       -1.482683 |              0.0 |      0.0 |          0.0 |         0.05 | -6.696769 |
    | 2017-04-08 |           0.0 |         8.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.6679 |                  1.6679 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -1.6679 |      0.0 |        -1.6679 |       -1.516257 |              0.0 |      0.0 |          0.0 |         0.05 | -6.848412 |
    | 2017-04-09 |           0.0 |        10.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          3.2075 |                  3.2075 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |           -3.2075 |      0.0 |        -3.2075 |       -1.822828 |              0.0 |      0.0 |          0.0 |         0.05 | -8.233084 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_urban_lost_surface_runoff:

unused surface runoff
_____________________

Zones with sealed surfaces are either connected or unconnected to the cistern.  We
first demonstrate the case of an unconnected zone:

>>> landtype(SEALED)
>>> cisternsource(False)

As typical for |whmod_rural| and demonstrated in the :ref:`whmod_rural_sealed` example,
all surface runoff leaves the considered system and is lost for irrigation:

.. integration-test::

    >>> interceptioncapacity.sealed = 2.0
    >>> landmonthfactor.sealed = 1.0
    >>> conditions = test("whmod_urban_unused_surface_runoff", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | cisterninflow | cisternoverflow | cisterndemand | cisternextraction | internalirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | cisternwater | deepwater |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-11 |           0.4 |        -1.5 |                  0.0 |                     0.4 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-12 |           0.0 |        -0.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-13 |           0.0 |        -1.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-14 |           0.0 |        -1.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-15 |           0.0 |         1.7 |                  0.0 |                     0.0 |         0.0 |              7.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-16 |           0.2 |         4.4 |                  0.0 |                     0.2 |         0.0 |              19.8 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.2 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-17 |           4.5 |         4.5 |                  0.0 |                     0.3 |         2.5 |             20.25 |      0.0 |     2.5 |           2.5 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.7 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-18 |           0.0 |         3.4 |                  0.0 |                     0.3 |         0.0 |              15.3 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.4 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-19 |           3.2 |         4.8 |                  0.0 |                     0.4 |         2.6 |              21.6 |      0.0 |     2.6 |           2.6 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.6 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-20 |           4.6 |         6.7 |                  0.0 |                     0.3 |         4.2 |             30.15 |      0.0 |     4.2 |           4.2 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.7 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-21 |           2.3 |         5.8 |                  0.0 |                     0.6 |         2.0 |              26.1 |      0.0 |     2.0 |           2.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.4 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-22 |          18.0 |         6.5 |                  0.0 |                     0.8 |        17.4 |             29.25 |      0.0 |    17.4 |          17.4 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.2 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-23 |          19.2 |         5.0 |                  0.0 |                     0.5 |        18.4 |              22.5 |      0.0 |    18.4 |          18.4 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-24 |           0.4 |         3.0 |                  0.0 |                     0.8 |         0.0 |              13.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.1 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-25 |           8.3 |         3.1 |                  0.0 |                     0.5 |         7.4 |             13.95 |      0.0 |     7.4 |           7.4 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-26 |           5.3 |         7.1 |                  0.0 |                     0.4 |         4.8 |             31.95 |      0.0 |     4.8 |           4.8 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.6 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-27 |           0.7 |         9.4 |                  0.0 |                     1.3 |         0.3 |              42.3 |      0.0 |     0.3 |           0.3 |         0.0 |                    0.0 |             0.0 |                     1.3 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.7 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-28 |           2.7 |         4.6 |                  0.0 |                     0.9 |         1.4 |              20.7 |      0.0 |     1.4 |           1.4 |         0.0 |                    0.0 |             0.0 |                     0.9 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.1 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-01 |           1.6 |         3.7 |                  0.0 |                     0.7 |         0.7 |             16.65 |      0.0 |     0.7 |           0.7 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.3 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-02 |           2.5 |         4.7 |                  0.0 |                     0.7 |         1.8 |             21.15 |      0.0 |     1.8 |           1.8 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.3 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-03 |           0.6 |         5.9 |                  0.0 |                     1.1 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     1.1 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.8 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-04 |           0.2 |         7.7 |                  0.0 |                     1.0 |         0.0 |             34.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     1.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-05 |           1.7 |         6.3 |                  0.0 |                     0.8 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.9 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-06 |           0.3 |         3.7 |                  0.0 |                     0.6 |         0.0 |             16.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.6 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-07 |           0.0 |         1.6 |                  0.0 |                     0.6 |         0.0 |               7.2 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-08 |           1.8 |         4.0 |                  0.0 |                     0.7 |         0.0 |              18.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.1 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-09 |           8.9 |         5.6 |                  0.0 |                     0.5 |         8.0 |              25.2 |      0.0 |     8.0 |           8.0 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-10 |           0.0 |         5.8 |                  0.0 |                     0.8 |         0.0 |              26.1 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.7 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-11 |           0.0 |         5.7 |                  0.0 |                     0.7 |         0.0 |             25.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-12 |           0.0 |         4.6 |                  0.0 |                     0.0 |         0.0 |              20.7 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-13 |           0.9 |         4.2 |                  0.0 |                     0.9 |         0.0 |              18.9 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.9 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-14 |           0.1 |         7.4 |                  0.0 |                     0.1 |         0.0 |              33.3 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.1 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-15 |           0.0 |         6.3 |                  0.0 |                     0.0 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-16 |           0.0 |         8.7 |                  0.0 |                     0.0 |         0.0 |             39.15 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-17 |           3.9 |         6.4 |                  0.0 |                     1.1 |         1.9 |              28.8 |      0.0 |     1.9 |           1.9 |         0.0 |                    0.0 |             0.0 |                     1.1 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.9 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-18 |           8.7 |         5.2 |                  0.0 |                     1.1 |         7.6 |              23.4 |      0.0 |     7.6 |           7.6 |         0.0 |                    0.0 |             0.0 |                     1.1 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.9 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-19 |          26.4 |         5.1 |                  0.0 |                     0.5 |        25.3 |             22.95 |      0.0 |    25.3 |          25.3 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-20 |          11.5 |         8.7 |                  0.0 |                     0.6 |        11.0 |             39.15 |      0.0 |    11.0 |          11.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.4 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-21 |           0.9 |         6.2 |                  0.0 |                     1.5 |         0.3 |              27.9 |      0.0 |     0.3 |           0.3 |         0.0 |                    0.0 |             0.0 |                     1.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.5 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-22 |           0.0 |         5.9 |                  0.0 |                     0.5 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-23 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-24 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-25 |           0.0 |         5.9 |                  0.0 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-26 |           0.0 |         6.7 |                  0.0 |                     0.0 |         0.0 |             30.15 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-27 |           0.0 |         7.0 |                  0.0 |                     0.0 |         0.0 |              31.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-28 |           0.0 |         8.3 |                  0.0 |                     0.0 |         0.0 |             37.35 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-29 |           1.5 |         9.0 |                  0.0 |                     0.8 |         0.0 |              40.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.7 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-30 |           0.3 |        12.4 |                  0.0 |                     1.0 |         0.0 |              55.8 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     1.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-03-31 |           0.2 |        15.0 |                  0.0 |                     0.2 |         0.0 |              67.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.2 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-04-01 |           4.5 |        11.8 |                  0.0 |                     1.5 |         2.5 |              53.1 |      0.0 |     2.5 |           2.5 |         0.0 |                    0.0 |             0.0 |                     1.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.5 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-04-02 |           0.0 |         9.4 |                  0.0 |                     0.5 |         0.0 |              42.3 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-04-03 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-04-04 |           0.0 |         7.9 |                  0.0 |                     0.0 |         0.0 |             35.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-04-05 |           0.4 |         7.5 |                  0.0 |                     0.4 |         0.0 |             33.75 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-04-06 |           0.0 |         7.2 |                  0.0 |                     0.0 |         0.0 |              32.4 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-04-07 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-04-08 |           0.0 |         8.6 |                  0.0 |                     0.0 |         0.0 |              38.7 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-04-09 |           0.0 |        10.5 |                  0.0 |                     0.0 |         0.0 |             47.25 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_urban_collected_surface_runoff:

collected surface runoff
________________________

However, connected sealed zones route all their surface runoff into the cistern, which
means their complete precipitation input, besides interception evaporation losses,
becomes available for irrigation:

.. integration-test::

    >>> cisternsource(True)
    >>> conditions = test("whmod_urban_collected_surface_runoff", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | cisterninflow | cisternoverflow | cisterndemand | cisternextraction | internalirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | cisternwater | deepwater |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-11 |           0.4 |        -1.5 |                  0.0 |                     0.4 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-12 |           0.0 |        -0.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-13 |           0.0 |        -1.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-14 |           0.0 |        -1.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-15 |           0.0 |         1.7 |                  0.0 |                     0.0 |         0.0 |              7.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-16 |           0.2 |         4.4 |                  0.0 |                     0.2 |         0.0 |              19.8 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.2 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |         0.05 |       0.0 |
    | 2017-02-17 |           4.5 |         4.5 |                  0.0 |                     0.3 |         2.5 |             20.25 |      0.0 |     2.5 |           2.5 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |         0.025 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.7 |      0.0 |          0.0 |        0.075 |       0.0 |
    | 2017-02-18 |           0.0 |         3.4 |                  0.0 |                     0.3 |         0.0 |              15.3 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.4 |      0.0 |          0.0 |        0.075 |       0.0 |
    | 2017-02-19 |           3.2 |         4.8 |                  0.0 |                     0.4 |         2.6 |              21.6 |      0.0 |     2.6 |           2.6 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |         0.026 |           0.001 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.6 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-02-20 |           4.6 |         6.7 |                  0.0 |                     0.3 |         4.2 |             30.15 |      0.0 |     4.2 |           4.2 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |         0.042 |           0.042 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.7 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-02-21 |           2.3 |         5.8 |                  0.0 |                     0.6 |         2.0 |              26.1 |      0.0 |     2.0 |           2.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |          0.02 |            0.02 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.4 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-02-22 |          18.0 |         6.5 |                  0.0 |                     0.8 |        17.4 |             29.25 |      0.0 |    17.4 |          17.4 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |         0.174 |           0.174 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.2 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-02-23 |          19.2 |         5.0 |                  0.0 |                     0.5 |        18.4 |              22.5 |      0.0 |    18.4 |          18.4 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |         0.184 |           0.184 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-02-24 |           0.4 |         3.0 |                  0.0 |                     0.8 |         0.0 |              13.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.1 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-02-25 |           8.3 |         3.1 |                  0.0 |                     0.5 |         7.4 |             13.95 |      0.0 |     7.4 |           7.4 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |         0.074 |           0.074 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-02-26 |           5.3 |         7.1 |                  0.0 |                     0.4 |         4.8 |             31.95 |      0.0 |     4.8 |           4.8 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |         0.048 |           0.048 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.6 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-02-27 |           0.7 |         9.4 |                  0.0 |                     1.3 |         0.3 |              42.3 |      0.0 |     0.3 |           0.3 |         0.0 |                    0.0 |             0.0 |                     1.3 |           0.0 |                0.0 |         0.003 |           0.003 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.7 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-02-28 |           2.7 |         4.6 |                  0.0 |                     0.9 |         1.4 |              20.7 |      0.0 |     1.4 |           1.4 |         0.0 |                    0.0 |             0.0 |                     0.9 |           0.0 |                0.0 |         0.014 |           0.014 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.1 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-01 |           1.6 |         3.7 |                  0.0 |                     0.7 |         0.7 |             16.65 |      0.0 |     0.7 |           0.7 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |         0.007 |           0.007 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.3 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-02 |           2.5 |         4.7 |                  0.0 |                     0.7 |         1.8 |             21.15 |      0.0 |     1.8 |           1.8 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |         0.018 |           0.018 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.3 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-03 |           0.6 |         5.9 |                  0.0 |                     1.1 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     1.1 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.8 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-04 |           0.2 |         7.7 |                  0.0 |                     1.0 |         0.0 |             34.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     1.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-05 |           1.7 |         6.3 |                  0.0 |                     0.8 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.9 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-06 |           0.3 |         3.7 |                  0.0 |                     0.6 |         0.0 |             16.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.6 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-07 |           0.0 |         1.6 |                  0.0 |                     0.6 |         0.0 |               7.2 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-08 |           1.8 |         4.0 |                  0.0 |                     0.7 |         0.0 |              18.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.1 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-09 |           8.9 |         5.6 |                  0.0 |                     0.5 |         8.0 |              25.2 |      0.0 |     8.0 |           8.0 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |          0.08 |            0.08 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-10 |           0.0 |         5.8 |                  0.0 |                     0.8 |         0.0 |              26.1 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.7 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-11 |           0.0 |         5.7 |                  0.0 |                     0.7 |         0.0 |             25.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-12 |           0.0 |         4.6 |                  0.0 |                     0.0 |         0.0 |              20.7 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-13 |           0.9 |         4.2 |                  0.0 |                     0.9 |         0.0 |              18.9 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.9 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-14 |           0.1 |         7.4 |                  0.0 |                     0.1 |         0.0 |              33.3 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.1 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-15 |           0.0 |         6.3 |                  0.0 |                     0.0 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-16 |           0.0 |         8.7 |                  0.0 |                     0.0 |         0.0 |             39.15 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-17 |           3.9 |         6.4 |                  0.0 |                     1.1 |         1.9 |              28.8 |      0.0 |     1.9 |           1.9 |         0.0 |                    0.0 |             0.0 |                     1.1 |           0.0 |                0.0 |         0.019 |           0.019 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.9 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-18 |           8.7 |         5.2 |                  0.0 |                     1.1 |         7.6 |              23.4 |      0.0 |     7.6 |           7.6 |         0.0 |                    0.0 |             0.0 |                     1.1 |           0.0 |                0.0 |         0.076 |           0.076 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.9 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-19 |          26.4 |         5.1 |                  0.0 |                     0.5 |        25.3 |             22.95 |      0.0 |    25.3 |          25.3 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |         0.253 |           0.253 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-20 |          11.5 |         8.7 |                  0.0 |                     0.6 |        11.0 |             39.15 |      0.0 |    11.0 |          11.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |          0.11 |            0.11 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.4 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-21 |           0.9 |         6.2 |                  0.0 |                     1.5 |         0.3 |              27.9 |      0.0 |     0.3 |           0.3 |         0.0 |                    0.0 |             0.0 |                     1.5 |           0.0 |                0.0 |         0.003 |           0.003 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.5 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-22 |           0.0 |         5.9 |                  0.0 |                     0.5 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-23 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-24 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-25 |           0.0 |         5.9 |                  0.0 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-26 |           0.0 |         6.7 |                  0.0 |                     0.0 |         0.0 |             30.15 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-27 |           0.0 |         7.0 |                  0.0 |                     0.0 |         0.0 |              31.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-28 |           0.0 |         8.3 |                  0.0 |                     0.0 |         0.0 |             37.35 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-29 |           1.5 |         9.0 |                  0.0 |                     0.8 |         0.0 |              40.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.7 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-30 |           0.3 |        12.4 |                  0.0 |                     1.0 |         0.0 |              55.8 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     1.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-03-31 |           0.2 |        15.0 |                  0.0 |                     0.2 |         0.0 |              67.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.2 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-04-01 |           4.5 |        11.8 |                  0.0 |                     1.5 |         2.5 |              53.1 |      0.0 |     2.5 |           2.5 |         0.0 |                    0.0 |             0.0 |                     1.5 |           0.0 |                0.0 |         0.025 |           0.025 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.5 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-04-02 |           0.0 |         9.4 |                  0.0 |                     0.5 |         0.0 |              42.3 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-04-03 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-04-04 |           0.0 |         7.9 |                  0.0 |                     0.0 |         0.0 |             35.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-04-05 |           0.4 |         7.5 |                  0.0 |                     0.4 |         0.0 |             33.75 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-04-06 |           0.0 |         7.2 |                  0.0 |                     0.0 |         0.0 |              32.4 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-04-07 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-04-08 |           0.0 |         8.6 |                  0.0 |                     0.0 |         0.0 |              38.7 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |
    | 2017-04-09 |           0.0 |        10.5 |                  0.0 |                     0.0 |         0.0 |             47.25 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |           0.0 |             0.0 |           0.0 |               0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |          0.1 |       0.0 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_urban_multiple_zones:

multiple zones
______________

This example modifies the :ref:`whmod_rural_multiple_zones` example of |whmod_rural| by
connecting the first (grassland) and last zone (sealed surface) to the cistern:

>>> name2value = {par.name: par.value for par in control}
>>> nmbzones(4)
>>> landtype(GRASS, CONIFER, WATER, SEALED)
>>> soiltype(SAND, SAND, NONE, NONE)
>>> cisternsource(True, False, False, True)
>>> zonearea(2.5)
>>> interceptioncapacity.conifer = 2.0
>>> landmonthfactor.conifer = 1.335

>>> for name, value in name2value.items():
...     if name not in ("nmbzones", "landtype", "soiltype", "cisternsource", "zonearea", "interceptioncapacity"):
...         control[name].value = value
>>> model.update_parameters()
>>> aetmodel.parameters.control.dissefactor(4.0)
>>> petmodel.parameters.control.dampingfactor(1.0)
>>> retmodel.parameters.control.evapotranspirationfactor(1.0)

The second zone (conifer forest) benefits from irrigation by a faster soil moisture
increase.  In contrast, the first zone's soil water content increases slightly slower
than in the original example.  This behaviour is due to the assumption of an
impermeable layer, which not only drains percolated water into the cistern but also
prevents any capillary rise:

.. integration-test::

    >>> conditions = test("whmod_urban_multiple_zones", get_conditions="2017-02-10")
    |       date | precipitation | temperature |                          relativesoilmoisture |                   interceptionevaporation |                     throughfall |                    potentialsnowmelt |                snowmelt |                     ponding |                surfacerunoff |                            percolation |                          soilevapotranspiration |                   lakeevaporation |                             totalevapotranspiration |                     capillaryrise |                            requiredirrigation | cisterninflow | cisternoverflow | cisterndemand | cisternextraction |                           internalirrigation |                externalirrigation |                          potentialrecharge |                         baseflow | actualrecharge | delayedrecharge |                   interceptedwater |                snowpack |                            soilmoisture | cisternwater | deepwater |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 | 0.102752  0.134748  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.7302              0.0 |      0.0       0.0  0.7302                      0.0 | 0.0      3.75  0.0            0.0 |      70.0      66.25  0.0                 0.0 |           0.0 |             0.0 |      0.340625 |              0.05 | 10.275229  9.724771  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0      -3.75  -0.7302                0.0 | 0.0       0.0      0.0       0.0 |       -1.12005 |       -0.203031 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 10.275229  13.474771  0.0           0.0 |          0.0 | -0.917019 |
    | 2017-02-11 |           0.4 |        -1.5 | 0.101923   0.15724  0.0                   0.0 | 0.4     0.4  0.0                      0.4 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |  0.08289  0.179951  0.0                     0.0 | 0.0  0.0  0.9736              0.0 |  0.48289  0.579951  0.9736                      0.4 | 0.0  2.429179  0.0            0.0 |  59.80766  54.276001  0.0                 0.0 |           0.0 |             0.0 |      0.285209 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -2.429179  -0.5736                0.0 | 0.0       0.0      0.0       0.0 |      -0.750695 |       -0.302305 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  10.19234  15.723999  0.0           0.0 |          0.0 | -1.365409 |
    | 2017-02-12 |           0.0 |        -0.9 | 0.100484  0.176771  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.143914  0.291527  0.0                     0.0 | 0.0  0.0  0.8519              0.0 | 0.143914  0.291527  0.8519                      0.0 | 0.0  2.244621  0.0            0.0 | 59.951575  52.322908  0.0                 0.0 |           0.0 |             0.0 |      0.280686 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -2.244621  -0.8519                0.0 | 0.0       0.0      0.0       0.0 |       -0.77413 |       -0.387833 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 10.048425  17.677092  0.0           0.0 |          0.0 | -1.751706 |
    | 2017-02-13 |           0.0 |        -1.6 | 0.099673  0.195834  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.081102   0.18586  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.081102   0.18586  0.4868                      0.0 | 0.0  2.092153  0.0            0.0 | 60.032677  50.416615  0.0                 0.0 |           0.0 |             0.0 |      0.276123 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -2.092153  -0.4868                0.0 | 0.0       0.0      0.0       0.0 |      -0.644738 |       -0.434402 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  9.967323  19.583385  0.0           0.0 |          0.0 | -1.962043 |
    | 2017-02-14 |           0.0 |        -1.3 | 0.098869  0.213293  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.080463  0.204208  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.080463  0.204208  0.4868                      0.0 | 0.0  1.950153  0.0            0.0 |  60.11314   48.67067  0.0                 0.0 |           0.0 |             0.0 |       0.27196 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -1.950153  -0.4868                0.0 | 0.0       0.0      0.0       0.0 |      -0.609238 |       -0.466094 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |   9.88686   21.32933  0.0           0.0 |          0.0 | -2.105187 |
    | 2017-02-15 |           0.0 |         1.7 |  0.09807  0.229346  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  7.65   7.65  0.0               7.65 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.079827  0.220582  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.079827  0.220582  0.4868                      0.0 | 0.0   1.82587  0.0            0.0 | 60.192967  47.065383  0.0                 0.0 |           0.0 |             0.0 |      0.268146 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   -1.82587  -0.4868                0.0 | 0.0       0.0      0.0       0.0 |      -0.578167 |        -0.48641 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  9.807033  22.934617  0.0           0.0 |          0.0 | -2.196945 |
    | 2017-02-16 |           0.2 |         4.4 | 0.097674  0.245038  0.0                   0.0 | 0.2     0.2  0.0                      0.2 |  0.0      0.0  0.0          0.0 |  19.8   19.8  0.0               19.8 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.039598  0.147143  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.239598  0.347143  0.4868                      0.2 | 0.0  1.716364  0.0            0.0 | 60.232566  45.496162  0.0                 0.0 |           0.0 |             0.0 |      0.264322 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -1.716364  -0.2868                0.0 | 0.0       0.0      0.0       0.0 |      -0.500791 |       -0.489017 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  9.767434  24.503838  0.0           0.0 |          0.0 | -2.208719 |
    | 2017-02-17 |           4.5 |         4.5 | 0.153005  0.296112  0.0                   0.0 | 0.3  0.4005  0.0                      0.3 |  4.1      2.5  0.0          2.5 | 20.25  20.25  0.0              20.25 | 0.0  0.0  0.0       0.0 |  4.1      2.5  0.0      2.5 | 0.0  0.0  0.0            2.5 |  0.011939   0.073252  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 |      0.3    0.4005  0.3651                      0.3 | 0.0  1.613637  0.0            0.0 | 56.144505  41.455777  0.0                 0.0 |       0.00628 |             0.0 |      0.244001 |           0.00628 |  1.444991  1.066948  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -1.540385   4.1349                0.0 | 0.0       0.0  0.82698       0.0 |       0.441884 |       -0.320273 | 0.1  1.5995  0.0               1.7 | 0.0  0.0  0.0       0.0 | 15.300487  29.611171  0.0           0.0 |          0.0 | -1.446563 |
    | 2017-02-18 |           0.0 |         3.4 | 0.152397   0.30919  0.0                   0.0 | 0.1  0.4005  0.0                      0.3 |  0.0      0.0  0.0          0.0 |  15.3   15.3  0.0               15.3 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.060805       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 | 0.160805    0.4005  0.3651                      0.3 | 0.0  1.307803  0.0            0.0 | 54.760318  39.081026  0.0                 0.0 |           0.0 |             0.0 |      0.234603 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -1.307803  -0.3651                0.0 | 0.0       0.0      0.0       0.0 |      -0.418226 |       -0.338029 | 0.0   1.199  0.0               1.4 | 0.0  0.0  0.0       0.0 | 15.239682  30.918974  0.0           0.0 |          0.0 |  -1.52676 |
    | 2017-02-19 |           3.2 |         4.8 | 0.195733  0.354946  0.0                   0.0 | 0.4   0.534  0.0                      0.4 |  2.8    2.399  0.0          2.6 |  21.6   21.6  0.0               21.6 | 0.0  0.0  0.0       0.0 |  2.8    2.399  0.0      2.6 | 0.0  0.0  0.0            2.6 |  0.024906   0.126012  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.4868              0.0 |      0.4     0.534  0.4868                      0.4 | 0.0  1.236254  0.0            0.0 | 51.985223  35.571784  0.0                 0.0 |      0.006562 |             0.0 |      0.218893 |          0.006562 |  1.558485   1.06642  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -1.110241   2.7132                0.0 | 0.0       0.0  0.54264       0.0 |        0.26508 |       -0.228704 | 0.0   1.466  0.0               1.6 | 0.0  0.0  0.0       0.0 | 19.573262  35.494636  0.0           0.0 |          0.0 | -1.032976 |
    | 2017-02-20 |           4.6 |         6.7 | 0.263033  0.419351  0.0                   0.0 | 0.3  0.4005  0.0                      0.3 |  4.2    4.066  0.0          4.2 | 30.15  30.15  0.0              30.15 | 0.0  0.0  0.0       0.0 |  4.2    4.066  0.0      4.2 | 0.0  0.0  0.0            4.2 |  0.070018   0.301997  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 |      0.3    0.4005  0.3651                      0.3 | 0.0  1.006512  0.0            0.0 | 46.296756   29.73485  0.0                 0.0 |      0.010675 |             0.0 |      0.190079 |          0.010675 |  2.600076  1.669942  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.704514   4.2349                0.0 | 0.0       0.0  0.84698       0.0 |       0.670851 |       -0.065642 | 0.1  1.5995  0.0               1.7 | 0.0  0.0  0.0       0.0 |  26.30332  41.935092  0.0           0.0 |          0.0 | -0.296483 |
    | 2017-02-21 |           2.3 |         5.8 | 0.294178  0.451406  0.0                   0.0 | 0.4   0.801  0.0                      0.6 |  2.0   1.8995  0.0          2.0 |  26.1   26.1  0.0               26.1 | 0.0  0.0  0.0       0.0 |  2.0   1.8995  0.0      2.0 | 0.0  0.0  0.0            2.0 |   0.07001   0.214412  0.0          0.0 | 0.099166       0.0  0.0                     0.0 | 0.0  0.0  0.7302              0.0 | 0.499166     0.801  0.7302                      0.6 | 0.0  0.734129  0.0            0.0 | 41.865856  25.645691  0.0                 0.0 |      0.005175 |             0.0 |      0.168779 |          0.005175 |  1.283673  0.786337  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.519718   1.5698                0.0 | 0.0       0.0  0.31396       0.0 |       0.184031 |       -0.020384 | 0.0   1.199  0.0               1.4 | 0.0  0.0  0.0       0.0 | 29.417817  45.140646  0.0           0.0 |          0.0 | -0.092068 |
    | 2017-02-22 |          18.0 |         6.5 | 0.642002  0.606231  0.0                   0.0 | 0.4   1.068  0.0                      0.8 | 17.6   17.199  0.0         17.4 | 29.25  29.25  0.0              29.25 | 0.0  0.0  0.0       0.0 | 17.6   17.199  0.0     17.4 | 0.0  0.0  0.0           17.4 |  0.815904   2.335676  0.0          0.0 | 0.217577       0.0  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.617577     1.068  0.9736                      0.8 | 0.0  0.619132  0.0            0.0 | 24.015665        0.0  0.0                 0.0 |       0.04554 |             0.0 |      0.060039 |           0.04554 | 18.215904       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   1.716544  17.0264                0.0 | 0.0  0.343309  3.40528       0.0 |       3.748589 |        0.662815 | 0.0   0.932  0.0               1.2 | 0.0  0.0  0.0       0.0 |  64.20024  60.623102  0.0           0.0 |          0.0 |  2.993706 |
    | 2017-02-23 |          19.2 |         5.0 | 0.767307  0.738219  0.0                   0.0 | 0.4  0.6675  0.0                      0.5 | 18.8   18.132  0.0         18.4 |  22.5   22.5  0.0               22.5 | 0.0  0.0  0.0       0.0 | 18.8   18.132  0.0     18.4 | 0.0  0.0  0.0           18.4 |  6.180788   5.162172  0.0          0.0 | 0.088775       0.0  0.0                     0.0 | 0.0  0.0  0.6085              0.0 | 0.488775    0.6675  0.6085                      0.5 | 0.0  0.228958  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.061452 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   4.933214  18.5915                0.0 | 0.0  0.986643   3.7183       0.0 |       4.704943 |        1.395528 | 0.0  1.3325  0.0               1.5 | 0.0  0.0  0.0       0.0 | 76.730677  73.821888  0.0           0.0 |     0.061452 |  6.303121 |
    | 2017-02-24 |           0.4 |         3.0 |  0.76353  0.738892  0.0                   0.0 | 0.4   1.068  0.0                      0.8 |  0.0      0.0  0.0          0.0 |  13.5   13.5  0.0               13.5 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.377706       0.0  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.777706     1.068  0.9736                      0.8 | 0.0  0.067274  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.067274  -0.5736                0.0 | 0.0       0.0      0.0       0.0 |      -0.160218 |        1.113519 | 0.0  0.6645  0.0               1.1 | 0.0  0.0  0.0       0.0 | 76.352971  73.889162  0.0           0.0 |     0.061452 |  5.029383 |
    | 2017-02-25 |           8.3 |         3.1 | 0.801454   0.77662  0.0                   0.0 | 0.4  0.6675  0.0                      0.5 |  7.9   6.9645  0.0          7.4 | 13.95  13.95  0.0              13.95 | 0.0  0.0  0.0       0.0 |  7.9   6.9645  0.0      7.4 | 0.0  0.0  0.0            7.4 |  4.013297   3.258415  0.0          0.0 |  0.09429       0.0  0.0                     0.0 | 0.0  0.0  0.6085              0.0 |  0.49429    0.6675  0.6085                      0.5 | 0.0  0.066757  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.028533 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   3.191659   7.6915                0.0 | 0.0  0.638332   1.5383       0.0 |       2.176632 |        1.306229 | 0.0  1.3325  0.0               1.5 | 0.0  0.0  0.0       0.0 | 80.145383  77.662003  0.0           0.0 |     0.089985 |  5.899786 |
    | 2017-02-26 |           5.3 |         7.1 |  0.82234  0.798803  0.0                   0.0 | 0.4   0.534  0.0                      0.4 |  4.9   4.6325  0.0          4.8 | 31.95  31.95  0.0              31.95 | 0.0  0.0  0.0       0.0 |  4.9   4.6325  0.0      4.8 | 0.0  0.0  0.0            4.8 |  2.811355    2.45596  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.4868              0.0 |      0.4     0.534  0.4868                      0.4 | 0.0  0.041799  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.019028 |        0.009014 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   2.414161   4.8132                0.0 | 0.0  0.482832  0.96264       0.0 |       1.445472 |        1.331469 | 0.0   1.466  0.0               1.6 | 0.0  0.0  0.0       0.0 | 82.234028  79.880342  0.0           0.0 |          0.1 |  6.013789 |
    | 2017-02-27 |           0.7 |         9.4 | 0.814845  0.799824  0.0                   0.0 | 0.4  1.7355  0.0                      1.3 |  0.3    0.166  0.0          0.3 |  42.3   42.3  0.0               42.3 | 0.0  0.0  0.0       0.0 |  0.3    0.166  0.0      0.3 | 0.0  0.0  0.0            0.3 |  0.183606   0.094453  0.0          0.0 |  0.86589       0.0  0.0                     0.0 | 0.0  0.0  1.5821              0.0 |  1.26589    1.7355  1.5821                      1.3 | 0.0  0.030542  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.001209 |        0.001209 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   0.063911  -0.8821                0.0 | 0.0  0.012782      0.0       0.0 |      -0.207743 |        1.052458 | 0.0  0.2645  0.0               0.7 | 0.0  0.0  0.0       0.0 | 81.484532  79.982431  0.0           0.0 |          0.1 |  4.753588 |
    | 2017-02-28 |           2.7 |         4.6 | 0.819289  0.804265  0.0                   0.0 | 0.4  1.2015  0.0                      0.9 |  2.3   0.9645  0.0          1.4 |  20.7   20.7  0.0               20.7 | 0.0  0.0  0.0       0.0 |  2.3   0.9645  0.0      1.4 | 0.0  0.0  0.0            1.4 |  1.375663   0.550558  0.0          0.0 | 0.479939       0.0  0.0                     0.0 | 0.0  0.0  1.0953              0.0 | 0.879939    1.2015  1.0953                      0.9 | 0.0  0.030079  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.006939 |        0.006939 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   0.520479   1.6047                0.0 | 0.0  0.104096  0.32094       0.0 |       0.425036 |        0.938725 | 0.0  0.7985  0.0               1.1 | 0.0  0.0  0.0       0.0 | 81.928929  80.426452  0.0           0.0 |          0.1 |  4.239899 |
    | 2017-03-01 |           1.6 |         3.7 | 0.822265  0.806224  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.0   0.3985  0.0          0.7 | 16.65  16.65  0.0              16.65 | 0.0  0.0  0.0       0.0 |  1.0   0.3985  0.0      0.7 | 0.0  0.0  0.0            0.7 |  0.606336   0.230656  0.0          0.0 |  0.09612       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 |  0.69612    0.9345  0.8792                      0.7 | 0.0  0.028122  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.003266 |        0.003266 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   0.202534   0.7208                0.0 | 0.0  0.040507  0.14416       0.0 |       0.184667 |        0.802038 | 0.0  1.0655  0.0               1.3 | 0.0  0.0  0.0       0.0 | 82.226473  80.622418  0.0           0.0 |          0.1 |  3.622528 |
    | 2017-03-02 |           2.5 |         4.7 | 0.828677  0.813035  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.9   1.5655  0.0          1.8 | 21.15  21.15  0.0              21.15 | 0.0  0.0  0.0       0.0 |  1.9   1.5655  0.0      1.8 | 0.0  0.0  0.0            1.8 |   1.16257   0.911679  0.0          0.0 | 0.096208       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.696208    0.9345  0.8792                      0.7 | 0.0  0.027285  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.007406 |        0.007406 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   0.884394   1.6208                0.0 | 0.0  0.176879  0.32416       0.0 |       0.501039 |        0.747476 | 0.0  1.0655  0.0               1.3 | 0.0  0.0  0.0       0.0 | 82.867695  81.303524  0.0           0.0 |          0.1 |  3.376091 |
    | 2017-03-03 |           0.6 |         5.9 | 0.823857   0.81328  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  0.0      0.0  0.0          0.0 | 26.55  26.55  0.0              26.55 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.481966       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 1.081966    1.4685  1.3816                      1.1 | 0.0  0.024508  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.024508  -0.7816                0.0 | 0.0       0.0      0.0       0.0 |      -0.201527 |        0.575451 | 0.0   0.197  0.0               0.8 | 0.0  0.0  0.0       0.0 | 82.385729  81.328032  0.0           0.0 |          0.1 |  2.599113 |
    | 2017-03-04 |           0.2 |         7.7 | 0.816157  0.804525  0.0                   0.0 | 0.2   0.397  0.0                      1.0 |  0.0      0.0  0.0          0.0 | 34.65  34.65  0.0              34.65 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.770034  0.899923  0.0                     0.0 | 0.0  0.0   1.256              0.0 | 0.970034  1.296923   1.256                      1.0 | 0.0  0.024412  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.024412   -1.056                0.0 | 0.0       0.0      0.0       0.0 |      -0.270103 |        0.422178 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 81.615695   80.45252  0.0           0.0 |          0.1 |  1.906832 |
    | 2017-03-05 |           1.7 |         6.3 | 0.818631  0.804805  0.0                   0.0 | 0.6   1.068  0.0                      0.8 |  1.1      0.0  0.0          0.0 | 28.35  28.35  0.0              28.35 | 0.0  0.0  0.0       0.0 |  1.1      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |  0.660588        0.0  0.0          0.0 | 0.192054       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.792054     1.068  1.0048                      0.8 | 0.0  0.028009  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.001651 |        0.001651 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.028009   0.6952                0.0 | 0.0       0.0  0.13904       0.0 |       0.132038 |        0.369584 | 0.0   0.632  0.0               0.9 | 0.0  0.0  0.0       0.0 | 81.863053   80.48053  0.0           0.0 |          0.1 |  1.669285 |
    | 2017-03-06 |           0.3 |         3.7 | 0.815748  0.805084  0.0                   0.0 | 0.3   0.801  0.0                      0.6 |  0.0      0.0  0.0          0.0 | 16.65  16.65  0.0              16.65 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.288302       0.0  0.0                     0.0 | 0.0  0.0  0.7536              0.0 | 0.588302     0.801  0.7536                      0.6 | 0.0  0.027889  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.027889  -0.4536                0.0 | 0.0       0.0      0.0       0.0 |      -0.120372 |         0.28077 | 0.0   0.131  0.0               0.6 | 0.0  0.0  0.0       0.0 |  81.57475  80.508419  0.0           0.0 |          0.1 |  1.268143 |
    | 2017-03-07 |           0.0 |         1.6 | 0.809026  0.797673  0.0                   0.0 | 0.0   0.131  0.0                      0.6 |  0.0      0.0  0.0          0.0 |   7.2    7.2  0.0                7.2 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.672104  0.768862  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.672104  0.899862  0.8792                      0.6 | 0.0   0.02777  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   -0.02777  -0.8792                0.0 | 0.0       0.0      0.0       0.0 |      -0.226742 |        0.188774 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 80.902646  79.767327  0.0           0.0 |          0.1 |  0.852627 |
    | 2017-03-08 |           1.8 |         4.0 | 0.813019  0.797984  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.2      0.0  0.0          0.0 |  18.0   18.0  0.0               18.0 | 0.0  0.0  0.0       0.0 |  1.2      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |  0.704941        0.0  0.0          0.0 | 0.095811       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.695811    0.9345  0.8792                      0.7 | 0.0  0.031059  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.001762 |        0.001762 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.031059   0.9208                0.0 | 0.0       0.0  0.18416       0.0 |       0.176395 |         0.18653 | 0.0  0.8655  0.0               1.1 | 0.0  0.0  0.0       0.0 | 81.301894  79.798386  0.0           0.0 |          0.1 |  0.842492 |
    | 2017-03-09 |           8.9 |         5.6 | 0.846654  0.831876  0.0                   0.0 | 0.5  0.6675  0.0                      0.5 |  8.3   7.7655  0.0          8.0 |  25.2   25.2  0.0               25.2 | 0.0  0.0  0.0       0.0 |  8.3   7.7655  0.0      8.0 | 0.0  0.0  0.0            8.0 |  4.936467    4.40716  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0   0.628              0.0 |      0.5    0.6675   0.628                      0.5 | 0.0  0.030916  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.032341 |        0.032341 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   4.376244    8.272                0.0 | 0.0  0.875249   1.6544       0.0 |       2.529649 |        0.611265 | 0.1  1.3325  0.0               1.5 | 0.0  0.0  0.0       0.0 | 84.665426  83.187642  0.0           0.0 |          0.1 |  2.760875 |
    | 2017-03-10 |           0.0 |         5.8 | 0.839872  0.832055  0.0                   0.0 | 0.1   1.068  0.0                      0.8 |  0.0      0.0  0.0          0.0 |  26.1   26.1  0.0               26.1 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.678229       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.778229     1.068  1.0048                      0.8 | 0.0   0.01782  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   -0.01782  -1.0048                0.0 | 0.0       0.0      0.0       0.0 |      -0.255655 |        0.454119 | 0.0  0.2645  0.0               0.7 | 0.0  0.0  0.0       0.0 | 83.987198  83.205463  0.0           0.0 |          0.1 |  2.051101 |
    | 2017-03-11 |           0.0 |         5.7 | 0.830201  0.821903  0.0                   0.0 | 0.0  0.2645  0.0                      0.7 |  0.0      0.0  0.0          0.0 | 25.65  25.65  0.0              25.65 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.967063  1.032915  0.0                     0.0 | 0.0  0.0   1.256              0.0 | 0.967063  1.297415   1.256                      0.7 | 0.0  0.017764  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.017764   -1.256                0.0 | 0.0       0.0      0.0       0.0 |      -0.318441 |        0.314078 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 83.020134  82.190312  0.0           0.0 |          0.1 |  1.418582 |
    | 2017-03-12 |           0.0 |         4.6 | 0.818629  0.806704  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  20.7   20.7  0.0               20.7 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.157239   1.54108  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 1.157239   1.54108  1.5072                      0.0 | 0.0  0.021184  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.021184  -1.5072                0.0 | 0.0       0.0      0.0       0.0 |      -0.382096 |        0.187883 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 81.862895  80.670415  0.0           0.0 |          0.1 |  0.848603 |
    | 2017-03-13 |           0.9 |         4.2 | 0.816931  0.804088  0.0                   0.0 | 0.6     0.9  0.0                      0.9 |  0.3      0.0  0.0          0.0 |  18.9   18.9  0.0               18.9 | 0.0  0.0  0.0       0.0 |  0.3      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |  0.181533        0.0  0.0          0.0 | 0.288302  0.288654  0.0                     0.0 | 0.0  0.0  1.1304              0.0 | 0.888302  1.188654  1.1304                      0.9 | 0.0  0.027083  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.000454 |        0.000454 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.027083  -0.2304                0.0 | 0.0       0.0      0.0       0.0 |      -0.064371 |        0.142157 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  81.69306  80.408844  0.0           0.0 |          0.1 |  0.642075 |
    | 2017-03-14 |           0.1 |         7.4 | 0.809247  0.793834  0.0                   0.0 | 0.1     0.1  0.0                      0.1 |  0.0      0.0  0.0          0.0 |  33.3   33.3  0.0               33.3 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.768402  1.053673  0.0                     0.0 | 0.0  0.0  1.1304              0.0 | 0.868402  1.153673  1.1304                      0.1 | 0.0  0.028198  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.028198  -1.0304                0.0 | 0.0       0.0      0.0       0.0 |      -0.264649 |        0.068416 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 80.924658  79.383369  0.0           0.0 |          0.1 |   0.30901 |
    | 2017-03-15 |           0.0 |         6.3 | 0.797748   0.77889  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 28.35  28.35  0.0              28.35 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.149814  1.527205  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 1.149814  1.527205  1.5072                      0.0 | 0.0  0.032861  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.032861  -1.5072                0.0 | 0.0       0.0      0.0       0.0 |      -0.385015 |       -0.013777 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 79.774844  77.889025  0.0           0.0 |          0.1 | -0.062228 |
    | 2017-03-16 |           0.0 |         8.7 | 0.784384  0.761571  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 39.15  39.15  0.0              39.15 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.336403  1.772416  0.0                     0.0 | 0.0  0.0  1.7584              0.0 | 1.336403  1.772416  1.7584                      0.0 | 0.0  0.040537  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.040537  -1.7584                0.0 | 0.0       0.0      0.0       0.0 |      -0.449734 |       -0.092803 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 78.438441  76.157147  0.0           0.0 |          0.1 | -0.419159 |
    | 2017-03-17 |           3.9 |         6.4 | 0.794696   0.77149  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  3.3      1.9  0.0          1.9 |  28.8   28.8  0.0               28.8 | 0.0  0.0  0.0       0.0 |  3.3      1.9  0.0      1.9 | 0.0  0.0  0.0            1.9 |  1.793761   0.959021  0.0          0.0 | 0.475094       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 1.075094    1.4685  1.3816                      1.1 | 0.0  0.050828  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.009234 |        0.009234 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   0.908193   2.5184                0.0 | 0.0  0.181639  0.50368       0.0 |       0.685319 |        0.048247 | 0.0  0.5315  0.0               0.9 | 0.0  0.0  0.0       0.0 | 79.469585  77.148953  0.0           0.0 |          0.1 |  0.217913 |
    | 2017-03-18 |           8.7 |         5.2 | 0.825432  0.806546  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  8.1   7.2315  0.0          7.6 |  23.4   23.4  0.0               23.4 | 0.0  0.0  0.0       0.0 |  8.1   7.2315  0.0      7.6 | 0.0  0.0  0.0            7.6 |  4.549601   3.770584  0.0          0.0 | 0.476795       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 1.076795    1.4685  1.3816                      1.1 | 0.0  0.044746  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.030374 |        0.030374 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   3.725839   7.3184                0.0 | 0.0  0.745168  1.46368       0.0 |       2.208848 |        0.439897 | 0.0  0.5315  0.0               0.9 | 0.0  0.0  0.0       0.0 | 82.543189  80.654615  0.0           0.0 |          0.1 |  1.986864 |
    | 2017-03-19 |          26.4 |         5.1 | 0.924036  0.910797  0.0                   0.0 | 0.5  0.6675  0.0                      0.5 | 25.8  24.9315  0.0         25.3 | 22.95  22.95  0.0              22.95 | 0.0  0.0  0.0       0.0 | 25.8  24.9315  0.0     25.3 | 0.0  0.0  0.0           25.3 | 15.939555  14.533584  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0   0.628              0.0 |      0.5    0.6675   0.628                      0.5 | 0.0   0.02715  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.103099 |        0.103099 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  14.506434   25.772                0.0 | 0.0  2.901287   5.1544       0.0 |       8.055687 |        1.820406 | 0.1  1.3325  0.0               1.5 | 0.0  0.0  0.0       0.0 | 92.403633  91.079681  0.0           0.0 |          0.1 |  8.222145 |
    | 2017-03-20 |          11.5 |         8.7 | 0.943824   0.93347  0.0                   0.0 | 0.6   0.801  0.0                      0.6 | 11.0  10.8325  0.0         11.0 | 39.15  39.15  0.0              39.15 | 0.0  0.0  0.0       0.0 | 11.0  10.8325  0.0     11.0 | 0.0  0.0  0.0           11.0 |  9.021249   8.567816  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.7536              0.0 |      0.6     0.801  0.7536                      0.6 | 0.0  0.002662  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.050053 |        0.050053 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   8.565154  10.7464                0.0 | 0.0  1.713031  2.14928       0.0 |       3.862311 |         2.19054 | 0.0   1.199  0.0               1.4 | 0.0  0.0  0.0       0.0 | 94.382385  93.347027  0.0           0.0 |          0.1 |  9.893915 |
    | 2017-03-21 |           0.9 |         6.2 | 0.935313  0.933614  0.0                   0.0 | 0.6     2.0  0.0                      1.5 |  0.3    0.099  0.0          0.3 |  27.9   27.9  0.0               27.9 | 0.0  0.0  0.0       0.0 |  0.3    0.099  0.0      0.3 | 0.0  0.0  0.0            0.3 |  0.259474   0.083288  0.0          0.0 | 0.891578  0.002472  0.0                     0.0 | 0.0  0.0   1.884              0.0 | 1.491578  2.002472   1.884                      1.5 | 0.0  0.001104  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.001399 |        0.001399 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   0.082184   -0.984                0.0 | 0.0  0.016437      0.0       0.0 |      -0.229563 |         1.75185 | 0.0     0.0  0.0               0.5 | 0.0  0.0  0.0       0.0 | 93.531333  93.361371  0.0           0.0 |          0.1 |  7.912502 |
    | 2017-03-22 |           0.0 |         5.9 | 0.915533  0.907226  0.0                   0.0 | 0.0     0.0  0.0                      0.5 |  0.0      0.0  0.0          0.0 | 26.55  26.55  0.0              26.55 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.978082  2.639869  0.0                     0.0 | 0.0  0.0   2.512              0.0 | 1.978082  2.639869   2.512                      0.5 | 0.0  0.001097  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.001097   -2.512                0.0 | 0.0       0.0      0.0       0.0 |      -0.628274 |        1.320407 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  91.55325  90.722599  0.0           0.0 |          0.1 |  5.963822 |
    | 2017-03-23 |           0.0 |         5.2 | 0.899771  0.886251  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  23.4   23.4  0.0               23.4 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |  1.57618  2.100485  0.0                     0.0 | 0.0  0.0  2.0096              0.0 |  1.57618  2.100485  2.0096                      0.0 | 0.0  0.002994  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.002994  -2.0096                0.0 | 0.0       0.0      0.0       0.0 |      -0.503149 |        0.989852 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  89.97707  88.625109  0.0           0.0 |          0.1 |  4.470821 |
    | 2017-03-24 |           0.0 |         5.2 | 0.884062  0.865401  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  23.4   23.4  0.0               23.4 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.570823  2.090565  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.570823  2.090565  2.0096                      0.0 | 0.0  0.005519  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.005519  -2.0096                0.0 | 0.0       0.0      0.0       0.0 |       -0.50378 |        0.719103 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 88.406247  86.540062  0.0           0.0 |          0.1 |  3.247939 |
    | 2017-03-25 |           0.0 |         5.9 | 0.872324  0.849893  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 26.55  26.55  0.0              26.55 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.173868  1.559923  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 1.173868  1.559923  1.5072                      0.0 | 0.0  0.009145  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.009145  -1.5072                0.0 | 0.0       0.0      0.0       0.0 |      -0.379086 |        0.520035 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 87.232379  84.989284  0.0           0.0 |          0.1 |  2.348818 |
    | 2017-03-26 |           0.0 |         6.7 | 0.859643   0.83319  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 30.15  30.15  0.0              30.15 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.268067  1.683015  0.0                     0.0 | 0.0  0.0  1.6328              0.0 | 1.268067  1.683015  1.6328                      0.0 | 0.0  0.012683  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.012683  -1.6328                0.0 | 0.0       0.0      0.0       0.0 |      -0.411371 |          0.3512 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 85.964312  83.318953  0.0           0.0 |          0.1 |  1.586247 |
    | 2017-03-27 |           0.0 |         7.0 | 0.844087  0.812747  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  31.5   31.5  0.0               31.5 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.555656  2.061688  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.555656  2.061688  2.0096                      0.0 | 0.0  0.017406  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.017406  -2.0096                0.0 | 0.0       0.0      0.0       0.0 |      -0.506752 |        0.195679 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 84.408656  81.274671  0.0           0.0 |          0.1 |  0.883816 |
    | 2017-03-28 |           0.0 |         8.3 | 0.825691  0.788662  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 37.35  37.35  0.0              37.35 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.839597  2.433125  0.0                     0.0 | 0.0  0.0  2.3864              0.0 | 1.839597  2.433125  2.3864                      0.0 | 0.0  0.024622  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.024622  -2.3864                0.0 | 0.0       0.0      0.0       0.0 |      -0.602755 |        0.050948 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 82.569059  78.866168  0.0           0.0 |          0.1 |  0.230113 |
    | 2017-03-29 |           1.5 |         9.0 |   0.8272  0.789016  0.0                   0.0 | 0.6   1.068  0.0                      0.8 |  0.9      0.0  0.0          0.0 |  40.5   40.5  0.0               40.5 | 0.0  0.0  0.0       0.0 |  0.9      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |  0.556469        0.0  0.0          0.0 | 0.192615       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.792615     1.068  1.0048                      0.8 | 0.0  0.035397  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.001391 |        0.001391 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.035397   0.4952                0.0 | 0.0       0.0  0.09904       0.0 |       0.090191 |        0.058061 | 0.0   0.432  0.0               0.7 | 0.0  0.0  0.0       0.0 | 82.719975  78.901564  0.0           0.0 |          0.1 |  0.262243 |
    | 2017-03-30 |           0.3 |        12.4 | 0.815638  0.777276  0.0                   0.0 | 0.3   0.732  0.0                      1.0 |  0.0      0.0  0.0          0.0 |  55.8   55.8  0.0               55.8 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |  1.15621  1.209177  0.0                     0.0 | 0.0  0.0   1.884              0.0 |  1.45621  1.941177   1.884                      1.0 | 0.0  0.035219  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.035219   -1.584                0.0 | 0.0       0.0      0.0       0.0 |      -0.404805 |       -0.025842 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 81.563765  77.727607  0.0           0.0 |          0.1 |  -0.11672 |
    | 2017-03-31 |           0.2 |        15.0 | 0.791635  0.745424  0.0                   0.0 | 0.2     0.2  0.0                      0.2 |  0.0      0.0  0.0          0.0 |  67.5   67.5  0.0               67.5 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 2.400291  3.226675  0.0                     0.0 | 0.0  0.0  3.3912              0.0 | 2.600291  3.426675  3.3912                      0.2 | 0.0  0.041432  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.041432  -3.1912                0.0 | 0.0       0.0      0.0       0.0 |      -0.808158 |       -0.167652 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 79.163475  74.542364  0.0           0.0 |          0.1 | -0.757226 |
    | 2017-04-01 |           4.5 |        11.8 | 0.801385  0.759061  0.0                   0.0 | 0.8     2.0  0.0                      1.5 |  3.7      2.5  0.0          2.5 |  53.1   53.1  0.0               53.1 | 0.0  0.0  0.0       0.0 |  3.7      2.5  0.0      2.5 | 0.0  0.0  0.0            2.5 |  2.058177    1.19578  0.0          0.0 | 0.666816   0.00234  0.0                     0.0 | 0.0  0.0  1.9245              0.0 | 1.466816   2.00234  1.9245                      1.5 | 0.0  0.061871  0.0            0.0 |       0.0        0.0  0.0                 0.0 |      0.011395 |        0.011395 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   1.133909   2.5755                0.0 | 0.0  0.226782   0.5151       0.0 |       0.741882 |       -0.002781 | 0.0     0.0  0.0               0.5 | 0.0  0.0  0.0       0.0 | 80.138481  75.906114  0.0           0.0 |          0.1 | -0.012563 |
    | 2017-04-02 |           0.0 |         9.4 | 0.786093   0.73948  0.0                   0.0 | 0.0     0.0  0.0                      0.5 |  0.0      0.0  0.0          0.0 |  42.3   42.3  0.0               42.3 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.529168  2.010553  0.0                     0.0 | 0.0  0.0  2.0528              0.0 | 1.529168  2.010553  2.0528                      0.5 | 0.0  0.052451  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.052451  -2.0528                0.0 | 0.0       0.0      0.0       0.0 |      -0.526313 |       -0.097682 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 78.609313  73.948012  0.0           0.0 |          0.1 | -0.441194 |
    | 2017-04-03 |           0.0 |         8.1 | 0.767078  0.715211  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 36.45  36.45  0.0              36.45 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.901523   2.49322  0.0                     0.0 | 0.0  0.0   2.566              0.0 | 1.901523   2.49322   2.566                      0.0 | 0.0  0.066306  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.066306   -2.566                0.0 | 0.0       0.0      0.0       0.0 |      -0.658077 |       -0.199264 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  76.70779  71.521097  0.0           0.0 |          0.1 | -0.900006 |
    | 2017-04-04 |           0.0 |         7.9 |  0.74725   0.69018  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 35.55  35.55  0.0              35.55 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.982784  2.589761  0.0                     0.0 | 0.0  0.0  2.6943              0.0 | 1.982784  2.589761  2.6943                      0.0 | 0.0  0.086617  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.086617  -2.6943                0.0 | 0.0       0.0      0.0       0.0 |      -0.695229 |       -0.289167 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 74.725006  69.017953  0.0           0.0 |          0.1 | -1.306068 |
    | 2017-04-05 |           0.4 |         7.5 | 0.735071  0.674236  0.0                   0.0 | 0.4     0.4  0.0                      0.4 |  0.0      0.0  0.0          0.0 | 33.75  33.75  0.0              33.75 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.217868  1.705859  0.0                     0.0 | 0.0  0.0  2.1811              0.0 | 1.617868  2.105859  2.1811                      0.4 | 0.0  0.111522  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.111522  -1.7811                0.0 | 0.0       0.0      0.0       0.0 |      -0.473156 |       -0.322519 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 73.507138  67.423615  0.0           0.0 |          0.1 | -1.456705 |
    | 2017-04-06 |           0.0 |         7.2 | 0.719227  0.654999  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  32.4   32.4  0.0               32.4 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.584455   2.05331  0.0                     0.0 | 0.0  0.0  2.1811              0.0 | 1.584455   2.05331  2.1811                      0.0 | 0.0   0.12964  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0   -0.12964  -2.1811                0.0 | 0.0       0.0      0.0       0.0 |      -0.577685 |       -0.368772 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 71.922683  65.499946  0.0           0.0 |          0.1 | -1.665618 |
    | 2017-04-07 |           0.0 |         8.1 | 0.711823  0.646983  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 36.45  36.45  0.0              36.45 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.740384  0.955678  0.0                     0.0 | 0.0  0.0  1.0264              0.0 | 0.740384  0.955678  1.0264                      0.0 | 0.0  0.153989  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.153989  -1.0264                0.0 | 0.0       0.0      0.0       0.0 |      -0.295097 |       -0.355417 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 71.182298  64.698257  0.0           0.0 |          0.1 | -1.605298 |
    | 2017-04-08 |           0.0 |         8.6 | 0.699833  0.633178  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  38.7   38.7  0.0               38.7 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.198976  1.545464  0.0                     0.0 | 0.0  0.0  1.6679              0.0 | 1.198976  1.545464  1.6679                      0.0 | 0.0  0.164976  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.164976  -1.6679                0.0 | 0.0       0.0      0.0       0.0 |      -0.458219 |       -0.374052 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 69.983322  63.317769  0.0           0.0 |          0.1 | -1.689465 |
    | 2017-04-09 |           0.0 |        10.5 |  0.67691  0.605566  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 47.25  47.25  0.0              47.25 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 2.292354  2.946224  0.0                     0.0 | 0.0  0.0  3.2075              0.0 | 2.292354  2.946224  3.2075                      0.0 | 0.0  0.185097  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 |       0.0       0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 0.0  -0.185097  -3.2075                0.0 | 0.0       0.0      0.0       0.0 |      -0.848149 |       -0.459991 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 67.690968  60.556642  0.0           0.0 |          0.1 | -2.077623 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_urban_snow:

snow
____

All snow processes are modelled exactly as by |whmod_rural| and described in the
:ref:`whmod_rural_snow` example:

.. integration-test::

    >>> cisternsource(False)
    >>> test.inits.cisternwater = 0.0
    >>> inputs.temperature.series -= 8.0
    >>> conditions = test("whmod_urban_snow", get_conditions="2017-02-10")
    |       date | precipitation | temperature |                          relativesoilmoisture |                   interceptionevaporation |                     throughfall |                    potentialsnowmelt |                     snowmelt |                      ponding |                surfacerunoff |                            percolation |                          soilevapotranspiration |                   lakeevaporation |                             totalevapotranspiration |                          capillaryrise |                            requiredirrigation | cisterninflow | cisternoverflow | cisterndemand | cisternextraction |                internalirrigation |                externalirrigation |                                potentialrecharge |                              baseflow | actualrecharge | delayedrecharge |                   interceptedwater |                       snowpack |                            soilmoisture | cisternwater | deepwater |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |       -10.8 |   0.0375    0.0375  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.7302              0.0 |      0.0       0.0  0.7302                      0.0 |     3.75      3.75  0.0            0.0 |     66.25      66.25  0.0                 0.0 |           0.0 |             0.0 |       0.33125 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |     -3.75      -3.75  -0.7302                0.0 |      0.0       0.0      0.0       0.0 |       -2.05755 |       -0.372971 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 |      3.75       3.75  0.0           0.0 |          0.0 | -1.684579 |
    | 2017-02-11 |           0.4 |        -9.5 | 0.070632  0.070427  0.0                   0.0 | 0.4     0.4  0.0                      0.4 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.030545  0.051011  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.430545  0.451011  0.9736                      0.4 | 3.343748  3.343748  0.0            0.0 | 62.936798  62.957263  0.0                 0.0 |           0.0 |             0.0 |      0.314735 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -3.343748  -3.343748  -0.5736                0.0 |      0.0       0.0      0.0       0.0 |      -1.815274 |       -0.634416 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 |  7.063202   7.042737  0.0           0.0 |          0.0 | -2.865437 |
    | 2017-02-12 |           0.0 |        -8.9 | 0.099731  0.099214  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.100328  0.133554  0.0                     0.0 | 0.0  0.0  0.8519              0.0 | 0.100328  0.133554  0.8519                      0.0 | 3.010193  3.012182  0.0            0.0 | 60.026932  60.078634  0.0                 0.0 |           0.0 |             0.0 |      0.300264 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -3.010193  -3.012182  -0.8519                0.0 |      0.0       0.0      0.0       0.0 |      -1.718569 |       -0.830939 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 |  9.973068   9.921366  0.0           0.0 |          0.0 | -3.753067 |
    | 2017-02-13 |           0.0 |        -9.6 | 0.126288  0.125554  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.080508  0.106933  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.080508  0.106933  0.4868                      0.0 | 2.736205  2.740922  0.0            0.0 | 57.371235  57.444646  0.0                 0.0 |           0.0 |             0.0 |       0.28704 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.736205  -2.740922  -0.4868                0.0 |      0.0       0.0      0.0       0.0 |      -1.490982 |       -0.950585 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 | 12.628765  12.555354  0.0           0.0 |          0.0 | -4.293464 |
    | 2017-02-14 |           0.0 |        -9.3 | 0.150286  0.149284  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.101248  0.134409  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.101248  0.134409  0.4868                      0.0 | 2.501132  2.507442  0.0            0.0 | 54.971351  55.071613  0.0                 0.0 |           0.0 |             0.0 |      0.275107 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.501132  -2.507442  -0.4868                0.0 |      0.0       0.0      0.0       0.0 |      -1.373844 |       -1.027309 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 | 15.028649  14.928387  0.0           0.0 |          0.0 | -4.639999 |
    | 2017-02-15 |           0.0 |        -6.3 | 0.172097  0.170786  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.119564  0.158608  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.119564  0.158608  0.4868                      0.0 | 2.300641  2.308794  0.0            0.0 | 52.790274  52.921427  0.0                 0.0 |           0.0 |             0.0 |      0.264279 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.300641  -2.308794  -0.4868                0.0 |      0.0       0.0      0.0       0.0 |      -1.274059 |       -1.072037 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 | 17.209726  17.078573  0.0           0.0 |          0.0 | -4.842021 |
    | 2017-02-16 |           0.2 |        -3.6 | 0.192698  0.191041  0.0                   0.0 | 0.2     0.2  0.0                      0.2 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.067899  0.112586  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.267899  0.312586  0.4868                      0.2 | 2.127988  2.138117  0.0            0.0 | 50.730184  50.895895  0.0                 0.0 |           0.0 |             0.0 |      0.254065 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.127988  -2.138117  -0.2868                0.0 |      0.0       0.0      0.0       0.0 |      -1.138226 |       -1.084035 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 | 19.269816  19.104105  0.0           0.0 |          0.0 | -4.896213 |
    | 2017-02-17 |           4.5 |        -3.5 | 0.212429  0.210893  0.0                   0.0 | 0.3  0.4005  0.0                      0.3 |  4.1      2.5  0.0          2.5 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 |      0.3    0.4005  0.3651                      0.3 | 1.973055   1.98523  0.0            0.0 |  48.75713  48.910666  0.0                 0.0 |           0.0 |             0.0 |      0.244169 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.973055   -1.98523   4.1349                0.0 |      0.0       0.0  0.82698       0.0 |      -0.162591 |       -0.917006 | 0.1  1.5995  0.0               1.7 |    4.1      2.5  0.0       2.5 |  21.24287  21.089334  0.0           0.0 |          0.0 | -4.141798 |
    | 2017-02-18 |           0.0 |        -4.6 | 0.229925   0.22932  0.0                   0.0 | 0.1  0.4005  0.0                      0.3 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.082315       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 | 0.182315    0.4005  0.3651                      0.3 | 1.831896  1.842631  0.0            0.0 | 47.007549  47.068035  0.0                 0.0 |           0.0 |             0.0 |      0.235189 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.831896  -1.842631  -0.3651                0.0 |      0.0       0.0      0.0       0.0 |      -1.009907 |       -0.933846 | 0.0   1.199  0.0               1.4 |    4.1      2.5  0.0       2.5 | 22.992451  22.931965  0.0           0.0 |          0.0 | -4.217859 |
    | 2017-02-19 |           3.2 |        -3.2 |  0.24705  0.246485  0.0                   0.0 | 0.4   0.534  0.0                      0.4 |  2.8    2.399  0.0          2.6 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.4868              0.0 |      0.4     0.534  0.4868                      0.4 | 1.712502  1.716541  0.0            0.0 | 45.295046  45.351494  0.0                 0.0 |           0.0 |             0.0 |      0.226616 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.712502  -1.716541   2.7132                0.0 |      0.0       0.0  0.54264       0.0 |      -0.314621 |       -0.821599 | 0.0   1.466  0.0               1.6 |    6.9    4.899  0.0       5.1 | 24.704954  24.648506  0.0           0.0 |          0.0 | -3.710881 |
    | 2017-02-20 |           4.6 |        -1.3 | 0.263057  0.262529  0.0                   0.0 | 0.3  0.4005  0.0                      0.3 |  4.2    4.066  0.0          4.2 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 |      0.3    0.4005  0.3651                      0.3 | 1.600776  1.604379  0.0            0.0 | 43.694271  43.747115  0.0                 0.0 |           0.0 |             0.0 |      0.218603 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.600776  -1.604379   4.2349                0.0 |      0.0       0.0  0.84698       0.0 |       0.045691 |       -0.664386 | 0.1  1.5995  0.0               1.7 |   11.1    8.965  0.0       9.3 | 26.305729  26.252885  0.0           0.0 |          0.0 | -3.000803 |
    | 2017-02-21 |           2.3 |        -2.2 | 0.277074  0.277569  0.0                   0.0 | 0.4   0.801  0.0                      0.6 |  2.0   1.8995  0.0          2.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.099174       0.0  0.0                     0.0 | 0.0  0.0  0.7302              0.0 | 0.499174     0.801  0.7302                      0.6 | 1.500833  1.504064  0.0            0.0 | 42.292611  42.243051  0.0                 0.0 |           0.0 |             0.0 |      0.211339 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.500833  -1.504064   1.5698                0.0 |      0.0       0.0  0.31396       0.0 |      -0.437264 |       -0.623216 | 0.0   1.199  0.0               1.4 |   13.1  10.8645  0.0      11.3 | 27.707389  27.756949  0.0           0.0 |          0.0 | -2.814852 |
    | 2017-02-22 |          18.0 |        -1.5 |  0.28917  0.291709  0.0                   0.0 | 0.4   1.068  0.0                      0.8 | 17.6   17.199  0.0         17.4 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.207154       0.0  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.607154     1.068  0.9736                      0.8 | 1.416815  1.413903  0.0            0.0 | 41.082951  40.829148  0.0                 0.0 |           0.0 |             0.0 |       0.20478 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.416815  -1.413903  17.0264                0.0 |      0.0       0.0  3.40528       0.0 |       2.697601 |       -0.021254 | 0.0   0.932  0.0               1.2 |   30.7  28.0635  0.0      28.7 | 28.917049  29.170852  0.0           0.0 |          0.0 | -0.095997 |
    | 2017-02-23 |          19.2 |        -3.0 | 0.302103  0.305034  0.0                   0.0 | 0.4  0.6675  0.0                      0.5 | 18.8   18.132  0.0         18.4 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.053641       0.0  0.0                     0.0 | 0.0  0.0  0.6085              0.0 | 0.453641    0.6675  0.6085                      0.5 | 1.346876    1.3325  0.0            0.0 | 39.789715  39.496648  0.0                 0.0 |           0.0 |             0.0 |      0.198216 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.346876    -1.3325  18.5915                0.0 |      0.0       0.0   3.7183       0.0 |       3.048456 |         0.53519 | 0.0  1.3325  0.0               1.5 |   49.5  46.1955  0.0      47.1 | 30.210285  30.503352  0.0           0.0 |          0.0 |  2.417269 |
    | 2017-02-24 |           0.4 |        -5.0 | 0.312627  0.317621  0.0                   0.0 | 0.4   1.068  0.0                      0.8 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.222284       0.0  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.622284     1.068  0.9736                      0.8 | 1.274693  1.258702  0.0            0.0 | 38.737306  38.237946  0.0                 0.0 |           0.0 |             0.0 |      0.192438 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.274693  -1.258702  -0.5736                0.0 |      0.0       0.0      0.0       0.0 |      -0.776749 |        0.297376 | 0.0  0.6645  0.0               1.1 |   49.5  46.1955  0.0      47.1 | 31.262694  31.762054  0.0           0.0 |          0.0 |  1.343144 |
    | 2017-02-25 |           8.3 |        -4.9 | 0.324235  0.329536  0.0                   0.0 | 0.4  0.6675  0.0                      0.5 |  7.9   6.9645  0.0          7.4 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.057104       0.0  0.0                     0.0 | 0.0  0.0  0.6085              0.0 | 0.457104    0.6675  0.6085                      0.5 | 1.217892  1.191541  0.0            0.0 | 37.576518  37.046405  0.0                 0.0 |           0.0 |             0.0 |      0.186557 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.217892  -1.191541   7.6915                0.0 |      0.0       0.0   1.5383       0.0 |       0.935942 |        0.413128 | 0.0  1.3325  0.0               1.5 |   57.4    53.16  0.0      54.5 | 32.423482  32.953595  0.0           0.0 |          0.0 |  1.865958 |
    | 2017-02-26 |           5.3 |        -0.9 | 0.335807  0.340838  0.0                   0.0 | 0.4   0.534  0.0                      0.4 |  4.9   4.6325  0.0          4.8 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.4868              0.0 |      0.4     0.534  0.4868                      0.4 | 1.157227  1.130206  0.0            0.0 | 36.419291  35.916199  0.0                 0.0 |           0.0 |             0.0 |      0.180839 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.157227  -1.130206   4.8132                0.0 |      0.0       0.0  0.96264       0.0 |       0.390782 |        0.409077 | 0.0   1.466  0.0               1.6 |   62.3  57.7925  0.0      59.3 | 33.580709  34.083801  0.0           0.0 |          0.0 |  1.847662 |
    | 2017-02-27 |           0.7 |         1.4 | 0.403097    0.4119  0.0                   0.0 | 0.4  1.7355  0.0                      1.3 |  0.3    0.166  0.0          0.3 |   6.3    6.3  0.0                6.3 |   6.3     6.3  0.0       6.3 |   6.6    6.466  0.0      6.6 | 0.0  0.0  0.0            6.6 |  0.426533   0.433766  0.0          0.0 | 0.543217       0.0  0.0                     0.0 | 0.0  0.0  1.5821              0.0 | 0.943217    1.7355  1.5821                      1.3 | 1.098788  1.074009  0.0            0.0 | 29.690253  28.809956  0.0                 0.0 |           0.0 |             0.0 |      0.146251 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.672255  -0.640242  -0.8821                0.0 |      0.0       0.0      0.0       0.0 |      -0.548649 |        0.235471 | 0.0  0.2645  0.0               0.7 |   56.0  51.4925  0.0      53.0 | 40.309747  41.190044  0.0           0.0 |          0.0 |  1.063542 |
    | 2017-02-28 |           2.7 |        -3.4 |  0.40763  0.419528  0.0                   0.0 | 0.4  1.2015  0.0                      0.9 |  2.3   0.9645  0.0          1.4 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.344257       0.0  0.0                     0.0 | 0.0  0.0  1.0953              0.0 | 0.744257    1.2015  1.0953                      0.9 |  0.79752  0.762753  0.0            0.0 |  29.23699  28.047204  0.0                 0.0 |           0.0 |             0.0 |       0.14321 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.79752  -0.762753   1.6047                0.0 |      0.0       0.0  0.32094       0.0 |      -0.069128 |        0.180257 | 0.0  0.7985  0.0               1.1 |   58.3   52.457  0.0      54.4 |  40.76301  41.952796  0.0           0.0 |          0.0 |  0.814157 |
    | 2017-03-01 |           1.6 |        -4.3 | 0.414731  0.426863  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.0   0.3985  0.0          0.7 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.069373       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.669373    0.9345  0.8792                      0.7 | 0.779489  0.733458  0.0            0.0 | 28.526874  27.313746  0.0                 0.0 |           0.0 |             0.0 |      0.139602 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.779489  -0.733458   0.7208                0.0 |      0.0       0.0  0.14416       0.0 |      -0.234077 |        0.105151 | 0.0  1.0655  0.0               1.3 |   59.3  52.8555  0.0      55.1 | 41.473126  42.686254  0.0           0.0 |          0.0 |  0.474929 |
    | 2017-03-02 |           2.5 |        -3.3 | 0.421547  0.433923  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.9   1.5655  0.0          1.8 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.070178       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.670178    0.9345  0.8792                      0.7 | 0.751791  0.706005  0.0            0.0 | 27.845261  26.607741  0.0                 0.0 |           0.0 |             0.0 |      0.136133 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.751791  -0.706005   1.6208                0.0 |      0.0       0.0  0.32416       0.0 |      -0.040289 |        0.078787 | 0.0  1.0655  0.0               1.3 |   61.2   54.421  0.0      56.9 | 42.154739  43.392259  0.0           0.0 |          0.0 |  0.355853 |
    | 2017-03-03 |           0.6 |        -2.1 | 0.425259  0.440725  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.354684       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 0.954684    1.4685  1.3816                      1.1 |  0.72583  0.680235  0.0            0.0 | 27.474116  25.927506  0.0                 0.0 |           0.0 |             0.0 |      0.133504 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.72583  -0.680235  -0.7816                0.0 |      0.0       0.0      0.0       0.0 |      -0.546916 |       -0.034634 | 0.0   0.197  0.0               0.8 |   61.2   54.421  0.0      56.9 | 42.525884  44.072494  0.0           0.0 |          0.0 | -0.156429 |
    | 2017-03-04 |           0.2 |        -0.3 | 0.426671  0.440438  0.0                   0.0 | 0.2   0.397  0.0                      1.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.570753  0.684711  0.0                     0.0 | 0.0  0.0   1.256              0.0 | 0.770753  1.081711   1.256                      1.0 | 0.711948  0.656006  0.0            0.0 | 27.332921  25.956212  0.0                 0.0 |           0.0 |             0.0 |      0.133223 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.711948  -0.656006   -1.056                0.0 |      0.0       0.0      0.0       0.0 |      -0.605988 |       -0.138203 | 0.0     0.0  0.0               0.0 |   61.2   54.421  0.0      56.9 | 42.667079  44.043788  0.0           0.0 |          0.0 | -0.624214 |
    | 2017-03-05 |           1.7 |        -1.7 | 0.432308  0.447008  0.0                   0.0 | 0.6   1.068  0.0                      0.8 |  1.1      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.142996       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.742996     1.068  1.0048                      0.8 | 0.706714  0.657016  0.0            0.0 | 26.769203  25.299195  0.0                 0.0 |           0.0 |             0.0 |      0.130171 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.706714  -0.657016   0.6952                0.0 |      0.0       0.0  0.13904       0.0 |      -0.201892 |       -0.149748 | 0.0   0.632  0.0               0.9 |   62.3   54.421  0.0      56.9 | 43.230797  44.700805  0.0           0.0 |          0.0 | -0.676359 |
    | 2017-03-06 |           0.3 |        -4.3 | 0.437006  0.453349  0.0                   0.0 | 0.3   0.801  0.0                      0.6 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.216318       0.0  0.0                     0.0 | 0.0  0.0  0.7536              0.0 | 0.516318     0.801  0.7536                      0.6 | 0.686072  0.634144  0.0            0.0 | 26.299449  24.665052  0.0                 0.0 |           0.0 |             0.0 |      0.127411 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.686072  -0.634144  -0.4536                0.0 |      0.0       0.0      0.0       0.0 |      -0.443454 |       -0.202988 | 0.0   0.131  0.0               0.6 |   62.3   54.421  0.0      56.9 | 43.700551  45.334948  0.0           0.0 |          0.0 | -0.916825 |
    | 2017-03-07 |           0.0 |        -6.4 | 0.438615  0.453506  0.0                   0.0 | 0.0   0.131  0.0                      0.6 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.508241  0.596954  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.508241  0.727954  0.8792                      0.6 | 0.669181  0.612577  0.0            0.0 | 26.138509  24.649429  0.0                 0.0 |           0.0 |             0.0 |       0.12697 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.669181  -0.612577  -0.8792                0.0 |      0.0       0.0      0.0       0.0 |      -0.540239 |       -0.264121 | 0.0     0.0  0.0               0.0 |   62.3   54.421  0.0      56.9 | 43.861491  45.350571  0.0           0.0 |          0.0 | -1.192944 |
    | 2017-03-08 |           1.8 |        -4.0 | 0.444522  0.459626  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.2      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.072776       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.672776    0.9345  0.8792                      0.7 | 0.663459  0.612052  0.0            0.0 | 25.547826  24.037377  0.0                 0.0 |           0.0 |             0.0 |      0.123963 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.663459  -0.612052   0.9208                0.0 |      0.0       0.0  0.18416       0.0 |      -0.134718 |       -0.240664 | 0.0  0.8655  0.0               1.1 |   63.5   54.421  0.0      56.9 | 44.452174  45.962623  0.0           0.0 |          0.0 | -1.086997 |
    | 2017-03-09 |           8.9 |        -2.4 | 0.450949  0.465543  0.0                   0.0 | 0.5  0.6675  0.0                      0.5 |  8.3   7.7655  0.0          8.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0   0.628              0.0 |      0.5    0.6675   0.628                      0.5 | 0.642736  0.591717  0.0            0.0 |  24.90509   23.44566  0.0                 0.0 |           0.0 |             0.0 |      0.120877 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.642736  -0.591717    8.272                0.0 |      0.0       0.0   1.6544       0.0 |       1.345787 |        0.046911 | 0.1  1.3325  0.0               1.5 |   71.8  62.1865  0.0      64.9 |  45.09491   46.55434  0.0           0.0 |          0.0 |  0.211879 |
    | 2017-03-10 |           0.0 |        -2.2 | 0.451972  0.471268  0.0                   0.0 | 0.1   1.068  0.0                      0.8 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.518358       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.618358     1.068  1.0048                      0.8 | 0.620682  0.572491  0.0            0.0 | 24.802766   22.87317  0.0                 0.0 |           0.0 |             0.0 |       0.11919 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.620682  -0.572491  -1.0048                0.0 |      0.0       0.0      0.0       0.0 |      -0.549493 |       -0.061199 | 0.0  0.2645  0.0               0.7 |   71.8  62.1865  0.0      64.9 | 45.197234   47.12683  0.0           0.0 |          0.0 | -0.276415 |
    | 2017-03-11 |           0.0 |        -2.3 | 0.450729  0.468669  0.0                   0.0 | 0.0  0.2645  0.0                      0.7 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |  0.74155  0.814195  0.0                     0.0 | 0.0  0.0   1.256              0.0 |  0.74155  1.078695   1.256                      0.7 | 0.617218   0.55429  0.0            0.0 | 24.927098  23.133074  0.0                 0.0 |           0.0 |             0.0 |       0.12015 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.617218   -0.55429   -1.256                0.0 |      0.0       0.0      0.0       0.0 |      -0.606877 |       -0.160114 | 0.0     0.0  0.0               0.0 |   71.8  62.1865  0.0      64.9 | 45.072902  46.866926  0.0           0.0 |          0.0 | -0.723179 |
    | 2017-03-12 |           0.0 |        -3.4 |  0.44806   0.46215  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.888345  1.214433  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 0.888345  1.214433  1.5072                      0.0 | 0.621429  0.562505  0.0            0.0 | 25.194015  23.785002  0.0                 0.0 |           0.0 |             0.0 |      0.122448 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.621429  -0.562505  -1.5072                0.0 |      0.0       0.0      0.0       0.0 |      -0.672783 |       -0.253045 | 0.0     0.0  0.0               0.0 |   71.8  62.1865  0.0      64.9 | 44.805985  46.214998  0.0           0.0 |          0.0 | -1.142917 |
    | 2017-03-13 |           0.9 |        -3.8 | 0.452152  0.465718  0.0                   0.0 | 0.6     0.9  0.0                      0.9 |  0.3      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.221269  0.226642  0.0                     0.0 | 0.0  0.0  1.1304              0.0 | 0.821269  1.126642  1.1304                      0.9 | 0.630532  0.583465  0.0            0.0 | 24.784751  23.428179  0.0                 0.0 |           0.0 |             0.0 |      0.120532 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.630532  -0.583465  -0.2304                0.0 |      0.0       0.0      0.0       0.0 |      -0.361099 |       -0.272632 | 0.0     0.0  0.0               0.0 |   72.1  62.1865  0.0      64.9 | 45.215249  46.571821  0.0           0.0 |          0.0 | -1.231384 |
    | 2017-03-14 |           0.1 |        -0.6 | 0.452385  0.463119  0.0                   0.0 | 0.1     0.1  0.0                      0.1 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.593386  0.831864  0.0                     0.0 | 0.0  0.0  1.1304              0.0 | 0.693386  0.931864  1.1304                      0.1 |  0.61661  0.571929  0.0            0.0 | 24.761528  23.688114  0.0                 0.0 |           0.0 |             0.0 |      0.121124 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.61661  -0.571929  -1.0304                0.0 |      0.0       0.0      0.0       0.0 |      -0.554735 |       -0.323768 | 0.0     0.0  0.0               0.0 |   72.1  62.1865  0.0      64.9 | 45.238472  46.311886  0.0           0.0 |          0.0 | -1.462351 |
    | 2017-03-15 |           0.0 |        -1.7 | 0.449639  0.456864  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.890361  1.205774  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 0.890361  1.205774  1.5072                      0.0 | 0.615826  0.580318  0.0            0.0 | 25.036063   24.31357  0.0                 0.0 |           0.0 |             0.0 |      0.123374 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.615826  -0.580318  -1.5072                0.0 |      0.0       0.0      0.0       0.0 |      -0.675836 |       -0.387587 | 0.0     0.0  0.0               0.0 |   72.1  62.1865  0.0      64.9 | 44.963937   45.68643  0.0           0.0 |          0.0 | -1.750599 |
    | 2017-03-16 |           0.0 |         0.7 | 0.472806  0.476012  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  3.15   3.15  0.0               3.15 |  3.15    3.15  0.0      3.15 |  3.15     3.15  0.0     3.15 | 0.0  0.0  0.0           3.15 |  0.423588   0.440881  0.0          0.0 | 1.034849  1.395158  0.0                     0.0 | 0.0  0.0  1.7584              0.0 | 1.034849  1.395158  1.7584                      0.0 | 0.625134  0.600837  0.0            0.0 | 22.719366  22.398773  0.0                 0.0 |           0.0 |             0.0 |      0.112795 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.201546  -0.159955  -1.7584                0.0 |      0.0       0.0      0.0       0.0 |      -0.529975 |       -0.413398 | 0.0     0.0  0.0               0.0 |  68.95  59.0365  0.0     61.75 | 47.280634  47.601227  0.0           0.0 |          0.0 | -1.867176 |
    | 2017-03-17 |           3.9 |        -1.6 | 0.474491  0.481407  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  3.3      1.9  0.0          1.9 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.381023       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 0.981023    1.4685  1.3816                      1.1 | 0.549467  0.539504  0.0            0.0 | 22.550921  21.859269  0.0                 0.0 |           0.0 |             0.0 |      0.111025 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.549467  -0.539504   2.5184                0.0 |      0.0       0.0  0.50368       0.0 |       0.231437 |       -0.296509 | 0.0  0.5315  0.0               0.9 |  72.25  60.9365  0.0     63.65 | 47.449079  48.140731  0.0           0.0 |          0.0 |  -1.33923 |
    | 2017-03-18 |           8.7 |        -2.8 | 0.476115  0.486637  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  8.1   7.2315  0.0          7.6 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.381825       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 0.981825    1.4685  1.3816                      1.1 | 0.544217  0.523011  0.0            0.0 | 22.388529  21.336258  0.0                 0.0 |           0.0 |             0.0 |      0.109312 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.544217  -0.523011   7.3184                0.0 |      0.0       0.0  1.46368       0.0 |       1.196873 |       -0.025805 | 0.0  0.5315  0.0               0.9 |  80.35   68.168  0.0     71.25 | 47.611471  48.663742  0.0           0.0 |          0.0 | -0.116552 |
    | 2017-03-19 |          26.4 |        -2.9 | 0.481507  0.491711  0.0                   0.0 | 0.5  0.6675  0.0                      0.5 | 25.8  24.9315  0.0         25.3 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0   0.628              0.0 |      0.5    0.6675   0.628                      0.5 | 0.539188  0.507346  0.0            0.0 | 21.849341  20.828913  0.0                 0.0 |           0.0 |             0.0 |      0.106696 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.539188  -0.507346   25.772                0.0 |      0.0       0.0   5.1544       0.0 |       4.892767 |        0.865781 | 0.1  1.3325  0.0               1.5 | 106.15  93.0995  0.0     96.55 | 48.150659  49.171087  0.0           0.0 |          0.0 |  3.910434 |
    | 2017-03-20 |          11.5 |         0.7 | 0.605637  0.612925  0.0                   0.0 | 0.6   0.801  0.0                      0.6 | 11.0  10.8325  0.0         11.0 |  3.15   3.15  0.0               3.15 |  3.15    3.15  0.0      3.15 | 14.15  13.9825  0.0    14.15 | 0.0  0.0  0.0          14.15 |  2.259628   2.353568  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.7536              0.0 |      0.6     0.801  0.7536                      0.6 |  0.52271  0.492452  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  1.736918   1.861116  10.7464                0.0 | 0.347384  0.372223  2.14928       0.0 |       2.868887 |        1.228882 | 0.0   1.199  0.0               1.4 |  103.0  89.9495  0.0      93.4 | 60.563741  61.292471  0.0           0.0 |          0.0 |  5.550438 |
    | 2017-03-21 |           0.9 |        -1.8 | 0.600142  0.615078  0.0                   0.0 | 0.6     2.0  0.0                      1.5 |  0.3    0.099  0.0          0.3 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.779586  0.002177  0.0                     0.0 | 0.0  0.0   1.884              0.0 | 1.379586  2.002177   1.884                      1.5 | 0.229995  0.217479  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.229995  -0.217479   -0.984                0.0 |      0.0       0.0      0.0       0.0 |      -0.357869 |        0.941253 | 0.0     0.0  0.0               0.5 |  103.3  90.0485  0.0      93.7 |  60.01415  61.507773  0.0           0.0 |          0.0 |  4.251317 |
    | 2017-03-22 |           0.0 |        -2.1 | 0.585285  0.593933  0.0                   0.0 | 0.0     0.0  0.0                      0.5 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.725432  2.328388  0.0                     0.0 | 0.0  0.0   2.512              0.0 | 1.725432  2.328388   2.512                      0.5 | 0.239745   0.21387  0.0            0.0 | 11.471537  10.606744  0.0                 0.0 |           0.0 |             0.0 |      0.055196 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.239745   -0.21387   -2.512                0.0 |      0.0       0.0      0.0       0.0 |      -0.741404 |        0.636239 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 58.528463  59.393256  0.0           0.0 |          0.0 |  2.873673 |
    | 2017-03-23 |           0.0 |        -2.8 | 0.574312  0.578102  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.364731  1.834174  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.364731  1.834174  2.0096                      0.0 | 0.267474  0.251088  0.0            0.0 | 12.568794   12.18983  0.0                 0.0 |           0.0 |             0.0 |      0.061897 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.267474  -0.251088  -2.0096                0.0 |      0.0       0.0      0.0       0.0 |       -0.63204 |        0.406339 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 57.431206   57.81017  0.0           0.0 |          0.0 |  1.835294 |
    | 2017-03-24 |           0.0 |        -2.8 | 0.563678  0.562803  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.352701  1.811468  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.352701  1.811468  2.0096                      0.0 | 0.289271  0.281614  0.0            0.0 | 13.632224  13.719684  0.0                 0.0 |           0.0 |             0.0 |       0.06838 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.289271  -0.281614  -2.0096                0.0 |      0.0       0.0      0.0       0.0 |      -0.645121 |        0.215742 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 56.367776  56.280316  0.0           0.0 |          0.0 |  0.974431 |
    | 2017-03-25 |           0.0 |        -2.1 | 0.556738  0.552524  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.005469  1.341289  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 1.005469  1.341289  1.5072                      0.0 | 0.311497  0.313374  0.0            0.0 | 14.326197  14.747599  0.0                 0.0 |           0.0 |             0.0 |      0.072684 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.311497  -0.313374  -1.5072                0.0 |      0.0       0.0      0.0       0.0 |      -0.533018 |        0.080015 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 55.673803  55.252401  0.0           0.0 |          0.0 |  0.361399 |
    | 2017-03-26 |           0.0 |        -1.3 | 0.549177  0.541485  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.082673  1.439933  0.0                     0.0 | 0.0  0.0  1.6328              0.0 | 1.082673  1.439933  1.6328                      0.0 | 0.326597  0.336001  0.0            0.0 | 15.082272  15.851531  0.0                 0.0 |           0.0 |             0.0 |      0.077335 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.326597  -0.336001  -1.6328                0.0 |      0.0       0.0      0.0       0.0 |       -0.57385 |       -0.038511 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 54.917728  54.148469  0.0           0.0 |          0.0 |  -0.17394 |
    | 2017-03-27 |           0.0 |        -1.0 | 0.539378  0.527557  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.323482  1.754272  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.323482  1.754272  2.0096                      0.0 | 0.343596  0.361487  0.0            0.0 | 16.062157  17.244316  0.0                 0.0 |           0.0 |             0.0 |      0.083266 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.343596  -0.361487  -2.0096                0.0 |      0.0       0.0      0.0       0.0 |      -0.678671 |       -0.154552 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 53.937843  52.755684  0.0           0.0 |          0.0 | -0.698059 |
    | 2017-03-28 |           0.0 |         0.3 | 0.538104  0.521748  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  1.35   1.35  0.0               1.35 |  1.35    1.35  0.0      1.35 |  1.35     1.35  0.0     1.35 | 0.0  0.0  0.0           1.35 |  0.286644   0.271135  0.0          0.0 | 1.557337  2.055221  0.0                     0.0 | 0.0  0.0  2.3864              0.0 | 1.557337  2.055221  2.3864                      0.0 | 0.366492  0.395439  0.0            0.0 | 16.189647  17.825232  0.0                 0.0 |           0.0 |             0.0 |      0.085037 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.079847  -0.124305  -2.3864                0.0 |      0.0       0.0      0.0       0.0 |      -0.647638 |       -0.243933 | 0.0     0.0  0.0               0.0 | 101.95  88.6985  0.0     92.35 | 53.810353  52.174768  0.0           0.0 |          0.0 | -1.101763 |
    | 2017-03-29 |           1.5 |         1.0 | 0.582764   0.56206  0.0                   0.0 | 0.6   1.068  0.0                      0.8 |  0.9      0.0  0.0          0.0 |   4.5    4.5  0.0                4.5 |   4.5     4.5  0.0       4.5 |   5.4      4.5  0.0      4.5 | 0.0  0.0  0.0            4.5 |  1.139787   0.879008  0.0          0.0 | 0.163731       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.763731     1.068  1.0048                      0.8 | 0.369543  0.410206  0.0            0.0 | 11.723622  13.794033  0.0                 0.0 |           0.0 |             0.0 |      0.063794 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.770244   0.468802   0.4952                0.0 | 0.154049   0.09376  0.09904       0.0 |       0.346849 |       -0.136843 | 0.0   0.432  0.0               0.7 |  97.45  84.1985  0.0     87.85 | 58.276378  56.205967  0.0           0.0 |          0.0 | -0.618072 |
    | 2017-03-30 |           0.3 |         4.4 |  0.72222  0.705958  0.0                   0.0 | 0.3   0.732  0.0                      1.0 |  0.0      0.0  0.0          0.0 |  19.8   19.8  0.0               19.8 |  19.8    19.8  0.0      19.8 |  19.8     19.8  0.0     19.8 | 0.0  0.0  0.0           19.8 |  5.105211    4.66207  0.0          0.0 | 1.021504  1.063053  0.0                     0.0 | 0.0  0.0   1.884              0.0 | 1.321504  1.795053   1.884                      1.0 | 0.272381  0.314975  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |   4.83283   4.347095   -1.584                0.0 | 0.966566  0.869419      0.0       0.0 |       1.439985 |        0.148988 | 0.0     0.0  0.0               0.0 |  77.65  64.3985  0.0     68.05 | 72.222044  70.595819  0.0           0.0 |          0.0 |  0.672926 |
    | 2017-03-31 |           0.2 |         7.0 | 0.875684  0.859162  0.0                   0.0 | 0.2     0.2  0.0                      0.2 |  0.0      0.0  0.0          0.0 |  31.5   31.5  0.0               31.5 |  31.5    31.5  0.0      31.5 |  31.5     31.5  0.0     31.5 | 0.0  0.0  0.0           31.5 | 13.917106  13.143815  0.0          0.0 | 2.316867  3.131125  0.0                     0.0 | 0.0  0.0  3.3912              0.0 | 2.516867  3.331125  3.3912                      0.2 | 0.080377  0.095336  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 13.836729  13.048479  -3.1912                0.0 | 2.767346  2.609696      0.0       0.0 |       4.579241 |        0.952056 | 0.0     0.0  0.0               0.0 |  46.15  32.8985  0.0     36.55 | 87.568449  85.916215  0.0           0.0 |          0.0 |  4.300111 |
    | 2017-04-01 |           4.5 |         3.8 | 0.927868  0.921345  0.0                   0.0 | 0.8     2.0  0.0                      1.5 |  3.7      2.5  0.0          2.5 |  17.1   17.1  0.0               17.1 |  17.1    17.1  0.0      17.1 |  20.8     19.6  0.0     19.6 | 0.0  0.0  0.0           19.6 | 14.905504  13.389799  0.0          0.0 | 0.683373   0.00243  0.0                     0.0 | 0.0  0.0  1.9245              0.0 | 1.483373   2.00243  1.9245                      1.5 | 0.007205  0.010476  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 14.898299  13.379323   2.5755                0.0 |  2.97966  2.675865   0.5151       0.0 |       6.170624 |        1.898022 | 0.0     0.0  0.0               0.5 |  29.05  15.7985  0.0     19.45 | 92.786777  92.134462  0.0           0.0 |          0.0 |  8.572713 |
    | 2017-04-02 |           0.0 |         1.4 | 0.922874  0.912005  0.0                   0.0 | 0.0     0.0  0.0                      0.5 |  0.0      0.0  0.0          0.0 |   6.3    6.3  0.0                6.3 |   6.3     6.3  0.0       6.3 |   6.3      6.3  0.0      6.3 | 0.0  0.0  0.0            6.3 |   5.22066   5.129019  0.0          0.0 | 1.580155  2.106732  0.0                     0.0 | 0.0  0.0  2.0528              0.0 | 1.580155  2.106732  2.0528                      0.5 | 0.001407  0.001825  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  5.219252   5.127194  -2.0528                0.0 |  1.04385  1.025439      0.0       0.0 |       1.556089 |         1.83604 | 0.0     0.0  0.0               0.0 |  22.75   9.4985  0.0     13.15 | 92.287369  91.200536  0.0           0.0 |          0.0 |  8.292762 |
    | 2017-04-03 |           0.0 |         0.1 |  0.90398  0.886677  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  0.45   0.45  0.0               0.45 |  0.45    0.45  0.0      0.45 |  0.45     0.45  0.0     0.45 | 0.0  0.0  0.0           0.45 |  0.367887   0.357108  0.0          0.0 |  1.97321  2.628296  0.0                     0.0 | 0.0  0.0   2.566              0.0 |  1.97321  2.628296   2.566                      0.0 |  0.00172  0.002555  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.366166   0.354553   -2.566                0.0 | 0.073233  0.070911      0.0       0.0 |      -0.497356 |        1.413067 | 0.0     0.0  0.0               0.0 |   22.3   9.0485  0.0      12.7 | 90.397992  88.667687  0.0           0.0 |          0.0 |  6.382338 |
    | 2017-04-04 |           0.0 |        -0.1 | 0.883377   0.85929  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 2.063624  2.744142  0.0                     0.0 | 0.0  0.0  2.6943              0.0 | 2.063624  2.744142  2.6943                      0.0 |  0.00332  0.005457  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.00332  -0.005457  -2.6943                0.0 |      0.0       0.0      0.0       0.0 |      -0.675769 |        1.034425 | 0.0     0.0  0.0               0.0 |   22.3   9.0485  0.0      12.7 | 88.337688  85.929002  0.0           0.0 |          0.0 |  4.672144 |
    | 2017-04-05 |           0.4 |        -0.5 | 0.870722  0.841219  0.0                   0.0 | 0.4     0.4  0.0                      0.4 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.271483  1.817519  0.0                     0.0 | 0.0  0.0  2.1811              0.0 | 1.671483  2.217519  2.1811                      0.4 | 0.005948  0.010447  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.005948  -0.010447  -1.7811                0.0 |      0.0       0.0      0.0       0.0 |      -0.449374 |        0.765458 | 0.0     0.0  0.0               0.0 |   22.3   9.0485  0.0      12.7 | 87.072153  84.121931  0.0           0.0 |          0.0 |  3.457311 |
    | 2017-04-06 |           0.0 |        -0.8 | 0.854227  0.819414  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.657579  2.195586  0.0                     0.0 | 0.0  0.0  2.1811              0.0 | 1.657579  2.195586  2.1811                      0.0 | 0.008102  0.015012  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.008102  -0.015012  -2.1811                0.0 |      0.0       0.0      0.0       0.0 |      -0.551053 |        0.526815 | 0.0     0.0  0.0               0.0 |   22.3   9.0485  0.0      12.7 | 85.422677  81.941357  0.0           0.0 |          0.0 |  2.379443 |
    | 2017-04-07 |           0.0 |         0.1 | 0.848046  0.811139  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  0.45   0.45  0.0               0.45 |  0.45    0.45  0.0      0.45 |  0.45     0.45  0.0     0.45 | 0.0  0.0  0.0           0.45 |  0.303005   0.272955  0.0          0.0 | 0.776714  1.026604  0.0                     0.0 | 0.0  0.0  1.0264              0.0 | 0.776714  1.026604  1.0264                      0.0 | 0.011616  0.022084  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.291389   0.250871  -1.0264                0.0 | 0.058278  0.050174      0.0       0.0 |      -0.148148 |        0.404465 | 0.0     0.0  0.0               0.0 |  21.85   8.5985  0.0     12.25 | 84.804574  81.113881  0.0           0.0 |          0.0 |   1.82683 |
    | 2017-04-08 |           0.0 |         0.6 | 0.844725  0.805787  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   2.7    2.7  0.0                2.7 |   2.7     2.7  0.0       2.7 |   2.7      2.7  0.0      2.7 | 0.0  0.0  0.0            2.7 |  1.785191   1.596533  0.0          0.0 |  1.26005  1.663922  0.0                     0.0 | 0.0  0.0  1.6679              0.0 |  1.26005  1.663922  1.6679                      0.0 | 0.013157  0.025262  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  1.772033   1.571272  -1.6679                0.0 | 0.354407  0.314254      0.0       0.0 |       0.251686 |        0.376771 | 0.0     0.0  0.0               0.0 |  19.15   5.8985  0.0      9.55 | 84.472491  80.578687  0.0           0.0 |          0.0 |  1.701745 |
    | 2017-04-09 |           0.0 |         2.5 | 0.859502  0.798799  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 11.25  11.25  0.0              11.25 | 11.25  5.8985  0.0      9.55 | 11.25   5.8985  0.0     9.55 | 0.0  0.0  0.0           9.55 |  7.365396   3.430356  0.0          0.0 | 2.420952  3.194353  0.0                     0.0 | 0.0  0.0  3.2075              0.0 | 2.420952  3.194353  3.2075                      0.0 | 0.014039  0.027471  0.0            0.0 |       0.0        0.0  0.0                 0.0 |           0.0 |             0.0 |           0.0 |               0.0 | 0.0  0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  7.351357   3.402885  -3.2075                0.0 | 1.470271  0.680577      0.0       0.0 |       1.348973 |        0.553001 | 0.0     0.0  0.0               0.0 |    7.9      0.0  0.0       0.0 | 85.950181  79.879949  0.0           0.0 |          0.0 |  2.497717 |

>>> round_(model.check_waterbalance(conditions))
0.0
"""

import numpy

from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.exe.modelimports import *
from hydpy.interfaces import aetinterfaces

from hydpy.models.whmod import whmod_masks
from hydpy.models.whmod import whmod_model
from hydpy.models.whmod.whmod_constants import *


class Model(
    whmod_model.Main_AETModel_V1,
    whmod_model.Sub_TempModel_V1,
    whmod_model.Sub_PrecipModel_V1,
    whmod_model.Sub_IntercModel_V1,
    whmod_model.Sub_SoilWaterModel_V1,
    whmod_model.Sub_SnowCoverModel_V1,
):
    """|whmod_urban.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="WHMod-Urban", description="WHMod for urban regions"
    )
    __HYDPY_ROOTMODEL__ = True

    INLET_METHODS = ()
    OBSERVER_METHODS = ()
    RECEIVER_METHODS = ()
    ADD_METHODS = (
        whmod_model.Calc_InterceptionEvaporation_InterceptedWater_AETModel_V1,
        whmod_model.Calc_LakeEvaporation_AETModel_V1,
        whmod_model.Calc_SoilEvapotranspiration_AETModel_V1,
    )
    INTERFACE_METHODS = (
        whmod_model.Get_Temperature_V1,
        whmod_model.Get_MeanTemperature_V1,
        whmod_model.Get_Precipitation_V1,
        whmod_model.Get_InterceptedWater_V1,
        whmod_model.Get_SoilWater_V1,
        whmod_model.Get_SnowCover_V1,
    )
    RUN_METHODS = (
        whmod_model.Calc_Throughfall_InterceptedWater_V1,
        whmod_model.Calc_InterceptionEvaporation_InterceptedWater_V1,
        whmod_model.Calc_LakeEvaporation_V1,
        whmod_model.Calc_PotentialSnowmelt_V1,
        whmod_model.Calc_Snowmelt_Snowpack_V1,
        whmod_model.Calc_Ponding_V1,
        whmod_model.Calc_SurfaceRunoff_V1,
        whmod_model.Calc_RelativeSoilMoisture_V1,
        whmod_model.Calc_Percolation_V1,
        whmod_model.Calc_CisternInflow_V1,
        whmod_model.Calc_CisternOverflow_CisternWater_V1,
        whmod_model.Calc_SoilEvapotranspiration_V1,
        whmod_model.Calc_TotalEvapotranspiration_V1,
        whmod_model.Calc_CapillaryRise_V2,
        whmod_model.Calc_SoilMoisture_V1,
        whmod_model.Calc_RelativeSoilMoisture_V1,
        whmod_model.Calc_RequiredIrrigation_V1,
        whmod_model.Calc_CisternDemand_V1,
        whmod_model.Calc_CisternExtraction_CisternWater_V1,
        whmod_model.Calc_InternalIrrigation_SoilMoisture_V1,
        whmod_model.Calc_RelativeSoilMoisture_V1,
        whmod_model.Calc_ExternalIrrigation_SoilMoisture_V2,
        whmod_model.Calc_RelativeSoilMoisture_V1,
        whmod_model.Calc_PotentialRecharge_V2,
        whmod_model.Calc_Baseflow_V1,
        whmod_model.Calc_ActualRecharge_V1,
        whmod_model.Calc_DelayedRecharge_DeepWater_V1,
    )
    OUTLET_METHODS = ()
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (aetinterfaces.AETModel_V1,)
    SUBMODELS = ()

    aetmodel = modeltools.SubmodelProperty(aetinterfaces.AETModel_V1)

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            Error = \Sigma In - \Sigma Out - \Delta Vol
            \\ \\
            \Sigma In = \sum_{t=t_0}^{t_1} \left(
            P_t + \sum_{k=1}^{N} Z^k \cdot E_t^k \right)
            \\
            \Sigma Out = \sum_{t=t_0}^{t_1} \left( T_t + O_t / A +
            \sum_{k=1}^{N} Z^k \cdot \left( B_t^k + R_t^k +
            \begin{cases}
            S_t^k &|\ \overline{C} \\
            0 &|\ C
            \end{cases} \right) \right)
            \\
            \Delta Vol = f_{\Delta}\big(D\big) + f_{\Delta}\big(W\big) / A +
            \sum_{k=1}^{N} Z^k \cdot \Big(
            f_{\Delta} \big( I^k \big) + f_{\Delta} \big( P^k \big) +
            f_{\Delta} \big( M^k \big) \Big)
            \\ \\
            f_{\Delta}(x) = x_{t1} - x_{t0}
            \\ \\
            A = Area \cdot 1000\\
            N = NmbZones \\
            Z = ZoneRatio \\
            C = CisternSource \\
            P = Precipitation \\
            T = TotalEvapotranspiration \\
            S = SurfaceRunoff \\
            O = CisternOverflow \\
            B = Baseflow \\
            R = DelayedRecharge \\
            I = InterceptedWater \\
            P = Snowpack \\
            M = SoilMoisture \\
            W = CisternWater \\
            D = DeepWater

        The returned error should always be in scale with numerical precision so that
        it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run via
        property |Sequences.conditions|.  See the integration tests of the application
        model |whmod_urban| for some examples.
        """

        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["model"]["states"]
        a = self.parameters.control.area.value
        c = self.parameters.control.cisternsource.values
        r = self.parameters.derived.zoneratio.values
        return (
            sum(self.sequences.inputs.precipitation.series)
            - numpy.sum(r * fluxes.totalevapotranspiration.series)
            - numpy.sum((r * fluxes.surfacerunoff.series)[:, ~c])
            - numpy.sum(fluxes.cisternoverflow.series) / a * 1000.0
            + numpy.sum(r * fluxes.externalirrigation.series)
            - numpy.sum(r * fluxes.baseflow.series)
            - sum(fluxes.delayedrecharge.series)
            - sum(r * (last.interceptedwater - first["interceptedwater"]))
            - sum(r * (last.snowpack - first["snowpack"]))
            - sum(r * (last.soilmoisture - first["soilmoisture"]))
            - (last.cisternwater - first["cisternwater"]) / a * 1000.0
            - (last.deepwater - first["deepwater"])
        )


class Masks(masktools.Masks):
    """Masks applicable to |whmod_urban|."""

    CLASSES = whmod_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
