"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const lib_1 = require("../lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const lambda = require("aws-cdk-lib/aws-lambda");
const iam = require("aws-cdk-lib/aws-iam");
const core_1 = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-kinesisstreams-lambda';
const lambdaRole = new iam.Role(stack, 'test-role', {
    assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
    inlinePolicies: {
        LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
            statements: [new iam.PolicyStatement({
                    actions: [
                        'logs:CreateLogGroup',
                        'logs:CreateLogStream',
                        'logs:PutLogEvents'
                    ],
                    resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                })]
        })
    }
});
const lambdaFn = new lambda.Function(stack, 'test-fn', {
    runtime: lambda.Runtime.NODEJS_16_X,
    handler: 'index.handler',
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    role: lambdaRole,
});
const cfnLambdafunction = lambdaFn.node.findChild('Resource');
cfnLambdafunction.cfnOptions.metadata = {
    cfn_nag: {
        rules_to_suppress: [{
                id: 'W89',
                reason: `This is not a rule for the general case, just for specific use cases/industries`
            },
            {
                id: 'W92',
                reason: `Impossible for us to define the correct concurrency for clients`
            }]
    }
};
const stream = new kinesis.Stream(stack, 'test-stream', {
    shardCount: 2,
    encryption: kinesis.StreamEncryption.MANAGED
});
// Definitions
const props = {
    existingStreamObj: stream,
    existingLambdaObj: lambdaFn,
    kinesisEventSourceProps: {
        startingPosition: lambda.StartingPosition.LATEST,
        batchSize: 1
    },
};
new lib_1.KinesisStreamsToLambda(stack, 'test-ks-lambda', props);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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