"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = void 0;
const cxapi = require("@aws-cdk/cx-api");
const tree_metadata_1 = require("./private/tree-metadata");
const stage_1 = require("./stage");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps.html
 */
class App extends stage_1.Stage {
    /**
     * Checks if an object is an instance of the `App` class.
     *
     * @param obj The object to evaluate.
     * @returns `true` if `obj` is an `App`.
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Initializes a CDK application.
     *
     * @param props initialization properties.
     */
    constructor(props = {}) {
        var _a, _b;
        super(undefined, '', {
            outdir: (_a = props.outdir) !== null && _a !== void 0 ? _a : process.env[cxapi.OUTDIR_ENV],
        });
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        const analyticsReporting = (_b = props.analyticsReporting) !== null && _b !== void 0 ? _b : props.runtimeInfo;
        if (analyticsReporting !== undefined) {
            this.node.setContext(cxapi.ANALYTICS_REPORTING_ENABLED_CONTEXT, analyticsReporting);
        }
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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