"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerVolumeConsistency = exports.BundlingDockerImage = void 0;
const child_process_1 = require("child_process");
const fs_1 = require("./fs");
/**
 * A Docker image used for asset bundling.
 */
class BundlingDockerImage {
    /** @param image The Docker image */
    constructor(image, _imageHash) {
        this.image = image;
        this._imageHash = _imageHash;
    }
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name.
     */
    static fromRegistry(image) {
        return new BundlingDockerImage(image);
    }
    /**
     * Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file.
     * @param options Docker build options.
     */
    static fromAsset(path, options = {}) {
        const buildArgs = options.buildArgs || {};
        const dockerArgs = [
            'build', '-q',
            ...(options.file ? ['-f', options.file] : []),
            ...flatten(Object.entries(buildArgs).map(([k, v]) => ['--build-arg', `${k}=${v}`])),
            path,
        ];
        const docker = dockerExec(dockerArgs);
        const match = docker.stdout.toString().match(/sha256:([a-z0-9]+)/);
        if (!match) {
            throw new Error('Failed to extract image ID from Docker build output');
        }
        // Fingerprints the directory containing the Dockerfile we're building and
        // differentiates the fingerprint based on build arguments. We do this so
        // we can provide a stable image hash. Otherwise, the image ID will be
        // different every time the Docker layer cache is cleared, due primarily to
        // timestamps.
        const hash = fs_1.FileSystem.fingerprint(path, { extraHash: JSON.stringify(options) });
        return new BundlingDockerImage(match[1], hash);
    }
    /**
     * Provides a stable representation of this image for JSON serialization.
     *
     * @returns The overridden image name if set or image hash name in that order
     */
    toJSON() {
        var _a;
        return (_a = this._imageHash) !== null && _a !== void 0 ? _a : this.image;
    }
    /**
     * Runs a Docker image.
     */
    run(options = {}) {
        const volumes = options.volumes || [];
        const environment = options.environment || {};
        const command = options.command || [];
        const dockerArgs = [
            'run', '--rm',
            ...options.user
                ? ['-u', options.user]
                : [],
            ...flatten(volumes.map(v => { var _a; return ['-v', `${v.hostPath}:${v.containerPath}:${(_a = v.consistency) !== null && _a !== void 0 ? _a : DockerVolumeConsistency.DELEGATED}`]; })),
            ...flatten(Object.entries(environment).map(([k, v]) => ['--env', `${k}=${v}`])),
            ...options.workingDirectory
                ? ['-w', options.workingDirectory]
                : [],
            this.image,
            ...command,
        ];
        dockerExec(dockerArgs, {
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
        });
    }
    /**
     * Copies a file or directory out of the Docker image to the local filesystem.
     */
    cp(imagePath, outputPath) {
        const { stdout } = dockerExec(['create', this.image]);
        const match = stdout.toString().match(/([0-9a-f]{16,})/);
        if (!match) {
            throw new Error('Failed to extract container ID from Docker create output');
        }
        const containerId = match[1];
        const containerPath = `${containerId}:${imagePath}`;
        try {
            dockerExec(['cp', containerPath, outputPath]);
        }
        catch (err) {
            throw new Error(`Failed to copy files from ${containerPath} to ${outputPath}: ${err}`);
        }
        finally {
            dockerExec(['rm', '-v', containerId]);
        }
    }
}
exports.BundlingDockerImage = BundlingDockerImage;
/**
 * Supported Docker volume consistency types.
 *
 * Only valid on macOS due to the way file storage works on Mac
 */
var DockerVolumeConsistency;
(function (DockerVolumeConsistency) {
    DockerVolumeConsistency["CONSISTENT"] = "consistent";
    DockerVolumeConsistency["DELEGATED"] = "delegated";
    DockerVolumeConsistency["CACHED"] = "cached";
})(DockerVolumeConsistency = exports.DockerVolumeConsistency || (exports.DockerVolumeConsistency = {}));
function flatten(x) {
    return Array.prototype.concat([], ...x);
}
function dockerExec(args, options) {
    var _a, _b, _c;
    const prog = (_a = process.env.CDK_DOCKER) !== null && _a !== void 0 ? _a : 'docker';
    const proc = child_process_1.spawnSync(prog, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_b = proc.stdout) === null || _b === void 0 ? void 0 : _b.toString().trim()}\n\n\nstderr: ${(_c = proc.stderr) === null || _c === void 0 ? void 0 : _c.toString().trim()}`);
        }
        throw new Error(`${prog} exited with status ${proc.status}`);
    }
    return proc;
}
//# sourceMappingURL=data:application/json;base64,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