"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApp = void 0;
const constructs_1 = require("constructs");
const cfn_resource_1 = require("../cfn-resource");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetCfnResources = findCfnResources(dependency.target);
        const sourceCfnResources = findCfnResources(dependency.source);
        for (const target of targetCfnResources) {
            for (const source of sourceCfnResources) {
                source.addDependsOn(target);
            }
        }
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare its template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    const includeStack = (stack) => {
        if (!stack_1.Stack.isStack(stack)) {
            return false;
        }
        if (!stack.nested) {
            return false;
        }
        // test: if we are not within a stage, then include it.
        if (!stage_1.Stage.of(stack)) {
            return true;
        }
        return stage_1.Stage.of(stack) === root;
    };
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (includeStack(stack)) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findCfnResources(root) {
    return root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource);
}
//# sourceMappingURL=data:application/json;base64,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