"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildSchema = exports.cleanUpApiNameForNamespace = exports.validatePolicy = exports.checkParseTemplate = exports.checkParsePolicy = exports.checkParseSchema = void 0;
const cedar = require("@cedar-policy/cedar-wasm/nodejs");
function checkParseSchema(schemaStr) {
    const schemaParseResult = cedar.checkParseSchema(schemaStr);
    if (schemaParseResult.type === 'error') {
        throw new Error(`Schema is invalid: ${schemaParseResult.errors}`);
    }
}
exports.checkParseSchema = checkParseSchema;
function checkParsePolicy(policyStatement) {
    const parsePolicyResult = cedar.checkParsePolicySet(policyStatement);
    if (parsePolicyResult.type == 'error') {
        throw new Error(`Invalid policy statement: ${policyStatement}. Errors: ${parsePolicyResult.errors.join(', ')}`);
    }
}
exports.checkParsePolicy = checkParsePolicy;
function checkParseTemplate(templateStatement) {
    const templateParseResult = cedar.checkParseTemplate(templateStatement);
    if (templateParseResult.type === 'error') {
        throw new Error(`Error parsing template: ${templateStatement}. Errors: ${templateParseResult.errors.join(', ')}`);
    }
}
exports.checkParseTemplate = checkParseTemplate;
function validatePolicy(policyStatement, schemaStr) {
    const validationResult = cedar.validate({
        validationSettings: { enabled: 'on' },
        schema: {
            json: JSON.parse(schemaStr),
        },
        policySet: policyStatement,
    });
    if (validationResult.type === 'failure') {
        throw new Error(`Policy (contents: ${policyStatement}) could not be parsed: ${validationResult.errors.join('; ')}`);
    }
    if (validationResult.type === 'success' && validationResult.validationErrors.length > 0) {
        throw new Error(`Policy (contents: ${policyStatement}) could not be validated against the schema:
      ${validationResult.validationErrors.join('; ')}`);
    }
}
exports.validatePolicy = validatePolicy;
function cleanUpApiNameForNamespace(apiName) {
    const validCedarName = apiName.replace(/[^a-zA-Z0-9_]/g, '').trim();
    if (validCedarName.length === 0) {
        return 'ImportedApi';
    }
    if (/[0-9_]/.exec(validCedarName[0])) {
        return `Api${validCedarName}`;
    }
    return validCedarName;
}
exports.cleanUpApiNameForNamespace = cleanUpApiNameForNamespace;
function buildSchema(namespace, actionNames, principalGroupType) {
    const additionalEntities = {};
    if (principalGroupType) {
        additionalEntities[principalGroupType] = {
            shape: {
                type: 'Record',
                attributes: {},
            },
        };
    }
    const actions = actionNames.reduce((acc, actionName) => {
        return {
            ...acc,
            [actionName]: {
                appliesTo: {
                    context: { type: 'Record', attributes: {} },
                    principalTypes: ['User'],
                    resourceTypes: ['Application'],
                },
            },
        };
    }, {});
    return {
        [namespace]: {
            entityTypes: {
                ...additionalEntities,
                User: {
                    shape: {
                        type: 'Record',
                        attributes: {},
                    },
                    memberOfTypes: principalGroupType ? [principalGroupType] : [],
                },
                Application: {
                    shape: {
                        type: 'Record',
                        attributes: {},
                    },
                },
            },
            actions,
        },
    };
}
exports.buildSchema = buildSchema;
//# sourceMappingURL=data:application/json;base64,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