"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStore = exports.ValidationSettingsMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
const policy_1 = require("./policy");
const permissions_1 = require("./private/permissions");
/**
 * Validation Settings mode, according to the Cloudformation PolicyStore resource
 */
var ValidationSettingsMode;
(function (ValidationSettingsMode) {
    ValidationSettingsMode["OFF"] = "OFF";
    ValidationSettingsMode["STRICT"] = "STRICT";
})(ValidationSettingsMode || (exports.ValidationSettingsMode = ValidationSettingsMode = {}));
class PolicyStoreBase extends core_1.Resource {
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.policyStoreArn],
            scope: this,
        });
    }
    grantAuth(grantee) {
        return this.grant(grantee, ...permissions_1.AUTH_ACTIONS);
    }
    grantRead(grantee) {
        return this.grant(grantee, ...permissions_1.READ_ACTIONS);
    }
    grantWrite(grantee) {
        return this.grant(grantee, ...permissions_1.WRITE_ACTIONS.concat(permissions_1.READ_ACTIONS));
    }
}
class PolicyStore extends PolicyStoreBase {
    /**
     * Create a PolicyStore construct that represents an external policy store via policy store id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreId The PolicyStore's id.
     */
    static fromPolicyStoreId(scope, id, policyStoreId) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreId });
    }
    /**
     * Create a PolicyStore construct that represents an external PolicyStore via policy store arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreArn The PolicyStore's ARN.
     */
    static fromPolicyStoreArn(scope, id, policyStoreArn) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreArn });
    }
    /**
     * Creates a PolicyStore construct that represents an external Policy Store.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `PolicyStoreAttributes` object.
     */
    static fromPolicyStoreAttributes(scope, id, attrs) {
        class Import extends PolicyStoreBase {
            constructor(policyStoreArn, policyStoreId) {
                super(scope, id);
                this.policyStoreArn = policyStoreArn;
                this.policyStoreId = policyStoreId;
            }
        }
        let policyStoreId;
        let policyStoreArn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.policyStoreId) {
            if (!attrs.policyStoreArn) {
                throw new Error('One of policyStoreId or policyStoreArn is required!');
            }
            policyStoreArn = attrs.policyStoreArn;
            const maybeId = stack.splitArn(attrs.policyStoreArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeId) {
                throw new Error(`ARN for PolicyStore must be in the form: ${core_1.ArnFormat.SLASH_RESOURCE_NAME}`);
            }
            policyStoreId = maybeId;
        }
        else {
            if (attrs.policyStoreArn) {
                throw new Error('Only one of policyStoreArn or policyStoreId can be provided');
            }
            policyStoreId = attrs.policyStoreId;
            policyStoreArn = stack.formatArn({
                resource: 'policy-store',
                resourceName: attrs.policyStoreId,
                service: 'verifiedpermissions',
            });
        }
        return new Import(policyStoreArn, policyStoreId);
    }
    /**
     * This method generates a schema based on an swagger file. It makes the same assumptions and decisions
     * made in the Amazon Verified Permissions console. This feature is built for swagger files generated from an Amazon API Gateway
     * export. It's possible that some swagger files generated by other tools will not work. In that case, please
     * file an issue.
     * @param swaggerFilePath absolute path to a swagger file in the local directory structure, in json format
     * @param groupEntityTypeName optional parameter to specify the group entity type name. If passed, the schema's User type will have a parent of this type.
     */
    static schemaFromOpenApiSpec(swaggerFilePath, groupEntityTypeName) {
        const RELEVANT_HTTP_METHODS = ['get', 'post', 'put', 'patch', 'delete', 'head'];
        const openApiSpecString = fs.readFileSync(swaggerFilePath, 'utf-8');
        const openApiSpec = JSON.parse(openApiSpecString);
        if (!openApiSpec.paths) {
            throw new Error('Invalid OpenAPI spec - missing paths object');
        }
        const namespace = (0, cedar_helpers_1.cleanUpApiNameForNamespace)(openApiSpec.info.title);
        const pathUrls = Object.keys(openApiSpec.paths);
        const actionNames = [];
        for (const pathUrl of pathUrls) {
            const pathDef = openApiSpec.paths[pathUrl];
            if (!pathDef) {
                continue;
            }
            let pathVerbs = Object.keys(pathDef);
            if (pathVerbs.includes('x-amazon-apigateway-any-method')) {
                pathVerbs = RELEVANT_HTTP_METHODS;
            }
            for (const httpVerb of pathVerbs) {
                if (!RELEVANT_HTTP_METHODS.includes(httpVerb)) {
                    continue;
                }
                const actionName = `${httpVerb} ${pathUrl}`;
                actionNames.push(actionName);
            }
        }
        return (0, cedar_helpers_1.buildSchema)(namespace, actionNames, groupEntityTypeName);
    }
    constructor(scope, id, props = {
        validationSettings: {
            mode: ValidationSettingsMode.OFF,
        },
    }) {
        super(scope, id);
        if (props.schema) {
            (0, cedar_helpers_1.checkParseSchema)(props.schema.cedarJson);
        }
        this.policyStore = new aws_verifiedpermissions_1.CfnPolicyStore(this, id, {
            schema: props.schema
                ? {
                    cedarJson: props.schema.cedarJson,
                }
                : undefined,
            validationSettings: props.validationSettings,
            description: props.description,
        });
        this.policyStoreArn = this.getResourceArnAttribute(this.policyStore.attrArn, {
            resource: 'policy-store',
            resourceName: this.physicalName,
            service: 'verifiedpermissions',
        });
        this.policyStoreName = this.getResourceNameAttribute(this.policyStore.ref);
        this.policyStoreId = this.policyStore.attrPolicyStoreId;
        this.schema = props.schema;
        this.validationSettings = props.validationSettings;
        this.description = props.description;
    }
    /**
     * Add multiple policies to the policy store
     *
     * @param policyDefinitions An array of policy options for the policy stores policies.
     * @returns An array of created policy constructs.
     */
    addPolicies(policyDefinitions) {
        let policies = policyDefinitions.map((policyOption) => {
            return new policy_1.Policy(this, policyOption.policyId, {
                policyStore: this,
                definition: policyOption.policyConfiguration,
            });
        });
        return policies;
    }
    /**
     * Takes in an absolute path to a directory containing .cedar files and adds the contents of each
     * .cedar file as policies to this policy store. Parses the policies with cedar-wasm and, if the policy store has a schema,
     * performs semantic validation of the policies as well.
     * @param absolutePath a string representing an absolute path to the directory containing your policies
     * @returns An array of created Policy constructs.
     */
    addPoliciesFromPath(absolutePath) {
        if (!fs.statSync(absolutePath).isDirectory()) {
            throw new Error(`The path ${absolutePath} does not appear to be a directory`);
        }
        const policyFileNames = fs
            .readdirSync(absolutePath)
            .map((f) => path.join(absolutePath, f))
            .filter((f) => !fs.statSync(f).isDirectory() && f.endsWith('.cedar'));
        if (this.validationSettings.mode === ValidationSettingsMode.STRICT) {
            if (!this.schema) {
                throw new Error('A schema must exist when adding policies to a policy store with strict validation mode.');
            }
            for (const policyFile of policyFileNames) {
                const policyStatement = fs.readFileSync(policyFile, 'utf-8');
                (0, cedar_helpers_1.validatePolicy)(policyStatement, this.schema.cedarJson);
            }
        }
        const policies = policyFileNames.map((cedarFile) => policy_1.Policy.fromFile(this, cedarFile, {
            path: cedarFile,
            policyStore: this,
        }));
        return policies;
    }
}
exports.PolicyStore = PolicyStore;
_a = JSII_RTTI_SYMBOL_1;
PolicyStore[_a] = { fqn: "@cdklabs/cdk-verified-permissions.PolicyStore", version: "0.0.5" };
//# sourceMappingURL=data:application/json;base64,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