import contextlib
import datetime
import io
import logging
import os
import pathlib
import re
import sys
import tempfile
import typing
from logging.handlers import RotatingFileHandler

import click
import humanize
from rich.console import Console, Group
from rich.panel import Panel
from rich.syntax import Syntax

from ideas import constants
from ideas.constants import EXIT_STATUS

ERROR_LOG_MAX_FILE_SIZE = 2 * 1024 * 1024  # in bytes
CONSOLE_LOGGER_NAME = "ideas-cli"
MAX_PANEL_WIDTH = 120

console = Console()
logger = logging.getLogger(__name__)


def pretty_print(obj):
    """
    Will print (or otherwise handle) output, dumping anything supporting JSON
    serialization.
    """
    return console.print_json(
        data=obj,
        indent=4,
        # Prevent the JSON serializer from escaping non-ascii characters
        # https://docs.python.org/3/library/json.html#character-encodings
        ensure_ascii=False,
    )


def bash(msg):
    """Helper function to generate rich Syntax object for bash"""
    return Syntax(msg, "bash")


def show_message(
    *args, title: typing.Optional[str] = None, border_color: str = "white", **kwargs
):
    """
    Show a message to the console.
    Try to print the message in a panel box, but if the message does not fit a certain width,
    do not print within the panel box to avoid terminal resizing issues.
    """

    def ascii_line_width(s, tab_width=4):
        # copied from https://stackoverflow.com/questions/61957429/is-it-possible-to-get-number-of-printable-characters-in-string
        if "\n" in s:
            raise ValueError("String contains mutliple lines.")
        ansi_be_gone = re.compile(r"(?:\x1B[@-_]|[\x80-\x9F])[0-?]*[ -/]*[@-~]")
        s = ansi_be_gone.sub("", s)
        return len(s) + s.count("\t") * (tab_width - 1)

    # print as a group if multiple message components
    msg = Group(*args) if len(args) > 1 else args[0]
    panel = Panel(msg, title=title, border_style=border_color, expand=False, **kwargs)

    # first capture the output if printed as a panel
    with console.capture() as capture:
        console.print(panel)

    # see the max terminal char count for the panel
    # if it exceeds a certain max count, just print without a panel
    lines = capture.get().split("\n")
    panel_width = ascii_line_width(lines[0])
    if panel_width > MAX_PANEL_WIDTH:
        console.print(f"\n[{border_color}]{title}:[/{border_color}]")
        console.print(msg)
    else:
        console.print()
        console.print(panel)


def show_error(
    *args,
    title: typing.Optional[str] = None,
    **kwargs,
):
    """
    Prints an angry-looking user-facing error to the console display.
    """
    show_message(*args, title=title, border_color="bold red", **kwargs)


def show_success(*args, title: typing.Optional[str] = None, **kwargs):
    """
    Prints a pleasant, calming message of hope to the console display.
    """
    show_message(*args, title=title, border_color="bold green", **kwargs)


def show_warning(message: str):
    """
    Prints an alert user-facing warning to the console display.
    """
    console.print(f"[bold dark_orange]Warning: {message}[/bold dark_orange]")


def get_error_log_path():
    error_log_path = os.getenv("IDEAS_CLI_ERROR_LOG_FILEPATH")
    if error_log_path is None:
        error_log_path = pathlib.Path(tempfile.gettempdir()) / "ideas-cli-error.log"
    return error_log_path


def setup_logging(
    debug: bool,
    error_log_filepath: typing.Optional[str],
    error_log_max_file_size: int = ERROR_LOG_MAX_FILE_SIZE,
) -> None:
    """
    Sends all logging to stderr and file to prevent the default StreamHandler from messing up the
    JSON we send to stdout.

    If the error_log_filepath is not specified (i.e., by the user), we log to a file in /tmp or
    similar, to avoid dropping log files in the current directory all over the place.

    TODO
    ----
    - add separate logger for the file_progress.txt file.
    - set PYTHONUNBUFFERED=1

    https://inscopix.atlassian.net/browse/ID-1929
    """
    if error_log_filepath is None:
        error_log_filepath = get_error_log_path()

    # Change root logger level from WARNING (default) to NOTSET in order for all messages to be delegated.
    logging.getLogger().setLevel(logging.NOTSET)

    # These libraries are super noisy, and it's not useful for debugging in most
    # cases
    for external_library in ("boto3", "botocore", "urllib3"):
        logging.getLogger(external_library).setLevel(logging.CRITICAL)

    # Create one StreamHandler that writes to stderr (to avoid messing with stdout JSON output) and one
    # FileHandler with a max size.
    log_handlers = [
        # logging.StreamHandler(stream=sys.stderr),
        RotatingFileHandler(
            filename=error_log_filepath, maxBytes=error_log_max_file_size, backupCount=1
        ),
    ]

    # Only enable DEBUG level logging if `debug` was set
    if debug:
        log_level = logging.DEBUG
    else:
        log_level = logging.INFO

    for log_handler in log_handlers:
        log_handler.setLevel(log_level)

    logging.basicConfig(
        format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
        datefmt="%Y-%m-%d %H:%M:%S",
        handlers=[*log_handlers],
    )

    setup_console_logger()

    # print title to indicate start of new log session
    logger = logging.getLogger()
    logger.info("Starting IDEAS CLI Session")


def setup_console_logger():
    """
    Sends all logging to stderr and file to prevent the default StreamHandler from messing up the
    JSON we send to stdout.

    If the error_log_filepath is not specified (i.e., by the user), we log to a file in /tmp or
    similar, to avoid dropping log files in the current directory all over the place.

    TODO
    ----
    - add separate logger for the file_progress.txt file.
    - set PYTHONUNBUFFERED=1

    https://inscopix.atlassian.net/browse/ID-1929
    """
    # Change logger level from WARNING (default) to NOTSET in order for all messages to be delegated.
    logger = logging.getLogger(CONSOLE_LOGGER_NAME)

    stdout_handler = logging.StreamHandler(sys.stdout)
    stdout_handler.setLevel(logging.INFO)

    logger.addHandler(stdout_handler)


def get_console_logger():
    return logging.getLogger(CONSOLE_LOGGER_NAME)


def abort(
    *args,
    title: typing.Optional[str] = "Unexpected Exception",
    log: typing.Optional[str] = "Unhandled exception",
    exit_code: EXIT_STATUS = EXIT_STATUS.UNSPECIFIED_ERROR,
) -> typing.NoReturn:
    """
    Display an optional error to the user before exiting with a specific status code.
    """
    logger.exception(log)

    if not args:
        # only print the exception to the console if it's an unexpected exception
        console.print_exception()
        show_error(
            "An unexpected error occured.",
            f"\nPlease contact support: [bold cyan]{constants.SUPPORT_EMAIL}[/bold cyan]",
            "\nError trace shown above ^^^",
            "\nQuickly view the latest logs with the following command:",
            bash("ideas log"),
            title=title,
        )
    else:
        show_error(*args, title=title)

    sys.exit(exit_code.value)


def log_to_progress_file(
    upload_file: io.TextIOWrapper, file_size: int, uploaded_bytes: int
) -> int:
    """
    Logs a line to the open file-handle representing the upload progress, like so:

        2023-06-15 13:14:29 - 3%
        2023-06-15 13:14:30 - 6%
        2023-06-15 13:14:30 - 10%
        2023-06-15 13:14:30 - 13%
        ...

    Returns the integer percentage completion.
    """
    percentage = min(100, int(uploaded_bytes / file_size * 100))

    upload_file.write(
        f"{datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - {percentage}% \n"
    )

    return percentage


def progress_bar(
    file_size: int, label: str, max_label_length: int, upload_file: io.TextIOWrapper
):
    readable_total_size = humanize.naturalsize(file_size)

    def label_format(updated: typing.Optional[tuple[int, int]]) -> str:
        """
        Return a human-readable form of the progress of a file upload, in terms
        of bytes uploaded versus total size.

        Called by click.progressbar on each iteration.
        """
        if updated is None:
            uploaded_bytes = 0
        else:
            _, uploaded_bytes = updated

        readable_current_size = humanize.naturalsize(uploaded_bytes, format="%.2f")
        log_to_progress_file(upload_file, file_size, uploaded_bytes)

        return f"|  {readable_current_size} / {readable_total_size}"

    # TODO the eta is very inaccurate to start when resuming a file download: this is because click
    # calculates the ETA based on the start time over the progress so far. When we are resuming a
    # file upload, this means the ETA is super low until we upload enough segments to become more
    # accurate.
    #
    # Unfortunately, click doesn't have an easy way to fix this or reset the ETA.
    return click.progressbar(
        length=file_size,
        # To keep things looking pretty when we display multiple progress bars, we pad the filename
        # label to the maximum length of a label (which corresponds to the maximum filename length).
        #
        # TODO: consider eliding this text with an ellipsis if it gets extremely long
        label=label.ljust(max_label_length),
        file=sys.stderr,
        item_show_func=label_format,
    )


def progress_log(
    file_size: int, label: str, log_file: io.TextIOWrapper, operation: str
):
    """
    Log progress updates to stderr, useful if stderr isn't a TTY.
    """

    class ProgressLogger:
        def __init__(self):
            self.percentage = 0

        def update(
            self, chunk_size: int, updated: typing.Optional[tuple[int, int]]
        ) -> None:
            if updated is None:
                return

            part, uploaded_bytes = updated
            percentage = log_to_progress_file(log_file, file_size, uploaded_bytes)
            self.percentage += percentage

            click.echo(
                f"{operation.capitalize()}ing part {part} size={chunk_size} bytes. Percentage {operation}ed={percentage}",
                err=True,
            )

    @contextlib.contextmanager
    def progress_logger(operation: str):
        click.echo(
            f"Starting {operation} of file {label} size={file_size} bytes", err=True
        )
        try:
            yield ProgressLogger()
        finally:
            pass

    return progress_logger(operation)


def file_progress(
    file_size: int,
    label: str,
    max_label_length: int,
    progress_file: io.TextIOWrapper,
    operation: str = "upload",
):
    """
    If stderr is a TTY, output a progress-bar for the file; otherwise, log a line to stderr each
    time we update the progress.

    Either way, returns a context manager that has one method: `update`. For example:

        with file_progress(4_000_000, 'myfile.isxd', 16) as progress:
            uploaded_bytes = 0
            for part_num in range(0, 4_000_000, 1_000_000):
                uploaded_bytes += 1_000_000

                progress.update(
                    1_000_000,  # how many bytes we uploaded this iteration
                    (
                        part_num,  # which part number this is
                        percentage,  # the float percentage of the total upload
                        uploaded_bytes,  # total uploaded bytes up-to-and-including this iteration
                    )
                )

        myfile.isxd     [####################################]  100%  |  4.00 MB / 4.0 MB

    We also log progress updates to the specified file handle at `progress_file`.
    """
    if sys.stderr.isatty():
        return progress_bar(file_size, label, max_label_length, progress_file)
    else:
        return progress_log(file_size, label, progress_file, operation)
