import math

def profit_simple(S0: float, ST: float, r: float, tau: float) -> float:
    """
    Calculate simple profit from holding a stock position.
    
    This function calculates the profit from buying a stock and holding it,
    accounting for the time value of money by compounding the initial investment.
    
    Args:
        S0: Initial stock price (purchase price)
        ST: Final stock price (sale price)
        r: Risk-free rate (as decimal, e.g., 0.05 for 5%)
        tau: Holding period in years
        
    Returns:
        Profit (positive) or loss (negative) from the investment
        
    Example:
        >>> profit_simple(100.0, 110.0, 0.05, 1.0)
        4.873...
    """
    return ST - S0 * math.exp(r * tau)

def profit_with_costs(S0: float, ST: float, r: float, tau: float, D_tau: float, C0: float) -> float:
    """
    Calculate profit from holding a stock position with transaction costs and dividends.
    
    This function accounts for dividends received and transaction costs,
    compounding the costs at the risk-free rate.
    
    Args:
        S0: Initial stock price (purchase price)
        ST: Final stock price (sale price)
        r: Risk-free rate (as decimal, e.g., 0.05 for 5%)
        tau: Holding period in years
        D_tau: Dividends received during the holding period
        C0: Transaction costs paid initially
        
    Returns:
        Net profit (positive) or loss (negative) including all costs and dividends
        
    Example:
        >>> profit_with_costs(100.0, 110.0, 0.05, 1.0, 2.0, 1.0)
        10.948...
    """
    return ST + D_tau - C0 * math.exp(r * tau)