"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
    get keyArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'kms',
            // aliasName already contains the '/'
            resource: this.aliasName,
        });
    }
    get keyId() {
        return this.aliasName;
    }
    addAlias(alias) {
        return this.aliasTargetKey.addAlias(alias);
    }
    addToResourcePolicy(statement, allowNoOp) {
        this.aliasTargetKey.addToResourcePolicy(statement, allowNoOp);
    }
    grant(grantee, ...actions) {
        return this.aliasTargetKey.grant(grantee, ...actions);
    }
    grantDecrypt(grantee) {
        return this.aliasTargetKey.grantDecrypt(grantee);
    }
    grantEncrypt(grantee) {
        return this.aliasTargetKey.grantEncrypt(grantee);
    }
    grantEncryptDecrypt(grantee) {
        return this.aliasTargetKey.grantEncryptDecrypt(grantee);
    }
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management
 * Service (AWS KMS). Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    constructor(scope, id, props) {
        let aliasName = props.aliasName;
        if (!core_1.Token.isUnresolved(aliasName)) {
            if (!aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                aliasName = REQUIRED_ALIAS_PREFIX + aliasName;
            }
            if (aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${aliasName}`);
            }
            if (aliasName.toLocaleLowerCase().startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${aliasName}`);
            }
            if (!aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error('Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-');
            }
        }
        super(scope, id, {
            physicalName: aliasName,
        });
        this.aliasTargetKey = props.targetKey;
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: this.physicalName,
            targetKeyId: this.aliasTargetKey.keyArn,
        });
        this.aliasName = this.getResourceNameAttribute(resource.aliasName);
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs the properties of the referenced KMS Alias
     */
    static fromAliasAttributes(scope, id, attrs) {
        // tslint:disable-next-line: class-name
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    generatePhysicalName() {
        return REQUIRED_ALIAS_PREFIX + super.generatePhysicalName();
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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