from typing import TYPE_CHECKING, Any, Protocol, Tuple, Union

from aiokafka import TopicPartition as AIOKafkaTopicPartition

from faststream.broker.message import StreamMessage

if TYPE_CHECKING:
    from aiokafka import ConsumerRecord


class ConsumerProtocol(Protocol):
    """A protocol for Kafka consumers."""

    async def commit(self) -> None: ...


class FakeConsumer:
    """A fake Kafka consumer."""

    async def commit(self) -> None:
        pass


FAKE_CONSUMER = FakeConsumer()


class KafkaMessage(
    StreamMessage[
        Union[
            "ConsumerRecord",
            Tuple["ConsumerRecord", ...],
        ]
    ]
):
    """Represents a Kafka message in the FastStream framework.

    This class extends `StreamMessage` and is specialized for handling Kafka ConsumerRecord objects.
    """

    def __init__(
        self,
        *args: Any,
        consumer: ConsumerProtocol,
        **kwargs: Any,
    ) -> None:
        super().__init__(*args, **kwargs)

        self.consumer = consumer

    async def nack(self) -> None:
        """Reject the Kafka message."""
        if not self.committed:
            topic_partition = AIOKafkaTopicPartition(
                self.raw_message.topic,  # type: ignore[union-attr]
                self.raw_message.partition,  # type: ignore[union-attr]
            )
            self.consumer.seek(  # type: ignore[attr-defined]
                partition=topic_partition,
                offset=self.raw_message.offset,  # type: ignore[union-attr]
            )
            await super().nack()


class KafkaAckableMessage(KafkaMessage):
    async def ack(self) -> None:
        """Acknowledge the Kafka message."""
        if not self.committed:
            await self.consumer.commit()
            await super().ack()
