"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const construct_1 = require("./construct");
const runtime_info_1 = require("./private/runtime-info");
const tree_metadata_1 = require("./private/tree-metadata");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps_and_stacks.html
 */
class App extends construct_1.Construct {
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        super(undefined, '');
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        // both are reverse logic
        this.runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? false : true;
        this.outdir = props.outdir || process.env[cxapi.OUTDIR_ENV];
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Synthesizes a cloud assembly for this app. Emits it to the directory
     * specified by `outdir`.
     *
     * @returns a `CloudAssembly` which can be used to inspect synthesized
     * artifacts such as CloudFormation templates and assets.
     */
    synth() {
        // we already have a cloud assembly, no-op for you
        if (this._assembly) {
            return this._assembly;
        }
        const assembly = construct_1.ConstructNode.synth(this.node, {
            outdir: this.outdir,
            runtimeInfo: this.runtimeInfo ? runtime_info_1.collectRuntimeInformation() : undefined
        });
        this._assembly = assembly;
        return assembly;
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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