"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const cx_api_1 = require("@aws-cdk/cx-api");
const construct_1 = require("../construct");
const stack_1 = require("../stack");
const tree_1 = require("../tree");
const FILE_PATH = 'tree.json';
/**
 * Construct that is automatically attached to the top-level `App`.
 * This generates, as part of synthesis, a file containing the construct tree and the metadata for each node in the tree.
 * The output is in a tree format so as to preserve the construct hierarchy.
 *
 * @experimental
 */
class TreeMetadata extends construct_1.Construct {
    constructor(scope) {
        super(scope, 'Tree');
    }
    synthesize(session) {
        const lookup = {};
        const visit = (construct) => {
            const children = construct.node.children.map((c) => {
                try {
                    return visit(c);
                }
                catch (e) {
                    this.node.addWarning(`Failed to render tree metadata for node [${c.node.id}]. Reason: ${e}`);
                    return undefined;
                }
            });
            const childrenMap = children
                .filter((child) => child !== undefined)
                .reduce((map, child) => Object.assign(map, { [child.id]: child }), {});
            const node = {
                id: construct.node.id || 'App',
                path: construct.node.path,
                children: Object.keys(childrenMap).length === 0 ? undefined : childrenMap,
                attributes: this.synthAttributes(construct)
            };
            lookup[node.path] = node;
            return node;
        };
        const tree = {
            version: 'tree-0.1',
            tree: visit(this.node.root),
        };
        const builder = session.assembly;
        fs.writeFileSync(path.join(builder.outdir, FILE_PATH), JSON.stringify(tree, undefined, 2), { encoding: 'utf-8' });
        builder.addArtifact('Tree', {
            type: cx_api_1.ArtifactType.CDK_TREE,
            properties: {
                file: FILE_PATH
            }
        });
    }
    synthAttributes(construct) {
        // check if a construct implements IInspectable
        function canInspect(inspectable) {
            return inspectable.inspect !== undefined;
        }
        const inspector = new tree_1.TreeInspector();
        // get attributes from the inspector
        if (canInspect(construct)) {
            construct.inspect(inspector);
            return stack_1.Stack.of(construct).resolve(inspector.attributes);
        }
        return undefined;
    }
}
exports.TreeMetadata = TreeMetadata;
//# sourceMappingURL=data:application/json;base64,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