"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const cx_api_1 = require("@aws-cdk/cx-api");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const assets_1 = require("./assets");
const construct_1 = require("./construct");
const context_provider_1 = require("./context-provider");
const asset_parameters_1 = require("./private/asset-parameters");
const cloudformation_lang_1 = require("./private/cloudformation-lang");
const logical_id_1 = require("./private/logical-id");
const resolve_1 = require("./private/resolve");
const uniqueid_1 = require("./private/uniqueid");
const STACK_SYMBOL = Symbol.for('@aws-cdk/core.Stack');
const VALID_STACK_NAME_REGEX = /^[A-Za-z][A-Za-z0-9-]*$/;
/**
 * A root construct which represents a single CloudFormation stack.
 */
class Stack extends construct_1.Construct {
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually a Program instance.
     * @param id The construct ID of this stack. If `stackName` is not explicitly
     * defined, this id (and any parent IDs) will be used to determine the
     * physical ID of the stack.
     * @param props Stack properties.
     */
    constructor(scope, id, props = {}) {
        // For unit test convenience parents are optional, so bypass the type check when calling the parent.
        super(scope, id);
        /**
         * Options for CloudFormation template (like version, transform, description).
         */
        this.templateOptions = {};
        /**
         * Other stacks this stack depends on
         */
        this._stackDependencies = new Set();
        /**
         * Lists all missing contextual information.
         * This is returned when the stack is synthesized under the 'missing' attribute
         * and allows tooling to obtain the context and re-synthesize.
         */
        this._missingContext = new Array();
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this._logicalIds = new logical_id_1.LogicalIDs();
        const { account, region, environment } = this.parseEnvironment(props.env);
        this.account = account;
        this.region = region;
        this.environment = environment;
        if (props.description !== undefined) {
            // Max length 1024 bytes
            // Typically 2 bytes per character, may be more for more exotic characters
            if (props.description.length > 512) {
                throw new Error(`Stack description must be <= 1024 bytes. Received description: '${props.description}'`);
            }
            this.templateOptions.description = props.description;
        }
        this._stackName = props.stackName !== undefined ? props.stackName : this.generateUniqueId();
        this.tags = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'aws:cdk:stack', props.tags);
        if (!VALID_STACK_NAME_REGEX.test(this.stackName)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${this.stackName}'`);
        }
        // the preferred behavior is to generate a unique id for this stack and use
        // it as the artifact ID in the assembly. this allows multiple stacks to use
        // the same name. however, this behavior is breaking for 1.x so it's only
        // applied under a feature flag which is applied automatically for new
        // projects created using `cdk init`.
        this.artifactId = this.node.tryGetContext(cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT)
            ? this.generateUniqueId()
            : this.stackName;
        this.templateFile = `${this.artifactId}.template.json`;
        this.templateUrl = lazy_1.Lazy.stringValue({ produce: () => this._templateUrl || '<unresolved>' });
    }
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     */
    static isStack(x) {
        return x !== null && typeof (x) === 'object' && STACK_SYMBOL in x;
    }
    /**
     * Looks up the first stack scope in which `construct` is defined. Fails if there is no stack up the tree.
     * @param construct The construct to start the search from.
     */
    static of(construct) {
        return _lookup(construct);
        function _lookup(c) {
            if (Stack.isStack(c)) {
                return c;
            }
            if (!c.node.scope) {
                throw new Error(`No stack could be identified for the construct at path ${construct.node.path}`);
            }
            return _lookup(c.node.scope);
        }
    }
    /**
     * Resolve a tokenized value in the context of the current stack.
     */
    resolve(obj) {
        return resolve_1.resolve(obj, {
            scope: this,
            prefix: [],
            resolver: cloudformation_lang_1.CLOUDFORMATION_TOKEN_RESOLVER,
            preparing: false
        });
    }
    /**
     * Convert an object, potentially containing tokens, to a JSON string
     */
    toJsonString(obj, space) {
        return cloudformation_lang_1.CloudFormationLang.toJSON(obj, space).toString();
    }
    /**
     * Indicate that a context key was expected
     *
     * Contains instructions which will be emitted into the cloud assembly on how
     * the key should be supplied.
     *
     * @param report The set of parameters needed to obtain the context
     */
    reportMissingContext(report) {
        this._missingContext.push(report);
    }
    /**
     * Rename a generated logical identities
     *
     * To modify the naming scheme strategy, extend the `Stack` class and
     * override the `createNamingScheme` method.
     */
    renameLogicalId(oldId, newId) {
        this._logicalIds.addRename(oldId, newId);
    }
    /**
     * Allocates a stack-unique CloudFormation-compatible logical identity for a
     * specific resource.
     *
     * This method is called when a `CfnElement` is created and used to render the
     * initial logical identity of resources. Logical ID renames are applied at
     * this stage.
     *
     * This method uses the protected method `allocateLogicalId` to render the
     * logical ID for an element. To modify the naming scheme, extend the `Stack`
     * class and override this method.
     *
     * @param element The CloudFormation element for which a logical identity is
     * needed.
     */
    getLogicalId(element) {
        const logicalId = this.allocateLogicalId(element);
        return this._logicalIds.applyRename(logicalId);
    }
    /**
     * Add a dependency between this stack and another stack
     */
    addDependency(stack, reason) {
        if (stack === this) {
            return;
        } // Can ignore a dependency on self
        reason = reason || 'dependency added using stack.addDependency()';
        const dep = stack.stackDependencyReasons(this);
        if (dep !== undefined) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`'${stack.node.path}' depends on '${this.node.path}' (${dep.join(', ')}). Adding this dependency (${reason}) would create a cyclic reference.`);
        }
        this._stackDependencies.add({ stack, reason });
        if (process.env.CDK_DEBUG_DEPS) {
            // tslint:disable-next-line:no-console
            console.error(`[CDK_DEBUG_DEPS] stack "${this.node.path}" depends on "${stack.node.path}" because: ${reason}`);
        }
    }
    /**
     * Return the stacks this stack depends on
     */
    get dependencies() {
        return Array.from(this._stackDependencies.values()).map(d => d.stack);
    }
    /**
     * The concrete CloudFormation physical stack name.
     *
     * This is either the name defined explicitly in the `stackName` prop or
     * allocated based on the stack's location in the construct tree. Stacks that
     * are directly defined under the app use their construct `id` as their stack
     * name. Stacks that are defined deeper within the tree will use a hashed naming
     * scheme based on the construct path to ensure uniqueness.
     *
     * If you wish to obtain the deploy-time AWS::StackName intrinsic,
     * you can use `Aws.stackName` directly.
     */
    get stackName() {
        return this._stackName;
    }
    /**
     * The partition in which this stack is defined
     */
    get partition() {
        // Always return a non-scoped partition intrinsic. These will usually
        // be used to construct an ARN, but there are no cross-partition
        // calls anyway.
        return cfn_pseudo_1.Aws.PARTITION;
    }
    /**
     * The Amazon domain suffix for the region in which this stack is defined
     */
    get urlSuffix() {
        // Since URL Suffix always follows partition, it is unscoped like partition is.
        return cfn_pseudo_1.Aws.URL_SUFFIX;
    }
    /**
     * The ID of the stack
     *
     * @example After resolving, looks like arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123
     */
    get stackId() {
        return new cfn_pseudo_1.ScopedAws(this).stackId;
    }
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     */
    get notificationArns() {
        return new cfn_pseudo_1.ScopedAws(this).notificationArns;
    }
    /**
     * Indicates if this is a nested stack, in which case `parentStack` will include a reference to it's parent.
     */
    get nested() {
        return this.parentStack !== undefined;
    }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    formatArn(components) {
        return arn_1.Arn.format(components, this);
    }
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN string to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    parseArn(arn, sepIfToken = '/', hasName = true) {
        return arn_1.Arn.parse(arn, sepIfToken, hasName);
    }
    /**
     * Returnst the list of AZs that are availability in the AWS environment
     * (account/region) associated with this stack.
     *
     * If the stack is environment-agnostic (either account and/or region are
     * tokens), this property will return an array with 2 tokens that will resolve
     * at deploy-time to the first two availability zones returned from CloudFormation's
     * `Fn::GetAZs` intrinsic function.
     *
     * If they are not available in the context, returns a set of dummy values and
     * reports them as missing, and let the CLI resolve them by calling EC2
     * `DescribeAvailabilityZones` on the target environment.
     */
    get availabilityZones() {
        // if account/region are tokens, we can't obtain AZs through the context
        // provider, so we fallback to use Fn::GetAZs. the current lowest common
        // denominator is 2 AZs across all AWS regions.
        const agnostic = token_1.Token.isUnresolved(this.account) || token_1.Token.isUnresolved(this.region);
        if (agnostic) {
            return this.node.tryGetContext(cxapi.AVAILABILITY_ZONE_FALLBACK_CONTEXT_KEY) || [
                cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.getAzs()),
                cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.getAzs())
            ];
        }
        const value = context_provider_1.ContextProvider.getValue(this, {
            provider: cxapi.AVAILABILITY_ZONE_PROVIDER,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        }).value;
        if (!Array.isArray(value)) {
            throw new Error(`Provider ${cxapi.AVAILABILITY_ZONE_PROVIDER} expects a list`);
        }
        return value;
    }
    addFileAsset(asset) {
        // assets are always added at the top-level stack
        if (this.parentStack) {
            return this.parentStack.addFileAsset(asset);
        }
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.FileAssetParameters(this.assetParameters, asset.sourceHash);
            const metadata = {
                path: asset.fileName,
                id: asset.sourceHash,
                packaging: asset.packaging,
                sourceHash: asset.sourceHash,
                s3BucketParameter: params.bucketNameParameter.logicalId,
                s3KeyParameter: params.objectKeyParameter.logicalId,
                artifactHashParameter: params.artifactHashParameter.logicalId,
            };
            this.node.addMetadata(cxapi.ASSET_METADATA, metadata);
        }
        const bucketName = params.bucketNameParameter.valueAsString;
        // key is prefix|postfix
        const encodedKey = params.objectKeyParameter.valueAsString;
        const s3Prefix = cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const s3Filename = cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const objectKey = `${s3Prefix}${s3Filename}`;
        const s3Url = `https://s3.${this.region}.${this.urlSuffix}/${bucketName}/${objectKey}`;
        return { bucketName, objectKey, s3Url };
    }
    addDockerImageAsset(asset) {
        if (this.parentStack) {
            return this.parentStack.addDockerImageAsset(asset);
        }
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.DockerImageAssetParameters(this.assetParameters, asset.sourceHash);
            const metadata = {
                id: asset.sourceHash,
                packaging: 'container-image',
                path: asset.directoryName,
                sourceHash: asset.sourceHash,
                imageNameParameter: params.imageNameParameter.logicalId,
                repositoryName: asset.repositoryName,
                buildArgs: asset.dockerBuildArgs,
                target: asset.dockerBuildTarget
            };
            this.node.addMetadata(cxapi.ASSET_METADATA, metadata);
        }
        // Parse repository name and tag from the parameter (<REPO_NAME>@sha256:<TAG>)
        // Example: cdk/cdkexampleimageb2d7f504@sha256:72c4f956379a43b5623d529ddd969f6826dde944d6221f445ff3e7add9875500
        const components = cfn_fn_1.Fn.split('@sha256:', params.imageNameParameter.valueAsString);
        const repositoryName = cfn_fn_1.Fn.select(0, components).toString();
        const imageSha = cfn_fn_1.Fn.select(1, components).toString();
        const imageUri = `${this.account}.dkr.ecr.${this.region}.${this.urlSuffix}/${repositoryName}@sha256:${imageSha}`;
        return {
            imageUri, repositoryName
        };
    }
    /**
     * Returns the naming scheme used to allocate logical IDs. By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * In order to make sure logical IDs are unique and stable, we hash the resource
     * construct tree path (i.e. toplevel/secondlevel/.../myresource) and add it as
     * a suffix to the path components joined without a separator (CloudFormation
     * IDs only allow alphanumeric characters).
     *
     * The result will be:
     *
     *   <path.join('')><md5(path.join('/')>
     *     "human"      "hash"
     *
     * If the "human" part of the ID exceeds 240 characters, we simply trim it so
     * the total ID doesn't exceed CloudFormation's 255 character limit.
     *
     * We only take 8 characters from the md5 hash (0.000005 chance of collision).
     *
     * Special cases:
     *
     * - If the path only contains a single component (i.e. it's a top-level
     *   resource), we won't add the hash to it. The hash is not needed for
     *   disamiguation and also, it allows for a more straightforward migration an
     *   existing CloudFormation template to a CDK stack without logical ID changes
     *   (or renames).
     * - For aesthetic reasons, if the last components of the path are the same
     *   (i.e. `L1/L2/Pipeline/Pipeline`), they will be de-duplicated to make the
     *   resulting human portion of the ID more pleasing: `L1L2Pipeline<HASH>`
     *   instead of `L1L2PipelinePipeline<HASH>`
     * - If a component is named "Default" it will be omitted from the path. This
     *   allows refactoring higher level abstractions around constructs without affecting
     *   the IDs of already deployed resources.
     * - If a component is named "Resource" it will be omitted from the user-visible
     *   path, but included in the hash. This reduces visual noise in the human readable
     *   part of the identifier.
     *
     * @param cfnElement The element for which the logical ID is allocated.
     */
    allocateLogicalId(cfnElement) {
        const scopes = cfnElement.node.scopes;
        const stackIndex = scopes.indexOf(cfnElement.stack);
        const pathComponents = scopes.slice(stackIndex + 1).map(x => x.node.id);
        return uniqueid_1.makeUniqueId(pathComponents);
    }
    /**
     * Validate stack name
     *
     * CloudFormation stack names can include dashes in addition to the regular identifier
     * character classes, and we don't allow one of the magic markers.
     *
     * @internal
     */
    _validateId(name) {
        if (name && !VALID_STACK_NAME_REGEX.test(name)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
    }
    /**
     * Prepare stack
     *
     * Find all CloudFormation references and tell them we're consuming them.
     *
     * Find all dependencies as well and add the appropriate DependsOn fields.
     */
    prepare() {
        const tokens = this.findTokens();
        // References (originating from this stack)
        for (const reference of tokens) {
            // skip if this is not a CfnReference
            if (!cfn_reference_1.CfnReference.isCfnReference(reference)) {
                continue;
            }
            const targetStack = Stack.of(reference.target);
            // skip if this is not a cross-stack reference
            if (targetStack === this) {
                continue;
            }
            // determine which stack should create the cross reference
            const factory = this.determineCrossReferenceFactory(targetStack);
            // if one side is a nested stack (has "parentStack"), we let it create the reference
            // since it has more knowledge about the world.
            const consumedValue = factory.prepareCrossReference(this, reference);
            // if the reference has already been assigned a value for the consuming stack, carry on.
            if (!reference.hasValueForStack(this)) {
                reference.assignValueForStack(this, consumedValue);
            }
        }
        // Resource dependencies
        for (const dependency of this.node.dependencies) {
            const theirStack = Stack.of(dependency.target);
            if (theirStack !== undefined && theirStack !== this && Stack.of(dependency.source) === this) {
                this.addDependency(theirStack, `"${dependency.source.node.path}" depends on "${dependency.target.node.path}"`);
            }
            else {
                for (const target of findResources([dependency.target])) {
                    for (const source of findResources([dependency.source])) {
                        source.addDependsOn(target);
                    }
                }
            }
        }
        if (this.tags.hasTags()) {
            this.node.addMetadata(cxapi.STACK_TAGS_METADATA_KEY, this.tags.renderTags());
        }
        if (this.parentStack) {
            // add the nested stack template as an asset
            const cfn = JSON.stringify(this._toCloudFormation());
            const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
            const parent = this.parentStack;
            const templateLocation = parent.addFileAsset({
                packaging: assets_1.FileAssetPackaging.FILE,
                sourceHash: templateHash,
                fileName: this.templateFile
            });
            // if bucketName/objectKey are cfn parameters from a stack other than the parent stack, they will
            // be resolved as cross-stack references like any other (see "multi" tests).
            this._templateUrl = `https://s3.${parent.region}.${parent.urlSuffix}/${templateLocation.bucketName}/${templateLocation.objectKey}`;
        }
    }
    synthesize(session) {
        const builder = session.assembly;
        // write the CloudFormation template as a JSON file
        const outPath = path.join(builder.outdir, this.templateFile);
        const text = JSON.stringify(this._toCloudFormation(), undefined, 2);
        fs.writeFileSync(outPath, text);
        // if this is a nested stack, do not emit it as a cloud assembly artifact (it will be registered as an s3 asset instead)
        if (this.nested) {
            return;
        }
        const deps = this.dependencies.map(s => s.artifactId);
        const meta = this.collectMetadata();
        // backwards compatibility since originally artifact ID was always equal to
        // stack name the stackName attribute is optional and if it is not specified
        // the CLI will use the artifact ID as the stack name. we *could have*
        // always put the stack name here but wanted to minimize the risk around
        // changes to the assembly manifest. so this means that as long as stack
        // name and artifact ID are the same, the cloud assembly manifest will not
        // change.
        const stackNameProperty = this.stackName === this.artifactId
            ? {}
            : { stackName: this.stackName };
        const properties = Object.assign({ templateFile: this.templateFile }, stackNameProperty);
        // add an artifact that represents this stack
        builder.addArtifact(this.artifactId, {
            type: cxapi.ArtifactType.AWS_CLOUDFORMATION_STACK,
            environment: this.environment,
            properties,
            dependencies: deps.length > 0 ? deps : undefined,
            metadata: Object.keys(meta).length > 0 ? meta : undefined,
        });
        for (const ctx of this._missingContext) {
            builder.addMissing(ctx);
        }
    }
    /**
     * Returns the CloudFormation template for this stack by traversing
     * the tree and invoking _toCloudFormation() on all Entity objects.
     *
     * @internal
     */
    _toCloudFormation() {
        if (this.templateOptions.transform) {
            // tslint:disable-next-line: max-line-length
            this.node.addWarning('This stack is using the deprecated `templateOptions.transform` property. Consider switching to `templateOptions.transforms`.');
            if (!this.templateOptions.transforms) {
                this.templateOptions.transforms = [];
            }
            if (this.templateOptions.transforms.indexOf(this.templateOptions.transform) === -1) {
                this.templateOptions.transforms.unshift(this.templateOptions.transform);
            }
        }
        const template = {
            Description: this.templateOptions.description,
            Transform: extractSingleValue(this.templateOptions.transforms),
            AWSTemplateFormatVersion: this.templateOptions.templateFormatVersion,
            Metadata: this.templateOptions.metadata
        };
        const elements = cfnElements(this);
        const fragments = elements.map(e => this.resolve(e._toCloudFormation()));
        // merge in all CloudFormation fragments collected from the tree
        for (const fragment of fragments) {
            merge(template, fragment);
        }
        // resolve all tokens and remove all empties
        const ret = this.resolve(template) || {};
        this._logicalIds.assertAllRenamesApplied();
        return ret;
    }
    /**
     * Exports a resolvable value for use in another stack.
     *
     * @returns a token that can be used to reference the value from the producing stack.
     */
    prepareCrossReference(sourceStack, reference) {
        const targetStack = Stack.of(reference.target);
        // Ensure a singleton "Exports" scoping Construct
        // This mostly exists to trigger LogicalID munging, which would be
        // disabled if we parented constructs directly under Stack.
        // Also it nicely prevents likely construct name clashes
        const exportsScope = targetStack.getCreateExportsScope();
        // Ensure a singleton CfnOutput for this value
        const resolved = targetStack.resolve(reference);
        const id = 'Output' + JSON.stringify(resolved);
        const exportName = targetStack.generateExportName(exportsScope, id);
        const output = exportsScope.node.tryFindChild(id);
        if (!output) {
            new cfn_output_1.CfnOutput(exportsScope, id, { value: token_1.Token.asString(reference), exportName });
        }
        // add a dependency on the producing stack - it has to be deployed before this stack can consume the exported value
        // if the producing stack is a nested stack (i.e. has a parent), the dependency is taken on the parent.
        const producerDependency = targetStack.parentStack ? targetStack.parentStack : targetStack;
        const consumerDependency = sourceStack.parentStack ? sourceStack.parentStack : sourceStack;
        consumerDependency.addDependency(producerDependency, `${sourceStack.node.path} -> ${reference.target.node.path}.${reference.displayName}`);
        // We want to return an actual FnImportValue Token here, but Fn.importValue() returns a 'string',
        // so construct one in-place.
        return new intrinsic_1.Intrinsic({ 'Fn::ImportValue': exportName });
    }
    getCreateExportsScope() {
        const exportsName = 'Exports';
        let stackExports = this.node.tryFindChild(exportsName);
        if (stackExports === undefined) {
            stackExports = new construct_1.Construct(this, exportsName);
        }
        return stackExports;
    }
    /**
     * Determine the various stack environment attributes.
     *
     */
    parseEnvironment(env = {}) {
        // if an environment property is explicitly specified when the stack is
        // created, it will be used. if not, use tokens for account and region but
        // they do not need to be scoped, the only situation in which
        // export/fn::importvalue would work if { Ref: "AWS::AccountId" } is the
        // same for provider and consumer anyway.
        const account = env.account || cfn_pseudo_1.Aws.ACCOUNT_ID;
        const region = env.region || cfn_pseudo_1.Aws.REGION;
        // this is the "aws://" env specification that will be written to the cloud assembly
        // manifest. it will use "unknown-account" and "unknown-region" to indicate
        // environment-agnosticness.
        const envAccount = !token_1.Token.isUnresolved(account) ? account : cxapi.UNKNOWN_ACCOUNT;
        const envRegion = !token_1.Token.isUnresolved(region) ? region : cxapi.UNKNOWN_REGION;
        return {
            account, region,
            environment: cx_api_1.EnvironmentUtils.format(envAccount, envRegion)
        };
    }
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    stackDependencyReasons(other) {
        if (this === other) {
            return [];
        }
        for (const dep of this._stackDependencies) {
            const ret = dep.stack.stackDependencyReasons(other);
            if (ret !== undefined) {
                return [dep.reason].concat(ret);
            }
        }
        return undefined;
    }
    collectMetadata() {
        const output = {};
        const stack = this;
        visit(this);
        return output;
        function visit(node) {
            // break off if we reached a node that is not a child of this stack
            const parent = findParentStack(node);
            if (parent !== stack) {
                return;
            }
            if (node.node.metadata.length > 0) {
                // Make the path absolute
                output[construct_1.ConstructNode.PATH_SEP + node.node.path] = node.node.metadata.map(md => stack.resolve(md));
            }
            for (const child of node.node.children) {
                visit(child);
            }
        }
        function findParentStack(node) {
            if (node instanceof Stack && node.parentStack === undefined) {
                return node;
            }
            if (!node.node.scope) {
                return undefined;
            }
            return findParentStack(node.node.scope);
        }
    }
    /**
     * Calculcate the stack name based on the construct path
     */
    generateUniqueId() {
        // In tests, it's possible for this stack to be the root object, in which case
        // we need to use it as part of the root path.
        const rootPath = this.node.scope !== undefined ? this.node.scopes.slice(1) : [this];
        const ids = rootPath.map(c => c.node.id);
        // Special case, if rootPath is length 1 then just use ID (backwards compatibility)
        // otherwise use a unique stack name (including hash). This logic is already
        // in makeUniqueId, *however* makeUniqueId will also strip dashes from the name,
        // which *are* allowed and also used, so we short-circuit it.
        if (ids.length === 1) {
            // Could be empty in a unit test, so just pretend it's named "Stack" then
            return ids[0] || 'Stack';
        }
        return uniqueid_1.makeUniqueId(ids);
    }
    generateExportName(stackExports, id) {
        const stack = Stack.of(stackExports);
        const components = [...stackExports.node.scopes.slice(2).map(c => c.node.id), id];
        const prefix = stack.stackName ? stack.stackName + ':' : '';
        const exportName = prefix + uniqueid_1.makeUniqueId(components);
        return exportName;
    }
    get assetParameters() {
        if (!this._assetParameters) {
            this._assetParameters = new construct_1.Construct(this, 'AssetParameters');
        }
        return this._assetParameters;
    }
    determineCrossReferenceFactory(target) {
        // unsupported: stacks from different apps
        if (target.node.root !== this.node.root) {
            throw new Error(`Cannot reference across apps. ` +
                `Consuming and producing stacks must be defined within the same CDK app.`);
        }
        // unsupported: stacks are not in the same environment
        if (target.environment !== this.environment) {
            throw new Error(`Stack "${this.node.path}" cannot consume a cross reference from stack "${target.node.path}". ` +
                `Cross stack references are only supported for stacks deployed to the same environment or between nested stacks and their parent stack`);
        }
        // if one of the stacks is a nested stack, go ahead and give it the right to make the cross reference
        if (target.nested) {
            return target;
        }
        if (this.nested) {
            return this;
        }
        // both stacks are top-level (non-nested), the taret (producing stack) gets to make the reference
        return target;
    }
    /**
     * Returns all the tokens used within the scope of the current stack.
     */
    findTokens() {
        const tokens = new Array();
        for (const element of cfnElements(this)) {
            try {
                tokens.push(...resolve_1.findTokens(element, () => element._toCloudFormation()));
            }
            catch (e) {
                // Note: it might be that the properties of the CFN object aren't valid.
                // This will usually be preventatively caught in a construct's validate()
                // and turned into a nicely descriptive error, but we're running prepare()
                // before validate(). Swallow errors that occur because the CFN layer
                // doesn't validate completely.
                //
                // This does make the assumption that the error will not be rectified,
                // but the error will be thrown later on anyway. If the error doesn't
                // get thrown down the line, we may miss references.
                if (e.type === 'CfnSynthesisError') {
                    continue;
                }
                throw e;
            }
        }
        return tokens;
    }
}
exports.Stack = Stack;
function merge(template, part) {
    for (const section of Object.keys(part)) {
        const src = part[section];
        // create top-level section if it doesn't exist
        let dest = template[section];
        if (!dest) {
            template[section] = dest = src;
        }
        else {
            // add all entities from source section to destination section
            for (const id of Object.keys(src)) {
                if (id in dest) {
                    throw new Error(`section '${section}' already contains '${id}'`);
                }
                dest[id] = src[id];
            }
        }
    }
}
/**
 * Collect all CfnElements from a Stack
 *
 * @param node Root node to collect all CfnElements from
 * @param into Array to append CfnElements to
 * @returns The same array as is being collected into
 */
function cfnElements(node, into = []) {
    if (cfn_element_1.CfnElement.isCfnElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (Stack.isStack(child)) {
            continue;
        }
        cfnElements(child, into);
    }
    return into;
}
// These imports have to be at the end to prevent circular imports
const arn_1 = require("./arn");
const cfn_element_1 = require("./cfn-element");
const cfn_fn_1 = require("./cfn-fn");
const cfn_output_1 = require("./cfn-output");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_1 = require("./cfn-resource");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const intrinsic_1 = require("./private/intrinsic");
const tag_manager_1 = require("./tag-manager");
const token_1 = require("./token");
/**
 * Find all resources in a set of constructs
 */
function findResources(roots) {
    const ret = new Array();
    for (const root of roots) {
        ret.push(...root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource));
    }
    return ret;
}
function extractSingleValue(array) {
    if (array && array.length === 1) {
        return array[0];
    }
    return array;
}
//# sourceMappingURL=data:application/json;base64,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