"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createKeyPolicyUpdaterCustomResource = void 0;
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const core_1 = require("@aws-solutions-constructs/core");
const utils_1 = require("./utils");
function createKeyPolicyUpdaterCustomResource(scope, props) {
    const lambdaFunction = core_1.buildLambdaFunction(scope, {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler',
            description: 'Custom resource function that updates a provided key policy to allow CloudFront access.',
            timeout: props.timeout,
            memorySize: props.memorySize,
            code: lambda.Code.fromAsset(`${__dirname}/key-policy-updater-custom-resource`),
            role: new iam.Role(scope, 'KmsKeyPolicyUpdateLambdaRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
                description: 'Role to update kms key policy to allow CloudFront access',
                inlinePolicies: {
                    KmsPolicy: new iam.PolicyDocument({
                        statements: [
                            new iam.PolicyStatement({
                                actions: ['kms:PutKeyPolicy', 'kms:GetKeyPolicy', 'kms:DescribeKey'],
                                effect: iam.Effect.ALLOW,
                                resources: [props.encryptionKey.keyArn]
                            })
                        ]
                    })
                }
            })
        }
    });
    const kmsKeyPolicyUpdateProvider = new custom_resources_1.Provider(scope, 'KmsKeyPolicyUpdateProvider', {
        onEventHandler: lambdaFunction
    });
    utils_1.addCfnSuppressRulesForCustomResourceProvider(kmsKeyPolicyUpdateProvider);
    const customResource = new aws_cdk_lib_1.CustomResource(scope, 'KmsKeyPolicyUpdater', {
        resourceType: 'Custom::KmsKeyPolicyUpdater',
        serviceToken: kmsKeyPolicyUpdateProvider.serviceToken,
        properties: {
            KmsKeyId: props.encryptionKey.keyId,
            CloudFrontDistributionId: props.distribution.distributionId,
            AccountId: aws_cdk_lib_1.Aws.ACCOUNT_ID
        },
    });
    return {
        lambdaFunction,
        customResource
    };
}
exports.createKeyPolicyUpdaterCustomResource = createKeyPolicyUpdaterCustomResource;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2V5LXBvbGljeS11cGRhdGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsia2V5LXBvbGljeS11cGRhdGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7O0FBR0gsMkNBQTJDO0FBQzNDLGlEQUFpRDtBQUNqRCw2Q0FBa0Q7QUFDbEQsbUVBQXdEO0FBQ3hELHlEQUFxRTtBQUdyRSxtQ0FBdUU7QUFnQnZFLFNBQWdCLG9DQUFvQyxDQUNsRCxLQUFnQixFQUNoQixLQUE0QjtJQUc1QixNQUFNLGNBQWMsR0FBRywwQkFBbUIsQ0FBQyxLQUFLLEVBQUU7UUFDaEQsbUJBQW1CLEVBQUU7WUFDbkIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUNuQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixXQUFXLEVBQUUseUZBQXlGO1lBQ3RHLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztZQUN0QixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxxQ0FBcUMsQ0FBQztZQUM5RSxJQUFJLEVBQUUsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSw4QkFBOEIsRUFBRTtnQkFDeEQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2dCQUMzRCxXQUFXLEVBQUUsMERBQTBEO2dCQUN2RSxjQUFjLEVBQUU7b0JBQ2QsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQzt3QkFDaEMsVUFBVSxFQUFFOzRCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQ0FDdEIsT0FBTyxFQUFFLENBQUMsa0JBQWtCLEVBQUUsa0JBQWtCLEVBQUUsaUJBQWlCLENBQUM7Z0NBQ3BFLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7Z0NBQ3hCLFNBQVMsRUFBRSxDQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFFOzZCQUMxQyxDQUFDO3lCQUNIO3FCQUNGLENBQUM7aUJBQ0g7YUFDRixDQUFDO1NBQ0g7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLDBCQUEwQixHQUFHLElBQUksMkJBQVEsQ0FBQyxLQUFLLEVBQUUsNEJBQTRCLEVBQUU7UUFDbkYsY0FBYyxFQUFFLGNBQWM7S0FDL0IsQ0FBQyxDQUFDO0lBRUgsb0RBQTRDLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUV6RSxNQUFNLGNBQWMsR0FBRyxJQUFJLDRCQUFjLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1FBQ3RFLFlBQVksRUFBRSw2QkFBNkI7UUFDM0MsWUFBWSxFQUFFLDBCQUEwQixDQUFDLFlBQVk7UUFDckQsVUFBVSxFQUFFO1lBQ1YsUUFBUSxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsS0FBSztZQUNuQyx3QkFBd0IsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLGNBQWM7WUFDM0QsU0FBUyxFQUFFLGlCQUFHLENBQUMsVUFBVTtTQUMxQjtLQUNGLENBQUMsQ0FBQztJQUVILE9BQU87UUFDTCxjQUFjO1FBQ2QsY0FBYztLQUNmLENBQUM7QUFDSixDQUFDO0FBbkRELG9GQW1EQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCAqIGFzIGNkayBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCB7IEF3cywgQ3VzdG9tUmVzb3VyY2UgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBQcm92aWRlciB9IGZyb20gXCJhd3MtY2RrLWxpYi9jdXN0b20tcmVzb3VyY2VzXCI7XG5pbXBvcnQgeyBidWlsZExhbWJkYUZ1bmN0aW9uIH0gZnJvbSBcIkBhd3Mtc29sdXRpb25zLWNvbnN0cnVjdHMvY29yZVwiO1xuaW1wb3J0IHsgSUtleSB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3Mta21zXCI7XG5pbXBvcnQgeyBEaXN0cmlidXRpb24gfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWNsb3VkZnJvbnRcIjtcbmltcG9ydCB7IGFkZENmblN1cHByZXNzUnVsZXNGb3JDdXN0b21SZXNvdXJjZVByb3ZpZGVyIH0gZnJvbSBcIi4vdXRpbHNcIjtcbi8vIE5vdGU6IFRvIGVuc3VyZSBDREt2MiBjb21wYXRpYmlsaXR5LCBrZWVwIHRoZSBpbXBvcnQgc3RhdGVtZW50IGZvciBDb25zdHJ1Y3Qgc2VwYXJhdGVcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG5leHBvcnQgaW50ZXJmYWNlIENyZWF0ZUtleVBvbGljeVVwZGF0ZXJSZXNwb25zZSB7XG4gIHJlYWRvbmx5IGxhbWJkYUZ1bmN0aW9uOiBsYW1iZGEuRnVuY3Rpb247XG4gIHJlYWRvbmx5IGN1c3RvbVJlc291cmNlOiBDdXN0b21SZXNvdXJjZTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBLZXlQb2xpY3lVcGRhdGVyUHJvcHMge1xuICByZWFkb25seSBlbmNyeXB0aW9uS2V5OiBJS2V5O1xuICByZWFkb25seSBkaXN0cmlidXRpb246IERpc3RyaWJ1dGlvbjtcbiAgcmVhZG9ubHkgdGltZW91dD86IGNkay5EdXJhdGlvbjtcbiAgcmVhZG9ubHkgbWVtb3J5U2l6ZT86IG51bWJlcjtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUtleVBvbGljeVVwZGF0ZXJDdXN0b21SZXNvdXJjZShcbiAgc2NvcGU6IENvbnN0cnVjdCxcbiAgcHJvcHM6IEtleVBvbGljeVVwZGF0ZXJQcm9wc1xuKTogQ3JlYXRlS2V5UG9saWN5VXBkYXRlclJlc3BvbnNlIHtcblxuICBjb25zdCBsYW1iZGFGdW5jdGlvbiA9IGJ1aWxkTGFtYmRhRnVuY3Rpb24oc2NvcGUsIHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMThfWCxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIGRlc2NyaXB0aW9uOiAnQ3VzdG9tIHJlc291cmNlIGZ1bmN0aW9uIHRoYXQgdXBkYXRlcyBhIHByb3ZpZGVkIGtleSBwb2xpY3kgdG8gYWxsb3cgQ2xvdWRGcm9udCBhY2Nlc3MuJyxcbiAgICAgIHRpbWVvdXQ6IHByb3BzLnRpbWVvdXQsXG4gICAgICBtZW1vcnlTaXplOiBwcm9wcy5tZW1vcnlTaXplLFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0va2V5LXBvbGljeS11cGRhdGVyLWN1c3RvbS1yZXNvdXJjZWApLFxuICAgICAgcm9sZTogbmV3IGlhbS5Sb2xlKHNjb3BlLCAnS21zS2V5UG9saWN5VXBkYXRlTGFtYmRhUm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIGRlc2NyaXB0aW9uOiAnUm9sZSB0byB1cGRhdGUga21zIGtleSBwb2xpY3kgdG8gYWxsb3cgQ2xvdWRGcm9udCBhY2Nlc3MnLFxuICAgICAgICBpbmxpbmVQb2xpY2llczoge1xuICAgICAgICAgIEttc1BvbGljeTogbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ2ttczpQdXRLZXlQb2xpY3knLCAna21zOkdldEtleVBvbGljeScsICdrbXM6RGVzY3JpYmVLZXknXSxcbiAgICAgICAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbIHByb3BzLmVuY3J5cHRpb25LZXkua2V5QXJuIF1cbiAgICAgICAgICAgICAgfSlcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9KVxuICAgICAgICB9XG4gICAgICB9KVxuICAgIH1cbiAgfSk7XG5cbiAgY29uc3Qga21zS2V5UG9saWN5VXBkYXRlUHJvdmlkZXIgPSBuZXcgUHJvdmlkZXIoc2NvcGUsICdLbXNLZXlQb2xpY3lVcGRhdGVQcm92aWRlcicsIHtcbiAgICBvbkV2ZW50SGFuZGxlcjogbGFtYmRhRnVuY3Rpb25cbiAgfSk7XG5cbiAgYWRkQ2ZuU3VwcHJlc3NSdWxlc0ZvckN1c3RvbVJlc291cmNlUHJvdmlkZXIoa21zS2V5UG9saWN5VXBkYXRlUHJvdmlkZXIpO1xuXG4gIGNvbnN0IGN1c3RvbVJlc291cmNlID0gbmV3IEN1c3RvbVJlc291cmNlKHNjb3BlLCAnS21zS2V5UG9saWN5VXBkYXRlcicsIHtcbiAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206Okttc0tleVBvbGljeVVwZGF0ZXInLFxuICAgIHNlcnZpY2VUb2tlbjoga21zS2V5UG9saWN5VXBkYXRlUHJvdmlkZXIuc2VydmljZVRva2VuLFxuICAgIHByb3BlcnRpZXM6IHtcbiAgICAgIEttc0tleUlkOiBwcm9wcy5lbmNyeXB0aW9uS2V5LmtleUlkLFxuICAgICAgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbklkOiBwcm9wcy5kaXN0cmlidXRpb24uZGlzdHJpYnV0aW9uSWQsXG4gICAgICBBY2NvdW50SWQ6IEF3cy5BQ0NPVU5UX0lEXG4gICAgfSxcbiAgfSk7XG5cbiAgcmV0dXJuIHtcbiAgICBsYW1iZGFGdW5jdGlvbixcbiAgICBjdXN0b21SZXNvdXJjZVxuICB9O1xufSJdfQ==