/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as cdk from "aws-cdk-lib";
import * as s3 from 'aws-cdk-lib/aws-s3';
import { CustomResource } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * The TemplateValue interface defines the id-value pair that will
 * be substituted in the template.
 *
 * For example, given the template:
 * template:
 *   hello name_placeholder, nice to meet you
 *
 * and an instantiation of TemplateValue { id: 'name_placeholder', value: 'jeff' }
 *
 * the template will be transformed to:
 * template:
 *   hello jeff, nice to meet you
 */
export interface TemplateValue {
    /**
     * The placeholder string to be replaced in the template.
     */
    readonly id: string;
    /**
     * The value to replace the placeholder in the template with.
     */
    readonly value: string;
}
export interface TemplateWriterProps {
    /**
     * The S3 bucket that holds the template to transform.
     * Upstream this can come either from an Asset or S3 bucket directly.
     * Internally it will always resolve to S3 bucket in either case (the cdk asset bucket or the customer-defined bucket).
     */
    readonly templateBucket: s3.IBucket;
    /**
     * The S3 object key of the template to transform.
     */
    readonly templateKey: string;
    /**
     * An array of TemplateValue objects, each defining a placeholder string in the
     * template that will be replaced with its corresponding value.
     */
    readonly templateValues: TemplateValue[];
    /**
     * Optional configuration for user-defined duration of the backing Lambda function, which may be necessary when transforming very large objects.
     *
     * @default Duration.seconds(3)
     */
    readonly timeout?: cdk.Duration;
    /**
     * Optional configuration for user-defined memorySize of the backing Lambda function, which may be necessary when transforming very large objects.
     *
     * @default 128
     */
    readonly memorySize?: number;
}
export interface CreateTemplateWriterResponse {
    readonly s3Bucket: s3.IBucket;
    readonly s3Key: string;
    readonly customResource: CustomResource;
}
export declare function createTemplateWriterCustomResource(scope: Construct, id: string, props: TemplateWriterProps): CreateTemplateWriterResponse;
