"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createTemplateWriterCustomResource = void 0;
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const path = require("path");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const core_1 = require("@aws-solutions-constructs/core");
const utils_1 = require("./utils");
function createTemplateWriterCustomResource(scope, id, props) {
    const outputAsset = new aws_s3_assets_1.Asset(scope, `${id}OutputAsset`, {
        path: path.join(__dirname, 'placeholder')
    });
    const templateWriterLambda = core_1.buildLambdaFunction(scope, {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/template-writer-custom-resource`),
            timeout: props.timeout,
            memorySize: props.memorySize,
            role: new iam.Role(scope, `${id}TemplateWriterLambdaRole`, {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
                description: 'Role used by the TemplateWriterLambda to transform the incoming asset',
                inlinePolicies: {
                    CloudWatchLogsPolicy: new iam.PolicyDocument({
                        statements: [
                            new iam.PolicyStatement({
                                actions: [
                                    'logs:CreateLogGroup',
                                    'logs:CreateLogStream',
                                    'logs:PutLogEvents'
                                ],
                                resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                            })
                        ]
                    }),
                    ReadInputAssetPolicy: new iam.PolicyDocument({
                        statements: [
                            new iam.PolicyStatement({
                                actions: ['s3:GetObject'],
                                effect: iam.Effect.ALLOW,
                                resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:s3:::${props.templateBucket.bucketName}/${props.templateKey}`]
                            })
                        ]
                    }),
                    WriteOutputAssetPolicy: new iam.PolicyDocument({
                        statements: [
                            new iam.PolicyStatement({
                                actions: ['s3:PutObject'],
                                effect: iam.Effect.ALLOW,
                                resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:s3:::${outputAsset.s3BucketName}/*`]
                            })
                        ]
                    })
                }
            })
        }
    });
    const templateWriterProvider = new custom_resources_1.Provider(scope, `${id}TemplateWriterProvider`, {
        onEventHandler: templateWriterLambda
    });
    utils_1.addCfnSuppressRulesForCustomResourceProvider(templateWriterProvider);
    const customResource = new aws_cdk_lib_1.CustomResource(scope, `${id}TemplateWriterCustomResource`, {
        resourceType: 'Custom::TemplateWriter',
        serviceToken: templateWriterProvider.serviceToken,
        properties: {
            TemplateValues: JSON.stringify({ templateValues: props.templateValues }),
            TemplateInputBucket: props.templateBucket.bucketName,
            TemplateInputKey: props.templateKey,
            TemplateOutputBucket: outputAsset.s3BucketName
        }
    });
    return {
        s3Bucket: outputAsset.bucket,
        s3Key: customResource.getAttString('TemplateOutputKey'),
        customResource
    };
}
exports.createTemplateWriterCustomResource = createTemplateWriterCustomResource;
//# sourceMappingURL=data:application/json;base64,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