"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_sdk_client_mock_1 = require("aws-sdk-client-mock");
const client_kms_1 = require("@aws-sdk/client-kms");
const key_policy_updater_custom_resource_1 = require("../lib/key-policy-updater-custom-resource");
const kmsMock = aws_sdk_client_mock_1.mockClient(client_kms_1.KMSClient);
beforeEach(() => {
    kmsMock.reset();
});
it('Should exit if an AWS managed key is provided; return a success code but give the exact reason', async () => {
    // Mocks
    kmsMock.on(client_kms_1.DescribeKeyCommand).resolves({
        KeyMetadata: {
            KeyId: 'sample-key-id',
            KeyManager: client_kms_1.KeyManagerType.AWS
        }
    });
    // Arrange
    const e = {
        RequestType: 'Create',
        ResourceProperties: {
            CloudFrontDistributionId: 'sample-cf-distro-id',
            AccountId: '111122223333'
        }
    };
    const context = {
    // ...
    };
    // Act
    const res = await key_policy_updater_custom_resource_1.handler(e, context);
    // Assert
    expect(res.Status).toBe('SUCCESS');
    expect(res.Data).toBe('An AWS managed key was provided, no action needed from the custom resource, exiting now.');
});
it('Should return an error if the key policy is returned as undefined', async () => {
    // Mocks
    kmsMock.on(client_kms_1.DescribeKeyCommand).resolves({
        KeyMetadata: {
            KeyId: 'sample-key-id',
            KeyManager: client_kms_1.KeyManagerType.CUSTOMER
        }
    });
    kmsMock.on(client_kms_1.GetKeyPolicyCommand).resolves({
        Policy: undefined
    });
    // Arrange
    const e = {
        RequestType: 'Update',
        ResourceProperties: {
            CloudFrontDistributionId: 'sample-cf-distro-id',
            AccountId: '111122223333'
        }
    };
    const context = {
    // ...
    };
    // Act
    const res = await key_policy_updater_custom_resource_1.handler(e, context);
    // Assert
    expect(res.Status).toBe('FAILED');
});
it('Should update the key policy if the proper params are given', async () => {
    // Mocks
    kmsMock.on(client_kms_1.DescribeKeyCommand).resolves({
        KeyMetadata: {
            KeyId: 'sample-key-id',
            KeyManager: client_kms_1.KeyManagerType.CUSTOMER
        }
    });
    kmsMock.on(client_kms_1.GetKeyPolicyCommand).resolves({
        Policy: `{\n
      \"Version\" : \"2012-10-17\",\n
      \"Id\" : \"sample-key-id\",\n
      \"Statement\" : [ {\n
          \"Sid\" : \"Grant-CloudFront-Distribution-Key-Usage\",\n
          \"Effect\" : \"Allow\",\n
          \"Principal\" : {\n
              \"AWS\" : \"arn:aws:iam::111122223333:root\"\n
          },\n
          \"Action\" : \"kms:*\",\n
          \"Resource\" : \"*\"\n
      } ]\n
    }`
    });
    // Arrange
    const e = {
        RequestType: 'Update',
        ResourceProperties: {
            CloudFrontDistributionId: 'sample-cf-distro-id',
            AccountId: '111122223333'
        }
    };
    const context = {
    // ...
    };
    // Act
    const res = await key_policy_updater_custom_resource_1.handler(e, context);
    // Assert
    expect(res.Status).toBe('SUCCESS');
});
it('Should fail if an error occurs with putting the new key policy, all other inputs valid', async () => {
    // Mocks
    kmsMock.on(client_kms_1.DescribeKeyCommand).resolves({
        KeyMetadata: {
            KeyId: 'sample-key-id',
            KeyManager: client_kms_1.KeyManagerType.CUSTOMER
        }
    });
    kmsMock.on(client_kms_1.GetKeyPolicyCommand).resolves({
        Policy: `{\n
      \"Version\" : \"2012-10-17\",\n
      \"Id\" : \"key-default-1\",\n
      \"Statement\" : [ {\n
          \"Sid\" : \"Grant-CloudFront-Distribution-Key-Usage\",\n
          \"Effect\" : \"Allow\",\n
          \"Principal\" : {\n
              \"AWS\" : \"arn:aws:iam::111122223333:root\"\n
          },\n
          \"Action\" : \"kms:*\",\n
          \"Resource\" : \"*\"\n
      } ]\n
    }`
    });
    kmsMock.on(client_kms_1.PutKeyPolicyCommand).rejects();
    const e = {
        RequestType: 'Update',
        ResourceProperties: {
            CloudFrontDistributionId: 'sample-cf-distro-id',
            AccountId: '111122223333'
        }
    };
    const context = {
    // ...
    };
    // Act
    const res = await key_policy_updater_custom_resource_1.handler(e, context);
    // Assert
    expect(res.Status).toBe('FAILED');
});
it('Should fail if the key policy has already been applied in a previous stack update or similar event (custom resource response)', async () => {
    // Mocks
    kmsMock.on(client_kms_1.DescribeKeyCommand).resolves({
        KeyMetadata: {
            KeyId: 'sample-key-id',
            KeyManager: client_kms_1.KeyManagerType.CUSTOMER
        }
    });
    kmsMock.on(client_kms_1.GetKeyPolicyCommand).resolves({
        Policy: `{\n
      \"Version\" : \"2012-10-17\",\n
      \"Id\" : \"key-default-1\",\n
      \"Statement\" : [ {\n
          \"Sid\" : \"Grant-CloudFront-Distribution-Key-Usage\",\n
          \"Effect\" : \"Allow\",\n
          \"Principal\" : {\n
              \"AWS\" : \"arn:aws:iam::111122223333:root\"\n
          },\n
          \"Action\" : \"kms:*\",\n
          \"Resource\" : \"*\"\n
      } ]\n
    }`
    });
    const e = {
        RequestType: 'Update',
        ResourceProperties: {
            CloudFrontDistributionId: 'sample-cf-distro-id',
            AccountId: '111122223333'
        }
    };
    const context = {
    // ...
    };
    // Act
    const res = await key_policy_updater_custom_resource_1.handler(e, context);
    // Assert
    expect(res.Status).toBe('SUCCESS');
});
it('updateKeyPolicy() should overwrite an existing key policy statement that matches on the sid', async () => {
    // Arrange
    const keyPolicy = {
        Version: "2012-10-17",
        Id: "key-default-1",
        Statement: [
            {
                Sid: 'Grant-CloudFront-Distribution-Key-Usage',
                Effect: "Allow"
            },
            {
                Sid: 'Some-Other-Key-Policy-Statement',
                Effect: "Allow"
            }
        ]
    };
    const keyPolicyStatement = {
        Sid: 'Grant-CloudFront-Distribution-Key-Usage',
        Effect: "Deny"
    };
    // Act
    const res = key_policy_updater_custom_resource_1.updateKeyPolicy(keyPolicy, keyPolicyStatement);
    // Assert
    expect(res.Statement[0].Sid).toBe('Grant-CloudFront-Distribution-Key-Usage');
    expect(res.Statement[0].Effect).toBe('Deny');
});
it('updateKeyPolicy() should add the key policy statement if one with matching sid does not already exist', async () => {
    // Arrange
    const keyPolicy = {
        Version: "2012-10-17",
        Id: "key-default-1",
        Statement: [
            {
                Sid: 'Some-Other-Key-Policy-Statement',
                Effect: "Allow"
            }
        ]
    };
    const keyPolicyStatement = {
        Sid: 'Grant-CloudFront-Distribution-Key-Usage',
        Effect: "Deny"
    };
    // Act
    const res = key_policy_updater_custom_resource_1.updateKeyPolicy(keyPolicy, keyPolicyStatement);
    // Assert
    expect(res.Statement[1].Sid).toBe('Grant-CloudFront-Distribution-Key-Usage');
    expect(res.Statement[1].Effect).toBe('Deny');
});
//# sourceMappingURL=data:application/json;base64,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