"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const path = require("path");
const template_writer_1 = require("../lib/template-writer");
test('TemplateWriter sets properties correctly', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const templateAsset = new aws_s3_assets_1.Asset(stack, 'TemplateAsset', {
        path: path.join(__dirname, 'template/sample-template')
    });
    const templateValues = new Array({
        id: 'placeholder',
        value: 'resolved_value'
    });
    template_writer_1.createTemplateWriterCustomResource(stack, 'Test', {
        templateBucket: templateAsset.bucket,
        templateKey: templateAsset.s3ObjectKey,
        templateValues
    });
    const cfnTemplate = assertions_1.Template.fromStack(stack);
    cfnTemplate.hasResourceProperties('Custom::TemplateWriter', {
        TemplateValues: JSON.stringify({ templateValues }),
        TemplateInputKey: templateAsset.s3ObjectKey,
        TemplateInputBucket: {
            "Fn::Sub": "cdk-hnb659fds-assets-${AWS::AccountId}-${AWS::Region}"
        },
        TemplateOutputBucket: {
            "Fn::Sub": "cdk-hnb659fds-assets-${AWS::AccountId}-${AWS::Region}"
        },
    });
});
test('TemplateWriter accepts custom lambda function timeout and memory size', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const templateAsset = new aws_s3_assets_1.Asset(stack, 'TemplateAsset', {
        path: path.join(__dirname, 'template/sample-template')
    });
    const templateValues = new Array({
        id: 'placeholder',
        value: 'resolved_value'
    });
    template_writer_1.createTemplateWriterCustomResource(stack, 'Test', {
        templateBucket: templateAsset.bucket,
        templateKey: templateAsset.s3ObjectKey,
        templateValues,
        timeout: aws_cdk_lib_1.Duration.minutes(7),
        memorySize: 4096
    });
    const cfnTemplate = assertions_1.Template.fromStack(stack);
    cfnTemplate.hasResourceProperties('AWS::Lambda::Function', {
        Timeout: 420,
        MemorySize: 4096
    });
});
//# sourceMappingURL=data:application/json;base64,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