"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalize = exports.patch = exports.merge = exports.decorateResourceTypes = exports.forEachSection = exports.massageSpec = void 0;
const fastJsonPatch = require("fast-json-patch");
const lib_1 = require("../lib");
const scrutiny_1 = require("./scrutiny");
function massageSpec(spec) {
    scrutiny_1.detectScrutinyTypes(spec);
    replaceIncompleteTypes(spec);
    dropTypelessAttributes(spec);
}
exports.massageSpec = massageSpec;
function forEachSection(spec, data, cb) {
    cb(spec.PropertyTypes, data.PropertyTypes, ['PropertyTypes']);
    cb(spec.ResourceTypes, data.ResourceTypes, ['ResourceTypes']);
    // Per-resource specs are keyed on ResourceType (singular), but we want it in ResourceTypes (plural)
    cb(spec.ResourceTypes, data.ResourceType, ['ResourceType']);
}
exports.forEachSection = forEachSection;
function decorateResourceTypes(data) {
    const requiredTransform = data.ResourceSpecificationTransform;
    if (!requiredTransform) {
        return;
    }
    const resourceTypes = data.ResourceTypes || data.ResourceType;
    for (const name of Object.keys(resourceTypes)) {
        resourceTypes[name].RequiredTransform = requiredTransform;
    }
}
exports.decorateResourceTypes = decorateResourceTypes;
/**
 * Fix incomplete type definitions in PropertyTypes
 *
 * Some user-defined types are defined to not have any properties, and not
 * be a collection of other types either. They have no definition at all.
 *
 * Add a property object type with empty properties.
 */
function replaceIncompleteTypes(spec) {
    for (const [name, definition] of Object.entries(spec.PropertyTypes)) {
        if (!lib_1.schema.isRecordType(definition)
            && !lib_1.schema.isCollectionProperty(definition)
            && !lib_1.schema.isScalarProperty(definition)
            && !lib_1.schema.isPrimitiveProperty(definition)) {
            // eslint-disable-next-line no-console
            console.log(`[${name}] Incomplete type, adding empty "Properties" field`);
            definition.Properties = {};
        }
    }
}
/**
 * Drop Attributes specified with the different ResourceTypes that have
 * no type specified.
 */
function dropTypelessAttributes(spec) {
    const resourceTypes = spec.ResourceTypes;
    Object.values(resourceTypes).forEach((resourceType) => {
        var _a;
        const attributes = (_a = resourceType.Attributes) !== null && _a !== void 0 ? _a : {};
        Object.keys(attributes).forEach((attrKey) => {
            const attrVal = attributes[attrKey];
            if (Object.keys(attrVal).length === 0) {
                delete attributes[attrKey];
            }
        });
    });
}
function merge(spec, fragment, jsonPath) {
    if (!fragment) {
        return;
    }
    for (const key of Object.keys(fragment)) {
        if (key in spec) {
            const specVal = spec[key];
            const fragVal = fragment[key];
            if (typeof specVal !== typeof fragVal) {
                // eslint-disable-next-line max-len
                throw new Error(`Attempted to merge ${JSON.stringify(fragVal)} into incompatible ${JSON.stringify(specVal)} at path ${jsonPath.join('/')}/${key}`);
            }
            if (typeof specVal !== 'object') {
                // eslint-disable-next-line max-len
                throw new Error(`Conflict when attempting to merge ${JSON.stringify(fragVal)} into ${JSON.stringify(specVal)} at path ${jsonPath.join('/')}/${key}`);
            }
            merge(specVal, fragVal, [...jsonPath, key]);
        }
        else {
            spec[key] = fragment[key];
        }
    }
}
exports.merge = merge;
function patch(spec, fragment) {
    if (!fragment) {
        return;
    }
    if ('patch' in fragment) {
        // eslint-disable-next-line no-console
        console.log(`Applying patch: ${fragment.patch.description}`);
        fastJsonPatch.applyPatch(spec, fragment.patch.operations);
    }
    else {
        for (const key of Object.keys(fragment)) {
            patch(spec[key], fragment[key]);
        }
    }
}
exports.patch = patch;
/**
 * Modifies the provided specification so that ``ResourceTypes`` and ``PropertyTypes`` are listed in alphabetical order.
 *
 * @param spec an AWS CloudFormation Resource Specification document.
 *
 * @returns ``spec``, after having sorted the ``ResourceTypes`` and ``PropertyTypes`` sections alphabetically.
 */
function normalize(spec) {
    spec.ResourceTypes = normalizeSection(spec.ResourceTypes);
    if (spec.PropertyTypes) {
        spec.PropertyTypes = normalizeSection(spec.PropertyTypes);
    }
    return spec;
    function normalizeSection(section) {
        const result = {};
        for (const key of Object.keys(section).sort()) {
            result[key] = section[key];
        }
        return result;
    }
}
exports.normalize = normalize;
//# sourceMappingURL=data:application/json;base64,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