"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Manifest = exports.VERSION_MISMATCH = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const assembly = require("./cloud-assembly");
/* eslint-disable @typescript-eslint/no-var-requires */
/* eslint-disable @typescript-eslint/no-require-imports */
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = 'Cloud assembly schema version mismatch';
const ASSETS_SCHEMA = require('../schema/assets.schema.json');
const ASSEMBLY_SCHEMA = require('../schema/cloud-assembly.schema.json');
/**
 * Version is shared for both manifests
 */
const SCHEMA_VERSION = require('../schema/cloud-assembly.version.json').version;
/**
 * Protocol utility class.
 *
 * @stability stable
 */
class Manifest {
    constructor() { }
    /**
     * Validates and saves the cloud assembly manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     * @stability stable
     */
    static saveAssemblyManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnWrite);
    }
    /**
     * Load and validates the cloud assembly manifest from file.
     *
     * @param filePath - path to the manifest file.
     * @stability stable
     */
    static loadAssemblyManifest(filePath) {
        return Manifest.loadManifest(filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnRead);
    }
    /**
     * Validates and saves the asset manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     * @stability stable
     */
    static saveAssetManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, ASSETS_SCHEMA, Manifest.patchStackTagsOnRead);
    }
    /**
     * Load and validates the asset manifest from file.
     *
     * @param filePath - path to the manifest file.
     * @stability stable
     */
    static loadAssetManifest(filePath) {
        return this.loadManifest(filePath, ASSETS_SCHEMA);
    }
    /**
     * Fetch the current schema version number.
     *
     * @stability stable
     */
    static version() {
        return SCHEMA_VERSION;
    }
    /**
     * (deprecated) Deprecated.
     *
     * @deprecated use `saveAssemblyManifest()`
     */
    static save(manifest, filePath) { return this.saveAssemblyManifest(manifest, filePath); }
    /**
     * (deprecated) Deprecated.
     *
     * @deprecated use `loadAssemblyManifest()`
     */
    static load(filePath) { return this.loadAssemblyManifest(filePath); }
    static validate(manifest, schema) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = parseVersion(Manifest.version());
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted.
        if (semver.gt(actual, maxSupported)) {
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}, but found ${actual}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
        });
        if (!result.valid) {
            throw new Error(`Invalid assembly manifest:\n${result}`);
        }
    }
    static saveManifest(manifest, filePath, schema, preprocess) {
        let withVersion = { ...manifest, version: Manifest.version() };
        Manifest.validate(withVersion, schema);
        if (preprocess) {
            withVersion = preprocess(withVersion);
        }
        fs.writeFileSync(filePath, JSON.stringify(withVersion, undefined, 2));
    }
    static loadManifest(filePath, schema, preprocess) {
        let obj = JSON.parse(fs.readFileSync(filePath, { encoding: 'utf-8' }));
        if (preprocess) {
            obj = preprocess(obj);
        }
        Manifest.validate(obj, schema);
        return obj;
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     * @link https://github.com/aws/aws-cdk/blob/v1.27.0/packages/aws-cdk/lib/api/cxapp/stacks.ts#L427.
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how CloudFormation expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how CloudFormation expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTagsOnRead(manifest) {
        return Manifest.replaceStackTags(manifest, tags => tags.map((diskTag) => ({
            key: diskTag.Key,
            value: diskTag.Value,
        })));
    }
    /**
     * See explanation on `patchStackTagsOnRead`
     *
     * Translate stack tags metadata if it has the "right" casing.
     */
    static patchStackTagsOnWrite(manifest) {
        return Manifest.replaceStackTags(manifest, tags => tags.map(memTag => 
        // Might already be uppercased (because stack synthesis generates it in final form yet)
        ('Key' in memTag ? memTag : { Key: memTag.key, Value: memTag.value })));
    }
    /**
     * Recursively replace stack tags in the stack metadata
     */
    static replaceStackTags(manifest, fn) {
        // Need to add in the `noUndefined`s because otherwise jest snapshot tests are going to freak out
        // about the keys with values that are `undefined` (even though they would never be JSON.stringified)
        return noUndefined({
            ...manifest,
            artifacts: mapValues(manifest.artifacts, artifact => {
                if (artifact.type !== assembly.ArtifactType.AWS_CLOUDFORMATION_STACK) {
                    return artifact;
                }
                return noUndefined({
                    ...artifact,
                    metadata: mapValues(artifact.metadata, metadataEntries => metadataEntries.map(metadataEntry => {
                        if (metadataEntry.type !== assembly.ArtifactMetadataEntryType.STACK_TAGS || !metadataEntry.data) {
                            return metadataEntry;
                        }
                        return {
                            ...metadataEntry,
                            data: fn(metadataEntry.data),
                        };
                    })),
                });
            }),
        });
    }
}
exports.Manifest = Manifest;
_a = JSII_RTTI_SYMBOL_1;
Manifest[_a] = { fqn: "@aws-cdk/cloud-assembly-schema.Manifest", version: "1.97.0" };
function mapValues(xs, fn) {
    if (!xs) {
        return undefined;
    }
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        ret[k] = fn(v);
    }
    return ret;
}
function noUndefined(xs) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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