"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !exports.hasOwnProperty(p)) __createBinding(exports, m, p);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.diffResource = exports.diffTemplate = void 0;
const impl = require("./diff");
const types = require("./diff/types");
const util_1 = require("./diff/util");
__exportStar(require("./diff/types"), exports);
const DIFF_HANDLERS = {
    AWSTemplateFormatVersion: (diff, oldValue, newValue) => diff.awsTemplateFormatVersion = impl.diffAttribute(oldValue, newValue),
    Description: (diff, oldValue, newValue) => diff.description = impl.diffAttribute(oldValue, newValue),
    Metadata: (diff, oldValue, newValue) => diff.metadata = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffMetadata)),
    Parameters: (diff, oldValue, newValue) => diff.parameters = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffParameter)),
    Mappings: (diff, oldValue, newValue) => diff.mappings = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffMapping)),
    Conditions: (diff, oldValue, newValue) => diff.conditions = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffCondition)),
    Transform: (diff, oldValue, newValue) => diff.transform = impl.diffAttribute(oldValue, newValue),
    Resources: (diff, oldValue, newValue) => diff.resources = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffResource)),
    Outputs: (diff, oldValue, newValue) => diff.outputs = new types.DifferenceCollection(util_1.diffKeyedEntities(oldValue, newValue, impl.diffOutput)),
};
/**
 * Compare two CloudFormation templates and return semantic differences between them.
 *
 * @param currentTemplate the current state of the stack.
 * @param newTemplate     the target state of the stack.
 *
 * @returns a +types.TemplateDiff+ object that represents the changes that will happen if
 *      a stack which current state is described by +currentTemplate+ is updated with
 *      the template +newTemplate+.
 */
function diffTemplate(currentTemplate, newTemplate) {
    // Base diff
    const theDiff = calculateTemplateDiff(currentTemplate, newTemplate);
    // We're going to modify this in-place
    const newTemplateCopy = deepCopy(newTemplate);
    let didPropagateReferenceChanges;
    let diffWithReplacements;
    do {
        diffWithReplacements = calculateTemplateDiff(currentTemplate, newTemplateCopy);
        // Propagate replacements for replaced resources
        didPropagateReferenceChanges = false;
        if (diffWithReplacements.resources) {
            diffWithReplacements.resources.forEachDifference((logicalId, change) => {
                if (change.changeImpact === types.ResourceImpact.WILL_REPLACE) {
                    if (propagateReplacedReferences(newTemplateCopy, logicalId)) {
                        didPropagateReferenceChanges = true;
                    }
                }
            });
        }
    } while (didPropagateReferenceChanges);
    // Copy "replaced" states from `diffWithReplacements` to `theDiff`.
    diffWithReplacements.resources
        .filter(r => isReplacement(r.changeImpact))
        .forEachDifference((logicalId, downstreamReplacement) => {
        const resource = theDiff.resources.get(logicalId);
        if (resource.changeImpact !== downstreamReplacement.changeImpact) {
            propagatePropertyReplacement(downstreamReplacement, resource);
        }
    });
    return theDiff;
}
exports.diffTemplate = diffTemplate;
function isReplacement(impact) {
    return impact === types.ResourceImpact.MAY_REPLACE || impact === types.ResourceImpact.WILL_REPLACE;
}
/**
 * For all properties in 'source' that have a "replacement" impact, propagate that impact to "dest"
 */
function propagatePropertyReplacement(source, dest) {
    for (const [propertyName, diff] of Object.entries(source.propertyUpdates)) {
        if (diff.changeImpact && isReplacement(diff.changeImpact)) {
            // Use the propertydiff of source in target. The result of this happens to be clear enough.
            dest.setPropertyChange(propertyName, diff);
        }
    }
}
function calculateTemplateDiff(currentTemplate, newTemplate) {
    const differences = {};
    const unknown = {};
    for (const key of util_1.unionOf(Object.keys(currentTemplate), Object.keys(newTemplate)).sort()) {
        const oldValue = currentTemplate[key];
        const newValue = newTemplate[key];
        if (util_1.deepEqual(oldValue, newValue)) {
            continue;
        }
        const handler = DIFF_HANDLERS[key]
            || ((_diff, oldV, newV) => unknown[key] = impl.diffUnknown(oldV, newV));
        handler(differences, oldValue, newValue);
    }
    if (Object.keys(unknown).length > 0) {
        differences.unknown = new types.DifferenceCollection(unknown);
    }
    return new types.TemplateDiff(differences);
}
/**
 * Compare two CloudFormation resources and return semantic differences between them
 */
function diffResource(oldValue, newValue) {
    return impl.diffResource(oldValue, newValue);
}
exports.diffResource = diffResource;
/**
 * Replace all references to the given logicalID on the given template, in-place
 *
 * Returns true iff any references were replaced.
 */
function propagateReplacedReferences(template, logicalId) {
    let ret = false;
    function recurse(obj) {
        if (Array.isArray(obj)) {
            obj.forEach(recurse);
        }
        if (typeof obj === 'object' && obj !== null) {
            if (!replaceReference(obj)) {
                Object.values(obj).forEach(recurse);
            }
        }
    }
    function replaceReference(obj) {
        const keys = Object.keys(obj);
        if (keys.length !== 1) {
            return false;
        }
        const key = keys[0];
        if (key === 'Ref') {
            if (obj.Ref === logicalId) {
                obj.Ref = logicalId + ' (replaced)';
                ret = true;
            }
            return true;
        }
        if (key.startsWith('Fn::')) {
            if (Array.isArray(obj[key]) && obj[key].length > 0 && obj[key][0] === logicalId) {
                obj[key][0] = logicalId + '(replaced)';
                ret = true;
            }
            return true;
        }
        return false;
    }
    recurse(template);
    return ret;
}
function deepCopy(x) {
    if (Array.isArray(x)) {
        return x.map(deepCopy);
    }
    if (typeof x === 'object' && x !== null) {
        const ret = {};
        for (const key of Object.keys(x)) {
            ret[key] = deepCopy(x[key]);
        }
        return ret;
    }
    return x;
}
//# sourceMappingURL=data:application/json;base64,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