"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConstructOrder = exports.Construct = exports.Node = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const metadata_1 = require("./metadata");
const dependency_1 = require("./private/dependency");
const stack_trace_1 = require("./private/stack-trace");
const uniqueid_1 = require("./private/uniqueid");
const CONSTRUCT_NODE_PROPERTY_SYMBOL = Symbol.for('constructs.Construct.node');
/**
 * Represents the construct node in the scope tree.
 *
 * @stability stable
 */
class Node {
    /**
     * @stability stable
     */
    constructor(host, scope, id) {
        this.host = host;
        this._locked = false; // if this is "true", addChild will fail
        this._aspects = [];
        this._children = {};
        this._context = {};
        this._metadata = new Array();
        this._dependencies = new Set();
        this.invokedAspects = [];
        this._validations = new Array();
        id = id || ''; // if undefined, convert to empty string
        this.id = sanitizeId(id);
        this.scope = scope;
        // We say that scope is required, but root scopes will bypass the type
        // checks and actually pass in 'undefined'.
        if (scope != null) {
            if (id === '') {
                throw new Error('Only root constructs may have an empty name');
            }
            // Has side effect so must be very last thing in constructor
            Node.of(scope).addChild(host, this.id);
        }
        else {
            // This is a root construct.
            this.id = id;
        }
    }
    /**
     * Returns the node associated with a construct.
     *
     * @param construct the construct.
     * @stability stable
     */
    static of(construct) {
        const node = construct[CONSTRUCT_NODE_PROPERTY_SYMBOL];
        if (!node) {
            throw new Error('construct does not have an associated node. All constructs must extend the "Construct" base class');
        }
        return node;
    }
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     *
     * @stability stable
     */
    get path() {
        const components = this.scopes.slice(1).map(c => Node.of(c).id);
        return components.join(Node.PATH_SEP);
    }
    /**
     * Returns an opaque tree-unique address for this construct.
     *
     * Addresses are 42 characters hexadecimal strings. They begin with "c8"
     * followed by 40 lowercase hexadecimal characters (0-9a-f).
     *
     * Addresses are calculated using a SHA-1 of the components of the construct
     * path.
     *
     * To enable refactorings of construct trees, constructs with the ID `Default`
     * will be excluded from the calculation. In those cases constructs in the
     * same tree may have the same addreess.
     *
     * @stability stable
     * @example
     *
     * c83a2846e506bcc5f10682b564084bca2d275709ee
     */
    get addr() {
        if (!this._addr) {
            this._addr = uniqueid_1.addressOf(this.scopes.map(c => Node.of(c).id));
        }
        return this._addr;
    }
    /**
     * (deprecated) A tree-global unique alphanumeric identifier for this construct.
     *
     * Includes
     * all components of the tree.
     *
     * @deprecated please avoid using this property and use `addr` to form unique names.
     * This algorithm uses MD5, which is not FIPS-complient and also excludes the
     * identity of the root construct from the calculation.
     */
    get uniqueId() {
        const components = this.scopes.slice(1).map(c => Node.of(c).id);
        return components.length > 0 ? uniqueid_1.makeLegacyUniqueId(components) : '';
    }
    /**
     * Return a direct child by id, or undefined.
     *
     * @param id Identifier of direct child.
     * @returns the child if found, or undefined
     * @stability stable
     */
    tryFindChild(id) {
        return this._children[sanitizeId(id)];
    }
    /**
     * Return a direct child by id.
     *
     * Throws an error if the child is not found.
     *
     * @param id Identifier of direct child.
     * @returns Child with the given id.
     * @stability stable
     */
    findChild(id) {
        const ret = this.tryFindChild(id);
        if (!ret) {
            throw new Error(`No child with id: '${id}'`);
        }
        return ret;
    }
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     *
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     *
     * @returns a construct or undefined if there is no default child
     * @stability stable
     * @throws if there is more than one child
     */
    get defaultChild() {
        if (this._defaultChild !== undefined) {
            return this._defaultChild;
        }
        const resourceChild = this.tryFindChild('Resource');
        const defaultChild = this.tryFindChild('Default');
        if (resourceChild && defaultChild) {
            throw new Error(`Cannot determine default child for ${this.path}. There is both a child with id "Resource" and id "Default"`);
        }
        return defaultChild || resourceChild;
    }
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     *
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     *
     * @returns a construct or undefined if there is no default child
     * @stability stable
     * @throws if there is more than one child
     */
    set defaultChild(value) {
        this._defaultChild = value;
    }
    /**
     * All direct children of this construct.
     *
     * @stability stable
     */
    get children() {
        return Object.values(this._children);
    }
    /**
     * Return this construct and all of its children in the given order.
     *
     * @stability stable
     */
    findAll(order = ConstructOrder.PREORDER) {
        const ret = new Array();
        visit(this.host);
        return ret;
        function visit(c) {
            if (order === ConstructOrder.PREORDER) {
                ret.push(c);
            }
            for (const child of Node.of(c).children) {
                visit(child);
            }
            if (order === ConstructOrder.POSTORDER) {
                ret.push(c);
            }
        }
    }
    /**
     * This can be used to set contextual values.
     *
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     *
     * @param key The context key.
     * @param value The context value.
     * @stability stable
     */
    setContext(key, value) {
        if (this.children.length > 0) {
            const names = this.children.map(c => Node.of(c).id);
            throw new Error('Cannot set context after children have been added: ' + names.join(','));
        }
        this._context[key] = value;
    }
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key.
     * @returns The context value or `undefined` if there is no context value for thie key.
     * @stability stable
     */
    tryGetContext(key) {
        const value = this._context[key];
        if (value !== undefined) {
            return value;
        }
        return this.scope && Node.of(this.scope).tryGetContext(key);
    }
    /**
     * An immutable array of metadata objects associated with this construct.
     *
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     *
     * @stability stable
     */
    get metadata() {
        return [...this._metadata];
    }
    /**
     * Adds a metadata entry to this construct.
     *
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata.
     * @param data the value of the metadata (can be a Token).
     * @param fromFunction a function under which to restrict the metadata entry's stack trace (defaults to this.addMetadata).
     * @stability stable
     */
    addMetadata(type, data, fromFunction) {
        if (data == null) {
            return;
        }
        const trace = this.tryGetContext(metadata_1.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA)
            ? undefined
            : stack_trace_1.captureStackTrace(fromFunction || this.addMetadata);
        this._metadata.push({ type, data, trace });
    }
    /**
     * Adds a { "info": <message> } metadata entry to this construct.
     *
     * The toolkit will display the info message when apps are synthesized.
     *
     * @param message The info message.
     * @stability stable
     */
    addInfo(message) {
        this.addMetadata(metadata_1.ConstructMetadata.INFO_METADATA_KEY, message);
    }
    /**
     * Adds a { "warning": <message> } metadata entry to this construct.
     *
     * The toolkit will display the warning when an app is synthesized, or fail
     * if run in --strict mode.
     *
     * @param message The warning message.
     * @stability stable
     */
    addWarning(message) {
        this.addMetadata(metadata_1.ConstructMetadata.WARNING_METADATA_KEY, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     *
     * The toolkit will fail synthesis when errors are reported.
     *
     * @param message The error message.
     * @stability stable
     */
    addError(message) {
        this.addMetadata(metadata_1.ConstructMetadata.ERROR_METADATA_KEY, message);
    }
    /**
     * Applies the aspect to this Constructs node.
     *
     * @stability stable
     */
    applyAspect(aspect) {
        this._aspects.push(aspect);
        return;
    }
    /**
     * All parent scopes of this construct.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be the current construct and the first element will be the root of the
     * tree.
     * @stability stable
     */
    get scopes() {
        const ret = new Array();
        let curr = this.host;
        while (curr) {
            ret.unshift(curr);
            curr = Node.of(curr).scope;
        }
        return ret;
    }
    /**
     * Returns the root of the construct tree.
     *
     * @returns The root of the construct tree.
     * @stability stable
     */
    get root() {
        return this.scopes[0];
    }
    /**
     * Returns true if this construct or the scopes in which it is defined are locked.
     *
     * @stability stable
     */
    get locked() {
        if (this._locked) {
            return true;
        }
        if (this.scope && Node.of(this.scope).locked) {
            return true;
        }
        return false;
    }
    /**
     * Add an ordering dependency on another Construct.
     *
     * All constructs in the dependency's scope will be deployed before any
     * construct in this construct's scope.
     *
     * @stability stable
     */
    addDependency(...dependencies) {
        for (const dependency of dependencies) {
            this._dependencies.add(dependency);
        }
    }
    /**
     * Return all dependencies registered on this node or any of its children.
     *
     * @stability stable
     */
    get dependencies() {
        const found = new Map(); // Deduplication map
        const ret = new Array();
        for (const source of this.findAll()) {
            for (const dependable of Node.of(source)._dependencies) {
                for (const target of dependency_1.DependableTrait.get(dependable).dependencyRoots) {
                    let foundTargets = found.get(source);
                    if (!foundTargets) {
                        found.set(source, foundTargets = new Set());
                    }
                    if (!foundTargets.has(target)) {
                        ret.push({ source, target });
                        foundTargets.add(target);
                    }
                }
            }
        }
        return ret;
    }
    /**
     * (experimental) Remove the child with the given name, if present.
     *
     * @returns Whether a child with the given name was deleted.
     * @experimental
     */
    tryRemoveChild(childName) {
        if (!(childName in this._children)) {
            return false;
        }
        delete this._children[childName];
        return true;
    }
    /**
     * Adds a validation to this construct.
     *
     * When `node.validate()` is called, the `validate()` method will be called on
     * all validations and all errors will be returned.
     *
     * @stability stable
     */
    addValidation(validation) {
        this._validations.push(validation);
    }
    /**
     * Synthesizes a CloudAssembly from a construct tree.
     *
     * @param options Synthesis options.
     * @stability stable
     */
    synthesize(options) {
        // the three holy phases of synthesis: prepare, validate and synthesize
        // prepare
        this.prepare();
        // validate
        const validate = options.skipValidation === undefined ? true : !options.skipValidation;
        if (validate) {
            const errors = this.validate();
            if (errors.length > 0) {
                const errorList = errors.map(e => `[${Node.of(e.source).path}] ${e.message}`).join('\n  ');
                throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
            }
        }
        // synthesize (leaves first)
        for (const construct of this.findAll(ConstructOrder.POSTORDER)) {
            const node = Node.of(construct);
            try {
                node._lock();
                const ctx = {
                    ...options.sessionContext,
                    outdir: options.outdir,
                };
                construct.onSynthesize(ctx); // "as any" is needed because we want to keep "synthesize" protected
            }
            finally {
                node._unlock();
            }
        }
    }
    /**
     * Invokes "prepare" on all constructs (depth-first, post-order) in the tree under `node`.
     *
     * @stability stable
     */
    prepare() {
        // Aspects are applied root to leaf
        for (const construct of this.findAll(ConstructOrder.PREORDER)) {
            Node.of(construct).invokeAspects();
        }
        // since constructs can be added to the tree during invokeAspects, call findAll() to recreate the list.
        // use PREORDER.reverse() for backward compatability
        for (const construct of this.findAll(ConstructOrder.PREORDER).reverse()) {
            const cn = construct;
            if ('onPrepare' in cn) {
                if (typeof (cn.onPrepare) !== 'function') {
                    throw new Error('expecting "onPrepare" to be a function');
                }
                cn.onPrepare();
            }
        }
    }
    /**
     * Validates tree (depth-first, pre-order) and returns the list of all errors.
     *
     * An empty list indicates that there are no errors.
     *
     * @stability stable
     */
    validate() {
        let errors = new Array();
        for (const child of this.children) {
            errors = errors.concat(Node.of(child).validate());
        }
        const localErrors = this.host.onValidate(); // "as any" is needed because we want to keep "validate" protected
        // invoke validations
        for (const v of this._validations) {
            localErrors.push(...v.validate());
        }
        return errors.concat(localErrors.map(msg => ({ source: this.host, message: msg })));
    }
    /**
     * Locks this construct from allowing more children to be added. After this
     * call, no more children can be added to this construct or to any children.
     * @internal
     */
    _lock() {
        this._locked = true;
    }
    /**
     * Unlocks this costruct and allows mutations (adding children).
     * @internal
     */
    _unlock() {
        this._locked = false;
    }
    /**
     * Adds a child construct to this node.
     *
     * @param child The child construct
     * @param childName The type name of the child construct.
     * @returns The resolved path part name of the child
     */
    addChild(child, childName) {
        if (this.locked) {
            // special error if root is locked
            if (!this.path) {
                throw new Error('Cannot add children during synthesis');
            }
            throw new Error(`Cannot add children to "${this.path}" during synthesis`);
        }
        if (childName in this._children) {
            const name = this.id || '';
            const typeName = this.host.constructor.name;
            throw new Error(`There is already a Construct with name '${childName}' in ${typeName}${name.length > 0 ? ' [' + name + ']' : ''}`);
        }
        this._children[childName] = child;
    }
    /**
     * Triggers each aspect to invoke visit
     */
    invokeAspects() {
        const descendants = this.findAll();
        for (const aspect of this._aspects) {
            if (this.invokedAspects.includes(aspect)) {
                continue;
            }
            descendants.forEach(member => aspect.visit(member));
            this.invokedAspects.push(aspect);
        }
    }
}
exports.Node = Node;
_a = JSII_RTTI_SYMBOL_1;
Node[_a] = { fqn: "constructs.Node", version: "3.3.75" };
/**
 * Separator used to delimit construct path components.
 *
 * @stability stable
 */
Node.PATH_SEP = '/';
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 *
 * @stability stable
 */
class Construct {
    /**
     * Creates a new construct node.
     *
     * @param scope The scope in which to define this construct.
     * @param id The scoped construct ID.
     * @param options Options.
     * @stability stable
     */
    constructor(scope, id, options = {}) {
        var _c;
        // attach the construct to the construct tree by creating a node
        const nodeFactory = (_c = options.nodeFactory) !== null && _c !== void 0 ? _c : { createNode: (host, nodeScope, nodeId) => new Node(host, nodeScope, nodeId) };
        Object.defineProperty(this, CONSTRUCT_NODE_PROPERTY_SYMBOL, {
            value: nodeFactory.createNode(this, scope, id),
            enumerable: false,
            configurable: false,
        });
        // implement IDependable privately
        dependency_1.DependableTrait.implement(this, {
            dependencyRoots: [this],
        });
    }
    /**
     * Returns a string representation of this construct.
     *
     * @stability stable
     */
    toString() {
        return Node.of(this).path || '<root>';
    }
    /**
     * (deprecated) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if there the construct is valid.
     * @deprecated use `Node.addValidation()` to subscribe validation functions on this construct
     * instead of overriding this method.
     */
    onValidate() {
        return [];
    }
    /**
     * Perform final modifications before synthesis.
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     *
     * @stability stable
     */
    onPrepare() {
        return;
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     * @stability stable
     */
    onSynthesize(session) {
        ignore(session);
    }
}
exports.Construct = Construct;
_b = JSII_RTTI_SYMBOL_1;
Construct[_b] = { fqn: "constructs.Construct", version: "3.3.75" };
/**
 * In what order to return constructs.
 *
 * @stability stable
 */
var ConstructOrder;
(function (ConstructOrder) {
    ConstructOrder[ConstructOrder["PREORDER"] = 0] = "PREORDER";
    ConstructOrder[ConstructOrder["POSTORDER"] = 1] = "POSTORDER";
})(ConstructOrder = exports.ConstructOrder || (exports.ConstructOrder = {}));
function ignore(_x) {
    return;
}
// Import this _after_ everything else to help node work the classes out in the correct order...
const PATH_SEP_REGEX = new RegExp(`${Node.PATH_SEP}`, 'g');
/**
 * Return a sanitized version of an arbitrary string, so it can be used as an ID
 */
function sanitizeId(id) {
    // Escape path seps as double dashes
    return id.replace(PATH_SEP_REGEX, '--');
}
//# sourceMappingURL=data:application/json;base64,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