"""
All known protocol constants.

Contains following types of constants:

 * `APP_`, application identifiers, e.g. `APP_3GPP_GX`
 * `VENDOR_`, vendor codes, e.g. `VENDOR_NOKIA`
 * `CMD_`, diameter command codes, e.g. `CMD_ACCOUNTING`
 * `AVP_`, diameter AVPs. These are either labeled `AVP_<name>`, for the
    diameter base protocol (e.g. `AVP_SERVICE_TYPE`), or `AVP_<vendor>_<name>`,
    for the vendor-specific AVPs (e.g. `AVP_TGPP_GGSN_ADDRESS`)
 * `E_`, for all the enumerations, e.g. `E_DISCONNECT_CAUSE_BUSY`

"""

# All known Application IDs
APP_DIAMETER_COMMON_MESSAGES = 0
APP_DIAMETER_BASE_ACCOUNTING = 3
APP_DIAMETER_QOS_APPLICATION = 9
APP_DIAMETER_CAPABILITIES_UPDATE = 10
APP_DIAMETER_IKE_SK_IKESK = 11
APP_DIAMETER_NAT_CONTROL_APPLICATION = 12
APP_DIAMETER_ERP = 13
APP_3GPP_RE_RF = 16777218
APP_3GPP_WX = 16777219
APP_3GPP_ZN = 16777220
APP_3GPP_ZH = 16777221
# Old Diameter Gx Auth-App-Id, until R7 spec was released, see ETSI TS 129 210
APP_3GPP_GX_R6 = 16777224
APP_3GPP_GX_OVER_GY = 16777225
APP_3GPP_MM10 = 16777226
APP_3GPP_RX_RELEASE_6 = 16777229
APP_3GPP_PR = 16777230
APP_ITU_T_RS = 16777235
# This is the current Diameter Gx Auth-App-Id as of 3GPP R7, see ETSI TS 129 212 V
APP_3GPP_GX = 16777238
APP_JUNIPER_CLUSTER = 16777239
APP_JUNIPER_POLICY_CONTROL_AAA = 16777240
APP_IPTEGO_USPI = 16777241
APP_COVERGENCE_SPECIFIC_SIP_ROUTING = 16777242
APP_POLICY_PROCESSING = 16777243
APP_JUNIPER_POLICY_CONTROL_JSRC = 16777244
APP_ITU_T_S_TC1 = 16777245
APP_3GPP2_CAN_ACCESS_AUTHENTICATION_AND_AUTHORIZATION = 16777247
APP_3GPP2_WLAN_INTERWORKING_ACCESS_AUTHENTICATION_AND_AUTHORIZATION = 16777248
APP_3GPP2_WLAN_INTERWORKING_ACCOUNTING = 16777249
APP_3GPP_STA = 16777250
APP_3GPP_S6A_S6D = 16777251
APP_3GPP_S13_S13 = 16777252
APP_ETSI_RE = 16777253
APP_ETSI_GOCAP = 16777254
APP_3GPP_SLG = 16777255
APP_ITU_T_RW = 16777256
APP_ETSI_A4 = 16777257
APP_ITU_T_RT = 16777258
APP_CARA = 16777259
APP_CAMA = 16777260
APP_FEMTOCELL_EXTENSION_TO_DIAMETER_EAP_APPLICATION = 16777261
APP_ITU_T_RU = 16777262
APP_ITU_T_NG = 16777263
APP_3GPP_SWM = 16777264
APP_3GPP_SWX = 16777265
APP_3GPP_GXX = 16777266
APP_3GPP_ZPN = 16777268
APP_JUNIPER_EXAMPLE = 16777270
APP_ITU_T_RI = 16777271
APP_3GPP_S6B = 16777272
APP_JUNIPER_JGX = 16777273
APP_ITU_T_RD = 16777274
APP_ADMI_NOTIFICATION_APPLICATION = 16777275
APP_ADMI_MESSAGING_INTERFACE_APPLICATION = 16777276
APP_PETER_SERVICE_VSI = 16777277
APP_ETSI_RR_REQUEST_MODEL = 16777278
APP_ETSI_RR_DELEGATED_MODEL = 16777279
APP_3GPP_SLH = 16777291
APP_CLOUDMARK_DIAMETER_INTERFACE = 16777293
APP_CAMIANT_DRMA = 16777294
APP_PILTE_INTERWORKING_DIAMETER_APPLICATION = 16777295
APP_JUNIPER_SESSIONS_RECOVERY = 16777296
APP_VEDICIS_LIVEPROXY = 16777297
APP_PI_3GPP2_DIAMETER_APPLICATION = 16777298
APP_SANDVINE_RFPLUS = 16777299
APP_SUBSCRIPTION_INFORMATION_APPLICATION = 16777300
APP_3GPP_SY = 16777302
APP_3GPP_SD = 16777303
APP_HP_DIAMETER_TOPOLOGY_DISCOVERY = 16777305
APP_ITU_T_M9 = 16777306
APP_ITU_T_M13 = 16777307
APP_3GPP_S7A = 16777308
APP_3GPP_TSP = 16777309
APP_3GPP_S6M = 16777310
APP_3GPP_T4 = 16777311
APP_3GPP_SGD = 16777313
APP_INTRADO_SLG = 16777314
APP_VERIZON_FEMTO_LOC = 16777316
APP_3GPP_S15 = 16777318
APP_3GPP_S9A = 16777319
APP_3GPP_S9A_STAR = 16777320
APP_GATEWAY_LOCATION_APPLICATION = 16777321
APP_VERIZON_SESSION_RECOVERY = 16777322
APP_MAGIC_CLIENT_INTERFACE_PROTOCOL_CIP = 16777324
APP_ITU_T_NC = 16777325
APP_ITU_T_NE = 16777326
APP_RIVADA_XD = 16777329
APP_RIVADA_XM = 16777330
APP_RIVADA_XH = 16777331
APP_RIVADA_XF = 16777332
APP_RIVADA_XP = 16777333
APP_RIVADA_XA = 16777334
APP_3GPP_MB2C = 16777335
APP_3GPP_PC2 = 16777337
APP_JUNIPER_DOMAIN_POLICY = 16777338
APP_HOST_OBSERVER = 16777339
APP_3GPP_PC6_PC7 = 16777340
APP_3GPP_NP = 16777342
APP_S6B_APPLICATION_ONE_AAA = 16777999
APP_3GPP_S6T = 16777345
APP_RELAY = 4294967295
APP_NASREQ_APPLICATION = 1
APP_EAP_APPLICATION = 5
APP_MOBILE_IPV4_APPLICATION = 2
APP_DIAMETER_CREDIT_CONTROL_APPLICATION = 4
APP_SUN_PING_APPLICATION = 555
APP_3GPP_GMB = 16777223
APP_3GPP_GQ = 16777222
APP_3GPP_RX = 16777236
APP_3GPP_CX = 16777216
APP_3GPP_SH = 16777217
APP_3GPP_SGMB = 16777292
APP_3GPP_S9 = 16777267
APP_3GPP_S6C = 16777312
APP_3GPP_PC4A = 16777336
APP_3GPP_T6A_T6B = 16777346
APP_3GPP2_M1 = 16777323
APP_3GPP2_TY = 16777237
APP_DIAMETER_SESSION_INITIATION_PROTOCOL_SIP_APPLICATION = 6
APP_DIAMETER_E2E4_APPLICATION = 16777231
APP_ERICSSON_SCAP = 19302
APP_ERICSSON_SRAP = 19310
APP_ERICSSON_MSI = 16777227
APP_ERICSSON_ZX = 16777228
APP_ERICSSON_CHARGING_CIP = 16777232
APP_ERICSSON_MM = 16777233
APP_ERICSSON_HSI = 16777269
APP_ERICSSON_CHARGING_DCIP = 16777301
APP_ERICSSON_SY = 16777304
APP_ERICSSON_DIAMETER_SIGNALLING_CONTROLLER_APPLICATION_DSC = 16777315
APP_ERICSSON_SX = 16777327
APP_DIAMETER_MOBILE_IPV6_IKE_MIP6I = 7
APP_DIAMETER_MOBILE_IPV6_AUTH_MIP6A = 8
APP_VODAFONE_GX = 16777234
APP_ALU_SY = 111
APP_NOKIA_SERVICE_EXTENSION_NSE = 16777328
APP_NOKIA_SDR = 16777341
APP_NOKIA_SOLUTIONS_AND_NETWORKS_NSN_HD_APPLICATION = 16777317
APP_NSN_UNIFIED_CHARGING_TRIGGER_FUNCTION_UCTF = 16777246

# All known Vendor IDs
VENDOR_NONE = 0
VENDOR_MERIT = 61
VENDOR_USR = 429
VENDOR_LUCENT = 1751
VENDOR_DEUTSCHE_TELEKOM_AG = 2937
VENDOR_ACISION = 3830
VENDOR_SKT = 5806
VENDOR_TGPP = 10415
VENDOR_ETSI = 13019
VENDOR_TANGO = 13421
VENDOR_CHINATELECOM = 81000
VENDOR_TGPPCX = 16777216
VENDOR_SUN = 42
VENDOR_TGPP2 = 5535
VENDOR_ERICSSON = 193
VENDOR_CISCO = 5771
VENDOR_STARENT = 8164
VENDOR_VODAFONE = 12645
VENDOR_ALU = 637
VENDOR_NOKIA = 94
VENDOR_NOKIA_SOLUTIONS = 28458
VENDOR_HP = 11
VENDOR_ORACLETEKELEC = 323
VENDOR_CISCOSYSTEMS = 9
VENDOR_JUNIPER = 2636
VENDOR_INOVAR = 41897
VENDOR_HUAWEI = 2011
VENDOR_VERIZONWIRELESS = 12951
VENDOR_TELEFONICA_MOVILES = 27640
VENDOR_SIEMENS_AG = 4329
VENDOR_MICROSOFT = 311
VENDOR_METASWITCH = 19444
VENDOR_ONEM2M = 45687
VENDOR_UNKNOWN = 99999

# Application ID-to-name mapping
APPLICATIONS = {
    0: "Diameter Common Messages",
    3: "Diameter Base Accounting",
    9: "Diameter QoS application",
    10: "Diameter Capabilities Update",
    11: "Diameter IKE SK (IKESK)",
    12: "Diameter NAT Control Application",
    13: "Diameter ERP",
    16777218: "3GPP Re/Rf",
    16777219: "3GPP Wx",
    16777220: "3GPP Zn",
    16777221: "3GPP Zh",
    16777224: "3GPP Gx",
    16777225: "3GPP Gx over Gy",
    16777226: "3GPP MM10",
    16777229: "3GPP Rx Release 6",
    16777230: "3GPP Pr",
    16777235: "ITU-T Rs",
    16777238: "3GPP Gx",
    16777239: "Juniper Cluster",
    16777240: "Juniper Policy-Control-AAA",
    16777241: "iptego USPI",
    16777242: "Covergence-specific SIP routing",
    16777243: "Policy Processing",
    16777244: "Juniper Policy-Control-JSRC",
    16777245: "ITU-T S-TC1",
    16777247: "3GPP2 CAN Access Authentication and Authorization",
    16777248: "3GPP2 WLAN Interworking Access Authentication and Authorization",
    16777249: "3GPP2 WLAN Interworking Accounting",
    16777250: "3GPP STa",
    16777251: "3GPP S6a/S6d",
    16777252: "3GPP S13/S13'",
    16777253: "ETSI Re",
    16777254: "ETSI GOCAP",
    16777255: "3GPP SLg",
    16777256: "ITU-T Rw",
    16777257: "ETSI a4",
    16777258: "ITU-T Rt",
    16777259: "CARA",
    16777260: "CAMA",
    16777261: "Femtocell extension to Diameter EAP Application",
    16777262: "ITU-T Ru",
    16777263: "ITU-T Ng",
    16777264: "3GPP SWm",
    16777265: "3GPP SWx",
    16777266: "3GPP Gxx",
    16777268: "3GPP Zpn",
    16777270: "Juniper-Example",
    16777271: "ITU-T Ri",
    16777272: "3GPP S6b",
    16777273: "Juniper JGx",
    16777274: "ITU-T Rd",
    16777275: "ADMI Notification Application",
    16777276: "ADMI Messaging Interface Application",
    16777277: "Peter-Service VSI",
    16777278: "ETSI Rr request model",
    16777279: "ETSI Rr delegated model",
    16777291: "3GPP SLh",
    16777293: "Cloudmark Diameter Interface",
    16777294: "Camiant DRMA",
    16777295: "PiLTE Interworking Diameter Application",
    16777296: "Juniper-Sessions-Recovery",
    16777297: "Vedicis LiveProxy",
    16777298: "Pi*3GPP2 Diameter Application",
    16777299: "Sandvine Rf+",
    16777300: "Subscription Information Application",
    16777302: "3GPP Sy",
    16777303: "3GPP Sd",
    16777305: "HP Diameter Topology Discovery",
    16777306: "ITU-T M9",
    16777307: "ITU-T M13",
    16777308: "3GPP S7a",
    16777309: "3GPP Tsp",
    16777310: "3GPP S6m",
    16777311: "3GPP T4",
    16777313: "3GPP SGd",
    16777314: "Intrado-SLg",
    16777316: "Verizon-Femto-Loc",
    16777318: "3GPP S15",
    16777319: "3GPP S9a",
    16777320: "3GPP S9a*",
    16777321: "Gateway Location Application",
    16777322: "Verizon Session Recovery",
    16777324: "MAGIC Client Interface Protocol (CIP)",
    16777325: "ITU-T Nc",
    16777326: "ITU-T Ne",
    16777329: "Rivada Xd",
    16777330: "Rivada Xm",
    16777331: "Rivada Xh",
    16777332: "Rivada Xf",
    16777333: "Rivada Xf",
    16777334: "Rivada Xp",
    16777335: "3GPP MB2c",
    16777337: "3GPP PC2",
    16777338: "Juniper Domain Policy",
    16777339: "Host Observer",
    16777340: "3GPP PC6/PC7",
    16777342: "3GPP Np",
    16777999: "S6b Application (One-AAA)",
    16777345: "3GPP S6t",
    4294967295: "Relay",
    1: "NASREQ Application",
    5: "EAP Application",
    2: "Mobile IPv4 Application",
    4: "Diameter Credit Control Application",
    555: "Sun Ping Application",
    16777223: "3GPP Gmb",
    16777222: "3GPP Gq",
    16777236: "3GPP Rx",
    16777216: "3GPP Cx",
    16777217: "3GPP Sh",
    16777292: "3GPP SGmb",
    16777267: "3GPP S9",
    16777312: "3GPP S6c",
    16777336: "3GPP PC4a",
    16777346: "3GPP T6a/T6b",
    16777323: "3GPP2 M1",
    16777237: "3GPP2 Ty",
    6: "Diameter Session Initiation Protocol (SIP) Application",
    16777231: "Diameter e2e4 Application",
    19302: "Ericsson SCAP",
    19310: "Ericsson SRAP",
    16777227: "Ericsson MSI",
    16777228: "Ericsson Zx",
    16777232: "Ericsson Charging-CIP",
    16777233: "Ericsson Mm",
    16777269: "Ericsson HSI",
    16777301: "Ericsson Charging-DCIP",
    16777304: "Ericsson Sy",
    16777315: "Ericsson Diameter Signalling Controller Application (DSC)",
    16777327: "Ericsson Sx",
    7: "Diameter Mobile IPv6 IKE   (MIP6I)",
    8: "Diameter Mobile IPv6 Auth  (MIP6A)",
    16777234: "Vodafone Gx+",
    111: "ALU Sy",
    16777328: "Nokia Service Extension, NSE",
    16777341: "Nokia Sdr",
    16777317: "Nokia Solutions and Networks (NSN) Hd Application",
    16777246: "NSN Unified Charging Trigger Function (UCTF)"}

# Vendor ID-to-name mapping
VENDORS = {
    0: "None",
    61: "Merit",
    429: "USR",
    1751: "Lucent",
    2937: "Deutsche_Telekom_AG",
    3830: "Acision",
    5806: "SKT",
    10415: "TGPP",
    13019: "ETSI",
    13421: "Tango",
    81000: "ChinaTelecom",
    16777216: "TGPPCX",
    42: "Sun",
    5535: "TGPP2",
    193: "Ericsson",
    5771: "Cisco",
    8164: "Starent",
    12645: "Vodafone",
    637: "ALU",
    94: "Nokia",
    28458: "NokiaSolutionsAndNetworks",
    11: "HP",
    323: "OracleTekelec",
    9: "CiscoSystems",
    2636: "Juniper",
    41897: "Inovar",
    2011: "Huawei",
    12951: "VerizonWireless",
    27640: "Telefonica_Moviles_Espa_S.A.",
    4329: "Siemens_AG",
    311: "Microsoft",
    19444: "Metaswitch",
    45687: "oneM2M"}

# All known command codes
CMD_CAPABILITIES_EXCHANGE = 257
CMD_RE_AUTH = 258
CMD_ACCOUNTING = 271
CMD_ABORT_SESSION = 274
CMD_SESSION_TERMINATION = 275
CMD_DEVICE_WATCHDOG = 280
CMD_DISCONNECT_PEER = 282
CMD_BOOSTRAPPING_INFO = 310
CMD_MESSAGE_PROCESS = 311
CMD_GBAPUSH_INFO = 312
CMD_POLICY_DATA = 314
CMD_POLICY_INSTALL = 315
CMD_3GPP_UPDATE_LOCATION = 316
CMD_3GPP_CANCEL_LOCATION = 317
CMD_3GPP_AUTHENTICATION_INFORMATION = 318
CMD_3GPP_INSERT_SUBSCRIBER_DATA = 319
CMD_3GPP_DELETE_SUBSCRIBER_DATA = 320
CMD_3GPP_PURGE_UE = 321
CMD_3GPP_RESET = 322
CMD_3GPP_NOTIFY = 323
CMD_3GPP_ME_IDENTITY_CHECK = 324
CMD_MIP6 = 325
CMD_QOS_AUTHORIZATION = 326
CMD_QOS_INSTALL = 327
CMD_CAPABILITIES_UPDATE = 328
CMD_IKEV2_SK = 329
CMD_NAT_CONTROL = 330
CMD_3GPP_PROVIDE_LOCATION = 8388620
CMD_3GPP_LOCATION_REPORT = 8388621
CMD_3GPP_LCS_ROUTING_INFO = 8388622
CMD_SUBSCRIPTION_INFORMATION_APPLICATION = 8388631
CMD_DISTRIBUTED_CHARGING = 8388632
CMD_ERICSSON_SL = 8388633
CMD_ERICSSON_SN = 8388634
CMD_SPENDING_LIMIT = 8388635
CMD_SPENDING_STATUS_NOTIFICATION = 8388636
CMD_TDF_SESSION = 8388637
CMD_3GPP_UPDATE_VCSG_LOCATION = 8388638
CMD_3GPP_DEVICE_ACTION = 8388639
CMD_3GPP_DEVICE_NOTIFICATION = 8388640
CMD_3GPP_SUBSCRIBER_INFORMATION = 8388641
CMD_CANCEL_VCSG_LOCATION = 8388642
CMD_3GPP_DEVICE_TRIGGER = 8388643
CMD_3GPP_DELIVERY_REPORT = 8388644
CMD_MO_FORWARD_SHORT_MESSAGE = 8388645
CMD_MT_FORWARD_SHORT_MESSAGE = 8388646
CMD_SEND_ROUTING_INFO_FOR_SM = 8388647
CMD_ALERT_SERVICE_CENTRE = 8388648
CMD_REPORT_SM_DELIVERY_STATUS = 8388649
CMD_NSN_CANCEL_LOCATIONMS = 8388650
CMD_NSN_USER_DATAMS = 8388651
CMD_NSN_PROFILE_UPDATEMS = 8388652
CMD_NSN_SUBSCRIBE_NOTIFICATIONSMS = 8388653
CMD_NSN_PUSH_NOTIFICATIONMS = 8388654
CMD_GET_GATEWAY = 8388655
CMD_TRIGGER_ESTABLISHMENT = 8388656
CMD_ERICSSON_BINDING_DATA = 8388657
CMD_ERICSSON_TRACE_REPORT = 8388717
CMD_CONFIGURATION_INFORMATION = 8388718
CMD_REPORTING_INFORMATION = 8388719
CMD_NON_AGGREGATED_RUCI_REPORT = 8388720
CMD_AGGREGATED_RUCI_REPORT = 8388721
CMD_MODIFY_UECONTEXT = 8388722
CMD_NIDD_INFORMATION = 8388726
CMD_CONNECTION_MANAGEMENT = 8388732
CMD_MO_DATA = 8388733
CMD_MT_DATA = 8388734
CMD_AA = 265
CMD_DIAMETER_EAP = 268
CMD_AA_MOBILE_NODE = 260
CMD_HOME_AGENT_MIP = 262
CMD_CREDIT_CONTROL = 272
CMD_SUN_PING = 511
CMD_TGPP_USER_AUTHORIZATION = 300
CMD_TGPP_SERVER_ASSIGNMENT = 301
CMD_TGPP_LOCATION_INFO = 302
CMD_TGPP_MULTIMEDIA_AUTH = 303
CMD_TGPP_REGISTRATION_TERMINATION = 304
CMD_TGPP_PUSH_PROFILE = 305
CMD_TGPP_USER_DATA = 306
CMD_TGPP_PROFILE_UPDATE = 307
CMD_TGPP_SUBSCRIBE_NOTIFICATIONS = 308
CMD_TGPP_PUSH_NOTIFICATION = 309
CMD_TGPP_GCS_ACTION = 8388662
CMD_TGPP_GCS_NOTIFICATION = 8388663
CMD_TGPP_PROSE_SUBSCRIBER_INFORMATION = 8388664
CMD_TGPP_UPDATE_PROSE_SUBSCRIBER_DATA = 8388665
CMD_TGPP_PROSE_NOTIFY = 8388666
CMD_TGPP_RESET = 8388667
CMD_TGPP_PROSE_INITIAL_LOCATION_INFORMATION = 8388713
CMD_USER_AUTHORIZATION = 283
CMD_SERVER_ASSIGNMENT = 284
CMD_LOCATION_INFO = 285
CMD_MULTIMEDIA_AUTH = 286
CMD_REGISTRATION_TERMINATION = 287
CMD_PUSH_PROFILE = 288
CMD_MIP6_REQUEST_ANSWER = 325
CMD_HP_PEER_INFORMATION = 100
CMD_HP_FETCH_PEERS = 101
CMD_HP_SUBSCRIBE_CHANGE = 102
CMD_HP_NOTIFY_CHANGE = 103

# 3GPP specific values for "Service-Context-Id" AVP, from TS 32.299 version 16.2.0
SERVICE_CONTEXT_PS_CHARGING = "32251@3gpp.org"
SERVICE_CONTEXT_CPDT_CHARGING = "32253@3gpp.org"
SERVICE_CONTEXT_EXPOSURE_FUNCTION_API_CHARGING = "32254@3gpp.org"
SERVICE_CONTEXT_IMS_CHARGING = "32260@3gpp.org"
SERVICE_CONTEXT_MMS_SERVICE_CHARGING = "32270@3gpp.org"
SERVICE_CONTEXT_LCS_SERVICE_CHARGING = "32271@3gpp.org"
SERVICE_CONTEXT_POC_SERVICE_CHARGING = "32272@3gpp.org"
SERVICE_CONTEXT_MBMS_SERVICE_CHARGING = "32273@3gpp.org"
SERVICE_CONTEXT_SMS_SERVICE_CHARGING = "32274@3gpp.org"
SERVICE_CONTEXT_MMTEL_SERVICE_CHARGING = "32275@3gpp.org"
SERVICE_CONTEXT_VCS_CHARGING = "32276@3gpp.org"
SERVICE_CONTEXT_PROSE_CHARGING = "32277@3gpp.org"
SERVICE_CONTEXT_MONITORING_EVENT_CHARGING = "32278@3gpp.org"
SERVICE_CONTEXT_AOC_SERVICE_INFORMATION = "32280@3gpp.org"
SERVICE_CONTEXT_ABMF_RC_INFORMATION = "32296@3gpp.org"

SERVICE_CONTEXTS = {
    SERVICE_CONTEXT_PS_CHARGING: "PS charging",
    SERVICE_CONTEXT_CPDT_CHARGING: "CPDT charging",
    SERVICE_CONTEXT_EXPOSURE_FUNCTION_API_CHARGING: "Exposure function API charging",
    SERVICE_CONTEXT_IMS_CHARGING: "IMS charging",
    SERVICE_CONTEXT_MMS_SERVICE_CHARGING: "MMS service charging",
    SERVICE_CONTEXT_LCS_SERVICE_CHARGING: "LCS service charging",
    SERVICE_CONTEXT_POC_SERVICE_CHARGING: "PoC service charging",
    SERVICE_CONTEXT_MBMS_SERVICE_CHARGING: "MBMS service charging",
    SERVICE_CONTEXT_SMS_SERVICE_CHARGING: "SMS service charging",
    SERVICE_CONTEXT_MMTEL_SERVICE_CHARGING: "MMTel service charging",
    SERVICE_CONTEXT_VCS_CHARGING: "VCS charging",
    SERVICE_CONTEXT_PROSE_CHARGING: "ProSe charging",
    SERVICE_CONTEXT_MONITORING_EVENT_CHARGING: "Monitoring Event charging",
    SERVICE_CONTEXT_AOC_SERVICE_INFORMATION: "AoC Service Information",
    SERVICE_CONTEXT_ABMF_RC_INFORMATION: "ABMF Rc information"
}

# All known AVPs
AVP_USER_NAME = 1
AVP_USER_PASSWORD = 2
AVP_CHAP_PASSWORD = 3
AVP_NAS_IP_ADDRESS = 4
AVP_NAS_PORT = 5
AVP_SERVICE_TYPE = 6
E_SERVICE_TYPE_UNKNOWN = 0
E_SERVICE_TYPE_LOGIN = 1
E_SERVICE_TYPE_FRAMED = 2
E_SERVICE_TYPE_CALLBACK_LOGIN = 3
E_SERVICE_TYPE_CALLBACK_FRAMED = 4
E_SERVICE_TYPE_OUTBOUND = 5
E_SERVICE_TYPE_ADMINISTRATIVE = 6
E_SERVICE_TYPE_NAS_PROMPT = 7
E_SERVICE_TYPE_AUTHENTICATE_ONLY = 8
E_SERVICE_TYPE_CALLBACK_NAS_PROMPT = 9
E_SERVICE_TYPE_CALL_CHECK = 10
E_SERVICE_TYPE_CALLBACK_ADMINISTRATIVE = 11
E_SERVICE_TYPE_VOICE = 12
E_SERVICE_TYPE_FAX = 13
E_SERVICE_TYPE_MODEM_RELAY = 14
E_SERVICE_TYPE_IAPP_REGISTER = 15
E_SERVICE_TYPE_IAPP_AP_CHECK = 16
E_SERVICE_TYPE_AUTHORIZE_ONLY = 17
E_SERVICE_TYPE_FRAMED_MANAGEMENT = 18

AVP_FRAMED_PROTOCOL = 7
E_FRAMED_PROTOCOL_PPP = 1
E_FRAMED_PROTOCOL_SLIP = 2
E_FRAMED_PROTOCOL_ARAP = 3
E_FRAMED_PROTOCOL_GANDALF = 4
E_FRAMED_PROTOCOL_XYLOGICS = 5
E_FRAMED_PROTOCOL_X_75 = 6
E_FRAMED_PROTOCOL_GPRS_PDP_CONTEXT = 7
E_FRAMED_PROTOCOL_ASCEND_ARA = 255
E_FRAMED_PROTOCOL_MPP = 256
E_FRAMED_PROTOCOL_EURAW = 257
E_FRAMED_PROTOCOL_EUUI = 258
E_FRAMED_PROTOCOL_X25 = 259
E_FRAMED_PROTOCOL_COMB = 260
E_FRAMED_PROTOCOL_FR = 261

AVP_FRAMED_IP_ADDRESS = 8
AVP_FRAMED_IP_NETMASK = 9
AVP_FRAMED_ROUTING = 10
E_FRAMED_ROUTING_NONE = 0
E_FRAMED_ROUTING_SEND_ROUTING_PACKETS = 1
E_FRAMED_ROUTING_LISTEN_FOR_ROUTING_PACKETS = 2
E_FRAMED_ROUTING_SEND_AND_LISTEN = 3

AVP_FILTER_ID = 11
AVP_FRAMED_MTU = 12
AVP_FRAMED_COMPRESSION = 13
E_FRAMED_COMPRESSION_NONE = 0
E_FRAMED_COMPRESSION_VAN_JACOBSON_TCP_IP_HEADER_COMPRESSION = 1
E_FRAMED_COMPRESSION_IPX_HEADER_COMPRESSION = 2
E_FRAMED_COMPRESSION_STAC_LZS_COMPRESSION = 3

AVP_LOGIN_IP_HOST = 14
AVP_LOGIN_SERVICE = 15
E_LOGIN_SERVICE_TELNET = 0
E_LOGIN_SERVICE_RLOGIN = 1
E_LOGIN_SERVICE_TCP_CLEAR = 2
E_LOGIN_SERVICE_PORTMASTER = 3
E_LOGIN_SERVICE_LAT = 4
E_LOGIN_SERVICE_X25_PAD = 5
E_LOGIN_SERVICE_X25_T3POS = 6
E_LOGIN_SERVICE_UNASSIGNED = 7
E_LOGIN_SERVICE_TCP_CLEAR_QUIET_SUPPRESSES_ANY_NAS_GENERATED_CONNECT_STRING = 8

AVP_LOGIN_TCP_PORT = 16
AVP_REPLY_MESSAGE = 18
AVP_CALLBACK_NUMBER = 19
AVP_CALLBACK_ID = 20
AVP_FRAMED_ROUTE = 22
AVP_FRAMED_IPX_NETWORK = 23
AVP_STATE = 24
AVP_CLASS = 25
AVP_VENDOR_SPECIFIC = 26
AVP_SESSION_TIMEOUT = 27
AVP_IDLE_TIMEOUT = 28
AVP_TERMINATION_ACTION = 29
E_TERMINATION_ACTION_DEFAULT = 0
E_TERMINATION_ACTION_RADIUS_REQUEST = 1

AVP_CALLED_STATION_ID = 30
AVP_CALLING_STATION_ID = 31
AVP_NAS_IDENTIFIER = 32
AVP_PROXY_STATE = 33
AVP_LOGIN_LAT_SERVICE = 34
AVP_LOGIN_LAT_NODE = 35
AVP_LOGIN_LAT_GROUP = 36
AVP_FRAMED_APPLETALK_LINK = 37
AVP_FRAMED_APPLETALK_NETWORK = 38
AVP_FRAMED_APPLETALK_ZONE = 39
AVP_ACCT_STATUS_TYPE = 40
E_ACCT_STATUS_TYPE_START = 1
E_ACCT_STATUS_TYPE_STOP = 2
E_ACCT_STATUS_TYPE_INTERIM_UPDATE = 3
E_ACCT_STATUS_TYPE_MODEM_START = 4
E_ACCT_STATUS_TYPE_MODEM_STOP = 5
E_ACCT_STATUS_TYPE_CANCEL = 6
E_ACCT_STATUS_TYPE_ACCOUNTING_ON = 7
E_ACCT_STATUS_TYPE_ACCOUNTING_OFF = 8
E_ACCT_STATUS_TYPE_TUNNEL_START = 9
E_ACCT_STATUS_TYPE_TUNNEL_STOP = 10
E_ACCT_STATUS_TYPE_TUNNEL_REJECT = 11
E_ACCT_STATUS_TYPE_TUNNEL_LINK_START = 12
E_ACCT_STATUS_TYPE_TUNNEL_LINK_STOP = 13
E_ACCT_STATUS_TYPE_TUNNEL_LINK_REJECTF = 14
E_ACCT_STATUS_TYPE_FAILED = 15

AVP_ACCT_DELAY_TIME = 41
AVP_ACCT_INPUT_OCTETS = 42
AVP_ACCT_OUTPUT_OCTETS = 43
AVP_ACCT_SESSION_ID = 44
AVP_ACCT_AUTHENTIC = 45
E_ACCT_AUTHENTIC_NONE = 0
E_ACCT_AUTHENTIC_RADIUS = 1
E_ACCT_AUTHENTIC_LOCAL = 2
E_ACCT_AUTHENTIC_REMOTE = 3
E_ACCT_AUTHENTIC_DIAMETER = 4

AVP_ACCT_SESSION_TIME = 46
AVP_ACCT_INPUT_PACKETS = 47
AVP_ACCT_OUTPUT_PACKETS = 48
AVP_ACCT_TERMINATE_CAUSE = 49
E_ACCT_TERMINATE_CAUSE_USER_REQUEST = 1
E_ACCT_TERMINATE_CAUSE_LOST_CARRIER = 2
E_ACCT_TERMINATE_CAUSE_LOST_SERVICE = 3
E_ACCT_TERMINATE_CAUSE_IDLE_TIMEOUT = 4
E_ACCT_TERMINATE_CAUSE_SESSION_TIMEOUT = 5
E_ACCT_TERMINATE_CAUSE_ADMIN_RESET = 6
E_ACCT_TERMINATE_CAUSE_ADMIN_REBOOT = 7
E_ACCT_TERMINATE_CAUSE_PORT_ERROR = 8
E_ACCT_TERMINATE_CAUSE_NAS_ERROR = 9
E_ACCT_TERMINATE_CAUSE_NAS_REQUEST = 10
E_ACCT_TERMINATE_CAUSE_NAS_REBOOT = 11
E_ACCT_TERMINATE_CAUSE_PORT_UNNEEDED = 12
E_ACCT_TERMINATE_CAUSE_PORT_PREEMPTED = 13
E_ACCT_TERMINATE_CAUSE_PORT_SUSPENDED = 14
E_ACCT_TERMINATE_CAUSE_SERVICE_UNAVAILABLE = 15
E_ACCT_TERMINATE_CAUSE_CALLBACK = 16
E_ACCT_TERMINATE_CAUSE_USER_ERROR = 17
E_ACCT_TERMINATE_CAUSE_HOST_REQUEST = 18
E_ACCT_TERMINATE_CAUSE_SUPPLICANT_RESTART = 19
E_ACCT_TERMINATE_CAUSE_REAUTHENTICATION_FAILURE = 20
E_ACCT_TERMINATE_CAUSE_PORT_REINITIALIZED = 21
E_ACCT_TERMINATE_CAUSE_PORT_ADMINISTRATIVELY_DISABLED = 22

AVP_ACCOUNTING_MULTI_SESSION_ID = 50
AVP_ACCT_LINK_COUNT = 51
AVP_ACCT_INPUT_GIGAWORDS = 52
AVP_ACCT_OUTPUT_GIGAWORDS = 53
AVP_EVENT_TIMESTAMP = 55
AVP_EGRESS_VLANID = 56
AVP_INGRESS_FILTERS = 57
E_INGRESS_FILTERS_ENABLED = 1
E_INGRESS_FILTERS_DISABLED = 2

AVP_EGRESS_VLAN_NAME = 58
AVP_USER_PRIORITY_TABLE = 59
AVP_CHAP_CHALLENGE = 60
AVP_NAS_PORT_TYPE = 61
E_NAS_PORT_TYPE_ASYNC = 0
E_NAS_PORT_TYPE_SYNC = 1
E_NAS_PORT_TYPE_ISDN_SYNC = 2
E_NAS_PORT_TYPE_ISDN_ASYNC_V120 = 3
E_NAS_PORT_TYPE_ISDN_ASYNC_V110 = 4
E_NAS_PORT_TYPE_VIRTUAL = 5
E_NAS_PORT_TYPE_PIAFS = 6
E_NAS_PORT_TYPE_HDLC_CLEAR_CHANNEL = 7
E_NAS_PORT_TYPE_X25 = 8
E_NAS_PORT_TYPE_X75 = 9
E_NAS_PORT_TYPE_G_3_FAX = 10
E_NAS_PORT_TYPE_SDSL_SYMMETRIC_DSL = 11
E_NAS_PORT_TYPE_ADSL_CAP_ASYMMETRIC_DSL_CARRIERLESS_AMPLITUDE_PHASE_MODULATION = 12
E_NAS_PORT_TYPE_ADSL_DMT_ASYMMETRIC_DSL_DISCRETE_MULTI_TONE = 13
E_NAS_PORT_TYPE_IDSL_ISDN_DIGITAL_SUBSCRIBER_LINE = 14
E_NAS_PORT_TYPE_ETHERNET = 15
E_NAS_PORT_TYPE_XDSL_DIGITAL_SUBSCRIBER_LINE_OF_UNKNOWN_TYPE = 16
E_NAS_PORT_TYPE_CABLE = 17
E_NAS_PORT_TYPE_WIRELESS_OTHER = 18
E_NAS_PORT_TYPE_WIRELESS_IEEE_802_11 = 19
E_NAS_PORT_TYPE_TOKEN_RING = 20
E_NAS_PORT_TYPE_FDDI = 21
E_NAS_PORT_TYPE_WIRELESS_CDMA2000 = 22
E_NAS_PORT_TYPE_WIRELESS_UMTS = 23
E_NAS_PORT_TYPE_WIRELESS_1X_EV = 24
E_NAS_PORT_TYPE_IAPP = 25
E_NAS_PORT_TYPE_FTTP_FIBER_TO_THE_PREMISES = 26
E_NAS_PORT_TYPE_WIRELESS_IEEE_802_16 = 27
E_NAS_PORT_TYPE_WIRELESS_IEEE_802_20 = 28
E_NAS_PORT_TYPE_WIRELESS_IEEE_802_22 = 29
E_NAS_PORT_TYPE_PPPOA_PPP_OVER_ATM = 30
E_NAS_PORT_TYPE_PPPOEOA_PPP_OVER_ETHERNET_OVER_ATM = 31
E_NAS_PORT_TYPE_PPPOEOE_PPP_OVER_ETHERNET_OVER_ETHERNET = 32
E_NAS_PORT_TYPE_PPPOEOVLAN_PPP_OVER_ETHERNET_OVER_VLAN = 33
E_NAS_PORT_TYPE_PPPOEOQINQ_PPP_OVER_ETHERNET_OVER_IEEE_802_1QINQ = 34
E_NAS_PORT_TYPE_XPON_PASSIVE_OPTICAL_NETWORK = 35
E_NAS_PORT_TYPE_WIRELESS_XGP = 36

AVP_PORT_LIMIT = 62
AVP_LOGIN_LAT_PORT = 63
AVP_TUNNEL_TYPE = 64
E_TUNNEL_TYPE_PPTP = 1
E_TUNNEL_TYPE_L2F = 2
E_TUNNEL_TYPE_L2TP = 3
E_TUNNEL_TYPE_ATMP = 4
E_TUNNEL_TYPE_VTP = 5
E_TUNNEL_TYPE_AH = 6
E_TUNNEL_TYPE_IP_IP_ENCAP = 7
E_TUNNEL_TYPE_MIN_IP_IP = 8
E_TUNNEL_TYPE_ESP = 9
E_TUNNEL_TYPE_GRE = 10
E_TUNNEL_TYPE_DVS = 11
E_TUNNEL_TYPE_IP_IN_IP_TUNNELING = 12
E_TUNNEL_TYPE_VLAN = 13

AVP_TUNNEL_MEDIUM_TYPE = 65
E_TUNNEL_MEDIUM_TYPE_IPV4 = 1
E_TUNNEL_MEDIUM_TYPE_IPV6 = 2
E_TUNNEL_MEDIUM_TYPE_NSAP = 3
E_TUNNEL_MEDIUM_TYPE_HDLC = 4
E_TUNNEL_MEDIUM_TYPE_BBN = 5
E_TUNNEL_MEDIUM_TYPE_IEEE_802 = 6
E_TUNNEL_MEDIUM_TYPE_E_163 = 7
E_TUNNEL_MEDIUM_TYPE_E_164 = 8
E_TUNNEL_MEDIUM_TYPE_F_69 = 9
E_TUNNEL_MEDIUM_TYPE_X_121 = 10
E_TUNNEL_MEDIUM_TYPE_IPX = 11
E_TUNNEL_MEDIUM_TYPE_APPLETALK_802 = 12
E_TUNNEL_MEDIUM_TYPE_DECNET4 = 13
E_TUNNEL_MEDIUM_TYPE_VINES = 14
E_TUNNEL_MEDIUM_TYPE_E_164_NSAP = 15

AVP_TUNNEL_CLIENT_ENDPOINT = 66
AVP_TUNNEL_SERVER_ENDPOINT = 67
AVP_TUNNEL_CONNECTION_ID = 68
AVP_TUNNEL_PASSWORD = 69
AVP_ARAP_PASSWORD = 70
AVP_ARAP_FEATURES = 71
AVP_ARAP_ZONE_ACCESS = 72
E_ARAP_ZONE_ACCESS_ONLY_ALLOW_ACCESS_TO_DEFAULT_ZONE = 1
E_ARAP_ZONE_ACCESS_USE_ZONE_FILTER_INCLUSIVELY = 2
E_ARAP_ZONE_ACCESS_USE_ZONE_FILTER_EXCLUSIVELY = 3

AVP_ARAP_SECURITY = 73
AVP_ARAP_SECURITY_DATA = 74
AVP_PASSWORD_RETRY = 75
AVP_PROMPT = 76
E_PROMPT_NO_ECHO = 0
E_PROMPT_ECHO = 1

AVP_CONNECT_INFO = 77
AVP_CONFIGURATION_TOKEN = 78
AVP_EAP_MESSAGE = 79
AVP_SIGNATURE = 80
AVP_TUNNEL_PRIVATE_GROUP_ID = 81
AVP_TUNNEL_ASSIGNMENT_ID = 82
AVP_TUNNEL_PREFERENCE = 83
AVP_ARAP_CHALLENGE_RESPONSE = 84
AVP_ACCT_INTERIM_INTERVAL = 85
AVP_ACCT_TUNNEL_PACKETS_LOST = 86
AVP_NAS_PORT_ID = 87
AVP_FRAMED_POOL = 88
AVP_CUI = 89
AVP_TUNNEL_CLIENT_AUTH_ID = 90
AVP_TUNNEL_SERVER_AUTH_ID = 91
AVP_RADIUS_NAS_FILTER_RULE = 92
AVP_ORIGINATING_LINE_INFO = 94
AVP_NAS_IPV6_ADDRESS = 95
AVP_FRAMED_INTERFACE_ID = 96
AVP_FRAMED_IPV6_PREFIX = 97
AVP_LOGIN_IPV6_HOST = 98
AVP_FRAMED_IPV6_ROUTE = 99
AVP_FRAMED_IPV6_POOL = 100
AVP_ERROR_CAUSE = 101
E_ERROR_CAUSE_DIAMETER_COMMON_MESSAGES = 0
E_ERROR_CAUSE_RESIDUAL_SESSION_CONTEXT_REMOVED = 201
E_ERROR_CAUSE_INVALID_EAP_PACKET_IGNORED = 202
E_ERROR_CAUSE_UNSUPPORTED_ATTRIBUTE = 401
E_ERROR_CAUSE_MISSING_ATTRIBUTE = 402
E_ERROR_CAUSE_NAS_IDENTIFICATION_MISMATCH = 403
E_ERROR_CAUSE_INVALID_REQUEST = 404
E_ERROR_CAUSE_UNSUPPORTED_SERVICE = 405
E_ERROR_CAUSE_UNSUPPORTED_EXTENSION = 406
E_ERROR_CAUSE_INVALID_ATTRIBUTE_VALUE = 407
E_ERROR_CAUSE_ADMINISTRATIVELY_PROHIBITED = 501
E_ERROR_CAUSE_REQUEST_NOT_ROUTABLE_PROXY = 502
E_ERROR_CAUSE_SESSION_CONTEXT_NOT_FOUND = 503
E_ERROR_CAUSE_SESSION_CONTEXT_NOT_REMOVABLE = 504
E_ERROR_CAUSE_OTHER_PROXY_PROCESSING_ERROR = 505
E_ERROR_CAUSE_RESOURCES_UNAVAILABLE = 506
E_ERROR_CAUSE_REQUEST_INITIATED = 507
E_ERROR_CAUSE_MULTIPLE_SESSION_SELECTION_UNSUPPORTED = 508

AVP_EAP_KEY_NAME = 102
AVP_DIGEST_RESPONSE = 103
AVP_DIGEST_REALM = 104
AVP_DIGEST_NONCE = 105
AVP_DIGEST_RESPONSE_AUTH = 106
AVP_DIGEST_NEXTNONCE = 107
AVP_DIGEST_METHOD = 108
AVP_DIGEST_URI = 109
AVP_DIGEST_QOP = 110
AVP_DIGEST_ALGORITHM = 111
AVP_DIGEST_ENTITY_BODY_HASH = 112
AVP_DIGEST_CNONCE = 113
AVP_DIGEST_NONCE_COUNT = 114
AVP_DIGEST_USERNAME = 115
AVP_DIGEST_OPAQUE = 116
AVP_DIGEST_AUTH_PARAM = 117
AVP_DIGEST_AKA_AUTS = 118
AVP_DIGEST_DOMAIN = 119
AVP_DIGEST_STALE = 120
AVP_DIGEST_HA1 = 121
AVP_SIP_AOR = 122
AVP_DELEGATED_IPV6_PREFIX = 123
AVP_MIP6_FEATURE_VECTOR = 124
AVP_MIP6_HOME_LINK_PREFIX = 125
AVP_OPERATOR_NAME = 126
AVP_LOCATION_INFORMATION = 127
AVP_LOCATION_DATA = 128
AVP_BASIC_LOCATION_POLICY_RULES = 129
AVP_EXTENDED_LOCATION_POLICY_RULES = 130
AVP_LOCATION_CAPABLE = 131
AVP_REQUESTED_LOCATION_INFO = 132
AVP_FRAMED_MANAGEMENT_PROTOCOL = 133
E_FRAMED_MANAGEMENT_PROTOCOL_SNMP = 1
E_FRAMED_MANAGEMENT_PROTOCOL_WEB_BASED = 2
E_FRAMED_MANAGEMENT_PROTOCOL_NETCONF = 3
E_FRAMED_MANAGEMENT_PROTOCOL_FTP = 4
E_FRAMED_MANAGEMENT_PROTOCOL_TFTP = 5
E_FRAMED_MANAGEMENT_PROTOCOL_SFTP = 6
E_FRAMED_MANAGEMENT_PROTOCOL_RCP = 7
E_FRAMED_MANAGEMENT_PROTOCOL_SCP = 8

AVP_MANAGEMENT_TRANSPORT_PROTECTION = 134
E_MANAGEMENT_TRANSPORT_PROTECTION_NO_PROTECTION = 1
E_MANAGEMENT_TRANSPORT_PROTECTION_INTEGRITY_PROTECTION = 2
E_MANAGEMENT_TRANSPORT_PROTECTION_INTEGRITY_CONFIDENTIALITY_PROTECTION = 3

AVP_MANAGEMENT_POLICY_ID = 135
AVP_MANAGEMENT_PRIVILEGE_LEVEL = 136
AVP_PKM_SS_CERT = 137
AVP_PKM_CA_CERT = 138
AVP_PKM_CONFIG_SETTINGS = 139
AVP_PKM_CRYPTOSUITE_LIST = 140
AVP_PPKM_SAID = 141
AVP_PKM_SA_DESCRIPTOR = 142
AVP_PKM_AUTH_KEY = 143
AVP_HOST_IP_ADDRESS = 257
AVP_AUTH_APPLICATION_ID = 258
AVP_ACCT_APPLICATION_ID = 259
AVP_VENDOR_SPECIFIC_APPLICATION_ID = 260
AVP_REDIRECT_HOST_USAGE = 261
E_REDIRECT_HOST_USAGE_DON_T_CARE = 0
E_REDIRECT_HOST_USAGE_ALL_SESSION = 1
E_REDIRECT_HOST_USAGE_ALL_REALM = 2
E_REDIRECT_HOST_USAGE_REALM_AND_APPLICATION = 3
E_REDIRECT_HOST_USAGE_ALL_APPLICATION = 4
E_REDIRECT_HOST_USAGE_ALL_HOST = 5
E_REDIRECT_HOST_USAGE_ALL_USER = 6

AVP_REDIRECT_MAX_CACHE_TIME = 262
AVP_SESSION_ID = 263
AVP_ORIGIN_HOST = 264
AVP_SUPPORTED_VENDOR_ID = 265
AVP_VENDOR_ID = 266
AVP_FIRMWARE_REVISION = 267
AVP_RESULT_CODE = 268
E_RESULT_CODE_DIAMETER_MULTI_ROUND_AUTH = 1001
E_RESULT_CODE_DIAMETER_SUCCESS = 2001
E_RESULT_CODE_DIAMETER_LIMITED_SUCCESS = 2002
E_RESULT_CODE_DIAMETER_FIRST_REGISTRATION = 2003
E_RESULT_CODE_DIAMETER_SUBSEQUENT_REGISTRATION = 2004
E_RESULT_CODE_DIAMETER_UNREGISTERED_SERVICE = 2005
E_RESULT_CODE_DIAMETER_SUCCESS_SERVER_NAME_NOT_STORED = 2006
E_RESULT_CODE_DIAMETER_SERVER_SELECTION = 2007
E_RESULT_CODE_DIAMETER_SUCCESS_AUTH_SENT_SERVER_NOT_STORED = 2008
E_RESULT_CODE_DIAMETER_SUCCESS_RELOCATE_HA = 2009
E_RESULT_CODE_DIAMETER_COMMAND_UNSUPPORTED = 3001
E_RESULT_CODE_DIAMETER_UNABLE_TO_DELIVER = 3002
E_RESULT_CODE_DIAMETER_REALM_NOT_SERVED = 3003
E_RESULT_CODE_DIAMETER_TOO_BUSY = 3004
E_RESULT_CODE_DIAMETER_LOOP_DETECTED = 3005
E_RESULT_CODE_DIAMETER_REDIRECT_INDICATION = 3006
E_RESULT_CODE_DIAMETER_APPLICATION_UNSUPPORTED = 3007
E_RESULT_CODE_DIAMETER_INVALID_HDR_BITS = 3008
E_RESULT_CODE_DIAMETER_INVALID_AVP_BITS = 3009
E_RESULT_CODE_DIAMETER_UNKNOWN_PEER = 3010
E_RESULT_CODE_DIAMETER_REALM_REDIRECT_INDICATION = 3011
E_RESULT_CODE_DIAMETER_AUTHENTICATION_REJECTED = 4001
E_RESULT_CODE_DIAMETER_OUT_OF_SPACE = 4002
E_RESULT_CODE_DIAMETER_ELECTION_LOST = 4003
E_RESULT_CODE_DIAMETER_ERROR_MIP_REPLY_FAILURE = 4005
E_RESULT_CODE_DIAMETER_ERROR_HA_NOT_AVAILABLE = 4006
E_RESULT_CODE_DIAMETER_ERROR_BAD_KEY = 4007
E_RESULT_CODE_DIAMETER_ERROR_MIP_FILTER_NOT_SUPPORTED = 4008
E_RESULT_CODE_DIAMETER_END_USER_SERVICE_DENIED = 4010
E_RESULT_CODE_DIAMETER_CREDIT_CONTROL_NOT_APPLICABLE = 4011
E_RESULT_CODE_DIAMETER_CREDIT_LIMIT_REACHED = 4012
E_RESULT_CODE_DIAMETER_USER_NAME_REQUIRED = 4013
E_RESULT_CODE_RESOURCE_FAILURE = 4014
E_RESULT_CODE_DIAMETER_AVP_UNSUPPORTED = 5001
E_RESULT_CODE_DIAMETER_UNKNOWN_SESSION_ID = 5002
E_RESULT_CODE_DIAMETER_AUTHORIZATION_REJECTED = 5003
E_RESULT_CODE_DIAMETER_INVALID_AVP_VALUE = 5004
E_RESULT_CODE_DIAMETER_MISSING_AVP = 5005
E_RESULT_CODE_DIAMETER_RESOURCES_EXCEEDED = 5006
E_RESULT_CODE_DIAMETER_CONTRADICTING_AVPS = 5007
E_RESULT_CODE_DIAMETER_AVP_NOT_ALLOWED = 5008
E_RESULT_CODE_DIAMETER_AVP_OCCURS_TOO_MANY_TIMES = 5009
E_RESULT_CODE_DIAMETER_NO_COMMON_APPLICATION = 5010
E_RESULT_CODE_DIAMETER_UNSUPPORTED_VERSION = 5011
E_RESULT_CODE_DIAMETER_UNABLE_TO_COMPLY = 5012
E_RESULT_CODE_DIAMETER_INVALID_BIT_IN_HEADER = 5013
E_RESULT_CODE_DIAMETER_INVALID_AVP_LENGTH = 5014
E_RESULT_CODE_DIAMETER_INVALID_MESSAGE_LENGTH = 5015
E_RESULT_CODE_DIAMETER_INVALID_AVP_BIT_COMBO = 5016
E_RESULT_CODE_DIAMETER_NO_COMMON_SECURITY = 5017
E_RESULT_CODE_DIAMETER_RADIUS_AVP_UNTRANSLATABLE = 5018
E_RESULT_CODE_DIAMETER_ERROR_NO_FOREIGN_HA_SERVICE = 5024
E_RESULT_CODE_DIAMETER_ERROR_END_TO_END_MIP_KEY_ENCRYPTION = 5025
E_RESULT_CODE_DIAMETER_USER_UNKNOWN = 5030
E_RESULT_CODE_DIAMETER_RATING_FAILED = 5031
E_RESULT_CODE_DIAMETER_ERROR_USER_UNKNOWN = 5032
E_RESULT_CODE_DIAMETER_ERROR_IDENTITIES_DONT_MATCH = 5033
E_RESULT_CODE_DIAMETER_ERROR_IDENTITY_NOT_REGISTERED = 5034
E_RESULT_CODE_DIAMETER_ERROR_ROAMING_NOT_ALLOWED = 5035
E_RESULT_CODE_DIAMETER_ERROR_IDENTITY_ALREADY_REGISTERED = 5036
E_RESULT_CODE_DIAMETER_ERROR_AUTH_SCHEME_NOT_SUPPORTED = 5037
E_RESULT_CODE_DIAMETER_ERROR_IN_ASSIGNMENT_TYPE = 5038
E_RESULT_CODE_DIAMETER_ERROR_TOO_MUCH_DATA = 5039
E_RESULT_CODE_DIAMETER_ERROR_NOT_SUPPORTED_USER_DATA = 5040
E_RESULT_CODE_DIAMETER_ERROR_MIP6_AUTH_MODE = 5041
E_RESULT_CODE_UNKNOWN_BINDING_TEMPLATE_NAME = 5042
E_RESULT_CODE_BINDING_FAILURE = 5043
E_RESULT_CODE_MAX_BINDINGS_SET_FAILURE = 5044
E_RESULT_CODE_MAXIMUM_BINDINGS_REACHED_FOR_ENDPOINT = 5045
E_RESULT_CODE_SESSION_EXISTS = 5046
E_RESULT_CODE_INSUFFICIENT_CLASSIFIERS = 5047
E_RESULT_CODE_DIAMETER_ERROR_EAP_CODE_UNKNOWN = 5048
E_RESULT_CODE_DIAMETER_END_USER_NOT_FOUND = 5241

AVP_PRODUCT_NAME = 269
AVP_SESSION_BINDING = 270
E_SESSION_BINDING_RE_AUTH = 1
E_SESSION_BINDING_STR = 2
E_SESSION_BINDING_UNASSIGNED = 3
E_SESSION_BINDING_ACCOUNTING = 4

AVP_SESSION_SERVER_FAILOVER = 271
E_SESSION_SERVER_FAILOVER_REFUSE_SERVICE = 0
E_SESSION_SERVER_FAILOVER_TRY_AGAIN = 1
E_SESSION_SERVER_FAILOVER_ALLOW_SERVICE = 2
E_SESSION_SERVER_FAILOVER_TRY_AGAIN_ALLOW_SERVICE = 3

AVP_MULTI_ROUND_TIME_OUT = 272
AVP_DISCONNECT_CAUSE = 273
E_DISCONNECT_CAUSE_REBOOTING = 0
E_DISCONNECT_CAUSE_BUSY = 1
E_DISCONNECT_CAUSE_DO_NOT_WANT_TO_TALK_TO_YOU = 2

AVP_AUTH_REQUEST_TYPE = 274
E_AUTH_REQUEST_TYPE_AUTHENTICATE_ONLY = 1
E_AUTH_REQUEST_TYPE_AUTHORIZE_ONLY = 2
E_AUTH_REQUEST_TYPE_AUTHORIZE_AUTHENTICATE = 3

AVP_ALTERNATE_PEER = 275
AVP_AUTH_GRACE_PERIOD = 276
AVP_AUTH_SESSION_STATE = 277
E_AUTH_SESSION_STATE_STATE_MAINTAINED = 0
E_AUTH_SESSION_STATE_NO_STATE_MAINTAINED = 1

AVP_ORIGIN_STATE_ID = 278
AVP_FAILED_AVP = 279
AVP_PROXY_HOST = 280
AVP_ERROR_MESSAGE = 281
AVP_ROUTE_RECORD = 282
AVP_DESTINATION_REALM = 283
AVP_PROXY_INFO = 284
AVP_RE_AUTH_REQUEST_TYPE = 285
E_RE_AUTH_REQUEST_TYPE_AUTHORIZE_ONLY = 0
E_RE_AUTH_REQUEST_TYPE_AUTHORIZE_AUTHENTICATE = 1

AVP_ACCOUNTING_SUB_SESSION_ID = 287
AVP_AUTHORIZATION_LIFETIME = 291
AVP_REDIRECT_HOST = 292
AVP_DESTINATION_HOST = 293
AVP_ERROR_REPORTING_HOST = 294
AVP_TERMINATION_CAUSE = 295
E_TERMINATION_CAUSE_DIAMETER_LOGOUT = 1
E_TERMINATION_CAUSE_DIAMETER_SERVICE_NOT_PROVIDED = 2
E_TERMINATION_CAUSE_DIAMETER_BAD_ANSWER = 3
E_TERMINATION_CAUSE_DIAMETER_ADMINISTRATIVE = 4
E_TERMINATION_CAUSE_DIAMETER_LINK_BROKEN = 5
E_TERMINATION_CAUSE_DIAMETER_AUTH_EXPIRED = 6
E_TERMINATION_CAUSE_DIAMETER_USER_MOVED = 7
E_TERMINATION_CAUSE_DIAMETER_SESSION_TIMEOUT = 8
E_TERMINATION_CAUSE_UNASSIGNED = 10
E_TERMINATION_CAUSE_USER_REQUEST = 11
E_TERMINATION_CAUSE_LOST_CARRIER = 12
E_TERMINATION_CAUSE_LOST_SERVICE = 13
E_TERMINATION_CAUSE_IDLE_TIMEOUT = 14
E_TERMINATION_CAUSE_SESSION_TIMEOUT = 15
E_TERMINATION_CAUSE_ADMIN_RESET = 16
E_TERMINATION_CAUSE_ADMIN_REBOOT = 17
E_TERMINATION_CAUSE_PORT_ERROR = 18
E_TERMINATION_CAUSE_NAS_ERROR = 19
E_TERMINATION_CAUSE_NAS_REQUEST = 20
E_TERMINATION_CAUSE_NAS_REBOOT = 21
E_TERMINATION_CAUSE_PORT_UNNEEDED = 22
E_TERMINATION_CAUSE_PORT_PREEMPTED = 23
E_TERMINATION_CAUSE_PORT_SUSPENDED = 24
E_TERMINATION_CAUSE_SERVICE_UNAVAILABLE = 25
E_TERMINATION_CAUSE_CALLBACK = 26
E_TERMINATION_CAUSE_USER_ERROR = 27
E_TERMINATION_CAUSE_HOST_REQUEST = 28
E_TERMINATION_CAUSE_SUPPLICANT_RESTART = 29
E_TERMINATION_CAUSE_REAUTHENTICATION_FAILURE = 31
E_TERMINATION_CAUSE_PORT_ADMINISTRATIVELY_DISABLED = 32

AVP_ORIGIN_REALM = 296
AVP_EXPERIMENTAL_RESULT = 297
AVP_EXPERIMENTAL_RESULT_CODE = 298
AVP_INBAND_SECURITY_ID = 299
E_INBAND_SECURITY_ID_NO_INBAND_SECURITY = 0
E_INBAND_SECURITY_ID_TLS = 1

AVP_E2E_SEQUENCE = 300
AVP_DRMP = 301
E_DRMP_PRIORITY_0 = 0
E_DRMP_PRIORITY_1 = 1
E_DRMP_PRIORITY_2 = 2
E_DRMP_PRIORITY_3 = 3
E_DRMP_PRIORITY_4 = 4
E_DRMP_PRIORITY_5 = 5
E_DRMP_PRIORITY_6 = 6
E_DRMP_PRIORITY_7 = 7
E_DRMP_PRIORITY_8 = 8
E_DRMP_PRIORITY_9 = 9
E_DRMP_PRIORITY_10 = 10
E_DRMP_PRIORITY_11 = 11
E_DRMP_PRIORITY_12 = 12
E_DRMP_PRIORITY_13 = 13
E_DRMP_PRIORITY_14 = 14
E_DRMP_PRIORITY_15 = 15

AVP_QOS_FILTER_RULE = 407
AVP_ACCOUNTING_RECORD_TYPE = 480
E_ACCOUNTING_RECORD_TYPE_EVENT_RECORD = 1
E_ACCOUNTING_RECORD_TYPE_START_RECORD = 2
E_ACCOUNTING_RECORD_TYPE_INTERIM_RECORD = 3
E_ACCOUNTING_RECORD_TYPE_STOP_RECORD = 4

AVP_ACCOUNTING_REALTIME_REQUIRED = 483
E_ACCOUNTING_REALTIME_REQUIRED_RESERVED = 0
E_ACCOUNTING_REALTIME_REQUIRED_DELIVER_AND_GRANT = 1
E_ACCOUNTING_REALTIME_REQUIRED_GRANT_AND_STORE = 2
E_ACCOUNTING_REALTIME_REQUIRED_GRANT_AND_LOSE = 3

AVP_ACCOUNTING_RECORD_NUMBER = 485
AVP_TMOD_1 = 495
AVP_TOKEN_RATE = 496
AVP_BUCKET_DEPTH = 497
AVP_PEAK_TRAFFIC_RATE = 498
AVP_MINIMUM_POLICED_UNIT = 499
AVP_MAXIMUM_PACKET_SIZE = 500
AVP_TMOD_2 = 501
AVP_BANDWIDTH = 502
AVP_PHB_CLASS = 503
AVP_QOS_RESOURCES = 508
AVP_FILTER_RULE = 509
AVP_FILTER_RULE_PRECEDENCE = 510
AVP_CLASSIFIER = 511
AVP_CLASSIFIER_ID = 512
AVP_PROTOCOL = 513
E_PROTOCOL_TCP_TRANSMISSION_CONTROL = 6
E_PROTOCOL_UDP_USER_DATAGRAM = 17
E_PROTOCOL_SCTP_STREAM_CONTROL_TRANSMISSION_PROTOCOL = 132

AVP_DIRECTION = 514
E_DIRECTION_IN = 0
E_DIRECTION_OUT = 1
E_DIRECTION_BOTH = 2

AVP_FROM_SPEC = 515
AVP_TO_SPEC = 516
AVP_NEGATED = 517
E_NEGATED_FALSE = 0
E_NEGATED_TRUE = 1

AVP_IP_ADDRESS = 518
AVP_IP_ADDRESS_RANGE = 519
AVP_IP_ADDRESS_START = 520
AVP_IP_ADDRESS_END = 521
AVP_IP_ADDRESS_MASK = 522
AVP_IP_BIT_MASK_WIDTH = 523
AVP_MAC_ADDRESS = 524
AVP_MAC_ADDRESS_MASK = 525
AVP_MAC_ADDRESS_MASK_PATTERN = 526
AVP_EUI64_ADDRESS = 527
AVP_EUI64_ADDRESS_MASK = 528
AVP_EUI64_ADDRESS_MASK_PATTERN = 529
AVP_PORT = 530
AVP_PORT_RANGE = 531
AVP_PORT_START = 532
AVP_PORT_END = 533
AVP_USE_ASSIGNED_ADDRESS = 534
E_USE_ASSIGNED_ADDRESS_FALSE = 0
E_USE_ASSIGNED_ADDRESS_TRUE = 1

AVP_DIFFSERV_CODE_POINT = 535
E_DIFFSERV_CODE_POINT_CS0 = 0
E_DIFFSERV_CODE_POINT_CS1 = 8
E_DIFFSERV_CODE_POINT_CS2 = 16
E_DIFFSERV_CODE_POINT_CS3 = 24
E_DIFFSERV_CODE_POINT_CS4 = 32
E_DIFFSERV_CODE_POINT_CS5 = 40
E_DIFFSERV_CODE_POINT_CS6 = 48
E_DIFFSERV_CODE_POINT_CS7 = 56
E_DIFFSERV_CODE_POINT_AF11 = 10
E_DIFFSERV_CODE_POINT_AF12 = 12
E_DIFFSERV_CODE_POINT_AF13 = 14
E_DIFFSERV_CODE_POINT_AF21 = 18
E_DIFFSERV_CODE_POINT_AF22 = 20
E_DIFFSERV_CODE_POINT_AF23 = 22
E_DIFFSERV_CODE_POINT_AF31 = 26
E_DIFFSERV_CODE_POINT_AF32 = 28
E_DIFFSERV_CODE_POINT_AF33 = 30
E_DIFFSERV_CODE_POINT_AF41 = 34
E_DIFFSERV_CODE_POINT_AF42 = 36
E_DIFFSERV_CODE_POINT_AF43 = 38
E_DIFFSERV_CODE_POINT_EF = 46
E_DIFFSERV_CODE_POINT_VOICE_ADMIT = 44

AVP_FRAGMENTATION_FLAG = 536
E_FRAGMENTATION_FLAG_DON_T_FRAGMENT_DF = 0
E_FRAGMENTATION_FLAG_MORE_FRAGMENTS_MF = 1

AVP_IP_OPTION = 537
AVP_IP_OPTION_TYPE = 538
E_IP_OPTION_TYPE_END_OF_OPTIONS_LIST = 0
E_IP_OPTION_TYPE_NO_OPERATION = 1
E_IP_OPTION_TYPE_SECURITY = 130
E_IP_OPTION_TYPE_LOOSE_SOURCE_ROUTE = 131
E_IP_OPTION_TYPE_TIME_STAMP = 68
E_IP_OPTION_TYPE_EXTENDED_SECURITY = 133
E_IP_OPTION_TYPE_COMMERCIAL_SECURITY = 134
E_IP_OPTION_TYPE_RECORD_ROUTE = 7
E_IP_OPTION_TYPE_STREAM_ID = 136
E_IP_OPTION_TYPE_STRICT_SOURCE_ROUTE = 137
E_IP_OPTION_TYPE_EXPERIMENTAL_MEASUREMENT = 10
E_IP_OPTION_TYPE_MTU_PROBE = 11
E_IP_OPTION_TYPE_MTU_REPLY = 12
E_IP_OPTION_TYPE_EXPERIMENTAL_FLOW_CONTROL = 205
E_IP_OPTION_TYPE_EXPERIMENTAL_ACCESS_CONTROL = 142
E_IP_OPTION_TYPE_ENCODE = 15
E_IP_OPTION_TYPE_IMI_TRAFFIC_DESCRIPTOR = 144
E_IP_OPTION_TYPE_EXTENDED_INTERNET_PROTOCOL = 145
E_IP_OPTION_TYPE_TRACEOUTE = 82
E_IP_OPTION_TYPE_ADDRESS_EXTENSION = 147
E_IP_OPTION_TYPE_ROUTER_ALERT = 148
E_IP_OPTION_TYPE_SELECTIVE_DIRECTED_BROADCAST = 149
E_IP_OPTION_TYPE_DYNAMIC_PACKET_STATE = 151
E_IP_OPTION_TYPE_UPSTREAM_MULTICAST_PKT = 152
E_IP_OPTION_TYPE_QUICK_START = 25
E_IP_OPTION_TYPE_RFC3692_STYLE_EXPERIMENT = 222

AVP_IP_OPTION_VALUE = 539
AVP_TCP_OPTION = 540
AVP_TCP_OPTION_TYPE = 541
E_TCP_OPTION_TYPE_END_OF_OPTION_LIST = 0
E_TCP_OPTION_TYPE_NO_OPERATION = 1
E_TCP_OPTION_TYPE_MAXIMUM_SEGMENT_SIZE = 2
E_TCP_OPTION_TYPE_WINDOW_SCALE = 3
E_TCP_OPTION_TYPE_SACK_PERMITTED = 4
E_TCP_OPTION_TYPE_SACK = 5
E_TCP_OPTION_TYPE_ECHO = 6
E_TCP_OPTION_TYPE_ECHO_REPLY = 7
E_TCP_OPTION_TYPE_TIMESTAMPS = 8
E_TCP_OPTION_TYPE_PARTIAL_ORDER_CONNECTION_PERMITTED = 9
E_TCP_OPTION_TYPE_PARTIAL_ORDER_SERVICE_PROFILE = 10
E_TCP_OPTION_TYPE_CC = 11
E_TCP_OPTION_TYPE_CC_NEW = 12
E_TCP_OPTION_TYPE_CC_ECHO = 13
E_TCP_OPTION_TYPE_TCP_ALTERNATE_CHECKSUM_REQUEST = 14
E_TCP_OPTION_TYPE_TCP_ALTERNATE_CHECKSUM_DATA = 15
E_TCP_OPTION_TYPE_SKEETER = 16
E_TCP_OPTION_TYPE_BUBBA = 17
E_TCP_OPTION_TYPE_TRAILER_CHECKSUM_OPTION = 18
E_TCP_OPTION_TYPE_MD5_SIGNATURE_OPTION = 19
E_TCP_OPTION_TYPE_SCPS_CAPABILITIES = 20
E_TCP_OPTION_TYPE_SELECTIVE_NEGATIVE_ACKNOWLEDGEMENTS = 21
E_TCP_OPTION_TYPE_RECORD_BOUNDARIES = 22
E_TCP_OPTION_TYPE_CORRUPTION_EXPERIENCED = 23
E_TCP_OPTION_TYPE_SNAP = 24
E_TCP_OPTION_TYPE_TCP_COMPRESSION_FILTER = 26
E_TCP_OPTION_TYPE_QUICK_START_RESPONSE = 27
E_TCP_OPTION_TYPE_USER_TIMEOUT_OPTION = 28
E_TCP_OPTION_TYPE_TCP_AUTHENTICATION_OPTION_TCP_AO = 29
E_TCP_OPTION_TYPE_MULTIPATH_TCP_MPTCP = 30
E_TCP_OPTION_TYPE_TCP_FAST_OPEN_COOKIE = 34
E_TCP_OPTION_TYPE_RFC3692_STYLE_EXPERIMENT_1 = 253
E_TCP_OPTION_TYPE_RFC3692_STYLE_EXPERIMENT_2 = 254

AVP_TCP_OPTION_VALUE = 542
AVP_TCP_FLAGS = 543
AVP_TCP_FLAG_TYPE = 544
AVP_ICMP_TYPE = 545
AVP_ICMP_TYPE_NUMBER = 546
E_ICMP_TYPE_NUMBER_ECHO_REPLY = 0
E_ICMP_TYPE_NUMBER_DESTINATION_UNREACHABLE = 3
E_ICMP_TYPE_NUMBER_SOURCE_QUENCH = 4
E_ICMP_TYPE_NUMBER_REDIRECT = 5
E_ICMP_TYPE_NUMBER_ALTERNATE_HOST_ADDRESS = 6
E_ICMP_TYPE_NUMBER_ECHO = 8
E_ICMP_TYPE_NUMBER_ROUTER_ADVERTISEMENT = 9
E_ICMP_TYPE_NUMBER_ROUTER_SOLICITATION = 10
E_ICMP_TYPE_NUMBER_TIME_EXCEEDED = 11
E_ICMP_TYPE_NUMBER_PARAMETER_PROBLEM = 12
E_ICMP_TYPE_NUMBER_TIMESTAMP = 13
E_ICMP_TYPE_NUMBER_TIMESTAMP_REPLY = 14
E_ICMP_TYPE_NUMBER_INFORMATION_REQUEST = 15
E_ICMP_TYPE_NUMBER_INFORMATION_REPLY = 16
E_ICMP_TYPE_NUMBER_ADDRESS_MASK_REQUEST = 17
E_ICMP_TYPE_NUMBER_ADDRESS_MASK_REPLY = 18
E_ICMP_TYPE_NUMBER_RESERVED_FOR_SECURITY = 19
E_ICMP_TYPE_NUMBER_TRACEROUTE = 30
E_ICMP_TYPE_NUMBER_DATAGRAM_CONVERSION_ERROR = 31
E_ICMP_TYPE_NUMBER_MOBILE_HOST_REDIRECT = 32
E_ICMP_TYPE_NUMBER_IPV6_WHERE_ARE_YOU = 33
E_ICMP_TYPE_NUMBER_IPV6_I_AM_HERE = 34
E_ICMP_TYPE_NUMBER_MOBILE_REGISTRATION_REQUEST = 35
E_ICMP_TYPE_NUMBER_MOBILE_REGISTRATION_REPLY = 36
E_ICMP_TYPE_NUMBER_DOMAIN_NAME_REQUEST = 37
E_ICMP_TYPE_NUMBER_DOMAIN_NAME_REPLY = 38
E_ICMP_TYPE_NUMBER_SKIP = 39
E_ICMP_TYPE_NUMBER_PHOTURIUS = 40
E_ICMP_TYPE_NUMBER_ICMP_MESSAGES_EXPERIMENTAL = 41
E_ICMP_TYPE_NUMBER_EXTENDED_ECHO_REQUEST = 42
E_ICMP_TYPE_NUMBER_EXTENDED_ECHO_REPLY = 43
E_ICMP_TYPE_NUMBER_RFC3692_STYLE_EXPERIMENT_1 = 253
E_ICMP_TYPE_NUMBER_RESERVED = 254

AVP_ICMP_CODE = 547
AVP_ETH_OPTION = 548
AVP_ETH_PROTO_TYPE = 549
AVP_ETH_ETHER_TYPE = 550
AVP_ETH_SAP = 551
AVP_VLAN_ID_RANGE = 552
AVP_S_VID_START = 553
AVP_S_VID_END = 554
AVP_C_VID_START = 555
AVP_C_VID_END = 556
AVP_USER_PRIORITY_RANGE = 557
AVP_LOW_USER_PRIORITY = 558
AVP_HIGH_USER_PRIORITY = 559
AVP_TIME_OF_DAY_CONDITION = 560
AVP_TIME_OF_DAY_START = 561
AVP_TIME_OF_DAY_END = 562
AVP_DAY_OF_WEEK_MASK = 563
E_DAY_OF_WEEK_MASK_SUNDAY = 0
E_DAY_OF_WEEK_MASK_MONDAY = 2
E_DAY_OF_WEEK_MASK_TUESDAY = 4
E_DAY_OF_WEEK_MASK_WEDNESDAY = 8
E_DAY_OF_WEEK_MASK_THURSDAY = 16
E_DAY_OF_WEEK_MASK_FRIDAY = 32
E_DAY_OF_WEEK_MASK_SATURDAY = 64

AVP_DAY_OF_MONTH_MASK = 564
AVP_MONTH_OF_YEAR_MASK = 565
E_MONTH_OF_YEAR_MASK_JANUARY = 0
E_MONTH_OF_YEAR_MASK_FEBRUARY = 2
E_MONTH_OF_YEAR_MASK_MARCH = 4
E_MONTH_OF_YEAR_MASK_APRIL = 8
E_MONTH_OF_YEAR_MASK_MAY = 16
E_MONTH_OF_YEAR_MASK_JUNE = 32
E_MONTH_OF_YEAR_MASK_JULY = 64
E_MONTH_OF_YEAR_MASK_AUGUST = 128
E_MONTH_OF_YEAR_MASK_SEPTEMBER = 256
E_MONTH_OF_YEAR_MASK_OCTOBER = 512
E_MONTH_OF_YEAR_MASK_NOVEMBER = 1024
E_MONTH_OF_YEAR_MASK_DECEMBER = 2048

AVP_ABSOLUTE_START_TIME = 566
AVP_ABSOLUTE_START_FRACTIONAL_SECONDS = 567
AVP_ABSOLUTE_END_TIME = 568
AVP_ABSOLUTE_END_FRACTIONAL_SECONDS = 569
AVP_TIMEZONE_FLAG = 570
E_TIMEZONE_FLAG_UTC = 0
E_TIMEZONE_FLAG_LOCAL = 1
E_TIMEZONE_FLAG_OFFSET = 2

AVP_TIMEZONE_OFFSET = 571
AVP_TREATMENT_ACTION = 572
E_TREATMENT_ACTION_DROP = 0
E_TREATMENT_ACTION_SHAPE = 1
E_TREATMENT_ACTION_MARK = 2
E_TREATMENT_ACTION_PERMIT = 3

AVP_QOS_PROFILE_ID = 573
AVP_QOS_PROFILE_TEMPLATE = 574
AVP_QOS_SEMANTICS = 575
E_QOS_SEMANTICS_QOS_DESIRED = 0
E_QOS_SEMANTICS_QOS_AVAILABLE = 1
E_QOS_SEMANTICS_QOS_DELIVERED = 2
E_QOS_SEMANTICS_MINIMUM_QOS = 3
E_QOS_SEMANTICS_QOS_AUTHORIZED = 4

AVP_QOS_PARAMETERS = 576
AVP_EXCESS_TREATMENT = 577
AVP_REDIRECT_REALM = 620
AVP_OC_SUPPORTED_FEATURES = 621
AVP_OC_FEATURE_VECTOR = 622
AVP_OC_OLR = 623
AVP_OC_SEQUENCE_NUMBER = 624
AVP_OC_VALIDITY_DURATION = 625
AVP_OC_REPORT_TYPE = 626
E_OC_REPORT_TYPE_HOST_REPORT = 0
E_OC_REPORT_TYPE_REALM_REPORT = 1
E_OC_REPORT_TYPE_PEER_REPORT = 2

AVP_OC_REDUCTION_PERCENTAGE = 627
AVP_OC_PEER_ALGO = 648
AVP_SOURCEID = 649
AVP_LOAD = 650
AVP_LOAD_TYPE = 651
E_LOAD_TYPE_HOST = 0
E_LOAD_TYPE_PEER = 1

AVP_LOAD_VALUE = 652
AVP_USER_EQUIPMENT_INFO_EXTENSION = 653
AVP_USER_EQUIPMENT_INFO_IMEISV = 654
AVP_USER_EQUIPMENT_INFO_MAC = 655
AVP_USER_EQUIPMENT_INFO_EUI64 = 656
AVP_USER_EQUIPMENT_INFO_MODIFIEDEUI64 = 657
AVP_USER_EQUIPMENT_INFO_IMEI = 658
AVP_TGPP_GBA_USERSECSETTINGS = 400
AVP_TGPP_TRANSACTION_IDENTIFIER = 401
AVP_TGPP_NAF_HOSTNAME = 402
AVP_TGPP_GAA_SERVICE_IDENTIFIER = 403
AVP_TGPP_KEY_EXPIRYTIME = 404
AVP_TGPP_ME_KEY_MATERIAL = 405
AVP_TGPP_UICC_KEY_MATERIAL = 406
AVP_TGPP_GBA_U_AWARENESS_INDICATOR = 407
E_GBA_U_AWARENESS_INDICATOR_NO = 0
E_GBA_U_AWARENESS_INDICATOR_YES = 1

AVP_TGPP_BOOTSTRAPINFOCREATIONTIME = 408
AVP_TGPP_GUSS_TIMESTAMP = 409
AVP_TGPP_GBA_TYPE = 410
E_GBA_TYPE_3G_GBA = 0
E_GBA_TYPE_2G_GBA = 1

AVP_TGPP_UE_ID = 411
AVP_TGPP_UE_ID_TYPE = 412
E_UE_ID_TYPE_PRIVATE_USER_IDENTITY = 0
E_UE_ID_TYPE_PUBLIC_USER_IDENTITY = 1

AVP_TGPP_UICC_APP_LABEL = 413
AVP_TGPP_UICC_ME = 414
E_UICC_ME_GBA_ME = 0
E_UICC_ME_GBA_U = 1

AVP_TGPP_REQUESTED_KEY_LIFETIME = 415
AVP_TGPP_PRIVATE_IDENTITY_REQUEST = 416
E_PRIVATE_IDENTITY_REQUEST_PRIVATE_IDENTITY_REQUESTED = 0
E_PRIVATE_IDENTITY_REQUEST_PRIVATE_IDENTITY_NOT_REQUESTED = 1

AVP_TGPP_GBA_PUSH_INFO = 417
AVP_TGPP_NAF_SA_IDENTIFIER = 418
AVP_TGPP_SECURITY_FEATURE_REQUEST = 419
AVP_TGPP_SECURITY_FEATURE_RESPONSE = 420
AVP_TGPP_CALLEE_INFORMATION = 565
AVP_TGPP_EVENT_TYPE = 823
AVP_TGPP_3GPP_SIP_METHOD = 824
AVP_TGPP_EVENT = 825
AVP_TGPP_CONTENT_TYPE = 826
AVP_TGPP_CONTENT_LENGTH = 827
AVP_TGPP_CONTENT_DISPOSITION = 828
AVP_TGPP_ROLE_OF_NODE = 829
E_ROLE_OF_NODE_ORIGINATING_ROLE = 0
E_ROLE_OF_NODE_TERMINATING_ROLE = 1
E_ROLE_OF_NODE_PROXY_ROLE = 2
E_ROLE_OF_NODE_B2BUA_ROLE = 3

AVP_TGPP_USER_SESSION_ID = 830
AVP_TGPP_CALLING_PARTY_ADDRESS = 831
AVP_TGPP_CALLED_PARTY_ADDRESS = 832
AVP_TGPP_TIME_STAMPS = 833
AVP_TGPP_SIP_REQUEST_TIMESTAMP = 834
AVP_TGPP_SIP_RESPONSE_TIMESTAMP = 835
AVP_TGPP_APPLICATION_SERVER = 836
AVP_TGPP_APPLICATION_PROVIDED_CALLED_PARTY_ADDRESS = 837
AVP_TGPP_INTER_OPERATOR_IDENTIFIER = 838
AVP_TGPP_ORIGINATING_IOI = 839
AVP_TGPP_TERMINATING_IOI = 840
AVP_TGPP_IMS_CHARGING_IDENTIFIER = 841
AVP_TGPP_SDP_SESSION_DESCRIPTION = 842
AVP_TGPP_SDP_MEDIA_COMPONENT = 843
AVP_TGPP_SDP_MEDIA_NAME = 844
AVP_TGPP_SDP_MEDIA_DESCRIPTION = 845
AVP_TGPP_CG_ADDRESS = 846
AVP_TGPP_GGSN_ADDRESS = 847
AVP_TGPP_SERVED_PARTY_IP_ADDRESS = 848
AVP_TGPP_AUTHORISED_QOS = 849
AVP_TGPP_APPLICATION_SERVER_INFORMATION = 850
AVP_TGPP_TRUNK_GROUP_ID = 851
AVP_TGPP_INCOMING_TRUNK_GROUP_ID = 852
AVP_TGPP_OUTGOING_TRUNK_GROUP_ID = 853
AVP_TGPP_BEARER_SERVICE = 854
AVP_TGPP_SERVICE_ID = 855
AVP_TGPP_ASSOCIATED_URI = 856
AVP_TGPP_CHARGED_PARTY = 857
AVP_TGPP_POC_CONTROLLING_ADDRESS = 858
AVP_TGPP_POC_GROUP_NAME = 859
AVP_TGPP_CAUSE = 860
AVP_TGPP_CAUSE_CODE = 861
E_CAUSE_CODE_END_OF_REGISTER_DIALOG = -3
E_CAUSE_CODE_END_OF_SUBSCRIBE_DIALOG = -2
E_CAUSE_CODE_SUCCESSFUL_TRANSACTION = -1
E_CAUSE_CODE_NORMAL_END_OF_SESSION = 0
E_CAUSE_CODE_UNSPECIFIED_ERROR = 1
E_CAUSE_CODE_UNSUCCESSFUL_SESSION_SETUP = 2
E_CAUSE_CODE_INTERNAL_ERROR = 3
E_CAUSE_CODE_MULTIPLE_CHOICES = 300
E_CAUSE_CODE_MOVED_PERMANENTLY = 301
E_CAUSE_CODE_MOVED_TEMPORARILY = 302
E_CAUSE_CODE_USE_PROXY = 305
E_CAUSE_CODE_ALTERNATIVE_SERVICE = 380
E_CAUSE_CODE_BAD_REQUEST = 400
E_CAUSE_CODE_UNAUTHORIZED = 401
E_CAUSE_CODE_PAYMENT_REQUIRED = 402
E_CAUSE_CODE_FORBIDDEN = 403
E_CAUSE_CODE_NOT_FOUND = 404
E_CAUSE_CODE_METHOD_NOT_ALLOWED = 405
E_CAUSE_CODE_NOT_ACCEPTABLE = 606
E_CAUSE_CODE_PROXY_AUTHENTICATION_REQUIRED = 407
E_CAUSE_CODE_REQUEST_TIMEOUT = 408
E_CAUSE_CODE_GONE = 410
E_CAUSE_CODE_CONDITIONAL_REQUEST_FAILED = 412
E_CAUSE_CODE_REQUEST_ENTITY_TOO_LARGE = 413
E_CAUSE_CODE_REQUEST_URI_TOO_LONG = 414
E_CAUSE_CODE_UNSUPPORTED_MEDIA_TYPE = 415
E_CAUSE_CODE_UNSUPPORTED_URI_SCHEME = 416
E_CAUSE_CODE_UNKNOWN_RESOURCE_PRIORITY = 417
E_CAUSE_CODE_BAD_EXTENSION = 420
E_CAUSE_CODE_EXTENSION_REQUIRED = 421
E_CAUSE_CODE_SESSION_INTERVAL_TOO_SMALL = 422
E_CAUSE_CODE_INTERVAL_TOO_BRIEF = 423
E_CAUSE_CODE_USE_IDENTITY_HEADER = 428
E_CAUSE_CODE_PROVIDE_REFERRER_IDENTITY = 429
E_CAUSE_CODE_BAD_IDENTITY_INFO = 436
E_CAUSE_CODE_UNSUPPORTED_CERTIFICATE = 437
E_CAUSE_CODE_INVALID_IDENTITY_HEADER = 438
E_CAUSE_CODE_TEMPORARILY_UNAVAILABLE = 480
E_CAUSE_CODE_CALL_TRANSACTION_DOES_NOT_EXIST = 481
E_CAUSE_CODE_LOOP_DETECTED = 482
E_CAUSE_CODE_TOO_MANY_HOPS = 483
E_CAUSE_CODE_ADDRESS_INCOMPLETE = 484
E_CAUSE_CODE_AMBIGUOUS = 485
E_CAUSE_CODE_BUSY_HERE = 486
E_CAUSE_CODE_REQUEST_TERMINATED = 487
E_CAUSE_CODE_NOT_ACCEPTABLE_HERE = 488
E_CAUSE_CODE_BAD_EVENT = 489
E_CAUSE_CODE_REQUEST_PENDING = 491
E_CAUSE_CODE_UNDECIPHERABLE = 493
E_CAUSE_CODE_SECURITY_AGREEMENT_REQUIRED = 494
E_CAUSE_CODE_SERVER_INTERNAL_ERROR = 500
E_CAUSE_CODE_NOT_IMPLEMENTED = 501
E_CAUSE_CODE_BAD_GATEWAY = 502
E_CAUSE_CODE_SERVICE_UNAVAILABLE = 503
E_CAUSE_CODE_SERVER_TIME_OUT = 504
E_CAUSE_CODE_VERSION_NOT_SUPPORTED = 505
E_CAUSE_CODE_MESSAGE_TOO_LARGE = 513
E_CAUSE_CODE_PRECONDITION_FAILURE = 580
E_CAUSE_CODE_BUSY_EVERYWHERE = 600
E_CAUSE_CODE_DECLINE = 603
E_CAUSE_CODE_DOES_NOT_EXIST_ANYWHERE = 604

AVP_TGPP_NODE_FUNCTIONALITY = 862
E_NODE_FUNCTIONALITY_S_CSCF = 0
E_NODE_FUNCTIONALITY_P_CSCF = 1
E_NODE_FUNCTIONALITY_I_CSCF = 2
E_NODE_FUNCTIONALITY_MRFC = 3
E_NODE_FUNCTIONALITY_MGCF = 4
E_NODE_FUNCTIONALITY_BGCF = 5
E_NODE_FUNCTIONALITY_AS = 6
E_NODE_FUNCTIONALITY_IBCF = 7
E_NODE_FUNCTIONALITY_S_GW = 8
E_NODE_FUNCTIONALITY_P_GW = 9
E_NODE_FUNCTIONALITY_HSGW = 10
E_NODE_FUNCTIONALITY_E_CSCF = 11
E_NODE_FUNCTIONALITY_MME = 12
E_NODE_FUNCTIONALITY_TRF = 13
E_NODE_FUNCTIONALITY_TF = 14
E_NODE_FUNCTIONALITY_ATCF = 15
E_NODE_FUNCTIONALITY_PROXY_FUNCTION = 16
E_NODE_FUNCTIONALITY_EPDG = 17

AVP_TGPP_SERVICE_SPECIFIC_DATA = 863
AVP_TGPP_ORIGINATOR = 864
E_ORIGINATOR_CALLING_PARTY = 0
E_ORIGINATOR_CALLED_PARTY = 1

AVP_TGPP_PS_FURNISH_CHARGING_INFORMATION = 865
AVP_TGPP_PS_FREE_FORMAT_DATA = 866
AVP_TGPP_PS_APPEND_FREE_FORMAT_DATA = 867
E_PS_APPEND_FREE_FORMAT_DATA_APPEND = 0
E_PS_APPEND_FREE_FORMAT_DATA_OVERWRITE = 1

AVP_TGPP_TIME_QUOTA_THRESHOLD = 868
AVP_TGPP_VOLUME_QUOTA_THRESHOLD = 869
AVP_TGPP_TRIGGER_TYPE = 870
E_TRIGGER_TYPE_CHANGE_IN_SGSN_IP_ADDRESS = 1
E_TRIGGER_TYPE_CHANGE_IN_QOS = 2
E_TRIGGER_TYPE_CHANGE_IN_LOCATION = 3
E_TRIGGER_TYPE_CHANGE_IN_RAT = 4
E_TRIGGER_TYPE_CHANGE_IN_UE_TIMEZONE = 5
E_TRIGGER_TYPE_UNDEFINED = 69
E_TRIGGER_TYPE_CHANGEINQOS_TRAFFIC_CLASS = 10
E_TRIGGER_TYPE_CHANGEINQOS_RELIABILITY_CLASS = 11
E_TRIGGER_TYPE_CHANGEINQOS_DELAY_CLASS = 12
E_TRIGGER_TYPE_CHANGEINQOS_PEAK_THROUGHPUT = 13
E_TRIGGER_TYPE_CHANGEINQOS_PRECEDENCE_CLASS = 14
E_TRIGGER_TYPE_CHANGEINQOS_MEAN_THROUGHPUT = 15
E_TRIGGER_TYPE_CHANGEINQOS_MAXIMUM_BIT_RATE_FOR_UPLINK = 16
E_TRIGGER_TYPE_CHANGEINQOS_MAXIMUM_BIT_RATE_FOR_DOWNLINK = 17
E_TRIGGER_TYPE_CHANGEINQOS_RESIDUAL_BER = 18
E_TRIGGER_TYPE_CHANGEINQOS_SDU_ERROR_RATIO = 19
E_TRIGGER_TYPE_CHANGEINQOS_TRANSFER_DELAY = 20
E_TRIGGER_TYPE_CHANGEINQOS_TRAFFIC_HANDLING_PRIORITY = 21
E_TRIGGER_TYPE_CHANGEINQOS_GUARANTEED_BIT_RATE_FOR_UPLINK = 22
E_TRIGGER_TYPE_CHANGEINQOS_GUARANTEED_BIT_RATE_FOR_DOWNLINK = 23
E_TRIGGER_TYPE_CHANGEINQOS_APN_AGGREGATE_MAXIMUM_BIT_RATE = 24
E_TRIGGER_TYPE_CHANGEINLOCATION_MCC = 30
E_TRIGGER_TYPE_CHANGEINLOCATION_MNC = 31
E_TRIGGER_TYPE_CHANGEINLOCATION_RAC = 32
E_TRIGGER_TYPE_CHANGEINLOCATION_LAC = 33
E_TRIGGER_TYPE_CHANGEINLOCATION_CELLID = 34
E_TRIGGER_TYPE_CHANGEINLOCATION_TAC = 35
E_TRIGGER_TYPE_CHANGEINLOCATION_ECGI = 36
E_TRIGGER_TYPE_CHANGE_IN_MEDIA_COMPOSITION = 40
E_TRIGGER_TYPE_CHANGEINPARTICIPANTS_NUMBER = 50
E_TRIGGER_TYPE_CHANGE_IN_THRSHLD_OF_PARTICIPANTS_NMB = 51
E_TRIGGER_TYPE_CHANGE_IN_USER_PARTICIPATING_TYPE = 52
E_TRIGGER_TYPE_CHANGE_IN_SERVICE_CONDITION = 60
E_TRIGGER_TYPE_CHANGE_IN_SERVING_NODE = 61
E_TRIGGER_TYPE_CHANGE_IN_ACCESS_FOR_A_SERVICE_DATA_FLOW = 62
E_TRIGGER_TYPE_CHANGE_IN_USER_CSG_INFORMATION = 70
E_TRIGGER_TYPE_CHANGE_IN_HYBRID_SUBSCRIBED_USER_CSG_INFORMATION = 71
E_TRIGGER_TYPE_CHANGE_IN_HYBRID_UNSUBSCRIBED_USER_CSG_INFORMATION = 72
E_TRIGGER_TYPE_CHANGE_OF_UE_PRESENCE_IN_PRESENCE_REPORTING_AREA = 73
E_TRIGGER_TYPE_CHANGE_IN_SERVING_PLMN_RATE_CONTROL = 74
E_TRIGGER_TYPE_CHANGE_IN_APN_RATE_CONTROL = 75

AVP_TGPP_QUOTA_HOLDING_TIME = 871
AVP_TGPP_3GPP_REPORTING_REASON = 872
E_3GPP_REPORTING_REASON_THRESHOLD = 0
E_3GPP_REPORTING_REASON_QHT = 1
E_3GPP_REPORTING_REASON_FINAL = 2
E_3GPP_REPORTING_REASON_QUOTA_EXHAUSTED = 3
E_3GPP_REPORTING_REASON_VALIDITY_TIME = 4
E_3GPP_REPORTING_REASON_OTHER_QUOTA_TYPE = 5
E_3GPP_REPORTING_REASON_RATING_CONDITION_CHANGE = 6
E_3GPP_REPORTING_REASON_FORCED_REAUTHORISATION = 7
E_3GPP_REPORTING_REASON_POOL_EXHAUSTED = 8

AVP_TGPP_SERVICE_INFORMATION = 873
AVP_TGPP_PS_INFORMATION = 874
AVP_TGPP_WLAN_INFORMATION = 875
AVP_TGPP_IMS_INFORMATION = 876
AVP_TGPP_MMS_INFORMATION = 877
AVP_TGPP_LCS_INFORMATION = 878
AVP_TGPP_POC_INFORMATION = 879
AVP_TGPP_MBMS_INFORMATION = 880
AVP_TGPP_QUOTA_CONSUMPTION_TIME = 881
AVP_TGPP_MEDIA_INITIATOR_FLAG = 882
E_MEDIA_INITIATOR_FLAG_CALLED_PARTY = 0
E_MEDIA_INITIATOR_FLAG_CALLING_PARTY = 1
E_MEDIA_INITIATOR_FLAG_UNKNOWN = 2

AVP_TGPP_POC_SERVER_ROLE = 883
E_POC_SERVER_ROLE_PARTICIPATING_POC_SERVER = 0
E_POC_SERVER_ROLE_CONTROLLING_POC_SERVER = 1

AVP_TGPP_POC_SESSION_TYPE = 884
E_POC_SESSION_TYPE_1_TO_1_POC_SESSION = 0
E_POC_SESSION_TYPE_CHAT_POC_GROUP_SESSION = 1
E_POC_SESSION_TYPE_PRE_ARRANGED_POC_GROUP_SESSION = 2
E_POC_SESSION_TYPE_AD_HOC_POC_GROUP_SESSION = 3

AVP_TGPP_NUMBER_OF_PARTICIPANTS = 885
AVP_TGPP_ORIGINATOR_ADDRESS = 886
AVP_TGPP_PARTICIPANTS_INVOLVED = 887
AVP_TGPP_EXPIRES = 888
AVP_TGPP_MESSAGE_BODY = 889
AVP_TGPP_WAG_ADDRESS = 890
AVP_TGPP_WAG_PLMN_ID = 891
AVP_TGPP_WLAN_RADIO_CONTAINER = 892
AVP_TGPP_WLAN_TECHNOLOGY = 893
AVP_TGPP_WLAN_UE_LOCAL_IPADDRESS = 894
AVP_TGPP_PDG_ADDRESS = 895
AVP_TGPP_PDG_CHARGING_ID = 896
AVP_TGPP_ADDRESS_DATA = 897
AVP_TGPP_ADDRESS_DOMAIN = 898
AVP_TGPP_ADDRESS_TYPE = 899
E_ADDRESS_TYPE_E_MAIL_ADDRESS = 0
E_ADDRESS_TYPE_MSISDN = 1
E_ADDRESS_TYPE_IPV4_ADDRESS = 2
E_ADDRESS_TYPE_IPV6_ADDRESS = 3
E_ADDRESS_TYPE_NUMERIC_SHORTCODE = 4
E_ADDRESS_TYPE_ALPHANUMERIC_SHORTCODE = 5
E_ADDRESS_TYPE_OTHER = 6

AVP_TGPP_BEARER_USAGE = 1000
E_BEARER_USAGE_GENERAL = 0
E_BEARER_USAGE_IMS_SIGNALLING = 1
E_BEARER_USAGE_DEDICATED = 2

AVP_TGPP_CHARGING_RULE_INSTALL = 1001
AVP_TGPP_CHARGING_RULE_REMOVE = 1002
AVP_TGPP_CHARGING_RULE_DEFINITION = 1003
AVP_TGPP_CHARGING_RULE_BASE_NAME = 1004
AVP_TGPP_CHARGING_RULE_NAME = 1005
AVP_TGPP_EVENT_TRIGGER = 1006
E_EVENT_TRIGGER_SGSN_CHANGE = 0
E_EVENT_TRIGGER_QOS_CHANGE = 1
E_EVENT_TRIGGER_RAT_CHANGE = 2
E_EVENT_TRIGGER_TFT_CHANGE = 3
E_EVENT_TRIGGER_PLMN_CHANGE = 4
E_EVENT_TRIGGER_LOSS_OF_BEARER = 1001
E_EVENT_TRIGGER_RECOVERY_OF_BEARER = 1002
E_EVENT_TRIGGER_IP_CAN_CHANGE = 7
E_EVENT_TRIGGER_GW_PCEF_MALFUNCTION = 8
E_EVENT_TRIGGER_RESOURCES_LIMITATION = 9
E_EVENT_TRIGGER_MAX_NR_BEARERS_REACHED = 10
E_EVENT_TRIGGER_QOS_CHANGE_EXCEEDING_AUTHORIZATION = 11
E_EVENT_TRIGGER_RAI_CHANGE = 12
E_EVENT_TRIGGER_USER_LOCATION_CHANGE = 13
E_EVENT_TRIGGER_NO_EVENT_TRIGGERS = 14
E_EVENT_TRIGGER_OUT_OF_CREDIT = 15
E_EVENT_TRIGGER_REALLOCATION_OF_CREDIT = 16
E_EVENT_TRIGGER_REVALIDATION_TIMEOUT = 17
E_EVENT_TRIGGER_UE_IP_ADDRESS_ALLOCATE = 18
E_EVENT_TRIGGER_UE_IP_ADDRESS_RELEASE = 19
E_EVENT_TRIGGER_DEFAULT_EPS_BEARER_QOS_CHANGE = 20
E_EVENT_TRIGGER_AN_GW_CHANGE = 21
E_EVENT_TRIGGER_SUCCESSFUL_RESOURCE_ALLOCATION = 22
E_EVENT_TRIGGER_RESOURCE_MODIFICATION_REQUEST = 23
E_EVENT_TRIGGER_PGW_TRACE_CONTROL = 24
E_EVENT_TRIGGER_UE_TIME_ZONE_CHANGE = 25
E_EVENT_TRIGGER_TAI_CHANGE = 26
E_EVENT_TRIGGER_ECGI_CHANGE = 27
E_EVENT_TRIGGER_CHARGING_CORRELATION_EXCHANGE = 28
E_EVENT_TRIGGER_APN_AMBR_MODIFICATION_FAILURE = 29
E_EVENT_TRIGGER_USER_CSG_INFORMATION_CHANGE = 30
E_EVENT_TRIGGER_USAGE_REPORT = 33
E_EVENT_TRIGGER_DEFAULT_EPS_BEARER_QOS_MODIFICATION_FAILURE = 34
E_EVENT_TRIGGER_USER_CSG_HYBRID_SUBSCRIBED_INFORMATION_CHANGE = 35
E_EVENT_TRIGGER_USER_CSG__HYBRID_UNSUBSCRIBED_INFORMATION_CHANGE = 36
E_EVENT_TRIGGER_ROUTING_RULE_CHANGE = 37
E_EVENT_TRIGGER_MAX_MBR_APN_AMBR_CHANGE = 38
E_EVENT_TRIGGER_APPLICATION_START = 39
E_EVENT_TRIGGER_APPLICATION_STOP = 40
E_EVENT_TRIGGER_ADC_REVALIDATION_TIMEOUT = 41
E_EVENT_TRIGGER_CS_TO_PS_HANDOVER = 42
E_EVENT_TRIGGER_UE_LOCAL_IP_ADDRESS_CHANGE = 43
E_EVENT_TRIGGER_H_E_NB_LOCAL_IP_ADDRESS_CHANGE = 44
E_EVENT_TRIGGER_ACCESS_NETWORK_INFO_REPORT = 45
E_EVENT_TRIGGER_CREDIT_MANAGEMENT_SESSION_FAILURE = 46
E_EVENT_TRIGGER_DEFAULT_QOS_CHANGE = 47
E_EVENT_TRIGGER_CHANGE_OF_UE_PRESENCE_IN_PRESENCE_REPORTING_AREA_REPORT = 48
E_EVENT_TRIGGER_TIME_CHANGE = 100
E_EVENT_TRIGGER_TFT_DELETED = 1000
E_EVENT_TRIGGER_POLICY_ENFORCEMENT_FAILED = 1003

AVP_TGPP_METERING_METHOD = 1007
E_METERING_METHOD_DURATION = 0
E_METERING_METHOD_VOLUME = 1
E_METERING_METHOD_DURATION_VOLUME = 2

AVP_TGPP_OFFLINE = 1008
E_OFFLINE_DISABLE_OFFLINE = 0
E_OFFLINE_ENABLE_OFFLINE = 1

AVP_TGPP_ONLINE = 1009
E_ONLINE_DISABLE_ONLINE = 0
E_ONLINE_ENABLE_ONLINE = 1

AVP_TGPP_PRECEDENCE = 1010
AVP_TGPP_REPORTING_LEVEL = 1011
E_REPORTING_LEVEL_SERVICE_IDENTIFIER_LEVEL = 0
E_REPORTING_LEVEL_RATING_GROUP_LEVEL = 1
E_REPORTING_LEVEL_SPONSORED_CONNECTIVITY_LEVEL = 2

AVP_TGPP_TFT_FILTER = 1012
AVP_TGPP_TFT_PACKET_FILTER_INFORMATION = 1013
AVP_TGPP_TOS_TRAFFIC_CLASS = 1014
AVP_TGPP_PDP_SESSION_OPERATION = 1015
E_PDP_SESSION_OPERATION_PDP_SESSION_TERMINATION = 0

AVP_TGPP_QOS_INFORMATION = 1016
AVP_TGPP_CHARGING_RULE_REPORT = 1018
AVP_TGPP_PCC_RULE_STATUS = 1019
E_PCC_RULE_STATUS_ACTIVE = 0
E_PCC_RULE_STATUS_INACTIVE = 1
E_PCC_RULE_STATUS_TEMPORARY_INACTIVE = 2

AVP_TGPP_BEARER_IDENTIFIER = 1020
AVP_TGPP_BEARER_OPERATION = 1021
E_BEARER_OPERATION_TERMINATION = 0
E_BEARER_OPERATION_ESTABLISHMENT = 1
E_BEARER_OPERATION_MODIFICATION = 2

AVP_TGPP_ACCESS_NETWORK_CHARGING_IDENTIFIER_GX = 1022
AVP_TGPP_BEARER_CONTROL_MODE = 1023
E_BEARER_CONTROL_MODE_UE_ONLY = 0
E_BEARER_CONTROL_MODE_RESERVED = 1
E_BEARER_CONTROL_MODE_UE_NW = 2

AVP_TGPP_NETWORK_REQUEST_SUPPORT = 1024
E_NETWORK_REQUEST_SUPPORT_NETWORK_REQUEST_NOT_SUPPORTED = 0
E_NETWORK_REQUEST_SUPPORT_NETWORK_REQUEST_SUPPORTED = 1

AVP_TGPP_GUARANTEED_BITRATE_DL = 1025
AVP_TGPP_GUARANTEED_BITRATE_UL = 1026
AVP_TGPP_IP_CAN_TYPE = 1027
E_IP_CAN_TYPE_3GPP_GPRS = 0
E_IP_CAN_TYPE_DOCSIS = 1
E_IP_CAN_TYPE_XDSL = 2
E_IP_CAN_TYPE_WIMAX = 3
E_IP_CAN_TYPE_3GPP2 = 4
E_IP_CAN_TYPE_3GPP_EPS = 5
E_IP_CAN_TYPE_NON_3GPP_EPS = 6
E_IP_CAN_TYPE_FBA = 7
E_IP_CAN_TYPE_3GPP_5GS = 8
E_IP_CAN_TYPE_NON_3GPP_5GS = 9

AVP_TGPP_QOS_CLASS_IDENTIFIER = 1028
E_QOS_CLASS_IDENTIFIER_QCI_1 = 1
E_QOS_CLASS_IDENTIFIER_QCI_2 = 2
E_QOS_CLASS_IDENTIFIER_QCI_3 = 3
E_QOS_CLASS_IDENTIFIER_QCI_4 = 4
E_QOS_CLASS_IDENTIFIER_QCI_5 = 5
E_QOS_CLASS_IDENTIFIER_QCI_6 = 6
E_QOS_CLASS_IDENTIFIER_QCI_7 = 7
E_QOS_CLASS_IDENTIFIER_QCI_8 = 8
E_QOS_CLASS_IDENTIFIER_QCI_9 = 9
E_QOS_CLASS_IDENTIFIER_QCI_65 = 65
E_QOS_CLASS_IDENTIFIER_QCI_66 = 66
E_QOS_CLASS_IDENTIFIER_QCI_69 = 69
E_QOS_CLASS_IDENTIFIER_QCI_70 = 70

AVP_TGPP_QOS_NEGOTIATION = 1029
E_QOS_NEGOTIATION_NO_QOS_NEGOTIATION = 0
E_QOS_NEGOTIATION_QOS_NEGOTIATION_SUPPORTED = 1

AVP_TGPP_QOS_UPGRADE = 1030
E_QOS_UPGRADE_QOS_UPGRADE_NOT_SUPPORTED = 0
E_QOS_UPGRADE_QOS_UPGRADE_SUPPORTED = 1

AVP_TGPP_RULE_FAILURE_CODE = 1031
E_RULE_FAILURE_CODE_UNKNOWN_RULE_NAME = 1
E_RULE_FAILURE_CODE_RATING_GROUP_ERROR = 2
E_RULE_FAILURE_CODE_SERVICE_IDENTIFIER_ERROR = 3
E_RULE_FAILURE_CODE_GW_PCEF_MALFUNCTION = 4
E_RULE_FAILURE_CODE_RESOURCES_LIMITATION = 5
E_RULE_FAILURE_CODE_MAX_NR_BEARERS_REACHED = 6
E_RULE_FAILURE_CODE_UNKNOWN_BEARER_ID = 7
E_RULE_FAILURE_CODE_MISSING_BEARER_ID = 8
E_RULE_FAILURE_CODE_MISSING_FLOW_DESCRIPTION = 9
E_RULE_FAILURE_CODE_RESOURCE_ALLOCATION_FAILURE = 10
E_RULE_FAILURE_CODE_UNSUCCESSFUL_QOS_VALIDATION = 11
E_RULE_FAILURE_CODE_INCORRECT_FLOW_INFORMATION = 12
E_RULE_FAILURE_CODE_PS_TO_CS_HANDOVER = 13
E_RULE_FAILURE_CODE_TDF_APPLICATION_IDENTIFIER_ERROR = 14
E_RULE_FAILURE_CODE_NO_BEARER_BOUND = 15
E_RULE_FAILURE_CODE_FILTER_RESTRICTIONS = 16
E_RULE_FAILURE_CODE_AN_GW_FAILED = 17
E_RULE_FAILURE_CODE_MISSING_REDIRECT_SERVER_ADDRESS = 18
E_RULE_FAILURE_CODE_CM_END_USER_SERVICE_DENIED = 19
E_RULE_FAILURE_CODE_CM_CREDIT_CONTROL_NOT_APPLICABLE = 20
E_RULE_FAILURE_CODE_CM_AUTHORIZATION_REJECTED = 21
E_RULE_FAILURE_CODE_CM_USER_UNKNOWN = 22
E_RULE_FAILURE_CODE_CM_RATING_FAILED = 23
E_RULE_FAILURE_CODE_ROUTING_RULE_REJECTION = 24

AVP_TGPP_RAT_TYPE = 1032
E_RAT_TYPE_WLAN = 0
E_RAT_TYPE_VIRTUAL = 1
E_RAT_TYPE_UTRAN = 1000
E_RAT_TYPE_GERAN = 1001
E_RAT_TYPE_GAN = 1002
E_RAT_TYPE_HSPA_EVOLUTION = 1003
E_RAT_TYPE_EUTRAN = 1004
E_RAT_TYPE_EUTRAN_NB_IOT = 1005
E_RAT_TYPE_NG_RAN = 1006
E_RAT_TYPE_LTE_M = 1007
E_RAT_TYPE_CDMA2000_1X = 2000
E_RAT_TYPE_HRPD = 2001
E_RAT_TYPE_UMB = 2002
E_RAT_TYPE_EHRPD = 2003

AVP_TGPP_EVENT_REPORT_INDICATION = 1033
AVP_TGPP_ALLOCATION_RETENTION_PRIORITY = 1034
AVP_TGPP_COA_IP_ADDRESS = 1035
AVP_TGPP_TUNNEL_HEADER_FILTER = 1036
AVP_TGPP_TUNNEL_HEADER_LENGTH = 1037
AVP_TGPP_TUNNEL_INFORMATION = 1038
AVP_TGPP_COA_INFORMATION = 1039
AVP_TGPP_APN_AGGREGATE_MAX_BITRATE_DL = 1040
AVP_TGPP_APN_AGGREGATE_MAX_BITRATE_UL = 1041
AVP_TGPP_REVALIDATION_TIME = 1042
AVP_TGPP_RULE_ACTIVATION_TIME = 1043
AVP_TGPP_RULE_DEACTIVATION_TIME = 1044
AVP_TGPP_SESSION_RELEASE_CAUSE = 1045
E_SESSION_RELEASE_CAUSE_UNSPECIFIED_REASON = 0
E_SESSION_RELEASE_CAUSE_UE_SUBSCRIPTION_REASON = 1
E_SESSION_RELEASE_CAUSE_INSUFFICIENT_SERVER_RESOURCES = 2

AVP_TGPP_PRIORITY_LEVEL = 1046
AVP_TGPP_PRE_EMPTION_CAPABILITY = 1047
E_PRE_EMPTION_CAPABILITY_PRE_EMPTION_CAPABILITY_ENABLED = 0
E_PRE_EMPTION_CAPABILITY_PRE_EMPTION_CAPABILITY_DISABLED = 1

AVP_TGPP_PRE_EMPTION_VULNERABILITY = 1048
E_PRE_EMPTION_VULNERABILITY_PRE_EMPTION_VULNERABILITY_ENABLED = 0
E_PRE_EMPTION_VULNERABILITY_PRE_EMPTION_VULNERABILITY_DISABLED = 1

AVP_TGPP_DEFAULT_EPS_BEARER_QOS = 1049
AVP_TGPP_AN_GW_ADDRESS = 1050
AVP_TGPP_QOS_RULE_INSTALL = 1051
AVP_TGPP_QOS_RULE_REMOVE = 1052
AVP_TGPP_QOS_RULE_DEFINITION = 1053
AVP_TGPP_QOS_RULE_NAME = 1054
AVP_TGPP_QOS_RULE_REPORT = 1055
AVP_TGPP_SECURITY_PARAMETER_INDEX = 1056
AVP_TGPP_FLOW_LABEL = 1057
AVP_TGPP_FLOW_INFORMATION = 1058
AVP_TGPP_PACKET_FILTER_CONTENT = 1059
AVP_TGPP_PACKET_FILTER_IDENTIFIER = 1060
AVP_TGPP_PACKET_FILTER_INFORMATION = 1061
AVP_TGPP_PACKET_FILTER_OPERATION = 1062
E_PACKET_FILTER_OPERATION_DELETION = 0
E_PACKET_FILTER_OPERATION_ADDITION = 1
E_PACKET_FILTER_OPERATION_MODIFICATION = 2

AVP_TGPP_RESOURCE_ALLOCATION_NOTIFICATION = 1063
E_RESOURCE_ALLOCATION_NOTIFICATION_ENABLE_NOTIFICATION = 0

AVP_TGPP_SESSION_LINKING_INDICATOR = 1064
E_SESSION_LINKING_INDICATOR_SESSION_LINKING_IMMEDIATE = 0
E_SESSION_LINKING_INDICATOR_SESSION_LINKING_DEFERRED = 1

AVP_TGPP_PDN_CONNECTION_ID = 1065
AVP_TGPP_MONITORING_KEY = 1066
AVP_TGPP_USAGE_MONITORING_INFORMATION = 1067
AVP_TGPP_USAGE_MONITORING_LEVEL = 1068
E_USAGE_MONITORING_LEVEL_SESSION_LEVEL = 0
E_USAGE_MONITORING_LEVEL_PCC_RULE_LEVEL = 1

AVP_TGPP_USAGE_MONITORING_REPORT = 1069
E_USAGE_MONITORING_REPORT_USAGE_MONITORING_REPORT_REQUIRED = 0

AVP_TGPP_USAGE_MONITORING_SUPPORT = 1070
E_USAGE_MONITORING_SUPPORT_USAGE_MONITORING_DISABLED = 0

AVP_TGPP_CSG_INFORMATION_REPORTING = 1071
E_CSG_INFORMATION_REPORTING_CHANGE_CSG_CELL = 0
E_CSG_INFORMATION_REPORTING_CHANGE_CSG_SUBSCRIBED_HYBRID_CELL = 1
E_CSG_INFORMATION_REPORTING_CHANGE_CSG_UNSUBSCRIBED_HYBRID_CELL = 2

AVP_TGPP_PACKET_FILTER_USAGE = 1072
E_PACKET_FILTER_USAGE_SEND_TO_UE = 1

AVP_TGPP_CHARGING_CORRELATION_INDICATOR = 1073
E_CHARGING_CORRELATION_INDICATOR_CHARGING_IDENTIFIER_REQUIRED = 0

AVP_TGPP_QOS_RULE_BASE_NAME = 1074
AVP_TGPP_ROUTING_RULE_REMOVE = 1075
AVP_TGPP_ROUTING_RULE_DEFINITION = 1076
AVP_TGPP_ROUTING_RULE_IDENTIFIER = 1077
AVP_TGPP_ROUTING_FILTER = 1078
AVP_TGPP_ROUTING_IP_ADDRESS = 1079
AVP_TGPP_FLOW_DIRECTION = 1080
E_FLOW_DIRECTION_UNSPECIFIED = 0
E_FLOW_DIRECTION_DOWNLINK = 1
E_FLOW_DIRECTION_UPLINK = 2
E_FLOW_DIRECTION_BIDIRECTIONAL = 3

AVP_TGPP_ROUTING_RULE_INSTALL = 1081
AVP_TGPP_CREDIT_MANAGEMENT_STATUS = 1082
AVP_TGPP_REDIRECT_INFORMATION = 1085
AVP_TGPP_REDIRECT_SUPPORT = 1086
E_REDIRECT_SUPPORT_REDIRECTION_DISABLED = 0
E_REDIRECT_SUPPORT_REDIRECTION_ENABLED = 1

AVP_TGPP_TDF_INFORMATION = 1087
AVP_TGPP_TDF_APPLICATION_IDENTIFIER = 1088
AVP_TGPP_TDF_DESTINATION_HOST = 1089
AVP_TGPP_TDF_DESTINATION_REALM = 1090
AVP_TGPP_TDF_IP_ADDRESS = 1091
AVP_TGPP_ADC_RULE_INSTALL = 1092
AVP_TGPP_ADC_RULE_REMOVE = 1093
AVP_TGPP_ADC_RULE_DEFINITION = 1094
AVP_TGPP_ADC_RULE_BASE_NAME = 1095
AVP_TGPP_ADC_RULE_NAME = 1096
AVP_TGPP_ADC_RULE_REPORT = 1097
AVP_TGPP_APPLICATION_DETECTION_INFORMATION = 1098
AVP_TGPP_PS_TO_CS_SESSION_CONTINUITY = 1099
E_PS_TO_CS_SESSION_CONTINUITY_VIDEO_PS2CS_CONT_CANDIDATE = 0

AVP_TGPP_SERVED_USER_IDENTITY = 1100
AVP_TGPP_VASP_ID = 1101
AVP_TGPP_VAS_ID = 1102
AVP_TGPP_TRIGGER_EVENT = 1103
E_TRIGGER_EVENT_MM1_MESSAGE_SUBMISSION_PROFILE_BASED = 0
E_TRIGGER_EVENT_MM1_MESSAGE_SUBMISSION_ADDRESS_BASED = 1
E_TRIGGER_EVENT_MM1_MESSAGE_DELIVERY = 2
E_TRIGGER_EVENT_MM7_MESSAGE_SUBMISSION_PROFILE_BASED = 3
E_TRIGGER_EVENT_MM7_MESSAGE_SUBMISSION_ADDRESS_BASED = 4

AVP_TGPP_SENDER_ADDRESS = 1104
AVP_TGPP_INITIAL_RECIPIENT_ADDRESS = 1105
AVP_TGPP_RESULT_RECIPIENT_ADDRESS = 1106
AVP_TGPP_MM10_SEQUENCE_NUMBER = 1107
AVP_TGPP_MM10_RECIPIENT_ADDRESS = 1108
AVP_TGPP_ROUTEING_ADDRESS = 1109
AVP_TGPP_ORIGINATING_INTERFACE = 1110
E_ORIGINATING_INTERFACE_MM1 = 0
E_ORIGINATING_INTERFACE_MM3 = 1
E_ORIGINATING_INTERFACE_MM4 = 2
E_ORIGINATING_INTERFACE_MM7 = 3

AVP_TGPP_DELIVERY_REPORT = 1111
E_DELIVERY_REPORT_NO_DELIVERY_REPORT_REQUESTED = 0
E_DELIVERY_REPORT_DELIVERY_REPORT_REQUESTED = 1

AVP_TGPP_READ_REPLY = 1112
E_READ_REPLY_NO_READ_REPLY_REQUESTED = 0
E_READ_REPLY_READ_REPLY_REQUESTED = 1

AVP_TGPP_SENDER_VISIBILITY = 1113
E_SENDER_VISIBILITY_SENDER_IDENTIFICATION_REQUESTED_NOT_TO_BE_HIDDEN = 0
E_SENDER_VISIBILITY_SENDER_IDENTIFICATION_REQUESTED_TO_BE_HIDDEN = 1

AVP_TGPP_SERVICE_KEY = 1114
AVP_TGPP_BILLING_INFORMATION = 1115
AVP_TGPP_STATUS = 1116
AVP_TGPP_STATUS_CODE = 1117
AVP_TGPP_STATUS_TEXT = 1118
AVP_TGPP_ROUTEING_ADDRESS_RESOLUTION = 1119
E_ROUTEING_ADDRESS_RESOLUTION_UNRESOLVED = 0
E_ROUTEING_ADDRESS_RESOLUTION_RESOLVED = 1

AVP_TGPP_DOMAIN_NAME = 1200
AVP_TGPP_RECIPIENT_ADDRESS = 1201
AVP_TGPP_SUBMISSION_TIME = 1202
AVP_TGPP_MM_CONTENT_TYPE = 1203
AVP_TGPP_TYPE_NUMBER = 1204
E_TYPE_NUMBER = 0
E_TYPE_NUMBER_TEXT = 1
E_TYPE_NUMBER_TEXT_HTML = 2
E_TYPE_NUMBER_TEXT_PLAIN = 3
E_TYPE_NUMBER_TEXT_X_HDML = 4
E_TYPE_NUMBER_TEXT_X_TTML = 5
E_TYPE_NUMBER_TEXT_X_VCALENDAR = 6
E_TYPE_NUMBER_TEXT_X_VCARD = 7
E_TYPE_NUMBER_TEXT_VND_WAP_WML = 8
E_TYPE_NUMBER_TEXT_VND_WAP_WMLSCRIPT = 9
E_TYPE_NUMBER_TEXT_VND_WAP_WTA_EVENT = 10
E_TYPE_NUMBER_MULTIPART = 11
E_TYPE_NUMBER_MULTIPART_MIXED = 12
E_TYPE_NUMBER_MULTIPART_FORM_DATA = 13
E_TYPE_NUMBER_MULTIPART_BYTERANTES = 14
E_TYPE_NUMBER_MULTIPART_ALTERNATIVE = 15
E_TYPE_NUMBER_APPLICATION = 16
E_TYPE_NUMBER_APPLICATION_JAVA_VM = 17
E_TYPE_NUMBER_APPLICATION_X_WWW_FORM_URLENCODED = 18
E_TYPE_NUMBER_APPLICATION_X_HDMLC = 19
E_TYPE_NUMBER_APPLICATION_VND_WAP_WMLC = 20
E_TYPE_NUMBER_APPLICATION_VND_WAP_WMLSCRIPTC = 21
E_TYPE_NUMBER_APPLICATION_VND_WAP_WTA_EVENTC = 22
E_TYPE_NUMBER_APPLICATION_VND_WAP_UAPROF = 23
E_TYPE_NUMBER_APPLICATION_VND_WAP_WTLS_CA_CERTIFICATE = 24
E_TYPE_NUMBER_APPLICATION_VND_WAP_WTLS_USER_CERTIFICATE = 25
E_TYPE_NUMBER_APPLICATION_X_X509_CA_CERT = 26
E_TYPE_NUMBER_APPLICATION_X_X509_USER_CERT = 27
E_TYPE_NUMBER_IMAGE = 28
E_TYPE_NUMBER_IMAGE_GIF = 29
E_TYPE_NUMBER_IMAGE_JPEG = 30
E_TYPE_NUMBER_IMAGE_TIFF = 31
E_TYPE_NUMBER_IMAGE_PNG = 32
E_TYPE_NUMBER_IMAGE_VND_WAP_WBMP = 33
E_TYPE_NUMBER_APPLICATION_VND_WAP_MULTIPART = 34
E_TYPE_NUMBER_APPLICATION_VND_WAP_MULTIPART_MIXED = 35
E_TYPE_NUMBER_APPLICATION_VND_WAP_MULTIPART_FORM_DATA = 36
E_TYPE_NUMBER_APPLICATION_VND_WAP_MULTIPART_BYTERANGES = 37
E_TYPE_NUMBER_APPLICATION_VND_WAP_MULTIPART_ALTERNATIVE = 38
E_TYPE_NUMBER_APPLICATION_XML = 39
E_TYPE_NUMBER_TEXT_XML = 40
E_TYPE_NUMBER_APPLICATION_VND_WAP_WBXML = 41
E_TYPE_NUMBER_APPLICATION_X_X968_CROSS_CERT = 42
E_TYPE_NUMBER_APPLICATION_X_X968_CA_CERT = 43
E_TYPE_NUMBER_APPLICATION_X_X968_USER_CERT = 44
E_TYPE_NUMBER_TEXT_VND_WAP_SI = 45
E_TYPE_NUMBER_APPLICATION_VND_WAP_SIC = 46
E_TYPE_NUMBER_TEXT_VND_WAP_SL = 47
E_TYPE_NUMBER_APPLICATION_VND_WAP_SLC = 48
E_TYPE_NUMBER_TEXT_VND_WAP_CO = 49
E_TYPE_NUMBER_APPLICATION_VND_WAP_COC = 50
E_TYPE_NUMBER_APPLICATION_VND_WAP_MULTIPART_RELATED = 51
E_TYPE_NUMBER_APPLICATION_VND_WAP_SIA = 52
E_TYPE_NUMBER_TEXT_VND_WAP_CONNECTIVITY_XML = 53
E_TYPE_NUMBER_APPLICATION_VND_WAP_CONNECTIVITY_WBXML = 54
E_TYPE_NUMBER_APPLICATION_PKCS7_MIME = 55
E_TYPE_NUMBER_APPLICATION_VND_WAP_HASHED_CERTIFICATE = 56
E_TYPE_NUMBER_APPLICATION_VND_WAP_SIGNED_CERTIFICATE = 57
E_TYPE_NUMBER_APPLICATION_VND_WAP_CERT_RESPONSE = 58
E_TYPE_NUMBER_APPLICATION_XHTML_XML = 59
E_TYPE_NUMBER_APPLICATION_WML_XML = 60
E_TYPE_NUMBER_TEXT_CSS = 61
E_TYPE_NUMBER_APPLICATION_VND_WAP_MMS_MESSAGE = 62
E_TYPE_NUMBER_APPLICATION_VND_WAP_ROLLOVER_CERTIFICATE = 63
E_TYPE_NUMBER_APPLICATION_VND_WAP_LOCC_WBXML = 64
E_TYPE_NUMBER_APPLICATION_VND_WAP_LOC_XML = 65
E_TYPE_NUMBER_APPLICATION_VND_SYNCML_DM_WBXML = 66
E_TYPE_NUMBER_APPLICATION_VND_SYNCML_DM_XML = 67
E_TYPE_NUMBER_APPLICATION_VND_SYNCML_NOTIFICATION = 68
E_TYPE_NUMBER_APPLICATION_VND_WAP_XHTML_XML = 69
E_TYPE_NUMBER_APPLICATION_VND_WV_CSP_CIR = 70
E_TYPE_NUMBER_APPLICATION_VND_OMA_DD_XML = 71
E_TYPE_NUMBER_APPLICATION_VND_OMA_DRM_MESSAGE = 72
E_TYPE_NUMBER_APPLICATION_VND_OMA_DRM_CONTENT = 73
E_TYPE_NUMBER_APPLICATION_VND_OMA_DRM_RIGHTS_XML = 74
E_TYPE_NUMBER_APPLICATION_VND_OMA_DRM_RIGHTS_WBXML = 75
E_TYPE_NUMBER_APPLICATION_VND_WV_CSP_XML = 76
E_TYPE_NUMBER_APPLICATION_VND_WV_CSP_WBXML = 77
E_TYPE_NUMBER_APPLICATION_VND_SYNCML_DS_NOTIFICATION = 78
E_TYPE_NUMBER_AUDIO = 79
E_TYPE_NUMBER_VIDEO = 80
E_TYPE_NUMBER_APPLICATION_VND_OMA_DD2_XML = 81
E_TYPE_NUMBER_APPLICATION_MIKEY = 82
E_TYPE_NUMBER_APPLICATION_VND_OMA_DCD = 83
E_TYPE_NUMBER_APPLICATION_VND_OMA_DCDC = 84

AVP_TGPP_ADDITIONAL_TYPE_INFORMATION = 1205
AVP_TGPP_CONTENT_SIZE = 1206
AVP_TGPP_ADDITIONAL_CONTENT_INFORMATION = 1207
AVP_TGPP_ADDRESSEE_TYPE = 1208
E_ADDRESSEE_TYPE_TO = 0
E_ADDRESSEE_TYPE_CC = 1
E_ADDRESSEE_TYPE_BCC = 2

AVP_TGPP_PRIORITY = 1209
E_PRIORITY_LOW = 0
E_PRIORITY_NORMAL = 1
E_PRIORITY_HIGH = 2

AVP_TGPP_MESSAGE_ID = 1210
AVP_TGPP_MESSAGE_TYPE = 1211
E_MESSAGE_TYPE_M_SEND_REQ = 1
E_MESSAGE_TYPE_M_SEND_CONF = 2
E_MESSAGE_TYPE_M_NOTIFICATION_IND = 3
E_MESSAGE_TYPE_M_NOTIFYRESP_IND = 4
E_MESSAGE_TYPE_M_RETRIEVE_CONF = 5
E_MESSAGE_TYPE_M_ACKNOWLEDGE_IND = 6
E_MESSAGE_TYPE_M_DELIVERY_IND = 7
E_MESSAGE_TYPE_M_READ_REC_IND = 8
E_MESSAGE_TYPE_M_READ_ORIG_IND = 9
E_MESSAGE_TYPE_M_FORWARD_REQ = 10
E_MESSAGE_TYPE_M_FORWARD_CONF = 11
E_MESSAGE_TYPE_M_MBOX_STORE_CONF = 12
E_MESSAGE_TYPE_M_MBOX_VIEW_CONF = 13
E_MESSAGE_TYPE_M_MBOX_UPLOAD_CONF = 14
E_MESSAGE_TYPE_M_MBOX_DELETE_CONF = 15

AVP_TGPP_MESSAGE_SIZE = 1212
AVP_TGPP_MESSAGE_CLASS = 1213
AVP_TGPP_CLASS_IDENTIFIER = 1214
E_CLASS_IDENTIFIER_PERSONAL = 0
E_CLASS_IDENTIFIER_ADVERTISEMENT = 1
E_CLASS_IDENTIFIER_INFORMATIONAL = 2
E_CLASS_IDENTIFIER_AUTO = 3

AVP_TGPP_TOKEN_TEXT = 1215
AVP_TGPP_DELIVERY_REPORT_REQUESTED = 1216
E_DELIVERY_REPORT_REQUESTED_NO = 0
E_DELIVERY_REPORT_REQUESTED_YES = 1

AVP_TGPP_ADAPTATIONS = 1217
E_ADAPTATIONS_NO = 0
E_ADAPTATIONS_YES = 1

AVP_TGPP_APPLIC_ID = 1218
AVP_TGPP_AUX_APPLIC_INFO = 1219
AVP_TGPP_CONTENT_CLASS = 1220
E_CONTENT_CLASS_TEXT = 0
E_CONTENT_CLASS_IMAGE_BASIC = 1
E_CONTENT_CLASS_IMAGE_RICH = 2
E_CONTENT_CLASS_VIDEO_BASIC = 3
E_CONTENT_CLASS_VIDEO_RICH = 4
E_CONTENT_CLASS_MEGAPIXEL = 5
E_CONTENT_CLASS_CONTENT_BASIC = 6
E_CONTENT_CLASS_CONTENT_RICH = 7

AVP_TGPP_DRM_CONTENT = 1221
E_DRM_CONTENT_NO = 0
E_DRM_CONTENT_YES = 1

AVP_TGPP_READ_REPLY_REPORT_REQUESTED = 1222
E_READ_REPLY_REPORT_REQUESTED_NO = 0
E_READ_REPLY_REPORT_REQUESTED_YES = 1

AVP_TGPP_REPLY_APPLIC_ID = 1223
AVP_TGPP_FILE_REPAIR_SUPPORTED = 1224
E_FILE_REPAIR_SUPPORTED_SUPPORTED = 1
E_FILE_REPAIR_SUPPORTED_NOT_SUPPORTED = 2

AVP_TGPP_MBMS_USER_SERVICE_TYPE = 1225
E_MBMS_USER_SERVICE_TYPE_DOWNLOAD = 1
E_MBMS_USER_SERVICE_TYPE_STREAMING = 2

AVP_TGPP_UNIT_QUOTA_THRESHOLD = 1226
AVP_TGPP_PDP_ADDRESS = 1227
AVP_TGPP_SGSN_ADDRESS = 1228
AVP_TGPP_POC_SESSION_ID = 1229
AVP_TGPP_DEFERRED_LOCATION_EVENT_TYPE = 1230
AVP_TGPP_LCS_APN = 1231
AVP_TGPP_LCS_CLIENT_ID = 1232
AVP_TGPP_LCS_CLIENT_DIALED_BY_MS = 1233
AVP_TGPP_LCS_CLIENT_EXTERNAL_ID = 1234
AVP_TGPP_LCS_CLIENT_NAME = 1235
AVP_TGPP_LCS_DATA_CODING_SCHEME = 1236
AVP_TGPP_LCS_FORMAT_INDICATOR = 1237
E_LCS_FORMAT_INDICATOR_LOGICAL_NAME = 0
E_LCS_FORMAT_INDICATOR_EMAIL_ADDRESS = 1
E_LCS_FORMAT_INDICATOR_MSISDN = 2
E_LCS_FORMAT_INDICATOR_URL = 3
E_LCS_FORMAT_INDICATOR_SIP_URL = 4

AVP_TGPP_LCS_NAME_STRING = 1238
AVP_TGPP_LCS_REQUESTOR_ID = 1239
AVP_TGPP_LCS_REQUESTOR_ID_STRING = 1240
AVP_TGPP_LCS_CLIENT_TYPE = 1241
E_LCS_CLIENT_TYPE_EMERGENCY_SERVICES = 0
E_LCS_CLIENT_TYPE_VALUE_ADDED_SERVICES = 1
E_LCS_CLIENT_TYPE_PLMN_OPERATOR_SERVICES = 2
E_LCS_CLIENT_TYPE_LAWFUL_INTERCEPT_SERVICES = 3

AVP_TGPP_LOCATION_ESTIMATE = 1242
AVP_TGPP_LOCATION_ESTIMATE_TYPE = 1243
E_LOCATION_ESTIMATE_TYPE_CURRENT_LOCATION = 0
E_LOCATION_ESTIMATE_TYPE_CURRENT_LAST_KNOWN_LOCATION = 1
E_LOCATION_ESTIMATE_TYPE_INITIAL_LOCATION = 2
E_LOCATION_ESTIMATE_TYPE_ACTIVATE_DEFERRED_LOCATION = 3
E_LOCATION_ESTIMATE_TYPE_CANCEL_DEFERRED_LOCATION = 4
E_LOCATION_ESTIMATE_TYPE_NOTIFICATION_VERIFICATION_ONLY = 5

AVP_TGPP_LOCATION_TYPE = 1244
AVP_TGPP_POSITIONING_DATA = 1245
AVP_TGPP_WLAN_SESSION_ID = 1246
AVP_TGPP_PDP_CONTEXT_TYPE = 1247
E_PDP_CONTEXT_TYPE_PRIMARY = 0
E_PDP_CONTEXT_TYPE_SECONDARY = 1

AVP_TGPP_MMBOX_STORAGE_REQUESTED = 1248
E_MMBOX_STORAGE_REQUESTED_NO = 0
E_MMBOX_STORAGE_REQUESTED_YES = 1

AVP_TGPP_SERVICE_SPECIFIC_INFO = 1249
AVP_TGPP_CALLED_ASSERTED_IDENTITY = 1250
AVP_TGPP_REQUESTED_PARTY_ADDRESS = 1251
AVP_TGPP_POC_USER_ROLE = 1252
AVP_TGPP_POC_USER_ROLE_IDS = 1253
AVP_TGPP_POC_USER_ROLE_INFO_UNITS = 1254
E_POC_USER_ROLE_INFO_UNITS_MODERATOR = 1
E_POC_USER_ROLE_INFO_UNITS_DISPATCHER = 2
E_POC_USER_ROLE_INFO_UNITS_SESSION_OWNER = 3
E_POC_USER_ROLE_INFO_UNITS_SESSION_PARTICIPANT = 4

AVP_TGPP_TALK_BURST_EXCHANGE = 1255
AVP_TGPP_SERVICE_GENERIC_INFORMATION = 1256
AVP_TGPP_SERVICE_SPECIFIC_TYPE = 1257
AVP_TGPP_EVENT_CHARGING_TIMESTAMP = 1258
AVP_TGPP_PARTICIPANT_ACCESS_PRIORITY = 1259
E_PARTICIPANT_ACCESS_PRIORITY_PRE_EMPTIVE_PRIORITY = 1
E_PARTICIPANT_ACCESS_PRIORITY_HIGH_PRIORITY = 2
E_PARTICIPANT_ACCESS_PRIORITY_NORMAL_PRIORITY = 3
E_PARTICIPANT_ACCESS_PRIORITY_LOW_PRIORITY = 4

AVP_TGPP_PARTICIPANT_GROUP = 1260
AVP_TGPP_POC_CHANGE_CONDITION = 1261
E_POC_CHANGE_CONDITION_SERVICECHANGE = 0
E_POC_CHANGE_CONDITION_VOLUMELIMIT = 1
E_POC_CHANGE_CONDITION_TIMELIMIT = 2
E_POC_CHANGE_CONDITION_NUMBEROFTALKBURSTLIMIT = 3
E_POC_CHANGE_CONDITION_NUMBEROFACTIVEPARTICIPANTS = 4
E_POC_CHANGE_CONDITION_TARIFFTIME = 5

AVP_TGPP_POC_CHANGE_TIME = 1262
AVP_TGPP_ACCESS_NETWORK_INFORMATION = 1263
AVP_TGPP_TRIGGER = 1264
AVP_TGPP_BASE_TIME_INTERVAL = 1265
AVP_TGPP_ENVELOPE = 1266
AVP_TGPP_ENVELOPE_END_TIME = 1267
AVP_TGPP_ENVELOPE_REPORTING = 1268
E_ENVELOPE_REPORTING_DO_NOT_REPORT_ENVELOPES = 0
E_ENVELOPE_REPORTING_REPORT_ENVELOPES = 1
E_ENVELOPE_REPORTING_REPORT_ENVELOPES_WITH_VOLUME = 2
E_ENVELOPE_REPORTING_REPORT_ENVELOPES_WITH_EVENTS = 3
E_ENVELOPE_REPORTING_REPORT_ENVELOPES_WITH_VOLUME_AND_EVENTS = 4

AVP_TGPP_ENVELOPE_START_TIME = 1269
AVP_TGPP_TIME_QUOTA_MECHANISM = 1270
AVP_TGPP_TIME_QUOTA_TYPE = 1271
E_TIME_QUOTA_TYPE_DISCRETE_TIME_PERIOD = 0
E_TIME_QUOTA_TYPE_CONTINUOUS_TIME_PERIOD = 1

AVP_TGPP_EARLY_MEDIA_DESCRIPTION = 1272
AVP_TGPP_SDP_TIMESTAMPS = 1273
AVP_TGPP_SDP_OFFER_TIMESTAMP = 1274
AVP_TGPP_SDP_ANSWER_TIMESTAMP = 1275
AVP_TGPP_AF_CORRELATION_INFORMATION = 1276
AVP_TGPP_POC_SESSION_INITIATION_TYPE = 1277
E_POC_SESSION_INITIATION_TYPE_PRE_ESTABLISHED = 0
E_POC_SESSION_INITIATION_TYPE_ON_DEMAND = 1

AVP_TGPP_OFFLINE_CHARGING = 1278
AVP_TGPP_USER_PARTICIPATING_TYPE = 1279
E_USER_PARTICIPATING_TYPE_NORMAL = 0
E_USER_PARTICIPATING_TYPE_NW_POC_BOX = 1
E_USER_PARTICIPATING_TYPE_UE_POC_BOX = 2

AVP_TGPP_ALTERNATE_CHARGED_PARTY_ADDRESS = 1280
AVP_TGPP_IMS_COMMUNICATION_SERVICE_IDENTIFIER = 1281
AVP_TGPP_NUMBER_OF_RECEIVED_TALK_BURSTS = 1282
AVP_TGPP_NUMBER_OF_TALK_BURSTS = 1283
AVP_TGPP_RECEIVED_TALK_BURST_TIME = 1284
AVP_TGPP_RECEIVED_TALK_BURST_VOLUME = 1285
AVP_TGPP_TALK_BURST_TIME = 1286
AVP_TGPP_TALK_BURST_VOLUME = 1287
AVP_TGPP_MEDIA_INITIATOR_PARTY = 1288
AVP_TGPP_PC5_RADIO_TECHNOLOGY = 1300
E_PC5_RADIO_TECHNOLOGY_EUTRA = 0
E_PC5_RADIO_TECHNOLOGY_WLAN = 1
E_PC5_RADIO_TECHNOLOGY_BOTH_EUTRA_AND_WLAN = 2

AVP_TGPP_RAN_END_TIMESTAMP = 1301
AVP_TGPP_RAN_SECONDARY_RAT_USAGE_REPORT = 1302
AVP_TGPP_RAN_START_TIMESTAMP = 1303
AVP_TGPP_SECONDARY_RAT_TYPE = 1304
AVP_TGPP_CIVIC_ADDRESS_INFORMATION = 1305
AVP_TGPP_WLAN_OPERATOR_ID = 1306
AVP_TGPP_WLAN_OPERATOR_NAME = 1307
AVP_TGPP_WLAN_PLMN_ID = 1308
AVP_TGPP_API_CONTENT = 1309
AVP_TGPP_API_DIRECTION = 1310
E_API_DIRECTION_INVOCATION = 0
E_API_DIRECTION_NOTIFICATION = 1

AVP_TGPP_API_IDENTIFIER = 1311
AVP_TGPP_API_INVOCATION_TIMESTAMP = 1312
AVP_TGPP_API_RESULT_CODE = 1313
AVP_TGPP_API_SIZE = 1314
AVP_TGPP_VOLTE_INFORMATION = 1323
AVP_TGPP_SUBSCRIPTION_DATA = 1400
AVP_TGPP_TERMINAL_INFORMATION = 1401
AVP_TGPP_IMEI = 1402
AVP_TGPP_SOFTWARE_VERSION = 1403
AVP_TGPP_QOS_SUBSCRIBED = 1404
AVP_TGPP_ULR_FLAGS = 1405
AVP_TGPP_ULA_FLAGS = 1406
AVP_TGPP_VISITED_PLMN_ID = 1407
AVP_TGPP_REQUESTED_EUTRAN_AUTHENTICATION_INFO = 1408
AVP_TGPP_REQUESTED_UTRAN_GERAN_AUTHENTICATION_INFO = 1409
AVP_TGPP_NUMBER_OF_REQUESTED_VECTORS = 1410
AVP_TGPP_RE_SYNCHRONIZATION_INFO = 1411
AVP_TGPP_IMMEDIATE_RESPONSE_PREFERRED = 1412
AVP_TGPP_AUTHENTICATION_INFO = 1413
AVP_TGPP_E_UTRAN_VECTOR = 1414
AVP_TGPP_UTRAN_VECTOR = 1415
AVP_TGPP_GERAN_VECTOR = 1416
AVP_TGPP_NETWORK_ACCESS_MODE = 1417
E_NETWORK_ACCESS_MODE_PACKET_AND_CIRCUIT = 0
E_NETWORK_ACCESS_MODE_RESERVED = 1
E_NETWORK_ACCESS_MODE_ONLY_PACKET = 2

AVP_TGPP_HPLMN_ODB = 1418
AVP_TGPP_ITEM_NUMBER = 1419
AVP_TGPP_CANCELLATION_TYPE = 1420
E_CANCELLATION_TYPE_MME_UPDATE_PROCEDURE = 0
E_CANCELLATION_TYPE_SGSN_UPDATE_PROCEDURE = 1
E_CANCELLATION_TYPE_SUBSCRIPTION_WITHDRAWAL = 2
E_CANCELLATION_TYPE_UPDATE_PROCEDURE_IWF = 3
E_CANCELLATION_TYPE_INITIAL_ATTACH_PROCEDURE = 4

AVP_TGPP_DSR_FLAGS = 1421
AVP_TGPP_DSA_FLAGS = 1422
AVP_TGPP_CONTEXT_IDENTIFIER = 1423
AVP_TGPP_SUBSCRIBER_STATUS = 1424
E_SUBSCRIBER_STATUS_SERVICE_GRANTED = 0
E_SUBSCRIBER_STATUS_OPERATOR_DETERMINED_BARRING = 1

AVP_TGPP_OPERATOR_DETERMINED_BARRING = 1425
AVP_TGPP_ACCESS_RESTRICTION_DATA = 1426
AVP_TGPP_APN_OI_REPLACEMENT = 1427
AVP_TGPP_ALL_APN_CONFIGURATIONS_INCLUDED_INDICATOR = 1428
E_ALL_APN_CONFIGURATIONS_INCLUDED_INDICATOR_ALL_APN_CONFIGURATIONS_INCLUDED = 0
E_ALL_APN_CONFIGURATIONS_INCLUDED_INDICATOR_MODIFIED_ADDED_APN_CONFIGURATIONS_INCLUDED = 1

AVP_TGPP_APN_CONFIGURATION_PROFILE = 1429
AVP_TGPP_APN_CONFIGURATION = 1430
AVP_TGPP_EPS_SUBSCRIBED_QOS_PROFILE = 1431
AVP_TGPP_VPLMN_DYNAMIC_ADDRESS_ALLOWED = 1432
E_VPLMN_DYNAMIC_ADDRESS_ALLOWED_NOTALLOWED = 0
E_VPLMN_DYNAMIC_ADDRESS_ALLOWED_ALLOWED = 1

AVP_TGPP_STN_SR = 1433
AVP_TGPP_ALERT_REASON = 1434
E_ALERT_REASON_UE_PRESENT = 0
E_ALERT_REASON_UE_MEMORY_AVAILABLE = 1

AVP_TGPP_AMBR = 1435
AVP_TGPP_CSG_SUBSCRIPTION_DATA = 1436
AVP_TGPP_CSG_ID = 1437
AVP_TGPP_PDN_GW_ALLOCATION_TYPE = 1438
E_PDN_GW_ALLOCATION_TYPE_STATIC = 0
E_PDN_GW_ALLOCATION_TYPE_DYNAMIC = 1

AVP_TGPP_EXPIRATION_DATE = 1439
AVP_TGPP_RAT_FREQUENCY_SELECTION_PRIORITY_ID = 1440
AVP_TGPP_IDA_FLAGS = 1441
AVP_TGPP_PUA_FLAGS = 1442
AVP_TGPP_NOR_FLAGS = 1443
AVP_TGPP_USER_ID = 1444
AVP_TGPP_EQUIPMENT_STATUS = 1445
E_EQUIPMENT_STATUS_WHITELISTED = 0
E_EQUIPMENT_STATUS_BLACKLISTED = 1
E_EQUIPMENT_STATUS_GREYLISTED = 2

AVP_TGPP_REGIONAL_SUBSCRIPTION_ZONE_CODE = 1446
AVP_TGPP_RAND = 1447
AVP_TGPP_XRES = 1448
AVP_TGPP_AUTN = 1449
AVP_TGPP_KASME = 1450
AVP_TGPP_TRACE_COLLECTION_ENTITY = 1452
AVP_TGPP_KC = 1453
AVP_TGPP_SRES = 1454
AVP_TGPP_REQUESTING_NODE_TYPE = 1455
E_REQUESTING_NODE_TYPE_MME = 0
E_REQUESTING_NODE_TYPE_SGSN = 1
E_REQUESTING_NODE_TYPE_MME_SGSN = 2

AVP_TGPP_PDN_TYPE = 1456
E_PDN_TYPE_IPV4 = 0
E_PDN_TYPE_IPV6 = 1
E_PDN_TYPE_IPV4V6 = 2
E_PDN_TYPE_IPV4_OR_IPV6 = 3
E_PDN_TYPE_NON_IP = 4

AVP_TGPP_ROAMING_RESTRICTED_DUE_TO_UNSUPPORTED_FEATURE = 1457
E_ROAMING_RESTRICTED_DUE_TO_UNSUPPORTED_FEATURE_ROAMING_RESTRICTED_DUE_TO_UNSUPPORTED_FEATURE = 0

AVP_TGPP_TRACE_DATA = 1458
AVP_TGPP_TRACE_REFERENCE = 1459
AVP_TGPP_TRACE_DEPTH = 1462
E_TRACE_DEPTH_MINIMUM = 0
E_TRACE_DEPTH_MEDIUM = 1
E_TRACE_DEPTH_MAXIMUM = 2
E_TRACE_DEPTH_MINIMUMWITHOUTVENDORSPECIFICEXTENSION = 3
E_TRACE_DEPTH_MEDIUMWITHOUTVENDORSPECIFICEXTENSION = 4
E_TRACE_DEPTH_MAXIMUMWITHOUTVENDORSPECIFICEXTENSION = 5

AVP_TGPP_TRACE_NE_TYPE_LIST = 1463
AVP_TGPP_TRACE_INTERFACE_LIST = 1464
AVP_TGPP_TRACE_EVENT_LIST = 1465
AVP_TGPP_OMC_ID = 1466
AVP_TGPP_GPRS_SUBSCRIPTION_DATA = 1467
AVP_TGPP_COMPLETE_DATA_LIST_INCLUDED_INDICATOR = 1468
E_COMPLETE_DATA_LIST_INCLUDED_INDICATOR_ALL_PDP_CONTEXTS_INCLUDED = 0
E_COMPLETE_DATA_LIST_INCLUDED_INDICATOR_MODIFIED_ADDED_PDP_CONTEXTS_INCLUDED = 1

AVP_TGPP_PDP_CONTEXT = 1469
AVP_TGPP_PDP_TYPE = 1470
AVP_TGPP_3GPP2_MEID = 1471
AVP_TGPP_SPECIFIC_APN_INFO = 1472
AVP_TGPP_LCS_INFO = 1473
AVP_TGPP_GMLC_NUMBER = 1474
AVP_TGPP_LCS_PRIVACYEXCEPTION = 1475
AVP_TGPP_SS_CODE = 1476
AVP_TGPP_SS_STATUS = 1477
AVP_TGPP_NOTIFICATION_TO_UE_USER = 1478
E_NOTIFICATION_TO_UE_USER_NOTIFY_LOCATION_ALLOWED = 0
E_NOTIFICATION_TO_UE_USER_NOTIFYANDVERIFY_LOCATION_ALLOWED_IF_NO_RESPONSE = 1
E_NOTIFICATION_TO_UE_USER_NOTIFYANDVERIFY_LOCATION_NOT_ALLOWED_IF_NO_RESPONSE = 2
E_NOTIFICATION_TO_UE_USER_LOCATION_NOT_ALLOWED = 3

AVP_TGPP_EXTERNAL_CLIENT = 1479
AVP_TGPP_CLIENT_IDENTITY = 1480
AVP_TGPP_GMLC_RESTRICTION = 1481
E_GMLC_RESTRICTION_GMLC_LIST = 0
E_GMLC_RESTRICTION_HOME_COUNTRY = 1

AVP_TGPP_PLMN_CLIENT = 1482
E_PLMN_CLIENT_BROADCAST_SERVICE = 0
E_PLMN_CLIENT_O_AND_M_HPLMN = 1
E_PLMN_CLIENT_O_AND_M_VPLMN = 2
E_PLMN_CLIENT_ANONYMOUS_LOCATION = 3
E_PLMN_CLIENT_TARGET_UE_SUBSCRIBED_SERVICE = 4

AVP_TGPP_3GPP_SERVICE_TYPE = 1483
AVP_TGPP_SERVICETYPEIDENTITY = 1484
AVP_TGPP_MO_LR = 1485
AVP_TGPP_TELESERVICE_LIST = 1486
AVP_TGPP_TS_CODE = 1487
AVP_TGPP_CALL_BARRING_INFOR_LIST = 1488
AVP_TGPP_SGSN_NUMBER = 1489
AVP_TGPP_IDR_FLAGS = 1490
AVP_TGPP_ICS_INDICATOR = 1491
E_ICS_INDICATOR_FALSE = 0
E_ICS_INDICATOR_TRUE = 1

AVP_TGPP_IMS_VOICE_OVER_PS_SESSIONS_SUPPORTED = 1492
E_IMS_VOICE_OVER_PS_SESSIONS_SUPPORTED_NOT_SUPPORTED = 0
E_IMS_VOICE_OVER_PS_SESSIONS_SUPPORTED_SUPPORTED = 1

AVP_TGPP_HOMOGENEOUS_SUPPORT_OF_IMS_VOICE_OVER_PS_SESSIONS = 1493
E_HOMOGENEOUS_SUPPORT_OF_IMS_VOICE_OVER_PS_SESSIONS_NOT_SUPPORTED = 0
E_HOMOGENEOUS_SUPPORT_OF_IMS_VOICE_OVER_PS_SESSIONS_SUPPORTED = 1

AVP_TGPP_LAST_UE_ACTIVITY_TIME = 1494
AVP_TGPP_EPS_USER_STATE = 1495
AVP_TGPP_EPS_LOCATION_INFORMATION = 1496
AVP_TGPP_MME_USER_STATE = 1497
AVP_TGPP_SGSN_USER_STATE = 1498
AVP_TGPP_USER_STATE = 1499
E_USER_STATE_DETACHED = 0
E_USER_STATE_ATTACHED_NOT_REACHABLE_FOR_PAGING = 1
E_USER_STATE_ATTACHED_REACHABLE_FOR_PAGING = 2
E_USER_STATE_CONNECTED_NOT_REACHABLE_FOR_PAGING = 3
E_USER_STATE_CONNECTED_REACHABLE_FOR_PAGING = 4
E_USER_STATE_NETWORK_DETERMINED_NOT_REACHABLE = 5

AVP_TGPP_NON_3GPP_USER_DATA = 1500
AVP_TGPP_NON_3GPP_IP_ACCESS = 1501
E_NON_3GPP_IP_ACCESS_NON_3GPP_SUBSCRIPTION_ALLOWED = 0
E_NON_3GPP_IP_ACCESS_NON_3GPP_SUBSCRIPTION_BARRED = 1

AVP_TGPP_NON_3GPP_IP_ACCESS_APN = 1502
E_NON_3GPP_IP_ACCESS_APN_NON_3GPP_APNS_ENABLE = 0
E_NON_3GPP_IP_ACCESS_APN_NON_3GPP_APNS_DISABLE = 1

AVP_TGPP_AN_TRUSTED = 1503
E_AN_TRUSTED_TRUSTED = 0
E_AN_TRUSTED_UNTRUSTED = 1

AVP_TGPP_ANID = 1504
AVP_TGPP_TRACE_INFO = 1505
AVP_TGPP_MIP_FA_RK = 1506
AVP_TGPP_MIP_FA_RK_SPI = 1507
AVP_TGPP_PPR_FLAGS = 1508
AVP_TGPP_WLAN_IDENTIFIER = 1509
AVP_TGPP_TWAN_ACCESS_INFO = 1510
AVP_TGPP_ACCESS_AUTHORIZATION_FLAGS = 1511
AVP_TGPP_TWAN_DEFAULT_APN_CONTEXT_ID = 1512
AVP_TGPP_FULL_NETWORK_NAME = 1516
AVP_TGPP_SHORT_NETWORK_NAME = 1517
AVP_TGPP_AAA_FAILURE_INDICATION = 1518
AVP_TGPP_TRANSPORT_ACCESS_TYPE = 1519
E_TRANSPORT_ACCESS_TYPE_BBF = 0

AVP_TGPP_DER_FLAGS = 1520
AVP_TGPP_DEA_FLAGS = 1521
AVP_TGPP_RAR_FLAGS = 1522
AVP_TGPP_DER_S6B_FLAGS = 1523
AVP_TGPP_SSID = 1524
AVP_TGPP_HESSID = 1525
AVP_TGPP_ACCESS_NETWORK_INFO = 1526
AVP_TGPP_TWAN_CONNECTION_MODE = 1527
AVP_TGPP_TWAN_CONNECTIVITY_PARAMETERS = 1528
AVP_TGPP_CONNECTIVITY_FLAGS = 1529
AVP_TGPP_TWAN_PCO = 1530
AVP_TGPP_TWAG_CP_ADDRESS = 1531
AVP_TGPP_TWAG_UP_ADDRESS = 1532
AVP_TGPP_TWAN_S2A_FAILURE_CAUSE = 1533
AVP_TGPP_SM_BACK_OFF_TIMER = 1534
AVP_TGPP_WLCP_KEY = 1535
AVP_TGPP_ORIGINATION_TIME_STAMP = 1536
AVP_TGPP_MAXIMUM_WAIT_TIME = 1537
AVP_TGPP_EMERGENCY_SERVICES = 1538
AVP_TGPP_AAR_FLAGS = 1539
AVP_TGPP_IMEI_CHECK_IN_VPLMN_RESULT = 1540
AVP_TGPP_ERP_AUTHORIZATION = 1541
AVP_TGPP_MME_LOCATION_INFORMATION = 1600
AVP_TGPP_SGSN_LOCATION_INFORMATION = 1601
AVP_TGPP_E_UTRAN_CELL_GLOBAL_IDENTITY = 1602
AVP_TGPP_TRACKING_AREA_IDENTITY = 1603
AVP_TGPP_CELL_GLOBAL_IDENTITY = 1604
AVP_TGPP_ROUTING_AREA_IDENTITY = 1605
AVP_TGPP_LOCATION_AREA_IDENTITY = 1606
AVP_TGPP_SERVICE_AREA_IDENTITY = 1607
AVP_TGPP_GEOGRAPHICAL_INFORMATION = 1608
AVP_TGPP_GEODETIC_INFORMATION = 1609
AVP_TGPP_CURRENT_LOCATION_RETRIEVED = 1610
E_CURRENT_LOCATION_RETRIEVED_ACTIVE_LOCATION_RETRIEVAL = 0

AVP_TGPP_AGE_OF_LOCATION_INFORMATION = 1611
AVP_TGPP_ACTIVE_APN = 1612
AVP_TGPP_SIPTO_PERMISSION = 1613
E_SIPTO_PERMISSION_SIPTO_ALLOWED = 0
E_SIPTO_PERMISSION_SIPTO_NOTALLOWED = 1

AVP_TGPP_ERROR_DIAGNOSTIC = 1614
E_ERROR_DIAGNOSTIC_GPRS_DATA_SUBSCRIBED = 0
E_ERROR_DIAGNOSTIC_NO_GPRS_DATA_SUBSCRIBED = 1
E_ERROR_DIAGNOSTIC_ODB_ALL_APN = 2
E_ERROR_DIAGNOSTIC_ODB_HPLMN_APN = 3
E_ERROR_DIAGNOSTIC_ODB_VPLMN_APN = 4

AVP_TGPP_UE_SRVCC_CAPABILITY = 1615
E_UE_SRVCC_CAPABILITY_UE_SRVCC_NOT_SUPPORTED = 0
E_UE_SRVCC_CAPABILITY_UE_SRVCC_SUPPORTED = 1

AVP_TGPP_MPS_PRIORITY = 1616
AVP_TGPP_VPLMN_LIPA_ALLOWED = 1617
E_VPLMN_LIPA_ALLOWED_LIPA_NOTALLOWED = 0
E_VPLMN_LIPA_ALLOWED_LIPA_ALLOWED = 1

AVP_TGPP_LIPA_PERMISSION = 1618
E_LIPA_PERMISSION_LIPA_PROHIBITED = 0
E_LIPA_PERMISSION_LIPA_ONLY = 1
E_LIPA_PERMISSION_LIPA_CONDITIONAL = 2

AVP_TGPP_SUBSCRIBED_PERIODIC_RAU_TAU_TIMER = 1619
AVP_TGPP_EXT_PDP_TYPE = 1620
AVP_TGPP_EXT_PDP_ADDRESS = 1621
AVP_TGPP_MDT_CONFIGURATION = 1622
AVP_TGPP_JOB_TYPE = 1623
E_JOB_TYPE_IMMEDIATE_MDT_ONLY = 0
E_JOB_TYPE_LOGGED_MDT_ONLY = 1
E_JOB_TYPE_TRACE_ONLY = 2
E_JOB_TYPE_IMMEDIATE_MDT_AND_TRACE = 3
E_JOB_TYPE_RLF_REPORTS_ONLY = 4

AVP_TGPP_AREA_SCOPE = 1624
AVP_TGPP_LIST_OF_MEASUREMENTS = 1625
AVP_TGPP_REPORTING_TRIGGER = 1626
AVP_TGPP_REPORT_INTERVAL = 1627
E_REPORT_INTERVAL_UMTS_250_MS = 0
E_REPORT_INTERVAL_UMTS_500_MS = 1
E_REPORT_INTERVAL_UMTS_1000_MS = 2
E_REPORT_INTERVAL_UMTS_2000_MS = 3
E_REPORT_INTERVAL_UMTS_3000_MS = 4
E_REPORT_INTERVAL_UMTS_4000_MS = 5
E_REPORT_INTERVAL_UMTS_6000_MS = 6
E_REPORT_INTERVAL_UMTS_8000_MS = 7
E_REPORT_INTERVAL_UMTS_12000_MS = 8
E_REPORT_INTERVAL_UMTS_16000_MS = 9
E_REPORT_INTERVAL_UMTS_20000_MS = 10
E_REPORT_INTERVAL_UMTS_24000_MS = 11
E_REPORT_INTERVAL_UMTS_28000_MS = 12
E_REPORT_INTERVAL_UMTS_32000_MS = 13
E_REPORT_INTERVAL_UMTS_64000_MS = 14
E_REPORT_INTERVAL_LTE_120_MS = 15
E_REPORT_INTERVAL_LTE_240_MS = 16
E_REPORT_INTERVAL_LTE_480_MS = 17
E_REPORT_INTERVAL_LTE_640_MS = 18
E_REPORT_INTERVAL_LTE_1024_MS = 19
E_REPORT_INTERVAL_LTE_2048_MS = 20
E_REPORT_INTERVAL_LTE_5120_MS = 21
E_REPORT_INTERVAL_LTE_10240_MS = 22
E_REPORT_INTERVAL_LTE_60000_MS = 23
E_REPORT_INTERVAL_LTE_360000_MS = 24
E_REPORT_INTERVAL_LTE_720000_MS = 25
E_REPORT_INTERVAL_LTE_1800000_MS = 26
E_REPORT_INTERVAL_LTE_3600000_MS = 27

AVP_TGPP_REPORT_AMOUNT = 1628
E_REPORT_AMOUNT_1 = 0
E_REPORT_AMOUNT_2 = 1
E_REPORT_AMOUNT_4 = 2
E_REPORT_AMOUNT_8 = 3
E_REPORT_AMOUNT_16 = 4
E_REPORT_AMOUNT_32 = 5
E_REPORT_AMOUNT_64 = 6
E_REPORT_AMOUNT_INFINITY = 7

AVP_TGPP_EVENT_THRESHOLD_RSRP = 1629
AVP_TGPP_EVENT_THRESHOLD_RSRQ = 1630
AVP_TGPP_LOGGING_INTERVAL = 1631
E_LOGGING_INTERVAL_1_28 = 0
E_LOGGING_INTERVAL_2_56 = 1
E_LOGGING_INTERVAL_5_12 = 2
E_LOGGING_INTERVAL_10_24 = 3
E_LOGGING_INTERVAL_20_48 = 4
E_LOGGING_INTERVAL_30_72 = 5
E_LOGGING_INTERVAL_40_96 = 6
E_LOGGING_INTERVAL_61_44 = 7

AVP_TGPP_LOGGING_DURATION = 1632
E_LOGGING_DURATION_600_SEC = 0
E_LOGGING_DURATION_1200_SEC = 1
E_LOGGING_DURATION_2400_SEC = 2
E_LOGGING_DURATION_3600_SEC = 3
E_LOGGING_DURATION_5400_SEC = 4
E_LOGGING_DURATION_7200_SEC = 5

AVP_TGPP_RELAY_NODE_INDICATOR = 1633
E_RELAY_NODE_INDICATOR_NOT_RELAY_NODE = 0
E_RELAY_NODE_INDICATOR_RELAY_NODE = 1

AVP_TGPP_MDT_USER_CONSENT = 1634
E_MDT_USER_CONSENT_CONSENT_NOT_GIVEN = 0
E_MDT_USER_CONSENT_CONSENT_GIVEN = 1

AVP_TGPP_PUR_FLAGS = 1635
AVP_TGPP_SUBSCRIBED_VSRVCC = 1636
E_SUBSCRIBED_VSRVCC_VSRVCC_SUBSCRIBED = 0

AVP_TGPP_EQUIVALENT_PLMN_LIST = 1637
AVP_TGPP_CLR_FLAGS = 1638
AVP_TGPP_UVR_FLAGS = 1639
AVP_TGPP_UVA_FLAGS = 1640
AVP_TGPP_VPLMN_CSG_SUBSCRIPTION_DATA = 1641
AVP_TGPP_TIME_ZONE = 1642
AVP_TGPP_A_MSISDN = 1643
AVP_TGPP_MME_NUMBER_FOR_MT_SMS = 1645
AVP_TGPP_SMS_REGISTER_REQUEST = 1648
E_SMS_REGISTER_REQUEST_SMS_REGISTRATION_REQUIRED = 0
E_SMS_REGISTER_REQUEST_SMS_REGISTRATION_NOT_PREFERRED = 1
E_SMS_REGISTER_REQUEST_NO_PREFERENCE = 2

AVP_TGPP_LOCAL_TIME_ZONE = 1649
AVP_TGPP_DAYLIGHT_SAVING_TIME = 1650
E_DAYLIGHT_SAVING_TIME_NO_ADJUSTMENT = 0
E_DAYLIGHT_SAVING_TIME_PLUS_ONE_HOUR_ADJUSTMENT = 1
E_DAYLIGHT_SAVING_TIME_PLUS_TWO_HOURS_ADJUSTMENT = 2

AVP_TGPP_SUBSCRIPTION_DATA_FLAGS = 1654
AVP_TGPP_MEASUREMENT_PERIOD_LTE = 1655
E_MEASUREMENT_PERIOD_LTE_1024_MS = 0
E_MEASUREMENT_PERIOD_LTE_1280_MS = 1
E_MEASUREMENT_PERIOD_LTE_2048_MS = 2
E_MEASUREMENT_PERIOD_LTE_2560_MS = 3
E_MEASUREMENT_PERIOD_LTE_5120_MS = 4
E_MEASUREMENT_PERIOD_LTE_10240_MS = 5
E_MEASUREMENT_PERIOD_LTE_1_MIN = 6

AVP_TGPP_MEASUREMENT_PERIOD_UMTS = 1656
E_MEASUREMENT_PERIOD_UMTS_250_MS = 0
E_MEASUREMENT_PERIOD_UMTS_500_MS = 1
E_MEASUREMENT_PERIOD_UMTS_1000_MS = 2
E_MEASUREMENT_PERIOD_UMTS_2000_MS = 3
E_MEASUREMENT_PERIOD_UMTS_3000_MS = 4
E_MEASUREMENT_PERIOD_UMTS_4000_MS = 5
E_MEASUREMENT_PERIOD_UMTS_6000_MS = 6
E_MEASUREMENT_PERIOD_UMTS_8000_MS = 7
E_MEASUREMENT_PERIOD_UMTS_12000_MS = 8
E_MEASUREMENT_PERIOD_UMTS_16000_MS = 9
E_MEASUREMENT_PERIOD_UMTS_20000_MS = 10
E_MEASUREMENT_PERIOD_UMTS_24000_MS = 11
E_MEASUREMENT_PERIOD_UMTS_28000_MS = 12
E_MEASUREMENT_PERIOD_UMTS_32000_MS = 13
E_MEASUREMENT_PERIOD_UMTS_64000_MS = 14

AVP_TGPP_COLLECTION_PERIOD_RRM_LTE = 1657
AVP_TGPP_COLLECTION_PERIOD_RRM_UMTS = 1658
AVP_TGPP_POSITIONING_METHOD = 1659
AVP_TGPP_MEASUREMENT_QUANTITY = 1660
AVP_TGPP_EVENT_THRESHOLD_EVENT_1F = 1661
AVP_TGPP_EVENT_THRESHOLD_EVENT_1I = 1662
AVP_TGPP_RESTORATION_PRIORITY = 1663
AVP_TGPP_SGS_MME_IDENTITY = 1664
AVP_TGPP_SIPTO_LOCAL_NETWORK_PERMISSION = 1665
E_SIPTO_LOCAL_NETWORK_PERMISSION_SIPTO_AT_LOCAL_NETWORK_ALLOWED = 0
E_SIPTO_LOCAL_NETWORK_PERMISSION_SIPTO_AT_LOCAL_NETWORK_NOTALLOWED = 1

AVP_TGPP_COUPLED_NODE_DIAMETER_ID = 1666
AVP_TGPP_WLAN_OFFLOADABILITY = 1667
AVP_TGPP_WLAN_OFFLOADABILITY_EUTRAN = 1668
AVP_TGPP_WLAN_OFFLOADABILITY_UTRAN = 1669
AVP_TGPP_RESET_ID = 1670
AVP_TGPP_MDT_ALLOWED_PLMN_ID = 1671
AVP_TGPP_ADJACENT_PLMNS = 1672
AVP_TGPP_ADJACENT_ACCESS_RESTRICTION_DATA = 1673
AVP_TGPP_DL_BUFFERING_SUGGESTED_PACKET_COUNT = 1674
AVP_TGPP_IMSI_GROUP_ID = 1675
AVP_TGPP_GROUP_SERVICE_ID = 1676
AVP_TGPP_GROUP_PLMN_ID = 1677
AVP_TGPP_LOCAL_GROUP_ID = 1678
AVP_TGPP_AIR_FLAGS = 1679
AVP_TGPP_UE_USAGE_TYPE = 1680
AVP_TGPP_NON_IP_PDN_TYPE_INDICATOR = 1681
E_NON_IP_PDN_TYPE_INDICATOR_FALSE = 0
E_NON_IP_PDN_TYPE_INDICATOR_TRUE = 1

AVP_TGPP_NON_IP_DATA_DELIVERY_MECHANISM = 1682
E_NON_IP_DATA_DELIVERY_MECHANISM_SGI_BASED_DATA_DELIVERY = 0
E_NON_IP_DATA_DELIVERY_MECHANISM_SCEF_BASED_DATA_DELIVERY = 1

AVP_TGPP_ADDITIONAL_CONTEXT_IDENTIFIER = 1683
AVP_TGPP_SCEF_REALM = 1684
AVP_TGPP_SUBSCRIPTION_DATA_DELETION = 1685
AVP_TGPP_PREFERRED_DATA_MODE = 1686
AVP_TGPP_EMERGENCY_INFO = 1687
AVP_TGPP_V2X_SUBSCRIPTION_DATA = 1688
AVP_TGPP_V2X_PERMISSION = 1689
AVP_TGPP_PDN_CONNECTION_CONTINUITY = 1690
E_PDN_CONNECTION_CONTINUITY_MAINTAIN_PDN_CONNECTION = 0
E_PDN_CONNECTION_CONTINUITY_DISCONNECT_PDN_CONNECTION_WITH_REACTIVATION_REQUEST = 1
E_PDN_CONNECTION_CONTINUITY_DISCONNECT_PDN_CONNECTION_WITHOUT_REACTIVATION_REQUEST = 2

AVP_TGPP_EDRX_CYCLE_LENGTH = 1691
AVP_TGPP_EDRX_CYCLE_LENGTH_VALUE = 1692
AVP_TGPP_UE_PC5_AMBR = 1693
AVP_TGPP_MBSFN_AREA = 1694
AVP_TGPP_MBSFN_AREA_ID = 1695
AVP_TGPP_CARRIER_FREQUENCY = 1696
AVP_TGPP_RDS_INDICATOR = 1697
E_RDS_INDICATOR_DISABLED = 0
E_RDS_INDICATOR_ENABLED = 1

AVP_TGPP_SERVICE_GAP_TIME = 1698
AVP_TGPP_AERIAL_UE_SUBSCRIPTION_INFORMATION = 1699
AVP_TGPP_BROADCAST_LOCATION_ASSISTANCE_DATA_TYPES = 1700
AVP_TGPP_PAGING_TIME_WINDOW = 1701
AVP_TGPP_OPERATION_MODE = 1702
E_OPERATION_MODE_IU_MODE = 1
E_OPERATION_MODE_WB_S1_MODE = 2
E_OPERATION_MODE_NB_S1_MODE = 3

AVP_TGPP_PAGING_TIME_WINDOW_LENGTH = 1703
AVP_TGPP_CORE_NETWORK_RESTRICTIONS = 1704
AVP_TGPP_EDRX_RELATED_RAT = 1705
AVP_TGPP_INTERWORKING_5GS_INDICATOR = 1706
E_INTERWORKING_5GS_INDICATOR_NOT_SUBSCRIBED = 0
E_INTERWORKING_5GS_INDICATOR_SUBSCRIBED = 1

AVP_TGPP_ETHERNET_PDN_TYPE_INDICATOR = 1707
E_ETHERNET_PDN_TYPE_INDICATOR_FALSE = 0
E_ETHERNET_PDN_TYPE_INDICATOR_TRUE = 1

AVP_TGPP_SUBSCRIBED_ARPI = 1708
AVP_TGPP_IAB_OPERATION_PERMISSION = 1709
E_IAB_OPERATION_PERMISSION_IAB_OPERATION_ALLOWED = 0
E_IAB_OPERATION_PERMISSION_IAB_OPERATION_NOTALLOWED = 1

AVP_TGPP_V2X_SUBSCRIPTION_DATA_NR = 1710
AVP_TGPP_UE_PC5_QOS = 1711
AVP_TGPP_PC5_QOS_FLOW = 1712
AVP_TGPP_5QI = 1713
AVP_TGPP_PC5_FLOW_BITRATES = 1714
AVP_TGPP_GUARANTEED_FLOW_BITRATES = 1715
AVP_TGPP_MAXIMUM_FLOW_BITRATES = 1716
AVP_TGPP_PC5_RANGE = 1717
AVP_TGPP_PC5_LINK_AMBR = 1718
AVP_TGPP_SMS_INFORMATION = 2000
AVP_TGPP_DATA_CODING_SCHEME = 2001
AVP_TGPP_DESTINATION_INTERFACE = 2002
AVP_TGPP_INTERFACE_ID = 2003
AVP_TGPP_INTERFACE_PORT = 2004
AVP_TGPP_INTERFACE_TEXT = 2005
AVP_TGPP_INTERFACE_TYPE = 2006
E_INTERFACE_TYPE_UNKNOWN = 0
E_INTERFACE_TYPE_MOBILE_ORIGINATING = 1
E_INTERFACE_TYPE_MOBILE_TERMINATING = 2
E_INTERFACE_TYPE_APPLICATION_ORIGINATING = 3
E_INTERFACE_TYPE_APPLICATION_TERMINATION = 4

AVP_TGPP_SM_MESSAGE_TYPE = 2007
E_SM_MESSAGE_TYPE_SUBMISSION = 0
E_SM_MESSAGE_TYPE_DELIVERY_REPORT = 1
E_SM_MESSAGE_TYPE_SM_SERVICE_REQUEST = 2

AVP_TGPP_ORIGINATOR_SCCP_ADDRESS = 2008
AVP_TGPP_ORIGINATOR_INTERFACE = 2009
AVP_TGPP_RECIPIENT_SCCP_ADDRESS = 2010
AVP_TGPP_REPLY_PATH_REQUESTED = 2011
E_REPLY_PATH_REQUESTED_NO_REPLY_PATH_SET = 0
E_REPLY_PATH_REQUESTED_REPLY_PATH_SET = 1

AVP_TGPP_SM_DISCHARGE_TIME = 2012
AVP_TGPP_SM_PROTOCOL_ID = 2013
AVP_TGPP_SM_STATUS = 2014
AVP_TGPP_SM_USER_DATA_HEADER = 2015
AVP_TGPP_SMS_NODE = 2016
E_SMS_NODE_SMS_ROUTER = 0
E_SMS_NODE_IP_SM_GW = 1
E_SMS_NODE_SMS_ROUTER_AND_IP_SM_GW = 2
E_SMS_NODE_SMS_SC = 3

AVP_TGPP_SMSC_ADDRESS = 2017
AVP_TGPP_CLIENT_ADDRESS = 2018
AVP_TGPP_NUMBER_OF_MESSAGES_SENT = 2019
AVP_TGPP_LOW_BALANCE_INDICATION = 2020
E_LOW_BALANCE_INDICATION_NOT_APPLICABLE = 0
E_LOW_BALANCE_INDICATION_YES = 1

AVP_TGPP_REMAINING_BALANCE = 2021
AVP_TGPP_REFUND_INFORMATION = 2022
AVP_TGPP_CARRIER_SELECT_ROUTING_INFORMATION = 2023
AVP_TGPP_NUMBER_PORTABILITY_ROUTING_INFORMATION = 2024
AVP_TGPP_POC_EVENT_TYPE = 2025
E_POC_EVENT_TYPE_NORMAL = 0
E_POC_EVENT_TYPE_INSTANT_PERSONAL_AALERT_EVENT = 1
E_POC_EVENT_TYPE_POC_GROUP_ADVERTISEMENT_EVENT = 2
E_POC_EVENT_TYPE_EARLY_SSESSION_SETTING_UP_EVENT = 3
E_POC_EVENT_TYPE_POC_TALK_BURST = 4

AVP_TGPP_RECIPIENT_INFO = 2026
AVP_TGPP_ORIGINATOR_RECEIVED_ADDRESS = 2027
AVP_TGPP_RECIPIENT_RECEIVED_ADDRESS = 2028
AVP_TGPP_SM_SERVICE_TYPE = 2029
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_CONTENT_PROCESSING = 0
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_FORWARDING = 1
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_FORWARDING_MULTIPLE_SUBSCRIPTIONS = 2
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_FILTERING = 3
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_RECEIPT = 4
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_NETWORK_STORAGE = 5
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_TO_MULTIPLE_DESTINATIONS = 6
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_VIRTUAL_PRIVATE_NETWORK_VPN = 7
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_AUTO_REPLY = 8
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_PERSONAL_SIGNATURE = 9
E_SM_SERVICE_TYPE_VAS4SMS_SHORT_MESSAGE_DEFERRED_DELIVERY = 10

AVP_TGPP_MMTEL_INFORMATION = 2030
AVP_TGPP_MMTEL_SERVICE_TYPE = 2031
E_MMTEL_SERVICE_TYPE_ORIGINATING_IDENTIFICATION_PRESENTATION_OIP = 0
E_MMTEL_SERVICE_TYPE_ORIGINATING_IDENTIFICATION_RESTRICTION_OIR = 1
E_MMTEL_SERVICE_TYPE_TERMINATING_IDENTIFICATION_PRESENTATION_TIP = 2
E_MMTEL_SERVICE_TYPE_TERMINATING_IDENTIFICATION_RESTRICTION_TIR = 3
E_MMTEL_SERVICE_TYPE_COMMUNICATION_HOLD_HOLD = 4
E_MMTEL_SERVICE_TYPE_COMMUNICATIONS_BARRING_CB = 5
E_MMTEL_SERVICE_TYPE_COMMUNICATION_DIVERSION_CDIV = 6
E_MMTEL_SERVICE_TYPE_COMMUNICATION_DIVERSION_NOTIFICATION_CDIVN = 7
E_MMTEL_SERVICE_TYPE_COMMUNICATION_WAITING_CW = 8
E_MMTEL_SERVICE_TYPE_MESSAGE_WAITING_INDICATION_MWI = 9
E_MMTEL_SERVICE_TYPE_CONFERENCE_CONF = 10
E_MMTEL_SERVICE_TYPE_FLEXIBLE_ALERTING_FA = 11
E_MMTEL_SERVICE_TYPE_COMPLETION_OF_COMMUNICATION_TO_BUSY_SUBSCRIBER_CCBS = 12
E_MMTEL_SERVICE_TYPE_COMPLETION_OF_COMMUNICATIONS_ON_NO_REPLY_CCNR = 13
E_MMTEL_SERVICE_TYPE_MALICIOUS_COMMUNICATION_IDENTIFICATION_MCID = 14
E_MMTEL_SERVICE_TYPE_CUSTOMIZED_ALERTING_TONE_CAT = 15
E_MMTEL_SERVICE_TYPE_CLOSED_USER_GROUP_CUG = 16
E_MMTEL_SERVICE_TYPE_PERSONAL_NETWORK_MANAGEMENT_PNM = 17
E_MMTEL_SERVICE_TYPE_CUSTOMIZED_RINGING_SIGNAL_CRS = 18
E_MMTEL_SERVICE_TYPE_ADVICE_OF_CHARGE_AOC = 19

AVP_TGPP_SERVICE_MODE = 2032
AVP_TGPP_SUBSCRIBER_ROLE = 2033
E_SUBSCRIBER_ROLE_ORIGINATING = 0
E_SUBSCRIBER_ROLE_TERMINATING = 1

AVP_TGPP_NUMBER_OF_DIVERSIONS = 2034
AVP_TGPP_ASSOCIATED_PARTY_ADDRESS = 2035
AVP_TGPP_SDP_TYPE = 2036
E_SDP_TYPE_SDP_OFFER = 0
E_SDP_TYPE_SDP_ANSWER = 1

AVP_TGPP_CHANGE_CONDITION = 2037
E_CHANGE_CONDITION_NORMAL_RELEASE = 0
E_CHANGE_CONDITION_ABNORMAL_RELEASE = 1
E_CHANGE_CONDITION_QOS_CHANGE = 2
E_CHANGE_CONDITION_VOLUME_LIMIT = 3
E_CHANGE_CONDITION_TIME_LIMIT = 4
E_CHANGE_CONDITION_SERVING_NODE_CHANGE = 5
E_CHANGE_CONDITION_SERVING_NODE_PLMN_CHANGE = 6
E_CHANGE_CONDITION_USER_LOCATION_CHANGE = 7
E_CHANGE_CONDITION_RAT_CHANGE = 8
E_CHANGE_CONDITION_UE_TIME_ZONE_CHANGE = 9
E_CHANGE_CONDITION_TARIFF_TIME_CHANGE = 10
E_CHANGE_CONDITION_SERVICE_IDLED_OUT = 11
E_CHANGE_CONDITION_SERVICE_SPECIFIC_TIME_LIMIT = 12
E_CHANGE_CONDITION_MAX_NUMBER_OF_CHANGES_IN_CHANGING_CONDITIONS = 13
E_CHANGE_CONDITION_CGI_SAI_CHANGE = 14
E_CHANGE_CONDITION_RAI_CHANGE = 15
E_CHANGE_CONDITION_ECGI_CHANGE = 16
E_CHANGE_CONDITION_TAI_CHANGE = 17
E_CHANGE_CONDITION_SERVICE_DATA_VOLUME_LIMIT = 18
E_CHANGE_CONDITION_SERVICE_DATA_TIME_LIMIT = 19
E_CHANGE_CONDITION_MANAGEMENT_INTERVANTION = 20
E_CHANGE_CONDITION_SERVICE_STOP = 21
E_CHANGE_CONDITION_USER_CSG_INFORMATION_CHANGE = 22
E_CHANGE_CONDITION_S_GW_CHANGE = 23
E_CHANGE_CONDITION_CHANGE_OF_UE_PRESENCE_IN_PRESENCE_REPORTING_AREA = 24
E_CHANGE_CONDITION_PROXIMITY_ALERTED = 25
E_CHANGE_CONDITION_TIME_EXPIRED_WITH_NO_RENEWAL = 26
E_CHANGE_CONDITION_REQUESTOR_CANCELLATION = 27
E_CHANGE_CONDITION_MAXIMUM_NUMBER_OF_REPORTS = 28
E_CHANGE_CONDITION_PLMN_CHANGE = 29
E_CHANGE_CONDITION_COVERAGE_STATUS_CHANGE = 30
E_CHANGE_CONDITION_REMOVAL_OF_ACCESS = 31
E_CHANGE_CONDITION_UNAVAILABILITY_OF_ACCESS = 32
E_CHANGE_CONDITION_ACCESS_CHANGE_OF_SERVICE_DATA_FLOW = 33
E_CHANGE_CONDITION_INDIRECT_CHANGE_CONDITION = 34
E_CHANGE_CONDITION_MAXIMUM_NUMBER_OF_NIDD_SUBMISSIONS = 35
E_CHANGE_CONDITION_CHANGE_IN_UE_TO_PE = 36
E_CHANGE_CONDITION_SERVING_PLMN_RATE_CONTROL_CHANGE = 37
E_CHANGE_CONDITION_APN_RATE_CONTROL_CHANGE = 38

AVP_TGPP_CHANGE_TIME = 2038
AVP_TGPP_DIAGNOSTICS = 2039
E_DIAGNOSTICS_UNSPECIFIED = 0
E_DIAGNOSTICS_SESSION_TIMEOUT = 1
E_DIAGNOSTICS_RESOURCE_LIMITATION = 2
E_DIAGNOSTICS_ADMIN_DISCONNECT = 3
E_DIAGNOSTICS_IDLE_TIMEOUT = 4
E_DIAGNOSTICS_PCRF_UNREACHABLE = 5
E_DIAGNOSTICS_AAA_UNREACHABLE = 6
E_DIAGNOSTICS_AAA_INITIATED_SESSION_TERMINATION = 7
E_DIAGNOSTICS_REAUTHENTICATION_FAILED = 8
E_DIAGNOSTICS_PCRF_INITIATED_SESSION_TERMINATION = 9
E_DIAGNOSTICS_PCRF_INITIATED_FLOW_TERMINATION = 10
E_DIAGNOSTICS_PCRF_ACCOUNTING_PARAMETERS_CHANGED = 11
E_DIAGNOSTICS_PMIP_INITIATED_SESSION_TERMINATION = 12
E_DIAGNOSTICS_PPP_INITIATED_SESSION_TERMINATION = 13
E_DIAGNOSTICS_GTP_INITIATED_SESSION_TERMINATION = 14
E_DIAGNOSTICS_PMIP_REVOCATION = 15
E_DIAGNOSTICS_HANDOVER_ERROR = 16
E_DIAGNOSTICS_PMIP_LIFETIME_EXPIRED = 17
E_DIAGNOSTICS_REAUTHORIZATION_FAILED = 18

AVP_TGPP_SERVICE_DATA_CONTAINER = 2040
AVP_TGPP_START_TIME = 2041
AVP_TGPP_STOP_TIME = 2042
AVP_TGPP_TIME_FIRST_USAGE = 2043
AVP_TGPP_TIME_LAST_USAGE = 2044
AVP_TGPP_TIME_USAGE = 2045
AVP_TGPP_TRAFFIC_DATA_VOLUMES = 2046
AVP_TGPP_SERVING_NODE_TYPE = 2047
E_SERVING_NODE_TYPE_SGSN = 0
E_SERVING_NODE_TYPE_PMIPSGW = 1
E_SERVING_NODE_TYPE_GTPSGW = 2
E_SERVING_NODE_TYPE_EPDG = 3
E_SERVING_NODE_TYPE_HSGW = 4
E_SERVING_NODE_TYPE_MME = 5

AVP_TGPP_SUPPLEMENTARY_SERVICE = 2048
AVP_TGPP_PARTICIPANT_ACTION_TYPE = 2049
E_PARTICIPANT_ACTION_TYPE_CREATE_CONF = 0
E_PARTICIPANT_ACTION_TYPE_JOIN_CONF = 1
E_PARTICIPANT_ACTION_TYPE_INVITE_INTO_CONF = 2
E_PARTICIPANT_ACTION_TYPE_QUIT_CONF = 3

AVP_TGPP_PDN_CONNECTION_CHARGING_ID = 2050
AVP_TGPP_DYNAMIC_ADDRESS_FLAG = 2051
E_DYNAMIC_ADDRESS_FLAG_STATIC = 0
E_DYNAMIC_ADDRESS_FLAG_DYNAMIC = 1

AVP_TGPP_ACCUMULATED_COST = 2052
AVP_TGPP_AOC_COST_INFORMATION = 2053
AVP_TGPP_AOC_INFORMATION = 2054
AVP_TGPP_AOC_REQUEST_TYPE = 2055
E_AOC_REQUEST_TYPE_AOC_NOT_REQUESTED = 0
E_AOC_REQUEST_TYPE_AOC_FULL = 1
E_AOC_REQUEST_TYPE_AOC_COST_ONLY = 2
E_AOC_REQUEST_TYPE_AOC_TARIFF_ONLY = 3

AVP_TGPP_CURRENT_TARIFF = 2056
AVP_TGPP_NEXT_TARIFF = 2057
AVP_TGPP_RATE_ELEMENT = 2058
AVP_TGPP_SCALE_FACTOR = 2059
AVP_TGPP_TARIFF_INFORMATION = 2060
AVP_TGPP_UNIT_COST = 2061
AVP_TGPP_INCREMENTAL_COST = 2062
AVP_TGPP_LOCAL_SEQUENCE_NUMBER = 2063
AVP_TGPP_NODE_ID = 2064
AVP_TGPP_SGW_CHANGE = 2065
E_SGW_CHANGE_ACR_START_NOT_DUE_TO_SGW_CHANGE = 0
E_SGW_CHANGE_ACR_START_DUE_TO_SGW_CHANGE = 1

AVP_TGPP_CHARGING_CHARACTERISTICS_SELECTION_MODE = 2066
E_CHARGING_CHARACTERISTICS_SELECTION_MODE_SERVING_NODE_SUPPLIED = 0
E_CHARGING_CHARACTERISTICS_SELECTION_MODE_SUBSCRIPTION_SPECIFIC = 1
E_CHARGING_CHARACTERISTICS_SELECTION_MODE_APN_SPECIFIC = 2
E_CHARGING_CHARACTERISTICS_SELECTION_MODE_HOME_DEFAULT = 3
E_CHARGING_CHARACTERISTICS_SELECTION_MODE_ROAMING_DEFAULT = 4
E_CHARGING_CHARACTERISTICS_SELECTION_MODE_VISITING_DEFAULT = 5

AVP_TGPP_SGW_ADDRESS = 2067
AVP_TGPP_DYNAMIC_ADDRESS_FLAG_EXTENSION = 2068
E_DYNAMIC_ADDRESS_FLAG_EXTENSION_STATIC = 0
E_DYNAMIC_ADDRESS_FLAG_EXTENSION_DYNAMIC = 1

AVP_TGPP_APPLICATION_SERVER_ID = 2101
AVP_TGPP_APPLICATION_SERVICE_TYPE = 2102
E_APPLICATION_SERVICE_TYPE_SENDING = 100
E_APPLICATION_SERVICE_TYPE_RECEIVING = 101
E_APPLICATION_SERVICE_TYPE_RETRIEVAL = 102
E_APPLICATION_SERVICE_TYPE_INVITING = 103
E_APPLICATION_SERVICE_TYPE_LEAVING = 104
E_APPLICATION_SERVICE_TYPE_JOINING = 105

AVP_TGPP_APPLICATION_SESSION_ID = 2103
AVP_TGPP_DELIVERY_STATUS = 2104
AVP_TGPP_IM_INFORMATION = 2110
AVP_TGPP_NUMBER_OF_MESSAGES_SUCCESSFULLY_EXPLODED = 2111
AVP_TGPP_NUMBER_OF_MESSAGES_SUCCESSFULLY_SENT = 2112
AVP_TGPP_TOTAL_NUMBER_OF_MESSAGES_EXPLODED = 2113
AVP_TGPP_TOTAL_NUMBER_OF_MESSAGES_SENT = 2114
AVP_TGPP_DCD_INFORMATION = 2115
AVP_TGPP_CONTENT_ID = 2116
AVP_TGPP_CONTENT_PROVIDER_ID = 2117
AVP_TGPP_CHARGE_REASON_CODE = 2118
E_CHARGE_REASON_CODE_UNKNOWN = 0
E_CHARGE_REASON_CODE_USAGE = 1
E_CHARGE_REASON_CODE_COMMUNICATION_ATTEMPT_CHARGE = 2
E_CHARGE_REASON_CODE_SETUP_CHARGE = 3
E_CHARGE_REASON_CODE_ADD_ON_CHARGE = 4

AVP_TGPP_SIP_REQUEST_TIMESTAMP_FRACTION = 2301
AVP_TGPP_SIP_RESPONSE_TIMESTAMP_FRACTION = 2302
AVP_TGPP_ONLINE_CHARGING_FLAG = 2303
E_ONLINE_CHARGING_FLAG_ECF_ADDRESS_NOT_PROVIDED = 0
E_ONLINE_CHARGING_FLAG_ECF_ADDRESS_PROVIDED = 1

AVP_TGPP_CUG_INFORMATION = 2304
AVP_TGPP_REAL_TIME_TARIFF_INFORMATION = 2305
AVP_TGPP_TARIFF_XML = 2306
AVP_TGPP_MBMS_GW_ADDRESS = 2307
AVP_TGPP_IMSI_UNAUTHENTICATED_FLAG = 2308
E_IMSI_UNAUTHENTICATED_FLAG_AUTHENTICATED = 0
E_IMSI_UNAUTHENTICATED_FLAG_UNAUTHENTICATED = 1

AVP_TGPP_ACCOUNT_EXPIRATION = 2309
AVP_TGPP_AOC_FORMAT = 2310
E_AOC_FORMAT_MONETARY = 0
E_AOC_FORMAT_NON_MONETARY = 1
E_AOC_FORMAT_CAI = 2

AVP_TGPP_AOC_SERVICE = 2311
AVP_TGPP_AOC_SERVICE_OBLIGATORY_TYPE = 2312
E_AOC_SERVICE_OBLIGATORY_TYPE_NON_BINDING = 0
E_AOC_SERVICE_OBLIGATORY_TYPE_BINDING = 1

AVP_TGPP_AOC_SERVICE_TYPE = 2313
E_AOC_SERVICE_TYPE_NONE = 0
E_AOC_SERVICE_TYPE_AOC_S = 1
E_AOC_SERVICE_TYPE_AOC_D = 2
E_AOC_SERVICE_TYPE_AOC_E = 3

AVP_TGPP_AOC_SUBSCRIPTION_INFORMATION = 2314
AVP_TGPP_PREFERRED_AOC_CURRENCY = 2315
AVP_TGPP_CSG_ACCESS_MODE = 2317
E_CSG_ACCESS_MODE_CLOSED_MODE = 0
E_CSG_ACCESS_MODE_HYBRID_MODE = 1

AVP_TGPP_CSG_MEMBERSHIP_INDICATION = 2318
E_CSG_MEMBERSHIP_INDICATION_NOT_CSG_MEMBER = 0
E_CSG_MEMBERSHIP_INDICATION_CSG_MEMBER = 1

AVP_TGPP_USER_CSG_INFORMATION = 2319
AVP_TGPP_OUTGOING_SESSION_ID = 2320
AVP_TGPP_INITIAL_IMS_CHARGING_IDENTIFIER = 2321
AVP_TGPP_IMS_EMERGENCY_INDICATOR = 2322
E_IMS_EMERGENCY_INDICATOR_NON_EMERGENCY = 0
E_IMS_EMERGENCY_INDICATOR_EMERGENCY = 1
AVP_TGPP_MBMS_CHARGED_PARTY = 2323
E_MBMS_CHARGED_PARTY_CONTENT_PROVIDER = 0
E_MBMS_CHARGED_PARTY_SUBSCRIBER = 1
AVP_TGPP_LMSI = 2400
AVP_TGPP_SERVING_NODE = 2401
AVP_TGPP_MME_NAME = 2402
AVP_TGPP_MSC_NUMBER = 2403
AVP_TGPP_LCS_CAPABILITIES_SETS = 2404
AVP_TGPP_GMLC_ADDRESS = 2405
AVP_TGPP_ADDITIONAL_SERVING_NODE = 2406
AVP_TGPP_PSR_ADDRESS = 2407
AVP_TGPP_MME_REALM = 2408
AVP_TGPP_SGSN_NAME = 2409
AVP_TGPP_SGSN_REALM = 2410
AVP_TGPP_RIA_FLAGS = 2411
AVP_TGPP_SLG_LOCATION_TYPE = 2500
E_SLG_LOCATION_TYPE_CURRENT_LOCATION = 0
E_SLG_LOCATION_TYPE_CURRENT_OR_LAST_KNOWN_LOCATION = 1
E_SLG_LOCATION_TYPE_INITIAL_LOCATION = 2
E_SLG_LOCATION_TYPE_ACTIVATE_DEFERRED_LOCATION = 3
E_SLG_LOCATION_TYPE_CANCEL_DEFERRED_LOCATION = 4
E_SLG_LOCATION_TYPE_NOTIFICATION_VERIFICATION_ONLY = 5

AVP_TGPP_LCS_EPS_CLIENT_NAME = 2501
AVP_TGPP_LCS_REQUESTOR_NAME = 2502
AVP_TGPP_LCS_PRIORITY = 2503
AVP_TGPP_LCS_QOS = 2504
AVP_TGPP_HORIZONTAL_ACCURACY = 2505
AVP_TGPP_VERTICAL_ACCURACY = 2506
AVP_TGPP_VERTICAL_REQUESTED = 2507
E_VERTICAL_REQUESTED_VERTICAL_COORDINATE_IS_NOT_REQUESTED = 0
E_VERTICAL_REQUESTED_VERTICAL_COORDINATE_IS_REQUESTED = 1

AVP_TGPP_VELOCITY_REQUESTED = 2508
E_VELOCITY_REQUESTED_VELOCITY_IS_NOT_REQUESTED = 0
E_VELOCITY_REQUESTED_BEST_VELOCITY_IS_REQUESTED = 1

AVP_TGPP_RESPONSE_TIME = 2509
E_RESPONSE_TIME_LOW_DELAY = 0
E_RESPONSE_TIME_DELAY_TOLERANT = 1

AVP_TGPP_SUPPORTED_GAD_SHAPES = 2510
AVP_TGPP_LCS_CODEWORD = 2511
AVP_TGPP_LCS_PRIVACY_CHECK = 2512
E_LCS_PRIVACY_CHECK_ALLOWED_WITHOUT_NOTIFICATION = 0
E_LCS_PRIVACY_CHECK_ALLOWED_WITH_NOTIFICATION = 1
E_LCS_PRIVACY_CHECK_ALLOWED_IF_NO_RESPONSE = 2
E_LCS_PRIVACY_CHECK_RESTRICTED_IF_NO_RESPONSE = 3
E_LCS_PRIVACY_CHECK_NOT_ALLOWED = 4

AVP_TGPP_ACCURACY_FULFILMENT_INDICATOR = 2513
E_ACCURACY_FULFILMENT_INDICATOR_REQUESTED_ACCURACY_FULFILLED = 0
E_ACCURACY_FULFILMENT_INDICATOR_REQUESTED_ACCURACY_NOT_FULFILLED = 1

AVP_TGPP_AGE_OF_LOCATION_ESTIMATE = 2514
AVP_TGPP_VELOCITY_ESTIMATE = 2515
AVP_TGPP_EUTRAN_POSITIONING_DATA = 2516
AVP_TGPP_ECGI = 2517
AVP_TGPP_LOCATION_EVENT = 2518
E_LOCATION_EVENT_EMERGENCY_CALL_ORIGINATION = 0
E_LOCATION_EVENT_EMERGENCY_CALL_RELEASE = 1
E_LOCATION_EVENT_MO_LR = 2
E_LOCATION_EVENT_EMERGENCY_CALL_HANDOVER = 3
E_LOCATION_EVENT_DEFERRED_MT_LR_RESPONSE = 4
E_LOCATION_EVENT_DEFERRED_MO_LR_TTTP_INITIATION = 5
E_LOCATION_EVENT_DELAYED_LOCATION_REPORTING = 6

AVP_TGPP_PSEUDONYM_INDICATOR = 2519
E_PSEUDONYM_INDICATOR_PSEUDONYM_NOT_REQUESTED = 0
E_PSEUDONYM_INDICATOR_PSEUDONYM_REQUESTED = 1

AVP_TGPP_LCS_SERVICE_TYPE_ID = 2520
AVP_TGPP_LCS_PRIVACY_CHECK_NON_SESSION = 2521
AVP_TGPP_LCS_PRIVACY_CHECK_SESSION = 2522
AVP_TGPP_LCS_QOS_CLASS = 2523
E_LCS_QOS_CLASS_ASSURED = 0
E_LCS_QOS_CLASS_BEST_EFFORT = 1

AVP_TGPP_GERAN_POSITIONING_INFO = 2524
AVP_TGPP_GERAN_POSITIONING_DATA = 2525
AVP_TGPP_GERAN_GANSS_POSITIONING_DATA = 2526
AVP_TGPP_UTRAN_POSITIONING_INFO = 2527
AVP_TGPP_UTRAN_POSITIONING_DATA = 2528
AVP_TGPP_UTRAN_GANSS_POSITIONING_DATA = 2529
AVP_TGPP_LRR_FLAGS = 2530
AVP_TGPP_LCS_REFERENCE_NUMBER = 2531
AVP_TGPP_DEFERRED_LOCATION_TYPE = 2532
AVP_TGPP_AREA_EVENT_INFO = 2533
AVP_TGPP_AREA_DEFINITION = 2534
AVP_TGPP_AREA = 2535
AVP_TGPP_AREA_TYPE = 2536
E_AREA_TYPE_COUNTRY_CODE = 0
E_AREA_TYPE_PLMN_ID = 1
E_AREA_TYPE_LOCATION_AREA_ID = 2
E_AREA_TYPE_ROUTING_AREA_ID = 3
E_AREA_TYPE_CELL_GLOBAL_ID = 4
E_AREA_TYPE_UTRAN_CELL_ID = 5
E_AREA_TYPE_TRACKING_AREA_ID = 6
E_AREA_TYPE_E_UTRAN_CELL_GLOBAL_ID = 7

AVP_TGPP_AREA_IDENTIFICATION = 2537
AVP_TGPP_OCCURRENCE_INFO = 2538
E_OCCURRENCE_INFO_ONE_TIME_EVENT = 0
E_OCCURRENCE_INFO_MULTIPLE_TIME_EVENT = 1

AVP_TGPP_INTERVAL_TIME = 2539
AVP_TGPP_PERIODIC_LDR_INFORMATION = 2540
AVP_TGPP_REPORTING_AMOUNT = 2541
AVP_TGPP_REPORTING_INTERVAL = 2542
AVP_TGPP_REPORTING_PLMN_LIST = 2543
AVP_TGPP_PLMN_ID_LIST = 2544
AVP_TGPP_PLR_FLAGS = 2545
AVP_TGPP_PLA_FLAGS = 2546
AVP_TGPP_DEFERRED_MT_LR_DATA = 2547
AVP_TGPP_TERMINATION_CAUSE_3GPP = 2548
E_TERMINATION_CAUSE_3GPP_NORMAL = 0
E_TERMINATION_CAUSE_3GPP_ERROR_UNDEFINED = 1
E_TERMINATION_CAUSE_3GPP_INTERNAL_TIMEOUT = 2
E_TERMINATION_CAUSE_3GPP_CONGESTION = 3
E_TERMINATION_CAUSE_3GPP_MT_LR_RESTART = 4
E_TERMINATION_CAUSE_3GPP_PRIVACY_VIOLATION = 5
E_TERMINATION_CAUSE_3GPP_SHAPE_OF_LOCATION_ESTIMATE_NOT_SUPPORTED = 6
E_TERMINATION_CAUSE_3GPP_SUBSCRIBER_TERMINATION = 7
E_TERMINATION_CAUSE_3GPP_UE_TERMINATION = 8
E_TERMINATION_CAUSE_3GPP_NETWORK_TERMINATION = 9

AVP_TGPP_LRA_FLAGS = 2549
AVP_TGPP_PERIODIC_LOCATION_SUPPORT_INDICATOR = 2550
E_PERIODIC_LOCATION_SUPPORT_INDICATOR_NOT_SUPPORTED = 0
E_PERIODIC_LOCATION_SUPPORT_INDICATOR_SUPPORTED = 1

AVP_TGPP_PRIORITIZED_LIST_INDICATOR = 2551
E_PRIORITIZED_LIST_INDICATOR_NOT_PRIORITIZED = 0
E_PRIORITIZED_LIST_INDICATOR_PRIORITIZED = 1

AVP_TGPP_ESMLC_CELL_INFO = 2552
AVP_TGPP_CELL_PORTION_ID = 2553
AVP_TGPP_ONEXRTT_RCID = 2554
AVP_TGPP_DELAYED_LOCATION_REPORTING_DATA = 2555
AVP_TGPP_CIVIC_ADDRESS = 2556
AVP_TGPP_BAROMETRIC_PRESSURE = 2557
AVP_TGPP_UTRAN_ADDITIONAL_POSITIONING_DATA = 2558
AVP_TGPP_MOTION_EVENT_INFO = 2559
AVP_TGPP_LINEAR_DISTANCE = 2560
AVP_TGPP_MAXIMUM_INTERVAL = 2561
AVP_TGPP_SAMPLING_INTERVAL = 2562
AVP_TGPP_REPORTING_DURATION = 2563
AVP_TGPP_REPORTING_LOCATION_REQUIREMENTS = 2564
AVP_TGPP_ADDITIONAL_AREA = 2565
AVP_TGPP_IMS_APPLICATION_REFERENCE_IDENTIFIER = 2601
AVP_TGPP_LOW_PRIORITY_INDICATOR = 2602
E_LOW_PRIORITY_INDICATOR_NO = 0
E_LOW_PRIORITY_INDICATOR_YES = 1

AVP_TGPP_IP_REALM_DEFAULT_INDICATOR = 2603
E_IP_REALM_DEFAULT_INDICATOR_DEFAULT_IP_REALM_NOT_USED = 0
E_IP_REALM_DEFAULT_INDICATOR_DEFAULT_IP_REALM_USED = 1

AVP_TGPP_LOCAL_GW_INSERTED_INDICATOR = 2604
E_LOCAL_GW_INSERTED_INDICATOR_LOCAL_GW_NOT_INSERTED = 0
E_LOCAL_GW_INSERTED_INDICATOR_LOCAL_GW_INSERTED = 1

AVP_TGPP_TRANSCODER_INSERTED_INDICATOR = 2605
E_TRANSCODER_INSERTED_INDICATOR_TRANSCODER_NOT_INSERTED = 0
E_TRANSCODER_INSERTED_INDICATOR_TRANSCODER_INSERTED = 1

AVP_TGPP_PDP_ADDRESS_PREFIX_LENGTH = 2606
AVP_TGPP_TRANSIT_IOI_LIST = 2701
AVP_TGPP_STATUS_AS_CODE = 2702
E_STATUS_AS_CODE_4XX = 0
E_STATUS_AS_CODE_5XX = 1
E_STATUS_AS_CODE_TIMEOUT = 2
AVP_TGPP_NNI_INFORMATION = 2703
AVP_TGPP_NNI_TYPE = 2704
E_NNI_TYPE_NON_ROAMING = 0
E_NNI_TYPE_ROAMING_WITHOUT_LOOPBACK = 1
E_NNI_TYPE_ROAMING_WITH_LOOPBACK = 2

AVP_TGPP_NEIGHBOUR_NODE_ADDRESS = 2705
AVP_TGPP_RELATIONSHIP_MODE = 2706
E_RELATIONSHIP_MODE_TRUSTED = 0
E_RELATIONSHIP_MODE_NON_TRUSTED = 1

AVP_TGPP_SESSION_DIRECTION = 2707
E_SESSION_DIRECTION_INBOUND = 0
E_SESSION_DIRECTION_OUTBOUND = 1

AVP_TGPP_FROM_ADDRESS = 2708
AVP_TGPP_ACCESS_TRANSFER_INFORMATION = 2709
AVP_TGPP_ACCESS_TRANSFER_TYPE = 2710
E_ACCESS_TRANSFER_TYPE_PS_TO_CS_TRANSFER = 0
E_ACCESS_TRANSFER_TYPE_CS_TO_PS_TRANSFER = 1
E_ACCESS_TRANSFER_TYPE_PS_TO_PS_TRANSFER = 2
E_ACCESS_TRANSFER_TYPE_CS_TO_CS_TRANSFER = 3

AVP_TGPP_RELATED_IMS_CHARGING_IDENTIFIER = 2711
AVP_TGPP_RELATED_IMS_CHARGING_IDENTIFIER_NODE = 2712
AVP_TGPP_IMS_VISITED_NETWORK_IDENTIFIER = 2713
AVP_TGPP_TWAN_USER_LOCATION_INFO = 2714
AVP_TGPP_BSSID = 2716
AVP_TGPP_TAD_IDENTIFIER = 2717
E_TAD_IDENTIFIER_CS = 0
E_TAD_IDENTIFIER_PS = 1

AVP_TGPP_TDF_APPLICATION_INSTANCE_IDENTIFIER = 2802
AVP_TGPP_HENB_LOCAL_IP_ADDRESS = 2804
AVP_TGPP_UE_LOCAL_IP_ADDRESS = 2805
AVP_TGPP_UDP_SOURCE_PORT = 2806
AVP_TGPP_CS_SERVICE_QOS_REQUEST_IDENTIFIER = 2807
AVP_TGPP_CS_SERVICE_QOS_REQUEST_OPERATION = 2808
E_CS_SERVICE_QOS_REQUEST_OPERATION_DELETION = 0
E_CS_SERVICE_QOS_REQUEST_OPERATION_MODIFICATION = 1

AVP_TGPP_MUTE_NOTIFICATION = 2809
E_MUTE_NOTIFICATION_MUTE_REQUIRED = 0

AVP_TGPP_MONITORING_TIME = 2810
AVP_TGPP_AN_GW_STATUS = 2811
E_AN_GW_STATUS_AN_GW_FAILED = 0

AVP_TGPP_USER_LOCATION_INFO_TIME = 2812
AVP_TGPP_CS_SERVICE_RESOURCE_REPORT = 2813
AVP_TGPP_CS_SERVICE_RESOURCE_FAILURE_CAUSE = 2814
E_CS_SERVICE_RESOURCE_FAILURE_CAUSE_RESOURCE_RELEASED = 0

AVP_TGPP_CS_SERVICE_RESOURCE_RESULT_OPERATION = 2815
E_CS_SERVICE_RESOURCE_RESULT_OPERATION_DELETION = 0

AVP_TGPP_DEFAULT_QOS_INFORMATION = 2816
AVP_TGPP_DEFAULT_QOS_NAME = 2817
AVP_TGPP_CONDITIONAL_APN_AGGREGATE_MAX_BITRATE = 2818
AVP_TGPP_RAN_NAS_RELEASE_CAUSE = 2819
AVP_TGPP_PRESENCE_REPORTING_AREA_ELEMENTS_LIST = 2820
AVP_TGPP_PRESENCE_REPORTING_AREA_IDENTIFIER = 2821
AVP_TGPP_PRESENCE_REPORTING_AREA_INFORMATION = 2822
AVP_TGPP_PRESENCE_REPORTING_AREA_STATUS = 2823
E_PRESENCE_REPORTING_AREA_STATUS_IN_AREA = 0
E_PRESENCE_REPORTING_AREA_STATUS_OUT_OF_AREA = 1
E_PRESENCE_REPORTING_AREA_STATUS_INACTIVE = 2

AVP_TGPP_NETLOC_ACCESS_SUPPORT = 2824
E_NETLOC_ACCESS_SUPPORT_NETLOC_ACCESS_NOT_SUPPORTED = 0

AVP_TGPP_FIXED_USER_LOCATION_INFO = 2825
AVP_TGPP_PCSCF_RESTORATION_INDICATION = 2826
E_PCSCF_RESTORATION_INDICATION_PCSCF_RESTORATION = 0

AVP_TGPP_IP_CAN_SESSION_CHARGING_SCOPE = 2827
E_IP_CAN_SESSION_CHARGING_SCOPE_IP_CAN_SESSION_SCOPE = 0

AVP_TGPP_MONITORING_FLAGS = 2828
AVP_TGPP_DEFAULT_ACCESS = 2829
E_DEFAULT_ACCESS_3GPP_GPRS = 0
E_DEFAULT_ACCESS_DOCSIS = 1
E_DEFAULT_ACCESS_XDSL = 2
E_DEFAULT_ACCESS_WIMAX = 3
E_DEFAULT_ACCESS_3GPP2 = 4
E_DEFAULT_ACCESS_3GPP_EPS = 5
E_DEFAULT_ACCESS_NON_3GPP_EPS = 6

AVP_TGPP_NBIFOM_MODE = 2830
E_NBIFOM_MODE_UE_INITIATED = 0
E_NBIFOM_MODE_NETWORK_INITIATED = 1

AVP_TGPP_NBIFOM_SUPPORT = 2831
E_NBIFOM_SUPPORT_NBIFOM_NOT_SUPPORTED = 0
E_NBIFOM_SUPPORT_NBIFOM_SUPPORTED = 1

AVP_TGPP_RAN_RULE_SUPPORT = 2832
AVP_TGPP_ACCESS_AVAILABILITY_CHANGE_REASON = 2833
AVP_TGPP_ROUTING_RULE_FAILURE_CODE = 2834
AVP_TGPP_ROUTING_RULE_REPORT = 2835
AVP_TGPP_TRAFFIC_STEERING_POLICY_IDENTIFIER_DL = 2836
AVP_TGPP_TRAFFIC_STEERING_POLICY_IDENTIFIER_UL = 2837
AVP_TGPP_REQUEST_TYPE = 2838
AVP_TGPP_EXECUTION_TIME = 2839
AVP_TGPP_CONDITIONAL_POLICY_INFORMATION = 2840
AVP_TGPP_RESOURCE_RELEASE_NOTIFICATION = 2841
E_RESOURCE_RELEASE_NOTIFICATION_ENABLE_NOTIFICATION = 0

AVP_TGPP_REMOVAL_OF_ACCESS = 2842
E_REMOVAL_OF_ACCESS_REMOVAL_OF_ACCESS = 0

AVP_TGPP_TCP_SOURCE_PORT = 2843
AVP_TGPP_DEFAULT_BEARER_INDICATION = 2844
E_DEFAULT_BEARER_INDICATION_BIND_TO_DEF_BEARER = 0
E_DEFAULT_BEARER_INDICATION_BIND_TO_APPLICABLE_BEARER = 1

AVP_TGPP_PRA_INSTALL = 2845
AVP_TGPP_PRA_REMOVE = 2846
AVP_TGPP_3GPP_PS_DATA_OFF_STATUS_GX = 2847
E_3GPP_PS_DATA_OFF_STATUS_GX_ACTIVE = 0
E_3GPP_PS_DATA_OFF_STATUS_GX_INACTIVE = 1

AVP_TGPP_EXTENDED_APN_AMBR_DL = 2848
AVP_TGPP_EXTENDED_APN_AMBR_UL = 2849
AVP_TGPP_EXTENDED_GBR_DL = 2850
AVP_TGPP_EXTENDED_GBR_UL = 2851
AVP_TGPP_MAX_PLR_DL = 2852
AVP_TGPP_MAX_PLR_UL = 2853
AVP_TGPP_UE_STATUS = 2854
AVP_TGPP_PRESENCE_REPORTING_AREA_NODE = 2855
E_PRESENCE_REPORTING_AREA_NODE_OCS = 0
E_PRESENCE_REPORTING_AREA_NODE_PCRF = 1

AVP_TGPP_POLICY_COUNTER_IDENTIFIER = 2901
AVP_TGPP_POLICY_COUNTER_STATUS = 2902
AVP_TGPP_POLICY_COUNTER_STATUS_REPORT = 2903
AVP_TGPP_SL_REQUEST_TYPE = 2904
E_SL_REQUEST_TYPE_INITIAL_REQUEST = 0
E_SL_REQUEST_TYPE_INTERMEDIATE_REQUEST = 1

AVP_TGPP_PENDING_POLICY_COUNTER_INFORMATION = 2905
AVP_TGPP_PENDING_POLICY_COUNTER_CHANGE_TIME = 2906
AVP_TGPP_DEVICE_ACTION = 3001
AVP_TGPP_DEVICE_NOTIFICATION = 3002
AVP_TGPP_TRIGGER_DATA = 3003
AVP_TGPP_PAYLOAD = 3004
AVP_TGPP_ACTION_TYPE = 3005
E_ACTION_TYPE_DEVICE_TRIGGER_REQUEST = 1
E_ACTION_TYPE_DELIVERY_REPORT = 2

AVP_TGPP_PRIORITY_INDICATION = 3006
E_PRIORITY_INDICATION_NON_PRIORITY = 0
E_PRIORITY_INDICATION_PRIORITY = 1

AVP_TGPP_REFERENCE_NUMBER = 3007
AVP_TGPP_REQUEST_STATUS = 3008
E_REQUEST_STATUS_SUCCESS = 0
E_REQUEST_STATUS_TEMPORARYERROR = 201
E_REQUEST_STATUS_INVPAYLOAD = 101
E_REQUEST_STATUS_INVEXTID = 102
E_REQUEST_STATUS_INVSCSID = 103
E_REQUEST_STATUS_INVPERIOD = 104
E_REQUEST_STATUS_NOTAUTHORIZED = 105
E_REQUEST_STATUS_SERVICEUNAVAILABLE = 106
E_REQUEST_STATUS_PERMANENTERROR = 107
E_REQUEST_STATUS_QUOTAEXCEEDED = 108
E_REQUEST_STATUS_RATEEXCEEDED = 109

AVP_TGPP_DELIVERY_OUTCOME = 3009
E_DELIVERY_OUTCOME_SUCCESS = 0
E_DELIVERY_OUTCOME_EXPIRED = 1
E_DELIVERY_OUTCOME_TEMPORARYERROR = 2
E_DELIVERY_OUTCOME_UNDELIVERABLE = 3

AVP_TGPP_APPLICATION_PORT_IDENTIFIER = 3010
AVP_TGPP_OLD_REFERENCE_NUMBER = 3011
AVP_TGPP_IP_SM_GW_NUMBER = 3100
AVP_TGPP_IP_SM_GW_NAME = 3101
AVP_TGPP_USER_IDENTIFIER = 3102
AVP_TGPP_S6_SERVICE_ID = 3103
E_S6_SERVICE_ID_DEVICE_TRIGGER = 0
E_S6_SERVICE_ID_SMS_MO = 1

AVP_TGPP_SCS_IDENTITY = 3104
AVP_TGPP_SERVICE_PARAMETERS = 3105
AVP_TGPP_T4_PARAMETERS = 3106
AVP_TGPP_SERVICE_DATA = 3107
AVP_TGPP_T4_DATA = 3108
AVP_TGPP_HSS_CAUSE = 3109
AVP_TGPP_SIR_FLAGS = 3110
AVP_TGPP_EXTERNAL_IDENTIFIER = 3111
AVP_TGPP_IP_SM_GW_REALM = 3112
AVP_TGPP_AESE_COMMUNICATION_PATTERN = 3113
AVP_TGPP_COMMUNICATION_PATTERN_SET = 3114
AVP_TGPP_PERIODIC_COMMUNICATION_INDICATOR = 3115
AVP_TGPP_COMMUNICATION_DURATION_TIME = 3116
AVP_TGPP_PERIODIC_TIME = 3117
AVP_TGPP_SCHEDULED_COMMUNICATION_TIME = 3118
AVP_TGPP_STATIONARY_INDICATION = 3119
AVP_TGPP_AESE_COMMUNICATION_PATTERN_CONFIG_STATUS = 3120
AVP_TGPP_AESE_ERROR_REPORT = 3121
AVP_TGPP_MONITORING_EVENT_CONFIGURATION = 3122
AVP_TGPP_MONITORING_EVENT_REPORT = 3123
AVP_TGPP_SCEF_REFERENCE_ID = 3124
AVP_TGPP_SCEF_ID = 3125
AVP_TGPP_SCEF_REFERENCE_ID_FOR_DELETION = 3126
AVP_TGPP_MONITORING_TYPE = 3127
AVP_TGPP_MAXIMUM_NUMBER_OF_REPORTS = 3128
AVP_TGPP_UE_REACHABILITY_CONFIGURATION = 3129
AVP_TGPP_MONITORING_DURATION = 3130
AVP_TGPP_MAXIMUM_DETECTION_TIME = 3131
AVP_TGPP_REACHABILITY_TYPE = 3132
AVP_TGPP_MAXIMUM_LATENCY = 3133
AVP_TGPP_MAXIMUM_RESPONSE_TIME = 3134
AVP_TGPP_LOCATION_INFORMATION_CONFIGURATION = 3135
AVP_TGPP_MONTE_LOCATION_TYPE = 3136
AVP_TGPP_ACCURACY = 3137
AVP_TGPP_ASSOCIATION_TYPE = 3138
AVP_TGPP_ROAMING_INFORMATION = 3139
AVP_TGPP_REACHABILITY_INFORMATION = 3140
AVP_TGPP_IMEI_CHANGE = 3141
AVP_TGPP_MONITORING_EVENT_CONFIG_STATUS = 3142
AVP_TGPP_SUPPORTED_SERVICES = 3143
AVP_TGPP_SUPPORTED_MONITORING_EVENTS = 3144
AVP_TGPP_CIR_FLAGS = 3145
AVP_TGPP_SERVICE_RESULT = 3146
AVP_TGPP_SERVICE_RESULT_CODE = 3147
AVP_TGPP_REFERENCE_ID_VALIDITY_TIME = 3148
AVP_TGPP_EVENT_HANDLING = 3149
AVP_TGPP_NIDD_AUTHORIZATION_REQUEST = 3150
AVP_TGPP_NIDD_AUTHORIZATION_RESPONSE = 3151
AVP_TGPP_SERVICE_REPORT = 3152
AVP_TGPP_NODE_TYPE = 3153
AVP_TGPP_S6T_HSS_CAUSE = 3154
AVP_TGPP_ENHANCED_COVERAGE_RESTRICTION = 3155
AVP_TGPP_ENHANCED_COVERAGE_RESTRICTION_DATA = 3156
AVP_TGPP_RESTRICTED_PLMN_LIST = 3157
AVP_TGPP_ALLOWED_PLMN_LIST = 3158
AVP_TGPP_REQUESTED_VALIDITY_TIME = 3159
AVP_TGPP_GRANTED_VALIDITY_TIME = 3160
AVP_TGPP_NIDD_AUTHORIZATION_UPDATE = 3161
AVP_TGPP_LOSS_OF_CONNECTIVITY_REASON = 3162
AVP_TGPP_GROUP_REPORTING_GUARD_TIMER = 3163
AVP_TGPP_CIA_FLAGS = 3164
AVP_TGPP_GROUP_MONITORING_EVENT_REPORT = 3165
AVP_TGPP_GROUP_MONITORING_EVENT_REPORT_ITEM = 3166
AVP_TGPP_RIR_FLAGS = 3167
AVP_TGPP_TYPE_OF_EXTERNAL_IDENTIFIER = 3168
AVP_TGPP_APN_VALIDITY_TIME = 3169
AVP_TGPP_SUGGESTED_NETWORK_CONFIGURATION = 3170
AVP_TGPP_MONITORING_EVENT_REPORT_STATUS = 3171
AVP_TGPP_PLMN_ID_REQUESTED = 3172
E_PLMN_ID_REQUESTED_TRUE = 0
E_PLMN_ID_REQUESTED_FALSE = 1

AVP_TGPP_ADDITIONALIDENTIFIERS = 3173
AVP_TGPP_NIR_FLAGS = 3174
AVP_TGPP_REPORTING_TIME_STAMP = 3175
AVP_TGPP_NIA_FLAGS = 3176
AVP_TGPP_GROUP_USER_IDENTIFIER = 3177
AVP_TGPP_MTC_PROVIDER_INFO = 3178
AVP_TGPP_MTC_PROVIDER_ID = 3179
AVP_TGPP_PDN_CONNECTIVITY_STATUS_CONFIGURATION = 3180
AVP_TGPP_PDN_CONNECTIVITY_STATUS_REPORT = 3181
AVP_TGPP_PDN_CONNECTIVITY_STATUS_TYPE = 3182
AVP_TGPP_TRAFFIC_PROFILE = 3183
AVP_TGPP_SM_DELIVERY_OUTCOME_T4 = 3200
E_SM_DELIVERY_OUTCOME_T4_ABSENT_SUBSCRIBER = 0
E_SM_DELIVERY_OUTCOME_T4_UE_MEMORTY_CAPACITY_EXCEEDED = 1
E_SM_DELIVERY_OUTCOME_T4_SUCCESSFUL_TRANSFER = 2

AVP_TGPP_ABSENT_SUBSCRIBER_DIAGNOSTIC_T4 = 3201
E_ABSENT_SUBSCRIBER_DIAGNOSTIC_T4_NO_PAGING_RESPONSE = 0
E_ABSENT_SUBSCRIBER_DIAGNOSTIC_T4_UE_DETACHED = 1
E_ABSENT_SUBSCRIBER_DIAGNOSTIC_T4_UE_DEREGISTERED = 2
E_ABSENT_SUBSCRIBER_DIAGNOSTIC_T4_UE_PURGED = 3
E_ABSENT_SUBSCRIBER_DIAGNOSTIC_T4_ROAMING_RESTRICTION = 4
E_ABSENT_SUBSCRIBER_DIAGNOSTIC_T4_UNIDENTIFIED_SUBSCRIBER = 5

AVP_TGPP_TRIGGER_ACTION = 3202
AVP_TGPP_MTC_ERROR_DIAGNOSTIC = 3203
AVP_TGPP_REASON_HEADER = 3401
AVP_TGPP_INSTANCE_ID = 3402
AVP_TGPP_ROUTE_HEADER_RECEIVED = 3403
AVP_TGPP_ROUTE_HEADER_TRANSMITTED = 3404
AVP_TGPP_SM_DEVICE_TRIGGER_INFORMATION = 3405
AVP_TGPP_MTC_IWF_ADDRESS = 3406
AVP_TGPP_SM_DEVICE_TRIGGER_INDICATOR = 3407
E_SM_DEVICE_TRIGGER_INDICATOR_NOT_DEVICETRIGGER = 0
E_SM_DEVICE_TRIGGER_INDICATOR_DEVICETRIGGER = 1

AVP_TGPP_SM_SEQUENCE_NUMBER = 3408
AVP_TGPP_SMS_RESULT = 3409
AVP_TGPP_VCS_INFORMATION = 3410
AVP_TGPP_BASIC_SERVICE_CODE = 3411
AVP_TGPP_BEARER_CAPABILITY = 3412
AVP_TGPP_TELESERVICE = 3413
AVP_TGPP_ISUP_LOCATION_NUMBER = 3414
AVP_TGPP_FORWARDING_PENDING = 3415
E_FORWARDING_PENDING_FORWARDING_NOT_PENDING = 0
E_FORWARDING_PENDING_FORWARDING_PENDING = 1

AVP_TGPP_ISUP_CAUSE = 3416
AVP_TGPP_ISUP_RELEASE_CAUSE = 3416  # format used in older 3gpp releases
AVP_TGPP_MSC_ADDRESS = 3417
AVP_TGPP_NETWORK_CALL_REFERENCE_NUMBER = 3418
AVP_TGPP_START_OF_CHARGING = 3419
AVP_TGPP_VLR_NUMBER = 3420
AVP_TGPP_CN_OPERATOR_SELECTION_ENTITY = 3421
E_CN_OPERATOR_SELECTION_ENTITY_THE_SERVING_NETWORK_HAS_BEEN_SELECTED_BY_THE_UE = 0
E_CN_OPERATOR_SELECTION_ENTITY_THE_SERVING_NETWORK_HAS_BEEN_SELECTED_BY_THE_NETWORK = 1

AVP_TGPP_ISUP_CAUSE_DIAGNOSTICS = 3422
AVP_TGPP_ISUP_CAUSE_LOCATION = 3423
AVP_TGPP_ISUP_CAUSE_VALUE = 3424
AVP_TGPP_EPDG_ADDRESS = 3425
AVP_TGPP_ANNOUNCING_UE_HPLMN_IDENTIFIER = 3426
AVP_TGPP_ANNOUNCING_UE_VPLMN_IDENTIFIER = 3427
AVP_TGPP_COVERAGE_STATUS = 3428
E_COVERAGE_STATUS_OUT_OF_COVERAGE = 0
E_COVERAGE_STATUS_IN_COVERAGE = 1

AVP_TGPP_LAYER_2_GROUP_ID = 3429
AVP_TGPP_MONITORED_PLMN_IDENTIFIER = 3430
AVP_TGPP_MONITORING_UE_HPLMN_IDENTIFIER = 3431
AVP_TGPP_MONITORING_UE_IDENTIFIER = 3432
AVP_TGPP_MONITORING_UE_VPLMN_IDENTIFIER = 3433
AVP_TGPP_PC3_CONTROL_PROTOCOL_CAUSE = 3434
AVP_TGPP_PC3_EPC_CONTROL_PROTOCOL_CAUSE = 3435
AVP_TGPP_REQUESTED_PLMN_IDENTIFIER = 3436
AVP_TGPP_REQUESTOR_PLMN_IDENTIFIER = 3437
AVP_TGPP_ROLE_OF_PROSE_FUNCTION = 3438
E_ROLE_OF_PROSE_FUNCTION_LOCAL_PLMN = 0

AVP_TGPP_USAGE_INFORMATION_REPORT_SEQUENCE_NUMBER = 3439
AVP_TGPP_PROSE_3RD_PARTY_APPLICATION_ID = 3440
AVP_TGPP_PROSE_DIRECT_COMMUNICATION_TRANSMISSION_DATA_CONTAINER = 3441
AVP_TGPP_PROSE_DIRECT_DISCOVERY_MODEL = 3442
E_PROSE_DIRECT_DISCOVERY_MODEL_MODEL_A = 0
E_PROSE_DIRECT_DISCOVERY_MODEL_MODEL_B = 1

AVP_TGPP_PROSE_EVENT_TYPE = 3443
E_PROSE_EVENT_TYPE_ANNOUNCING = 0
E_PROSE_EVENT_TYPE_MONITORING = 1
E_PROSE_EVENT_TYPE_MATCH_REPORT = 2

AVP_TGPP_PROSE_FUNCTION_IP_ADDRESS = 3444
AVP_TGPP_PROSE_GROUP_IP_MULTICAST_ADDRESS = 3446
AVP_TGPP_PROSE_INFORMATION = 3447
AVP_TGPP_PROSE_RANGE_CLASS = 3448
E_PROSE_RANGE_CLASS_RESERVED = 0
E_PROSE_RANGE_CLASS_50M = 1
E_PROSE_RANGE_CLASS_100M = 2
E_PROSE_RANGE_CLASS_200M = 3
E_PROSE_RANGE_CLASS_500M = 4
E_PROSE_RANGE_CLASS_1000M = 5

AVP_TGPP_PROSE_REASON_FOR_CANCELLATION = 3449
E_PROSE_REASON_FOR_CANCELLATION_PROXIMITY_ALERT_SENT = 0
E_PROSE_REASON_FOR_CANCELLATION_TIME_EXPIRED_WITH_NO_RENEWAL = 1
E_PROSE_REASON_FOR_CANCELLATION_REQUESTOR_CANCELLATION = 2

AVP_TGPP_PROSE_REQUEST_TIMESTAMP = 3450
AVP_TGPP_PROSE_ROLE_OF_UE = 3451
E_PROSE_ROLE_OF_UE_ANNOUNCING_UE = 0
E_PROSE_ROLE_OF_UE_MONITORING_UE = 1
E_PROSE_ROLE_OF_UE_REQUESTOR_UE = 2
E_PROSE_ROLE_OF_UE_REQUESTED_UE = 3

AVP_TGPP_PROSE_SOURCE_IP_ADDRESS = 3452
AVP_TGPP_PROSE_UE_ID = 3453
AVP_TGPP_PROXIMITY_ALERT_INDICATION = 3454
E_PROXIMITY_ALERT_INDICATION_ALERT = 0
E_PROXIMITY_ALERT_INDICATION_NO_ALERT = 1

AVP_TGPP_PROXIMITY_ALERT_TIMESTAMP = 3455
# these two AVPs have the same value in 3GPP TS 32.299 version 16.2.0
AVP_TGPP_PROXIMITY_CANCELLATION_TIMESTAMP = 3456
AVP_TGPP_TIME_FIRST_RECEPTION = 3456
AVP_TGPP_PROSE_FUNCTION_PLMN_IDENTIFIER = 3457
AVP_TGPP_APPLICATION_SPECIFIC_DATA = 3458
AVP_TGPP_COVERAGE_INFO = 3459
AVP_TGPP_LOCATION_INFO = 3460
AVP_TGPP_PROSE_DIRECT_COMMUNICATION_RECEPTION_DATA_CONTAINER = 3461
AVP_TGPP_RADIO_PARAMETER_SET_INFO = 3463
AVP_TGPP_RADIO_PARAMETER_SET_VALUES = 3464
AVP_TGPP_RADIO_RESOURCES_INDICATOR = 3465
AVP_TGPP_TIME_FIRST_TRANSMISSION = 3467
AVP_TGPP_TRANSMITTER_INFO = 3468
AVP_TGPP_ORIGIN_APP_LAYER_USER_ID = 3600
AVP_TGPP_TARGET_APP_LAYER_USER_ID = 3601
AVP_TGPP_PROSE_FUNCTION_ID = 3602
AVP_TGPP_PROSE_REQUEST_TYPE = 3603
AVP_TGPP_PDUID = 3604
AVP_TGPP_APPLICATION_DATA = 3605
AVP_TGPP_ALLOWED_SUFFIXES_NUMBER = 3606
AVP_TGPP_MONITOR_TARGET = 3607
AVP_TGPP_PROSE_CODE_SUFFIX_MASK = 3608
AVP_TGPP_SUFFIX_CODE = 3609
AVP_TGPP_SUFFIX_MASK = 3610
AVP_TGPP_BANNED_USER_TARGET = 3611
AVP_TGPP_METADATA_INDICATOR = 3612
AVP_TGPP_APP_LAYER_USER_ID = 3801
AVP_TGPP_ASSISTANCE_INFO = 3802
AVP_TGPP_ASSISTANCE_INFO_VALIDITY_TIMER = 3803
AVP_TGPP_DISCOVERY_TYPE = 3804
AVP_TGPP_FILTER_ID = 3805
AVP_TGPP_MAC_ADDRESS = 3806
AVP_TGPP_MATCH_REPORT = 3807
AVP_TGPP_OPERATING_CHANNEL = 3808
AVP_TGPP_P2P_FEATURES = 3809
AVP_TGPP_PROSE_APP_CODE = 3810
AVP_TGPP_PROSE_APP_ID = 3811
AVP_TGPP_PROSE_APP_MASK = 3812
AVP_TGPP_PROSE_DISCOVERY_FILTER = 3813
AVP_TGPP_PRR_FLAGS = 3814
AVP_TGPP_PROSE_VALIDITY_TIMER = 3815
AVP_TGPP_REQUESTING_EPUID = 3816
AVP_TGPP_TARGETED_EPUID = 3817
AVP_TGPP_TIME_WINDOW = 3818
AVP_TGPP_WIFI_P2P_ASSISTANCE_INFO = 3819
AVP_TGPP_WLAN_ASSISTANCE_INFO = 3820
AVP_TGPP_WLAN_LINK_LAYER_ID = 3821
AVP_TGPP_WLAN_LINK_LAYER_ID_LIST = 3822
AVP_TGPP_LOCATION_UPDATE_TRIGGER = 3823
AVP_TGPP_LOCATION_UPDATE_EVENT_TYPE = 3824
AVP_TGPP_CHANGE_OF_AREA_TYPE = 3825
AVP_TGPP_LOCATION_UPDATE_EVENT_TRIGGER = 3826
AVP_TGPP_REPORT_CARDINALITY = 3827
E_REPORT_CARDINALITY_SINGLE = 0
E_REPORT_CARDINALITY_MULTIPLE = 1

AVP_TGPP_MINIMUM_INTERVAL_TIME = 3828
AVP_TGPP_PERIODIC_LOCATION_TYPE = 3829
AVP_TGPP_LOCATION_REPORT_INTERVAL_TIME = 3830
AVP_TGPP_TOTAL_NUMBER_OF_REPORTS = 3831
AVP_TGPP_VALIDITY_TIME_ANNOUNCE = 3832
AVP_TGPP_VALIDITY_TIME_MONITOR = 3833
AVP_TGPP_VALIDITY_TIME_COMMUNICATION = 3834
AVP_TGPP_PROSE_APP_CODE_INFO = 3835
AVP_TGPP_MIC = 3836
AVP_TGPP_UTC_BASED_COUNTER = 3837
AVP_TGPP_PROSE_MATCH_REFRESH_TIMER = 3838
AVP_TGPP_PROSE_METADATA_INDEX_MASK = 3839
AVP_TGPP_APP_IDENTIFIER = 3840
AVP_TGPP_OS_ID = 3841
AVP_TGPP_OS_APP_ID = 3842
AVP_TGPP_REQUESTING_RPAUID = 3843
AVP_TGPP_TARGET_RPAUID = 3844
AVP_TGPP_TARGET_PDUID = 3845
AVP_TGPP_PROSE_RESTRICTED_CODE = 3846
AVP_TGPP_PROSE_RESTRICTED_CODE_SUFFIXRANGE = 3847
AVP_TGPP_BEGINNING_SUFFIX = 3848
AVP_TGPP_ENDING_SUFFIX = 3849
AVP_TGPP_DISCOVERY_ENTRY_ID = 3850
AVP_TGPP_MATCH_TIMESTAMP = 3851
AVP_TGPP_PMR_FLAGS = 3852
AVP_TGPP_PROSE_APPLICATION_METADATA = 3853
AVP_TGPP_DISCOVERY_AUTH_REQUEST = 3854
AVP_TGPP_DISCOVERY_AUTH_RESPONSE = 3855
AVP_TGPP_MATCH_REQUEST = 3856
AVP_TGPP_MATCH_REPORT_INFO = 3857
AVP_TGPP_BANNED_RPAUID = 3858
AVP_TGPP_BANNED_PDUID = 3859
AVP_TGPP_CODE_RECEIVING_SECURITYMATERIAL = 3860
AVP_TGPP_DUSK = 3862
AVP_TGPP_DUIK = 3863
AVP_TGPP_DUCK = 3864
AVP_TGPP_MIC_CHECK_INDICATOR = 3865
AVP_TGPP_ENCRYPTED_BITMASK = 3866
AVP_TGPP_PROSE_APP_CODE_SUFFIX_RANGE = 3867
AVP_TGPP_PC5_TECH = 3868
AVP_TGPP_ENHANCED_DIAGNOSTICS = 3901
AVP_TGPP_INTER_UE_TRANSFER = 3902
E_INTER_UE_TRANSFER_INTRA_UE_TRANSFER = 0
E_INTER_UE_TRANSFER_INTER_UE_TRANSFER = 1

AVP_TGPP_TWAG_ADDRESS = 3903
AVP_TGPP_ANNOUNCEMENT_INFORMATION = 3904
AVP_TGPP_ANNOUNCEMENT_IDENTIFIER = 3905
AVP_TGPP_ANNOUNCEMENT_ORDER = 3906
AVP_TGPP_VARIABLE_PART = 3907
AVP_TGPP_VARIABLE_PART_ORDER = 3908
AVP_TGPP_VARIABLE_PART_TYPE = 3909
AVP_TGPP_VARIABLE_PART_VALUE = 3910
AVP_TGPP_TIME_INDICATOR = 3911
AVP_TGPP_QUOTA_INDICATOR = 3912
E_QUOTA_INDICATOR_QUOTA_IS_NOT_USED_DURING_PLAYBACK = 0
E_QUOTA_INDICATOR_QUOTA_IS_USED_DURING_PLAYBACK = 1

AVP_TGPP_PLAY_ALTERNATIVE = 3913
E_PLAY_ALTERNATIVE_SERVED_PARTY = 0
E_PLAY_ALTERNATIVE_REMOTE_PARTY = 1

AVP_TGPP_LANGUAGE = 3914
AVP_TGPP_PRIVACY_INDICATOR = 3915
E_PRIVACY_INDICATOR_NOT_PRIVATE = 0
E_PRIVACY_INDICATOR_PRIVATE = 1

AVP_TGPP_CALLED_IDENTITY = 3916
AVP_TGPP_CALLED_IDENTITY_CHANGE = 3917
AVP_TGPP_UWAN_USER_LOCATION_INFO = 3918
AVP_TGPP_MONITORING_EVENT_CONFIGURATION_ACTIVITY = 3919
AVP_TGPP_MONITORING_EVENT_REPORT_DATA = 3920
AVP_TGPP_MONITORING_EVENT_INFORMATION = 3921
AVP_TGPP_MONITORING_EVENT_FUNCTIONALITY = 3922
AVP_TGPP_MONITORING_EVENT_REPORT_NUMBER = 3923
AVP_TGPP_CELLULAR_NETWORK_INFORMATION = 3924
AVP_TGPP_RELATED_CHANGE_CONDITION_INFORMATION = 3925
AVP_TGPP_RELATED_TRIGGER = 3926
AVP_TGPP_CPDT_INFORMATION = 3927
AVP_TGPP_NIDD_SUBMISSION = 3928
AVP_TGPP_SERVING_NODE_IDENTITY = 3929
AVP_TGPP_CP_CIOT_EPS_OPTIMISATION_INDICATOR = 3930
E_CP_CIOT_EPS_OPTIMISATION_INDICATOR_NOT_APPLY = 0
E_CP_CIOT_EPS_OPTIMISATION_INDICATOR_APPLY = 1

AVP_TGPP_SGI_PTP_TUNNELLING_METHOD = 3931
E_SGI_PTP_TUNNELLING_METHOD_UDP_IP_BASED = 0
E_SGI_PTP_TUNNELLING_METHOD_OTHERS = 1

AVP_TGPP_UNI_PDU_CP_ONLY_FLAG = 3932
E_UNI_PDU_CP_ONLY_FLAG_UNI_PDU_BOTH_UP_CP = 0
E_UNI_PDU_CP_ONLY_FLAG_UNI_PDU_CP_ONLY = 1

AVP_TGPP_APN_RATE_CONTROL = 3933
AVP_TGPP_APN_RATE_CONTROL_DOWNLINK = 3934
AVP_TGPP_APN_RATE_CONTROL_UPLINK = 3935
AVP_TGPP_ADDITIONAL_EXCEPTION_REPORTS = 3936
E_ADDITIONAL_EXCEPTION_REPORTS_NOT_ALLOWED = 0
E_ADDITIONAL_EXCEPTION_REPORTS_ALLOWED = 1

AVP_TGPP_RATE_CONTROL_MAX_MESSAGE_SIZE = 3937
AVP_TGPP_RATE_CONTROL_MAX_RATE = 3938
AVP_TGPP_RATE_CONTROL_TIME_UNIT = 3939
AVP_TGPP_SCS_AS_ADDRESS = 3940
AVP_TGPP_SCS_ADDRESS = 3941
AVP_TGPP_SCS_REALM = 3942
AVP_TGPP_ENODEB_ID = 4008
AVP_TGPP_EXTENDED_ENODEB_ID = 4013
AVP_TGPP_CHARGING_PER_IP_CAN_SESSION_INDICATOR = 4400
E_CHARGING_PER_IP_CAN_SESSION_INDICATOR_INACTIVE = 0
E_CHARGING_PER_IP_CAN_SESSION_INDICATOR_ACTIVE = 1

AVP_TGPP_ACCESS_NETWORK_INFO_CHANGE = 4401
AVP_TGPP_DISCOVEREE_UE_HPLMN_IDENTIFIER = 4402
AVP_TGPP_DISCOVEREE_UE_VPLMN_IDENTIFIER = 4403
AVP_TGPP_DISCOVERER_UE_HPLMN_IDENTIFIER = 4404
AVP_TGPP_DISCOVERER_UE_VPLMN_IDENTIFIER = 4405
AVP_TGPP_3GPP_PS_DATA_OFF_STATUS = 4406
E_3GPP_PS_DATA_OFF_STATUS_ACTIVE = 0
E_3GPP_PS_DATA_OFF_STATUS_INACTIVE = 1

AVP_TGPP_UNUSED_QUOTA_TIMER = 4407
AVP_TGPP_ANNOUNCING_PLMN_ID = 4408
AVP_TGPP_PROSE_UE_TO_NETWORK_RELAY_UE_ID = 4409
AVP_TGPP_PROSE_TARGET_LAYER_2_ID = 4410
AVP_TGPP_RELAY_IP_ADDRESS = 4411
AVP_TGPP_TARGET_IP_ADDRESS = 4412
AVP_TGPP_FE_IDENTIFIER_LIST = 4413
AVP_ACCOUNTING_INPUT_OCTETS = 363
AVP_ACCOUNTING_OUTPUT_OCTETS = 364
AVP_ACCOUNTING_INPUT_PACKETS = 365
AVP_ACCOUNTING_OUTPUT_PACKETS = 366
AVP_NAS_FILTER_RULE = 400
AVP_TUNNELING = 401
AVP_CHAP_AUTH = 402
AVP_CHAP_ALGORITHM = 403
E_CHAP_ALGORITHM_CHAP_WITH_MD5 = 5

AVP_CHAP_IDENT = 404
AVP_CHAP_RESPONSE = 405
AVP_ACCOUNTING_AUTH_METHOD = 406
E_ACCOUNTING_AUTH_METHOD_PAP = 1
E_ACCOUNTING_AUTH_METHOD_CHAP = 2
E_ACCOUNTING_AUTH_METHOD_MS_CHAP_1 = 3
E_ACCOUNTING_AUTH_METHOD_MS_CHAP_2 = 4
E_ACCOUNTING_AUTH_METHOD_EAP = 5
E_ACCOUNTING_AUTH_METHOD_UNDEFINED = 6
E_ACCOUNTING_AUTH_METHOD_NONE = 7

AVP_ORIGIN_AAA_PROTOCOL = 408
E_ORIGIN_AAA_PROTOCOL_RADIUS = 1

AVP_EAP_PAYLOAD = 462
AVP_EAP_REISSUED_PAYLOAD = 463
AVP_EAP_MASTER_SESSION_KEY = 464
AVP_ACCOUNTING_EAP_AUTH_METHOD = 465
AVP_MIP_FA_TO_HA_SPI = 318
AVP_MIP_FA_TO_MN_SPI = 319
AVP_MIP_REG_REQUEST = 320
AVP_MIP_REG_REPLY = 321
AVP_MIP_MN_AAA_AUTH = 322
AVP_MIP_HA_TO_FA_SPI = 323
# This might not actually exist. rfc4004 does not define it, but there's a hole
# in the defined AVP codes that might have been for this one.
AVP_MIP_MN_TO_FA_SPI = 324
AVP_MIP_MN_TO_FA_MSA = 325
AVP_MIP_FA_TO_MN_MSA = 326
AVP_MIP_FA_TO_HA_MSA = 328
AVP_MIP_HA_TO_FA_MSA = 329
AVP_MIP_MN_TO_HA_MSA = 331
AVP_MIP_HA_TO_MN_MSA = 332
AVP_MIP_MOBILE_NODE_ADDRESS = 333
AVP_MIP_HOME_AGENT_ADDRESS = 334
AVP_MIP_NONCE = 335
AVP_MIP_CANDIDATE_HOME_AGENT_HOST = 336
AVP_MIP_FEATURE_VECTOR = 337
E_MIP_FEATURE_VECTOR_MOBILE_NODE_HOME_ADDRESS_REQUESTED = 1
E_MIP_FEATURE_VECTOR_HOME_ADDRESS_ALLOCATABLE_ONLY_IN_HOME_REALM = 2
E_MIP_FEATURE_VECTOR_HOME_AGENT_REQUESTED = 4
E_MIP_FEATURE_VECTOR_FOREIGN_HOME_AGENT_AVAILABLE = 8
E_MIP_FEATURE_VECTOR_MN_HA_KEY_REQUEST = 16
E_MIP_FEATURE_VECTOR_MN_FA_KEY_REQUEST = 32
E_MIP_FEATURE_VECTOR_FA_HA_KEY_REQUEST = 64
E_MIP_FEATURE_VECTOR_HOME_AGENT_IN_FOREIGN_NETWORK = 128
E_MIP_FEATURE_VECTOR_CO_LOCATED_MOBILE_NODE = 256

AVP_MIP_AUTH_INPUT_DATA_LENGTH = 338
AVP_MIP_AUTHENTICATOR_LENGTH = 339
AVP_MIP_AUTHENTICATOR_OFFSET = 340
AVP_MIP_MN_AAA_SPI = 341
AVP_MIP_FILTER_RULE = 342
AVP_MIP_SESSION_KEY = 343
AVP_MIP_FA_CHALLENGE = 344
AVP_MIP_ALGORITHM_TYPE = 345
E_MIP_ALGORITHM_TYPE_HMAC_SHA_1 = 2

AVP_MIP_REPLAY_MODE = 346
E_MIP_REPLAY_MODE_NONE = 1
E_MIP_REPLAY_MODE_TIMESTAMPS = 2
E_MIP_REPLAY_MODE_NONCES = 3

AVP_MIP_ORIGINATING_FOREIGN_AAA = 347
AVP_MIP_HOME_AGENT_HOST = 348
AVP_MIP_MSA_LIFETIME = 367
AVP_CC_CORRELATION_ID = 411
AVP_CC_INPUT_OCTETS = 412
AVP_CC_MONEY = 413
AVP_CC_OUTPUT_OCTETS = 414
AVP_CC_REQUEST_NUMBER = 415
AVP_CC_REQUEST_TYPE = 416
E_CC_REQUEST_TYPE_INITIAL_REQUEST = 1
E_CC_REQUEST_TYPE_UPDATE_REQUEST = 2
E_CC_REQUEST_TYPE_TERMINATION_REQUEST = 3
E_CC_REQUEST_TYPE_EVENT_REQUEST = 4

AVP_CC_SERVICE_SPECIFIC_UNITS = 417
AVP_CC_SESSION_FAILOVER = 418
E_CC_SESSION_FAILOVER_FAILOVER_NOT_SUPPORTED = 0
E_CC_SESSION_FAILOVER_FAILOVER_SUPPORTED = 1

AVP_CC_SUB_SESSION_ID = 419
AVP_CC_TIME = 420
AVP_CC_TOTAL_OCTETS = 421
AVP_CHECK_BALANCE_RESULT = 422
E_CHECK_BALANCE_RESULT_ENOUGH_CREDIT = 0
E_CHECK_BALANCE_RESULT_NO_CREDIT = 1

AVP_COST_INFORMATION = 423
AVP_COST_UNIT = 424
AVP_CURRENCY_CODE = 425
AVP_CREDIT_CONTROL = 426
E_CREDIT_CONTROL_CREDIT_AUTHORIZATION = 0
E_CREDIT_CONTROL_RE_AUTHORIZATION = 1

AVP_CREDIT_CONTROL_FAILURE_HANDLING = 427
E_CREDIT_CONTROL_FAILURE_HANDLING_TERMINATE = 0
E_CREDIT_CONTROL_FAILURE_HANDLING_CONTINUE = 1
E_CREDIT_CONTROL_FAILURE_HANDLING_RETRY_AND_TERMINATE = 2

AVP_DIRECT_DEBITING_FAILURE_HANDLING = 428
E_DIRECT_DEBITING_FAILURE_HANDLING_CONTINUE = 1
E_DIRECT_DEBITING_FAILURE_HANDLING_TERMINATE_OR_BUFFER = 0

AVP_EXPONENT = 429
AVP_FINAL_UNIT_INDICATION = 430
AVP_GRANTED_SERVICE_UNIT = 431
AVP_RATING_GROUP = 432
AVP_REDIRECT_ADDRESS_TYPE = 433
E_REDIRECT_ADDRESS_TYPE_IPV6_ADDRESS = 1
E_REDIRECT_ADDRESS_TYPE_SIP_URI = 3
E_REDIRECT_ADDRESS_TYPE_URL = 2
E_REDIRECT_ADDRESS_TYPE_IPV4_ADDRESS = 0

AVP_REDIRECT_SERVER = 434
AVP_REDIRECT_SERVER_ADDRESS = 435
AVP_REQUESTED_ACTION = 436
E_REQUESTED_ACTION_DIRECT_DEBITING = 0
E_REQUESTED_ACTION_REFUND_ACCOUNT = 1
E_REQUESTED_ACTION_CHECK_BALANCE = 2
E_REQUESTED_ACTION_PRICE_ENQUIRY = 3

AVP_REQUESTED_SERVICE_UNIT = 437
AVP_RESTRICTION_FILTER_RULE = 438
AVP_SERVICE_IDENTIFIER = 439
AVP_SERVICE_PARAMETER_INFO = 440
AVP_SERVICE_PARAMETER_TYPE = 441
AVP_SERVICE_PARAMETER_VALUE = 442
AVP_SUBSCRIPTION_ID = 443
AVP_SUBSCRIPTION_ID_DATA = 444
AVP_UNIT_VALUE = 445
AVP_USED_SERVICE_UNIT = 446
AVP_VALUE_DIGITS = 447
AVP_VALIDITY_TIME = 448
AVP_FINAL_UNIT_ACTION = 449
E_FINAL_UNIT_ACTION_TERMINATE = 0
E_FINAL_UNIT_ACTION_REDIRECT = 1
E_FINAL_UNIT_ACTION_RESTRICT_ACCESS = 2

AVP_SUBSCRIPTION_ID_TYPE = 450
E_SUBSCRIPTION_ID_TYPE_END_USER_E164 = 0
E_SUBSCRIPTION_ID_TYPE_END_USER_IMSI = 1
E_SUBSCRIPTION_ID_TYPE_END_USER_SIP_URI = 2
E_SUBSCRIPTION_ID_TYPE_END_USER_NAI = 3
E_SUBSCRIPTION_ID_TYPE_END_USER_PRIVATE = 4

AVP_TARIFF_TIME_CHANGE = 451
AVP_TARIFF_CHANGE_USAGE = 452
E_TARIFF_CHANGE_USAGE_UNIT_AFTER_TARIFF_CHANGE = 1
E_TARIFF_CHANGE_USAGE_UNIT_INDETERMINATE = 2
E_TARIFF_CHANGE_USAGE_UNIT_BEFORE_TARIFF_CHANGE = 0

AVP_G_S_U_POOL_IDENTIFIER = 453
AVP_CC_UNIT_TYPE = 454
E_CC_UNIT_TYPE_TIME = 0
E_CC_UNIT_TYPE_MONEY = 1
E_CC_UNIT_TYPE_TOTAL_OCTETS = 2
E_CC_UNIT_TYPE_OUTPUT_OCTETS = 4
E_CC_UNIT_TYPE_INPUT_OCTETS = 3
E_CC_UNIT_TYPE_SERVICE_SPECIFIC_UNITS = 5

AVP_MULTIPLE_SERVICES_INDICATOR = 455
E_MULTIPLE_SERVICES_INDICATOR_MULTIPLE_SERVICES_NOT_SUPPORTED = 0
E_MULTIPLE_SERVICES_INDICATOR_MULTIPLE_SERVICES_SUPPORTED = 1

AVP_MULTIPLE_SERVICES_CREDIT_CONTROL = 456
AVP_G_S_U_POOL_REFERENCE = 457
AVP_USER_EQUIPMENT_INFO = 458
AVP_USER_EQUIPMENT_INFO_TYPE = 459
E_USER_EQUIPMENT_INFO_TYPE_IMEISV = 0
E_USER_EQUIPMENT_INFO_TYPE_MAC = 1
E_USER_EQUIPMENT_INFO_TYPE_EUI64 = 2
E_USER_EQUIPMENT_INFO_TYPE_MODIFIED_EUI64 = 3

AVP_USER_EQUIPMENT_INFO_VALUE = 460
AVP_SERVICE_CONTEXT_ID = 461
AVP_REDIRECT_SERVER_EXTENSION = 665
AVP_REDIRECT_ADDRESS_IPADDRESS = 666
AVP_REDIRECT_ADDRESS_URL = 667
AVP_REDIRECT_ADDRESS_SIP_URI = 668
AVP_QOS_FINAL_UNIT_INDICATION = 669
AVP_SUN_PING_TIMESTAMP_SECS = 1
AVP_SUN_PING_TIMESTAMP_USECS = 2
AVP_SUN_PING_TIMESTAMP = 3
AVP_TGPP_3GPP_IMSI = 1
AVP_TGPP_3GPP_CHARGING_ID = 2
AVP_TGPP_3GPP_PDP_TYPE = 3
E_3GPP_PDP_TYPE_IPV4 = 0
E_3GPP_PDP_TYPE_PPP = 1
E_3GPP_PDP_TYPE_IPV6 = 2
E_3GPP_PDP_TYPE_IPV4V6 = 3

AVP_TGPP_3GPP_CG_ADDRESS = 4
AVP_TGPP_3GPP_GPRS_NEGOTIATED_QOS_PROFILE = 5
AVP_TGPP_3GPP_SGSN_ADDRESS = 6
AVP_TGPP_3GPP_GGSN_ADDRESS = 7
AVP_TGPP_3GPP_IMSI_MCC_MNC = 8
AVP_TGPP_3GPP_GGSN_MCC_MNC = 9
AVP_TGPP_3GPP_NSAPI = 10
AVP_TGPP_3GPP_SESSION_STOP_INDICATOR = 11
AVP_TGPP_3GPP_SELECTION_MODE = 12
AVP_TGPP_3GPP_CHARGING_CHARACTERISTICS = 13
AVP_TGPP_3GPP_CG_IPV6_ADDRESS = 14
AVP_TGPP_3GPP_SGSN_IPV6_ADDRESS = 15
AVP_TGPP_3GPP_GGSN_IPV6_ADDRESS = 16
AVP_TGPP_3GPP_IPV6_DNS_SERVER = 17
AVP_TGPP_3GPP_SGSN_MCC_MNC = 18
AVP_TGPP_3GPP_TEARDOWN_INDICATOR = 19
AVP_TGPP_3GPP_IMEISV = 20
AVP_TGPP_3GPP_RAT_TYPE = 21
AVP_TGPP_3GPP_USER_LOCATION_INFO = 22
AVP_TGPP_3GPP_MS_TIMEZONE = 23
AVP_TGPP_3GPP_CAMEL_CHARGING_INFO = 24
AVP_TGPP_3GPP_PACKET_FILTER = 25
AVP_TGPP_3GPP_NEGOTIATED_DSCP = 26
AVP_TGPP_3GPP_ALLOCATE_IP_TYPE = 27
AVP_TGPP_3GPP_TWAN_IDENTIFIER = 29
AVP_TGPP_ABORT_CAUSE = 500
E_ABORT_CAUSE_BEARER_RELEASED = 0
E_ABORT_CAUSE_INSUFFICIENT_SERVER_RESOURCES = 1
E_ABORT_CAUSE_INSUFFICIENT_BEARER_RESOURCES = 2
E_ABORT_CAUSE_PS_TO_CS_HANDOVER = 3
E_ABORT_CAUSE_SPONSORED_DATA_CONNECTIVITY__DISALLOWED = 4

AVP_TGPP_ACCESS_NETWORK_CHARGING_ADDRESS = 501
AVP_TGPP_ACCESS_NETWORK_CHARGING_IDENTIFIER = 502
AVP_TGPP_ACCESS_NETWORK_CHARGING_IDENTIFIER_VALUE = 503
AVP_TGPP_AF_APPLICATION_IDENTIFIER = 504
AVP_TGPP_AF_CHARGING_IDENTIFIER = 505
AVP_TGPP_AUTHORIZATION_TOKEN = 506
AVP_TGPP_FLOW_DESCRIPTION = 507
AVP_TGPP_FLOW_GROUPING = 508
AVP_TGPP_FLOW_NUMBER = 509
AVP_TGPP_FLOWS = 510
AVP_TGPP_FLOW_STATUS = 511
E_FLOW_STATUS_ENABLED_UPLINK = 0
E_FLOW_STATUS_ENABLED_DOWNLINK = 1
E_FLOW_STATUS_ENABLED = 2
E_FLOW_STATUS_DISABLED = 3
E_FLOW_STATUS_REMOVED = 4

AVP_TGPP_FLOW_USAGE = 512
E_FLOW_USAGE_NO_INFORMATION = 0
E_FLOW_USAGE_RTCP = 1
E_FLOW_USAGE_AF_SIGNALLING = 2

AVP_TGPP_SPECIFIC_ACTION = 513
E_SPECIFIC_ACTION_SERVICE_INFORMATION_REQUEST_NOW_VOID = 0
E_SPECIFIC_ACTION_CHARGING_CORRELATION_EXCHANGE = 1
E_SPECIFIC_ACTION_INDICATION_OF_LOSS_OF_BEARER = 2
E_SPECIFIC_ACTION_INDICATION_OF_RECOVERY_OF_BEARER = 3
E_SPECIFIC_ACTION_INDICATION_OF_RELEASE_OF_BEARER = 4
E_SPECIFIC_ACTION_INDICATION_OF_ESTABLISHMENT_OF_BEARER_NOW_VOID = 5
E_SPECIFIC_ACTION_IP_CAN_CHANGE = 6
E_SPECIFIC_ACTION_INDICATION_OF_OUT_OF_CREDIT = 7
E_SPECIFIC_ACTION_INDICATION_OF_SUCCESSFUL_RESOURCES_ALLOCATION = 8
E_SPECIFIC_ACTION_INDICATION_OF_FAILED_RESOURCES_ALLOCATION = 9
E_SPECIFIC_ACTION_INDICATION_OF_LIMITED_PCC_DEPLOYMENT = 10
E_SPECIFIC_ACTION_USAGE_REPORT = 11
E_SPECIFIC_ACTION_ACCESS_NETWORK_INFO_REPORT = 12
E_SPECIFIC_ACTION_INDICATION_OF_RECOVERY_FROM_LIMITED_PCC_DEPLOYMENT = 13
E_SPECIFIC_ACTION_INDICATION_OF_ACCESS_NETWORK_INFO_REPORTING_FAILURE = 14
E_SPECIFIC_ACTION_INDICATION_OF_TRANSFER_POLICY_EXPIRED = 15
E_SPECIFIC_ACTION_PLMN_CHANGE = 16

AVP_TGPP_MAX_REQUESTED_BANDWIDTH_DL = 515
AVP_TGPP_MAX_REQUESTED_BANDWIDTH_UL = 516
AVP_TGPP_MEDIA_COMPONENT_DESCRIPTION = 517
AVP_TGPP_MEDIA_COMPONENT_NUMBER = 518
AVP_TGPP_MEDIA_SUB_COMPONENT = 519
AVP_TGPP_MEDIA_TYPE = 520
E_MEDIA_TYPE_AUDIO = 0
E_MEDIA_TYPE_VIDEO = 1
E_MEDIA_TYPE_DATA = 2
E_MEDIA_TYPE_APPLICATION = 3
E_MEDIA_TYPE_CONTROL = 4
E_MEDIA_TYPE_TEXT = 5
E_MEDIA_TYPE_MESSAGE = 6
E_MEDIA_TYPE_OTHER = 4294967295

AVP_TGPP_RR_BANDWIDTH = 521
AVP_TGPP_RS_BANDWIDTH = 522
AVP_TGPP_SIP_FORKING_INDICATION = 523
E_SIP_FORKING_INDICATION_SINGLE_DIALOGUE = 0
E_SIP_FORKING_INDICATION_SEVERAL_DIALOGUES = 1

AVP_TGPP_CODEC_DATA = 524
AVP_TGPP_SERVICE_URN = 525
AVP_TGPP_ACCEPTABLE_SERVICE_INFO = 526
AVP_TGPP_SERVICE_INFO_STATUS = 527
E_SERVICE_INFO_STATUS_FINAL_SERVICE_INFORMATION = 0
E_SERVICE_INFO_STATUS_PRELIMINARY_SERVICE_INFORMATION = 1

AVP_TGPP_MPS_IDENTIFIER = 528
AVP_TGPP_AF_SIGNALLING_PROTOCOL = 529
E_AF_SIGNALLING_PROTOCOL_NO_INFORMATION = 0
E_AF_SIGNALLING_PROTOCOL_SIP = 1

AVP_TGPP_SPONSORED_CONNECTIVITY_DATA = 530
AVP_TGPP_SPONSOR_IDENTITY = 531
AVP_TGPP_APPLICATION_SERVICE_PROVIDER_IDENTITY = 532
AVP_TGPP_RX_REQUEST_TYPE = 533
E_RX_REQUEST_TYPE_INITIAL_REQUEST = 0
E_RX_REQUEST_TYPE_UPDATE_REQUEST = 1
E_RX_REQUEST_TYPE_PCSCF_RESTORATION = 2

AVP_TGPP_MIN_REQUESTED_BANDWIDTH_DL = 534
AVP_TGPP_MIN_REQUESTED_BANDWIDTH_UL = 535
AVP_TGPP_REQUIRED_ACCESS_INFO = 536
E_REQUIRED_ACCESS_INFO_USER_LOCATION = 0
E_REQUIRED_ACCESS_INFO_MS_TIME_ZONE = 1

AVP_TGPP_IP_DOMAIN_ID = 537
AVP_TGPP_GCS_IDENTIFIER = 538
AVP_TGPP_SHARING_KEY_DL = 539
AVP_TGPP_SHARING_KEY_UL = 540
AVP_TGPP_RETRY_INTERVAL = 541
AVP_TGPP_SPONSORING_ACTION = 542
E_SPONSORING_ACTION_DISABLE_SPONSORING = 0
E_SPONSORING_ACTION_ENABLE_SPONSORING = 1

AVP_TGPP_MAX_SUPPORTED_BANDWIDTH_DL = 543
AVP_TGPP_MAX_SUPPORTED_BANDWIDTH_UL = 544
AVP_TGPP_MIN_DESIRED_BANDWIDTH_DL = 545
AVP_TGPP_MIN_DESIRED_BANDWIDTH_UL = 546
AVP_TGPP_MCPTT_IDENTIFIER = 547
AVP_TGPP_SERVICE_AUTHORIZATION_INFO = 548
AVP_TGPP_MEDIA_COMPONENT_STATUS = 549
AVP_TGPP_PRIORITY_SHARING_INDICATOR = 550
E_PRIORITY_SHARING_INDICATOR_PRIORITY_SHARING_ENABLED = 0
E_PRIORITY_SHARING_INDICATOR_PRIORITY_SHARING_DISABLED = 1

AVP_TGPP_AF_REQUESTED_DATA = 551
AVP_TGPP_CONTENT_VERSION = 552
AVP_TGPP_PRE_EMPTION_CONTROL_INFO = 553
AVP_TGPP_EXTENDED_MAX_REQUESTED_BW_DL = 554
AVP_TGPP_EXTENDED_MAX_REQUESTED_BW_UL = 555
AVP_TGPP_EXTENDED_MAX_SUPPORTED_BW_DL = 556
AVP_TGPP_EXTENDED_MAX_SUPPORTED_BW_UL = 557
AVP_TGPP_EXTENDED_MIN_DESIRED_BW_DL = 558
AVP_TGPP_EXTENDED_MIN_DESIRED_BW_UL = 559
AVP_TGPP_EXTENDED_MIN_REQUESTED_BW_DL = 560
AVP_TGPP_EXTENDED_MIN_REQUESTED_BW_UL = 561
AVP_TGPP_MCVIDEO_IDENTIFIER = 562
AVP_TGPP_IMS_CONTENT_IDENTIFIER = 563
AVP_TGPP_IMS_CONTENT_TYPE = 564
E_IMS_CONTENT_TYPE_NO_CONTENT_DETAIL = 0
E_IMS_CONTENT_TYPE_CAT = 1
E_IMS_CONTENT_TYPE_CONFERENCE = 2

AVP_TGPP_VISITED_NETWORK_IDENTIFIER = 600
AVP_TGPP_PUBLIC_IDENTITY = 601
AVP_TGPP_SERVER_NAME = 602
AVP_TGPP_SERVER_CAPABILITIES = 603
AVP_TGPP_MANDATORY_CAPABILITY = 604
AVP_TGPP_OPTIONAL_CAPABILITY = 605
AVP_TGPP_CX_USER_DATA = 606
AVP_TGPP_3GPP_SIP_NUMBER_AUTH_ITEMS = 607
AVP_TGPP_3GPP_SIP_AUTHENTICATION_SCHEME = 608
AVP_TGPP_3GPP_SIP_AUTHENTICATE = 609
AVP_TGPP_3GPP_SIP_AUTHORIZATION = 610
AVP_TGPP_SIP_AUTHENTICATION_CONTEXT = 611
AVP_TGPP_3GPP_SIP_AUTH_DATA_ITEM = 612
AVP_TGPP_3GPP_SIP_ITEM_NUMBER = 613
AVP_TGPP_SERVER_ASSIGNMENT_TYPE = 614
E_SERVER_ASSIGNMENT_TYPE_NO_ASSIGNMENT = 0
E_SERVER_ASSIGNMENT_TYPE_REGISTRATION = 1
E_SERVER_ASSIGNMENT_TYPE_RE_REGISTRATION = 2
E_SERVER_ASSIGNMENT_TYPE_UNREGISTERED_USER = 3
E_SERVER_ASSIGNMENT_TYPE_TIMEOUT_DEREGISTRATION = 4
E_SERVER_ASSIGNMENT_TYPE_USER_DEREGISTRATION = 5
E_SERVER_ASSIGNMENT_TYPE_TIMEOUT_DEREGISTRATION_STORE_SERVER_NAME = 6
E_SERVER_ASSIGNMENT_TYPE_USER_DEREGISTRATION_STORE_SERVER_NAME = 7
E_SERVER_ASSIGNMENT_TYPE_ADMINISTRATIVE_DEREGISTRATION = 8
E_SERVER_ASSIGNMENT_TYPE_AUTHENTICATION_FAILURE = 9
E_SERVER_ASSIGNMENT_TYPE_AUTHENTICATION_TIMEOUT = 10
E_SERVER_ASSIGNMENT_TYPE_DEREGISTRATION_TOO_MUCH_DATA = 11
E_SERVER_ASSIGNMENT_TYPE_AAA_USER_DATA_REQUEST = 12
E_SERVER_ASSIGNMENT_TYPE_PGW_UPDATE = 13
E_SERVER_ASSIGNMENT_TYPE_RESTORATION = 14

AVP_TGPP_DEREGISTRATION_REASON = 615
AVP_TGPP_REASON_CODE = 616
E_REASON_CODE_PERMANENT_TERMINATION = 0
E_REASON_CODE_NEW_SERVER_ASSIGNED = 1
E_REASON_CODE_SERVER_CHANGE = 2
E_REASON_CODE_REMOVE_S_CSCF = 3

AVP_TGPP_REASON_INFO = 617
AVP_TGPP_CHARGING_INFORMATION = 618
AVP_TGPP_PRIMARY_EVENT_CHARGING_FUNCTION_NAME = 619
AVP_TGPP_SECONDARY_EVENT_CHARGING_FUNCTION_NAME = 620
AVP_TGPP_PRIMARY_CHARGING_COLLECTION_FUNCTION_NAME = 621
AVP_TGPP_SECONDARY_CHARGING_COLLECTION_FUNCTION_NAME = 622
AVP_TGPP_USER_AUTHORIZATION_TYPE = 623
E_USER_AUTHORIZATION_TYPE_REGISTRATION = 0
E_USER_AUTHORIZATION_TYPE_DE_REGISTRATION = 1
E_USER_AUTHORIZATION_TYPE_REGISTRATION_AND_CAPABILITIES = 2

AVP_TGPP_USER_DATA_ALREADY_AVAILABLE = 624
E_USER_DATA_ALREADY_AVAILABLE_USER_DATA_NOT_AVAILABLE = 0
E_USER_DATA_ALREADY_AVAILABLE_USER_DATA_ALREADY_AVAILABLE = 1

AVP_TGPP_CONFIDENTIALITY_KEY = 625
AVP_TGPP_INTEGRITY_KEY = 626
AVP_TGPP_USER_DATA_REQUEST_TYPE_OBSOLETE = 627
E_USER_DATA_REQUEST_TYPE_OBSOLETE_COMPLETE_PROFILE = 0
E_USER_DATA_REQUEST_TYPE_OBSOLETE_REGISTERED_PROFILE = 1
E_USER_DATA_REQUEST_TYPE_OBSOLETE_UNREGISTERED_PROFILE = 2

AVP_TGPP_SUPPORTED_FEATURES = 628
AVP_TGPP_FEATURE_LIST_ID = 629
AVP_TGPP_FEATURE_LIST = 630
AVP_TGPP_SUPPORTED_APPLICATIONS = 631
AVP_TGPP_ASSOCIATED_IDENTITIES = 632
AVP_TGPP_ORIGINATING_REQUEST = 633
E_ORIGINATING_REQUEST_ORIGINATING = 0

AVP_TGPP_WILDCARDED_PSI = 634
AVP_TGPP_SIP_DIGEST_AUTHENTICATE = 635
AVP_TGPP_WILDCARDED_IMPU = 636
AVP_TGPP_UAR_FLAGS = 637
AVP_TGPP_LOOSE_ROUTE_INDICATION = 638
E_LOOSE_ROUTE_INDICATION_LOOSE_ROUTE_NOT_REQUIRED = 0
E_LOOSE_ROUTE_INDICATION_LOOSE_ROUTE_REQUIRED = 1

AVP_TGPP_SCSCF_RESTORATION_INFO = 639
AVP_TGPP_PATH = 640
AVP_TGPP_CONTACT = 641
AVP_TGPP_SUBSCRIPTION_INFO = 642
AVP_TGPP_CALL_ID_SIP_HEADER = 643
AVP_TGPP_FROM_SIP_HEADER = 644
AVP_TGPP_TO_SIP_HEADER = 645
AVP_TGPP_RECORD_ROUTE = 646
AVP_TGPP_ASSOCIATED_REGISTERED_IDENTITIES = 647
AVP_TGPP_MULTIPLE_REGISTRATION_INDICATION = 648
E_MULTIPLE_REGISTRATION_INDICATION_NOT_MULTIPLE_REGISTRATION = 0
E_MULTIPLE_REGISTRATION_INDICATION_MULTIPLE_REGISTRATION = 1

AVP_TGPP_RESTORATION_INFO = 649
AVP_TGPP_SESSION_PRIORITY = 650
E_SESSION_PRIORITY_PRIORITY_0 = 0
E_SESSION_PRIORITY_PRIORITY_1 = 1
E_SESSION_PRIORITY_PRIORITY_2 = 2
E_SESSION_PRIORITY_PRIORITY_3 = 3
E_SESSION_PRIORITY_PRIORITY_4 = 4

AVP_TGPP_IDENTITY_WITH_EMERGENCY_REGISTRATION = 651
AVP_TGPP_PRIVILEDGED_SENDER_INDICATION = 652
E_PRIVILEDGED_SENDER_INDICATION_NOT_PRIVILEDGED_SENDER = 0
E_PRIVILEDGED_SENDER_INDICATION_PRIVILEDGED_SENDER = 1

AVP_TGPP_LIA_FLAGS = 653
AVP_TGPP_INITIAL_CSEQ_SEQUENCE_NUMBER = 654
AVP_TGPP_SAR_FLAGS = 655
AVP_TGPP_ALLOWED_WAF_WWSF_IDENTITIES = 656
AVP_TGPP_WEBRTC_AUTHENTICATION_FUNCTION_NAME = 657
AVP_TGPP_WEBRTC_WEB_SERVER_FUNCTION_NAME = 658
AVP_TGPP_USER_IDENTITY = 700
AVP_TGPP_MSISDN = 701
AVP_TGPP_SH_USER_DATA = 702
AVP_TGPP_DATA_REFERENCE = 703
E_DATA_REFERENCE_REPOSITORYDATA = 0
E_DATA_REFERENCE_UNDEFINED = 9
E_DATA_REFERENCE_IMSPUBLICIDENTITY = 10
E_DATA_REFERENCE_IMSUSERSTATE = 11
E_DATA_REFERENCE_S_CSCFNAME = 12
E_DATA_REFERENCE_INITIALFILTERCRITERIA = 13
E_DATA_REFERENCE_LOCATIONINFORMATION = 14
E_DATA_REFERENCE_USERSTATE = 15
E_DATA_REFERENCE_CHARGINGINFORMATION = 16
E_DATA_REFERENCE_MSISDN = 17
E_DATA_REFERENCE_PSIACTIVATION = 18
E_DATA_REFERENCE_DSAI = 19
E_DATA_REFERENCE_RESERVED = 20
E_DATA_REFERENCE_SERVICELEVELTRACEINFO = 21
E_DATA_REFERENCE_IPADDRESSSECUREBINDINGINFORMATION = 22
E_DATA_REFERENCE_SERVICEPRIORITYLEVEL = 23
E_DATA_REFERENCE_SMSREGISTRATIONINFO = 24
E_DATA_REFERENCE_UEREACHABILITYFORIP = 25
E_DATA_REFERENCE_TADSINFORMATION = 26
E_DATA_REFERENCE_STN_SR = 27
E_DATA_REFERENCE_UE_SRVCC_CAPABILITY = 28
E_DATA_REFERENCE_EXTENDEDPRIORITY = 29
E_DATA_REFERENCE_CSRN = 30
E_DATA_REFERENCE_REFERENCELOCATIONINFORMATION = 31
E_DATA_REFERENCE_IMSI = 32
E_DATA_REFERENCE_IMSPRIVATEUSERIDENTITY = 33
E_DATA_REFERENCE_IMEISV = 34
E_DATA_REFERENCE_UE_5G_SRVCC_CAPABILITY = 35

AVP_TGPP_SERVICE_INDICATION = 704
AVP_TGPP_SUBS_REQ_TYPE = 705
E_SUBS_REQ_TYPE_SUBSCRIBE = 0
E_SUBS_REQ_TYPE_UNSUBSCRIBE = 1

AVP_TGPP_REQUESTED_DOMAIN = 706
E_REQUESTED_DOMAIN_CS_DOMAIN = 0
E_REQUESTED_DOMAIN_PS_DOMAIN = 1

AVP_TGPP_CURRENT_LOCATION = 707
E_CURRENT_LOCATION_DONOTNEEDINITIATEACTIVELOCATIONRETRIEVAL = 0
E_CURRENT_LOCATION_INITIATEACTIVELOCATIONRETRIEVAL = 1

AVP_TGPP_IDENTITY_SET = 708
E_IDENTITY_SET_ALL_IDENTITIES = 0
E_IDENTITY_SET_REGISTERED_IDENTITIES = 1
E_IDENTITY_SET_IMPLICIT_IDENTITIES = 2
E_IDENTITY_SET_ALIAS_IDENTITIES = 3

AVP_TGPP_EXPIRY_TIME = 709
AVP_TGPP_SEND_DATA_INDICATION = 710
E_SEND_DATA_INDICATION_USER_DATA_NOT_REQUESTED = 0
E_SEND_DATA_INDICATION_USER_DATA_REQUESTED = 1

AVP_TGPP_DSAI_TAG = 711
AVP_TGPP_ONE_TIME_NOTIFICATION = 712
E_ONE_TIME_NOTIFICATION_ONE_TIME_NOTIFICATION_REQUESTED = 0

AVP_TGPP_REQUESTED_NODES = 713
AVP_TGPP_SERVING_NODE_INDICATION = 714
E_SERVING_NODE_INDICATION_ONLY_SERVING_NODES_REQUIRED = 0

AVP_TGPP_REPOSITORY_DATA_ID = 715
AVP_TGPP_SEQUENCE_NUMBER = 716
AVP_TGPP_PRE_PAGING_SUPPORTED = 717
E_PRE_PAGING_SUPPORTED_PREPAGING_NOT_SUPPORTED = 0
E_PRE_PAGING_SUPPORTED_PREPAGING_SUPPORTED = 1

AVP_TGPP_LOCAL_TIME_ZONE_INDICATION = 718
E_LOCAL_TIME_ZONE_INDICATION_ONLY_LOCAL_TIME_ZONE_REQUESTED = 0
E_LOCAL_TIME_ZONE_INDICATION_LOCAL_TIME_ZONE_WITH_LOCATION_INFO_REQUESTED = 1

AVP_TGPP_UDR_FLAGS = 719
E_UDR_FLAGS_LOCATION_INFORMATION_EPS_SUPPORTED = 1
E_UDR_FLAGS_RAT_TYPE_REQUESTED = 2

AVP_TGPP_3GPP_WLAN_APN_ID = 100
AVP_TGPP_AUTHENTICATION_METHOD = 300
E_AUTHENTICATION_METHOD_WLAN_EAP_SIM = 0
E_AUTHENTICATION_METHOD_WLAN_EAP_AKA = 1

AVP_TGPP_AUTHENTICATION_INFORMATION_SIM = 301
AVP_TGPP_AUTHORIZATION_INFORMATION_SIM = 302
AVP_TGPP_WLAN_USER_DATA = 303
AVP_TGPP_CHARGING_DATA = 304
AVP_TGPP_WLAN_ACCESS = 305
E_WLAN_ACCESS_WLAN_SUBSCRIPTION_ALLOWED = 0
E_WLAN_ACCESS_WLAN_SUBSCRIPTION_BARRED = 1

AVP_TGPP_WLAN_3GPP_IP_ACCESS = 306
E_WLAN_3GPP_IP_ACCESS_WLAN_APNS_ENABLE = 0
E_WLAN_3GPP_IP_ACCESS_WLAN_APNS_DISABLE = 1

AVP_TGPP_APN_AUTHORIZED = 307
AVP_TGPP_APN_BARRING_TYPE = 309
E_APN_BARRING_TYPE_WLAN_APN_NO_BARRING = 0
E_APN_BARRING_TYPE_WLAN_APN_HOME_BARRED_WHEN_ROAMING = 1
E_APN_BARRING_TYPE_WLAN_APN_VISITED_BARRED = 2
E_APN_BARRING_TYPE_WLAN_APN_HOME_BARRED = 3
E_APN_BARRING_TYPE_WLAN_APN_INTERNET_ACCESS_BARRED = 4

AVP_TGPP_WLAN_DIRECT_IP_ACCESS = 310
E_WLAN_DIRECT_IP_ACCESS_WLAN_DIRECT_IP_ACCESS = 0
E_WLAN_DIRECT_IP_ACCESS_WLAN_NO_DIRECT_IP_ACCESS = 1

AVP_TGPP_SESSION_REQUEST_TYPE = 311
E_SESSION_REQUEST_TYPE_AUTHORIZATION_REQUEST = 0
E_SESSION_REQUEST_TYPE_ROUTING_POLICY = 1

AVP_TGPP_ROUTING_POLICY = 312
AVP_TGPP_MAX_REQUESTED_BANDWIDTH = 313
AVP_TGPP_CHARGING_CHARACTERISTICS = 314
AVP_TGPP_CHARGING_NODES = 315
AVP_TGPP_PRIMARY_OCS_CHARGING_FUNCTION_NAME = 316
AVP_TGPP_SECONDARY_OCS_CHARGING_FUNCTION_NAME = 317
AVP_TGPP_3GPP_AAA_SERVER_NAME = 318
AVP_TGPP_MAXIMUM_NUMBER_ACCESSES = 319
AVP_TGPP_TMGI = 900
AVP_TGPP_REQUIRED_MBMS_BEARER_CAPABILITIES = 901
AVP_TGPP_MBMS_STARTSTOP_INDICATION = 902
E_MBMS_STARTSTOP_INDICATION_START = 0
E_MBMS_STARTSTOP_INDICATION_STOP = 1
E_MBMS_STARTSTOP_INDICATION_UPDATE = 2

AVP_TGPP_MBMS_SERVICE_AREA = 903
AVP_TGPP_MBMS_SESSION_DURATION = 904
AVP_TGPP_ALTERNATIVE_APN = 905
AVP_TGPP_MBMS_SERVICE_TYPE = 906
E_MBMS_SERVICE_TYPE_MULTICAST = 0
E_MBMS_SERVICE_TYPE_BROADCAST = 1

AVP_TGPP_MBMS_2G_3G_INDICATOR = 907
E_MBMS_2G_3G_INDICATOR_2G = 0
E_MBMS_2G_3G_INDICATOR_3G = 1
E_MBMS_2G_3G_INDICATOR_2G_AND_3G = 2

AVP_TGPP_MBMS_SESSION_IDENTITY = 908
AVP_TGPP_RAI = 909
AVP_TGPP_ADDITIONAL_MBMS_TRACE_INFO = 910
AVP_TGPP_MBMS_TIME_TO_DATA_TRANSFER = 911
AVP_TGPP_MBMS_SESSION_REPETITION_NUMBER = 912
AVP_TGPP_MBMS_REQUIRED_QOS = 913
AVP_TGPP_MBMS_COUNTING_INFORMATION = 914
E_MBMS_COUNTING_INFORMATION_COUNTING_NOT_APPLICABLE = 0
E_MBMS_COUNTING_INFORMATION_COUNTING_APPLICABLE = 1

AVP_TGPP_MBMS_USER_DATA_MODE_INDICATION = 915
E_MBMS_USER_DATA_MODE_INDICATION_UNICAST = 0
E_MBMS_USER_DATA_MODE_INDICATION_MULTICAST_AND_UNICAST = 1

AVP_TGPP_MBMS_GGSN_ADDRESS = 916
AVP_TGPP_MBMS_GGSN_IPV6_ADDRESS = 917
AVP_TGPP_MBMS_BMSC_SSM_IP_ADDRESS = 918
AVP_TGPP_MBMS_BMSC_SSM_IPV6_ADDRESS = 919
AVP_TGPP_MBMS_FLOW_IDENTIFIER = 920
AVP_TGPP_CN_IP_MULTICAST_DISTRIBUTION = 921
E_CN_IP_MULTICAST_DISTRIBUTION_NO_IP_MULTICAST = 0
E_CN_IP_MULTICAST_DISTRIBUTION_IP_MULTICAST = 1

AVP_TGPP_MBMS_HC_INDICATOR = 922
E_MBMS_HC_INDICATOR_UNCOMPRESSED_HEADER = 0
E_MBMS_HC_INDICATOR_COMPRESSED_HEADER = 1

AVP_TGPP_MBMS_ACCESS_INDICATOR = 923
E_MBMS_ACCESS_INDICATOR_UTRAN = 0
E_MBMS_ACCESS_INDICATOR_E_UTRAN = 1
E_MBMS_ACCESS_INDICATOR_UTRAN_AND_E_UTRAN = 2

AVP_TGPP_MBMS_GW_SSM_IP_ADDRESS = 924
AVP_TGPP_MBMS_GW_SSM_IPV6_ADDRESS = 925
AVP_TGPP_MBMS_BMSC_SSM_UDP_PORT = 926
AVP_TGPP_MBMS_GW_UDP_PORT = 927
AVP_TGPP_MBMS_GW_UDP_PORT_INDICATOR = 928
E_MBMS_GW_UDP_PORT_INDICATOR_UDP_PORT_REQUIRED = 1

AVP_TGPP_MBMS_DATA_TRANSFER_START = 929
AVP_TGPP_MBMS_DATA_TRANSFER_STOP = 930
AVP_TGPP_MBMS_FLAGS = 931
AVP_TGPP_RESTART_COUNTER = 932
AVP_TGPP_DIAGNOSTIC_INFO = 933
AVP_TGPP_MBMS_CELL_LIST = 934
AVP_TGPP_SUBSESSION_DECISION_INFO = 2200
AVP_TGPP_SUBSESSION_ENFORCEMENT_INFO = 2201
AVP_TGPP_SUBSESSION_ID = 2202
AVP_TGPP_SUBSESSION_OPERATION = 2203
E_SUBSESSION_OPERATION_TERMINATION = 0
E_SUBSESSION_OPERATION_ESTABLISHMENT = 1
E_SUBSESSION_OPERATION_MODIFICATION = 2

AVP_TGPP_MULTIPLE_BBERF_ACTION = 2204
E_MULTIPLE_BBERF_ACTION_ESTABLISHMENT = 0
E_MULTIPLE_BBERF_ACTION_TERMINATION = 1

AVP_TGPP_SC_ADDRESS = 3300
AVP_TGPP_SM_RP_UI = 3301
AVP_TGPP_TFR_FLAGS = 3302
AVP_TGPP_SM_DELIVERY_FAILURE_CAUSE = 3303
AVP_TGPP_SM_ENUMERATED_DELIVERY_FAILURE_CAUSE = 3304
E_SM_ENUMERATED_DELIVERY_FAILURE_CAUSE_MEMORY_CAPACITY_EXCEEDED = 0
E_SM_ENUMERATED_DELIVERY_FAILURE_CAUSE_EQUIPMENT_PROTOCOL_ERROR = 1
E_SM_ENUMERATED_DELIVERY_FAILURE_CAUSE_EQUIPMENT_NOT_SM_EQUIPPED = 2
E_SM_ENUMERATED_DELIVERY_FAILURE_CAUSE_UNKNOWN_SERVICE_CENTRE = 3
E_SM_ENUMERATED_DELIVERY_FAILURE_CAUSE_SC_CONGESTION = 4
E_SM_ENUMERATED_DELIVERY_FAILURE_CAUSE_INVALID_SME_ADDRESS = 5
E_SM_ENUMERATED_DELIVERY_FAILURE_CAUSE_USER_NOT_SC_USER = 6

AVP_TGPP_SM_DIAGNOSTIC_INFO = 3305
AVP_TGPP_SM_DELIVERY_TIMER = 3306
AVP_TGPP_SM_DELIVERY_START_TIME = 3307
AVP_TGPP_SM_RP_MTI = 3308
E_SM_RP_MTI_SM_DELIVER = 0
E_SM_RP_MTI_SM_STATUS_REPORT = 1

AVP_TGPP_SM_RP_SMEA = 3309
AVP_TGPP_SRR_FLAGS = 3310
AVP_TGPP_SM_DELIVERY_NOT_INTENDED = 3311
E_SM_DELIVERY_NOT_INTENDED_ONLY_IMSI_REQUESTED = 0
E_SM_DELIVERY_NOT_INTENDED_ONLY_MCC_MNC_REQUESTED = 1

AVP_TGPP_MWD_STATUS = 3312
AVP_TGPP_MME_ABSENT_USER_DIAGNOSTIC_SM = 3313
AVP_TGPP_MSC_ABSENT_USER_DIAGNOSTIC_SM = 3314
AVP_TGPP_SGSN_ABSENT_USER_DIAGNOSTIC_SM = 3315
AVP_TGPP_SM_DELIVERY_OUTCOME = 3316
AVP_TGPP_MME_SM_DELIVERY_OUTCOME = 3317
AVP_TGPP_MSC_SM_DELIVERY_OUTCOME = 3318
AVP_TGPP_SGSN_SM_DELIVERY_OUTCOME = 3319
AVP_TGPP_IP_SM_GW_SM_DELIVERY_OUTCOME = 3320
AVP_TGPP_SM_DELIVERY_CAUSE = 3321
E_SM_DELIVERY_CAUSE_SM_DELIVERY_CAUSE = 0
E_SM_DELIVERY_CAUSE_ABSENT_USER = 1
E_SM_DELIVERY_CAUSE_SUCCESSFUL_TRANSFER = 2

AVP_TGPP_ABSENT_USER_DIAGNOSTIC_SM = 3322
AVP_TGPP_RDR_FLAGS = 3323
AVP_TGPP_MAXIMUM_UE_AVAILABILITY_TIME = 3329
AVP_TGPP_MAXIMUM_RETRANSMISSION_TIME = 3330
AVP_TGPP_REQUESTED_RETRANSMISSION_TIME = 3331
AVP_TGPP_SMS_GMSC_ADDRESS = 3332
AVP_TGPP_SMS_GMSC_ALERT_EVENT = 3333
AVP_TGPP_SMSF_3GPP_ABSENT_USER_DIAGNOSTIC_SM = 3334
AVP_TGPP_SMSF_NON_3GPP_ABSENT_USER_DIAGNOSTIC_SM = 3335
AVP_TGPP_SMSF_3GPP_SM_DELIVERY_OUTCOME = 3336
AVP_TGPP_SMSF_NON_3GPP_SM_DELIVERY_OUTCOME = 3337
AVP_TGPP_SMSF_3GPP_NUMBER = 3338
AVP_TGPP_SMSF_NON_3GPP_NUMBER = 3339
AVP_TGPP_SMSF_3GPP_NAME = 3340
AVP_TGPP_SMSF_NON_3GPP_NAME = 3341
AVP_TGPP_SMSF_3GPP_REALM = 3342
AVP_TGPP_SMSF_NON_3GPP_REALM = 3343
AVP_TGPP_SMSF_3GPP_ADDRESS = 3344
AVP_TGPP_SMSF_NON_3GPP_ADDRESS = 3345
AVP_TGPP_BMSC_ADDRESS = 3500
AVP_TGPP_BMSC_PORT = 3501
AVP_TGPP_MBMS_BEARER_EVENT = 3502
AVP_TGPP_MBMS_BEARER_EVENT_NOTIFICATION = 3503
AVP_TGPP_MBMS_BEARER_REQUEST = 3504
AVP_TGPP_MBMS_BEARER_RESPONSE = 3505
AVP_TGPP_MBMS_BEARER_RESULT = 3506
AVP_TGPP_MBMS_START_TIME = 3507
AVP_TGPP_RADIO_FREQUENCY = 3508
AVP_TGPP_TMGI_ALLOCATION_REQUEST = 3509
AVP_TGPP_TMGI_ALLOCATION_RESPONSE = 3510
AVP_TGPP_TMGI_ALLOCATION_RESULT = 3511
AVP_TGPP_TMGI_DEALLOCATION_REQUEST = 3512
AVP_TGPP_TMGI_DEALLOCATION_RESPONSE = 3513
AVP_TGPP_TMGI_DEALLOCATION_RESULT = 3514
AVP_TGPP_TMGI_EXPIRY = 3515
AVP_TGPP_TMGI_NUMBER = 3516
AVP_TGPP_MB2U_SECURITY = 3517
AVP_TGPP_PROSE_SUBSCRIPTION_DATA = 3701
AVP_TGPP_PROSE_PERMISSION = 3702
AVP_TGPP_PROSE_ALLOWED_PLMN = 3703
AVP_TGPP_PROSE_DIRECT_ALLOWED = 3704
AVP_TGPP_UPR_FLAGS = 3705
AVP_TGPP_PNR_FLAGS = 3706
AVP_TGPP_PROSE_INITIAL_LOCATION_INFORMATION = 3707
AVP_TGPP_AUTHORIZED_DISCOVERY_RANGE = 3708
AVP_TGPP_COMMUNICATION_FAILURE_INFORMATION = 4300
AVP_TGPP_CAUSE_TYPE = 4301
AVP_TGPP_S1AP_CAUSE = 4302
AVP_TGPP_RANAP_CAUSE = 4303
AVP_TGPP_GMM_CAUSE = 4304
AVP_TGPP_SM_CAUSE = 4305
AVP_TGPP_NUMBER_OF_UE_PER_LOCATION_CONFIGURATION = 4306
AVP_TGPP_NUMBER_OF_UE_PER_LOCATION_REPORT = 4307
AVP_TGPP_UE_COUNT = 4308
AVP_TGPP_BSSGP_CAUSE = 4309
AVP_TGPP_SERVING_PLMN_RATE_CONTROL = 4310
AVP_TGPP_UPLINK_RATE_LIMIT = 4311
AVP_TGPP_DOWNLINK_RATE_LIMIT = 4312
AVP_TGPP_EXTENDED_PCO = 4313
AVP_TGPP_CONNECTION_ACTION = 4314
AVP_TGPP_NON_IP_DATA = 4315
AVP_TGPP_SCEF_WAIT_TIME = 4316
AVP_TGPP_CMR_FLAGS = 4317
AVP_TGPP_RRC_CAUSE_COUNTER = 4318
AVP_TGPP_COUNTER_VALUE = 4319
AVP_TGPP_RRC_COUNTER_TIMESTAMP = 4320
AVP_TGPP_TDA_FLAGS = 4321
AVP_TGPP_IDLE_STATUS_INDICATION = 4322
AVP_TGPP_IDLE_STATUS_TIMESTAMP = 4323
AVP_TGPP_ACTIVE_TIME = 4324
AVP_TGPP_REACHABILITY_CAUSE = 4325
AVP_TGPP2_ACCESS_NETWORK_PHYSICAL_ACCESS_ID_REALM = 898
AVP_TGPP2_ACCESS_NETWORK_PHYSICAL_ACCESS_ID_VALUE = 899
AVP_TGPP2_ACCESS_NETWORK_PHYSICAL_ACCESS_ID = 900
AVP_TGPP2_FLOW_OPERATION = 800
E_FLOW_OPERATION_TERMINATION = 0
E_FLOW_OPERATION_ESTABLISHMENT = 1
E_FLOW_OPERATION_MODIFICATION = 2

AVP_TGPP2_3GPP2_CHARGING_RULE_INSTALL = 801
AVP_TGPP2_3GPP2_CHARGING_RULE_DEFINITION = 802
AVP_TGPP2_3GPP2_EVENT_TRIGGER = 803
E_3GPP2_EVENT_TRIGGER_PCF_CHANGE = 0
E_3GPP2_EVENT_TRIGGER_QOS_CHANGE = 1
E_3GPP2_EVENT_TRIGGER_RAT_CHANGE = 2
E_3GPP2_EVENT_TRIGGER_TFT_CHANGE = 3
E_3GPP2_EVENT_TRIGGER_PLMN_CHANGE = 4
E_3GPP2_EVENT_TRIGGER_LOSS_OF_FLOW = 5
E_3GPP2_EVENT_TRIGGER_RECOVERY_OF_FLOW = 6
E_3GPP2_EVENT_TRIGGER_IP_CAN_CHANGE = 7
E_3GPP2_EVENT_TRIGGER_PCC_RULE_FAILURE = 8
E_3GPP2_EVENT_TRIGGER_ACCESS_NETWORK_PHYSICAL_ACCESS_ID_CHANGE = 9

AVP_TGPP2_3GPP2_QOS_INFORMATION = 804
AVP_TGPP2_3GPP2_CHARGING_RULE_REPORT = 805
AVP_TGPP2_AGW_IP_ADDRESS = 806
AVP_TGPP2_AGW_IPV6_ADDRESS = 807
AVP_TGPP2_3GPP2_RAT_TYPE = 808
E_3GPP2_RAT_TYPE_CDMA2000_1X = 0
E_3GPP2_RAT_TYPE_HRPD = 1
E_3GPP2_RAT_TYPE_WLAN = 2
E_3GPP2_RAT_TYPE_UMB = 3

AVP_TGPP2_FLOW_INFO = 809
AVP_TGPP2_FLOW_IDENTIFIER = 810
AVP_TGPP2_GRANTED_QOS = 811
AVP_TGPP2_REQUESTED_QOS = 812
AVP_TGPP2_FLOW_DESCRIPTION_INFO = 813
AVP_TGPP2_RULE_REASON_CODE = 814
E_RULE_REASON_CODE_RESOURCES_LIMITATION = 0

AVP_TGPP2_AGW_MCC_MNC = 815
AVP_TGPP2_3GPP2_BSID = 9010
AVP_SIP_ACCOUNTING_INFORMATION = 368
AVP_SIP_ACCOUNTING_SERVER_URI = 369
AVP_SIP_CREDIT_CONTROL_SERVER_URI = 370
AVP_SIP_SERVER_URI = 371
AVP_SIP_SERVER_CAPABILITIES = 372
AVP_SIP_MANDATORY_CAPABILITY = 373
AVP_SIP_OPTIONAL_CAPABILITY = 374
AVP_SIP_SERVER_ASSIGNMENT_TYPE = 375
E_SIP_SERVER_ASSIGNMENT_TYPE_NO_ASSIGNMENT = 0
E_SIP_SERVER_ASSIGNMENT_TYPE_REGISTRATION = 1
E_SIP_SERVER_ASSIGNMENT_TYPE_RE_REGISTRATION = 2
E_SIP_SERVER_ASSIGNMENT_TYPE_UNREGISTERED_USER = 3
E_SIP_SERVER_ASSIGNMENT_TYPE_TIMEOUT_DEREGISTRATION = 4
E_SIP_SERVER_ASSIGNMENT_TYPE_USER_DEREGISTRATION = 5
E_SIP_SERVER_ASSIGNMENT_TYPE_TIMEOUT_DEREGISTRATION_STORE_SERVER_NAME = 6
E_SIP_SERVER_ASSIGNMENT_TYPE_USER_DEREGISTRATION_STORE_SERVER_NAME = 7
E_SIP_SERVER_ASSIGNMENT_TYPE_ADMINISTRATIVE_DEREGISTRATION = 8
E_SIP_SERVER_ASSIGNMENT_TYPE_AUTHENTICATION_FAILURE = 9
E_SIP_SERVER_ASSIGNMENT_TYPE_AUTHENTICATION_TIMEOUT = 10
E_SIP_SERVER_ASSIGNMENT_TYPE_DEREGISTRATION_TOO_MUCH_DATA = 11

AVP_SIP_AUTH_DATA_ITEM = 376
AVP_SIP_AUTHENTICATION_SCHEME = 377
E_SIP_AUTHENTICATION_SCHEME_DIGEST = 0

AVP_SIP_ITEM_NUMBER = 378
AVP_SIP_AUTHENTICATE = 379
AVP_SIP_AUTHORIZATION = 380
AVP_SIP_AUTHENTICATION_INFO = 381
AVP_SIP_NUMBER_AUTH_ITEMS = 382
AVP_SIP_DEREGISTRATION_REASON = 383
AVP_SIP_REASON_CODE = 384
E_SIP_REASON_CODE_PERMANENT_TERMINATION = 0
E_SIP_REASON_CODE_NEW_SIP_SERVER_ASSIGNED = 1
E_SIP_REASON_CODE_SIP_SERVER_CHANGE = 2
E_SIP_REASON_CODE_REMOVE_SIP_SERVER = 3

AVP_SIP_REASON_INFO = 385
AVP_SIP_VISITED_NETWORK_ID = 386
AVP_SIP_USER_AUTHORIZATION_TYPE = 387
E_SIP_USER_AUTHORIZATION_TYPE_REGISTRATION = 0
E_SIP_USER_AUTHORIZATION_TYPE_DEREGISTRATION = 1
E_SIP_USER_AUTHORIZATION_TYPE_REGISTRATION_AND_CAPABILITIES = 2

AVP_SIP_SUPPORTED_USER_DATA_TYPE = 388
AVP_SIP_USER_DATA = 389
AVP_SIP_USER_DATA_TYPE = 390
AVP_SIP_USER_DATA_CONTENTS = 391
AVP_SIP_USER_DATA_ALREADY_AVAILABLE = 392
E_SIP_USER_DATA_ALREADY_AVAILABLE_USER_DATA_NOT_AVAILABLE = 0
E_SIP_USER_DATA_ALREADY_AVAILABLE_USER_DATA_ALREADY_AVAILABLE = 1

AVP_SIP_METHOD = 393
AVP_ETSI_ETSI_EXPERIMENTAL_RESULT_CODE = 298
E_ETSI_EXPERIMENTAL_RESULT_CODE_INSUFFICIENT_RESOURCES = 4041
E_ETSI_EXPERIMENTAL_RESULT_CODE_COMMIT_FAILURE = 4043
E_ETSI_EXPERIMENTAL_RESULT_CODE_REFRESH_FAILURE = 4044
E_ETSI_EXPERIMENTAL_RESULT_CODE_QOS_PROFILE_FAILURE = 4045
E_ETSI_EXPERIMENTAL_RESULT_CODE_ACCESS_PROFILE_FAILURE = 4046
E_ETSI_EXPERIMENTAL_RESULT_CODE_PRIORITY_NOT_GRANTED = 4047
E_ETSI_EXPERIMENTAL_RESULT_CODE_MODIFICATION_FAILURE = 5041

AVP_ETSI_GLOBALLY_UNIQUE_ADDRESS = 300
AVP_ETSI_ADDRESS_REALM = 301
AVP_ETSI_LOGICAL_ACCESS_ID = 302
AVP_ETSI_INITIAL_GATE_SETTING = 303
AVP_ETSI_QOS_PROFILE = 304
AVP_ETSI_IP_CONNECTIVITY_STATUS = 305
E_IP_CONNECTIVITY_STATUS_IP_CONNECTIVITY_ON = 0
E_IP_CONNECTIVITY_STATUS_IP_CONNECTIVITY_LOST = 1

AVP_ETSI_ACCESS_NETWORK_TYPE = 306
AVP_ETSI_AGGREGATION_NETWORK_TYPE = 307
E_AGGREGATION_NETWORK_TYPE_UNKNOWN = 0
E_AGGREGATION_NETWORK_TYPE_ATM = 1
E_AGGREGATION_NETWORK_TYPE_ETHERNET = 2

AVP_ETSI_MAXIMUM_ALLOWED_BANDWIDTH_UL = 308
AVP_ETSI_MAXIMUM_ALLOWED_BANDWIDTH_DL = 309
AVP_ETSI_MAXIMUM_PRIORITY_DEPRECATED = 310
AVP_ETSI_TRANSPORT_CLASS = 311
AVP_ETSI_APPLICATION_CLASS_ID = 312
AVP_ETSI_PHYSICAL_ACCESS_ID = 313
AVP_ETSI_INITIAL_GATE_SETTING_ID = 314
AVP_ETSI_QOS_PROFILE_ID = 315
AVP_ETSI_ETSI_LOCATION_INFORMATION = 350
AVP_ETSI_RACS_CONTACT_POINT = 351
AVP_ETSI_TERMINAL_TYPE = 352
AVP_ETSI_REQUESTED_INFORMATION_353 = 353
E_REQUESTED_INFORMATION_353_SUBSCRIBER_ID = 0
E_REQUESTED_INFORMATION_353_LOCATION_INFORMATION = 1
E_REQUESTED_INFORMATION_353_RACS_CONTACT_POINT = 2
E_REQUESTED_INFORMATION_353_ACCESS_NETWORK_TYPE = 7
E_REQUESTED_INFORMATION_353_TERMINAL_TYPE = 4
E_REQUESTED_INFORMATION_353_LOGICAL_ACCESS_ID = 5
E_REQUESTED_INFORMATION_353_PHYSICAL_ACCESS_ID = 6
E_REQUESTED_INFORMATION_353_INITIAL_GATE_SETTING = 8
E_REQUESTED_INFORMATION_353_QOS_PROFILE = 9
E_REQUESTED_INFORMATION_353_IP_CONNECTIVITY_STATUS = 10

AVP_ETSI_ETSI_EVENT_TYPE_354 = 354
E_ETSI_EVENT_TYPE_354_USER_LOGON = 0
E_ETSI_EVENT_TYPE_354_LOCATION_INFORMATION_CHANGED = 1
E_ETSI_EVENT_TYPE_354_RACS_CONTACT_POINT_CHANGED = 2
E_ETSI_EVENT_TYPE_354_ACCESS_NETWORK_TYPE = 3
E_ETSI_EVENT_TYPE_354_TERMINAL_TYPE_CHANGED = 4
E_ETSI_EVENT_TYPE_354_LOGICAL_ACCESS_ID_CHANGED = 5
E_ETSI_EVENT_TYPE_354_PHYSICAL_ACCESS_ID_CHANGED = 6
E_ETSI_EVENT_TYPE_354_IP_ADDRESS_CHANGED = 7
E_ETSI_EVENT_TYPE_354_INITIAL_GATE_SETTING_CHANGED = 8
E_ETSI_EVENT_TYPE_354_QOS_PROFILE_CHANGED = 9
E_ETSI_EVENT_TYPE_354_USER_LOGOFF = 10

AVP_ETSI_CIVIC_LOCATION = 355
AVP_ETSI_GEOSPATIAL_LOCATION = 356
AVP_ETSI_SESSION_BUNDLE_ID = 400
AVP_ETSI_ETSI_EVENT_TYPE_420 = 420
E_ETSI_EVENT_TYPE_420_UPDATE = 0
E_ETSI_EVENT_TYPE_420_REMOVE = 1

AVP_ETSI_BINDING_INFORMATION = 450
AVP_ETSI_BINDING_INPUT_LIST = 451
AVP_ETSI_BINDING_OUTPUT_LIST = 452
AVP_ETSI_V6_TRANSPORT_ADDRESS = 453
AVP_ETSI_V4_TRANSPORT_ADDRESS = 454
AVP_ETSI_PORT_NUMBER = 455
AVP_ETSI_RESERVATION_CLASS = 456
AVP_ETSI_REQUESTED_INFORMATION_457 = 457
E_REQUESTED_INFORMATION_457_LATCH = 0
E_REQUESTED_INFORMATION_457_RELATCH = 1

AVP_ETSI_RESERVATION_PRIORITY = 458
E_RESERVATION_PRIORITY_DEFAULT = 0
E_RESERVATION_PRIORITY_PRIORITY_ONE = 1
E_RESERVATION_PRIORITY_PRIORITY_TWO = 2
E_RESERVATION_PRIORITY_PRIORITY_THREE = 3
E_RESERVATION_PRIORITY_PRIORITY_FOUR = 4
E_RESERVATION_PRIORITY_PRIORITY_FIVE = 5
E_RESERVATION_PRIORITY_PRIORITY_SIX = 6
E_RESERVATION_PRIORITY_PRIORITY_SEVEN = 7
E_RESERVATION_PRIORITY_PRIORITY_EIGHT = 8
E_RESERVATION_PRIORITY_PRIORITY_NINE = 9
E_RESERVATION_PRIORITY_PRIORITY_TEN = 10
E_RESERVATION_PRIORITY_PRIORITY_ELEVEN = 11
E_RESERVATION_PRIORITY_PRIORITY_TWELVE = 12
E_RESERVATION_PRIORITY_PRIORITY_THIRTEEN = 13
E_RESERVATION_PRIORITY_PRIORITY_FOURTEEN = 14
E_RESERVATION_PRIORITY_PRIORITY_FIFTEEN = 15

AVP_ETSI_ETSI_SERVICE_CLASS = 459
AVP_ETSI_MEDIA_AUTHORIZATION_CONTEXT_ID = 462
AVP_ETSI_LINE_IDENTIFIER = 500
AVP_ETSI_ETSI_SIP_AUTHENTICATE = 501
AVP_ETSI_ETSI_SIP_AUTHORIZATION = 502
AVP_ETSI_ETSI_SIP_AUTHENTICATION_INFO = 503
AVP_ETSI_ETSI_DIGEST_REALM = 504
AVP_ETSI_ETSI_DIGEST_NONCE = 505
AVP_ETSI_ETSI_DIGEST_DOMAIN = 506
AVP_ETSI_ETSI_DIGEST_OPAQUE = 507
AVP_ETSI_ETSI_DIGEST_STALE = 508
AVP_ETSI_ETSI_DIGEST_ALGORITHM = 509
AVP_ETSI_ETSI_DIGEST_QOP = 510
AVP_ETSI_ETSI_DIGEST_HA1 = 511
AVP_ETSI_ETSI_DIGEST_AUTH_PARAM = 512
AVP_ETSI_ETSI_DIGEST_USERNAME = 513
AVP_ETSI_ETSI_DIGEST_URI = 514
AVP_ETSI_ETSI_DIGEST_RESPONSE = 515
AVP_ETSI_ETSI_DIGEST_CNONCE = 516
AVP_ETSI_ETSI_DIGEST_NONCE_COUNT = 517
AVP_ETSI_ETSI_DIGEST_METHOD = 518
AVP_ETSI_ETSI_DIGEST_ENTITY_BODY_HASH = 519
AVP_ETSI_ETSI_DIGEST_NEXTNONCE = 520
AVP_ETSI_ETSI_DIGEST_RESPONSE_AUTH = 521
AVP_MIP6_AGENT_INFO = 486
AVP_MIP_CAREOF_ADDRESS = 487
AVP_MIP_AUTHENTICATOR = 488
AVP_MIP_MAC_MOBILITY_DATA = 489
AVP_MIP_TIMESTAMP = 490
AVP_MIP_MN_HA_SPI = 491
AVP_MIP_MN_HA_MSA = 492
AVP_SERVICE_SELECTION = 493
AVP_MIP6_AUTH_MODE = 494
E_MIP6_AUTH_MODE_RESERVED = 0
E_MIP6_AUTH_MODE_IP6_AUTH_MN_AAA = 1

AVP_PMIP6_DHCP_SERVER_ADDRESS = 504
AVP_PMIP6_IPV4_HOME_ADDRESS = 505
AVP_MOBILE_NODE_IDENTIFIER = 506
AVP_SERVICE_CONFIGURATION = 507
AVP_VODAFONE_CONTEXT_TYPE = 256
E_CONTEXT_TYPE_PRIMARY = 0
E_CONTEXT_TYPE_SECONDARY = 1

AVP_VODAFONE_VODAFONE_QUOTA_CONSUMPTION_TIME = 257
AVP_VODAFONE_VODAFONE_QUOTA_HOLDING_TIME = 258
AVP_VODAFONE_VODAFONE_TIME_QUOTA_THRESHOLD = 259
AVP_VODAFONE_RADIO_ACCESS_TECHNOLOGY = 260
E_RADIO_ACCESS_TECHNOLOGY_UTRAN = 0
E_RADIO_ACCESS_TECHNOLOGY_GERAN = 1
E_RADIO_ACCESS_TECHNOLOGY_WLAN = 2

AVP_VODAFONE_VODAFONE_REPORTING_REASON = 261
E_VODAFONE_REPORTING_REASON_THRESHOLD = 0
E_VODAFONE_REPORTING_REASON_QHT = 1
E_VODAFONE_REPORTING_REASON_FINAL = 2
E_VODAFONE_REPORTING_REASON_QUOTA_EXHAUSTED = 3
E_VODAFONE_REPORTING_REASON_VALIDITY_TIME = 4
E_VODAFONE_REPORTING_REASON_OTHER_QUOTA_TYPE = 5
E_VODAFONE_REPORTING_REASON_RATING_CONDITION_CHANGE = 6
E_VODAFONE_REPORTING_REASON_FORCED_REAUTHORISATION = 7

AVP_VODAFONE_VODAFONE_RULEBASE_ID = 262
AVP_VODAFONE_VODAFONE_TIME_OF_FIRST_USAGE = 263
AVP_VODAFONE_VODAFONE_TIME_OF_LAST_USAGE = 264
AVP_VODAFONE_VODAFONE_TRIGGER = 265
AVP_VODAFONE_VODAFONE_TRIGGER_TYPE = 266
E_VODAFONE_TRIGGER_TYPE_CHANGE_IN_SGSN_IP_ADDRESS = 1
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_ANY = 2
E_VODAFONE_TRIGGER_TYPE_CHANGEINLOCATION_ANY = 3
E_VODAFONE_TRIGGER_TYPE_CHANGEINRAT = 4
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_TRAFFIC_CLASS = 10
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_RELIABILITY_CLASS = 11
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_DELAY_CLASS = 12
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_PEAK_THROUGHPUT = 13
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_PRECEDENCE_CLASS = 14
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_MEAN_THROUGHPUT = 15
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_MAXIMUM_BIT_RATE_FOR_UPLINK = 16
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_MAXIMUM_BIT_RATE_FOR_DOWNLINK = 17
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_RESIDUAL_BER = 18
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_SDU_ERROR_RATIO = 19
E_VODAFONE_TRIGGER_TYPE_CHANGEINQOS_TRANSFER_DELAY = 20

AVP_VODAFONE_USER_LOCATION_INFORMATION = 267
AVP_VODAFONE_VODAFONE_VOLUME_QUOTA_THRESHOLD = 268
AVP_ALU_DETAILED_RESULT = 15
AVP_ALU_DETAILED_RESULT_CAUSE = 16
AVP_ALU_DETAILED_RESULT_CODE = 17
AVP_ALU_DPA_INSTRUCTION = 1016
E_DPA_INSTRUCTION_SESSION_CREATED = 0
E_DPA_INSTRUCTION_IP_ADDRESS_UPDATED = 1
E_DPA_INSTRUCTION_AUX_GX_ESTABLISHMENT = 2

AVP_ALU_CHARGING_POLICY_REPORT = 1134
AVP_ALU_POLICY_COUNTER = 1135
AVP_ALU_POLICY_COUNTER_ID = 1136
AVP_ALU_POLICY_COUNTER_VALUE = 1137
AVP_ALU_SUBSCRIBER_USER_ID = 1139
AVP_NOKIASOLUTIONSANDNETWORKS_USER_AGENT_TYPE = 2016
E_USER_AGENT_TYPE_NON_MOBILE_BROWSER = 0
E_USER_AGENT_TYPE_MOBILE_BROWSER = 1

AVP_HP_PEER_STATE = 200
E_PEER_STATE_DOWN = 0
E_PEER_STATE_UP = 1

AVP_HP_PEER_NAME = 201
AVP_HP_PEER_IDENTITY = 202
AVP_HP_PEER_STATE_CHANGE = 203
AVP_HP_MORE_PEERS = 204
AVP_HP_PEER_TYPE = 205
E_PEER_TYPE_CLIENT = 0
E_PEER_TYPE_SERVER = 1
E_PEER_TYPE_CLIENTANDSERVER = 2
E_PEER_TYPE_PROXY = 3

AVP_ERICSSON_ACC_SERVICE_TYPE = 261
E_ACC_SERVICE_TYPE_AUDIO_CONFERENCE = 0
E_ACC_SERVICE_TYPE_VIDEO_CONFERENCE = 1

AVP_ERICSSON_SSO_STATUS = 280
E_SSO_STATUS_NON_VALID = 0
E_SSO_STATUS_NON_TRUSTED = 1

AVP_ERICSSON_IMS_SERVICE_IDENTIFICATION = 284
AVP_ERICSSON_ERICSSON_SERVICE_INFORMATION = 285
AVP_ERICSSON_ACTIVITY_INFORMATION = 288
AVP_ERICSSON_FEATURE_TAG = 289
AVP_ERICSSON_RULE_SPACE_SUGGESTION = 290
AVP_ERICSSON_RULE_SPACE_DECISION = 291
AVP_ERICSSON_BEARER_CONTROL_OPTIONS = 292
AVP_ERICSSON_PDP_CONTEXT_TYPE = 321
AVP_ERICSSON_ACTIVE_TIME_REPORTING = 322
E_ACTIVE_TIME_REPORTING_NO_REPORTING = 0
E_ACTIVE_TIME_REPORTING_REPORT_TIMESTAMP = 1
E_ACTIVE_TIME_REPORTING_REPORT_TIME_STAMP_VOLUME = 2

AVP_ERICSSON_ACTIVE_TIME_REPORT = 323
AVP_ERICSSON_ACTIVE_TIME_REPORT_START_TIME = 324
AVP_ERICSSON_ACTIVE_TIME_REPORT_END_TIME = 325
AVP_ERICSSON_TIME_QUOTA_MEASUREMENT = 326
AVP_ERICSSON_TIME_QUOTA_METHOD = 327
E_TIME_QUOTA_METHOD_DURATION = 1
E_TIME_QUOTA_METHOD_INACTIVITY_INCLUDED = 2
E_TIME_QUOTA_METHOD_INACTIVITY = 3
E_TIME_QUOTA_METHOD_ACTIVE_PERIODS = 4

AVP_ERICSSON_TIME_QUOTA_RESOLUTION = 328
AVP_ERICSSON_TIME_QUOTA_INACTIVITY_TIME = 329
AVP_ERICSSON_USER_REDIRECTED = 330
E_USER_REDIRECTED_APPEND_URL = 1

AVP_ERICSSON_URL_MODIFIER = 332
E_URL_MODIFIER_APPEND_URL = 1

AVP_ERICSSON_GPRS_ROAMING_STATUS = 333
E_GPRS_ROAMING_STATUS_HOME = 0
E_GPRS_ROAMING_STATUS_VISITED = 1

AVP_ERICSSON_ERICSSON_WILDCARDED_PUBLIC_IDENTITY = 334
AVP_ERICSSON_SIP_RINGING_TIMESTAMP = 338
AVP_ERICSSON_SCAP_CURRENCY_CODE = 544
AVP_ERICSSON_SCAP_SUBSCRIPTION_ID = 553
AVP_ERICSSON_SCAP_SUBSCRIPTION_ID_DATA = 554
AVP_ERICSSON_SCAP_SUBSCRIPTION_ID_TYPE = 555
E_SCAP_SUBSCRIPTION_ID_TYPE_END_USER_MSISDN = 0
E_SCAP_SUBSCRIPTION_ID_TYPE_END_USER_IMSI = 1
E_SCAP_SUBSCRIPTION_ID_TYPE_END_USER_SIP_URI = 2
E_SCAP_SUBSCRIPTION_ID_TYPE_END_USER_NAI = 3
E_SCAP_SUBSCRIPTION_ID_TYPE_END_USER_PRIVATE_OPERATOR_DEFINED = 4

AVP_ERICSSON_ORIGINAL_SUBSCRIPTION_ID = 559
AVP_ERICSSON_ABNORMAL_TERMINATION_REASON = 600
E_ABNORMAL_TERMINATION_REASON_SERVICE_ELEMENT_TERMINATION = 0
E_ABNORMAL_TERMINATION_REASON_CONNECTION_TO_USER_BROKEN = 1

AVP_ERICSSON_SCAP_FINAL_UNIT_INDICATION = 601
AVP_ERICSSON_SCAP_GRANTED_SERVICE_UNIT = 602
AVP_ERICSSON_COST = 603
AVP_ERICSSON_SCAP_COST_INFORMATION = 604
AVP_ERICSSON_ACCOUNTING_CORRELATION_ID = 605
AVP_ERICSSON_SCAP_REQUESTED_SERVICE_UNIT = 606
AVP_ERICSSON_SCAP_SERVICE_PARAMETER_INFO = 607
AVP_ERICSSON_SCAP_SERVICE_PARAMETER_TYPE = 608
E_SCAP_SERVICE_PARAMETER_TYPE_SERVICE_PROVIDER_ID_CCN = 0
E_SCAP_SERVICE_PARAMETER_TYPE_EXTENSION_NUMBER_1_CCN = 1
E_SCAP_SERVICE_PARAMETER_TYPE_EXTENSION_NUMBER_2_CCN = 2
E_SCAP_SERVICE_PARAMETER_TYPE_EXTENSION_NUMBER_3_CCN = 3
E_SCAP_SERVICE_PARAMETER_TYPE_EXTENSION_NUMBER_4_CCN = 4
E_SCAP_SERVICE_PARAMETER_TYPE_EXTENSION_TEXT_CCN = 5
E_SCAP_SERVICE_PARAMETER_TYPE_GPRS_QUALITY_OF_SERVICE_CCN = 6
E_SCAP_SERVICE_PARAMETER_TYPE_REDIRECTING_PARTY_NUMBER_CCN = 7
E_SCAP_SERVICE_PARAMETER_TYPE_ORIGINATING_LOCATION_INFORMATION_CCN = 8
E_SCAP_SERVICE_PARAMETER_TYPE_TERMINATING_LOCATION_INFORMATION_CCN = 9
E_SCAP_SERVICE_PARAMETER_TYPE_REGION_CHARGING_ORIGIN_CCN = 10
E_SCAP_SERVICE_PARAMETER_TYPE_SUBSCRIPTION_TYPE_CCN = 11
E_SCAP_SERVICE_PARAMETER_TYPE_SMS_DELIVERY_STATUS_CCN = 12
E_SCAP_SERVICE_PARAMETER_TYPE_TIME_ZONE_CCN = 13
E_SCAP_SERVICE_PARAMETER_TYPE_TRAFFIC_CASE_CCN = 14
E_SCAP_SERVICE_PARAMETER_TYPE_DEDICATED_ACCOUNT_ID_CCN = 15
E_SCAP_SERVICE_PARAMETER_TYPE_CALLING_PARTY_NUMBER_CCN = 16
E_SCAP_SERVICE_PARAMETER_TYPE_CALLED_PARTY_NUMBER_CCN = 17
E_SCAP_SERVICE_PARAMETER_TYPE_TELE_SERVICE_CODE_CCN = 18
E_SCAP_SERVICE_PARAMETER_TYPE_SERVICE_KEY_CCN = 19

AVP_ERICSSON_SCAP_SERVICE_PARAMETER_VALUE = 609
AVP_ERICSSON_SCAP_EVENT_TIMESTAMP = 610
AVP_ERICSSON_SCAP_UNIT_TYPE = 611
E_SCAP_UNIT_TYPE_SERVICE_CREDIT_TIME_SECONDS = 0
E_SCAP_UNIT_TYPE_SERVICE_CREDIT_VOLUME_BYTES = 1
E_SCAP_UNIT_TYPE_SERVICE_CREDIT_EVENTS_NUMBER_OF_EVENTS = 2
E_SCAP_UNIT_TYPE_SERIVCE_CREDIT_MONEY_MONETARY_VALUE = 3

AVP_ERICSSON_SCAP_UNIT_VALUE = 612
AVP_ERICSSON_SCAP_USED_SERVICE_UNIT = 613
AVP_ERICSSON_SCAP_CHECK_BALANCE_RESULT = 614
E_SCAP_CHECK_BALANCE_RESULT_ENOUGH_CREDIT = 0
E_SCAP_CHECK_BALANCE_RESULT_NO_CREDIT = 1

AVP_ERICSSON_SCAP_REQUESTED_ACTION = 615
E_SCAP_REQUESTED_ACTION_DIRECT_DEBITING = 0
E_SCAP_REQUESTED_ACTION_REFUND_ACCOUNT = 1
E_SCAP_REQUESTED_ACTION_CHECK_BALANCE = 2
E_SCAP_REQUESTED_ACTION_PRICE_ENQUIRY = 3

AVP_ERICSSON_SCAP_EXPONENT = 616
AVP_ERICSSON_SCAP_VALUE_DIGITS = 617
AVP_ERICSSON_REQUESTED_SERVICE = 701
AVP_NONE_S_CSCF_NAME_ORIGINATING = 1008
AVP_NONE_S_CSCF_NAME_TERMINATING = 1009
AVP_NONE_SIP_SERVER_NAME = 1012
AVP_ERICSSON_CHARGING_RULE_AUTHORIZATION = 1055
AVP_ERICSSON_AUTHORIZATION_STATE = 1056
E_AUTHORIZATION_STATE_AUTHORIZED = 0
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_CALENDAR_TIME = 1
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_ROAMING = 2
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_QOS = 3
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_BLACKLISTING = 4
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_TERMINAL_LIMITATIONS = 5
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_USER_DEFINED_REASON_1 = 6
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_USER_DEFINED_REASON_2 = 7
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_USER_DEFINED_REASON_3 = 8
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_USER_DEFINED_REASON_4 = 9
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_USER_DEFINED_REASON_5 = 10
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_UNKNOWN_REASON = 11
E_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_USAGE_REPORTING_OVER_GX = 12

AVP_ERICSSON_AUTHORIZATION_STATE_CHANGE_TIME = 1057
AVP_ERICSSON_NEXT_AUTHORIZATION_STATE = 1059
E_NEXT_AUTHORIZATION_STATE_AUTHORIZED = 0
E_NEXT_AUTHORIZATION_STATE_UNAUTHORIZED_DUE_TO_CALENDAR_TIME = 1

AVP_ERICSSON_GX_CAPABILITY_LIST = 1060
AVP_ERICSSON_MMT_INFORMATION = 1061
AVP_ERICSSON_RESULT_CODE_EXTENSION = 1067
AVP_ERICSSON_SUBSCRIPTION_ID_LOCATION = 1074
AVP_ERICSSON_OTHER_PARTY_ID = 1075
AVP_ERICSSON_OTHER_PARTY_ID_NATURE = 1076
E_OTHER_PARTY_ID_NATURE_UNKNOWN = 0
E_OTHER_PARTY_ID_NATURE_INTERNATIONAL = 1
E_OTHER_PARTY_ID_NATURE_NATIONAL = 2
E_OTHER_PARTY_ID_NATURE_SUBSCRIBER_NUMBER = 4
E_OTHER_PARTY_ID_NATURE_ALPHANUMERIC = 5

AVP_ERICSSON_OTHER_PARTY_ID_DATA = 1077
AVP_ERICSSON_OTHER_PARTY_ID_TYPE = 1078
E_OTHER_PARTY_ID_TYPE_END_USER_MSISDN = 0
E_OTHER_PARTY_ID_TYPE_END_USER_IMSI = 1
E_OTHER_PARTY_ID_TYPE_END_USER_SIP_URI = 2
E_OTHER_PARTY_ID_TYPE_END_USER_NAI = 3
E_OTHER_PARTY_ID_TYPE_END_USER_PRIVATE = 4

AVP_ERICSSON_SERVICE_PROVIDER_ID = 1081
AVP_ERICSSON_TRAFFIC_CASE = 1082
AVP_ERICSSON_RELATED_ICID = 1128
AVP_ERICSSON_SUPPLEMENTARY_SERVICE_INFORMATION = 1129
AVP_ERICSSON_SUPPLEMENTARY_SERVICE_IDENTITY = 1130
AVP_ERICSSON_SUPPLEMENTARY_SERVICE_ACTION = 1131
AVP_ERICSSON_REDIRECTING_PARTY_ADDRESS = 1133
AVP_ERICSSON_CALLING_PARTY_ADDRESS_PRESENTATION_STATUS = 1141
E_CALLING_PARTY_ADDRESS_PRESENTATION_STATUS_PRESENTATION_ALLOWED = 0
E_CALLING_PARTY_ADDRESS_PRESENTATION_STATUS_PRESENTATION_RESTRICTED = 1
E_CALLING_PARTY_ADDRESS_PRESENTATION_STATUS_UNKNOWN = 2

AVP_ERICSSON_CALLED_ASSERTED_IDENTITY_PRESENTATION_STATUS = 1142
E_CALLED_ASSERTED_IDENTITY_PRESENTATION_STATUS_PRESENTATION_ALLOWED = 0
E_CALLED_ASSERTED_IDENTITY_PRESENTATION_STATUS_PRESENTATION_RESTRICTED = 1

AVP_ERICSSON_CREDIT_INSTANCE_ID = 1143
AVP_ERICSSON_SERVICE_START_TIMESTAMP = 1144
AVP_ERICSSON_CUMULATIVE_USED_SERVICE_UNIT = 1145
AVP_ERICSSON_ERICSSON_CUSTOMER_ID = 1146
AVP_ERICSSON_FROM_HEADER = 1153
AVP_ERICSSON_SECONDARY_DIGEST_HA1 = 1192
AVP_ERICSSON_SIP_RINGING_TIMESTAMP_FRACTION = 1256
AVP_ERICSSON_AUTHENTICATION_METHOD = 1261
E_AUTHENTICATION_METHOD_NOAUTHENTICATION = 0
E_AUTHENTICATION_METHOD_AKAAUTHENTICATION = 1
E_AUTHENTICATION_METHOD_NASSBUNDLEDAUTHENTICATION = 2
E_AUTHENTICATION_METHOD_DIGESTAUTHENTICATION = 3
E_AUTHENTICATION_METHOD_SSOAUTHENTICATION = 4

AVP_ERICSSON_FROM_HEADER_PRESENTATION_STATUS = 1262
E_FROM_HEADER_PRESENTATION_STATUS_PRESENTATION_ALLOWED = 0
E_FROM_HEADER_PRESENTATION_STATUS_PRESENTATION_RESTRICTED = 1
E_FROM_HEADER_PRESENTATION_STATUS_UNKNOWN = 2

AVP_ERICSSON_TRANSACTION_INFO = 1264
AVP_ERICSSON_TRANSACTION_TYPE = 1265
E_TRANSACTION_TYPE_SIP_REQUEST = 0
E_TRANSACTION_TYPE_SIP_RESPONSE = 1
E_TRANSACTION_TYPE_DIAMETER_REQUEST = 2
E_TRANSACTION_TYPE_DIAMETER_ANSWER = 3

AVP_ERICSSON_TRANSACTION_DATA_NAME = 1266
AVP_ERICSSON_TRANSACTION_DATA_VALUE = 1267
AVP_ERICSSON_CHARGING_PROFILE_ID = 1268
AVP_ERICSSON_SERVICE_NUMBER_TYPE = 1307
E_SERVICE_NUMBER_TYPE_OSN = 0
E_SERVICE_NUMBER_TYPE_NSN = 1
E_SERVICE_NUMBER_TYPE_NON_SERVICE_E164 = 2
E_SERVICE_NUMBER_TYPE_UNKNOWN = 3

AVP_ERICSSON_SCC_SERVICE_IDENTITY = 1314
E_SCC_SERVICE_IDENTITY_T_ADS = 0
E_SCC_SERVICE_IDENTITY_SDS = 100
E_SCC_SERVICE_IDENTITY_SRVCC = 200

AVP_ERICSSON_SDD_TADS_DECISION = 1315
E_SDD_TADS_DECISION_NO_SELECTION = 0
E_SDD_TADS_DECISION_NO_SELECTION_AND_REJECT = 1
E_SDD_TADS_DECISION_VOLTE_UE_ON_PS = 2
E_SDD_TADS_DECISION_VOLTE_UE_ON_CS = 3
E_SDD_TADS_DECISION_VOLTE_UE_ON_PS_OR_PS = 4
E_SDD_TADS_DECISION_BREAKOUT_TO_CS = 5
E_SDD_TADS_DECISION_FIXED_UE_ON_PS_AND_BREAKOUT_TO_CS = 6

AVP_ERICSSON_SERVED_USER = 1330
AVP_ERICSSON_TENANT = 1380
AVP_ERICSSON_UHTZ_OFFSET = 1388
AVP_ERICSSON_AS_TYPE = 1433
E_AS_TYPE_MMTEL_AS = 0
E_AS_TYPE_REDIRECTION_AS = 1
E_AS_TYPE_SCHEDULED_CONFERENCE_AS = 2
E_AS_TYPE_SCC_AS = 3
E_AS_TYPE_PARLAY_X_AS = 4
E_AS_TYPE_ST_AS = 5

AVP_ERICSSON_TRANSACTION_SIP_MESSAGE = 1460
AVP_CISCO_CISCO_FLOW_DESCRIPTION_507 = 507
AVP_CISCO_CISCO_CHARGING_RULE_DEFINITION = 131072
AVP_CISCO_CONTENT_DEFINITION = 131073
AVP_CISCO_BILLING_POLICY_DEFINITION = 131074
AVP_CISCO_POLICY_MAP_DEFINITION = 131075
AVP_CISCO_SERVICE_DEFINITION = 131076
AVP_CISCO_CONTENT_POLICY_MAP = 131077
AVP_CISCO_SERVICE_INFO = 131078
AVP_CISCO_BILLING_PLAN_DEFINITION = 131079
AVP_CISCO_VOLUME_THRESHOLD = 131080
AVP_CISCO_TIME_THRESHOLD = 131081
AVP_CISCO_CONTENT_IDLE_TIMER = 131082
AVP_CISCO_NEXTHOP_UPLINK = 131083
AVP_CISCO_NEXTHOP_DOWNLINK = 131084
AVP_CISCO_L7_PARSE_PROTOCOL_TYPE = 131085
E_L7_PARSE_PROTOCOL_TYPE_HTTP = 0
E_L7_PARSE_PROTOCOL_TYPE_IMAP = 1
E_L7_PARSE_PROTOCOL_TYPE_OTHER = 2
E_L7_PARSE_PROTOCOL_TYPE_POP3 = 3
E_L7_PARSE_PROTOCOL_TYPE_RTSP = 4
E_L7_PARSE_PROTOCOL_TYPE_SMTP = 5
E_L7_PARSE_PROTOCOL_TYPE_WAP_CONNECTION_ORIENTED = 6
E_L7_PARSE_PROTOCOL_TYPE_WAP_CONNECTION_LESS = 7
E_L7_PARSE_PROTOCOL_TYPE_SIP = 8
E_L7_PARSE_PROTOCOL_TYPE_FTP = 9
E_L7_PARSE_PROTOCOL_TYPE_NBAR = 10
E_L7_PARSE_PROTOCOL_TYPE_DNS = 11
E_L7_PARSE_PROTOCOL_TYPE_HTTP_INSERT = 12

AVP_CISCO_SERVICE_STATUS = 131086
AVP_CISCO_SERVICE_NAME = 131087
AVP_CISCO_BILLING_POLICY_NAME = 131088
AVP_CISCO_POLICY_MAP_NAME = 131089
AVP_CISCO_POLICY_MAP_MATCH = 131090
AVP_CISCO_MATCH_STRING = 131091
AVP_CISCO_ATTRIBUTE_STRING = 131092
AVP_CISCO_ONLINE_BILLING_BASIS = 131093
E_ONLINE_BILLING_BASIS_EVENT = 1
E_ONLINE_BILLING_BASIS_IP_BYTE = 2
E_ONLINE_BILLING_BASIS_TCP_BYTE = 3
E_ONLINE_BILLING_BASIS_DURATION = 4
E_ONLINE_BILLING_BASIS_DURATION_CONNECT = 5
E_ONLINE_BILLING_BASIS_DURATION_TRANSACTION = 6

AVP_CISCO_SERVICE_ACTIVATION = 131094
E_SERVICE_ACTIVATION_USER_PROFILE = 0
E_SERVICE_ACTIVATION_AUTOMATIC = 1

AVP_CISCO_CDR_VOLUME_THRESHOLD = 131095
AVP_CISCO_CDR_TIME_THRESHOLD = 131096
AVP_CISCO_ADVICE_OF_CHARGE = 131097
AVP_CISCO_APPEND_URL = 131098
E_APPEND_URL_DISABLE_APPEND_URL = 0
E_APPEND_URL_ENABLE_APPEND_URL = 1

AVP_CISCO_CONFIRM_TOKEN_131099 = 131099
AVP_CISCO_CISCO_SERVICE_CLASS = 131100
AVP_CISCO_SERVICE_IDLE_TIME = 131101
AVP_CISCO_OWNER_ID = 131102
AVP_CISCO_OWNER_NAME = 131103
AVP_CISCO_ONLINE_PASSTHROUGH_QUOTA = 131104
AVP_CISCO_ONLINE_REAUTHORIZATION_THRESHOLD = 131105
AVP_CISCO_ONLINE_REAUTHORIZATION_TIMEOUT = 131106
AVP_CISCO_INITIAL_TIMEOUT = 131107
AVP_CISCO_MAXIMUM_TIMEOUT = 131108
AVP_CISCO_REFUND_POLICY = 131109
AVP_CISCO_METER_EXCLUDE = 131110
E_METER_EXCLUDE_MMS_WAP = 0
E_METER_EXCLUDE_RTSP_PAUSE = 1
E_METER_EXCLUDE_SERVICE_IDLE = 2
E_METER_EXCLUDE_NETWORK_INIT_SIP = 3

AVP_CISCO_METER_INCLUDE_IMAP = 131111
E_METER_INCLUDE_IMAP_BODY_AND_HEADER = 0
E_METER_INCLUDE_IMAP_BODY_ONLY = 1
E_METER_INCLUDE_IMAP_BODY_AND_OTHER = 2

AVP_CISCO_METERING_GRANULARITY = 131112
AVP_CISCO_METER_INCREMENT = 131113
AVP_CISCO_METER_INITIAL = 131114
AVP_CISCO_METER_MINIMUM = 131115
AVP_CISCO_VERIFY = 131116
AVP_CISCO_CONFIRM_TOKEN_131117 = 131117
AVP_CISCO_WEIGHT = 131118
AVP_CISCO_USER_IDLE_TIMER = 131119
AVP_CISCO_POLICY_PRELOAD_REQ_TYPE = 131120
E_POLICY_PRELOAD_REQ_TYPE_POLICY_PRELOAD_REQ = 0
E_POLICY_PRELOAD_REQ_TYPE_POLICY_PRELOAD_RESP = 1
E_POLICY_PRELOAD_REQ_TYPE_POLICY_PRELOAD_PUSH = 2
E_POLICY_PRELOAD_REQ_TYPE_POLICY_PRELOAD_PUSH_ACK = 3

AVP_CISCO_POLICY_PRELOAD_OBJECT_TYPE = 131121
E_POLICY_PRELOAD_OBJECT_TYPE_POLICY_MAP = 0
E_POLICY_PRELOAD_OBJECT_TYPE_BILLING_POLICY = 1
E_POLICY_PRELOAD_OBJECT_TYPE_CONTENT = 2
E_POLICY_PRELOAD_OBJECT_TYPE_SERVICE = 3
E_POLICY_PRELOAD_OBJECT_TYPE_BILLING_PLAN = 4
E_POLICY_PRELOAD_OBJECT_TYPE_DOMAIN_GROUP = 5
E_POLICY_PRELOAD_OBJECT_TYPE_HEADER_INSERT = 6
E_POLICY_PRELOAD_OBJECT_TYPE_HEADER_GROUP = 7
E_POLICY_PRELOAD_OBJECT_TYPE_QOS_PROFILE = 8

AVP_CISCO_POLICY_PRELOAD_STATUS = 131122
E_POLICY_PRELOAD_STATUS_INITIATE = 0
E_POLICY_PRELOAD_STATUS_COMPLETE = 1

AVP_CISCO_CHARGING_RULE_TRIGGER_TYPE = 131123
E_CHARGING_RULE_TRIGGER_TYPE_NO_CHARGING_RULE_EVENT_TRIGGERS = 0
E_CHARGING_RULE_TRIGGER_TYPE_VOLUME_THRESHOLD = 1
E_CHARGING_RULE_TRIGGER_TYPE_TIME_THRESHOLD = 2
E_CHARGING_RULE_TRIGGER_TYPE_SVC_FLOW_DETECTION = 3
E_CHARGING_RULE_TRIGGER_TYPE_CHARGING_RULE_REMOVE = 4

AVP_CISCO_CHARGING_RULE_EVENT = 131124
AVP_CISCO_SERVICE_REPORTING_LEVEL = 131125
E_SERVICE_REPORTING_LEVEL_TRANSACTION = 0
E_SERVICE_REPORTING_LEVEL_SERVICE = 1

AVP_CISCO_ACCOUNTING = 131126
AVP_CISCO_ACCOUNTING_CUSTOMER_STRING = 131127
AVP_CISCO_L7_PARSE_LENGTH = 131128
AVP_CISCO_SERVICE_CDR_THRESHOLD = 131129
AVP_CISCO_INTERMEDIATE_CDR_THRESHOLD = 131130
AVP_CISCO_CDR_GENERATION_DELAY = 131131
AVP_CISCO_REPLICATE_SESSION = 131132
AVP_CISCO_REPLICATE_SESSION_DELAY = 131133
AVP_CISCO_CONTENT_PENDING_TIMER = 131134
AVP_CISCO_OPERATION_STATUS = 131135
E_OPERATION_STATUS_OUT_OF_SERVICE = 0
E_OPERATION_STATUS_IN_SERVICE = 1

AVP_CISCO_SUBSCRIBER_IP_SOURCE = 131136
E_SUBSCRIBER_IP_SOURCE_DEFAULT = 0
E_SUBSCRIBER_IP_SOURCE_HTTP_X_FORWARDED_FOR = 1

AVP_CISCO_NEXTHOP = 131137
AVP_CISCO_NEXTHOP_REVERSE = 131138
AVP_CISCO_CHARGING_RULE_EVENT_TRIGGER = 131139
AVP_CISCO_BILLING_PLAN_NAME = 131140
AVP_CISCO_CONTENT_FLOW_DESCRIPTION = 131141
AVP_CISCO_CONTENT_FLOW_FILTER = 131142
AVP_CISCO_CLIENT_GROUP_ID = 131143
AVP_CISCO_ACL_NUMBER = 131144
AVP_CISCO_ACL_NAME = 131145
AVP_CISCO_DESTINATION_IP_ADDRESS = 131146
AVP_CISCO_DESTINATION_MASK = 131147
AVP_CISCO_PROTOCOL_ID = 131148
AVP_CISCO_START_OF_PORT_RANGE = 131149
AVP_CISCO_END_OF_PORT_RANGE = 131150
AVP_CISCO_CONTENT_NAME = 131151
AVP_CISCO_FAILED_PRELOAD_OBJECT = 131152
AVP_CISCO_VRF_NAME = 131153
AVP_CISCO_VLAN_ID = 131154
AVP_CISCO_VOLUME_USAGE = 131155
AVP_CISCO_CISCO_TIME_USAGE = 131156
AVP_CISCO_DURATION = 131157
AVP_CISCO_FIRST_PACKET_TIMESTANP = 131158
AVP_CISCO_LAST_PACKET_TIMESTANP = 131159
AVP_CISCO_CISCO_FLOW_DESCRIPTION = 131160
AVP_CISCO_TERMINATE_BEARER = 131161
AVP_CISCO_SERVICE_RATING_GROUP = 131162
AVP_CISCO_CONTENT_SCOPE = 131163
E_CONTENT_SCOPE_GLOBAL = 0
E_CONTENT_SCOPE_USER = 1

AVP_CISCO_FLOW_STATUS_POLICY_MISMATCH = 131164
E_FLOW_STATUS_POLICY_MISMATCH_FORWARD = 0
E_FLOW_STATUS_POLICY_MISMATCH_BLOCK = 1

AVP_CISCO_POLICY_MAP_TYPE = 131165
E_POLICY_MAP_TYPE_URL_MAP = 0
E_POLICY_MAP_TYPE_HEADER_MAP = 1
E_POLICY_MAP_TYPE_METHOD_MAP = 2
E_POLICY_MAP_TYPE_ATTRIBUTE_MAP = 3

AVP_CISCO_POLICY_MAP_MATCH_INSTALL = 131166
AVP_CISCO_POLICY_MAP_MATCH_REMOVE = 131167
AVP_CISCO_POLICY_MAP_REPLACE = 131168
E_POLICY_MAP_REPLACE_DISABLED = 0
E_POLICY_MAP_REPLACE_ENABLED = 1

AVP_CISCO_CISCO_FLOW_STATUS = 131169
E_CISCO_FLOW_STATUS_FORWARD = 0
E_CISCO_FLOW_STATUS_BLOCK = 1
E_CISCO_FLOW_STATUS_REDIRECT = 2

AVP_CISCO_SERVICE_QOS = 131170
AVP_CISCO_QOS_RATE_LIMIT_UL = 131171
AVP_CISCO_QOS_RATE_LIMIT_DL = 131172
AVP_CISCO_QOS_RATE_LIMIT = 131173
AVP_CISCO_MAX_BANDWIDTH = 131174
AVP_CISCO_RATE_LIMIT_CONFORM_ACTION = 131175
AVP_CISCO_RATE_LIMIT_EXCEED_ACTION = 131176
AVP_CISCO_CISCO_RATE_LIMIT_ACTION = 131177
E_CISCO_RATE_LIMIT_ACTION_FORWARD = 0
E_CISCO_RATE_LIMIT_ACTION_DROP = 1
E_CISCO_RATE_LIMIT_ACTION_MARK_DSCP = 2

AVP_CISCO_CISCO_DSCP = 131178
AVP_CISCO_POLICY_MAP_INSTALL = 131179
AVP_CISCO_POLICY_MAP_REMOVE = 131180
AVP_CISCO_BILLING_POLICY_INSTALL = 131181
AVP_CISCO_BILLING_POLICY_REMOVE = 131182
AVP_CISCO_CONTENT_INSTALL = 131183
AVP_CISCO_CONTENT_REMOVE = 131184
AVP_CISCO_SERVICE_INSTALL = 131185
AVP_CISCO_SERVICE_REMOVE = 131186
AVP_CISCO_BILLING_PLAN_INSTALL = 131187
AVP_CISCO_BILLING_PLAN_REMOVE = 131188
AVP_CISCO_POLICY_PRELOAD_ERROR_CODE = 131189
E_POLICY_PRELOAD_ERROR_CODE_INCONSISTENT_PRELOAD_DATA = 0
E_POLICY_PRELOAD_ERROR_CODE_MANDATORY_AVP_MISSING = 1
E_POLICY_PRELOAD_ERROR_CODE_FAILURE_TO_ENFORCE = 2
E_POLICY_PRELOAD_ERROR_CODE_WRONG_ORDER = 3
E_POLICY_PRELOAD_ERROR_CODE_CONFLICT_WITH_STATIC_CONFIG = 4

AVP_CISCO_MAX_BURST_SIZE = 131190
AVP_CISCO_FAILED_PRELOAD_OBJ_NAME = 131191
AVP_CISCO_CISCO_EVENT_TRIGGER_TYPE = 131192
E_CISCO_EVENT_TRIGGER_TYPE_NO_CISCO_TRIGGERS = 0
E_CISCO_EVENT_TRIGGER_TYPE_TCP_SYN_DETECTION = 1
E_CISCO_EVENT_TRIGGER_TYPE_VOLUME_THRESHOLD = 2
E_CISCO_EVENT_TRIGGER_TYPE_TIME_THRESHOLD = 3
E_CISCO_EVENT_TRIGGER_TYPE_USER_AGENT_DETECTION = 4
E_CISCO_EVENT_TRIGGER_TYPE_VOLUME_THRESHOLD_64 = 5

AVP_CISCO_CISCO_EVENT_TRIGGER = 131193
AVP_CISCO_TCP_SYN = 131194
AVP_CISCO_CISCO_EVENT = 131195
AVP_CISCO_INTERLEAVED = 131196
E_INTERLEAVED_DISABLED = 0
E_INTERLEAVED_ENABLED = 1

AVP_CISCO_CONTROL_URL = 131197
AVP_CISCO_RELATIVE_URL = 131198
E_RELATIVE_URL_DISABLED = 0
E_RELATIVE_URL_ENABLED = 1

AVP_CISCO_MINING = 131199
E_MINING_DISABLED = 0
E_MINING_ENABLED = 1

AVP_CISCO_USER_DEFAULT = 131200
E_USER_DEFAULT_DISABLED = 0
E_USER_DEFAULT_ENABLED = 1

AVP_CISCO_CISCO_PRIORITY = 131201
AVP_CISCO_DOMAIN_GROUP_NAME = 131202
AVP_CISCO_DOMAIN_GROUP_DEFINITION = 131203
AVP_CISCO_DOMAIN_GROUP_INSTALL = 131204
AVP_CISCO_DOMAIN_GROUP_REMOVE = 131205
AVP_CISCO_DOMAIN_GROUP_ACTIVATION = 131206
E_DOMAIN_GROUP_ACTIVATION_DISABLED = 0
E_DOMAIN_GROUP_ACTIVATION_ENABLED = 1

AVP_CISCO_DUAL_BILLING_BASIS = 131207
E_DUAL_BILLING_BASIS_EVENT = 1
E_DUAL_BILLING_BASIS_IP_BYTE = 2
E_DUAL_BILLING_BASIS_TCP_BYTE = 3
E_DUAL_BILLING_BASIS_DURATION = 4
E_DUAL_BILLING_BASIS_DURATION_CONNECT = 5
E_DUAL_BILLING_BASIS_DURATION_TRANSACTION = 6

AVP_CISCO_DUAL_PASSTHROUGH_QUOTA = 131208
AVP_CISCO_DUAL_REAUTHORIZATION_THRESHOLD = 131209
AVP_CISCO_VIRTUAL_ONLINE = 131210
E_VIRTUAL_ONLINE_DISABLED = 0
E_VIRTUAL_ONLINE_ENABLED = 1

AVP_CISCO_NEXTHOP_MEDIA = 131211
AVP_CISCO_NEXTHOP_OVERRIDE = 131212
E_NEXTHOP_OVERRIDE_DISABLED = 0
E_NEXTHOP_OVERRIDE_ENABLED = 1

AVP_CISCO_CISCO_QUOTA_CONSUMPTION_TIME = 131213
AVP_CISCO_CLASS_MAP_NAME = 131214
AVP_CISCO_HEADER_GROUP_NAME = 131215
AVP_CISCO_HEADER_GROUP_DEFINITION = 131216
AVP_CISCO_HEADER_GROUP_INSTALL = 131217
AVP_CISCO_HEADER_GROUP_REMOVE = 131218
AVP_CISCO_HEADER_INSERT_NAME = 131219
AVP_CISCO_HEADER_FIELD_NAME = 131220
AVP_CISCO_HEADER_CLASS_NAME = 131221
AVP_CISCO_HEADER_CLASS_MODE = 131222
E_HEADER_CLASS_MODE_EXCLUDE = 0
E_HEADER_CLASS_MODE_INCLUDE = 1

AVP_CISCO_HEADER_CLASS = 131223
AVP_CISCO_RADIUS_ATTRIBUTE_TYPE = 131224
AVP_CISCO_RADIUS_VSA_VENDOR_ID = 131225
AVP_CISCO_RADIUS_VSA_SUBATTRIBUTE_TYPE = 131226
AVP_CISCO_HEADER_ITEM_RADIUS = 131227
AVP_CISCO_HEADER_ITEM = 131228
E_HEADER_ITEM_TIMESTAMP = 0
E_HEADER_ITEM_QUOTA_SERVER = 1

AVP_CISCO_HEADER_ITEM_STRING = 131229
AVP_CISCO_HEADER_ITEMS_ENCRYPTED = 131230
AVP_CISCO_HEADER_INSERT_DEFINITION = 131231
AVP_CISCO_HEADER_INSERT_INSTALL = 131232
AVP_CISCO_HEADER_INSERT_REMOVE = 131233
AVP_CISCO_USER_IDLE_POD = 131234
E_USER_IDLE_POD_DISABLED = 0
E_USER_IDLE_POD_ENABLED = 1

AVP_CISCO_DOMAIN_GROUP_CLEAR = 131235
E_DOMAIN_GROUP_CLEAR_DISABLED = 0
E_DOMAIN_GROUP_CLEAR_ENABLED = 1

AVP_CISCO_CISCO_QOS_PROFILE_NAME = 131236
AVP_CISCO_CISCO_QOS_PROFILE = 131237
AVP_CISCO_CISCO_QOS_PROFILE_INSTALL = 131238
AVP_CISCO_CISCO_QOS_PROFILE_REMOVE = 131239
AVP_CISCO_CISCO_QOS_PROFILE_UPLINK = 131240
AVP_CISCO_CISCO_QOS_PROFILE_DOWNLINK = 131241
AVP_CISCO_HEADER_ITEM_ENCRYPTION = 131242
E_HEADER_ITEM_ENCRYPTION_UNENCRYPTED = 0
E_HEADER_ITEM_ENCRYPTION_ENCRYPTED = 1

AVP_CISCO_SERVICE_GROUP_NAME = 131243
AVP_CISCO_SERVICE_GROUP_DEFINITION = 131244
AVP_CISCO_SERVICE_GROUP_INSTALL = 131245
AVP_CISCO_SERVICE_GROUP_REMOVE = 131246
AVP_CISCO_SERVICE_GROUP_EVENT = 131247
AVP_CISCO_CISCO_REPORT_USAGE = 131248
AVP_CISCO_ACCEL = 131249
E_ACCEL_DISABLE = 0
E_ACCEL_ENABLE = 1

AVP_CISCO_CISCO_ANSWER_USER_USAGE = 131250
AVP_CISCO_CISCO_REQUEST_USAGE_TYPE = 131251
E_CISCO_REQUEST_USAGE_TYPE_NO_USAGE = 0
E_CISCO_REQUEST_USAGE_TYPE_VOL_USAGE = 1
E_CISCO_REQUEST_USAGE_TYPE_TIME_USAGE = 2

AVP_CISCO_CISCO_REQUEST_CHARGING_RULE_USAGE = 131252
AVP_CISCO_CISCO_REQUEST_SERVICE_GROUP_USAGE = 131253
AVP_CISCO_CISCO_ANSWER_CHARGING_RULE_USAGE = 131254
AVP_CISCO_CISCO_ANSWER_SERVICE_GROUP_USAGE = 131255
AVP_CISCO_USER_AGENT = 131256
AVP_CISCO_SERVICE_LIFE_TIME = 131257
AVP_CISCO_VOLUME_THRESHOLD_64 = 131258
AVP_CISCO_DELEGATED_IP_INSTALL = 131259
AVP_CISCO_DELEGATED_IPV4_DEFINITION = 131260
AVP_CISCO_DELEGATED_IPV6_DEFINITION = 131261
AVP_CISCO_AGGR_PREFIX_LEN = 131262
AVP_CISCO_SERVICE_IDENTIFIER_LO = 131263
AVP_CISCO_SERVICE_IDENTIFIER_HI = 131264
AVP_CISCO_SERVICE_IDENTIFIER_RANGE = 131265
AVP_STARENT_ACCESS_NETWORK_CHARGING_PHYSICAL_ACCESS_ID = 1472
AVP_STARENT_ACCESS_NETWORK_CHARGING_PHYSICAL_ACCESS_ID_REALM = 1474
AVP_STARENT_ACCESS_NETWORK_CHARGING_PHYSICAL_ACCESS_ID_VALUE = 1473
AVP_STARENT_SN_CUSTOMER_ID = 1146
AVP_STARENT_SN_ABSOLUTE_VALIDITY_TIME = 505
AVP_STARENT_SN_BANDWIDTH_CONTROL = 512
E_SN_BANDWIDTH_CONTROL_HIGH = 0
E_SN_BANDWIDTH_CONTROL_LOW = 1

AVP_STARENT_SN_CF_POLICY_ID = 529
AVP_STARENT_SN_CHARGING_COLLECTION_FUNCTION_NAME = 530
AVP_STARENT_SN_CHARGING_ID = 525
AVP_STARENT_SN_FAST_REAUTH_USERNAME_11010 = 11010
AVP_STARENT_SN_FIREWALL_POLICY_515 = 515
AVP_STARENT_SN_MONITORING_KEY = 518
AVP_STARENT_SN_PHASE0_PSAPNAME = 523
AVP_STARENT_SN_PSEUDONYM_USERNAME_11011 = 11011
AVP_STARENT_SN_REMAINING_SERVICE_UNIT = 526
AVP_STARENT_SN_RULEBASE_ID = 528
AVP_STARENT_SN_SERVICE_FLOW_DETECTION = 520
E_SN_SERVICE_FLOW_DETECTION_ENABLE_DETECTION = 0

AVP_STARENT_SN_SERVICE_START_TIMESTAMP = 527
AVP_STARENT_SN_TIME_QUOTA_THRESHOLD = 503
AVP_STARENT_SN_TOTAL_USED_SERVICE_UNIT = 504
AVP_STARENT_SN_TRAFFIC_POLICY = 514
AVP_STARENT_SN_TRANSPARENT_DATA = 513
AVP_STARENT_SN_UNIT_QUOTA_THRESHOLD = 502
AVP_STARENT_SN_USAGE_MONITORING = 521
E_SN_USAGE_MONITORING_USAGE_MONITORING_DISABLED = 0
E_SN_USAGE_MONITORING_USAGE_MONITORING_ENABLED = 1

AVP_STARENT_SN_USAGE_MONITORING_CONTROL = 517
AVP_STARENT_SN_USAGE_VOLUME = 519
AVP_STARENT_SN_VOLUME_QUOTA_THRESHOLD = 501
AVP_STARENT_SN_SESSION_START_INDICATOR = 522
AVP_STARENT_STARENT_SUBSCRIBER_PERMISSION = 20
E_STARENT_SUBSCRIBER_PERMISSION_NONE = 0
E_STARENT_SUBSCRIBER_PERMISSION_SIMPLE_IP = 1
E_STARENT_SUBSCRIBER_PERMISSION_MOBILE_IP = 2
E_STARENT_SUBSCRIBER_PERMISSION_SIMPLE_IP_MOBILE_IP = 3
E_STARENT_SUBSCRIBER_PERMISSION_HA_MOBILE_IP = 4
E_STARENT_SUBSCRIBER_PERMISSION_SIMPLE_IP_HA_MOBILE_IP = 5
E_STARENT_SUBSCRIBER_PERMISSION_MOBILE_IP_HA_MOBILE_IP = 6
E_STARENT_SUBSCRIBER_PERMISSION_SIP_MIP_HA_MIP = 7
E_STARENT_SUBSCRIBER_PERMISSION_GGSN_PDP_TYPE_IP = 8
E_STARENT_SUBSCRIBER_PERMISSION_GGSN_PDP_TYPE_PPP = 16
E_STARENT_SUBSCRIBER_PERMISSION_NETWORK_MOBILITY = 32
E_STARENT_SUBSCRIBER_PERMISSION_FA_HA_NEMO = 38
E_STARENT_SUBSCRIBER_PERMISSION_ALL = 63

AVP_STARENT_PROHIBIT_PAYLOAD_COMPRESSION = 237
E_PROHIBIT_PAYLOAD_COMPRESSION_ALLOWED = 0
E_PROHIBIT_PAYLOAD_COMPRESSION_PROHIBITED = 1

AVP_STARENT_SN_ACCESS_LINK_IP_FRAG = 63
E_SN_ACCESS_LINK_IP_FRAG_NORMAL = 0
E_SN_ACCESS_LINK_IP_FRAG_DF_IGNORE = 1
E_SN_ACCESS_LINK_IP_FRAG_DF_FRAGMENT_ICMP_NOTIFY = 2

AVP_STARENT_SN_ACCT_INPUT_GIGA_DROPPED = 230
AVP_STARENT_SN_ACCT_INPUT_OCTETS_DROPPED = 228
AVP_STARENT_SN_ACCT_INPUT_PACKETS_DROPPED = 226
AVP_STARENT_SN_ACCT_OUTPUT_GIGA_DROPPED = 231
AVP_STARENT_SN_ACCT_OUTPUT_OCTETS_DROPPED = 229
AVP_STARENT_SN_ACCT_OUTPUT_PACKETS_DROPPED = 227
AVP_STARENT_SN_ACS_CREDIT_CONTROL_GROUP = 301
AVP_STARENT_SN_ADMIN_EXPIRY = 204
AVP_STARENT_SN_ADMIN_PERMISSION = 21
E_SN_ADMIN_PERMISSION_NONE = 0
E_SN_ADMIN_PERMISSION_CLI = 1
E_SN_ADMIN_PERMISSION_FTP = 2
E_SN_ADMIN_PERMISSION_CLI_FTP = 3
E_SN_ADMIN_PERMISSION_INTERCEPT = 4
E_SN_ADMIN_PERMISSION_CLI_INTERCEPT = 5
E_SN_ADMIN_PERMISSION_CLI_INTERCEPT_FTP = 7
E_SN_ADMIN_PERMISSION_ECS = 8
E_SN_ADMIN_PERMISSION_CLI_ECS = 9
E_SN_ADMIN_PERMISSION_CLI_FTP_ECS = 11
E_SN_ADMIN_PERMISSION_CLI_INTERCEPT_ECS = 13
E_SN_ADMIN_PERMISSION_CLI_INTERCEPT_FTP_ECS = 15

AVP_STARENT_SN_ASSIGNED_VLAN_ID = 152
AVP_STARENT_SN_AUTHORISED_QOS = 266
AVP_STARENT_SN_BANDWIDTH_POLICY = 300
AVP_STARENT_SN_CALL_ID = 251
AVP_STARENT_SN_CAUSE_CODE = 267
E_SN_CAUSE_CODE_NORMAL_END_OF_SESSION = 0
E_SN_CAUSE_CODE_SUCCESSFUL_TRANSACTION = 1
E_SN_CAUSE_CODE_END_OF_SUBSCRIBER_DIALOG = 2
E_SN_CAUSE_CODE_3XX_REDIRECTION = 3
E_SN_CAUSE_CODE_4XX_REQUEST_FAILURE = 4
E_SN_CAUSE_CODE_5XX_SERVER_FAILURE = 5
E_SN_CAUSE_CODE_6XX_GLOBAL_FAILURE = 6
E_SN_CAUSE_CODE_UNSPECIFIED_ERROR = 7
E_SN_CAUSE_CODE_UNSUCCESSFUL_SESSION_SETUP = 8
E_SN_CAUSE_CODE_INTERNAL_ERROR = 9

AVP_STARENT_SN_CAUSE_FOR_REC_CLOSING = 139
AVP_STARENT_SN_CBB_POLICY = 302
AVP_STARENT_SN_CF_CALL_INTERNATIONAL = 293
E_SN_CF_CALL_INTERNATIONAL_DISABLE = 0
E_SN_CF_CALL_INTERNATIONAL_ENABLE = 1

AVP_STARENT_SN_CF_CALL_LOCAL = 291
E_SN_CF_CALL_LOCAL_DISABLE = 0
E_SN_CF_CALL_LOCAL_ENABLE = 1

AVP_STARENT_SN_CF_CALL_LONGDISTANCE = 292
AVP_STARENT_SN_CF_CALL_PREMIUM = 294
AVP_STARENT_SN_CF_CALL_ROAMINGINTERNATNL = 298
AVP_STARENT_SN_CF_CALL_TRANSFER = 285
AVP_STARENT_SN_CF_CALL_WAITING = 284
AVP_STARENT_SN_CF_CID_DISPLAY = 282
AVP_STARENT_SN_CF_CID_DISPLAY_BLOCKED = 283
AVP_STARENT_SN_CF_FOLLOW_ME = 281
AVP_STARENT_SN_CF_FORWARD_BUSY_LINE = 279
AVP_STARENT_SN_CF_FORWARD_NO_ANSWER = 278
AVP_STARENT_SN_CF_FORWARD_NOT_REGD = 280
AVP_STARENT_SN_CF_FORWARD_UNCONDITIONAL = 277
AVP_STARENT_SN_CFPOLICY_ID = 220
AVP_STARENT_SN_CHANGE_CONDITION = 140
AVP_STARENT_SN_CHARGING_VPN_NAME = 137
AVP_STARENT_SN_CHRG_CHAR_SELECTION_MODE = 138
AVP_STARENT_SN_CONTENT_DISPOSITION = 272
AVP_STARENT_SN_CONTENT_LENGTH = 271
AVP_STARENT_SN_CONTENT_TYPE = 270
AVP_STARENT_SN_CR_INTERNATIONAL_CID = 295
AVP_STARENT_SN_CR_LONGDISTANCE_CID = 296
AVP_STARENT_SN_CSCF_APP_SERVER_INFO = 275
AVP_STARENT_SN_CSCF_RF_SDP_MEDIA_COMPONENTS = 273
AVP_STARENT_SN_CSCF_SUBSCRIBER_IP_ADDRESS = 287
AVP_STARENT_SN_DATA_TUNNEL_IGNORE_DF_BIT = 49
AVP_STARENT_SN_DHCP_LEASE_EXPIRY_POLICY = 157
AVP_STARENT_SN_DHCP_OPTIONS = 309
AVP_STARENT_SN_DIRECTION = 153
AVP_STARENT_SN_DISCONNECT_REASON = 3
AVP_STARENT_SN_DNS_PROXY_INTERCEPT_LIST = 214
AVP_STARENT_SN_DNS_PROXY_USE_SUBSCR_ADDR = 25
AVP_STARENT_SN_DYNAMIC_ADDR_ALLOC_IND_FLAG = 141
AVP_STARENT_SN_ECS_DATA_VOLUME = 176
AVP_STARENT_SN_ENABLE_QOS_RENEGOTIATION = 144
AVP_STARENT_SN_EVENT = 255
AVP_STARENT_SN_EXT_INLINE_SRVR_CONTEXT = 41
AVP_STARENT_SN_EXT_INLINE_SRVR_DOWN_ADDR = 56
AVP_STARENT_SN_EXT_INLINE_SRVR_DOWN_VLAN = 59
AVP_STARENT_SN_EXT_INLINE_SRVR_PREFERENCE = 57
AVP_STARENT_SN_EXT_INLINE_SRVR_UP_ADDR = 55
AVP_STARENT_SN_EXT_INLINE_SRVR_UP_VLAN = 58
AVP_STARENT_SN_FAST_REAUTH_USERNAME_304 = 304
AVP_STARENT_SN_FIREWALL_ENABLED = 198
AVP_STARENT_SN_FIREWALL_POLICY_239 = 239
AVP_STARENT_SN_FMC_LOCATION = 171
AVP_STARENT_SN_GGSN_ADDRESS = 264
AVP_STARENT_SN_GGSN_MIP_REQUIRED = 68
AVP_STARENT_SN_GRATUITOUS_ARP_AGGRESSIVE = 54
AVP_STARENT_SN_GTP_VERSION = 62
AVP_STARENT_SN_HA_SEND_DNS_ADDRESS = 47
AVP_STARENT_SN_HANDOFF_INDICATOR = 310
AVP_STARENT_SN_HOME_BEHAVIOR = 119
AVP_STARENT_SN_HOME_PROFILE = 109
AVP_STARENT_SN_HOME_SUB_USE_GGSN = 106
AVP_STARENT_SN_IMS_AM_ADDRESS = 167
AVP_STARENT_SN_IMS_AM_DOMAIN_NAME = 168
AVP_STARENT_SN_IMS_CHARGING_IDENTIFIER = 260
AVP_STARENT_SN_IMSI = 252
AVP_STARENT_SN_INACTIVITY_TIME = 232
AVP_STARENT_SN_INTERNAL_SM_INDEX = 122
AVP_STARENT_SN_IP_ALLOC_METHOD = 53
AVP_STARENT_SN_IPV6_ALLOC_METHOD = 314
AVP_STARENT_SN_IP_FILTER_IN = 10
AVP_STARENT_SN_IP_FILTER_OUT = 11
AVP_STARENT_SN_IP_HEADER_COMPRESSION = 150
AVP_STARENT_SN_IP_HIDE_SERVICE_ADDRESS = 60
AVP_STARENT_SN_IP_IN_ACL = 17
AVP_STARENT_SN_IP_IN_PLCY_GRP = 193
AVP_STARENT_SN_IP_OUT_ACL = 18
AVP_STARENT_SN_IP_OUT_PLCY_GRP = 194
AVP_STARENT_SN_IP_POOL_NAME = 8
AVP_STARENT_SN_IP_SOURCE_VALIDATION = 14
AVP_STARENT_SN_IP_SOURCE_VIOLATE_NO_ACCT = 196
AVP_STARENT_SN_IP_SRC_VALIDATION_DROP_LIMIT = 110
AVP_STARENT_SN_IPV6_DNS_PROXY = 126
AVP_STARENT_SN_IPV6_EGRESS_FILTERING = 103
AVP_STARENT_SN_IPV6_MIN_LINK_MTU = 136
AVP_STARENT_SN_IPV6_NUM_RTR_ADVT = 97
AVP_STARENT_SN_IPV6_PRIMARY_DNS = 101
AVP_STARENT_SN_IPV6_RTR_ADVT_INTERVAL = 96
AVP_STARENT_SN_IPV6_SECONDARY_DNS = 102
AVP_STARENT_SN_IPV6_SEC_POOL = 124
AVP_STARENT_SN_IPV6_SEC_PREFIX = 125
AVP_STARENT_SN_ISC_TEMPLATE_NAME = 276
AVP_STARENT_SN_IS_UNREGISTERED_SUBSCRIBER = 269
AVP_STARENT_SN_L3_TO_L2_TUN_ADDR_POLICY = 43
AVP_STARENT_SN_LI_DEST_ADDRESS = 240
AVP_STARENT_SN_LOCAL_IP_ADDRESS = 13
AVP_STARENT_SN_LONG_DURATION_ACTION = 45
AVP_STARENT_SN_LONG_DURATION_NOTIFICATION = 253
AVP_STARENT_SN_LONG_DURATION_TIMEOUT = 44
AVP_STARENT_SN_MAX_SEC_CONTEXTS_PER_SUBS = 290
AVP_STARENT_SN_MEDIATION_ACCT_RSP_ACTION = 105
AVP_STARENT_SN_MEDIATION_ENABLED = 123
AVP_STARENT_SN_MEDIATION_NO_INTERIMS = 146
AVP_STARENT_SN_MEDIATION_VPN_NAME = 104
AVP_STARENT_SN_MIN_COMPRESS_SIZE = 23
AVP_STARENT_SN_MIP_AAA_ASSIGN_ADDR = 50
AVP_STARENT_SN_MIP_ANCID = 166
AVP_STARENT_SN_MIP_DUAL_ANCHOR = 165
AVP_STARENT_SN_MIP_HA_ASSIGNMENT_TABLE = 154
AVP_STARENT_SN_MIP_MATCH_AAA_ASSIGN_ADDR = 51
AVP_STARENT_SN_MIP_REG_LIFETIME_REALM = 175
AVP_STARENT_SN_MIP_SEND_ANCID = 163
AVP_STARENT_SN_MIP_SEND_CORRELATION_INFO = 188
AVP_STARENT_SN_MIP_SEND_HOST_CONFIG = 311
AVP_STARENT_SN_MIP_SEND_IMSI = 164
AVP_STARENT_SN_MIP_SEND_TERM_VERIFICATION = 48
AVP_STARENT_SN_MN_HA_HASH_ALGORITHM = 99
AVP_STARENT_SN_MN_HA_TIMESTAMP_TOLERANCE = 30
AVP_STARENT_SN_MODE = 151
AVP_STARENT_SN_MS_ISDN = 248
AVP_STARENT_SN_NAI_CONSTRUCTION_DOMAIN = 37
AVP_STARENT_SN_NAT_IP_ADDRESS = 297
AVP_STARENT_SN_NODE_FUNCTIONALITY = 268
AVP_STARENT_SN_NPU_QOS_PRIORITY = 98
AVP_STARENT_SN_NTK_INITIATED_CTX_IND_FLAG = 142
AVP_STARENT_SN_NTK_SESSION_DISCONNECT_FLAG = 143
AVP_STARENT_SN_NW_REACHABILITY_SERVER_NAME = 65
AVP_STARENT_SN_ORIGINATING_IOI = 261
AVP_STARENT_SN_OVERLOAD_DISC_CONNECT_TIME = 233
AVP_STARENT_SN_OVERLOAD_DISC_INACT_TIME = 234
AVP_STARENT_SN_OVERLOAD_DISCONNECT = 235
AVP_STARENT_SN_PDG_TTG_REQUIRED = 299
AVP_STARENT_SN_PDIF_MIP_RELEASE_TIA = 172
AVP_STARENT_SN_PDIF_MIP_REQUIRED = 170
AVP_STARENT_SN_PDIF_MIP_SIMPLE_IP_FALLBACK = 173
AVP_STARENT_SN_PDSN_HANDOFF_REQ_IP_ADDR = 46
AVP_STARENT_SN_PDSN_NAS_ID = 190
AVP_STARENT_SN_PDSN_NAS_IP_ADDRESS = 191
AVP_STARENT_SN_PERMIT_USER_MCAST_PDUS = 134
AVP_STARENT_SN_PPP_ACCEPT_PEER_V6IFID = 95
AVP_STARENT_SN_PPP_ALWAYS_ON_VSE = 130
AVP_STARENT_SN_PPP_DATA_COMPRESSION = 9
AVP_STARENT_SN_PPP_DATA_COMPRESSION_MODE = 19
AVP_STARENT_SN_PPP_KEEPALIVE = 16
AVP_STARENT_SN_PPP_NW_LAYER_IPV4 = 92
AVP_STARENT_SN_PPP_NW_LAYER_IPV6 = 93
AVP_STARENT_SN_PPP_OUTBOUND_PASSWORD = 15
AVP_STARENT_SN_PPP_OUTBOUND_USERNAME = 61
AVP_STARENT_SN_PPP_PROGRESS_CODE = 4
AVP_STARENT_SN_PPP_RENEG_DISC = 187
AVP_STARENT_SN_PREPAID = 128
AVP_STARENT_SN_PREPAID_COMPRESSED_COUNT = 31
AVP_STARENT_SN_PREPAID_FINAL_DURATION_ALG = 135
AVP_STARENT_SN_PREPAID_INBOUND_OCTETS = 32
AVP_STARENT_SN_PREPAID_OUTBOUND_OCTETS = 33
AVP_STARENT_SN_PREPAID_PREFERENCE = 129
AVP_STARENT_SN_PREPAID_TIMEOUT = 35
AVP_STARENT_SN_PREPAID_TOTAL_OCTETS = 34
AVP_STARENT_SN_PREPAID_WATERMARK = 36
AVP_STARENT_SN_PRIMARY_DCCA_PEER = 223
AVP_STARENT_SN_PRIMARY_DNS_SERVER = 5
AVP_STARENT_SN_PRIMARY_NBNS_SERVER = 148
AVP_STARENT_SN_PROXY_MIP = 52
AVP_STARENT_SN_PROXY_MIPV6 = 65530
AVP_STARENT_SN_PSEUDONYM_USERNAME_305 = 305
AVP_STARENT_SN_QOS_BACKGROUND_CLASS = 91
AVP_STARENT_SN_QOS_CLASS_BACKGROUND_PHB = 113
AVP_STARENT_SN_QOS_CLASS_CONVERSATIONAL_PHB = 111
AVP_STARENT_SN_QOS_CLASS_INTERACTIVE_1_PHB = 114
AVP_STARENT_SN_QOS_CLASS_INTERACTIVE_2_PHB = 115
AVP_STARENT_SN_QOS_CLASS_INTERACTIVE_3_PHB = 116
AVP_STARENT_SN_QOS_CLASS_STREAMING_PHB = 112
AVP_STARENT_SN_QOS_CONVERSATION_CLASS = 86
AVP_STARENT_SN_QOS_HLR_PROFILE = 303
AVP_STARENT_SN_QOS_INTERACTIVE1_CLASS = 88
AVP_STARENT_SN_QOS_INTERACTIVE2_CLASS = 89
AVP_STARENT_SN_QOS_INTERACTIVE3_CLASS = 90
AVP_STARENT_SN_QOS_NEGOTIATED = 147
AVP_STARENT_SN_QOS_RENEGOTIATION_TIMEOUT = 145
AVP_STARENT_SN_QOS_STREAMING_CLASS = 87
AVP_STARENT_SN_QOS_TP_DNLK = 73
AVP_STARENT_SN_QOS_TP_UPLK = 79
AVP_STARENT_SN_QOS_TRAFFIC_POLICY = 177
AVP_STARENT_SN_RAD_APN_NAME = 162
AVP_STARENT_SN_RADIUS_RETURNED_USERNAME = 236
AVP_STARENT_SN_RE_CHAP_INTERVAL = 7
AVP_STARENT_SN_ROAMING_BEHAVIOR = 121
AVP_STARENT_SN_ROAMING_PROFILE = 118
AVP_STARENT_SN_ROAMING_SUB_USE_GGSN = 108
AVP_STARENT_SN_ROHC_FLOW_MARKING_MODE = 274
AVP_STARENT_SN_ROHC_PROFILE_NAME = 238
AVP_STARENT_SN_ROLE_OF_NODE = 256
AVP_STARENT_SN_ROUTING_AREA_ID = 249
AVP_STARENT_SN_RULEBASE = 250
AVP_STARENT_SN_SDP_SESSION_DESCRIPTION = 263
AVP_STARENT_SN_SEC_IP_POOL_NAME = 265
AVP_STARENT_SN_SECONDARY_DCCA_PEER = 224
AVP_STARENT_SN_SECONDARY_DNS_SERVER = 6
AVP_STARENT_SN_SECONDARY_NBNS_SERVER = 149
AVP_STARENT_SN_SERVICE_ADDRESS = 169
AVP_STARENT_SN_SERVICE_TYPE = 24
AVP_STARENT_SN_SESSION_ID = 257
AVP_STARENT_SN_SIMULTANEOUS_SIP_MIP = 22
AVP_STARENT_SN_SIP_METHOD = 254
AVP_STARENT_SN_SIP_REQUEST_TIME_STAMP = 258
AVP_STARENT_SN_SIP_RESPONSE_TIME_STAMP = 259
AVP_STARENT_SN_SOFTWARE_VERSION = 288
AVP_STARENT_SN_SUBS_ACC_FLOW_TRAFFIC_VALID = 225
AVP_STARENT_SN_SUBSCRIBER_ACCOUNTING = 64
AVP_STARENT_SN_SUBSCRIBER_ACCT_INTERIM = 70
AVP_STARENT_SN_SUBSCRIBER_ACCT_MODE = 192
AVP_STARENT_SN_SUBSCRIBER_ACCT_RSP_ACTION = 100
AVP_STARENT_SN_SUBSCRIBER_ACCT_START = 69
AVP_STARENT_SN_SUBSCRIBER_ACCT_STOP = 71
AVP_STARENT_SN_SUBSCRIBER_CLASS = 219
AVP_STARENT_SN_SUBSCRIBER_IP_HDR_NEG_MODE = 67
AVP_STARENT_SN_SUBSCRIBER_IP_TOS_COPY = 85
AVP_STARENT_SN_SUBSCRIBER_NEXTHOP_ADDRESS = 127
AVP_STARENT_SN_SUBSCRIBER_NO_INTERIMS = 133
AVP_STARENT_SN_SUBSCRIBER_TEMPLATE_NAME = 158
AVP_STARENT_SN_SUBS_IMSA_SERVICE_NAME = 159
AVP_STARENT_SN_SUBS_VJ_SLOTID_CMP_NEG_MODE = 221
AVP_STARENT_SN_TP_DNLK_BURST_SIZE = 76
AVP_STARENT_SN_TP_DNLK_COMMITTED_DATA_RATE = 74
AVP_STARENT_SN_TP_DNLK_EXCEED_ACTION = 77
AVP_STARENT_SN_TP_DNLK_PEAK_DATA_RATE = 75
AVP_STARENT_SN_TP_DNLK_VIOLATE_ACTION = 78
AVP_STARENT_SN_TP_UPLK_BURST_SIZE = 82
AVP_STARENT_SN_TP_UPLK_COMMITTED_DATA_RATE = 80
AVP_STARENT_SN_TP_UPLK_EXCEED_ACTION = 83
AVP_STARENT_SN_TP_UPLK_PEAK_DATA_RATE = 81
AVP_STARENT_SN_TP_UPLK_VIOLATE_ACTION = 84
AVP_STARENT_SN_TPO_POLICY = 308
AVP_STARENT_SN_TRAFFIC_GROUP = 161
AVP_STARENT_SN_TRAFFICSELECTOR_CLASS = 307
AVP_STARENT_SN_TUN_ADDR_POLICY = 156
AVP_STARENT_SN_TUNNEL_GN = 174
AVP_STARENT_SN_TUNNEL_ISAKMP_CRYPTO_MAP = 38
AVP_STARENT_SN_TUNNEL_ISAKMP_SECRET = 39
AVP_STARENT_SN_TUNNEL_LOAD_BALANCING = 27
AVP_STARENT_SN_TUNNEL_PASSWORD = 26
AVP_STARENT_SN_UNCLASSIFY_LIST_NAME = 132
AVP_STARENT_SN_USER_PRIVILEGE = 313
AVP_STARENT_SN_VIRTUAL_APN_NAME = 94
AVP_STARENT_SN_VISITING_BEHAVIOR = 120
AVP_STARENT_SN_VISITING_PROFILE = 117
AVP_STARENT_SN_VISITING_SUB_USE_GGSN = 107
AVP_STARENT_SN_VOICE_PUSH_LIST_NAME = 131
AVP_STARENT_SN_VPN_ID = 1
AVP_STARENT_SN_VPN_NAME = 2
AVP_STARENT_SN_WIMAX_AUTH_ONLY = 306
AVP_STARENT_SN_WLAN_AP_IDENTIFIER = 319
AVP_STARENT_SN_WLAN_UE_IDENTIFIER = 320
AVP_NOKIA_NOKIA_IMS_MEDIA_COMPONENT_ID = 5101
AVP_NOKIA_NOKIA_TIME_OF_FIRST_USAGE = 5103
AVP_NOKIA_NOKIA_TIME_OF_LAST_USAGE = 5104
AVP_NOKIA_NOKIA_SESSION_START_INDICATOR = 5105
AVP_NOKIA_NOKIA_RULEBASE_ID = 5106
AVP_NOKIA_NOKIA_QUOTA_CONSUMPTION_TIME = 5109
AVP_NOKIA_NOKIA_QUOTA_HOLDING_TIME = 5110
AVP_NOKIA_DEFAULT_QUOTA = 5111
AVP_NOKIA_NOKIA_URI = 5112
AVP_NOKIA_NSN_TOKEN_VALUE = 5113
AVP_NOKIASOLUTIONSANDNETWORKS_NSN_IN_INFORMATION = 100
AVP_NOKIASOLUTIONSANDNETWORKS_NSN_ACCOUNT_LOCATION_ID = 223
AVP_NOKIASOLUTIONSANDNETWORKS_ACCOUNT_STATE = 60528
AVP_NOKIASOLUTIONSANDNETWORKS_SUBSCRIBER_SPECIFIC_INFO = 60530
AVP_NOKIASOLUTIONSANDNETWORKS_OFFER_NAME = 60551
AVP_NOKIASOLUTIONSANDNETWORKS_VERSION = 60552
AVP_NOKIASOLUTIONSANDNETWORKS_ACTIVE_PRODUCT_OFFER = 60553
AVP_NOKIASOLUTIONSANDNETWORKS_EXPIRY_DATE = 60554
AVP_NOKIASOLUTIONSANDNETWORKS_BILLING_ACCOUNT = 60557
AVP_NOKIASOLUTIONSANDNETWORKS_BILLING_ACCOUNT_NUMBER = 60558
AVP_NOKIASOLUTIONSANDNETWORKS_ACCOUNT_TYPE = 60559
AVP_NOKIASOLUTIONSANDNETWORKS_ACCOUNT_SUBTYPE = 60560
AVP_ORACLETEKELEC_DSR_APPLICATIONINVOKED = 2468
E_DSR_APPLICATIONINVOKED_RBAR_ID = 3
E_DSR_APPLICATIONINVOKED_FABR_ID = 4
E_DSR_APPLICATIONINVOKED_CPA_ID = 5
E_DSR_APPLICATIONINVOKED_POLICY_DRA_ID = 6

AVP_ORACLETEKELEC_PDRA_EARLY_BINDING = 2500
AVP_ORACLETEKELEC_SESSION_RELEASE_REASON = 2501
AVP_ORACLETEKELEC_MSGCOPYANSWER = 2516
AVP_CISCOSYSTEMS_DSCP = 131178
AVP_CISCOSYSTEMS_RATE_LIMIT_ACTION = 131177
AVP_CISCOSYSTEMS_QOS_GROUP_RULE_INSTALL = 132001
AVP_CISCOSYSTEMS_QOS_GROUP_RULE_DEFINITION = 132003
AVP_CISCOSYSTEMS_QOS_GROUP_RULE_NAME = 132004
AVP_CISCOSYSTEMS_MBR_LIMIT_EXCEED_ACTION_UL = 132006
AVP_CISCOSYSTEMS_MBR_LIMIT_EXCEED_ACTION_DL = 132008
AVP_CISCOSYSTEMS_MBR_BURST_SIZE_UL = 132009
AVP_CISCOSYSTEMS_MBR_BURST_SIZE_DL = 132010
AVP_CISCOSYSTEMS_QOS_LEVEL = 132011
AVP_CISCOSYSTEMS_OVERRIDE_CONTROL = 132017
AVP_CISCOSYSTEMS_OVERRIDE_RULE_NAME = 132018
AVP_CISCOSYSTEMS_OVERRIDE_CHARGING_ACTION_PARAMETERS = 132019
AVP_CISCOSYSTEMS_OVERRIDE_CHARGING_ACTION_NAME = 132020
AVP_CISCOSYSTEMS_OVERRIDE_CHARGING_ACTION_EXCLUDE_RULE = 132021
AVP_CISCOSYSTEMS_OVERRIDE_CHARGING_PARAMETERS = 132022
AVP_CISCOSYSTEMS_OVERRIDE_SERVICE_IDENTIFIER = 132023
AVP_CISCOSYSTEMS_OVERRIDE_RATING_GROUP = 132024
AVP_CISCOSYSTEMS_CISCOSYSTEMS_EXECUTION_TIME = 132025
AVP_CISCOSYSTEMS_OVERRIDE_ONLINE = 132026
E_OVERRIDE_ONLINE_DISABLE_ONLINE = 0
E_OVERRIDE_ONLINE_ENABLE_ONLINE = 1

AVP_CISCOSYSTEMS_OVERRIDE_OFFLINE = 132027
E_OVERRIDE_OFFLINE_DISABLE_OFFLINE = 0
E_OVERRIDE_OFFLINE_ENABLE_OFFLINE = 1

AVP_CISCOSYSTEMS_OVERRIDE_CONTENT_FILTERING_STATE = 132028
E_OVERRIDE_CONTENT_FILTERING_STATE_DISABLE_CF = 0
E_OVERRIDE_CONTENT_FILTERING_STATE_ENABLE_CF = 1

AVP_CISCOSYSTEMS_OVERRIDE_POLICY_PARAMETERS = 132029
AVP_CISCOSYSTEMS_OVERRIDE_QOS_INFORMATION = 132030
AVP_CISCOSYSTEMS_OVERRIDE_QOS_CLASS_IDENTIFIER = 132039
E_OVERRIDE_QOS_CLASS_IDENTIFIER_TRAFFIC_CLASS_A = 1
E_OVERRIDE_QOS_CLASS_IDENTIFIER_TRAFFIC_CLASS_B = 2
E_OVERRIDE_QOS_CLASS_IDENTIFIER_TRAFFIC_CLASS_C = 3
E_OVERRIDE_QOS_CLASS_IDENTIFIER_TRAFFIC_CLASS_D = 4
E_OVERRIDE_QOS_CLASS_IDENTIFIER_TRAFFIC_CLASS_E = 5
E_OVERRIDE_QOS_CLASS_IDENTIFIER_TRAFFIC_CLASS_F = 6
E_OVERRIDE_QOS_CLASS_IDENTIFIER_TRAFFIC_CLASS_G = 7
E_OVERRIDE_QOS_CLASS_IDENTIFIER_TRAFFIC_CLASS_H = 8
E_OVERRIDE_QOS_CLASS_IDENTIFIER_TRAFFIC_CLASS_I = 9

AVP_CISCOSYSTEMS_OVERRIDE_MAX_REQUESTED_BANDWIDTH_UL = 132032
AVP_CISCOSYSTEMS_OVERRIDE_MAX_REQUESTED_BANDWIDTH_DL = 132033
AVP_CISCOSYSTEMS_OVERRIDE_GUARANTEED_BITRATE_UL = 132034
AVP_CISCOSYSTEMS_OVERRIDE_GUARANTEED_BITRATE_DL = 132035
AVP_CISCOSYSTEMS_OVERRIDE_ALLOCATION_RETENTION_PRIORITY = 132036
AVP_CISCOSYSTEMS_OVERRIDE_PRIORITY_LEVEL = 132037
AVP_CISCOSYSTEMS_OVERRIDE_PRE_EMPTION_CAPABILITY = 132038
E_OVERRIDE_PRE_EMPTION_CAPABILITY_PRE_EMPTION_CAPABILITY_ENABLED = 0
E_OVERRIDE_PRE_EMPTION_CAPABILITY_PRE_EMPTION_CAPABILITY_DISABLED = 1

AVP_CISCOSYSTEMS_OVERRIDE_TOS_VALUE = 132046
AVP_CISCOSYSTEMS_OVERRIDE_TOS_DIRECTION = 132047
E_OVERRIDE_TOS_DIRECTION_UPLINK_DIRECTION = 0
E_OVERRIDE_TOS_DIRECTION_DOWNLINK_DIRECTION = 1
E_OVERRIDE_TOS_DIRECTION_BIDIRECTIONAL = 2

AVP_CISCOSYSTEMS_OVERRIDE_TOS_VALUE_STANDARD = 132048
E_OVERRIDE_TOS_VALUE_STANDARD_BE = 0
E_OVERRIDE_TOS_VALUE_STANDARD_AF11 = 10
E_OVERRIDE_TOS_VALUE_STANDARD_AF12 = 12
E_OVERRIDE_TOS_VALUE_STANDARD_AF13 = 14
E_OVERRIDE_TOS_VALUE_STANDARD_AF21 = 18
E_OVERRIDE_TOS_VALUE_STANDARD_AF22 = 20
E_OVERRIDE_TOS_VALUE_STANDARD_AF23 = 22
E_OVERRIDE_TOS_VALUE_STANDARD_AF31 = 26
E_OVERRIDE_TOS_VALUE_STANDARD_AF32 = 28
E_OVERRIDE_TOS_VALUE_STANDARD_AF33 = 30
E_OVERRIDE_TOS_VALUE_STANDARD_AF41 = 34
E_OVERRIDE_TOS_VALUE_STANDARD_AF42 = 36
E_OVERRIDE_TOS_VALUE_STANDARD_AF43 = 38
E_OVERRIDE_TOS_VALUE_STANDARD_EF = 46

AVP_CISCOSYSTEMS_OVERRIDE_TOS_VALUE_CUSTOM = 132049
AVP_CISCOSYSTEMS_OVERRIDE_CONTROL_NAME = 132052
AVP_CISCOSYSTEMS_OVERRIDE_NEXTHOP_ADDRESS = 132054
AVP_CISCOSYSTEMS_OVERRIDE_CONTROL_PENDING_QUEUE_ACTION = 132078
E_OVERRIDE_CONTROL_PENDING_QUEUE_ACTION_FLUSH = 0
E_OVERRIDE_CONTROL_PENDING_QUEUE_ACTION_RETAIN = 1

AVP_CISCOSYSTEMS_OVERRIDE_CONTROL_MERGE_WILDCARD = 132079
E_OVERRIDE_CONTROL_MERGE_WILDCARD_TRUE = 0
E_OVERRIDE_CONTROL_MERGE_WILDCARD_FALSE = 1

AVP_JUNIPER_TDF_APPLICATION_INSTANCE_IDENTIFIER_BASE = 1100
AVP_JUNIPER_SERVICE_CHAINING_INFORMATION = 1101
AVP_JUNIPER_LRF_PROFILE_NAME = 1102
AVP_JUNIPER_HCM_PROFILE_NAME = 1103
AVP_JUNIPER_FORWARDING_CLASS_NAME = 1104
AVP_JUNIPER_REDIRECT_VRF = 1105
AVP_JUNIPER_REQUESTED_BURSTSIZE_UL = 1106
AVP_JUNIPER_REQUESTED_BURSTSIZE_DL = 1107
AVP_JUNIPER_STEERING_INFORMATION = 1108
AVP_JUNIPER_STEERING_UPLINK_VRF = 1109
AVP_JUNIPER_STEERING_DOWNLINK_VRF = 1110
AVP_JUNIPER_STEERING_IP_ADDRESS = 1111
AVP_INOVAR_INOVAR_FORWARD_TO_PARTY = 4001
AVP_INOVAR_INOVAR_APP_CODE = 4002
E_INOVAR_APP_CODE__MY5 = 1
E_INOVAR_APP_CODE__VIRTUAL_CALLER_ID_VCID = 2

AVP_INOVAR_INOVAR_SERVICE_CODE = 4003
E_INOVAR_SERVICE_CODE__UE_REGISTRATION_NOTIFICATION_BUDDY_LIST_LOOKUP = 1
E_INOVAR_SERVICE_CODE__UNSUCCESSFUL_CALL_SETUP_NOTIFICATION = 2

AVP_INOVAR_INOVAR_SERVICE_ID = 4004
AVP_INOVAR_INOVAR_UNSUCCESSFUL_CALL_REASON = 4005
AVP_INOVAR_INOVAR_SIP_RESPONSE_CODE = 4006
AVP_INOVAR_INOVAR_ISUP_CAUSE = 4007
AVP_INOVAR_INOVAR_SUPPLEMENTARY_SERVICE_ID = 4008
AVP_INOVAR_INOVAR_PS_REGISTRATION_STATUS = 4009
E_INOVAR_PS_REGISTRATION_STATUS__SUBSCRIBER_IS_NOT_REGISTERED_IN_PS_DOMAIN = 0
E_INOVAR_PS_REGISTRATION_STATUS__SUBSCRIBER_IS_REGISTERED_IN_PS_DOMAIN = 1

AVP_INOVAR_INOVAR_SERVICE_RESULT = 4010
E_INOVAR_SERVICE_RESULT__MY5_BUDDY_LIST_MATCH_FOUND = 1001
E_INOVAR_SERVICE_RESULT__MY5_INTERNAL_FAILURE = 1002
E_INOVAR_SERVICE_RESULT__MY5_SUBS_NOT_FOUND = 1003
E_INOVAR_SERVICE_RESULT__MY5_SUBS_BUDDY_LIST_NOT_CONFIGURED = 1004
E_INOVAR_SERVICE_RESULT__MY5_SUBS_BUDDY_LIST_CFG_NOT_ENABLED = 1005
E_INOVAR_SERVICE_RESULT__MY5_SUBS_BUDDY_LIST_CFG_ENABLED_BUT_NO_MATCH_FOUND_4_THE_OTHER_PARTY_NUMBER = 1006
E_INOVAR_SERVICE_RESULT__VCID_SUCCESS = 2001
E_INOVAR_SERVICE_RESULT__VCID_INTERNAL_FAILURE = 2002
E_INOVAR_SERVICE_RESULT__VCID_SUBSCRIBER_NOT_FOUND = 2003

AVP_INOVAR_INOVAR_CALLING_PARTY_INFO = 4011
AVP_HUAWEI_P2PSMS_INFORMATION = 20400
AVP_HUAWEI_SMSC_ADDRESS_HUAWEI = 20401
AVP_HUAWEI_SM_ID = 20402
AVP_HUAWEI_SM_LENGTH = 20403
AVP_HUAWEI_MO_MSC_ADDRESS = 20404
AVP_HUAWEI_MT_MSC_ADDRESS = 20405
AVP_HUAWEI_SOURCE_ADDR = 20406
AVP_HUAWEI_DEST_ADDR = 20407
AVP_HUAWEI_FEE_FLAG = 20409
E_FEE_FLAG_CHARGE_IN_THE_ORIGINAL_MODE = 0
E_FEE_FLAG_CHARGE_THE_CALLING_NUMBER = 1
E_FEE_FLAG_CHARGE_THE_CALLED_NUMBER = 2
E_FEE_FLAG_CHARGE_THE_CHARGING_NUMBER = 3

AVP_HUAWEI_FEE_TYPE = 20410
E_FEE_TYPE_CHARGE_BY_DEFAULT_MODE = 0
E_FEE_TYPE_CHARGE_BY_ITEM = 1
E_FEE_TYPE_CHARGE_BY_MONTH_WITH_AUTHENTICATION_AND_FEE_DEDUCTION = 2
E_FEE_TYPE_CHARGE_BY_MONTH_WITH_AUTHENTICATION_BUT_NO_FEE_DEDUCTION = 3
E_FEE_TYPE_CHARGE_UPPER_LIMIT_AMOUNT_BY_THE_NUMBER_OF_SMS = 4
E_FEE_TYPE_CHARGE_BY_MONTH_WITH_LIMITED_NUMBER_OF_SMS_WITH_AUTHENTICATION_AND_FEE_DEDUCTION = 5
E_FEE_TYPE_CHARGE_BY_MONTH_WITH_LIMITED_NUMBER_OF_SMS_WITH_AUTHENTICATION_BUT_NO_FEE_DEDUCTION = 6

AVP_HUAWEI_STATUS_REPORT_REQUESTED = 20415
E_STATUS_REPORT_REQUESTED_NO = 0
E_STATUS_REPORT_REQUESTED_YES = 1
E_STATUS_REPORT_REQUESTED_REFUND_ON_FAILURE = 2

AVP_HUAWEI_SEND_RESULT = 20418
E_SEND_RESULT_DELIVERED = 0

AVP_HUAWEI_OPERATION_RESULT = 20430
E_OPERATION_RESULT_SUCCESS = 0
E_OPERATION_RESULT_PROTOCOL_IS_NOT_SUPPORTED = 1
E_OPERATION_RESULT_OTHER_REASON = 10
E_OPERATION_RESULT_MO_ACCOUNT_DOES_NOT_EXIST = 11
E_OPERATION_RESULT_MT_ACCOUNT_DOES_NOT_EXIST = 12
E_OPERATION_RESULT_MO_ACCOUNT_STATE_IS_ABNORMAL = 13
E_OPERATION_RESULT_MT_ACCOUNT_STATE_IS_ABNORMAL = 14
E_OPERATION_RESULT_MO_BALANCE_IS_NOT_ENOUGH = 15
E_OPERATION_RESULT_MT_BALANCE_IS_NOT_ENOUGH = 16

AVP_HUAWEI_NOTIFY_MODE = 20431
E_NOTIFY_MODE_SEND_THE_REPORT_REGARDLESS_OF_WHETHER_THE_MESSAGE_SENDING_SUCCEEDS_OR_FAILS = 1
E_NOTIFY_MODE_SEND_THE_REPORT_WHEN_THE_MESSAGE_SENDING_FAILS = 2
E_NOTIFY_MODE_SEND_THE_REPORT_WHEN_THE_MESSAGE_SUCCEEDS = 3
E_NOTIFY_MODE_NEVER_SEND_THE_REPORT = 4

AVP_VERIZONWIRELESS_IDLE_TO_CONNECTED_TRANSITION_COUNT = 5001
AVP_VERIZONWIRELESS_CONNECTED_DURATION = 5002
AVP_VERIZONWIRELESS_CHARGING_GATEWAY_FUNCTION_HOST = 6068
AVP_VERIZONWIRELESS_CHARGING_GROUP_ID = 6069
AVP_VERIZONWIRELESS_LI_INDICATOR_GX = 6092
AVP_VERIZONWIRELESS_RANDOM_VECTOR = 6093
AVP_VERIZONWIRELESS_INTERCEPTION_ACTION = 6094
AVP_VERIZONWIRELESS_LI_CORRELATION_ID = 6095
AVP_VERIZONWIRELESS_LI_TARGETMFADDRESS = 6096
AVP_VERIZONWIRELESS_LI_TARGETID = 6097
AVP_VERIZONWIRELESS_MF_ADDRESS = 6098
AVP_VERIZONWIRELESS_MF_PORT = 6099
AVP_VERIZONWIRELESS_SELF_ACTIVATION_STATUS = 6115
E_SELF_ACTIVATION_STATUS_CONTINUE = 0
E_SELF_ACTIVATION_STATUS_REACTIVATION_DISALLOWED_TO_APN = 1

AVP_VERIZONWIRELESS_VIRTUAL_APN_NAME = 6116
AVP_VERIZONWIRELESS_ORIGINATION_TIMESTAMP = 7102
AVP_VERIZONWIRELESS_MAX_WAIT_TIME = 7103
AVP_VERIZONWIRELESS_ONLINE_RESPONSE_REQUIRED = 8105
E_ONLINE_RESPONSE_REQUIRED_WAIT = 0
E_ONLINE_RESPONSE_REQUIRED_DON_T_WAIT = 1
E_ONLINE_RESPONSE_REQUIRED_SPARE2 = 2
E_ONLINE_RESPONSE_REQUIRED_SPARE3 = 3
E_ONLINE_RESPONSE_REQUIRED_SPARE4 = 4

AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ADDITIONAL_CALLING_PARTY_NUMBER = 1
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ANNOUNCEMENT = 2
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ANNOUNCEMENT_CODE = 3
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ANNOUNCEMENT_DATA = 4
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ANNOUNCEMENT_DATA_TYPE = 5
E_CS_ANNOUNCEMENT_DATA_TYPE_INTEGER32 = 1
E_CS_ANNOUNCEMENT_DATA_TYPE_OCTETSTRING = 2
E_CS_ANNOUNCEMENT_DATA_TYPE_TIME = 3

AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ANNOUNCEMENT_DATA_VALUE = 6
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_BEARER_CAPABILITY = 7
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_CALL_REFERENCE_NUMBER = 8
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_CALLED_PARTY_NUMBER = 9
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_CALLING_PARTYS_CATEGORY = 10
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_CALLING_PARTY_NUMBER = 11
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_EVENT_TYPE_BCSM = 12
E_CS_EVENT_TYPE_BCSM_COLLECTEDINFO = 2
E_CS_EVENT_TYPE_BCSM_ANALYZEDINFORMATION = 3
E_CS_EVENT_TYPE_BCSM_ROUTESELECTFAILURE = 4
E_CS_EVENT_TYPE_BCSM_OCALLEDPARTYBUSY = 5
E_CS_EVENT_TYPE_BCSM_ONOANSWER = 6
E_CS_EVENT_TYPE_BCSM_OANSWER = 7
E_CS_EVENT_TYPE_BCSM_ODISCONNECT = 9
E_CS_EVENT_TYPE_BCSM_OABANDON = 10
E_CS_EVENT_TYPE_BCSM_TERMATTEMPTAUTHORIZED = 12
E_CS_EVENT_TYPE_BCSM_TBUSY = 13
E_CS_EVENT_TYPE_BCSM_TNOANSWER = 14
E_CS_EVENT_TYPE_BCSM_TANSWER = 15
E_CS_EVENT_TYPE_BCSM_TDISCONNECT = 17
E_CS_EVENT_TYPE_BCSM_TABANDON = 18

AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_FIRST_CALL_INDICATION = 13
E_CS_FIRST_CALL_INDICATION_FIRST_CALL_FALSE = 0
E_CS_FIRST_CALL_INDICATION_FIRST_CALL_TRUE = 1

AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_FREE_FORMAT_DATA = 14
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_FURNISH_CHARGING_INFORMATION = 15
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_HIGH_LAYER_COMPATIBILITY = 16
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_INFORMATION = 17
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_LOCATION_INFORMATION = 18
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_MSC_ADDRESS = 19
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ORIGINAL_CALLED_PARTY_ID = 20
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ORIGINATING_LOCATION_NUMBER = 21
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_PARTY_TO_CHARGE = 22
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_REDIRECTING_PARTY_NUMBER = 23
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_REDIRECTION_INFORMATION = 24
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_RELEASE_CAUSE_CODE = 25
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_REROUTING_NUMBER = 26
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_SERVICE_KEY = 27
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_TERMINATING_VLR_NUMBER = 28
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_WARNING_PERIOD = 29
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ACCUMULATED_TIME = 30
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ANNOUNCEMENT_LOGIC = 31
E_CS_ANNOUNCEMENT_LOGIC_NOT_TO_BE_PLAYED = 0
E_CS_ANNOUNCEMENT_LOGIC_PLAYED_TO_CALLING_PARTY = 1
E_CS_ANNOUNCEMENT_LOGIC_PLAYED_TO_CALLED_PARTY = 2
E_CS_ANNOUNCEMENT_LOGIC_PLAYED_TO_BOTH_PARTIES = 3

AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ANNOUNCEMENT_TYPE = 32
E_CS_ANNOUNCEMENT_TYPE_IMMEDIATE_ANNOUNCEMENT = 0
E_CS_ANNOUNCEMENT_TYPE_MIDCALL_ANNOUNCEMENT = 1
E_CS_ANNOUNCEMENT_TYPE_ENDCALL_ANNOUNCEMENT = 2

AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_BALANCE_TOPUP_DATA = 33
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_CALL_SETUP_RESULT_CODE = 34
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_CUSTOMER_DATA = 35
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_DIALOGUE_HANDLING = 36
E_CS_DIALOGUE_HANDLING_NORMAL = 0
E_CS_DIALOGUE_HANDLING_OPTIMIZED = 1

AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_ISUP_RELEASE_CAUSE_CODE = 37
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_MAX_CALL_DURATION = 38
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_RESULT_CODE = 39
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_START_OF_CHARGING = 40
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_TELESERVICE_CODE = 41
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_TRAFFIC_CASE = 42
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_AOC_INFO = 43
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_AOC_INFO_E1 = 44
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_AOC_INFO_E2 = 45
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_AOC_INFO_E3 = 46
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_AOC_INFO_E4 = 47
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_AOC_INFO_E5 = 48
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_AOC_INFO_E6 = 49
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_AOC_INFO_E7 = 50
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_CALLED_PARTY_BCD_NUMBER = 51
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_CONFERENCE_ID = 52
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_MULTIPARTY_CALL_INDICATION = 53
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_NETWORK_TYPE = 54
AVP_TELEFONICA_MOVILES_ESPA_S_A__CS_SERVICE_ID = 55
AVP_TELEFONICA_MOVILES_ESPA_S_A__OCS_ACCOUNT_MANAGEMENT_OP = 56
AVP_SIEMENS_AG_SIEMENS_IMSI = 1106
AVP_SIEMENS_AG_SIEMENS_MSIDN = 1107
AVP_SIEMENS_AG_SIEMENS_SERVER_NAME = 1110
AVP_MICROSOFT_MS_CHAP_RESPONSE = 1
AVP_MICROSOFT_MS_CHAP_ERROR = 2
AVP_MICROSOFT_MS_CHAP_NT_ENC_PW = 6
AVP_MICROSOFT_MS_CHAP_DOMAIN = 10
AVP_MICROSOFT_MS_CHAP_CHALLENGE = 11
AVP_MICROSOFT_MS_CHAP2_RESPONSE = 25
AVP_MICROSOFT_MS_CHAP2_SUCCESS = 26
AVP_MICROSOFT_MS_CHAP2_CPW = 27
AVP_METASWITCH_METASWITCH_INFORMATION = 10000
AVP_METASWITCH_ADJACENCY_NAME = 10001
AVP_METASWITCH_INCOMING_ADJACENCY_NAME = 10002
AVP_METASWITCH_OUTGOING_ADJACENCY_NAME = 10003
AVP_METASWITCH_ACCOUNT_NAME = 10004
AVP_METASWITCH_INCOMING_ACCOUNT_NAME = 10005
AVP_METASWITCH_OUTGOING_ACCOUNT_NAME = 10006
AVP_METASWITCH_SIP_DIVERSION = 10007
AVP_METASWITCH_INPUT_OUTPUT_OCTETS = 10008
AVP_METASWITCH_INGRESS_INPUT_OCTETS = 10009
AVP_METASWITCH_INGRESS_OUTPUT_OCTETS = 10010
AVP_METASWITCH_EGRESS_INPUT_OCTETS = 10011
AVP_METASWITCH_EGRESS_OUTPUT_OCTETS = 10012
AVP_METASWITCH_DISCONNECT_DIRECTION = 10013
E_DISCONNECT_DIRECTION_DISCONNECT_SIDE_ORIG = 1
E_DISCONNECT_DIRECTION_DISCONNECT_SIDE_TERM = 2
E_DISCONNECT_DIRECTION_DISCONNECT_SIDE_NEITHER = 3

AVP_METASWITCH_MEDIA_DESCRIPTION = 10014

AVP_ONEM2M_ACCESS_NETWORK_IDENTIFIER = 1000
AVP_ONEM2M_APPLICATION_ENTITY_ID = 1001
AVP_ONEM2M_CONTROL_MEMORY_SIZE = 1002
AVP_ONEM2M_CURRENT_NUMBER_MEMBERS = 1003
AVP_ONEM2M_DATA_MEMORY_SIZE = 1004
AVP_ONEM2M_EXTERNAL_ID = 1005
AVP_ONEM2M_GROUP_NAME = 1006
AVP_ONEM2M_HOSTING_CSE_ID = 1007
AVP_ONEM2M_ORIGINATOR = 1008
AVP_ONEM2M_MAXIMUM_NUMBER_MEMBERS = 1009
AVP_ONEM2M_M2M_EVENT_RECORD_TIMESTAMP = 1010
AVP_ONEM2M_M2M_INFORMATION = 1011
AVP_ONEM2M_OCCUPANCY = 1012
AVP_ONEM2M_PROTOCOL_TYPE = 1013
E_PROTOCOL_TYPE_HTTP = 0
E_PROTOCOL_TYPE_COAP = 1
E_PROTOCOL_TYPE_MQTT = 2

AVP_ONEM2M_RECEIVER = 1014
AVP_ONEM2M_REQUEST_BODY_SIZE = 1015
AVP_ONEM2M_REQUEST_HEADERS_SIZE = 1016
AVP_ONEM2M_REQUEST_OPERATION = 1017
E_REQUEST_OPERATION_CREATE = 1
E_REQUEST_OPERATION_RETRIEVE = 2
E_REQUEST_OPERATION_UPDATE = 3
E_REQUEST_OPERATION_DELETE = 4
E_REQUEST_OPERATION_NOTIFY = 5

AVP_ONEM2M_RESPONSE_BODY_SIZE = 1018
AVP_ONEM2M_RESPONSE_HEADERS_SIZE = 1019
AVP_ONEM2M_RESPONSE_STATUS_CODE = 1020
E_RESPONSE_STATUS_CODE_ACCEPTED = 1000
E_RESPONSE_STATUS_CODE_OK = 2000
E_RESPONSE_STATUS_CODE_CREATED = 2001
E_RESPONSE_STATUS_CODE_DELETED = 2002
E_RESPONSE_STATUS_CODE_UPDATED = 2004
E_RESPONSE_STATUS_CODE_CONTENT_EMPTY = 2100
E_RESPONSE_STATUS_CODE_BAD_REQUEST = 4000
E_RESPONSE_STATUS_CODE_NOT_FOUND = 4004
E_RESPONSE_STATUS_CODE_OPERATION_NOT_ALLOWED = 4005
E_RESPONSE_STATUS_CODE_REQUEST_TIMEOUT = 4008
E_RESPONSE_STATUS_CODE_SUBSCRIPTION_CREATOR_HAS_NO_PRIVILEGE = 4101
E_RESPONSE_STATUS_CODE_CONTENTS_UNACCEPTABLE = 4102
E_RESPONSE_STATUS_CODE_ACCESS_DENIED = 4103
E_RESPONSE_STATUS_CODE_GROUP_REQUEST_IDENTIFIER_EXISTS = 4104
E_RESPONSE_STATUS_CODE_CONFLICT = 4105
E_RESPONSE_STATUS_CODE_INTERNAL_SERVER_ERROR = 5000
E_RESPONSE_STATUS_CODE_NOT_IMPLEMENTED = 5001
E_RESPONSE_STATUS_CODE_TARGET_NOT_REACHABLE = 5103
E_RESPONSE_STATUS_CODE_NO_PRIVILEGE = 5105
E_RESPONSE_STATUS_CODE_ALREADY_EXISTS = 5106
E_RESPONSE_STATUS_CODE_TARGET_NOT_SUBSCRIBABLE = 5203
E_RESPONSE_STATUS_CODE_SUBSCRIPTION_VERIFICATION_INITIATION_FAILED = 5204
E_RESPONSE_STATUS_CODE_SUBSCRIPTION_HOST_HAS_NO_PRIVILEGE = 5205
E_RESPONSE_STATUS_CODE_NON_BLOCKING_REQUEST_NOT_SUPPORTED = 5206
E_RESPONSE_STATUS_CODE_NOT_ACCEPTABLE = 5207
E_RESPONSE_STATUS_CODE_EXTERNAL_OBJECT_NOT_REACHABLE = 6003
E_RESPONSE_STATUS_CODE_EXTERNAL_OBJECT_NOT_FOUND = 6005
E_RESPONSE_STATUS_CODE_MAX_NUMBER_OF_MEMBER_EXCEEDED = 6010
E_RESPONSE_STATUS_CODE_MEMBER_TYPE_INCONSISTENT = 6011
E_RESPONSE_STATUS_CODE_MGMT_SESSION_CANNOT_BE_ESTABLISHED = 6020
E_RESPONSE_STATUS_CODE_MGMT_SESSION_ESTABLISHMENT_TIMEOUT = 6021
E_RESPONSE_STATUS_CODE_INVALID_CMDTYPE = 6022
E_RESPONSE_STATUS_CODE_INVALID_ARGUMENTS = 6023
E_RESPONSE_STATUS_CODE_INSUFFICIENT_ARGUMENTS = 6024
E_RESPONSE_STATUS_CODE_MGMT_CONVERSION_ERROR = 6025
E_RESPONSE_STATUS_CODE_MGMT_CANCELLATION_FAILED = 6026
E_RESPONSE_STATUS_CODE_ALREADY_COMPLETE = 6028
E_RESPONSE_STATUS_CODE_MGMT_COMMAND_NOT_CANCELLABLE = 6029

AVP_ONEM2M_SUBGROUP_NAME = 1021
AVP_ONEM2M_TARGET_ID = 1022
