# Copyright (c) [2025] [OpenAI]
# Copyright (c) [2025] [ByteDance Ltd. and/or its affiliates.]
# SPDX-License-Identifier: Apache-2.0
#
# This file has been modified by [ByteDance Ltd. and/or its affiliates.] on 2025.7
#
# Original file was released under Apache License Version 2.0, with the full license text
# available at https://github.com/openai/openai-python/blob/main/LICENSE.
#
# This modified file is released under the same license.

from typing import List, Optional, Union

from typing_extensions import Literal, TypeAlias
from volcenginesdkarkruntime.types.shared.reasoning import Reasoning
from volcenginesdkarkruntime.types.shared.thinking import Thinking

from ..._models import BaseModel
from .response_caching import ResponseCaching
from .response_error import ResponseError
from .response_incomplete_details import IncompleteDetails
from .response_output_item import ResponseOutputItem
from .response_text_config import ResponseTextConfig
from .response_usage import ResponseUsage
from .tool import Tool
from .tool_choice_function import ToolChoiceFunction
from .tool_choice_knowledge_search import ToolChoiceKnowledgeSearch
from .tool_choice_mcp import ToolChoiceMcp
from .tool_choice_options import ToolChoiceOptions
from .tool_choice_web_search import ToolChoiceWebSearch

__all__ = ["Response", "ToolChoice"]


ToolChoice: TypeAlias = Union[
    ToolChoiceOptions,
    ToolChoiceFunction,
    ToolChoiceMcp,
    ToolChoiceWebSearch,
    ToolChoiceKnowledgeSearch,
]


class Response(BaseModel):
    created_at: int
    """Unix timestamp (in seconds) of when this Response was created."""

    error: Optional[ResponseError] = None
    """An error object returned when the model fails to generate a Response."""

    id: str
    """Unique identifier for this Response."""

    incomplete_details: Optional[IncompleteDetails] = None
    """Details about why the response is incomplete."""

    max_output_tokens: Optional[int] = None
    """An upper bound for the number of tokens that can be generated for a response."""

    model: str
    """Model Name or endpoint id."""

    object: Literal["response"]
    """The object type of this resource - always set to `response`."""

    output: List[ResponseOutputItem]
    """An array of content items generated by the model."""

    previous_response_id: Optional[str] = None
    """The unique ID of the previous response to the model. Use this to create multi-turn conversations."""

    thinking: Optional[Thinking] = None
    """The thinking mode to use for the model."""

    service_tier: Optional[Literal["auto", "default"]] = None
    """Specifies the latency tier to use for processing the request."""

    status: Literal["in_progress", "completed", "incomplete", "failed"]
    """The status of the response generation.  One of `completed`, `failed`, `in_progress`, or `incomplete`."""

    temperature: Optional[float] = None
    """What sampling temperature to use, between 0 and 2."""

    tools: List[Tool]
    """An array of tools the model may call while generating a response. You can specify which tool to use by setting the `tool_choice` parameter."""

    top_p: Optional[float] = None
    """An alternative to sampling with temperature, called nucleus sampling, where the model considers the results of the tokens with top_p probability mass. So 0.1 means only the tokens comprising the top 10% probability mass are considered."""

    usage: Optional[ResponseUsage] = None
    """Represents token usage details including input tokens, output tokens, a breakdown of output tokens, and the total tokens used."""

    caching: Optional[ResponseCaching] = None
    """Represents the caching configuration for the response."""

    text: Optional[ResponseTextConfig] = None
    """Configuration options for a text response from the model. Can be plain text or json object"""

    instructions: Optional[str] = None
    """Inserts a system (or developer) message as the first item in the model's context."""

    store: Optional[bool] = None
    """Whether to store the response object"""

    expire_at: Optional[int] = None
    """The expiration time of the response object"""

    tool_choice: Optional[ToolChoice] = None
    """How the model should select which tool (or tools) to use when generating a response. See the `tools` parameter to see how to specify which tools the model can call."""

    parallel_tool_calls: Optional[bool] = None
    """Whether to allow the model to run tool calls in parallel."""

    max_tool_calls: Optional[int] = None
    """The maximum number of total calls to built-in tools that can be processed in a response. This maximum number applies across all built-in tool calls, not per individual tool. Any further attempts to call a tool by the model will be ignored."""

    reasoning: Optional[Reasoning] = None
    """Configuration options for reasoning models"""
