# -*- coding: utf-8 -*-
"""
Components that do not require PyWake nor Floris.

@author: ricriv
"""

# %% Import.

from abc import ABC

import matplotlib as mpl
import matplotlib.pyplot as plt
import numpy as np
import pandas as pd
import xarray as xr
from scipy.interpolate import PchipInterpolator
from scipy.special import erf

# %% Classes and functions.


class SmoothPotFunctions(ABC):
    """Collection of functions to clip the output."""

    def pot_tanh(r, R, exponent=20):
        r"""
        Smooth pot function based on tanh.

        .. math::
          y = \mathrm{tanh}\left(\left(\frac{r}{R}\right)^{a}\right)

        Values where :math:`|r| < R` are mapped to 0, while values outside of it are mapped to 1.

        Parameters
        ----------
        r : ndarray
            Radial distance :math:`r`.
        R : ndarray
            Distance at which to clip :math:`R`.
        exponent : int, optional
            Exponent :math:`a`. The default is 20.

        Returns
        -------
        y : ndarray
            Smooth pot of radius :math:`R`.

        """
        return np.tanh((r / R) ** exponent)

    def pot_arctan(r, R, exponent=100):
        r"""
        Smooth pot function based on arctan.

        .. math::
          y = \frac{2}{\pi} \mathrm{arctan}\left(\left(\frac{r}{R}\right)^{a}\right)

        Values where :math:`|r| < R` are mapped to 0, while values outside of it are mapped to 1.

        Parameters
        ----------
        r : ndarray
            Radial distance :math:`r`.
        R : ndarray
            Distance at which to clip :math:`R`.
        exponent : int, optional
            Exponent :math:`a`. The default is 100.

        Returns
        -------
        y : ndarray
            Smooth pot of radius :math:`R`.

        """
        return 2.0 / np.pi * np.arctan((r / R) ** exponent)

    def pot_erf(r, R, exponent=20):
        r"""
        Smooth pot function based on the error function.

        .. math::
          y = \mathrm{erf}\left(\left(\frac{r}{R}\right)^{a}\right)

        Values where :math:`|r| < R` are mapped to 0, while values outside of it are mapped to 1.

        Parameters
        ----------
        r : ndarray
            Radial distance :math:`r`.
        R : ndarray
            Distance at which to clip :math:`R`.
        exponent : int, optional
            Exponent :math:`a`. The default is 20.

        Returns
        -------
        y : ndarray
            Smooth pot of radius :math:`R`.

        """
        return erf((r / R) ** exponent)

    def pot_sharp(r, R):
        r"""
        Sharp pot function.
        
        .. math::
          y = \begin{cases}
                  0  &  \left| r \right| < R,    \\
                  1  &  \left| r \right| \ge R
              \end{cases}

        Parameters
        ----------
        r : ndarray
            Radial distance :math:`r`.
        R : ndarray
            Distance at which to clip :math:`R`.

        Returns
        -------
        y : ndarray
            Sharp pot of radius :math:`R`.

        """
        y = np.ones_like(r)
        y[np.abs(r) < R] = 0.0
        return y


class PchipInterpolatorWithExtrap(PchipInterpolator):
    r"""PCHIP 1-D monotonic cubic interpolation, with bounded output.

    This class is the same as PchipInterpolator from Scipy. The difference is that
    it provides a new argument ``y_extrapolate``, which sets the extrapolation value.

    Parameters
    ----------
    x : ndarray, shape (npoints, )
        1D input grid.
    y : ndarray, shape (npoints, ny)
        2D output grid.
    y_extrapolate : array_like, shape (ny)
        1D output used for extrapolation.
    """

    def __init__(self, x, y, y_extrapolate):
        super().__init__(x=x, y=y, axis=0, extrapolate=False)
        self.y_extrapolate = y_extrapolate

    def __call__(self, x):
        """
        Evaluate the interpolator.

        Parameters
        ----------
        x : (N, M, ...) ndarray
            Input.

        Returns
        -------
        yp : (N, M, ..., ny) ndarray
            Output.

        """
        # Predict output.
        yp = super().__call__(x)
        # Replace the NaN in each output with the correct extrapolation value.
        for j in range(yp.shape[-1]):
            np.nan_to_num(yp[..., j], copy=False, nan=self.y_extrapolate[j])
        return yp


def load_ccblade_performance(file):
    """
    Load performance data generated by CCBlade.

    Parameters
    ----------
    file : str
        File path.

    Returns
    -------
    df : pandas DataFrame
        Table with the performance data.

    """
    df = pd.read_table(file, index_col=0)
    df.index.name = "Wind speed (m/s)"  # Remove #
    df.columns = [s.strip() for s in df.columns]
    return df


def _rot_x(x):
    """
    Rotation tensor around x axis.

    Parameters
    ----------
    x : float
        Angle in radian.

    Returns
    -------
    (3, 3) float ndarray
        Rotation tensor.

    Notes
    -----
    See `Wikipedia <https://en.wikipedia.org/wiki/Rotation_matrix#Basic_3D_rotations>`__.

    """
    s = np.sin(x)
    c = np.cos(x)
    # fmt: off
    return np.array([[1.0, 0.0, 0.0],
                     [0.0,   c,  -s],
                     [0.0,   s,   c]])
    # fmt: on


def _rot_y(x):
    """
    Rotation tensor around y axis.

    Parameters
    ----------
    x : float
        Angle in radian.

    Returns
    -------
    (3, 3) float ndarray
        Rotation tensor.

    Notes
    -----
    See `Wikipedia <https://en.wikipedia.org/wiki/Rotation_matrix#Basic_3D_rotations>`__.

    """
    s = np.sin(x)
    c = np.cos(x)
    # fmt: off
    return np.array([[c,   0.0,   s],
                     [0.0, 1.0, 0.0],
                     [-s,  0.0,   c]])
    # fmt: on


def _rot_z(x):
    """
    Rotation tensor around z axis.

    Parameters
    ----------
    x : float
        Angle in radian.

    Returns
    -------
    (3, 3) float ndarray
        Rotation tensor.

    Notes
    -----
    See `Wikipedia <https://en.wikipedia.org/wiki/Rotation_matrix#Basic_3D_rotations>`__.

    """
    s = np.sin(x)
    c = np.cos(x)
    # fmt: off
    return np.array([[c,    -s, 0.0],
                     [s,     c, 0.0],
                     [0.0, 0.0, 1.0]])
    # fmt: on


def make_rectangular_grid(y, z):
    """
    Make a rectangular grid. The grid is in the vertical plane for 0 yaw and tilt.

    This class is meant to work with PyWake, and therefore it uses the same reference frame.
    Usually, the grid will be generated with the rotor center in (0, 0).

    Parameters
    ----------
    y : (N, ) array_like
        1D array with the y coordinate (crosswind).
    z : (M, ) array_like
        1D array with the z coordinate (up).

    Returns
    -------
    x_2d : (N, M) ndarray
        x coordinate (downwind) of the grid points. Always 0.
    y_2d : (N, M) ndarray
        y coordinate (crosswind) of the grid points.
    z_2d : (N, M) ndarray
        z coordinate (up) of the grid points.

    """
    y_2d, z_2d = np.meshgrid(y, z, indexing="ij")
    # For yaw = tilt = 0 the grid is in the vertical plane.
    # Here we might return a view, but choose instead to allocate because later we might rotate this grid.
    x_2d = np.zeros_like(y_2d)
    return x_2d, y_2d, z_2d


def make_polar_grid(radius, azimuth, degrees=False):
    """
    Make a regular grid using polar coordinates. The grid is in the vertical plane for 0 yaw and tilt.

    This class is meant to work with PyWake, and therefore it uses the same reference frame.
    Usually, the grid will be generated with the rotor center in (0, 0).

    Parameters
    ----------
    radius : (N, ) array_like
        Array of radius. The rotor center is at 0 and grows towards the blade tip.
        For example, `radius = np.linspace(0.0, 100.0, 10)`
    azimuth : (M, ) array_like
        Array of azimuth angles in radians, typically [0, 2*pi).
        0 is horizontal (crosswind) and grows clockwise looking downwind.
        For example, `azimuth = np.linspace(0.0, 2.0*np.pi, 20, endpoint=False)`.
        Due to periodicity, it is best to skip 2*pi, and thus set `endpoint=False`.
    degrees : bool, optional
        If `True`, then the given angles are assumed to be in degrees. Default is `False`, which means radian.

    Returns
    -------
    x_2d : (N, M) ndarray
        x coordinate (downwind) of the grid points. Always 0.
    y_2d : (N, M) ndarray
        y coordinate (crosswind) of the grid points.
    z_2d : (N, M) ndarray
        z coordinate (up) of the grid points.

    """
    # This is more efficient than:
    #    radius_2d, azimuth_2d = np.meshgrid(radius, azimuth, indexing="ij")
    radius_2d = np.broadcast_to(radius[:, np.newaxis], (radius.size, azimuth.size))
    azimuth_2d = np.broadcast_to(azimuth[np.newaxis, :], (radius.size, azimuth.size))
    if degrees:
        azimuth_2d = np.deg2rad(azimuth_2d)
    y_2d = radius_2d * np.cos(azimuth_2d)
    z_2d = radius_2d * np.sin(azimuth_2d)
    # For yaw = tilt = 0 the grid is in the vertical plane.
    # Here we might return a view, but choose instead to allocate because later we might rotate this grid.
    x_2d = np.zeros_like(y_2d)
    return x_2d, y_2d, z_2d


def _cross(a, b):
    """
    Return the cross product of two vectors.

    Parameters
    ----------
    a : (3,) array_like
        First vector.
    a : (3,) array_like
        Second vector.

    Returns
    -------
    v : (3,) ndarray
        Cross product of a and b.

    """
    # Adapted from _cross3 in
    # https://github.com/scipy/scipy/blob/main/scipy/spatial/transform/_rotation.pyx
    # This function is much simpler than the one in numpy, and consequently has less overhead.
    return np.array(
        [
            a[1] * b[2] - a[2] * b[1],
            a[2] * b[0] - a[0] * b[2],
            a[0] * b[1] - a[1] * b[0],
        ]
    )


def _compose_quat_single(p, q):
    # Adapted from _compose_quat_single in
    # https://github.com/scipy/scipy/blob/main/scipy/spatial/transform/_rotation.pyx

    cross = _cross(p[:3], q[:3])

    return np.array(
        [
            p[3] * q[0] + q[3] * p[0] + cross[0],
            p[3] * q[1] + q[3] * p[1] + cross[1],
            p[3] * q[2] + q[3] * p[2] + cross[2],
            p[3] * q[3] - p[0] * q[0] - p[1] * q[1] - p[2] * q[2],
        ]
    )


def _compose_quat(p, q):
    """Compose quaternions."""
    # Adapted from _compose_quat in
    # https://github.com/scipy/scipy/blob/main/scipy/spatial/transform/_rotation.pyx
    n = q.shape[0] if p.shape[0] == 1 else p.shape[0]

    product = np.zeros((n, 4))

    # dealing with broadcasting
    if p.shape[0] == 1:
        for ind in range(n):
            product[ind, :] = _compose_quat_single(p[0, :], q[ind, :])
    elif q.shape[0] == 1:
        for ind in range(n):
            product[ind, :] = _compose_quat_single(p[ind, :], q[0, :])
    else:
        for ind in range(n):
            product[ind, :] = _compose_quat_single(p[ind, :], q[ind, :])

    return product


def _make_elementary_quat(axis, angles):
    """
    Convert from axis-angle representation to quaternion using the Hamilton representation.

    Parameters
    ----------
    axis : int
        Rotation axes. Must be 0, 1 or 2.
    angles : (N, 3) ndarray
        Rotation angles in rad.

    Returns
    -------
    quat : (N, 4) ndarray
        Quaternions.

    """
    # Adapted from _make_elementary_quat in
    # https://github.com/scipy/scipy/blob/main/scipy/spatial/transform/_rotation.pyx
    # See formula at https://en.wikipedia.org/wiki/Conversion_between_quaternions_and_Euler_angles#Definition
    n = angles.shape[0]
    quat = np.zeros((n, 4))

    match axis:  # Requires python >= 3.10
        case "x":
            axis_ind = 0
        case "y":
            axis_ind = 1
        case "z":
            axis_ind = 2

    half_angle = angles / 2
    for ind in range(n):
        quat[ind, axis_ind] = np.sin(half_angle[ind])
        quat[ind, 3] = np.cos(half_angle[ind])

    return quat


def _elementary_quat_compose(seq, angles, intrinsic):
    """Compose a sequence of rotations using quaternions."""
    # Adapted from _elementary_quat_compose in
    # https://github.com/scipy/scipy/blob/main/scipy/spatial/transform/_rotation.pyx

    result = _make_elementary_quat(seq[0], angles[:, 0])
    seq_len = len(seq)

    for idx in range(1, seq_len):
        if intrinsic:
            result = _compose_quat(
                result, _make_elementary_quat(seq[idx], angles[:, idx])
            )
        else:
            result = _compose_quat(
                _make_elementary_quat(seq[idx], angles[:, idx]), result
            )
    return result


def _quat_as_matrix(quat):
    """
    Represent a quaternion as a rotation tensor.

    Parameters
    ----------
    quat : (4,) array_like
        Quaternion.

    Returns
    -------
    matrix : (3, 3) ndarray
        Rotation tensor.

    """
    # Adapted from as_matrix in
    # https://github.com/scipy/scipy/blob/main/scipy/spatial/transform/_rotation.pyx
    # Formula from the wiki
    # https://en.wikipedia.org/wiki/Conversion_between_quaternions_and_Euler_angles#Rotation_matrices
    x, y, z, w = quat

    x2 = x * x
    y2 = y * y
    z2 = z * z
    w2 = w * w

    xy = x * y
    zw = z * w
    xz = x * z
    yw = y * w
    yz = y * z
    xw = x * w

    matrix = np.zeros((3, 3))

    matrix[0, 0] = x2 - y2 - z2 + w2
    matrix[1, 0] = 2 * (xy + zw)
    matrix[2, 0] = 2 * (xz - yw)

    matrix[0, 1] = 2 * (xy - zw)
    matrix[1, 1] = -x2 + y2 - z2 + w2
    matrix[2, 1] = 2 * (yz + xw)

    matrix[0, 2] = 2 * (xz + yw)
    matrix[1, 2] = 2 * (yz - xw)
    matrix[2, 2] = -x2 - y2 + z2 + w2

    return matrix


def rotate_grid(x_2d, y_2d, z_2d, yaw=0.0, tilt=0.0, degrees=False):
    """
    Rotate in yaw and tilt the grid generated by `rectangular_grid` or `polar_grid`.

    This class is meant to work with PyWake, and therefore it uses the same reference frame.
    The PyWake reference frame convention is illustrated in
    `R. Riva et al., Incorporation of floater rotation and displacement in a static wind farm simulator <https://iopscience.iop.org/article/10.1088/1742-6596/2767/6/062019>`_
    and described hereafter.

     - Cartesian reference frame.
     - Origin in the rotor center.
     - x axis horizontal, positive downwind.
     - y axis horizontal, positive left when looking downwind. It is the crosswind.
     - z axis vertical, positive up.
     - yaw angle is measured starting from y, positive counter-clockwise when looking down.
     - tilt angle is measured starting from z, positive clockwise when looking crosswind.
       That is, tilt causes the top of the rotor to move downwind.

    It is assumed that the turbine is bottom-fixed. Therefore, the order of rotations is first yaw around the vertical axis (z),
    and then tilt.

    Parameters
    ----------
    x_2d : (N, M) ndarray
        x coordinates (downwind) of the grid points.
    y_2d : (N, M) ndarray
        y coordinates (crosswind) of the grid points.
    z_2d : (N, M) ndarray
        z coordinates (up) of the grid points.
    yaw : float, optional
        Rotor yaw angle. The default is 0.
    tilt : float, optional
        Rotor tilt angle. The default is 0.
    degrees : bool, optional
        If `True`, then the given angles are assumed to be in degrees. Default is `False`, which means radian.

    Returns
    -------
    x_2d : (N, M) ndarray
        x coordinates (downwind) of the grid points.
    y_2d : (N, M) ndarray
        y coordinates (crosswind) of the grid points.
    z_2d : (N, M) ndarray
        z coordinates (up) of the grid points.

    """
    angles = np.array([[tilt, yaw]])

    if degrees:
        angles = np.deg2rad(angles)

    # Compute quaternion for this rotation.
    quat = _elementary_quat_compose("yz", angles, intrinsic=False)[0, :]

    # Get equivalent rotation tensor.
    matrix = _quat_as_matrix(quat)

    # Rotate the grid.
    xyz = np.vstack((x_2d.ravel(), y_2d.ravel(), z_2d.ravel()))
    xyz = matrix @ xyz

    # Get each component.
    x_2d_ = xyz[0, :].reshape(x_2d.shape)
    y_2d_ = xyz[1, :].reshape(y_2d.shape)
    z_2d_ = xyz[2, :].reshape(z_2d.shape)

    return x_2d_, y_2d_, z_2d_


def plot_grid(x_2d, y_2d, z_2d, close_grid=False, fig=None):
    """
    Make a 3D plot with the grid generated by `rectangular_grid` or `polar_grid`.

    Parameters
    ----------
    x_2d : (N, M) ndarray
        X coordinates of the grid points.
    y_2d : (N, M) ndarray
        Y coordinates of the grid points.
    z_2d : (N, M) ndarray
        Z coordinates of the grid points.
    close_grid : bool
        If `True` the first azimuth value (column) is repeated to allow closing the mesh.
        This only makes sense for polar grids, and therefore the default is `False`.
    fig : matplotlib Figure, optional
        Figure to plot the grid. The default is `None`, in which case a new one will be created.

    Returns
    -------
    ax :  matplotlib Axes
        Matplotlib axes.

    """
    # Make the data periodic?
    if close_grid:
        x = np.concatenate((x_2d, x_2d[:, [0]]), axis=1)
        y = np.concatenate((y_2d, y_2d[:, [0]]), axis=1)
        z = np.concatenate((z_2d, z_2d[:, [0]]), axis=1)
    else:
        x = x_2d
        y = y_2d
        z = z_2d

    # Plot.
    if fig is None:
        fig = plt.figure()
    ax = fig.add_subplot(projection="3d")
    ax.set_proj_type("ortho")
    ax.set_aspect("equal")
    range_x = np.maximum(np.ptp(x), 2.0)
    range_y = np.maximum(np.ptp(y), 2.0)
    range_z = np.maximum(np.ptp(z), 2.0)
    ax.set_box_aspect((range_x, range_y, range_z))
    ax.set_xlabel("x (downwind)")
    ax.set_ylabel("y (crosswind)")
    ax.set_zlabel("z (up)")
    ax.invert_xaxis()
    ax.invert_yaxis()
    ax.scatter(x_2d.ravel(), y_2d.ravel(), z_2d.ravel())
    ax.plot_wireframe(x, y, z)
    return ax


def _preallocate_ilktn(wt, wd=None, ws=None, time=None, dtype=np.float32, **names):
    """
    Preallocate a xarray `DataArray` to store a quantity dependent on inflow and names. Typically used to store loads and sector-average.

    The shape of the quantity depends if time is used or not.

        - Without time the shape is `(turbine, wind direction, wind speed, name_0, name_1, ...)`.
        - With time the shape is `(turbine, time, name_0, name_1, ...)`. If provided, wind direction and wind speed are saved as time-dependent coordinates.

    Parameters
    ----------
    wt : list_like
        Wind turbines.
    wd : array_like, optional
        Wind directions. Required if `time` is not `None`.
    ws : array_like, optional
        Wind speeds. Required if `time` is not `None`.
    time : array_like, optional
        Time. If provided, `wd` and `ws` become time-dependent coordinates.
    dtype : data-type, optional
        The desired data-type for the result. The default is single precision,
        which should be enough for all outputs. The properties of each type can
        be checked with `np.finfo(np.float32(1.0))`.
    names : dict of list_like
        Additional coordinate names.

    Returns
    -------
    quantity : xarray.DataArray
        Quantity for each turbine, inflow condition and name.

    """
    if time is None:
        quantity = xr.DataArray(
            data=np.full(
                (
                    len(wt),
                    len(wd),
                    len(ws),
                    *[len(x) for x in names.values()],
                ),
                np.nan,
                dtype=dtype,
            ),
            coords={
                "wt": np.asarray(wt),
                "wd": np.asarray(wd),
                "ws": np.asarray(ws),
                **names,
            },
        )
    else:
        # Set the independent coordinates: turbine, time and names.
        time_ = np.asarray(time)
        coords = {
            "wt": np.asarray(wt),
            "time": time_,
            **names,
        }
        dims = list(coords)

        # Set the dependent coordinates: wind direction and wind speed.
        if wd is not None:
            wd_ = np.asarray(wd)
            if wd_.size == 1:
                wd_ = np.broadcast_to(wd_, time_.shape)
            coords["wd"] = (["time"], wd_)
        if ws is not None:
            ws_ = np.asarray(ws)
            if ws_.size == 1:
                ws_ = np.broadcast_to(ws_, time_.shape)
            coords["ws"] = (["time"], ws_)

        # Allocate the quantity.
        quantity = xr.DataArray(
            data=np.full(
                (
                    len(wt),
                    time_.size,
                    *[len(x) for x in names.values()],
                ),
                np.nan,
                dtype=dtype,
            ),
            coords=coords,
            dims=dims,
        )
    return quantity


def _get_sensor_names(surrogates):
    """
    Get the sensor names from a dictionary of surrogate models.

    When a surrogate model has only 1 output, then its name is used directly.
    If instead the surrogate has multiple outputs, then a set of names in the
    form `surrogate_name.out_0, surrogate_name.out_1, ...` is created., where
    `out_i` are the surrogate `output_names`.

    Parameters
    ----------
    surrogates : dict of surrogates_interface.surrogates.SurrogateModel
        Dictionary of surrogate models.

    Returns
    -------
    names : list of str
        Names for all output sensors.

    """
    names = []
    for key, val in surrogates.items():
        if val.n_outputs == 1:
            names.append(key)
        else:
            names.extend([f"{key}.{out}" for out in val.output_names])
    return names


def plot_flow_map(flow_map, wd=None, ws=None, wt=None, time=None, quantity="WS_eff"):
    """
    Plot the effective wind speed and Turbulence Intensity over each rotor for any inflow condition.

    Parameters
    ----------
    flow_map : xarray DataSet
        Flow map generated by `compute_flow_map`. It contains the effective wind speed,
        effective turbulence intensity and corresponding grid points for each turbine and flow case.
    wt : int, (I) array_like, optional
        Wind turbines. Must be a subset of the one contained in `flow_map`.
        The default is `None`, which means to use all available wind turbines.
    wd : float, (L) array_like, optional
        Wind direction, in deg. Must be a subset of the one contained in `flow_map`.
        The default is `None`, which means to use all available wind directions.
        It is ignored if `time` is not `None`.
    ws : float, (K) array_like, optional
        Wind speed. Must be a subset of the one contained in `flow_map`.
        The default is `None`, which means to use all available wind speeds.
        It is ignored if `time` is not `None`.
    time : float, (Time) array_like, optional
        Time. Must be a subset of the one contained in `flow_map`.
        The default is `None`, which means to use all available time instants.
    quantity : str, optional
        Quantity to plot. Can be `"WS_eff"` or `"TI_eff"`. The default is `"WS_eff"`.

    Returns
    -------
    None.

    """
    if quantity == "WS_eff":
        cmap = "Blues_r"
        label = "Effective wind speed [m/s]"
    elif quantity == "TI_eff":
        cmap = "Oranges"
        label = "Effective Turbulence Intensity [-]"
    else:
        raise ValueError("quantity must be 'WS_eff' or 'TI_eff'.")

    wt_ = flow_map["wt"].values if wt is None else np.atleast_1d(wt)
    if "time" in flow_map.dims:
        time_ = flow_map["time"].values if time is None else np.atleast_1d(time)
        for wt_i in wt_:
            for time_t in time_:
                fig = plt.figure()
                fig.suptitle(f"Turbine {wt_i}, Time {time_t}")
                ax = fig.add_subplot(projection="3d")
                ax.set_proj_type("ortho")
                ax.set_aspect("equal")
                x_grid = (
                    flow_map["grid"]
                    .loc[{"wt": wt_i, "time": time_t, "axis": "x"}]
                    .values
                )
                y_grid = (
                    flow_map["grid"]
                    .loc[{"wt": wt_i, "time": time_t, "axis": "y"}]
                    .values
                )
                z_grid = (
                    flow_map["grid"]
                    .loc[{"wt": wt_i, "time": time_t, "axis": "z"}]
                    .values
                )
                ptp_x = np.ptp(x_grid)
                ptp_y = np.ptp(y_grid)
                ptp_z = np.ptp(z_grid)
                ptp_max = np.max(np.array([ptp_x, ptp_y, ptp_z]))
                ptp_min = 0.1 * ptp_max
                range_x = np.maximum(ptp_x, ptp_min)
                range_y = np.maximum(ptp_y, ptp_min)
                range_z = np.maximum(ptp_z, ptp_min)
                ax.set_box_aspect((range_x, range_y, range_z))
                ax.set_xlabel("x (east)")
                ax.set_ylabel("y (north)")
                ax.set_zlabel("z (up)")
                # ax.invert_xaxis()
                # ax.invert_yaxis()
                patch_ws = ax.scatter(
                    x_grid.ravel(),
                    y_grid.ravel(),
                    z_grid.ravel(),
                    c=flow_map["flow"]
                    .loc[{"wt": wt_i, "time": time_t, "quantity": quantity}]
                    .values,
                    cmap=cmap,
                )
                plt.colorbar(patch_ws, label=label, ax=ax)
                plt.tight_layout()

    else:  # "time" not in flow_map.dims
        wd_ = flow_map["wd"].values if wd is None else np.atleast_1d(wd)
        ws_ = flow_map["ws"].values if ws is None else np.atleast_1d(ws)
        for wt_i in wt_:
            for wd_l in wd_:
                for ws_k in ws_:

                    fig = plt.figure()
                    fig.suptitle(
                        f"Turbine {wt_i}, Wind direction {wd_l}, Wind speed {ws_k}"
                    )
                    ax = fig.add_subplot(projection="3d")
                    ax.set_proj_type("ortho")
                    ax.set_aspect("equal")
                    x_grid = (
                        flow_map["grid"]
                        .loc[{"wt": wt_i, "wd": wd_l, "ws": ws_k, "axis": "x"}]
                        .values
                    )
                    y_grid = (
                        flow_map["grid"]
                        .loc[{"wt": wt_i, "wd": wd_l, "ws": ws_k, "axis": "y"}]
                        .values
                    )
                    z_grid = (
                        flow_map["grid"]
                        .loc[{"wt": wt_i, "wd": wd_l, "ws": ws_k, "axis": "z"}]
                        .values
                    )
                    ptp_x = np.ptp(x_grid)
                    ptp_y = np.ptp(y_grid)
                    ptp_z = np.ptp(z_grid)
                    ptp_max = np.max(np.array([ptp_x, ptp_y, ptp_z]))
                    ptp_min = 0.1 * ptp_max
                    range_x = np.maximum(ptp_x, ptp_min)
                    range_y = np.maximum(ptp_y, ptp_min)
                    range_z = np.maximum(ptp_z, ptp_min)
                    ax.set_box_aspect((range_x, range_y, range_z))
                    ax.set_xlabel("x (east)")
                    ax.set_ylabel("y (north)")
                    ax.set_zlabel("z (up)")
                    # ax.invert_xaxis()
                    # ax.invert_yaxis()
                    patch_ws = ax.scatter(
                        x_grid.ravel(),
                        y_grid.ravel(),
                        z_grid.ravel(),
                        c=flow_map["flow"]
                        .loc[{"wt": wt_i, "wd": wd_l, "ws": ws_k, "quantity": quantity}]
                        .values,
                        cmap=cmap,
                    )
                    plt.colorbar(patch_ws, label=label, ax=ax)
                    plt.tight_layout()


def compute_sector_average(
    sim,
    n_radius=100,
    n_azimuth=181,
    look="downwind",
    align_in_yaw=True,
    align_in_tilt=True,
    axial_wind=False,
    wt=None,
    wd=None,
    ws=None,
    time=None,
    dtype=np.float32,
):
    """
    Compute the sector-averaged effective wind speed and effective turbulence intensity, assuming 4 sectors.

    Each sector spans 90 deg in azimuth and they are oriented as left, up, right and down, as
    described in `Guilloré, A., Campagnolo, F. & Bottasso, C. L. (2024). A control-oriented load surrogate model based on sector-averaged inflow quantities: capturing damage for unwaked, waked, wake-steering and curtailed wind turbines <https://doi.org/10.1088/1742-6596/2767/3/032019>`_

    The result can be visualized via `wind_farm_loads.tool_agnostic.plot_sector_average()`.

    Parameters
    ----------
    sim : py_wake SimulationResult or floris FlorisModel
        Either:

         - Floris model. Must follow a call to `run()`.
         - Simulation result computed by PyWake. Must follow a call to the wind farm model.

    n_radius : int, optional
        Number of points along the rotor radius. The default is 100.
    n_azimuth : int, optional
        Number of points along the rotor azimuth. Must be odd.
        The default is 181, which ensures 2 deg per azimuth step.
    look : str, optional
        The left and right sectors are determined by an observer that can look
        `"upwind"` or `"downwind"`. The default is `"downwind"`.
    align_in_yaw : bool, optional
        If `True` (default) the grid is aligned in yaw with the rotor plane.
    align_in_tilt : bool, optional
        If `True` (default) the grid is aligned in tilt with the rotor plane.
    axial_wind : bool, optional
        If `True` the axial wind speed and TI are returned. The default is `False`.
    wt : int, (I) array_like, optional
        Wind turbines. Must be a subset of the one contained in `sim_res`.
        The default is `None`, which means to use all available wind turbines.
    wd : float, (L) array_like, optional
        Wind direction, in deg. Must be a subset of the one contained in `sim_res`.
        The default is `None`, which means to use all available wind directions.
    ws : float, (K) array_like, optional
        Wind speed. Must be a subset of the one contained in `sim_res`.
        The default is `None`, which means to use all available wind speeds.
    time : float, (Time) array_like, optional
        Time. Must be a subset of the one contained in `sim_res`.
        The default is `None`, which means to use all available time instants.
    dtype : data-type, optional
        The desired data-type for the result. The default is single precision,
        which should be enough for all outputs. The properties of each type can
        be checked with `np.finfo(np.float32(1.0))`.

    Returns
    -------
    sa : xarray DataArray
        Sector-averaged wind speed, effective turbulence intensity for each turbine and flow case.

    """
    # Part of this function that depends on the tool.
    if type(sim).__name__ == "FlorisModel":
        diameters = sim.core.farm.rotor_diameters[0, :]  # First findex.
        # In this context, we only care about the diameter to determine the type.
        # An alternative would be:
        #   len(fmodel.core.farm.turbine_type)
        # But that might create more types than intended.
        n_types = np.unique(diameters).size
        from wind_farm_loads.floris import compute_flow_map
    elif type(sim).__name__ == "SimulationResult":  # PyWake
        diameters = sim.windFarmModel.windTurbines._diameters
        n_types = len(diameters)
        from wind_farm_loads.py_wake import compute_flow_map
    else:
        raise TypeError(
            f"Input 'sim' must be a floris FlorisModel or a py-wake SimulationResult. Received {type(sim).__name__}."
        )

    # Tool-agnostic part.

    # Check that n_azimuth is odd.
    assert n_azimuth % 2 == 1

    # Make the azimuth grid. We cover 360 deg starting from -45, which is the
    # beginning of the left sector.
    # We also add 1 more point at the end for easier slicing later on.
    dazim = 360.0 / (n_azimuth - 1)  # Azimuth step.
    azimuth = np.linspace(-45.0, 360.0 - 45.0 + dazim, n_azimuth + 1, endpoint=True)

    # Make the polar grid for each turbine type.
    x_grid = np.zeros((n_radius, n_azimuth + 1, n_types))
    y_grid = np.zeros((n_radius, n_azimuth + 1, n_types))
    z_grid = np.zeros((n_radius, n_azimuth + 1, n_types))
    for i in range(n_types):
        radius = np.linspace(0.0, 0.5 * diameters[i], n_radius)
        x_grid[:, :, i], y_grid[:, :, i], z_grid[:, :, i] = make_polar_grid(
            radius, azimuth, degrees=True
        )

    # Get the flow map.
    flow_map = compute_flow_map(
        sim,
        x_grid,
        y_grid,
        z_grid,
        align_in_yaw=align_in_yaw,
        align_in_tilt=align_in_tilt,
        axial_wind=axial_wind,
        wt=wt,
        wd=wd,
        ws=ws,
        time=time,
        dtype=dtype,
    )

    # Preallocate a xarray `DataArray` to store the sector-averaged wind speed and turbulence intensity.
    # Assume 4 sectors: up, right, down and left.
    sa = _preallocate_ilktn(
        wt=flow_map["wt"].values,
        wd=flow_map["wd"].values,
        ws=flow_map["ws"].values,
        time=flow_map["time"].values if "time" in flow_map.dims else None,
        dtype=dtype,
        quantity=["WS_eff", "TI_eff"],
        direction=["up", "right", "down", "left"],
    )

    # Number of azimuth steps in each sector.
    i_sector_size = (n_azimuth - 1) // 4
    # Compute the sector average for the up and down sectors.
    # q0 is radius and q1 is azimuth.
    sa.loc[{"direction": "up"}] = flow_map["flow"][
        ..., i_sector_size : 2 * i_sector_size
    ].mean(dim=["q0", "q1"])
    sa.loc[{"direction": "down"}] = flow_map["flow"][
        ..., 3 * i_sector_size : n_azimuth
    ].mean(dim=["q0", "q1"])

    # Compute the sector average for the left and right sectors.
    if look == "upwind":
        sa.loc[{"direction": "left"}] = flow_map["flow"][..., 0:i_sector_size].mean(
            dim=["q0", "q1"]
        )
        sa.loc[{"direction": "right"}] = flow_map["flow"][
            ..., 2 * i_sector_size : 3 * i_sector_size
        ].mean(dim=["q0", "q1"])

    elif look == "downwind":
        sa.loc[{"direction": "right"}] = flow_map["flow"][..., 0:i_sector_size].mean(
            dim=["q0", "q1"]
        )
        sa.loc[{"direction": "left"}] = flow_map["flow"][
            ..., 2 * i_sector_size : 3 * i_sector_size
        ].mean(dim=["q0", "q1"])

    else:
        raise ValueError("Parameter look must be 'upwind' or 'downwind'")

    return sa


def plot_sector_average(
    sector_average, wd=None, ws=None, wt=None, time=None, quantity="WS_eff"
):
    """
    Plot the sector-average effective wind speed and Turbulence Intensity for each rotor and inflow condition.

    Parameters
    ----------
    sector_average : xarray DataArray
        Sector average generated by `compute_sector_average`. It contains the effective wind speed,
        effective turbulence intensity and farm layout for each turbine and flow case.
    wt : int, (I) array_like, optional
        Wind turbines. Must be a subset of the one contained in `flow_map`.
        The default is `None`, which means to use all available wind turbines.
    wd : float, (L) array_like, optional
        Wind direction, in deg. Must be a subset of the one contained in `flow_map`.
        The default is `None`, which means to use all available wind directions.
        It is ignored if `time` is not `None`.
    ws : float, (K) array_like, optional
        Wind speed. Must be a subset of the one contained in `flow_map`.
        The default is `None`, which means to use all available wind speeds.
        It is ignored if `time` is not `None`.
    time : float, (Time) array_like, optional
        Time. Must be a subset of the one contained in `sim_res`.
        The default is `None`, which means to use all available time instants.
    quantity : str, optional
        Quantity to plot. Must be `"WS_eff"` or `"TI_eff"`. The default is `"WS_eff"`.

    Returns
    -------
    None.

    """
    if quantity == "WS_eff":
        cmap = plt.get_cmap("Blues_r")
        label = "Effective wind speed [m/s]"
        format_str = ".2f"
    elif quantity == "TI_eff":
        cmap = plt.get_cmap("Oranges")
        label = "Effective Turbulence Intensity [-]"
        format_str = ".2%"
    else:
        raise ValueError("Parameter quantity must be 'WS_eff' or 'TI_eff'.")

    wt_ = sector_average["wt"].values if wt is None else np.atleast_1d(wt)
    if "time" in sector_average.dims:
        time_ = sector_average["time"].values if time is None else np.atleast_1d(time)
        for wt_i in wt_:
            for time_t in time_:
                fig, ax = plt.subplots()
                fig.suptitle(f"Turbine {wt_i}, Time {time_t}")
                data = sector_average.loc[
                    {"wt": wt_i, "time": time_t, "quantity": quantity}
                ].values
                normalizer = mpl.colors.Normalize(
                    vmin=0.9 * data.min(), vmax=1.1 * data.max()
                )
                colors = cmap(normalizer(data))

                def get_data_str(_):
                    """Get data for the pie chart."""
                    # This function will be called 4 times (once per sector).
                    # Remember the last index and increment it by 1 to get the current sector.
                    try:
                        get_data_str.i_sector += 1
                    except AttributeError:
                        get_data_str.i_sector = 0
                    return f"{data[get_data_str.i_sector]:{format_str}}"

                # To draw the sectors we use a pie chart.
                # Unfortunately, this means that the visualization is 2D.
                ax.pie(
                    (1, 1, 1, 1),
                    explode=None,
                    labels=sector_average["direction"].values,
                    colors=colors,
                    autopct=get_data_str,
                    startangle=135,
                    counterclock=False,
                    wedgeprops={"edgecolor": "k"},
                )
                plt.colorbar(
                    mpl.cm.ScalarMappable(norm=normalizer, cmap=cmap),
                    label=label,
                    ax=ax,
                )
                plt.tight_layout()

    else:  # "time" not in sector_average.dims
        wd_ = sector_average["wd"].values if wd is None else np.atleast_1d(wd)
        ws_ = sector_average["ws"].values if ws is None else np.atleast_1d(ws)
        for wt_i in wt_:
            for wd_l in wd_:
                for ws_k in ws_:

                    fig, ax = plt.subplots()
                    fig.suptitle(
                        f"Turbine {wt_i}, Wind direction {wd_l}, Wind speed {ws_k}"
                    )
                    data = sector_average.loc[
                        {"wt": wt_i, "wd": wd_l, "ws": ws_k, "quantity": quantity}
                    ].values
                    normalizer = mpl.colors.Normalize(
                        vmin=0.9 * data.min(), vmax=1.1 * data.max()
                    )
                    colors = cmap(normalizer(data))

                    def get_data_str(_):
                        """Get data for the pie chart."""
                        # This function will be called 4 times (once per sector).
                        # Remember the last index and increment it by 1 to get the current sector.
                        try:
                            get_data_str.i_sector += 1
                        except AttributeError:
                            get_data_str.i_sector = 0
                        return f"{data[get_data_str.i_sector]:{format_str}}"

                    # To draw the sectors we use a pie chart.
                    # Unfortunately, this means that the visualization is 2D.
                    ax.pie(
                        (1, 1, 1, 1),
                        explode=None,
                        labels=sector_average["direction"].values,
                        colors=colors,
                        autopct=get_data_str,
                        startangle=135,
                        counterclock=False,
                        wedgeprops={"edgecolor": "k"},
                    )
                    plt.colorbar(
                        mpl.cm.ScalarMappable(norm=normalizer, cmap=cmap),
                        label=label,
                        ax=ax,
                    )
                    plt.tight_layout()


def _predict_loads_pod(
    surrogates,
    flow_map,
    expand_shape_fun,
    *additional_inputs,
    dtype=np.float32,
    ti_in_percent=True,
):
    """Tool-agnostic part of `predict_loads_pod()`."""
    # Preallocate a DataArray for the results.
    loads = _preallocate_ilktn(
        wt=flow_map["wt"].values,
        wd=flow_map["wd"].values,
        ws=flow_map["ws"].values,
        time=flow_map["time"].values if "time" in flow_map.dims else None,
        name=_get_sensor_names(surrogates),
        dtype=dtype,
    )

    # Multiply the turbulence intensity by 100?
    if ti_in_percent:
        ti = flow_map["flow"].loc[{"quantity": "TI_eff"}].values * 100.0
    else:
        ti = flow_map["flow"].loc[{"quantity": "TI_eff"}].values

    # Ensure that the additional inputs have shape (wt, wd, ws) or (wt, time), then reshape to column array.
    # We take the sizes from the flow map, rather than sim_res, because the flow map might contain a subset of it.
    if "time" in flow_map.dims:
        theta = [
            expand_shape_fun(th, flow_map["wt"].size, flow_map["time"].size)
            .astype(dtype)
            .reshape(-1, 1)
            for th in additional_inputs
        ]
    else:
        theta = [
            expand_shape_fun(
                th,
                flow_map["wt"].size,
                flow_map["wd"].size,
                flow_map["ws"].size,
            )
            .astype(dtype)
            .reshape(-1, 1)
            for th in additional_inputs
        ]

    # Compose input for load surrogate.
    # We want points on the rows and features on the columns, hence the reshape.
    # Points are: wt, wd and ws or wt and time.
    # Features are: WS, TI and additional inputs.
    if "time" in flow_map.dims:
        shape_flow = (
            flow_map["wt"].size * flow_map["time"].size,
            1  # 1 because each quantity (WS and TI) is kept separate.
            * flow_map["q0"].size
            * flow_map["q1"].size,
        )
        shape_load = (flow_map["wt"].size, flow_map["time"].size)
    else:
        shape_flow = (
            flow_map["wt"].size * flow_map["wd"].size * flow_map["ws"].size,
            1  # 1 because each quantity (WS and TI) is kept separate.
            * flow_map["q0"].size
            * flow_map["q1"].size,
        )
        shape_load = (flow_map["wt"].size, flow_map["wd"].size, flow_map["ws"].size)
    x = np.concatenate(
        (
            flow_map["flow"]
            .loc[{"quantity": "WS_eff"}]
            .values.astype(dtype)
            .reshape(shape_flow),
            ti.astype(dtype).reshape(shape_flow),
            *theta,
        ),
        axis=1,
    )

    # Evaluate loads.
    for sensor in list(surrogates.keys()):
        loads.loc[{"name": sensor}] = (
            surrogates[sensor].predict_output(x).reshape(shape_load)
        )

    return loads


def _predict_loads_sector_average(
    surrogates,
    sector_average,
    expand_shape_fun,
    *additional_inputs,
    dtype=np.float32,
    ti_in_percent=True,
):
    """Tool-agnostic part of `predict_loads_sector_average()`."""
    # Preallocate a DataArray for the results.
    loads = _preallocate_ilktn(
        wt=sector_average["wt"].values,
        wd=sector_average["wd"].values,
        ws=sector_average["ws"].values,
        time=sector_average["time"].values if "time" in sector_average.dims else None,
        name=_get_sensor_names(surrogates),
        dtype=dtype,
    )

    # Multiply the turbulence intensity by 100?
    if ti_in_percent:
        ti = sector_average.loc[{"quantity": "TI_eff"}].values * 100.0
    else:
        ti = sector_average.loc[{"quantity": "TI_eff"}].values

    # Ensure that the additional inputs have shape (wt, wd, ws) or (wt, time), then reshape to column array.
    # We take the sizes from the sector average, rather than sim_res, because the sector average might contain a subset of it.
    if "time" in sector_average.dims:
        theta = [
            expand_shape_fun(
                th,
                sector_average["wt"].size,
                sector_average["time"].size,
            )
            .astype(dtype)
            .reshape(-1, 1)
            for th in additional_inputs
        ]
        shape_load = (
            sector_average["wt"].size,
            sector_average["time"].size,
        )
    else:
        theta = [
            expand_shape_fun(
                th,
                sector_average["wt"].size,
                sector_average["wd"].size,
                sector_average["ws"].size,
            )
            .astype(dtype)
            .reshape(-1, 1)
            for th in additional_inputs
        ]
        shape_load = (
            sector_average["wt"].size,
            sector_average["wd"].size,
            sector_average["ws"].size,
        )

    # Compose input for load surrogate.
    # We want points on the rows and features on the columns, hence the reshape.
    # Points are: wt, wd and ws or wt and time.
    # Features are: sector-averaged WS, sector-averaged TI and additional inputs.
    # Assumes that the order of the sectors matches the one of the surrogate.
    # In this case, the sectors must be ordered as: up, right, down and left.
    n_sectors = sector_average["direction"].size
    x = np.concatenate(
        (
            sector_average.loc[{"quantity": "WS_eff"}]
            .values.astype(dtype)
            .reshape((-1, n_sectors)),
            ti.astype(dtype).reshape((-1, n_sectors)),
            *theta,
        ),
        axis=1,
    )

    # Evaluate loads.
    for sensor in list(surrogates.keys()):
        loads.loc[{"name": sensor}] = (
            surrogates[sensor].predict_output(x).reshape(shape_load)
        )

    return loads


def compute_lifetime_del(
    probability, loads, material, factor=20.0 * 365.0 * 24.0 * 3600.0 * 1e-8
):
    r"""
    Compute Lifetime Damage Equivalent Loads.

    The Lifetime DEL for each turbine, load channel and Wöhler exponent is computed as

    .. math::
      \mathrm{LDEL}
          = \left(
                \phi \int_{-\pi}^{+\pi} \int_{V_{\text{cut-in}}}^{V_{\text{cut-out}}} p(V, \theta) \mathrm{DEL}(V, \theta)^m dV d\theta
            \right)^{1/m}

    where:
        - :math:`V` is the wind speed;
        - :math:`\theta` is the wind direction;
        - :math:`\mathrm{DEL}` is the Damage Equivalent Load for a given turbine and Wöhler exponent;
        - :math:`p(V, \theta)` is the probability of the flow case;
        - :math:`\phi` is the number of seconds in 20 years divided by the estimated number of cycles;
        - :math:`m` is the Wöhler exponent.

    The integral is numerically computed with the trapezoidal method.

    Parameters
    ----------
    probability : (L, K) xarray DataArray
        Probability for each wind direction (l) and wind speed (k).
        The dimensions must be `"wd"` and `"ws"`.
    loads : (I, L, K, N) xarray DataArray
        Damage Equivalent Loads for each turbine (i), wind direction (l), wind speed (k) and output channel (n).
        The dimensions must be `"wt"`, `"wd"`, `"ws"` and `"name"`.
    material : (N) xarray DataArray
        Wöhler exponent for each output channel (n).
        The dimension must be `"name"`.
    factor : float, optional
        Number of seconds in 20 years divided by the estimated number of cycles.
        The default is :math:`20 \cdot 365 \cdot 24 \cdot 3600 / 10^8`.

    Raises
    ------
    ValueError
        If the probability is not normalized to 1. That is, it must be

        .. math::
          \int_{-\pi}^{+\pi} \int_{0}^{+\infty} p(V, \theta) dV d\theta = 1

    Returns
    -------
    ldel : (I, N) xarray DataArray
        Lifetime Damage Equivalent Loads for each turbine (i) and output channel (n).
        The dimensions are `"wt"` and `"name"`.

    """
    if abs(float(probability.integrate("wd").integrate("ws"))) - 1.0 > 1e-14:
        raise ValueError("The probability must integrate to 1.")

    # Carry out the numerical integration with the trapezoidal method.
    # Wind direction is first, followed by wind speed, so that should be already optimal for the integration.
    ldel = probability * loads**material  # Shape: (wd, ws, wt, name)
    ldel = ldel.integrate("wd").integrate("ws")  # Shape: (wt, name)
    one_over_m = 1.0 / material  # Shape: (name,)
    ldel = (factor * ldel) ** one_over_m  # Shape: (wt, name)
    return ldel


# %% Main.

if __name__ == "__main__":

    plt.close("all")

    radius_1d = np.linspace(0.0, 50.0, 5)
    azimuth_1d = np.linspace(0.0, 2.0 * np.pi, 8, endpoint=False)
    x_2d, y_2d, z_2d = make_polar_grid(radius_1d, azimuth_1d)
    plot_grid(x_2d, y_2d, z_2d, close_grid=True)

    y = np.linspace(-50.0, +50.0, 5)
    z = np.linspace(-50.0, +50.0, 5)
    x_2d, y_2d, z_2d = make_rectangular_grid(y, z)
    plot_grid(x_2d, y_2d, z_2d, close_grid=False)
