"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const core_1 = require("@aws-solutions-constructs/core");
function deployNewFunc(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        }
    };
    return new lib_1.S3ToLambda(stack, 'test-s3-lambda', props);
}
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.lambdaFunction !== null);
    expect(construct.s3Bucket !== null);
    expect(construct.s3LoggingBucket !== null);
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new cdk.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.S3ToLambda(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: cdk.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// s3 bucket with bucket, loggingBucket, and auto delete objects
// --------------------------------------------------------------
test('s3 bucket with bucket, loggingBucket, and auto delete objects', () => {
    const stack = new cdk.Stack();
    new lib_1.S3ToLambda(stack, 's3-lambda', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        },
        loggingBucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::S3::Bucket", 2);
    template.hasResourceProperties("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "s3lambdaS3LoggingBucketAC6FF14E"
        }
    });
});
// --------------------------------------------------------------
// s3 bucket with one content bucket and no logging bucket
// --------------------------------------------------------------
test('s3 bucket with one content bucket and no logging bucket', () => {
    const stack = new cdk.Stack();
    new lib_1.S3ToLambda(stack, 's3-lambda', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        },
        logS3AccessLogs: false
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::S3::Bucket", 1);
});
test('check properties with existing S3 bucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = core_1.CreateScrapBucket(stack, {});
    const construct = new lib_1.S3ToLambda(stack, 's3-lambda', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler'
        },
        existingBucketObj: existingBucket
    });
    expect(construct.s3Bucket).toEqual(undefined);
    expect(construct.s3BucketInterface).not.toEqual(undefined);
    expect(construct.s3LoggingBucket).toEqual(undefined);
});
//# sourceMappingURL=data:application/json;base64,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