"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const minimatch = require("minimatch");
const assets_1 = require("./assets");
const bundling_1 = require("./bundling");
const fs_1 = require("./fs");
const names_1 = require("./names");
const cache_1 = require("./private/cache");
const stack_1 = require("./stack");
const stage_1 = require("./stage");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * Stages a file or directory from a location on the file system into a staging directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 *
 * @stability stable
 */
class AssetStaging extends construct_compat_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this.sourcePath = path.resolve(props.sourcePath);
        this.fingerprintOptions = props;
        if (!fs.existsSync(this.sourcePath)) {
            throw new Error(`Cannot find asset at ${this.sourcePath}`);
        }
        this.sourceStats = fs.statSync(this.sourcePath);
        const outdir = (_b = stage_1.Stage.of(this)) === null || _b === void 0 ? void 0 : _b.assetOutdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly asset output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        this.assetOutdir = outdir;
        // Determine the hash type based on the props as props.assetHashType is
        // optional from a caller perspective.
        this.customSourceFingerprint = props.assetHash;
        this.hashType = determineHashType(props.assetHashType, this.customSourceFingerprint);
        // Decide what we're going to do, without actually doing it yet
        let stageThisAsset;
        let skip = false;
        if (props.bundling) {
            // Check if we actually have to bundle for this stack
            const bundlingStacks = (_c = this.node.tryGetContext(cxapi.BUNDLING_STACKS)) !== null && _c !== void 0 ? _c : ['*'];
            skip = !bundlingStacks.find(pattern => minimatch(stack_1.Stack.of(this).stackName, pattern));
            const bundling = props.bundling;
            stageThisAsset = () => this.stageByBundling(bundling, skip);
        }
        else {
            stageThisAsset = () => this.stageByCopying();
        }
        // Calculate a cache key from the props. This way we can check if we already
        // staged this asset and reuse the result (e.g. the same asset with the same
        // configuration is used in multiple stacks). In this case we can completely
        // skip file system and bundling operations.
        //
        // The output directory and whether this asset is skipped or not should also be
        // part of the cache key to make sure we don't accidentally return the wrong
        // staged asset from the cache.
        this.cacheKey = calculateCacheKey({
            outdir: this.assetOutdir,
            sourcePath: path.resolve(props.sourcePath),
            bundling: props.bundling,
            assetHashType: this.hashType,
            customFingerprint: this.customSourceFingerprint,
            extraHash: props.extraHash,
            exclude: props.exclude,
            ignoreMode: props.ignoreMode,
            skip,
        });
        const staged = AssetStaging.assetCache.obtain(this.cacheKey, stageThisAsset);
        this.stagedPath = staged.stagedPath;
        this.absoluteStagedPath = staged.stagedPath;
        this.assetHash = staged.assetHash;
        this.packaging = staged.packaging;
        this.isArchive = staged.isArchive;
    }
    /**
     * Clears the asset hash cache.
     *
     * @stability stable
     */
    static clearAssetHashCache() {
        this.assetCache.clear();
    }
    /**
     * (deprecated) A cryptographic hash of the asset.
     *
     * @deprecated see `assetHash`.
     */
    get sourceHash() {
        return this.assetHash;
    }
    /**
     * Return the path to the staged asset, relative to the Cloud Assembly (manifest) directory of the given stack.
     *
     * Only returns a relative path if the asset was staged, returns an absolute path if
     * it was not staged.
     *
     * A bundled asset might end up in the outDir and still not count as
     * "staged"; if asset staging is disabled we're technically expected to
     * reference source directories, but we don't have a source directory for the
     * bundled outputs (as the bundle output is written to a temporary
     * directory). Nevertheless, we will still return an absolute path.
     *
     * A non-obvious directory layout may look like this:
     *
     * ```
     *    CLOUD ASSEMBLY ROOT
     *      +-- asset.12345abcdef/
     *      +-- assembly-Stage
     *            +-- MyStack.template.json
     *            +-- MyStack.assets.json <- will contain { "path": "../asset.12345abcdef" }
     * ```
     *
     * @stability stable
     */
    relativeStagedPath(stack) {
        var _b;
        const asmManifestDir = (_b = stage_1.Stage.of(stack)) === null || _b === void 0 ? void 0 : _b.outdir;
        if (!asmManifestDir) {
            return this.stagedPath;
        }
        const isOutsideAssetDir = path.relative(this.assetOutdir, this.stagedPath).startsWith('..');
        if (isOutsideAssetDir || this.stagingDisabled) {
            return this.stagedPath;
        }
        return path.relative(asmManifestDir, this.stagedPath);
    }
    /**
     * Stage the source to the target by copying
     *
     * Optionally skip if staging is disabled, in which case we pretend we did something but we don't really.
     */
    stageByCopying() {
        const assetHash = this.calculateHash(this.hashType);
        const stagedPath = this.stagingDisabled
            ? this.sourcePath
            : path.resolve(this.assetOutdir, renderAssetFilename(assetHash, path.extname(this.sourcePath)));
        if (!this.sourceStats.isDirectory() && !this.sourceStats.isFile()) {
            throw new Error(`Asset ${this.sourcePath} is expected to be either a directory or a regular file`);
        }
        this.stageAsset(this.sourcePath, stagedPath, 'copy');
        return {
            assetHash,
            stagedPath,
            packaging: this.sourceStats.isDirectory() ? assets_1.FileAssetPackaging.ZIP_DIRECTORY : assets_1.FileAssetPackaging.FILE,
            isArchive: this.sourceStats.isDirectory() || ARCHIVE_EXTENSIONS.includes(path.extname(this.sourcePath).toLowerCase()),
        };
    }
    /**
     * Stage the source to the target by bundling
     *
     * Optionally skip, in which case we pretend we did something but we don't really.
     */
    stageByBundling(bundling, skip) {
        var _b;
        if (!this.sourceStats.isDirectory()) {
            throw new Error(`Asset ${this.sourcePath} is expected to be a directory when bundling`);
        }
        if (skip) {
            // We should have bundled, but didn't to save time. Still pretend to have a hash.
            // If the asset uses OUTPUT or BUNDLE, we use a CUSTOM hash to avoid fingerprinting
            // a potentially very large source directory. Other hash types are kept the same.
            let hashType = this.hashType;
            if (hashType === assets_1.AssetHashType.OUTPUT || hashType === assets_1.AssetHashType.BUNDLE) {
                this.customSourceFingerprint = names_1.Names.uniqueId(this);
                hashType = assets_1.AssetHashType.CUSTOM;
            }
            return {
                assetHash: this.calculateHash(hashType, bundling),
                stagedPath: this.sourcePath,
                packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
                isArchive: true,
            };
        }
        // Try to calculate assetHash beforehand (if we can)
        let assetHash = this.hashType === assets_1.AssetHashType.SOURCE || this.hashType === assets_1.AssetHashType.CUSTOM
            ? this.calculateHash(this.hashType, bundling)
            : undefined;
        const bundleDir = this.determineBundleDir(this.assetOutdir, assetHash);
        this.bundle(bundling, bundleDir);
        // Check bundling output content and determine if we will need to archive
        const bundlingOutputType = (_b = bundling.outputType) !== null && _b !== void 0 ? _b : bundling_1.BundlingOutput.AUTO_DISCOVER;
        const bundledAsset = determineBundledAsset(bundleDir, bundlingOutputType);
        // Calculate assetHash afterwards if we still must
        assetHash = assetHash !== null && assetHash !== void 0 ? assetHash : this.calculateHash(this.hashType, bundling, bundledAsset.path);
        const stagedPath = path.resolve(this.assetOutdir, renderAssetFilename(assetHash, bundledAsset.extension));
        this.stageAsset(bundledAsset.path, stagedPath, 'move');
        // If bundling produced a single archive file we "touch" this file in the bundling
        // directory after it has been moved to the staging directory. This way if bundling
        // is skipped because the bundling directory already exists we can still determine
        // the correct packaging type.
        if (bundledAsset.packaging === assets_1.FileAssetPackaging.FILE) {
            fs.closeSync(fs.openSync(bundledAsset.path, 'w'));
        }
        return {
            assetHash,
            stagedPath,
            packaging: bundledAsset.packaging,
            isArchive: true,
        };
    }
    /**
     * Whether staging has been disabled
     */
    get stagingDisabled() {
        return !!this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
    }
    /**
     * Copies or moves the files from sourcePath to targetPath.
     *
     * Moving implies the source directory is temporary and can be trashed.
     *
     * Will not do anything if source and target are the same.
     */
    stageAsset(sourcePath, targetPath, style) {
        // Is the work already done?
        const isAlreadyStaged = fs.existsSync(targetPath);
        if (isAlreadyStaged) {
            if (style === 'move' && sourcePath !== targetPath) {
                fs.removeSync(sourcePath);
            }
            return;
        }
        // Moving can be done quickly
        if (style == 'move') {
            fs.renameSync(sourcePath, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        if (this.sourceStats.isFile()) {
            fs.copyFileSync(sourcePath, targetPath);
        }
        else if (this.sourceStats.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${sourcePath}`);
        }
    }
    /**
     * Determine the directory where we're going to write the bundling output
     *
     * This is the target directory where we're going to write the staged output
     * files if we can (if the hash is fully known), or a temporary directory
     * otherwise.
     */
    determineBundleDir(outdir, sourceHash) {
        if (sourceHash) {
            return path.resolve(outdir, renderAssetFilename(sourceHash));
        }
        // When the asset hash isn't known in advance, bundler outputs to an
        // intermediate directory named after the asset's cache key
        return path.resolve(outdir, `bundling-temp-${this.cacheKey}`);
    }
    /**
     * Bundles an asset to the given directory
     *
     * If the given directory already exists, assume that everything's already
     * in order and don't do anything.
     *
     * @param options Bundling options
     * @param bundleDir Where to create the bundle directory
     * @returns The fully resolved bundle output directory.
     */
    bundle(options, bundleDir) {
        var _b, _c, _d, _e;
        if (fs.existsSync(bundleDir)) {
            return;
        }
        fs.ensureDirSync(bundleDir);
        // Chmod the bundleDir to full access.
        fs.chmodSync(bundleDir, 0o777);
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_b = options.volumes) !== null && _b !== void 0 ? _b : [],
        ];
        let localBundling;
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            localBundling = (_c = options.local) === null || _c === void 0 ? void 0 : _c.tryBundle(bundleDir, options);
            if (!localBundling) {
                options.image.run({
                    command: options.command,
                    user,
                    volumes,
                    environment: options.environment,
                    workingDirectory: (_d = options.workingDirectory) !== null && _d !== void 0 ? _d : AssetStaging.BUNDLING_INPUT_DIR,
                    securityOpt: (_e = options.securityOpt) !== null && _e !== void 0 ? _e : '',
                });
            }
        }
        catch (err) {
            // When bundling fails, keep the bundle output for diagnosability, but
            // rename it out of the way so that the next run doesn't assume it has a
            // valid bundleDir.
            const bundleErrorDir = bundleDir + '-error';
            if (fs.existsSync(bundleErrorDir)) {
                // Remove the last bundleErrorDir.
                fs.removeSync(bundleErrorDir);
            }
            fs.renameSync(bundleDir, bundleErrorDir);
            throw new Error(`Failed to bundle asset ${this.node.path}, bundle output is located at ${bundleErrorDir}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            const outputDir = localBundling ? bundleDir : AssetStaging.BUNDLING_OUTPUT_DIR;
            throw new Error(`Bundling did not produce any output. Check that content is written to ${outputDir}.`);
        }
    }
    calculateHash(hashType, bundling, outputDir) {
        var _b;
        // When bundling a CUSTOM or SOURCE asset hash type, we want the hash to include
        // the bundling configuration. We handle CUSTOM and bundled SOURCE hash types
        // as a special case to preserve existing user asset hashes in all other cases.
        if (hashType == assets_1.AssetHashType.CUSTOM || (hashType == assets_1.AssetHashType.SOURCE && bundling)) {
            const hash = crypto.createHash('sha256');
            // if asset hash is provided by user, use it, otherwise fingerprint the source.
            hash.update((_b = this.customSourceFingerprint) !== null && _b !== void 0 ? _b : fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions));
            // If we're bundling an asset, include the bundling configuration in the hash
            if (bundling) {
                hash.update(JSON.stringify(bundling));
            }
            return hash.digest('hex');
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
            case assets_1.AssetHashType.OUTPUT:
                if (!outputDir) {
                    throw new Error(`Cannot use \`${hashType}\` hash type when \`bundling\` is not specified.`);
                }
                return fs_1.FileSystem.fingerprint(outputDir, this.fingerprintOptions);
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
_a = JSII_RTTI_SYMBOL_1;
AssetStaging[_a] = { fqn: "@aws-cdk/core.AssetStaging", version: "1.113.0" };
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 *
 * @stability stable
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 *
 * @stability stable
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
/**
 * Cache of asset hashes based on asset configuration to avoid repeated file
 * system and bundling operations.
 */
AssetStaging.assetCache = new cache_1.Cache();
function renderAssetFilename(assetHash, extension = '') {
    return `asset.${assetHash}${extension}`;
}
/**
 * Determines the hash type from user-given prop values.
 *
 * @param assetHashType Asset hash type construct prop
 * @param customSourceFingerprint Asset hash seed given in the construct props
 */
function determineHashType(assetHashType, customSourceFingerprint) {
    const hashType = customSourceFingerprint
        ? (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.CUSTOM)
        : (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.SOURCE);
    if (customSourceFingerprint && hashType !== assets_1.AssetHashType.CUSTOM) {
        throw new Error(`Cannot specify \`${assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
    }
    if (hashType === assets_1.AssetHashType.CUSTOM && !customSourceFingerprint) {
        throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
    }
    return hashType;
}
/**
 * Calculates a cache key from the props. Normalize by sorting keys.
 */
function calculateCacheKey(props) {
    return crypto.createHash('sha256')
        .update(JSON.stringify(sortObject(props)))
        .digest('hex');
}
/**
 * Recursively sort object keys
 */
function sortObject(object) {
    if (typeof object !== 'object' || object instanceof Array) {
        return object;
    }
    const ret = {};
    for (const key of Object.keys(object).sort()) {
        ret[key] = sortObject(object[key]);
    }
    return ret;
}
/**
 * Returns the single archive file of a directory or undefined
 */
function singleArchiveFile(directory) {
    if (!fs.existsSync(directory)) {
        throw new Error(`Directory ${directory} does not exist.`);
    }
    if (!fs.statSync(directory).isDirectory()) {
        throw new Error(`${directory} is not a directory.`);
    }
    const content = fs.readdirSync(directory);
    if (content.length === 1) {
        const file = path.join(directory, content[0]);
        const extension = path.extname(content[0]).toLowerCase();
        if (fs.statSync(file).isFile() && ARCHIVE_EXTENSIONS.includes(extension)) {
            return file;
        }
    }
    return undefined;
}
/**
 * Returns the bundled asset to use based on the content of the bundle directory
 * and the type of output.
 */
function determineBundledAsset(bundleDir, outputType) {
    const archiveFile = singleArchiveFile(bundleDir);
    // auto-discover means that if there is an archive file, we take it as the
    // bundle, otherwise, we will archive here.
    if (outputType === bundling_1.BundlingOutput.AUTO_DISCOVER) {
        outputType = archiveFile ? bundling_1.BundlingOutput.ARCHIVED : bundling_1.BundlingOutput.NOT_ARCHIVED;
    }
    switch (outputType) {
        case bundling_1.BundlingOutput.NOT_ARCHIVED:
            return { path: bundleDir, packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY };
        case bundling_1.BundlingOutput.ARCHIVED:
            if (!archiveFile) {
                throw new Error('Bundling output directory is expected to include only a single .zip or .jar file when `output` is set to `ARCHIVED`');
            }
            return { path: archiveFile, packaging: assets_1.FileAssetPackaging.FILE, extension: path.extname(archiveFile) };
    }
}
//# sourceMappingURL=data:application/json;base64,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