"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Arn = exports.ArnFormat = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_fn_1 = require("./cfn-fn");
const token_1 = require("./token");
const util_1 = require("./util");
/**
 * An enum representing the various ARN formats that different services use.
 *
 * @stability stable
 */
var ArnFormat;
(function (ArnFormat) {
    ArnFormat["NO_RESOURCE_NAME"] = "arn:aws:service:region:account:resource";
    ArnFormat["COLON_RESOURCE_NAME"] = "arn:aws:service:region:account:resource:resourceName";
    ArnFormat["SLASH_RESOURCE_NAME"] = "arn:aws:service:region:account:resource/resourceName";
    ArnFormat["SLASH_RESOURCE_SLASH_RESOURCE_NAME"] = "arn:aws:service:region:account:/resource/resourceName";
})(ArnFormat = exports.ArnFormat || (exports.ArnFormat = {}));
/**
 * @stability stable
 */
class Arn {
    constructor() { }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *    arn:{partition}:{service}:{region}:{account}:{resource}{sep}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     *
     * @stability stable
     */
    static format(components, stack) {
        var _b, _c, _d, _e;
        jsiiDeprecationWarnings._aws_cdk_core_ArnComponents(components);
        jsiiDeprecationWarnings._aws_cdk_core_Stack(stack);
        const partition = (_b = components.partition) !== null && _b !== void 0 ? _b : stack.partition;
        const region = (_c = components.region) !== null && _c !== void 0 ? _c : stack.region;
        const account = (_d = components.account) !== null && _d !== void 0 ? _d : stack.account;
        const sep = (_e = components.sep) !== null && _e !== void 0 ? _e : (components.arnFormat === ArnFormat.COLON_RESOURCE_NAME ? ':' : '/');
        const values = [
            'arn', ':', partition, ':', components.service, ':', region, ':', account, ':',
            ...(components.arnFormat === ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME ? ['/'] : []),
            components.resource,
        ];
        if (sep !== '/' && sep !== ':' && sep !== '') {
            throw new Error('resourcePathSep may only be ":", "/" or an empty string');
        }
        if (components.resourceName != null) {
            values.push(sep);
            values.push(components.resourceName);
        }
        return values.join('');
    }
    /**
     * (deprecated) Given an ARN, parses it and returns components.
     *
     * IF THE ARN IS A CONCRETE STRING...
     *
     * ...it will be parsed and validated. The separator (`sep`) will be set to '/'
     * if the 6th component includes a '/', in which case, `resource` will be set
     * to the value before the '/' and `resourceName` will be the rest. In case
     * there is no '/', `resource` will be set to the 6th components and
     * `resourceName` will be set to the rest of the string.
     *
     * IF THE ARN IS A TOKEN...
     *
     * ...it cannot be validated, since we don't have the actual value yet at the
     * time of this function call. You will have to supply `sepIfToken` and
     * whether or not ARNs of the expected format usually have resource names
     * in order to parse it properly. The resulting `ArnComponents` object will
     * contain tokens for the subexpressions of the ARN, not string literals.
     *
     * If the resource name could possibly contain the separator char, the actual
     * resource name cannot be properly parsed. This only occurs if the separator
     * char is '/', and happens for example for S3 object ARNs, IAM Role ARNs,
     * IAM OIDC Provider ARNs, etc. To properly extract the resource name from a
     * Tokenized ARN, you must know the resource type and call
     * `Arn.extractResourceName`.
     *
     * @param arn The ARN to parse.
     * @param sepIfToken The separator used to separate resource from resourceName.
     * @param hasName Whether there is a name component in the ARN at all.
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     * @deprecated use split instead
     */
    static parse(arn, sepIfToken = '/', hasName = true) {
        jsiiDeprecationWarnings.print("@aws-cdk/core.Arn#parse", "use split instead");
        let arnFormat;
        if (!hasName) {
            arnFormat = ArnFormat.NO_RESOURCE_NAME;
        }
        else {
            arnFormat = sepIfToken === '/' ? ArnFormat.SLASH_RESOURCE_NAME : ArnFormat.COLON_RESOURCE_NAME;
        }
        return this.split(arn, arnFormat);
    }
    /**
     * Splits the provided ARN into its components.
     *
     * Works both if 'arn' is a string like 'arn:aws:s3:::bucket',
     * and a Token representing a dynamic CloudFormation expression
     * (in which case the returned components will also be dynamic CloudFormation expressions,
     * encoded as Tokens).
     *
     * @param arn the ARN to split into its components.
     * @param arnFormat the expected format of 'arn' - depends on what format the service 'arn' represents uses.
     * @stability stable
     */
    static split(arn, arnFormat) {
        jsiiDeprecationWarnings._aws_cdk_core_ArnFormat(arnFormat);
        const components = parseArnShape(arn);
        if (components === 'token') {
            return parseTokenArn(arn, arnFormat);
        }
        const [, partition, service, region, account, resourceTypeOrName, ...rest] = components;
        let resource;
        let resourceName;
        let sep;
        let resourcePartStartIndex = 0;
        let detectedArnFormat;
        let slashIndex = resourceTypeOrName.indexOf('/');
        if (slashIndex === 0) {
            // new-style ARNs are of the form 'arn:aws:s4:us-west-1:12345:/resource-type/resource-name'
            slashIndex = resourceTypeOrName.indexOf('/', 1);
            resourcePartStartIndex = 1;
            detectedArnFormat = ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME;
        }
        if (slashIndex !== -1) {
            // the slash is only a separator if ArnFormat is not NO_RESOURCE_NAME
            if (arnFormat === ArnFormat.NO_RESOURCE_NAME) {
                sep = undefined;
                slashIndex = -1;
                detectedArnFormat = ArnFormat.NO_RESOURCE_NAME;
            }
            else {
                sep = '/';
                detectedArnFormat = resourcePartStartIndex === 0
                    ? ArnFormat.SLASH_RESOURCE_NAME
                    // need to repeat this here, as otherwise the compiler thinks 'detectedArnFormat' is not initialized in all paths
                    : ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME;
            }
        }
        else if (rest.length > 0) {
            sep = ':';
            slashIndex = -1;
            detectedArnFormat = ArnFormat.COLON_RESOURCE_NAME;
        }
        else {
            sep = undefined;
            detectedArnFormat = ArnFormat.NO_RESOURCE_NAME;
        }
        if (slashIndex !== -1) {
            resource = resourceTypeOrName.substring(resourcePartStartIndex, slashIndex);
            resourceName = resourceTypeOrName.substring(slashIndex + 1);
        }
        else {
            resource = resourceTypeOrName;
        }
        if (rest.length > 0) {
            if (!resourceName) {
                resourceName = '';
            }
            else {
                resourceName += ':';
            }
            resourceName += rest.join(':');
        }
        // "|| undefined" will cause empty strings to be treated as "undefined".
        // Optional ARN attributes (e.g. region, account) should return as empty string
        // if they are provided as such.
        return util_1.filterUndefined({
            service: service || undefined,
            resource: resource || undefined,
            partition: partition || undefined,
            region,
            account,
            resourceName,
            sep,
            arnFormat: detectedArnFormat,
        });
    }
    /**
     * Extract the full resource name from an ARN.
     *
     * Necessary for resource names (paths) that may contain the separator, like
     * `arn:aws:iam::111111111111:role/path/to/role/name`.
     *
     * Only works if we statically know the expected `resourceType` beforehand, since we're going
     * to use that to split the string on ':<resourceType>/' (and take the right-hand side).
     *
     * We can't extract the 'resourceType' from the ARN at hand, because CloudFormation Expressions
     * only allow literals in the 'separator' argument to `{ Fn::Split }`, and so it can't be
     * `{ Fn::Select: [5, { Fn::Split: [':', ARN] }}`.
     *
     * Only necessary for ARN formats for which the type-name separator is `/`.
     *
     * @stability stable
     */
    static extractResourceName(arn, resourceType) {
        const components = parseArnShape(arn);
        if (components === 'token') {
            return cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(`:${resourceType}/`, arn));
        }
        // Apparently we could just parse this right away. Validate that we got the right
        // resource type (to notify authors of incorrect assumptions right away).
        const parsed = Arn.split(arn, ArnFormat.SLASH_RESOURCE_NAME);
        if (!token_1.Token.isUnresolved(parsed.resource) && parsed.resource !== resourceType) {
            throw new Error(`Expected resource type '${resourceType}' in ARN, got '${parsed.resource}' in '${arn}'`);
        }
        if (!parsed.resourceName) {
            throw new Error(`Expected resource name in ARN, didn't find one: '${arn}'`);
        }
        return parsed.resourceName;
    }
}
exports.Arn = Arn;
_a = JSII_RTTI_SYMBOL_1;
Arn[_a] = { fqn: "@aws-cdk/core.Arn", version: "1.133.0" };
/**
 * Given a Token evaluating to ARN, parses it and returns components.
 *
 * The ARN cannot be validated, since we don't have the actual value yet
 * at the time of this function call. You will have to know the separator
 * and the type of ARN.
 *
 * The resulting `ArnComponents` object will contain tokens for the
 * subexpressions of the ARN, not string literals.
 *
 * WARNING: this function cannot properly parse the complete final
 * 'resourceName' part if it contains colons,
 * like 'arn:aws:cognito-sync:region:account:identitypool/us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla'.
 *
 * @param arnToken The input token that contains an ARN
 * @param arnFormat the expected format of 'arn' - depends on what format the service the ARN represents uses
 */
function parseTokenArn(arnToken, arnFormat) {
    // ARN looks like:
    // arn:partition:service:region:account:resource
    // arn:partition:service:region:account:resource:resourceName
    // arn:partition:service:region:account:resource/resourceName
    // arn:partition:service:region:account:/resource/resourceName
    const components = cfn_fn_1.Fn.split(':', arnToken);
    const partition = cfn_fn_1.Fn.select(1, components).toString();
    const service = cfn_fn_1.Fn.select(2, components).toString();
    const region = cfn_fn_1.Fn.select(3, components).toString();
    const account = cfn_fn_1.Fn.select(4, components).toString();
    let resource;
    let resourceName;
    let sep;
    if (arnFormat === ArnFormat.NO_RESOURCE_NAME || arnFormat === ArnFormat.COLON_RESOURCE_NAME) {
        // we know that the 'resource' part will always be the 6th segment in this case
        resource = cfn_fn_1.Fn.select(5, components);
        if (arnFormat === ArnFormat.COLON_RESOURCE_NAME) {
            resourceName = cfn_fn_1.Fn.select(6, components);
            sep = ':';
        }
        else {
            resourceName = undefined;
            sep = undefined;
        }
    }
    else {
        // we know that the 'resource' and 'resourceName' parts are separated by slash here,
        // so we split the 6th segment from the colon-separated ones with a slash
        const lastComponents = cfn_fn_1.Fn.split('/', cfn_fn_1.Fn.select(5, components));
        if (arnFormat === ArnFormat.SLASH_RESOURCE_NAME) {
            resource = cfn_fn_1.Fn.select(0, lastComponents);
            resourceName = cfn_fn_1.Fn.select(1, lastComponents);
        }
        else {
            // arnFormat is ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
            // which means there's an extra slash there at the beginning that we need to skip
            resource = cfn_fn_1.Fn.select(1, lastComponents);
            resourceName = cfn_fn_1.Fn.select(2, lastComponents);
        }
        sep = '/';
    }
    return { partition, service, region, account, resource, resourceName, sep, arnFormat };
}
/**
 * Validate that a string is either unparseable or looks mostly like an ARN
 */
function parseArnShape(arn) {
    // assume anything that starts with 'arn:' is an ARN,
    // so we can report better errors
    const looksLikeArn = arn.startsWith('arn:');
    if (!looksLikeArn) {
        if (token_1.Token.isUnresolved(arn)) {
            return 'token';
        }
        else {
            throw new Error(`ARNs must start with "arn:" and have at least 6 components: ${arn}`);
        }
    }
    // If the ARN merely contains Tokens, but otherwise *looks* mostly like an ARN,
    // it's a string of the form 'arn:${partition}:service:${region}:${account}:resource/xyz'.
    // Parse fields out to the best of our ability.
    // Tokens won't contain ":", so this won't break them.
    const components = arn.split(':');
    // const [/* arn */, partition, service, /* region */ , /* account */ , resource] = components;
    const partition = components.length > 1 ? components[1] : undefined;
    if (!partition) {
        throw new Error('The `partition` component (2nd component) of an ARN is required: ' + arn);
    }
    const service = components.length > 2 ? components[2] : undefined;
    if (!service) {
        throw new Error('The `service` component (3rd component) of an ARN is required: ' + arn);
    }
    const resource = components.length > 5 ? components[5] : undefined;
    if (!resource) {
        throw new Error('The `resource` component (6th component) of an ARN is required: ' + arn);
    }
    // Region can be missing in global ARNs (such as used by IAM)
    // Account can be missing in some ARN types (such as used for S3 buckets)
    return components;
}
//# sourceMappingURL=data:application/json;base64,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