"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParameter = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * A CloudFormation parameter.
 *
 * Use the optional Parameters section to customize your templates.
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 *
 * @stability stable
 */
class CfnParameter extends cfn_element_1.CfnElement {
    /**
     * Creates a parameter construct.
     *
     * Note that the name (logical ID) of the parameter will derive from it's `coname` and location
     * within the stack. Therefore, it is recommended that parameters are defined at the stack level.
     *
     * @param scope The parent construct.
     * @param props The parameter properties.
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_core_CfnParameterProps(props);
        this._type = props.type || 'String';
        this._default = props.default;
        this._allowedPattern = props.allowedPattern;
        this._allowedValues = props.allowedValues;
        this._constraintDescription = props.constraintDescription;
        this._description = props.description;
        this._maxLength = props.maxLength;
        this._maxValue = props.maxValue;
        this._minLength = props.minLength;
        this._minValue = props.minValue;
        this._noEcho = props.noEcho;
    }
    /**
     * The data type for the parameter (DataType).
     *
     * @default String
     * @stability stable
     */
    get type() {
        return this._type;
    }
    /**
     * The data type for the parameter (DataType).
     *
     * @default String
     * @stability stable
     */
    set type(type) {
        this._type = type;
    }
    /**
     * A value of the appropriate type for the template to use if no value is specified when a stack is created.
     *
     * If you define constraints for the parameter, you must specify
     * a value that adheres to those constraints.
     *
     * @default - No default value for parameter.
     * @stability stable
     */
    get default() {
        return this._default;
    }
    /**
     * A value of the appropriate type for the template to use if no value is specified when a stack is created.
     *
     * If you define constraints for the parameter, you must specify
     * a value that adheres to those constraints.
     *
     * @default - No default value for parameter.
     * @stability stable
     */
    set default(value) {
        this._default = value;
    }
    /**
     * A regular expression that represents the patterns to allow for String types.
     *
     * @default - No constraints on patterns allowed for parameter.
     * @stability stable
     */
    get allowedPattern() {
        return this._allowedPattern;
    }
    /**
     * A regular expression that represents the patterns to allow for String types.
     *
     * @default - No constraints on patterns allowed for parameter.
     * @stability stable
     */
    set allowedPattern(pattern) {
        this._allowedPattern = pattern;
    }
    /**
     * An array containing the list of values allowed for the parameter.
     *
     * @default - No constraints on values allowed for parameter.
     * @stability stable
     */
    get allowedValues() {
        return this._allowedValues;
    }
    /**
     * An array containing the list of values allowed for the parameter.
     *
     * @default - No constraints on values allowed for parameter.
     * @stability stable
     */
    set allowedValues(values) {
        this._allowedValues = values;
    }
    /**
     * A string that explains a constraint when the constraint is violated.
     *
     * For example, without a constraint description, a parameter that has an allowed
     * pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
     * an invalid value:
     *
     * @default - No description with customized error message when user specifies invalid values.
     * @stability stable
     */
    get constraintDescription() {
        return this._constraintDescription;
    }
    /**
     * A string that explains a constraint when the constraint is violated.
     *
     * For example, without a constraint description, a parameter that has an allowed
     * pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
     * an invalid value:
     *
     * @default - No description with customized error message when user specifies invalid values.
     * @stability stable
     */
    set constraintDescription(desc) {
        this._constraintDescription = desc;
    }
    /**
     * A string of up to 4000 characters that describes the parameter.
     *
     * @default - No description for the parameter.
     * @stability stable
     */
    get description() {
        return this._description;
    }
    /**
     * A string of up to 4000 characters that describes the parameter.
     *
     * @default - No description for the parameter.
     * @stability stable
     */
    set description(desc) {
        this._description = desc;
    }
    /**
     * An integer value that determines the largest number of characters you want to allow for String types.
     *
     * @default - None.
     * @stability stable
     */
    get maxLength() {
        return this._maxLength;
    }
    /**
     * An integer value that determines the largest number of characters you want to allow for String types.
     *
     * @default - None.
     * @stability stable
     */
    set maxLength(len) {
        this._maxLength = len;
    }
    /**
     * An integer value that determines the smallest number of characters you want to allow for String types.
     *
     * @default - None.
     * @stability stable
     */
    get minLength() {
        return this._minLength;
    }
    /**
     * An integer value that determines the smallest number of characters you want to allow for String types.
     *
     * @default - None.
     * @stability stable
     */
    set minLength(len) {
        this._minLength = len;
    }
    /**
     * A numeric value that determines the largest numeric value you want to allow for Number types.
     *
     * @default - None.
     * @stability stable
     */
    get maxValue() {
        return this._maxValue;
    }
    /**
     * A numeric value that determines the largest numeric value you want to allow for Number types.
     *
     * @default - None.
     * @stability stable
     */
    set maxValue(len) {
        this._maxValue = len;
    }
    /**
     * A numeric value that determines the smallest numeric value you want to allow for Number types.
     *
     * @default - None.
     * @stability stable
     */
    get minValue() {
        return this._minValue;
    }
    /**
     * A numeric value that determines the smallest numeric value you want to allow for Number types.
     *
     * @default - None.
     * @stability stable
     */
    set minValue(len) {
        this._minValue = len;
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     *
     * @stability stable
     */
    get noEcho() {
        return !!this._noEcho;
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     *
     * @stability stable
     */
    set noEcho(echo) {
        this._noEcho = echo;
    }
    /**
     * The parameter value as a Token.
     *
     * @stability stable
     */
    get value() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
    /**
     * The parameter value, if it represents a string.
     *
     * @stability stable
     */
    get valueAsString() {
        if (!isStringType(this.type) && !isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string or number type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     *
     * @stability stable
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * The parameter value, if it represents a number.
     *
     * @stability stable
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Parameters: {
                [this.logicalId]: {
                    Type: this.type,
                    Default: this.default,
                    AllowedPattern: this.allowedPattern,
                    AllowedValues: this.allowedValues,
                    ConstraintDescription: this.constraintDescription,
                    Description: this.description,
                    MaxLength: this.maxLength,
                    MaxValue: this.maxValue,
                    MinLength: this.minLength,
                    MinValue: this.minValue,
                    NoEcho: this._noEcho,
                },
            },
        };
    }
    /**
     * @stability stable
     */
    resolve(_context) {
        jsiiDeprecationWarnings._aws_cdk_core_IResolveContext(_context);
        return this.value;
    }
}
exports.CfnParameter = CfnParameter;
_a = JSII_RTTI_SYMBOL_1;
CfnParameter[_a] = { fqn: "@aws-cdk/core.CfnParameter", version: "1.133.0" };
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
//# sourceMappingURL=data:application/json;base64,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