"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdktfAdapter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdktf_1 = require("cdktf");
const github_actions_cdk_1 = require("github-actions-cdk");
/**
 * Adapter to integrate CDKTF (Cloud Development Kit for Terraform) projects with GitHub Actions.
 * This class extends the base `Project` class and allows for GitHub Actions workflow generation with annotation handling and validation.
 */
class CdktfAdapter extends github_actions_cdk_1.Project {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@github-actions-cdk/cdktf.CdktfAdapter", version: "0.0.19" };
    awsCdkScope;
    hasValidationErrors;
    /**
     * Initializes a new instance of the CdktfAdapter.
     *
     * @param awsCdkScope - The scope of the AWS CDK project.
     * @param props - Optional properties for project configuration.
     */
    constructor(awsCdkScope, props = {}) {
        super(props);
        this.awsCdkScope = awsCdkScope;
        this.hasValidationErrors = false;
        // Add an aspect to trigger synthesis when visiting the root scope node.
        cdktf_1.Aspects.of(this.awsCdkScope).add({
            visit: (node) => {
                if (node === this.awsCdkScope) {
                    this.synth();
                }
            },
        });
    }
    /**
     * Handles errors occurring during the synthesis process, particularly validation errors.
     * Adds validation error messages as annotations to the CDK scope node.
     *
     * @param error - The error encountered during synthesis.
     */
    handleSynthesisError(error) {
        if ((0, github_actions_cdk_1.isValidationError)(error)) {
            this.hasValidationErrors = true;
            this.awsCdkScope.node.addValidation({
                validate: () => {
                    return error.errors.map(({ message, source }) => `- [${source.node.path}]: ${message}`);
                },
            });
        }
        else {
            throw error;
        }
    }
    /**
     * Finalizes the synthesis process by adding annotations based on workflow metadata.
     * Adds informational, warning, and error messages to the AWS CDK scope and handles whether synthesis should continue on error annotations.
     */
    finalizeSynthesis() {
        const workflows = Object.values(this.manifest.workflows);
        // Loop through all annotations in workflows and apply appropriate annotation levels.
        for (const workflow of workflows) {
            for (const annotation of workflow.annotations) {
                switch (annotation.level) {
                    case github_actions_cdk_1.AnnotationMetadataEntryType.INFO:
                        cdktf_1.Annotations.of(this.awsCdkScope).addInfo(annotation.message);
                        break;
                    case github_actions_cdk_1.AnnotationMetadataEntryType.WARN:
                        cdktf_1.Annotations.of(this.awsCdkScope).addWarning(annotation.message);
                        break;
                    case github_actions_cdk_1.AnnotationMetadataEntryType.ERROR:
                        cdktf_1.Annotations.of(this.awsCdkScope).addError(annotation.message);
                        break;
                    default:
                        throw new Error(`Unknown annotation level: ${annotation.level}`);
                }
            }
        }
        // If not configured to continue on errors and error annotations are present, halt synthesis.
        if (!this.continueOnErrorAnnotations && this.manifest.hasErrorAnnotation()) {
            return;
        }
        // Halt synthesis if there are validation errors.
        if (this.hasValidationErrors) {
            return;
        }
        // Log successful workflow generation.
        cdktf_1.Annotations.of(this.awsCdkScope).addInfo(`GitHub Actions workflows generated at ${this.outdir}`);
    }
}
exports.CdktfAdapter = CdktfAdapter;
//# sourceMappingURL=data:application/json;base64,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