"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * @deprecated use `fromBucket`
     */
    static bucket(bucket, key, objectVersion) {
        return this.fromBucket(bucket, key, objectVersion);
    }
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * @deprecated use `fromInline`
     */
    static inline(code) {
        return this.fromInline(code);
    }
    /**
     * Loads the function code from a local disk path.
     *
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static fromAsset(path, options) {
        return new AssetCode(path, options);
    }
    /**
     * @deprecated use `fromAsset`
     */
    static asset(path) {
        return this.fromAsset(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static fromCfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * @deprecated use `fromCfnParameters`
     */
    static cfnParameters(props) {
        return this.fromCfnParameters(props);
    }
    /**
     * Called after the CFN function resource has been created to allow the code
     * class to bind to it. Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindToResource(_resource, _options) {
        return;
    }
}
exports.Code = Code;
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        this.isInline = true;
        if (code.length === 0) {
            throw new Error('Lambda inline code cannot be empty');
        }
        if (code.length > 4096) {
            throw new Error('Lambda source is too large, must be <= 4096 but is ' + code.length);
        }
    }
    bind(_scope) {
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        this.isInline = false;
    }
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.path,
                ...this.options,
            });
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    bindToResource(resource, options = {}) {
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const resourceProperty = options.resourceProperty || 'Code';
        // https://github.com/aws/aws-cdk/issues/1432
        this.asset.addResourceMetadata(resource, resourceProperty);
    }
}
exports.AssetCode = AssetCode;
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
class CfnParametersCode extends Code {
    constructor(props = {}) {
        super();
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    bind(scope) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(scope, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(scope, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
        return {
            s3Location: {
                bucketName: this._bucketNameParam.valueAsString,
                objectKey: this._objectKeyParam.valueAsString,
            },
        };
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
//# sourceMappingURL=data:application/json;base64,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