"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const event_invoke_config_1 = require("./event-invoke-config");
const event_source_mapping_1 = require("./event-source-mapping");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
class FunctionBase extends core_1.Resource {
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     * @param permission The permission to grant to this Lambda function. @see Permission for details.
     */
    addPermission(id, permission) {
        if (!this.canCreatePermissions) {
            // FIXME: Report metadata
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const action = permission.action || 'lambda:InvokeFunction';
        const scope = permission.scope || this;
        new lambda_generated_1.CfnPermission(scope, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: permission.sourceAccount,
            sourceArn: permission.sourceArn,
        });
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections() {
        if (!this._connections) {
            // tslint:disable-next-line:max-line-length
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    get latestVersion() {
        // Dynamic to avoid infinite recursion when creating the LatestVersion instance...
        return new LatestVersion(this);
    }
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    addEventSourceMapping(id, options) {
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options,
        });
    }
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['lambda:InvokeFunction'],
            resourceArns: [this.functionArn],
            // Fake resource-like object on which to call addToResourcePolicy(), which actually
            // calls addPermission()
            resource: {
                addToResourcePolicy: (_statement) => {
                    // Couldn't add permissions to the principal, so add them locally.
                    const identifier = `Invoke${grantee.grantPrincipal}`; // calls the .toString() of the princpal
                    this.addPermission(identifier, {
                        principal: grantee.grantPrincipal,
                        action: 'lambda:InvokeFunction',
                    });
                    return { statementAdded: true, policyDependable: this._functionNode().findChild(identifier) };
                },
                node: this.node,
            },
        });
    }
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source) {
        source.bind(this);
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this,
            ...options,
        });
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * For use internally for constructs, when the tree is set up in non-standard ways. Ex: SingletonFunction.
     * @internal
     */
    _functionNode() {
        return this.node;
    }
    parsePermissionPrincipal(principal) {
        if (!principal) {
            return undefined;
        }
        // use duck-typing, not instance of
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if ('service' in principal) {
            return principal.service;
        }
        if ('arn' in principal) {
            return principal.arn;
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ArnPrincipal, ServicePrincipal');
    }
}
exports.FunctionBase = FunctionBase;
class QualifiedFunctionBase extends FunctionBase {
    constructor() {
        super(...arguments);
        this.permissionsNode = this.node;
    }
    get latestVersion() {
        return this.lambda.latestVersion;
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the qualified function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this.lambda,
            qualifier: this.qualifier,
            ...options,
        });
    }
}
exports.QualifiedFunctionBase = QualifiedFunctionBase;
/**
 * The $LATEST version of a function, useful when attempting to create aliases.
 */
class LatestVersion extends FunctionBase {
    constructor(lambda) {
        super(lambda, '$LATEST');
        this.version = '$LATEST';
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.lambda = lambda;
    }
    get functionArn() {
        return `${this.lambda.functionArn}:${this.version}`;
    }
    get functionName() {
        return `${this.lambda.functionName}:${this.version}`;
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get latestVersion() {
        return this;
    }
    get role() {
        return this.lambda.role;
    }
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
}
//# sourceMappingURL=data:application/json;base64,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