"use strict";
const core_1 = require("@aws-cdk/core");
const path = require("path");
const lambda = require("../lib");
const function_hash_1 = require("../lib/function-hash");
module.exports = {
    'trimFromStart': {
        'trim not needed'(test) {
            test.deepEqual(function_hash_1.trimFromStart('foo', 100), 'foo');
            test.deepEqual(function_hash_1.trimFromStart('foo', 3), 'foo');
            test.deepEqual(function_hash_1.trimFromStart('', 3), '');
            test.done();
        },
        'trim required'(test) {
            test.deepEqual(function_hash_1.trimFromStart('hello', 3), 'llo');
            test.deepEqual(function_hash_1.trimFromStart('hello', 4), 'ello');
            test.deepEqual(function_hash_1.trimFromStart('hello', 1), 'o');
            test.done();
        },
    },
    'calcHash': {
        'same configuration and code yields the same hash'(test) {
            const stack1 = new core_1.Stack();
            const fn1 = new lambda.Function(stack1, 'MyFunction1', {
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'handler.zip')),
                handler: 'index.handler',
            });
            const stack2 = new core_1.Stack();
            const fn2 = new lambda.Function(stack2, 'MyFunction1', {
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'handler.zip')),
                handler: 'index.handler',
            });
            test.deepEqual(function_hash_1.calculateFunctionHash(fn1), function_hash_1.calculateFunctionHash(fn2));
            test.deepEqual(function_hash_1.calculateFunctionHash(fn1), 'aea5463dba236007afe91d2832b3c836');
            test.done();
        },
    },
    'code impacts hash'(test) {
        const stack1 = new core_1.Stack();
        const fn1 = new lambda.Function(stack1, 'MyFunction1', {
            runtime: lambda.Runtime.NODEJS_12_X,
            code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
            handler: 'index.handler',
        });
        test.notDeepEqual(function_hash_1.calculateFunctionHash(fn1), 'aea5463dba236007afe91d2832b3c836');
        test.deepEqual(function_hash_1.calculateFunctionHash(fn1), '979b4a14c6f174c745cdbcd1036cf844');
        test.done();
    },
    'environment variables impact hash'(test) {
        const stack1 = new core_1.Stack();
        const fn1 = new lambda.Function(stack1, 'MyFunction', {
            runtime: lambda.Runtime.NODEJS_12_X,
            code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
            handler: 'index.handler',
            environment: {
                Foo: 'bar',
            },
        });
        const stack2 = new core_1.Stack();
        const fn2 = new lambda.Function(stack2, 'MyFunction', {
            runtime: lambda.Runtime.NODEJS_12_X,
            code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
            handler: 'index.handler',
            environment: {
                Foo: 'beer',
            },
        });
        test.deepEqual(function_hash_1.calculateFunctionHash(fn1), 'd1bc824ac5022b7d62d8b12dbae6580c');
        test.deepEqual(function_hash_1.calculateFunctionHash(fn2), '3b683d05465012b0aa9c4ff53b32f014');
        test.done();
    },
    'runtime impacts hash'(test) {
        const stack1 = new core_1.Stack();
        const fn1 = new lambda.Function(stack1, 'MyFunction', {
            runtime: lambda.Runtime.NODEJS_12_X,
            code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
            handler: 'index.handler',
            environment: {
                Foo: 'bar',
            },
        });
        const stack2 = new core_1.Stack();
        const fn2 = new lambda.Function(stack2, 'MyFunction', {
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
            handler: 'index.handler',
            environment: {
                Foo: 'beer',
            },
        });
        test.deepEqual(function_hash_1.calculateFunctionHash(fn1), 'd1bc824ac5022b7d62d8b12dbae6580c');
        test.deepEqual(function_hash_1.calculateFunctionHash(fn2), '0f168f0772463e8e547bb3800937e54d');
        test.done();
    },
    'inline code change impacts the hash'(test) {
        const stack1 = new core_1.Stack();
        const fn1 = new lambda.Function(stack1, 'MyFunction', {
            runtime: lambda.Runtime.NODEJS_12_X,
            code: lambda.Code.fromInline('foo'),
            handler: 'index.handler',
        });
        const stack2 = new core_1.Stack();
        const fn2 = new lambda.Function(stack2, 'MyFunction', {
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo bar'),
            handler: 'index.handler',
        });
        test.deepEqual(function_hash_1.calculateFunctionHash(fn1), 'ebf2e871fc6a3062e8bdcc5ebe16db3f');
        test.deepEqual(function_hash_1.calculateFunctionHash(fn2), 'ffedf6424a18a594a513129dc97bf53c');
        test.done();
    },
    'impact of env variables order on hash': {
        'without "currentVersion", we preserve old behavior to avoid unnesesary invalidation of templates'(test) {
            const stack1 = new core_1.Stack();
            const fn1 = new lambda.Function(stack1, 'MyFunction', {
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
                handler: 'index.handler',
                environment: {
                    Foo: 'bar',
                    Bar: 'foo',
                },
            });
            const stack2 = new core_1.Stack();
            const fn2 = new lambda.Function(stack2, 'MyFunction', {
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
                handler: 'index.handler',
                environment: {
                    Bar: 'foo',
                    Foo: 'bar',
                },
            });
            test.notDeepEqual(function_hash_1.calculateFunctionHash(fn1), function_hash_1.calculateFunctionHash(fn2));
            test.done();
        },
        'with "currentVersion", we sort env keys so order is consistent'(test) {
            const stack1 = new core_1.Stack();
            const fn1 = new lambda.Function(stack1, 'MyFunction', {
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
                handler: 'index.handler',
                environment: {
                    Foo: 'bar',
                    Bar: 'foo',
                },
            });
            new core_1.CfnOutput(stack1, 'VersionArn', { value: fn1.currentVersion.functionArn });
            const stack2 = new core_1.Stack();
            const fn2 = new lambda.Function(stack2, 'MyFunction', {
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
                handler: 'index.handler',
                environment: {
                    Bar: 'foo',
                    Foo: 'bar',
                },
            });
            new core_1.CfnOutput(stack2, 'VersionArn', { value: fn2.currentVersion.functionArn });
            test.deepEqual(function_hash_1.calculateFunctionHash(fn1), function_hash_1.calculateFunctionHash(fn2));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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