"use strict";
const assert_1 = require("@aws-cdk/assert");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const _ = require("lodash");
const nodeunit_1 = require("nodeunit");
const path = require("path");
const lambda = require("../lib");
module.exports = nodeunit_1.testCase({
    'add incompatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const func = new lambda.Function(stack, 'myFunc', {
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            code,
        });
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS],
        });
        // THEN
        test.throws(() => func.addLayers(layer), /This lambda function uses a runtime that is incompatible with this layer/);
        test.done();
    },
    'add compatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const func = new lambda.Function(stack, 'myFunc', {
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            code,
        });
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [lambda.Runtime.PYTHON_3_7],
        });
        // THEN
        // should not throw
        func.addLayers(layer);
        test.done();
    },
    'add compatible layer for deep clone'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const runtime = lambda.Runtime.PYTHON_3_7;
        const func = new lambda.Function(stack, 'myFunc', {
            runtime,
            handler: 'index.handler',
            code,
        });
        const clone = _.cloneDeep(runtime);
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [clone],
        });
        // THEN
        // should not throw
        func.addLayers(layer);
        test.done();
    },
    'empty inline code is not allowed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN/THEN
        test.throws(() => new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline(''),
        }), /Lambda inline code cannot be empty/);
        test.done();
    },
    'logGroup is correctly returned'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
        });
        const logGroup = fn.logGroup;
        test.ok(logGroup.logGroupName);
        test.ok(logGroup.logGroupArn);
        test.done();
    },
    'dlq is returned when provided by user'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14),
        });
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            deadLetterQueue: dlQueue,
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueArn);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueName);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueUrl);
        test.done();
    },
    'dlq is returned when setup by cdk'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            deadLetterQueueEnabled: true,
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueArn);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueName);
        test.ok(deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueUrl);
        test.done();
    },
    'dlq is undefined when not setup'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === undefined);
        test.done();
    },
    'one and only one child LogRetention construct will be created'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            logRetention: logs.RetentionDays.FIVE_DAYS,
        });
        // tslint:disable:no-unused-expression
        // Call logGroup a few times. If more than one instance of LogRetention was created,
        // the second call will fail on duplicate constructs.
        fn.logGroup;
        fn.logGroup;
        fn.logGroup;
        // tslint:enable:no-unused-expression
        test.done();
    },
    'fails when inline code is specified on an incompatible runtime'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.PROVIDED,
            code: lambda.Code.fromInline('foo'),
        }), /Inline source not allowed for/);
        test.done();
    },
    'currentVersion': {
        // see test.function-hash.ts for more coverage for this
        'logical id of version is based on the function hash'(test) {
            // GIVEN
            const stack1 = new cdk.Stack();
            const fn1 = new lambda.Function(stack1, 'MyFunction', {
                handler: 'foo',
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'handler.zip')),
                environment: {
                    FOO: 'bar',
                },
            });
            const stack2 = new cdk.Stack();
            const fn2 = new lambda.Function(stack2, 'MyFunction', {
                handler: 'foo',
                runtime: lambda.Runtime.NODEJS_12_X,
                code: lambda.Code.fromAsset(path.join(__dirname, 'handler.zip')),
                environment: {
                    FOO: 'bear',
                },
            });
            // WHEN
            new cdk.CfnOutput(stack1, 'CurrentVersionArn', {
                value: fn1.currentVersion.functionArn,
            });
            new cdk.CfnOutput(stack2, 'CurrentVersionArn', {
                value: fn2.currentVersion.functionArn,
            });
            // THEN
            assert_1.expect(stack1).to(assert_1.haveOutput({
                outputName: 'CurrentVersionArn',
                outputValue: {
                    Ref: 'MyFunctionCurrentVersion197490AF1a9a73cf5c46aec5e40fb202042eb60b',
                },
            }));
            assert_1.expect(stack2).to(assert_1.haveOutput({
                outputName: 'CurrentVersionArn',
                outputValue: {
                    Ref: 'MyFunctionCurrentVersion197490AF8360a045031060e3117269037b7bffd6',
                },
            }));
            test.done();
        },
    },
});
//# sourceMappingURL=data:application/json;base64,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