"use strict";
const AWS = require("aws-sdk-mock");
const nock = require("nock");
const sinon = require("sinon");
const provider = require("../lib/log-retention-provider");
AWS.setSDK(require.resolve('aws-sdk'));
const eventCommon = {
    ServiceToken: 'token',
    ResponseURL: 'https://localhost',
    StackId: 'stackId',
    RequestId: 'requestId',
    LogicalResourceId: 'logicalResourceId',
    PhysicalResourceId: 'group',
    ResourceType: 'Custom::LogRetention',
};
const context = {
    functionName: 'provider',
};
function createRequest(type) {
    return nock('https://localhost')
        .put('/', (body) => body.Status === type && body.PhysicalResourceId === 'group')
        .reply(200);
}
module.exports = {
    'tearDown'(callback) {
        AWS.restore();
        nock.cleanAll();
        callback();
    },
    async 'create event'(test) {
        const createLogGroupFake = sinon.fake.resolves({});
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group',
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: 'group',
            retentionInDays: 30,
        });
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: '/aws/lambda/provider',
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: '/aws/lambda/provider',
            retentionInDays: 1,
        });
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'update event with new log retention'(test) {
        const error = new Error();
        error.code = 'ResourceAlreadyExistsException';
        const createLogGroupFake = sinon.fake.rejects(error);
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Update',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '365',
                LogGroupName: 'group',
            },
            OldResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
                RetentionInDays: '30',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group',
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: 'group',
            retentionInDays: 365,
        });
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'update event with log retention undefined'(test) {
        const error = new Error();
        error.code = 'ResourceAlreadyExistsException';
        const createLogGroupFake = sinon.fake.rejects(error);
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Update',
            PhysicalResourceId: 'group',
            ResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
            },
            OldResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
                RetentionInDays: '365',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group',
        });
        sinon.assert.calledWith(deleteRetentionPolicyFake, {
            logGroupName: 'group',
        });
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'delete event'(test) {
        const createLogGroupFake = sinon.fake.resolves({});
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Delete',
            PhysicalResourceId: 'group',
            ResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.notCalled(createLogGroupFake);
        sinon.assert.notCalled(putRetentionPolicyFake);
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'responds with FAILED on error'(test) {
        const createLogGroupFake = sinon.fake.rejects(new Error('UnkownError'));
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group',
            },
        };
        const request = createRequest('FAILED');
        await provider.handler(event, context);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'does not fail when operations on provider log group fail'(test) {
        const createLogGroupFake = (params) => {
            if (params.logGroupName === '/aws/lambda/provider') {
                return Promise.reject(new Error('OperationAbortedException'));
            }
            return Promise.resolve({});
        };
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'response data contains the log group name'(test) {
        AWS.mock('CloudWatchLogs', 'createLogGroup', sinon.fake.resolves({}));
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', sinon.fake.resolves({}));
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', sinon.fake.resolves({}));
        const event = {
            ...eventCommon,
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group',
            },
        };
        async function withOperation(operation) {
            const request = nock('https://localhost')
                .put('/', (body) => { var _a; return ((_a = body.Data) === null || _a === void 0 ? void 0 : _a.LogGroupName) === 'group'; })
                .reply(200);
            const opEvent = { ...event, RequestType: operation };
            await provider.handler(opEvent, context);
            test.equal(request.isDone(), true);
        }
        await withOperation('Create');
        await withOperation('Update');
        await withOperation('Delete');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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