"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return false;
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    constructor(principalJson, conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
class ArnPrincipal extends PrincipalBase {
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
class AccountPrincipal extends ArnPrincipal {
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString()
            ]
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
class FederatedPrincipal extends PrincipalBase {
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    toString() {
        return `AccountRootPrincipal()`;
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return `AnyPrincipal()`;
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error(`Cannot add multiple principals with different "assumeRoleAction". ` +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error(`Components of a CompositePrincipal must not have conditions. ` +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        return this.fn(cdk.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    toJSON() {
        return `<unresolved-token>`;
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || cdk.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, cdk.Aws.URL_SUFFIX);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service
        });
    }
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,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