import { IUserPool, IUserPoolClient } from 'aws-cdk-lib/aws-cognito';
import { IResource, Resource } from 'aws-cdk-lib/core';
import { Construct } from 'constructs';
import { IPolicyStore } from './policy-store';
export interface CognitoGroupConfiguration {
    /**
     * The name of the schema entity type that's mapped to the user pool group
     */
    readonly groupEntityType: string;
}
export interface CognitoUserPoolConfiguration {
    /**
     * Client identifiers.
     *
     * @default - empty list.
     */
    readonly clientIds?: string[];
    /**
     * Cognito Group Configuration
     *
     * @default - no Cognito Group configuration provided
     */
    readonly groupConfiguration?: CognitoGroupConfiguration;
    /**
     * Cognito User Pool.
     *
     * @default - no Cognito User Pool
     */
    readonly userPool: IUserPool;
}
export interface OpenIdConnectGroupConfiguration {
    /**
     * The token claim that you want Verified Permissions to interpret as group membership
     *
     */
    readonly groupClaim: string;
    /**
     * The policy store entity type that you want to map your users' group claim to
     *
     */
    readonly groupEntityType: string;
}
export interface OpenIdConnectAccessTokenConfiguration {
    /**
     * The access token aud claim values that you want to accept in your policy store
     *
     * @default - no audiences
     *
     */
    readonly audiences?: string[];
    /**
     * The claim that determines the principal in OIDC access tokens
     *
     * @default - no principal claim
     */
    readonly principalIdClaim?: string;
}
export interface OpenIdConnectIdentityTokenConfiguration {
    /**
     * The ID token audience, or client ID, claim values that you want to accept in your policy store from an OIDC identity provider
     *
     * @default - no client IDs
     *
     */
    readonly clientIds?: string[];
    /**
     * The claim that determines the principal in OIDC access tokens
     *
     * @default - no principal claim
     */
    readonly principalIdClaim?: string;
}
export interface OpenIdConnectConfiguration {
    /**
     * A descriptive string that you want to prefix to user entities from your OIDC identity provider
     *
     * @default - no Entity ID Prefix
     */
    readonly entityIdPrefix?: string;
    /**
     * The claim in OIDC identity provider tokens that indicates a user's group membership, and the entity type that you want to map it to
     *
     * @default - no Group Config
     */
    readonly groupConfiguration?: OpenIdConnectGroupConfiguration;
    /**
     * The issuer URL of an OIDC identity provider. This URL must have an OIDC discovery endpoint at the path .well-known/openid-configuration
     *
     */
    readonly issuer: string;
    /**
     * The configuration for processing access tokens from your OIDC identity provider
     * Exactly one between accessTokenOnly and identityTokenOnly must be defined
     *
     * @default - no Access Token Config
     */
    readonly accessTokenOnly?: OpenIdConnectAccessTokenConfiguration;
    /**
     * The configuration for processing identity (ID) tokens from your OIDC identity provider
     * Exactly one between accessTokenOnly and identityTokenOnly must be defined
     *
     * @default - no ID Token Config
     */
    readonly identityTokenOnly?: OpenIdConnectIdentityTokenConfiguration;
}
export interface IdentitySourceConfiguration {
    /**
     * Cognito User Pool Configuration.
     *
     * @default - no Cognito User Pool Config
     *
     */
    readonly cognitoUserPoolConfiguration?: CognitoUserPoolConfiguration;
    /**
     * OpenID Connect Idp configuration
     *
     * @default - no OpenID Provider config
     *
     */
    readonly openIdConnectConfiguration?: OpenIdConnectConfiguration;
}
export interface IdentitySourceProps {
    /**
     *  Identity Source configuration.
     */
    readonly configuration: IdentitySourceConfiguration;
    /**
     * Policy Store in which you want to store this identity source
     *
     */
    readonly policyStore: IPolicyStore;
    /**
     * Principal entity type
     *
     * @default - No principal entity type for the identity source.
     */
    readonly principalEntityType?: string;
}
export interface IIdentitySource extends IResource {
    /**
     * Identity Source identifier.
     *
     * @attribute
     */
    readonly identitySourceId: string;
}
declare abstract class IdentitySourceBase extends Resource implements IIdentitySource {
    abstract readonly identitySourceId: string;
}
export interface IdentitySourceAttributes {
    readonly identitySourceId: string;
}
export declare class IdentitySource extends IdentitySourceBase {
    /**
     * Creates Identity Source from its attributes
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs An `IdentitySourceAttributes` object.
     */
    static fromIdentitySourceAttributes(scope: Construct, id: string, attrs: IdentitySourceAttributes): IIdentitySource;
    /**
     * Create an Identity Source from its identifier
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param identitySourceId The Identity Source identifier.
     */
    static fromIdentitySourceId(scope: Construct, id: string, identitySourceId: string): IIdentitySource;
    private readonly configurationMode;
    private readonly identitySource;
    readonly clientIds: string[];
    readonly identitySourceId: string;
    readonly issuer: string;
    readonly userPoolArn?: string;
    readonly cognitoGroupEntityType?: string;
    readonly policyStore: IPolicyStore;
    readonly audiencesOIDC: string[];
    readonly principalIdClaimOIDC?: string;
    readonly groupConfigGroupClaimOIDC?: string;
    readonly groupConfigGroupEntityTypeOIDC?: string;
    constructor(scope: Construct, id: string, props: IdentitySourceProps);
    /**
     * Add a User Pool Client
     * The method can be called only when the Identity Source is configured with Cognito auth provider
     *
     * @param userPoolClient The User Pool Client Construct.
     */
    addUserPoolClient(userPoolClient: IUserPoolClient): void;
    /**
     * Add a clientId to the list
     * The method can be called only when the Identity Source is configured with one of these configs:
     *  - Cognito auth provider
     *  - OIDC auth provider and ID Token Selection mode
     *
     * @param clientId The clientId to be added
     */
    addClientId(clientId: string): void;
    /**
     * Add an audience to the list.
     * The method can be called only when the Identity Source is configured with OIDC auth provider and Access Token Selection mode
     *
     * @param audience the audience to be added
     */
    addAudience(audience: string): void;
}
export {};
