"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyType = exports.Policy = exports.POLICY_DESC_SUFFIX_FROM_FILE = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
exports.POLICY_DESC_SUFFIX_FROM_FILE = ' - Created by CDK';
class PolicyBase extends core_1.Resource {
}
class Policy extends PolicyBase {
    /**
     *  Import a policy into the CDK using its id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param policyId The policy id.
     */
    static fromPolicyId(scope, id, policyId) {
        return Policy.fromPolicyAttributes(scope, id, { policyId });
    }
    /**
     * Import a Policy construct from attributes.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param attrs A `PolicyAttributes` object.
     * @returns
     */
    static fromPolicyAttributes(scope, id, attrs) {
        class Import extends PolicyBase {
            constructor(policyType, policyId) {
                super(scope, id);
                this.policyType = policyType;
                this.policyId = policyId;
            }
        }
        const policyType = attrs.policyType ?? PolicyType.STATIC;
        return new Import(policyType, attrs.policyId);
    }
    /**
     * Create a policy based on a file containing a cedar policy. Best practice would be
     * for the file name to end in `.cedar` but this is not required. Policy is parsed for valid
     * syntax but not validated against schema. In order to validate against schema, use
     * `PolicyStore.addPoliciesFromPath()`
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct id.
     * @param props A `StaticPolicyFromFileProps` object.
     */
    static fromFile(scope, id, props) {
        const policyFileContents = fs.readFileSync(props.path).toString();
        (0, cedar_helpers_1.checkParsePolicy)(policyFileContents);
        let relativePath = path.basename(props.path);
        let policyDescription = props.description || (0, cedar_helpers_1.getPolicyDescription)(policyFileContents) || `${relativePath}${exports.POLICY_DESC_SUFFIX_FROM_FILE}`;
        return new Policy(scope, id, {
            definition: {
                static: {
                    statement: policyFileContents,
                    description: policyDescription,
                },
            },
            policyStore: props.policyStore,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        // validations
        if (props.definition.static && props.definition.templateLinked) {
            throw new Error('Policy can either be static or templateLinked');
        }
        let cfnDefinitionAttr;
        let definitionProperty = props.definition;
        if (props.definition.static) {
            (0, cedar_helpers_1.checkParsePolicy)(props.definition.static.statement);
            let description = props.definition.static.description || (0, cedar_helpers_1.getPolicyDescription)(props.definition.static.statement) || undefined;
            definitionProperty = {
                static: {
                    ...props.definition.static,
                    ...{
                        description,
                    },
                },
            };
            cfnDefinitionAttr = {
                static: {
                    description: description,
                    statement: props.definition.static.statement,
                },
            };
        }
        else if (props.definition.templateLinked) {
            cfnDefinitionAttr = {
                templateLinked: {
                    policyTemplateId: props.definition.templateLinked.policyTemplate.policyTemplateId,
                    principal: props.definition.templateLinked.principal,
                    resource: props.definition.templateLinked.resource,
                },
            };
        }
        else {
            throw new Error('Policy must either be static or templateLinked');
        }
        // resource
        this.policy = new aws_verifiedpermissions_1.CfnPolicy(this, id, {
            definition: cfnDefinitionAttr,
            policyStoreId: props.policyStore.policyStoreId,
        });
        // assign construct props
        this.policyId = this.policy.attrPolicyId;
        this.policyType = definitionProperty.static
            ? PolicyType.STATIC
            : PolicyType.TEMPLATELINKED;
        this.definition = definitionProperty;
        this.policyStoreId = props.policyStore.policyStoreId;
    }
}
exports.Policy = Policy;
_a = JSII_RTTI_SYMBOL_1;
Policy[_a] = { fqn: "@cdklabs/cdk-verified-permissions.Policy", version: "0.1.1" };
/**
 * PolicyType options
 */
var PolicyType;
(function (PolicyType) {
    PolicyType["STATIC"] = "Static";
    PolicyType["TEMPLATELINKED"] = "TemplateLinked";
})(PolicyType || (exports.PolicyType = PolicyType = {}));
//# sourceMappingURL=data:application/json;base64,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