"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ManagedPolicy = void 0;
const core_1 = require("@aws-cdk/core");
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * Managed policy
 *
 */
class ManagedPolicy extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.managedPolicyName,
        });
        /**
         * The policy document.
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        this.description = props.description || '';
        this.path = props.path || '/';
        if (props.document) {
            this.document = props.document;
        }
        const resource = new iam_generated_1.CfnManagedPolicy(this, 'Resource', {
            policyDocument: this.document,
            managedPolicyName: this.physicalName,
            description: this.description,
            path: this.path,
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatements(p));
        }
        // arn:aws:iam::123456789012:policy/teststack-CreateTestDBPolicy-16M23YE3CS700
        this.managedPolicyName = this.getResourceNameAttribute(core_1.Stack.of(this).parseArn(resource.ref, '/').resourceName);
        this.managedPolicyArn = this.getResourceArnAttribute(resource.ref, {
            region: '',
            service: 'iam',
            resource: 'policy',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a customer managed policy from the managedPolicyName.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     */
    static fromManagedPolicyName(scope, id, managedPolicyName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = core_1.Stack.of(scope).formatArn({
                    service: 'iam',
                    region: '',
                    account: core_1.Stack.of(scope).account,
                    resource: 'policy',
                    resourceName: managedPolicyName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an external managed policy by ARN.
     *
     * For this managed policy, you only need to know the ARN to be able to use it.
     * This can be useful if you got the ARN from a CloudFormation Export.
     *
     * If the imported Managed Policy ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * managed policy has a `path` (like `arn:...:policy/AdminPolicy/AdminAllow`), the
     * `managedPolicyName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Managed Policy ARN should be supplied without the
     * `path` in order to resolve the correct managed policy resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param managedPolicyArn the ARN of the managed policy to import
     */
    static fromManagedPolicyArn(scope, id, managedPolicyArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.managedPolicyArn = managedPolicyArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a managed policy from one of the policies that AWS manages.
     *
     * For this managed policy, you only need to know the name to be able to use it.
     *
     * Some managed policy names start with "service-role/", some start with
     * "job-function/", and some don't start with anything. Do include the
     * prefix when constructing this object.
     */
    static fromAwsManagedPolicyName(managedPolicyName) {
        class AwsManagedPolicy {
            constructor() {
                this.managedPolicyArn = core_1.Lazy.stringValue({
                    produce(ctx) {
                        return core_1.Stack.of(ctx.scope).formatArn({
                            service: 'iam',
                            region: '',
                            account: 'aws',
                            resource: 'policy',
                            resourceName: managedPolicyName,
                        });
                    },
                });
            }
        }
        return new AwsManagedPolicy();
    }
    /**
     * Adds a statement to the policy document.
     */
    addStatements(...statement) {
        this.document.addStatements(...statement);
    }
    /**
     * Attaches this policy to a user.
     */
    attachToUser(user) {
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
    }
    /**
     * Attaches this policy to a role.
     */
    attachToRole(role) {
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
    }
    /**
     * Attaches this policy to a group.
     */
    attachToGroup(group) {
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
    }
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            result.push('Managed Policy is empty. You must add statements to the policy');
        }
        return result;
    }
}
exports.ManagedPolicy = ManagedPolicy;
//# sourceMappingURL=data:application/json;base64,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