"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositePrincipal = exports.Anyone = exports.AnyPrincipal = exports.AccountRootPrincipal = exports.OpenIdConnectPrincipal = exports.WebIdentityPrincipal = exports.FederatedPrincipal = exports.CanonicalUserPrincipal = exports.OrganizationPrincipal = exports.ServicePrincipal = exports.AccountPrincipal = exports.ArnPrincipal = exports.PrincipalPolicyFragment = exports.PrincipalWithConditions = exports.PrincipalBase = void 0;
const cdk = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return { statementAdded: false };
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    /**
     * Returns a new PrincipalWithConditions using this principal as the base, with the
     * passed conditions added.
     *
     * When there is a value for the same operator and key in both the principal and the
     * conditions parameter, the value from the conditions parameter will be used.
     *
     * @returns a new PrincipalWithConditions object.
     */
    withConditions(conditions) {
        return new PrincipalWithConditions(this, conditions);
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * An IAM principal with additional conditions specifying when the policy is in effect.
 *
 * For more information about conditions, see:
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html
 */
class PrincipalWithConditions {
    constructor(principal, conditions) {
        this.principal = principal;
        this.grantPrincipal = this;
        this.assumeRoleAction = this.principal.assumeRoleAction;
        this.additionalConditions = conditions;
    }
    /**
     * Add a condition to the principal
     */
    addCondition(key, value) {
        const existingValue = this.conditions[key];
        this.conditions[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Adds multiple conditions to the principal
     *
     * Values from the conditions parameter will overwrite existing values with the same operator
     * and key.
     */
    addConditions(conditions) {
        Object.entries(conditions).forEach(([key, value]) => {
            this.addCondition(key, value);
        });
    }
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    get conditions() {
        return this.mergeConditions(this.principal.policyFragment.conditions, this.additionalConditions);
    }
    get policyFragment() {
        return new PrincipalPolicyFragment(this.principal.policyFragment.principalJson, this.conditions);
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(statement) {
        return this.principal.addToPrincipalPolicy(statement);
    }
    toString() {
        return this.principal.toString();
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    mergeConditions(principalConditions, additionalConditions) {
        const mergedConditions = {};
        Object.entries(principalConditions).forEach(([operator, condition]) => {
            mergedConditions[operator] = condition;
        });
        Object.entries(additionalConditions).forEach(([operator, condition]) => {
            // merge the conditions if one of the additional conditions uses an
            // operator that's already used by the principal's conditions merge the
            // inner structure.
            const existing = mergedConditions[operator];
            if (!existing) {
                mergedConditions[operator] = condition;
                return; // continue
            }
            // if either the existing condition or the new one contain unresolved
            // tokens, fail the merge. this is as far as we go at this point.
            if (cdk.Token.isUnresolved(condition) || cdk.Token.isUnresolved(existing)) {
                throw new Error(`multiple "${operator}" conditions cannot be merged if one of them contains an unresolved token`);
            }
            mergedConditions[operator] = { ...existing, ...condition };
        });
        return mergedConditions;
    }
}
exports.PrincipalWithConditions = PrincipalWithConditions;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    /**
     *
     * @param principalJson JSON of the "Principal" section in a policy statement
     * @param conditions conditions that need to be applied to this policy
     */
    constructor(principalJson, 
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
/**
 * Specify a principal by the Amazon Resource Name (ARN).
 * You can specify AWS accounts, IAM users, Federated SAML users, IAM roles, and specific assumed-role sessions.
 * You cannot specify IAM groups or instance profiles as principals
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
 */
class ArnPrincipal extends PrincipalBase {
    /**
     *
     * @param arn Amazon Resource Name (ARN) of the principal entity (i.e. arn:aws:iam::123456789012:user/user-name)
     */
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
/**
 * Specify AWS account ID as the principal entity in a policy to delegate authority to the account.
 */
class AccountPrincipal extends ArnPrincipal {
    /**
     *
     * @param accountId AWS account ID (i.e. 123456789012)
     */
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    /**
     *
     * @param service AWS service (i.e. sqs.amazonaws.com)
     */
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString(),
            ],
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    /**
     *
     * @param organizationId The unique identifier (ID) of an organization (i.e. o-12345abcde)
     */
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy principal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    /**
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     *   root user and IAM users for an account all see the same ID.
     *   (i.e. 79a59df900b949e55d96a1e698fbacedfd6e09d98eacf8f8d5218e7cd47ef2be)
     */
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
/**
 * Principal entity that represents a federated identity provider such as Amazon Cognito,
 * that can be used to provide temporary security credentials to users who have been authenticated.
 * Additional condition keys are available when the temporary security credentials are used to make a request.
 * You can use these keys to write policies that limit the access of federated users.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#condition-keys-wif
 */
class FederatedPrincipal extends PrincipalBase {
    /**
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
/**
 * A principal that represents a federated identity provider as Web Identity such as Cognito, Amazon,
 * Facebook, Google, etc.
 */
class WebIdentityPrincipal extends FederatedPrincipal {
    /**
     *
     * @param identityProvider identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(identityProvider, conditions = {}) {
        super(identityProvider, conditions !== null && conditions !== void 0 ? conditions : {}, 'sts:AssumeRoleWithWebIdentity');
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `WebIdentityPrincipal(${this.federated})`;
    }
}
exports.WebIdentityPrincipal = WebIdentityPrincipal;
/**
 * A principal that represents a federated identity provider as from a OpenID Connect provider.
 */
class OpenIdConnectPrincipal extends WebIdentityPrincipal {
    /**
     *
     * @param openIdConnectProvider OpenID Connect provider
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(openIdConnectProvider, conditions = {}) {
        super(openIdConnectProvider.openIdConnectProviderArn, conditions !== null && conditions !== void 0 ? conditions : {});
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `OpenIdConnectPrincipal(${this.federated})`;
    }
}
exports.OpenIdConnectPrincipal = OpenIdConnectPrincipal;
/**
 * Use the AWS account into which a stack is deployed as the principal entity in a policy
 */
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    toString() {
        return 'AccountRootPrincipal()';
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return 'AnyPrincipal()';
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
/**
 * Represents a principal that has multiple types of principals. A composite principal cannot
 * have conditions. i.e. multiple ServicePrincipals that form a composite principal
 */
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    /**
     * Adds IAM principals to the composite principal. Composite principals cannot have
     * conditions.
     *
     * @param principals IAM principals that will be added to the composite principal
     */
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error('Cannot add multiple principals with different "assumeRoleAction". ' +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error('Components of a CompositePrincipal must not have conditions. ' +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        return this.fn(cdk.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return '<unresolved-token>';
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || cdk.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, cdk.Aws.URL_SUFFIX);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service,
        });
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,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