"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Role = void 0;
const core_1 = require("@aws-cdk/core");
const grant_1 = require("./grant");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const policy_document_1 = require("./policy-document");
const policy_statement_1 = require("./policy-statement");
const principals_1 = require("./principals");
const immutable_role_1 = require("./private/immutable-role");
const util_1 = require("./util");
/**
 * IAM Role
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 */
class Role extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.roleName,
        });
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.managedPolicies = [];
        this.attachedPolicies = new util_1.AttachedPolicies();
        const externalIds = props.externalIds || [];
        if (props.externalId) {
            externalIds.push(props.externalId);
        }
        this.assumeRolePolicy = createAssumeRolePolicy(props.assumedBy, externalIds);
        this.managedPolicies.push(...props.managedPolicies || []);
        this.permissionsBoundary = props.permissionsBoundary;
        const maxSessionDuration = props.maxSessionDuration && props.maxSessionDuration.toSeconds();
        validateMaxSessionDuration(maxSessionDuration);
        const description = (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 0) ? props.description : undefined;
        if (description && description.length > 1000) {
            throw new Error('Role description must be no longer than 1000 characters.');
        }
        const role = new iam_generated_1.CfnRole(this, 'Resource', {
            assumeRolePolicyDocument: this.assumeRolePolicy,
            managedPolicyArns: core_1.Lazy.listValue({ produce: () => this.managedPolicies.map(p => p.managedPolicyArn) }, { omitEmpty: true }),
            policies: _flatten(props.inlinePolicies),
            path: props.path,
            permissionsBoundary: this.permissionsBoundary ? this.permissionsBoundary.managedPolicyArn : undefined,
            roleName: this.physicalName,
            maxSessionDuration,
            description,
        });
        this.roleId = role.attrRoleId;
        this.roleArn = this.getResourceArnAttribute(role.attrArn, {
            region: '',
            service: 'iam',
            resource: 'role',
            resourceName: this.physicalName,
        });
        this.roleName = this.getResourceNameAttribute(role.ref);
        this.policyFragment = new principals_1.ArnPrincipal(this.roleArn).policyFragment;
        function _flatten(policies) {
            if (policies == null || Object.keys(policies).length === 0) {
                return undefined;
            }
            const result = new Array();
            for (const policyName of Object.keys(policies)) {
                const policyDocument = policies[policyName];
                result.push({ policyName, policyDocument });
            }
            return result;
        }
    }
    /**
     * Import an external role by ARN.
     *
     * If the imported Role ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * role has a `path` (like `arn:...:role/AdminRoles/Alice`), the
     * `roleName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Role ARN should be supplied without the
     * `path` in order to resolve the correct role resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param roleArn the ARN of the role to import
     * @param options allow customizing the behavior of the returned role
     */
    static fromRoleArn(scope, id, roleArn, options = {}) {
        const scopeStack = core_1.Stack.of(scope);
        const parsedArn = scopeStack.parseArn(roleArn);
        const resourceName = parsedArn.resourceName;
        // service roles have an ARN like 'arn:aws:iam::<account>:role/service-role/<roleName>'
        // we want to support these as well, so strip out the 'service-role/' prefix if we see it
        const roleName = resourceName.startsWith('service-role/')
            ? resourceName.slice('service-role/'.length)
            : resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.grantPrincipal = this;
                this.assumeRoleAction = 'sts:AssumeRole';
                this.policyFragment = new principals_1.ArnPrincipal(roleArn).policyFragment;
                this.roleArn = roleArn;
                this.roleName = roleName;
                this.attachedPolicies = new util_1.AttachedPolicies();
            }
            addToPolicy(statement) {
                return this.addToPrincipalPolicy(statement).statementAdded;
            }
            addToPrincipalPolicy(statement) {
                if (!this.defaultPolicy) {
                    this.defaultPolicy = new policy_1.Policy(this, 'Policy');
                    this.attachInlinePolicy(this.defaultPolicy);
                }
                this.defaultPolicy.addStatements(statement);
                return { statementAdded: true, policyDependable: this.defaultPolicy };
            }
            attachInlinePolicy(policy) {
                const policyAccount = core_1.Stack.of(policy).account;
                if (accountsAreEqualOrOneIsUnresolved(policyAccount, roleAccount)) {
                    this.attachedPolicies.attach(policy);
                    policy.attachToRole(this);
                }
            }
            addManagedPolicy(_policy) {
                // FIXME: Add warning that we're ignoring this
            }
            /**
             * Grant permissions to the given principal to pass this role.
             */
            grantPassRole(identity) {
                return this.grant(identity, 'iam:PassRole');
            }
            /**
             * Grant the actions defined in actions to the identity Principal on this resource.
             */
            grant(grantee, ...actions) {
                return grant_1.Grant.addToPrincipal({
                    grantee,
                    actions,
                    resourceArns: [this.roleArn],
                    scope: this,
                });
            }
        }
        const roleAccount = parsedArn.account;
        const scopeAccount = scopeStack.account;
        return options.mutable !== false && accountsAreEqualOrOneIsUnresolved(scopeAccount, roleAccount)
            ? new Import(scope, id)
            : new immutable_role_1.ImmutableRole(scope, `ImmutableRole${id}`, new Import(scope, id));
        function accountsAreEqualOrOneIsUnresolved(account1, account2) {
            return core_1.Token.isUnresolved(account1) || core_1.Token.isUnresolved(account2) ||
                account1 === account2;
        }
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPrincipalPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.attachInlinePolicy(this.defaultPolicy);
        }
        this.defaultPolicy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.defaultPolicy };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The the managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.managedPolicies.find(mp => mp === policy)) {
            return;
        }
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToRole(this);
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(grantee, ...actions) {
        return grant_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.roleArn],
            scope: this,
        });
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        return this.grant(identity, 'iam:PassRole');
    }
    /**
     * Return a copy of this Role object whose Policies will not be updated
     *
     * Use the object returned by this method if you want this Role to be used by
     * a construct without it automatically updating the Role's Policies.
     *
     * If you do, you are responsible for adding the correct statements to the
     * Role's policies yourself.
     */
    withoutPolicyUpdates() {
        if (!this.immutableRole) {
            this.immutableRole = new immutable_role_1.ImmutableRole(this.node.scope, `ImmutableRole${this.node.id}`, this);
        }
        return this.immutableRole;
    }
}
exports.Role = Role;
function createAssumeRolePolicy(principal, externalIds) {
    const statement = new AwsStarStatement();
    statement.addPrincipals(principal);
    statement.addActions(principal.assumeRoleAction);
    if (externalIds.length) {
        statement.addCondition('StringEquals', { 'sts:ExternalId': externalIds.length === 1 ? externalIds[0] : externalIds });
    }
    const doc = new policy_document_1.PolicyDocument();
    doc.addStatements(statement);
    return doc;
}
function validateMaxSessionDuration(duration) {
    if (duration === undefined) {
        return;
    }
    if (duration < 3600 || duration > 43200) {
        throw new Error(`maxSessionDuration is set to ${duration}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`);
    }
}
/**
 * A PolicyStatement that normalizes its Principal field differently
 *
 * Normally, "anyone" is normalized to "Principal: *", but this statement
 * normalizes to "Principal: { AWS: * }".
 */
class AwsStarStatement extends policy_statement_1.PolicyStatement {
    toStatementJson() {
        const stat = super.toStatementJson();
        if (stat.Principal === '*') {
            stat.Principal = { AWS: '*' };
        }
        return stat;
    }
}
//# sourceMappingURL=data:application/json;base64,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