import torch
from torch import nn
from fmot.nn import Sequencer, MIMOSequencer, ParameterQuantizer, LUT, Loop
from .. import torchscript_utils as utils
from .default_patchings import DEFAULT_PATCHINGS
import textwrap
from importlib import import_module
from .lut_registry import LUT_REGISTRY
import inspect
from typing import *
from .apply_tags import copy_tags

DEBUG = False


def _rename_vars(varnames):
    newvarnames = []
    for v in varnames:
        if v[0].isnumeric():
            v = "var" + v
        newvarnames.append(v)
    return newvarnames


class PatchIR:
    """
    A serializable graph IR, replacing torcscript IR for use in PatchedModules

    :param str fname: The name of the function being represented in the graph
        (typically ``forward``).
    """

    def __init__(self, fname, signature: List[str] = None):
        self.fname = fname
        self.inputs = []
        self.nodes = []
        self.outputs = []
        self.signature = signature

    def add_input(self, x):
        """
        Add an input argument to the PatchIR.

        :param str x: the name of the variable. Should be unique. If it is
            numeric, it is prepended with ``var``. For example, an input
            with name ``2`` would be renamed ``var2``.
        """
        assert x not in self.inputs, f"Input {x} already in inputs"
        if x != "self":
            self.inputs.append(_rename_vars([x])[0])

    def add_output(self, x):
        """
        Add an output argument to the PatchIR.

        :param str x: the name of the variable. Should be unique. If it is
            numeric, it is prepended with ``var``. For example, an output
            with name ``2`` would be renamed ``var2``.
        """
        assert x not in self.outputs, f"Output {x} already in inputs"
        self.outputs.append(_rename_vars([x])[0])

    def add_node(self, node):
        """
        Add a node to the PatchIR graph. Order matters!

        :param PatchIRNode node: The node to be added to the graph.
        """
        self.nodes.append(node)

    def populate_inputs(self, *args, **kwargs):
        objs = {}

        for i, arg in enumerate(args):
            objs[self.inputs[i]] = arg

        if len(kwargs) != 0:
            if self.signature is None:
                raise ValueError(
                    f"Could not run PatchIR with kwargs... unknown signature"
                )
            for key, arg in kwargs.items():
                idx = self.signature.index(key)
                objs[self.inputs[idx]] = arg

        return objs

    def call(self, module, *args, **kwargs):
        """
        Execute PatchIR graph, using a pytorch module.

        :param torch.nn.Module module: The module used to resource any
            references to ``self`` in the PatchIR graph.
        :param args: Arguments to the PatchIR graph.

        :returns: Outputs, as enumerated by `add_outputs`
        """
        try:
            # Collect inputs into object dictionary
            objs = self.populate_inputs(*args, **kwargs)
            objs["self"] = module

            # Call all nodes
            for node in self.nodes:
                try:
                    node(objs)
                except Exception as e:
                    message = str(node) + "\n" + str(e)
                    print(message)
                    raise e
                if DEBUG:
                    print(f"Calling node: {node}")

            # Collect outputs
            outputs = []
            for name in self.outputs:
                outputs.append(objs[name])
                if DEBUG:
                    print(f"Collecting output: {name}")

            outputs = tuple(outputs)
        except:
            message = "Could not run PatchIR. \nCode Dump:\n"
            message += textwrap.indent(str(self), "   ")
            message += "\n" + "-" * 45 + "\n"
            print(message)
            raise

        if len(outputs) == 0:
            return None
        elif len(outputs) == 1:
            return outputs[0]
        else:
            return outputs

    def __repr__(self):
        t = "    "
        rep = f'def {self.fname}({", ".join(["self"] + self.inputs)}):\n{t}'
        rep += f"\n{t}".join([n.__repr__() for n in self.nodes])
        rep += f'\n{t}return {", ".join(self.outputs)}'
        return rep


##############################
# > PatchIR Nodes


class PatchIRNode:
    """
    Base PatchIRNode class, subclassed to perform different actions

    :param str[] inputs: A list of input variable names
    :param str[] outputs: A list of output variable names
    :param str sourceref: A reference to user's code (usually generated by TorchScript IR)
    """

    def __init__(self, inputs, outputs, sourceref, **kwargs):
        self.inputs = _rename_vars(inputs)
        self.outputs = _rename_vars(outputs)
        self.sourceref = sourceref
        for k, v in kwargs.items():
            self.__setattr__(k, v)

    def input(self):
        assert len(self.inputs) == 1
        return self.inputs[0]

    def output(self):
        assert len(self.outputs) == 1
        return self.outputs[0]

    def get_inputs(self, objs):
        try:
            return [objs[x] for x in self.inputs]
        except:
            print(f"Failed to pull {self.inputs} from {objs.keys()}")
            raise

    def attach_outputs(self, objs, *outputs):
        for name, output in zip(self.outputs, outputs):
            objs[name] = output

    def __call__(self, objs):
        inputs = self.get_inputs(objs)
        outputs = self.act(*inputs)
        if not isinstance(outputs, tuple):
            outputs = (outputs,)
        self.attach_outputs(objs, *outputs)

    def act(self, *args):
        """
        Needs to be implemented for each PatchIR Subclass.
        """
        raise NotImplementedError

    @classmethod
    def from_torchscript(cls, node):
        """
        This classmethod constructs and returns a node from a TorchScript IR node

        :param node: TorchScript IR node
        """
        raise NotImplementedError


class GetParamNode(PatchIRNode):
    param_name = None  #: name of parameter to get

    def act(self, parent):
        """
        Run ``parent.param_name`` through a quantizer
        """
        quantizer = parent.quantdict[self.param_name]
        x = getattr(parent, self.param_name)
        x.dimensions = list(quantizer.dimensions)
        return quantizer(x)

    @classmethod
    def from_torchscript(cls, node):
        """
        Factory method to create a GetParamNode from a TorchScript IR node.
        """
        assert utils.isgetparam(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        param_name = utils.get_attr_name(node)
        return cls(inputs, outputs, sourceref, param_name=param_name)

    def __repr__(self):
        rep = f'{self.output()} = {self.input()}.quantdict["{self.param_name}"]'
        rep += f"({self.input()}.{self.param_name})"
        return rep


class GetAttrNode(PatchIRNode):
    attr_name = None  #: Name of attribute to get

    def act(self, parent):
        """
        Get ``parent.attr_name``
        """
        return getattr(parent, self.attr_name)

    @classmethod
    def from_torchscript(cls, node):
        """
        Factory method to create a GetAttrNode from a TorchScript IR node.
        """
        assert utils.isgetattrmodule(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        attr_name = utils.get_attr_name(node)
        return cls(inputs, outputs, sourceref, attr_name=attr_name)

    def __repr__(self):
        return f"{self.output()} = {self.input()}.{self.attr_name}"


class CallMethodNode(PatchIRNode):
    method_name = None  #: name of method to call

    def act(self, parent, *args):
        """
        Call ``parent.method_name(*args)``
        """
        method = parent.__getattribute__(self.method_name)
        parent._sourceref = self.sourceref
        return (method(*args),)

    @classmethod
    def from_torchscript(cls, node):
        """
        Factory method to create a CallMethodNode from a TorchScript IR node.
        """
        assert utils.iscallmethod(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        method_name = utils.get_attr_name(node)
        if method_name.startswith("forward"):
            method_name = "__call__"
        return cls(inputs, outputs, sourceref, method_name=method_name)

    def __repr__(self):
        method_name = self.method_name
        if method_name == "__call__":
            method_name = "forward"
        rep = f'{", ".join(self.outputs)} = {self.inputs[0]}.{method_name}'
        rep += f'({", ".join(self.inputs[1:])})'
        return rep


class PrimCallMethodNode(CallMethodNode):
    @classmethod
    def from_torchscript(cls, node):
        assert utils.isprimcallmethod(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        method_name = "__call__"
        return cls(inputs, outputs, sourceref, method_name=method_name)


class AtenPatchNode(PatchIRNode):
    fname = None  #: Original name of the function
    patch_idx = None  #: Patch index in the patchdict

    def act(self, parent, *args):
        """
        Calls ``parent.patchlist[patch_idx](*args)``
        """
        patch = parent.patchlist[self.patch_idx]
        patch._sourceref = self.sourceref
        try:
            return (patch(*args),)
        except:
            raise

    @classmethod
    def from_torchscript(cls, node, patch_idx):
        """
        Factory method to create an AtenPatchNode from a TorchScript IR node.
        """
        assert utils.isaten(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(
            node, tensorial_inputs=True
        )
        inputs = ["self"] + inputs
        fname = node.kind()
        return cls(inputs, outputs, sourceref, patch_idx=patch_idx, fname=fname)

    def __repr__(self):
        rep = (
            f'{", ".join(self.outputs)} = {self.inputs[0]}.patchlist[{self.patch_idx}]'
        )
        rep += f'({", ".join(self.inputs[1:])}) #{self.fname}'
        return rep


class NonePatchNode(PatchIRNode):
    def act(self, parent, *args):
        """
        Calls ``parent.patchlist[patch_idx](*args)``
        """
        return (None,)

    @classmethod
    def from_torchscript(cls, node):
        """
        Factory method to create an AtenPatchNode from a TorchScript IR node.
        """
        assert utils.isNone(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        inputs = ["self"] + inputs
        return cls(inputs, outputs, sourceref)

    def __repr__(self):
        rep = f'{", ".join(self.outputs)} = None'
        return rep


class FunctionalPatchNode(AtenPatchNode):
    @classmethod
    def from_torchscript(cls, node, patch_idx):
        assert utils.isfunctional(node) or utils.isfmotfunctional(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(
            node, tensorial_inputs=True
        )
        inputs = ["self"] + inputs
        fname = utils.getfunctionalname(node)
        return cls(inputs, outputs, sourceref, patch_idx=patch_idx, fname=fname)


class UserFunctionPatchNode(AtenPatchNode):
    @classmethod
    def from_torchscript(cls, node, patch_idx):
        assert utils.isuserfunction(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(
            node, tensorial_inputs=True
        )
        inputs = ["self"] + inputs
        fname = f"custom_fn: {utils.getuserfuncname(node)}"
        return cls(inputs, outputs, sourceref, patch_idx=patch_idx, fname=fname)


class ListConstructNode(PatchIRNode):
    def act(self, *args):
        """
        Returns ``[a for a in args]``
        """
        return [a for a in args]

    @classmethod
    def from_torchscript(cls, node):
        assert utils.islistconstruct(node, tensorlist=True)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        return cls(inputs, outputs, sourceref)

    def __repr__(self):
        return f"{self.output()} = {self.inputs}"


class GetItemNode(PatchIRNode):
    def act(self, obj, key):
        return obj[key]

    @classmethod
    def from_torchscript(cls, node):
        assert utils.islistgetitem(node) or utils.isdictgetitem(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        return cls(inputs, outputs, sourceref)

    def __repr__(self):
        return f"{self.output()} = {self.inputs[0]}[{self.inputs[1]}]"


class PrimConstantNode(PatchIRNode):
    constant = None  #: constant value

    def act(self, *args):
        return self.constant

    @classmethod
    def from_torchscript(cls, node):
        assert utils.isprimconstant(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        ret = cls(inputs, outputs, sourceref)
        ret.constant = utils.get_constant_value(node)
        return ret

    def __repr__(self):
        return f"{self.output()} = {self.constant}"


class ListAppendNode(PatchIRNode):
    def act(self, tlist, item):
        tlist.append(item)
        return tlist

    @classmethod
    def from_torchscript(cls, node):
        assert utils.islistappend(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        return cls(inputs, outputs, sourceref)

    def __repr__(self):
        return f"{self.output()} = {self.inputs[0]}.append({self.inputs[1]})"


class TupleConstructNode(PatchIRNode):
    def act(self, *args):
        """
        Returns ``tuple(args)``
        """
        return (tuple([a for a in args]),)

    @classmethod
    def from_torchscript(cls, node):
        assert utils.istupleconstruct(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        return cls(inputs, outputs, sourceref)

    def __repr__(self):
        return f'{self.output()} = ({", ".join([str(x) for x in self.inputs])})'


class TupleUnpackNode(PatchIRNode):
    def act(self, x):
        return x

    @classmethod
    def from_torchscript(cls, node):
        assert utils.istupleunpack(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        return cls(inputs, outputs, sourceref)

    def __repr__(self):
        return f'{", ".join([str(x) for x in self.outputs])} = {self.input()}'


class ListUnpackNode(PatchIRNode):
    def act(self, x):
        """
        Unpacks a list
        """
        return tuple(x)

    @classmethod
    def from_torchscript(cls, node):
        assert utils.islistunpack(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        return cls(inputs, outputs, sourceref)

    def __repr__(self):
        return f'{", ".join([str(x) for x in self.outputs])} = {self.input()}'


class AtenSizeNode(PatchIRNode):
    def act(self, x):
        return list(x.shape)

    @classmethod
    def from_torchscript(cls, node):
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        return cls(inputs, outputs, sourceref)

    def __repr__(self):
        return f"{self.output()} = {self.input()}.shape"


class PrintNode(PatchIRNode):
    def act(self, x):
        print(x)

    @classmethod
    def from_torchscript(cls, node):
        assert utils.isprint(node)
        inputs, outputs, sourceref = utils.get_inputs_outputs_sourceref(node)
        return cls(inputs, outputs, sourceref)

    def __repr__(self):
        return f"print({self.input()})"


def importuserfunction(fname):
    p, m = fname.rsplit(".", 1)
    mod = import_module(p)
    fn = getattr(mod, m)
    return fn


###############################
# > CONSTRUCTING FROM TORCH


def quantdict_from_torch(module):
    """
    :param torch.nn.Module module: Parent module

    :returns: A ModuleDict of parameter quantizers for each direct child parameter
    """
    quantdict = nn.ModuleDict()
    for name, param in module.named_parameters(recurse=False):
        quantdict[name] = ParameterQuantizer(is_weight=param.dim() >= 2)
    return quantdict


def get_graph(module, step=False):
    module: torch.nn.Module = module
    if not hasattr(module, "graph"):
        module = torch.jit.script(module)

    if step:
        graph = module.step.graph

    else:
        try:
            graph = module.graph
        except:
            print(module)
            print(step)
            raise
    return graph


def patchlist_from_torch(module, step=False, extra_patchings=None):
    """
    Inspects a module's TorchScript IR graph to create a list of patch modules.
    Each operator that need patching (i.e. direct calls to ``aten::*`` or
    ``torch.nn.functional``) will result in a new patch module being instantiated.

    :param torch.nn.Module module: Parent module
    :param bool step: Whether to use ``.step`` method or ``.forward`` method to
        determine which TorchScript IR nodes need to be patched. Default is False
        (i.e. ``.forward``).
    :param dict extra_patchings: Patch rules to supplement DEFAULT_PATCHINGS

    :returns: A ModuleList of patch modules
    """

    # Get/update patching dict

    patchings = DEFAULT_PATCHINGS
    if extra_patchings is not None:
        patchings.update(extra_patchings)
    # Get torchscript IR graph
    graph = get_graph(module, step)
    # Instantiate patch modules in patchlist where necessary
    patchlist = nn.ModuleList()
    for node in graph.nodes():
        patch_key = None
        patch = None
        if utils.ispythonicaten(node):
            pass
        elif utils.isaten(node):
            patch_key = node.kind()
        elif utils.isfunctional(node):
            patch_key = utils.getfunctionalname(node)
        elif utils.isuserfunction(node):
            patch_key = utils.getuserfuncname(node)

        if patch_key in LUT_REGISTRY:
            patch = LUT(LUT_REGISTRY[patch_key])
        elif patch_key is not None:
            if patch_key in patchings:
                patch_rule = patchings[patch_key]
                if hasattr(patch_rule, "_from_node"):
                    patch = patch_rule._from_node(node, module)
                else:
                    patch = patch_rule()
            else:
                print("PatchIR Conversion Failed")
                print(graph)
                raise KeyError(f"patch_key={patch_key} not supported")

        if patch is not None:
            patchlist.append(patch)
    return patchlist


def patchir_from_torch(module, step=False):
    """
    Given a model, translate its TorchScript IR graph into a PatchIR graph.

    :param torch.nn.Module module: Pytorch model from which to construct a
        PatchIR graph
    :param bool step: Whether to use the model's step method or forward method.

    :returns: PatchIR graph
    """
    fname = "step" if step else "forward"

    signature = inspect.signature(module.forward)
    signature = list(signature.parameters.keys())

    pir = PatchIR(fname, signature=signature)
    graph = get_graph(module, step)
    patch_idx = 0
    for x in graph.inputs():
        if DEBUG:
            print(f"Registering input {x.debugName()}")
        pir.add_input(x.debugName())
    for x in graph.outputs():
        if DEBUG:
            print(f"Registering output {x.debugName()}")
        pir.add_output(x.debugName())
    for node in graph.nodes():
        if utils.isgetparam(node):
            if DEBUG:
                print(f"Registering GetParamNode")
            pir.add_node(GetParamNode.from_torchscript(node))
        elif utils.isgetattrmodule(node):
            if DEBUG:
                print(f"Registering GetAttrNode")
            pir.add_node(GetAttrNode.from_torchscript(node))
        elif utils.iscallmethod(node):
            if DEBUG:
                print("Registering CallMethodNode")
            pir.add_node(CallMethodNode.from_torchscript(node))
        elif utils.isprimcallmethod(node):
            pir.add_node(PrimCallMethodNode.from_torchscript(node))
        elif utils.islistgetitem(node):
            pir.add_node(GetItemNode.from_torchscript(node))
        elif utils.isdictgetitem(node):
            pir.add_node(GetItemNode.from_torchscript(node))
        elif utils.islistappend(node):
            pir.add_node(ListAppendNode.from_torchscript(node))
        elif utils.isprimconstant(node):
            pir.add_node(PrimConstantNode.from_torchscript(node))
        elif utils.isatensize(node):
            pir.add_node(AtenSizeNode.from_torchscript(node))
        elif utils.isaten(node):
            if DEBUG:
                print("Registering AtenPatchNode")
            pir.add_node(AtenPatchNode.from_torchscript(node, patch_idx))
            patch_idx += 1
        elif utils.isfunctional(node):
            if DEBUG:
                print("Registering FunctionalPatchNode")
            pir.add_node(FunctionalPatchNode.from_torchscript(node, patch_idx))
            patch_idx += 1
        elif utils.isfmotfunctional(node):
            if DEBUG:
                print("Registering FMOTFunctionalPatchNode")
            pir.add_node(FunctionalPatchNode.from_torchscript(node, patch_idx))
            patch_idx += 1
        elif utils.islistconstruct(node, tensorlist=True):
            if DEBUG:
                print("Registering ListConstructNode")
            pir.add_node(ListConstructNode.from_torchscript(node))
        elif utils.islistunpack(node):
            if DEBUG:
                print("Registering ListUnpackNode")
            pir.add_node(ListUnpackNode.from_torchscript(node))
        elif utils.istupleconstruct(node):
            if DEBUG:
                print("Registering TupleConstructNode")
            pir.add_node(TupleConstructNode.from_torchscript(node))
        elif utils.istupleunpack(node):
            if DEBUG:
                print("Registering TupleUnpackNode")
            pir.add_node(TupleUnpackNode.from_torchscript(node))
        elif utils.isuserfunction(node):
            pir.add_node(UserFunctionPatchNode.from_torchscript(node, patch_idx))
            patch_idx += 1
        elif utils.isprint(node):
            pir.add_node(PrintNode.from_torchscript(node))
        elif utils.isNone(node):
            pass
            pir.add_node(NonePatchNode.from_torchscript(node))
    return pir


class PatchedModule(nn.Module):
    """
    PatchedModules remove any direct function calls in a model's computational
    graph, replacing them with modular patches. For example, a model that
    calls ``torch.relu(x)`` would be patched by instantiating a ``torch.nn.ReLU``
    module and calling its ``forward`` method.

    PatchedModules also patch direct parameter accesses through quantizers.
    Instead of directly accessing parameters, each parameter access needs to be
    wrapped by a parameter quantizer.
    """

    def __init__(self, orig_name, patchlist, quantdict, patchIR):
        super().__init__()
        self.orig_name = orig_name  #: original name of the model
        self.patchlist = (
            patchlist  #: ``torch.nn.ModuleList`` containing the patch modules
        )
        self.quantdict = (
            quantdict  #: ``torch.nn.ModuleDict`` containing the parameter quantizers
        )
        self.patchIR = patchIR  #: PatchIR graph, making the PatchedModule executable

    def forward(self, *args, **kwargs):
        return self.patchIR.call(self, *args, **kwargs)

    @classmethod
    def from_torch(cls, module, extra_patchings=None):
        """
        Instantiate a PatchedModule from an arbitrary ``torch.nn.Module``.

        Patchlist is constructed by calling ``patchlist_from_torch``
        Quantdict is constructed by calling ``quantdict_from_torch``
        PatchIR is constructed by calling ``patchir_from_torch``

        :param torch.nn.Module module: Original pytorch model
        :param bool step: Whether to use the pytorch model's step method, or its
            forward method when constructing the PatchIR graph. Default is False (i.e. forward)
        :param dict extra_patchings: Patching rules to supplement DEFAULT_PATCHINGS

        :returns: A PatchedModule
        """
        orig_name = type(module).__name__
        patchlist = patchlist_from_torch(
            module, step=False, extra_patchings=extra_patchings
        )
        quantdict = quantdict_from_torch(module)
        if len(patchlist) == 0:
            patchlist = None
        if len(quantdict) == 0:
            quantdict = None
        pir = patchir_from_torch(module, step=False)
        pmodule = cls(orig_name, patchlist, quantdict, pir)
        for name, param in module.named_parameters(recurse=False):
            pmodule.__setattr__(name, param)
        for name, mod in module.named_children():
            pmodule.__setattr__(name, mod)
        return pmodule

    def code(self):
        """
        Semi-human-legible python code, generated from the PatchIR
        """
        return self.patchIR.__repr__()


class PatchedSequencer(Sequencer):
    """
    PatchedModules remove any direct function calls in a model's computational
    graph, replacing them with modular patches. For example, a model that
    calls ``torch.relu(x)`` would be patched by instantiating a ``torch.nn.ReLU``
    module and calling its ``forward`` method.

    PatchedModules also patch direct parameter accesses through quantizers.
    Instead of directly accessing parameters, each parameter access needs to be
    wrapped by a parameter quantizer.
    """

    def __init__(
        self,
        orig_name,
        patchlist,
        quantdict,
        patchIR,
        state_shapes,
        seq_dim,
        return_hidden_state,
    ):
        super().__init__(
            state_shapes=state_shapes,
            seq_dim=seq_dim,
            return_hidden_state=return_hidden_state,
        )
        self.orig_name = orig_name  #: original name of the model
        self.patchlist = (
            patchlist  #: ``torch.nn.ModuleList`` containing the patch modules
        )
        self.quantdict = (
            quantdict  #: ``torch.nn.ModuleDict`` containing the parameter quantizers
        )
        self.patchIR = patchIR  #: PatchIR graph, making the PatchedModule executable

    def step(self, *args):
        return self.patchIR.call(self, *args)

    @classmethod
    def from_torch(cls, module, extra_patchings=None):
        """
        Instantiate a PatchedSequencer from an arbitrary ``torch.nn.Module``.

        Patchlist is constructed by calling ``patchlist_from_torch``
        Quantdict is constructed by calling ``quantdict_from_torch``
        PatchIR is constructed by calling ``patchir_from_torch``

        :param torch.nn.Module module: Original pytorch model
        :param bool step: Whether to use the pytorch model's step method, or its
            forward method when constructing the PatchIR graph. Default is False (i.e. forward)
        :param dict extra_patchings: Patching rules to supplement DEFAULT_PATCHINGS

        :returns: A PatchedModule
        """
        assert isinstance(module, Sequencer)
        orig_name = type(module).__name__
        patchlist = patchlist_from_torch(
            module, step=True, extra_patchings=extra_patchings
        )
        quantdict = quantdict_from_torch(module)
        if len(patchlist) == 0:
            patchlist = None
        if len(quantdict) == 0:
            quantdict = None
        pir = patchir_from_torch(module, step=True)
        pmodule = cls(
            orig_name,
            patchlist,
            quantdict,
            pir,
            state_shapes=module.state_shapes,
            seq_dim=module.seq_dim,
            return_hidden_state=module.return_hidden_state,
        )
        for name, param in module.named_parameters(recurse=False):
            pmodule.__setattr__(name, param)
        for name, mod in module.named_children():
            pmodule.__setattr__(name, mod)
        return pmodule

    def code(self):
        """
        Semi-human-legible python code, generated from the PatchIR
        """
        return self.patchIR.__repr__()


class PatchedMIMOSequencer(MIMOSequencer):
    """
    PatchedModules remove any direct function calls in a model's computational
    graph, replacing them with modular patches. For example, a model that
    calls ``torch.relu(x)`` would be patched by instantiating a ``torch.nn.ReLU``
    module and calling its ``forward`` method.

    PatchedModules also patch direct parameter accesses through quantizers.
    Instead of directly accessing parameters, each parameter access needs to be
    wrapped by a parameter quantizer.
    """

    def __init__(
        self,
        orig_name: str,
        patchlist: torch.nn.ModuleList,
        quantdict: torch.nn.ModuleDict,
        patchIR: PatchIR,
        state_shapes: list[list[int]],
        seq_dim: int,
        num_inputs: int,
        num_outputs: int,
        return_hidden_state: bool = True,
    ):
        super().__init__(
            num_inputs=num_inputs,
            num_outputs=num_outputs,
            state_shapes=state_shapes,
            seq_dim=seq_dim,
            return_hidden_state=return_hidden_state,
        )
        self.orig_name = orig_name  #: original name of the model
        self.patchlist = (
            patchlist  #: ``torch.nn.ModuleList`` containing the patch modules
        )
        self.quantdict = (
            quantdict  #: ``torch.nn.ModuleDict`` containing the parameter quantizers
        )
        self.patchIR = patchIR  #: PatchIR graph, making the PatchedModule executable

    def step(self, *args):
        return self.patchIR.call(self, *args)

    @classmethod
    def from_torch(cls, module, extra_patchings=None):
        """
        Instantiate a PatchedSequencer from an arbitrary ``torch.nn.Module``.

        Patchlist is constructed by calling ``patchlist_from_torch``
        Quantdict is constructed by calling ``quantdict_from_torch``
        PatchIR is constructed by calling ``patchir_from_torch``

        :param torch.nn.Module module: Original pytorch model
        :param bool step: Whether to use the pytorch model's step method, or its
            forward method when constructing the PatchIR graph. Default is False (i.e. forward)
        :param dict extra_patchings: Patching rules to supplement DEFAULT_PATCHINGS

        :returns: A PatchedModule
        """
        assert isinstance(module, MIMOSequencer)
        orig_name = type(module).__name__
        patchlist = patchlist_from_torch(
            module, step=True, extra_patchings=extra_patchings
        )
        quantdict = quantdict_from_torch(module)
        if len(patchlist) == 0:
            patchlist = None
        if len(quantdict) == 0:
            quantdict = None
        pir = patchir_from_torch(module, step=True)

        pmodule = cls(
            orig_name=orig_name,
            patchlist=patchlist,
            quantdict=quantdict,
            patchIR=pir,
            state_shapes=module.state_shapes,
            seq_dim=module.seq_dim,
            num_inputs=module.num_inputs,
            num_outputs=module.num_outputs,
            return_hidden_state=module.return_hidden_state,
        )
        for name, param in module.named_parameters(recurse=False):
            pmodule.__setattr__(name, param)
        for name, mod in module.named_children():
            pmodule.__setattr__(name, mod)
        return pmodule

    def code(self):
        """
        Semi-human-legible python code, generated from the PatchIR
        """
        return self.patchIR.__repr__()


class PatchedLoop(Loop):
    def __init__(
        self,
        orig_name: str,
        patchlist: torch.nn.ModuleList,
        quantdict: torch.nn.ModuleDict,
        patchIR: PatchIR,
        n_iter: int,
        n_recurse: int,
        slice_blocksizes: list[int],
        slice_reversed: Union[bool, list[bool]] = False,
        concat_reversed: Union[bool, list[bool]] = False,
        dim: int = -1,
    ):
        super().__init__(
            n_iter=n_iter,
            n_recurse=n_recurse,
            slice_blocksizes=slice_blocksizes,
            slice_reversed=slice_reversed,
            concat_reversed=concat_reversed,
            dim=dim,
        )
        self.orig_name = orig_name
        self.patchlist = patchlist
        self.quantdict = quantdict
        self.patchIR = patchIR

    def step(self, *args):
        return self.patchIR.call(self, *args)

    @classmethod
    def from_torch(cls, module, extra_patchings=None):
        """
        Instantiate a PatchedSequencer from an arbitrary ``torch.nn.Module``.

        Patchlist is constructed by calling ``patchlist_from_torch``
        Quantdict is constructed by calling ``quantdict_from_torch``
        PatchIR is constructed by calling ``patchir_from_torch``

        :param torch.nn.Module module: Original pytorch model
        :param bool step: Whether to use the pytorch model's step method, or its
            forward method when constructing the PatchIR graph. Default is False (i.e. forward)
        :param dict extra_patchings: Patching rules to supplement DEFAULT_PATCHINGS

        :returns: A PatchedModule
        """
        assert isinstance(module, Loop)
        orig_name = type(module).__name__
        patchlist = patchlist_from_torch(
            module, step=True, extra_patchings=extra_patchings
        )
        quantdict = quantdict_from_torch(module)
        if len(patchlist) == 0:
            patchlist = None
        if len(quantdict) == 0:
            quantdict = None
        pir = patchir_from_torch(module, step=True)

        pmodule = cls(
            orig_name=orig_name,
            patchlist=patchlist,
            quantdict=quantdict,
            patchIR=pir,
            n_iter=module.n_iter,
            n_recurse=module.n_recurse,
            slice_blocksizes=module.slice_blocksizes,
            slice_reversed=module.slice_reversed,
            concat_reversed=module.concat_reversed,
            dim=module.dim,
        )
        for name, param in module.named_parameters(recurse=False):
            pmodule.__setattr__(name, param)
        for name, mod in module.named_children():
            pmodule.__setattr__(name, mod)
        return pmodule

    def code(self):
        """
        Semi-human-legible python code, generated from the PatchIR
        """
        return self.patchIR.__repr__()


def get_patched_module_from_torch(module, extra_patchings=None):
    if isinstance(module, Sequencer):
        if isinstance(module, MIMOSequencer):
            out = PatchedMIMOSequencer.from_torch(module, extra_patchings)
        else:
            out = PatchedSequencer.from_torch(module, extra_patchings)
    elif isinstance(module, Loop):
        out = PatchedLoop.from_torch(module, extra_patchings)
    else:
        out = PatchedModule.from_torch(module, extra_patchings)

    # copy tags
    # TODO: use tag appliers for precision tags
    if hasattr(module, "precision"):
        out.precision = module.precision

    # automagically copy tags from the parent module to the PatchIR child
    copy_tags(module, out)

    return out
