import type { AutoScalingGroup } from "aws-cdk-lib/aws-autoscaling";
import { IConnectable, Instance, LaunchTemplate } from "aws-cdk-lib/aws-ec2";
import { IGrantable } from "aws-cdk-lib/aws-iam";
/**
 * A component involved in the startup process of an EC2 instance.
 */
export interface IStarterAddOn {
    /**
     * Any configuration or customization of the virtual machine takes place here.
     *
     * @param starter - The starter that can be configured.
     * @returns The scripts to include in the user data
     */
    configure(starter: Starter): void;
}
/**
 * Orchestrates the startup process of EC2 instances.
 *
 * A `Starter` is a registry for add-ons. Each add-on can add permissions to the
 * role, network rules to the security group, or scripts to the user data.
 *
 * Scripts are prioritized, so add-ons can be registered out of order but their
 * scripts will appear in the user data in order of priority.
 */
export declare class Starter implements IConnectable, IGrantable {
    #private;
    /**
     * Create a Starter for an auto-scaling group.
     *
     * @param group - The auto-scaling group
     * @returns a new Starter
     */
    static forAutoScalingGroup(group: AutoScalingGroup): Starter;
    /**
     * Create a Starter for a Launch Template.
     *
     * The launch template _must_ have a defined user data property, or this
     * method will throw an error.
     *
     * @param template - The launch template
     * @returns a new Starter
     * @throws Error if the Launch Template user data is undefined
     */
    static forLaunchTemplate(template: LaunchTemplate): Starter;
    /**
     * Create a Starter for a single EC2 instance3
     *
     * @param instance - The instance
     * @returns a new Starter
     */
    static forInstance(instance: Instance): Starter;
    /**
     * @ignore
     */
    private constructor();
    /**
     * @ignore
     */
    private registerUserData;
    get grantPrincipal(): import("aws-cdk-lib/aws-iam").IPrincipal;
    get connections(): import("aws-cdk-lib/aws-ec2").Connections;
    /**
     * All lines of the startup script in priority order.
     */
    get orderedLines(): string[];
    /**
     * Add one or more commands to the user data at a specific priority.
     *
     * @param priority - The priority of these lines (lower executes earlier)
     * @param commands - The lines to add
     * @returns provides a fluent interface
     */
    addScript(priority: number, ...commands: string[]): Starter;
    /**
     * Register add-ons with this Starter.
     *
     * @param addons - The add-ons to register
     * @returns provides a fluent interface
     */
    withAddOns(...addons: IStarterAddOn[]): Starter;
}
