"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const codedeploy = require("../../lib");
const lib_1 = require("../../lib");
function mockFunction(stack, id) {
    return new lambda.Function(stack, id, {
        code: lambda.Code.inline('mock'),
        handler: 'index.handler',
        runtime: lambda.Runtime.NODEJS_8_10
    });
}
function mockAlias(stack) {
    return new lambda.Alias(stack, 'Alias', {
        aliasName: 'my-alias',
        version: new lambda.Version(stack, 'Version', {
            lambda: mockFunction(stack, 'Function')
        })
    });
}
module.exports = {
    "CodeDeploy Lambda DeploymentGroup": {
        "can be created with default AllAtOnce IN_PLACE configuration"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: "MyApp3CE31C26"
                },
                ServiceRoleArn: {
                    "Fn::GetAtt": [
                        "MyDGServiceRole5E94FD88",
                        "Arn"
                    ]
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE"
                    ]
                },
                DeploymentConfigName: "CodeDeployDefault.LambdaAllAtOnce",
                DeploymentStyle: {
                    DeploymentOption: "WITH_TRAFFIC_CONTROL",
                    DeploymentType: "BLUE_GREEN"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
                Type: "AWS::Lambda::Alias",
                Properties: {
                    FunctionName: {
                        Ref: "Function76856677"
                    },
                    FunctionVersion: {
                        "Fn::GetAtt": [
                            "Version6A868472",
                            "Version"
                        ]
                    },
                    Name: "my-alias"
                },
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [{
                            Action: "sts:AssumeRole",
                            Effect: "Allow",
                            Principal: {
                                Service: { "Fn::Join": ["", ["codedeploy.", { Ref: "AWS::Region" }, ".", { Ref: "AWS::URLSuffix" }]] }
                            }
                        }],
                    Version: "2012-10-17"
                },
                ManagedPolicyArns: [
                    {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                { Ref: "AWS::Partition" },
                                ':iam::aws:policy/service-role/AWSCodeDeployRoleForLambda'
                            ]
                        ]
                    }
                ]
            }));
            test.done();
        },
        "can be created with explicit name"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE,
                deploymentGroupName: 'test',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                DeploymentGroupName: "test",
            }));
            test.done();
        },
        "can be created with explicit role"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const serviceRole = new iam.Role(stack, 'MyRole', {
                assumedBy: new iam.ServicePrincipal('not-codedeploy.test')
            });
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE,
                role: serviceRole
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [{
                            Action: "sts:AssumeRole",
                            Effect: "Allow",
                            Principal: {
                                Service: "not-codedeploy.test"
                            }
                        }],
                    Version: "2012-10-17"
                },
                ManagedPolicyArns: [
                    {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                { Ref: "AWS::Partition" },
                                ':iam::aws:policy/service-role/AWSCodeDeployRoleForLambda'
                            ]
                        ]
                    }
                ]
            }));
            test.done();
        },
        "can configure blue/green traffic shifting"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: "MyApp3CE31C26"
                },
                ServiceRoleArn: {
                    "Fn::GetAtt": [
                        "MyDGServiceRole5E94FD88",
                        "Arn"
                    ]
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE"
                    ]
                },
                DeploymentConfigName: "CodeDeployDefault.LambdaLinear10PercentEvery1Minute",
                DeploymentStyle: {
                    DeploymentOption: "WITH_TRAFFIC_CONTROL",
                    DeploymentType: "BLUE_GREEN"
                }
            }));
            test.done();
        },
        "can rollback on alarm"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: codedeploy.LambdaDeploymentConfig.ALL_AT_ONCE,
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                        threshold: 1,
                        evaluationPeriods: 1
                    })]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AlarmConfiguration: {
                    Alarms: [{
                            Name: {
                                Ref: "Failures8A3E1A2F"
                            }
                        }],
                    Enabled: true
                },
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE",
                        "DEPLOYMENT_STOP_ON_ALARM"
                    ]
                },
            }));
            test.done();
        },
        "onPreHook throws error if pre-hook already defined"(test) {
            const stack = new cdk.Stack();
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                alias,
                preHook: mockFunction(stack, 'PreHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            test.throws(() => group.addPreHook(mockFunction(stack, 'PreHook2')));
            test.done();
        },
        "onPostHook throws error if post-hook already defined"(test) {
            const stack = new cdk.Stack();
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            test.throws(() => group.addPostHook(mockFunction(stack, 'PostHook2')));
            test.done();
        },
        "can run pre hook lambda function before deployment"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                preHook: mockFunction(stack, 'PreHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        },
                        BeforeAllowTrafficHook: {
                            Ref: "PreHook8B53F672"
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: "MyDGServiceRoleDefaultPolicy65E8E1EA",
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                "Fn::GetAtt": [
                                    "PreHook8B53F672",
                                    "Arn"
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: "2012-10-17"
                }
            }));
            test.done();
        },
        "can add pre hook lambda function after creating the deployment group"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            group.addPreHook(mockFunction(stack, 'PreHook'));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        },
                        BeforeAllowTrafficHook: {
                            Ref: "PreHook8B53F672"
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: "MyDGServiceRoleDefaultPolicy65E8E1EA",
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                "Fn::GetAtt": [
                                    "PreHook8B53F672",
                                    "Arn"
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: "2012-10-17"
                }
            }));
            test.done();
        },
        "can run post hook lambda function before deployment"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        },
                        AfterAllowTrafficHook: {
                            Ref: 'PostHookF2E49B30'
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: "MyDGServiceRoleDefaultPolicy65E8E1EA",
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                "Fn::GetAtt": [
                                    "PostHookF2E49B30",
                                    "Arn"
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: "2012-10-17"
                },
            }));
            test.done();
        },
        "can add post hook lambda function after creating the deployment group"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            const group = new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE
            });
            group.addPostHook(mockFunction(stack, 'PostHook'));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
                UpdatePolicy: {
                    CodeDeployLambdaAliasUpdate: {
                        ApplicationName: {
                            Ref: "MyApp3CE31C26"
                        },
                        DeploymentGroupName: {
                            Ref: "MyDGC350BD3F"
                        },
                        AfterAllowTrafficHook: {
                            Ref: 'PostHookF2E49B30'
                        }
                    }
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyName: "MyDGServiceRoleDefaultPolicy65E8E1EA",
                Roles: [{
                        Ref: 'MyDGServiceRole5E94FD88'
                    }],
                PolicyDocument: {
                    Statement: [{
                            Action: 'lambda:InvokeFunction',
                            Resource: {
                                "Fn::GetAtt": [
                                    "PostHookF2E49B30",
                                    "Arn"
                                ]
                            },
                            Effect: 'Allow'
                        }],
                    Version: "2012-10-17"
                },
            }));
            test.done();
        },
        "can disable rollback when alarm polling fails"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE,
                ignorePollAlarmsFailure: true,
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                        threshold: 1,
                        evaluationPeriods: 1
                    })]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AlarmConfiguration: {
                    Alarms: [{
                            Name: {
                                Ref: "Failures8A3E1A2F"
                            }
                        }],
                    Enabled: true,
                    IgnorePollAlarmFailure: true
                },
            }));
            test.done();
        },
        "can disable rollback when deployment fails"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE,
                autoRollback: {
                    failedDeployment: false
                }
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CodeDeploy::DeploymentGroup', {
                ApplicationName: {
                    Ref: "MyApp3CE31C26"
                },
                ServiceRoleArn: {
                    "Fn::GetAtt": [
                        "MyDGServiceRole5E94FD88",
                        "Arn"
                    ]
                },
                DeploymentConfigName: "CodeDeployDefault.LambdaAllAtOnce",
                DeploymentStyle: {
                    DeploymentOption: "WITH_TRAFFIC_CONTROL",
                    DeploymentType: "BLUE_GREEN"
                }
            }));
            test.done();
        },
        "can enable rollback when deployment stops"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE,
                autoRollback: {
                    stoppedDeployment: true
                }
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE",
                        "DEPLOYMENT_STOP_ON_REQUEST"
                    ]
                },
            }));
            test.done();
        },
        "can disable rollback when alarm in failure state"(test) {
            const stack = new cdk.Stack();
            const application = new codedeploy.LambdaApplication(stack, 'MyApp');
            const alias = mockAlias(stack);
            new codedeploy.LambdaDeploymentGroup(stack, 'MyDG', {
                application,
                alias,
                postHook: mockFunction(stack, 'PostHook'),
                deploymentConfig: lib_1.LambdaDeploymentConfig.ALL_AT_ONCE,
                autoRollback: {
                    deploymentInAlarm: false
                },
                alarms: [new cloudwatch.Alarm(stack, 'Failures', {
                        metric: alias.metricErrors(),
                        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
                        threshold: 1,
                        evaluationPeriods: 1
                    })]
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodeDeploy::DeploymentGroup', {
                AutoRollbackConfiguration: {
                    Enabled: true,
                    Events: [
                        "DEPLOYMENT_FAILURE",
                    ]
                },
            }));
            test.done();
        },
        'imported with fromLambdaDeploymentGroupAttributes': {
            'defaults the Deployment Config to Canary10Percent5Minutes'(test) {
                const stack = new cdk.Stack();
                const lambdaApp = codedeploy.LambdaApplication.fromLambdaApplicationName(stack, 'LA', 'LambdaApplication');
                const importedGroup = codedeploy.LambdaDeploymentGroup.fromLambdaDeploymentGroupAttributes(stack, 'LDG', {
                    application: lambdaApp,
                    deploymentGroupName: 'LambdaDeploymentGroup',
                });
                test.equal(importedGroup.deploymentConfig, lib_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES);
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5kZXBsb3ltZW50LWdyb3VwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5kZXBsb3ltZW50LWdyb3VwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBdUY7QUFDdkYsc0RBQXVEO0FBQ3ZELHdDQUF5QztBQUN6Qyw4Q0FBK0M7QUFDL0MscUNBQXNDO0FBRXRDLHdDQUF5QztBQUN6QyxtQ0FBbUQ7QUFFbkQsU0FBUyxZQUFZLENBQUMsS0FBZ0IsRUFBRSxFQUFVO0lBQ2hELE9BQU8sSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7UUFDcEMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQztRQUNoQyxPQUFPLEVBQUUsZUFBZTtRQUN4QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO0tBQ3BDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFDRCxTQUFTLFNBQVMsQ0FBQyxLQUFnQjtJQUNqQyxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO1FBQ3RDLFNBQVMsRUFBRSxVQUFVO1FBQ3JCLE9BQU8sRUFBRSxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtZQUM1QyxNQUFNLEVBQUUsWUFBWSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUM7U0FDeEMsQ0FBQztLQUNILENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRCxpQkFBUztJQUNQLG1DQUFtQyxFQUFFO1FBQ25DLDhEQUE4RCxDQUFDLElBQVU7WUFDdkUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsZ0JBQWdCLEVBQUUsNEJBQXNCLENBQUMsV0FBVzthQUNyRCxDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0NBQWtDLEVBQUU7Z0JBQ2hFLGVBQWUsRUFBRTtvQkFDZixHQUFHLEVBQUUsZUFBZTtpQkFDckI7Z0JBQ0QsY0FBYyxFQUFFO29CQUNkLFlBQVksRUFBRTt3QkFDWix5QkFBeUI7d0JBQ3pCLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0QseUJBQXlCLEVBQUU7b0JBQ3pCLE9BQU8sRUFBRSxJQUFJO29CQUNiLE1BQU0sRUFBRTt3QkFDTixvQkFBb0I7cUJBQ3JCO2lCQUNGO2dCQUNELG9CQUFvQixFQUFFLG1DQUFtQztnQkFDekQsZUFBZSxFQUFFO29CQUNmLGdCQUFnQixFQUFFLHNCQUFzQjtvQkFDeEMsY0FBYyxFQUFFLFlBQVk7aUJBQzdCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsb0JBQW9CLEVBQUU7Z0JBQ2xELElBQUksRUFBRSxvQkFBb0I7Z0JBQzFCLFVBQVUsRUFBRTtvQkFDVixZQUFZLEVBQUU7d0JBQ1osR0FBRyxFQUFFLGtCQUFrQjtxQkFDeEI7b0JBQ0QsZUFBZSxFQUFFO3dCQUNmLFlBQVksRUFBRTs0QkFDWixpQkFBaUI7NEJBQ2pCLFNBQVM7eUJBQ1Y7cUJBQ0Y7b0JBQ0QsSUFBSSxFQUFFLFVBQVU7aUJBQ2pCO2dCQUNELFlBQVksRUFBRTtvQkFDWiwyQkFBMkIsRUFBRTt3QkFDM0IsZUFBZSxFQUFFOzRCQUNmLEdBQUcsRUFBRSxlQUFlO3lCQUNyQjt3QkFDRCxtQkFBbUIsRUFBRTs0QkFDbkIsR0FBRyxFQUFFLGNBQWM7eUJBQ3BCO3FCQUNGO2lCQUNGO2FBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUVyQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQzlDLHdCQUF3QixFQUFFO29CQUN4QixTQUFTLEVBQUUsQ0FBQzs0QkFDVixNQUFNLEVBQUUsZ0JBQWdCOzRCQUN4QixNQUFNLEVBQUUsT0FBTzs0QkFDZixTQUFTLEVBQUU7Z0NBQ1QsT0FBTyxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsYUFBYSxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsRUFBRTs2QkFDdkc7eUJBQ0YsQ0FBQztvQkFDRixPQUFPLEVBQUUsWUFBWTtpQkFDdEI7Z0JBQ0QsaUJBQWlCLEVBQUU7b0JBQ2pCO3dCQUNFLFVBQVUsRUFBRTs0QkFDVixFQUFFOzRCQUNGO2dDQUNFLE1BQU07Z0NBQ04sRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7Z0NBQ3pCLDBEQUEwRDs2QkFDM0Q7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxtQ0FBbUMsQ0FBQyxJQUFVO1lBQzVDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsV0FBVztnQkFDWCxLQUFLO2dCQUNMLGdCQUFnQixFQUFFLDRCQUFzQixDQUFDLFdBQVc7Z0JBQ3BELG1CQUFtQixFQUFFLE1BQU07YUFDNUIsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDcEUsbUJBQW1CLEVBQUUsTUFBTTthQUM1QixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxtQ0FBbUMsQ0FBQyxJQUFVO1lBQzVDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ2hELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxxQkFBcUIsQ0FBQzthQUMzRCxDQUFDLENBQUM7WUFFSCxJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsZ0JBQWdCLEVBQUUsNEJBQXNCLENBQUMsV0FBVztnQkFDcEQsSUFBSSxFQUFFLFdBQVc7YUFDbEIsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGdCQUFnQixFQUFFO2dCQUM5Qyx3QkFBd0IsRUFBRTtvQkFDeEIsU0FBUyxFQUFFLENBQUM7NEJBQ1YsTUFBTSxFQUFFLGdCQUFnQjs0QkFDeEIsTUFBTSxFQUFFLE9BQU87NEJBQ2YsU0FBUyxFQUFFO2dDQUNULE9BQU8sRUFBRSxxQkFBcUI7NkJBQy9CO3lCQUNGLENBQUM7b0JBQ0YsT0FBTyxFQUFFLFlBQVk7aUJBQ3RCO2dCQUNELGlCQUFpQixFQUFFO29CQUNqQjt3QkFDRSxVQUFVLEVBQUU7NEJBQ1YsRUFBRTs0QkFDRjtnQ0FDRSxNQUFNO2dDQUNOLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO2dDQUN6QiwwREFBMEQ7NkJBQzNEO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0QsMkNBQTJDLENBQUMsSUFBVTtZQUNwRCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFdBQVcsR0FBRyxJQUFJLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDckUsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQy9CLElBQUksVUFBVSxDQUFDLHFCQUFxQixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ2xELFdBQVc7Z0JBQ1gsS0FBSztnQkFDTCxnQkFBZ0IsRUFBRSw0QkFBc0IsQ0FBQyw4QkFBOEI7YUFDeEUsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtDQUFrQyxFQUFFO2dCQUNoRSxlQUFlLEVBQUU7b0JBQ2YsR0FBRyxFQUFFLGVBQWU7aUJBQ3JCO2dCQUNELGNBQWMsRUFBRTtvQkFDZCxZQUFZLEVBQUU7d0JBQ1oseUJBQXlCO3dCQUN6QixLQUFLO3FCQUNOO2lCQUNGO2dCQUNELHlCQUF5QixFQUFFO29CQUN6QixPQUFPLEVBQUUsSUFBSTtvQkFDYixNQUFNLEVBQUU7d0JBQ04sb0JBQW9CO3FCQUNyQjtpQkFDRjtnQkFDRCxvQkFBb0IsRUFBRSxxREFBcUQ7Z0JBQzNFLGVBQWUsRUFBRTtvQkFDZixnQkFBZ0IsRUFBRSxzQkFBc0I7b0JBQ3hDLGNBQWMsRUFBRSxZQUFZO2lCQUM3QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELHVCQUF1QixDQUFDLElBQVU7WUFDaEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsZ0JBQWdCLEVBQUUsVUFBVSxDQUFDLHNCQUFzQixDQUFDLFdBQVc7Z0JBQy9ELE1BQU0sRUFBRSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO3dCQUMvQyxNQUFNLEVBQUUsS0FBSyxDQUFDLFlBQVksRUFBRTt3QkFDNUIsa0JBQWtCLEVBQUUsVUFBVSxDQUFDLGtCQUFrQixDQUFDLHNCQUFzQjt3QkFDeEUsU0FBUyxFQUFFLENBQUM7d0JBQ1osaUJBQWlCLEVBQUUsQ0FBQztxQkFDckIsQ0FBQyxDQUFDO2FBQ0osQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDcEUsa0JBQWtCLEVBQUU7b0JBQ2xCLE1BQU0sRUFBRSxDQUFDOzRCQUNQLElBQUksRUFBRTtnQ0FDSixHQUFHLEVBQUUsa0JBQWtCOzZCQUN4Qjt5QkFDRixDQUFDO29CQUNGLE9BQU8sRUFBRSxJQUFJO2lCQUNkO2dCQUNELHlCQUF5QixFQUFFO29CQUN6QixPQUFPLEVBQUUsSUFBSTtvQkFDYixNQUFNLEVBQUU7d0JBQ04sb0JBQW9CO3dCQUNwQiwwQkFBMEI7cUJBQzNCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0Qsb0RBQW9ELENBQUMsSUFBVTtZQUM3RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDaEUsS0FBSztnQkFDTCxPQUFPLEVBQUUsWUFBWSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUM7Z0JBQ3ZDLGdCQUFnQixFQUFFLDRCQUFzQixDQUFDLFdBQVc7YUFDckQsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3JFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxzREFBc0QsQ0FBQyxJQUFVO1lBQy9ELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNoRSxLQUFLO2dCQUNMLFFBQVEsRUFBRSxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQztnQkFDekMsZ0JBQWdCLEVBQUUsNEJBQXNCLENBQUMsV0FBVzthQUNyRCxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDdkUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELG9EQUFvRCxDQUFDLElBQVU7WUFDN0QsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsT0FBTyxFQUFFLFlBQVksQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDO2dCQUN2QyxnQkFBZ0IsRUFBRSw0QkFBc0IsQ0FBQyxXQUFXO2FBQ3JELENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0JBQW9CLEVBQUU7Z0JBQ3RELFlBQVksRUFBRTtvQkFDWiwyQkFBMkIsRUFBRTt3QkFDM0IsZUFBZSxFQUFFOzRCQUNmLEdBQUcsRUFBRSxlQUFlO3lCQUNyQjt3QkFDRCxtQkFBbUIsRUFBRTs0QkFDbkIsR0FBRyxFQUFFLGNBQWM7eUJBQ3BCO3dCQUNELHNCQUFzQixFQUFFOzRCQUN0QixHQUFHLEVBQUUsaUJBQWlCO3lCQUN2QjtxQkFDRjtpQkFDRjthQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7WUFFckMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO2dCQUNoRCxVQUFVLEVBQUUsc0NBQXNDO2dCQUNsRCxLQUFLLEVBQUUsQ0FBQzt3QkFDTixHQUFHLEVBQUUseUJBQXlCO3FCQUMvQixDQUFDO2dCQUNGLGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUUsQ0FBQzs0QkFDVixNQUFNLEVBQUUsdUJBQXVCOzRCQUMvQixRQUFRLEVBQUU7Z0NBQ1IsWUFBWSxFQUFFO29DQUNaLGlCQUFpQjtvQ0FDakIsS0FBSztpQ0FDTjs2QkFDRjs0QkFDRCxNQUFNLEVBQUUsT0FBTzt5QkFDaEIsQ0FBQztvQkFDRixPQUFPLEVBQUUsWUFBWTtpQkFDdEI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxzRUFBc0UsQ0FBQyxJQUFVO1lBQy9FLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDaEUsV0FBVztnQkFDWCxLQUFLO2dCQUNMLGdCQUFnQixFQUFFLDRCQUFzQixDQUFDLFdBQVc7YUFDckQsQ0FBQyxDQUFDO1lBQ0gsS0FBSyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDLENBQUM7WUFFakQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxvQkFBb0IsRUFBRTtnQkFDdEQsWUFBWSxFQUFFO29CQUNaLDJCQUEyQixFQUFFO3dCQUMzQixlQUFlLEVBQUU7NEJBQ2YsR0FBRyxFQUFFLGVBQWU7eUJBQ3JCO3dCQUNELG1CQUFtQixFQUFFOzRCQUNuQixHQUFHLEVBQUUsY0FBYzt5QkFDcEI7d0JBQ0Qsc0JBQXNCLEVBQUU7NEJBQ3RCLEdBQUcsRUFBRSxpQkFBaUI7eUJBQ3ZCO3FCQUNGO2lCQUNGO2FBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUVyQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELFVBQVUsRUFBRSxzQ0FBc0M7Z0JBQ2xELEtBQUssRUFBRSxDQUFDO3dCQUNOLEdBQUcsRUFBRSx5QkFBeUI7cUJBQy9CLENBQUM7Z0JBQ0YsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRSxDQUFDOzRCQUNWLE1BQU0sRUFBRSx1QkFBdUI7NEJBQy9CLFFBQVEsRUFBRTtnQ0FDUixZQUFZLEVBQUU7b0NBQ1osaUJBQWlCO29DQUNqQixLQUFLO2lDQUNOOzZCQUNGOzRCQUNELE1BQU0sRUFBRSxPQUFPO3lCQUNoQixDQUFDO29CQUNGLE9BQU8sRUFBRSxZQUFZO2lCQUN0QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELHFEQUFxRCxDQUFDLElBQVU7WUFDOUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsUUFBUSxFQUFFLFlBQVksQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDO2dCQUN6QyxnQkFBZ0IsRUFBRSw0QkFBc0IsQ0FBQyxXQUFXO2FBQ3JELENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0JBQW9CLEVBQUU7Z0JBQ3RELFlBQVksRUFBRTtvQkFDWiwyQkFBMkIsRUFBRTt3QkFDM0IsZUFBZSxFQUFFOzRCQUNmLEdBQUcsRUFBRSxlQUFlO3lCQUNyQjt3QkFDRCxtQkFBbUIsRUFBRTs0QkFDbkIsR0FBRyxFQUFFLGNBQWM7eUJBQ3BCO3dCQUNELHFCQUFxQixFQUFFOzRCQUNyQixHQUFHLEVBQUUsa0JBQWtCO3lCQUN4QjtxQkFDRjtpQkFDRjthQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7WUFFckMsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO2dCQUNoRCxVQUFVLEVBQUUsc0NBQXNDO2dCQUNsRCxLQUFLLEVBQUUsQ0FBQzt3QkFDTixHQUFHLEVBQUUseUJBQXlCO3FCQUMvQixDQUFDO2dCQUNGLGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUUsQ0FBQzs0QkFDVixNQUFNLEVBQUUsdUJBQXVCOzRCQUMvQixRQUFRLEVBQUU7Z0NBQ1IsWUFBWSxFQUFFO29DQUNaLGtCQUFrQjtvQ0FDbEIsS0FBSztpQ0FDTjs2QkFDRjs0QkFDRCxNQUFNLEVBQUUsT0FBTzt5QkFDaEIsQ0FBQztvQkFDRixPQUFPLEVBQUUsWUFBWTtpQkFDdEI7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCx1RUFBdUUsQ0FBQyxJQUFVO1lBQ2hGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsTUFBTSxLQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDaEUsV0FBVztnQkFDWCxLQUFLO2dCQUNMLGdCQUFnQixFQUFFLDRCQUFzQixDQUFDLFdBQVc7YUFDckQsQ0FBQyxDQUFDO1lBQ0gsS0FBSyxDQUFDLFdBQVcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFFbkQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxvQkFBb0IsRUFBRTtnQkFDdEQsWUFBWSxFQUFFO29CQUNaLDJCQUEyQixFQUFFO3dCQUMzQixlQUFlLEVBQUU7NEJBQ2YsR0FBRyxFQUFFLGVBQWU7eUJBQ3JCO3dCQUNELG1CQUFtQixFQUFFOzRCQUNuQixHQUFHLEVBQUUsY0FBYzt5QkFDcEI7d0JBQ0QscUJBQXFCLEVBQUU7NEJBQ3JCLEdBQUcsRUFBRSxrQkFBa0I7eUJBQ3hCO3FCQUNGO2lCQUNGO2FBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUVyQyxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELFVBQVUsRUFBRSxzQ0FBc0M7Z0JBQ2xELEtBQUssRUFBRSxDQUFDO3dCQUNOLEdBQUcsRUFBRSx5QkFBeUI7cUJBQy9CLENBQUM7Z0JBQ0YsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRSxDQUFDOzRCQUNWLE1BQU0sRUFBRSx1QkFBdUI7NEJBQy9CLFFBQVEsRUFBRTtnQ0FDUixZQUFZLEVBQUU7b0NBQ1osa0JBQWtCO29DQUNsQixLQUFLO2lDQUNOOzZCQUNGOzRCQUNELE1BQU0sRUFBRSxPQUFPO3lCQUNoQixDQUFDO29CQUNGLE9BQU8sRUFBRSxZQUFZO2lCQUN0QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELCtDQUErQyxDQUFDLElBQVU7WUFDeEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsUUFBUSxFQUFFLFlBQVksQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDO2dCQUN6QyxnQkFBZ0IsRUFBRSw0QkFBc0IsQ0FBQyxXQUFXO2dCQUNwRCx1QkFBdUIsRUFBRSxJQUFJO2dCQUM3QixNQUFNLEVBQUUsQ0FBQyxJQUFJLFVBQVUsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTt3QkFDL0MsTUFBTSxFQUFFLEtBQUssQ0FBQyxZQUFZLEVBQUU7d0JBQzVCLGtCQUFrQixFQUFFLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxzQkFBc0I7d0JBQ3hFLFNBQVMsRUFBRSxDQUFDO3dCQUNaLGlCQUFpQixFQUFFLENBQUM7cUJBQ3JCLENBQUMsQ0FBQzthQUNKLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsa0NBQWtDLEVBQUU7Z0JBQ3BFLGtCQUFrQixFQUFFO29CQUNsQixNQUFNLEVBQUUsQ0FBQzs0QkFDUCxJQUFJLEVBQUU7Z0NBQ0osR0FBRyxFQUFFLGtCQUFrQjs2QkFDeEI7eUJBQ0YsQ0FBQztvQkFDRixPQUFPLEVBQUUsSUFBSTtvQkFDYixzQkFBc0IsRUFBRSxJQUFJO2lCQUM3QjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELDRDQUE0QyxDQUFDLElBQVU7WUFDckQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxXQUFXLEdBQUcsSUFBSSxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMvQixJQUFJLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNsRCxXQUFXO2dCQUNYLEtBQUs7Z0JBQ0wsUUFBUSxFQUFFLFlBQVksQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDO2dCQUN6QyxnQkFBZ0IsRUFBRSw0QkFBc0IsQ0FBQyxXQUFXO2dCQUNwRCxZQUFZLEVBQUU7b0JBQ1osZ0JBQWdCLEVBQUUsS0FBSztpQkFDeEI7YUFDRixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0NBQWtDLEVBQUU7Z0JBQ2hFLGVBQWUsRUFBRTtvQkFDZixHQUFHLEVBQUUsZUFBZTtpQkFDckI7Z0JBQ0QsY0FBYyxFQUFFO29CQUNkLFlBQVksRUFBRTt3QkFDWix5QkFBeUI7d0JBQ3pCLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0Qsb0JBQW9CLEVBQUUsbUNBQW1DO2dCQUN6RCxlQUFlLEVBQUU7b0JBQ2YsZ0JBQWdCLEVBQUUsc0JBQXNCO29CQUN4QyxjQUFjLEVBQUUsWUFBWTtpQkFDN0I7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCwyQ0FBMkMsQ0FBQyxJQUFVO1lBQ3BELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsV0FBVztnQkFDWCxLQUFLO2dCQUNMLFFBQVEsRUFBRSxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQztnQkFDekMsZ0JBQWdCLEVBQUUsNEJBQXNCLENBQUMsV0FBVztnQkFDcEQsWUFBWSxFQUFFO29CQUNaLGlCQUFpQixFQUFFLElBQUk7aUJBQ3hCO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDcEUseUJBQXlCLEVBQUU7b0JBQ3pCLE9BQU8sRUFBRSxJQUFJO29CQUNiLE1BQU0sRUFBRTt3QkFDTixvQkFBb0I7d0JBQ3BCLDRCQUE0QjtxQkFDN0I7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFDRCxrREFBa0QsQ0FBQyxJQUFVO1lBQzNELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sV0FBVyxHQUFHLElBQUksVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyRSxNQUFNLEtBQUssR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDL0IsSUFBSSxVQUFVLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDbEQsV0FBVztnQkFDWCxLQUFLO2dCQUNMLFFBQVEsRUFBRSxZQUFZLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQztnQkFDekMsZ0JBQWdCLEVBQUUsNEJBQXNCLENBQUMsV0FBVztnQkFDcEQsWUFBWSxFQUFFO29CQUNaLGlCQUFpQixFQUFFLEtBQUs7aUJBQ3pCO2dCQUNELE1BQU0sRUFBRSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO3dCQUMvQyxNQUFNLEVBQUUsS0FBSyxDQUFDLFlBQVksRUFBRTt3QkFDNUIsa0JBQWtCLEVBQUUsVUFBVSxDQUFDLGtCQUFrQixDQUFDLHNCQUFzQjt3QkFDeEUsU0FBUyxFQUFFLENBQUM7d0JBQ1osaUJBQWlCLEVBQUUsQ0FBQztxQkFDckIsQ0FBQyxDQUFDO2FBQ0osQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDcEUseUJBQXlCLEVBQUU7b0JBQ3pCLE9BQU8sRUFBRSxJQUFJO29CQUNiLE1BQU0sRUFBRTt3QkFDTixvQkFBb0I7cUJBQ3JCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsbURBQW1ELEVBQUU7WUFDbkQsMkRBQTJELENBQUMsSUFBVTtnQkFDcEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBRTlCLE1BQU0sU0FBUyxHQUFHLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLG1CQUFtQixDQUFDLENBQUM7Z0JBQzNHLE1BQU0sYUFBYSxHQUFHLFVBQVUsQ0FBQyxxQkFBcUIsQ0FBQyxtQ0FBbUMsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO29CQUN2RyxXQUFXLEVBQUUsU0FBUztvQkFDdEIsbUJBQW1CLEVBQUUsdUJBQXVCO2lCQUM3QyxDQUFDLENBQUM7Z0JBRUgsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLEVBQUUsNEJBQXNCLENBQUMseUJBQXlCLENBQUMsQ0FBQztnQkFFN0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztTQUNGO0tBQ0Y7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhwZWN0LCBoYXZlUmVzb3VyY2UsIGhhdmVSZXNvdXJjZUxpa2UsIFJlc291cmNlUGFydCB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgY2xvdWR3YXRjaCA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJyk7XG5pbXBvcnQgaWFtID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWlhbScpO1xuaW1wb3J0IGxhbWJkYSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnKTtcbmltcG9ydCBjZGsgPSByZXF1aXJlKCdAYXdzLWNkay9jb3JlJyk7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IGNvZGVkZXBsb3kgPSByZXF1aXJlKCcuLi8uLi9saWInKTtcbmltcG9ydCB7IExhbWJkYURlcGxveW1lbnRDb25maWcgfSBmcm9tICcuLi8uLi9saWInO1xuXG5mdW5jdGlvbiBtb2NrRnVuY3Rpb24oc3RhY2s6IGNkay5TdGFjaywgaWQ6IHN0cmluZykge1xuICByZXR1cm4gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgaWQsIHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5pbmxpbmUoJ21vY2snKSxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzhfMTBcbiAgfSk7XG59XG5mdW5jdGlvbiBtb2NrQWxpYXMoc3RhY2s6IGNkay5TdGFjaykge1xuICByZXR1cm4gbmV3IGxhbWJkYS5BbGlhcyhzdGFjaywgJ0FsaWFzJywge1xuICAgIGFsaWFzTmFtZTogJ215LWFsaWFzJyxcbiAgICB2ZXJzaW9uOiBuZXcgbGFtYmRhLlZlcnNpb24oc3RhY2ssICdWZXJzaW9uJywge1xuICAgICAgbGFtYmRhOiBtb2NrRnVuY3Rpb24oc3RhY2ssICdGdW5jdGlvbicpXG4gICAgfSlcbiAgfSk7XG59XG5cbmV4cG9ydCA9IHtcbiAgXCJDb2RlRGVwbG95IExhbWJkYSBEZXBsb3ltZW50R3JvdXBcIjoge1xuICAgIFwiY2FuIGJlIGNyZWF0ZWQgd2l0aCBkZWZhdWx0IEFsbEF0T25jZSBJTl9QTEFDRSBjb25maWd1cmF0aW9uXCIodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRVxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgQXBwbGljYXRpb25OYW1lOiB7XG4gICAgICAgICAgUmVmOiBcIk15QXBwM0NFMzFDMjZcIlxuICAgICAgICB9LFxuICAgICAgICBTZXJ2aWNlUm9sZUFybjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIk15REdTZXJ2aWNlUm9sZTVFOTRGRDg4XCIsXG4gICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBBdXRvUm9sbGJhY2tDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICBFdmVudHM6IFtcbiAgICAgICAgICAgIFwiREVQTE9ZTUVOVF9GQUlMVVJFXCJcbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIERlcGxveW1lbnRDb25maWdOYW1lOiBcIkNvZGVEZXBsb3lEZWZhdWx0LkxhbWJkYUFsbEF0T25jZVwiLFxuICAgICAgICBEZXBsb3ltZW50U3R5bGU6IHtcbiAgICAgICAgICBEZXBsb3ltZW50T3B0aW9uOiBcIldJVEhfVFJBRkZJQ19DT05UUk9MXCIsXG4gICAgICAgICAgRGVwbG95bWVudFR5cGU6IFwiQkxVRV9HUkVFTlwiXG4gICAgICAgIH1cbiAgICAgIH0pKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpBbGlhcycsIHtcbiAgICAgICAgVHlwZTogXCJBV1M6OkxhbWJkYTo6QWxpYXNcIixcbiAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgIEZ1bmN0aW9uTmFtZToge1xuICAgICAgICAgICAgUmVmOiBcIkZ1bmN0aW9uNzY4NTY2NzdcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgRnVuY3Rpb25WZXJzaW9uOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcIlZlcnNpb242QTg2ODQ3MlwiLFxuICAgICAgICAgICAgICBcIlZlcnNpb25cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgTmFtZTogXCJteS1hbGlhc1wiXG4gICAgICAgIH0sXG4gICAgICAgIFVwZGF0ZVBvbGljeToge1xuICAgICAgICAgIENvZGVEZXBsb3lMYW1iZGFBbGlhc1VwZGF0ZToge1xuICAgICAgICAgICAgQXBwbGljYXRpb25OYW1lOiB7XG4gICAgICAgICAgICAgIFJlZjogXCJNeUFwcDNDRTMxQzI2XCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBEZXBsb3ltZW50R3JvdXBOYW1lOiB7XG4gICAgICAgICAgICAgIFJlZjogXCJNeURHQzM1MEJEM0ZcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFN0YXRlbWVudDogW3tcbiAgICAgICAgICAgIEFjdGlvbjogXCJzdHM6QXNzdW1lUm9sZVwiLFxuICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgU2VydmljZTogeyBcIkZuOjpKb2luXCI6IFtcIlwiLCBbXCJjb2RlZGVwbG95LlwiLCB7IFJlZjogXCJBV1M6OlJlZ2lvblwiIH0sIFwiLlwiLCB7IFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiIH1dXSB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfV0sXG4gICAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgfSxcbiAgICAgICAgTWFuYWdlZFBvbGljeUFybnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgIFwiYXJuOlwiLFxuICAgICAgICAgICAgICAgIHsgUmVmOiBcIkFXUzo6UGFydGl0aW9uXCIgfSxcbiAgICAgICAgICAgICAgICAnOmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTQ29kZURlcGxveVJvbGVGb3JMYW1iZGEnXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICBcImNhbiBiZSBjcmVhdGVkIHdpdGggZXhwbGljaXQgbmFtZVwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBuZXcgY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdNeURHJywge1xuICAgICAgICBhcHBsaWNhdGlvbixcbiAgICAgICAgYWxpYXMsXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IExhbWJkYURlcGxveW1lbnRDb25maWcuQUxMX0FUX09OQ0UsXG4gICAgICAgIGRlcGxveW1lbnRHcm91cE5hbWU6ICd0ZXN0JyxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZURlcGxveTo6RGVwbG95bWVudEdyb3VwJywge1xuICAgICAgICBEZXBsb3ltZW50R3JvdXBOYW1lOiBcInRlc3RcIixcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICBcImNhbiBiZSBjcmVhdGVkIHdpdGggZXhwbGljaXQgcm9sZVwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBjb25zdCBzZXJ2aWNlUm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ015Um9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ25vdC1jb2RlZGVwbG95LnRlc3QnKVxuICAgICAgfSk7XG5cbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgICAgcm9sZTogc2VydmljZVJvbGVcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFN0YXRlbWVudDogW3tcbiAgICAgICAgICAgIEFjdGlvbjogXCJzdHM6QXNzdW1lUm9sZVwiLFxuICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgU2VydmljZTogXCJub3QtY29kZWRlcGxveS50ZXN0XCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XSxcbiAgICAgICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICB9LFxuICAgICAgICBNYW5hZ2VkUG9saWN5QXJuczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgICAgeyBSZWY6IFwiQVdTOjpQYXJ0aXRpb25cIiB9LFxuICAgICAgICAgICAgICAgICc6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NDb2RlRGVwbG95Um9sZUZvckxhbWJkYSdcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgXVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgIFwiY2FuIGNvbmZpZ3VyZSBibHVlL2dyZWVuIHRyYWZmaWMgc2hpZnRpbmdcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGFwcGxpY2F0aW9uID0gbmV3IGNvZGVkZXBsb3kuTGFtYmRhQXBwbGljYXRpb24oc3RhY2ssICdNeUFwcCcpO1xuICAgICAgY29uc3QgYWxpYXMgPSBtb2NrQWxpYXMoc3RhY2spO1xuICAgICAgbmV3IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnTXlERycsIHtcbiAgICAgICAgYXBwbGljYXRpb24sXG4gICAgICAgIGFsaWFzLFxuICAgICAgICBkZXBsb3ltZW50Q29uZmlnOiBMYW1iZGFEZXBsb3ltZW50Q29uZmlnLkxJTkVBUl8xMFBFUkNFTlRfRVZFUllfMU1JTlVURVxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgQXBwbGljYXRpb25OYW1lOiB7XG4gICAgICAgICAgUmVmOiBcIk15QXBwM0NFMzFDMjZcIlxuICAgICAgICB9LFxuICAgICAgICBTZXJ2aWNlUm9sZUFybjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIk15REdTZXJ2aWNlUm9sZTVFOTRGRDg4XCIsXG4gICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgICBBdXRvUm9sbGJhY2tDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICBFdmVudHM6IFtcbiAgICAgICAgICAgIFwiREVQTE9ZTUVOVF9GQUlMVVJFXCJcbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIERlcGxveW1lbnRDb25maWdOYW1lOiBcIkNvZGVEZXBsb3lEZWZhdWx0LkxhbWJkYUxpbmVhcjEwUGVyY2VudEV2ZXJ5MU1pbnV0ZVwiLFxuICAgICAgICBEZXBsb3ltZW50U3R5bGU6IHtcbiAgICAgICAgICBEZXBsb3ltZW50T3B0aW9uOiBcIldJVEhfVFJBRkZJQ19DT05UUk9MXCIsXG4gICAgICAgICAgRGVwbG95bWVudFR5cGU6IFwiQkxVRV9HUkVFTlwiXG4gICAgICAgIH1cbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICBcImNhbiByb2xsYmFjayBvbiBhbGFybVwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBuZXcgY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdNeURHJywge1xuICAgICAgICBhcHBsaWNhdGlvbixcbiAgICAgICAgYWxpYXMsXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgICAgYWxhcm1zOiBbbmV3IGNsb3Vkd2F0Y2guQWxhcm0oc3RhY2ssICdGYWlsdXJlcycsIHtcbiAgICAgICAgICBtZXRyaWM6IGFsaWFzLm1ldHJpY0Vycm9ycygpLFxuICAgICAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX1RIUkVTSE9MRCxcbiAgICAgICAgICB0aHJlc2hvbGQ6IDEsXG4gICAgICAgICAgZXZhbHVhdGlvblBlcmlvZHM6IDFcbiAgICAgICAgfSldXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgQWxhcm1Db25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgQWxhcm1zOiBbe1xuICAgICAgICAgICAgTmFtZToge1xuICAgICAgICAgICAgICBSZWY6IFwiRmFpbHVyZXM4QTNFMUEyRlwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfV0sXG4gICAgICAgICAgRW5hYmxlZDogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBBdXRvUm9sbGJhY2tDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICBFdmVudHM6IFtcbiAgICAgICAgICAgIFwiREVQTE9ZTUVOVF9GQUlMVVJFXCIsXG4gICAgICAgICAgICBcIkRFUExPWU1FTlRfU1RPUF9PTl9BTEFSTVwiXG4gICAgICAgICAgXVxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgIFwib25QcmVIb29rIHRocm93cyBlcnJvciBpZiBwcmUtaG9vayBhbHJlYWR5IGRlZmluZWRcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIGNvbnN0IGdyb3VwID0gbmV3IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnTXlERycsIHtcbiAgICAgICAgYWxpYXMsXG4gICAgICAgIHByZUhvb2s6IG1vY2tGdW5jdGlvbihzdGFjaywgJ1ByZUhvb2snKSxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRVxuICAgICAgfSk7XG4gICAgICB0ZXN0LnRocm93cygoKSA9PiBncm91cC5hZGRQcmVIb29rKG1vY2tGdW5jdGlvbihzdGFjaywgJ1ByZUhvb2syJykpKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgXCJvblBvc3RIb29rIHRocm93cyBlcnJvciBpZiBwb3N0LWhvb2sgYWxyZWFkeSBkZWZpbmVkXCIodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBjb25zdCBncm91cCA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFsaWFzLFxuICAgICAgICBwb3N0SG9vazogbW9ja0Z1bmN0aW9uKHN0YWNrLCAnUG9zdEhvb2snKSxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRVxuICAgICAgfSk7XG4gICAgICB0ZXN0LnRocm93cygoKSA9PiBncm91cC5hZGRQb3N0SG9vayhtb2NrRnVuY3Rpb24oc3RhY2ssICdQb3N0SG9vazInKSkpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICBcImNhbiBydW4gcHJlIGhvb2sgbGFtYmRhIGZ1bmN0aW9uIGJlZm9yZSBkZXBsb3ltZW50XCIodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgcHJlSG9vazogbW9ja0Z1bmN0aW9uKHN0YWNrLCAnUHJlSG9vaycpLFxuICAgICAgICBkZXBsb3ltZW50Q29uZmlnOiBMYW1iZGFEZXBsb3ltZW50Q29uZmlnLkFMTF9BVF9PTkNFXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkxhbWJkYTo6QWxpYXMnLCB7XG4gICAgICAgIFVwZGF0ZVBvbGljeToge1xuICAgICAgICAgIENvZGVEZXBsb3lMYW1iZGFBbGlhc1VwZGF0ZToge1xuICAgICAgICAgICAgQXBwbGljYXRpb25OYW1lOiB7XG4gICAgICAgICAgICAgIFJlZjogXCJNeUFwcDNDRTMxQzI2XCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBEZXBsb3ltZW50R3JvdXBOYW1lOiB7XG4gICAgICAgICAgICAgIFJlZjogXCJNeURHQzM1MEJEM0ZcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIEJlZm9yZUFsbG93VHJhZmZpY0hvb2s6IHtcbiAgICAgICAgICAgICAgUmVmOiBcIlByZUhvb2s4QjUzRjY3MlwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgICBQb2xpY3lOYW1lOiBcIk15REdTZXJ2aWNlUm9sZURlZmF1bHRQb2xpY3k2NUU4RTFFQVwiLFxuICAgICAgICBSb2xlczogW3tcbiAgICAgICAgICBSZWY6ICdNeURHU2VydmljZVJvbGU1RTk0RkQ4OCdcbiAgICAgICAgfV0sXG4gICAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgICAgU3RhdGVtZW50OiBbe1xuICAgICAgICAgICAgQWN0aW9uOiAnbGFtYmRhOkludm9rZUZ1bmN0aW9uJyxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgXCJQcmVIb29rOEI1M0Y2NzJcIixcbiAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdydcbiAgICAgICAgICB9XSxcbiAgICAgICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICB9XG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgXCJjYW4gYWRkIHByZSBob29rIGxhbWJkYSBmdW5jdGlvbiBhZnRlciBjcmVhdGluZyB0aGUgZGVwbG95bWVudCBncm91cFwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBjb25zdCBncm91cCA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRVxuICAgICAgfSk7XG4gICAgICBncm91cC5hZGRQcmVIb29rKG1vY2tGdW5jdGlvbihzdGFjaywgJ1ByZUhvb2snKSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpMYW1iZGE6OkFsaWFzJywge1xuICAgICAgICBVcGRhdGVQb2xpY3k6IHtcbiAgICAgICAgICBDb2RlRGVwbG95TGFtYmRhQWxpYXNVcGRhdGU6IHtcbiAgICAgICAgICAgIEFwcGxpY2F0aW9uTmFtZToge1xuICAgICAgICAgICAgICBSZWY6IFwiTXlBcHAzQ0UzMUMyNlwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgRGVwbG95bWVudEdyb3VwTmFtZToge1xuICAgICAgICAgICAgICBSZWY6IFwiTXlER0MzNTBCRDNGXCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBCZWZvcmVBbGxvd1RyYWZmaWNIb29rOiB7XG4gICAgICAgICAgICAgIFJlZjogXCJQcmVIb29rOEI1M0Y2NzJcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgUG9saWN5TmFtZTogXCJNeURHU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NjVFOEUxRUFcIixcbiAgICAgICAgUm9sZXM6IFt7XG4gICAgICAgICAgUmVmOiAnTXlER1NlcnZpY2VSb2xlNUU5NEZEODgnXG4gICAgICAgIH1dLFxuICAgICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFN0YXRlbWVudDogW3tcbiAgICAgICAgICAgIEFjdGlvbjogJ2xhbWJkYTpJbnZva2VGdW5jdGlvbicsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgIFwiUHJlSG9vazhCNTNGNjcyXCIsXG4gICAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnXG4gICAgICAgICAgfV0sXG4gICAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgfVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgIFwiY2FuIHJ1biBwb3N0IGhvb2sgbGFtYmRhIGZ1bmN0aW9uIGJlZm9yZSBkZXBsb3ltZW50XCIodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgcG9zdEhvb2s6IG1vY2tGdW5jdGlvbihzdGFjaywgJ1Bvc3RIb29rJyksXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IExhbWJkYURlcGxveW1lbnRDb25maWcuQUxMX0FUX09OQ0VcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6TGFtYmRhOjpBbGlhcycsIHtcbiAgICAgICAgVXBkYXRlUG9saWN5OiB7XG4gICAgICAgICAgQ29kZURlcGxveUxhbWJkYUFsaWFzVXBkYXRlOiB7XG4gICAgICAgICAgICBBcHBsaWNhdGlvbk5hbWU6IHtcbiAgICAgICAgICAgICAgUmVmOiBcIk15QXBwM0NFMzFDMjZcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIERlcGxveW1lbnRHcm91cE5hbWU6IHtcbiAgICAgICAgICAgICAgUmVmOiBcIk15REdDMzUwQkQzRlwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgQWZ0ZXJBbGxvd1RyYWZmaWNIb29rOiB7XG4gICAgICAgICAgICAgIFJlZjogJ1Bvc3RIb29rRjJFNDlCMzAnXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgICBQb2xpY3lOYW1lOiBcIk15REdTZXJ2aWNlUm9sZURlZmF1bHRQb2xpY3k2NUU4RTFFQVwiLFxuICAgICAgICBSb2xlczogW3tcbiAgICAgICAgICBSZWY6ICdNeURHU2VydmljZVJvbGU1RTk0RkQ4OCdcbiAgICAgICAgfV0sXG4gICAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgICAgU3RhdGVtZW50OiBbe1xuICAgICAgICAgICAgQWN0aW9uOiAnbGFtYmRhOkludm9rZUZ1bmN0aW9uJyxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgXCJQb3N0SG9va0YyRTQ5QjMwXCIsXG4gICAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnXG4gICAgICAgICAgfV0sXG4gICAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICBcImNhbiBhZGQgcG9zdCBob29rIGxhbWJkYSBmdW5jdGlvbiBhZnRlciBjcmVhdGluZyB0aGUgZGVwbG95bWVudCBncm91cFwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBjb25zdCBncm91cCA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRVxuICAgICAgfSk7XG4gICAgICBncm91cC5hZGRQb3N0SG9vayhtb2NrRnVuY3Rpb24oc3RhY2ssICdQb3N0SG9vaycpKTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkxhbWJkYTo6QWxpYXMnLCB7XG4gICAgICAgIFVwZGF0ZVBvbGljeToge1xuICAgICAgICAgIENvZGVEZXBsb3lMYW1iZGFBbGlhc1VwZGF0ZToge1xuICAgICAgICAgICAgQXBwbGljYXRpb25OYW1lOiB7XG4gICAgICAgICAgICAgIFJlZjogXCJNeUFwcDNDRTMxQzI2XCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBEZXBsb3ltZW50R3JvdXBOYW1lOiB7XG4gICAgICAgICAgICAgIFJlZjogXCJNeURHQzM1MEJEM0ZcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIEFmdGVyQWxsb3dUcmFmZmljSG9vazoge1xuICAgICAgICAgICAgICBSZWY6ICdQb3N0SG9va0YyRTQ5QjMwJ1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgICAgUG9saWN5TmFtZTogXCJNeURHU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NjVFOEUxRUFcIixcbiAgICAgICAgUm9sZXM6IFt7XG4gICAgICAgICAgUmVmOiAnTXlER1NlcnZpY2VSb2xlNUU5NEZEODgnXG4gICAgICAgIH1dLFxuICAgICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFN0YXRlbWVudDogW3tcbiAgICAgICAgICAgIEFjdGlvbjogJ2xhbWJkYTpJbnZva2VGdW5jdGlvbicsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgIFwiUG9zdEhvb2tGMkU0OUIzMFwiLFxuICAgICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93J1xuICAgICAgICAgIH1dLFxuICAgICAgICAgIFZlcnNpb246IFwiMjAxMi0xMC0xN1wiXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gICAgXCJjYW4gZGlzYWJsZSByb2xsYmFjayB3aGVuIGFsYXJtIHBvbGxpbmcgZmFpbHNcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGFwcGxpY2F0aW9uID0gbmV3IGNvZGVkZXBsb3kuTGFtYmRhQXBwbGljYXRpb24oc3RhY2ssICdNeUFwcCcpO1xuICAgICAgY29uc3QgYWxpYXMgPSBtb2NrQWxpYXMoc3RhY2spO1xuICAgICAgbmV3IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnTXlERycsIHtcbiAgICAgICAgYXBwbGljYXRpb24sXG4gICAgICAgIGFsaWFzLFxuICAgICAgICBwb3N0SG9vazogbW9ja0Z1bmN0aW9uKHN0YWNrLCAnUG9zdEhvb2snKSxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgICAgaWdub3JlUG9sbEFsYXJtc0ZhaWx1cmU6IHRydWUsXG4gICAgICAgIGFsYXJtczogW25ldyBjbG91ZHdhdGNoLkFsYXJtKHN0YWNrLCAnRmFpbHVyZXMnLCB7XG4gICAgICAgICAgbWV0cmljOiBhbGlhcy5tZXRyaWNFcnJvcnMoKSxcbiAgICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9USFJFU0hPTEQsXG4gICAgICAgICAgdGhyZXNob2xkOiAxLFxuICAgICAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxXG4gICAgICAgIH0pXVxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIEFsYXJtQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIEFsYXJtczogW3tcbiAgICAgICAgICAgIE5hbWU6IHtcbiAgICAgICAgICAgICAgUmVmOiBcIkZhaWx1cmVzOEEzRTFBMkZcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1dLFxuICAgICAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgSWdub3JlUG9sbEFsYXJtRmFpbHVyZTogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgIFwiY2FuIGRpc2FibGUgcm9sbGJhY2sgd2hlbiBkZXBsb3ltZW50IGZhaWxzXCIodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uKHN0YWNrLCAnTXlBcHAnKTtcbiAgICAgIGNvbnN0IGFsaWFzID0gbW9ja0FsaWFzKHN0YWNrKTtcbiAgICAgIG5ldyBjb2RlZGVwbG95LkxhbWJkYURlcGxveW1lbnRHcm91cChzdGFjaywgJ015REcnLCB7XG4gICAgICAgIGFwcGxpY2F0aW9uLFxuICAgICAgICBhbGlhcyxcbiAgICAgICAgcG9zdEhvb2s6IG1vY2tGdW5jdGlvbihzdGFjaywgJ1Bvc3RIb29rJyksXG4gICAgICAgIGRlcGxveW1lbnRDb25maWc6IExhbWJkYURlcGxveW1lbnRDb25maWcuQUxMX0FUX09OQ0UsXG4gICAgICAgIGF1dG9Sb2xsYmFjazoge1xuICAgICAgICAgIGZhaWxlZERlcGxveW1lbnQ6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIEFwcGxpY2F0aW9uTmFtZToge1xuICAgICAgICAgIFJlZjogXCJNeUFwcDNDRTMxQzI2XCJcbiAgICAgICAgfSxcbiAgICAgICAgU2VydmljZVJvbGVBcm46IHtcbiAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgXCJNeURHU2VydmljZVJvbGU1RTk0RkQ4OFwiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgIF1cbiAgICAgICAgfSxcbiAgICAgICAgRGVwbG95bWVudENvbmZpZ05hbWU6IFwiQ29kZURlcGxveURlZmF1bHQuTGFtYmRhQWxsQXRPbmNlXCIsXG4gICAgICAgIERlcGxveW1lbnRTdHlsZToge1xuICAgICAgICAgIERlcGxveW1lbnRPcHRpb246IFwiV0lUSF9UUkFGRklDX0NPTlRST0xcIixcbiAgICAgICAgICBEZXBsb3ltZW50VHlwZTogXCJCTFVFX0dSRUVOXCJcbiAgICAgICAgfVxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgIFwiY2FuIGVuYWJsZSByb2xsYmFjayB3aGVuIGRlcGxveW1lbnQgc3RvcHNcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGFwcGxpY2F0aW9uID0gbmV3IGNvZGVkZXBsb3kuTGFtYmRhQXBwbGljYXRpb24oc3RhY2ssICdNeUFwcCcpO1xuICAgICAgY29uc3QgYWxpYXMgPSBtb2NrQWxpYXMoc3RhY2spO1xuICAgICAgbmV3IGNvZGVkZXBsb3kuTGFtYmRhRGVwbG95bWVudEdyb3VwKHN0YWNrLCAnTXlERycsIHtcbiAgICAgICAgYXBwbGljYXRpb24sXG4gICAgICAgIGFsaWFzLFxuICAgICAgICBwb3N0SG9vazogbW9ja0Z1bmN0aW9uKHN0YWNrLCAnUG9zdEhvb2snKSxcbiAgICAgICAgZGVwbG95bWVudENvbmZpZzogTGFtYmRhRGVwbG95bWVudENvbmZpZy5BTExfQVRfT05DRSxcbiAgICAgICAgYXV0b1JvbGxiYWNrOiB7XG4gICAgICAgICAgc3RvcHBlZERlcGxveW1lbnQ6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlRGVwbG95OjpEZXBsb3ltZW50R3JvdXAnLCB7XG4gICAgICAgIEF1dG9Sb2xsYmFja0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgICAgIEV2ZW50czogW1xuICAgICAgICAgICAgXCJERVBMT1lNRU5UX0ZBSUxVUkVcIixcbiAgICAgICAgICAgIFwiREVQTE9ZTUVOVF9TVE9QX09OX1JFUVVFU1RcIlxuICAgICAgICAgIF1cbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgICBcImNhbiBkaXNhYmxlIHJvbGxiYWNrIHdoZW4gYWxhcm0gaW4gZmFpbHVyZSBzdGF0ZVwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYXBwbGljYXRpb24gPSBuZXcgY29kZWRlcGxveS5MYW1iZGFBcHBsaWNhdGlvbihzdGFjaywgJ015QXBwJyk7XG4gICAgICBjb25zdCBhbGlhcyA9IG1vY2tBbGlhcyhzdGFjayk7XG4gICAgICBuZXcgY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50R3JvdXAoc3RhY2ssICdNeURHJywge1xuICAgICAgICBhcHBsaWNhdGlvbixcbiAgICAgICAgYWxpYXMsXG4gICAgICAgIHBvc3RIb29rOiBtb2NrRnVuY3Rpb24oc3RhY2ssICdQb3N0SG9vaycpLFxuICAgICAgICBkZXBsb3ltZW50Q29uZmlnOiBMYW1iZGFEZXBsb3ltZW50Q29uZmlnLkFMTF9BVF9PTkNFLFxuICAgICAgICBhdXRvUm9sbGJhY2s6IHtcbiAgICAgICAgICBkZXBsb3ltZW50SW5BbGFybTogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgYWxhcm1zOiBbbmV3IGNsb3Vkd2F0Y2guQWxhcm0oc3RhY2ssICdGYWlsdXJlcycsIHtcbiAgICAgICAgICBtZXRyaWM6IGFsaWFzLm1ldHJpY0Vycm9ycygpLFxuICAgICAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX1RIUkVTSE9MRCxcbiAgICAgICAgICB0aHJlc2hvbGQ6IDEsXG4gICAgICAgICAgZXZhbHVhdGlvblBlcmlvZHM6IDFcbiAgICAgICAgfSldXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVEZXBsb3k6OkRlcGxveW1lbnRHcm91cCcsIHtcbiAgICAgICAgQXV0b1JvbGxiYWNrQ29uZmlndXJhdGlvbjoge1xuICAgICAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgRXZlbnRzOiBbXG4gICAgICAgICAgICBcIkRFUExPWU1FTlRfRkFJTFVSRVwiLFxuICAgICAgICAgIF1cbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdpbXBvcnRlZCB3aXRoIGZyb21MYW1iZGFEZXBsb3ltZW50R3JvdXBBdHRyaWJ1dGVzJzoge1xuICAgICAgJ2RlZmF1bHRzIHRoZSBEZXBsb3ltZW50IENvbmZpZyB0byBDYW5hcnkxMFBlcmNlbnQ1TWludXRlcycodGVzdDogVGVzdCkge1xuICAgICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgICAgICBjb25zdCBsYW1iZGFBcHAgPSBjb2RlZGVwbG95LkxhbWJkYUFwcGxpY2F0aW9uLmZyb21MYW1iZGFBcHBsaWNhdGlvbk5hbWUoc3RhY2ssICdMQScsICdMYW1iZGFBcHBsaWNhdGlvbicpO1xuICAgICAgICBjb25zdCBpbXBvcnRlZEdyb3VwID0gY29kZWRlcGxveS5MYW1iZGFEZXBsb3ltZW50R3JvdXAuZnJvbUxhbWJkYURlcGxveW1lbnRHcm91cEF0dHJpYnV0ZXMoc3RhY2ssICdMREcnLCB7XG4gICAgICAgICAgYXBwbGljYXRpb246IGxhbWJkYUFwcCxcbiAgICAgICAgICBkZXBsb3ltZW50R3JvdXBOYW1lOiAnTGFtYmRhRGVwbG95bWVudEdyb3VwJyxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgdGVzdC5lcXVhbChpbXBvcnRlZEdyb3VwLmRlcGxveW1lbnRDb25maWcsIExhbWJkYURlcGxveW1lbnRDb25maWcuQ0FOQVJZXzEwUEVSQ0VOVF81TUlOVVRFUyk7XG5cbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9LFxuICAgIH0sXG4gIH0sXG59O1xuIl19