import * as cdk from '@aws-cdk/core';
import { IOpenIdConnectProvider } from './oidc-provider';
import { Condition, Conditions, PolicyStatement } from './policy-statement';
import { ISamlProvider } from './saml-provider';
/**
 * Any object that has an associated principal that a permission can be granted to.
 *
 * @stability stable
 */
export interface IGrantable {
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    readonly grantPrincipal: IPrincipal;
}
/**
 * Represents a logical IAM principal.
 *
 * An IPrincipal describes a logical entity that can perform AWS API calls
 * against sets of resources, optionally under certain conditions.
 *
 * Examples of simple principals are IAM objects that you create, such
 * as Users or Roles.
 *
 * An example of a more complex principals is a `ServicePrincipal` (such as
 * `new ServicePrincipal("sns.amazonaws.com")`, which represents the Simple
 * Notifications Service).
 *
 * A single logical Principal may also map to a set of physical principals.
 * For example, `new OrganizationPrincipal('o-1234')` represents all
 * identities that are part of the given AWS Organization.
 *
 * @stability stable
 */
export interface IPrincipal extends IGrantable {
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     *
     * @stability stable
     */
    readonly assumeRoleAction: string;
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    readonly policyFragment: PrincipalPolicyFragment;
    /**
     * The AWS account ID of this principal.
     *
     * Can be undefined when the account is not known
     * (for example, for service principals).
     * Can be a Token - in that case,
     * it's assumed to be AWS::AccountId.
     *
     * @stability stable
     */
    readonly principalAccount?: string;
    /**
     * (deprecated) Add to the policy of this principal.
     *
     * @returns true if the statement was added, false if the principal in
     * question does not have a policy document to add the statement to.
     * @deprecated Use `addToPrincipalPolicy` instead.
     */
    addToPolicy(statement: PolicyStatement): boolean;
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPrincipalPolicy(statement: PolicyStatement): AddToPrincipalPolicyResult;
}
/**
 * Result of calling `addToPrincipalPolicy`.
 *
 * @stability stable
 */
export interface AddToPrincipalPolicyResult {
    /**
     * Whether the statement was added to the identity's policies.
     *
     * @stability stable
     */
    readonly statementAdded: boolean;
    /**
     * Dependable which allows depending on the policy change being applied.
     *
     * @default - Required if `statementAdded` is true.
     * @stability stable
     */
    readonly policyDependable?: cdk.IDependable;
}
/**
 * Base class for policy principals.
 *
 * @stability stable
 */
export declare abstract class PrincipalBase implements IPrincipal {
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The AWS account ID of this principal.
     *
     * Can be undefined when the account is not known
     * (for example, for service principals).
     * Can be a Token - in that case,
     * it's assumed to be AWS::AccountId.
     *
     * @stability stable
     */
    readonly principalAccount: string | undefined;
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    abstract readonly policyFragment: PrincipalPolicyFragment;
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     *
     * @stability stable
     */
    readonly assumeRoleAction: string;
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPolicy(statement: PolicyStatement): boolean;
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPrincipalPolicy(_statement: PolicyStatement): AddToPrincipalPolicyResult;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
    /**
     * JSON-ify the principal.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toJSON(): {
        [key: string]: string[];
    };
    /**
     * Returns a new PrincipalWithConditions using this principal as the base, with the passed conditions added.
     *
     * When there is a value for the same operator and key in both the principal and the
     * conditions parameter, the value from the conditions parameter will be used.
     *
     * @returns a new PrincipalWithConditions object.
     * @stability stable
     */
    withConditions(conditions: Conditions): IPrincipal;
}
/**
 * An IAM principal with additional conditions specifying when the policy is in effect.
 *
 * For more information about conditions, see:
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html
 *
 * @stability stable
 */
export declare class PrincipalWithConditions implements IPrincipal {
    private readonly principal;
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     *
     * @stability stable
     */
    readonly assumeRoleAction: string;
    private additionalConditions;
    /**
     * @stability stable
     */
    constructor(principal: IPrincipal, conditions: Conditions);
    /**
     * Add a condition to the principal.
     *
     * @stability stable
     */
    addCondition(key: string, value: Condition): void;
    /**
     * Adds multiple conditions to the principal.
     *
     * Values from the conditions parameter will overwrite existing values with the same operator
     * and key.
     *
     * @stability stable
     */
    addConditions(conditions: Conditions): void;
    /**
     * The conditions under which the policy is in effect.
     *
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     *
     * @stability stable
     */
    get conditions(): Record<string, any>;
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPolicy(statement: PolicyStatement): boolean;
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPrincipalPolicy(statement: PolicyStatement): AddToPrincipalPolicyResult;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
    /**
     * JSON-ify the principal.
     *
     * Used when JSON.stringify() is called
     *
     * @stability stable
     */
    toJSON(): {
        [key: string]: string[];
    };
    private mergeConditions;
}
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 *
 * @stability stable
 */
export declare class PrincipalPolicyFragment {
    readonly principalJson: {
        [key: string]: string[];
    };
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    readonly conditions: Conditions;
    /**
     * @param principalJson JSON of the "Principal" section in a policy statement.
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    constructor(principalJson: {
        [key: string]: string[];
    }, 
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    conditions?: Conditions);
}
/**
 * Specify a principal by the Amazon Resource Name (ARN).
 *
 * You can specify AWS accounts, IAM users, Federated SAML users, IAM roles, and specific assumed-role sessions.
 * You cannot specify IAM groups or instance profiles as principals
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
 * @stability stable
 */
export declare class ArnPrincipal extends PrincipalBase {
    readonly arn: string;
    /**
     * @param arn Amazon Resource Name (ARN) of the principal entity (i.e. arn:aws:iam::123456789012:user/user-name).
     * @stability stable
     */
    constructor(arn: string);
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * Specify AWS account ID as the principal entity in a policy to delegate authority to the account.
 *
 * @stability stable
 */
export declare class AccountPrincipal extends ArnPrincipal {
    readonly accountId: any;
    /**
     * @param accountId AWS account ID (i.e. 123456789012).
     * @stability stable
     */
    constructor(accountId: any);
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * Options for a service principal.
 *
 * @stability stable
 */
export interface ServicePrincipalOpts {
    /**
     * The region in which the service is operating.
     *
     * @default the current Stack's region.
     * @stability stable
     */
    readonly region?: string;
    /**
     * Additional conditions to add to the Service Principal.
     *
     * @default - No conditions
     * @stability stable
     */
    readonly conditions?: {
        [key: string]: any;
    };
}
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 *
 * @stability stable
 */
export declare class ServicePrincipal extends PrincipalBase {
    readonly service: string;
    private readonly opts;
    /**
     * @param service AWS service (i.e. sqs.amazonaws.com).
     * @stability stable
     */
    constructor(service: string, opts?: ServicePrincipalOpts);
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * A principal that represents an AWS Organization.
 *
 * @stability stable
 */
export declare class OrganizationPrincipal extends PrincipalBase {
    readonly organizationId: string;
    /**
     * @param organizationId The unique identifier (ID) of an organization (i.e. o-12345abcde).
     * @stability stable
     */
    constructor(organizationId: string);
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * A policy principal for canonicalUserIds - useful for S3 bucket policies that use Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 * @stability stable
 */
export declare class CanonicalUserPrincipal extends PrincipalBase {
    readonly canonicalUserId: string;
    /**
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     * @stability stable
     */
    constructor(canonicalUserId: string);
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * Principal entity that represents a federated identity provider such as Amazon Cognito, that can be used to provide temporary security credentials to users who have been authenticated.
 *
 * Additional condition keys are available when the temporary security credentials are used to make a request.
 * You can use these keys to write policies that limit the access of federated users.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#condition-keys-wif
 * @stability stable
 */
export declare class FederatedPrincipal extends PrincipalBase {
    readonly federated: string;
    readonly conditions: Conditions;
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     *
     * @stability stable
     */
    readonly assumeRoleAction: string;
    /**
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito).
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    constructor(federated: string, conditions: Conditions, assumeRoleAction?: string);
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * A principal that represents a federated identity provider as Web Identity such as Cognito, Amazon, Facebook, Google, etc.
 *
 * @stability stable
 */
export declare class WebIdentityPrincipal extends FederatedPrincipal {
    /**
     * @param identityProvider identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito).
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    constructor(identityProvider: string, conditions?: Conditions);
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * A principal that represents a federated identity provider as from a OpenID Connect provider.
 *
 * @stability stable
 */
export declare class OpenIdConnectPrincipal extends WebIdentityPrincipal {
    /**
     * @param openIdConnectProvider OpenID Connect provider.
     * @param conditions The conditions under which the policy is in effect.
     * @stability stable
     */
    constructor(openIdConnectProvider: IOpenIdConnectProvider, conditions?: Conditions);
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * Principal entity that represents a SAML federated identity provider.
 *
 * @stability stable
 */
export declare class SamlPrincipal extends FederatedPrincipal {
    /**
     * @stability stable
     */
    constructor(samlProvider: ISamlProvider, conditions: Conditions);
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * Principal entity that represents a SAML federated identity provider for programmatic and AWS Management Console access.
 *
 * @stability stable
 */
export declare class SamlConsolePrincipal extends SamlPrincipal {
    /**
     * @stability stable
     */
    constructor(samlProvider: ISamlProvider, conditions?: Conditions);
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * Use the AWS account into which a stack is deployed as the principal entity in a policy.
 *
 * @stability stable
 */
export declare class AccountRootPrincipal extends AccountPrincipal {
    /**
     * @stability stable
     */
    constructor();
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * A principal representing all identities in all accounts.
 *
 * @stability stable
 */
export declare class AnyPrincipal extends ArnPrincipal {
    /**
     * @stability stable
     */
    constructor();
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
/**
 * (deprecated) A principal representing all identities in all accounts.
 *
 * @deprecated use `AnyPrincipal`
 */
export declare class Anyone extends AnyPrincipal {
}
/**
 * Represents a principal that has multiple types of principals.
 *
 * A composite principal cannot
 * have conditions. i.e. multiple ServicePrincipals that form a composite principal
 *
 * @stability stable
 */
export declare class CompositePrincipal extends PrincipalBase {
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     *
     * @stability stable
     */
    readonly assumeRoleAction: string;
    private readonly principals;
    /**
     * @stability stable
     */
    constructor(...principals: PrincipalBase[]);
    /**
     * Adds IAM principals to the composite principal.
     *
     * Composite principals cannot have
     * conditions.
     *
     * @param principals IAM principals that will be added to the composite principal.
     * @stability stable
     */
    addPrincipals(...principals: PrincipalBase[]): this;
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    get policyFragment(): PrincipalPolicyFragment;
    /**
     * Returns a string representation of an object.
     *
     * @stability stable
     */
    toString(): string;
}
