"""
Copied from https://github.com/AikidoSec/firewall-node
"""

import pytest
import time
from ipaddress import ip_network, ip_address
from . import IPMatcher

test_ip_ranges = [
    "8.8.4.0/24",
    "8.8.8.0/24",
    "8.34.208.0/20",
    "8.35.192.0/20",
    "23.236.48.0/20",
    "23.251.128.0/19",
    "34.0.0.0/15",
    "34.2.0.0/16",
    "34.3.0.0/23",
    "34.3.3.0/24",
    "34.3.4.0/24",
    "34.3.8.0/21",
    "34.3.16.0/20",
    "34.3.32.0/19",
    "34.3.64.0/18",
    "34.4.0.0/14",
    "34.8.0.0/13",
    "34.16.0.0/12",
    "34.32.0.0/11",
    "34.64.0.0/10",
    "34.128.0.0/10",
    "35.184.0.0/13",
    "35.192.0.0/14",
    "35.196.0.0/15",
    "35.198.0.0/16",
    "35.199.0.0/17",
    "35.199.128.0/18",
    "35.200.0.0/13",
    "35.208.0.0/12",
    "35.224.0.0/12",
    "35.240.0.0/13",
    "57.140.192.0/18",
    "64.15.112.0/20",
    "64.233.160.0/19",
    "66.22.228.0/23",
    "66.102.0.0/20",
    "66.249.64.0/19",
    "70.32.128.0/19",
    "72.14.192.0/18",
    "74.125.0.0/16",
    "104.154.0.0/15",
    "104.196.0.0/14",
    "104.237.160.0/19",
    "107.167.160.0/19",
    "107.178.192.0/18",
    "108.59.80.0/20",
    "108.170.192.0/18",
    "108.177.0.0/17",
    "130.211.0.0/16",
    "136.22.160.0/20",
    "136.22.176.0/21",
    "136.22.184.0/23",
    "136.22.186.0/24",
    "136.124.0.0/15",
    "142.250.0.0/15",
    "146.148.0.0/17",
    "152.65.208.0/22",
    "152.65.214.0/23",
    "152.65.218.0/23",
    "152.65.222.0/23",
    "152.65.224.0/19",
    "162.120.128.0/17",
    "162.216.148.0/22",
    "162.222.176.0/21",
    "172.110.32.0/21",
    "172.217.0.0/16",
    "172.253.0.0/16",
    "173.194.0.0/16",
    "173.255.112.0/20",
    "192.158.28.0/22",
    "192.178.0.0/15",
    "193.186.4.0/24",
    "199.36.154.0/23",
    "199.36.156.0/24",
    "199.192.112.0/22",
    "199.223.232.0/21",
    "207.223.160.0/20",
    "208.65.152.0/22",
    "208.68.108.0/22",
    "208.81.188.0/22",
    "208.117.224.0/19",
    "209.85.128.0/17",
    "216.58.192.0/19",
    "216.73.80.0/20",
    "216.239.32.0/19",
    "2001:4860::/32",
    "2404:6800::/32",
    "2404:f340::/32",
    "2600:1900::/28",
    "2605:ef80::/32",
    "2606:40::/32",
    "2606:73c0::/32",
    "2607:1c0:241:40::/60",
    "2607:1c0:300::/40",
    "2607:f8b0::/32",
    "2620:11a:a000::/40",
    "2620:120:e000::/40",
    "2800:3f0::/32",
    "2a00:1450::/32",
    "2c0f:fb50::/32",
    "157.55.39.0/24",
    "207.46.13.0/24",
    "40.77.167.0/24",
    "13.66.139.0/24",
    "13.66.144.0/24",
    "52.167.144.0/24",
    "13.67.10.16/28",
    "13.69.66.240/28",
    "13.71.172.224/28",
    "139.217.52.0/28",
    "191.233.204.224/28",
    "20.36.108.32/28",
    "20.43.120.16/28",
    "40.79.131.208/28",
    "40.79.186.176/28",
    "52.231.148.0/28",
    "20.79.107.240/28",
    "51.105.67.0/28",
    "20.125.163.80/28",
    "40.77.188.0/22",
    "65.55.210.0/24",
    "199.30.24.0/23",
    "40.77.202.0/24",
    "40.77.139.0/25",
    "20.74.197.0/28",
    "20.15.133.160/27",
    "40.77.177.0/24",
    "40.77.178.0/23",
    "40.80.242.63",
    "20.12.141.99",
    "20.49.136.28",
    "51.116.131.221",
    "51.107.40.209",
    "20.40.133.240",
    "20.50.168.91",
    "51.120.48.122",
    "20.193.45.113",
    "40.76.173.151",
    "40.76.163.7",
    "20.185.79.47",
    "52.142.26.175",
    "20.185.79.15",
    "52.142.24.149",
    "40.76.162.208",
    "40.76.163.23",
    "40.76.162.191",
    "40.76.162.247",
    "40.88.21.235",
    "20.191.45.212",
    "52.146.59.12",
    "52.146.59.156",
    "52.146.59.154",
    "52.146.58.236",
    "20.62.224.44",
    "51.104.180.53",
    "51.104.180.47",
    "51.104.180.26",
    "51.104.146.225",
    "51.104.146.235",
    "20.73.202.147",
    "20.73.132.240",
    "20.71.12.143",
    "20.56.197.58",
    "20.56.197.63",
    "20.43.150.93",
    "20.43.150.85",
    "20.44.222.1",
    "40.89.243.175",
    "13.89.106.77",
    "52.143.242.6",
    "52.143.241.111",
    "52.154.60.82",
    "20.197.209.11",
    "20.197.209.27",
    "20.226.133.105",
    "191.234.216.4",
    "191.234.216.178",
    "20.53.92.211",
    "20.53.91.2",
    "20.207.99.197",
    "20.207.97.190",
    "40.81.250.205",
    "40.64.106.11",
    "40.64.105.247",
    "20.72.242.93",
    "20.99.255.235",
    "20.113.3.121",
    "52.224.16.221",
    "52.224.21.53",
    "52.224.20.204",
    "52.224.21.19",
    "52.224.20.249",
    "52.224.20.203",
    "52.224.20.190",
    "52.224.16.229",
    "52.224.21.20",
    "52.146.63.80",
    "52.224.20.227",
    "52.224.20.193",
    "52.190.37.160",
    "52.224.21.23",
    "52.224.20.223",
    "52.224.20.181",
    "52.224.21.49",
    "52.224.21.55",
    "52.224.21.61",
    "52.224.19.152",
    "52.224.20.186",
    "52.224.21.27",
    "52.224.21.51",
    "52.224.20.174",
    "52.224.21.4",
    "51.104.164.109",
    "51.104.167.71",
    "51.104.160.177",
    "51.104.162.149",
    "51.104.167.95",
    "51.104.167.54",
    "51.104.166.111",
    "51.104.167.88",
    "51.104.161.32",
    "51.104.163.250",
    "51.104.164.189",
    "51.104.167.19",
    "51.104.160.167",
    "51.104.167.110",
    "20.191.44.119",
    "51.104.167.104",
    "20.191.44.234",
    "51.104.164.215",
    "51.104.167.52",
    "20.191.44.22",
    "51.104.167.87",
    "51.104.167.96",
    "20.191.44.16",
    "51.104.167.61",
    "51.104.164.147",
    "20.50.48.159",
    "40.114.182.172",
    "20.50.50.130",
    "20.50.50.163",
    "20.50.50.46",
    "40.114.182.153",
    "20.50.50.118",
    "20.50.49.55",
    "20.50.49.25",
    "40.114.183.251",
    "20.50.50.123",
    "20.50.49.237",
    "20.50.48.192",
    "20.50.50.134",
    "51.138.90.233",
    "40.114.183.196",
    "20.50.50.146",
    "40.114.183.88",
    "20.50.50.145",
    "20.50.50.121",
    "20.50.49.40",
    "51.138.90.206",
    "40.114.182.45",
    "51.138.90.161",
    "20.50.49.0",
    "40.119.232.215",
    "104.43.55.167",
    "40.119.232.251",
    "40.119.232.50",
    "40.119.232.146",
    "40.119.232.218",
    "104.43.54.127",
    "104.43.55.117",
    "104.43.55.116",
    "104.43.55.166",
    "52.154.169.50",
    "52.154.171.70",
    "52.154.170.229",
    "52.154.170.113",
    "52.154.171.44",
    "52.154.172.2",
    "52.143.244.81",
    "52.154.171.87",
    "52.154.171.250",
    "52.154.170.28",
    "52.154.170.122",
    "52.143.243.117",
    "52.143.247.235",
    "52.154.171.235",
    "52.154.171.196",
    "52.154.171.0",
    "52.154.170.243",
    "52.154.170.26",
    "52.154.169.200",
    "52.154.170.96",
    "52.154.170.88",
    "52.154.171.150",
    "52.154.171.205",
    "52.154.170.117",
    "52.154.170.209",
    "191.235.202.48",
    "191.233.3.202",
    "191.235.201.214",
    "191.233.3.197",
    "191.235.202.38",
    "20.53.78.144",
    "20.193.24.10",
    "20.53.78.236",
    "20.53.78.138",
    "20.53.78.123",
    "20.53.78.106",
    "20.193.27.215",
    "20.193.25.197",
    "20.193.12.126",
    "20.193.24.251",
    "20.204.242.101",
    "20.207.72.113",
    "20.204.242.19",
    "20.219.45.67",
    "20.207.72.11",
    "20.219.45.190",
    "20.204.243.55",
    "20.204.241.148",
    "20.207.72.110",
    "20.204.240.172",
    "20.207.72.21",
    "20.204.246.81",
    "20.207.107.181",
    "20.204.246.254",
    "20.219.43.246",
    "52.149.25.43",
    "52.149.61.51",
    "52.149.58.139",
    "52.149.60.38",
    "52.148.165.38",
    "52.143.95.162",
    "52.149.56.151",
    "52.149.30.45",
    "52.149.58.173",
    "52.143.95.204",
    "52.149.28.83",
    "52.149.58.69",
    "52.148.161.87",
    "52.149.58.27",
    "52.149.28.18",
    "20.79.226.26",
    "20.79.239.66",
    "20.79.238.198",
    "20.113.14.159",
    "20.75.144.152",
    "20.43.172.120",
    "20.53.134.160",
    "20.201.15.208",
    "20.93.28.24",
    "20.61.34.40",
    "52.242.224.168",
    "20.80.129.80",
    "57.152.72.128",
    "51.8.253.152",
    "20.195.108.47",
    "4.195.133.120",
    "4.228.76.163",
    "4.182.131.108",
    "4.209.224.56",
    "108.141.83.74",
    "4.213.46.14",
    "172.169.17.165",
    "51.8.71.117",
    "20.3.1.178",
    "54.236.1.0/24",
    "31.13.24.0/21",
    "31.13.64.0/18",
    "31.13.65.0/24",
    "31.13.66.0/24",
    "31.13.67.0/24",
    "31.13.70.0/24",
    "31.13.71.0/24",
    "31.13.72.0/24",
    "31.13.73.0/24",
    "31.13.80.0/24",
    "31.13.82.0/24",
    "31.13.83.0/24",
    "31.13.84.0/24",
    "31.13.86.0/24",
    "31.13.88.0/24",
    "31.13.89.0/24",
    "31.13.93.0/24",
    "31.13.96.0/19",
    "45.64.40.0/22",
    "57.141.0.0/24",
    "57.141.3.0/24",
    "57.141.5.0/24",
    "57.141.7.0/24",
    "57.141.8.0/24",
    "57.141.10.0/24",
    "57.141.13.0/24",
    "57.144.0.0/14",
    "57.144.102.0/23",
    "57.144.104.0/23",
    "57.144.108.0/23",
    "57.144.110.0/23",
    "57.144.112.0/23",
    "57.144.116.0/23",
    "57.144.120.0/23",
    "57.144.132.0/23",
    "57.144.134.0/23",
    "57.144.144.0/23",
    "57.144.150.0/23",
    "57.144.152.0/23",
    "57.144.160.0/23",
    "57.144.162.0/23",
    "57.144.172.0/23",
    "57.144.174.0/23",
    "57.144.180.0/23",
    "66.220.144.0/20",
    "66.220.144.0/21",
    "66.220.152.0/21",
    "69.63.176.0/20",
    "69.63.176.0/21",
    "69.63.184.0/21",
    "69.171.224.0/19",
    "69.171.224.0/20",
    "69.171.240.0/20",
    "69.171.250.0/24",
    "74.119.76.0/22",
    "102.132.96.0/20",
    "102.132.101.0/24",
    "103.4.96.0/22",
    "129.134.0.0/17",
    "129.134.25.0/24",
    "129.134.26.0/24",
    "129.134.27.0/24",
    "129.134.28.0/24",
    "129.134.29.0/24",
    "129.134.30.0/23",
    "129.134.30.0/24",
    "129.134.31.0/24",
    "157.240.0.0/17",
    "157.240.0.0/24",
    "157.240.3.0/24",
    "157.240.5.0/24",
    "157.240.7.0/24",
    "157.240.9.0/24",
    "157.240.11.0/24",
    "157.240.13.0/24",
    "157.240.14.0/24",
    "157.240.15.0/24",
    "157.240.17.0/24",
    "157.240.19.0/24",
    "157.240.22.0/24",
    "157.240.24.0/24",
    "157.240.25.0/24",
    "157.240.26.0/24",
    "157.240.27.0/24",
    "157.240.30.0/24",
    "157.240.31.0/24",
    "157.240.192.0/18",
    "157.240.195.0/24",
    "157.240.196.0/24",
    "157.240.199.0/24",
    "157.240.200.0/24",
    "157.240.201.0/24",
    "157.240.202.0/24",
    "157.240.203.0/24",
    "157.240.205.0/24",
    "157.240.209.0/24",
    "157.240.210.0/24",
    "157.240.211.0/24",
    "157.240.212.0/24",
    "157.240.214.0/24",
    "157.240.221.0/24",
    "157.240.223.0/24",
    "157.240.229.0/24",
    "157.240.231.0/24",
    "157.240.234.0/24",
    "157.240.235.0/24",
    "157.240.241.0/24",
    "157.240.243.0/24",
    "157.240.244.0/24",
    "157.240.245.0/24",
    "157.240.247.0/24",
    "157.240.249.0/24",
    "157.240.251.0/24",
    "157.240.252.0/24",
    "157.240.253.0/24",
    "157.240.254.0/24",
    "163.70.128.0/17",
    "163.70.128.0/24",
    "163.70.147.0/24",
    "163.70.151.0/24",
    "163.70.158.0/24",
    "163.70.159.0/24",
    "173.252.64.0/19",
    "173.252.88.0/21",
    "173.252.96.0/19",
    "179.60.192.0/22",
    "179.60.195.0/24",
    "185.60.216.0/22",
    "185.60.217.0/24",
    "185.60.218.0/24",
    "185.60.219.0/24",
    "185.89.216.0/22",
    "185.89.218.0/23",
    "185.89.218.0/24",
    "185.89.219.0/24",
    "204.15.20.0/22",
    "2620:0:1c00::/40",
    "2a03:2880::/32",
    "2a03:2880::/36",
    "2a03:2880:1000::/36",
    "2a03:2880:2000::/36",
    "2a03:2880:3000::/36",
    "2a03:2880:5000::/36",
    "2a03:2880:f001::/48",
    "2a03:2880:f003::/48",
    "2a03:2880:f004::/48",
    "2a03:2880:f007::/48",
    "2a03:2880:f008::/48",
    "2a03:2880:f00a::/48",
    "2a03:2880:f00c::/48",
    "2a03:2880:f00d::/48",
    "2a03:2880:f00e::/48",
    "2a03:2880:f00f::/48",
    "2a03:2880:f011::/48",
    "2a03:2880:f012::/48",
    "2a03:2880:f013::/48",
    "2a03:2880:f01d::/48",
    "2a03:2880:f021::/48",
    "2a03:2880:f023::/48",
    "2a03:2880:f028::/48",
    "2a03:2880:f02c::/48",
    "2a03:2880:f031::/48",
    "2a03:2880:f032::/48",
    "2a03:2880:f034::/48",
    "2a03:2880:f035::/48",
    "2a03:2880:f036::/48",
    "2a03:2880:f03a::/48",
    "2a03:2880:f03d::/48",
    "2a03:2880:f03f::/48",
    "2a03:2880:f042::/48",
    "2a03:2880:f045::/48",
    "2a03:2880:f04e::/48",
    "2a03:2880:f050::/48",
    "2a03:2880:f052::/48",
    "2a03:2880:f053::/48",
    "2a03:2880:f057::/48",
    "2a03:2880:f058::/48",
    "2a03:2880:f05b::/48",
    "2a03:2880:f05e::/48",
    "2a03:2880:f065::/48",
    "2a03:2880:f06f::/48",
    "2a03:2880:f074::/48",
    "2a03:2880:f078::/48",
    "2a03:2880:f07d::/48",
    "2a03:2880:f07e::/48",
    "2a03:2880:f080::/48",
    "2a03:2880:f082::/48",
    "2a03:2880:f083::/48",
    "2a03:2880:f084::/48",
    "2a03:2880:f085::/48",
    "2a03:2880:f08e::/48",
    "2a03:2880:f0a5::/48",
    "2a03:2880:f0fc::/47",
    "2a03:2880:f0fc::/48",
    "2a03:2880:f0fd::/48",
    "2a03:2880:f0ff::/48",
    "2a03:2880:f101::/48",
    "2a03:2880:f103::/48",
    "2a03:2880:f104::/48",
    "2a03:2880:f107::/48",
    "2a03:2880:f108::/48",
    "2a03:2880:f10a::/48",
    "2a03:2880:f10c::/48",
    "2a03:2880:f10d::/48",
    "2a03:2880:f10e::/48",
    "2a03:2880:f10f::/48",
    "2a03:2880:f111::/48",
    "2a03:2880:f112::/48",
    "2a03:2880:f113::/48",
    "2a03:2880:f115::/48",
    "2a03:2880:f121::/48",
    "2a03:2880:f123::/48",
    "2a03:2880:f128::/48",
    "2a03:2880:f12c::/48",
    "2a03:2880:f131::/48",
    "2a03:2880:f132::/48",
    "2a03:2880:f134::/48",
    "2a03:2880:f135::/48",
    "2a03:2880:f136::/48",
    "2a03:2880:f13a::/48",
    "2a03:2880:f13d::/48",
    "2a03:2880:f13f::/48",
    "2a03:2880:f142::/48",
    "2a03:2880:f145::/48",
    "2a03:2880:f14e::/48",
    "2a03:2880:f150::/48",
    "2a03:2880:f152::/48",
    "2a03:2880:f153::/48",
    "2a03:2880:f157::/48",
    "2a03:2880:f158::/48",
    "2a03:2880:f15b::/48",
    "2a03:2880:f15e::/48",
    "2a03:2880:f162::/48",
    "2a03:2880:f169::/48",
    "2a03:2880:f16d::/48",
    "2a03:2880:f170::/48",
    "2a03:2880:f171::/48",
    "2a03:2880:f172::/48",
    "2a03:2880:f173::/48",
    "2a03:2880:f175::/48",
    "2a03:2880:f176::/48",
    "2a03:2880:f177::/48",
    "2a03:2880:f178::/48",
    "2a03:2880:f17b::/48",
    "2a03:2880:f189::/48",
    "2a03:2880:f1fc::/47",
    "2a03:2880:f1fc::/48",
    "2a03:2880:f1fd::/48",
    "2a03:2880:f1ff::/48",
    "2a03:2880:f201::/48",
    "2a03:2880:f203::/48",
    "2a03:2880:f204::/48",
    "2a03:2880:f207::/48",
    "2a03:2880:f208::/48",
    "2a03:2880:f20a::/48",
    "2a03:2880:f20c::/48",
    "2a03:2880:f20d::/48",
    "2a03:2880:f20e::/48",
    "2a03:2880:f20f::/48",
    "2a03:2880:f211::/48",
    "2a03:2880:f212::/48",
    "2a03:2880:f213::/48",
    "2a03:2880:f215::/48",
    "2a03:2880:f221::/48",
    "2a03:2880:f223::/48",
    "2a03:2880:f228::/48",
    "2a03:2880:f22c::/48",
    "2a03:2880:f231::/48",
    "2a03:2880:f232::/48",
    "2a03:2880:f234::/48",
    "2a03:2880:f235::/48",
    "2a03:2880:f236::/48",
    "2a03:2880:f23a::/48",
    "2a03:2880:f23d::/48",
    "2a03:2880:f23f::/48",
    "2a03:2880:f242::/48",
    "2a03:2880:f245::/48",
    "2a03:2880:f24e::/48",
    "2a03:2880:f250::/48",
    "2a03:2880:f252::/48",
    "2a03:2880:f253::/48",
    "2a03:2880:f257::/48",
    "2a03:2880:f258::/48",
    "2a03:2880:f25b::/48",
    "2a03:2880:f25e::/48",
    "2a03:2880:f262::/48",
    "2a03:2880:f269::/48",
    "2a03:2880:f26d::/48",
    "2a03:2880:f270::/48",
    "2a03:2880:f271::/48",
    "2a03:2880:f272::/48",
    "2a03:2880:f273::/48",
    "2a03:2880:f275::/48",
    "2a03:2880:f276::/48",
    "2a03:2880:f277::/48",
    "2a03:2880:f278::/48",
    "2a03:2880:f27b::/48",
    "2a03:2880:f289::/48",
    "2a03:2880:f2ff::/48",
    "2a03:2880:f32a::/48",
    "2a03:2880:f32b::/48",
    "2a03:2880:f32d::/48",
    "2a03:2880:f32e::/48",
    "2a03:2880:f330::/48",
    "2a03:2880:f33a::/48",
    "2a03:2880:f33c::/48",
    "2a03:2880:f342::/48",
    "2a03:2880:f343::/48",
    "2a03:2880:f348::/48",
    "2a03:2880:f34b::/48",
    "2a03:2880:f34c::/48",
    "2a03:2880:f350::/48",
    "2a03:2880:f351::/48",
    "2a03:2880:f356::/48",
    "2a03:2880:f357::/48",
    "2a03:2880:f35a::/48",
    "2a03:2880:f800::/48",
    "2a03:2880:f803::/48",
    "2a03:2880:f804::/48",
    "2a03:2880:f806::/48",
    "2a03:2880:f809::/48",
    "2a03:2880:f80a::/48",
    "2a03:2880:f80c::/48",
    "2a03:2880:ff08::/48",
    "2a03:2880:ff09::/48",
    "2a03:2880:ff0a::/48",
    "2a03:2880:ff0b::/48",
    "2a03:2880:ff0c::/48",
    "64.63.0.0/18",
    "64.63.30.0/24",
    "64.63.31.0/24",
    "64.63.46.0/24",
    "64.63.47.0/24",
    "69.195.160.0/19",
    "69.195.160.0/24",
    "69.195.162.0/24",
    "69.195.166.0/24",
    "69.195.168.0/24",
    "69.195.169.0/24",
    "69.195.171.0/24",
    "69.195.174.0/24",
    "69.195.178.0/24",
    "69.195.179.0/24",
    "69.195.181.0/24",
    "69.195.182.0/24",
    "69.195.183.0/24",
    "69.195.184.0/24",
    "69.195.186.0/24",
    "69.195.187.0/24",
    "103.252.112.0/23",
    "103.252.114.0/23",
    "104.244.40.0/24",
    "104.244.41.0/24",
    "104.244.42.0/24",
    "104.244.44.0/24",
    "104.244.45.0/24",
    "104.244.46.0/24",
    "104.244.47.0/24",
    "188.64.224.0/21",
    "192.133.76.0/22",
    "199.16.156.0/22",
    "199.16.156.0/23",
    "199.59.148.0/22",
    "199.96.56.0/23",
    "202.160.128.0/24",
    "202.160.129.0/24",
    "202.160.130.0/24",
    "202.160.131.0/24",
    "209.237.192.0/19",
    "209.237.192.0/24",
    "209.237.193.0/24",
    "209.237.194.0/24",
    "209.237.195.0/24",
    "209.237.196.0/24",
    "209.237.198.0/24",
    "209.237.199.0/24",
    "2400:6680:f000::/36",
    "2606:1f80:f000::/36",
    "2a04:9d40:f000::/36",
]

ips_to_check = [
    {"ip": "202.160.131.0", "shouldExist": True},
    {"ip": "20.50.168.91", "shouldExist": True},
    {"ip": "2a03:2880:f10c::12", "shouldExist": True},
    {"ip": "35.196.0.12", "shouldExist": True},
    {"ip": "34.3.64.8", "shouldExist": True},
    {"ip": "2a03:2880:f20c::ff", "shouldExist": True},
    {"ip": "192.168.2.1", "shouldExist": False},
    {"ip": "::1", "shouldExist": False},
]

iterations = 500


class SimpleBlockList:
    def __init__(self):
        self.subnets = []

    def add_subnet(self, ip, mask, family=None):
        subnet = ip_network(f"{ip}/{mask}", strict=False)
        self.subnets.append(subnet)

    def add_address(self, ip):
        try:
            addr = ip_address(ip)
            if addr.version == 4:
                subnet = ip_network(f"{ip}/32", strict=False)
            else:
                subnet = ip_network(f"{ip}/128", strict=False)
            self.subnets.append(subnet)
        except ValueError:
            # Skip invalid addresses
            pass

    def check(self, ip, family=None):
        try:
            addr = ip_address(ip)
            for subnet in self.subnets:
                if addr in subnet:
                    return True
            return False
        except ValueError:
            return False


def test_performance_comparison():
    # Create IPMatcher instance
    ip_matcher = IPMatcher(test_ip_ranges)

    # Create SimpleBlockList instance
    block_list = SimpleBlockList()
    for ip_range in test_ip_ranges:
        if "/" in ip_range:
            ip, mask = ip_range.split("/")
            mask = int(mask)
            block_list.add_subnet(ip, mask)
        else:
            block_list.add_address(ip_range)

    # Time IPMatcher checks
    start_ip_matcher = time.perf_counter()
    for _ in range(iterations):
        for to_check in ips_to_check:
            blocked = ip_matcher.has(to_check["ip"])
            assert (
                blocked == to_check["shouldExist"]
            ), f"Blocklist check returned wrong result for {to_check['ip']}"
    end_ip_matcher = time.perf_counter()
    ip_matcher_time = (
        (end_ip_matcher - start_ip_matcher) / iterations * 1000
    )  # Convert to milliseconds per iteration

    # Time SimpleBlockList checks
    start_block_list = time.perf_counter()
    for _ in range(iterations):
        for to_check in ips_to_check:
            blocked = block_list.check(to_check["ip"])
            assert (
                blocked == to_check["shouldExist"]
            ), f"Blocklist check returned wrong result for {to_check['ip']}"
    end_block_list = time.perf_counter()
    block_list_time = (
        (end_block_list - start_block_list) / iterations * 1000
    )  # Convert to milliseconds per iteration

    # Calculate percentage difference
    if ip_matcher_time == 0:
        percentage_diff = float("inf")
    else:
        percentage_diff = ((block_list_time - ip_matcher_time) / ip_matcher_time) * 100

    # Print the results for information
    print(f"IPMatcher average time per iteration: {ip_matcher_time:.6f} ms")
    print(f"BlockList average time per iteration: {block_list_time:.6f} ms")
    print(f"Percentage difference: {percentage_diff:.2f}%")

    # Assert that IPMatcher is faster than BlockList by at least 25%
    assert (
        percentage_diff > 25
    ), f"IPMatcher was not sufficiently faster (only {percentage_diff:.2f}% faster)"
