# Copyright 2024 Market Logic Software AG. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


"""
This module contains the model classes for answers.
"""

from typing import Optional, List
from datetime import datetime
from pydantic import Field
from deepsights.utils import DeepSightsIdModel
from deepsights.userclient.resources._model import (
    DocumentEvidence,
    ContentStoreEvidence,
)


#################################################
class AnswerV2(DeepSightsIdModel):
    """
    Represents a answer V2 generated by the DeepSights API.

    Attributes:

        status (str): The processing status of the answer.
        permission_validation (str): The permission validation of the answer for the caller.
        question (str): The question that the answer is answering.
        answer (Optional[str]): The AI-generated answer in markdown format.
        watchouts (Optional[str]): The AI-generated watchouts in markdown format.
        document_sources (List[ReportEvidence]): List of evidence from documents used in the answer.
        secondary_sources (List[ReportEvidence]): List of evidence from secondary sources used in the answer.
        news_sources (List[ReportEvidence]): List of evidence from news sources used in the answer.
        document_suggestions (List[ReportEvidence]): List of suggestions from documents as further reading for the question.
        secondary_suggestions (List[ReportEvidence]): List of suggestions from secondary sources as further reading for the question.
        news_suggestions (List[ReportEvidence]): List of suggestions from news sources as further reading for the question.
    """

    permission_validation: str = Field(
        description="The permission validation of the answer for the caller.",
    )
    status: str = Field(description="The processing status of the answer.")
    question: str = Field(description="The question that the answer is answering.")
    answer: Optional[str] = Field(
        default=None, description="The AI-generated answer in markdown format."
    )
    watchouts: Optional[str] = Field(
        default=None, description="The AI-generated watchouts in markdown format."
    )
    document_sources: List[DocumentEvidence] = Field(
        default=[], description="List of evidence from documents used in the answer."
    )
    secondary_sources: List[ContentStoreEvidence] = Field(
        default=[],
        description="List of evidence from secondary sources used in the answer.",
    )
    news_sources: List[ContentStoreEvidence] = Field(
        default=[], description="List of evidence from news sources used in the answer."
    )
    document_suggestions: List[DocumentEvidence] = Field(
        default=[], description="List of suggestions from documents as further reading for the question."
    )
    secondary_suggestions: List[ContentStoreEvidence] = Field(
        default=[],
        description="List of suggestions from secondary sources as further reading for the question.",
    )
    news_suggestions: List[ContentStoreEvidence] = Field(
        default=[], description="List of suggestions from news sources as further reading for the question."
    )